/**
 * ITQ's for use in OSL. Based on the media library simple-question types.
 * The main difference is that the questions are created in the HTML file and identified using HTML ids.
 * Please see the file for an example.
 */

// Data used for questions. Based on media library simple question types data format.
// This is usually supplied by editorial team in this format.
var questions = {

    "mc1_data": {
        "type": "mc",
        "corrects": [1],
        "extraFeedback": "",
        "extraCorrectFeedback": "",
        "extraIncorrectFeedback": "",
        "numTries": 0
    },
    "mc2_data": {
        "type": "mc",
        "corrects": [3],
        "extraFeedback": "",
        "extraCorrectFeedback": "",
        "extraIncorrectFeedback": "",
        "numTries": 0
    },
    "mc3_data": {
        "type": "mc",
        "corrects": [1, 3],
        "extraFeedback": "<p>As bands are visible in the other lanes, it is very unlikely that the proteins did not transfer onto the nitrocellulose from the lane in question. Sometimes air bubbles between the nitrocellulose and the gel can prevent transfer, but these are usually apparent as circular ‘spots’ in the developed blot. If either the primary or secondary antibody had been omitted, then no bands would be detected.</p>",
        "extraCorrectFeedback": "",
        "extraIncorrectFeedback": "",
        "numTries": 0
    },
    "qn3_data": {
        "type": "sc",
        "corrects": [1],
        "extraFeedback": "",
        "extraCorrectFeedback": "",
        "extraIncorrectFeedback": "",
        "numTries": 0
    },
    "scom1_data": {
        "type": "senc",
        "corrects": [3, 11, 23, 33],
        "extraFeedback": "",
        "extraCorrectFeedback": "",
        "extraIncorrectFeedback": "",
        "numTries": 0
    },
    "scom2_data": {
        "type": "senc",
        "corrects": [2, 11, 22],
        "extraFeedback": "",
        "extraCorrectFeedback": "",
        "extraIncorrectFeedback": "",
        "numTries": 0
    },
    "scom3_data": {
        "type": "senc",
        "corrects": [1, 12, 23, 32],
        "extraFeedback": "<p></p><p>Given the <em>M</em><sub>r</sub> of 170 000 for the unreduced protein and <em>M</em><sub>r</sub> values of 58 000 and 25 000 for the two polypeptide constituents, the protein may be composed of two chains of 58 000 and two chains of 25 000, or it could be a trimer of the 58 000 band and the 25 000 band is a contaminant. It is also possible that the protein has at least one chain of <em>M</em><sub>r</sub> = 25 000 and one chain of <em>M</em><sub>r</sub> = 58 000, but may have additional chains that were not detected. The intensity of the bands gives an indication of the relative amounts of protein in each and the fact that these were similar suggests that a composition of two chains of 58 000 and two chains of 25 000 is most likely. </p>",
        "extraCorrectFeedback": "",
        "extraIncorrectFeedback": "",
        "numTries": 0
    },
    "qnsc1_data": {
        "type": "sc",
        "corrects": [2],
        "extraFeedback": "<p></p><p>The protein could be resolved by increasing the concentration of acrylamide in the gel, to effectively retard its movement through the gel. Running the gel for less time or reducing the voltage would not affect the resolution of the proteins in the samples. Decreasing the concentration of SDS in the sample may reduce the overall net negative charge associated with the protein but would affect other proteins similarly and would not improve resolution.</p>",
        "extraCorrectFeedback": "",
        "extraIncorrectFeedback": "",
        "numTries": 0
    }

};


/*
 * oslItqs namespace.
 */
var oslItqs = (function () {
    /*
     * @Object sharedProps Options for adding properties that are shared by all itqs in the same package.
     */

    var sharedProps = {
        "maxTries": 2,
        "removeIncorrect": true,
        "partialFeedback": "Your answer is partially correct. ",
        "correctFeedback": "Your answer is correct. ",
        //"extraCorrectFeedback":"",
        "incorrectFeedback": "Your answer is incorrect. ",
        //"extraIncorrectFeedback": "",
        "removeIncorrectFeedback": "Wrong answers have been removed. ",
        "intermediateFeedback": "Try again. ",
        "noSelectionFeedback": "Please select an answer. "
        //"extraFeedback":""

    };

    /*
     * multiChoice - Function to check answers for multiple choice questions. This function is called from a button in the
     * HTML next to a list of checkboxes.
     * @param {string} questionId - ID of question to be checked. Used to find answers in object above.
     * @param {string} groupName - Name of the checkbox input group, used to identify what is selected.
     */
    var multiChoice = function (questionId, groupName) {
        var answerArray = [];
        $("#" + questionId + " input[name='" + groupName + "']").each(function () {
            if (this.checked) {
                answerArray.push(parseInt($(this).val()));
            }
        });
        checkAnswer(questionId, answerArray);
    };

    /*
     * singleChoice - Function to check answer for single choice question. This function is called from a button in the
     * HTML next to a list of radio buttons.
     * @param {string} questionId - ID of question to be checked. Used to find answers in object above.
     * @param {string} groupName - Name of the radio input group, used to identify what is selected.
     */
    var singleChoice = function (questionId, groupName) {
        var answerArray = [];
        if (!isNaN(parseInt($("input[name='" + groupName + "']:checked").val()))) {
            answerArray.push(parseInt($("input[name='" + groupName + "']:checked").val()));
        }
        checkAnswer(questionId, answerArray);
    };
    /* sentenceComplete - Function to check answers for sentence complete question. The function is called from a button in
     * the HTML next to a paragraph containing drop down sentence complete questions.
     * @param {string} questionId - ID of question to be checked. Used to find answers in the object above.
     */
    var sentenceComplete = function (questionId) {
        var answerArray = [];
        $("#" + questionId + " select option:selected").each(function () {
            if (!isNaN(parseInt($(this).val()))) {
                answerArray.push(parseInt($(this).val()));
            }
        });
        checkAnswer(questionId, answerArray);
    };

    /*
     * numericInput - Function to check answers for numeric input questions. The function is called from a button in the
     * HTML next to a list of numeric inputs. This is a much more simple version than is available in the Media library.
     * This is the functionality that was required at the time. Would expect more functionality to be added over time.
     * @param {string} questionId - ID of question to be checked. Used to find answers in the object above.
     */

    var numericInput = function (questionId) {
        var answerArray = [];
        $("#" + questionId + " input").each(function () {
            if (!isNaN(parseFloat($(this).val()))) {
                answerArray.push(parseFloat($(this).val()));
            }
            checkAnswer(questionId, answerArray);
        });
    };

    /*
     * checkAnswer - Function called by every question type.
     * @param {string} questionId - The id of the question to check above.
     * @param {*} - Array of numbers corresponding to the answers selected by student.
     */
    function checkAnswer(questionId, answer) {
        var feedback = "";
        var item = questionId + "_data";
        var obj = questions[item];
        var qnType = obj.type;
        // Check student has selected an answer.
        if (answer.length == 0) {
            feedback = "Please select an answer. "
        } else {
            obj.numTries++;
        }

        var currentCorrects = 0;
        var currentIncorrects = 0;
        var incorrectAnswers = [];
        for (var i = 0; i < answer.length; i++) {
            if (obj.corrects.indexOf(answer[i]) >= 0) {
                currentCorrects++;
            } else {
                currentIncorrects++;
                if (qnType == "ni") {
                    incorrectAnswers.push(i);
                } else {
                    incorrectAnswers.push(answer[i]);
                }
            }
        }

        // Check to see if the student has any tries left.
        if (currentCorrects != obj.corrects.length && obj.numTries >= sharedProps.maxTries) {
            $("#" + questionId).find("input, select, button").attr('disabled', 'disabled');
            // If student has used maximum number of tries show the correct answers.
            showCorrectAnswers(questionId, obj, qnType);
            var feedbackString = "Correct answers are now shown. " + obj.extraFeedback;
            displayFeedback(questionId, feedbackString);

        } else {
            //If student still has tries left, check if the answers are correct.
            var numCorrects = obj.corrects.length;

            if (currentCorrects == obj.corrects.length && currentIncorrects == 0) {
                feedback = sharedProps.correctFeedback + obj.extraFeedback;
                $("#" + questionId).find("input, button, select").attr('disabled', 'disabled');
            } else if ((currentCorrects > 0 && currentCorrects < numCorrects) || (currentCorrects > 0 && currentIncorrects > 0)) {
                feedback = sharedProps.partialFeedback;
                if (obj.numTries > 1) {
                    feedback += obj.extraIncorrectFeedback;
                }
            } else if (currentCorrects == 0 && currentIncorrects > 0) {
                feedback = sharedProps.incorrectFeedback;
                if (obj.numTries > 1) {
                    feedback += obj.extraIncorrectFeedback;
                }
            }
            if (sharedProps.removeIncorrect && incorrectAnswers.length > 0) {
                feedback += " " + sharedProps.removeIncorrectFeedback;
                removeIncorrectAnswers(questionId, incorrectAnswers, qnType);
            }
            displayFeedback(questionId, feedback);
        }
    }

    function removeIncorrectAnswers(questionId, incorrectArray, qnType) {
        if (qnType == "mc" || qnType == "sc") {
            $.each(incorrectArray, function (i) {
                $("#" + questionId + " input").eq(parseInt(incorrectArray[i]) - 1).prop("checked", false);
            });
        } else if (qnType == "ni") {
            $.each(incorrectArray, function (i) {
                var sel = incorrectArray[i] + 1;
                $("#" + questionId).find("input:nth-of-type(" + sel + ")").val("");
            });
        } else if (qnType == "senc") {
            $.each(incorrectArray, function (i) {
                $("#" + questionId + " select option[value = '" + incorrectArray[i] + "']").prop("selected", false);
            });
        }
    }

    function showCorrectAnswers(questionId, question, qnType) {

        if (qnType == "ni") {
            $.each(question.corrects, function (k, v) {
                var sel = k + 1;
                $("#" + questionId + " input:nth-of-type(" + sel + ")").val(v);
            });
        } else if (qnType == "senc") {
            $.each(question.corrects, function (k, v) {
                $("#" + questionId + " select option[value = '" + v + "']").prop("selected", true);
            });
        }
        else if (qnType == "mc" || qnType == "sc") {
            var i;
            for (i = 1; i <= $("#" + questionId + " input").length; ++i) {
                $("#" + questionId + " input[value = '" + i + "']").prop("checked", false);
            }

            $.each(question.corrects, function (k, v) {
                $("#" + questionId + " input[value = '" + v + "']").prop("checked", true);
            });
        }
        displayFeedback(questionId, "Correct answers are now shown. ");
    }

    /*
     * Uses question ID and feedback string to show a div with the class 'feedback' to student.
     * @param {string} questionId - The div id used for this question.
     * @param {string} feedback - Text to be displayed to student. HTML formatting available.
     *
     */
    function displayFeedback(questionId, feedback) {
        $("#" + questionId + " div.feedback").html(feedback);
    }

    /* Used for the 'reveal answer' questions.
     * @param {Object} event - click or keypress event
     * @param {string} id - the question ID.
     */
    var revealAnswerBtn = function (event, id) {
        $("#" + id + "_answer").removeClass("notvisible");
        $("#" + id).attr("disabled", "disabled");
    };

    return {
        oslItqs: oslItqs,
        multiChoice: multiChoice,
        singleChoice: singleChoice,
        sentenceComplete: sentenceComplete,
        numericInput: numericInput,
        revealAnswerBtn: revealAnswerBtn
    }
})();

$(document).ready(function () {
        $("#rev1").on("click", function (evt) {
            oslItqs.revealAnswerBtn(evt, this.id)
        });
        $("#rev2").on("click", function (evt) {
            oslItqs.revealAnswerBtn(evt, this.id)
        });
        $("#rev3").on("click", function (evt) {
            oslItqs.revealAnswerBtn(evt, this.id)
        });
        $("#rev4").on("click", function (evt) {
            oslItqs.revealAnswerBtn(evt, this.id)
        });
        $("#rev5").on("click", function (evt) {
            oslItqs.revealAnswerBtn(evt, this.id)
        });

        $("#mc1-submit").bind("click", function () {
            oslItqs.multiChoice("mc1", "mc1_data");
        });
        $("#mc2-submit").bind("click", function () {
            oslItqs.multiChoice("mc2", "mc2_data");
        });
        $("#mc3-submit").bind("click", function () {
            oslItqs.multiChoice("mc3", "mc3_data");
        });
        $("#sc1-submit").bind("click", function () {
            oslItqs.singleChoice("qnsc1", "sc1");
        });
        $("#scom1-submit").bind("click", function () {
            oslItqs.sentenceComplete("scom1");
        });
        $("#scom2-submit").bind("click", function () {
            oslItqs.sentenceComplete("scom2");
        });
        $("#scom3-submit").bind("click", function () {
            oslItqs.sentenceComplete("scom3");
        });
    }
);

