/**
 * @fileOverview Trumps2 - A generic "top trumps" game
 * *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.ImageLoader');
OU.require('OU.util.ModalMessage');
OU.require('OU.util.ModalInfo');
OU.require('OU.util.Instruction');

KEY_LEFT = 37;
KEY_UP = 38;
KEY_RIGHT = 39;
KEY_DOWN = 40;

/**
 * @class Trumps2 - Top Trumps2 card game
 * @extends OU.util.Activity
 * @param {Object} data - Holds the data content for a specific instance of the activity
 * @param {String} instance - (Optional) A unique identifier name for this instance, defaults to 'a1'
 * @param {OU.util.Controller} controller - (Optional) A reference to the controller that initialised this instance, if undefined, the superclass will generate a new controller and create the reference to it as 'this.controller'
 */
OU.activity.Trumps2 = function(data, instance, controller) {
    var trumps = this;
    var cardWidth, cardHeight, gapWidth,
            visibleCards, yourCard, cpuCard, coverCard, yourPack, cpuPack,
            fullPack, progressBar, browseOffset, targetBrowseOffset, return2MenuDiv,
            playing = false;

    //Override the timeout function, so we can cancel all pending timeouts
    var _timeoutIDs = [];
    var setTimeout = function(fn, ms) {
        var id = window.setTimeout(fn, ms);
        _timeoutIDs.push(id);
    };
    var clearAllTimeouts = function() {
        for (var i = _timeoutIDs.length; i--; ) {
            window.clearTimeout(_timeoutIDs[i]);
        }
        _timeoutIDs = [];
    };

    OU.Trumps2 = this;

    /**
     * canvasView - this is the function that is first called when the activity is launched,
     *              assuming you are in a browser that supports canvas.
     *
     * Typical tasks for this function are:
     * <ul>
     * <li> Define and initialise and activity wide variables</li>
     * <li> Initialise Layers & Divs</li>
     * <li> Call a loading function</li>
     * <li> Initiating the activity by calling any methods that</li>
     * </ul>
     */
    OU.activity.Trumps2.prototype.canvasView = function() {

        this.computerPlays = true;
        this.dumbComputer = false;

        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.data.activityContent,
            onLoad: function() {
                //Load welcome screen
                trumps.backgroundDiv = new OU.util.Div({
                    style: "text-align:center;background: url('" + trumps.data.activityContent.welcomeBackground + "') center center no-repeat; background-size:cover",
                    innerHTML: '<div id="menuDiv" class="goneAway"></div>',
                    showScroll: false
                });
                trumps.menuDiv = document.getElementById('menuDiv');
                trumps.menuDiv.innerHTML = '<p><img style="width:70%;" src="' + trumps.data.activityContent.titleImage + '"/></p><div id="playButton" class="menuButton" tabIndex="11">Play</div><br/><div id="optionsButton" class="menuButton" tabIndex="12">Game options</div><br/><div id="instructionsButton" class="menuButton" tabIndex="13">How to play</div><br/><div id="browseButton" class="menuButton" tabIndex="14">Browse cards</div>';
                trumps.resize();
                OU.addClass(trumps.menuDiv, 'ease');
                var playButton = document.getElementById('playButton');
                OU.events.addListener(playButton, function() {
                    trumps.startPlay();
                }, 'tap');
                var optionsButton = document.getElementById('optionsButton');
                OU.events.addListener(optionsButton, function() {
                    trumps.settingsMenu();
                }, 'tap');
                var instructionsButton = document.getElementById('instructionsButton');
                OU.events.addListener(instructionsButton, function() {
                    if (!playing) {
                        trumps.instructionsView();
                    }
                }, 'tap');
                var browseButton = document.getElementById('browseButton');
                OU.events.addListener(browseButton, function() {
                    if (!playing) {
                        trumps.hideMenu();
                        setTimeout(function() {
                            trumps.browseView();
                        }, 1000);
                    }
                }, 'tap');
                OU.events.addSwipeListener(function(events) {
                    trumps.browseViewSwipe(events);
                });
                OU.events.addKeyListener(function(key) {
                    trumps.keyPress(key);
                });
                setTimeout(function() {
                    trumps.showMenu();
                    trumps.orientationWarning();
                }, 200);
                trumps.loadCards();
            }
        });

    };
    OU.activity.Trumps2.prototype.keyPress = function(keycode) {
        switch (keycode) {
            case 27: //escape
                this.showMenu();
                break;
            case KEY_UP:
            case KEY_RIGHT:
                if (fullPack) {
                    trumps.moveBrowseView(1);
                }
                break;
            case KEY_DOWN:
            case KEY_LEFT:
                if (fullPack) {
                    trumps.moveBrowseView(-1);
                }
                break;
        }

    };
    /**
     * Advise the user to rotate their device if they are trying to use landscape on a smaller device
     */
    OU.activity.Trumps2.prototype.orientationWarning = function() {
        if (OU.modalInfo) {
            OU.modalInfo.remove();
        }
        if (this.w > this.h && this.h < 580) {
            if (!OU.modalInfo) {
                OU.modalInfo = new OU.util.ModalInfo();
            }
            OU.modalInfo.view({
                html: "<div class='winMessage'>This activity works best in portrait view on mobile devices. On a desktop/laptop, please resize your browser window.</div>",
                width: this.w * .66
            });
        }
    };
    /**
     * load the cards into a card object array
     */
    OU.activity.Trumps2.prototype.loadCards = function() {
        var g, c, card, group, groups = this.data.activityContent.groups;

        this.cards = [];
        for (g = groups.length; g--; ) {
            group = groups[g];
            for (c = group.details.length; c--; ) {
                card = group.details[c];
                this.cards.push(new this.Card({
                    card: card,
                    group: group
                }));
            }
        }
    };

    Array.prototype.shuffle = function() {
        var i = this.length, j, temp;
        if (i === 0)
            return this;
        while (--i) {
            j = Math.floor(Math.random() * (i + 1));
            temp = this[i];
            this[i] = this[j];
            this[j] = temp;
        }
        return this;
    };
    /**
     * Shuffle pack - randomise the order of the card array and deal the cards into 2 hands
     *
     */
    OU.activity.Trumps2.prototype.shufflePack = function() {
        var i, j;
        this.cards.shuffle();
        this.userCards = [];
        this.cpuCards = [];
        this.minValues = [];
        this.maxValues = [];
        for (i = this.data.activityContent.categories.length; i--; ) {
            this.minValues[i] = 10000000;
            this.maxValues[i] = -10000000;
        }
        for (i = this.cards.length; i--; ) {
            var card = this.cards[i];
            card.reset();
            if (i < this.cards.length * 0.5) { // TODO make it optional to set the split percentage
                this.userCards.push(card);
            }
            else {
                this.cpuCards.push(card);
            }
            // Calculate the float value for each of the categories
            card.floatValues = [];
            for (var j = card.values.length; j--; ) {
                card.floatValues[j] = parseFloat(card.values[j].value.replace(/,/g, ""))
                // Calculate category ranges for Smart computer to use
                if (card.floatValues[j] > this.maxValues[j]) {
                    this.maxValues[j] = card.floatValues[j];
                }
                if (card.floatValues[j] < this.minValues[j]) {
                    this.minValues[j] = card.floatValues[j];
                }
            }
        }
    };
    /**
     * @class Card - card object
     */
    OU.activity.Trumps2.prototype.Card = function(data) {
        this.group = data.group;
        this.color = data.group.color;
        this.name = data.card.name;
        this.desc = data.card.desc1 + " " + data.card.desc2;
        this.itemImage = data.card.imgElement;
        this.cardImage = this.group.imgElement;
        this.values = [
            {
                value: data.card.value1
            },
            {
                value: data.card.value2
            },
            {
                value: data.card.value3
            },
            {
                value: data.card.value4
            },
            {
                value: data.card.value5
            }
        ];
        var scaleFactor, width, height, cardX, cardY, textLeft, textWidth, textRight;

        /**
         * Reset card state
         */
        OU.activity.Trumps2.prototype.Card.prototype.reset = function() {
            this.selectedOption = null;
            this.winner = null;
        };
        /**
         * Is Hit checks which option has been selected
         */
        OU.activity.Trumps2.prototype.Card.prototype.isHit = function(x, y, state) {
            var mousePointer = false;
            if (trumps.usersTurn) {
                if (x >= textLeft && x <= textLeft + textWidth) {
                    var optionHeight = (height * 0.34) / this.values.length;
                    for (var i = this.values.length; i--; ) {
                        if (y >= cardY + height * 0.58 + (i * optionHeight) + 3 && y < cardY + height * 0.58 + ((i + 1) * optionHeight) + 3) {
                            if (state && this.selectedOption === null) {
                                this.selectedOption = i;
                                trumps.choiceMade();
                            }
                            else { // just hovering
                                mousePointer = true;
                            }
                        }
                    }
                }
                if (mousePointer) {
                    yourCard.canvas.style.cursor = 'pointer';
                }
                else {
                    yourCard.canvas.style.cursor = '';
                }
            }
        };
        /**
         * Render the card to a layer object's canvas
         */
        OU.activity.Trumps2.prototype.Card.prototype.render = function(layer, specificCardX) {
            if (!layer) {
                return;
            }
            var ctx = layer.context, layerWidth = layer.w, layerHeight = layer.h;
            if (this.cardImage.width / layerWidth > this.cardImage.height / layerHeight) { // fit width
                scaleFactor = layerWidth / this.cardImage.width;
            }
            else { // fit height
                scaleFactor = layerHeight / this.cardImage.height;
            }
            width = this.cardImage.width * scaleFactor;
            height = this.cardImage.height * scaleFactor;
            cardX = specificCardX === undefined ? (layerWidth - width) / 2 : specificCardX;
            cardY = (layerHeight - height) / 2;
            ctx.drawImage(this.cardImage, cardX, cardY, width, height);
            ctx.drawImage(this.itemImage, cardX + width * 0.08128, cardY + height * 0.2, width * 0.8374, height * 0.4022);

            textLeft = cardX + width * 0.13;
            textWidth = width * 0.74;
            textRight = textLeft + textWidth * 0.8;

            new OU.util.DynText({
                txt: this.name,
                context: ctx,
                x: textLeft,
                y: cardY + height * 0.11,
                w: textWidth,
                h: height * 0.06,
                align: "left",
                colour: "#fff"
            });

            new OU.util.DynText({
                txt: this.desc,
                context: ctx,
                x: textLeft,
                y: cardY + height * 0.17,
                w: textWidth,
                h: height * 0.07,
                align: "left",
                colour: "#363636",
                fontSize: 16
            });
            var fontSize = trumps.h < 550 ? 10 : (trumps.h < 650 ? 12 : (trumps.h < 800 ? 17 : 20));
            var optionHeight = (height * 0.34) / this.values.length;
            for (var i = this.values.length; i--; ) {
                var optionBackground = this.selectedOption === i ? (this.winner ? 'rgba(100,235,100,0.6)' : 'rgba(235,100,100,0.6)') : this.color;
                var textCol = '#363636';
                if (this.selectedOption === null && this.tabFocus === i) {
                    optionBackground = '#c00';
                    textCol = '#fff';
                }
                new OU.util.DynText({
                    txt: trumps.data.activityContent.categories[i],
                    context: ctx,
                    x: textLeft,
                    y: cardY + height * 0.58 + (i * optionHeight) + 3,
                    w: textWidth,
                    h: optionHeight - 6,
                    align: "left",
                    colour: textCol,
                    fontSize: fontSize,
                    background: {
                        col: optionBackground,
                        radius: 5
                    }
                });
                new OU.util.DynText({
                    txt: this.values[i].value,
                    context: ctx,
                    x: textLeft,
                    y: cardY + height * 0.58 + (i * optionHeight) + 3,
                    w: textWidth,
                    h: optionHeight - 6,
                    align: "right",
                    fontWeight: "bolder",
                    colour: textCol,
                    fontSize: fontSize
                });
            }
        };
        this.reset();
    };
    /**
     * Start play - provide game options screen and then start play
     */
    OU.activity.Trumps2.prototype.startPlay = function() {
        if (!playing) {
            playing = true;
            this.hideMenu();
            this.play();
        }
    };
    /**
     * Display a return to menu button. Used when in play or browse view
     */
    OU.activity.Trumps2.prototype.showReturn2Menu = function() {
        if (!return2MenuDiv) {
            return2MenuDiv = new OU.util.Div({
                container: this,
                x: 0,
                y: 0,
                w: 70,
                h: 60,
                htmlClass: "return2Menu",
                zIndex: OU.POP_UP_LEVEL + 10
            });
            OU.events.addListener(return2MenuDiv.div, function() {
                trumps.showMenu();
            }, 'tap');
        }
    };
    /**
     * Play - provide game options screen and then start play
     */
    OU.activity.Trumps2.prototype.play = function() {

        //init play
        this.usersTurn = true;
        this.shufflePack();
        yourPack = new OU.util.Div({
            container: this,
            x: gapWidth,
            y: 75 + cardHeight,
            w: this.w - gapWidth,
            h: 30,
            style: "color:#eee;",
            htmlClass: "packDetails"
        });
        if (2 * (cardWidth + gapWidth) > this.w - 10) {
            cpuPack = new OU.util.Div({
                container: this,
                x: 0,
                y: 15,
                w: this.w - 10,
                h: 30,
                style: "text-align:right; color:#eee;",
                htmlClass: "packDetails"
            });
        }
        else {
            cpuPack = new OU.util.Div({
                container: this,
                x: 0,
                y: 15,
                w: 2 * (cardWidth + gapWidth),
                h: 30,
                style: "text-align:right; color:#eee;",
                htmlClass: "packDetails"
            });
        }
        yourCard = new OU.util.Layer({
            container: this,
            x: gapWidth,
            y: 60,
            w: cardWidth,
            h: cardHeight,
            hasEvents: true,
            themeClass: "object3D",
            transform: {
                translate: {
                    x: 0,
                    y: this.h
                },
                rotate: {
                    y: 140
                }
            }
        });
        cpuCard = new OU.util.Layer({
            container: this,
            x: cardWidth + 2 * gapWidth,
            y: 60,
            w: cardWidth,
            h: cardHeight,
            themeClass: "object3D",
            transform: {
                translate: {
                    x: 0, // this.w / 2 - (cardWidth + 2 * gapWidth),
                    y: -this.h
                },
                rotate: {
                    y: 180
                }
            }
        });
        coverCard = new OU.util.Layer({
            container: this,
            x: cardWidth + 2 * gapWidth,
            y: 60,
            w: cardWidth,
            h: cardHeight,
            themeClass: "object3D",
            transform: {
                translate: {
                    x: 0, //this.w / 2 - (cardWidth + 2 * gapWidth),
                    y: -this.h
                },
                rotate: {
                    y: 140
                }
            }
        });
        this.showReturn2Menu();

        this.renderPlay();
    };
    /**
     * User or CPU has selected an option from their card
     */
    OU.activity.Trumps2.prototype.choiceMade = function() {
        var userCard = this.userCards[0], compCard = this.cpuCards[0],
                userVal = userCard.floatValues[userCard.selectedOption],
                cpuVal = compCard.floatValues[userCard.selectedOption],
                highestWins = this.data.activityContent.highestWins[userCard.selectedOption];
        if (!yourCard) {
            return; // user has exit, so do nothing
        }

         if (userVal === cpuVal) {userCard.winner = true;
            compCard.winner = false;
            setTimeout(function() {
                if (yourCard) {
                    new OU.util.ModalMessage({
                        html: "<div class='winMessage'>Equal! You win these cards</div>",
                        timeout: 2000
                    });
                }
            }, 3000);
            setTimeout(function() {
                if (yourCard) {
                    trumps.userWins();
                }
            }, 5000);
        }

        else if ((userVal > cpuVal) === highestWins) {
            userCard.winner = true;
            compCard.winner = false;
            setTimeout(function() {
                if (yourCard) {
                    new OU.util.ModalMessage({
                        html: "<div class='winMessage'>You win these cards</div>",
                        timeout: 2000
                    });
                }
            }, 3000);
            setTimeout(function() {
                if (yourCard) {
                    trumps.userWins();
                }
            }, 5000);
        }
        else {
            userCard.winner = false;
            compCard.winner = true;
            setTimeout(function() {
                if (yourCard) {
                    new OU.util.ModalMessage({
                        html: "<div class='winMessage'>Computer wins these cards</div>",
                        timeout: 2000
                    });
                }
            }, 3000);
            setTimeout(function() {
                if (yourCard) {
                    trumps.cpuWins();
                }
            }, 5000);
        }
        compCard.selectedOption = userCard.selectedOption;
        this.renderPlay();
        this.showCPUCard();

    };
    /**
     * CPU chooses
     */
    OU.activity.Trumps2.prototype.CPUchoice = function() {
        var userCard = this.userCards[0], compCard = this.cpuCards[0], choice, userVal = 0, cpuVal = 0,
                categories = trumps.data.activityContent.categories, range, chancePerc, i, bestChance;
        if (userCard && cpuCard) {
            if (this.dumbComputer) { // Dumb CPU chooses categories at random
                while (userVal === cpuVal) {
                    choice = (Math.random() * userCard.values.length) | 0;
                    userVal = userCard.floatValues[choice];
                    cpuVal = compCard.floatValues[choice];
                }
            }
            else { // smart CPU - uses category ranges to pick best chance of it winning
                choice = categories.length - 1;
                bestChance = 0;
                for (i = categories.length; i--; ) {
                    range = this.maxValues[i] - this.minValues[i];
                    chancePerc = (compCard.floatValues[i] - this.minValues[i]) / range;
                    if (!this.data.activityContent.highestWins[i]) {
                        chancePerc = 1 - chancePerc;
                    }
                    if (chancePerc > bestChance) {
                        bestChance = chancePerc;
                        choice = i;
                    }
                }
                /*
                 userCard.selectedOption = choice;
                 new OU.util.ModalMessage({
                 html: "<div class='winMessage'>Computer chooses " + categories[choice] + " with a " + (bestChance * 100 | 0) + "% chance to win</div>",
                 timeout: 2000
                 });
                 //*/
            }
            userCard.selectedOption = choice;
            new OU.util.ModalMessage({
                html: "<div class='winMessage'>Computer chooses " + categories[choice] + "</div>",
                timeout: 2000
            });

            setTimeout(function() {
                trumps.choiceMade();
            }, 2000);
        }
    };
    OU.activity.Trumps2.prototype.showCPUCard = function() {
        if (yourCard) {
            setTimeout(function() {
                if (visibleCards === 2) {
                    cpuCard.transform({
                        rotate: {
                            y: 0
                        },
                        origin: {
                            y: -cardWidth
                        }
                    });
                    coverCard.transform({
                        rotate: {
                            y: 180
                        },
                        origin: {
                            y: -cardWidth
                        }
                    });
                }
                else {
                    yourCard.transform({
                        translate: {
                            x: -cardWidth
                        }
                    });
                    cpuCard.transform({
                        translate: {
                            x: -cardWidth
                        },
                        rotate: {
                            y: 180
                        }
                    });
                    coverCard.transform({
                        translate: {
                            x: -cardWidth
                        }
                    });
                    setTimeout(function() {
                        yourCard.transform({
                            translate: {
                                x: -cardWidth
                            }
                        });
                        cpuCard.transform({
                            translate: {
                                x: -cardWidth
                            },
                            rotate: {
                                y: 0
                            },
                            origin: {
                                x: -cardWidth
                            }
                        });
                        coverCard.transform({
                            translate: {
                                x: -cardWidth
                            },
                            rotate: {
                                y: 180
                            },
                            origin: {
                                x: -cardWidth
                            }
                        });
                    }, 1000);
                }
            }, 10);
        }
    };
    OU.activity.Trumps2.prototype.userWins = function() {
        if (yourCard) {
            yourCard.transform({
                translate: {
                    x: -gapWidth - cardWidth,
                    y: this.h,
                    z: 1000
                }
            });
            cpuCard.transform({
                translate: {
                    x: -2 * (cardWidth + gapWidth),
                    y: this.h,
                    z: 1000
                }
            });
            coverCard.transform({
                translate: {
                    x: -(cardWidth + 2 * gapWidth),
                    y: this.h,
                    z: 1000
                },
                rotate: {
                    y: 180
                }
            });
            this.userCards[0].reset();
            this.cpuCards[0].reset();
            this.userCards.push(this.userCards.shift()); // move our card to back
            this.userCards.push(this.cpuCards.shift()); // take cpu's card a put at back of our pack
            this.usersTurn = true;
            this.loadNext();
        }
    };
    OU.activity.Trumps2.prototype.cpuWins = function() {
        if (yourCard) {
            yourCard.transform({
                translate: {
                    x: this.w - gapWidth,
                    y: -this.h,
                    z: 1000
                }
            });
            cpuCard.transform({
                translate: {
                    x: this.w - (cardWidth + 2 * gapWidth),
                    y: -this.h,
                    z: 1000
                }
            });
            coverCard.transform({
                translate: {
                    x: this.w - (cardWidth + 2 * gapWidth),
                    y: -this.h,
                    z: 1000
                },
                rotate: {
                    y: 180
                }
            });
            this.userCards[0].reset();
            this.cpuCards[0].reset();
            this.cpuCards.push(this.cpuCards.shift()); // move our card to back
            this.cpuCards.push(this.userCards.shift()); // take cpu's card a put at back of our pack
            this.usersTurn = !this.computerPlays;
            this.loadNext();
            if (this.computerPlays) {
                setTimeout(function() {
                    trumps.CPUchoice();
                }, 2000);
            }
        }
    };
    OU.activity.Trumps2.prototype.loadNext = function() {
        if (yourCard) {

            setTimeout(function() {
                OU.removeClass(yourCard.canvas, 'fastease');
                OU.removeClass(cpuCard.canvas, 'fastease');
                OU.removeClass(coverCard.canvas, 'fastease');
                yourCard.transform({
                    translate: {
                        x: 0,
                        y: trumps.h
                    },
                    rotate: {
                        y: 140
                    }
                });
                cpuCard.transform({
                    translate: {
//                    x: trumps.w / 2 - (cardWidth + 2 * gapWidth),
                        y: -trumps.h
                    },
                    rotate: {
                        y: 180
                    }
                });
                coverCard.transform({
                    translate: {
//                    x: trumps.w / 2 - (cardWidth + 2 * gapWidth),
                        y: -trumps.h
                    },
                    rotate: {
                        y: 140
                    }
                });
                setTimeout(function() {
                    OU.addClass(yourCard.canvas, 'fastease');
                    OU.addClass(cpuCard.canvas, 'fastease');
                    OU.addClass(coverCard.canvas, 'fastease');
                    trumps.renderPlay();
                }, 200);
            }, 1000);
        }
    };
    /**
     * render the playing view
     */
    OU.activity.Trumps2.prototype.renderCoverCard = function() {
        var layer = coverCard, ctx = layer.context, layerWidth = layer.w, layerHeight = layer.h, scaleFactor,
                width, height, cardX, cardY,
                coverImage = this.data.activityContent.coverCard.imgElement;
        if (coverImage.width / layerWidth > coverImage.height / layerHeight) { // fit width
            scaleFactor = layerWidth / coverImage.width;
        }
        else { // fit height
            scaleFactor = layerHeight / coverImage.height;
        }
        width = coverImage.width * scaleFactor;
        height = coverImage.height * scaleFactor;
        cardX = (layerWidth - width) / 2;
        cardY = (layerHeight - height) / 2;
        ctx.drawImage(coverImage, cardX, cardY, width, height);
    };
    /**
     * render the playing view
     */
    OU.activity.Trumps2.prototype.renderPlay = function() {
        if (yourCard && cpuCard) {
            if (this.userCards.length < 1 || this.cpuCards.length < 1) {
                yourPack.div.innerHTML = '';
                cpuPack.div.innerHTML = '';
                if (this.userCards.length === 0) {
                    setTimeout(function() {
                        new OU.util.ModalMessage({
                            html: "<div class='winMessage'>Unlucky, you didn't win this time.</div>",
                            timeout: 4000
                        });
                    }, 100);
                }
                else {
                    setTimeout(function() {
                        new OU.util.ModalMessage({
                            html: "<div class='winMessage'>Congratulations, You win the game!</div>",
                            timeout: 4000
                        });
                    }, 100);

                }
                setTimeout(function() {
                    trumps.showMenu();
                }, 4000);
            }
            else {
                var numCards = this.userCards.length;
                if (numCards === 1) {
                    yourPack.div.innerHTML = "<img style='float:left' src='data/" + this.data.activityContent.packGraphics[0] + "'/> Last card!";
                }
                else if (numCards === 2) {
                    yourPack.div.innerHTML = "<img style='float:left' src='data/" + this.data.activityContent.packGraphics[1] + "'/> only 2 cards left.";
                }
                else {
                    yourPack.div.innerHTML = "<img style='float:left' src='data/" + this.data.activityContent.packGraphics[2] + "'/> You have " + numCards + " cards.";
                }
                var numCards = this.cpuCards.length;
                if (numCards === 1) {
                    cpuPack.div.innerHTML = "Computer's last card! <img style='float:right' src='data/" + this.data.activityContent.packGraphics[0] + "'/>";
                }
                else if (numCards === 2) {
                    cpuPack.div.innerHTML = "Computer has 2 card left <img style='float:right' src='data/" + this.data.activityContent.packGraphics[1] + "'/>";
                }
                else {
                    cpuPack.div.innerHTML = "Computer has " + numCards + " cards <img style='float:right' src='data/" + this.data.activityContent.packGraphics[2] + "'/>";
                }
                this.userCards[0].render(yourCard);
                this.cpuCards[0].render(cpuCard);
                this.renderCoverCard();
                yourCard.events.clickable = [this.userCards[0]];

                this.resetTabs();

                setTimeout(function() {
                    yourCard.transform({
                        translate: {
                            x: 0
                        }
                    });
                    cpuCard.transform({
                        translate: {
                            x: 0
                        },
                        rotate: {
                            y: 180
                        }
                    });
                    coverCard.transform({
                        translate: {
                            x: 0
                        }
                    });
                }, 0);
            }
        }
    };

    OU.activity.Trumps2.prototype.resetTabs = function() {
        this.removeTabs();
        this.tabOptions = [];
        for (var i = 0; i < this.userCards[0].values.length; i++) {
            this.tabOptions.push(new this.TabOption({
                trumps: this,
                getOption: function(index) {
                    return function() {
                        return index;
                    }
                }(i)
            }));
        }
    };
    OU.activity.Trumps2.prototype.removeTabs = function() {
        if (this.tabOptions) {
            for (var i = this.tabOptions.length; i--; ) {
                this.tabOptions[i].remove();
            }
        }
        this.tabOptions = null;
    };
    /**
     * Tababble options
     */
    OU.activity.Trumps2.prototype.TabOption = function(params) {
        var trumps = params.trumps;
        this.getOption = params.getOption;
        OU.activity.Trumps2.prototype.TabOption.prototype.focus = function() {
            trumps.userCards[0].tabFocus = this.getOption();
            trumps.userCards[0].render(yourCard);
        };
        OU.activity.Trumps2.prototype.TabOption.prototype.blur = function() {
            trumps.userCards[0].tabFocus = null;
            trumps.userCards[0].render(yourCard);
        };
        OU.activity.Trumps2.prototype.TabOption.prototype.hit = function() {
            trumps.userCards[0].selectedOption = this.getOption();
            trumps.choiceMade();
        };
        OU.base(this, params);
    };
    OU.inherits(OU.activity.Trumps2.prototype.TabOption, OU.util.Tabbable);


    /**
     * Hides the menu
     */
    OU.activity.Trumps2.prototype.hideMenu = function() {
        OU.addClass(this.menuDiv, 'goneAway');
        setTimeout(function() {
            trumps.menuDiv.style.display = 'none';
        }, 1000);
    };
    /**
     * Re-shows the menu
     */
    OU.activity.Trumps2.prototype.showMenu = function() {
        clearAllTimeouts();

        this.removeInstructionsView();
        if (yourCard) {
            yourCard.remove();
            cpuCard.remove();
            coverCard.remove();
            yourPack.remove();
            cpuPack.remove();
        }
        if (fullPack) {
            fullPack.remove();
        }
        this.removeArrows();
        if (return2MenuDiv) {
            return2MenuDiv.remove();
        }
        if (progressBar) {
            progressBar.remove();
        }
        if (this.settingsMenuDiv) {
            this.settingsMenuDiv.remove();
            this.settingsMenuDiv = null;
        }
        yourCard = null;
        cpuCard = null;
        coverCard = null;
        yourPack = null;
        cpuPack = null;
        fullPack = null;
        return2MenuDiv = null;
        this.menuDiv.style.display = 'block';
        setTimeout(function() {
            OU.removeClass(trumps.menuDiv, 'goneAway');
        }, 100);
        playing = false;
    };
    /**
     * resize - called whenever the outer bounds of the activity change, ie when the browser window is resized, or a mobile device is rotated.
     *
     * This function should do the following:
     * <ul>
     * <li>Resize all visible elements</li>
     * <li>Re-layout all visible elements</li>
     * <li>Actually re-render the current view of the activity</li>
     * </ul>
     */
    OU.activity.Trumps2.prototype.resize = function() {
        OU.activity.Trumps2.superClass_.resize.call(this); // call the superclass resize

        this.backgroundDiv.resize({
            x: 0,
            y: 0,
            w: this.w,
            h: this.h
        });
        console.log('Y:' + this.backgroundDiv.div.style.top + ' H:' + this.backgroundDiv.div.style.height);

        cardHeight = this.h - 120;
        cardWidth = cardHeight * this.data.activityOptions.cardAspectRatio;

        if (cardWidth * 2.2 < this.w) { // show 2 cards in landscape
            visibleCards = 2;
        }
        else { // single card view
            visibleCards = 1;
        }
        gapWidth = (this.w - (cardWidth * visibleCards)) / 3;
        if (yourPack) {
            yourPack.resize({
                x: gapWidth,
                y: 75 + cardHeight,
                w: this.w - gapWidth - 10,
                h: 30
            });
        }
        if (cpuPack) {
            if (2 * (cardWidth + gapWidth) > this.w - 10) {
                cpuPack.resize({
                    x: 0,
                    y: 15,
                    w: this.w - 10,
                    h: 30,
                    style: "text-align:right"
                });
            }
            else {
                cpuPack.resize({
                    x: 0,
                    y: 15,
                    w: 2 * (cardWidth + gapWidth),
                    h: 30,
                    style: "text-align:right"
                });
            }
        }
        if (yourCard) {
            yourCard.resize({
                x: gapWidth,
                y: 60,
                w: cardWidth,
                h: cardHeight
            });
        }
        if (cpuCard) {
            cpuCard.resize({
                x: cardWidth + 2 * gapWidth,
                y: 60,
                w: cardWidth,
                h: cardHeight
            });
        }
        if (coverCard) {
            coverCard.resize({
                x: cardWidth + 2 * gapWidth,
                y: 60,
                w: cardWidth,
                h: cardHeight
            });
        }
        if (fullPack) {
            fullPack.resize({
                x: 0,
                y: (this.h - cardHeight) / 2,
                w: this.w,
                h: cardHeight
            });
        }
        this.renderArrows();
        if (progressBar) {
            progressBar.resize();
        }
        if (return2MenuDiv) {
            return2MenuDiv.resize({
                x: 0,
                y: 0,
                w: 70,
                h: 60
            });
        }
        this.renderPlay();
        if (fullPack) {
            this.renderBrowseView();
        }
        this.moveBrowseView();
        if (this.settingsMenuDiv) {
            this.settingsMenuDiv.resize({
                x: this.x + (this.w - 350) / 2,
                y: this.y + this.h * .25,
                w: 350,
                h: "auto"
            });
            this.settingsMenuDiv.div.style.opacity = 1;
        }
        this.orientationWarning();
        window.scrollTo(0, 0);
    };
    OU.activity.Trumps2.prototype.toggleCPULevel = function(level) {
        switch (level) {
            case 'none':
                this.computerPlays = false;
                this.dumbComputer = true;
                break;
            case 'dumb':
                this.computerPlays = true;
                this.dumbComputer = true;
                break;
            case 'smart':
                this.computerPlays = true;
                this.dumbComputer = false;
                break;
        }
        this.renderSettingsMenu();
    };
    OU.activity.Trumps2.prototype.settingsMenu = function() {
        if (playing) {
            return;
        }
        var self = this;
        if (this.settingsMenuDiv) {
            this.settingsMenuDiv.div.style.opacity = 0;
            setTimeout(function() {
                if (self.settingsMenuDiv) {
                    self.settingsMenuDiv.remove();
                    self.settingsMenuDiv = null;
                }
                trumps.showMenu();
            }, 1000);

        }
        else {
            this.menuDiv.style.display = 'none';

            this.settingsMenuDiv = new OU.util.Div({
                x: this.x + (this.w - 350) / 2,
                y: this.y + this.h * .25,
                w: 350,
                h: "auto",
                htmlClass: "settingsPanel"
            });
            setTimeout(function() { // sight delay otherwise the transition doesn't work
                self.settingsMenuDiv.div.style.opacity = 1;
            }, 2);
            this.renderSettingsMenu();
        }
    };
    OU.activity.Trumps2.prototype.renderSettingsMenu = function() {
        var cpuLevel = this.computerPlays ? (this.dumbComputer ? 1 : 2) : 0,
                html = "<table class='settingsTable'><tr><th colspan='2'>Difficulty Level <a tabindex='1' class='settingsClose' id='_settingsClose'></a></th></tr>";

        html += "<tr><td colspan='2'></tr><tr><td>Computer always plays 2<sup>nd</sup></td><td><a tabindex='1' id='levelNone' class='radiotop" + (cpuLevel === 0 ? ' checked' : '') + "'></a></td></tr>"
                + "<tr><td>Dumb Computer</td><td><a tabindex='2' id='levelDumb' class='radiomiddle" + (cpuLevel === 1 ? ' checked' : '') + "'></a></td></tr>"
                + "<tr><td>Smart Computer</td><td><a tabindex='3' id='levelSmart' class='radiobottom" + (cpuLevel === 2 ? ' checked' : '') + "'></a></td></tr>";

        html += "<tr><td colspan='2'></tr></table>";
        this.settingsMenuDiv.div.innerHTML = html;
        var levelObj = document.getElementById('levelNone');
        OU.events.addListener(levelObj, function() {
            OU.Trumps2.toggleCPULevel("none");
        });
        var levelObj = document.getElementById('levelDumb');
        OU.events.addListener(levelObj, function() {
            OU.Trumps2.toggleCPULevel("dumb");
        });
        var levelObj = document.getElementById('levelSmart');
        OU.events.addListener(levelObj, function() {
            OU.Trumps2.toggleCPULevel("smart");
        });
        var levelObj = document.getElementById('_settingsClose');
        OU.events.addListener(levelObj, function() {
            OU.Trumps2.settingsMenu();
        });
    };
    OU.activity.Trumps2.prototype.renderArrows = function() {
        if (fullPack) {
            if (this.nextArrowDiv) {
                this.nextArrowDiv.resize({
                    x: this.w * .75,
                    y: 0,
                    w: this.w * .25,
                    h: this.h
                });
            }
            else {
                this.nextArrowDiv = new OU.util.Div({
                    x: this.w * .75,
                    y: 0,
                    w: this.w * .25,
                    h: this.h,
                    zIndex: OU.POP_UP_LEVEL,
                    htmlClass: 'mousePoint'
                });
                OU.events.addListener(this.nextArrowDiv.div, function() {
                    trumps.moveBrowseView(1);
                });
            }
            this.nextArrowDiv.div.style.backgroundImage = 'url("' + this.dataDir + this.data.activityOptions.nextArrow.image + '")';
            this.nextArrowDiv.div.style.backgroundPosition = 'center right';
            this.nextArrowDiv.div.style.backgroundRepeat = 'no-repeat';

            if (this.previousArrowDiv) {
                this.previousArrowDiv.resize({
                    x: 0,
                    y: 0,
                    w: this.w / 4,
                    h: this.h
                });
            }
            else {
                this.previousArrowDiv = new OU.util.Div({
                    x: 0,
                    y: 0,
                    w: this.w / 4,
                    h: this.h,
                    zIndex: OU.POP_UP_LEVEL,
                    htmlClass: 'mousePoint'
                });
                OU.events.addListener(this.previousArrowDiv.div, function() {
                    trumps.moveBrowseView(-1);
                });
            }
            this.previousArrowDiv.div.style.backgroundImage = 'url("' + this.dataDir + this.data.activityOptions.previousArrow.image + '")';
            this.previousArrowDiv.div.style.backgroundPosition = 'center left';
            this.previousArrowDiv.div.style.backgroundRepeat = 'no-repeat';
        }
    };
    OU.activity.Trumps2.prototype.removeArrows = function() {
        if (this.nextArrowDiv) {
            this.nextArrowDiv.remove();
            this.nextArrowDiv = null;
        }
        if (this.previousArrowDiv) {
            this.previousArrowDiv.remove();
            this.previousArrowDiv = null;
        }
    };
    OU.activity.Trumps2.prototype.ProgressBar = function(params) {
        var container = params.container,
                position = params.position || 0,
                width = params.width || 1,
                height = 10,
                progressBg = params.progressBackground || 'rgba(255,255,255,0.4)',
                progressHighlight = params.progressHighlight || 'rgba(255,255,255,0.9)',
                progressBorder = params.progressBorder || '#fff',
                div = new OU.util.Div({
                    x: container.x + container.w * .1,
                    y: container.y + container.h - height * 3,
                    w: container.w * .8,
                    h: height,
                    zIndex: OU.POP_UP_LEVEL - 1
                });
        var render = function() {
            if (div) {
                if (width < 100 && width > 0) {
                    div.html('<div style="height:' + height + 'px;border: 1px solid ' + progressBorder + ';background:' + progressBg + ';"><div style="background: ' + progressHighlight + '; width:' + width + '%;height:10px; float:left; margin-left:' + position + '%;"></div></div>');
                }
                else {
                    div.html('');
                }
            }
        };

        OU.activity.Trumps2.prototype.ProgressBar.prototype.setPosition = function(params) {
            position = params.position;
            width = (params.width * 100) | 0;
            position = position > 0 ? (position < 1 ? position : 1) : 0;
            position = (position * 100) | 0;
            render();
        };
        OU.activity.Trumps2.prototype.ProgressBar.prototype.resize = function() {
            if (div) {
                div.resize({
                    x: container.x + container.w * .1,
                    y: container.y + container.h - height * 3,
                    w: container.w * .8,
                    h: height
                });
                render();
            }
        };
        OU.activity.Trumps2.prototype.ProgressBar.prototype.remove = function() {
            if (div) {
                div.remove();
                div = null;
            }
        };
    };
    /**
     * Browse the cards
     */
    OU.activity.Trumps2.prototype.browseView = function() {
        browseOffset = this.w;
        targetBrowseOffset = 0;
        fullPack = new OU.util.Layer({
            container: this,
            x: 0,
            y: (this.h - cardHeight) / 2,
            w: this.w,
            h: cardHeight
        });
        progressBar = new this.ProgressBar({
            container: this,
            position: 0,
            width: 1,
            progressBackground: this.data.activityOptions.progressBackground,
            progressHighlight: this.data.activityOptions.progressHighlight,
            progressBorder: this.data.activityOptions.progressBorder
        });
        this.renderArrows();
        this.showReturn2Menu();
        this.browseCycle();
    };
    OU.activity.Trumps2.prototype.browseCycle = function(slide) {
        if (fullPack) {
            if (targetBrowseOffset !== browseOffset) {
                this.renderBrowseView();
            }
            setTimeout(function() {
                trumps.browseCycle();
            }, 40);
        }
    };
    OU.activity.Trumps2.prototype.moveBrowseView = function(slide) {
        if (this.cards) {
            var cardSpacing = cardWidth + 20,
                    numCards = this.cards.length;
            if (slide === 1) {
                targetBrowseOffset -= cardSpacing;
            }
            if (slide === -1) {
                targetBrowseOffset += cardSpacing;
            }
            if (targetBrowseOffset > 0) {
                targetBrowseOffset = 0;
            }
            if (targetBrowseOffset < this.w - 20 - (numCards * cardSpacing)) {
                targetBrowseOffset = this.w - 20 - (numCards * cardSpacing);
            }
        }
    };
    OU.activity.Trumps2.prototype.browseViewSwipe = function(events) {
        if (fullPack) {
            if (events.dx > 0) {
                trumps.moveBrowseView(-1);
            }
            else {
                trumps.moveBrowseView(1);
            }
        }
    };
    OU.activity.Trumps2.prototype.renderBrowseView = function() {
        var i, cardSpacing = cardWidth + 20,
                numCards = this.cards.length;

        fullPack.clear();
        if (browseOffset !== targetBrowseOffset) {
            browseOffset = browseOffset + ((targetBrowseOffset - browseOffset) / 2);
        }
        if (Math.abs(browseOffset - targetBrowseOffset) < 2) {
            browseOffset = targetBrowseOffset;
        }
        for (i = numCards; i--; ) {
            if (browseOffset + (i * cardSpacing) < this.w && browseOffset + (i * cardSpacing) + cardWidth > 0) { // card is in view so render it
                this.cards[i].reset();
                this.cards[i].render(fullPack, browseOffset + 20 + (i * cardSpacing));
            }
        }
        var fullWidth = cardSpacing * numCards - 20;
        progressBar.setPosition({
            position: -browseOffset / (cardSpacing * numCards),
            width: this.w / fullWidth
        });
    };

    // call the superclass's constructor
    OU.base(this, data, instance, controller);
};
// Call our inherits function to implement the class inheritance
OU.inherits(OU.activity.Trumps2, OU.util.Activity);
