/**
 * @fileOverview Div - Adds a div to the body, and positions accordingly
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @class Adds a DIV element to the page
 *
 * @param {object} params, various params to define the behaviour:
 * <ul>
 * <li><strong>{OU.util.Activity} container</strong>: (required) reference to the parent activity</li>
 * <li><strong>x</strong>: x position (optional, defaults to 25% of container width)</li>
 * <li><strong>y</strong>: y position (optional, defaults to 25% of container height)</li>
 * <li><strong>w</strong>: width (optional, defaults to 50% width of container)</li>
 * <li><strong>{int} zIndex</strong>: (optional) zIndex to apply to this div(Gets added to the activity zOffset</li>
 * <li><strong>{string} style</strong>: (optional) CSS style to add to the div</li>
 * </ul>
 */
OU.util.HTMLEditor = function(params) {
    if (params === undefined)
        params = {};
    this.div = null;
    this.container = params.container || {};
    this.zIndexStart = params.zIndex || OU.ABSOLUTE_TOP;
    this.zOffset = this.container.zOffset || 0;
    this.zIndexStart = this.zIndexStart + this.zOffset;
    this.params = params;
    this.style = params.style || '';
    this.addStyle = '';
    this.value;
    this.onClose = params.onClose || function() {
    };
    /**
     * sets everything up and adds the div element into the parent DOM object
     * @private
     */
    OU.util.HTMLEditor.prototype.init = function(params) {
        var self = this, container = this.container,
                x = params.x || container.x + container.w * .25,
                y = params.y || container.y + container.h * .25,
                w = params.w || container.w * .5 || window.innerWidth || 480;

        this.x = x;
        this.y = y;
        this.w = w;
        //Add the event listener to the targetDiv
        this.targetDiv = params.targetDiv;
        this.value = this.insertStandard(params.targetDiv.innerHTML);
        this.targetListener = this.targetDiv.addEventListener('click', function() {
            self.open();
        }, false);
        this.targetListener = this.targetDiv.addEventListener('keyup', function(e) {
            var k;
            if (e._dealtWith)
                return;
            if (window.event)
                k = window.event.keyCode;
            else if (e)
                k = e.which || e.keyCode;
            if (k === 32) {
                self.open();
            }
        }, false);
    };
    OU.util.HTMLEditor.prototype.open = function() {
        var self = this;
        if (OU._popupHighlander)
            OU._popupHighlander.remove();
        OU._popupHighlander = this;
        this.div = document.createElement("div");
        this.div.setAttribute("style", this.addStyle + 'left:' + this.x + 'px; top:' + this.y + 'px;' + 'width:' + this.w + 'px; height:auto; ' + this.style);
        this.div.setAttribute("class", "_overlaydiv htmleditor hidden3d" + (this.params.htmlClass || ''));
        this.div.style.zIndex = this.zIndexStart;
        document.body.appendChild(this.div);

        this.setFunctions();
        this.div.innerHTML = "<div id='_htmlView'>" + this.replaceStandard(this.value) + "</div>"
                + "<input type=text id='_htmlInput' onkeyup='OU._HTMLEditor.change();' value='" + this.value + "'/>"
                + "<p><input type=submit value='Bold' onclick='OU._HTMLEditor.bold();'/>"
                + "<input type=submit value='Italic' onclick='OU._HTMLEditor.italic();'/>"
                + "<input type=submit value='Superscript' onclick='OU._HTMLEditor.super();'/>"
                + "<input type=submit value='Subscript' onclick='OU._HTMLEditor.sub();'/>"
                + this.specialCharsSelect()
                + "</p><p><input type=submit value='Done' onclick='OU._HTMLEditor.done();' style='margin:10px 0;float:right;'/></p>";
        OU.addRemovable(this, this.zOffset);
        setTimeout(function() {
            OU.addClass(self.div, "show3d");
        }, 20);
    };

    OU.util.HTMLEditor.prototype.specialCharsSelect = function() {
        var i, html = '&nbsp;&nbsp;&nbsp;&nbsp;<select onchange="OU._HTMLEditor.specialChar(this);">';
        var specials = "αβγδεζηθικλμνξοπρστυφχψωΓΔΘΛΞΠΣΦΨΩ";

        html += '<option value="">symbols...</option><option value="*S*">Standard symbol</option>';
        for (i = 0; i < specials.length; i++) {
            html += '<option>' + specials[i] + '</option>';
        }
        html += '</select>';
        return html;
    };
    OU.util.HTMLEditor.prototype.done = function() {
        this.targetDiv.innerHTML = this.replaceStandard(this.value);
        this.onClose();
        this.remove();
    };
    OU.util.HTMLEditor.prototype.insertStandard = function(str) {
        return str.replace('<img src="standard.png"/>', '*S*');
    };
    OU.util.HTMLEditor.prototype.replaceStandard = function(str) {
        return str.replace('*S*', '<img src="standard.png"/>');
    };
    /**
     * Initialise the functions that will action the buttons
     */
    OU.util.HTMLEditor.prototype.setFunctions = function() {
        var self = this;
        OU._HTMLEditor = {
            tag: function(tag) {
                var inBox = document.getElementById('_htmlInput'),
                        val = inBox.value,
                        start = inBox.selectionStart, end = inBox.selectionEnd,
                        chunk1 = val.substr(0, start),
                        chunk2 = val.substr(start, end - start),
                        chunk3 = val.substr(end);

                self.value = chunk1 + "<" + tag + ">" + chunk2 + "</" + tag + ">" + chunk3;
                document.getElementById('_htmlInput').value = self.value;
                document.getElementById('_htmlView').innerHTML = self.replaceStandard(self.value);
            },
            insertChar: function(c) {
                if (c === '')
                    return;
                var inBox = document.getElementById('_htmlInput'),
                        val = inBox.value,
                        start = inBox.selectionStart,
                        chunk1 = val.substr(0, start),
                        chunk2 = val.substr(start);

                self.value = chunk1 + c + chunk2;
                document.getElementById('_htmlInput').value = self.value;
                document.getElementById('_htmlView').innerHTML = self.replaceStandard(self.value);
            },
            change: function() {
                self.value = document.getElementById('_htmlInput').value;
                document.getElementById('_htmlView').innerHTML = self.replaceStandard(self.value);
                return false;
            },
            specialChar: function(select) {
                OU._HTMLEditor.insertChar(select.value);
                return false;
            },
            bold: function() {
                OU._HTMLEditor.tag('b');
                return false;
            },
            italic: function() {
                OU._HTMLEditor.tag('i');
                return false;
            },
            super: function() {
                OU._HTMLEditor.tag('sup');
                return false;
            },
            sub: function() {
                OU._HTMLEditor.tag('sub');
                return false;
            },
            done: function() {
                self.done();
                return false;
            }
        };
    };
    /**
     * Resizes the div
     * @param {object} p - optional resize parameters, all default to current value if not specified:
     * <ul>
     * <li><strong>x</strong>: x position</li>
     * <li><strong>y</strong>: y position</li>
     * <li><strong>w</strong>: width </li>
     * <li><strong>h</strong>: height</li>
     * </ul>
     */
    OU.util.HTMLEditor.prototype.resize = function(p) {
        p = p || {};
        var x = p.x || this.container.x + this.container.w * .25,
                y = p.y || this.container.y + this.container.h * .25,
                w = p.w || this.container.w * .5;
        if (x === this.x && y === this.y && w === this.w)
            return; // nothing to do
        this.div.setAttribute("style", this.addStyle + 'left:' + x + 'px; top:' + y + 'px;' + 'width:' + w + 'px; height:auto; ' + this.style);
        if (this.zIndexStart)
            this.div.style.zIndex = this.zIndexStart;
        this.x = x;
        this.y = y;
        this.w = w;
        if (this._scroll) {
            this._scroll.resize({
                x: x + w,
                y: y,
                w: 20,
                h: 'auto'
            });
            this.handleScroll();
        }
    };

    /**
     * Removes the div from the page and tidies up event handlers, scroll bars , etc.
     */
    OU.util.HTMLEditor.prototype.remove = function() {
        var self = this;
        // remove targetDiv listener
        this.targetDiv.removeEventListener("click", this.targetListener, false);
        // remove scrollbar
        if (this._scroll) {
            this.div.removeEventListener("scroll", this._scrollListener, false);
            this.div.removeEventListener("DOMSubtreeModified", this._scrollListener, false);
            this._scroll.remove();
        }
        this._scrollctx = null;
        this._scroll = null;
        if (this.div) {
            OU.removeClass(this.div, 'show3d');
            if (this.div.parentNode) {
                setTimeout(function() { // remove after css transform has finished
                    if (self.div.parentNode)
                        self.div.parentNode.removeChild(self.div);
                }, 600);
            }
        }
        OU._popupHighlander = null;
    };
    /**
     * Handles the manual scroll bar
     * @private
     */
    OU.util.HTMLEditor.prototype.handleScroll = function() {
        if (!this._scroll || this.h === 'auto')
            return;
        var contentHeight = this.div.scrollHeight,
                scrollTop = this.div.scrollTop,
                h = this.h,
                visiblePerc = h / contentHeight,
                barTopPerc = scrollTop / contentHeight,
                barTop = h * barTopPerc,
                barHeight = h * visiblePerc;

        this._scroll.clear();
        if (contentHeight <= h)
            return;
        this.hasScrollBar = true;
        this._scrollctx.fillStyle = "rgba(128,128,128,0.2)";
        this._scrollctx.fillRect(5, 0, 10, h);
        this._scrollctx.fillStyle = "rgba(0,0,0,0.2)";
        this._scrollctx.fillRect(5, barTop, 10, barHeight);
    };
    this.init(params);
};
