/**
 * @fileOverview PopUpForm - opens a form in a pop up div
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @class Opens a form in a "pop-up" div and calls a function when closed
 *
 * @param {object} params, various params to define the behaviour:
 * <ul>
 * <li><strong>{OU.util.Activity} container</strong>: (required) reference to the parent activity</li>
 * <li><strong>x</strong>: x position (optional, defaults to 25% of container width)</li>
 * <li><strong>y</strong>: y position (optional, defaults to 25% of container height)</li>
 * <li><strong>w</strong>: width (optional, defaults to 50% width of container)</li>
 * <li><strong>{int} zIndex</strong>: (optional) zIndex to apply to this div(Gets added to the activity zOffset</li>
 * <li><strong>{string} style</strong>: (optional) CSS style to add to the div</li>
 * </ul>
 */
OU.util.PopUpForm = function(params) {
    if (params === undefined)
        params = {};
    this.div = null;
    this.form = params.formHTML || 'Oops, form is missing.';
    this.container = params.container || {};
    this.zIndexStart = params.zIndex || OU.ABSOLUTE_TOP;
    this.zOffset = this.container.zOffset || 0;
    this.zIndexStart = this.zIndexStart + this.zOffset;
    this.params = params;
    this.style = params.style || '';
    this.addStyle = '';
    this.value;
    this.onSubmit = params.onSubmit || function() {
    };
    /**
     * sets everything up and adds the div element into the parent DOM object
     * @private
     */
    OU.util.PopUpForm.prototype.init = function(params) {
        var self = this,container = this.container,
                x = params.x || container.x + container.w * .25,
                y = params.y || container.y + container.h * .25,
                w = params.w || container.w * .5 || window.innerWidth || 480;

        this.x = x;
        this.y = y;
        this.w = w;

        if (OU._popupHighlander)
            OU._popupHighlander.remove();
        OU._popupHighlander = this;
        this.div = document.createElement("div");
        this.div.setAttribute("style", this.addStyle + 'left:' + this.x + 'px; top:' + this.y + 'px;' + 'width:' + this.w + 'px; height:auto; ' + this.style);
        this.div.setAttribute("class", "_overlaydiv htmleditor hidden3d" + (this.params.htmlClass || ''));
        this.div.style.zIndex = this.zIndexStart;
        document.body.appendChild(this.div);

        this.div.innerHTML = "<form onsubmit='OU._popupHighlander.done(); return false;'>"
                + this.form+ "</form>";
        OU.addRemovable(this, this.zOffset);
        setTimeout(function() {
            OU.addClass(self.div, "show3d");
        }, 20);
    };

    OU.util.PopUpForm.prototype.done = function(e) {
        try {
            this.onSubmit();
        }
        catch(e) {
            console.log('caught an error: '+e.message);
        }
        this.remove();
    };

    /**
     * Resizes the div
     * @param {object} p - optional resize parameters, all default to current value if not specified:
     * <ul>
     * <li><strong>x</strong>: x position</li>
     * <li><strong>y</strong>: y position</li>
     * <li><strong>w</strong>: width </li>
     * <li><strong>h</strong>: height</li>
     * </ul>
     */
    OU.util.PopUpForm.prototype.resize = function(p) {
        p=p || {};
        var x = p.x || this.container.x+this.container.w*.25,
                y = p.y || this.container.y+this.container.h*.25,
                w = p.w || this.container.w*.5;
        if (x === this.x && y === this.y && w === this.w)
            return; // nothing to do
        this.div.setAttribute("style", this.addStyle + 'left:' + x + 'px; top:' + y + 'px;' + 'width:' + w + 'px; height:auto; ' + this.style);
        if (this.zIndexStart)
            this.div.style.zIndex = this.zIndexStart;
        this.x = x;
        this.y = y;
        this.w = w;
        if (this._scroll) {
            this._scroll.resize({
                x: x + w,
                y: y,
                w: 20,
                h: 'auto'
            });
            this.handleScroll();
        }
    };

    /**
     * Removes the div from the page and tidies up event handlers, scroll bars , etc.
     */
    OU.util.PopUpForm.prototype.remove = function() {
        var self=this;
        // remove scrollbar
        if (this._scroll) {
            this.div.removeEventListener("scroll", this._scrollListener, false);
            this.div.removeEventListener("DOMSubtreeModified", this._scrollListener, false);
            this._scroll.remove();
        }
        this._scrollctx = null;
        this._scroll = null;
        if (this.div) {
            OU.removeClass(this.div, 'show3d');
            if (this.div.parentNode) {
                setTimeout(function() { // remove after css transform has finished
                    if(self.div && self.div.parentNode)
                        self.div.parentNode.removeChild(self.div);
                },600);
            }
        }
        OU._popupHighlander = null;
    };
    /**
     * Handles the manual scroll bar
     * @private
     */
    OU.util.PopUpForm.prototype.handleScroll = function() {
        if (!this._scroll || this.h === 'auto')
            return;
        var contentHeight = this.div.scrollHeight,
                scrollTop = this.div.scrollTop,
                h = this.h,
                visiblePerc = h / contentHeight,
                barTopPerc = scrollTop / contentHeight,
                barTop = h * barTopPerc,
                barHeight = h * visiblePerc;

        this._scroll.clear();
        if (contentHeight <= h)
            return;
        this.hasScrollBar = true;
        this._scrollctx.fillStyle = "rgba(128,128,128,0.2)";
        this._scrollctx.fillRect(5, 0, 10, h);
        this._scrollctx.fillStyle = "rgba(0,0,0,0.2)";
        this._scrollctx.fillRect(5, barTop, 10, barHeight);
    };
    this.init(params);
};
