/**
 * @fileOverview Drag2Queue - Drag and drop items into an existing queue - placing the items in the correct position
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.Div');
OU.require('OU.util.DynText');
OU.require('OU.util.Draggable');
OU.require('OU.util.Layer');

KEY_LEFT=37;
KEY_UP=38;
KEY_RIGHT=39;
KEY_DOWN=40;

/**
 * @class Drag and drop items into an existing queue - placing the items in the correct position
 * @extends OU.util.Activity
 * @param {object} data - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.Drag2Queue = function(data,instance,controller) {

    /**
     * Starting point when running with HTML5 Canvas Support
     */
    OU.activity.Drag2Queue.prototype.canvasView = function() {
        var itemHeight,i,item,itemMinW;

        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer

        this.descriptionArea = new OU.util.Div({
            container: this,
            x: this.x+10,
            y: this.y+10,
            w: this.w/2 -20,
            h: this.h-20
        });

        this.imageLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });

        this.order = this.data.order;
        this.items = [];
        this.numDraggable=0;
        this.checkCount=0;

        itemHeight=(this.h-40)/this.data.items.length;
        itemHeight = itemHeight>40?40:itemHeight;

        this.listArea = {
            x:this.w/2 + 10*OU.dpr,
            y:10*OU.dpr,
            w:this.w/2-20*OU.dpr,
            h:this.h-20*OU.dpr,
            padding: 10*OU.dpr,
            itemHeight: itemHeight
        };

        this.newItemsArea = {
            x: 10,
            y: this.h/2 + 10,
            w: this.w/2 -20,
            h: this.h/2-20,
            padding: 10
        };

        this.minWidth = 0;
        for (i = 0; i < this.data.items.length; i++ ) {
            var color = "#ccd";
            item = this.data.items[i];
            if(item.draggable===true) {
                this.items.push( new this.Item( {
                    "queue": this,
                    "events": this.events,
                    "id": item.id,
                    "sortOrder": this.numDraggable,
                    "order": item.order,
                    "name": item.name,
                    "draggable": true,
                    "color": color,
                    "x": this.newItemsArea.x+this.newItemsArea.padding,
                    "y": this.newItemsArea.y+this.newItemsArea.padding +(this.numDraggable * itemHeight),
                    "w": this.newItemsArea.w-(2*this.newItemsArea.padding),
                    "h": itemHeight*.9
                } ) );
                this.numDraggable++;
            }
            else {
                this.items.push( new this.Item( {
                    "queue": this,
                    "events": this.events,
                    "id": item.id,
                    "sortOrder": i-this.numDraggable,
                    "order": item.order,
                    "name": item.name,
                    "draggable": false,
                    "disableTab":true,
                    "color": color,
                    "x": this.listArea.x+this.listArea.padding,
                    "y": this.listArea.y+this.listArea.padding +((i-this.numDraggable) * itemHeight),
                    "w": this.listArea.w-(2*this.listArea.padding),
                    "h": itemHeight*.9
                } ) );
            }
            itemMinW = this.items[this.items.length-1].minWidth();
            if(itemMinW>this.minWidth)
                this.minWidth=itemMinW;
        }
        this.minWidth += 60;

        this.view='norm';

        this.renderBackdrop();
        this.doRender=true;
        this.renderLoop();
        this.resize();
    };
    /**
     * Resize the activity
     */
    OU.activity.Drag2Queue.prototype.resize = function() {
        OU.activity.Drag2Queue.superClass_.resize.call(this); // call the parent class resize
        var i,item,itemHeight=(this.h-40)/this.data.items.length,
        placed=0,notPlaced=0;

        itemHeight = itemHeight>40*OU.dpr?40*OU.dpr:itemHeight;
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.imageLayer.resize();
        this.listArea = {
            x:this.x + this.w/2 + 10*OU.dpr,
            y:10*OU.dpr,
            w:this.w/2-20*OU.dpr,
            h:this.h-20*OU.dpr,
            padding: 10*OU.dpr,
            itemHeight: itemHeight
        };

        this.descriptionArea.resize({
            x: this.x+10,
            y: this.y+10,
            w: this.w/2 -20,
            h: 'auto'
        });
        var descHeight = this.descriptionArea.div.clientHeight + 20;
        this.newItemsArea = {
            x: 10,
            y: descHeight,
            w: this.w/2 -20,
            h: (this.h-descHeight-OU.controlHeight),
            padding: 10
        };
        for (i=0;i<this.data.items.length; i++ ) {
            item = this.items[i];
            if(item.placed) {
                item.resize({
                    "x": this.listArea.x+(this.listArea.w-this.minWidth)/2,
                    "y": this.listArea.y+this.listArea.padding +(placed++ * itemHeight),
                    "w": this.minWidth*OU.dpr,
                    "h": itemHeight*.9
                });
            }
            else {
                item.resize({
                    "x": this.newItemsArea.x+(this.newItemsArea.w-this.minWidth)/2,
                    "y": this.newItemsArea.y+this.newItemsArea.padding +(notPlaced++ * itemHeight),
                    "w": this.minWidth*OU.dpr,
                    "h": itemHeight*.9
                });
            }
        }
        this.renderBackdrop();
        this.doRender=true;
    };
    /**
     * Perform a render loop, to enable animation of objects
     */
    OU.activity.Drag2Queue.prototype.renderLoop = function() {
        var self=this;
        if(this.doRender)
            this.render();
        setTimeout(function() {
            self.renderLoop();
        },40);
    };
    /**
     * Render the background to the activity
     */
    OU.activity.Drag2Queue.prototype.renderBackdrop = function() {
        var ctx = this.bgLayer.context;

        // draw list area
        ctx.gradRect( {
            x:this.listArea.x,
            y:this.listArea.y,
            w:this.listArea.w,
            h:this.listArea.h,
            radius:this.listArea.padding/2,
            col2: '#eef',
            col1:'#eef'
        });
    };
    /**
     * Move the items into the correct target positions.
     */
    OU.activity.Drag2Queue.prototype.solve = function() {
        var i,item;
        for (i = 0; i < this.items.length; i++ ) {
            item=this.items[i];
            item.sortOrder=item.order-1;
            item.placed=true;
        }
        this.doRender=true;
    };
    /**
     * Render the activity depending on the current state.
     * Also implements the animation of items, as it renders them.
     */
    OU.activity.Drag2Queue.prototype.render = function() {
        var ctx = this.imageLayer.context,self=this,item,
        events = this.imageLayer.events,
        feedback,matches,i,shuffling,targetX,targetY,newX,newY;

        ctx.clearRect(0,0,this.w,this.h); // clear background

        events.clickable.length = 0;

        if(this.solveButton_) {
            this.solveButton_.remove();
            this.solveButton_=undefined;
        }
        if(this.feedbackButton_) {
            this.feedbackButton_.remove();
            this.feedbackButton_=undefined;
        }
        if(this.checkButton_) {
            this.checkButton_.remove();
            this.checkButton_=undefined;
        }

        if(this.view!=='feedback') {
            this.descriptionArea.div.innerHTML = this.data.instructions;


                this.checkButton_ = new OU.util.Button({
                    layer: this.imageLayer,
                    txt:"Check your answer",
                    x:this.newItemsArea.x+20,
                    y:this.newItemsArea.y+this.newItemsArea.h-OU.controlHeight-10,
                    w:this.newItemsArea.w-40,
                    h:OU.controlHeight,
                    onClick: function() {
                        self.checkCount++;
                        self.view='feedback';
                        self.render();
                    }
                });
        }
        else {
            matches=0;
            for ( i = 0; i < this.items.length; i++ ) {
                if(this.items[i].order===this.items[i].sortOrder+1)
                    matches++;
            }
            if(matches===this.items.length)
                feedback = this.data.matchedFeedback;
            else {
                feedback = this.data.unmatchedFeedback;
                if(this.checkCount>2) {
                    this.solveButton_ = new OU.util.Button({
                        layer: this.imageLayer,
                        x:this.newItemsArea.x+20,
                        y:this.newItemsArea.y+this.newItemsArea.h-OU.controlHeight-10,
                        w:this.newItemsArea.w-40,
                        h:OU.controlHeight,
                        txt: "Show answer",
                        onClick: function() {
                            self.solve();
                        }
                    });
                }
            }

            this.descriptionArea.div.innerHTML = feedback;
            if(this.data.finished && this.data.finished.buttonText!=='' && matches===this.items.length) {
                this.feedbackButton_ = new OU.util.Button({
                    layer: this.imageLayer,
                    x:this.newItemsArea.x+20,
                    y:this.newItemsArea.y+this.newItemsArea.h-OU.controlHeight-10,
                    w:this.newItemsArea.w-40,
                    h:OU.controlHeight,
                    txt: this.data.finished.buttonText,
                    onClick: function() {
                        self.controller.step(self.data.finished.controllerStep || 1);
                    }
                });
            }
        }
        // animate shuffle
        shuffling = false;
        for (i = 0; i < this.items.length; i++ ) {
            item =this.items[i];
            if(item.draggable===false || item.draggable.dragId === 0 && !item.hasFocus) { // item is not being dragged, so shuffle to appropriate position
                if(item.placed) {
                    targetX=this.listArea.x+(this.listArea.w-this.minWidth)/2;
                    targetY=this.listArea.y+this.listArea.padding +(item.sortOrder * this.listArea.itemHeight);
                }
                else {
                    targetX = this.newItemsArea.x+(this.newItemsArea.w-this.minWidth)/2;
                    targetY=this.newItemsArea.y+this.newItemsArea.padding +(item.sortOrder * this.listArea.itemHeight);
                }
                if(item.x!==targetX || item.y!==targetY) {
                    shuffling=true;
                    newX = item.x + (targetX-item.x)/2;
                    newY = item.y + (targetY-item.y)/2;
                    if(Math.abs(targetX-newX)<0.2)
                        newX=targetX;
                    if(Math.abs(targetY-newY)<0.2)
                        newY=targetY;
                    item.move(newX,newY);
                }
            }
            else {
                shuffling=true;
            }
            // render item
            item.render();
        }

        if(!shuffling) {
            this.doRender=false;
        }
    };
    /**
     * @class Draggable item to be moved into the queue.
     * @extends OU.util.Tabbable
     * @param {object} params
     */
    OU.activity.Drag2Queue.prototype.Item = function( params ) {
        this.queue = params.queue;
        this.events = params.events;
        this.id = params.id;
        this.order = params.order;
        this.sortOrder = params.sortOrder;
        this.name = params.name;
        this.color = params.color,
        this.x = params.x;
        this.y = params.y;
        this.h = params.h;
        this.w = params.w;
        this.startX = this.x;
        this.startY = this.y;
        /**
         * resizes the item and it's corresponding draggable element
         * @param {object} p - new dimensions
         */
        OU.activity.Drag2Queue.prototype.Item.prototype.resize = function(p) {
            this.x = p.x;
            this.y = p.y;
            this.h = p.h;
            this.w = p.w;
            if(this.draggable) {
                this.draggable.x = p.x;
                this.draggable.y = p.y;
                this.draggable.h = p.h;
                this.draggable.w = p.w;
            }
            this.startX = this.x;
            this.startY = this.y;
        };
        /**
         * Calculates minimum width based on the text for this item
         * @returns {double} width in pixels of the text when rendered
         */
        OU.activity.Drag2Queue.prototype.Item.prototype.minWidth = function() {
            var test = new OU.util.DynText( {
                "measureOnly":true,
                "txt": this.name,
                "x": 0,
                "y": 0,
                "w": this.w,
                "h": this.h,
                "context": this.queue.imageLayer.context,
                fontWeight: "bold",
                fontSize: 18
            } );
            return test.w;
        };
        /**
         * renders this item
         */
        OU.activity.Drag2Queue.prototype.Item.prototype.render = function( ) {
            var ctx = this.queue.imageLayer.context,sF=this.draggable?(this.draggable.dragId!==0?1.4:1):1,bg,
            x = this.x-(this.w*(sF-1)/2),
            y = this.y-(this.h*(sF-1)/2),
            w = this.w*sF,
            h = this.h*sF;

            if(this.hasFocus) {
                bg = {
                    "borderCol": "#444",
                    "col": "#c00",
                    "radius": this.draggable?h/3:0
                };
            }
            else {
                bg = {
                    "borderCol": "#444",
                    "col": this.color,
                    "radius": this.draggable?h/3:0
                };
            }
            new OU.util.DynText( {
                "txt": this.name,
                "x": x,
                "y": y,
                "w": w,
                "h": h,
                "context": ctx,
                background: bg,
                fontWeight: "bold",
                fontSize: 18*sF*OU.dpr
            } );
            if(this.draggable) {
                ctx.gripPattern({
                    x: w*.9+x-h/4,
                    y:y+h/4,
                    w: h/2,
                    h: h/2,
                    col:'#222'
                });
                this.queue.imageLayer.events.clickable.push(this.draggable);
            }
        };
        /**
         * Called when the item is clicked/touched at the start of a drag.
         * Moves the item to the top of the items array, so that it gets drawn last,
         * so that the currently dragged item is always on top of other items
         * @param {object} p - details
         */
        OU.activity.Drag2Queue.prototype.Item.prototype.startDrag = function( p ) {
            //sort the items so that the latest dragged is at the end of the queue, therefore render last next time (on top)
            p.me.queue.items.sort( function (a,b) {
                if(a.draggable===false && b.draggable===false)
                    return 0;
                if(a.draggable===false)
                    return -1;
                if(b.draggable===false)
                    return 1;
                return a.draggable.dragId - b.draggable.dragId;
            });
            p.me.queue.view='norm';
            p.me.queue.doRender=true;
        };
        /**
         * Called when the item is dropped.
         * Not much to do, but re-render
         * @param {object} p - details
         */
        OU.activity.Drag2Queue.prototype.Item.prototype.endDrag = function( p ) {
            p.me.queue.doRender=true;
        };
        /**
         * Called in mid drag. Moves the item and also changes its position in the queue
         * @param {object} p - details
         */
        OU.activity.Drag2Queue.prototype.Item.prototype.newPos = function( p ) {
            var item = p.me,ic,numPlaced=0, numNew=0,q = item.queue;
            item.x = p.x;
            item.y = p.y;

            // Shuffle sort order based on new position
            q.items.sort( function (a,b) {
                return a.y - b.y;
            });

            for(var i=0; i<q.items.length; i++) {
                ic = q.items[i].x+q.items[i].w/2;
                if(ic > q.newItemsArea.x && ic<q.newItemsArea.x+q.newItemsArea.w) {
                    q.items[i].sortOrder=numNew++;
                    q.items[i].placed=false;
                }
                else {
                    q.items[i].sortOrder=numPlaced++;
                    q.items[i].placed=true;
                }
            }
            q.items.sort( function (a,b) {
                if(a.draggable===false && b.draggable===false)
                    return 0;
                if(a.draggable===false)
                    return -1;
                if(b.draggable===false)
                    return 1;
                return a.draggable.dragId - b.draggable.dragId;
            });
            q.doRender=true;
        };
        /**
         * Moves the item to a new X,Y
         * @param {double} x - new x coordinate
         * @param {double} y - new y coordinate
         */
        OU.activity.Drag2Queue.prototype.Item.prototype.move = function( x,y ) {
            this.x=x;
            this.y=y;
            if(this.draggable) {
                this.draggable.x=x;
                this.draggable.y=y;
            }
        };
        if(params.draggable) { // If this item is draggable then attach a draggable object to it.
            this.draggable = new OU.util.Draggable( {
                "me":this,
                "events": this.queue.imageLayer.events,
                "x": this.x,
                "y": this.y,
                "h": this.h,
                "w": this.w,
                "onStart": this.startDrag,
                "onEnd": this.endDrag,
                "onMove": this.newPos
            } );
            this.placed=false;
        }
        else {
            this.draggable=false;
            this.placed=true;
        }
        /**
         * Focus handler, called when the item receives tab focus
         */
        OU.activity.Drag2Queue.prototype.Item.prototype.focus = function () {
            if(this.draggable) {
                this.hasFocus = true;
                this.render();

                this.queue.items.sort( function (a,b) { // Ensure draggable items hover over non-draggables
                    if(a.draggable===false && b.draggable===false)
                        return 0;
                    if(a.draggable===false)
                        return -1;
                    if(b.draggable===false)
                        return 1;
                    return a.draggable.dragId - b.draggable.dragId;
                });
                this.queue.view='norm';
            }
            return false;
        };
        /**
         * Blur handler, called when the item loses tab focus
         */
        OU.activity.Drag2Queue.prototype.Item.prototype.blur = function () {
            if(this.draggable) {
                this.hasFocus = false;
                this.render();
            }
            return false;
        };
        /**
         * Called when the item has tab focus and an arrow key is pressed
         * @param {int} k - the keycode of the key pressed
         */
        OU.activity.Drag2Queue.prototype.Item.prototype.arrowKey = function (k) {
            switch(k){
                case KEY_UP:
                    this.move(this.x,this.y-10);
                    break;
                case KEY_DOWN:
                    this.move(this.x,this.y+10);
                    break;
                case KEY_LEFT:
                    this.move(this.x-10,this.y);
                    break;
                case KEY_RIGHT:
                    this.move(this.x+10,this.y);
                    break;
            }
            this.newPos({
                x:this.x,
                y:this.y,
                me:this
            });
            this.queue.doRender=true;
        };

        OU.base(this, params);
    };
    OU.inherits(OU.activity.Drag2Queue.prototype.Item, OU.util.Tabbable);

    /**
     * Starting point of the activity is canvas is not supported by the environment we are in.
     * Lists the data as text.
     */
    OU.activity.Drag2Queue.prototype.accessibleView = function() {
        clearInterval(this.renderCycle);

        var h = '<div id="accessibleView"><h1>'+this.data.title+'</h1><p>'+this.data.description+'</p>';

        for(var i=0; i<this.data.items.length; i++) {
            h += '<h2>'+this.data.items[i].name+'</h2>';
        }
        if(this.data.matchedFeedback !== undefined) {
            h += "<h3>If the order is correct the feedback is:</h3>";
            if(typeof this.data.matchedFeedback === 'string') {
                h += '<p>'+this.data.matchedFeedback+'</p>';
            }
            else { // must be a string array
                for(var j=0; j<this.data.matchedFeedback.length; j++)
                    h += '<p>'+this.data.matchedFeedback[j]+'</p>';
            }
            h += "<h3>If the order is incorrect the feedback is:</h3>";
            if(typeof this.data.unmatchedFeedback === 'string') {
                h += '<p>'+this.data.unmatchedFeedback+'</p>';
            }
            else { // must be a string array
                for(var j=0; j<this.data.unmatchedFeedback.length; j++)
                    h += '<p>'+this.data.unmatchedFeedback[j]+'</p>';
            }
            // randomise order of the items
            this.data.items.sort( function(a,b) {
                return a.order-b.order;
            } );
            h += "<h3>The correct order is:</h3><p>";
            for(i=0; i<this.data.items.length; i++) {
                h += this.data.items[i].name+' : ';
            }
            h += '</p>';
        }
        else if(this.data.unmatchedFeedback !== undefined) {
            h += "<h3>Feedback after your choice is:</h3>";
            if(typeof this.data.unmatchedFeedback === 'string') {
                h += '<p>'+this.data.unmatchedFeedback+'</p>';
            }
            else { // must be a string array
                for(var j=0; j<this.data.unmatchedFeedback.length; j++)
                    h += '<p>'+this.data.unmatchedFeedback[j]+'</p>';
            }
        }

        h += '</div>';
        document.body.innerHTML='';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this,data,instance,controller);
};
OU.inherits(OU.activity.Drag2Queue,OU.util.Activity);
