/**
 * @fileOverview Dynamic Text utility - renders text on a canvas within a defined area.
 * Scales text down to make it fit if required
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

/**
 * @class Dynamic Text Object - renders text within a defined space - resizing font to ensure the text fits within the area
 *
 * @param {object} params - Various parameters, mostly optional:
 * <ul>
 * <li> <strong>{canvas.context} context </strong>: (Required) Canvas context to render to</li>
 * <li> <strong>{string | json object} txt </strong>: the text to render, in either a string or a json object containing paragraphs.
 * Txt may contain some simple html formatting tags: (STRONG,B,I,EM,SUB,SUP,BR)</li>
 * <li> <strong>{int} x </strong>: X value of text area</li>
 * <li> <strong>{int} y </strong>: Y value of text area</li>
 * <li> <strong>{int} w </strong>: Width of text area</li>
 * <li> <strong>{int} h </strong>: Height of text area</li>
 * <li> <strong>{int} maxLines </strong>: maximum number of lines to use if automatically breaking up.</li>
 * <li> <strong>{int} fontSize </strong>: Font size in pixels, defaults to 24 (also multiplied by OU.dpr for retina display)</li>
 * <li> <strong>{double} propTextHeight </strong>: (optional) Propotional font size sets the font height as a proportion of the DynText height, ie 0.5 would make fontSize 50% of DynText box height</li>
 * <li> <strong>{double} relLineHeight </strong>: Relative line height, defaults to 1.1 (ie space between lines is 10% of font height)</li>
 * <li> <strong>{string} colour </strong>: Text colour in web format ie. "#fff" or "white" or "rgba(255,255,255,1)"</li>
 * <li> <strong>{string} rgb </strong>: Text Colour in RGB format, ie. "26,26,26", requires the alpha parameter, to make up the full rgba string</li>
 * <li> <strong>{double|string} alpha </strong>: Text Opacity value between 0 and 1</li>
 * <li> <strong>{string} fontFamily </strong>: Font family to use, ie. tahoma, verdana, etc. - uses default if not specified</li>
 * <li> <strong>{string} fontWeight </strong>: Weight of the font - ie. normal, bold,etc.</li>
 * <li> <strong>{boolean} notDynamic </strong>: if true, then text is not resized to fit in the area (note: this assumes it fits, so be careful)</li>
 * <li> <strong>{string} align </strong>: Text alignment, defaults to centre</li>
 * <li> <strong>{double} padding </strong>: Left and Right padding around the text</li>
 * <li> <strong>{double} verticalPadding </strong>: Padding on top and below of the text</li>
 * <li> <strong>{string} alignPara </strong>: Text alignment for paragraphs, defaults to left</li>
 * <li> <strong>{boolean} autoWidth </strong>: if true, object will reduce its width to fit the text, default is false</li>
 * <li> <strong>{boolean} autoCentre </strong>: if true then maintain centre when reducing width with autoWidth</li>
 * <li> <strong>{boolean} noBreak </strong>: if true then text isn't broken across lines, defaults to false</li>
 * <li> <strong>{boolean} measureOnly </strong>: if true, then render is not run but all the calculations are done to determine the size of the output, defaults to false</li>
 * </ul>
 *
 */
OU.util.DynText = function(params) {
    if (params.context === undefined) {
        alert('Incorrect use of DynText Object - Context Required');
        return;
    }
    this.context = params.context;
    this.params = params;
    this.txt = params.txt || '';
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || 200;
    this.h = params.h || 100;
    this.maxLines = params.maxLines || 1000;
    this.relLineHeight = params.relLineHeight || 1.1; // relative line height - default to 10% line spacing
    this.paraBottomMargin = 1; // proportionate to fontSize
    if (params.propTextHeight !== undefined)
        this.fontSize = this.h * params.propTextHeight * OU.dpr;
    else
        this.fontSize = params.fontSize || 24 * OU.dpr;
    this.font = new OU.font({
        family: params.fontFamily,
        size: this.fontSize,
        weight: params.fontWeight
    });
    this.dynamic = params.notDynamic === true ? false : true;
    this.align = params.align || 'center';
    this.padding = params.padding || 10;
    this.verticalPadding = params.verticalPadding || 0;
    this.alignPara = params.alignPara || 'left';
    this.alpha = params.alpha || 1;
    this.autoWidth = params.autoWidth || false;
    this.autoCentre = params.autoCentre || false;
    this.beenDisabled = false;
    this.noBreak = params.noBreak || false;
    this.measureOnly = params.measureOnly || false;
    this.textMetrics = [];
    this.textBoxes = [];
    this.colour = params.colour || undefined;
    this.rgb = params.rgb || undefined;
    if (this.colour === undefined && this.rgb === undefined)
        this.rgb = '26,26,26';
    if (typeof(this.txt) === 'string' || typeof(this.txt) === 'number') {
        this.paras = {};
        this.paras.p = [];
        this.paras.p.push({
            line: '' + this.txt,
            colour: this.colour,
            rgb: this.rgb,
            alpha: this.alpha
        });
    }
    else if (typeof(this.txt[0]) === 'string' || typeof(this.txt[0]) === 'number') {
        this.paras = {};
        this.paras.p = [];
        for (var i = 0; i < this.txt.length; i++) {
            this.paras.p.push({
                line: '' + this.txt[i],
                colour: this.colour,
                rgb: this.rgb,
                alpha: this.alpha
            });
        }
    }
    else {
        this.paras = this.txt;
    }
    this.render();
};
/**
 * Renders the DynText to the canvas, including background if present
 */
OU.util.DynText.prototype.render = function() {
    var maxWidth = this.w - (2 * this.padding), b, seg, x, wordW, lineW, wboxX, textHeight,
            para, i, j, line, lineWidth, bits, yStart, lineSize, oldX,
            wrd, wrdx, wrdy, wrdw, wrdh, phrases,
            pa = this.paras.p,
            nParas = pa.length, tl,
            maxHeight = (this.h * .8) - (this.font.size * this.paraBottomMargin * (pa.length - 1)) + (this.font.size * (this.relLineHeight - 1)) - (2 * this.verticalPadding),
            maxParaW = 0,
            lineCount = 0, lineTxt,
            paraX = this.x + this.padding,
            remaining, resize = true,
            ctx = this.context;
    ctx.save();
    /* debug - show outline of textbox
     ctx.save();
     ctx.strokeStyle="#ff0000";
     ctx.strokeRect(this.x,this.y,this.w,this.h);
     ctx.restore();
     //*/

    /*
     * Calculate the font size required to make all the text fit in the dimensions given
     */
    if (!this.noBreak) {
        do {
            resize = false;
            ctx.font = this.font.str();
            lineCount = 0;
            maxParaW = 0;
            for (i = 0; i < nParas; i++) {
                para = pa[i];
                remaining = para.line;
                para.lines = [];
                line = 0;
                para.lines[line] = '';
                lineWidth = 0;
                while (remaining !== undefined && remaining.length > 0) {
                    bits = remaining.split(' ', 1);
                    seg = para.lines[line] + bits[0];
                    seg = seg.replace(/<\/?[a-z][a-z0-9]*[^<>]*>/ig, ""); // Strips out HTML tags, so measurement doesn't consider them
                    lineWidth = ctx.measureText(seg, this.x, this.y);
                    // build a list of the text dimensions.
                    this.textMetrics.push(ctx.measureText(bits[0].replace(/<\/?[a-z][a-z0-9]*[^<>]*>/ig, ""), this.x, this.y));
                    if (maxWidth < lineWidth.width && para.lines[line] !== '' && this.dynamic) {
                        line++;
                        para.lines[line] = '';
                    }
                    else {
                        if (lineWidth.width > maxParaW)
                            maxParaW = lineWidth.width;
                    }
                    para.lines[line] += bits[0] + ' ';
                    remaining = remaining.substr(bits[0].length + 1);
                    if (maxParaW > maxWidth) { // overshot width so break the loops and resize
                        resize = true;
                        i = nParas;
                        break;
                    }
                }
                lineCount = lineCount + line + 1;
                if ((lineCount) * this.font.size * this.relLineHeight >= maxHeight) { // Overshot height so break and resize
                    resize = true;
                    i = nParas;
                    break;
                }
            }
            this.font.size--;
        }
        while (resize && this.font.size > 5 && this.dynamic && lineCount < this.maxLines);
    }
    else {

// I'm afraid this is a bit of a hack to make sure that a single line of text will not be split up onto
// a second line.
// It resizes the text until it fits correctly within the required space.
// Removing it shouldn't cause any problems and it will only be used if noBreak is set to true.
// MD
        para = pa[0];
        remaining = para.line;
        para.lines = [];
        para.lines.push(para.line);
        lineWidth = 0;
        seg = para.line;
        seg = seg.replace(/<\/?[a-z][a-z0-9]*[^<>]*>/ig, ""); // .replace Strips out HTML tags, so measurement doesn't consider them
        ctx.font = this.font.str();
        lineWidth = ctx.measureText(seg, this.x, this.y).width;
        lineCount = 1;
        while (lineWidth > maxWidth) {
            this.font.size--;
            ctx.font = this.font.str();
            lineWidth = ctx.measureText(seg, this.x, this.y).width;
            maxParaW = lineWidth;
        }
    }
    this.font.size++;
    if (this.autoWidth || this.measureOnly) {
        if (this.autoCentre) { // if Auto centre, then maintain centre when reducing width
            this.x = this.x + (this.w - (maxParaW + (2 * this.padding))) / 2;
        }
        this.w = maxParaW + (2 * this.padding);
    }
    if (this.params.background)
        ctx.background(this.params.background, this);
    //    ctx.textAlign = this.align;
    if (this.align === "center")
        ctx.textAlign = this.align;
    else
        ctx.textAlign = "left";
    if (this.alignPara === 'center')
        paraX = (this.w - maxParaW) / 2 + this.x;
    //TODO add verticalAlign here - currently just centers vertically
    textHeight = lineCount * this.font.size * this.relLineHeight; // number of lines * lineHeight
    textHeight = textHeight - this.font.size * (this.relLineHeight - 1); // minus the padding from the last line
    textHeight = textHeight + (pa.length - 1) * this.font.size * this.paraBottomMargin; // plus paragraph bottom margins

    if (this.measureOnly) { //  If only measuring the sizes then return now.
        this.h = (textHeight + (this.font.size * this.paraBottomMargin * (pa.length - 1)) + (this.font.size * (this.relLineHeight - 1)) - (2 * this.verticalPadding)) / .8;
        return;
    }

    if (this.params.verticalAlign === 'top')
        yStart = this.y + (this.font.size / 2);
    else if (this.params.verticalAlign === 'bottom')
        yStart = this.y + (this.h - textHeight) + (this.font.size / 2);
    else
        yStart = this.y + ((this.h - textHeight) / 2) + (this.font.size / 2);
    /*
     * Now render the text
     */
    lineSize = this.font.size;
    lineCount = 0;
    for (i = 0; i < nParas; i++) {
        para = pa[i];
        if (para.colour)
            ctx.fillStyle = para.colour;
        if (para.rgb) {
            if (para.alpha)
                ctx.fillStyle = 'rgba(' + para.rgb + ',' + para.alpha + ')';
            else
                ctx.fillStyle = 'rgba(' + para.rgb + ',1)';
        }
        tl = para.lines.length;
        for (j = 0; j < tl; j++) {
// split it on spaces first then do the reg ex on each block
// add up everything between spaces to get the hitbox
            remaining = '' + para.lines[j];
            remaining = remaining.replace(/<\/?[a-z][a-z0-9]*[^<>]*>/ig, ""); // .replace Strips out HTML tags, so measurement doesn't consider them
            lineW = ctx.measureText(remaining, this.x, yStart).width;
            if (this.align === 'center')
                oldX = this.x + (this.w - lineW) / 2;
            else if (this.align === 'right')
                oldX = this.x + this.w - this.padding - lineW;
            else // left
                oldX = paraX;
            phrases = para.lines[j].split(' ');
            for (wrd = 0; wrd < phrases.length; wrd++) {
                lineTxt = phrases[wrd].split(/(<\/?[a-z][a-z0-9]*[^<>]*>)\s*/gi) || [para.lines[j]];
                x = oldX;
                for (b = 0; b < lineTxt.length; b++) {
                    seg = lineTxt[b];
                    switch (seg) {
                        case "<strong>":
                        case "<b>":
                            this.font.weight = 'bold';
                            ctx.font = this.font.str();
                            break;
                        case "</strong>":
                        case "</b>":
                            this.font.weight = '';
                            ctx.font = this.font.str();
                            break;
                        case "<i>":
                        case "<em>":
                            this.font.style = 'italic';
                            ctx.font = this.font.str();
                            break;
                        case "</i>":
                        case "</em>":
                            this.font.style = '';
                            ctx.font = this.font.str();
                            break;
                        case "<sub>":
                            yStart += this.font.size * .4;
                            this.font.size *= .6;
                            ctx.font = this.font.str();
                            break;
                        case "</sub>":
                            this.font.size /= .6;
                            ctx.font = this.font.str();
                            yStart -= this.font.size * .4;
                            break;
                        case "<sup>":
                            yStart -= this.font.size * .4;
                            this.font.size *= .6;
                            ctx.font = this.font.str();
                            break;
                        case "</sup>":
                            this.font.size /= .6;
                            ctx.font = this.font.str();
                            yStart += this.font.size * .4;
                            break;
                        case "<br/>": // convert breaks to 'space' - //TODO consider inserting a line break instead
                        case "<br>":
                            wordW = ctx.measureText(' ', this.x, yStart).width;
                            x += wordW;
                            break;
                        default:
                            wordW = ctx.measureText(seg, this.x, yStart).width;
                            if (this.align === 'center') {
                                wboxX = x + wordW / 2;
                                ctx.fillText(seg, x + wordW / 2, yStart + (lineCount * lineSize * this.relLineHeight)); // note last character is trimmed, as all lines will have a trailing space
                            }
                            else if (this.align === 'right') {
                                wboxX = x;
                                ctx.fillText(seg, x, yStart + (lineCount * lineSize * this.relLineHeight)); // note last character is trimmed, as all lines will have a trailing space
                            }
                            else { // left
                                wboxX = x;
                                ctx.fillText(seg, x, yStart + (lineCount * lineSize * this.relLineHeight));
                            }
                            var ly = yStart + (lineCount * lineSize * this.relLineHeight);
                            if (b === 0) { // first character, load up the hitbox.
                                wrdx = wboxX;
                                wrdy = ly - (this.font.size / 2);
                                wrdw = wboxX + wordW;
                                wrdh = (ly - (this.font.size / 2)) + this.font.size;
                            }
                            else {
                                if (wboxX < wrdx)
                                    wrdx = wboxX;
                                if ((ly - (this.font.size / 2)) < wrdy)
                                    wrdy = (ly - (this.font.size / 2));
                                if ((wboxX + wordW) > wrdw)
                                    wrdw = wboxX + wordW;
                                if (((ly - (this.font.size / 2)) + this.font.size) > wrdh)
                                    wrdh = (ly - (this.font.size / 2)) + this.font.size;
                            }
                            /*
                             // Shows bounding boxes for words..
                             ctx.save();
                             ctx.strokeStyle="#0000ff";
                             ctx.strokeRect(x,ly-(this.font.size/2),wordW,this.font.size);
                             ctx.restore();
                             // */
//                        this.textBoxes.push({seg:seg, x:x,y:( yStart + (lineCount * lineSize * this.relLineHeight)),w:wordW,h:this.font.size});
                            x += wordW;
                            break;
                    }
// Lets add a space after every word.
                    oldX = x;
                }
                wordW = ctx.measureText(' ', this.x, yStart).width;
                oldX += wordW;
                this.textBoxes.push({
                    seg: seg,
                    x: wrdx,
                    y: wrdy,
                    w: wrdw - wrdx,
                    h: wrdh - wrdy
                });
            }
            lineCount++;
        }
        yStart += lineSize * this.paraBottomMargin;
    }
    ctx.restore();
};
/**
 * @deprecated - Use Headers via the controller
 */
OU.util.pageTitle = function(params) {
    var ctx = params.context,
            w = params.w || ctx.canvas.width,
            h = params.h || ctx.canvas.height * 0.05,
            x = params.x || 0,
            y = params.y || 0;
    ctx.fillStyle = params.col || 'rgba(255,255,255,1)';
    ctx.fillRect(x, y, w, h);
    new OU.util.DynText({
        txt: params.txt,
        x: x,
        y: y,
        w: w,
        h: h,
        context: ctx,
        propTextHeight: 0.5,
        fontWeight: 'bold',
        maxLines: 1,
        align: 'left'
    });
};
/**
 * @deprecated - Use Headers via the controller
 */
OU.util.pageStrap = function(params) {
    var ctx = params.context,
            w = params.w || ctx.canvas.width,
            h = params.h || ctx.canvas.height * 0.05,
            x = params.x || 0,
            y = params.y || ctx.canvas.height * 0.05,
            gradient = ctx.createLinearGradient(x, y, w, h);
    gradient.addColorStop(0, params.col1 || 'rgba(255,255,255,0.5)');
    gradient.addColorStop(1, params.col2 || 'rgba(255,255,255,0)');
    ctx.fillStyle = gradient;
    ctx.fillRect(x, y, w, h);
    new OU.util.DynText({
        txt: params.txt,
        x: x,
        y: y,
        w: w,
        h: h,
        context: ctx,
        propTextHeight: 0.2,
        maxLines: 1,
        align: 'left'
    });
};
