/**
 * @fileOverview VarHist  - displays a variable, showing a historical graph as well as the current value
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
/**
 * @class Variable history graph - Shows the historical value of a variable on a moving graph.
 * 
 * @param {object} params - options:
 * <ul>
 * <li><strong>{double} x:</strong> X Co-ordinate</li>
 * <li><strong>{double} y:</strong> Y Co-ordinate</li>
 * <li><strong>{double} w:</strong> Width</li>
 * <li><strong>{double} h:</strong> Height</li>
 * <li><strong>{double} min:</strong> Minimum value of the variable, to set the scale</li>
 * <li><strong>{double} max:</strong> Maximum value of the variable, to set the scale</li>
 * <li><strong>{canvas.context} context:</strong> Context of the canvas to render to</li>
 * <li><strong>{string} title:</strong> Optional title to display</li>
 * <li><strong>{double} val:</strong> Optional initial value, defaults to 0 if undefined</li>
 * </ul>
 */
OU.util.VarHist = function ( params ) {
    this.params = params;
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || 100;
    this.h = params.h || 40;
    this.min = params.min;
    this.max = params.max;
    this.h = this.h - 20;
    this.title = params.title || '';
    /**
     * Init the history array and some settings
     * @private
     */
    OU.util.VarHist.prototype.init = function () {
        var v = this.val = this.params.val || 0;
        this.hist = [];
        this.hist.push(v);
        if (v < this.min)
            this.min = v;
        if (v > this.max)
            this.max = v;
        this.numPoints = this.w - 60 | 0;
    };
    /**
     * Resizes the graph
     * @param {object} params - new dims:
     * <ul>
     * <li><strong>{double} x:</strong> X Co-ordinate</li>
     * <li><strong>{double} y:</strong> Y Co-ordinate</li>
     * <li><strong>{double} w:</strong> Width</li>
     * <li><strong>{double} h:</strong> Height</li>
     * </ul>
     */
    OU.util.VarHist.prototype.resize = function ( params ) {
        this.x = params.x || 0;
        this.y = params.y || 0;
        this.w = params.w || 100;
        this.h = params.h || 40;
        this.h = this.h - 20;
        this.numPoints = this.w - 60 | 0;
        this.render();
    };
    /**
     * Adds a new value to the history.
     * Note, this does not call render, as it it assumed render is being called separately.
     * @param {double} value - current value
     */
    OU.util.VarHist.prototype.update = function ( value ) {
        this.hist.push(value);
        this.val = value;
        if (value < this.min)
            this.min = value;
        if (value > this.max)
            this.max = value;
    };
    /**
     * Renders the graph of the historical value
     */
    OU.util.VarHist.prototype.render = function () {
        var ctx = this.params.context, idx,
            i, v, x = this.x, y = this.y, w = this.w, h = this.h,
            dy = this.max - this.min, histSize = this.hist.length;
        dy = (dy==0) ? 1 : dy;
        ctx.save();
        ctx.textAlign = 'center';
        ctx.strokeStyle = '#999';
        ctx.fillStyle = '#999';
        ctx.fillRect(x, y + h, w, 20);
        ctx.strokeRect(x, y, w, h + 20);
        ctx.font = '12px ' + OU.theme.font;
        ctx.fillStyle = '#fff';
        ctx.fillText(this.title, this.x + this.w / 2, this.y + this.h + 10);
        ctx.strokeStyle = '#c00';
        ctx.font = 'bold 16px ' + OU.theme.font;
        ctx.fillStyle = '#444';
        ctx.fillText(this.val, this.x + 30, this.y + this.h / 2);
        ctx.beginPath();
        v = this.h * (this.hist[histSize - 1] - this.min) / dy;
        ctx.moveTo(this.x + this.w, this.y + this.h - v);
        for (i = this.numPoints - 1; i--;) {
            idx = histSize - (this.numPoints - i) - 1;
            if (idx >= 0) {
                v = this.h * (this.hist[idx] - this.min) / dy;
                ctx.lineTo(this.x + i + 60, this.y + this.h - v);
            }
            else {
                i = 0;
            }
        }
        ctx.stroke();
        ctx.restore();
    };
    /**
     * Resets the data and graph
     */
    OU.util.VarHist.prototype.reset = function () {
        this.init();
    };
    this.init();
};