/**
 * @fileOverview  Layers Object
 * @param {JSON array} data Data as included in the data/data.js file
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Slider');
OU.require('OU.util.Layer');
OU.require('OU.util.varControlBank');
OU.require('OU.util.ImageLoader');
/**
 * @class Layers activity
 * @extends OU.util.Activity
 * @param {object} data - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.Layers = function(data, instance, controller) {
    /**
     * Starting point when running with HTML5 Canvas Support
     */
    OU.activity.Layers.prototype.canvasView = function() {

        var self = this;

        this.config = {
            imageLoadTimeout: 10000, // allow max 10 seconds to load images

            // animation settings
            fps: 40, // 40ms = 25 fps
            slideDuration: 10000, // each slide takes 10 seconds
            transPerc: 0.2 // percentage of slideDuration to face in next slide (20%)
        };
        if (this.data.hasOnOff === undefined)
            this.data.hasOnOff = true;
        if (this.data.hasGrip === undefined)
            this.data.hasGrip = true;
        if (this.data.hasSlider === undefined)
            this.data.hasSlider = false;
        if (this.data.hasVal === undefined)
            this.data.hasVal = false;

        this.controls = [];
        this.view = 'norm';

        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.data.images,
            onLoad: function() {
                self.start();
            }
        });
    };
    /**
     * resize all slides based on new dimensions
     */
    OU.activity.Layers.prototype.resize = function() {
        OU.activity.Layers.superClass_.resize.call(this); // call the parent class resize

        var i, slide, images = this.data.images, bH = OU.controlHeight, maxSlideHeight = 0;

        for (i = 0; i < images.length; i++) {
            slide = images[i];
            if (slide.w > this.w) {
                slide.h = slide.h * (this.w / slide.w);
                slide.w = this.w;
            }
            if (slide.h > this.h - bH) {
                slide.w = slide.w * ((this.h - bH) / slide.h);
                slide.h = this.h - bH;
            }
            if (slide.h > maxSlideHeight)
                maxSlideHeight = slide.h;
            if (slide.layer) {
                slide.layer.resize();
                this.renderLayer(slide);
            }
        }
        this.controlY = (this.h - bH - maxSlideHeight) / 2 + maxSlideHeight;
        this.controlY = this.controlY + bH / 2 < this.h - bH ? this.controlY + bH / 2 : this.h - bH;

        if (this.controlLayer) {
            this.controlLayer.resize();
            this.controlBank.resize({
                x: 0,
                y: this.controlY,
                w: this.w,
                h: bH
            });
            this.doRender = true;
        }
    };
    /**
     * Initialise a new layer object for each slide in the data
     */
    OU.activity.Layers.prototype.initLayers = function() {
        var i, slide, images = this.data.images,
                bH = OU.controlHeight, maxSlideHeight = 0;

        for (i = 0; i < images.length; i++) {
            slide = images[i];
            slide.val = slide.opacity;
            slide.min = 0;
            slide.max = 1;
            if (slide.w > this.w) {
                slide.h = slide.h * (this.w / slide.w);
                slide.w = this.w;
            }
            if (slide.h > this.h - bH) {
                slide.w = slide.w * ((this.h - bH) / slide.h);
                slide.h = this.h - bH;
            }
            if (slide.h > maxSlideHeight)
                maxSlideHeight = slide.h;
            slide.layer = new OU.util.Layer({
                container: this,
                zIndex: this.zOffset + 25 + i
            });
            this.renderLayer(slide);
        }
        this.controlY = (this.h - bH - maxSlideHeight) / 2 + maxSlideHeight;
        this.controlY = this.controlY + bH / 2 < this.h - bH ? this.controlY + bH / 2 : this.h - bH;

        this.initControls();
    };
    /**
     * switch render on
     * @param {object} me - reference to the activity
     */
    OU.activity.Layers.prototype.render = function(me) {
        me.renderData = true;
    };
    /**
     * Update the opacity of each layer, according to the controls
     */
    OU.activity.Layers.prototype.performRender = function() {
        var i, slide, control, images = this.data.images;

        for (i = images.length; i--; ) {
            slide = images[i];
            control = this.controlBank.controls[i];
            control.variable.layer.canvas.style.zIndex = this.zOffset + 25 + control.sortOrder;
            slide.layer.opacity(slide.val);
        }
        this.renderData = false;
    };
    /**
     * sets the opacity and renders the slide to the layer
     * @param {object} layer
     */
    OU.activity.Layers.prototype.renderLayer = function(layer) {
        var ctx = layer.layer.context,
                x = (this.w - layer.w) / 2,
                y = (this.h - OU.controlHeight - layer.h) / 2;

        layer.layer.opacity(layer.val);
        ctx.drawImage(layer.image, x, y, layer.w, layer.h);
    };
    /**
     * Initialise the controls - sets up a varControlBank to manipulate the behaviour of each layer.
     */
    OU.activity.Layers.prototype.initControls = function() {
        var bH = OU.controlHeight,
                images = this.data.images,
                clickable;

        this.controlLayer = new OU.util.Layer({
            container: this,
            hasEvents: true,
            zIndex: this.zOffset + OU.CONTROL_LEVEL,
            id: 'control'
        });
        clickable = this.controlLayer.events.clickable;

        if (this.controlBank)
            this.controlBank.close();
        this.controlBank = new OU.util.varControlBank({
            x: 0,
            y: this.controlY,
            w: this.w,
            h: bH,
            vars: images,
            layer: this.controlLayer,
            clickable: clickable,
            callback: this.render,
            parent: this,
            hasOnOff: this.data.hasOnOff, // true,
            hasGrip: this.data.hasGrip, // true,
            hasSlider: this.data.hasSlider, // true,
            hasVal: this.data.hasVal, // false,
            fillWidth: true
        });

        this.renderControls();
    };
    /**
     * re-renders the control elements
     */
    OU.activity.Layers.prototype.renderControls = function() {
        this.controlLayer.clear();
        this.controlBank.render();
        this.doRender = false;
    };
    /**
     * set up a render loop so that we can animate the dragging of the layers' order
     */
    OU.activity.Layers.prototype.renderCycle = function() {
        var self = this;
        if (this.controlLayer.context) {
            if (this.doRender) {
                this.renderControls();
            }
            if (this.renderData) {
                this.performRender();
            }
            setTimeout(function() {
                self.renderCycle();
            }, 40);
        }
    };
    /**
     * Starts the activity, called after the imageLoader is finished
     */
    OU.activity.Layers.prototype.start = function() {
        this.initLayers();

        this.doRender = true;
        this.renderData = true;
        this.renderCycle();
    };
    /**
     * Output a text only version of the activity if canvas is not supported.
     */
    OU.activity.Layers.prototype.accessible = function() {
        var h = '<div id="accessibleView">';
        h += '<h1>' + this.data.title + '</h1>';
        h += '<p>' + this.data.description + '</p>';
        h += '<h2>This activity has the following layers:</h2>';
        for (var i = 0; i < this.data.images.length; i++)
            h += '<h3>' + this.data.images[i].name + '</h3>';

        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };

    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Layers, OU.util.Activity);
