/**
 * @fileOverview htmlBox - places a div element on the page (possibly over the canvas) and inserts html
 *           Benefiting from browser parsing of content
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 *
 */
/**
 * @class HtmlBox - Provides a div element combined with a frame and close button.
 * Only one HtmlBox is allows to be open at any time.
 *
 * @param {object} params - A range of options:
 * <ul>
 * <li><strong>{object} container:</strong> (required) reference to the calling object (typically an OU.util.Activity)</li>
 * <li><strong>{string} html:</strong> The HTML content to put in the div</li>
 * <li><strong>{int} x:</strong> Left co-ordinate of the div</li>
 * <li><strong>{int} y:</strong> Top co-ordinate of the div</li>
 * <li><strong>{int} w:</strong> Width of the div</li>
 * <li><strong>{int} h:</strong> Height of the div</li>
 * <li><strong>{string} style:</strong> Optional CSS to add to the div</li>
 * <li><strong>{boolean} unclosable:</strong> if true, no close button is included, defaults to false</li>
 * <li><strong>{boolean} handleScrolling:</strong> if true, then we handle the scrolling of the div manually</li>
 * <li><strong>{boolean} showScroll:</strong> if true, then manually render some scroll bars</li>
 * <li><strong>{boolean} dontRegister:</strong> if true, then the HtmlBox isn't put into the removables array and won't be removed by a controller</li>
 * </ul>
 */
OU.util.HtmlBox = function ( params ) {
    var htmlBox = this;

    if (document.getElementById('htmlBox')!=null) // only render 1 box - important incase this is called within a rendering cycle
        return;
    if (params===undefined)
        params = {};
    var style = params.style || '';
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || 400;
    this.h = params.h || 300;
    this.container = params.container || {};
    if (this.w > 40) // adjust for padding
        this.w -= 40;
    if (this.h > 40)
        this.h -= 40;
    this.startY = -1;
    this.dead = false;
    this.unclosable = false;
    if (params.unclosable!==undefined)
        this.unclosable = params.unclosable;
    this.handleScrolling = true;
    if (params.handleScrolling!==undefined)
        this.handleScrolling = params.handleScrolling;
    this.div = document.createElement('div');
    this.showScroll = true;
    if (params.showScroll!==undefined) this.showScroll = params.showScroll;
    if (params.unclosable)
        this.div.setAttribute('id', "uncloseableHtmlBox");
    else
        this.div.setAttribute('id', "htmlBox");
    this.div.setAttribute('style', "overflow:auto; position:absolute; top:" + this.y + "px; left:" + this.x + "px; width:" + this.w + "px; height:" + this.h + "px; padding:0 20px;" + style);
    this.zOffset = this.container.zOffset || 0;
    this.div.style.zIndex = this.zOffset + OU.POP_UP_LEVEL;
    this.div.innerHTML = params.html;
    document.body.appendChild(this.div);
    if (this.showScroll && OU.IS_IPAD) {
        this._scroll = document.createElement('canvas');
        document.body.appendChild(this._scroll);
        this._scroll.height = this.h;
        this._scroll.width = "20";
        this._scroll.top = this.y;
        this._scrollctx = this._scroll.getContext('2d');
        this._scroll.setAttribute('style', 'z-index:10000;position:absolute;top:' + this.y + 'px; left:' + (this.x + this.w + 25) + 'px; height:' + this.h + ';width:20px;');
    }
    if (params.dontRegister===undefined) {
            OU.addRemovable(this, this.zOffset);
    }
    /**
     * Resizes the HtmlBox
     * @param {object} dims - new position with following elements:
     * <ul>
     * <li><strong>{int} x:</strong> Left co-ordinate of the div</li>
     * <li><strong>{int} y:</strong> Top co-ordinate of the div</li>
     * <li><strong>{int} w:</strong> Width of the div</li>
     * <li><strong>{int} h:</strong> Height of the div</li>
     * </ul>
     */
    OU.util.HtmlBox.prototype.resize = function ( dims ) {
        if (dims.x!==undefined)
            this.div.style.left = dims.x + 'px';
        if (dims.y!==undefined)
            this.div.style.top = dims.y + 'px';
        if (dims.w!==undefined)
            this.div.style.width = dims.w + 'px';
        if (dims.h!==undefined)
            this.div.style.height = dims.h + 'px';
    };
    /**
     * @private
     */
    OU.util.HtmlBox.prototype.touchStart = function ( e ) {
        e = OU.combineMouseTouch(e);
        htmlBox.startY = e.pageY;
    };
    /**
     * @private
     */
    OU.util.HtmlBox.prototype.touchUp = function ( e ) {
        e = OU.combineMouseTouch(e);
        htmlBox.startY = -1;
    };
    /**
     * @private
     */
    OU.util.HtmlBox.prototype.touchMove = function ( e ) {
        e.preventDefault();
        e = OU.combineMouseTouch(e);
        if (htmlBox.startY==-1)
            return;
        var dy = (htmlBox.startY - e.pageY) * 0.1;
        var newTop = htmlBox.div.scrollTop + dy;
        if (newTop < 0)
            newTop = 0;
        if (newTop > htmlBox.div.scrollHeight - htmlBox.h)
            newTop = htmlBox.div.scrollHeight - htmlBox.h;
        htmlBox.div.scrollTop = newTop;
    };
    /**
     * @private
     */
    OU.util.HtmlBox.prototype.handleScroll = function () {
        var sh = htmlBox.div.scrollHeight, st = htmlBox.div.scrollTop, ch = htmlBox.div.clientHeight;
        var p, th, tt, w, h;
        w = htmlBox._scroll.width;
        h = htmlBox._scroll.height;
        p = ch / sh;
        th = ch * p;
        tt = ((ch - th) * (st / (sh - ch)));
        if (ch==sh) return;
        htmlBox._scrollctx.clearRect(0, 0, w, h);
        htmlBox._scrollctx.fillStyle = "rgba(128,128,128,0.1)";
        htmlBox._scrollctx.fillRect(5, 0, 10, h);
        htmlBox._scrollctx.fillStyle = "rgba(0,0,0,0.2)";
        htmlBox._scrollctx.fillRect(5, tt, 10, th);
    };
    /**
     * Removes the HtmlBox and corresponding elements
     */
    OU.util.HtmlBox.prototype.remove = OU.util.HtmlBox.prototype.close = function () {
        if (this.div!=null) {
            this.div.removeEventListener("mousedown", this.touchStart, false);
            this.div.removeEventListener("touchstart", this.touchStart, false);
            this.div.removeEventListener("touchend", this.touchUp, false);
            this.div.removeEventListener("touchmove", this.touchMove, false);
            if (this._scroll!=null)   this.div.removeEventListener("scroll", this.handleScroll, false);
            this.div.parentNode.removeChild(this.div);
            this.div = null;
            // kill the scroll too
            if (this._scroll!=null) {
                document.body.removeChild(this._scroll);
                this._scrollctx = null;
                this._scroll = null;
            }
        }
    };
    if (this.handleScrolling) {
        this.div.addEventListener("mousedown", this.touchStart, false);
        this.div.addEventListener("touchstart", this.touchStart, false);
        this.div.addEventListener("touchend", this.touchUp, false);
        this.div.addEventListener("touchmove", this.touchMove, false);
        if (this._scroll!=null) {
            this.div.addEventListener("scroll", this.handleScroll, false);
            this.handleScroll();
        }
    }
};