/**
 * @fileOverview Viewport Slider
 * 
 * A bit clunky in use, especially if the view covers a large area
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Draggable');
/**
 * @class A 2D slider, similar to a scroll bar, but works in 2 directions.
 * @param {object} params - options:
 * <ul>
 * <li><strong>{double} x:</strong> X Co-ordinate</li>
 * <li><strong>{double} y:</strong> Y Co-ordinate</li>
 * <li><strong>{double} w:</strong> Width</li>
 * <li><strong>{double} h:</strong> Height</li>
 * <li><strong>{string} instance:</strong> unique id</li>
 * <li><strong>{canvas.context} context:</strong> context of the layer to render to</li>
 * <li><strong>{OU.util.Events} events:</strong> Events object for the layer this is on</li>
 * <li><strong>{function} callback:</strong> Function to call when position changes.</li>
 * <li><strong>{object} container:</strong> Reference to the calling object, passed back to the callback function as "me"</li>
 * <li><strong>{object} map:</strong> object defining the viewport and overall image dimensions {vpW,vpH,w,h}</li>
 * </ul>
 */
OU.util.ViewSlider = function( params ) {
    this.events = params.events;
    this.instance = params.instance || 's1';
    if(OU.sliders===undefined)
        OU.sliders = [];
    OU.sliders[this.instance] = this;

    this.params = params;
    this.container = params.container;
    this.context = params.context;
    if(OU.IS_IPAD)
        this.frames = data.SLIDER_RATE_IPAD || 2;
    else
        this.frames = data.SLIDER_RATE_PC || 10;
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || 100;
    this.h = params.h || 20;
    this.map = params.map;
    this.callback = params.callback || function() {};
    this.target = 0;
    this.timer=null;
    this.slider = {
        x:this.x,
        y:this.y,
        w:this.w * this.map.vpW/this.map.w,
        h:this.h * this.map.vpH/this.map.h
    };
    if(this.slider.w>this.w)
        this.slider.w = this.w;
    this.halted=false;
    this.init();
};
/**
 * Sets up the control
 * @private
 */
OU.util.ViewSlider.prototype.init = function() {
    this.draggable = new OU.util.Draggable( {
        "me":this,
        "events": this.events,
        "x": this.slider.x,
        "y": this.slider.y,
        "h": this.slider.h,
        "w": this.slider.w,
        "onStart": this.startDrag,
        "onEnd": this.endDrag,
        "onMove": this.newPos
    } );
    this.events.clickable.push(this.draggable);
};
/** @private */
OU.util.ViewSlider.prototype.startDrag = function() {};
/** @private */
OU.util.ViewSlider.prototype.endDrag = function() {};
/** @private */
OU.util.ViewSlider.prototype.newPos = function( sliderPos ) {
    var vs = sliderPos.me;

    vs.slider.x = sliderPos.x;
    vs.slider.y = sliderPos.y;

    vs.render();
    vs.callback( {
        x:(vs.slider.x-vs.x)*vs.map.w/vs.w,
        y:(vs.slider.y-vs.y)*vs.map.h/vs.h,
        me: vs.container
    } );
};
/**
 * Renders the slider - optionally updating the position
 * @param {object} sliderPos, json object of new position {x,y,w,h}
 */
OU.util.ViewSlider.prototype.render = function( sliderPos ) {
    var ctx = this.context;

    if(sliderPos!==undefined) {
        if(sliderPos.propHeight!==undefined) {
            this.slider.h = this.h*sliderPos.propHeight;
            if(this.slider.h>this.h)
                this.slider.h=this.h;
        }
        if(sliderPos.propWidth!==undefined) {
            this.slider.w = this.w*sliderPos.propWidth;
            if(this.slider.w>this.w)
                this.slider.w=this.w;
        }
        this.slider.x = this.x + sliderPos.x*this.w/this.map.w;
        this.slider.y = this.y + sliderPos.y*this.h/this.map.h;
    }

    if(this.slider.x<this.x)
        this.slider.x = this.x;
    if(this.slider.x>this.x+this.w-this.slider.w)
        this.slider.x = this.x+this.w-this.slider.w;
    if(this.slider.y<this.y)
        this.slider.y = this.y;
    if(this.slider.y>this.y+this.h-this.slider.h)
        this.slider.y = this.y+this.h-this.slider.h;

    this.draggable.x = this.slider.x;
    this.draggable.y = this.slider.y;

    ctx.background({
        clear:true,
        RGB:'255,255,255',
        alpha:0.8
    },{
        x:this.x-1,
        y:this.y-1,
        w:this.w+2,
        h:this.h+2
    });

    ctx.save();
    ctx.lineWidth=0.25;
    ctx.beginPath();
    ctx.fillStyle = "#ccc";
    ctx.fillRect(this.slider.x,this.slider.y,this.slider.w,this.slider.h);
    ctx.strokeStyle = "#444";
    ctx.strokeRect(this.slider.x,this.slider.y,this.slider.w,this.slider.h);
    ctx.closePath();
    ctx.restore();
    ctx.gripPattern(this.calcGrip());
};
/**
 * Calculates the visible "grip" area
 * @private
 */
OU.util.ViewSlider.prototype.calcGrip = function() {
    var w = this.slider.w/2>30 ? 30 : this.slider.w/2,
    h = this.slider.h/2>40 ? 40 : this.slider.h/2,
    x = this.slider.x+(this.slider.w-w)/2,
    y = this.slider.y+(this.slider.h-h)/2;
    return {
        x: x,
        y: y,
        w: w,
        h:h
    };
};