/**
 * @fileOverview Choice2Table - user makes choices about a number of items which are then moved into a table structure
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Instruction');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
OU.require('OU.util.CanvasTable');

KEY_LEFT = 37;
KEY_UP = 38;
KEY_RIGHT = 39;
KEY_DOWN = 40;

/**
 * @class Choice2Table - renders a table in the bottom half of the activity and places items in the top half on a scrolling backdrop.
 * The user makes a decision for each item which is then moved into the correct sections of the table.
 *
 * @extends OU.util.Activity
 * @param {object} data - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.Choice2Table = function(data, instance, controller) {
    OU.activity.Choice2Table.prototype.canvasView = function() {
        var self = this, bH = OU.controlHeight;
        this.bgLayer = new OU.util.Layer({
            container: this
        });
        this.tableLayer = new OU.util.Layer({
            container: this
        });

        this.table = new OU.util.CanvasTable({
            container: this,
            layer: this.tableLayer,
            sections: this.data.table.sections,
            x: this.x + this.w * 0.05,
            y: this.y + this.h * 0.5,
            w: this.w * 0.9,
            h: this.h * 0.475,
            borderColour: this.data.table.borderColour,
            bgColour: this.data.table.bgColour,
            titleBgColour: this.data.table.titleBgColour,
            titleTextColour: this.data.table.titleTextColour,
            hasInfo: this.data.table.hasInfo
        });
        this.plinthLayer = new OU.util.Layer({
            container: this
        });
        this.itemsLayer = new OU.util.Layer({
            container: this
        });
        this.itemsLayer.context.font = "bold 20px " + OU.theme.font;

        this.navigatorLayer = new OU.util.Layer({
            container: this
        });
        this.questionLayer = new OU.util.Layer({
            container: this,
            hasEvents: true
        });

        if (this.data.preload === true) {
            this.controlLayer = new OU.util.Layer({
                x: this.x + this.w / 2 - bH,
                y: this.y + this.h - bH,
                w: bH * 2,
                h: bH,
                container: this,
                hasEvents: true
            });
        }
        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.data,
            onLoad: function() {
                self.start();
                self.started = true;
            }
        });
    };
    /**
     * Starts the activity off after all images have been loaded.
     * @private
     */
    OU.activity.Choice2Table.prototype.start = function() {
        var i, item, x, y;
        //Init data
        this.visibles = []; // onscreen items
        this.initScale();
        this.foregroundFactor = 2;
        this.plinthPosition = {
            x: 0,
            y: 0
        };
        this.popped = 0;
        this.strictOrder = this.data.strictOrder || false;
        for (i = 0; i < this.data.items.length; i++) {
            item = this.data.items[i];
            item.visible = false;
            if (item.image) {
                item.w = item.image.width * this.scale;
                item.h = item.image.height * this.scale;
            }
            else {
                item.w = this.itemsLayer.context.measureText(item.label, 0, 0).width * 1.5;
                item.h = 30;
            }

            if (item.startPos !== undefined) {
                x = this.foregroundFactor * item.startPos.x * this.scale;
                y = item.startPos.y * this.scale;
            }
            else {
                x = (this.w - item.w) / 2;
                y = 10;
            }
            this.visibles.push(new this.Item({
                "events": this.itemsLayer.events,
                "item": item,
                "sortOrder": this.popped,
                "order": this.popped,
                "name": item.label,
                "color": '#fcfcfc',
                "x": x,
                "y": y,
                "w": item.w,
                "h": item.h,
                disabled: this.strictOrder,
                parent: this
            }));
            this.popped++;
        }
        this.currentItem = 0;

        this.bgOffset = 0;

        // start render loop
        this.doRender = true;
        this.renderLoop();

        this.restart();
        this.renderBackground();
    };
    OU.activity.Choice2Table.prototype.startNavigator = function() {
        var nav = this.data.navigator;
        this.navigator = new this.Navigator({
            parent: this,
            layer: this.navigatorLayer,
            x: nav.startPos.x * this.scale,
            y: nav.startPos.y * this.scale,
            imageMoving: nav.moving.image,
            imageStatic: nav.still.image,
            targetItem: this.visibles[this.currentItem]
        });
    };

    /**
     * Resets the activity and starts it off again.
     * @private
     */
    OU.activity.Choice2Table.prototype.restart = function() {
        var self=this,shrinkArea, infoSize = 2 * OU.controlHeight;
        this._feedbackRendered = false;
        this.backgroundX = 0;
        if (this.data.infoButtonLocation === 'topRight')
            shrinkArea = {
                x: this.w - infoSize,
                y: OU.controlHeight,
                w: infoSize,
                h: infoSize
            };
        if (this.data.infoButtonLocation === 'bottomRight')
            shrinkArea = {
                x: this.w - infoSize,
                y: this.h - infoSize,
                w: infoSize,
                h: infoSize
            };
        if (this.data.infoButtonLocation === 'bottomLeft')
            shrinkArea = {
                x: OU.controlHeight,
                y: this.h - infoSize,
                w: infoSize,
                h: infoSize
            };
        this.infoButton = new OU.util.Instruction({
            container: this,
            message: this.data.instructions,
            shrinkArea:shrinkArea,
            onClose: function() {
                self.startNavigator();
            }
        });
    };
    OU.activity.Choice2Table.prototype.renderBackground = function() {
        var ctx = this.bgLayer.context,
                bg = this.data.background,
                w = bg.image.width * this.scale;
        this.bgLayer.resize({
            w: w
        });
        this.bgLayer.clear();
        ctx.fillStyle = this.data.backgroundColor;
        ctx.fillRect(0, 0, w, this.h);
        ctx.drawImage(bg.image, this.backgroundX, 0, w, bg.image.height * this.scale);
    };
    /**
     * Resizes all elements
     * @private
     */
    OU.activity.Choice2Table.prototype.resize = function() {
        OU.activity.Choice2Table.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, i, item, visible;
        if (!this.started)
            return;

        this.renderBackground();

        this.tableLayer.resize();
        this.table.resize({
            x: this.x + this.w * 0.05,
            y: this.y + this.h * 0.5,
            w: this.w * 0.9,
            h: this.h * 0.475
        });
        this.plinthLayer.resize();
        this.itemsLayer.resize();
        if (this.data.preload === true) {
            this.controlLayer.resize({
                x: this.x + this.w / 2 - bH,
                y: this.y + this.h - bH,
                w: bH * 2,
                h: bH
            });
            this.doneButton.render();
        }
        this.initScale();

        // reposition all items
        for (i = this.visibles.length; i--; ) {
            visible = this.visibles[i];
            item = visible.item;
            if (!visible.tableInfo) { // item is not in the table yet so handle position
                if (item.image) {
                    visible.resize(item.image.width * this.scale, item.image.height * this.scale);
                }

                if (item.startPos) {
                    x = item.startPos.x * this.scale;
                    y = item.startPos.y * this.scale;
                }
                visible.move(x, y);
            }
        }
        this.navigator.resize();

        this.doRender = true;
    };
    /**
     * Navigator token has arrived at a new item, so render question and options
     * @private
     */
    OU.activity.Choice2Table.prototype.arrived = function() {
        this.question = this.visibles[this.currentItem].question;
        this.doRender = true;
    };

    /**
     * Render question and options
     * @private
     */
    OU.activity.Choice2Table.prototype.renderQuestion = function() {
        var i, option, ctx = this.questionLayer.context,
                pctx = this.plinthLayer.context,
                visible, question, padding, bH = OU.controlHeight,
                numOptions, optionWidth, self = this,
                sections = this.data.table.sections,
                sc = this.scale;
        this.questionLayer.clear();
        if (this.question) {
            visible = this.visibles[this.currentItem];
            question = visible.question;
            padding = question.padding || {};
            if (!this.validationState) {
                this.plinthPosition.x = visible.x - visible.w / 2 + this.bgOffset;
                this.plinthPosition.y = visible.y + visible.h / 2 - (question.h - bH) / 2;
            }
            if (question.image)
                pctx.drawImage(question.image, this.plinthPosition.x, this.plinthPosition.y, question.w, question.h);
            new OU.util.DynText({
                txt: this.validationState ? this.data.validationQuestion : question.text,
                colour: (this.validationState ? this.data.validationColour : question.textColour) || '#ccc',
                x: this.plinthPosition.x + (padding.left || 0) * sc,
                y: this.plinthPosition.y + (padding.top || 0) * sc,
                w: question.w - (padding.left || 0) * sc - (padding.right || 0) * sc,
                h: question.h - (padding.top || 0) * sc - (padding.bottom || 0) * sc - bH,
                align: 'left',
                context: ctx
            });
            numOptions = sections.length;
            optionWidth = question.w / numOptions;
            if (this.options) {
                for (i = this.options.length; i--; ) {
                    this.options[i].resize({
                        x: this.plinthPosition.x + i * optionWidth,
                        y: this.plinthPosition.y + question.h - bH,
                        w: optionWidth,
                        h: bH
                    });
                    this.options[i].render();
                }
            }
            else {
                this.options = [];
                for (i = numOptions; i--; ) {
                    option = sections[i];
                    if (option.optionText) {
                        this.options[i] = new OU.util.ControlButton({
                            layer: this.questionLayer,
                            txt: option.optionText,
                            x: this.plinthPosition.x + i * optionWidth,
                            y: this.plinthPosition.y + question.h - bH,
                            w: optionWidth,
                            h: bH,
                            onClick: function(i) {
                                self.choose(i);
                            },
                            onClickParam: i
                        });
                    }
                }
            }
        }
    };
    OU.activity.Choice2Table.prototype.choose = function(optionIndex) {
        var i, item = this.visibles[this.currentItem],
                bH = OU.controlHeight, visible, question,
                self = this;

        visible = this.visibles[this.currentItem];
        question = visible.question;

        if (this.validationState) { // choosing after validation question

            if (optionIndex === 0) { // User wants to stick with decision
                //do nothing to the item.
            }
            else { // user wants to change.
                this.table.switchItem(item, (item.tableInfo.section.sectionIndex + 1) % this.table.sections.length);
            }

            // now clear question and move on
            this.plinthLayer.clear();
            this.table.info("");
            if (this.options) {
                for (i = this.options.length; i--; ) {
                    this.options[i].remove();
                }
            }
            this.options = null;
            this.question = null;
            if (this.currentItem === this.visibles.length) { // reached last item

            }
            else { //move to next item
                this.navigator.targetItem = this.visibles[++this.currentItem];
                this.targetItem++;
            }
            this.validationState = false;
        }
        else { // ask validation question
            this.table.addItem(item, optionIndex);
            item.x = item.x + this.bgOffset;
            this.table.info(this.question.validation);

            // clear question options
            if (this.options) {
                for (i = this.options.length; i--; ) {
                    this.options[i].remove();
                }
            }
            this.options = [];
            // Set new options for swap or continue

            this.options[0] = new OU.util.ControlButton({
                layer: this.questionLayer,
                txt: "Yes",
                x: this.plinthPosition.x,
                y: this.plinthPosition.y + question.h - bH,
                w: question.w / 2,
                h: bH,
                onClick: function() {
                    self.choose(0);
                }
            });
            this.options[1] = new OU.util.ControlButton({
                layer: this.questionLayer,
                txt: "No",
                x: this.plinthPosition.x + question.w / 2,
                y: this.plinthPosition.y + question.h - bH,
                w: question.w / 2,
                h: bH,
                onClick: function() {
                    self.choose(1);
                }
            });
            this.validationState = true;
        }
        this.doRender = true;
    };
    /**
     * Stops the animation loop
     */
    OU.activity.Choice2Table.prototype.remove = function() {
        OU.activity.Choice2Table.superClass_.remove.call(this); // call the superclass method
        this.doRender = false;
    };
    /**
     * Calculates the scale of the activity
     * @private
     */
    OU.activity.Choice2Table.prototype.initScale = function() {
        var bg = this.data.background.image;
        this.scale = (this.h / 2) / bg.height;
    };
    /**
     * Performs the render loop to enable animations
     * @private
     */
    OU.activity.Choice2Table.prototype.renderLoop = function() {
        var self = this;
        if (this.doRender && this.navigator)
            this.render();
        setTimeout(function() {
            self.renderLoop();
        }, 40);
    };

    /**
     * Renders the activity
     * @private
     */
    OU.activity.Choice2Table.prototype.render = function() {
        var bg = this.data.background,
                visible, i,
                navctx = this.navigatorLayer.context,
                ictx = this.itemsLayer.context,
                item = this.visibles[this.currentItem],
                targetTranslate, dT; // delta translation offset
        this.doRender = false;

        if (!this.validationState) {
            targetTranslate = (item.x / bg.image.width) * (this.w / (2 * this.foregroundFactor)) - item.x + item.w / 2;
            dT = targetTranslate - this.bgOffset; // delta translation offset
            dT = dT / (this.navigator.friction / 4);
            if (Math.abs(dT) < 1) {
                this.bgOffset = targetTranslate;
            }
            else if (Math.abs(dT) < 2) {
                this.bgOffset = targetTranslate;
//            this.arrived();
            }
            else {
                this.bgOffset = this.bgOffset + dT / 12;
                this.doRender = true;
            }
        }

        this.plinthLayer.clear();
        this.itemsLayer.clear();
        this.navigatorLayer.clear();

        // Translate the layers so that the navigator is always visible
        // this has the effect of scrolling the backdrop
        navctx.save();
        ictx.save();
        this.bgLayer.move({x: this.bgOffset / this.foregroundFactor});
        navctx.translate(this.bgOffset, 0);
        ictx.translate(this.bgOffset, 0);


        // Render items
        for (i = this.visibles.length; i--; ) {
            visible = this.visibles[i];
            // animate shuffle if out of place

            if (visible.tableInfo) { // visible is in the Table,so will have a position
                if (this.table.moveItem(visible))
                    this.doRender = true; // if item is in motion then enable shuffling
                visible.render();
            }
            else if (visible.x + this.bgOffset < this.x + this.w && visible.x + visible.w > -this.bgOffset) {
                visible.render();
            }
        }

        //animate and render navigator icon
        if (this.navigator.animate())
            this.doRender = true;

        // If currently on a question then render the question
        this.renderQuestion();

        // restore translations
        navctx.restore();
        ictx.restore();
    };

    /**
     * @class Navigator item/token that moves through the items
     * @param {object} params - options
     */
    OU.activity.Choice2Table.prototype.Navigator = function(params) {
        this.parent = params.parent;
        this.layer = params.layer;
        this.x = params.x;
        this.y = params.y;
        this.image = this.imageMoving = params.imageMoving;
        this.imageStatic = params.imageStatic;
        this.w = this.image.width * this.parent.scale;
        this.h = this.image.height * this.parent.scale;
        this.targetItem = params.targetItem;
        this.angle = 0;
        this.targetAngle = 0;
        this.friction = 10;
        this.scale = 1;
        this.targetScale = 1;
        /**
         * Animate the navigator icon
         */
        OU.activity.Choice2Table.prototype.Navigator.prototype.animate = function() {
            var targetX = this.targetItem.x + this.targetItem.w / 2 - this.w / 2,
                    targetY = this.targetItem.y + this.targetItem.h / 2 - this.h / 2,
                    dX = targetX - this.x,
                    dY = targetY - this.y,
                    theta = Math.atan(dY / dX);

            if (this.parent.validationState) { // don't move if in validation state
                this.render();
                return;
            }

            // Determine the current target angle
            if (Math.abs(dX) > this.w / 2 || Math.abs(dY) > this.h / 2) { // travelling, so point towards target
                if (dX > 0) {
                    if (dY > 0)
                        this.targetAngle = theta;
                    else
                        this.targetAngle = theta;
                }
                else {
                    if (dY > 0)
                        this.targetAngle = Math.PI + theta;
                    else
                        this.targetAngle = Math.PI + theta;
                }
                this.targetScale = 1;
            }
            else { // arriving, so land upright
                this.targetAngle = -Math.PI / 2;
                this.targetScale = 0.5;
            }
            if (this.scale !== this.targetScale) {
                this.scale = this.scale + (this.targetScale - this.scale) / 2;
            }

            // move the current angle towards the target angle
            if (this.angle !== this.targetAngle) {
                if (Math.abs(this.targetAngle - this.angle) < (1 / 180) * Math.PI) { // if with 1 degree, then make equal
                    this.angle = this.targetAngle;
                }
                else { // ensure we spin in the right direction - through the smallest angle
                    if (Math.abs(this.targetAngle - this.angle) > Math.PI) {
                        if (this.targetAngle > this.angle)
                            this.angle = this.angle + Math.PI * 2;
                        else
                            this.angle = this.angle - Math.PI * 2;
                    }
                    if (this.targetAngle > this.angle)
                        this.angle = this.angle + (this.targetAngle - this.angle) / 4;
                    else {
                        this.angle = this.angle - (this.angle - this.targetAngle) / 4;
                    }
                }

            }
            // we are at the correct angle, check for correct location
            else if (Math.abs(dX) === 0 && Math.abs(dY) === 0) {
                this.image = this.imageStatic;
                this.render();
                return false;
            }
            // check if we are really close to destination
            if (Math.abs(dX) < 2 && Math.abs(dY) < 2 && (this.angle === this.targetAngle)) {
                // Just arrived
                this.x = targetX;
                this.y = targetY;
                this.parent.arrived();
                this.friction = 20; // reset friction so we have a slow start when we move
                this.angle = this.targetAngle = -Math.PI / 2; // reset angles so we don't spin wildly
            }
            else {
                // Still on the way, so move according to distance away and friction
                this.x = this.x + dX / (6 * this.friction);
                this.y = this.y + dY / (6 * this.friction);
                if (this.friction > 1)
                    this.friction--;
            }
            this.image = this.imageMoving;
            this.render();
            return true;
        };

        OU.activity.Choice2Table.prototype.Navigator.prototype.resize = function() {
            this.w = this.image.width * this.parent.scale;
            this.h = this.image.height * this.parent.scale;
        };

        OU.activity.Choice2Table.prototype.Navigator.prototype.render = function() {
            var ctx = this.layer.context;
            ctx.save();
            ctx.translate(this.x + this.w / 2, this.y + this.h / 2);
            ctx.rotate(this.angle);
            ctx.drawImage(this.image, -this.w * this.scale / 2, -this.h * this.scale / 2, this.w * this.scale, this.h * this.scale);
            ctx.restore();
        };
    };
    /**
     * @class Item - An item in the list of choices
     * @param {object} params - options
     */
    OU.activity.Choice2Table.prototype.Item = function(params) {
        this.events = params.events;
        this.item = params.item;
        this.order = params.order;
        this.sortOrder = params.sortOrder;
        this.parent = params.parent;
        this.container = this.parent.container;
        this.name = params.name;
        this.color = params.color;
        this.x = params.x;
        this.y = params.y;
        this.h = params.h * this.parent.scale;
        this.w = params.w * this.parent.scale;
        this.ctx=params.parent.itemsLayer.context;
        this.image = params.item.image;
        this.startX = this.x;
        this.startY = this.y;
        if (this.image !== undefined) {
            this.w = this.image.width * this.parent.scale;
            this.h = this.image.height * this.parent.scale;
        }
        this.question = this.item.question;
        if (this.question.image) {
            this.question.w = this.question.image.width * this.parent.scale;
            this.question.h = this.question.image.height * this.parent.scale;
        }
        OU.activity.Choice2Table.prototype.Item.prototype.render = function() {
            var parent = this.parent,ctx = this.ctx,
                    bg,
                    sF = parent.scale,
                    x = this.x - (this.w * (sF - 1) / 2),
                    y = this.y - (this.h * (sF - 1) / 2),
                    w = this.w * sF,
                    h = this.h * sF,
                    realX = x+parent.bgOffset;
            if (this.tableInfo) {// if we are already in the table, then negate the translation
                x = x - parent.bgOffset;
            }
            else if (realX > parent.x + parent.w || realX + w < parent.x)
                return;
            if (this.image !== undefined) {
                ctx.drawImage(this.image, x | 0, y | 0, w, h);
            }
            else {
                bg = {
                    "borderCol": "#444",
                    "col": this.color,
                    "radius": h / 2
                };
                new OU.util.DynText({
                    "txt": this.name,
                    "x": x,
                    "y": y,
                    "w": w,
                    "h": h,
                    "context": ctx,
                    background: bg,
                    fontWeight: "bold",
                    fontSize: 20 * OU.dpr
                });
            }
        };
        OU.activity.Choice2Table.prototype.Item.prototype.move = function(x, y) {
            this.x = x;
            this.y = y;
        };
        OU.activity.Choice2Table.prototype.Item.prototype.resize = function(w, h) {
            this.w = w;
            this.h = h;
            if (this.question.image) {
                this.question.w = this.question.image.width * this.parent.scale;
                this.question.h = this.question.image.height * this.parent.scale;
            }
        };
    };

    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Choice2Table, OU.util.Activity);
