/**
 * @fileOverview OU.activity.Molecules - renders a 3D representation of a molecule - ball and stick stylie !
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.HtmlBox');
OU.require('OU.util.Slider');
OU.require('OU.util.varControlBank');
OU.require('OU.util.Layer');

KEY_LEFT=37;
KEY_UP=38;
KEY_RIGHT=39;
KEY_DOWN=40;

// Atom Types
OU.OTHER = 0;
OU.HYDROGEN = 1;
OU.CARBON = 2;
OU.NITROGEN = 3;
OU.OXYGEN = 4;
OU.FLUORINE = 5;
OU.CHLORINE = 6;
OU.BROMINE = 7;
OU.IODINE = 8;
OU.NOBLEGAS = 9;
OU.PHOSPHORUS = 10;
OU.SULFUR = 11;
OU.BORON = 12;
OU.ALKALIMETALS = 13;
OU.ALKALINEEARTHMETALS = 14;
OU.TITANIUM = 15;
OU.IRON = 16;
OU.atoms = [];
OU.atoms[OU.OTHER] =
{
    name:'?',
    col:'#DD77FF',
    r:221,
    g:119,
    b:255,
    text:'#444',
    radius:0.05
}
;
OU.atoms[OU.HYDROGEN] =
{
    name:'H',
    col:'#dddddd',
    r:221,
    g:221,
    b:221,
    text:'#444',
    radius:0.037
}
;
OU.atoms[OU.CARBON] =
{
    name:'C',
    col:'#000000',
    r:0,
    g:0,
    b:0,
    text:'#eee',
    radius:0.077
}
;
OU.atoms[OU.NITROGEN] =
{
    name:'N',
    col:'#2233FF',
    r:34,
    g:51,
    b:255,
    text:'#fff',
    radius:0.070
}
;
OU.atoms[OU.OXYGEN] =
{
    name:'O',
    col:'#FF2200',
    r:255,
    g:34,
    b:0,
    text:'#fff',
    radius:0.073
}
;
OU.atoms[OU.FLUORINE] =
{
    name:'F',
    col:'#55BB00',
    r:85,
    g:187,
    b:0,
    text:'#fff',
    radius:0.072
}
;
OU.atoms[OU.CHLORINE] =
{
    name:'Cl',
    col:'#55BB00',
    r:85,
    g:187,
    b:0,
    text:'#fff',
    radius:0.100
}
;
OU.atoms[OU.BROMINE] =
{
    name:'Br',
    col:'#992200',
    r:153,
    g:34,
    b:0,
    text:'#fff',
    radius:0.114
}
;
OU.atoms[OU.IODINE] =
{
    name:'I',
    col:'#6600BB',
    r:102,
    g:0,
    b:187,
    text:'#fff',
    radius:0.133
}
;
OU.atoms[OU.NOBLEGAS] =
{
    name:'He/Ne/Ar/Xe/Kr',
    col:'#00FFFF',
    r:0,
    g:255,
    b:255,
    text:'#444',
    radius:0.05
}
;
OU.atoms[OU.PHOSPHORUS] =
{
    name:'P',
    col:'#FF9900',
    r:255,
    g:153,
    b:0,
    text:'#444',
    radius:0.11
}
;
OU.atoms[OU.SULFUR] =
{
    name:'S',
    col:'#DDDD00',
    r:221,
    g:221,
    b:0,
    text:'#444',
    radius:0.103
}
;
OU.atoms[OU.BORON] =
{
    name:'B',
    col:'#FFAA77',
    r:255,
    g:170,
    b:119,
    text:'#444',
    radius:0.09
}
;
OU.atoms[OU.ALKALIMETALS] =
{
    name:'Li/Na/K/Rb/Cs',
    col:'#7700FF',
    r:119,
    g:0,
    b:255,
    radius:0.152
}
;
OU.atoms[OU.ALKALINEEARTHMETALS] =
{
    name:'Be/Mg/Ca/Sr/Ba/Ra',
    col:'#007700',
    r:0,
    g:119,
    b:0,
    text:'#fff',
    radius:0.222
}
;
OU.atoms[OU.TITANIUM] =
{
    name:'Ti',
    col:'#999999',
    r:153,
    g:153,
    b:153,
    text:'#000',
    radius:0.14
};
OU.atoms[OU.IRON] =
{
    name:'Fe',
    col:'#DD7700',
    r:221,
    g:119,
    b:0,
    text:'#fff',
    radius:0.076
};
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Molecules = function ( data, instance, controller ) {
    OU.activity.Molecules.prototype.canvasView = function () {
        this.config = {
            depth:5,
            fps:40, // 40ms = 25 fps
            spaceFillFactor:3.5
        };
        this.PAD = this.h * .045;
        this.inRender = false;
        this.controls = [];
        this.rotatable = this.data.settings.rotatable;
        this.spinOn = this.data.settings.spin;
        this.atomicAnimation = this.data.settings.animate;
        this.showZoomSlider = this.data.settings.zoomSlider;
        this.measureOn = this.data.settings.measure;
        this.namesOn = this.data.settings.symbols;
        this.wireframeOn = this.data.settings.wireframe;
        this.editable = this.data.settings.editable;
        this.addBonds = false;
        this.deleteAtoms = false;
        this.zOrder = [];
        this.validateCounts=0;
        this.moleculeIndex = OU.LocalStorage.load("OU.molecule3d.eq") || 0;
        this.setMolecules();
        if (this.data.waitForCue)
            this.state = OU.PAUSED;
        else
            this.start();
    };
    OU.activity.Molecules.prototype.start = function () {
        var bH = OU.controlHeight, ctx, bctx,self=this;
        // create Canvas Layers & Contexts
        this.state = OU.RUNNING;
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        bctx = this.bgLayer.context;
        bctx.gradRect();
        // draw background on backdrop layer
        this.pyLayer = new OU.util.Layer({
            container:this,
            id:'pitchyaw'
        });
        this.modelLayer = new OU.util.Layer({
            container:this,
            'id':'model',
            y:this.y,
            'h':this.h - bH,
            'hasEvents':true,
            pinch:this.pinch,
            pinchMe:this,
            keyPress:function(k) {
                self.keyPress(k);
            }
        });
        this.modelLayer.events.clickable.push(this);
        ctx = this.modelLayer.context;
        ctx.textAlign = 'center';
        ctx.font = '16px ' + OU.theme.font;
        this.controlsLayer = new OU.util.Layer({
            container:this,
            'y':this.y + this.h - bH,
            'h':bH,
            'id':'controls',
            hasEvents:true
        });
        if (this.editable) {
            this.rotatable = false; // no rotation on builder
            this.window = {
                x:0,
                y:0,
                w:this.w * .8,
                h:this.h - bH
            };
            bctx.grid({
                x:0,
                y:0,
                w:this.w * .8,
                h:this.h - bH
            });
            bctx.grid({
                x:this.w * .8,
                y:0,
                w:this.w * .2,
                h:this.h - bH,
                strokeStyle:'#33f'
            });
            this.editList = new this.EditList({
                parent:this
            });
        }
        else {
            this.window = {
                x:0,
                y:0,
                w:this.w,
                h:this.h - bH
            };
        }
        ctx.translate(this.window.x, this.window.y + this.window.h);
        this.midDrag = false;
        this.renderRange = {
            minX:0,
            maxX:0,
            minY:0,
            maxY:0
        };
        this.gridSpaces = 40;
        // Set equation to 1st and render
        this.changeMolecule(this.moleculeIndex);
        this.moleculeSelector();
        this.inertia = {
            x:0,
            y:0
        };
        this.requireRender = false;
        this.renderLoop();
        this.countToSpin = 0;
        this.spinning = true;
    };
    OU.activity.Molecules.prototype.remove = function () {
        OU.activity.Molecules.superClass_.remove.call(this); // call the superclass method
        var i;
        if (this.molecule && this.molecule.atoms) {
            for (i = this.molecule.atoms.length; i--;) {
                this.molecule.removeAtom(i);
            }
        }
        this.molecule = undefined;
        this.molecules = undefined;
    };
    OU.activity.Molecules.prototype.resize = function () {
        OU.activity.Molecules.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, ctx, bctx, i,
        buttons = this.data.settings.showButtons,
        bX = this.w - bH * 3;
        if (!this.molecule)
            return;
        // create Canvas Layers & Contexts
        this.bgLayer.resize();
        this.bgLayer.context.gradRect();
        // draw background on backdrop layer
        if (this.editable) {
            this.window = {
                x:0,
                y:0,
                w:this.w * .8,
                h:this.h - bH
            };
            bctx = this.bgLayer.context;
            bctx.grid({
                x:0,
                y:0,
                w:this.w * .8,
                h:this.h - bH
            });
            bctx.grid({
                x:this.w * .8,
                y:0,
                w:this.w * .2,
                h:this.h - bH,
                strokeStyle:'#33f'
            });
            this.editList = new this.EditList({
                parent:this
            });
        }
        else {
            this.window = {
                x:0,
                y:0,
                w:this.w,
                h:this.h - bH
            };
        }
        if (this.selectionLayer) {
            this.selectionLayer.resize({
                'x':this.x + this.window.x,
                'y':this.y + this.window.y + this.window.h - bH * 2,
                'w':this.window.w,
                'h':bH * 2
            });
            this.renderInstructions();
        }
        this.bgLabels();
        this.pyLayer.resize();
        this.modelLayer.resize({
            'h':this.h - bH
        });
        ctx = this.modelLayer.context;
        ctx.translate(this.window.x, this.window.y + this.window.h);
        ctx.textAlign = 'center';
        ctx.font = '16px ' + OU.theme.font;
        this.controlsLayer.resize({
            'y':this.y + this.h - bH,
            'h':bH
        });
        if (buttons.measure) {
            this.measureButton.resize({
                x:bX,
                y:0,
                w:bH * 3,
                h:bH
            });
            bX -= bH * 3;
        }
        if (buttons.symbols) {
            this.namesButton.resize({
                x:bX,
                y:0,
                w:bH * 3,
                h:bH
            });
            bX -= bH * 3.5;
        }
        if (buttons.wireframe) {
            this.wireframeButton.resize({
                x:bX,
                y:0,
                w:bH * 3.5,
                h:bH
            });
            bX -= bH * 3;
        }
        if (buttons.spaceFill) {
            this.spaceFillButton.resize({
                x:bX,
                y:0,
                w:bH * 3,
                h:bH
            });
            bX -= bH * 3;
        }
        if (buttons.spin) {
            this.spinButton.resize({
                x:bX,
                y:0,
                w:bH * 3,
                h:bH
            });
            bX -= bH * 3;
        }
        if (this.animateButtons) {
            for (i = this.animateButtons.length; i--;) {
                this.animateButtons[i].resize({
                    x:bX,
                    y:0,
                    w:bH * 3,
                    h:bH
                });
                bX -= bH * 3;
            }
        }
        if (this.editable) {
            this.bondsButton.resize({
                x:bX,
                y:0,
                w:bH * 3,
                h:bH
            });
            bX -= bH * 3;
            this.deleteButton.resize({
                x:bX,
                y:0,
                w:bH * 3,
                h:bH
            });
            bX -= bH * 3;
        }
        else {
            if (this.zoomSlider) {
                this.zoomSlider.resize({
                    x:0,
                    y:0,
                    w:bX + bH * 2,
                    h:bH - 2
                });
            }
        }
        this.render();
    };
    OU.activity.Molecules.prototype.setMolecules = function () {
        this.molecules = [];
        for (var i = 0; i < this.data.molecules.length;i++) {
            this.molecules.push(new this.Molecule(this.data.molecules[i], this));
        }
    };
    OU.activity.Molecules.prototype.moleculeSelector = function () {
        if (this.molecules.length < 2 || this.data.settings.showMenu===false)// don't include if only 1 equation
            return;
        OU.obj[instance] = this; // push this into object array, so we can call it back from the page
        var eq, h = '<form>Molecule: <select onchange="OU.obj.' + instance + '.changeMolecule();" id="eqList' + this.instance + '">';
        for (eq = 0; eq < this.molecules.length; eq++) {
            h += '<option value="' + eq + '">' + this.molecules[eq].name + '</option>';
        }
        h += '</select></form>';
        this.eqSelector = new OU.util.HtmlBox({
            container:this,
            html:h,
            x:this.x + 0,
            y:this.y + 10,
            w:5,
            h:5,
            unclosable:true,
            handleScrolling:false,
            style:'overflow:visible'
        });
    };
    OU.activity.Molecules.prototype.step = function ( n ) {
        if (this.state==OU.PAUSED)
            this.start();
        else if (n - 1!=this.moleculeIndex)
            this.changeMolecule(n - 1);
    };
    OU.activity.Molecules.prototype.changeMolecule = function ( eqNum ) {
        if (eqNum===undefined) {
            var eqList = document.getElementById('eqList' + this.instance);
            eqNum = eqList.value;
        }
        this.molecule = this.molecules[eqNum || 0];
        if (this.molecule===undefined)
            this.molecule = this.molecules[0];
        OU.LocalStorage.save("OU.molecule3d.eq", eqNum);
        this.moleculeIndex = eqNum;
        this.initMolecule();
    };
    OU.activity.Molecules.prototype.initMolecule = function () {
        var bH = OU.controlHeight, bctx, bgctx = this.bgLayer.context;
        this.bgLayer.resize();
        bgctx.gradRect();
        // draw background on backdrop layer
        if (this.editable) {
            bctx = this.bgLayer.context;
            bctx.grid({
                x:0,
                y:0,
                w:this.w * .8,
                h:this.h - bH
            });
            bctx.grid({
                x:this.w * .8,
                y:0,
                w:this.w * .2,
                h:this.h - bH,
                strokeStyle:'#33f'
            });
        }
        this.zoom = 0.25;
        if (this.controls.length > 0)
            this.controls[0].closeControls();
        if (this.molecule.selection!==undefined) {
            if (this.selectionLayer===undefined) {
                this.selectionLayer = new OU.util.Layer({
                    container:this,
                    'x':this.window.x,
                    'y':this.window.y + this.window.h - bH * 2,
                    'w':this.window.w,
                    'h':bH * 2,
                    'id':'selLayer',
                    hasEvents:true
                });
            }
            this.renderInstructions();
        }
        else {
            if (this.selectionLayer!==undefined) {
                this.selectionLayer.remove();
                this.selectionLayer = undefined;
            }
        }
        this.controlsWidth = 0;
        this.renderRange.radius = 0;
        this.initControls();
        this.resetView();
        this.bgLabels();
    };
    OU.activity.Molecules.prototype.solve = function () {
        var solved = new this.Molecule({
            molfile:this.molecule.selection.targetMolfile
        }, this);
        this.molecule.atoms=solved.atoms;
        this.solved=true;
        this.initMolecule();
    };
    OU.activity.Molecules.prototype.validateSelection = function () {
        var i, as = this.molecule.selection.atoms, correct = 0, selected = 0, msg = '', ctx,
        bH = OU.controlHeight, failed = true, target, targetCount = {}, molCount = {}, t,buttonsWidth=0,self=this;
        this.validateCounts++;
        if (this.data.settings.editable) {
            // compare the built structure with the target structure
            failed = false;
            target = new this.Molecule({
                molfile:this.molecule.selection.targetMolfile
            }
            , this);
            for (i = target.atoms.length; i--;) {
                if (targetCount[target.atoms[i].type]===undefined)
                    targetCount[target.atoms[i].type] = 1;
                else
                    targetCount[target.atoms[i].type]++;
            }
            for (i = this.molecule.atoms.length; i--;) {
                if (this.molecule.atoms[i].pt.in2d.x > this.w * .8) { // molecule is in editlist area, remove it
                    this.molecule.removeAtom(i);
                    this.render();
                }
                else {
                    if (molCount[this.molecule.atoms[i].type]===undefined)
                        molCount[this.molecule.atoms[i].type] = 1;
                    else
                        molCount[this.molecule.atoms[i].type]++;
                }
            }
            for (t in molCount) { // use of 'in' is intentional
                if (molCount[t]!=targetCount[t]) {
                    failed = true;
                    msg = "Sorry, you do not have the correct number of elements. Update your model to try again."
                }
            }
            for (t in targetCount) { // use of 'in' is intentional
                if (molCount[t]!=targetCount[t]) {
                    failed = true;
                    msg = "Sorry, you do not have the correct number of elements. Update your model to try again."
                }
            }
            if (!failed) {
                if (this.compMolStruct(this.molecule, target)) {
                    msg = "Well done, that is the correct molecular structure.";
                }
                else {
                    failed = true;
                    msg = "Sorry, you have the correct elements present, but the bonds between them are not correct. Update your model to try again."
                }
            }
        }
        else {
            // Compare the selected atoms with the target atoms
            for (i = as.length; i--;) {
                if (this.molecule.atoms[as[i] - 1].selected)
                    correct++;
            }
            for (i = this.molecule.atoms.length; i--;) {
                if (this.molecule.atoms[i].selected)
                    selected++;
            }
            if (selected==correct && correct==as.length) {
                msg = "Well done, you have correctly identified the required elements.";
                failed = false;
            }
            else if (correct==0) {
                msg = "Sorry, you have not selected any of the required elements. Change your selection to try again."
            }
            else if (correct==as.length) {
                msg = "Sorry, you have selected more than the required elements. Change your selection to try again."
            }
            else {
                msg = "Sorry, you have included some of the elements, but have not got it quite right. Change your selection to try again.";
            }
        }

        ctx = this.selectionLayer.context;
        this.selectionLayer.clear();

        if (this.validateCounts>2) {
            buttonsWidth = buttonsWidth+bH*4.25;
            if (this.solveButton) {
                this.solveButton.render({
                    x:this.window.x + this.window.w - buttonsWidth,
                    y:bH * .5,
                    w:bH * 4,
                    h:bH
                });
            }
            else {
                this.solveButton = new OU.util.ControlButton({
                    txt:"Show Answer",
                    x:this.window.x + this.window.w - buttonsWidth,
                    y:bH * .5,
                    w:bH * 4,
                    h:bH,
                    layer:this.selectionLayer,
                    onClick:function () {
                        self.solve();
                    }
                });
            }
        }

        this.selectionText = new OU.util.DynText({
            context:ctx,
            align:'left',
            x:this.window.x,
            y:0,
            w:this.window.w - buttonsWidth,
            h:bH * 2,
            txt:msg
        });
    };
    OU.activity.Molecules.prototype.compMolStruct = function ( m1, m2 ) {
        // Compare Molecular structures - Atoms & Bonds only - physical location is not important
        var i, j;
        for (i = m1.atoms.length; i--;) {
            for (j = m2.atoms.length; j--;) {
                m1.atoms[j].visited = m2.atoms[j].visited = undefined;
            }
            if (this.compAtomStruct(m1.atoms[i], m2.atoms[0], m1, m2, 0))
                return true;
        }
        return false;
    };
    OU.activity.Molecules.prototype.compAtomStruct = function ( a, b, m1, m2, depth ) {
        // Recurse through the molecular structure checking for correct matches of atomic type and bonds
        var i, j, aCount = {}, bCount = {}, t, possible, c, d;
        if (a.type!=b.type)
            return false;
        if (a.links.length!=b.links.length)
            return false;
        for (i = a.links.length; i--;) {
            if (aCount[m1.atoms[a.links[i]].type]===undefined)
                aCount[m1.atoms[a.links[i]].type] = 1;
            else
                aCount[m1.atoms[a.links[i]].type]++;
        }
        for (i = b.links.length; i--;) {
            if (bCount[m2.atoms[b.links[i]].type]===undefined)
                bCount[m2.atoms[b.links[i]].type] = 1;
            else
                bCount[m2.atoms[b.links[i]].type]++;
        }
        for (t = aCount.length; t--;) {
            if (aCount[t]!=bCount[t]) {
                return false;
            }
        }
        for (t = bCount.length; t--;) {
            if (aCount[t]!=bCount[t]) {
                return false;
            }
        }
        a.visited = b.visited = true;
        // the 2 atoms have the same type and links, so now recurse through their links to check them
        for (i = a.links.length; i--;) {
            c = m1.atoms[a.links[i]];
            if (c.visited===undefined) {
                possible = false;
                for (j = b.links.length; j-- && !possible;) {
                    d = m2.atoms[b.links[j]];
                    if (d.visited===undefined) {
                        if (this.compAtomStruct(c, d, m1, m2, depth + 1)) {
                            possible = true;
                        }
                    }
                }
                if (!possible) {
                    a.visited = b.visited = undefined;
                    return false;
                }
            }
        }
        return true;
    };
    OU.activity.Molecules.prototype.varChange = function () {
        this.parent.calcModel();
        this.parent.render();
    };
    OU.activity.Molecules.prototype.renderInstructions = function () {
        if(!this.selectionLayer)
            return;
        var ctx = this.selectionLayer.context,bH=OU.controlHeight,self=this,
        buttonsWidth=bH * 5.25;

        this.selectionLayer.clear();

        if(this.solved) {
            if (this.selectionDone) {
                this.selectionDone.remove();
            }
                this.selectionDone = new OU.util.ControlButton({
                    txt:"Restart",
                    x:this.window.x + this.window.w - buttonsWidth,
                    y:bH * .5,
                    w:bH * 5,
                    h:bH,
                    layer:this.selectionLayer,
                    onClick:function () {
                        self.solved=false;
                        self.molecule.atoms=[];
                        self.validateCounts=0;
                        self.selectionDone.remove();
                        self.solveButton.remove();
                        self.solveButton=self.selectionDone=undefined;
                        self.initMolecule();
                    }
                });
            return;
        }

        if (this.selectionDone) {
            this.selectionDone.render({
                x:this.window.x + this.window.w - buttonsWidth,
                y:bH * .5,
                w:bH * 5,
                h:bH
            });
        }
        else {
            this.selectionDone = new OU.util.ControlButton({
                txt:"Check my answer",
                x:this.window.x + this.window.w - buttonsWidth,
                y:bH * .5,
                w:bH * 5,
                h:bH,
                layer:this.selectionLayer,
                onClick:function () {
                    self.validateSelection();
                }
            });
        }
        if (this.validateCounts>2) {
            buttonsWidth = buttonsWidth+bH*3.25;
            if (this.solveButton) {
                this.solveButton.render({
                    x:this.window.x + this.window.w - buttonsWidth,
                    y:bH * .5,
                    w:bH * 4,
                    h:bH
                });
            }
            else {
                this.solveButton = new OU.util.ControlButton({
                    txt:"Show Answer",
                    x:this.window.x + this.window.w - buttonsWidth,
                    y:bH * .5,
                    w:bH * 4,
                    h:bH,
                    layer:this.selectionLayer,
                    onClick:function () {
                        self.solve();
                    }
                });
            }
        }

        this.selectionText = new OU.util.DynText({
            context:ctx,
            align:'left',
            x:this.window.x,
            y:0,
            w:this.window.w - buttonsWidth,
            h:bH * 2,
            txt:this.molecule.selection.instruction
        });

    };
    OU.activity.Molecules.prototype.isHit = function ( x, y, pressed ) {
        var i, j, s, mol = this.molecule, a, ax, ay, radius, dx, dy, d, damper = 0.25, numAtoms = this.zOrder.length;
        if (pressed) {
            y = -(this.h - OU.controlHeight) + y;
            this.zOrder.sort(function ( a, b ) {
                return b.z - a.z;
            });
            this.renderInstructions();
            for (i = numAtoms; i--;) {
                a = mol.atoms[this.zOrder[i].idx];
                if (a.pt===undefined)
                    a.pt = new this.Point(a.x, a.y, a.z, this);
                ax = a.pt.in2d.x,
                ay = a.pt.in2d.y,
                radius = this.ballRadius(a);
                dx = ax - x;
                dy = ay - y;
                d = Math.sqrt(dx * dx + dy * dy);
                if (d <= radius) {
                    if (this.editable) {
                        if (this.addBonds) {
                            s = -1;
                            for (j = numAtoms; j--;) {
                                if (this.molecule.atoms[j].selected)
                                    s = j;
                            }
                            if (s!=-1) {
                                mol.addLink(this.zOrder[i].idx, s);
                                mol.clearSelected();
                            }
                            else {
                                a.selected = true;
                            }
                            this.render();
                            return;
                        }
                        if (this.deleteAtoms) {
                            mol.removeAtom(this.zOrder[i].idx);
                            this.render();
                        }
                        else if (this.measureOn) {
                            a.selected = !a.selected;
                            i = 0;
                            this.render();
                        }
                        else {
                            this.movingAtom = a;
                        }
                    }
                    else {
                        a.selected = !a.selected;
                        i = 0;
                        this.render();
                    }
                    return;
                }
            }
            if (this.editable) {
                if (this.midDrag && this.movingAtom!==undefined) {
                    this.movingAtom.pt.upd2Dto3D(x - this.dragStart.x, y - this.dragStart.y);
                    this.render();
                }
            }
            if (this.editable && this.editList.isHit(x, y)) {
                this.addBonds = false;
                this.deleteAtoms = false;
                this.measureOn = false;
                if (this.bondsButton)
                    this.bondsButton.state(this.addBonds);
                if (this.deleteButton)
                    this.deleteButton.state(this.deleteAtoms);
                if (this.measureButton)
                    this.measureButton.state(this.measureOn);
                this.render();
                return;
            }
            if (this.rotatable) {
                // touch is in white space so rotate
                if (this.midDrag && this.movingAtom===undefined) {
                    this.spinning = false;
                    this.countToSpin = 0;
                    this.rotation.y = this.rotation.y - (x - this.dragStart.x) * damper;
                    this.rotation.x = this.rotation.x - (y - this.dragStart.y) * damper;
                    var r2d = Math.PI / 180;
                    this.rotCX = Math.cos(this.rotation.x * r2d);
                    this.rotSX = Math.sin(this.rotation.x * r2d);
                    this.rotCY = Math.cos(this.rotation.y * r2d);
                    this.rotSY = Math.sin(this.rotation.y * r2d);
                    this.render();
                // this.renderPitchYaw();
                }
            }
            this.dragStart = {
                'x':x,
                'y':y
            };
            this.midDrag = true;
        }
        else {
            this.midDrag = false;
            this.movingAtom = undefined;
        }
    };

    OU.activity.Molecules.prototype.keyPress = function ( k ) {

        var dX=0,dY=0;
        switch(k){
            case KEY_UP:
                dY=-10;
                break;
            case KEY_DOWN:
                dY=10;
                break;
            case KEY_LEFT:
                dX=-10;
                break;
            case KEY_RIGHT:
                dX=10;
                break;
        }
        this.rotation.y = this.rotation.y - (dX);
        this.rotation.x = this.rotation.x - (dY);

        this.spinning = false;
        this.countToSpin = 0;

        var r2d = Math.PI / 180;
        this.rotCX = Math.cos(this.rotation.x * r2d);
        this.rotSX = Math.sin(this.rotation.x * r2d);
        this.rotCY = Math.cos(this.rotation.y * r2d);
        this.rotSY = Math.sin(this.rotation.y * r2d);
        this.render();

    };
    OU.activity.Molecules.prototype.animateAtoms = function () {
        var i, j, a, mol = this.molecule, atomArray;
        if (mol.animations) {
            for (i = mol.animations.length; i--;) {
                a = mol.animations[i];
                if (a.inMotion) {
                    this.requireRender = true;
                    atomArray = [];
                    for (j = a.atoms.length; j--;) {
                        atomArray.push(mol.atoms[a.atoms[j] - 1]);
                    }
                    this.rotateAboutAnyAxis({
                        points:atomArray,
                        axisP1:a.axisP1,
                        axisP2:a.axisP2
                    });
                }
            }
        }
    };
    OU.activity.Molecules.prototype.rotateAboutAnyAxis = function ( p ) {
        // Rotates an array of points around an axis defined by 2 points P1 & P2
        var i, pt, points = p.points, o, x, y, z,
        p1 = p.axisP1,
        p2 = p.axisP2,
        theta = p.angle || Math.PI / 36, // default to 5 degree
        dx = p2.x - p1.x,
        dy = p2.y - p1.y,
        dz = p2.z - p1.z,
        vlength = Math.sqrt(dx * dx + dy * dy + dz * dz), // vectore length
        u = {
            x:dx / vlength,
            y:dy / vlength,
            z:dz / vlength
        }
        , // unit vector
        c = Math.cos(theta),
        s = Math.sin(theta),
        t = 1 - c;
        for (i = points.length; i--;) {
            pt = points[i];
            o = {
                x:pt.x - p1.x,
                y:pt.y - p1.y,
                z:pt.z - p1.z
            };
            x = (t * (u.x * u.x) + c) * o.x + (t * (u.x * u.y) - s * u.z) * o.y + (t * (u.x * u.z) + s * u.y) * o.z;
            y = (t * (u.x * u.y) + s * u.z) * o.x + (t * (u.y * u.y) + c) * o.y + (t * (u.y * u.z) - s * u.x) * o.z;
            z = (t * (u.x * u.z) - s * u.y) * o.x + (t * (u.y * u.z) + s * u.x) * o.y + (t * (u.z * u.z) + c) * o.z;
            pt.x = x + p1.x;
            pt.y = y + p1.y;
            pt.z = z + p1.z;
            pt.pt = undefined;
        // delete previous pt calculation to force new calc in render
        }
    };
    OU.activity.Molecules.prototype.spin = function () {
        var r2d = Math.PI / 180, i = this.inertia, maxVelocity = 3;
        if (this.spinOn && this.spinning) {
            i.x += Math.random() - 0.5;
            i.y += Math.random() - 0.5;
            i.x = i.x > maxVelocity ? maxVelocity : (i.x < -maxVelocity ? -maxVelocity : i.x);
            i.y = i.y > maxVelocity ? maxVelocity : (i.y < -maxVelocity ? -maxVelocity : i.y);
            this.rotation.y += i.y;
            this.rotation.x += i.x;
            this.rotCX = Math.cos(this.rotation.x * r2d);
            this.rotSX = Math.sin(this.rotation.x * r2d);
            this.rotCY = Math.cos(this.rotation.y * r2d);
            this.rotSY = Math.sin(this.rotation.y * r2d);
            this.render();
        // this.renderPitchYaw();
        }
        else if (this.countToSpin++ > 100) {
            this.spinning = true;
        }
        else {
            this.inertia = {
                x:0,
                y:0
            };
        }
    };
    OU.activity.Molecules.prototype.resetView = function () {
        var r2d = Math.PI / 180, tooSmall = true,
        numAtoms = this.molecule ? this.molecule.atoms.length : 0,
        self = this, rr;
        this.rotation = {
            'x':330,
            'y':30
        };
        this.rotCX = Math.cos(this.rotation.x * r2d);
        this.rotSX = Math.sin(this.rotation.x * r2d);
        this.rotCY = Math.cos(this.rotation.y * r2d);
        this.rotSY = Math.sin(this.rotation.y * r2d);
        if (this.molecule)
            this.molecule.clearSelected();
        rr = this.renderRange;
        this.render();
        // this.renderPitchYaw();
        if (numAtoms > 0 && (rr.minX < this.window.w * .25 || rr.maxX > this.window.w * .75
            && rr.minY < this.window.h * .25 || rr.maxY > this.window.h * .75 )) {
            tooSmall = false;
        }
        if (rr.radius > this.w / 35 || rr.radius > this.h / 35) {
            tooSmall = false;
        }
        if (tooSmall || rr.radius==0) {
            this.zoom += 0.05;
            setTimeout(function () {
                self.resetView()
            }
            , 40);
        }
        if (this.zoomSlider)
            this.zoomSlider.render(this.zoom);
    };
    OU.activity.Molecules.prototype.zoomIn = function () {
        this.zoom += 0.2;
        this.render();
    };
    OU.activity.Molecules.prototype.zoomOut = function () {
        this.zoom -= 0.2;
        this.render();
    };
    OU.activity.Molecules.prototype.pinch = function ( e, s, x, y, dx, dy, me ) {
        var nz = ((e - s) / me.h) + me.zoom;
        nz = nz > 1 ? 1 : (nz < 0 ? 0 : nz);
        me.zoom = nz;
        me.setZoom(nz, me);
        me.render();
    };
    OU.activity.Molecules.prototype.setZoom = function ( zoom, graph ) {
        graph.zoom = zoom;
        if (graph.zoomSlider)
            graph.zoomSlider.render(zoom);
        graph.render();
    };
    OU.activity.Molecules.prototype.renderLoop = function () {
        var self = this;
        if (this.molecule) {
            this.animateAtoms();
            this.spin();
            if (this.requireRender)
                this.performRender();
            setTimeout(function () {
                self.renderLoop();
            }, 40);
        }
    };
    OU.activity.Molecules.prototype.render = function () {
        this.requireRender = true;
    };
    OU.activity.Molecules.prototype.performRender = function () {
        var bH = OU.controlHeight, ctx = this.modelLayer.context;
        ctx.clearRect(0, 0, this.w, -this.h - bH);
        // clear background
        if (this.molecule.atoms===undefined)
            return;
        // establish the offset to lock the view central
        this.viewXOffset = 0;
        this.viewYOffset = 0;
        var topPt = new this.Point(
            this.molecule.x1 + (this.molecule.x2 - this.molecule.x1) / 2,
            this.molecule.y1 + (this.molecule.y2 - this.molecule.y1) / 2,
            this.molecule.z1 + (this.molecule.z2 - this.molecule.z1) / 2,
            this);
        this.viewXOffset = this.window.x + (this.window.w / 2) - topPt.in2d.x;
        this.viewYOffset = ( -this.window.h / 2) - topPt.in2d.y;
        this.requireRender = false;
        this.renderMolecule();
        this.renderControls();
    };
    OU.activity.Molecules.prototype.renderControls = function () {
        var buttons = this.data.settings.showButtons, i;
        if (!this.molecule)
            return;
        this.controlsLayer.clear();
        if (buttons.measure)
            this.measureButton.render();
        if (buttons.symbols)
            this.namesButton.render();
        if (buttons.spin)
            this.spinButton.render();
        if (buttons.wireframe)
            this.wireframeButton.render();
        if (buttons.spaceFill)
            this.spaceFillButton.render();
        if (this.animateButtons) {
            for (i = this.animateButtons.length; i--;)
                this.animateButtons[i].render();
        }
        if (this.infoButton)
            this.infoButton.render();
        if (this.zoomSlider)
            this.zoomSlider.render();
        if (this.editable) {
            this.bondsButton.render();
            this.deleteButton.render();
        }
    };
    OU.activity.Molecules.prototype.initControls = function () {
        var self = this, ctx = this.controlsLayer.context, bH = OU.controlHeight, buttons, bX,
        mol = this.molecule, i, a,
        clickable = this.controlsLayer.events.clickable;
        buttons = this.data.settings.showButtons;
        bX = this.w - bH * 3;
        if(this.measureButton)
            this.measureButton.remove();
        if(this.namesButton)
            this.namesButton.remove();
        if(this.wireframeButton)
            this.wireframeButton.remove();
        if(this.spaceFillButton)
            this.spaceFillButton.remove();
        if(this.spinButton)
            this.spinButton.remove();
        if(this.animateButtons!==undefined) {
            console.log('removing animation buttons');
            for(i=this.animateButtons.length; i--;)
                this.animateButtons[i].remove();
            this.animateButtons.length=0;
        }
        if(this.bondsButton)
            this.bondsButton.remove();
        if(this.deleteButton)
            this.deleteButton.remove();
        if(this.zoomSlider)
            this.zoomSlider.remove();
        if(this.infoButton)
            this.infoButton.remove();

        if (buttons.measure) {
            this.measureButton = new OU.util.CheckBoxButton({
                txt:'Measure',
                tabIndex: 55,
                x:bX,
                y:0,
                w:bH * 3,
                h:bH,
                layer:this.controlsLayer,
                state:this.measureOn,
                onClick:function () {
                    self.measureOn = !self.measureOn;
                    self.measureButton.state(self.measureOn);
                    if (self.editable) {
                        self.deleteAtoms = false;
                        self.addBonds = false;
                        if (self.deleteButton)
                            self.deleteButton.state(self.deleteAtoms);
                        if (self.bondsButton)
                            self.bondsButton.state(self.addBonds);
                    }
                    self.molecule.clearSelected();
                    self.render();
                }
            });
            bX -= bH * 3;
        }
        if (buttons.symbols) {
            this.namesButton = new OU.util.CheckBoxButton({
                txt:'Symbols',
                tabIndex: 54,
                x:bX,
                y:0,
                w:bH * 3,
                h:bH,
                layer:this.controlsLayer,
                state:this.namesOn,
                onClick:function () {
                    self.toggleNames();
                }
            });
            bX -= bH * 3;
        }
        if (buttons.wireframe) {
            bX -= bH * 0.5;
            this.wireframeButton = new OU.util.CheckBoxButton({
                txt:'Wireframe',
                tabIndex: 53,
                x:bX,
                y:0,
                w:bH * 3.5,
                h:bH,
                layer:this.controlsLayer,
                state:this.wireframeOn,
                onClick:function () {
                    self.toggleWireframe();
                }
            });
            bX -= bH * 3;
        }
        if (buttons.spaceFill) {
            this.spaceFillButton = new OU.util.CheckBoxButton({
                txt:'SpaceFill',
                tabIndex: 52,
                x:bX,
                y:0,
                w:bH * 3,
                h:bH,
                layer:this.controlsLayer,
                state:this.spaceFillOn,
                onClick:function () {
                    self.toggleSpaceFill();
                }
            });
            bX -= bH * 3;
        }
        if (buttons.spin) {
            this.spinButton = new OU.util.CheckBoxButton({
                txt:'Spin',
                tabIndex: 51,
                x:bX,
                y:0,
                w:bH * 3,
                h:bH,
                layer:this.controlsLayer,
                state:this.spinOn,
                onClick:function () {
                    self.toggleSpin();
                }
            });
            bX -= bH * 3;
        }
        if (mol.animations) {
            this.animateButtons = [];
            console.log('Adding Animation buttons');
            for (i = mol.animations.length; i--;) {
                a = mol.animations[i];
                this.animateButtons[i] = new OU.util.CheckBoxButton({
                    txt:a.buttonLabel || 'Animate',
                    tabIndex: 40+i,
                    x:bX,
                    y:0,
                    w:bH * 3,
                    h:bH,
                    layer:this.controlsLayer,
                    state:this.atomicAnimation,
                    onClick:function ( p ) {
                        p.a.inMotion = !p.a.inMotion;
                        self.animateButtons[p.i].state(p.a.inMotion);
                        self.render();
                    },
                    onClickParam:{
                        a:a,
                        i:i
                    }
                });
                bX -= bH * 3;
            }
        }
        if (this.editable) {
            this.bondsButton = new OU.util.CheckBoxButton({
                txt:'Bonds',
                tabIndex: 39,
                x:bX,
                y:0,
                w:bH * 3,
                h:bH,
                layer:this.controlsLayer,
                state:false,
                onClick:function () {
                    self.addBonds = !self.addBonds;
                    self.deleteAtoms = false;
                    self.measureOn = false;
                    self.bondsButton.state(self.addBonds);
                    if (self.deleteButton)
                        self.deleteButton.state(self.deleteAtoms);
                    if (self.measureButton)
                        self.measureButton.state(self.measureOn);
                    self.molecule.clearSelected();
                    self.render();
                }
            });
            bX -= bH * 3;
            this.deleteButton = new OU.util.CheckBoxButton({
                txt:'Delete',
                tabIndex: 38,
                x:bX,
                y:0,
                w:bH * 3,
                h:bH,
                layer:this.controlsLayer,
                state:false,
                onClick:function () {
                    self.deleteAtoms = !self.deleteAtoms;
                    self.addBonds = false;
                    self.measureOn = false;
                    self.deleteButton.state(self.deleteAtoms);
                    if (self.bondsButton)
                        self.bondsButton.state(self.addBonds);
                    if (self.measureButton)
                        self.measureButton.state(self.measureOn);
                    self.render();
                }
            });
            bX -= bH * 3;
        }
        else {
            if (this.showZoomSlider) {
                this.zoomSlider = new OU.util.Slider({
                    container:this,
                    tabIndex: 38,
                    instance:'zoom' + this.instance,
                    x:0,
                    y:0,
                    w:bX + bH * 2,
                    h:bH - 2,
                    sliderHeight:(bH - 2) / 2,
                    drawContainer:true,
                    callback:this.setZoom,
                    callbackParam:this,
                    frames:4,
                    background:{
                        color:'#e0e0e0'
                    },
                    context:ctx
                });
                clickable.push(this.zoomSlider);
            }
            bX -= bH * 3;
        }
        if (this.data.infoHtml && this.data.infoHtml!='') {
            this.infoButton = new OU.util.InfoButton({
                layer:this.controlsLayer,
                tabIndex: 37,
                x:bX,
                y:0,
                onClick:function () {
                    self.showInfo();
                }
            });
        }
        this.bgLabels();
    };
    OU.activity.Molecules.prototype.showInfo = function () {
        var self = this;
        if (this.infoDesp)
            return;
        this.infoDisp = true;
        new OU.util.PopUpInfo({
            container:this,
            txt:this.data.infoHtml,
            x:this.w * .01,
            y:this.h * .01,
            w:this.w * .75,
            h:this.h * .7,
            onClose:function () {
                self.infoDisp = false;
            }
        });
    };
    OU.activity.Molecules.prototype.renderAxisGuide = function () {
        var ctx = this.modelLayer.context, mol = this.molecule;
        ctx.save();
        ctx.beginPath();
        ctx.strokeStyle = "#666";
        ctx.fillStyle = "#444";
        ctx.globalAlpha = 0.3;
        ctx.lineWidth = 1;
        // Axis
        this.moveTo(0, mol.y1, 0);
        this.lineTo(0, mol.y2, 0);
        this.moveTo(mol.x1, 0, 0);
        this.lineTo(mol.x2, 0, 0);
        this.moveTo(0, 0, mol.z1);
        this.lineTo(0, 0, mol.z2);
        ctx.stroke();
        this.text({
            txt:'X',
            x:mol.x2 * 1.1,
            y:0,
            z:0,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        this.text({
            txt:'Y',
            x:0,
            y:mol.y2 * 1.1,
            z:0,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        this.text({
            txt:'Z',
            x:0,
            y:0,
            z:mol.z2 * 1.1,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        ctx.restore();
    };
    OU.activity.Molecules.prototype.renderMolecule = function () {
        this.renderRange = {
            minX:this.window.w,
            maxX:0,
            minY:this.window.h,
            maxY:0,
            radius:0
        };
        if (this.spaceFillOn) {
            this.renderSpaceFill();
        }
        else if (this.wireframeOn) {
            this.renderWireframe();
        }
        else {
            this.renderBallStick();
        }
    };
    OU.activity.Molecules.prototype.renderSpaceFill = function () {
        var i, mol = this.molecule, a, a3, r, x, y, z, sp, s2, atom,
        sff = this.config.spaceFillFactor,
        xStep, yStep, xp, bestX1, bestX2, lastX1, lastX2, x2, s, lastStyle, res,
        minX = 1000,
        maxX = -1000,
        minY = 1000,
        maxY = -1000,
        ctx = this.modelLayer.context, f, x1, y1, zd;
        //        debugTimer.start();
        for (i = mol.atoms.length; i--;) {
            // determine the X / Y range of the atoms
            atom = mol.atoms[i];
            if (atom.pt===undefined)
                atom.pt = new this.Point(atom.x, atom.y, atom.z, this);
            else
                atom.pt.calc();
            a3 = atom.pt.in3d;
            r = atom.r * sff;
            if (a3.x - r < minX)
                minX = a3.x - r;
            if (a3.x + r > maxX)
                maxX = a3.x + r;
            if (a3.y - r < minY)
                minY = a3.y - r;
            if (a3.y + r > maxY)
                maxY = a3.y + r;
        }
        res = 250;
        // Resolution - higher (300) = high - res + slow : lower (50) = low - res + fast
        if (this.atomicAnimation || this.spinOn)
            res = 75;
        xStep = (maxX - minX) / res;
        yStep = (maxY - minY) / res;
        ctx.beginPath();
        for (x = minX; x < maxX; x = x + xStep) {
            // Step through every X / Y point and find the nearest atomic surface + render in 2d on canvase
            xp = x + xStep;
            lastX1 = undefined;
            lastX2 = undefined;
            for (y = minY; y < maxY; y = y + yStep) {
                bestX1 = undefined;
                bestX2 = undefined;
                for (i = mol.atoms.length; i--;) {
                    a = mol.atoms[i];
                    a3 = a.pt.in3d;
                    x1 = a3.x - x;
                    x2 = a3.x - xp;
                    y1 = a3.y - y;
                    r = a.r * sff;
                    if (r * r >= x1 * x1 + y1 * y1) {
                        // atom is within XY range of current ray
                        zd = Math.sqrt(r * r - x1 * x1 - y1 * y1);
                        z = a3.z - zd;
                        if (bestX1===undefined || z < bestX1.z) {
                            sp = new this.Point(x, y, z, this, true);
                            s2 = sp.in2d;
                            bestX1 = {
                                x:s2.x | 0,
                                y:s2.y | 0,
                                t:a.type,
                                f:zd / r - 0.5,
                                z:z
                            };
                        }
                    }
                    if (r * r >= x2 * x2 + y1 * y1) {
                        // atom is within XY range of current ray step
                        zd = Math.sqrt(r * r - x2 * x2 - y1 * y1);
                        z = a3.z - zd;
                        if (bestX2===undefined || z < bestX2.z) {
                            sp = new this.Point(xp, y, z, this, true);
                            s2 = sp.in2d;
                            bestX2 = {
                                x:s2.x | 0,
                                y:s2.y | 0,
                                t:a.type,
                                f:zd / r - 0.5,
                                z:z
                            };
                        }
                    }
                }
                if (bestX1===undefined)
                    bestX1 = bestX2;
                else if (bestX2===undefined)
                    bestX2 = bestX1;
                if (bestX1!==undefined && lastX1!==undefined) {
                    a = OU.atoms[lastX1.t];
                    f = lastX1.f > 0 ? lastX1.f * 128 : 0;
                    s = (a.r + f | 0) + ',' + (a.g + f | 0) + ',' + (a.b + f | 0);
                    if (s!=lastStyle) {
                        ctx.fill();
                        ctx.beginPath();
                        ctx.fillStyle = 'rgba(' + s + ',1)';
                        lastStyle = s;
                    }
                    ctx.moveTo(lastX1.x, lastX1.y);
                    ctx.lineTo(lastX2.x, lastX2.y);
                    ctx.lineTo(bestX2.x, bestX2.y - 1);
                    ctx.lineTo(bestX1.x - 1, bestX1.y - 1);
                }
                lastX1 = bestX1;
                lastX2 = bestX2;
            }
        }
        ctx.fill();
    //        debugTimer.save('render');
    //        debugTimer.report();
    };
    OU.activity.Molecules.prototype.renderWireframe = function () {
        var ctx = this.modelLayer.context, r = 0, i, a, j, b, b2, p, t, xd, yd, br,
        a2, a3, c = Math.PI * 2, stickGradient,
        mol = this.molecule,
        z, rr = this.renderRange, aType, f;
        this.zOrder = [];
        ctx.lineWidth = 0.5;
        ctx.strokeStyle = '#999';
        rr.minZ = 1000;
        rr.maxZ = -1000;
        for (i = mol.atoms.length; i--;) {
            a = mol.atoms[i];
            if (a.pt===undefined)
                a.pt = new this.Point(a.x, a.y, a.z, this);
            else
                a.pt.calc();
            z = a.pt.in3d.z;
            this.zOrder.push({
                idx:i,
                z:z
            });
            if (a.links===undefined)
                a.links = [];
            if (a.duallinks===undefined)
                a.duallinks = [];
            if (z > rr.maxZ)
                rr.maxZ = z;
            if (z < rr.minZ)
                rr.minZ = z;
            br = this.ballRadius(a);
            if (br > r)
                r = br;
        }
        rr.radius = r;
        this.zOrder.sort(function ( a, b ) {
            return a.z - b.z;
        });
        ctx.lineCap = 'round';
        for (i = this.zOrder.length; i--;) {
            a = mol.atoms[this.zOrder[i].idx];
            a2 = a.pt.in2d;
            a3 = a.pt.in3d;
            for (j = a.links.length; j--;) {
                b = mol.atoms[a.links[j]];
                b2 = b.pt.in2d;
                if (b.pt.in3d.z <= a3.z) {
                    ctx.beginPath();
                    stickGradient = ctx.createLinearGradient(a2.x, a2.y, b2.x, b2.y);
                    p = a.r / (a.r + b.r);
                    aType = OU.atoms[a.type];
                    f = (128 * (a.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                    stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    aType = OU.atoms[b.type];
                    f = (128 * (b.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                    stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    ctx.strokeStyle = stickGradient;
                    ctx.moveTo(a2.x, a2.y);
                    ctx.lineTo(b2.x, b2.y);
                    ctx.lineWidth = r / 3;
                    ctx.stroke();
                    if (Math.abs(a2.x - a2.x) < r * .99) {
                        ctx.beginPath();
                        ctx.fillStyle = stickGradient;
                        ctx.arc(a2.x, a2.y, r / 6, 0, c, false);
                        ctx.fill();
                    }
                }
            }
            for (j = a.duallinks.length; j--;) {
                b = mol.atoms[a.duallinks[j]];
                b2 = b.pt.in2d;
                if (b.pt.in3d.z <= a3.z) {
                    t = b2.x==a2.x ? 0 : ((Math.PI / 2) - Math.atan((b2.y - a2.y) / (b2.x - a2.x)));
                    xd = -Math.cos(t) / 4;
                    yd = Math.sin(t) / 4;
                    ctx.beginPath();
                    stickGradient = ctx.createLinearGradient(a2.x, a2.y, b2.x, b2.y);
                    p = a.r / (a.r + b.r);
                    aType = OU.atoms[a.type];
                    f = (128 * (a.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                    stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    aType = OU.atoms[b.type];
                    f = (128 * (b.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                    stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    ctx.strokeStyle = stickGradient;
                    ctx.moveTo(a2.x, a2.y);
                    ctx.bezierCurveTo(a2.x - xd * r * 2,
                        a2.y - yd * r * 2,
                        b2.x - xd * r * 2,
                        b2.y - yd * r * 2,
                        b2.x,
                        b2.y);
                    ctx.moveTo(a2.x, a2.y);
                    ctx.bezierCurveTo(a2.x + xd * r * 2, a2.y + yd * r * 2, b2.x + xd * r * 2, b2.y + yd * r * 2, b2.x, b2.y);
                    ctx.lineWidth = r / 3;
                    ctx.stroke();
                }
            }
        }
    };
    OU.activity.Molecules.prototype.renderBallStick = function () {
        var ctx = this.modelLayer.context, r, r1, r2, i, a, j, b, b2, bO, p, t, xd, yd, spb, spb2,
        aO, a2, a3, d2, dX, dY, dZ, d, sp, sp2, sdX, sdY, sdZ, c = Math.PI * 2, stickGradient, numSelected = 0,
        mol = this.molecule, r2d = 180 / Math.PI,
        abD, acD, dotP, dxB, dyB, dzB, b3, c3, z, rr = this.renderRange, aType, f;
        this.zOrder = [];
        ctx.lineWidth = 0.5;
        ctx.strokeStyle = '#999';
        //        debugTimer.start();
        // atoms
        rr.minZ = 1000;
        rr.maxZ = -1000;
        for (i = mol.atoms.length; i--;) {
            a = mol.atoms[i];
            if (a.pt===undefined)
                a.pt = new this.Point(a.x, a.y, a.z, this);
            else
                a.pt.calc();
            z = a.pt.in3d.z;
            this.zOrder.push({
                idx:i,
                z:z
            });
            if (a.links===undefined)
                a.links = [];
            if (a.duallinks===undefined)
                a.duallinks = [];
            if (z > rr.maxZ)
                rr.maxZ = z;
            if (z < rr.minZ)
                rr.minZ = z;
        }
        this.zOrder.sort(function ( a, b ) {
            return a.z - b.z;
        });
        for (i = this.zOrder.length; i--;) {
            a = mol.atoms[this.zOrder[i].idx];
            aO = a.pt.original;
            a2 = a.pt.in2d;
            a3 = a.pt.in3d;
            r1 = this.ball(a);
            for (j = a.links.length; j--;) {
                b = mol.atoms[a.links[j]];
                b2 = b.pt.in2d;
                bO = b.pt.original;
                if (b.pt.in3d.z <= a3.z) {
                    r2 = this.ballRadius(b);
                    r = r1 < r2 ? r1 : r2;
                    dX = bO.x - aO.x;
                    dY = bO.y - aO.y;
                    dZ = bO.z - aO.z;
                    d = Math.sqrt(dX * dX + dY * dY + dZ * dZ);
                    sdX = 0.9 * a.r * dX / d;
                    sdY = 0.9 * a.r * dY / d;
                    sdZ = 0.9 * a.r * dZ / d;
                    sp = new this.Point(aO.x + sdX, aO.y + sdY, aO.z + sdZ, this);
                    sp2 = sp.in2d;
                    ctx.beginPath();
                    stickGradient = ctx.createLinearGradient(a2.x, a2.y, b2.x, b2.y);
                    p = a.r / (a.r + b.r);
                    aType = OU.atoms[a.type];
                    f = (128 * (a.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                    stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    aType = OU.atoms[b.type];
                    f = (128 * (b.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                    stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    ctx.strokeStyle = stickGradient;
                    ctx.moveTo(sp2.x, sp2.y);
                    ctx.lineTo(b2.x, b2.y);
                    ctx.lineWidth = r / 3;
                    ctx.stroke();
                    if (Math.abs(sp2.x - a2.x) < r1 * .99) {
                        ctx.beginPath();
                        ctx.fillStyle = stickGradient;
                        ctx.arc(sp2.x, sp2.y, r / 6, 0, c, false);
                        ctx.fill();
                    }
                }
            }
            for (j = a.duallinks.length; j--;) {
                b = mol.atoms[a.duallinks[j]];
                b2 = b.pt.in2d;
                bO = b.pt.original;
                if (b.pt.in3d.z <= a3.z) {
                    r2 = this.ballRadius(b);
                    r = r1 < r2 ? r1 : r2;
                    dX = bO.x - aO.x;
                    dY = bO.y - aO.y;
                    dZ = bO.z - aO.z;
                    d = Math.sqrt(dX * dX + dY * dY + dZ * dZ);
                    sdX = 0.8 * a.r * dX / d;
                    sdY = 0.8 * a.r * dY / d;
                    sdZ = 0.8 * a.r * dZ / d;
                    sp = new this.Point(aO.x + sdX, aO.y + sdY, aO.z + sdZ, this);
                    sp2 = sp.in2d;
                    dX = aO.x - bO.x;
                    dY = aO.y - bO.y;
                    dZ = aO.z - bO.z;
                    sdX = 0.8 * b.r * dX / d;
                    sdY = 0.8 * b.r * dY / d;
                    sdZ = 0.8 * b.r * dZ / d;
                    spb = new this.Point(bO.x + sdX, bO.y + sdY, bO.z + sdZ, this);
                    spb2 = spb.in2d;
                    t = b2.x==sp2.x ? 0 : ((Math.PI / 2) - Math.atan((b2.y - sp2.y) / (b2.x - sp2.x)));
                    xd = -Math.cos(t) / 4;
                    yd = Math.sin(t) / 4;
                    ctx.beginPath();
                    stickGradient = ctx.createLinearGradient(a2.x, a2.y, b2.x, b2.y);
                    p = a.r / (a.r + b.r);
                    aType = OU.atoms[a.type];
                    f = (128 * (a.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                    stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    aType = OU.atoms[b.type];
                    f = (128 * (b.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                    stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    ctx.strokeStyle = stickGradient;
                    ctx.moveTo(sp2.x - xd * r1, sp2.y - yd * r1);
                    ctx.bezierCurveTo(sp2.x - xd * r1 * 2,
                        sp2.y - yd * r1 * 2,
                        spb2.x - xd * r2 * 2,
                        spb2.y - yd * r2 * 2,
                        spb2.x - xd * r2,
                        spb2.y - yd * r2);
                    ctx.moveTo(sp2.x + xd * r1, sp2.y + yd * r1);
                    ctx.bezierCurveTo(sp2.x + xd * r1 * 2, sp2.y + yd * r1 * 2, spb2.x + xd * r2 * 2, spb2.y + yd * r2 * 2, spb2.x + xd * r2, spb2.y + yd * r2);
                    ctx.lineWidth = r / 3;
                    ctx.stroke();
                    if (Math.abs(sp2.x - a2.x) < r1 * .99) {
                        ctx.beginPath();
                        ctx.fillStyle = stickGradient;
                        ctx.arc(sp2.x - xd * r1, sp2.y - yd * r1, r / 6, 0, c, false);
                        ctx.arc(sp2.x + xd * r1, sp2.y + yd * r1, r / 6, 0, c, false);
                        ctx.fill();
                    }
                }
            }
        }
        if (this.editable)
            this.editList.render();
        if (this.measureOn) {
            for (i = this.zOrder.length; i--;) {
                a = mol.atoms[this.zOrder[i].idx];
                a2 = a.pt.in2d;
                if (a.selected) {
                    numSelected++;
                    if (i > 0) {
                        for (j = i; j--;) {
                            b = mol.atoms[this.zOrder[j].idx];
                            b2 = b.pt.in2d;
                            if (b.selected) {
                                dX = b.pt.original.x - a.pt.original.x;
                                dY = b.pt.original.y - a.pt.original.y;
                                dZ = b.pt.original.z - a.pt.original.z;
                                d = Math.sqrt(dX * dX + dY * dY + dZ * dZ);
                                ctx.save();
                                ctx.beginPath();
                                ctx.moveTo(a2.x, a2.y);
                                ctx.lineTo(b2.x, b2.y);
                                ctx.lineWidth = 1;
                                ctx.strokeStyle = '#c00';
                                ctx.stroke();
                                dX = b2.x - a2.x;
                                dY = b2.y - a2.y;
                                d2 = Math.sqrt(dX * dX + dY * dY);
                                if (dX==0)
                                    t = Math.PI / 2;
                                else
                                    t = Math.atan(dY / dX);
                                if (a2.x < b2.x)
                                    ctx.translate(a2.x, a2.y);
                                else
                                    ctx.translate(b2.x, b2.y);
                                ctx.rotate(t);
                                ctx.fillStyle = 'rgba(255,255,255,0.7)';
                                ctx.fillRect(d2 / 4, -20, d2 / 2, 20);
                                ctx.fillStyle = '#c00';
                                ctx.font = 'bold 12px ' + OU.theme.font;
                                ctx.fillText((d * 1000 | 0) / 1000 + 'nm', d2 / 2, -10);
                                ctx.restore();
                            }
                        }
                    }
                }
            }
            ctx.save();
            ctx.font = 'bold 12px ' + OU.theme.font;
            if (numSelected==3) {
                for (i = mol.atoms.length; i--;) {
                    a = mol.atoms[i];
                    a2 = a.pt.in2d;
                    a3 = a.pt.original;
                    if (a.selected) {
                        for (j = mol.atoms.length; j--;) {
                            b = mol.atoms[j];
                            if (b.selected && j!=i) {
                                b3 = b.pt.original;
                                break;
                            }
                        }
                        for (j = j; j--;) {
                            c = mol.atoms[j];
                            if (c.selected & j!=i) {
                                c3 = c.pt.original;
                                j = 0;
                            }
                        }
                        dX = b3.x - a3.x;
                        dY = b3.y - a3.y;
                        dZ = b3.z - a3.z;
                        abD = Math.sqrt(dX * dX + dY * dY + dZ * dZ);
                        dxB = dX / abD;
                        dyB = dY / abD;
                        dzB = dZ / abD;
                        dX = c3.x - a3.x;
                        dY = c3.y - a3.y;
                        dZ = c3.z - a3.z;
                        acD = Math.sqrt(dX * dX + dY * dY + dZ * dZ);
                        dX = dX / acD;
                        dY = dY / acD;
                        dZ = dZ / acD;
                        dotP = (dxB * dX) + (dyB * dY) + (dzB * dZ);
                        t = (Math.acos(dotP > 1 ? 1 : dotP) * r2d * 100 | 0) / 100;
                        ctx.fillStyle = 'rgba(255,255,255,0.7)';
                        ctx.fillRect(a2.x - 40, a2.y - 30, 80, 20);
                        ctx.fillStyle = '#444';
                        ctx.fillText(t + 'Â°', a2.x, a2.y - 20);
                    }
                }
            }
            ctx.restore();
        }
    //        debugTimer.save('render');
    //        debugTimer.report();
    };
    OU.activity.Molecules.prototype.EditList = function ( p ) {
        this.parent = p.parent;
        OU.activity.Molecules.prototype.EditList.prototype.init = function () {
            var i, parent = this.parent, es = parent.data.editElements, e, t;
            for (i = es.length; i--;) {
                e = es[i];
                t = parent.atomType(e.type);
                e.a = {
                    x:0,
                    y:0,
                    z:0,
                    type:t,
                    r:OU.atoms[t].radius * (parent.atomicRadiusFactor || 5),
                    selected:false
                };
            }
        };
        OU.activity.Molecules.prototype.EditList.prototype.isHit = function ( x, y ) {
            var i, parent = this.parent, es = parent.data.editElements, dx, dy, d, e, atom;
            for (i = es.length; i--;) {
                e = es[i];
                dx = e.x - x;
                dy = e.y - y;
                d = Math.sqrt(dx * dx + dy * dy);
                if (d < e.r) {
                    atom = {
                    };
                    atom.type = e.a.type;
                    atom.pt = new parent.Point(0, 0, 0, parent);
                    atom.pt.in2d.x = e.x;
                    atom.pt.in2d.y = e.y;
                    atom.pt.upd2Dto3D(0, 0);
                    atom.x = atom.pt.original.x;
                    atom.y = atom.pt.original.r;
                    atom.z = atom.pt.original.z;
                    atom.r = e.a.r;
                    atom.selected = false;
                    atom.links = [];
                    atom.duallinks = [];
                    this.parent.molecule.atoms.push(atom);
                    return true;
                }
            }
            return false;
        };
        OU.activity.Molecules.prototype.EditList.prototype.render = function () {
            var i, parent = this.parent,
            es = parent.data.editElements, e, r, x = parent.w * .90, y = -(parent.h - OU.controlHeight * 4);
            for (i = es.length; i--;) {
                e = es[i];
                e.a.pt = new parent.Point(0, 0, 0, parent);
                r = parent.ballRadius(e.a);
                y += 20 + r;
                e.r = parent.ball(e.a, {
                    x:x,
                    y:y
                });
                e.x = x;
                e.y = y;
                y += r;
            }
        };
        this.init();
    };
    OU.activity.Molecules.prototype.ballRadius = function ( a ) {
        var mol = this.molecule,
        a3 = a.pt.in3d,
        dz = a3.z - (mol.z1 - ((mol.dz) * 60 * (1 - this.zoom))),
        rz = this.window.h, cX, eX;
        cX = (a3.x * 15) / (dz) * rz + this.viewXOffset;
        eX = ((a3.x + a.r) * 15) / (dz) * rz + this.viewXOffset;
        return Math.abs(cX - eX);
    };
    OU.activity.Molecules.prototype.ball = function ( a, ov, s ) {
        var override = ov || {},
        gradient, c = Math.PI * 2, ctx = this.modelLayer.context,
        x = override.x || a.pt.in2d.x,
        y = override.y || a.pt.in2d.y,
        scale = s || 1,
        radius = this.ballRadius(a) / scale,
        aType, f,
        rr = this.renderRange;
        if (a.selected && (this.measureOn || this.molecule.selection!==undefined)) {
            ctx.beginPath();
            ctx.save();
            ctx.arc(x, y, radius * 1.2, 0, c, false);
            ctx.strokeStyle = '#c00';
            ctx.fillStyle = '#fff';
            ctx.lineWidth = 4;
            ctx.fill();
            ctx.stroke();
            ctx.restore();
        }
        if (a.highlighted) {
            ctx.beginPath();
            ctx.save();
            ctx.arc(x, y, radius * 1.2, 0, c, false);
            ctx.strokeStyle = '#0c0';
            ctx.fillStyle = '#fff';
            ctx.lineWidth = 4;
            ctx.fill();
            ctx.stroke();
            ctx.restore();
        }
        ctx.beginPath();
        ctx.arc(x, y, radius, 0, c, false);
        aType = OU.atoms[a.type];
        if (this.editable)
            f = 0;
        else
            f = (128 * (a.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
        ctx.fillStyle = 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)';
        ctx.fill();
        ctx.beginPath();
        ctx.arc(x, y, radius * .9, 0, c, false);
        gradient = ctx.createRadialGradient(x - radius / 2, y - radius / 2, 0, x, y, radius);
        gradient.addColorStop(0, 'rgba(255,255,255,0.5)');
        gradient.addColorStop(0.3, 'rgba(255,255,255,0.4)');
        gradient.addColorStop(0.8, 'rgba(255,255,255,0)');
        gradient.addColorStop(1, 'rgba(0,0,0,0)');
        ctx.fillStyle = gradient;
        ctx.fill();
        if (this.namesOn) {
            ctx.save();
            ctx.font = radius + 'px ' + OU.theme.font;
            ctx.fillStyle = OU.atoms[a.type].text;
            ctx.fillText(OU.atoms[a.type].name, x, y);
            ctx.restore();
        }
        if (radius > this.renderRange.radius)
            this.renderRange.radius = radius;
        return radius * scale;
    };
    OU.activity.Molecules.prototype.toggleNames = function () {
        this.namesOn = !this.namesOn;
        this.namesButton.state(this.namesOn);
        this.render();
    };
    OU.activity.Molecules.prototype.toggleSpaceFill = function () {
        this.spaceFillOn = !this.spaceFillOn;
        this.spaceFillButton.state(this.spaceFillOn);
        if (this.wireframeButton!==undefined) {
            this.wireframeOn = false;
            this.wireframeButton.state(this.wireframeOn);
        }
        this.render();
    };
    OU.activity.Molecules.prototype.toggleWireframe = function () {
        this.wireframeOn = !this.wireframeOn;
        this.wireframeButton.state(this.wireframeOn);
        if (this.spaceFillButton!==undefined) {
            this.spaceFillOn = false;
            this.spaceFillButton.state(this.spaceFillOn);
        }
        this.render();
    };
    OU.activity.Molecules.prototype.toggleSpin = function () {
        this.spinOn = !this.spinOn;
        this.spinButton.state(this.spinOn);
        this.render();
    };
    OU.activity.Molecules.prototype.bgLabels = function () {
        var bH = OU.controlHeight,
        bgctx = this.bgLayer.context;
        bgctx.clearRect(0, this.h - bH, this.w, bH);
        bgctx.textAlign = 'right';
        bgctx.font = 'bold ' + ((bH / 2) | 0) + 'px ' + OU.theme.font;
        bgctx.fillStyle = '#666';
        bgctx.textAlign = 'center';
    };
    OU.activity.Molecules.prototype.renderPitchYaw = function () {
        if (this.editable)
            return;
        var yaw = Math.abs(this.rotation.y) % 360 | 0,
        pitch = Math.abs(this.rotation.x) % 360 | 0,
        bH = OU.controlHeight,
        ctx = this.pyLayer.context;
        ctx.textAlign = 'right';
        ctx.font = 'bold ' + ((bH * .5) | 0) + 'px ' + OU.theme.font;
        this.pyLayer.clear();
        ctx.fillStyle = '#AFDFE4';
        ctx.fillText(pitch + ' \u00B0', this.w * .85, this.h * 0.12);
        ctx.fillText(yaw + ' \u00B0', this.w * .95, this.h * 0.12);
    };
    OU.activity.Molecules.prototype.moveTo = function ( x, y, z ) {
        var pt = new this.Point(x, y, z, this);
        this.modelLayer.context.moveTo(pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Molecules.prototype.lineTo = function ( x, y, z ) {
        var pt = new this.Point(x, y, z, this);
        this.modelLayer.context.lineTo(pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Molecules.prototype.text = function ( params ) {
        var pt = new this.Point(params.x, params.y, params.z, this);
        this.modelLayer.context.fillText(params.txt, pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Molecules.prototype.Point = function ( x, y, z, graph, noRotate ) {
        this.graph = graph;
        this.noRotate = noRotate;
        this.original = {
            'x':x,
            'y':y,
            'z':z
        };
        OU.activity.Molecules.prototype.Point.prototype.calc = function () {
            var graph = this.graph, x = this.original.x, y = this.original.y, z = this.original.z,
            nz, ny, rz = graph.window.h, newX, newY, mol = graph.molecule, dz, rr = graph.renderRange;
            if (mol===undefined) {
                mol = {
                    z1:0,
                    z2:0
                };
            }
            if (this.noRotate===undefined) {
                // rotate about Y axis
                nz = (z * graph.rotCY) - (x * graph.rotSY);
                x = (z * graph.rotSY) + (x * graph.rotCY);
                z = nz;
                // rotate about X axis
                ny = (y * graph.rotCX) - (z * graph.rotSX);
                z = (y * graph.rotSX) + (z * graph.rotCX);
                y = ny;
            }
            this.in3d = {
                'x':x,
                'y':y,
                'z':z
            };
            // Note 3D to 2D projection is performed following the method
            // described at http : // en.wikipedia.org / wiki / 3D_projection
            dz = z - (mol.z1 - ((mol.z2 - mol.z1) * 60 * (1 - graph.zoom)));
            newX = (rz * x * 15) / dz + graph.viewXOffset;
            newY = -((rz * y * 15) / dz) + graph.viewYOffset;
            this.in2d = {
                // Point in 2d relative to the window in pixels
                'x':newX,
                'y':newY
            };
            newY = -newY;
            if (newX > rr.maxX)
                rr.maxX = newX;
            if (newX < rr.minX)
                rr.minX = newX;
            if (newY > rr.maxY)
                rr.maxY = newY;
            if (newY < rr.minY)
                rr.minY = newY;
        };
        OU.activity.Molecules.prototype.Point.prototype.upd2Dto3D = function ( dx, dy ) {
            // calculate and update 3d co - ords, based on current 2d co - ords
            var newX = this.in2d.x = this.in2d.x + dx, newY = this.in2d.y = this.in2d.y + dy, g = this.graph,
            eq = g.molecule, rz = g.window.h, p3, nz, ny,
            dz = this.in3d.z - (eq.z1 - ((eq.z2 - eq.z1) * 60 * (1 - g.zoom))),
            r2d = Math.PI / 180,
            rotCX = Math.cos(-g.rotation.x * r2d),
            rotSX = Math.sin(-g.rotation.x * r2d),
            rotCY = Math.cos(-g.rotation.y * r2d),
            rotSY = Math.sin(-g.rotation.y * r2d);
            this.in3d.x = ((newX - g.viewXOffset) / (15 * rz)) * dz;
            this.in3d.y = ((g.viewYOffset - newY) / (15 * rz)) * dz;
            // note in3d.Z remains the same - we only move in current XY plane
            // Calculate Original XYZ from current in3d XYZ
            p3 = this.in3d;
            x = p3.x;
            y = p3.y;
            z = p3.z;
            ny = (y * rotCX) - (z * rotSX);
            z = (y * rotSX) + (z * rotCX);
            y = ny;
            nz = (z * rotCY) - (x * rotSY);
            x = (z * rotSY) + (x * rotCY);
            z = nz;
            this.original = {
                x:x,
                y:y,
                z:z
            }
        };
        this.calc();
    };
    OU.activity.Molecules.prototype.atomType = function ( s ) {
        var i;
        s = s.replace(/^\s+|\s+$/g, "");
        for (i = OU.atoms.length; i--;) {
            if (OU.atoms[i].name==s) {
                return i;
            }
        }
        return 0;
    };
    /** @class */
    OU.activity.Molecules.prototype.Molecule = function ( params, parent ) {
        // TODO : ensure x2 > x1, y2 > y1, z2 > z1 - if not, swap round values
        this.name = params.name || 'y=?';
        this.parent = parent;
        this.x1 = params.x1 || -10;
        this.x2 = params.x2 || 10;
        this.y1 = params.y1 || -5;
        this.y2 = params.y2 || 5;
        this.z1 = params.z1 || -10;
        this.z2 = params.z2 || 10;
        this.reset_x1 = this.x1;
        this.reset_x2 = this.x2;
        this.reset_y1 = this.y1;
        this.reset_y2 = this.y2;
        this.reset_z1 = this.z1;
        this.reset_z2 = this.z2;
        this.variables = params.variables || [];
        this.selection = params.selection;
        this.atoms = params.atoms;
        this.animations = params.animations;
        this.atomicRadiusFactor = params.atomicRadiusFactor;
        this.molFile = params.molfile;
        if (params.colour!==undefined)
            this.colour = params.colour;
        if (params.rgb!==undefined)
            this.rgb = params.rgb;
        OU.activity.Molecules.prototype.Molecule.prototype.removeAtom = function ( idx ) {
            var i, j, b;
            for (i = this.atoms.length; i--;) {
                b = this.atoms[i];
                for (j = b.links.length; j--;) {
                    if (b.links[j]==idx) {
                        b.links.splice(j, 1);
                    }
                    else if (b.links[j] > idx) {
                        b.links[j]--;
                    }
                }
                for (j = b.duallinks.length; j--;) {
                    if (b.duallinks[j]==idx) {
                        b.duallinks.splice(j, 1);
                    }
                    else if (b.duallinks[j] > idx) {
                        b.duallinks[j]--;
                    }
                }
            }
            this.atoms.splice(idx, 1);
        };
        OU.activity.Molecules.prototype.Molecule.prototype.addLink = function ( aidx, bidx ) {
            var i, j, linked = false,
            a = this.atoms[aidx],
            b = this.atoms[bidx];
            if (aidx==bidx)
                return;
            for (i = a.links.length; i--;) {
                if (a.links[i]==bidx) {
                    linked = true;
                    a.links.splice(i, 1);
                    for (j = b.links.length; j--;) {
                        if (b.links[j]==aidx) {
                            b.links.splice(j, 1);
                            i = 0;
                            j = 0;
                        }
                    }
                }
            }
            if (linked) {
                this.atoms[aidx].duallinks.push(bidx);
                this.atoms[bidx].duallinks.push(aidx);
            }
            else {
                this.atoms[aidx].links.push(bidx);
                this.atoms[bidx].links.push(aidx);
            }
        };
        OU.activity.Molecules.prototype.Molecule.prototype.clearSelected = function () {
            var i;
            for (i = this.atoms.length; i--;) {
                this.atoms[i].selected = false;
            }
        };
        OU.activity.Molecules.prototype.Molecule.prototype.parseMOL = function () {
            if (this.molFile===undefined) {
                if(this.selection && this.selection.targetMolfile)
                    this.parseTargetMOL();
                return;
            }
            var i, a, b, l, r, t, s, atom, lines = this.molFile.split("\n"),
            sff = this.parent.config.spaceFillFactor,
            counts = lines[3],
            numAtoms = parseInt(counts.substr(0, 3)),
            numLinks = parseInt(counts.substr(3, 3));
            this.x1 = 2000;
            this.x2 = -2000;
            this.y1 = 2000;
            this.y2 = -2000;
            this.z1 = 2000;
            this.z2 = -2000;
            this.atoms = [];
            for (i = 0; i < numAtoms; i++) {
                a = lines[4 + i];
                s = a.substr(31, 3);
                atom = {
                };
                atom.id = i + 1;
                atom.type = this.parent.atomType(s);
                atom.x = parseFloat(a.substr(0, 10));
                atom.y = parseFloat(a.substr(10, 10));
                atom.z = parseFloat(a.substr(20, 10));
                atom.r = OU.atoms[atom.type].radius * (this.atomicRadiusFactor || 5);
                atom.sfrs = (atom.r * sff) * (atom.r * sff);
                atom.selected = false;
                if (atom.x < this.x1)
                    this.x1 = atom.x;
                if (atom.x > this.x2)
                    this.x2 = atom.x;
                if (atom.y < this.y1)
                    this.y1 = atom.y;
                if (atom.y > this.y2)
                    this.y2 = atom.y;
                if (atom.z < this.z1)
                    this.z1 = atom.z;
                if (atom.z > this.z2)
                    this.z2 = atom.z;
                atom.links = [];
                atom.duallinks = [];
                this.atoms.push(atom);
            }
            this.x1 -= 1;
            this.x2 += 1;
            this.y1 -= 1;
            this.y2 += 1;
            this.z1 -= 10;
            this.z2 += 10;
            for (i = 0; i < numLinks; i++) {
                b = lines[4 + i + numAtoms];
                l = parseInt(b.substr(0, 3)) - 1;
                r = parseInt(b.substr(3, 3)) - 1;
                t = parseInt(b.substr(6, 3));
                if (t==1) {
                    this.atoms[l].links.push(r);
                    this.atoms[r].links.push(l);
                }
                else {
                    this.atoms[l].duallinks.push(r);
                    this.atoms[r].duallinks.push(l);
                }
            }
        };
        OU.activity.Molecules.prototype.Molecule.prototype.parseTargetMOL = function () {
            if (this.selection.targetMolfile===undefined)
                return;
            var i, a, b, l, r, t, s, atom, lines = this.selection.targetMolfile.split("\n"),
            sff = this.parent.config.spaceFillFactor,
            counts = lines[3],
            numAtoms = parseInt(counts.substr(0, 3));
            this.x1 = 2000;
            this.x2 = -2000;
            this.y1 = 2000;
            this.y2 = -2000;
            this.z1 = 2000;
            this.z2 = -2000;
            this.atoms = [];
            for (i = 0; i < numAtoms; i++) {
                a = lines[4 + i];
                s = a.substr(31, 3);
                atom = {
                };
                atom.id = i + 1;
                atom.type = this.parent.atomType(s);
                atom.x = parseFloat(a.substr(0, 10));
                atom.y = parseFloat(a.substr(10, 10));
                atom.z = parseFloat(a.substr(20, 10));
                atom.r = OU.atoms[atom.type].radius * (this.atomicRadiusFactor || 5);
                atom.sfrs = (atom.r * sff) * (atom.r * sff);
                atom.selected = false;
                if (atom.x < this.x1)
                    this.x1 = atom.x;
                if (atom.x > this.x2)
                    this.x2 = atom.x;
                if (atom.y < this.y1)
                    this.y1 = atom.y;
                if (atom.y > this.y2)
                    this.y2 = atom.y;
                if (atom.z < this.z1)
                    this.z1 = atom.z;
                if (atom.z > this.z2)
                    this.z2 = atom.z;
                atom.links = [];
                atom.duallinks = [];
                this.atoms.push(atom);
            }
            this.atoms = [];
            this.x1 -= 1;
            this.x2 += 1;
            this.y1 -= 1;
            this.y2 += 1;
            this.z1 -= 10;
            this.z2 += 10;
        };
        OU.activity.Molecules.prototype.Molecule.prototype.deltas = function () {
            this.dx = Math.abs(this.x2 - this.x1);
            this.dy = Math.abs(this.y2 - this.y1);
            this.dz = Math.abs(this.z2 - this.z1);
        };
        this.parseMOL();
        if (params.highlightedAtoms!==undefined) {
            for (var i = params.highlightedAtoms.length; i--;) {
                this.atoms[params.highlightedAtoms[i] - 1].highlighted = true;
            }
        }
        this.deltas();
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Molecules, OU.util.Activity);
