/**
 * @fileOverview varControl - Variable control object, allows users to change a variable using a slider, and various other sub-elements
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.Draggable');
OU.require('OU.util.Layer');
/**
 * @class Var Control - Provides a cluster of control elements to manipulate a "Variable", for use in models, etc.
 * Note: VarControls are intended to be used within a VarControlBank.
 * @param {object} params - options:
 * <ul>
 * <li><strong>{double} x:</strong> X Co-ordinate</li>
 * <li><strong>{double} y:</strong> Y Co-ordinate</li>
 * <li><strong>{double} w:</strong> Width</li>
 * <li><strong>{double} h:</strong> Height (Note: this is the height of the button, not the stack)</li>
 * <li><strong>{double} stackH:</strong> Stack height - height of the "open" elements</li>
 * <li><strong>{int} tabIndex :</strong> tabIndex</li>
 * <li><strong>{object} variable:</strong> Reference to the variable to change the value of (must be an object with a field called "val")</li>
 * <li><strong>{OU.util.varControlBank} bank:</strong> reference to the parent control Bank that this varcontrol belongs to</li>
 * <li><strong>{boolean} valInButton:</strong> If true, then display the value of the variable in the button (default: false)</li>
 * <li><strong>{function} callback:</strong> Callback function that is called whenever the value changes.</li>
 * <li><strong>{object} parent:</strong> Reference to the calling object, typically an OU.util.Activity</li>
 * <li><strong>{OU.util.Layer} layer:</strong> The Layer that the control is to be rendered to.</li>
 * <li><strong>{string} instance:</strong> unique ID for this control</li>
 * <li><strong>{int} sortOrder:</strong> the sort order of this control</li>
 * </ul>
 */
OU.util.varControl = function ( params ) {
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || 0;
    this.h = params.h || 0;
    this.tabIndex = params.tabIndex || 0;
    this.layerOffset = params.layerOffset || 0;
    this.variable = params.variable;
    this.valInButton = params.valInButton || false;
    this.clickable = params.clickable; // @deprecated - no longer used
    this.callback = params.callback || function() {};
    this.parent = params.parent;
    this.bank = params.bank;
    this.context = params.layer.context;
    this.layer = params.layer;
    this.instance = params.instance || 'lc1';
    this.sW = this.w > 50 ? 50 : this.w; // stack width
    this.sX = this.parent.x + this.x + (this.w / 2) - (this.sW / 2); // stack X
    this.sOff = this.sX - this.x; // stack X relative to this.x
    this.isOpen = false;
    this.stackH = params.stackH || 0;
    this.hasOnOff = this.bank.hasOnOff;
    this.hasSlider = this.bank.hasSlider;
    this.hasVal = this.bank.hasVal;
    this.hasGrip = this.bank.hasGrip;
    this.sortOrder = params.sortOrder;
    this.showOnOff = (this.hasOnOff && !this.hasSlider);
    /**
     * Init the control
     * @private
     */
    OU.util.varControl.prototype.init = function () {
        this.isOpen = false;
        this.button = new OU.util.ControlButton({
            layer:this.layer,
            x:this.x,
            y:this.y,
            w:this.w,
            h:this.h,
            txt:this.variable.name,
            onClick:this.toggle,
            onClickParam:this,
            onOffIndicator:this.showOnOff,
            tabIndex: this.tabIndex
        });
        if (this.hasGrip) {
            this.draggable = new OU.util.Draggable({
                "me":this,
                x:this.sX - this.parent.x,
                y:this.bank.y - this.stackH,
                w:this.sW,
                h:this.h * .8,
                disabled:true,
                "onStart":this.startDrag,
                "onEnd":this.endDrag,
                "onMove":this.newPos
            });
        }
    };
    /**
     * Resizes the control
     * @param {object} params - dims & options:
     * <ul>
     * <li><strong>{int} x:</strong> X Co-ordinate</li>
     * <li><strong>{int} y:</strong> Y Co-ordinate</li>
     * <li><strong>{int} w:</strong> Width</li>
     * <li><strong>{int} h:</strong> Height</li>
     * <li><strong>{int} layerOffset:</strong> New Layer offset</li>
     * </ul>
     */
    OU.util.varControl.prototype.resize = function ( params ) {
        this.close();
        this.x = params.x || this.x || 0;
        this.y = params.y || this.y || 0;
        this.w = params.w || this.w || 100;
        this.h = params.h || this.h || OU.controlHeight;
        this.layerOffset = params.layerOffset || this.layerOffset || 0;
        this.sW = this.w > 50 ? 50 : this.w;
        this.sX = this.parent.x + this.x + (this.w / 2) - (this.sW / 2);
        this.button.resize({
            x:this.x,
            y:this.y,
            w:this.w,
            h:this.h
        });
        if (this.draggable!==undefined) {
            this.draggable.resize({
                x:this.sX - this.parent.x,
                y:this.bank.y - this.stackH,
                w:this.sW,
                h:this.h * .8
            });
        }
    };
    /**
     * Renders the control elements that are currently visible
     */
    OU.util.varControl.prototype.render = function () {
        var v = this.variable,
            nv = ((v.val * 100) | 0) / 100,
            t = this.h * .1, uH = this.h * .8, invButton,
            x = this.hasGrip ? this.sX - this.parent.x : 0,
            y = this.hasGrip ? this.bank.y - this.stackH : 0,
            ctx, clickable;
        this.button.params.onOff = (v.val > v.min + (v.max - v.min) / 2);
        if (this.valInButton) {
            this.button.params.txt = v.name + ": " + nv;
        }
        if (this.isOpen) {
            clickable = this.openLayer.events.clickable;
            clickable.removeType('vcTmp');
            ctx = this.openLayer.context;
            ctx.beginPath();
            ctx.roundRect(x, y, this.sW, this.stackH, 5);
            ctx.fillStyle = '#ddd';
            ctx.fill();
            ctx.fillStyle = '#000';
            ctx.strokeStyle = '#0f0';
            ctx.textAlign = 'center';
            if (this.hasGrip) {
                ctx.gripPattern({
                    x:x + this.sW / 4,
                    y:y + uH * .25,
                    w:this.sW / 2,
                    h:uH * .5
                });
                y = y + uH;
                clickable.push(this.draggable, 'vcTmp');
            }
            if (this.hasVal) {
                ctx.save();
                ctx.fillStyle = 'rgba(26,26,26,0.95)';
                ctx.roundRect(0, 7, this.sW, y + ((uH / 3) * 2), 3);
                ctx.fill();
                ctx.fillStyle = '#AFDFE4';
                ctx.font = this.sW / 4 + 'px ' + OU.theme.font;
                ctx.fillText(nv, x + this.sW / 2, y + ((uH / 3)) + 7);
                y = y + uH;
                ctx.roundRect(x, y, this.sW, this.stackH - uH, 3);
                ctx.fillStyle = 'rgba(26,26,26,0.95)';
                ctx.fill();
                ctx.restore();
            }
            if (this.hasSlider) {
                clickable.push(this.slider, 'vcTmp');
                this.slider.render((v.val - v.min) / (v.max - v.min));
                y = y + this.h * 3;
            }
            if (this.hasOnOff) {
                if (v.val < v.min + (v.max - v.min) / 2) {
                    ctx.untickedBox({
                        x:x + this.sW / 4,
                        y:y + uH / 4,
                        w:this.sW / 2,
                        h:uH / 2
                    });
                    invButton = new OU.util.InvisibleButton({
                        x:x,
                        y:y,
                        w:this.sW,
                        h:uH,
                        onClick:this.onOff,
                        onClickParam:{
                            me:this,
                            val:v.max
                        }
                    });
                    clickable.push(invButton, 'vcTmp');
                }
                else {
                    ctx.tickBox({
                        x:x + this.sW / 4,
                        y:y + uH / 4,
                        w:this.sW / 2,
                        h:uH / 2
                    });
                    invButton = new OU.util.InvisibleButton({
                        x:x,
                        y:y,
                        w:this.sW,
                        h:uH,
                        onClick:this.onOff,
                        onClickParam:{
                            me:this,
                            val:v.min
                        }
                    });
                    clickable.push(invButton, 'vcTmp');
                }
            }
            this.context.clearRect(this.x + this.w * .1, this.y, this.w * .8, this.h);
            this.button.render();
            ctx = this.context;
            ctx.fillStyle = this.button.params.colour;
            ctx.beginPath();
            ctx.moveTo(this.x + this.w / 2 - t / 2, this.y + t);
            ctx.lineTo(this.x + this.w / 2 + t / 2, this.y + t);
            ctx.lineTo(this.x + this.w / 2, this.y + 2 * t);
            ctx.closePath();
            ctx.fill();
        }
        else {
            this.context.clearRect(this.x + this.w * .1, this.y, this.w * .8, this.h);
            this.button.render();
            ctx = this.context;
            ctx.fillStyle = this.button.params.colour;
            ctx.beginPath();
            ctx.moveTo(this.x + this.w / 2 - t / 2, this.y + 2 * t);
            ctx.lineTo(this.x + this.w / 2 + t / 2, this.y + 2 * t);
            ctx.lineTo(this.x + this.w / 2, this.y + t);
            ctx.closePath();
            ctx.fill();
        }
    };
    /**
     * Toggle the Open/Close state of the control.
     * If their are no "open" elements, then this simply changes the state of the control to OPEN, and closes all other varcontrols
     * @param {OU.util.varControl} me - reference to this
     */
    OU.util.varControl.prototype.toggle = function ( me ) {
        var v = me.variable;
        if (!(me.hasSlider || me.hasGrip || me.hasVal)) {
            // nothing to see when open, so revert to a simple on for this control - turn all others off
            v.val = v.max;
            me.highlander(me);
            me.callback(me.parent);
        }
        else {
            if (me.isOpen) {
                me.closeControls();
                me.render();
            }
            else {
                me.closeControls();
                me.open();
                me.render();
            }
        }
    };
    /**
     * Open the control - closing any others that are open. Makes any visible elements visible.
     */
    OU.util.varControl.prototype.open = function () {
        var sliderY = this.hasGrip ? this.bank.y - this.stackH : 0, uH = this.h * .8, clickable;
        if (this.hasGrip)
            sliderY = sliderY + uH;
        if (this.hasVal)
            sliderY = sliderY + uH;
        if (this.hasGrip) {
            this.openLayer = this.bank.layer;
        }
        else {
            this.openLayer = new OU.util.Layer({
                container:this,
                x:this.sX,
                y:this.y - this.stackH + this.layerOffset - 1,
                w:this.sW,
                h:this.stackH,
                'id':'varControlSlider',
                hasEvents:true
            });
        }
        clickable = this.openLayer.events.clickable;
        if (this.hasSlider) {
            this.slider = new OU.util.Slider({
                x:this.hasGrip ? this.sX : 0,
                y:sliderY,
                w:this.sW,
                h:this.h * 3,
                context:this.openLayer.context,
                orientation:'vertical',
                frames:10,
                callback:this.varValue,
                callbackParam:this,
                instance:'s' + this.instance,
                sliderPos:(this.variable.val - this.variable.min) / (this.variable.max - this.variable.min),
                sliderStyle:'circle',
                tabIndex: this.tabIndex+1
            });
            clickable.push(this.slider, 'vcTmp');
        }
        if (this.hasGrip) {
            this.draggable.events = this.openLayer.events;
            this.draggable.disabled = false;
            clickable.push(this.draggable, 'vcTmp');
        }
        this.isOpen = true;
    };
    /**
     * Closes this control - removes the visible elements
     */
    OU.util.varControl.prototype.close = function () {
        if(this.slider)
            this.slider.remove();
        if (!this.hasGrip && this.openLayer!==undefined && this.openLayer!=null)
            this.openLayer.remove();
        this.openLayer = null;
        this.isOpen = false;
    };
    /**
     * Part of the drag functionality, which allows var controls to be re-ordered
     * @private
     */
    OU.util.varControl.prototype.startDrag = function ( p ) {
        //sort the items so that the latest dragged is at the end of the layers, therefore render last next time (on top)
        p.me.bank.controls.sort(function ( a, b ) {
            return b.draggable.dragId - a.draggable.dragId;
        });
        p.me.bank.doRender = true;
    };
    /**
     * Part of the drag functionality, which allows var controls to be re-ordered
     * @private
     */
    OU.util.varControl.prototype.endDrag = function ( p ) {
        p.me.bank.doRender = true;
    };
    /**
     * Part of the drag functionality, which allows var controls to be re-ordered.
     * Moves the controller and Changes the sort order of the var controls if they have changed.
     * @private
     */
    OU.util.varControl.prototype.newPos = function ( p ) {
        var i, me = p.me,
            controls = me.bank.controls,
            numItems = controls.length;
        if (p.x < me.bank.x)
            p.x = me.bank.x;
        if (p.x > me.bank.x + me.bank.w - me.w)
            p.x = me.bank.x + me.bank.w - me.w;
        me.move(p.x);
        me.draggable.x = p.x; // keep draggable in line with control
        me.draggable.y = me.bank.y - me.stackH; // fix draggable height
        // Shuffle sort order based on new position
        controls.sort(function ( a, b ) {
            return a.x - b.x;
        });
        for (i = numItems; i--;) {
            controls[i].sortOrder = i;
        }
        controls.sort(function ( a, b ) {
            return b.draggable.dragId - a.draggable.dragId;
        });
        me.bank.doRender = true;
    };
    /**
     * Part of the drag functionality, which allows var controls to be re-ordered.
     * Moves the controller and all sub elements.
     * @param {double} x - X Co-ordinate
     * @private
     */
    OU.util.varControl.prototype.move = function ( x ) {
        if (x) {
            this.x = x;
            this.button.move({
                x:this.x
            });
            this.sX = this.parent.x + this.x + (this.w / 2) - (this.sW / 2);
            if (this.slider)
                this.slider.resize({
                    x:this.hasGrip ? this.sX : 0
                });
            this.draggable.x = this.x;
        }
    };
    /**
     * Closes all Var Controls in the Bank
     */
    OU.util.varControl.prototype.closeControls = function () {
        var controls = this.bank.controls, i, control;
        for (i = controls.length; i--;) {
            control = controls[i];
            control.close();
            control.disabled = true;
        }
        this.bank.render();
    };
    /**
     * Sets value to minimum of all var controls in the bank, other than this one.
     * @param {OU.util.varControl} thisController - reference to itself
     */
    OU.util.varControl.prototype.highlander = function ( thisController ) { // There can be only one!
        var controls = this.bank.controls, i, control;
        for (i = controls.length; i--;) {
            control = controls[i];
            if (control!==thisController)
                control.variable.val = control.variable.min;
        }
        this.bank.render();
    };
    /**
     * Sets new value for the control, in terms of the actual variable value
     * @param {object} val - new value and self reference with structure:
     * <ul>
     * <li><strong>{double} val:</strong> New value</li>
     * <li><strong>{OU.util.varControl} me:</strong> reference to self</li>
     * </ul> 
     */
    OU.util.varControl.prototype.onOff = function ( val ) {
        if (val.me!==undefined) {
            var v = val.me.variable;
            v.val = val.val;
            if (val.me.slider!==undefined) {
                val.me.slider.halted = true;
                val.me.slider.sliderPos = (v.val - v.min) / (v.max - v.min);
            }
            val.me.bank.render();
            val.me.callback(val.me.parent);
        }
    };
    /**
     * Sets new value for the control, in terms of a percentage of the slider (0 to 1)
     * @param {double} val - New value
     * @param {OU.util.varControl} me - reference to self
     */
    OU.util.varControl.prototype.varValue = function ( val, me ) {
        if (me!==undefined) {
            me.variable.val = me.variable.min + (val * (me.variable.max - me.variable.min));
            me.callback(me.parent);
            me.bank.render();
        }
    };
    this.init();
};
