/**
 * @fileOverview TimelineController - Displays a timeline and loads in activities when events are clicked
 *
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
OU.require('OU.util.Controller');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.TimelineController = function ( controllerData, instance, controller ) {
    OU.activity.TimelineController.prototype.canvasView = function () {
        OU.activity.TimelineController.superClass_.canvasView.call(this); // call the parent class canvasView
        var self = this;

        this.activities[this.instance] = {
            data:'data/'
        };
        this.backdropLayer = new OU.util.Layer({
            container:this,
            hasEvents:true,
            dontRegister:true
        });
        this.timelineLayer = new OU.util.Layer({
            container:this,
            hasEvents:true,
            zIndex:OU.OVERLAY_CONTROLLER_LEVEL,
            y:this.h * 0.9,
            h:this.h * 0.1,
            dontRegister:true
        });
        this.startPage = this.data.startPage || {};
        this.endPage = this.data.endPage || {};

        this.backdropLayer.events.clickable.push(this);
        this.timelineLayer.events.clickable.push(this);
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data,
            onLoad:function () {
                self._started=true;
                self.start();
            }
        });
    };
    OU.activity.TimelineController.prototype.resize = function () {
        OU.activity.TimelineController.superClass_.resize.call(this); // call the parent class resize 
        var i, ev;
        if(!this._started)
            return;
        if(this.backdropLayer.w==this.w && this.backdropLayer.h==this.h) // ensure there has actually been a resize
            return;
        this.backdropLayer.resize();
        this.timelineLayer.resize({
            y:this.h * 0.9,
            h:this.h * 0.1
        });
        if(this.events) {
            for (i = this.events.length; i--;) {
                ev = this.events[i];
                if (ev.thumbLayer!==undefined) {
                    ev.thumbLayer.remove();
                    ev.thumbLayer = undefined;
                    ev.decay = 0;
                }
            }
        }
        this.initTimeline();
        this.doRender = true;
    };
    OU.activity.TimelineController.prototype.remove = function () {
        OU.activity.TimelineController.superClass_.remove.call(this); // call the parent class resize 
        this.backdropLayer.remove();
        this.timelineLayer.remove();
    };
    OU.activity.TimelineController.prototype.start = function () {
        this.initTimeline();
        this.doRender = true;
        this.renderCycle();
    };
    OU.activity.TimelineController.prototype.initTimeline = function () {
        var i, j, ev, placed, h, d = this.data, clashing,
        ctx = this.timelineLayer.context,
        self = this;
        this.fontSize = this.data.fontSize || 12;
        ctx.font = (this.data.eventFontWeight || 'bold')+ ' ' + this.fontSize + 'px ' + (this.data.eventFont || OU.theme.font);
        ctx.textAlign = 'left';
        this.scale = this.w / d.windowPerc;
        
        this.leftBoundaryMax = this.leftBoundary = 0;
        if(this.startPage.on) {
            this.leftBoundaryMax = -this.w*.75;
        }
        else {
            this.startPage=undefined;
        }
        this.rightBoundaryMax = this.rightBoundary = this.scale - this.w*0.6;
        if(this.endPage)
            this.rightBoundaryMax = this.rightBoundary+this.w*.5;
        if(this.tlOffset===undefined)
            this.targetX=this.tlOffset = this.leftBoundaryMax;
        d.events.sort(function ( a, b ) {
            return a.position - b.position;
        });
        this.numRows = 1;
        this.events = [];
        if(this.tabbables) {
            for(i=this.tabbables.length; i--;) {
                this.tabbables[i].remove();
            }
        }
        this.tabbables=[];
        for (i = d.events.length; i--;) {
            ev = d.events[i];
            this.events.push({
                data:ev,
                position:ev.position
            });
            ev = this.events[this.events.length - 1];
            ev.w = ctx.measureText(ev.data.title, 0, 0).width + 20;
            ev.h = this.fontSize * 1.2;
            ev.x = this.x + ev.position * this.scale;
            if (ev.data.image!==undefined) {
                ev.thumbWidth = ev.data.image.width + 30; // plus 30 to include shadow
                ev.thumbHeight = ev.data.image.height;
            }
            else {
                ev.thumbWidth = ctx.measureText(ev.data.data + ': ' + ev.data.title, 0, 0).width + 20;
                ev.thumbHeight = 200;
            }
            ev.decay = 0;
            if (this.events.length > 1) {
                ev.row = 0;
                do {
                    ev.row++;
                    clashing = false;
                    for (j = 0; j < this.events.length - 1 && !clashing; j++) {
                        placed = this.events[j];
                        if (placed.row==ev.row) {
                            if (placed.x < ev.x + ev.w + 10 && ev.x < placed.x + placed.w + 10)
                                clashing = true;
                        }
                    }
                } while (clashing);
            }
            else {
                ev.row = 1;
            }
            if (this.numRows < ev.row)
                this.numRows = ev.row;
            ev.isHit = function ( x, y, state ) {
                if (state) {
                    self.clearSection();
                    self.addActivity(this.data.section.activities[0]);
                    self.timelineLayer.events.flush();
                }
            };
            if (ev.data.section) {
                if (ev.data.section.options===undefined)
                    ev.data.section.options = {};
                ev.data.section.options.maintainLayout = true; // force section to maintain layout
            }
            
            this.tabbables.push(new OU.util.TimelineControllerEventTitle({
                container:this,
                position: ev.position*this.rightBoundary,
                event: ev,
                tabIndex: i+100
            }));
        }
        this.timelineHeight = h = (this.numRows + 2.5) * this.fontSize * 2.25;
        this.timelineLayer.resize({ // resize timeline according to number of rows
            y:this.h - h,
            h:h
        });
        ctx.font = (this.data.eventFontWeight || 'bold')+ ' ' + this.fontSize + 'px ' + (this.data.eventFont || OU.theme.font);
        this.events.sort(function ( a, b ) {
            return a.position - b.position;
        });
        this.timePoint = 0;
        this.setBGFont();
        
        if(this.startPage) {
            if(this.startHTML) {
                this.startHTML.resize({
                    x: this.startPageX(),
                    y:this.y,
                    w:this.w
                });
            }
            else {
                this.startHTML = new OU.util.Div({
                    innerHTML: this.startPage.html,
                    x:this.startPageX(),
                    y:this.y,
                    w:this.w,
                    h:"auto",
                    container: this
                });
            }
        }
        if(this.endPage) {
            if(this.endHTML) {
                this.endHTML.resize({
                    x:this.endPageX(),
                    w:this.w
                });
            }
            else {
                this.endHTML = new OU.util.Div({
                    innerHTML: this.endPage.html,
                    x:this.endPageX(),
                    y:this.y,
                    w:this.w,
                    h:"auto",
                    container: this
                });
            }
        }
        this.tidyDates();
    };
    OU.activity.TimelineController.prototype.tidyDates = function () {
        var datePadding=20,
        i,j,eD,ev,ctx = this.timelineLayer.context,x,w,merged;
        this.eventDates=[];
        for (i = this.events.length; i--;) {
            ev = this.events[i];
            x = this.x + ev.position * this.scale;
            w = ctx.measureText(ev.data.date, x, 0).width;
            merged=false;
            for(j=this.eventDates.length;j--;) {
                eD = this.eventDates[j];
                if((x<=eD.bounds.r+datePadding && x>=eD.bounds.l-datePadding) || (x+w<=eD.bounds.r+datePadding && x+w>=eD.bounds.l-datePadding)) { // clash
                    merged=true;
                    if(ev.position<eD.positions.l) {
                        eD.positions.l=ev.position;
                        eD.dates.l = ev.data.date;
                        eD.bounds.l=x;
                    }
                    else if(ev.position>eD.positions.r) {
                        eD.positions.r=ev.position;
                        eD.dates.r = ev.data.date;
                        eD.bounds.r=x+w;
                    }
                    eD.positions.c = eD.positions.l + (eD.positions.r-eD.positions.l)/2;
                    if(eD.dates.l != eD.dates.r)
                        eD.txt = eD.dates.l+'-'+eD.dates.r;
                    else
                        eD.txt = eD.dates.l;
                }
            }
            if(!merged) {
                this.eventDates.push({
                    txt: ev.data.date,
                    positions: {
                        l:ev.position,
                        r:ev.position,
                        c:ev.position
                    },
                    bounds: {
                        l:x,
                        r:x+w
                    },
                    dates: {
                        l:ev.data.date,
                        r:ev.data.date
                    }
                });
            }
        }    
    }; 
    OU.activity.TimelineController.prototype.renderCycle = function () {
        var self = this;
        if (this.doRender==true)
            this.render();
        setTimeout(function () {
            self.renderCycle();
        }, 40);
    };
    OU.activity.TimelineController.prototype.setBGFont = function () {
        var bctx = this.backdropLayer.context,
        time = this.roundedTime(),
        size = this.h / 2;
        bctx.font = (this.data.timeFontWeight || 'bold') +' '+size + 'px ' + (this.data.timeFont || OU.theme.font);
        while (bctx.measureText(time, 0, 0).width > this.w * .8 && size > 24) {
            size = size * .9;
            bctx.font = size + 'px bold ' + OU.theme.font;
        }
        bctx.textAlign = 'center';
        bctx.fillStyle = '#999';
    };
    
    OU.activity.TimelineController.prototype.startPageX = function () {
        var x = -(this.tlOffset-this.leftBoundaryMax);
        return x<-this.w?-this.w:x;
    };
    OU.activity.TimelineController.prototype.endPageX = function () {
        var x=this.rightBoundaryMax-this.tlOffset;
        return x>this.x+this.w?this.x+this.w:x;           
    };
    OU.activity.TimelineController.prototype.render = function () {
        if(!this._started)
            return;
        var i, ev, bctx = this.backdropLayer.context,layer=this.timelineLayer,
        ctx = layer.context, time, timeCursor, nearestEvent, sT = 10000, dT,
        x,x1,x2, h = this.fontSize * 3, y = this.timelineHeight - h,
        lineH = this.fontSize * 2.25, bgGrad,
        circ = Math.PI * 2, hasEvents = false,
        delay = 10,fills,endLeftX,startLeftX,
        normalFills = { 
            bg: this.data.titleBgColour || '#fff',
            fg: this.data.titleTextColour || '#000'
        },
        highlightFills = { 
            bg: this.data.titleHightlightBgColour || '#fff',
            fg: this.data.titleHightlightTextColour || '#000'
        },
        offset = this.tlOffset, totalDecay;
        this.timePoint = offset / this.rightBoundary;
        if(this.timePoint<0)
            this.timePoint=0;
        if(this.timePoint>1)
            this.timePoint=1;
        time = this.roundedTime();
        timeCursor = this.x + this.timePoint * this.scale - offset;
        this.backdropLayer.clear();
        layer.clear();
        bgGrad = bctx.createLinearGradient(0, this.h - this.timelineHeight, 0, 0);
        bgGrad.addColorStop(0.1, '#fff');
        bgGrad.addColorStop(0, this.data.bottomFadeColour);
        bgGrad.addColorStop(1, this.data.topFadeColour);
        bctx.fillStyle = bgGrad;
        bctx.rect(0, 0, this.w, this.h);
        bctx.fill();
        bctx.fillStyle = this.data.timeColour || '#ccc';

        bctx.fillText(time, this.w / 2, (this.h - this.timelineHeight) / 2);
        ctx.save();
        ctx.fillStyle = this.data.timelineColour;
        ctx.fillRect(this.x, y, this.w, h);
        ctx.beginPath();
        ctx.fillStyle = this.data.timepointColour || 'rgba(0,0,0,0.5)';
        ctx.moveTo(timeCursor, y + h / 4);
        ctx.lineTo(timeCursor + 10, y + h);
        ctx.lineTo(timeCursor - 10, y + h);
        ctx.fill();
        ctx.fillStyle = this.data.dateColour || '#fff';
        ctx.lineWidth = 1;
        ctx.textAlign = 'center';
        totalDecay = 0;
   
        if(this.startPage) {
            this.startHTML.resize({
                x: this.startPageX()
            });
        }       
        if(this.endPage) {
            this.endHTML.resize({
                x: this.endPageX()
            });
        }       

        for (i = this.events.length; i--;) {
            ev = this.events[i];
            x = this.x + ev.position * this.scale - offset;
            dT = Math.abs(x - timeCursor);
            if (dT < sT) {
                sT = dT;
                nearestEvent = ev;
            }
            totalDecay = totalDecay + ev.decay;
        } 
        for (i = this.events.length; i--;) {
            ev = this.events[i];
            x = this.x + ev.position * this.scale - offset;
            fills = normalFills;
            if (totalDecay > 2) {
                if (ev==nearestEvent)
                    fills = highlightFills;
            }            
            if (x < this.x + this.w) {
                ctx.beginPath();
                ctx.moveTo(x + 10, y - lineH * ev.row);
                ctx.bezierCurveTo(x + 5, y - lineH * ev.row, x, y - lineH * ev.row, x, y);
                ctx.stroke();
                ctx.beginPath();
                ctx.arc(x, y, lineH / 4, 0, circ, false);
                ctx.fillStyle=fills.bg;
                ctx.fill();
                ctx.stroke();
            }
        }
        for (i = this.eventDates.length; i--;) {
            ev = this.eventDates[i];
            x1 = this.x + ev.positions.l * this.scale - offset;
            x2 = this.x + ev.positions.r * this.scale - offset;
            if (x1 < this.x + this.w && x2>0) {
                ctx.fillStyle = '#222';
                ctx.fillText(ev.txt, this.x + ev.positions.c * this.scale-offset+2, 1+y + lineH * .8);
                ctx.fillStyle = this.data.dateColour || '#fff';
                ctx.fillText(ev.txt, this.x + ev.positions.c * this.scale-offset, y + lineH * .8);
            }
        }
        if (this.touchState || this.sliding)
            nearestEvent.decay = delay * 8;
        ctx.textAlign = 'left';
        layer.events.clickable.length=0;
        layer.events.clickable.push(this);
       
        for (i = this.events.length; i--;) {
            ev = this.events[i];
            x = this.x + ev.position * this.scale - offset;
            if (x < this.x + this.w) {
                fills = normalFills;
                if (totalDecay > 2) {
                    if (ev==nearestEvent)
                        fills = highlightFills;
                }
                ctx.beginPath();
                ctx.rect(x + 10, y - (ev.row + 0.4) * lineH, ev.w, lineH * .8);
                ctx.stroke();
                ctx.fillStyle = fills.bg;
                ctx.fill();
                ctx.fillStyle = fills.fg;
                ctx.fillText(ev.data.title, x + 20, y - lineH * ev.row);
                
                layer.events.clickable.push(new this.EventTitle({
                    layer: layer,
                    container:this,
                    position: ev.position*this.rightBoundary,
                    event: ev,
                    x: x+10,
                    y: y - (ev.row + 0.4) * lineH,
                    w: ev.w,
                    h:lineH * .8
                }));
            }
            if (ev.decay > 0) {
                hasEvents = true;
                ev.decay--;
                if (ev.thumbLayer===undefined) {
                    if (x + ev.thumbWidth < this.w && x + ev.thumbWidth > 0) {
                        ev.thumbLayer = new OU.util.Layer({
                            container:this,
                            hasEvents:true,
                            zIndex:OU.OVERLAY_CONTROLLER_LEVEL,
                            x:x,
                            y:this.h - this.timelineHeight - ev.thumbHeight,
                            w:ev.thumbWidth,
                            h:ev.thumbHeight
                        });
                        ev.thumbLayer.events.clickable.push(ev);
                        ev.thumbLayer.context.font = 'bold 12px ' + OU.theme.font;
                        this.renderThumb(ev);
                    }
                }
                else {
                    if (x + ev.thumbWidth < this.w && x + ev.thumbWidth > 0) {
                        ev.thumbLayer.move({
                            x:x,
                            y:this.h - this.timelineHeight - ev.thumbHeight,
                            w:ev.thumbWidth,
                            h:ev.thumbHeight
                        });
                        ev.thumbLayer.opacity(ev.decay / delay);
                        ev.thumbLayer.context.font = 'bold 12px ' + OU.theme.font;
                        if (ev==nearestEvent)
                            ev.thumbLayer.zIndex(OU.OVERLAY_CONTROLLER_LEVEL + 1);
                        else {
                            ev.thumbLayer.zIndex(OU.OVERLAY_CONTROLLER_LEVEL);
                            if (ev.decay > delay)
                                ev.decay = delay;
                        }
                        this.renderThumb(ev);
                    }
                    else {
                        ev.thumbLayer.remove();
                        ev.thumbLayer = undefined;
                        ev.decay = 0;
                    }
                }
            }
            else {
                if (ev.thumbLayer!==undefined) {
                    ev.thumbLayer.remove();
                    ev.thumbLayer = undefined;
                }
            }
        }
        ctx.restore();
        if (!hasEvents)
            this.doRender = false;
    };
    OU.activity.TimelineController.prototype.EventTitle = function (params) {
        this.container = params.container;
        this.event = params.event;
        this.layer = params.layer;
        this.tlPosition=params.position;
        this.x=params.x;
        this.y=params.y;
        this.w=params.w;
        this.h=params.h;
        this.layer.events.clickable.push(this);
        this.isHit = function (x,y,state) {
            if(state && x>=this.x && x<=this.x+this.w && y>=this.y && y<=this.y+this.h) {
                this.container.clearSection();
                this.container.addActivity(this.event.data.section.activities[0]);
                this.container.slideTo(this.tlPosition);
            }
        };
        
    };
    OU.activity.TimelineController.prototype.renderThumb = function ( ev ) {
        var ctx = ev.thumbLayer.context, t, grad, w = ev.thumbWidth, h = ev.thumbHeight,
        img = ev.data.image, iw, ih, d;
        if (img) {
            iw = img.width;
            ih = img.height;
            d = ih / 3;
            if (d > w - iw)
                d = w - iw;
            grad = ctx.createLinearGradient(iw, ih, iw, ih - d);
            grad.addColorStop(0, 'rgba(0,0,0,0.8)');
            grad.addColorStop(1, 'rgba(0,0,0,0.1)');
            ctx.fillStyle = grad;
            ctx.beginPath();
            ctx.moveTo(iw, ih);
            ctx.lineTo(iw + d, ih - d);
            ctx.lineTo(0, ih - d);
            ctx.fill();
            ctx.drawImage(img, 0, 0, iw, ih);
            ctx.beginPath();
            ctx.rect(0, 0, iw, ih);
            ctx.strokeStyle = '#ccc';
            ctx.stroke();
        }
        else {
            ctx.beginPath();
            ctx.rect(0, h - 24, w, 18);
            ctx.fillStyle = '#fff';
            ctx.fill();
            ctx.stroke();
            ctx.fillStyle = '#000';
            if (ev.data.date!='')
                t = ev.data.date + ': ' + ev.data.title;
            else
                t = ev.data.title;
            ctx.fillText(t, 10, h - 14);
        }
    };
    OU.activity.TimelineController.prototype.slideTo = function ( x ) {
        var self=this;
        if(x!==undefined) {
            this.targetX = x;
        }
        this.tlOffset = this.tlOffset+(this.targetX-this.tlOffset)/8; // slow on approach animation
        if(Math.abs(this.tlOffset-this.targetX)<2) {
            this.tlOffset=this.targetX;
            this.sliding=false;
        }
        else {
            this.sliding=true;
            setTimeout(function() {
                self.slideTo();
            },40);
        }
        this.doRender=true;
    };
    OU.activity.TimelineController.prototype.isHit = function ( x, y, evState ) {
        var dX;
        this.touchState = evState;
        if (evState) {
            if (this.inDrag) {
                dX = (x - this.dragStartX)*3;
                this.tlOffset = this.tlOffset - dX;
                this.dragStartX = x;                
                if (this.tlOffset < this.leftBoundaryMax)
                    this.tlOffset = this.leftBoundaryMax;
                if (this.tlOffset > this.rightBoundaryMax+this.w*.25)
                    this.tlOffset = this.rightBoundaryMax+this.w*.25;
                
                if(this.tlOffset<-this.w*.25)
                    this.clearSection();
                if(this.tlOffset>this.rightBoundaryMax)
                    this.clearSection();
                this.doRender = true;
            }
            else {
                this.dragStartX = x;
                this.inDrag = true;
            }
        }
        else {
            this.inDrag = false;
        }
    };
    OU.activity.TimelineController.prototype.roundedTime = function () {
        var d = this.data,
        t = this.timePoint * (d.endScale - d.startScale) + d.startScale;
        return ((t / d.scaleRounding) | 0) * d.scaleRounding;
    };
    OU.base(this, controllerData, instance, controller);
};

OU.util.TimelineControllerEventTitle = function (params) {
    this.container = params.container;
    this.event = params.event;
    this.tlPosition=params.position;

    OU.util.TimelineControllerEventTitle.prototype.hit = function () {
        this.container.clearSection();
        this.container.addActivity(this.event.data.section.activities[0]);
        this.container.slideTo(this.tlPosition);
    };
    OU.util.TimelineControllerEventTitle.prototype.focus = function () {
        this.hit();
        return false;
    };
    OU.util.TimelineControllerEventTitle.prototype.blur = function () {
        return false;
    };        
    OU.base(this, params);        
};

OU.inherits(OU.util.TimelineControllerEventTitle, OU.util.Tabbable);


if (OU.util.Controller) {
    OU.inherits(OU.activity.TimelineController, OU.util.Controller);
}
else {
    OU.preOnLoad.push(function () {
        OU.inherits(OU.activity.TimelineController, OU.util.Controller);
    });
}