/**
 * @fileOverview Treeslideshow - A slideshow that can have a linear or branched structure, with hotspots (specified in data.js) on each slide
 * @params {JSON array} data Data as included in the data/data.js file
 * @author Greg Black & Nigel Clarke <nigel.clarke@pentahedra.com>
 */

// Load in any util elements that are required
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');


/**
 * @class Treeslideshow - A slideshow that can have a linear or branched structure, with hotspots (specified in data.js) on each slide
 * @extends OU.util.Activity
 * @param {Object} data - Holds the data content for a specific instance of the activity
 * @param {String|undefined} instance - A unique identifier name for this instance, defaults to 'a1'
 * @param {Controller Object|undefined} controller - A reference to the controller that initialised this instance, if undefined, the superclass will generate a new controller and create the reference to it as 'this.controller'
 */
OU.activity.Treeslideshow = function(data,instance,controller) {

    /**
     * canvasView - this is the function that is first called when the activity is launched,
     * assuming you are in a browser that supports canvas.
     *
     * Typical tasks for this function are:
     * <ul>
     * <li> Define and initialise and activity wide variables</li>
     * <li> Initialise Layers & Divs</li>
     * <li> Call a loading function</li>
     * <li> Initiating the activity by calling any methods that</li>
     * </ul>
     */
    OU.activity.Treeslideshow.prototype.canvasView = function() {
        var bH = OU.controlHeight, self = this;

        //background layer
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw default background

        /** @constant **/
        this.IMAGE_LOAD_TIMEOUT = 10000; // allow max 10 seconds to load images

        //settings for caption object
        this.useCaptions = true;
        this.captionHeight = this.h*.1;
        this.slideHeight = this.h-bH; //height of slides -recalced, based on control placement
        this.slides = this.data.slides;
        
        this.currentSlide = 0; //image number that is currently in focus
        this.currentSelection = "";
        this.imageLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        
        this.slideHeight = this.h-bH;
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data: this.slides,
            onLoad: function() {
                self.started = true;
                self.scaleImages();
                self.render();
            }
        });
        //this.loadVars();
    };
    
    /**
     * The main renderer. Draws the current slide to the image layer. Creates markers if they do not already exist and renders a caption at the bottom of the image layer.
     */
    OU.activity.Treeslideshow.prototype.render = function() {
        var marker,mk,ctx = this.imageLayer.context,
        i,x,y,w,h,slide,clickable = this.imageLayer.events.clickable;
        
        if(this.currentSlide < 0) return;
        
        ctx.clearRect(0,0,this.w,this.h); // clear layer
        slide = this.slides[this.currentSlide];
        //render the current slide
        w = slide.w;
        h = slide.h;
        y = slide.y;
        x = slide.x;
        ctx.drawImage(slide.image,x,y,w,h);
        
        // Process any controller API requests that are in the data file - will have no effect if they aren't defined
        this.controller.processAPI(slide.controllerAPI);

        if(slide.markers==undefined) return;
        
        //if markers already exist, position and scale them, otherwise create them
        if(this.markers) {
            for(i=0; i<this.markers.length; i++) {
                marker = this.markers[i].marker;
                this.markers[i].render({
                    x: x + (marker.x * this.scaleFactor),
                    y: y + (marker.y * this.scaleFactor),
                    r: marker.radius ? marker.radius * this.scaleFactor : undefined,
                    w: marker.w ? marker.w * this.scaleFactor : undefined,
                    h: marker.h ? marker.h * this.scaleFactor : undefined
                });
            }
        }
        else{  
            this.markers=[];
            clickable.length = 0;
            for (i=0; i < slide.markers.length; i++) {
                marker = slide.markers[i];

                mk = new this.Marker({
                    parent: this,
                    x: x + (marker.x * this.scaleFactor),
                    y: y + (marker.y * this.scaleFactor),
                    r: marker.radius ? marker.radius * this.scaleFactor : undefined,
                    w: marker.w ? marker.w * this.scaleFactor : undefined,
                    h: marker.h ? marker.h * this.scaleFactor : undefined,
                    shape: marker.shape || "circle",
                    showGuide: marker.showGuide===false?false:true,
                    marker: marker,
                    context: ctx,
                    activityPopup: marker.activityPopup
                });
                clickable.push(mk);
                this.markers.push(mk);
            }
        }
        
        //render caption
        var captionTxt = "";
        if(this.currentSelection == ""){
            captionTxt = slide.label+" "+this.currentSelection;
        }
        else{
            captionTxt = this.currentSelection;
        }
        if(this.useCaptions){
            new OU.util.DynText({
                txt:captionTxt,
                fontSize: 16,
                x:0,
                y:this.h-this.captionHeight,
                w:this.w,
                h:this.captionHeight,
                context:ctx,
                background: {
                    RGB: '255,255,255',
                    alpha: 0.5
                }
            });
        }
    };
    /**
     * Removes the markers from the image layer. Also removes the tabbable element of each marker.
     */
    OU.activity.Treeslideshow.prototype.clearMarkers = function () {
        var i;
        if(this.markers) {
            for(i=this.markers.length; i--;) {
                this.markers[i].remove();
            }
        }
        this.markers=null;
    };
    /**
     * Called when the activity is being removed by a controller.
     */
    OU.activity.Treeslideshow.prototype.remove = function() {
        OU.activity.Treeslideshow.superClass_.remove.call(this); // call the superclass method
    };
    /**
     * Save some (optional) data associated with a particular slide.
     */
    OU.activity.Treeslideshow.prototype.saveVars = function (name, value) {
        OU.LocalStorage.save(this.data.storageItem+"."+name, value);
    };
    /**
     * This function is called instead of canvasView if the browser does not support HTML5 canvas.
     */
    OU.activity.Treeslideshow.prototype.accessibleView = function() {
        OU.activity.Treeslideshow.superClass_.accessibleView.call(this); // call the superclass method if you want to extend it, remove this line to override the method instead
    };
    /**
     * Resize all visible elements, calling a method to scale the images and finally call the main renderer.
     */
    OU.activity.Treeslideshow.prototype.resize = function() {
        OU.activity.Treeslideshow.superClass_.resize.call(this); // call the superclass resize
        if(!this.started) return;
        
        this.bgLayer.resize();
        this.bgLayer.context.gradRect();

        this.imageLayer.resize();
        this.focusOffset = this.w; //start the first slide off the right of the canvas
        this.clearMarkers();
        
        //set proportions of slideshow elements
        this.scaleImages();
        this.render();
    };
    /**
     * Function to scale all the images, based on the largest image.
     */
    OU.activity.Treeslideshow.prototype.scaleImages = function() {
        var j,w,h,img,minScale=10000,wScale,hScale,offsetX,offsetY;
        this.captionHeight = this.h*.1;
        this.slideHeight = this.h-this.captionHeight;
        for(j=this.slides.length; j--;) {
            img = this.slides[j].image;
            w = img.width;
            h = img.height;
            wScale = this.w/w;
            hScale = this.h/h;
            if(wScale<minScale){
                minScale=wScale;
                offsetX = (this.w-w*minScale)/2;
                offsetY = (this.slideHeight-h*minScale)/2;
            }
            if(hScale<minScale){
                minScale=hScale;
                offsetX = (this.w-w*minScale)/2;
                offsetY = (this.slideHeight-h*minScale)/2;
            }
        }
        for(j=this.slides.length; j--;) {
            img = this.slides[j].image;
            w = img.width;
            h = img.height;
            this.slides[j].w=w*minScale;
            this.slides[j].h=h*minScale;
            this.slides[j].y = offsetY;
            this.slides[j].x = offsetX;
        }
        this.scaleFactor = minScale;
    };
    /**
     * @class Marker
     * @extends OU.util.Tabbable
     * @param {object} params containing {x,y,w,h,nextSlide,radius,marker,showGuide,shape,parent}  
     * <ul>
     *  <li>{string} x - x-coordinate of marker</li>
     *  <li>{string} y - y-coordinate of marker</li>
     *  <li>{string} w - width of marker</li>
     *  <li>{string} h - height of marker</li>
     *  <li>{string} nextSlide - next slide to display when marker is clicked</li>
     *  <li>{string} radius - radius of marker</li>
     *  <li>{string} marker - a reference to the marker</li>
     *  <li>{string} showGuide - whether to show an outline of the marker (optional)</li>
     *  <li>{string} shape - rectangle, circle or polygon</li>
     *  <li>{string} parent - marker's parent</li>
     * </ul>
     */
    OU.activity.Treeslideshow.prototype.Marker = function (params) {
        this.context = params.context;
        this.x = params.x;
        this.y = params.y;
        this.w = params.w;
        this.h = params.h;
        this.nextSlide = params.nextSlide;
        this.radius = params.radius;
        this.marker = params.marker;
        this.showGuide = params.showGuide;
        this.shape = params.shape;
        this.parent = params.parent;

        this.isclicked=false;
        OU.activity.Treeslideshow.prototype.Marker.prototype.render = function () {
            if (this.showGuide || this.isclicked || this.hasFocus)
                this.renderClickArea();
        };
        /**
         * Function to render the clickable area of the marker.
         */
        OU.activity.Treeslideshow.prototype.Marker.prototype.renderClickArea = function () {
            var i, px, py,ctx=this.context;

            ctx.save();
            if(this.hasFocus) {
                ctx.strokeStyle = '#c00';
                ctx.lineWidth=3;
            }
            else {
                ctx.strokeStyle = '#444';
                ctx.lineWidth=1;
            }
            ctx.beginPath();
            switch (this.shape) {
                case "rectangle":
                    ctx.rect(this.x,this.y,this.w,this.h);
                    ctx.stroke();
                    break;
                case "poly":
                    var points = this.points;
                    ctx.beginPath();
                    for (i=points.length; i--;) {
                        px = this.x + (points[i].x * this.scaleFactor);
                        py = this.y + (points[i].y * this.scaleFactor);
                        if (i == 0) {
                            ctx.moveTo(px, py);
                        }
                        else {
                            ctx.lineTo(px, py);
                        }
                    }
                    ctx.fill();
                    ctx.closePath();
                    break;

                case "circle":
                default:
                    ctx.beginPath();
                    ctx.arc(this.x, this.y, this.radius * 6, 0, Math.PI * 2, true);
                    ctx.stroke();
                    ctx.closePath();
                    break;
            }
            ctx.restore();
        };
        /**
         * When marker receives focus, set flag and call render which will draw a focus rectangle.
         */
        OU.activity.Treeslideshow.prototype.Marker.prototype.focus = function () {
            this.hasFocus=true;
            this.render();
        };
        /**
         * When marker loses focus, set flag and call parent render and marker render which will clear the focus rectangle.
         */
        OU.activity.Treeslideshow.prototype.Marker.prototype.blur = function () {
            this.hasFocus=false;
            this.parent.render();
            this.render();
        };
        /**
         * Check whether a point is within a polygon. Returns true if an odd number of polygon edges are crossed by a vertical line extending upwards from the point.
         * @param {Number} x - the x-coordinate of the point
         * @param {Number} y - the y-coordinate of the point
         */
        OU.activity.Treeslideshow.prototype.Marker.prototype.countPolysides = function (x, y) {
            var points = this.marker.points,i, j = points.length - 1,
            sf = this.scaleFactor,
            xI, xJ, yI, yJ,oddNodes = false;

            for (i = 0; i < points.length; i++) {
                xI = this.x + points[i].x * sf;
                xJ = this.x + points[j].x * sf;
                yI = this.y + points[i].y * sf;
                yJ = this.y + points[j].y * sf;
                if ((yI < y && yJ >= y || yJ < y && yI >= y) && (xI <= x || xJ <= x)) {
                    if (xI + (y - yI) / (yJ - yI) * (xJ - xI) < x) {
                        oddNodes = !oddNodes;
                    }
                }
                j = i;
            }
            return oddNodes;
        };
        /**
         * Determine whether the marker has been clicked/pressed.
         * @param {Number} x - the x-coordinate of the point
         * @param {Number} y - the y-coordinate of the point
         * @param {Boolean} pressed - only respond to click/press event
         * @param {Object} events - the events object
         */
        OU.activity.Treeslideshow.prototype.Marker.prototype.isHit = function (x, y, pressed, events) {
            if (!pressed) return;
            var isHit = false,dx = x - this.x,dy = y - this.y;
            switch (this.marker.shape) {

                case "rectangle":
                    if (((x >= this.x) && (x <= this.x + this.w)) && ((y >= this.y) && (y <= this.y + this.h)))
                        isHit = true;
                    break;
                case "poly":
                    isHit = this.countPolysides(x, y);
                    break;
                case "circle":
                default:
                    if (Math.sqrt((dx * dx) + (dy * dy)) < this.radius * 6)
                        isHit = true; // mouse/touch is within circle
                    break;

            }
            if (isHit)
                this.hit();
        };
        /**
         * When a marker has been pressed, go to the next slide as specified in the data file. Also, store data if required.
         */
        OU.activity.Treeslideshow.prototype.Marker.prototype.hit = function () {
            this.parent.currentSlide=this.marker.nextSlide-1;
            this.parent.currentSelection = "";
            if(this.marker.showValue) this.parent.currentSelection = this.marker.displayText;
            //if a marker has storageName and storageValue, store that data when marker clicked
            if(this.marker.storageName!=undefined){
//                console.log(this.parent.data.storageItem+", "+this.marker.storageName+", "+this.marker.storageValue);
                this.parent.saveVars(this.marker.storageName, this.marker.storageValue);
            }
            this.parent.clearMarkers();
            this.parent.render();
        };
        OU.base(this, params);
        this.render();
    };
    OU.inherits(OU.activity.Treeslideshow.prototype.Marker, OU.util.Tabbable);

    // call the superclass's constructor
    OU.base(this,data,instance,controller);
};
// Call our inherits function to implement the class inheritance
OU.inherits(OU.activity.Treeslideshow,OU.util.Activity);
