/**
 * @fileOverview Dom object events handler - differs from OU.util.Events which is intended for use with Canvas objects
 *
 * Provides a uniform interface to the events associated with all DOM objects
 *
 * @auther Nigel Clarke <nigel.clarke@pentahedra.com>
 * @auther Gavin Hellyer <gavin.hellyer@pentahedra.com>
 */
/**
 * Attach event listener to a dom object.
 * Events are triggered on 'touchend' if a swipe has not been detected.
 * If the user moves more than the threshold horizontally or vertically during a single touch,
 * then a swipe event is triggered instead.
 */

/* JSLint directives */
/*global OU:true */

OU.util.DomEvents = function() {
    var self = this;
    var events = {};
    var types = {
        tap: 'onTap',
        longpress: 'onLongPress',
        doubletap: 'onDoubleTap',
        multitouch: 'onMultiTouch',
        multitouchend: 'onMultiTouchEnd'
    };
    var eventDefault = 'tap';
    var swipeThreshold = 50;
    var gestureTimeout = 450;
    var dragListeners = [];
    var swipeListeners = [];
    var __swipePrevented = false;

    this.attachToDom = function(domObj) { // domObj = document.body
        var start;
        var move;
        var end;
        var cancel;
        var propagate;

        start = function(e) {
            //console.log("DE.start()");
            //e.preventDefault();

            //console.log(JSON.stringify(events, null, 2));
            __swipePrevented = self.getOption(e.target, 'preventSwipe', true);
            events.type = 'tap';
            events.touching = true;
            events.swiped = false;
            events.startPosition = OU.combineMouseTouch(e);

            events.touches = e.touches;

//            console.log('Starting with ' + events.touches.length + ' touch events');

            if (!events.timer) {
                events.longpressCheck = true;
                events.timer = setTimeout(function() {
                    events.timer = null;
                    if (events.longpressCheck) {
                        events.type = 'longpress';

                        end(e);

                        events.override = true;
                    }
                }, gestureTimeout);
            }
            else {
                events.type = 'doubletap';
            }

            if (events.touches && events.touches.length > 1) {
                e.preventDefault();

                if (events.timer) {
                    clearTimeout(events.timer);
                    events.timer = null;
                }

                events.type = 'multitouch';
                events.override = true;
                events.propagateMove = true;
            }
        };
        move = function(e) {
            //console.log("DE.move()");
            e.preventDefault();

            var dx, dy;
            events.position = OU.combineMouseTouch(e);
            events.startPosition = events.startPosition || events.position;
            dx = events.position.pageX - events.startPosition.pageX;
            dy = events.position.pageY - events.startPosition.pageY;
            if ((Math.abs(dx) > swipeThreshold) || (Math.abs(dy) > swipeThreshold)) {
                events.swiped = true;
                if (events.timer) {
                    clearTimeout(events.timer);
                    events.timer = null;
                }
            }
            events.dx = dx;
            events.dy = dy;

            events.touches = e.touches;
            if (events.propagateMove) {
                var type = events.type || eventDefault;
                var event = e; // Set to whatever we need (start / end) event

                propagate(this, type, event.target, event);
            }

            // Handle drag
            var i;
            if (!__swipePrevented && events.touching) {
                events.startPosition = events.position;
                for (i = dragListeners.length; i--; ) {
                    if (dragListeners[i] !== null) {
                        dragListeners[i].onDrag(events);
                    }
                }
            }
        };
        end = function(e) {
            var i;

            //console.log("DE.end()");
            e.preventDefault();

            events.touches = e.touches;
            events.touching = false;

            if (!events.override) {
                if (!events.swiped || __swipePrevented) {

                    if (!events.swiped || events.type === 'multitouch') {
                        var type = events.type || eventDefault;
                        if (type === 'multitouch') {
                            type = 'multitouchend';
                        }
                        var event = e; // Set to whatever we need (start / end) event

                        propagate(this, type, event.target, event);
                    }
                }
                else {
                    // Trigger Swipe listeners
                    for (i = swipeListeners.length; i--; ) {
                        if (swipeListeners[i] !== null) {
                            swipeListeners[i].onSwipe(events);
                        }
                    }
                }
                // Handle drag
                if (!__swipePrevented) {
                    for (i = dragListeners.length; i--; ) {
                        if (dragListeners[i] !== null) {
                            dragListeners[i].endDrag(events);
                        }
                    }
                }

            }
            else {
                events.override = false;
            }

            //console.log('Ending with ' + events.touches.length + ' touch events');
            if (events.type === 'multitouch' && events.touches.length > 0) {
                // Do nothing.
            }
            else {
                events.type = null;
                events.longpressCheck = false;
                events.startDrag = events.startPosition = null;
                events.stopPropagation = false;
                events.propagateMove = false;
                __swipePrevented = false; // reset swipe prevention flag
            }
        };
        cancel = function(e) {
            events.touches = [];
        };
        propagate = function(domObj, type, target, event) {
            //console.log("DE.propagate()");
            var bubble = true;
            // Logic to check what is happening
            if (target.events) {
                // We have events, lets execute the one we need.
                if (target.events.hasOwnProperty(types[type]) && typeof target.events[types[type]] === 'function') {
                    target.events[types[type]](event, events, target);
                }
                else {
                    if (!target.parentNode || target.localName === 'body') {
                        // Reached the top. Don't continue
                        bubble = false;
                    }
                }

                if (target.events.options) {
                    events.stopPropagation = self.getOption(target, 'stopPropagation', false);
                    if (events.stopPropagation) {
                        bubble = false;
                    }
                }
            }
            else {
                if (!target.parentNode || target.localName === 'body') {
                    // Reached the top. Don't continue
                    bubble = false;
                }
            }

            if (bubble && target.parentNode) {
                propagate(domObj, type, target.parentNode, event);
            }
        };
        try {
            domObj.addEventListener("touchstart", start, false);
            domObj.addEventListener("touchend", end, false);
            domObj.addEventListener("touchmove", move, false);
            domObj.addEventListener("touchcancel", cancel, false);
        }
        catch (e) {
            console.log("Warning: Touch events not supported by browser");
        }
        domObj.addEventListener("mousedown", start, false);
        domObj.addEventListener("mouseup", end, false);
        domObj.addEventListener("mousemove", move, false);
    };

    this.addKeyListener = function(fn) {
        var listener, keyDown = function(key) {
            fn(key);
        };
        document.addEventListener("keydown", listener = function(e) {
            var k = null;
            if (e._dealtWith)
                return;
            e._dealtWith = true;
            if (window.event) {
                k = window.event.keyCode;
            }
            else if (e) {
                k = e.which || e.keyCode;
            }
            if (k !== 9) { // tab key - don't prevent field, otherwise buggers tabbing to inputs
                e.preventDefault();
            }
            if (k === 32 || k === 13) { // if SPACE or ENTER, then see if focussed element has a tap event
                var target = document.activeElement;
                if (target.events && target.events.hasOwnProperty(types["tap"]) && typeof target.events[types["tap"]] === 'function') {
                    target.events[types["tap"]](window.event, events, target);
                }
            }
            if (k !== null) {
                keyDown(k);
            }
        }, false, 0, true);
        return listener;
    };
    this.removeKeyListener = function(listener) {
        document.removeEventListener(listener);
    };

    this.addDragListener = function(params) {
        dragListeners.push({
            onDrag: params.onDrag || function() {
            },
            endDrag: params.endDrag || function() {
            }
        });
        return dragListeners.length - 1;
    };
    this.removeDragListener = function(index) {
        dragListeners[index] = null;
    };
    this.addSwipeListener = function(callback) {
        swipeListeners.push({
            onSwipe: callback
        });
        return swipeListeners.length - 1;
    };
    this.removeSwipeListener = function(index) {
        swipeListeners[index] = null;
    };
    this.addListener = function(domObj, callback, type) {
        type = type || eventDefault;
        if (types.hasOwnProperty(type)) {
            if (!domObj.events) {
                domObj.events = {};
            }
            if (domObj.events.hasOwnProperty(types[type])) {
                console.log('Overriding Event Listener: ' + type);
            }

            domObj.events[types[type]] = callback;
        }
    };

    this.addOption = function(domObj, option, value) {
        var options = {
            stopPropagation: false,
            preventSwipe: false
        };
        if (options.hasOwnProperty(option)) {
            if (!domObj.events) {
                domObj.events = {};
            }
            if (!domObj.events.options) {
                domObj.events.options = {};
            }
            domObj.events.options[option] = value || options[option];
        }
    };

    this.getOption = function(domObj, option, bubble) {
        if (domObj) {
            if (domObj.events && domObj.events.options) {
                if (domObj.events.options.hasOwnProperty(option)) {
                    return domObj.events.options[option];
                }
            }

            if (bubble) {
                if (domObj.localName !== 'body') {
                    return this.getOption(domObj.parentNode, option, bubble);
                }
            }
        }
        return false;
    };
    /**
     * Prevent a dom object triggering swipe events. IE AV position scrub bars.
     *
     * @param {object} domObj
     */
    this.preventSwipe = function(domObj) {
        this.addOption(domObj, 'preventSwipe', true);
    };
};

