/**
 * @fileOverview  Equation2D - renders a equation on a 2D graph
 *
 * @param {JSON array} data Data as included in the data/data.js file
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.HtmlBox');
OU.require('OU.util.Slider');
OU.require('OU.util.varControlBank');
OU.require('OU.util.Layer');

KEY_LEFT = 37;
KEY_UP = 38;
KEY_RIGHT = 39;
KEY_DOWN = 40;

/**
 * @class Equation2D - renders an equation on a 2D graph
 * @extends OU.util.Activity
 * @param {object} data - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.Equation2D = function(data, instance, controller) {

    /**
     * Starting point when running with HTML5 Canvas Support
     */
    OU.activity.Equation2D.prototype.canvasView = function() {
        var bH = OU.controlHeight, self = this;
        this.config = {
            depth: 5,
            fps: 40 // 40ms = 25 fps
        };
        // create Canvas Layers & Contexts
        this.bgLayer = new OU.util.Layer({
            container: this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.PAD = this.h * .045;
        this.modelLayer = new OU.util.Layer({
            container: this,
            'id': 'model',
            'h': this.h - bH
        });
        this.controlsLayer = new OU.util.Layer({
            container: this,
            'id': 'controls',
            hasEvents: true,
            pinch: this.pinch,
            pinchMe: this,
            keyPress: function(k) {
                self.keyPress(k);
            }
        });
        this.inRender = false;
        this.equation = function() {
        };
        // Set equation to 1st and render
        this.setEquations();
        this.changeEquation(0); // OU.LocalStorage.load("OU.equation2d.eq") || 0);
        // Don't remember which equation as it conflicts with drop down menu
        //TODO fix drop down menu so that it matches remembered equation'
        this.doRender = true;
        this.renderCycle();
        this.equationSelector();
    };
    /**
     * Determines the scale of the activity to ensure it all fits on screen
     */
    OU.activity.Equation2D.prototype.resize = function() {
        OU.activity.Equation2D.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight;
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.PAD = this.h * .045;
        this.modelLayer.resize({
            'h': this.h - bH
        });
        this.controlsLayer.resize();
        this.xScale = (this.w - this.PAD * 2) / this.equation.dx;
        this.yScale = ((this.h * .9) - this.PAD * 2) / this.equation.dy;
        this.controlsLayer.context.clearRect(0, 0, this.w, this.h);
        this.initControls();
        this.renderFrame();
        this.eqSelector.resize({
            x: this.x + this.w - 400,
            y: this.y + this.h * .025,
            w: 5,
            h: 5
        });
        this.doRender = true;
    };
    /**
     * Halts animation when the acitivity is removed to ensure the activity is not held in memory
     */
    OU.activity.Equation2D.prototype.remove = function() {
        OU.activity.Equation2D.superClass_.remove.call(this); // call the superclass method
        this.doRender = false;
    };
    /**
     * Event handler reacts to touch and click events over the equation
     * @param {int} x - x coordinate of the event
     * @param {int} y - y coordinate of the event
     * @param {boolean} evState - true if mouse or touch down
     */
    OU.activity.Equation2D.prototype.isHit = function(x, y, evState) {
        var eq = this.equation, dX, dY, edX, edY, bH = OU.controlHeight, PAD = this.PAD;
        if (evState) {
            if (this.inDrag) {
                dX = x - this.dragStartX;
                dY = y - this.dragStartY;
                edX = dX / (this.w - 2 * PAD) * (eq.x2 - eq.x1);
                edY = dY / (this.h - bH - 2 * PAD) * (eq.y2 - eq.y1);
                eq.x1 = eq.x1 - edX;
                eq.x2 = eq.x2 - edX;
                eq.y1 = eq.y1 + edY;
                eq.y2 = eq.y2 + edY;
                this.dragStartX = x;
                this.dragStartY = y;
                this.renderFrame();
                this.doRender = true;
            }
            else {
                this.dragStartX = x;
                this.dragStartY = y;
                this.inDrag = true;
            }
        }
        else {
            this.inDrag = false;
        }
    };
    /**
     * Event handler if a key is pressed
     * @param {int} k - the keycode of the key pressed
     */
    OU.activity.Equation2D.prototype.keyPress = function(k) {
        var eq = this.equation, dX = 0, dY = 0, edX, edY, bH = OU.controlHeight, PAD = this.PAD;

        switch (k) {
            case KEY_UP:
                dY = -10;
                break;
            case KEY_DOWN:
                dY = 10;
                break;
            case KEY_LEFT:
                dX = -10;
                break;
            case KEY_RIGHT:
                dX = 10;
                break;
        }

        edX = dX / (this.w - 2 * PAD) * (eq.x2 - eq.x1);
        edY = dY / (this.h - bH - 2 * PAD) * (eq.y2 - eq.y1);
        eq.x1 = eq.x1 - edX;
        eq.x2 = eq.x2 - edX;
        eq.y1 = eq.y1 + edY;
        eq.y2 = eq.y2 + edY;
        this.renderFrame();
        this.doRender = true;
    };
    /**
     * Event handler called when a pinch event is detected by the event handler.
     * Pinch effects zooming in and out of the graph
     * @param {double} e - end distance between the 2 poinch points
     * @param {double} s - start distance between the 2 poinch points
     * @param {double} x - x coordinate of the centre of the pinch
     * @param {double} y - y coordinate of the centre of the pinch
     * @param {double} dX - difference in x coordinates of the 2 end points
     * @param {double} dY - difference in y coordinates of the 2 end points
     * @param {object} me - reference back to the activity
     */
    OU.activity.Equation2D.prototype.pinch = function(e, s, x, y, dX, dY, me) {
        var eq = me.equation,
                mX = eq.x1 + (eq.x2 - eq.x1) / 2,
                mY = eq.y1 + (eq.y2 - eq.y1) / 2;
        dX = 1 - (dX - 1) * .5;
        dY = 1 - (dY - 1) * .5;
        eq.x1 = mX - (mX - eq.x1) * dX;
        eq.x2 = mX + (eq.x2 - mX) * dX;
        eq.y1 = mY - (mY - eq.y1) * dY;
        eq.y2 = mY + (eq.y2 - mY) * dY;
        eq.x1 = eq.x1 > eq.x2 - 1 ? eq.x2 - 1 : eq.x1; // ensure we never flip scale
        eq.y1 = eq.y1 > eq.y2 - 1 ? eq.y2 - 1 : eq.y1;
        eq.deltas();
        me.xScale = (me.w - me.PAD * 2) / eq.dx;
        me.yScale = ((me.h * .9) - me.PAD * 2) / eq.dy;
        me.renderFrame();
        me.doRender = true;
    };
    /**
     * Creates a new Equation object for each equation in the data
     */
    OU.activity.Equation2D.prototype.setEquations = function() {
        this.equations = [];
        for (var i = 0; i < this.data.equations.length; i++) {
            this.equations.push(new this.Equation(this.data.equations[i]));
        }
    };
    /**
     * Adds a select box to the page to allow the user to change the equation
     */
    OU.activity.Equation2D.prototype.equationSelector = function() {
        if (this.equations.length < 2)
            return; // don't include if only 1 equation
        OU.obj[instance] = this; // push this into object array, so we can call it back from the page
        var eq, h = '<form>Equation:<select onchange="OU.obj.' + instance + '.changeEquation();" id="eqList' + this.instance + '">';
        for (eq = 0; eq < this.equations.length; eq++) {
            h += '<option value="' + eq + '">' + this.equations[eq].name + '</option>';
        }
        h += '</select></form>';
        this.eqSelector = new OU.util.HtmlBox({
            container: this,
            html: h,
            x: this.x + this.w - 400,
            y: this.y + this.h * .025,
            w: 5,
            h: 5,
            unclosable: true,
            handleScrolling: false,
            style: 'overflow:visible'
        });
    };
    /**
     * Changes to the current view to another equestion
     * @param {int} eq - index of the equation in the equations array
     */
    OU.activity.Equation2D.prototype.changeEquation = function(eq) {
        if (eq === undefined) {
            var eqList = document.getElementById('eqList' + this.instance);
            eq = eqList.value;
        }
        this.equation = this.equations[eq || 0];
        OU.LocalStorage.save("OU.equation2d.eq", eq);
        // Calculate the viewport
        this.xScale = (this.w - this.PAD * 2) / this.equation.dx;
        this.yScale = ((this.h * .9) - this.PAD * 2) / this.equation.dy;
        this.controlsLayer.context.clearRect(0, 0, this.w, this.h);
        this.initControls();
        this.renderFrame();
        this.resetView();
    };
    /**
     * Resets the current scale and position of the view to the original settings
     */
    OU.activity.Equation2D.prototype.resetView = function() {
        var i, v, eq = this.equation, bH = OU.controlHeight;
        for (i = eq.variables.length; i--; ) {
            v = eq.variables[i];
            v.val = v.init;
        }
        eq.reset();
        this.xScale = (this.w - this.PAD * 2) / this.equation.dx;
        this.yScale = ((this.h - bH) - this.PAD * 2) / this.equation.dy;
        this.renderFrame();
        this.doRender = true;
    };
    /**
     * Loops every 40 milliseconds, performs a render in each iteration if required
     */
    OU.activity.Equation2D.prototype.renderCycle = function() {
        var self = this;
        if (this.modelLayer.context) { // If context is null, then activity has been removed, so stop looping
            if (this.doRender)
                this.render();
            setTimeout(function() {
                self.renderCycle();
            }, 40);
        }
    };
    /**
     * re-renders the current view
     */
    OU.activity.Equation2D.prototype.render = function() {
        var ctx = this.modelLayer.context, x, step, y1,
                variables = {}, // vairiable object that gets passed to calcY function
                j, y,bH = OU.controlHeight,
                eq = this.equation,
                xS = this.xScale,
                yS = this.yScale,
                PAD = this.PAD,
                yOff = this.h - bH - PAD;
        ctx.clearRect(0, 0, this.w, this.h - bH);
        ctx.save();
        ctx.lineWidth = 1;
        ctx.strokeStyle = "#44f";
        step = eq.dx / (this.w - 2 * PAD); // equivalent to a pixel
        // Set the value of the parameters for the equation in the variables object v
        for (j = eq.variables.length; j--; )
            variables[eq.variables[j].name] = eq.variables[j].val;
        ctx.beginPath();
        for (x = eq.x1; x < eq.x2; x = x + step) {
            variables.x = x;
            y1 = eq.calcY(variables);
            y = yOff - (y1 - eq.y1) * yS;
            if (isNaN(y))
                y = -1;
            if (x === eq.x1)
                ctx.moveTo(PAD + (x - eq.x1) * xS, y);
            else
                ctx.lineTo(PAD + (x - eq.x1) * xS, y);
        }
        ctx.stroke();
        ctx.restore();
        this.renderControls();
        this.doRender = false;
    };
    /**
     * Draws the axis for the graph
     */
    OU.activity.Equation2D.prototype.renderFrame = function() {
        var ctx = this.bgLayer.context,
                bH = OU.controlHeight,
                eq = this.equation,
                xAxis, yAxis, i, step, val,
                xS = this.xScale,
                yS = this.yScale,
                PAD = this.PAD;
        ctx.gradRect();
        ctx.save();
        ctx.fillStyle = '#fff';
        ctx.fillRect(PAD, PAD, this.w - 2 * PAD, this.h - bH - (2 * PAD));
        ctx.strokeStyle = "#000";
        ctx.lineWidth = 0.25;
        ctx.fillStyle = "#444";
        yAxis = -eq.x1 * xS + PAD;
        yAxis = yAxis < PAD ? PAD : (yAxis > this.w - PAD ? this.w - PAD : yAxis);
        ctx.beginPath();
        ctx.moveTo(yAxis, PAD);
        ctx.lineTo(yAxis, this.h - bH - PAD);
        ctx.closePath();
        ctx.stroke();
        step = (eq.dy / 20);
        ctx.textAlign = 'right';
        //      ctx.fillText("x:"+eq.x1+" to "+eq.x2+"    y:"+eq.y1+" to "+eq.y2+"    dX:"+eq.dx+"    dY:"+eq.dy,this.w-10,10);
        for (i = eq.y2; i >= eq.y1; i = i - step) {
            ctx.beginPath();
            ctx.moveTo(yAxis, PAD + (eq.y2 - i) * yS);
            ctx.lineTo(yAxis + 10, PAD + (eq.y2 - i) * yS);
            ctx.closePath();
            ctx.stroke();
            val = ((i * 100) | 0) / 100;
            ctx.fillText(val, yAxis - 5, PAD + (eq.y2 - i) * yS);
        }
        xAxis = eq.y2 * yS + PAD;
        xAxis = xAxis < PAD ? PAD : (xAxis > this.h - PAD - bH ? this.h - PAD - bH : xAxis);
        ctx.beginPath();
        ctx.moveTo(PAD, xAxis);
        ctx.lineTo(this.w - PAD, xAxis);
        ctx.closePath();
        ctx.stroke();
        step = (eq.dx / 20);
        ctx.textAlign = 'center';
        for (i = eq.x1; i <= eq.x2; i = i + step) {
            ctx.beginPath();
            ctx.moveTo(PAD + (i - eq.x1) * xS, xAxis);
            ctx.lineTo(PAD + (i - eq.x1) * xS, xAxis - 10);
            ctx.closePath();
            ctx.stroke();
            val = ((i * 100) | 0) / 100;
            ctx.fillText(val, PAD + (i - eq.x1) * xS, xAxis + 10);
        }
        ctx.restore();
    };
    /**
     * Set up the controls for the current equation
     */
    OU.activity.Equation2D.prototype.initControls = function() {
        var ctx = this.controlsLayer.context, bH = OU.controlHeight,
                clickable = this.controlsLayer.events.clickable,
                eq = this.equation, self = this;
        new OU.util.pageTitle({
            txt: '2D Equations',
            context: ctx
        });
        new OU.util.pageStrap({
            txt: this.equation.name,
            context: ctx,
            col1: 'rgba(170,204,255,0.5)',
            col2: 'rgba(170,204,255,0.0)'

        });
        clickable.length = 0;
        this.resetButton = new OU.util.ControlButton({
            txt: 'Reset',
            x: 0,
            y: this.h - bH,
            w: bH * 2,
            h: bH,
            layer: this.controlsLayer,
            onClick: function() {
                self.resetView();
            }
        });
        if (this.controlBank)
            this.controlBank.close();
        this.controlBank = new OU.util.varControlBank({
            x: bH * 2,
            y: this.h - bH,
            w: this.w - (bH * 2),
            h: bH,
            vars: eq.variables,
            valInButton: true,
            layer: this.controlsLayer,
            clickable: clickable,
            callback: this.varChange,
            parent: this
        });
        clickable.push(this);
    };
    /**
     * called when a variable has changed to ensure render is performed in the next loop
     * @param {object} me - refere to activity
     */
    OU.activity.Equation2D.prototype.varChange = function(me) {
        me.doRender = true;
    };
    /**
     * re-renders the parameter controls
     */
    OU.activity.Equation2D.prototype.renderControls = function() {
        this.controlBank.render();
    };
    /**
     * @class Equation class, keeps track of the equations
     * @param {object} params
     */
    OU.activity.Equation2D.prototype.Equation = function(params) {
        var t;
        this.name = params.name || 'y=?';
        this.x1 = params.x1 || 0;
        this.x2 = params.x2 || 10;
        this.y1 = params.y1 || 0;
        this.y2 = params.y2 || 10;
        this.z1 = params.z1 || 0;
        this.z2 = params.z2 || 10;
        if (this.x1 > this.x2) {
            t = this.x2;
            this.x2 = this.x1;
            this.x1 = t;
        }
        if (this.y1 > this.y2) {
            t = this.y2;
            this.y2 = this.y1;
            this.y1 = t;
        }
        this.variables = params.variables || [];
        this.reset_x1 = this.x1;
        this.reset_x2 = this.x2;
        this.reset_y1 = this.y1;
        this.reset_y2 = this.y2;
        this.calcY = params.calcY;
        if (params.colour !== undefined)
            this.colour = params.colour;
        if (params.rgb !== undefined)
            this.rgb = params.rgb;
        /**
         * calculates the delta X and Y values
         */
        OU.activity.Equation2D.prototype.Equation.prototype.deltas = function() {
            this.dx = Math.abs(this.x2 - this.x1);
            this.dy = Math.abs(this.y2 - this.y1);
        };
        /**
         * Sets current view to the original settings
         */
        OU.activity.Equation2D.prototype.Equation.prototype.reset = function() {
            this.x1 = this.reset_x1;
            this.x2 = this.reset_x2;
            this.y1 = this.reset_y1;
            this.y2 = this.reset_y2;
            this.deltas();
        };
        this.deltas();
    };
    /**
     * Alternative output in case the platform doesn't support canvas.
     * Outs a text list of the equations
     */
    OU.activity.Equation2D.prototype.accessibleView = function() {
        var i, h = '<div id="equation3DAccessible">';
        h += '<h1>' + this.data.title + '</h1>';
        h += '<p>' + this.data.description + '</p>';
        for (i = 0; i < this.data.equations.length; i++) {
            var equation = this.data.equations[i];
            h += '<h2>' + equation.name + '</h2><p>' + equation.description + '</p>';
        }
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Equation2D, OU.util.Activity);
