/**
 * info box menu activity
 *
 * @fileOverview InfoBoxMenu - An activity for clicking on hotspots to reveal a description
 *
 * @author Tegai Lansdell
 */

OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
OU.require('OU.activity.video');


/**
 * @class accessibleText
 * @extends OU
 * @param {object} params containing defaultText and id
 * <ul>
 *  <li>{string} defaultText - Default text to be spoken (optional)</li>
 *  <li>{String} id - Id of the accessbile Div (optional) </li>
 * </ul>
 * 
 * class for creating aria text with can be seen and spoken
 * by screen readers. I recommend  you don't set the is parameter
 * unless you really need to.
 */
OU.accessibleText = function(params) {
    
     this.currentText  = params.defaultText;
     this.previousText = "";
     this.id           = params.id;
     this.initialise();
};


/**
 * initialise - initialises the accessibility div
 *
 * Do not call this method.
 */
OU.accessibleText.prototype.initialise = function()
{
  
   var accessible = document.createElement('div');
   
   if (this.id === undefined)
       { this.id="accessibleTextId"; }
       
   if (this.currentText === undefined)
       { this.currentText = ""; }
      
   accessible.setAttribute("id",this.id);
   accessible.setAttribute("style","position:absolute;margin-top:-9999px");
   accessible.innerHTML = this.currentText;
   accessible.setAttribute("aria-live","polite");
   document.body.appendChild(accessible);
   

    
};

/**
 * clearText - clear the currently stored text
 *
 */
OU.accessibleText.prototype.clearText = function()
{
    this.currentText = "";
};

/**
 * setText - overwites the current text value with the supplied text
 * 
 * @param {string} txt - text string to be set
 *
 */
OU.accessibleText.prototype.setText = function(txt)
{
    this.currentText = txt;
};

/**
 * addText - concatenates the current text value with the supplied text
 * 
 * @param {string} txt - text string to be concatenated
 *
 */
OU.accessibleText.prototype.addText = function(txt)
{
    this.currentText += txt;
};

/**
 * getText - returns the stored text value
 * 
 * @returns (string) currently stored text
 */
OU.accessibleText.prototype.getText = function()
{
    return this.currentText;
};

/**
 * render - call this function when you are ready to have the text 
 * spoken by the screen reader
 * 
 * Basically the stored text value is written to the accessibilty div so that 
 * the screen reader can 'speak' the text
 * 
 * The text will only be updated if it is different from the previously 
 * displayed text.
 *
 */
OU.accessibleText.prototype.render = function()
{
    if (this.previousText !== this.currentText)
    {
        var domElm = document.getElementById(this.id),
            accessible = document.createElement('div');
        
        if (this.currentText === undefined)
           { this.currentText = ""; }
        
        domElm.parentNode.removeChild(domElm);
       
        accessible.setAttribute("id",this.id);
        accessible.setAttribute("style","position:absolute;margin-top:-9999px");
        accessible.innerHTML = this.currentText;
        accessible.setAttribute("aria-live","polite");
        accessible.setAttribute("tabindex","-1");
        document.body.appendChild(accessible);
        
        accessible.focus();
        
        //Do we need to reset the focus?
       
    }
    
};
     


/** 
 * @class InfoBoxMenu
 * @extends OU.util.Activity
 * @param {Object} data - Holds the data content for a specific instance of the activity
 * @param {String|undefined} instance - A unique identifier name for this instance, defaults to 'a1'
 * @param {Controller Object|undefined} controller - A reference to the controller that initialised this instance, if undefined, the superclass will generate a new controller and create the reference to it as 'this.controller'
 */
OU.activity.InfoBoxMenu = function(data,instance,controller) {

    /**
     * canvasView - this is the function that is first called when the activity is launched,
     *              assuming you are in a browser that supports canvas.
     *
     * Typical tasks for this function are:
     * <ul>
     * <li> Define and initialise and activity wide variables</li>
     * <li> Initialise the scaling and transformation variables</li>
     * <li> Initialise Layers & Divs</li>
     * <li> Call a loading function</li>
     * <li> Initiating the activity by calling any methods that</li>
     * </ul>
     */

    OU.activity.InfoBoxMenu.prototype.canvasView = function () {
        var self = this;
       
        this.bgLayer = new OU.util.Layer({
            container:this
        });
       
        this.accessibleTextView = new  OU.accessibleText({
            defaultText:this.data.accessibleDescription
        });
        
        this.activityIntsance = "";    
        this.scaleFactor = 1;
        this.translateX = 0;
        this.translateY =0;
        this.currentLinkIndex = -1;
        
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        
        this.currentDescription=this.data.defaultDescription;
        
         if (this.data.fps) {
            this.fps=this.data.fps;
        }else{
            this.fps = 80; // 12.5 frames per second - used for the render loop timeout
        
        }
        
        this.margin = this.data.activityMargin;
        
        this.bgLayer.context.fillStyle = "#ffffff";
        this.bgLayer.context.roundRect(this.x+this.margin , this.y+this.margin , (this.w-this.margin*2), (this.h-this.margin*2) , 0);
        this.bgLayer.context.fill();
        
        this.setScaleFactor();
               
        this.mainLayer = new OU.util.Layer({
            container:this,
            hasEvents:true,
            keyPress: function(key) {
                self.keyPress(key);
            }
        });
        
       
        this.boundary = {
            x:this.margin,
            y:this.margin,
            w:this.w - (this.margin *2),
            h:this.h - OU.controlHeight * 2 - this.margin
        };
      
        
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data,
            onLoad:function () {
                self.start();
            }
        });
           
        
    };
   
    /** 
     * drawObject - Draw the different types of objects on to the canvas
     *  
     *  Currently handles the following objects
     *  <ul>
     *       <li>poly - a series of points to make a shape. Can have a drop shadow</li>
     *       <li>line - a series of points to make a line.</li>
     *       <li>text - displays a text string</li>
     *       <li>circle - draws a circle with out with out a border</li>
     *       <li>rect   - draw a rectangle with rounded corners</li>
     *       <li>rect-simple - no rounded corners (faster)</li>
     *   <ul>
     *   
     *   @Todo
     *   <ul>
     *       <li>Add Oval shape</li>
     *       <li>Add image</li>
     *       <li>Add more checks for some data
     *   </li>
     *       
     */
     
    OU.activity.InfoBoxMenu.prototype.drawObject = function (obj,ctx,bHighlight) {
   
        var count,shadowCount,index;

        switch (obj.type)
        {
            case "poly":
            {
                for (shadowCount=((obj.shadow)?2:1);shadowCount>0;shadowCount--)
                {
                    ctx.save();
                  
                    if (obj.shadow && shadowCount===2)
                    {
                        ctx.fillStyle   = obj.bgColour;
                        ctx.strokeStyle = obj.bgColour;
                        
                        
                        ctx.shadowOffsetX = obj.shadow.x; 
                        ctx.shadowOffsetY = obj.shadow.y; 
                        ctx.shadowBlur = obj.shadow.blur; 
                        ctx.shadowColor = obj.shadow.colour;
                        
                    }
                    else
                    {
                        if (!bHighlight)
                        {
                            ctx.strokeStyle= obj.borderColour;
                            ctx.fillStyle= obj.bgColour;
                        }else{
                            ctx.strokeStyle= obj.borderColourActive;
                            ctx.fillStyle= obj.bgColourActive;
                        }
                    }
                    
                    ctx.lineWidth = obj.lineWidth;
                    ctx.beginPath();
                    ctx.moveTo(obj.points[0].x,obj.points[0].y);
                    index=1;
                    for (count=obj.points.length-1; count>0; count--, index++)
                    {
                        ctx.lineTo(obj.points[index].x,obj.points[index].y);
                    }
                    ctx.closePath();
                    ctx.fill();
                    ctx.stroke();
                    ctx.restore();
                }
                break;
            }
            case "line":
            {
                ctx.save();
                ctx.strokeStyle= obj.borderColour;
                ctx.lineWidth = obj.lineWidth;
                ctx.beginPath();
                ctx.moveTo(obj.points[0].x,obj.points[0].y);
                index=1;
                for (count=obj.points.length-1; count>0; count--, index++)
                {
                    ctx.lineTo(obj.points[index].x,obj.points[index].y);
                }
                ctx.stroke();
                ctx.restore();
                break;
            }
            case "arrowhead":
            {
                ctx.save();
                ctx.beginPath(); 
                ctx.fillStyle = obj.borderColour; 
                ctx.moveTo(obj.points[0], obj.points[1]);
                ctx.lineTo(obj.points[2], obj.points[3]);
                ctx.lineTo(obj.points[4], obj.points[5]);
                ctx.lineTo(obj.points[6], obj.points[7]);
                ctx.closePath();
                ctx.fill();
                ctx.restore();
                break;
            }
            case "curve":
            {
                ctx.save();
                ctx.moveTo(obj.points[0], obj.points[1]);
                ctx.quadraticCurveTo(obj.points[2],obj.points[3],obj.points[4],obj.points[5]);
                ctx.lineWidth = obj.lineWidth;
                ctx.strokeStyle = obj.borderColour; 
                ctx.stroke();
                ctx.restore();
                break;
            }
            case "text":
            {
                
                ctx.fillStyle= obj.colour;
                ctx.font = obj.fontStyle+" "+obj.fontSize+" "+obj.fontFamily;
                ctx.fillText(obj.text,obj.x,obj.y);
                
                break;
            }
            
            case "circle":
            {
                ctx.beginPath();
                if (!bHighlight)
                {
                   if (obj.borderColour) {ctx.strokeStyle= obj.borderColour;ctx.lineWidth = obj.lineWidth;}
                    ctx.fillStyle= obj.bgColour;
                }else{
                    if (obj.borderColourActive) {ctx.strokeStyle= obj.borderColourActive;ctx.lineWidth = obj.lineWidth;}
                    ctx.fillStyle= obj.bgColourActive;
                }           
                
                ctx.arc( obj.cx, obj.cy, obj.r, 0, Math.PI*2, false);
                ctx.closePath();
                ctx.fill();
                if (obj.borderColour) {ctx.stroke();}
                break;
            }
            case "rect":
            {
                
               for (count=((obj.shadow)?2:1);count>0;count--)
               {
                ctx.save();
                if (obj.shadow && count===2)
                {
                    ctx.fillStyle   = obj.bgColour;
                    ctx.strokeStyle = obj.bgColour;
                    ctx.lineWidth = obj.lineWidth*2;
                    
                    ctx.shadowOffsetX = obj.shadow.x; 
                    ctx.shadowOffsetY = obj.shadow.y; 
                    ctx.shadowBlur = obj.shadow.blur; 
                    ctx.shadowColor = obj.shadow.colour;
                }
                else
                {
                    if (!bHighlight)
                    {
                        ctx.strokeStyle= obj.borderColour;
                        ctx.fillStyle= obj.bgColour;
                    }else{
                        ctx.strokeStyle= obj.borderColourActive;
                        ctx.fillStyle= obj.bgColourActive;
                    }
                }
                
                ctx.lineWidth = obj.lineWidth; 
                
                ctx.beginPath();
                             
                ctx.moveTo(obj.x + obj.r, obj.y);
                ctx.lineTo(obj.x + obj.w - obj.r, obj.y);
                ctx.quadraticCurveTo(obj.x + obj.w, obj.y, obj.x + obj.w, obj.y + obj.r);
                ctx.lineTo(obj.x + obj.w, obj.y + obj.h - obj.r);
                ctx.quadraticCurveTo(obj.x + obj.w, obj.y + obj.h, obj.x + obj.w - obj.r, obj.y + obj.h);
                ctx.lineTo(obj.x + obj.r, obj.y + obj.h);
                ctx.quadraticCurveTo(obj.x, obj.y + obj.h, obj.x, obj.y + obj.h - obj.r);
                ctx.lineTo(obj.x, obj.y + obj.r);
                ctx.quadraticCurveTo(obj.x, obj.y, obj.x + obj.r, obj.y);
                
                ctx.closePath();
                ctx.fill();
                ctx.stroke();
                ctx.restore();
              }
                
              break;  
            }
            case "rect-simple":
            {
                ctx.save();
                if (!bHighlight)
                {
                    ctx.fillStyle= obj.bgColour;
                }else{
                    ctx.fillStyle= obj.bgColourActive;
                }
                ctx.fillRect(obj.x,obj.y,obj.w,obj.h);
                ctx.restore();
                
                break;
            }
            case "oval":
            {
              
               	var sakiyo = 0.551784,
                x1 = (obj.w / 2) * sakiyo, 
                y1 = (obj.h / 2) * sakiyo, 
                x2 = obj.x + obj.w,           
                y2 = obj.y +obj.h,           
                xmid = obj.x + obj.w / 2,       
                ymid = obj.y + obj.h / 2;      

                ctx.save();
                ctx.beginPath();
                ctx.moveTo(obj.x, ymid);
                ctx.bezierCurveTo(obj.x, ymid - y1, xmid - x1, obj.y, xmid, obj.y);
                ctx.bezierCurveTo(xmid + x1, obj.y, x2, ymid - y1, x2, ymid);
                ctx.bezierCurveTo(x2, ymid + y1, xmid + x1, y2, xmid, y2);
                ctx.bezierCurveTo(xmid - x1, y2, obj.x, ymid + y1, obj.x, ymid);
             
                if (!bHighlight)
                {
                    ctx.fillStyle= obj.bgColour;
                }else{
                    ctx.fillStyle= obj.bgColourActive;
                }
                ctx.fill();
                
                ctx.closePath();
                
                ctx.lineWidth = obj.lineWidth;
                ctx.strokeStyle = obj.borderColour;
                
                ctx.stroke();
                ctx.restore();
                break;
            }
        }
   
    };
    
    /** 
     * start - run the activity and start the render loop
     *         also creates the link object array.
     *  
     *       
     */
    OU.activity.InfoBoxMenu.prototype.start = function () {
        
        var count;
        
        this.descriptionLinks = [];
        this.descriptionTextData = [];
        this.areThereVideoLinks = false;
        
        
        for (count=this.data.description.length-1; count>=0; count--)
        {
            this.descriptionTextData[count] = new this.descriptionTextHandler({
                    InfoBoxMenu:this,
                    "descriptionId":this.data.description[count].linkId,
                    "data":this.data.description[count],
                    "ctx":this.mainLayer.context

                });   
            this.descriptionTextData[count].parseText();
        }  
            
        
        for (count=this.data.menuItems.length-1; count>=0; count--)
        {
            this.descriptionLinks[count] = new this.navLinks({
                InfoBoxMenu:this,
                "events":this.mainLayer.events,
                "data":this.data.menuItems[count],
                "ctx":this.mainLayer.context,
                "scaleFactor":this.scaleFactor,
                "translate":{x:this.translateX,y:this.translateY},
                "rollover":false
            });               
        }                     
        
        if (this.data.videos !== undefined)
        {
             this.areThereVideoLinks = true;  
  
        }
        
        // start render loop
        this.doRender = true;
        this.renderLoop();
    };
    
    
    /**
    *
    * resize - Main resize function
    * calls the setScaleFactor function
    */
    OU.activity.InfoBoxMenu.prototype.resize = function() {
        
               
        OU.activity.InfoBoxMenu.superClass_.resize.call(this); // call the parent class resize   
        
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
                      
        this.mainLayer.resize();
        
        this.boundary = {
            x:this.margin,
            y:this.margin,
            w:this.w - (this.margin *2),
            h:this.h - OU.controlHeight * 2 - this.margin
        };
       
          
        this.bgLayer.context.fillStyle = "#ffffff";
        this.bgLayer.context.roundRect(this.x+this.margin , this.y+this.margin , (this.w-this.margin*2), (this.h-this.margin*2) , 0);

        this.bgLayer.context.fill();
        
        this.setScaleFactor();
        
        this.doRender = true;
        this.renderLoop();
    };
    
    /**
    *
    * renderLoop - Main Render Loop
    *
    * Note setting the fps to zero (in the data) turns off the render loop
    */
    OU.activity.InfoBoxMenu.prototype.renderLoop = function () {
        var self = this;
        if (this.doRender)
            this.render();
        
        if (this.fps>0)
        {
            setTimeout(function () {
                self.renderLoop();
            }, this.fps);
        }
    };
    
    /**
    *
    * render - Main Render called from the render timeout
    *
    * The following tasks are carried out in this order
    * <ul>
    *   <li>clear the canvas</li>
    *   <li>Scale and center the activity</li>
    *   <li>display and background objects</li>
    *   <li>draw the links</li>
    *   <li>display any foreground objects</li>
    *   <li>display the current description</li>
    * </ul>
    * 
    */
    
    OU.activity.InfoBoxMenu.prototype.render = function () {
        
        this.accessibleTextView.clearText();
        
        
        //save the canvas context       
        this.mainLayer.context.save();
        
        //clear the canvas
        this.mainLayer.clear();
       
        //Scale and center the activity 
        this.mainLayer.context.translate(this.translateX,this.translateY);
        this.mainLayer.context.scale(this.scaleFactor, this.scaleFactor);  
        
        // display and background objects
        if (this.data.bgObjects)
            this.displayObjects(this.data.bgObjects);
          
        //draw the links  
        this.renderLinks();
        
        //display any foreground objects
         if (this.data.fgObjects)
            this.displayObjects(this.data.fgObjects);
        
        //display the current description
        this.displayDescription(this.currentDescription);
        
        //display any video content
        this.displayVideo(this.currentDescription);
               
        //restore the canvas context
        this.mainLayer.context.restore();
        
        
        this.accessibleTextView.render();
        
        
    };
    
   
   /**
    *
    * renderLinks - Render The Links
    */
    OU.activity.InfoBoxMenu.prototype.renderLinks = function () {
      var index=0,count,bHighlight;
       for (count=this.data.menuItems.length-1; count>=0; count--,index++)
       {
        if (this.currentDescription === this.data.menuItems[index].linkId){bHighlight = true;}else{bHighlight=false}
        this.descriptionLinks[index].render(bHighlight, this.scaleFactor, {x:this.translateX,y:this.translateY});
       }
              
    };
  
      
    /**
    *
    * setScaleFactor - function which sets the scaling factor based on the original width and height of the
    * activity (this is in the data file) and the current width and height of the canvas. The scaling
    * maintain the aspect ratio.
    *
    * The values for centering the activity (translateX and translateY) are also calculated here.
    */
    OU.activity.InfoBoxMenu.prototype.setScaleFactor = function () {
        
        
        var xfac = this.w / this.data.originalWidth,
            yfac = this.h / this.data.originalHeight;
        
        if (xfac <= yfac)
        {
            this.scaleFactor = xfac;
            
        }else{
            this.scaleFactor = yfac;
            
        }
        
        this.translateX = (this.w-(this.data.originalWidth * this.scaleFactor))/2;
        this.translateY = (this.h-(this.data.originalHeight * this.scaleFactor))/2;
       
    };
    
    /**
    * displayObjects - iterates through a list of objects and calls the drawObject function if they are to be
    * displayed based on the currentDescription variable.
    *
    * @param obj - list of objects to display from the data file
    *
    * the all ALLID is specified in the data file then the object will always be displyed
    */
    OU.activity.InfoBoxMenu.prototype.displayObjects = function (obj) {
       var ctx=this.mainLayer.context, theObjects=null, count, index, subcount;
       
       if (obj !== undefined)
       {
                
           for (count=obj.length-1, index=0; count>=0; count--,index++)
           {

                for (subcount=obj[index].linkId.length-1; subcount>=0; subcount--)
                {

                    if (obj[index].linkId[subcount] === this.currentDescription || obj[index].linkId[subcount]==="ALLID")
                    {
                    
                        theObjects = obj[index];
                        this.drawObject(theObjects,ctx,false);

                    }
                }
           }
        }
    };
    
    /**
    *
    * displayDescription - Draw the current description to then canvas
    * 
    * @param descriptionId - description id to be displyed
    *
    * This function can parse the following html tags
    *
    * <ul>
    *    
    *    <li> html b tag- bold text</li>
    * </ul>
    *
    * Todo
    * tidy up the drawing routine and more to the draw object function
    */
    OU.activity.InfoBoxMenu.prototype.displayDescription = function (descriptionId) {
      
       var count;
      
       //match IDs
       for (count=this.data.description.length-1; count>=0; count--)
       {
        if (this.data.description[count].linkId === descriptionId ){
            this.descriptionTextData[count].render();
            break;
        }
       }
    };
    
   
    /**
    *
    * displayVideo - Draw the current video to the canvas
    * 
    * @param descriptionId - video id to be displyed
    *
    
    */
    OU.activity.InfoBoxMenu.prototype.displayVideo = function (descriptionId) {
      var foundDescriptionIndex = -1, count=0;
      
      if ( this.areThereVideoLinks === true)
      {
          
        for (count=this.data.videos.length-1;count>=0;count--){
            if (descriptionId === this.data.videos[count].linkId){
                foundDescriptionIndex = count;
                break;
            }
        }
        
       if (foundDescriptionIndex>=0){
                        
            if (this.mainLayer.container.controller !== undefined) {   
                
                if (this.activityIntsance!=="")
                {
                    this.mainLayer.container.controller.removeActivity(this.activityIntsance);
                }
                this.activityIntsance = this.mainLayer.container.controller.addActivity(this.data.videos[foundDescriptionIndex].activityPopup, true);
                
               
            }
        }
      }
   }
    
    
    /** 
     * @class descriptionTextHandler
     * @extends OU.util.InfoBoxMenu
     * Class to handle the links on the activity 
     */
     OU.activity.InfoBoxMenu.prototype.descriptionTextHandler = function ( params ) {
        this.InfoBoxMenu = params.InfoBoxMenu;
        this.descriptionId = params.descriptionId;
        this.data = params.data;
        this.ctx = params.ctx;
        this.textDataArray = new Array(); 
                
        /**
        * render - renders the current description
        *
        * Todo 
        * move the scale factor and translate object into one object
        */
        OU.activity.InfoBoxMenu.prototype.descriptionTextHandler.prototype.render = function () {
            var count,index=0;
            
                
            for (count=this.textDataArray.length-1;count>=0;count--,index++)
            {

                if (this.textDataArray[index].bulletUsed === true)
                {
                    this.ctx.save();
                    this.ctx.beginPath();
                    this.ctx.fillStyle= this.textDataArray[index].bulletColour;
                   
                    this.ctx.arc( (this.textDataArray[index].x-(this.textDataArray[index].bulletRadius*4)), this.textDataArray[index].y, this.textDataArray[index].bulletRadius, 0, Math.PI*2, false);
                    this.ctx.closePath();
                    this.ctx.fill();
                    this.ctx.restore();
                }
                
                this.ctx.save();
                this.ctx.fillStyle= this.textDataArray[index].fillStyle;
                this.ctx.font = this.textDataArray[index].font;
                this.ctx.fillText(this.textDataArray[index].txt, this.textDataArray[index].x , this.textDataArray[index].y);
                this.ctx.restore();
                

                this.InfoBoxMenu.accessibleTextView.addText(this.textDataArray[index].txt) ;
            } 
  
            
            return;
            
        };
        
        OU.activity.InfoBoxMenu.prototype.descriptionTextHandler.prototype.parseText = function () {

            var count=0,
                theDescription = this.data,
                displayX=0,
                szString="", splitString="",
                lineWidth=0,
                textOffsetY=0,
                splitStingIndex=0, textWidth=0, textArrayCount=0;
                
            
            for (count=theDescription.text.length-1; count>=0; count--)
            {
                 displayX=theDescription.text[count].x;
                 
                                  
                 szString = theDescription.text[count].str;

                 this.ctx.font = theDescription.text[count].fontStyle+" "+theDescription.text[count].fontSize+" "+theDescription.text[count].fontFamily;
                 lineWidth   = this.ctx.measureText(szString, 0, 0).width;

                 szString = szString.replace(/<b>/gi," <b> ");
                 szString = szString.replace(/<i>/gi," <i> ");
                 szString = szString.replace(/<\/b>/gi," </end> ");
                 szString = szString.replace(/<\/i>/gi," </end> ");
                 szString = szString.replace(/<br>/gi," <br> ");
                 szString = szString.replace(/\\n/gi," <br> ");
                  
                 splitString = szString.split(" ");

                 textOffsetY = 0;
                 splitStingIndex = 0;

                 while (splitStingIndex<splitString.length)
                 {

                 
                  
                   displayX=theDescription.text[count].x;
                   this.textDataArray[textArrayCount] = new Object();
                   this.textDataArray[textArrayCount].x = displayX;
                   this.textDataArray[textArrayCount].y = textOffsetY + theDescription.text[count].y;
                   this.textDataArray[textArrayCount].txt = "";
                   this.textDataArray[textArrayCount].fillStyle= theDescription.text[count].colour;
                   this.textDataArray[textArrayCount].font = theDescription.text[count].fontStyle+" "+theDescription.text[count].fontSize+" "+theDescription.text[count].fontFamily;
                   
                   if (theDescription.text[count].bullet)
                    {
                        this.textDataArray[textArrayCount].bulletUsed = true;
                        this.textDataArray[textArrayCount].bulletColour=theDescription.text[count].bullet.colour;
                        this.textDataArray[textArrayCount].bulletRadius=theDescription.text[count].bullet.r;
                    }
                    else{

                        this.textDataArray[textArrayCount].bulletUsed = false;
                    }
                   
                   
                   lineWidth = 0;

                   while (splitStingIndex<splitString.length)
                   {

                     if (splitString[splitStingIndex]==="<br>")
                     {
                        
                        displayX=theDescription.text[count].x;
                                                              
                        lineWidth = 0;
                        splitStingIndex++;
                        break;

                     }
                     else if (splitString[splitStingIndex]==="<b>")
                     {
                         textArrayCount++;
                         
                         this.textDataArray[textArrayCount] = new Object();
                         this.textDataArray[textArrayCount].x = displayX;
                         this.textDataArray[textArrayCount].y = theDescription.text[count].y + textOffsetY;
                         this.textDataArray[textArrayCount].txt = "";
                         this.textDataArray[textArrayCount].fillStyle= theDescription.text[count].colour;
                         this.ctx.font = "bold"+" "+theDescription.text[count].fontSize+" "+theDescription.text[count].fontFamily;
                         this.textDataArray[textArrayCount].font = "bold"+" "+theDescription.text[count].fontSize+" "+theDescription.text[count].fontFamily;

                         splitStingIndex++;

                     }
                     else if (splitString[splitStingIndex]==="<i>")
                     {
                         textArrayCount++;
                         
                         this.textDataArray[textArrayCount] = new Object();
                         this.textDataArray[textArrayCount].x = displayX;
                         this.textDataArray[textArrayCount].y = theDescription.text[count].y + textOffsetY;
                         this.textDataArray[textArrayCount].txt = "";
                         this.textDataArray[textArrayCount].fillStyle= theDescription.text[count].colour;
                         this.ctx.font = "italic"+" "+theDescription.text[count].fontSize+" "+theDescription.text[count].fontFamily;
                         this.textDataArray[textArrayCount].font = "italic"+" "+theDescription.text[count].fontSize+" "+theDescription.text[count].fontFamily;

                         splitStingIndex++;

                     }
                     else if (splitString[splitStingIndex]==="</end>")
                     {
                         this.ctx.font = theDescription.text[count].fontStyle+" "+theDescription.text[count].fontSize+" "+theDescription.text[count].fontFamily;
                       
                         textArrayCount++;
                         
                         this.textDataArray[textArrayCount] = new Object();
                         this.textDataArray[textArrayCount].x = displayX;
                         this.textDataArray[textArrayCount].y = theDescription.text[count].y + textOffsetY;
                         this.textDataArray[textArrayCount].txt = "";
                         this.textDataArray[textArrayCount].fillStyle= theDescription.text[count].colour;
                         this.textDataArray[textArrayCount].font = theDescription.text[count].fontStyle+" "+theDescription.text[count].fontSize+" "+theDescription.text[count].fontFamily;
                   
                         
                         splitStingIndex++;

                     }
                     else 
                     {
                       
                        if (splitString[splitStingIndex]!=='' && splitString[splitStingIndex]!==' ')
                        {
                            textWidth =  this.ctx.measureText(" " + splitString[splitStingIndex], 0, 0).width;
                        }
                        else
                        {
                            textWidth =  this.ctx.measureText(splitString[splitStingIndex], 0, 0).width;
                        }
                        
                        lineWidth += textWidth;
                        
                        if (lineWidth<(theDescription.text[count].w) )
                        {
                            this.textDataArray[textArrayCount].txt += splitString[splitStingIndex] + " ";

                            displayX += textWidth;
                            splitStingIndex++;
                        }
                        else
                        {
                          break;
                        }
                      }
                     
                   


                   }
                   textArrayCount++;
                   textOffsetY+=theDescription.text[count].lineSpacing;
                 }
                }  
              
            return;
            
        };
        
      };
    
     /** 
     * @class navLinks
     * @extends OU.util.Activity
     * Class to handle the links on the activity 
     */
     OU.activity.InfoBoxMenu.prototype.navLinks = function ( params ) {
        this.InfoBoxMenu = params.InfoBoxMenu;
        this.events = params.events;
        this.data = params.data;
        this.ctx = params.ctx;
        this.id  = params.data.linkId;
        this.scaleFactor = params.scaleFactor;
        this.translate = params.translate;
        this.rollover  = params.rollover;
        this.hover     = false;
        this.tw        =0;       
        
        /**
        * render - renders the current link
        *
        * @param bHighlight  - (boolean) denotes where the link shoudl be highlighted or not
        * @param scaleFactor - (float)   the scale factor, used for checking x and y in the event handler
        * @param translate   - (object)  translation x and y (centering values), used for checking x and y in the event handler
        *
        * Todo 
        * move the scale factor and translate object into one object
        */
        OU.activity.InfoBoxMenu.prototype.navLinks.prototype.render = function (bHighlight, scaleFactor, translate) {
            
            var count, bHighlightColour=false;
    
            if (this.hover || bHighlight){bHighlightColour=true;}
            if (this.data.type!=="text"){
                this.InfoBoxMenu.drawObject(this.data,this.ctx,bHighlightColour);
            }
           
            for (count=this.data.label.length-1; count>=0; count--)
            {
              if ((bHighlightColour===true) && this.data.label[count].colourActive!==undefined)
              {
                this.ctx.fillStyle= this.data.label[count].colourActive; 
              }else{
                this.ctx.fillStyle= this.data.label[count].colour;
              }
              this.ctx.font = this.data.label[count].fontStyle+" "+this.data.label[count].fontSize+" "+this.data.label[count].fontFamily;
              this.ctx.fillText(this.data.label[count].text, this.data.label[count].x , this.data.label[count].y);
              
              
              this.tw = this.ctx.measureText(this.data.label[count].text).width;
              
              
            }
            
            this.scaleFactor = scaleFactor;
            this.translate = translate;
            return;
            
        };
              
         /**
        * EventClick - event handler for the link
        *
        * @param evt         - (object) event object
        * @param navLinks    - (object)  links object
        *
        * Todo 
        * <ul>
        *   <li>add a better bounds checking for polyshapes</li>
        *   <li>handle overlapping rectangles</li>
        *   <li>add hover state to the rectangles</li>
        * </ul>
        */
        OU.activity.InfoBoxMenu.prototype.navLinks.EventClick = function ( evt, navLinks ) {
          this.event = evt;
          this.navLinks = navLinks;
          this.isHit = function ( x, y, pressed ) {
           
           
            switch (this.navLinks.data.type)
            {
                case "text":
                {
                    
                    var linktX ,linktY ,linktW ,linktH;
                    linktX = (this.navLinks.data.label[0].x * this.navLinks.scaleFactor)+this.navLinks.translate.x;
                    linktY = ((this.navLinks.data.label[0].y-(this.navLinks.data.label[0].lineSpacing/2)) * this.navLinks.scaleFactor)+this.navLinks.translate.y;
                    linktW = this.navLinks.tw * this.navLinks.scaleFactor;
                    linktH = this.navLinks.data.label[0].lineSpacing * this.navLinks.scaleFactor;
                    
                   
                     if (pressed) {
                       if (((x>=linktX) && (x<=linktX+linktW)) && ((y>=linktY) && (y<=linktY+linktH)) ){
                            
                            this.navLinks.InfoBoxMenu.currentDescription=this.navLinks.id;
                            if (this.navLinks.InfoBoxMenu.data.fps<=0){this.navLinks.InfoBoxMenu.renderLoop();}
                      }
                    }
                    break;
                }
                case "rect":
                case "poly":
                case "rect-simple":
                {
                    var linkX ,linkY ,linkW ,linkH;
                    
                    linkX = (this.navLinks.data.x * this.navLinks.scaleFactor)+this.navLinks.translate.x;
                    linkY = (this.navLinks.data.y * this.navLinks.scaleFactor)+this.navLinks.translate.y;
                    linkW = this.navLinks.data.w * this.navLinks.scaleFactor;
                    linkH = this.navLinks.data.h * this.navLinks.scaleFactor;
                
                
                    if (pressed) {
                       if (((x>=linkX) && (x<=linkX+linkW)) && ((y>=linkY) && (y<=linkY+linkH)) ){
                            
                            this.navLinks.InfoBoxMenu.currentDescription=this.navLinks.id;
                            if (this.navLinks.InfoBoxMenu.data.fps<=0){this.navLinks.InfoBoxMenu.renderLoop();}
                      }
                    }
                    break;
                }
                case "circle":
                {
                    var linkCX ,linkCY ,linkR, pointA, pointB, pointC=0, linkCH;
                    
                    linkCX = (this.navLinks.data.cx * this.navLinks.scaleFactor)+this.navLinks.translate.x;
                    linkCY = (this.navLinks.data.cy * this.navLinks.scaleFactor)+this.navLinks.translate.y;
                    linkR  = (this.navLinks.data.r * this.navLinks.scaleFactor);
                    linkCH  = (this.navLinks.data.ch * this.navLinks.scaleFactor);
                    pointC = (linkR-linkCH)*(linkR-linkCH);
                  
                    pointA = ((x-linkCX)*(x-linkCX)) +((y - linkCY)*(y - linkCY));
                    pointB = linkR*linkR;
                   
                
                    if (pressed) {
                       // Is the point within the circle
                       if ( pointA < pointB && pointA > pointC)
                       {
                            this.navLinks.InfoBoxMenu.currentDescription=this.navLinks.id;
                            if (this.navLinks.InfoBoxMenu.data.fps<=0){this.navLinks.InfoBoxMenu.renderLoop();}
                       }
                    }else if (this.navLinks.rollover && this.navLinks.data.ch){
                                          
                       if ( pointA < pointB && pointA > pointC ){
                            this.navLinks.hover = true;
                       }else{
                            this.navLinks.hover = false;                 
                       }
                    }
                    break;
                }
                default:
                    console.log("Error link type not handled");
            }
           }
        };
        this.events.clickable.push(new OU.activity.InfoBoxMenu.prototype.navLinks.EventClick(this.events, this)); // push event click area
      };
  
    /**
     *  keyPress - handles user key presses
     */
    OU.activity.InfoBoxMenu.prototype.keyPress = function ( keyCode ) {
            
        if ( keyCode === 87 || keyCode === 38) // UP arrow and W
            {this.currentLinkIndex--;
                if (this.currentLinkIndex <0){ 
                    this.currentLinkIndex=this.data.menuItems.length-1;
                }
            }
        else if ( keyCode === 83 || keyCode === 40) // DOWN arrow and S
            {
                this.currentLinkIndex++;
                if (this.currentLinkIndex >= this.data.menuItems.length)
                {
                    this.currentLinkIndex=0;
                }
            }
        if (this.currentLinkIndex<0 || this.currentLinkIndex>=this.data.menuItems.length )
            {this.currentLinkIndex=0;}
        
        this.currentDescription = this.data.menuItems[this.currentLinkIndex].linkId;

        this.render();
        
    };
    /**
     * accessibleView - this function is called instead of canvasView if the browser does not support HTML5 canvas
     */
  OU.activity.InfoBoxMenu.prototype.accessibleView = function() {
        var h;
        clearInterval(this.renderCycle);
        h = '<div id="accessibleView">';
        h += '<h1>' + this.data.title + '</h1>';
        h += '<p>' + this.data.accessibleDescription + '</p>';
        h += '<p></p>';
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.InfoBoxMenu, OU.util.Activity);