/**
 * @fileOverview Virtual Microscope - HTML5 version
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Slider');
OU.require('OU.util.Layer');
OU.require('OU.util.TileViewer');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Microscope = function(data, instance, controller) {
    OU.activity.Microscope.prototype.canvasView = function() {
        var self = this, bH = OU.controlHeight, aspectRatio, thumbnail, thumbWidth, thumbHeight;

        OU.VM = this; // set a global var, so outside buttons can access functions directly

        // Forwards/backwards compatibility with the FL framework data structure
        // microscope expects the old structure, so move data elements to top level of this.data
        if (this.data.activityOptions) {
            for (var key in this.data.activityOptions) {
                this.data[key] = this.data.activityOptions[key];
            }
        }
        if (this.data.activityContent) {
            this.data.datasets = this.data.activityContent;
        }

        // create Canvas Layers & Contexts
        this.bgLayer = new OU.util.Layer({
            container: this,
            id: 'bg'
        });
        if (this.data.backgroundColour !== undefined) {
            this.bgLayer.context.gradRect({
                col1: this.data.backgroundColour,
                col2: this.data.backgroundColour
            }); // use specified background colour
        }
        else {
            this.bgLayer.context.gradRect(); // use default background
        }
        this.scopeLayer = new OU.util.Layer({
            container: this,
            id: 'canvas',
            h: this.h, // - bH,
            hasEvents: true,
            pinch: this.pinch,
            pinchMe: this,
            keyPress: function(key) {
                self.keypress(key);
            }
        });

        this.controlsBackLayer = new OU.util.Layer({
            container: this,
            id: 'oucontrols',
            y: this.h - bH,
            h: bH,
            hasEvents: true
        });
        OU.events.addKeyListener(function() {
        });

        this.controlsLayer = new OU.util.Layer({
            container: this,
            id: 'oucontrols',
            y: this.h - bH * 1.5,
            h: bH,
            hasEvents: true
        });
        this.inRotationView = false;
        this.showRulers = this.data.showRulers === undefined ? true : this.data.showRulers;

        if (this.data.datasets !== undefined) { // if has datasets, then use multiple specimen config
            var URLVars = OU.getUrlVars();
            if (URLVars['d']) {
                this.specimen = URLVars['d'] | 0;
            }
            else {
                this.specimen = 0;
            }
            this.imageBase = this.data.datasets[this.specimen];
            this.setupDatasetsDiv();
        }
        else { // if no datasets specified in data, then use original single specimen config
            this.imageBase = this.data;
        }
        if (this.data.thumbnail) {
            aspectRatio = this.imageBase.width / this.imageBase.height;
            if (aspectRatio > 1) { // landscape
                thumbWidth = this.imageBase.tileSize + 2;
                thumbHeight = (thumbWidth / aspectRatio + 2) | 0;
            }
            else { // portrait
                thumbHeight = this.imageBase.tileSize + 2;
                thumbWidth = (thumbHeight * aspectRatio + 2) | 0;
            }
            if (this.data.externalThumbnail) {
                thumbWidth = this.data.externalThumbnailWidth;
                thumbHeight = (thumbWidth / aspectRatio + 2) | 0;
                var canvas = document.getElementById(this.data.externalThumbnail);
                canvas.setAttribute("height", thumbHeight);
                canvas.setAttribute("width", thumbWidth);
                if (canvas) {
                    this.thumbLayer = new OU.util.Layer({
                        container: this,
                        canvas: canvas,
                        hasEvents: true
                    });
                }
            }
            else {
                this.thumbLayer = new OU.util.Layer({
                    container: this,
                    id: 'thumbnail',
                    x: this.w - thumbWidth,
                    w: thumbWidth,
                    y: this.h - bH - thumbHeight,
                    h: thumbHeight,
                    hasEvents: true
                });
            }
            thumbnail = {
                layer: this.thumbLayer,
                x: 1,
                y: 1,
                w: thumbWidth - 1,
                h: thumbHeight - 1
            };

        }
        this.specimenDescription();
        this.tileView = new OU.util.TileViewer({
            image: {
                w: this.imageBase.width,
                h: this.imageBase.height
            },
            tileSize: this.imageBase.tileSize,
            useTileGroups: this.data.useTileGroups,
            imageBase: this.imageBase.images,
            rotationImage: this.data.rotationImage,
            numberMarkers: true,
            markerCallback: function(rot) {
                self.loadRotation(rot);
            },
            zoomLevels: this.imageBase.zoomLevels,
            renderAngle: this.data.renderAngle,
            showRulers: this.showRulers,
            showCrosshairs: this.data.showCrosshairs,
            centreReadout: this.data.centreReadout,
            noEdgeSnap: this.data.noEdgeSnap,
            window: {
                w: this.w,
                h: this.h //  - bH
            },
            container: this,
            context: this.scopeLayer.context,
            zoomCallback: function(z) {
                self.setSlider(z);
            },
            thumbnail: thumbnail
        });
        this.scopeLayer.events.clickable.push(this.tileView);
        this.initControls();

        // Register the activity's Messenger API
        // pass an array of Parameters which define the API
        // each Parameter can have a get, set and monitor function
        this.defineActivityAPI([
            {
                name: "PositionPixels",
                getFunction: function() {
                    var position = self.tileView.getPositionPixels();
                    return {
                        x: position.x,
                        y: position.y,
                        zoom: position.zoom
                    };
                },
                setFunction: function(newPos) {
                    self.tileView.setPosition({
                        zoom: newPos.zoom,
                        xPixels: newPos.x,
                        yPixels: newPos.y
                    });
                    self.zoomSlider.sliderPos = newPos.zoom;
                    self.zoomSlider.render();
                },
                monitor: function(params) {
                    self.tileView.monitorPositionPixels = params.callback;
                }
            },
            {
                name: "PositionMM",
                getFunction: function() {
                    var position = self.tileView.getPositionMM();
                    return {
                        x: position.x,
                        y: position.y,
                        zoom: position.zoom
                    };
                },
                setFunction: function(newPos) {
                    self.tileView.setPosition({
                        zoom: newPos.zoom,
                        xMM: newPos.x,
                        yMM: newPos.y
                    });
                    self.zoomSlider.sliderPos = newPos.zoom;
                    self.zoomSlider.render();
                }
            },
            {
                name: "MeasureMM",
                getFunction: function() {
                    return self.tileView.getMeasureMM();
                },
                setFunction: function(newPos) {
                    self.setMeasure(newPos.state);
                    self.tileView.setMeasureMM(newPos);
                },
                monitor: function(params) {
                    self.tileView.monitorMeasureMM = params.callback;
                }
            },
            {
                name: "Slide",
                getFunction: function() {
                    return self.tileView.imageIdx;
                },
                setFunction: function(newSlide) {
                    self.tileView.changeImage(newSlide);
                    self.tileView.render();
                    self.renderControls();
                }
            },
            {
                name: "FeatureState",
                getFunction: function() {
                    return {
                        rulers: self.showRulers,
                        measure: self.tileView.measureOn
                    };
                },
                setFunction: function(params) {
                    var features = {};

                    if (params.rulers !== undefined) {
                        features.showRulers = self.showRulers = params.rulers;
                        self.tileView.setFeatures(features);
                        return;
                    }
                    if (params.measure === true) {
                        self.setMeasure('measure');
                    }
                    else if (params.angle === true) {
                        self.setMeasure('angle');
                    }
                    else {
                        self.setMeasure('off');
                    }
                    self.renderControls();
                }
            },
            {
                name: "viewURL",
                getFunction: function() {
                    return self.shareURL() + self.tileView.getDirectParams();
                }
            },
            {
                name: "Snapshot",
                getFunction: function(params) {
                    return self.scopeLayer.canvas.toDataURL(params.format, params.quality);
                }
            }
        ]);
        var URLVars = OU.getUrlVars();
        if (URLVars['rot'] > 0) {
            if (URLVars['deg']) {
                this._rotationStart = URLVars['deg'];
            }
            var rot = this.data.images[URLVars['s'] || 0].rotations[URLVars['rot'] - 1];
            rot.arrayIndex = URLVars['rot'] - 1;
            this.loadRotation(rot);
        }
        if (window.parent && window.parent.postMessage && window.parent !== window) {
            window.parent.postMessage(JSON.stringify({
                msg: "VM ready",
                width: this.w,
                height: this.h
            }), "*");
        }
        else if (window.top && window.top.postMessage && window.top !== window) {
            window.top.postMessage(JSON.stringify({
                msg: "VM ready",
                width: this.w,
                height: this.h
            }), "*");
        }
    };
    OU.activity.Microscope.prototype.remove = function() {
        OU.activity.Microscope.superClass_.remove.call(this);
        if (this.tileView) {
            this.tileView.remove();
        }
        if (this.tabbableSettings) {
            this.tabbableSettings.remove();
        }
    };
    OU.activity.Microscope.prototype.render = function() {
        this.tileView.doRender = true;
        if (this.settingsMenuDiv) {
            this.renderSettingsMenu();
        }
    };
    OU.activity.Microscope.prototype.gridColour = function(col) {
        if (col) {
            this.tileView.gridColour = col;
            this.render();
        }
        return false;
    };
    OU.activity.Microscope.prototype.gotoXY = function() {
        var x = document.getElementById('gotoX'),
                y = document.getElementById('gotoY');
        if (x && y) {
            this.tileView.setPosition({
                xMM: x.value,
                yMM: y.value
            });
        }
        return false;
    };
    OU.activity.Microscope.prototype.nudgeLeft = function() {
        this.tileView.nudge('left');
        return false;
    };
    OU.activity.Microscope.prototype.nudgeRight = function() {
        this.tileView.nudge('right');
        return false;
    };
    OU.activity.Microscope.prototype.nudgeUp = function() {
        this.tileView.nudge('up');
        return false;
    };
    OU.activity.Microscope.prototype.nudgeDown = function() {
        this.tileView.nudge('down');
        return false;
    };
    OU.activity.Microscope.prototype.gotoXYPixels = function() {
        var x = document.getElementById('gotoXpx'),
                y = document.getElementById('gotoYpx');
        if (x && y) {
            this.tileView.setPosition({
                xPixels: x.value,
                yPixels: y.value
            });
        }
        return false;
    };
    OU.activity.Microscope.prototype.resize = function() {
        OU.activity.Microscope.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, i, ctx = this.scopeLayer.context,
                aspectRatio, w, h,
                rightButtonsWidth = 0,
                leftButtonsWidth = 0;

        if (this._sharePopUp)
            this._sharePopUp.remove();
        this.bgLayer.resize();
        if (this.data.backgroundColour !== undefined) {
            this.bgLayer.context.gradRect({
                col1: this.data.backgroundColour,
                col2: this.data.backgroundColour
            }); // use specified background colour
        }
        else {
            this.bgLayer.context.gradRect(); // use default background
        }
        this.scopeLayer.resize({
            h: this.h // - bH
        });
        ctx.font = '18px ' + OU.theme.font;
        ctx.lineWidth = 2;
        ctx.strokeStyle = '#c00';
        ctx.fillStyle = '#c00';
        ctx.textAlign = 'center';

        if (this.data.thumbnail) {
            aspectRatio = this.imageBase.width / this.imageBase.height;
            if (aspectRatio > 1) {
                w = this.imageBase.tileSize + 2;
                h = (w / aspectRatio + 2) | 0;
            }
            else {
                h = this.imageBase.tileSize + 2;
                w = (h * aspectRatio + 2) | 0;
            }
            this.thumbLayer.resize({
                x: this.w - w,
                w: w,
                y: this.h - bH - h,
                h: h
            });
        }

        this.controlsBackLayer.resize({
            y: this.h - bH,
            h: bH
        });
        this.controlsLayer.resize({
            y: this.h - bH * 1.5,
            h: bH
        });
        this.tileView.resize({
            /* optionally resize the thumbnail - for testing
             thumbnail:{
             x:1,
             y:1,
             w:128,
             h:128
             }, //*/
            w: this.w,
            h: this.h - bH
        });
        for (i = 0; i < this.imageBase.images.length; i++) {
            if (this.viewButtons && this.viewButtons[i]) {
                this.viewButtons[i].resize({
                    x: i * bH * 2,
                    y: 0,
                    w: bH * 2,
                    h: bH
                });
                this.viewButtons[i].render();
            }
        }
        if (this.measureButton) {
            rightButtonsWidth = rightButtonsWidth + bH * 3;
            this.measureButton.resize({
                x: this.w - rightButtonsWidth,
                y: 0,
                w: bH * 3,
                h: bH
            });
            this.measureButton.render();
        }
        if (this.data.noShare === undefined && this.shareButton) {
            rightButtonsWidth = rightButtonsWidth + bH * 3;
            this.shareButton.resize({
                x: this.w - rightButtonsWidth,
                y: 0,
                w: bH * 3,
                h: bH
            });
            this.shareButton.render();
        }
        if (this.settingsButtonDiv) {
            rightButtonsWidth = rightButtonsWidth + bH * 2;
            this.settingsButtonDiv.resize({
                x: this.w - rightButtonsWidth,
                y: this.y + this.h - bH * 1.5,
                w: bH * 2,
                h: bH * .8
            });
        }
        if (this.settingsMenuDiv) {
            this.settingsMenuDiv.resize({
                x: this.x + this.w - 300,
                y: 0,
                w: 290,
                h: this.h - bH * 2 - 10
            });
            this.settingsMenuDiv.div.style.opacity = 1;
        }
        this.zoomSlider.resize({
            x: leftButtonsWidth,
            y: 0,
            w: this.w - leftButtonsWidth - rightButtonsWidth,
            h: bH
        });
        this.tileView.render();
        if (this.inRotationView) {
            this.loadedRotation.remove();
            this.loadRotation(this.openRotation);
        }
    };
    OU.activity.Microscope.prototype.initControls = function() {
        var i, ctx = this.controlsLayer.context, bH = OU.controlHeight, self = this,
                URLVars = OU.getUrlVars(), slideNum = URLVars['s'] || 0,
                initZoom = URLVars['zoom'] === undefined ? 0 : parseFloat(URLVars['zoom']),
                clickable = this.controlsLayer.events.clickable,
                rightButtonsWidth = 0,
                leftButtonsWidth = this.imageBase.images.length * bH * 2,
                config = this.data || {};

        this.measureSize = new OU.util.DynText({
            txt: 'Measure',
            x: this.w - rightButtonsWidth,
            y: 0,
            w: bH * 3 - (24 * OU.dpr + 20), // text will be button width, minus icon width&padding
            h: bH,
            context: this.controlsLayer.context,
            measureOnly: true
        });
        this.buttonFontSize = this.measureSize.font.size;
        clickable.length = 0;
        this.viewButtons = [];
        if (config.viewButtons !== false) {
            for (i = 0; i < this.imageBase.length; i++) {
                this.viewButtons[i] = new OU.util.CheckBoxButton({
                    txt: this.imageBase.images[i].label,
                    x: i * bH * 2,
                    y: 0,
                    w: bH * 2,
                    h: bH,
                    layer: this.controlsLayer,
                    fontSize: this.buttonFontSize,
                    onClick: function(p) {
                        self.changeImage(p);
                    },
                    onClickParam: i,
                    state: i === slideNum ? true : false
                });
            }
            rightButtonsWidth = rightButtonsWidth + bH * 3;
            this.measureButton = new OU.util.CheckBoxButton({
                txt: 'Measure',
                x: this.w - rightButtonsWidth,
                y: 0,
                w: bH * 3,
                h: bH,
                layer: this.controlsLayer,
                fontSize: this.buttonFontSize,
                state: false,
                onClick: function() {
                    self.toggleMeasure();
                }
            });
            this.tileView.measureOn = false;
            if (this.data.noShare === undefined) {
                rightButtonsWidth = rightButtonsWidth + bH * 3;
                this.shareButton = new OU.util.ControlButton({
                    txt: 'Share',
                    x: this.w - rightButtonsWidth,
                    y: bH * .1,
                    w: bH * 3,
                    h: bH * .8,
                    fontSize: this.buttonFontSize,
                    layer: this.controlsLayer,
                    onClick: function() {
                        self.share();
                    }
                });
            }
        }
        else {
            rightButtonsWidth = 0;
            leftButtonsWidth = 0;
        }
        if (this.data.settingsButton) {
            rightButtonsWidth = rightButtonsWidth + bH * 2;

            this.settingsButtonDiv = new OU.util.Div({
                x: this.w - rightButtonsWidth,
                y: this.y + this.h - bH * 1.5,
                w: bH * 2,
                h: bH * .8,
                htmlClass: "VMSettingsButton fastease"
            });
            OU.events.addListener(this.settingsButtonDiv.div, function() {
                self.settingsMenu();
            }, "tap");
            this.tabbableSettings = new this.SettingsTabbable({});

        }
        this.zoomSlider = new OU.util.Slider({
            container: this,
            sliderPos: initZoom,
            instance: 'zoom',
            x: leftButtonsWidth,
            y: 0,
            w: this.w - leftButtonsWidth - rightButtonsWidth,
            h: bH,
            sliderHeight: bH / 2,
            drawContainer: false,
            callback: function(val) {
                self.setZoom(val);
            },
            background: {
                clear: true
            },
            context: ctx
        });
        clickable.push(this.zoomSlider);
        this.renderSlider();
    };

    OU.activity.Microscope.prototype.SettingsTabbable = function(params) {
        console.log('settings Tabbable');
        OU.activity.Microscope.prototype.SettingsTabbable.prototype.focus = function() {
            OU.VM.settingsButtonDiv.addClass('tabbed');
        };
        OU.activity.Microscope.prototype.SettingsTabbable.prototype.blur = function() {
            OU.VM.settingsButtonDiv.removeClass('tabbed');
        };
        OU.activity.Microscope.prototype.SettingsTabbable.prototype.hit = function() {
            OU.VM.settingsMenu();
        };
        OU.base(this, params);
    };
    OU.inherits(OU.activity.Microscope.prototype.SettingsTabbable, OU.util.Tabbable);

    OU.activity.Microscope.prototype.settingsMenu = function() {
        var self = this;
        if (this.settingsMenuDiv) {
            this.settingsMenuDiv.div.style.opacity = 0;
            setTimeout(function() {
                if (self.settingsMenuDiv) {
                    self.settingsMenuDiv.remove();
                    self.settingsMenuDiv = null;
                }
            }, 1000);
        }
        else {
            this.settingsMenuDiv = new OU.util.Div({
                x: this.x + this.w - 300,
                y: 10,
                w: 290,
                h: this.h - OU.controlHeight * 2 - 10,
                htmlClass: "settingsPanel"
            });
            setTimeout(function() { // sight delay otherwise the transition doesn't work
                self.settingsMenuDiv.div.style.opacity = 1;
            }, 2);
            this.renderSettingsMenu();
        }
    };
    OU.activity.Microscope.prototype.renderSettingsMenu = function() {
        var numLights = this.imageBase.images.length,
                tv = this.tileView,
                currentLight = tv.imageIdx,
                rulers = tv.showRulers,
                gridType = tv.showGrid ? 1 : (tv.showGraticule ? 2 : 0),
                gridAngle = (180 * tv.gridAngle / Math.PI) | 0;
        var html = "<table class='settingsTable'><tr><th colspan='2'>Light <a tabindex='1' class='settingsClose' id='_settingsClose'></a></th></tr>";

        if (numLights === 3) { // PPL, XPL & REF
            html += "<tr><td>Plane-polarised</td><td><a tabIndex='1' id='_image1' class='radiotop" + (currentLight === 0 ? ' checked' : '') + "'></a></td>"
                    + "<tr><td>Crossed polars</td><td><a tabIndex='2' id='_image2' class='radiomiddle" + (currentLight === 1 ? ' checked' : '') + "'></a></td>"
                    + "<tr><td>Reflected</td><td><a tabIndex='3' id='_image3' class='radiobottom" + (currentLight === 2 ? ' checked' : '') + "'></a></td>";
        }
        else { // 2 PPL & XPL
            html += "<tr ><td>Plane-polarised</td><td><a tabindex='1' id='_image1' class='radiotop" + (currentLight === 0 ? ' checked' : '') + "'></a></td>"
                    + "<tr ><td>Crossed polars</td><td><a tabindex='2' id='_image2' class='radiobottom" + (currentLight === 1 ? ' checked' : '') + "'></a></td>";
        }

        html += "<tr><th colspan='2'>Measurements</th>"
                + "<tr><td>None</td><td><a tabIndex='4' id='_noGrid' class='radiotop" + (gridType === 0 ? ' checked' : '') + "'></a></td>"
//                + "<tr><td>Grid</td><td><a tabIndex='5' id='_grid' class='radiomiddle" + (gridType === 1 ? ' checked' : '') + "'></a></td>"
                + "<tr><td>10mm Graticule</td><td><a tabIndex='6' id='_graticule' class='radiobottom" + (gridType === 2 ? ' checked' : '') + "'></a></td>";

        // overlay colour and rotation options - comment out entire section to remove
        if (gridType > 0) {
            html += "<tr><td colspan=2><div><p>Rotate grid/graticule</p><p><input tabindex='7' type='button' class='rotateRight' id='_rotRight'/><input tabindex='7' type='button' class='rotateLeft' id='_rotLeft'/>&nbsp;&nbsp;<span id='gridAngle'>" + gridAngle + "</span>&deg;</p>"
                    + "<br clear='left'/>Overlay Colour: <br clear='left'/>"
                    + "<p><span tabindex='10' style='background:#c00;' id='_colc00'>&nbsp;</span>&nbsp;"
                    + "<span tabindex='11' style='background:#0c0;' id='_col0c0'>&nbsp;</span>&nbsp;"
                    + "<span tabindex='12' style='background:#00c;' id='_col00c'>&nbsp;</span>&nbsp;"
                    + "<span tabindex='13' style='background:#0cc;' id='_col0cc'>&nbsp;</span>&nbsp;"
                    + "<span tabindex='14' style='background:#cc0;' id='_colcc0'>&nbsp;</span>&nbsp;"
                    + "</p></div></td></tr>";
        }

        html += "<tr><td>Rulers?</td><td><a tabIndex='16' id='_rulers' class='checkbox" + (rulers ? ' checked' : '') + "'></a></td>";
        this.settingsMenuDiv.div.innerHTML = html;
        // attach listeners
        var obj = document.getElementById('_settingsClose');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.settingsMenu();
            }, 'tap');
        }
        obj = document.getElementById('_image1');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.changeImage(0);
            }, 'tap');
        }
        obj = document.getElementById('_image2');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.changeImage(1);
            }, 'tap');
        }
        obj = document.getElementById('_image3');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.changeImage(2);
            }, 'tap');
        }
        obj = document.getElementById('_noGrid');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.toggleGrid('none');
            }, 'tap');
        }
        obj = document.getElementById('_grid');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.toggleGrid('grid');
            }, 'tap');
        }
        obj = document.getElementById('_graticule');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.toggleGrid('graticule');
            }, 'tap');
        }
        obj = document.getElementById('_rotRight');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.rotateGridRight();
            }, 'tap');
        }
        obj = document.getElementById('_rotLeft');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.rotateGridLeft();
            }, 'tap');
        }
        obj = document.getElementById('_colc00');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.gridColour("#c00");
            }, 'tap');
        }
        obj = document.getElementById('_col0c0');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.gridColour("#0c0");
            }, 'tap');
        }
        obj = document.getElementById('_col00c');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.gridColour("#00c");
            }, 'tap');
        }
        obj = document.getElementById('_colcc0');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.gridColour("#cc0");
            }, 'tap');
        }
        obj = document.getElementById('_col0cc');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.gridColour("#0cc");
            }, 'tap');
        }

        obj = document.getElementById('_rulers');
        if (obj) {
            OU.events.addListener(obj, function() {
                OU.VM.toggleRulers();
            }, 'tap');
        }
    };
    OU.activity.Microscope.prototype.changeSpecimenAction = function() {
        var specimenChoice = document.getElementById('specimenChoices'),
                specimen = specimenChoice.value;
        if (specimen !== -1) {
            this.changeSpecimen(specimen);
        }
    };
    OU.activity.Microscope.prototype.setupDatasetsDiv = function() {
        var i, ds, datasets = this.data.datasets, html = '<select onchange="OU.VM.changeSpecimenAction();" id="specimenChoices">',
                specimenDiv = document.getElementById('specimens');
        if (!specimenDiv) {
            return;
        }

        for (i = 0; i < datasets.length; i++) {
            ds = datasets[i];
            if (i === this.specimen) {
                html = html + '<option value="' + i + '" selected="selected">' + ds.name + '</option>';
            }
            else {
                html = html + '<option value="' + i + '">' + ds.name + '</option>';
            }

        }
        html = html + "</select>";
        specimenDiv.innerHTML = html;

    };
    OU.activity.Microscope.prototype.changeSpecimen = function(specimenIdx) {
        this.specimen = specimenIdx;
        this.imageBase = this.data.datasets[this.specimen];
        this.tileView.newDims({
            w: this.imageBase.width,
            h: this.imageBase.height});
        this.tileView.nZ = this.imageBase.zoomLevels;
        this.tileView.changeDataset(this.imageBase.images);
        this.zoomSlider.setTarget(0);
        this.specimenDescription();
    };
    OU.activity.Microscope.prototype.specimenDescription = function() {
        var descDiv = document.getElementById('vmDescription');
        if (descDiv) {
            descDiv.innerHTML = this.imageBase.description;
        }
    };
    OU.activity.Microscope.prototype.changeImage = function(idx) {
        var i;
        this.tileView.changeImage(idx);
        for (i = 0; i < this.viewButtons.length; i++) {
            this.viewButtons[i].state(i === idx ? true : false);
        }
        this.render();
        this.renderControls();
    };
    OU.activity.Microscope.prototype.shareURL = function() {
        var match = /(.+:\/\/.+?\/).+\/html5Assets\/(.+)\//i.exec(window.location.href);
        if (match) { // found JISC project URL format
            return match[1] + 'rock_sample?asset=' + match[2] + '/index.html?';
        }
        match = /.+\??/i.exec(window.location.href);
        if (match) // strip any old parameters
            return match[0] + '?';
        // otherwise just return current URL
        return window.location.href + '?';
    };
    OU.activity.Microscope.prototype.share = function(rotationInfo) {
        var self = this, h, view = this.tileView.getDirectParams(), link;
        if (rotationInfo)
            view += rotationInfo;
        h = '<p>Share the current view using the link below.</p><p>'
                + '<input style="width:80%" id="shareLink" value="' + this.shareURL() + view + '"/>';
        this._sharePopUp = new OU.util.PopUpInfo({
            container: this,
            x: this.w * .1,
            y: 50,
            w: this.w * .5,
            h: 200,
            txt: h,
            zIndex: 9200,
            onClose: function() {
                self._sharePopUp = null;
            }
        });
        link = document.getElementById('shareLink');
        link.select();
    };
    OU.activity.Microscope.prototype.toggleRulers = function() {
        this.tileView.showRulers = !this.tileView.showRulers;
        this.render();
    };
    OU.activity.Microscope.prototype.toggleGrid = function(state) {
        var gridDetailDiv = document.getElementById('gridDetails'),
                chondruleDetailsDiv = document.getElementById('chondruleDetails');
        switch (state) {
            case 'none':
                this.tileView.setFeatures({
                    showGrid: false,
                    showGraticule: false,
                    measureChondrules: false
                });
                if (gridDetailDiv) {
                    gridDetailDiv.setAttribute('style', 'display:none');
                }
                if (chondruleDetailsDiv) {
                    chondruleDetailsDiv.setAttribute('style', 'display:none');
                }
                break;
            case 'grid':
                if (gridDetailDiv) {
                    gridDetailDiv.setAttribute('style', 'display:block');
                }
                if (chondruleDetailsDiv) {
                    chondruleDetailsDiv.setAttribute('style', 'display:none');
                }
                this.tileView.setFeatures({
                    showGrid: true,
                    showGraticule: false,
                    measureChondrules: false
                });
                break;
            case 'graticule':
                if (gridDetailDiv) {
                    gridDetailDiv.setAttribute('style', 'display:block');
                }
                if (chondruleDetailsDiv) {
                    chondruleDetailsDiv.setAttribute('style', 'display:none');
                }
                this.tileView.setFeatures({
                    showGrid: false,
                    showGraticule: true,
                    measureChondrules: false
                });
                break;
            case 'chondrule':
                if (gridDetailDiv) {
                    gridDetailDiv.setAttribute('style', 'display:none');
                }
                if (chondruleDetailsDiv) {
                    chondruleDetailsDiv.setAttribute('style', 'display:block');
                }
                this.tileView.setFeatures({
                    showGrid: false,
                    showGraticule: false,
                    measureChondrules: true
                });
                break;
        }
        this.render();
    };
    OU.activity.Microscope.prototype.undoChondrules = function() {
        this.tileView.undoChondrules();
        this.render();
        return false;
    };
    OU.activity.Microscope.prototype.clearChondrules = function() {
        this.tileView.clearChondrules();
        this.render();
        return false;
    };
    OU.activity.Microscope.prototype.exportChondrules = function() {
        this.tileView.exportChondrules();
        return false;
    };
    OU.activity.Microscope.prototype.rotateGridLeft = function() {
        this.rotateGrid(-Math.PI / 180);
    };
    OU.activity.Microscope.prototype.rotateGridRight = function() {
        this.rotateGrid(Math.PI / 180);
    };
    OU.activity.Microscope.prototype.rotateGrid = function(angle) {
        var angleDisplay = document.getElementById('gridAngle');
        this.tileView.gridAngle = this.tileView.gridAngle + angle;
        this.render();
        if (angleDisplay) {
            angleDisplay.innerHTML = (180 * this.tileView.gridAngle / Math.PI) | 0;
        }
    };
    OU.activity.Microscope.prototype.toggleMeasure = function() {
        switch (this._measureState) {
            default:
            case 'off':
                this._measureState = 'measure';
                this.tileView.measureOn = true;
                this.tileView.vectorAngles = false;
                this.measureButton.state(true);
                this.measureButton.modify({txt: "Measure"});
                break;
            case 'measure':
                this._measureState = 'angle';
                this.tileView.measureOn = true;
                this.tileView.vectorAngles = true;
                this.measureButton.state(true);
                this.measureButton.modify({txt: "Angle"});
                break;
            case 'angle':
                this._measureState = 'off';
                this.tileView.measureOn = false;
                this.measureButton.state(false);
                this.measureButton.modify({txt: "Measure"});
                break;
        }
        this.renderControls();
        this.render();
    };
    OU.activity.Microscope.prototype.setMeasure = function(state) {
        switch (state) {
            default:
            case 'measure':
                this._measureState = 'measure';
                this.tileView.measureOn = true;
                this.tileView.vectorAngles = false;
                this.measureButton.state(true);
                this.measureButton.modify({txt: "Measure"});
                break;
            case 'angle':
                this._measureState = 'angle';
                this.tileView.measureOn = true;
                this.tileView.vectorAngles = true;
                this.measureButton.state(true);
                this.measureButton.modify({txt: "Angle"});
                break;
            case 'off':
                this._measureState = 'off';
                this.tileView.measureOn = false;
                this.measureButton.state(false);
                this.measureButton.modify({txt: "Measure"});
                break;
        }
        this.renderControls();
        this.render();
    };
    OU.activity.Microscope.prototype.renderControls = function() {
        var i;
        this.controlsLayer.context.clearRect(0, 0, this.w, OU.controlHeight);
        if (this.measureButton) {
            this.measureButton.render();
        }
        if (this.shareButton) {
            this.shareButton.render();
        }
        for (i = this.viewButtons.length; i--; ) {
            if (this.viewButtons[i]) {
                this.viewButtons[i].render();
            }
        }
        this.renderSlider();
    };
    OU.activity.Microscope.prototype.renderSlider = function() {
        var self = this;
        this.zoomSlider.render();
        setTimeout(function() {
            self.renderSlider();
        }, 40);
    };
    OU.activity.Microscope.prototype.setZoom = function(val) { // Called when scrubBar is moved
        this.zoomSlider.render();
        this.tileView.scale(val);
    };
    OU.activity.Microscope.prototype.setSlider = function(val) { // Called when TileViewer changed via mousewheel, etc.
        this.zoomSlider.setTarget(val);
        this.tileView.scale(val);
    };
    OU.activity.Microscope.prototype.keypress = function(keycode) {
        var ns = null;

//        console.log('keycode: ' + keycode);
        if (keycode === 27) { // Escape key
            if (this.loadedRotation) {
                this.loadedRotation.remove();
                this.loadedRotation = null;
            }
            if (this.settingsMenuDiv) {
                this.settingsMenuDiv.remove();
                this.settingsMenuDiv = null;
            }
        }
        // Load a rotation if a number key pressed
        if (document.activeElement.id.substr(0, 4) !== 'goto') { // ignore number keys if in one of the goto fields
            if (keycode >= 49 && keycode <= 57) {
                if (this.imageBase.images[this.tileView.imageIdx].rotations &&
                        this.imageBase.images[this.tileView.imageIdx].rotations[keycode - 49]) {
                    this.loadRotation(this.imageBase.images[this.tileView.imageIdx].rotations[keycode - 49]);
                }
            }
        }
        // Handle Arrow keys
        if (this.loadedRotation) {
            switch (keycode) {
                case 40: //down
                case 37: //left
                    this.loadedRotation.rotateCCW();
                    break;
                case 38: // up
                case 39: //right
                    this.loadedRotation.rotateCW();
                    break;
            }
        }
        else if (this.data.arrowKeysMoveView) {
            if (keycode === 40) // DOWN
                this.nudgeDown();
            if (keycode === 38) // UP
                this.nudgeUp();
            if (keycode === 37) // LEFT
                this.nudgeLeft();
            if (keycode === 39) // RIGHT
                this.nudgeRight();
        }
        else {
            if (keycode === 40) // DOWN
                this.tileView.blurRadius -= 5;
            if (keycode === 38) // UP
                this.tileView.blurRadius += 5;
            if (keycode === 37) // LEFT
                this.tileView.blurRadius -= 1;
            if (keycode === 39) // RIGHT
                this.tileView.blurRadius += 1;
        }
        if (keycode === 79) // o
            this.tileView.brightenFactor -= .1;
        if (keycode === 80) // p
            this.tileView.brightenFactor += .1;
        this.tileView.render();

        if (keycode === 65) { // 'A' - Zoom in
            ns = this.tileView.s + 0.1;
            ns = ns > 1 ? 1 : ns;
        }
        if (keycode === 90) { // 'Z' - Zoom out
            ns = this.tileView.s - 0.1;
            ns = ns < 0 ? 0 : ns;
        }
        if (ns) {
            this.setZoom(ns);
            this.zoomSlider.sliderPos = ns;
            this.zoomSlider.render();
        }

    };
    OU.activity.Microscope.prototype.pinch = function(end, start, x, y, dx, dy, me) { // called when pinch event detected
        var ns = ((end - start) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x: x,
            y: y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };
    OU.activity.Microscope.prototype.loadRotation = function(rot) {
        if (this.inRotationView)
            return;
        this.inRotationView = true;
        this.scopeLayer.events.touched = this.scopeLayer.events.pressed = false;
        this.openRotation = rot;
        rot.container = this;
        this.loadedRotation = new this.Rotation(rot);
    };
    /**
     * @class
     */
    OU.activity.Microscope.prototype.Rotation = function(params) {
        var rotation = this, bH = OU.controlHeight, sW;

        this.title = params.title || '';
        this.pplData = params.pplData || '';
        this.xplData = params.xplData || '';
        this.arrayIndex = params.arrayIndex || 0;
        this.container = params.container;
        this.w = this.container.controller.w;
        this.h = this.container.controller.h;
        this.fileExt = params.fileType || ".jpg";
        this.reverse = params.reverseRotation;
        this.noFade = params.noFade;
        this.scaleTxt = params.scale || '1mm';

        if (this.w > OU.dpr * 1024) {
            this.w = OU.dpr * 1024;
        }
        if (this.h > OU.dpr * 768) {
            this.h = OU.dpr * 768;
        }
        this.x = (this.container.controller.w - this.w) / 2;
        this.y = (this.container.controller.h - this.h) / 2;

        if (this.w > this.h) {
            sW = this.h * .95 < (this.w / 2) * .95 ? this.h * .95 : (this.w / 2) * .95;
            sW = sW > 460 ? 460 : sW;
            this.slideWidth = params.slideWidth || sW;
            this.slideHeight = params.slideHeight || sW;
            this.orientation = 'landscape';
            this.leftX = this.x + (this.w / 2 - this.slideWidth) / 2;
            this.rightX = this.w / 2 + this.leftX;
            this.lefttopY = this.righttopY = this.y + (this.h - bH - this.slideHeight) / 2;
        }
        else {
            sW = this.w * .95 < (this.h / 2) * .95 ? this.w * .95 : (this.h / 2) * .95;
            sW = sW > 460 ? 460 : sW;
            this.slideWidth = params.slideWidth || sW;
            this.slideHeight = params.slideHeight || sW;
            this.orientation = 'portrait';
            this.rightX = this.leftX = this.x + (this.w - this.slideWidth) / 2;
            this.lefttopY = this.y + (this.h / 2 - this.slideWidth) / 2;
            this.righttopY = this.y + this.h / 2 + (this.h / 2 - this.slideWidth) / 2;
        }
        this.leftCenterX = this.leftX + this.slideWidth / 2;
        this.rightCenterX = this.rightX + this.slideWidth / 2;
        this.leftcenterY = this.lefttopY + this.slideHeight / 2;
        this.rightcenterY = this.righttopY + this.slideHeight / 2;
        this.numberOfImagesInFullRotation = params.numberOfImagesInFullRotation || 72;
//        this.reuseCount = params.reuseCount || 2;
        if (OU.IS_IPAD || OU.IS_IPHONE) {
            this.reuseCount = 2;
        }
        else {
            this.reuseCount = 1;
        }
        this.degrees = ((this.container._rotationStart || 0) * Math.PI / 180) % (Math.PI * 2);
        /**
         * @private
         */
        OU.activity.Microscope.prototype.Rotation.prototype.init = function() {
            var i, index, imageLayer, ctx, self = this, bH = OU.controlHeight, scaleX, scaleY, scaleInPixels,
                    w = this.w,
                    h = this.h,
                    x = this.x,
                    y = this.y;
            this.modalLayer = new OU.util.Layer({// Background layer masks the microscope underneath
                zIndex: OU.POP_UP_LEVEL,
                hasEvents: true
            });
            this.modalLayer.events.clickable.push({
                isHit: function(x, y, state) {
                    if (state) {
                        self.remove();
                    }
                }
            });
            this.bgLayer = new OU.util.Layer({// Background layer masks the microscope underneath
                x: x,
                y: y,
                w: w,
                h: h,
                zIndex: OU.POP_UP_LEVEL + 1
            });
            ctx = this.modalLayer.context;
            ctx.fillStyle = 'rgba(0,0,0,0.7)';
            ctx.fillRect(0, 0, this.container.controller.w, this.container.controller.h);
            ctx.fillStyle = 'rgba(0,0,0,0.4)';
            ctx.roundRect(x + 8, y + 8, w, h, 20);
            ctx.fill();

            ctx = this.bgLayer.context;
            ctx.fillStyle = '#fff';
            ctx.roundRect(0, 0, w, h, 20);
            ctx.fill();
            // Create 2 DIVs to contain the left and right image sets
            this.leftBox = document.createElement("div"),
                    this.rightBox = document.createElement("div");
            this.leftBox.style.width = this.rightBox.style.width = this.slideWidth + "px";
            this.leftBox.style.height = this.rightBox.style.height = this.slideHeight + "px";
            this.leftBox.style.left = this.leftX + "px";
            this.rightBox.style.left = this.rightX + "px";
            this.leftBox.style.top = this.lefttopY + "px";
            this.rightBox.style.top = this.righttopY + "px";
            this.leftBox.style.zIndex = this.rightBox.style.zIndex = OU.POP_UP_LEVEL + 1;
            this.leftBox.className = this.rightBox.className = "microscopeRotation";
            var parentDiv = OU._screenWrapper ? OU._screenWrapper.div : document.body;

            parentDiv.appendChild(this.leftBox);
            parentDiv.appendChild(this.rightBox);
            // Load the images into the divs
            this.segmentAngle = 2.0 * Math.PI / this.numberOfImagesInFullRotation;
            for (i = 0; i < this.numberOfImagesInFullRotation / this.reuseCount; i++) {
                if (this.reverse) {
                    index = (this.numberOfImagesInFullRotation / this.reuseCount) - i;
                }
                else {
                    index = i + 1;
                }
                // Add PPL image
                imageLayer = document.createElement("img");
                imageLayer.style.visibility = "hidden";
                imageLayer.src = this.container.dataDir + this.pplData + ((index < 10) ? "0" : "") + index + this.fileExt;
                imageLayer.width = this.slideWidth;
                imageLayer.height = this.slideHeight;
                imageLayer.style.zIndex = OU.POP_UP_LEVEL + 1 + i;
                this.leftBox.appendChild(imageLayer);
                // Add XPL image
                imageLayer = document.createElement("img");
                imageLayer.style.visibility = "hidden";
                imageLayer.src = this.container.dataDir + this.xplData + ((index < 10) ? "0" : "") + index + this.fileExt;
                imageLayer.width = this.slideWidth;
                imageLayer.height = this.slideHeight;
                imageLayer.style.zIndex = OU.POP_UP_LEVEL + 1 + i;
                this.rightBox.appendChild(imageLayer);
            }
            // mask layer covers up the corners of data has not been rounded off
            this.maskLayer = new OU.util.Layer({
                x: x,
                y: y,
                w: w,
                h: h,
                zIndex: OU.POP_UP_LEVEL + this.numberOfImagesInFullRotation + 10
            });
            // Add a control layer to capture touch/mouse events & contain other visual elements
            this.controlLayer = new OU.util.Layer({
                x: x,
                y: y,
                w: w,
                h: h,
                zIndex: OU.POP_UP_LEVEL + this.numberOfImagesInFullRotation + 11,
                hasEvents: true
            });
            ctx = this.controlLayer.context;
            ctx.font = this.slideWidth * .05 + 'px ' + OU.theme.font;
            this.controlLayer.events.clickable.push(this);
            ctx.closeIcon({
                x: this.w - bH,
                y: bH / 2,
                r: bH / 2
            });
            this.backButton = new OU.util.InvisibleButton({
                x: this.w - bH * 2,
                y: 0,
                w: bH * 2,
                h: bH + 10,
                layer: this.controlLayer,
                onClick: function() {
                    rotation.close();
                }
            });
            if (this.container.data.viewButtons !== false) {
                this.shareButton = new OU.util.ControlButton({
                    txt: 'Share',
                    x: bH * 2.5,
                    y: h - bH,
                    w: bH * 2,
                    h: bH,
                    layer: this.controlLayer,
                    fontSize: this.buttonFontSize,
                    onClick: function() {
                        var degrees = (rotation.degrees / (Math.PI / 180)) | 0;
                        var rot = (rotation.arrayIndex || 0) + 1;
                        rotation.container.share('&rot=' + rot + "&deg=" + degrees);
                    }
                });
            }
            // Title
            ctx.textAlign = 'center';
            if (this.orientation === 'landscape') {
                ctx.fillStyle = 'rgba(256,256,256,0.65)';
                ctx.fillRect(0, h * .1 - 16, w, 32);
                ctx.fillStyle = '#222';
                ctx.fillText(this.title, w / 2, h * .05);
                ctx.fillText("plane polarised light", w / 4, h * .1);
                ctx.fillText("between crossed polars", w * .75, h * .1);
            }
            else {
                ctx.save();

                ctx.fillStyle = 'rgba(256,256,256,0.65)';
                ctx.fillRect(0, bH - 16, ctx.measureText("plane polarised light").width + 20, 32);
                ctx.fillRect(w - ctx.measureText(this.title).width - 20, bH - 16, ctx.measureText(this.title).width + 20, 32);
                ctx.fillRect(0, h * .5 + bH - 16, ctx.measureText("between crossed polars").width + 20, 32);
                ctx.fillStyle = '#222';
                ctx.textAlign = 'left';
                ctx.fillText(this.title, w - ctx.measureText(this.title).width - 10, bH);
                ctx.fillText("plane polarised light", 10, bH);
                ctx.fillText("between crossed polars", 10, h * .5 + bH);
                ctx.restore();
            }
            // scale
            scaleInPixels = 234 * (this.slideWidth / 460); // TODO this should be dynamic, provided by data
            if (this.orientation === 'landscape') {
                scaleY = h - bH - ((this.lefttopY - this.y) / 2);
                scaleX = w / 2 - scaleInPixels / 2;
            }
            else {
                scaleY = h / 2;
                scaleX = w - 10 - scaleInPixels;
            }
            ctx.save();
            ctx.font = '12px ' + OU.theme.font;
            ctx.beginPath();
            ctx.fillText(this.scaleTxt, scaleX + scaleInPixels / 2, scaleY + 10);
            ctx.moveTo(scaleX, scaleY - 10);
            ctx.lineTo(scaleX, scaleY + 10);
            ctx.moveTo(scaleX, scaleY);
            ctx.lineTo(scaleX + scaleInPixels, scaleY);
            ctx.moveTo(scaleX + scaleInPixels, scaleY - 10);
            ctx.lineTo(scaleX + scaleInPixels, scaleY + 10);
            ctx.lineWidth = 2;
            ctx.strokeStyle = '#000';
            ctx.stroke();
            ctx.restore();
            ctx = this.maskLayer.context;
            // outer frame
            ctx.save();
            ctx.fillStyle = '#fff';
            ctx.roundRect(0, 0, w, h, 20);
            ctx.fill();
            ctx.globalCompositeOperation = 'destination-out';
            ctx.beginPath();
            ctx.arc(this.leftX - this.x + this.slideWidth / 2, this.lefttopY - this.y + this.slideHeight / 2, this.slideHeight / 2, 0, Math.PI * 2, false);
            ctx.arc(this.rightX - this.x + this.slideWidth / 2, this.righttopY - this.y + this.slideHeight / 2, this.slideHeight / 2, 0, Math.PI * 2, false);
            ctx.fill();
            ctx.restore();
            // Crosshairs
            ctx.moveTo(this.leftX - this.x, this.lefttopY - this.y + this.slideHeight / 2);
            ctx.lineTo(this.leftX - this.x + this.slideWidth, this.lefttopY - this.y + this.slideHeight / 2);
            ctx.moveTo(this.leftX - this.x + this.slideWidth / 2, this.lefttopY - this.y);
            ctx.lineTo(this.leftX - this.x + this.slideWidth / 2, this.lefttopY - this.y + this.slideHeight);
            ctx.moveTo(this.rightX - this.x, this.righttopY - this.y + this.slideHeight / 2);
            ctx.lineTo(this.rightX - this.x + this.slideWidth, this.righttopY - this.y + this.slideHeight / 2);
            ctx.moveTo(this.rightX - this.x + this.slideWidth / 2, this.righttopY - this.y);
            ctx.lineTo(this.rightX - this.x + this.slideWidth / 2, this.righttopY - this.y + this.slideHeight);
            ctx.strokeStyle = '#fff';
            ctx.stroke();

            OU.VM.ROTATION = this;
            if (OU.VM.data.includeRotationButtons) {
                // Rotation buttons
                this.rotationButtonsDiv = new OU.util.Div({
                    x: this.x + this.w / 2 - 32,
                    y: this.y + h - 74,
                    w: 64,
                    h: 74,
                    zIndex: OU.ABSOLUTE_TOP,
                    innerHTML: '<input id="__rotateLeft" type="button" class="rotateLeft" onClick="return OU.VM.ROTATION.rotateCCW();"/><input type="button" class="rotateRight" onClick="return OU.VM.ROTATION.rotateCW();"/>'
                });
                var leftRotate = document.getElementById('__rotateLeft');
                if (leftRotate) {
                    leftRotate.focus();
                }
            }
            this.doRender = true;
        };
        OU.activity.Microscope.prototype.Rotation.prototype.rotateCCW = function() {
            this.degrees -= Math.PI / 36;
            this.degrees = (this.degrees + (Math.PI * 2)) % (Math.PI * 2);
            this.render();
        };
        OU.activity.Microscope.prototype.Rotation.prototype.rotateCW = function() {
            this.degrees += Math.PI / 36;
            this.degrees = (this.degrees + (Math.PI * 2)) % (Math.PI * 2);
            this.render();
        };
        OU.activity.Microscope.prototype.Rotation.prototype.renderCycle = function() {
            if (this.doRender)
                this.performRender();
            setTimeout(function() {
                rotation.renderCycle();
            }, 20);
        };
        OU.activity.Microscope.prototype.Rotation.prototype.render = function() {
            this.doRender = true;
        };
        OU.activity.Microscope.prototype.Rotation.prototype.share = function() {

        };
        OU.activity.Microscope.prototype.Rotation.prototype.performRender = function() {
            var i, leftLayer, rightLayer, rotationDegrees = this.degrees, temp,
                    r2d = Math.PI / 180, ctx = this.controlLayer.context,
                    nImgs = this.numberOfImagesInFullRotation / this.reuseCount,
                    lower = parseInt(this.degrees / this.segmentAngle),
                    upper = (lower + 1), // % nImgs,
                    upOpacity = this.noFade ? 1 : (this.degrees - lower * this.segmentAngle) / this.segmentAngle,
                    w = this.w,
                    h = this.h;
            this.doRender = false;
            lower = lower % nImgs;
            upper = upper % nImgs;
            if (lower > upper) { // Swap order if required
                temp = upper;
                upper = lower;
                lower = temp;
                upOpacity = 1 - upOpacity;
            }
            // step through slides and set visibility,opacity & rotation accordingly
            for (i = nImgs; i--; ) {
                leftLayer = this.leftBox.childNodes[i];
                rightLayer = this.rightBox.childNodes[i];
                if (i === lower) {
                    if (leftLayer.style.visibility !== "visible")
                        leftLayer.style.visibility = rightLayer.style.visibility = "visible";
                    leftLayer.style.webkitTransform = rightLayer.style.webkitTransform = "rotate3d(0,0,1," + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.MozTransform = rightLayer.style.MozTransform = "rotate(" + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.opacity = rightLayer.style.opacity = 1;
                } else if (i === upper) {
                    if (leftLayer.style.visibility !== "visible")
                        rightLayer.style.visibility = leftLayer.style.visibility = "visible";
                    leftLayer.style.webkitTransform = rightLayer.style.webkitTransform = "rotate3d(0,0,1," + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.MozTransform = rightLayer.style.MozTransform = "rotate(" + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.opacity = rightLayer.style.opacity = upOpacity;
                }
                else {
                    if (leftLayer.style.visibility !== "hidden")
                        rightLayer.style.visibility = leftLayer.style.visibility = "hidden";
                }
            }
            if (this.orientation === 'landscape') {
                ctx.clearRect(0, this.lefttopY - this.y + this.slideWidth * .9, this.w, this.slideWidth * .1);
                ctx.fillText((this.degrees / r2d | 0) + '\u00B0', w / 2, this.lefttopY - this.y + this.slideWidth * .95);
            }
            else {
                ctx.clearRect(this.x, this.lefttopY - this.y + this.slideWidth * .95 - 10, w, 20);
                ctx.fillText((this.degrees / r2d | 0) + '\u00B0', w / 4, this.lefttopY - this.y + this.slideWidth * .95);
            }
        };
        OU.activity.Microscope.prototype.Rotation.prototype.isHit = function(x, y, evState) {
            var dX, dY, oldTheta, newTheta;
            if (evState) {
                if (y < this.righttopY - this.y)
                    dY = y - (this.leftcenterY - this.y);
                else
                    dY = y - (this.rightcenterY - this.y);
                if (x < this.rightX - this.x)
                    dX = x - (this.leftCenterX - this.x);
                else
                    dX = x - (this.rightCenterX - this.x);
                this.distance = Math.sqrt(dX * dX + dY * dY);
                if (this.distance > this.slideWidth / 2)
                    return;
                if (this.inDrag) {
                    oldTheta = Math.atan2(this.startdY, this.startdX),
                            newTheta = Math.atan2(dY, dX);
                    this.degrees += newTheta - oldTheta;
                    this.degrees = (this.degrees + (Math.PI * 2)) % (Math.PI * 2);
                    this.startdY = dY;
                    this.startdX = dX;
                    this.render();
                }
                else {
                    this.startdY = dY;
                    this.startdX = dX;
                    this.inDrag = true;
                }
            }
            else {
                this.inDrag = false;
            }
        };
        OU.activity.Microscope.prototype.Rotation.prototype.remove = OU.activity.Microscope.prototype.Rotation.prototype.close = function() {
            OU.activity.Microscope.superClass_.remove.call(this); // call the superclass method
            this.controlLayer.remove();
            if (this.rotationButtonsDiv) {
                this.rotationButtonsDiv.remove();
            }
            if (this.leftBox.parentNode !== null) {
                this.leftBox.parentNode.removeChild(this.leftBox);
                this.rightBox.parentNode.removeChild(this.rightBox);
            }
            this.bgLayer.remove();
            this.maskLayer.remove();
            this.modalLayer.remove();
            this.container.inRotationView = false;
            this.container.loadedRotation = null;
        };
        this.init();
        this.renderCycle();
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Microscope, OU.util.Activity);
