/**
 * @fileoverview Canvas Table
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Layer');
OU.require('OU.util.DynText');

/**
 * CanvasTable - renders and manipulates a table object on canvas
 *
 * @param {object} params - options:
 * <ul>
 * <li><strong>{object} container:</strong> (required) Calling object, typically a OU.util.Activity</li>
 * <li><strong>{OU.util.Layer} layer:</strong> (required) Layer to render the table to</li>
 * <li><strong>{json object} sections:</strong> (required) defines the structure, ie. sections, columns, rows, etc.</li>
 * <li><strong>{int} x:</strong> X co-ordinate</li>
 * <li><strong>{int} y:</strong> Y co-ordinate</li>
 * <li><strong>{int} w:</strong> Width</li>
 * <li><strong>{int} h:</strong> Height</li>
 * </ul>
 */
OU.util.CanvasTable = function(params) {
    if (params.container === undefined) {
        console.log("ERROR: CanvasTable requires a container");
        return;
    }
    if (params.layer === undefined) {
        console.log("ERROR: CanvasTable requires a layer");
        return;
    }
    if (params.sections === undefined) {
        console.log("ERROR: CanvasTable requires sections");
        return;
    }
    this.container = params.container;
    this.layer = params.layer;
    this.context = params.layer.context;
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || this.container.w;
    this.h = params.h || this.container.h;
    this.sections = params.sections;
    this.borderColour = params.borderColour || "#000";
    this.bgColour = params.bgColour || "#fff";
    this.titleTextColour = params.titleTextColour || "#fff";
    this.titleBgColour = params.titleBgColour || "#000";
    this.hasInfo = params.hasInfo || false;
    this.totalItems = 0;

    /**
     * Initialise the sections and use resize to render
     * @private
     */
    OU.util.CanvasTable.prototype.init = function() {
        var i;
        for (i = this.sections.length; i--; ) {
            this.sections[i]._items = [];
        }

        this.resize();
    };
    /**
     * Renders the table
     */
    OU.util.CanvasTable.prototype.render = function() {
        var section, sectionNum, row, column, ctx = this.context, sectionOffset = this.x,
                minFont = 1000, sectionText;

        ctx.strokeStyle = this.borderColour;
        ctx.fillStyle = this.bgColour;
        ctx.lineWidth = 6;
        ctx.rect(sectionOffset, this.y, this.w, this.h);
        ctx.stroke();
        ctx.fill();
        for (sectionNum = 0; sectionNum < this.sections.length; sectionNum++) {
            ctx.lineWidth = 6;
            section = this.sections[sectionNum];
            section.sectionIndex = sectionNum;
            ctx.rect(sectionOffset, this.y, this.columnWidth * section.columns, this.h - this.rowHeight - this.titleHeight); //this.totalRows*this.rowHeight);
            ctx.stroke();
            ctx.save();
            ctx.lineWidth = 1;
            for (column = section.columns; column--; ) {
                ctx.beginPath();
                ctx.moveTo(sectionOffset + column * this.columnWidth, this.y);
                ctx.lineTo(sectionOffset + column * this.columnWidth, this.y + this.totalRows * this.rowHeight);
                ctx.stroke();
                for (row = section.rows + 1; row--; ) {
                    ctx.beginPath();
                    ctx.moveTo(sectionOffset, this.y + row * this.rowHeight);
                    ctx.lineTo(sectionOffset + section.columns * this.columnWidth, this.y + row * this.rowHeight);
                    ctx.stroke();
                }
            }
            ctx.save();
            ctx.fillStyle = this.titleBgColour;
            ctx.fillRect(sectionOffset, this.y + this.h - this.titleHeight, this.columnWidth * section.columns, this.titleHeight);
            ctx.fillStyle = this.titleTextColour;
            // Calculate smallest font used to render all titles - do not render yet
            sectionText = new OU.util.DynText({
                context: ctx,
                txt: section.title,
                fontSize: this.titleHeight * .8,
                x: sectionOffset,
                y: this.y + this.h - this.titleHeight,
                w: this.columnWidth * section.columns,
                h: this.titleHeight,
                measureOnly: true
            });
            ctx.restore();
            if (sectionText.font.size < minFont)
                minFont = sectionText.font.size;
            sectionOffset = sectionOffset + section.columns * this.columnWidth;
            ctx.restore();
        }
        this.titleFontSize = minFont;
        // Now render all titles at the same size (the smallest needed to fit)
        sectionOffset = this.x;
        ctx.save();
        ctx.fillStyle = this.titleTextColour;
        for (sectionNum = 0; sectionNum < this.sections.length; sectionNum++) {
            section = this.sections[sectionNum];
            section._x = sectionOffset;
            sectionText = new OU.util.DynText({
                context: ctx,
                txt: section.title,
                fontSize: minFont,
                x: sectionOffset,
                y: this.y + this.h - this.titleHeight,
                w: this.columnWidth * section.columns,
                h: this.titleHeight
            });
            sectionOffset = sectionOffset + section.columns * this.columnWidth;
        }
        ctx.restore();
    };
    /**
     * Resizes the table dimensions and re-calculates the section sizes, etc.
     * @param {object} dims - optional new dims, default to current value unless specified:
     * <ul>
     * <li><strong>{int} x:</strong> X co-ordinate</li>
     * <li><strong>{int} y:</strong> Y co-ordinate</li>
     * <li><strong>{int} w:</strong> Width</li>
     * <li><strong>{int} h:</strong> Height</li>
     * </ul>
     */
    OU.util.CanvasTable.prototype.resize = function(dims) {
        var i, j, section, hasTitles = false;
        if (dims === undefined)
            dims = {};
        this.x = dims.x || this.x;
        this.y = dims.y || this.y;
        this.w = dims.w || this.w;
        this.h = dims.h || this.h;

        this.totalColumns = 0;
        this.totalRows = 0;
        for (i = this.sections.length; i--; ) {
            section = this.sections[i];
            this.totalColumns += section.columns;
            if (section.rows > this.totalRows)
                this.totalRows = section.rows;
            if (section.title && section.title !== "")
                hasTitles = true;
            for (j = section._items.length; j--; ) {
                this.setTargetPosition(section._items[j]);
            }
        }
        this.columnWidth = this.w / this.totalColumns;
        if (hasTitles)
            this.titleHeight = OU.controlHeight;
        else
            this.titleHeight = 0;
        if (this.hasInfo) // if has feedback, then add another row for it
            this.rowHeight = (this.h - this.titleHeight) / (this.totalRows + 1);
        else
            this.rowHeight = (this.h - this.titleHeight) / this.totalRows;

        this.render();
    };
    /**
     * Adds an item to the table, if it is over the table
     * @param {object} item - and item that has x,y,w,h dimensions and a move() function
     */
    OU.util.CanvasTable.prototype.addItemIfOver = function(item) {
        var i,
                centreX = item.x + item.w / 2,
                centreY = item.y + item.h / 2,
                section;
        //        console.log("add item, X,Y:"+centreX+","+centreY);
        if (centreX < this.x || centreX > this.x + this.w || centreY < this.y || centreY > this.y + this.h)
            return false; //return false if item is not over the table

        //determine which section it is over
        for (i = 0; i < this.sections.length; i++) {
            section = this.sections[i];
            if (centreX > section._x && centreX < section._x + section.columns * this.columnWidth) {
                this.addItemToSection(section, item);
                return true;
            }
        }
        return false;
    };
    /**
     * Renders the given text into the info box
     * @param {string} info
     */
    OU.util.CanvasTable.prototype.info = function(info) {
        new OU.util.DynText({
            context: this.context,
            txt: info,
            colour: this.container.data.validationColour,
            fontSize: this.titleFontSize,
            background: {
                clear: true
            },
            x: this.x,
            y: this.y + this.h - this.titleHeight - this.rowHeight,
            w: this.w,
            h: this.rowHeight
        });
    };
    /**
     * Adds an item to the table, into a specific section
     * @param {object} item - and item that has x,y,w,h dimensions and a move() function
     * @param {int} sectionIndex
     */
    OU.util.CanvasTable.prototype.addItem = function(item, sectionIndex) {
        var section = this.sections[sectionIndex];
        this.addItemToSection(section, item);
    };
    /**
     * Moves an item to the other section
     * @param {object} item - and item that has x,y,w,h dimensions and a move() function
     * @param {int} sectionIndex
     */
    OU.util.CanvasTable.prototype.switchItem = function(item, sectionIndex) {
        this.removeItem(item);
        this.addItem(item, sectionIndex);
    };
    /**
     * Sets the target position of an item based on the index number in a specific section
     * @param {object} item - the item to position
     * @private
     */
    OU.util.CanvasTable.prototype.setTargetPosition = function(item) {
        var section = item.tableInfo.section, itemIndex = item.tableInfo.itemIndex,
                column = itemIndex % section.columns,
                row = (itemIndex - column) / section.columns;

        item.tableInfo.targetX = section._x + this.columnWidth * column + (this.columnWidth - item.w) / 2;
        item.tableInfo.targetY = this.y + this.rowHeight * row + (this.rowHeight - item.h) / 2;
    };
    /**
     * Adds an item to a table section
     * @param {object} section - the section that the item is going in
     * @param {object} item - the item to put in the section
     * @private
     */
    OU.util.CanvasTable.prototype.addItemToSection = function(section, item) {
        var itemIndex = section._items.length;
        section._items[itemIndex] = item;
        item.tableInfo = {
            section: section,
            itemIndex: itemIndex
        };
        this.totalItems++;
        this.setTargetPosition(item);
    };
    /**
     * Removes an item from a section and closes any gap left by adjusting the indexes of all the items left.
     * @param {object} item - the item to remove
     */
    OU.util.CanvasTable.prototype.removeItem = function(item) {
        var i, section = item.tableInfo.section;
        section._items.splice(item.tableInfo.itemIndex, 1);
        item.tableInfo = null;
        this.totalItems--;
        for (i = 0; i < section._items.length; i++) {
            section._items[i].tableInfo.itemIndex = i;
            this.setTargetPosition(section._items[i]);
        }
        return true;
    };
    /**
     * Moves an item towards it's target position. Calls the items own move() function with a new value in the direction of the target
     * @param {object} item - the item to move
     */
    OU.util.CanvasTable.prototype.moveItem = function(item) {
        var x, y, dX = item.tableInfo.targetX - item.x,
                dY = item.tableInfo.targetY - item.y;
        if (dX === 0 && dY === 0)
            return false; // already at target, so return false, meaning no movement
        if (Math.abs(dX) < 2) {
            x = item.tableInfo.targetX;
        }
        else {
            x = item.x + dX / 6;
        }
        if (Math.abs(dY) < 2) {
            y = item.tableInfo.targetY;
        }
        else {
            y = item.y + dY / 6;
        }
        item.move(x, y);
        return true;
    };
    this.init();
};