/**
 * @fileoverview Button Classes
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

/* Some Constants to define Button State */
OU.UNSELECTED = 0;
OU.SELECTED = 1;
OU.CORRECT = 2;
OU.INCORRECT = 3;
/**
 *  @class BaseButton - Canvas based button for rendering various button types to canvas.
 *
 *  Requires a Layer parameter which hasEvents set to true, as this pushes itself into the clickable array.
 *  Note: Only intended to be initialised via a sub class.
 *
 *  @param {object} params - various parameters (optional unless otherwise stated):
 * <ul>
 * <li><strong>layer</strong>: (required) {OU.util.Layer} A reference to the Layer object this button is rendered to</li>
 * <li><strong>txt</strong>: {String} The button's label</li>
 * <li><strong>group</strong>: {String} Optional button group name - all buttons in the same group will have the same size font, defaults to a single activity group. It's advisable to put radio buttons in their own group.</li>
 * <li><strong>onClick</strong>: (required) {function} function to call when the button is clicked</li>
 * <li><strong>onClickParam</strong>: {object} optional parameter to send the onClick function</li>
 * <li><strong>fontSize</strong>: {int} maximum font size to use (shrinks to fit if too big)</li>
 * <li><strong>glyph</strong>: {object} Glyph to render in the button (see canvas.context.glyph extension)</li>
 * <li><strong>background</strong>: {object} Background to render (see canvas.context.background extension)</li>
 * <li><strong>visiblePadding</strong>: {int} Visible padding to the render (ie hit area bigger than visible button)</li>
 * <li><strong>renderStyle</strong>: {int} Render style (0== render Flat (default), else render new style)</li>
 * <li><strong>gloss</strong>: {boolean} Optionally add a gloss effect to the button</li>
 * <li><strong>disabled</strong>: {boolean} Set true to disable the click event</li>
 * <li><strong>autoWidth</strong>: {boolean} Set true to make the button resize down to fit the label text</li>
 * </ul>
 *
 * @extends OU.util.Tabbable
 */
OU.util.BaseButton = function(params) {
    params.fontSize = params.fontSize || OU.theme.fontSize * OU.dpr;
    this.params = params;
    this.onClickParam = params.onClickParam;
    this.layer = params.layer;
    this.container = params.layer.container;
    this.context = this.layer.context;
    this.glyph = params.glyph || this.glyph;
    this.background = params.background;    // Button renders its own background
    params.background = undefined;            // So remove from params that will later get passed to a DynText
    this.visiblePadding = params.visiblePadding;
    this.visibleDims = {}; // defines the visible area of the button ( a sub area of the button Dims (hit area)
    this.textParams = this.params; // the params to pass through to DynText - again dims are a sub area of visibleDims area
    this.textParams._origFontSize = params.fontSize;
    this.renderStyle = params.renderStyle || 0;
    this._state = false; // initial state if button type requires it
    this.instanceId = '_button_' + OU._buttonCount++;
    if (params.layer === undefined)
        OU.log("Bad use of Button class - no Layer given");

    var groupName = params.group || "defaultGroup";
    OU._buttonGroups = OU._buttonGroups || []; // ensure buttonGroups is defined
    this._group = OU._buttonGroups[groupName] = OU._buttonGroups[groupName] || []; // ensure this button's group is defined
    this._group.push(this); // add this button to the group
    if (this._group.minFont && this._group.minFont < this.textParams.fontSize)
        this.textParams.fontSize = this._group.minFont;
    /**
     * Sets some default values, adds the button to removable array and renders the button
     *
     *@private
     */
    OU.util.BaseButton.prototype.init = function() {
        this.textParams.context = this.layer.context;
        this.resize(this.params);
        //ensure some required params are present
        this.textParams.colour = this.textParams.colour || OU.theme.buttonTextCol;
        this.onClick = params.onClick || function() {
        };
        if (this.background)
            this.background.gloss = params.gloss || true;
        this.layer.events.clickable.push(this, this.instanceId); // push this button into the clickable array of the layer
        OU.addRemovable(this, this.instanceId); // push into removables array so that the tabbable element is removed
        this.render(); // render the button
    };
    /**
     * Modifies the button - allows change of text label, onClick function and onClickParam.
     * Render is called after modification.
     * @param {object} params - optional paramters, if undefined remain as before:
     * <ul>
     * <li><strong>{string} txt:</strong> New button label</li>
     * <li><strong>{function} onClick:</strong> New function to call when button is hit</li>
     * <li><strong>{object} onClickParam:</strong> New onClick Parameter</li>
     * </ul>
     */
    OU.util.BaseButton.prototype.modify = function(params) {
        if (params.txt)
            this.textParams.txt = params.txt;
        if (params.onClick)
            this.onClick = params.onClick;
        if (params.onClickParam)
            this.onClickParam = params.onClickParam;
        this.render();
    };
    /**
     * Renders a "flat" button, used for check box, radio buttons, etc.
     * @private
     */
    OU.util.BaseButton.prototype.renderFlat = function() {
        // used by check boxes etc.
        var ctx = this.context;
        if (ctx) {
            ctx.save();
            if (this.params.disabled) // if disabled, simply render at 50% opacity
                ctx.globalAlpha = 0.5;

            if (this.background)
                ctx.background(this.background, this.visibleDims); // render button background
            else
                ctx.clearRect(this.visibleDims.x - 1, this.visibleDims.y - 1, this.visibleDims.w + 2, this.visibleDims.h + 2);
            if (this.params.txt && this.params.txt !== '') {
                this._dyntext = new OU.util.DynText(this.textParams);
            }
            if (this.glyph) {
                ctx.glyph(this.glyph);
            }
            if (this.hasFocus) {
                this.focusOutline();
            }
            ctx.restore();
        }
    };
    /**
     * Renders an outline around the button
     */
    OU.util.BaseButton.prototype.focusOutline = function() {
        var ctx = this.context;
        if (ctx) {
            ctx.save();
            ctx.strokeStyle = '#c00';
            ctx.lineWidth = 3;
//        ctx.beginPath();
            ctx.roundRect(this.visibleDims.x + 2, this.visibleDims.y + 2, this.visibleDims.w - 4, this.visibleDims.h - 4, this.background ? this.background.radius : 0);
            ctx.stroke();
            ctx.restore();
        }
    };
    /**
     * Renders a button with default styling
     * @private
     */
    OU.util.BaseButton.prototype.renderNew = function() {
        var getBrightness = function(hex) {
            var r, g, b;
            if (!hex || hex === 'transparent')
                return 0;
            if (hex.length === 4) {
                r = (hex.substr(1, 1), 16) | 0,
                        g = (hex.substr(2, 1), 16) | 0,
                        b = (hex.substr(3, 1), 16) | 0;
            }
            else {
                r = (hex.substr(1, 2), 16) | 0,
                        g = (hex.substr(3, 2), 16) | 0,
                        b = (hex.substr(5, 2), 16) | 0;
            }
            return (0.2126 * (r / 255)) + (0.7152 * (g / 255)) + (0.0722 * (b / 255));
        };
        // used by proper buttons.
        var ctx = this.context, oc, oy, ob = {}, ovd = {}, lum;
        ctx.save();
        oc = this.textParams.colour;
        oy = this.textParams.y;
        lum = getBrightness(this.background.col);
        ob = {
            col: "rgba(0,0,0,0.1)",
            radius: this.background.radius
        };
        ovd = {
            x: this.visibleDims.x - 2,
            y: this.visibleDims.y - 2,
            w: this.visibleDims.w + 4,
            h: this.visibleDims.h + 4
        };
        this.textParams.y -= 1;
        this.textParams.colour = "rgba(100,100,100,0.7)";
        if (this.params.disabled) // if disabled, simply render at 50% opacity
            ctx.globalAlpha = 0.5;
        ctx.background(ob, ovd); // render button background
        ctx.background(this.background, this.visibleDims); // render button background
        /* debug* ctx.clearRect(this.visibleDims.x,this.visibleDims.y,this.visibleDims.w,this.visibleDims.h); //*/
        if (this.params.txt && this.params.txt !== '') {
            if (lum > 0.5) {
                this.textParams.colour = "#ffffff";
                this._dyntext = new OU.util.DynText(this.textParams);
                this.textParams.colour = oc;
                this.textParams.y = oy;
                new OU.util.DynText(this.textParams);
            }
            else {
                this._dyntext = new OU.util.DynText(this.textParams);
                this.textParams.colour = "#ffffff";
                this.textParams.y = oy;
                new OU.util.DynText(this.textParams);
            }
        }
        if (this.glyph) {
            ctx.glyph(this.glyph);
        }
        if (this.hasFocus) {
            this.focusOutline();
        }
        ctx.restore();
    };
    /**
     * Redraws the button on the canvas
     * @param {object} dims (optional) new dimensions, passed to resize if present
     */
    OU.util.BaseButton.prototype.render = function(dims) {
        var g;
        if (dims)
            this.resize(dims);
        if (this.renderStyle === 0)
            this.renderFlat();
        else
            this.renderNew();
        if (this._dyntext && (!this._group.minFont || this._dyntext.font.size < this._group.minFont)) {
            this._group.minFont = this._dyntext.font.size;
            for (var i = this._group.length; i--; ) {
                g = this._group[i];
                if (g.render) {
                    g.textParams.fontSize = this._group.minFont;
                    g.render();
                }
            }
        }
    };
    /**
     * Old render function
     * @deprecated
     * @private
     */
    OU.util.BaseButton.prototype.origrender = function() {
        var ctx = this.context, oy;
        ctx.save();
        oy = this.textParams.y;
        this.textParams.y = this.textParams.y - 1;
        this.textParams.colour = "rgba(100,100,100,0.7)";
        if (this.params.disabled) // if disabled, simply render at 50% opacity
            ctx.globalAlpha = 0.5;
        ctx.background(this.background, this.visibleDims); // render button background
        /* debug* ctx.clearRect(this.visibleDims.x,this.visibleDims.y,this.visibleDims.w,this.visibleDims.h); //*/
        if (this.params.txt && this.params.txt !== '') {
            new OU.util.DynText(this.textParams);
            this.textParams.colour = "#ffffff";
            this.textParams.y = oy;
            new OU.util.DynText(this.textParams);
        }
        if (this.glyph) {
            ctx.glyph(this.glyph);
        }
        ctx.restore();
    };
    /**
     * @deprecated - should use resize
     * @private
     */
    OU.util.BaseButton.prototype.move = function(p) {
        return this.resize(p);
    };
    /**
     * resize the button to the given dimensions, defaults to current state if elements omitted
     * @param {object} dims - ie {x:0,y:0,w:100,h:40}
     */
    OU.util.BaseButton.prototype.resize = function(dims) {
        this.visibleDims.x = this.x = dims.x || this.x || 0;
        this.visibleDims.y = this.y = dims.y || this.y || 0;
        this.visibleDims.w = this.w = dims.w || this.w || 100;
        this.visibleDims.h = this.h = dims.h || this.h || 40;
        if (this.visiblePadding) { // visiblePadding is a percentage value between 0.0 & 1.0
            this.visibleDims.w = this.visibleDims.w * (1 - (2 * this.visiblePadding));
            this.visibleDims.h = this.visibleDims.h * (1 - (2 * this.visiblePadding));
            this.visibleDims.x += this.w * this.visiblePadding;
            this.visibleDims.y += this.h * this.visiblePadding;
        }
        this.textParams.x = this.visibleDims.x;
        this.textParams.y = this.visibleDims.y;
        this.textParams.w = this.visibleDims.w;
        this.textParams.h = this.visibleDims.h;
        if (this.glyph) { // if we have a glyph, then adjust the text dimensions accordingly and position the glyph
            switch (this.glyph.align.toLowerCase()) {
                case "right":
                    this.glyph.x = this.textParams.x + this.textParams.w - (this.glyph.w + 5);
                    this.textParams.w = this.textParams.w - (this.glyph.w + 10);
                    this.textParams.align = 'right';
                    break;
                case "center":
                case "middle":
                    this.glyph.x = this.textParams.x + ((this.textParams.w - this.glyph.w) / 2);
                    break;
                case "left":
                default:
                    this.glyph.x = this.textParams.x + 5;
                    this.textParams.x += this.glyph.w + 10;
                    this.textParams.w = this.textParams.w - (this.glyph.w + 10);
                    this.textParams.align = 'left';
                    break;
            }
            this.glyph.y = this.textParams.y + ((this.textParams.h - this.glyph.h) / 2);
        }
        if (this.params.autoWidth) {
            this.textParams.measureOnly = true;
            var measure = new OU.util.DynText(this.textParams);
            this.textParams.measureOnly = false;
            this.textParams.autoWidth = false;
            this.textParams.w = this.w = this.visibleDims.w = measure.w + 40;
        }
    };
    /**
     * Called whenever a mouse or touch event happens on the canvas that this button is associated with.
     * Determines if the button is hit and calls hit() if true
     * @param {int} x - x co-ordinate in pixels, relative to the canvas
     * @param {int} y - y co-ordinate in pixels, relative to the canvas
     * @param {boolean} pressed - state of the event, true if in "touchdown" or "mousedown" state
     *
     * @private
     */
    OU.util.BaseButton.prototype.isHit = function(x, y, pressed) {
        if (pressed) {
            if (x < this.x || x > this.x + this.w)
                return false;
            if (y < this.y || y > this.y + this.h)
                return false;
            this.hit();
            return true;
        }
        return false;
    };
    /**
     * Change the button state to enabled
     */
    OU.util.BaseButton.prototype.enable = function() {
        this.params.disabled = false;
        this.render();
    };
    /**
     * Change the button state to disabled
     */
    OU.util.BaseButton.prototype.disable = function() {
        this.params.disabled = true;
        this.render();
    };
    /**
     * Action the onclick callback function
     * @returns false
     * @private
     */
    OU.util.BaseButton.prototype.hit = function() {
        if (this.params.disabled !== true) {

            if (this.defaultAction) // If the button has a default action built into it, then call it now
                this.defaultAction();
            this.onClick(this.onClickParam); // Call the supplied onClick function
        }
        return false;
    };
    /**
     * Enables the button's hit function
     */
    OU.util.BaseButton.prototype.enable = function() {
        this.params.disabled = false;
        this.render();
    };
    /**
     * Disables the button's hit function
     */
    OU.util.BaseButton.prototype.disable = function() {
        this.params.disabled = true;
        this.render();
    };
    /**
     * Overrides OU.util.Tabbable.focus() to visually show the button has tab focus
     * @private
     */
    OU.util.BaseButton.prototype.focus = function() {
        this.hasFocus = true;
        this.render();
        return false;
    };
    /**
     * Overrides OU.util.Tabbable.blur() to remove the visual effect pf the button having tab focus
     * @private
     */
    OU.util.BaseButton.prototype.blur = function() {
        this.hasFocus = undefined;
        this.render();
        return false;
    };
    /**
     * Removes the button from the clickable array and nulls some elements to aid garbage collection.
     * Note, this will not remove the buttons visible appearance on a canvas
     */
    OU.util.BaseButton.prototype.remove = function() {
        if (this.layer && this.layer.events && this.layer.events.clickable) {
            this.layer.events.clickable.removeType(this.instanceId); // remove the button from the clickable array
            if (this.onClick) {
                delete this.onClick;
                this.onClick = null;
            }
            if (this.context) {
                delete this.context;
                this.context = null;
            }
            if (this.container) {
                delete this.container;
                this.container = null;
            }
            if (this.layer) {
                delete this.layer;
                this.layer = null;
            }
        }
        OU.util.BaseButton.superClass_.remove.call(this); // call the parent class method
        OU.removables.removeType(this.instanceId);
    };
    OU.base(this, params);
    this.init();
};
OU.inherits(OU.util.BaseButton, OU.util.Tabbable);

OU.resetButtonGroupSizes = function(groups) {
    var g, i, j;
    groups = groups || [];
    groups.push('defaultGroup');

    OU._buttonGroups = OU._buttonGroups || []; // ensure buttonGroups is defined
    for (i = groups.length; i--; ) {
        g = OU._buttonGroups[groups[i]] || []; // ensure this button's group is defined
        g.minFont = null;
        for (j = g.length; j--; ) {
            g[j].textParams.fontSize = g[j].textParams._origFontSize;
        }
    }
};
/**
 * @class Basic button extends BaseButton with a specific UI
 *
 * @param {object} params - See Params for BaseButton
 *
 * @extends OU.util.BaseButton
 */
OU.util.Button = function(params) {
    var iconHeight = (params.h > 28 ? 24 : params.h - 4) * OU.dpr;
    params.visiblePadding = params.visiblePadding === undefined ? .1 : params.visiblePadding;
    params.colour = params.colour || '#222';
    params.align = 'center';
    params.renderStyle = 0;
    if (params.glyph) {
        params.glyph.w = iconHeight;
        params.glyph.h = iconHeight;
    }
    if (!params.background) {
        params.background = {
            col: '#76787F', //OU.theme.buttonBgCol - #ddd,
            radius: 3,
            borderCol: '#999' //OU.theme.buttonBorderCol
        };
    }
    OU.base(this, params);
};
OU.inherits(OU.util.Button, OU.util.BaseButton);
/**
 * @class Control Buttons, has a specific visual appearance
 * @param {object} params - See Params for BaseButton
 * @extends OU.util.BaseButton
 */
OU.util.ControlButton = function(params) {
    params.visiblePadding = .1;
    if (params.background === undefined) {
        params.background = {
            col: OU.theme.buttonBgCol,
            radius: 3,
            borderCol: OU.theme.buttonBorderCol
        };
    }
    OU.base(this, params);
};
OU.inherits(OU.util.ControlButton, OU.util.BaseButton);
/**
 * @class Info button, has appearance of a standard button with the addition of a "info" icon
 * @param {object} params - See Params for BaseButton
 * @extends OU.util.BaseButton
 */
OU.util.InfoButton = function(params) {
    params.txt = "";
    params.w = params.w || 32 * OU.dpr;
    params.h = params.h || 32 * OU.dpr;
    params.padding = 0;
    params.visiblePadding = 0;
    params.glyph = {
        type: 'infoIcon',
        align: 'center',
        w: 30 * OU.dpr,
        h: 30 * OU.dpr
    };
    OU.base(this, params);
};
OU.inherits(OU.util.InfoButton, OU.util.BaseButton);
/**
 * @class Button has appearance of a standard button with the addition of a "right arrow" icon
 * @param {object} params - See Params for BaseButton
 * @extends OU.util.BaseButton
 */
OU.util.NextButton = function(params) {
    var iconHeight = (params.h > 28 ? 24 : params.h - 4) * OU.dpr;
    params.visiblePadding = undefined; //.1;
    params.background = undefined;
    params.colour = '#000';
    params.glyph = {
        type: 'rightArrow',
        align: 'right',
        w: iconHeight,
        h: iconHeight
    };
    OU.base(this, params);
};
OU.inherits(OU.util.NextButton, OU.util.BaseButton);
/**
 * @class Button has appearance of a standard button with the addition of a "left arrow" icon
 * @param {object} params - See Params for BaseButton
 * @extends OU.util.BaseButton
 */
OU.util.PrevButton = function(params) {
    var iconHeight = (params.h > 28 ? 24 : params.h - 4) * OU.dpr;
    params.visiblePadding = undefined; //.1;
    params.background = undefined;
    params.colour = '#000';
    params.glyph = {
        type: 'leftArrow',
        align: 'left',
        w: iconHeight,
        h: iconHeight
    };
    OU.base(this, params);
};
OU.inherits(OU.util.PrevButton, OU.util.BaseButton);
/**
 * @class Button has appearance of a standard button with the addition of a "radio button" icon depending on state.
 * Note: Radio buttons only have the appearance of radio buttons, and not to characteristics.
 * ie. they are not grouped - changing state of one will not affect other radio buttons
 * @param {object} params - See Params for BaseButton
 * @extends OU.util.BaseButton
 */
OU.util.RadioButton = function(params) {
    var iconHeight = (params.h > 28 ? 24 : params.h - 4) * OU.dpr;
    params.visiblePadding = .1;
//    params.background = undefined;
    params.colour = '#000';
    params.glyph = undefined;
    this.glyph = {
        type: 'radioButton',
        align: 'left',
        w: iconHeight,
        h: iconHeight
    };

    params.group = params.group || "radioGroup";

    /**
     * Default action for RadioButtons is to change the state for all buttons in the group and re-render
     */
    OU.util.RadioButton.prototype.defaultAction = function() {
        for (var i = this._group.length; i--; )
            this._group[i].state && this._group[i].state(false);
        this.state(true);
        for (i = this._group.length; i--; )
            this._group[i].render && this._group[i].render();
    };
    /**
     * Changes the 'state' of the radio button
     * @param {const | boolean} s - the new state, can be [OU.UNSELECTED | OU.SELECTED | OU.CORRECT | OU.INCORRECT | true | false]
     */
    OU.util.RadioButton.prototype.state = function(s) {
        this._state = s;
        switch (s) {
            default:
            case OU.UNSELECTED:
            case false:
                this.glyph.type = 'radioButton'; // 'radioButton';
                break;
            case OU.SELECTED:
            case true:
                this.glyph.type = 'radioButtonSelected';
                break;
            case OU.CORRECT:
                this.glyph.type = 'radioButtonCorrect';
                break;
            case OU.INCORRECT:
                this.glyph.type = 'radioButtonIncorrect';
                break;
        }
    };
    this.state(params.state);
    OU.base(this, params);
};
OU.inherits(OU.util.RadioButton, OU.util.BaseButton);
/**
 * @class Button has appearance of a standard button with the addition of "checkbox" icons depending on it's state
 * @param {object} params - See Params for BaseButton
 * @extends OU.util.BaseButton
 */
OU.util.CheckBoxButton = function(params) {
    var iconHeight = (params.h > 28 ? 24 : params.h - 4) * OU.dpr;
    params.visiblePadding = .03;
    params.background = undefined;
    params.colour = '#000';
    params.glyph = undefined;
    this.glyph = {
        type: 'checkbox',
        align: 'left',
        w: iconHeight,
        h: iconHeight
    };
    /**
     * Default action for CheckBoxButtons is to change the state and re-render
     */
    OU.util.CheckBoxButton.prototype.defaultAction = function() {
        this._state = !this._state;
        this.state(this._state);
        this.render();
    };
    /**
     * Changes the 'state' of the checkbox
     * @param {const | boolean} s - the new state, can be [OU.UNSELECTED | OU.SELECTED | OU.CORRECT | OU.INCORRECT | true | false]
     */
    OU.util.CheckBoxButton.prototype.state = function(s) {
        this._state = s;
        switch (s) {
            default:
            case OU.UNSELECTED:
            case false:
                this.glyph.type = 'checkbox';
                break;
            case OU.SELECTED:
            case true:
                this.glyph.type = 'checkboxSelected';
                break;
            case OU.CORRECT:
                this.glyph.type = 'tickBox';
                break;
            case OU.INCORRECT:
                this.glyph.type = 'crossBox';
                break;
        }
    };
    this.state(params.state);
    OU.base(this, params);
};
OU.inherits(OU.util.CheckBoxButton, OU.util.BaseButton);
/**
 * @class Acts like a button - but has no render
 * @param {object} params - parameters to define the button:
 * <ul>
 * <li><strong>{int} x</strong> x position </li>
 * <li><strong>{int} y</strong> y position</li>
 * <li><strong>{int} w</strong> width</li>
 * <li><strong>{int} h</strong> height</li>
 * <li><strong>{function} onClick</strong> function to call on button hit</li>
 * <li><strong>{object} onClickParam</strong> optional parameter to pass back to the onClick function</li>
 * </ul>
 */
OU.util.InvisibleButton = function(params) {
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || 200;
    this.h = params.h || 100;
    this.onClick = params.onClick || function() {
    };
    this.onClickParam = params.onClickParam;
    if (params.layer)
        params.layer.events.clickable.push(this);
    /**
     * @private
     */
    OU.util.InvisibleButton.prototype.isHit = function(x, y, pressed) {
        if (pressed) {
            if (x < this.x || x > this.x + this.w)
                return;
            if (y < this.y || y > this.y + this.h)
                return;
            this.onClick(this.onClickParam);
        }
    };
    return this;
};
