require(
    ['../main'],
    function () {
        "use strict";
        require(
            [
                'utils',
                'table',
                'loader',
                'buttons',
                'feedback',
                'table-extend',
                'table-input',
                'table-save'
            ],
            function (utils, table, ldr, btn, fbk) {
                var type = 'table',
                    index = 0,
                    url, data;

                if (VLE.serverversion) {
                    url = VLE.get_param('settings'); // VLE attachment name for JSON file
                    data = {};

                    if (url.split('.').pop() === 'json') {
                        data.file = [url];
                    }
                    else {
                        data.folder = ['settings'];
                    }

                    TableActivity(data);
                }
                else { //local testing
                    require(['table-data'], function(data) {
                        if (typeof index === 'number') {
                            TableActivity(data.examples[index]);
                        }
                        else {
                            data.examples.forEach(function(data) {
                                TableActivity(data);
                            });
                        }
                    });
                }

                function TableActivity(data) {
                    console.log(JSON.stringify(data)); //for copy/pasting into data file

                    TableActivity.index = ++TableActivity.index || 0;

                    var index = TableActivity.index,
                        id = type + '-activity' + index,
                        container = document.body.appendChild(utils.createElement('div', 'container', id)),
                        question = table.construct(),
                        loader = ldr.construct(),
                        buttons, feedback;

                    var load = function() {
                        if (VLE.serverversion) {
                            loader.load({
                                container: container,
                                load: data,
                                callback: function(result) {
                                    data = result.file[0];
                                    init();
                                }
                            });
                        }
                        else {
                            if (data.images && Array.isArray(data.images.files) && data.images.files.length) {
                                loader.load({
                                    container: container,
                                    dir: data.images.dir || 'img/',
                                    load: {
                                        image: data.images.files
                                    },
                                    callback: function (result) {
                                        if (Array.isArray(result.image)) {
                                            data.images.files = result.image;
                                        }

                                        init();
                                    }
                                });
                            }
                            else {
                                init();
                            }
                        }
                    };

                    var init = function() {
                        if (typeof data.css != 'undefined') {
                            addCSS(data.css);
                        }

                        ensureBackwardCompatibility();
                        setOverrides();
                        setCfg();
                        create();
                        setEventHandlers();
                    };

                    //TODO support for old data files - remove in next version
                    var ensureBackwardCompatibility = function () {
                        //TODO this needs to become data.answer (part of the activity cfg)...
                        if (data.input && Array.isArray(data.input.answer)) {
                            data.input.answer = {
                                data: data.input.answer
                            };
                        }
                    };

                    //TODO remove in next version/move to activity class
                    var setOverrides = function () {
                        data.buttons.enabled = data.buttons.enabled || [];

                        //enforce retry button - remove in next version
                        if (data.buttons.enabled.indexOf('check') > -1 && data.buttons.enabled.indexOf('retry') < 0) {
                            data.buttons.enabled.push('retry');
                        }
                    };

                    var setCfg = function () {
                        data.buttons.enabled = data.buttons.enabled || [];

                        if (data.input && data.buttons.enabled.indexOf('retry') > -1) {
                            data.input.answer = data.input.answer || {};

                            if (['text', 'textarea'].indexOf(data.input.type) > -1 && data.input.answer.clearIncorrect !== true) {
                                data.input.answer.clearIncorrect = false;
                            }
                        }
                    };

                    var create = function() {
                        question.init(data, container);

                        if (!container.contains(question.container)) {
                            container.appendChild(question.container);
                        }

                        if (data.buttons) {
                            createButtons();

                            if (buttons.active('check')) {
                                createFeedback();
                            }
                        }

                        utils.triggerCustomEvent(container, 'activity-loaded');
                        //utils.makeResponsive(container);
                        container.style.visibility = 'visible';
                        //utils.resizeIFrame();
                    };

                    var createButtons = function () {
                        if (question.plugins.input && !question.plugins.input.answer.overlay) {
                            data.buttons.data = data.buttons.data || {};

                            if (question.plugins.input && !question.plugins.input.answer.overlay) {
                                data.buttons.data.reveal = {
                                    html: 'Reveal answer'
                                };
                            }
                        }

                        buttons = btn.construct(data.buttons);
                        buttons.init();
                        question.buttons = buttons;
                        container.appendChild(buttons.el);
                    };

                    var createFeedback = function () {
                        feedback = fbk.construct(data.feedback, container);
                        container.appendChild(feedback.el);

                        if (buttons.active('retry')) {
                            feedback.retry.replaceChild(buttons.get('retry').el, feedback.retry.firstChild);
                        }
                    };

                    var addCSS = function(css) {
                        var el, ss;

                        if (typeof css == 'string') {
                            el = document.head.appendChild(document.createElement('link'));
                            el.rel = 'stylesheet';
                            el.href = css;
                        }
                        else {
                            el = document.head.appendChild(document.createElement('style'));
                            //ss = document.head.styleSheets[document.head.styleSheets.length - 1];
                            ss = el.sheet;

                            utils.forIn(css, function(rules, selector) {
                                rules = Array.isArray(rules) ? rules : [rules];

                                rules.forEach(function(rule) {
                                    ss.insertRule('#' + id + ' ' + selector + '{' + rule + '}', ss.cssRules.length);
                                    console.log(ss);
                                });
                            });
                        }
                    };

                    var setEventHandlers = function () {
                        //container.addEventListener('check', handleEvent);
                        //container.addEventListener('retry', handleEvent);
                        container.addEventListener('feedback-result', handleEvent);
                        container.addEventListener('max-attempts-reached', handleEvent);
                    };

                    var triggerEvent = function (el, type) {
                        utils.triggerCustomEvent(el, type, {
                            triggered: true
                        });
                    };

                    var handleEvent = function (e) {
                        switch (e.type) {
                            case 'max-attempts-reached':
                                buttons.get('check').disable(); //has to exist if feedback

                                if (buttons.active('reveal')) {
                                    buttons.get('reveal').disable();
                                }
                                break;
                            case 'feedback-result':
                                if (e.detail.attempted) {
                                    if (!e.detail.correct) {
                                        feedback.retry.style.display = 'block';

                                        if (!buttons.active('retry')) {
                                            triggerEvent(feedback.retry, 'retry');
                                        }
                                    }
                                }
                                else {
                                    buttons.get('check').enable();
                                    feedback.retry.style.display = 'none';
                                    //TODO this should be a simple call to question.retry (once setup)...
                                    triggerEvent(feedback.retry, 'retry');
                                }

                                utils.resizeIFrame();
                        }
                    };

                    load();
                }
            }
        );
    }
);
define("activities/table_activity", function(){});

define(
    'table',['utils'],
	function (utils) {
        "use strict";

        var plugins = {};

        function Table() {
            //keep count for saving data where multiple instances
            Table.count = ++Table.count || 0;
            console.log('Table count: ' + Table.count);

            this.index = Table.count;
            this.id = 'table' + this.index;
            this.events = []; //subscribed events

            //cfg
            this.pseudo = false;
            this.plugins = [];
            this.placeholder = ''; //content for empty cells (better name?!)
            this.width = 'auto';
            this.height = 'auto';
            this.layout = 'auto';
            this.classes = [];
            this.content = [];
            this.caption = '';
            this.foot = '';
            this.rows = {
                num: 0,
                lang: {}, //where key is row index and value is lang
                headers: false, //true/false/array of indices
                class: true, //applies .row0 etc.
                classes: {}, //where key is row index and value is lang
                height: 'auto'
            };
            this.cols = {
                num: 0,
                lang: {},
                headers: false, //true/false/array of indices
                class: true, //applies .col0 etc.
                classes: {},
                width: 'auto'
            };
            //move to activity class?
            this.images = {
                dir: 'img/',
                lang: '',
                placeholder: '[[image]]',
                files: [],
                alt: []
            };

            //dom
            this.target = null;
            this.container = utils.createElement('div', 'table-container');
            this.el = null;
            this.body = null;
            this.rows.el = [];
            this.cols.el = [];
        }

        Table.prototype = {
            modules: {},

            init: function (cfg, target) {
                this.target = (typeof target == 'string') ? (document.body.querySelector(target) || document.body) : target;

                this.setCfg(cfg);
                this.normaliseData();
                this.resolveMergedCells();
                this.createTable();
                //this.setWidth();
                this.initButtonEvents();
                this.initPlugins();

                return this;
            },

            setCfg: function (cfg) {
                utils.extendDeep(this, cfg);

                if (!Array.isArray(this.rows.headers)) {
                    this.rows.headers = (this.rows.headers === true) ? [0] : [];
                }
                if (!Array.isArray(this.cols.headers)) {
                    this.cols.headers = (this.cols.headers === true) ? [0] : [];
                }
            },

            normaliseData: function(data) {
                data = data || this.content;

                if (Array.isArray(data)) {
                    if (Array.isArray(data[0])) {
                        return this.normaliseContentData(data);
                    }
                    else {
                        return this.normaliseRowData(data);
                    }
                }

                return this.normaliseCellData(data);
            },

            normaliseContentData: function(data) {
                var i;

                for (i = 0; i < this.rows.num; i++) {
                    data[i] = Array.isArray(data[i]) ? data[i] : [];
                    this.normaliseRowData(data[i]);
                }

                return data;
            },

            normaliseRowData: function(data) {
                var i;

                for (i = 0; i < this.cols.num; i++) {
                    data[i] = this.normaliseCellData(data[i]);
                }

                return data;
            },

            normaliseCellData: function(data) {
                if (data === null) {
                    return data;
                }

                if (utils.isObject(data)) {
                    if (data instanceof Element) {
                        data = {
                            el: data
                        };
                    }
                }
                else {
                    data = {
                        html: data
                    };
                }

                if (typeof data.html == 'number') {
                    data.html = data.html.toString();
                }

                //clean these up for use later
                if (typeof data.colspan != 'number' || data.colspan === 0) {
                    data.colspan = 1;
                }
                if (typeof data.rowspan != 'number' || data.rowspan === 0) {
                    data.rowspan = 1;
                }

                return data;
            },

            //TODO deal with merges that are outside of the table, i.e. no row/col for the span to go - data attributes??
            resolveMergedCells: function() {
                var i, span, array;

                //TODO find a 'nicer' way, i.e. without removing existing values
                //clear null values and start again
                this.removeValueFromArray(this.content, null);

                //resolve all colspans first so they aren't screwed up when resolving rowspans
                this.content.forEach(function(data, row, content) {
                    data.forEach(function(data, col) {
console.log('out', col, data);
                        if (data === null) {
                            return;
                        }

                        if (data.colspan > 1) {
                            array = utils.getPopulatedArray(data.colspan - 1, null);
                            utils.spliceArray(content[row], col + 1, array);
                        }
                    });
                });

                this.content.forEach(function(data, row, content) {
                    data.forEach(function(data, col) {
                        if (data === null) {
                            return;
                        }

                        if (data.rowspan > 1) {
                            for (i = row + 1, span = i + data.rowspan - 1; i < span; i++) {
                                if (typeof content[i] != 'undefined') {
                                    array = utils.getPopulatedArray(data.colspan, null);
                                    utils.spliceArray(content[i], col, array);
                                }
                            }
                        }
                    });
                });

                this.removeExtraneousData();
            },

            removeExtraneousData: function() {
                if (this.content.length > this.rows.num) {
                    this.content.splice(this.rows.num);
                }

                this.content.forEach(function(data) {
                    if (data.length > this.cols.num) {
                        data.splice(this.cols.num);
                    }
                }.bind(this));
            },

            removeValueFromArray: function(array, val) {
                for (var i = 0; i < array.length; i++) {
                    if (Array.isArray(array[i])) {
                        this.removeValueFromArray(array[i], val);
                    }
                    else {
                        if (array[i] === val) {
                            array.splice(i, 1);
                            i--;
                        }
                    }
                }

                return array;
            },

            setLanguage: function (el, lang) {
                if (typeof lang == 'string') {
                    el.lang = lang;
                }
            },

            /**
             * create dom elements and store any refs to them
             */
            createTable: function () {
                this.el = document.createElement(this.pseudo ? 'div' : 'table');
                this.body = utils.createElement((this.pseudo ? 'div' : 'tbody'), 'table-body');

                this.el.id = this.id;

                if (this.layout !== 'auto') {
                    this.el.style.tableLayout = this.layout;
                }

                this.setTableClasses();
                this.setLanguage(this.el, this.lang);

                if (this.width !== 'auto') {
                    this.el.style.width = (typeof this.width == 'number') ? (this.width + 'px') : this.width;
                }

                if (typeof this.caption == 'string' && this.caption !== '') {
                    this.createCaption();
                }

                this.content.forEach(function(data, i) {
                    this.rows.el.push(this.body.appendChild(this.createRow(data, i)));
                }.bind(this));

                this.el.appendChild(this.body);

                if (typeof this.foot == 'string' && this.foot !== '') {
                    this.createFoot();
                }

                this.container.appendChild(this.el);

                if (this.target instanceof Element) {
                    this.target.appendChild(this.container);
                }
            },

            setTableClasses: function() {
                this.classes.push('table');

                if (this.borders === false) {
                    this.classes.push('table-no-rules');
                }

                this.el.className = this.classes.join(' ');
            },

            createCaption: function () {
                var tag = (this.pseudo ? 'div' : 'caption');

                this.el.insertAdjacentHTML('beforeend', '<' + tag + ' class="table-caption">' +
                    this.caption +
                    '</' + tag + '>');
            },

            createFoot: function () {
                var tag = (this.pseudo ? 'div' : 'tfoot');

                this.el.insertAdjacentHTML('beforeend', '<' + tag + ' class="table-foot">' +
                    this.foot +
                    '</' + tag + '>');
            },

            createRow: function(data, row) {
                var tr = document.createElement(this.pseudo ? 'div' : 'tr');

                tr.className = 'table-row';

                if (this.rows.lang && this.rows.lang.hasOwnProperty(row)) {
                    this.setLanguage(tr, this.rows.lang[row]);
                }

                data.forEach(function(data, i) {
                    var cell;

                    if (typeof this.cols.el[i] == 'undefined') {
                        this.cols.el[i] = [];
                    }

                    cell = this.createCell(data, row, i);

                    if (cell) {
                        tr.appendChild(cell);
                    }
                }.bind(this));

                return tr;
            },

            createCell: function(data, row, col) {
                var cell;

                if (data === null) {
                    this.cols.el[col].push(null);
                    return data;
                }

                data.row = row;
                data.col = col;
                data.colHead = (this.cols.headers.indexOf(row) > -1);
                data.rowHead = (this.rows.headers.indexOf(col) > -1);
                data.type = (data.colHead || data.rowHead) ? 'th' : 'td';
                cell = document.createElement(this.pseudo ? 'div' : data.type);

                if (data.rowspan > 1) {
                    cell.rowSpan = data.rowspan;
                }
                if (data.colspan > 1) {
                    cell.colSpan = data.colspan;
                }

                if (this.cols.lang && this.cols.lang.hasOwnProperty(col)) {
                    data.lang = this.cols.lang[col];
                }

                this.setLanguage(cell, data.lang);
                this.setCellClasses(data, cell);

                //set dimensions
                if (typeof this.rows.height == 'number' && this.cols.headers.indexOf(row) < 0) {
                    cell.style.height = this.rows.height + 'px';
                }
                if (typeof this.cols.width == 'number' && this.rows.headers.indexOf(col) < 0) {
                    cell.style.width = this.cols.width + 'px';
                }

                if (data.el) {
                    cell.appendChild(data.el);
                }

                if (data.html) {
                    data.html = this.replaceImagePlaceholders(data.html);
                }

                cell.setAttribute('data-row', row.toString());
                cell.setAttribute('data-col', col.toString());
                cell.insertAdjacentHTML('beforeend', ((typeof data.html === 'string') ? data.html : this.placeholder || ''));

                this.cols.el[col].push(cell);

                return cell;
            },

            replaceImagePlaceholders: function(str) {
                var re = new RegExp(utils.escapeRegExp(this.images.placeholder), 'g'),
                    src, alt;

                return str.replace(re, function () {
                    this.images.applied = ++this.images.applied || 0;

                    src = this.images.dir + this.images.files[this.images.applied].src;
                    alt = this.images.alt[this.images.applied] || '';

                    return '<img src="' + src + '" alt="' + alt + '" lang="' + this.images.lang + '" />';
                }.bind(this));
            },

            setCellClasses: function (data, cell) {
                data.classes = Array.isArray(data.classes) ? data.classes : [];
                data.classes.push('table-cell');
                data.classes.push('table-' + data.type);

                if (this.rows.class) {
                    data.classes.push('row' + data.row);
                }
                if (this.rows.classes && this.rows.classes.hasOwnProperty(data.row)) {
                    data.classes.push(this.rows.classes[data.row]);
                }
                if (this.cols.class) {
                    data.classes.push('col' + data.col);
                }
                if (this.cols.classes && this.cols.classes.hasOwnProperty(data.col)) {
                    data.classes.push(this.cols.classes[data.col]);
                }
                if (!data.html && !data.el && this.placeholder) {
                    data.classes.push('table-cell-placeholder');
                }

                cell.className = data.classes.join(' ');
            },

            /**
             * set table (container) width
             */
            setWidth: function () {
                //col width takes precedence over table width
                if (typeof this.cols.width != 'number' && typeof this.width == 'number') {
                    this.container.style.width = this.width + 'px';
                    this.el.style.width = '100%';
                }
            },

            /**
             * loop enabled buttons and add event listeners to container
             */
            initButtonEvents: function () {
                if (this.buttons && Array.isArray(this.buttons.enabled)) {
                    this.buttons.enabled.forEach(function (name) {
                        this.on(name);
                    }.bind(this));
                }
            },

            /**
             * initialise enabled plugins and add event listeners to container
             */
            initPlugins: function () {
                var plugin;

                this.plugins.forEach(function (name) {
                    if (plugins[name]) {
                        plugin = plugins[name].construct();
                        plugin.table = this;
                        //utils.extendDeep(plugin, this[name]);

                        if (Array.isArray(plugin.events)) {
                            plugin.events.forEach(function (e) {
                                this.on(e);
                            }.bind(this));
                        }

                        plugin.init(this[name]);
                        this.plugins[name] = plugin;
                    }
                    else {
                        throw new Error('Plugin not found: ' + name);
                    }
                }.bind(this));
            },

            /**
             * utility function for iterating over content data
             */
            iterateContent: function (callback, args) {
                this.content.forEach(function (row, i) {
                    row.forEach(function (cell, j) {
                        callback.apply(this, [row, cell, i, j].concat(args));
                    }.bind(this));
                }.bind(this));
            },

            getCell: function(row, col) {
                return this.cols.el[col][row] || null;
            },

            /**
             * add event listener to container and push event type to event array
             * @param {string} type - event type
             */
            on: function (type) {
                //check event isn't already there to avoid duplicates
                if (this.events.indexOf(type) < 0) {
                    this.events.push(type);
                    this.target.addEventListener(type, this);
                }
            },

            /**
             * trigger an event on the container
             * @param {string} type - event type
             * @param {object} detail - an object to be passed in e.detail
             * @param {boolean} [bubbles] - allow the event to bubble (false unless === true)
             */
            trigger: function (type, detail, bubbles) {
                if (this.events.indexOf(type) > -1) {
                    utils.triggerCustomEvent(this.container, type, detail, true);
                }
            },

            /**
             * handle events for this and all plugins
             * @param {Event} e
             */
            handleEvent: function (e) {
                var prop = utils.toCamelCase(e.type.replace('table-', ''));

                console.log(e.type);

                if (e.target.nodeName.toLowerCase() != 'button') {
                    e.stopPropagation();
                }

                //trigger main event if one exists (needed?)
                if (prop in this && typeof this[prop] == 'function') {
                    this[prop](e);
                }

                //pass event to plugin.handleEvent
                this.plugins.forEach(function (plugin, i, array) {
                    if (plugin in array && typeof array[plugin].handleEvent == 'function') {
                        array[plugin].handleEvent(e);
                    }
                });
            }
        };

        /**
         * instantiates and a new table instance and returns the table element
         * @param {object} cfg
         * @param {Element|string} [target] - element/selector for element for table to be appended
         * @returns {null|Element|*}
         */
        var create = function(cfg, target) {
            return new Table().init(cfg, target).el;
        };

        /**
         * instantiates and returns a new table instance
         * @param cfg
         * @param target
         * @returns {Table}
         */
        var construct = function(cfg, target) {
            return new Table(cfg, target);
        };

        /**
         * adds plugin to module
         * @param {string} name - plugin name
         * @param {object} module - plugin module
         */
        var extend = function(name, module) {
            plugins[name] = module;
        };

        return {
            create: create,
            construct: construct,
            extend: extend
        };
    }
);
define(
    'table-extend',[
        'table',
        'buttons',
        'utils'
    ],
    function (table, buttons, utils) {
        "use strict";

        table.extend('extend', (function () {
            return {
                construct: function (cfg) {
                    return new TableExtend(cfg);
                }
            };
        })());

        function TableExtend() {
            this.table = null;
            this.events = ['table-extend', 'table-reduce'];

            //cfg
            this.rows = {
                enable: false,
                min: 0,
                max: 20,
                controls: {
                    height: 40,
                    add: {
                        html: '+',
                        label: 'Add row'
                    },
                    del: {
                        html: '&#8722;',
                        label: 'Delete row'
                    }
                }
            };
            this.cols = {
                enable: false,
                min: 0,
                max: 20,
                controls: {
                    width: 40,
                    add: {
                        html: '+',
                        label: 'Add col'
                    },
                    del: {
                        html: '&#8722;',
                        label: 'Delete col'
                    }
                }
            };
        }

        TableExtend.prototype = {
            init: function (cfg) {
                this.setCfg(cfg);

                if (this.cols.enable) {
                    this.initColControls();
                }
                if (this.rows.enable) {
                    this.initRowControls();
                }
            },

            setCfg: function (cfg) {
                utils.extendDeep(this, cfg);

                if (this.rows.min === 0) {
                    this.rows.min = this.table.rows.num;
                }
                if (this.cols.min === 0) {
                    this.cols.min = this.table.cols.num;
                }
            },

            initRowControls: function () {
                var control = utils.createElement('div', 'table-extend-controls table-extend-controls-row'),
                    container = utils.createElement('div', 'table-container-lower');

                control.style.height = this.rows.controls.height + 'px';

                control.appendChild(this.createControl('add', 'row', container));
                control.appendChild(this.createControl('del', 'row', container));

                container.appendChild(control);
                container.insertAdjacentHTML('beforeend', '<div></div>');

                this.table.container.appendChild(container);
            },

            initColControls: function () {
                var control = utils.createElement('div', 'table-extend-controls table-extend-controls-col'),
                    container = utils.createElement('div', 'table-container-upper');

                control.style.width = this.cols.controls.width + 'px';

                control.appendChild(this.createControl('add', 'col', container));
                control.appendChild(this.createControl('del', 'col', container));

                container.appendChild(utils.createElement('div')).appendChild(this.table.el);
                container.appendChild(control);

                this.table.container.appendChild(container);
            },

            createControl: function (action, type, container) {
                var control = this[type + 's'].controls[action],
                    event = (action == 'add') ? 'table-extend' : 'table-reduce';

                Object.assign(control, {
                    type: action + '-' + type,
                    classes: ['btn-no-style', 'btn-table-control'],
                    click: utils.triggerCustomEvent.bind(null, container, event, {type: type})
                });

                control = buttons.construct(control);

                return control.el;
            },

            addRow: function (data, row) {
                if (this.table.rows.num == this.rows.max) {
                    return false;
                }

                row = row || this.table.rows.num++;
                data = data || utils.getPopulatedArray(this.table.cols.num, {});

                //update content with data or array of empty objects if non passed
                this.table.content.push(data);

                //create and append new row
                this.table.rows.el.push(this.table.createRow(data, row));
                this.table.body.appendChild(this.table.rows.el[row]);
            },

            addCol: function (data, col) {
                if (this.table.cols.num == this.cols.max) {
                    return false;
                }

                col = col || this.table.cols.num++;
                data = data || utils.getPopulatedArray(this.table.rows.num, {});

                this.table.cols.el.push([]);

                this.table.content.forEach(function (row, i) {
                    var cell = this.table.createCell(data[i], i, col);

                    //update content with data
                    row.push(data[i]);

                    if (cell) {
                        //create and append new cell
                        this.table.rows.el[i].appendChild(cell);
                    }
                }.bind(this));
            },

            deleteRow: function () {
                if (this.table.rows.num == this.rows.min) {
                    return false;
                }

                this.table.rows.num--;
                this.table.content.pop();

                this.table.body.removeChild(this.table.rows.el[this.table.rows.num]);

                this.table.rows.el.pop();
                this.table.cols.el.forEach(function (col) {
                    col.pop();
                });
            },

            deleteCol: function () {
                if (this.table.cols.num == this.cols.min) {
                    return false;
                }

                this.table.cols.num--;
                this.table.content.forEach(function (row) {
                    row.pop();
                });

                this.table.cols.el.pop();
                this.table.rows.el.forEach(function (row) {
                    row.removeChild(row.lastChild);
                }.bind(this));

                VLE.resize_iframe();
            },

            update: function () {
                var row = this.table.content.splice(this.table.rows.num),
                    col = [];

                this.table.content.forEach(function (data) {
                    data = data.splice(this.table.cols.num);

                    data.forEach(function (data, i) {
                        col[i] = col[i] || [];
                        col[i].push(data);
                    });
                }.bind(this));

                col.forEach(function (data) {
                    this.addCol(data);
                }.bind(this));

                row.forEach(function (data) {
                    this.addRow(data);
                }.bind(this));
            },

            reset: function () {
                while (this.table.rows.num > this.rows.min) {
                    this.deleteRow();
                }

                while (this.table.cols.num > this.cols.min) {
                    this.deleteCol();
                }

                VLE.resize_iframe();
            },

            handleEvent: function (e) {
                switch (e.type) {
                    case 'data-loaded':
                        this.update();
                        break;
                    case 'table-extend':
                        /*if ((e.detail.type == 'col' && this.table.cols.num == this.cols.max) ||
                         (e.detail.type == 'row' && this.table.rows.num == this.rows.max)) {
                         e.stopImmediatePropagation();
                         return;
                         }*/

                        if (e.detail.type == 'col') {
                            this.addCol();
                        }
                        else {
                            this.addRow();
                        }
                        break;
                    case 'table-reduce':
                        /*if ((e.detail.type == 'col' && this.table.cols.num == this.cols.min) ||
                         (e.detail.type == 'row' && this.table.rows.num == this.rows.min)) {
                         e.stopImmediatePropagation();
                         return;
                         }*/

                        if (e.detail.type == 'col') {
                            this.deleteCol();
                        }
                        else {
                            this.deleteRow();
                        }
                        break;
                    case 'reset':
                        this.reset();
                }
            }
        };
    }
);
define(
	'table-input',[
		'table',
		'input',
		'utils'
	],
	function (table, input, utils) {
		"use strict";

		table.extend('input', (function () {
			return {
				construct: function(cfg) {
					return new TableInput(cfg);
				}
			};
		})());

        //TODO add showCorrect/showIncorrect
        //TODO add removeIncorrect
        function TableInput() {
            this.table = null;
            this.events = ['change', 'feedback', 'retry'];
            this.attempt = null;
            this.rows = {
                num: 0
            };
            this.cols = {
                num: 0
            };

            //cfg
            this.type = '';
            this.placeholder = {
                start: '[[',
                end: ']]'
            };
            //TODO move each input type cfg to an object/class along with all specific functionality?
            this.checkbox = {
                markUnchecked: false
            };
            this.radio = {
                group: 'rows'
            };
            this.text = {
                width: '100%',
                borders: false,
                showAsHTML: false
            };
            this.textarea = {
                width: '100%',
                height: '100%',
                borders: false,
                resizeOnReveal: true,
                showAsHTML: false
            };
            this.select = {
                options: [],
                shuffle: false
            };
            this.answer = {
                data: [],
                overlay: true,
                showBelowInput: false,
                hideInputOnReveal: false,
                clearIncorrect: true
            };

            //dom
            this.nodes = [];
            this.answer.el = null;
        }

        TableInput.prototype = {
            init: function(cfg) {
                this.setCfg(cfg);
                this.initInputs();

                if (this.answer.data.length > 0) {
                    this.setAnswer();
                }

                this.initAnswer();
                this.getAnswer();
            },

            setCfg: function(cfg) {
                if (cfg && ['text', 'textarea'].indexOf(cfg.type) > -1) {
                    this.answer.overlay = false;
                }

                utils.extendDeep(this, cfg);

                if (this.text.showAsHTML || this.textarea.showAsHTML) {
                    this.answer.showBelowInput = true;
                }

                //support for old data file format - remove in next version
                if (this.hasOwnProperty('options')) {
                    this.select.options = this.options;
                    delete this.options;
                }

                if (this.hasOwnProperty('shuffle')) {
                    this.select.shuffle = this.shuffle;
                    delete this.shuffle;
                }

                if (this.rows.hasOwnProperty('group')) {
                    if (this.rows.group === false) {
                        this.radio.group = false;
                    }
                    delete this.rows.group;
                }

                if (this.cols.hasOwnProperty('group')) {
                    if (this.cols.group === true) {
                        this.radio.group = 'cols';
                    }
                    delete this.cols.group;
                }

                if (Array.isArray(this.answer.data) && this.answer.data.length) {
                    if (Array.isArray(this.answer.data[0])) {
                        this.answer.data = [].concat.apply([], this.answer.data);
                    }
                }
                //end old support
            },

            initInputs: function() {
                var start = utils.escapeRegExp(this.placeholder.start),
                    end = utils.escapeRegExp(this.placeholder.end),
                    re = new RegExp(start + 'input' + end, 'g');

                this.table.content.forEach(function (data, row) {
                    data.forEach(function (data, col) {
                        if (row < this.rows.num && col < this.cols.num) {
                            return;
                        }

                        if (data === null) {
                            return;
                        }

                        //support for input array shorthand (e.g. ['radio', 'textarea']).
                        if (Array.isArray(data.input)) {
                            data.input.forEach(function (data, i, arr) {
                                if (typeof data == 'string' && input.isInput(data)) {
                                    arr[i] = {
                                        type: data
                                    };
                                }
                            });
                        }

                        if (!data.hasOwnProperty('input')) {
                            if (this.type == 'select' || !input.isInput(this.type)) {
                                return;
                            }

                            if (data.html && re.test(data.html)) {
                                data.input = [];

                                data.html = data.html.replace(re, function () {
                                    data.input.push({
                                        type: this.type
                                    });

                                    return '';
                                }.bind(this));


                                if (data.input.length == 1) {
                                    data.input = data.input[0];
                                }

                                this.table.getCell(row, col).innerHTML = data.html;
                            }

                            if (typeof data.html == 'undefined') {
                                data.input = {
                                    type: this.type
                                };
                            }
                        }

                        if (data.input) {
                            this.createInputs(data, row, col);
                        }
                    }.bind(this));
                }.bind(this));

                this.rows.num = this.table.rows.num;
                this.cols.num = this.table.cols.num;

                //console.log(this.rows.num, this.cols.num);
            },

            createInputs: function(data, row, col) {
                var cell = this.table.getCell(row, col),
                    obj, arr, group;

                data.input = Array.isArray(data.input) ? data.input : [data.input]; //ensure array

                arr = data.input.map(function(data, i) {
					obj = this.createInputObject(data, cell, row, col, i);

					if (!obj.label) {
                        this.setLabel(obj, cell, row, col);
                    }

                    if (!obj.prefilled) {
                    	this.nodes.push(obj);
					}

                    return obj;
                }.bind(this));

                group = input.create(arr);
                cell.appendChild(group);

                //any further configuration
				arr.forEach(function (input, i) {
                    utils.addClass(cell, 'cell-' + input.type);

					switch (input.type) {
						case 'text':
						    if (!this.text.borders) {
						        utils.addClass(input.el, 'no-borders');
                            }
                            utils.setBoxDimension(input.el, 'width', input.width);
						    break;
						case 'textarea':
                            if (!this.textarea.borders) {
                                utils.addClass(input.el, 'no-borders');
                            }
                            utils.setBoxDimension(input.el, 'width', input.width);
                            utils.setBoxDimension(input.el, 'height', input.height);
					}
                }.bind(this));
			},

            /**
             * take a copy of data storing a ref on the new object. This keeps data clean for saving etc.
             * @param {object} data - input data
			 * @param {Element} cell
			 * @param {number} row
			 * @param {number} col
             * @param {number} index
             */
			createInputObject: function(data, cell, row, col, index) {
                var obj = Object.assign({}, data);

                obj.data = data;
                obj.type = obj.type || this.type;
                obj.cell = cell;
                obj.row = row;
                obj.col = col;
                obj.index = index;
                obj.prefilled = !!obj.prefilled;
                obj.answer = null; //will hold ref to answer element if present

                switch (obj.type) {
                    case 'radio':
                        if (this.radio.group !== false) {
                            //where single radio to a cell, group as rows unless set to cols
                            obj.name = 'radio-group' + (this.radio.group == 'cols' ? col : row);
                        }
                        break;
                    case 'text':
                    	obj.width = obj.width || this.text.width;
                        break;
                    case 'textarea':
                        obj.width = obj.width || this.textarea.width;
                    	obj.height = obj.height || this.textarea.height;
                        break;
                    case 'select':
                        if (typeof data.options == 'number') {
                            obj.shuffle = !!this.select.options[data.options].shuffle;
                            obj.options = this.select.options[data.options].data.slice(0);
                        }
                }

                if (obj.prefilled) {
                    obj.disabled = true;
                }

                return obj;
            },

            setLabel: function(data, cell, row, col) {
                var header = (this.table.cols.headers.indexOf(row) > -1 || this.table.rows.headers.indexOf(col) > -1),
                    str = '';

                //where cell isn't a header, use the nearest col/row header if one/both exist
                if (!header) {
                    if (this.table.cols.headers.length > 0 || this.table.rows.headers.length > 0) {
                        if (this.table.cols.headers.length > 0) {
                            data.label = [];
                            data.label.push(this.table.cols.el[col][utils.nextLowest(this.table.cols.headers, row)]);
                        }
                        if (this.table.rows.headers.length > 0) {
                            data.label = data.label || [];
                            data.label.push(this.table.cols.el[utils.nextLowest(this.table.rows.headers, col)][row]);
                        }
                        return;
                    }
                }

                //last resort - indicate if head and/or give cell position
                str += (header ? 'Table head: ' : '');
                str += 'Row ' + (row + 1) + ', column ' + (col + 1);

                data.label = document.createElement('label');
                data.label.textContent = str;
                data.label.className = 'screen-reader';

                cell.appendChild(data.label);
            },

            /**
             * Set answer data in this.table.content if answer provided separately (this.answer.data)
             */
            setAnswer: function() {
                console.log(this.answer.data);

                this.nodes.forEach(function(input, i) {
                    input.data.answer = this.answer.data[i];
                }.bind(this));
            },

            /**
             * Any type specific initialisation of answer
             */
            initAnswer: function() {
                this.initInputAnswerData();
                this.disableInputs();

                if (this.answer.showBelowInput) {
                    this.initInputAnswerElement();
                }

                if (!this.answer.overlay) {
                    this.toggle();
                    this.answer.el = utils.createElement('div', 'table-answer');
                    this.answer.el.appendChild(this.table.container.cloneNode(true));
                    this.toggle(true);
                }

                this.enableInputs();
            },

            initInputAnswerData: function () {
                this.nodes.forEach(function (input) {
                    switch (input.type) {
                        case 'checkbox':
                            if (!this.checkbox.markUnchecked && input.answer === 0) {
                                input.data.answer = null;
                            }
                            break;
                        case 'radio':
                            if (input.answer === 0) {
                                input.data.answer = null;
                            }
                            break;
                        case 'select':
                            if (typeof input.data.answer == 'number') {
                                //account for shuffle/placeholder in answer
                                utils.toArray(input.el.options).some(function (el, i) {
                                    if (el.value == input.data.answer) {
                                        input.data.answer = i;
                                        return true;
                                    }
                                });
                            }
                    }
                }.bind(this));
            },

            initInputAnswerElement: function () {
                this.nodes.forEach(function (input) {
                    input.answer = input.el.cloneNode(true);
                    input.display = getComputedStyle(input.el).display;

                    switch (input.type) {
                        case 'checkbox':
                        case 'radio':
                            input.answer.checked = !!input.data.answer;
                            input.answer.name = ''; //break link with other radios
                            break;
                        case 'text':
                            if (this.text.showAsHTML) {
                                input.answer = document.createElement('div');
                                input.answer.insertAdjacentHTML('beforeEnd', (input.data.answer || ''));
                                utils.setBoxDimension(input.answer, 'width', input.width);
                            }
                            else {
                                input.answer.value = input.data.answer || '';
                            }
                            break;
                        case 'textarea':
                            if (this[input.type].showAsHTML) {
                                input.answer = document.createElement('div');
                                input.answer.insertAdjacentHTML('beforeEnd', (input.data.answer || ''));
                                utils.setBoxDimension(input.answer, 'width', input.width);

                                if (!this.textarea.resizeOnReveal) {
                                    utils.setBoxDimension(input.answer, 'height', input.height);
                                }
                            }
                            else {
                                input.answer.value = input.data.answer || '';
                            }
                            break;
                        case 'select':
                            input.answer.selectedIndex = input.data.answer || 0;
                    }

                    utils.addClass(input.answer, 'input-answer');
                    input.el.parentNode.insertBefore(input.answer, input.el.nextSibling);
                }.bind(this));
            },

            getInputObject: function(target) {
				var obj = null;

				this.nodes.some(function(node) {
					if (node.el == target) {
						obj = node;
						return true;
					}
				});

				return obj;
			},

            /**
             * updates relevant input data on change
             * @param {Event} e
             */
            change: function(e) {
                var input = this.getInputObject(e.target);

                switch (input.type) {
                    case 'checkbox':
                        if (e.target.checked) {
                            input.data.attempt = 1;
                        }
                        else {
                            delete input.data.attempt;
                        }
                        break;
                    case 'radio':
                        input.data.attempt = 1;

                        if (['cols', 'rows'].indexOf(this.radio.group)) {
                            this.updateRadioGroup(input);
                        }
                        break;
                    case 'text':
                    case 'textarea':
                        input.data.attempt = e.target.value.trim();

                        if (!input.data.attempt) {
                            delete input.data.attempt;
                        }
                        break;
                    case 'select':
                        //TODO start index at 1 so we can delete all falsy answer after switch
                        input.data.attempt = parseInt(e.target.selectedIndex, 10);
                }
            },

            /**
             * update data for radio group
             * @param {object} target - target input of original event
             */
            updateRadioGroup: function(target) {
                this.nodes.forEach(function(input) {
                    if (input.type != 'radio' || input == target) {
                        return;
                    }

                    if (input.name == target.name) {
                        delete input.data.attempt;
                    }
                }.bind(this));
            },

            /**
             * remove input data and any references held in for attributes
             */
            removeInputs: function() {
                var index = this.nodes.length - 1,
                    node;

                this.rows.num = this.table.rows.num;
                this.cols.num = this.table.cols.num;

                while (index) {
                    node = this.nodes[index];

                    if (node.row >= this.rows.num || node.col >= this.cols.num) {
                        if (Array.isArray(node.label)) {
                            node.label.forEach(function (el) {
                                 if (el instanceof Element && this.table.el.contains(el)) {
                                     el.setAttribute('for', el.getAttribute('for').replace(node.el.id, '').trim());
                                 }
                            }.bind(this));
                        }

                        this.nodes.splice(index, 1);
                    }

                    index -= 1;
                }
            },

            updateInputs: function() {
				this.nodes.forEach(function (node) {
					switch (node.type) {
						case 'radio':
						case 'checkbox':
							node.el.checked = !!node.data.attempt;
							break;
                        case 'text':
                        case 'textarea':
                            node.el.value = node.data.attempt || node.data.value || '';
                            break;
						case 'select':
							node.el.selectedIndex = node.data.attempt;
                    }
				});
            },

			hideInputs: function() {
            	this.nodes.forEach(function (input) {
            		input.el.style.display = 'none';
				});
			},

			showInputs: function() {
                this.nodes.forEach(function (input) {
                    input.el.style.display = input.display;
                });
			},

            resizeTextAreaToContent: function (el) {
                el.style.height = 'auto';
                el.style.overflow = 'hidden';
                el.style.height = el.scrollHeight + 'px';
                el.style.overflow = 'auto';
            },

            /**
             * get answer to check against
             */
            getAnswer: function() {
                this.answer.data = this.nodes.map(function(input) {
                    return input.data.answer || null;
                });
            },

            /**
			 * get attempt to check
             */
            getAttempt: function() {
                this.attempt = this.nodes.map(function(input) {
                    return input.data.attempt || null;
                });
            },

            submit: function() {
                this.disableInputs();
                this.getAttempt();

                this.table.trigger('feedback', {
                    attempt: this.attempt,
                    answer: this.answer.data
                }, true);
            },

            reveal: function(attempt) {
                if (this.answer.overlay) {
                    if (attempt) {
                        //this.enableInputs();
                        this.toggle(true);
                    }
                    else {
                        this.disableInputs();
                        this.toggle();
                    }
                }
                else {
                    this.disableInputs();
                    this.table.target.appendChild(this.answer.el);
                }

                utils.resizeIFrame();
            },

            /**
             * Toggle attempt/answer
             * @param {boolean} [attempt] - attempt if true, answer otherwise
             */
            toggle: function(attempt) {
                var answer = attempt ? 'attempt' : 'answer';

                if (this.answer.showBelowInput) {
                    this.nodes.forEach(function(input) {
                        console.log(input.display);
                        input.answer.style.display = attempt ? 'none' : input.display;

                        if (this.answer.hideInputOnReveal) {
                            input.el.style.display = attempt ? input.display : 'none';
                        }
                    }.bind(this));
                }
                else {
                    this.resetInputElements(); //reset to initial state before showing answer/attempt

                    this.nodes.forEach(function (input) {
                        if (input.data.hasOwnProperty(answer)) {
                            switch (input.type) {
                                case 'radio':
                                case 'checkbox':
                                    input.el.checked = input.data[answer];
                                    break;
                                case 'text':
                                    input.el.value = input.data[answer];
                                    break;
                                case 'textarea':
                                    input.el.value = input.data[answer];
                                    break;
                                case 'select':
                                    input.el.selectedIndex = input.data[answer];
                            }
                        }

                        //do any type specific stuff
                        switch (input.type) {
                            case 'textarea':
                                if (this.textarea.resizeOnReveal) {
                                    if (attempt) {
                                        utils.setBoxDimension(input.el, 'height', input.height);
                                    }
                                    else {
                                        this.resizeTextAreaToContent(input.el);
                                    }
                                }
                        }
                    }.bind(this));
                }
            },

            /**
             * check if col/row contains user content
             * @param {string} type - col/row
             * @param {number} [index] - col/row index, otherwise last col/row
             * @returns {boolean}
             */
            hasContent: function(type, index) {
                index = (typeof index == 'number') ? index : this.table[type + 's'].num - 1;

                return this.nodes.some(function(input) {
                    return (input[type] === index && input.data.attempt);
                });
            },

            enableInputs: function(inputs) {
                inputs = inputs || this.nodes;

                inputs.forEach(function(input) {
                    input.el.removeAttribute('disabled');
                });
            },

            disableInputs: function(inputs) {
                inputs = inputs || this.nodes;

                inputs.forEach(function(input) {
                    input.el.disabled = true;
                });
            },

            retry: function() {
                this.enableInputs();

                if (this.answer.clearIncorrect) {
                    this.clearIncorrect();
                }
            },

            clearIncorrect: function () {
                this.nodes.forEach(function (input) {
                    if (input.data.hasOwnProperty('attempt')) {
                        if (input.data.attempt !== input.data.answer) {
                            delete input.data.attempt;
                            this.resetInputElement(input);
                        }
                    }
                }.bind(this));
            },

            /**
             * Reset input data and show.
             */
            reset: function() {
                this.removeInputs();
                this.resetInputElements();
                this.resetInputData();
                this.enableInputs();

                if (this.answer.overlay) {
                    this.toggle(true);
                }
                else {
                    if (this.table.target.contains(this.answer.el)) {
                        this.table.target.removeChild(this.answer.el);
                    }
                }

                utils.resizeIFrame();
            },

            resetInputElement: function(input) {
                switch (input.type) {
                    case 'checkbox':
                        input.el.checked = !!input.data.checked;
                        break;
                    case 'radio':
                        input.el.checked = !!input.data.checked;
                        break;
                    case 'text':
                    case 'textarea':
                        input.el.value = input.data.value || '';
                        break;
                    case 'select':
                        input.el.selectedIndex = 0;
                }
            },

            resetInputElements: function() {
                this.nodes.forEach(function(input) {
                    this.resetInputElement(input);
                }.bind(this));
            },

            resetInputData: function() {
                this.nodes.forEach(function(input) {
                    if (input.data.hasOwnProperty('attempt')) {
                        delete input.data.attempt;
                    }
                });
            },

            /**
             * Handles all module events
             * @param {event} e
             * @param {string} e.type
             * @param {Element} e.target
             */
            handleEvent: function(e) {
                console.log(e.type);

                switch (e.type) {
                    case 'data-loaded':
                        this.updateInputs();
                        break;
                    case 'check':
                        this.submit();
                        break;
                    case 'change':
                        this.change(e);
                        break;
                    case 'reveal':
                        this.reveal(e.target.getAttribute('aria-pressed') == 'false');
                        break;
                    case 'reset':
                        this.reset();
                        break;
                    case 'retry':
                        console.log('RETRY.............');
                        this.retry();
                        break;
                    case 'table-extend':
                        this.initInputs();
                        break;
                    case 'table-reduce':
                        this.removeInputs();
                }
            }
        };
	}
);
define(
	'table-save',[
		'utils',
		'table',
		'saver'
	],
	function (utils, table, saver) {
		"use strict";

		table.extend('save', (function (){
			return {
				construct: function(cfg) {
					return new TableSave(cfg);
				}
			};
		})());

        function TableSave() {
            this.table = null;
            this.events = ['save', 'data-loaded'];
        }

        TableSave.prototype = {
            init: function (cfg) {
                this.setCfg(cfg);
                this.retrieve();
            },

            setCfg: function (cfg) {
                utils.extendDeep(this, cfg);
            },

            retrieve: function () {
                saver.retrieve({
                    names: this.getSaveObj(true),
                    callback: function (data) {
                        var p, i, num, obj = {};

                        if (data[this.table.id + '-rows'] !== '') {
                            //this.table.rows.num = parseInt(data[this.table.id + '-rows'], 10);
                            //this.table.cols.num = parseInt(data[this.table.id + '-cols'], 10);

                            //for (i = 0; i < this.table.rows.num; i++) {
                            for (i = 0, num = parseInt(data[this.table.id + '-rows'], 10); i < num; i++) {
                                obj[this.table.id + '-row' + i] = '';
                            }

                            saver.retrieve({
                                names: obj,
                                callback: function (data) {
                                    for (p in data) {
                                        if (data.hasOwnProperty(p) && data[p] !== '') {
                                            utils.extendDeep(this.table.content[p.replace(this.table.id + '-row', '')], JSON.parse(data[p]));
                                        }
                                    }

                                    this.table.trigger('data-loaded');
                                }.bind(this)
                            });
                        }
                    }.bind(this)
                });
            },

            getSaveObj: function (retrieve) {
                var obj = {};

                obj[this.table.id + '-rows'] = this.table.rows.num;
                obj[this.table.id + '-cols'] = this.table.cols.num;

                if (!retrieve) {
                    //chunk content in case large amounts of data (saving to the VLE)
                    this.table.content.forEach(function (row, i) {
                        obj[this.table.id + '-row' + i] = JSON.stringify(row);
                    }.bind(this));
                }

                return obj;
            },

            saveData: function () {
                saver.save({
                    values: this.getSaveObj()
                });
            },

            handleEvent: function (e) {
                //console.log(e.type);

                if (e.type == 'save') {
                    this.saveData();
                }
            }
        };
	}
);
