/*global $:false, data */

function isCanvasSupported() {
    var elem = document.createElement('canvas');
    return !!(elem.getContext && elem.getContext('2d'));
}

var canvas,
    context,
    sprite = {},
    sight,
    hit = false,
    hitPosition;

function init() {
    sprite.image = new Image();
    sprite.image.src = 'images/image.png';
    canvas = document.getElementById('canvas1');
    context = canvas.getContext("2d");
    sight = {
        x: canvas.width / 2,
        y: canvas.height / 2,
        width: 30,
        height: 30
    };
    hitPosition = {
        x: 0,
        y: 0
    };

    $("body").append("<input id='submit' type='button' value='Save'/>");
    $("body").append("<input id='reset' type='button' value='Reset'/>");

    retrieveDataFromVLE();
}

function draw() {
    context.clearRect(0, 0, canvas.width, canvas.height);

    context.drawImage(sprite.image, 0, 0, 500, 566);

    context.strokeStyle = 'black';
    context.lineWidth = 1;
    context.beginPath();
    context.moveTo(sight.x - sight.width / 2, sight.y);
    context.lineTo(sight.x + sight.width / 2, sight.y);
    context.moveTo(sight.x, sight.y - sight.height / 2);
    context.lineTo(sight.x, sight.y + sight.height / 2);
    context.stroke();

    context.beginPath();
    context.arc(sight.x, sight.y, 10, 0, 2 * Math.PI);
    context.stroke();

    if (hit) {
        context.strokeStyle = 'red';
        context.lineWidth = 8;
        var width = 20;
        context.beginPath();
        context.moveTo(hitPosition.x - width, hitPosition.y - width);
        context.lineTo(hitPosition.x + width, hitPosition.y + width);
        context.moveTo(hitPosition.x + width, hitPosition.y - width);
        context.lineTo(hitPosition.x - width, hitPosition.y + width);
        context.stroke();
    }
}

function render() {
    draw();
    requestAnimationFrame(render);
}

function reset(feedbackData) {
    hit = false;
    hitPosition.x = 0;
    hitPosition.y = 0;

    saveDataToVLE();
}

function saveDataToVLE() {
    var courseId = VLE.get_param("courseId") || VLE.get_param("_c");
    var documentId = VLE.get_param("documentId") || VLE.get_param("_i");
    var activityId = VLE.get_param("activityId") || VLE.get_param("_a");

    var values = {
        x: 0,
        y: 0
    };

    values.x = hitPosition.x;
    values.y = hitPosition.y;

    var previousValues;

    VLE.set_server_data(true, values, function () {
        console.log("set_server_data was successful");
    }, function (msg) {
        console.log("set_server_data has failed");
        if (msg !== null) {
            console.log(msg);
        }
    }, previousValues, null, activityId, documentId, courseId);
}

function retrieveDataFromVLE() {
    var courseId = VLE.get_param("courseId") || VLE.get_param("_c");
    var documentId = VLE.get_param("documentId") || VLE.get_param("_i");
    var activityId = VLE.get_param("activityId") || VLE.get_param("_a");

    var array = [];
    array.push("x");
    array.push("y");

    VLE.get_server_data(true, array, function (values) {
        if (values.x !== "" && values.y !== "") {
            hitPosition.x = parseFloat(values.x);
            hitPosition.y = parseFloat(values.y);
        }

        if (hitPosition.x !== 0 && hitPosition.y !== 0) {
            hit = true;
        }
    }, function (msg) {
        console.log("Failed to get server data.");
        if (msg !== null) {
            console.log(msg);
        }
    }, activityId, documentId, courseId);
}

$(function () {
    if (isCanvasSupported()) {
        var lastTime = 0;
        var vendors = ['webkit', 'moz'];
        var i;
        for (i = 0; i < vendors.length && !window.requestAnimationFrame; ++i) {
            window.requestAnimationFrame = window[vendors[i] + 'RequestAnimationFrame'];
            window.cancelAnimationFrame =
                window[vendors[i] + 'CancelAnimationFrame'] || window[vendors[i] + 'CancelRequestAnimationFrame'];
        }

        if (!window.requestAnimationFrame) {
            window.requestAnimationFrame = function (callback, element) {
                var currTime = new Date().getTime();
                var timeToCall = Math.max(0, 16 - (currTime - lastTime));
                var id = window.setTimeout(function () {
                    callback(currTime + timeToCall);
                }, timeToCall);

                lastTime = currTime + timeToCall;
                return id;
            };
        }

        if (!window.cancelAnimationFrame) {
            window.cancelAnimationFrame = function (id) {
                clearTimeout(id);
            };
        }

        init();

        requestAnimationFrame(render);
    }

    $("body").on("contextmenu", "#canvas1", function (e) {
        return false;
    });

    $("#canvas1").mouseup(function (event) {
        var rect = this.getBoundingClientRect();
        var offsetX = event.clientX - rect.left;
        var offsetY = event.clientY - rect.top;

        if (event.which === 1) {
            hit = true;
            hitPosition.x = offsetX;
            hitPosition.y = offsetY;
        }
    });

    $("#canvas1").mousemove(function (event) {
        var rect = this.getBoundingClientRect();
        sight.x = event.clientX - rect.left;
        sight.y = event.clientY - rect.top;
    });

    $("#canvas1").keydown(function (event) {
        if ([13, 32, 37, 38, 39, 40].indexOf(event.keyCode) > -1) {
            event.preventDefault();
        }

        if (event.which === 32 || event.which == 13) {
            hit = true;
            hitPosition.x = sight.x;
            hitPosition.y = sight.y;
        }
        else if (event.which == 37) {
            sight.x -= 3;
        }
        else if (event.which == 39) {
            sight.x += 3;
        }
        else if (event.which == 38) {
            sight.y -= 3;
        }
        else if (event.which == 40) {
            sight.y += 3;
        }
    });

    $("#submit").click(function () {
        saveDataToVLE();
    });

    $("#reset").click(function () {
        reset();
    });
});
