/**
 * @fileOverview  Overlay Object
 * @param {JSON array} data Data as included in the data/data.js file
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 * @modified Martin Donnelly

 Additions: 2011-11-03
 -------------------------------------------------------------------------------
 In the data.js file under slides there are three new settings:
 clickcolour:'rgba( 102,255,102, 0.5)'
 highlight:true / false
 persistant:true / false

 clickcolour sets the click colour for that particular slide
 highlight: if true a marker is displayed when clicked on
 persistant: if true, all the markers on a slide can be turned on / off
 simultaneously

 Under Markers:

 shape:'circle', radius:10
 This sets the marker to a circle with the specified radius

 shape:'rectangle',w:440,h:140
 This sets the marker to be a rectangle with the specified width / height

 shape:'poly',points:[{x:100,y:20},{x:120,y:80},{x:180,y:80},{x:140,y:120},#
 {x:180,y:180},{x:100,y:140},{x:20,y:180},{x:60,y:120},{x:20,y:80},{x:80,y:80}]
 This sets the marker to be a polygon with the specified points

 */
OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.NavButtons');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
/**
 * @class Overlay activity, works like a slideshow, with added hotspots.
 * @extends OU.util.Activity
 * @param {object} data - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.Overlay = function(data, instance, controller) {

    /**
     * Starting point when running with HTML5 Canvas Support
     */
    OU.activity.Overlay.prototype.canvasView = function() {
        var bH = OU.controlHeight, self = this;
        this.config = {
            imageLoadTimeout: 10000,
            // allow max 10 seconds to load images
            captionFont: "bold 24px georgia, serif",
            useNavButtons: true,
            markerRadius: 10,
            markerCol1: '#c00',
            // outer ring
            markerCol2: '#fff',
            // inner ring
            // Settings for Nav Buttons
            overlayControls: false,
            // set to false to seperate controls from images
            // animation settings
            autoScroll: false,
            sliderSpeed: 5,
            fps: 40 // 40ms = 25 fps
        };

        this.imageLayer = new OU.util.Layer({
            container: this,
            hasEvents: true,
            zIndex: OU.DATA_LEVEL
        });
        this.controlLayer = new OU.util.Layer({
            container: this,
            hasEvents: true,
            y: this.y + this.h - bH,
            h: bH,
            zIndex: OU.CONTROL_LEVEL,
            id: 'cl'
        });
        this.displayClickLayer = new OU.util.Layer({
            container: this,
            zIndex: OU.CONTROL_LEVEL - 1,
            id: 'm',
            h: this.h - bH,
            hasEvents: true
        });

        // determine Proportions
        if (this.config.overlayControls)
            this.config.slideHeight = this.h;
        else
            this.config.slideHeight = this.h - bH;
        this.config.maxHeight = this.config.slideHeight;
        this.slideNumber = OU.LocalStorage.load("OU.overlay.slideNum") || 0;

        if (this.slideNumber > this.data.slides.length)
            this.slideNumber = 0;
        this.slideNumber = this.slideNumber | 0;

        if (this.data.slides.length === 1)
            this.config.useNavButtons = false;
        this.slides = this.data.slides;
        // setup callbacks for events
        this.displayClickLayer.events.moveRight = function() {
            self.nextSlide();
        };
        this.displayClickLayer.events.moveLeft = function() {
            self.prevSlide();
        };
        // Add Nav buttons if required and more than 1 slide
        if (this.config.useNavButtons && this.data.slides.length > 1)
            this.navButtons = new OU.util.NavButtons({
                x: 0,
                y: 0,
                w: this.w,
                h: bH,
                bw: bH,
                pad: bH * .1,
                context: this.controlLayer.context,
                layer: this.controlLayer,
                leftFunc: function() {
                    self.prevSlide();
                },
                rightFunc: function() {
                    self.nextSlide();
                }
            });
        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.data.slides,
            onLoad: function() {
                self.render();
            }
        });
    };
    /**
     * Resizes the activity
     */
    OU.activity.Overlay.prototype.resize = function() {
        OU.activity.Overlay.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight;
        if (this.popUp)
            this.popUp.close();
        this.displayClickLayer.resize({
            h: this.h - bH
        });
        this.imageLayer.resize();
        this.controlLayer.resize({
            x: this.x,
            y: this.y + this.h - bH,
            w: this.w,
            h: bH
        });
        if (this.navButtons) {
            this.navButtons.move({
                w: this.w
            });
            this.navButtons.render();
        }

        if (this.config.overlayControls)
            this.config.slideHeight = this.h;
        else
            this.config.slideHeight = this.h - bH;
        this.config.maxHeight = this.config.slideHeight;
        this.clearMarkers();
        this.render();
    };
    /**
     * Render the control UI elements
     * @private
     */
    OU.activity.Overlay.prototype.renderControls = function() {
        // render nav buttons
        this.controlLayer.clear();
        if (this.config.useNavButtons) {
            if (this.slideNumber === 0)
                this.navButtons.leftOn = false;
            else
                this.navButtons.leftOn = true;
            if (this.slideNumber === this.data.slides.length - 1)
                this.navButtons.rightOn = false;
            else
                this.navButtons.rightOn = true;
            this.navButtons.render();
        }
    };
    /**
     * Render the current view
     */
    OU.activity.Overlay.prototype.render = function() {
        OU.LocalStorage.save("OU.overlay.slideNum", this.slideNumber);

        this.displayClickLayer.clear();
        if (this.data.solution) {
            this.placeAnswerButton();
        }
        this.renderSlide();
        this.renderControls();
    };
    /**
     * Render the slide
     * @private
     */
    OU.activity.Overlay.prototype.renderSlide = function() {
        var bH = OU.controlHeight, slide = this.slides[this.slideNumber],
                ctx = this.imageLayer.context,
                img = slide.image,
                x, y, w = img.width,
                h = img.height,
                progress = ' (' + (this.slideNumber + 1) + ' of ' + this.data.slides.length + ')';
        if (this.data.slides.length < 2)
            progress = '';
        ctx.clearRect(0, 0, this.w, this.h);

        // Resize the image to fit canvas
        if (w > this.w) {
            h = img.height * (this.w / w); // maintain aspect ratio
            w = this.w;
        }
        if (h > this.h) {
            w = w * (this.h / h); // maintain aspect ratio
            h = this.h;
        }
        this.scaleFactor = w / img.width;
        // draw the image in the center of the image area
        x = (this.w - w) / 2;
        y = (this.config.slideHeight - h) / 2;
        ctx.drawImage(img, x, y, w, h);
        // render slide title
        if (slide.showLabel === undefined || slide.showLabel) {
            new OU.util.DynText({
                txt: slide.label + progress,
                x: bH,
                y: this.h - bH,
                w: this.w - 2 * bH,
                h: bH,
                context: ctx
            });
        }
        this.placeHotspots();
    };
    /**
     * Displays an answer button that checks the users selections.
     * Only used if (optional) marker selection is being used
     * @private
     */
    OU.activity.Overlay.prototype.placeAnswerButton = function() {
        var self = this,
                bH = 36, bW = 120, iH = 24,
                bX = this.w - (bW + 10),
                bY = this.h - (bH + 10) - OU.controlHeight;

        if (this.answerButton) {
            this.answerButton.render();
        }
        else {
            this.answerButton = new OU.util.Button({
                layer: this.displayClickLayer,
                txt: 'Answer',
                x: bX,
                y: bY,
                w: bW,
                h: bH,
                onClick: function() {
                    self.validate();
                },
                glyph: {
                    type: 'rightArrow',
                    align: 'left',
                    w: iH,
                    h: iH,
                    back: 'rgba(0,0,0,0)',
                    fore: '#1A1A1A'
                }
            });
        }
    };
    /**
     * Places (and Renders if required) the hotspots over the current slide
     * @private
     */
    OU.activity.Overlay.prototype.placeHotspots = function() {
        var i, marker, mk, x, y, w, h, slide = this.slides[this.slideNumber],
                dcctx = this.displayClickLayer.context,
                clickable = this.displayClickLayer.events.clickable,
                img = slide.image;
        w = img.width;
        h = img.height;
        // Resize the image to fit canvas
        if (w > this.w) {
            h = img.height * (this.w / w); // maintain aspect ratio
            w = this.w;
        }
        if (h > this.h) {
            w = w * (this.h / h); // maintain aspect ratio
            h = this.h;
        }
        this.scaleFactor = w / img.width;
        x = (this.w - w) / 2;
        y = (this.config.slideHeight - h) / 2;
        this.displayClickLayer.clear();
        if (this.markers) {
            for (i = this.markers.length; i--; ) {
                marker = this.markers[i].marker;
                this.markers[i].render({
                    x: x + (marker.x * this.scaleFactor),
                    y: y + (marker.y * this.scaleFactor),
                    radius: marker.radius ? marker.radius * this.scaleFactor : undefined,
                    w: marker.w ? marker.w * this.scaleFactor : undefined,
                    h: marker.h ? marker.h * this.scaleFactor : undefined
                });
            }
        }
        else {
            this.markers = [];
            clickable.length = 0;
            for (i = slide.markers.length; i--; ) {
                marker = slide.markers[i];

                mk = new this.Marker({
                    overlay: this,
                    x: x + (marker.x * this.scaleFactor),
                    y: y + (marker.y * this.scaleFactor),
                    radius: marker.radius ? marker.radius * this.scaleFactor : undefined,
                    w: marker.w ? marker.w * this.scaleFactor : undefined,
                    h: marker.h ? marker.h * this.scaleFactor : undefined,
                    shape: marker.shape || "circle",
                    showGuide: marker.showGuide === false ? false : true,
                    showMarkerWhenHit:slide.showMarkerWhenHit || slide.persistant,
                    slide: slide,
                    marker: marker,
                    context: dcctx,
                    activityPopup: marker.activityPopup
                });
                clickable.push(mk);
                this.markers.push(mk);
            }
        }
    };
    /**
     * @class Marker class is the object that acts as a hotspot
     * @param {object} params - options:
     * <ul>
     * <li><strong>{OU.activity.Overlay} overlay:</strong> reference to parent activity</li>
     * <li><strong>{Canvas.context} context:</strong> canvas context to render to</li>
     * <li><strong>{object} slide:</strong> reference to the slide the marker is on</li>
     * <li><strong>{object} marker:</strong> reference to the marker's data object</li>
     * <li><strong>{int} x:</strong> X co-ordinate of the marker</li>
     * <li><strong>{int} y:</strong> Y co-ordinate of the marker</li>
     * <li><strong>{int} w:</strong> Width of the marker</li>
     * <li><strong>{int} h:</strong> Height of the marker</li>
     * <li><strong>{int} r:</strong> Radius (if hotspot if circular)</li>
     * <li><strong>{boolean} showGuide:</strong> True to render the marker graphic</li>
     * <li><strong>{string} shape:</strong> Optional shape of the hotspot [circle | rectangle | poly]</li>
     * </ul>
     * @extends OU.util.Tabbable
     */
    OU.activity.Overlay.prototype.Marker = function(params) {
        this.overlay = params.overlay;
        this.context = params.context;
        this.x = params.x;
        this.y = params.y;
        this.w = params.w;
        this.h = params.h;
        this.radius = params.radius ? params.radius : (this.overlay.config.markerRadius * this.overlay.scaleFactor);
        this.marker = params.marker;
        this.showGuide = params.showGuide;
        this.showMarkerWhenHit = params.showMarkerWhenHit;
        this.slide = params.slide;
        this.shape = params.shape;

        this.isclicked = false;
        /**
         * Renders the marker
         * @param {object} params - optional dimensions, (if not specified, then current values are used):
         * <ul>
         * <li><strong>{int} x:</strong> X co-ordinate of the marker</li>
         * <li><strong>{int} y:</strong> Y co-ordinate of the marker</li>
         * <li><strong>{int} w:</strong> Width of the marker</li>
         * <li><strong>{int} h:</strong> Height of the marker</li>
         * <li><strong>{int} r:</strong> Radius (if hotspot if circular)</li>
         * </ul>
         */
        OU.activity.Overlay.prototype.Marker.prototype.render = function(params) {
            params = params || {};
            this.x = params.x || this.x;
            this.y = params.y || this.y;
            this.w = params.w || this.w;
            this.h = params.h || this.h;
            this.radius = params.radius || this.radius;

            if (this.showGuide || (this.isclicked&&this.showMarkerWhenHit) || this.hasFocus)
                this.renderClickArea();
            if (this.showGuide)
                this.renderLabel();
        };
        /**
         * Renders the clickable ares for the marker.
         * @private
         */
        OU.activity.Overlay.prototype.Marker.prototype.renderClickArea = function() {
            var i, px, py, ctx = this.context;

            ctx.save();
            if (this.hasFocus) {
                ctx.strokeStyle = '#c00';
                ctx.lineWidth = 3;
            }
            else {
                ctx.strokeStyle = '#444';
                ctx.lineWidth = 1;
            }
            ctx.beginPath();
            switch (this.shape) {
                case "rectangle":
                    ctx.rect(this.x, this.y, this.w, this.h);
                    ctx.stroke();
                    break;
                case "poly":
                    var points = this.points;
                    ctx.beginPath();
                    for (i = points.length; i--; ) {
                        px = this.x + (points[i].x * this.scaleFactor);
                        py = this.y + (points[i].y * this.scaleFactor);
                        if (i === 0) {
                            ctx.moveTo(px, py);
                        }
                        else {
                            ctx.lineTo(px, py);
                        }
                    }
                    ctx.fill();
                    //			dcctx.stroke();
                    ctx.closePath();
                    break;

                case "circle":
                default:
                    ctx.beginPath();
                    ctx.arc(this.x, this.y, this.radius * 6, 0, Math.PI * 2, true);
                    ctx.stroke();
                    ctx.closePath();
                    break;
            }
            ctx.restore();
        };
        /**
         * Renders the marker label, if applicable
         * @private
         */
        OU.activity.Overlay.prototype.Marker.prototype.renderLabel = function() {
            var ctx = this.context;

            //TODO determine if overlap and move labels left,right,up, down to accommodate
            if (this.marker.label && this.marker.label !== '') {
                this.labelW = this.overlay.w;
                if (this.x + (this.radius * 2) + this.labelW > this.overlay.w)
                    this.labelW = this.overlay.w - (this.x + (this.radius * 2));
                this.markerLabel = new OU.util.DynText({
                    txt: this.marker.label,
                    x: this.x + (this.radius * 2),
                    y: this.y - (this.radius * 2),
                    w: this.labelW,
                    h: this.radius * 4,
                    context: ctx,
                    align: 'left',
                    background: {
                        radius: this.radius,
                        alpha: 0.6,
                        RGB: '255,255,255',
                        shadow: true
                    },
                    colour: this.isclicked ? '#999' : '#000',
                    autoWidth: true,
                    padding: 10,
                    fontFamily: 'tahoma,arial,sans',
                    fontWeight: 'bold',
                    fontSize: 14
                });
                this.labelW = this.markerLabel.w;
            }
        };
        /**
         * Override the tabbable class focus function to make the marker highlight when tabbed to
         * @private
         */
        OU.activity.Overlay.prototype.Marker.prototype.focus = function() {
            this.hasFocus = true;
            this.overlay.renderSlide();
        };
        /**
         * Override the tabbable class blur function to lose the highlight when tabbed away from
         * @private
         */
        OU.activity.Overlay.prototype.Marker.prototype.blur = function() {
            this.hasFocus = false;
            this.overlay.renderSlide();
        };
        /**
         * Determines if a polygon shaped hotspot has been hit.
         * Don't worry about how this works, but it does!
         * @param {double} x - x co-ordinate of the hit
         * @param {double} y - y co-ordinate of the hit
         * @returns {int} oddNodes - False if hit outside the polygon. True if hit inside
         * @private
         */
        OU.activity.Overlay.prototype.Marker.prototype.countPolysides = function(x, y) {
            var points = this.marker.points, i, j = points.length - 1,
                    sf = this.overlay.scaleFactor,
                    xI, xJ, yI, yJ, oddNodes = false;

            for (i = 0; i < points.length; i++) {
                xI = this.x + points[i].x * sf;
                xJ = this.x + points[j].x * sf;
                yI = this.y + points[i].y * sf;
                yJ = this.y + points[j].y * sf;
                if ((yI < y && yJ >= y || yJ < y && yI >= y) && (xI <= x || xJ <= x)) {
                    if (xI + (y - yI) / (yJ - yI) * (xJ - xI) < x) {
                        oddNodes = !oddNodes;
                    }
                }
                j = i;
            }
            return oddNodes;
        };
        /**
         * Determine if the marker is hit, depending on the shape
         * @param {double} x - x co-ordinate of the hit
         * @param {double} y - y co-ordinate of the hit
         * @param {boolean} pressed - true if mouseDown or touchDown is currently true
         */
        OU.activity.Overlay.prototype.Marker.prototype.isHit = function(x, y, pressed) {
            if (!pressed)
                return;
            var isHit = false, dx = x - this.x, dy = y - this.y;
            switch (this.marker.shape) {

                case "rectangle":
                    if (((x >= this.x) && (x <= this.x + this.w)) && ((y >= this.y) && (y <= this.y + this.h)))
                        isHit = true;
                    break;
                case "poly":
                    isHit = this.countPolysides(x, y);
                    break;
                case "circle":
                default:
                    if (Math.sqrt((dx * dx) + (dy * dy)) < this.radius * 6)
                        isHit = true; // mouse/touch is within circle
                    break;

            }
            if (this.labelW > 0 && x > this.x + (this.radius * 2) && x < this.x + (this.radius * 2) + this.labelW && y > this.y - (this.radius * 2) && y < this.y + (this.radius * 2))
                isHit = true; // mouse/touch is within Label
            if (isHit)
                this.hit();
        };
        /**
         * Act upon marker being hit, will either:
         * <ul>
         * <li>Pop up some info using (OU.util.PopUpInfo), if marker data contains an 'info' field</li>
         * <li>Load a pop-up activity via the controller, if marker data contains 'addActivity' field (For backwards compatability)</li>
         * <li>Make a ControllerAPI call, if marker data contains 'controllerAPI' field (Prefered use for loading activities)</li>
         * </ul>
         */
        OU.activity.Overlay.prototype.Marker.prototype.hit = function() {
            var l, events = this.overlay.displayClickLayer.events, x, y, w, h;
            if (this.slide.persistant) {
                this.isclicked = !this.isclicked;
            }
            else {
                for (l = this.slide.markers.length; l--; ) {
                    this.slide.markers[l].isclicked = false;
                }
                this.isclicked = true;
            }

            if (this.marker.activityPopup !== undefined) {
                if (this.controller !== undefined) {
                    this.controller.addActivity(this.marker.activityPopup);

                }
            } else {
                if (this.marker.info) {
                    events.disableSwipe();
                    if (this.marker.proportionateDims && this.marker.proportionateDims.switchedOn) {
                        x = this.context.canvas.width * this.marker.proportionateDims.x;
                        y = this.context.canvas.height * this.marker.proportionateDims.y;
                        w = this.context.canvas.width * this.marker.proportionateDims.w;
                        h = this.context.canvas.height * this.marker.proportionateDims.h;
                    }
                    if (this.marker.fixedDims && this.marker.fixedDims.switchedOn) {
                        w = this.marker.fixedDims.w;
                        h = this.marker.fixedDims.h;
                        x = (this.w - w) / 2;
                        y = (this.h - h) / 2;
                    }
                    this.overlay.popUp = new OU.util.PopUpInfo({
                        container: this.overlay,
                        x: x,
                        y: y,
                        w: w,
                        h: h,
                        txt: this.marker.info,
                        onClose: function() {
                            events.enableSwipe();
                        }
                    });
                }
            }
            this.render();
            events.flush();

            // Process any controller API requests that are in the data file - will have no effect if they aren't defined
            this.overlay.controller.processAPI(this.marker.controllerAPI);

        };
        OU.base(this, params);
        this.render();
    };
    OU.inherits(OU.activity.Overlay.prototype.Marker, OU.util.Tabbable);
    /**
     * Moves activity to the next slide, unless we're already at the end
     */
    OU.activity.Overlay.prototype.nextSlide = function() {
        if (this.slideNumber >= this.data.slides.length - 1)
            return;
        this.slideNumber++;
        this.clearMarkers();
        this.render();
    };
    /**
     * Removes the current markers, as we are leaving this slide
     * @private
     */
    OU.activity.Overlay.prototype.clearMarkers = function() {
        var i;
        if (this.markers) {
            for (i = this.markers.length; i--; ) {
                this.markers[i].remove(); // call each marker's remove function to remove the tabbable element
            }
        }
        this.markers = null;
    };
    /**
     * Moves activity to the previous slide, unless we're already at the beginning
     */
    OU.activity.Overlay.prototype.prevSlide = function() {
        if (this.slideNumber < 1)
            return;
        this.slideNumber--;
        this.clearMarkers();
        this.render();
    };
    /**
     * Validate selection if optional hotspot selection is being used
     */
    OU.activity.Overlay.prototype.validate = function() {
        var i, j, slide = this.slides[this.slideNumber], markers = slide.markers,
                self = this,
                solution = this.data.solution,
                wanted = 0, notwanted = 0, maxscore = 0,
                wantFlag = false,
                txt = '';

        maxscore = solution.correct.length;

        // ok lets check whats been hit..
        //  var markers = slide.markers ? slide.markers : slide.slide.markers;

        for (i = markers.length; i--; ) {
            if (markers[i].isclicked) {
                wantFlag = false;
                for (j = solution.correct.length; j--; ) {
                    if (solution.correct[j] === markers[i].id) {
                        wantFlag = true;
                    }
                }
                if (wantFlag) {
                    wanted++;
                }
                else {
                    notwanted++;
                }
            }
        }
        if (wanted === maxscore && notwanted === 0) {
            txt = solution.correctTxt;
        }
        else {
            if (wanted === maxscore) {
                txt += 'You have selected all the correct items';

                if (notwanted > 0) {
                    txt += ' but you have also selected incorrect items';
                }
            }
            if (wanted < maxscore) {
                txt += 'You have not selected all the correct items';
                if (notwanted > 0) {
                    txt += ' and you have also selected incorrect items';
                }
            }
        }

        this.overlay.popUp = new OU.util.PopUpInfo({
            container: this,
            txt: txt,
            onClose: function() {
                self.displayClickLayer.events.enableSwipe();
            }
        });
    };
    /**
     * Determines which markers are selected (if optional selection is in use)
     * @returns an array of marker indexes, of the markers that are currently selected
     * @private
     */
    OU.activity.Overlay.prototype.currentActive = function() {
        var slide = this.slides[this.slideNumber].slide, l, c = [];
        for (l = slide.markers.length; l--; ) {
            if (slide.markers[l].isclicked) {
                c[c.length] = l;
            }
        }
        return c;
    };
    /**
     * Text only version of the activity, used when we have no canvas support
     */
    OU.activity.Overlay.prototype.accessibleView = function() {
        var i, j, marker, slide, h = '<div id="overlayAccessible">';
        h += '<h1>' + this.data.title + '</h1>';
        h += '<p>' + this.data.description + '</p>';
        for (i = 0; i < this.data.slides.length; i++) {
            slide = this.data.slides[i];
            h += '<h2>' + slide.label + '</h2><p>Markers in this slide are:</p>';
            for (j = 0; j < slide.markers.length; j++) {
                marker = slide.markers[j];
                h += '<h3>' + marker.label + '</h3><p>' + marker.info + '</p>';
            }
        }
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Overlay, OU.util.Activity);
