/// <reference path="../typings/webrtc/MediaStream.d.ts" />
/// <reference path="../typings/webaudioapi/waa.d.ts" />
/// <reference path="../typings/es6-promise/es6-promise.d.ts" />
/// <reference path="MediaRecorder.d.ts" />
/// <reference path="vorbis_encoder.d.ts" />
var window;
if (window && !window.BlobEvent) {
    window.BlobEvent = function BlobEvent(type, init) {
        this.type = type;
        this.data = init.data;
    };
}
// END BlobEvent shim
var VorbisWorkerScript = (function () {
    function VorbisWorkerScript() {
    }
    VorbisWorkerScript.createWorker = function () {
        return new Worker(VorbisWorkerScript.getCurrentScriptURL());
    };
    // NOTE `self` should be type `WorkerGlobalScope`
    // see https://developer.mozilla.org/en-US/docs/Web/API/WorkerGlobalScope
    VorbisWorkerScript.main = function (self) {
        var Module = makeVorbisEncoderModule({
            onRuntimeInitialized: function () {
                self.postMessage({ type: 'load' });
            }
        });
        var handle;
        function flush() {
            var dataLength = Module._encoder_get_data_len(handle);
            if (dataLength === 0)
                return;
            var dataPointer = Module._encoder_get_data(handle);
            var chunk = Module.HEAPU8.subarray(dataPointer, dataPointer + dataLength);
            var data = new Uint8Array(chunk); // copy
            var buffer = data.buffer;
            Module._encoder_clear_data(handle);
            self.postMessage({ type: 'data', buffer: buffer }, [buffer]);
        }
        self.addEventListener('message', function (ev) {
            var data = ev.data;
            switch (data.type) {
                case 'start':
                    handle = Module._encoder_create_vbr(data.channels, data.sampleRate, data.quality);
                    Module._encoder_write_headers(handle);
                    flush();
                    break;
                case 'data':
                    Module._encoder_prepare_analysis_buffers(handle, data.samples);
                    for (var ch = 0; ch < data.channels; ++ch) {
                        var bufferPtr = Module._encoder_get_analysis_buffer(handle, ch);
                        var array = new Float32Array(data.buffers[ch]);
                        Module.HEAPF32.set(array, bufferPtr >> 2);
                    }
                    Module._encoder_encode(handle);
                    flush();
                    break;
                case 'finish':
                    Module._encoder_finish(handle);
                    flush();
                    Module._encoder_destroy(handle);
                    self.postMessage({ type: 'finish' });
                    break;
            }
        });
    };
    VorbisWorkerScript.getCurrentScriptURL = (function () {
        if (!this.document) {
            return null;
        }
        var script = document.currentScript;
        var scriptSrc = script.getAttribute('src');
        var absoluteRegex = /^(blob\:|http\:|https\:)/;
        var url;
        if (absoluteRegex.test(scriptSrc)) {
            url = scriptSrc;
        }
        else {
            var dirname = location.pathname.split('/').slice(0, -1).join('/');
            url = location.protocol + "//" + location.host;
            if (scriptSrc[0] === '/') {
                url += scriptSrc;
            }
            else {
                url += dirname + '/' + scriptSrc;
            }
        }
        return function () { return url; };
    })();
    return VorbisWorkerScript;
}());
function noop() { }
var VorbisEncoder = (function () {
    // ---
    function VorbisEncoder() {
        this._worker = VorbisWorkerScript.createWorker();
        // ---
        this._ondata = noop;
        this._onfinish = noop;
        // ---
        this._worker.onmessage = this.handleEncoderMessage.bind(this);
    }
    Object.defineProperty(VorbisEncoder.prototype, "ondata", {
        get: function () {
            return this._ondata;
        },
        set: function (value) {
            this._ondata = value || noop;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(VorbisEncoder.prototype, "onfinish", {
        get: function () {
            return this._onfinish;
        },
        set: function (value) {
            this._onfinish = value || noop;
        },
        enumerable: true,
        configurable: true
    });
    VorbisEncoder.prototype.init = function (channels, sampleRate, quality) {
        this._worker.postMessage({
            type: 'start',
            sampleRate: sampleRate,
            channels: channels,
            quality: quality
        });
    };
    VorbisEncoder.prototype.encode = function (buffers, samples, channels) {
        this._worker.postMessage({
            type: 'data',
            samples: samples,
            channels: channels,
            buffers: buffers
        }, buffers);
    };
    VorbisEncoder.prototype.finish = function () {
        this._worker.postMessage({ type: 'finish' });
    };
    VorbisEncoder.prototype.handleEncoderMessage = function (ev) {
        var data = ev.data;
        switch (data.type) {
            case 'load':
                // TODO
                break;
            case 'data':
                this._ondata(data.buffer);
                break;
            case 'finish':
                this._onfinish(new Event('finish'));
                break;
        }
    };
    return VorbisEncoder;
}());
var RecordingState;
(function (RecordingState) {
    RecordingState[RecordingState["inactive"] = 0] = "inactive";
    RecordingState[RecordingState["recording"] = 1] = "recording";
    RecordingState[RecordingState["paused"] = 2] = "paused";
})(RecordingState || (RecordingState = {}));
function makeBlobEvent(type, blob) {
    return new BlobEvent(type, { data: blob, blob: blob });
}
var VorbisMediaRecorder = (function () {
    // ---
    function VorbisMediaRecorder(stream, options) {
        this._state = RecordingState.inactive;
        this._stream = stream;
        this._encoder = new VorbisEncoder();
        this._chunks = [];
        this._ctx = new AudioContext();
        this._sourceNode = this._ctx.createMediaStreamSource(stream);
        this._procNode = this._ctx.createScriptProcessor(4096);
        this._onstart = noop;
        this._ondataavailable = noop;
        this._onstop = noop;
        // ---
        this._encoder.ondata = this.handleEncoderData.bind(this);
        this._encoder.onfinish = this.handleEncoderFinish.bind(this);
        this._procNode.onaudioprocess = this.handleAudioProcess.bind(this);
    }
    Object.defineProperty(VorbisMediaRecorder.prototype, "stream", {
        get: function () {
            return this._stream;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(VorbisMediaRecorder.prototype, "mimeType", {
        get: function () {
            return 'audio/ogg';
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(VorbisMediaRecorder.prototype, "state", {
        get: function () {
            return RecordingState[this._state];
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(VorbisMediaRecorder.prototype, "onstart", {
        get: function () {
            return this._onstart;
        },
        set: function (value) {
            this._onstart = value || noop;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(VorbisMediaRecorder.prototype, "ondataavailable", {
        get: function () {
            return this._ondataavailable;
        },
        set: function (value) {
            this._ondataavailable = value || noop;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(VorbisMediaRecorder.prototype, "onstop", {
        get: function () {
            return this._onstop;
        },
        set: function (value) {
            this._onstop = value || noop;
        },
        enumerable: true,
        configurable: true
    });
    VorbisMediaRecorder.prototype.start = function (timeslice) {
        var _this = this;
        if (timeslice !== undefined) {
            throw new Error('not implemented');
        }
        if (this._state !== RecordingState.inactive) {
            throw new Error('invalid state');
        }
        setTimeout(function () {
            _this._state = RecordingState.recording;
            _this._chunks = [];
            _this._sourceNode.connect(_this._procNode);
            _this._procNode.connect(_this._ctx.destination);
            var channels = _this._sourceNode.channelCount;
            var sampleRate = _this._ctx.sampleRate;
            _this._encoder.init(channels, sampleRate, 0.4);
            _this.onStart();
        });
    };
    VorbisMediaRecorder.prototype.stop = function () {
        var _this = this;
        if (this._state === RecordingState.inactive) {
            throw new Error('invalid state');
        }
        setTimeout(function () {
            _this._state = RecordingState.inactive;
            _this._sourceNode.disconnect(_this._procNode);
            _this._procNode.disconnect(_this._ctx.destination);
            _this._encoder.finish();
        });
    };
    VorbisMediaRecorder.prototype.onStart = function () {
        this._onstart(new Event('start'));
    };
    VorbisMediaRecorder.prototype.onDataAvailable = function (data) {
        this._ondataavailable(makeBlobEvent('dataavailable', data));
    };
    VorbisMediaRecorder.prototype.onStop = function () {
        this._onstop(new Event('stop'));
    };
    VorbisMediaRecorder.prototype.handleEncoderData = function (data) {
        this._chunks.push(data);
    };
    VorbisMediaRecorder.prototype.handleEncoderFinish = function () {
        var blob = new Blob(this._chunks, { type: this.mimeType });
        this.onDataAvailable(blob);
        this.onStop();
    };
    VorbisMediaRecorder.prototype.handleAudioProcess = function (ev) {
        var buffers = [];
        var audioBuffer = ev.inputBuffer;
        var samples = audioBuffer.length;
        var channels = audioBuffer.numberOfChannels;
        for (var ch = 0; ch < channels; ++ch) {
            // make a copy
            var array = audioBuffer.getChannelData(ch).slice();
            buffers.push(array.buffer);
        }
        this._encoder.encode(buffers, samples, channels);
    };
    return VorbisMediaRecorder;
}());
function makeVorbisEncoderModule(Module, module) {
// The Module object: Our interface to the outside world. We import
// and export values on it, and do the work to get that through
// closure compiler if necessary. There are various ways Module can be used:
// 1. Not defined. We create it here
// 2. A function parameter, function(Module) { ..generated code.. }
// 3. pre-run appended it, var Module = {}; ..generated code..
// 4. External script tag defines var Module.
// We need to do an eval in order to handle the closure compiler
// case, where this code here is minified but Module was defined
// elsewhere (e.g. case 4 above). We also need to check if Module
// already exists (e.g. case 3 above).
// Note that if you want to run closure, and also to use Module
// after the generated code, you will need to define   var Module = {};
// before the code. Then that object will be used in the code, and you
// can continue to use Module afterwards as well.
var Module;
if (!Module) Module = (typeof Module !== 'undefined' ? Module : null) || {};

// Sometimes an existing Module object exists with properties
// meant to overwrite the default module functionality. Here
// we collect those properties and reapply _after_ we configure
// the current environment's defaults to avoid having to be so
// defensive during initialization.
var moduleOverrides = {};
for (var key in Module) {
  if (Module.hasOwnProperty(key)) {
    moduleOverrides[key] = Module[key];
  }
}

// The environment setup code below is customized to use Module.
// *** Environment setup code ***
var ENVIRONMENT_IS_WEB = typeof window === 'object';
// Three configurations we can be running in:
// 1) We could be the application main() thread running in the main JS UI thread. (ENVIRONMENT_IS_WORKER == false and ENVIRONMENT_IS_PTHREAD == false)
// 2) We could be the application main() thread proxied to worker. (with Emscripten -s PROXY_TO_WORKER=1) (ENVIRONMENT_IS_WORKER == true, ENVIRONMENT_IS_PTHREAD == false)
// 3) We could be an application pthread running in a worker. (ENVIRONMENT_IS_WORKER == true and ENVIRONMENT_IS_PTHREAD == true)
var ENVIRONMENT_IS_WORKER = typeof importScripts === 'function';
var ENVIRONMENT_IS_NODE = typeof process === 'object' && typeof require === 'function' && !ENVIRONMENT_IS_WEB && !ENVIRONMENT_IS_WORKER;
var ENVIRONMENT_IS_SHELL = !ENVIRONMENT_IS_WEB && !ENVIRONMENT_IS_NODE && !ENVIRONMENT_IS_WORKER;

if (ENVIRONMENT_IS_NODE) {
  // Expose functionality in the same simple way that the shells work
  // Note that we pollute the global namespace here, otherwise we break in node
  if (!Module['print']) Module['print'] = function print(x) {
    process['stdout'].write(x + '\n');
  };
  if (!Module['printErr']) Module['printErr'] = function printErr(x) {
    process['stderr'].write(x + '\n');
  };

  var nodeFS = require('fs');
  var nodePath = require('path');

  Module['read'] = function read(filename, binary) {
    filename = nodePath['normalize'](filename);
    var ret = nodeFS['readFileSync'](filename);
    // The path is absolute if the normalized version is the same as the resolved.
    if (!ret && filename != nodePath['resolve'](filename)) {
      filename = path.join(__dirname, '..', 'src', filename);
      ret = nodeFS['readFileSync'](filename);
    }
    if (ret && !binary) ret = ret.toString();
    return ret;
  };

  Module['readBinary'] = function readBinary(filename) {
    var ret = Module['read'](filename, true);
    if (!ret.buffer) {
      ret = new Uint8Array(ret);
    }
    assert(ret.buffer);
    return ret;
  };

  Module['load'] = function load(f) {
    globalEval(read(f));
  };

  if (!Module['thisProgram']) {
    if (process['argv'].length > 1) {
      Module['thisProgram'] = process['argv'][1].replace(/\\/g, '/');
    } else {
      Module['thisProgram'] = 'unknown-program';
    }
  }

  Module['arguments'] = process['argv'].slice(2);

  if (typeof module !== 'undefined') {
    module['exports'] = Module;
  }

  process['on']('uncaughtException', function(ex) {
    // suppress ExitStatus exceptions from showing an error
    if (!(ex instanceof ExitStatus)) {
      throw ex;
    }
  });

  Module['inspect'] = function () { return '[Emscripten Module object]'; };
}
else if (ENVIRONMENT_IS_SHELL) {
  if (!Module['print']) Module['print'] = print;
  if (typeof printErr != 'undefined') Module['printErr'] = printErr; // not present in v8 or older sm

  if (typeof read != 'undefined') {
    Module['read'] = read;
  } else {
    Module['read'] = function read() { throw 'no read() available (jsc?)' };
  }

  Module['readBinary'] = function readBinary(f) {
    if (typeof readbuffer === 'function') {
      return new Uint8Array(readbuffer(f));
    }
    var data = read(f, 'binary');
    assert(typeof data === 'object');
    return data;
  };

  if (typeof scriptArgs != 'undefined') {
    Module['arguments'] = scriptArgs;
  } else if (typeof arguments != 'undefined') {
    Module['arguments'] = arguments;
  }

}
else if (ENVIRONMENT_IS_WEB || ENVIRONMENT_IS_WORKER) {
  Module['read'] = function read(url) {
    var xhr = new XMLHttpRequest();
    xhr.open('GET', url, false);
    xhr.send(null);
    return xhr.responseText;
  };

  if (typeof arguments != 'undefined') {
    Module['arguments'] = arguments;
  }

  if (typeof console !== 'undefined') {
    if (!Module['print']) Module['print'] = function print(x) {
      console.log(x);
    };
    if (!Module['printErr']) Module['printErr'] = function printErr(x) {
      console.log(x);
    };
  } else {
    // Probably a worker, and without console.log. We can do very little here...
    var TRY_USE_DUMP = false;
    if (!Module['print']) Module['print'] = (TRY_USE_DUMP && (typeof(dump) !== "undefined") ? (function(x) {
      dump(x);
    }) : (function(x) {
      // self.postMessage(x); // enable this if you want stdout to be sent as messages
    }));
  }

  if (ENVIRONMENT_IS_WORKER) {
    Module['load'] = importScripts;
  }

  if (typeof Module['setWindowTitle'] === 'undefined') {
    Module['setWindowTitle'] = function(title) { document.title = title };
  }
}
else {
  // Unreachable because SHELL is dependant on the others
  throw 'Unknown runtime environment. Where are we?';
}

function globalEval(x) {
  eval.call(null, x);
}
if (!Module['load'] && Module['read']) {
  Module['load'] = function load(f) {
    globalEval(Module['read'](f));
  };
}
if (!Module['print']) {
  Module['print'] = function(){};
}
if (!Module['printErr']) {
  Module['printErr'] = Module['print'];
}
if (!Module['arguments']) {
  Module['arguments'] = [];
}
if (!Module['thisProgram']) {
  Module['thisProgram'] = './this.program';
}

// *** Environment setup code ***

// Closure helpers
Module.print = Module['print'];
Module.printErr = Module['printErr'];

// Callbacks
Module['preRun'] = [];
Module['postRun'] = [];

// Merge back in the overrides
for (var key in moduleOverrides) {
  if (moduleOverrides.hasOwnProperty(key)) {
    Module[key] = moduleOverrides[key];
  }
}



// {{PREAMBLE_ADDITIONS}}

// === Preamble library stuff ===

// Documentation for the public APIs defined in this file must be updated in: 
//    site/source/docs/api_reference/preamble.js.rst
// A prebuilt local version of the documentation is available at: 
//    site/build/text/docs/api_reference/preamble.js.txt
// You can also build docs locally as HTML or other formats in site/
// An online HTML version (which may be of a different version of Emscripten)
//    is up at http://kripken.github.io/emscripten-site/docs/api_reference/preamble.js.html

//========================================
// Runtime code shared with compiler
//========================================

var Runtime = {
  setTempRet0: function (value) {
    tempRet0 = value;
  },
  getTempRet0: function () {
    return tempRet0;
  },
  stackSave: function () {
    return STACKTOP;
  },
  stackRestore: function (stackTop) {
    STACKTOP = stackTop;
  },
  getNativeTypeSize: function (type) {
    switch (type) {
      case 'i1': case 'i8': return 1;
      case 'i16': return 2;
      case 'i32': return 4;
      case 'i64': return 8;
      case 'float': return 4;
      case 'double': return 8;
      default: {
        if (type[type.length-1] === '*') {
          return Runtime.QUANTUM_SIZE; // A pointer
        } else if (type[0] === 'i') {
          var bits = parseInt(type.substr(1));
          assert(bits % 8 === 0);
          return bits/8;
        } else {
          return 0;
        }
      }
    }
  },
  getNativeFieldSize: function (type) {
    return Math.max(Runtime.getNativeTypeSize(type), Runtime.QUANTUM_SIZE);
  },
  STACK_ALIGN: 16,
  prepVararg: function (ptr, type) {
    if (type === 'double' || type === 'i64') {
      // move so the load is aligned
      if (ptr & 7) {
        assert((ptr & 7) === 4);
        ptr += 4;
      }
    } else {
      assert((ptr & 3) === 0);
    }
    return ptr;
  },
  getAlignSize: function (type, size, vararg) {
    // we align i64s and doubles on 64-bit boundaries, unlike x86
    if (!vararg && (type == 'i64' || type == 'double')) return 8;
    if (!type) return Math.min(size, 8); // align structures internally to 64 bits
    return Math.min(size || (type ? Runtime.getNativeFieldSize(type) : 0), Runtime.QUANTUM_SIZE);
  },
  dynCall: function (sig, ptr, args) {
    if (args && args.length) {
      assert(args.length == sig.length-1);
      if (!args.splice) args = Array.prototype.slice.call(args);
      args.splice(0, 0, ptr);
      assert(('dynCall_' + sig) in Module, 'bad function pointer type - no table for sig \'' + sig + '\'');
      return Module['dynCall_' + sig].apply(null, args);
    } else {
      assert(sig.length == 1);
      assert(('dynCall_' + sig) in Module, 'bad function pointer type - no table for sig \'' + sig + '\'');
      return Module['dynCall_' + sig].call(null, ptr);
    }
  },
  functionPointers: [],
  addFunction: function (func) {
    for (var i = 0; i < Runtime.functionPointers.length; i++) {
      if (!Runtime.functionPointers[i]) {
        Runtime.functionPointers[i] = func;
        return 2*(1 + i);
      }
    }
    throw 'Finished up all reserved function pointers. Use a higher value for RESERVED_FUNCTION_POINTERS.';
  },
  removeFunction: function (index) {
    Runtime.functionPointers[(index-2)/2] = null;
  },
  warnOnce: function (text) {
    if (!Runtime.warnOnce.shown) Runtime.warnOnce.shown = {};
    if (!Runtime.warnOnce.shown[text]) {
      Runtime.warnOnce.shown[text] = 1;
      Module.printErr(text);
    }
  },
  funcWrappers: {},
  getFuncWrapper: function (func, sig) {
    assert(sig);
    if (!Runtime.funcWrappers[sig]) {
      Runtime.funcWrappers[sig] = {};
    }
    var sigCache = Runtime.funcWrappers[sig];
    if (!sigCache[func]) {
      sigCache[func] = function dynCall_wrapper() {
        return Runtime.dynCall(sig, func, arguments);
      };
    }
    return sigCache[func];
  },
  getCompilerSetting: function (name) {
    throw 'You must build with -s RETAIN_COMPILER_SETTINGS=1 for Runtime.getCompilerSetting or emscripten_get_compiler_setting to work';
  },
  stackAlloc: function (size) { var ret = STACKTOP;STACKTOP = (STACKTOP + size)|0;STACKTOP = (((STACKTOP)+15)&-16);(assert((((STACKTOP|0) < (STACK_MAX|0))|0))|0); return ret; },
  staticAlloc: function (size) { var ret = STATICTOP;STATICTOP = (STATICTOP + (assert(!staticSealed),size))|0;STATICTOP = (((STATICTOP)+15)&-16); return ret; },
  dynamicAlloc: function (size) { var ret = DYNAMICTOP;DYNAMICTOP = (DYNAMICTOP + (assert(DYNAMICTOP > 0),size))|0;DYNAMICTOP = (((DYNAMICTOP)+15)&-16); if (DYNAMICTOP >= TOTAL_MEMORY) { var success = enlargeMemory(); if (!success) { DYNAMICTOP = ret;  return 0; } }; return ret; },
  alignMemory: function (size,quantum) { var ret = size = Math.ceil((size)/(quantum ? quantum : 16))*(quantum ? quantum : 16); return ret; },
  makeBigInt: function (low,high,unsigned) { var ret = (unsigned ? ((+((low>>>0)))+((+((high>>>0)))*4294967296.0)) : ((+((low>>>0)))+((+((high|0)))*4294967296.0))); return ret; },
  GLOBAL_BASE: 8,
  QUANTUM_SIZE: 4,
  __dummy__: 0
}



Module["Runtime"] = Runtime;



//========================================
// Runtime essentials
//========================================

var __THREW__ = 0; // Used in checking for thrown exceptions.

var ABORT = false; // whether we are quitting the application. no code should run after this. set in exit() and abort()
var EXITSTATUS = 0;

var undef = 0;
// tempInt is used for 32-bit signed values or smaller. tempBigInt is used
// for 32-bit unsigned values or more than 32 bits. TODO: audit all uses of tempInt
var tempValue, tempInt, tempBigInt, tempInt2, tempBigInt2, tempPair, tempBigIntI, tempBigIntR, tempBigIntS, tempBigIntP, tempBigIntD, tempDouble, tempFloat;
var tempI64, tempI64b;
var tempRet0, tempRet1, tempRet2, tempRet3, tempRet4, tempRet5, tempRet6, tempRet7, tempRet8, tempRet9;

function assert(condition, text) {
  if (!condition) {
    abort('Assertion failed: ' + text);
  }
}

var globalScope = this;

// Returns the C function with a specified identifier (for C++, you need to do manual name mangling)
function getCFunc(ident) {
  var func = Module['_' + ident]; // closure exported function
  if (!func) {
    try { func = eval('_' + ident); } catch(e) {}
  }
  assert(func, 'Cannot call unknown function ' + ident + ' (perhaps LLVM optimizations or closure removed it?)');
  return func;
}

var cwrap, ccall;
(function(){
  var JSfuncs = {
    // Helpers for cwrap -- it can't refer to Runtime directly because it might
    // be renamed by closure, instead it calls JSfuncs['stackSave'].body to find
    // out what the minified function name is.
    'stackSave': function() {
      Runtime.stackSave()
    },
    'stackRestore': function() {
      Runtime.stackRestore()
    },
    // type conversion from js to c
    'arrayToC' : function(arr) {
      var ret = Runtime.stackAlloc(arr.length);
      writeArrayToMemory(arr, ret);
      return ret;
    },
    'stringToC' : function(str) {
      var ret = 0;
      if (str !== null && str !== undefined && str !== 0) { // null string
        // at most 4 bytes per UTF-8 code point, +1 for the trailing '\0'
        ret = Runtime.stackAlloc((str.length << 2) + 1);
        writeStringToMemory(str, ret);
      }
      return ret;
    }
  };
  // For fast lookup of conversion functions
  var toC = {'string' : JSfuncs['stringToC'], 'array' : JSfuncs['arrayToC']};

  // C calling interface. 
  ccall = function ccallFunc(ident, returnType, argTypes, args, opts) {
    var func = getCFunc(ident);
    var cArgs = [];
    var stack = 0;
    assert(returnType !== 'array', 'Return type should not be "array".');
    if (args) {
      for (var i = 0; i < args.length; i++) {
        var converter = toC[argTypes[i]];
        if (converter) {
          if (stack === 0) stack = Runtime.stackSave();
          cArgs[i] = converter(args[i]);
        } else {
          cArgs[i] = args[i];
        }
      }
    }
    var ret = func.apply(null, cArgs);
    if ((!opts || !opts.async) && typeof EmterpreterAsync === 'object') {
      assert(!EmterpreterAsync.state, 'cannot start async op with normal JS calling ccall');
    }
    if (opts && opts.async) assert(!returnType, 'async ccalls cannot return values');
    if (returnType === 'string') ret = Pointer_stringify(ret);
    if (stack !== 0) {
      if (opts && opts.async) {
        EmterpreterAsync.asyncFinalizers.push(function() {
          Runtime.stackRestore(stack);
        });
        return;
      }
      Runtime.stackRestore(stack);
    }
    return ret;
  }

  var sourceRegex = /^function\s*\(([^)]*)\)\s*{\s*([^*]*?)[\s;]*(?:return\s*(.*?)[;\s]*)?}$/;
  function parseJSFunc(jsfunc) {
    // Match the body and the return value of a javascript function source
    var parsed = jsfunc.toString().match(sourceRegex).slice(1);
    return {arguments : parsed[0], body : parsed[1], returnValue: parsed[2]}
  }
  var JSsource = {};
  for (var fun in JSfuncs) {
    if (JSfuncs.hasOwnProperty(fun)) {
      // Elements of toCsource are arrays of three items:
      // the code, and the return value
      JSsource[fun] = parseJSFunc(JSfuncs[fun]);
    }
  }

  
  cwrap = function cwrap(ident, returnType, argTypes) {
    argTypes = argTypes || [];
    var cfunc = getCFunc(ident);
    // When the function takes numbers and returns a number, we can just return
    // the original function
    var numericArgs = argTypes.every(function(type){ return type === 'number'});
    var numericRet = (returnType !== 'string');
    if ( numericRet && numericArgs) {
      return cfunc;
    }
    // Creation of the arguments list (["$1","$2",...,"$nargs"])
    var argNames = argTypes.map(function(x,i){return '$'+i});
    var funcstr = "(function(" + argNames.join(',') + ") {";
    var nargs = argTypes.length;
    if (!numericArgs) {
      // Generate the code needed to convert the arguments from javascript
      // values to pointers
      funcstr += 'var stack = ' + JSsource['stackSave'].body + ';';
      for (var i = 0; i < nargs; i++) {
        var arg = argNames[i], type = argTypes[i];
        if (type === 'number') continue;
        var convertCode = JSsource[type + 'ToC']; // [code, return]
        funcstr += 'var ' + convertCode.arguments + ' = ' + arg + ';';
        funcstr += convertCode.body + ';';
        funcstr += arg + '=(' + convertCode.returnValue + ');';
      }
    }

    // When the code is compressed, the name of cfunc is not literally 'cfunc' anymore
    var cfuncname = parseJSFunc(function(){return cfunc}).returnValue;
    // Call the function
    funcstr += 'var ret = ' + cfuncname + '(' + argNames.join(',') + ');';
    if (!numericRet) { // Return type can only by 'string' or 'number'
      // Convert the result to a string
      var strgfy = parseJSFunc(function(){return Pointer_stringify}).returnValue;
      funcstr += 'ret = ' + strgfy + '(ret);';
    }
    funcstr += "if (typeof EmterpreterAsync === 'object') { assert(!EmterpreterAsync.state, 'cannot start async op with normal JS calling cwrap') }";
    if (!numericArgs) {
      // If we had a stack, restore it
      funcstr += JSsource['stackRestore'].body.replace('()', '(stack)') + ';';
    }
    funcstr += 'return ret})';
    return eval(funcstr);
  };
})();
Module["ccall"] = ccall;
Module["cwrap"] = cwrap;

function setValue(ptr, value, type, noSafe) {
  type = type || 'i8';
  if (type.charAt(type.length-1) === '*') type = 'i32'; // pointers are 32-bit
    switch(type) {
      case 'i1': HEAP8[((ptr)>>0)]=value; break;
      case 'i8': HEAP8[((ptr)>>0)]=value; break;
      case 'i16': HEAP16[((ptr)>>1)]=value; break;
      case 'i32': HEAP32[((ptr)>>2)]=value; break;
      case 'i64': (tempI64 = [value>>>0,(tempDouble=value,(+(Math_abs(tempDouble))) >= 1.0 ? (tempDouble > 0.0 ? ((Math_min((+(Math_floor((tempDouble)/4294967296.0))), 4294967295.0))|0)>>>0 : (~~((+(Math_ceil((tempDouble - +(((~~(tempDouble)))>>>0))/4294967296.0)))))>>>0) : 0)],HEAP32[((ptr)>>2)]=tempI64[0],HEAP32[(((ptr)+(4))>>2)]=tempI64[1]); break;
      case 'float': HEAPF32[((ptr)>>2)]=value; break;
      case 'double': HEAPF64[((ptr)>>3)]=value; break;
      default: abort('invalid type for setValue: ' + type);
    }
}
Module["setValue"] = setValue;


function getValue(ptr, type, noSafe) {
  type = type || 'i8';
  if (type.charAt(type.length-1) === '*') type = 'i32'; // pointers are 32-bit
    switch(type) {
      case 'i1': return HEAP8[((ptr)>>0)];
      case 'i8': return HEAP8[((ptr)>>0)];
      case 'i16': return HEAP16[((ptr)>>1)];
      case 'i32': return HEAP32[((ptr)>>2)];
      case 'i64': return HEAP32[((ptr)>>2)];
      case 'float': return HEAPF32[((ptr)>>2)];
      case 'double': return HEAPF64[((ptr)>>3)];
      default: abort('invalid type for setValue: ' + type);
    }
  return null;
}
Module["getValue"] = getValue;

var ALLOC_NORMAL = 0; // Tries to use _malloc()
var ALLOC_STACK = 1; // Lives for the duration of the current function call
var ALLOC_STATIC = 2; // Cannot be freed
var ALLOC_DYNAMIC = 3; // Cannot be freed except through sbrk
var ALLOC_NONE = 4; // Do not allocate
Module["ALLOC_NORMAL"] = ALLOC_NORMAL;
Module["ALLOC_STACK"] = ALLOC_STACK;
Module["ALLOC_STATIC"] = ALLOC_STATIC;
Module["ALLOC_DYNAMIC"] = ALLOC_DYNAMIC;
Module["ALLOC_NONE"] = ALLOC_NONE;

// allocate(): This is for internal use. You can use it yourself as well, but the interface
//             is a little tricky (see docs right below). The reason is that it is optimized
//             for multiple syntaxes to save space in generated code. So you should
//             normally not use allocate(), and instead allocate memory using _malloc(),
//             initialize it with setValue(), and so forth.
// @slab: An array of data, or a number. If a number, then the size of the block to allocate,
//        in *bytes* (note that this is sometimes confusing: the next parameter does not
//        affect this!)
// @types: Either an array of types, one for each byte (or 0 if no type at that position),
//         or a single type which is used for the entire block. This only matters if there
//         is initial data - if @slab is a number, then this does not matter at all and is
//         ignored.
// @allocator: How to allocate memory, see ALLOC_*
function allocate(slab, types, allocator, ptr) {
  var zeroinit, size;
  if (typeof slab === 'number') {
    zeroinit = true;
    size = slab;
  } else {
    zeroinit = false;
    size = slab.length;
  }

  var singleType = typeof types === 'string' ? types : null;

  var ret;
  if (allocator == ALLOC_NONE) {
    ret = ptr;
  } else {
    ret = [typeof _malloc === 'function' ? _malloc : Runtime.staticAlloc, Runtime.stackAlloc, Runtime.staticAlloc, Runtime.dynamicAlloc][allocator === undefined ? ALLOC_STATIC : allocator](Math.max(size, singleType ? 1 : types.length));
  }

  if (zeroinit) {
    var ptr = ret, stop;
    assert((ret & 3) == 0);
    stop = ret + (size & ~3);
    for (; ptr < stop; ptr += 4) {
      HEAP32[((ptr)>>2)]=0;
    }
    stop = ret + size;
    while (ptr < stop) {
      HEAP8[((ptr++)>>0)]=0;
    }
    return ret;
  }

  if (singleType === 'i8') {
    if (slab.subarray || slab.slice) {
      HEAPU8.set(slab, ret);
    } else {
      HEAPU8.set(new Uint8Array(slab), ret);
    }
    return ret;
  }

  var i = 0, type, typeSize, previousType;
  while (i < size) {
    var curr = slab[i];

    if (typeof curr === 'function') {
      curr = Runtime.getFunctionIndex(curr);
    }

    type = singleType || types[i];
    if (type === 0) {
      i++;
      continue;
    }
    assert(type, 'Must know what type to store in allocate!');

    if (type == 'i64') type = 'i32'; // special case: we have one i32 here, and one i32 later

    setValue(ret+i, curr, type);

    // no need to look up size unless type changes, so cache it
    if (previousType !== type) {
      typeSize = Runtime.getNativeTypeSize(type);
      previousType = type;
    }
    i += typeSize;
  }

  return ret;
}
Module["allocate"] = allocate;

// Allocate memory during any stage of startup - static memory early on, dynamic memory later, malloc when ready
function getMemory(size) {
  if (!staticSealed) return Runtime.staticAlloc(size);
  if ((typeof _sbrk !== 'undefined' && !_sbrk.called) || !runtimeInitialized) return Runtime.dynamicAlloc(size);
  return _malloc(size);
}
Module["getMemory"] = getMemory;

function Pointer_stringify(ptr, /* optional */ length) {
  if (length === 0 || !ptr) return '';
  // TODO: use TextDecoder
  // Find the length, and check for UTF while doing so
  var hasUtf = 0;
  var t;
  var i = 0;
  while (1) {
    assert(ptr + i < TOTAL_MEMORY);
    t = HEAPU8[(((ptr)+(i))>>0)];
    hasUtf |= t;
    if (t == 0 && !length) break;
    i++;
    if (length && i == length) break;
  }
  if (!length) length = i;

  var ret = '';

  if (hasUtf < 128) {
    var MAX_CHUNK = 1024; // split up into chunks, because .apply on a huge string can overflow the stack
    var curr;
    while (length > 0) {
      curr = String.fromCharCode.apply(String, HEAPU8.subarray(ptr, ptr + Math.min(length, MAX_CHUNK)));
      ret = ret ? ret + curr : curr;
      ptr += MAX_CHUNK;
      length -= MAX_CHUNK;
    }
    return ret;
  }
  return Module['UTF8ToString'](ptr);
}
Module["Pointer_stringify"] = Pointer_stringify;

// Given a pointer 'ptr' to a null-terminated ASCII-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

function AsciiToString(ptr) {
  var str = '';
  while (1) {
    var ch = HEAP8[((ptr++)>>0)];
    if (!ch) return str;
    str += String.fromCharCode(ch);
  }
}
Module["AsciiToString"] = AsciiToString;

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in ASCII form. The copy will require at most str.length+1 bytes of space in the HEAP.

function stringToAscii(str, outPtr) {
  return writeAsciiToMemory(str, outPtr, false);
}
Module["stringToAscii"] = stringToAscii;

// Given a pointer 'ptr' to a null-terminated UTF8-encoded string in the given array that contains uint8 values, returns
// a copy of that string as a Javascript String object.

function UTF8ArrayToString(u8Array, idx) {
  var u0, u1, u2, u3, u4, u5;

  var str = '';
  while (1) {
    // For UTF8 byte structure, see http://en.wikipedia.org/wiki/UTF-8#Description and https://www.ietf.org/rfc/rfc2279.txt and https://tools.ietf.org/html/rfc3629
    u0 = u8Array[idx++];
    if (!u0) return str;
    if (!(u0 & 0x80)) { str += String.fromCharCode(u0); continue; }
    u1 = u8Array[idx++] & 63;
    if ((u0 & 0xE0) == 0xC0) { str += String.fromCharCode(((u0 & 31) << 6) | u1); continue; }
    u2 = u8Array[idx++] & 63;
    if ((u0 & 0xF0) == 0xE0) {
      u0 = ((u0 & 15) << 12) | (u1 << 6) | u2;
    } else {
      u3 = u8Array[idx++] & 63;
      if ((u0 & 0xF8) == 0xF0) {
        u0 = ((u0 & 7) << 18) | (u1 << 12) | (u2 << 6) | u3;
      } else {
        u4 = u8Array[idx++] & 63;
        if ((u0 & 0xFC) == 0xF8) {
          u0 = ((u0 & 3) << 24) | (u1 << 18) | (u2 << 12) | (u3 << 6) | u4;
        } else {
          u5 = u8Array[idx++] & 63;
          u0 = ((u0 & 1) << 30) | (u1 << 24) | (u2 << 18) | (u3 << 12) | (u4 << 6) | u5;
        }
      }
    }
    if (u0 < 0x10000) {
      str += String.fromCharCode(u0);
    } else {
      var ch = u0 - 0x10000;
      str += String.fromCharCode(0xD800 | (ch >> 10), 0xDC00 | (ch & 0x3FF));
    }
  }
}
Module["UTF8ArrayToString"] = UTF8ArrayToString;

// Given a pointer 'ptr' to a null-terminated UTF8-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

function UTF8ToString(ptr) {
  return UTF8ArrayToString(HEAPU8,ptr);
}
Module["UTF8ToString"] = UTF8ToString;

// Copies the given Javascript String object 'str' to the given byte array at address 'outIdx',
// encoded in UTF8 form and null-terminated. The copy will require at most str.length*4+1 bytes of space in the HEAP.
// Use the function lengthBytesUTF8() to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outU8Array: the array to copy to. Each index in this array is assumed to be one 8-byte element.
//   outIdx: The starting offset in the array to begin the copying.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array. This count should include the null 
//                    terminator, i.e. if maxBytesToWrite=1, only the null terminator will be written and nothing else.
//                    maxBytesToWrite=0 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF8Array(str, outU8Array, outIdx, maxBytesToWrite) {
  if (!(maxBytesToWrite > 0)) // Parameter maxBytesToWrite is not optional. Negative values, 0, null, undefined and false each don't write out any bytes.
    return 0;

  var startIdx = outIdx;
  var endIdx = outIdx + maxBytesToWrite - 1; // -1 for string null terminator.
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! So decode UTF16->UTF32->UTF8.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    // For UTF8 byte structure, see http://en.wikipedia.org/wiki/UTF-8#Description and https://www.ietf.org/rfc/rfc2279.txt and https://tools.ietf.org/html/rfc3629
    var u = str.charCodeAt(i); // possibly a lead surrogate
    if (u >= 0xD800 && u <= 0xDFFF) u = 0x10000 + ((u & 0x3FF) << 10) | (str.charCodeAt(++i) & 0x3FF);
    if (u <= 0x7F) {
      if (outIdx >= endIdx) break;
      outU8Array[outIdx++] = u;
    } else if (u <= 0x7FF) {
      if (outIdx + 1 >= endIdx) break;
      outU8Array[outIdx++] = 0xC0 | (u >> 6);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else if (u <= 0xFFFF) {
      if (outIdx + 2 >= endIdx) break;
      outU8Array[outIdx++] = 0xE0 | (u >> 12);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else if (u <= 0x1FFFFF) {
      if (outIdx + 3 >= endIdx) break;
      outU8Array[outIdx++] = 0xF0 | (u >> 18);
      outU8Array[outIdx++] = 0x80 | ((u >> 12) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else if (u <= 0x3FFFFFF) {
      if (outIdx + 4 >= endIdx) break;
      outU8Array[outIdx++] = 0xF8 | (u >> 24);
      outU8Array[outIdx++] = 0x80 | ((u >> 18) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 12) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    } else {
      if (outIdx + 5 >= endIdx) break;
      outU8Array[outIdx++] = 0xFC | (u >> 30);
      outU8Array[outIdx++] = 0x80 | ((u >> 24) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 18) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 12) & 63);
      outU8Array[outIdx++] = 0x80 | ((u >> 6) & 63);
      outU8Array[outIdx++] = 0x80 | (u & 63);
    }
  }
  // Null-terminate the pointer to the buffer.
  outU8Array[outIdx] = 0;
  return outIdx - startIdx;
}
Module["stringToUTF8Array"] = stringToUTF8Array;

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF8 form. The copy will require at most str.length*4+1 bytes of space in the HEAP.
// Use the function lengthBytesUTF8() to compute the exact number of bytes (excluding null terminator) that this function will write.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF8(str, outPtr, maxBytesToWrite) {
  assert(typeof maxBytesToWrite == 'number', 'stringToUTF8(str, outPtr, maxBytesToWrite) is missing the third parameter that specifies the length of the output buffer!');
  return stringToUTF8Array(str, HEAPU8,outPtr, maxBytesToWrite);
}
Module["stringToUTF8"] = stringToUTF8;

// Returns the number of bytes the given Javascript string takes if encoded as a UTF8 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF8(str) {
  var len = 0;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! So decode UTF16->UTF32->UTF8.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var u = str.charCodeAt(i); // possibly a lead surrogate
    if (u >= 0xD800 && u <= 0xDFFF) u = 0x10000 + ((u & 0x3FF) << 10) | (str.charCodeAt(++i) & 0x3FF);
    if (u <= 0x7F) {
      ++len;
    } else if (u <= 0x7FF) {
      len += 2;
    } else if (u <= 0xFFFF) {
      len += 3;
    } else if (u <= 0x1FFFFF) {
      len += 4;
    } else if (u <= 0x3FFFFFF) {
      len += 5;
    } else {
      len += 6;
    }
  }
  return len;
}
Module["lengthBytesUTF8"] = lengthBytesUTF8;

// Given a pointer 'ptr' to a null-terminated UTF16LE-encoded string in the emscripten HEAP, returns
// a copy of that string as a Javascript String object.

function UTF16ToString(ptr) {
  var i = 0;

  var str = '';
  while (1) {
    var codeUnit = HEAP16[(((ptr)+(i*2))>>1)];
    if (codeUnit == 0)
      return str;
    ++i;
    // fromCharCode constructs a character from a UTF-16 code unit, so we can pass the UTF16 string right through.
    str += String.fromCharCode(codeUnit);
  }
}
Module["UTF16ToString"] = UTF16ToString;

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF16 form. The copy will require at most str.length*4+2 bytes of space in the HEAP.
// Use the function lengthBytesUTF16() to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outPtr: Byte address in Emscripten HEAP where to write the string to.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array. This count should include the null 
//                    terminator, i.e. if maxBytesToWrite=2, only the null terminator will be written and nothing else.
//                    maxBytesToWrite<2 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF16(str, outPtr, maxBytesToWrite) {
  assert(typeof maxBytesToWrite == 'number', 'stringToUTF16(str, outPtr, maxBytesToWrite) is missing the third parameter that specifies the length of the output buffer!');
  // Backwards compatibility: if max bytes is not specified, assume unsafe unbounded write is allowed.
  if (maxBytesToWrite === undefined) {
    maxBytesToWrite = 0x7FFFFFFF;
  }
  if (maxBytesToWrite < 2) return 0;
  maxBytesToWrite -= 2; // Null terminator.
  var startPtr = outPtr;
  var numCharsToWrite = (maxBytesToWrite < str.length*2) ? (maxBytesToWrite / 2) : str.length;
  for (var i = 0; i < numCharsToWrite; ++i) {
    // charCodeAt returns a UTF-16 encoded code unit, so it can be directly written to the HEAP.
    var codeUnit = str.charCodeAt(i); // possibly a lead surrogate
    HEAP16[((outPtr)>>1)]=codeUnit;
    outPtr += 2;
  }
  // Null-terminate the pointer to the HEAP.
  HEAP16[((outPtr)>>1)]=0;
  return outPtr - startPtr;
}
Module["stringToUTF16"] = stringToUTF16;

// Returns the number of bytes the given Javascript string takes if encoded as a UTF16 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF16(str) {
  return str.length*2;
}
Module["lengthBytesUTF16"] = lengthBytesUTF16;

function UTF32ToString(ptr) {
  var i = 0;

  var str = '';
  while (1) {
    var utf32 = HEAP32[(((ptr)+(i*4))>>2)];
    if (utf32 == 0)
      return str;
    ++i;
    // Gotcha: fromCharCode constructs a character from a UTF-16 encoded code (pair), not from a Unicode code point! So encode the code point to UTF-16 for constructing.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    if (utf32 >= 0x10000) {
      var ch = utf32 - 0x10000;
      str += String.fromCharCode(0xD800 | (ch >> 10), 0xDC00 | (ch & 0x3FF));
    } else {
      str += String.fromCharCode(utf32);
    }
  }
}
Module["UTF32ToString"] = UTF32ToString;

// Copies the given Javascript String object 'str' to the emscripten HEAP at address 'outPtr',
// null-terminated and encoded in UTF32 form. The copy will require at most str.length*4+4 bytes of space in the HEAP.
// Use the function lengthBytesUTF32() to compute the exact number of bytes (excluding null terminator) that this function will write.
// Parameters:
//   str: the Javascript string to copy.
//   outPtr: Byte address in Emscripten HEAP where to write the string to.
//   maxBytesToWrite: The maximum number of bytes this function can write to the array. This count should include the null 
//                    terminator, i.e. if maxBytesToWrite=4, only the null terminator will be written and nothing else.
//                    maxBytesToWrite<4 does not write any bytes to the output, not even the null terminator.
// Returns the number of bytes written, EXCLUDING the null terminator.

function stringToUTF32(str, outPtr, maxBytesToWrite) {
  assert(typeof maxBytesToWrite == 'number', 'stringToUTF32(str, outPtr, maxBytesToWrite) is missing the third parameter that specifies the length of the output buffer!');
  // Backwards compatibility: if max bytes is not specified, assume unsafe unbounded write is allowed.
  if (maxBytesToWrite === undefined) {
    maxBytesToWrite = 0x7FFFFFFF;
  }
  if (maxBytesToWrite < 4) return 0;
  var startPtr = outPtr;
  var endPtr = startPtr + maxBytesToWrite - 4;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! We must decode the string to UTF-32 to the heap.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var codeUnit = str.charCodeAt(i); // possibly a lead surrogate
    if (codeUnit >= 0xD800 && codeUnit <= 0xDFFF) {
      var trailSurrogate = str.charCodeAt(++i);
      codeUnit = 0x10000 + ((codeUnit & 0x3FF) << 10) | (trailSurrogate & 0x3FF);
    }
    HEAP32[((outPtr)>>2)]=codeUnit;
    outPtr += 4;
    if (outPtr + 4 > endPtr) break;
  }
  // Null-terminate the pointer to the HEAP.
  HEAP32[((outPtr)>>2)]=0;
  return outPtr - startPtr;
}
Module["stringToUTF32"] = stringToUTF32;

// Returns the number of bytes the given Javascript string takes if encoded as a UTF16 byte array, EXCLUDING the null terminator byte.

function lengthBytesUTF32(str) {
  var len = 0;
  for (var i = 0; i < str.length; ++i) {
    // Gotcha: charCodeAt returns a 16-bit word that is a UTF-16 encoded code unit, not a Unicode code point of the character! We must decode the string to UTF-32 to the heap.
    // See http://unicode.org/faq/utf_bom.html#utf16-3
    var codeUnit = str.charCodeAt(i);
    if (codeUnit >= 0xD800 && codeUnit <= 0xDFFF) ++i; // possibly a lead surrogate, so skip over the tail surrogate.
    len += 4;
  }

  return len;
}
Module["lengthBytesUTF32"] = lengthBytesUTF32;

function demangle(func) {
  var hasLibcxxabi = !!Module['___cxa_demangle'];
  if (hasLibcxxabi) {
    try {
      var buf = _malloc(func.length);
      writeStringToMemory(func.substr(1), buf);
      var status = _malloc(4);
      var ret = Module['___cxa_demangle'](buf, 0, 0, status);
      if (getValue(status, 'i32') === 0 && ret) {
        return Pointer_stringify(ret);
      }
      // otherwise, libcxxabi failed, we can try ours which may return a partial result
    } catch(e) {
      // failure when using libcxxabi, we can try ours which may return a partial result
    } finally {
      if (buf) _free(buf);
      if (status) _free(status);
      if (ret) _free(ret);
    }
  }
  var i = 3;
  // params, etc.
  var basicTypes = {
    'v': 'void',
    'b': 'bool',
    'c': 'char',
    's': 'short',
    'i': 'int',
    'l': 'long',
    'f': 'float',
    'd': 'double',
    'w': 'wchar_t',
    'a': 'signed char',
    'h': 'unsigned char',
    't': 'unsigned short',
    'j': 'unsigned int',
    'm': 'unsigned long',
    'x': 'long long',
    'y': 'unsigned long long',
    'z': '...'
  };
  var subs = [];
  var first = true;
  function dump(x) {
    //return;
    if (x) Module.print(x);
    Module.print(func);
    var pre = '';
    for (var a = 0; a < i; a++) pre += ' ';
    Module.print (pre + '^');
  }
  function parseNested() {
    i++;
    if (func[i] === 'K') i++; // ignore const
    var parts = [];
    while (func[i] !== 'E') {
      if (func[i] === 'S') { // substitution
        i++;
        var next = func.indexOf('_', i);
        var num = func.substring(i, next) || 0;
        parts.push(subs[num] || '?');
        i = next+1;
        continue;
      }
      if (func[i] === 'C') { // constructor
        parts.push(parts[parts.length-1]);
        i += 2;
        continue;
      }
      var size = parseInt(func.substr(i));
      var pre = size.toString().length;
      if (!size || !pre) { i--; break; } // counter i++ below us
      var curr = func.substr(i + pre, size);
      parts.push(curr);
      subs.push(curr);
      i += pre + size;
    }
    i++; // skip E
    return parts;
  }
  function parse(rawList, limit, allowVoid) { // main parser
    limit = limit || Infinity;
    var ret = '', list = [];
    function flushList() {
      return '(' + list.join(', ') + ')';
    }
    var name;
    if (func[i] === 'N') {
      // namespaced N-E
      name = parseNested().join('::');
      limit--;
      if (limit === 0) return rawList ? [name] : name;
    } else {
      // not namespaced
      if (func[i] === 'K' || (first && func[i] === 'L')) i++; // ignore const and first 'L'
      var size = parseInt(func.substr(i));
      if (size) {
        var pre = size.toString().length;
        name = func.substr(i + pre, size);
        i += pre + size;
      }
    }
    first = false;
    if (func[i] === 'I') {
      i++;
      var iList = parse(true);
      var iRet = parse(true, 1, true);
      ret += iRet[0] + ' ' + name + '<' + iList.join(', ') + '>';
    } else {
      ret = name;
    }
    paramLoop: while (i < func.length && limit-- > 0) {
      //dump('paramLoop');
      var c = func[i++];
      if (c in basicTypes) {
        list.push(basicTypes[c]);
      } else {
        switch (c) {
          case 'P': list.push(parse(true, 1, true)[0] + '*'); break; // pointer
          case 'R': list.push(parse(true, 1, true)[0] + '&'); break; // reference
          case 'L': { // literal
            i++; // skip basic type
            var end = func.indexOf('E', i);
            var size = end - i;
            list.push(func.substr(i, size));
            i += size + 2; // size + 'EE'
            break;
          }
          case 'A': { // array
            var size = parseInt(func.substr(i));
            i += size.toString().length;
            if (func[i] !== '_') throw '?';
            i++; // skip _
            list.push(parse(true, 1, true)[0] + ' [' + size + ']');
            break;
          }
          case 'E': break paramLoop;
          default: ret += '?' + c; break paramLoop;
        }
      }
    }
    if (!allowVoid && list.length === 1 && list[0] === 'void') list = []; // avoid (void)
    if (rawList) {
      if (ret) {
        list.push(ret + '?');
      }
      return list;
    } else {
      return ret + flushList();
    }
  }
  var parsed = func;
  try {
    // Special-case the entry point, since its name differs from other name mangling.
    if (func == 'Object._main' || func == '_main') {
      return 'main()';
    }
    if (typeof func === 'number') func = Pointer_stringify(func);
    if (func[0] !== '_') return func;
    if (func[1] !== '_') return func; // C function
    if (func[2] !== 'Z') return func;
    switch (func[3]) {
      case 'n': return 'operator new()';
      case 'd': return 'operator delete()';
    }
    parsed = parse();
  } catch(e) {
    parsed += '?';
  }
  if (parsed.indexOf('?') >= 0 && !hasLibcxxabi) {
    Runtime.warnOnce('warning: a problem occurred in builtin C++ name demangling; build with  -s DEMANGLE_SUPPORT=1  to link in libcxxabi demangling');
  }
  return parsed;
}

function demangleAll(text) {
  return text.replace(/__Z[\w\d_]+/g, function(x) { var y = demangle(x); return x === y ? x : (x + ' [' + y + ']') });
}

function jsStackTrace() {
  var err = new Error();
  if (!err.stack) {
    // IE10+ special cases: It does have callstack info, but it is only populated if an Error object is thrown,
    // so try that as a special-case.
    try {
      throw new Error(0);
    } catch(e) {
      err = e;
    }
    if (!err.stack) {
      return '(no stack trace available)';
    }
  }
  return err.stack.toString();
}

function stackTrace() {
  return demangleAll(jsStackTrace());
}
Module["stackTrace"] = stackTrace;

// Memory management

var PAGE_SIZE = 4096;

function alignMemoryPage(x) {
  if (x % 4096 > 0) {
    x += (4096 - (x % 4096));
  }
  return x;
}

var HEAP;
var buffer;
var HEAP8, HEAPU8, HEAP16, HEAPU16, HEAP32, HEAPU32, HEAPF32, HEAPF64;

function updateGlobalBuffer(buf) {
  Module['buffer'] = buffer = buf;
}

function updateGlobalBufferViews() {
  Module['HEAP8'] = HEAP8 = new Int8Array(buffer);
  Module['HEAP16'] = HEAP16 = new Int16Array(buffer);
  Module['HEAP32'] = HEAP32 = new Int32Array(buffer);
  Module['HEAPU8'] = HEAPU8 = new Uint8Array(buffer);
  Module['HEAPU16'] = HEAPU16 = new Uint16Array(buffer);
  Module['HEAPU32'] = HEAPU32 = new Uint32Array(buffer);
  Module['HEAPF32'] = HEAPF32 = new Float32Array(buffer);
  Module['HEAPF64'] = HEAPF64 = new Float64Array(buffer);
}

var STATIC_BASE = 0, STATICTOP = 0, staticSealed = false; // static area
var STACK_BASE = 0, STACKTOP = 0, STACK_MAX = 0; // stack area
var DYNAMIC_BASE = 0, DYNAMICTOP = 0; // dynamic area handled by sbrk


function abortOnCannotGrowMemory() {
  abort('Cannot enlarge memory arrays. Either (1) compile with  -s TOTAL_MEMORY=X  with X higher than the current value ' + TOTAL_MEMORY + ', (2) compile with  -s ALLOW_MEMORY_GROWTH=1  which adjusts the size at runtime but prevents some optimizations, (3) set Module.TOTAL_MEMORY to a higher value before the program runs, or if you want malloc to return NULL (0) instead of this abort, compile with  -s ABORTING_MALLOC=0 ');
}

function enlargeMemory() {
  abortOnCannotGrowMemory();
}


var TOTAL_STACK = Module['TOTAL_STACK'] || 5242880;
var TOTAL_MEMORY = Module['TOTAL_MEMORY'] || 16777216;

var totalMemory = 64*1024;
while (totalMemory < TOTAL_MEMORY || totalMemory < 2*TOTAL_STACK) {
  if (totalMemory < 16*1024*1024) {
    totalMemory *= 2;
  } else {
    totalMemory += 16*1024*1024
  }
}
if (totalMemory !== TOTAL_MEMORY) {
  Module.printErr('increasing TOTAL_MEMORY to ' + totalMemory + ' to be compliant with the asm.js spec (and given that TOTAL_STACK=' + TOTAL_STACK + ')');
  TOTAL_MEMORY = totalMemory;
}

// Initialize the runtime's memory
// check for full engine support (use string 'subarray' to avoid closure compiler confusion)
assert(typeof Int32Array !== 'undefined' && typeof Float64Array !== 'undefined' && !!(new Int32Array(1)['subarray']) && !!(new Int32Array(1)['set']),
       'JS engine does not provide full typed array support');



// Use a provided buffer, if there is one, or else allocate a new one
if (Module['buffer']) {
  buffer = Module['buffer'];
  assert(buffer.byteLength === TOTAL_MEMORY, 'provided buffer should be ' + TOTAL_MEMORY + ' bytes, but it is ' + buffer.byteLength);
} else {
  buffer = new ArrayBuffer(TOTAL_MEMORY);
}
updateGlobalBufferViews();


// Endianness check (note: assumes compiler arch was little-endian)
HEAP32[0] = 255;
assert(HEAPU8[0] === 255 && HEAPU8[3] === 0, 'Typed arrays 2 must be run on a little-endian system');

Module['HEAP'] = HEAP;
Module['buffer'] = buffer;
Module['HEAP8'] = HEAP8;
Module['HEAP16'] = HEAP16;
Module['HEAP32'] = HEAP32;
Module['HEAPU8'] = HEAPU8;
Module['HEAPU16'] = HEAPU16;
Module['HEAPU32'] = HEAPU32;
Module['HEAPF32'] = HEAPF32;
Module['HEAPF64'] = HEAPF64;

function callRuntimeCallbacks(callbacks) {
  while(callbacks.length > 0) {
    var callback = callbacks.shift();
    if (typeof callback == 'function') {
      callback();
      continue;
    }
    var func = callback.func;
    if (typeof func === 'number') {
      if (callback.arg === undefined) {
        Runtime.dynCall('v', func);
      } else {
        Runtime.dynCall('vi', func, [callback.arg]);
      }
    } else {
      func(callback.arg === undefined ? null : callback.arg);
    }
  }
}

var __ATPRERUN__  = []; // functions called before the runtime is initialized
var __ATINIT__    = []; // functions called during startup
var __ATMAIN__    = []; // functions called when main() is to be run
var __ATEXIT__    = []; // functions called during shutdown
var __ATPOSTRUN__ = []; // functions called after the runtime has exited

var runtimeInitialized = false;
var runtimeExited = false;


function preRun() {
  // compatibility - merge in anything from Module['preRun'] at this time
  if (Module['preRun']) {
    if (typeof Module['preRun'] == 'function') Module['preRun'] = [Module['preRun']];
    while (Module['preRun'].length) {
      addOnPreRun(Module['preRun'].shift());
    }
  }
  callRuntimeCallbacks(__ATPRERUN__);
}

function ensureInitRuntime() {
  if (runtimeInitialized) return;
  runtimeInitialized = true;
  callRuntimeCallbacks(__ATINIT__);
}

function preMain() {
  callRuntimeCallbacks(__ATMAIN__);
}

function exitRuntime() {
  callRuntimeCallbacks(__ATEXIT__);
  runtimeExited = true;
}

function postRun() {
  // compatibility - merge in anything from Module['postRun'] at this time
  if (Module['postRun']) {
    if (typeof Module['postRun'] == 'function') Module['postRun'] = [Module['postRun']];
    while (Module['postRun'].length) {
      addOnPostRun(Module['postRun'].shift());
    }
  }
  callRuntimeCallbacks(__ATPOSTRUN__);
}

function addOnPreRun(cb) {
  __ATPRERUN__.unshift(cb);
}
Module["addOnPreRun"] = addOnPreRun;

function addOnInit(cb) {
  __ATINIT__.unshift(cb);
}
Module["addOnInit"] = addOnInit;

function addOnPreMain(cb) {
  __ATMAIN__.unshift(cb);
}
Module["addOnPreMain"] = addOnPreMain;

function addOnExit(cb) {
  __ATEXIT__.unshift(cb);
}
Module["addOnExit"] = addOnExit;

function addOnPostRun(cb) {
  __ATPOSTRUN__.unshift(cb);
}
Module["addOnPostRun"] = addOnPostRun;

// Tools


function intArrayFromString(stringy, dontAddNull, length /* optional */) {
  var len = length > 0 ? length : lengthBytesUTF8(stringy)+1;
  var u8array = new Array(len);
  var numBytesWritten = stringToUTF8Array(stringy, u8array, 0, u8array.length);
  if (dontAddNull) u8array.length = numBytesWritten;
  return u8array;
}
Module["intArrayFromString"] = intArrayFromString;

function intArrayToString(array) {
  var ret = [];
  for (var i = 0; i < array.length; i++) {
    var chr = array[i];
    if (chr > 0xFF) {
      assert(false, 'Character code ' + chr + ' (' + String.fromCharCode(chr) + ')  at offset ' + i + ' not in 0x00-0xFF.');
      chr &= 0xFF;
    }
    ret.push(String.fromCharCode(chr));
  }
  return ret.join('');
}
Module["intArrayToString"] = intArrayToString;

function writeStringToMemory(string, buffer, dontAddNull) {
  var array = intArrayFromString(string, dontAddNull);
  var i = 0;
  while (i < array.length) {
    var chr = array[i];
    HEAP8[(((buffer)+(i))>>0)]=chr;
    i = i + 1;
  }
}
Module["writeStringToMemory"] = writeStringToMemory;

function writeArrayToMemory(array, buffer) {
  for (var i = 0; i < array.length; i++) {
    HEAP8[((buffer++)>>0)]=array[i];
  }
}
Module["writeArrayToMemory"] = writeArrayToMemory;

function writeAsciiToMemory(str, buffer, dontAddNull) {
  for (var i = 0; i < str.length; ++i) {
    assert(str.charCodeAt(i) === str.charCodeAt(i)&0xff);
    HEAP8[((buffer++)>>0)]=str.charCodeAt(i);
  }
  // Null-terminate the pointer to the HEAP.
  if (!dontAddNull) HEAP8[((buffer)>>0)]=0;
}
Module["writeAsciiToMemory"] = writeAsciiToMemory;

function unSign(value, bits, ignore) {
  if (value >= 0) {
    return value;
  }
  return bits <= 32 ? 2*Math.abs(1 << (bits-1)) + value // Need some trickery, since if bits == 32, we are right at the limit of the bits JS uses in bitshifts
                    : Math.pow(2, bits)         + value;
}
function reSign(value, bits, ignore) {
  if (value <= 0) {
    return value;
  }
  var half = bits <= 32 ? Math.abs(1 << (bits-1)) // abs is needed if bits == 32
                        : Math.pow(2, bits-1);
  if (value >= half && (bits <= 32 || value > half)) { // for huge values, we can hit the precision limit and always get true here. so don't do that
                                                       // but, in general there is no perfect solution here. With 64-bit ints, we get rounding and errors
                                                       // TODO: In i64 mode 1, resign the two parts separately and safely
    value = -2*half + value; // Cannot bitshift half, as it may be at the limit of the bits JS uses in bitshifts
  }
  return value;
}


// check for imul support, and also for correctness ( https://bugs.webkit.org/show_bug.cgi?id=126345 )
if (!Math['imul'] || Math['imul'](0xffffffff, 5) !== -5) Math['imul'] = function imul(a, b) {
  var ah  = a >>> 16;
  var al = a & 0xffff;
  var bh  = b >>> 16;
  var bl = b & 0xffff;
  return (al*bl + ((ah*bl + al*bh) << 16))|0;
};
Math.imul = Math['imul'];


if (!Math['clz32']) Math['clz32'] = function(x) {
  x = x >>> 0;
  for (var i = 0; i < 32; i++) {
    if (x & (1 << (31 - i))) return i;
  }
  return 32;
};
Math.clz32 = Math['clz32']

var Math_abs = Math.abs;
var Math_cos = Math.cos;
var Math_sin = Math.sin;
var Math_tan = Math.tan;
var Math_acos = Math.acos;
var Math_asin = Math.asin;
var Math_atan = Math.atan;
var Math_atan2 = Math.atan2;
var Math_exp = Math.exp;
var Math_log = Math.log;
var Math_sqrt = Math.sqrt;
var Math_ceil = Math.ceil;
var Math_floor = Math.floor;
var Math_pow = Math.pow;
var Math_imul = Math.imul;
var Math_fround = Math.fround;
var Math_min = Math.min;
var Math_clz32 = Math.clz32;

// A counter of dependencies for calling run(). If we need to
// do asynchronous work before running, increment this and
// decrement it. Incrementing must happen in a place like
// PRE_RUN_ADDITIONS (used by emcc to add file preloading).
// Note that you can add dependencies in preRun, even though
// it happens right before run - run will be postponed until
// the dependencies are met.
var runDependencies = 0;
var runDependencyWatcher = null;
var dependenciesFulfilled = null; // overridden to take different actions when all run dependencies are fulfilled
var runDependencyTracking = {};

function getUniqueRunDependency(id) {
  var orig = id;
  while (1) {
    if (!runDependencyTracking[id]) return id;
    id = orig + Math.random();
  }
  return id;
}

function addRunDependency(id) {
  runDependencies++;
  if (Module['monitorRunDependencies']) {
    Module['monitorRunDependencies'](runDependencies);
  }
  if (id) {
    assert(!runDependencyTracking[id]);
    runDependencyTracking[id] = 1;
    if (runDependencyWatcher === null && typeof setInterval !== 'undefined') {
      // Check for missing dependencies every few seconds
      runDependencyWatcher = setInterval(function() {
        if (ABORT) {
          clearInterval(runDependencyWatcher);
          runDependencyWatcher = null;
          return;
        }
        var shown = false;
        for (var dep in runDependencyTracking) {
          if (!shown) {
            shown = true;
            Module.printErr('still waiting on run dependencies:');
          }
          Module.printErr('dependency: ' + dep);
        }
        if (shown) {
          Module.printErr('(end of list)');
        }
      }, 10000);
    }
  } else {
    Module.printErr('warning: run dependency added without ID');
  }
}
Module["addRunDependency"] = addRunDependency;

function removeRunDependency(id) {
  runDependencies--;
  if (Module['monitorRunDependencies']) {
    Module['monitorRunDependencies'](runDependencies);
  }
  if (id) {
    assert(runDependencyTracking[id]);
    delete runDependencyTracking[id];
  } else {
    Module.printErr('warning: run dependency removed without ID');
  }
  if (runDependencies == 0) {
    if (runDependencyWatcher !== null) {
      clearInterval(runDependencyWatcher);
      runDependencyWatcher = null;
    }
    if (dependenciesFulfilled) {
      var callback = dependenciesFulfilled;
      dependenciesFulfilled = null;
      callback(); // can add another dependenciesFulfilled
    }
  }
}
Module["removeRunDependency"] = removeRunDependency;

Module["preloadedImages"] = {}; // maps url to image data
Module["preloadedAudios"] = {}; // maps url to audio data



var memoryInitializer = null;



// === Body ===

var ASM_CONSTS = [];




STATIC_BASE = 8;

STATICTOP = STATIC_BASE + 546448;
  /* global initializers */  __ATINIT__.push();
  

/* memory initializer */ allocate([0,0,0,0,0,0,0,0,0,0,0,0,0,0,224,63,0,0,0,0,0,0,240,63,0,0,0,0,0,0,248,63,0,0,0,0,0,0,4,64,0,0,0,0,0,0,18,64,0,0,0,0,0,0,33,64,0,0,0,0,0,128,48,64,0,0,0,4,107,244,52,66,0,0,0,0,0,0,0,0,0,0,0,0,0,0,224,63,0,0,0,0,0,0,240,63,0,0,0,0,0,0,248,63,0,0,0,0,0,0,0,64,0,0,0,0,0,0,4,64,0,0,0,0,0,0,18,64,0,0,0,0,0,0,33,64,0,0,0,4,107,244,52,66,255,255,255,255,0,0,12,195,0,0,12,195,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,32,194,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,128,0,0,0,63,0,0,0,63,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,191,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,191,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,191,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,191,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,210,66,0,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,249,213,64,0,0,0,0,0,64,223,64,0,0,0,0,0,136,227,64,0,0,0,0,0,112,231,64,0,0,0,0,0,88,235,64,0,0,0,0,0,64,239,64,0,0,0,0,0,136,243,64,0,0,0,0,0,112,247,64,0,0,0,0,0,88,251,64,0,0,0,0,0,64,255,64,0,0,0,0,0,136,3,65,0,0,0,0,136,132,14,65,154,153,153,153,153,153,185,191,0,0,0,0,0,0,0,0,154,153,153,153,153,153,185,63,154,153,153,153,153,153,201,63,51,51,51,51,51,51,211,63,154,153,153,153,153,153,217,63,0,0,0,0,0,0,224,63,51,51,51,51,51,51,227,63,102,102,102,102,102,102,230,63,154,153,153,153,153,153,233,63,205,204,204,204,204,204,236,63,0,0,0,0,0,0,240,63,0,0,0,0,0,0,224,63,0,0,0,0,0,0,240,63,0,0,0,0,0,0,240,63,205,204,204,204,204,204,244,63,154,153,153,153,153,153,249,63,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,64,0,0,0,0,0,0,12,64,0,0,0,0,0,0,16,64,0,0,0,0,0,0,16,64,51,51,51,51,51,51,17,64,102,102,102,102,102,102,18,64,0,0,0,0,0,0,20,64,0,0,0,0,0,0,20,64,0,0,0,0,0,0,20,64,0,0,0,0,0,0,20,64,0,0,0,0,0,0,20,64,0,0,0,0,0,0,20,64,0,0,0,0,0,0,20,64,154,153,153,153,153,153,201,63,154,153,153,153,153,153,201,63,154,153,153,153,153,153,201,63,154,153,153,153,153,153,217,63,51,51,51,51,51,51,227,63,0,0,0,0,128,135,195,64,0,0,0,0,128,135,195,64,0,0,0,0,128,135,195,64,0,0,0,0,128,135,195,64,0,0,0,0,128,135,195,64,0,0,0,0,128,135,195,64,205,204,204,204,204,204,43,64,51,51,51,51,51,51,46,64,154,153,153,153,153,153,47,64,0,0,0,0,0,128,48,64,51,51,51,51,51,51,49,64,102,102,102,102,102,230,50,64,154,153,153,153,153,25,52,64,0,0,0,0,0,0,72,64,0,0,0,0,0,56,143,64,0,0,0,0,0,56,143,64,0,0,0,0,0,56,143,64,0,0,0,0,0,56,143,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,240,63,0,0,0,0,0,0,240,63,0,0,0,0,0,0,248,63,0,0,0,0,0,0,0,64,0,0,0,0,0,0,0,64,0,0,0,0,0,0,4,64,154,153,153,153,153,153,5,64,0,0,0,0,0,0,8,64,154,153,153,153,153,153,13,64,0,0,0,0,0,0,16,64,0,0,0,0,0,0,16,64,0,0,0,0,0,88,203,64,0,0,0,0,0,136,211,64,0,0,0,0,0,88,219,64,0,0,0,0,0,142,226,64,0,0,0,0,0,118,230,64,0,0,0,0,0,94,234,64,0,0,0,0,128,79,242,64,0,0,0,0,0,112,247,64,0,0,0,0,0,76,253,64,0,0,0,0,0,23,1,65,0,0,0,0,0,249,5,65,0,0,0,0,8,76,13,65,0,0,0,0,0,64,223,64,0,0,0,0,0,112,231,64,0,0,0,0,0,76,237,64,0,0,0,0,0,23,241,64,0,0,0,0,0,136,243,64,0,0,0,0,0,255,244,64,0,0,0,0,0,112,247,64,0,0,0,0,0,219,250,64,0,0,0,0,0,76,253,64,0,0,0,0,0,23,1,65,0,0,0,0,0,136,3,65,0,0,0,0,8,76,13,65,0,0,0,0,0,148,209,64,0,0,0,0,0,88,219,64,0,0,0,0,0,23,225,64,0,0,0,0,0,249,229,64,0,0,0,0,0,88,235,64,0,0,0,0,0,76,237,64,0,0,0,0,128,79,242,64,0,0,0,0,0,249,245,64,0,0,0,0,0,106,248,64,0,0,0,0,128,19,252,64,0,0,0,0,128,79,2,65,0,0,0,0,128,49,7,65,154,153,153,153,153,153,40,64,0,0,0,0,0,0,42,64,0,0,0,0,0,0,42,64,0,0,0,0,0,0,44,64,0,0,0,0,0,0,46,64,0,0,0,0,0,192,88,64,0,0,0,0,0,192,88,64,0,0,0,0,0,192,88,64,0,0,0,0,0,192,88,64,0,0,0,0,0,192,88,64,0,0,0,0,0,192,88,64,0,0,0,0,0,192,88,64,0,0,0,0,0,76,221,64,0,0,0,0,0,130,228,64,0,0,0,0,0,100,233,64,0,0,0,0,0,64,239,64,0,0,0,0,0,148,241,64,0,0,0,0,0,11,243,64,0,0,0,0,0,255,244,64,0,0,0,0,0,118,246,64,0,0,0,0,0,219,250,64,0,0,0,0,0,76,253,64,0,0,0,0,0,23,1,65,0,0,0,0,128,49,7,65,0,0,0,0,0,76,205,64,0,0,0,0,0,136,211,64,0,0,0,0,0,124,229,64,0,0,0,0,0,255,244,64,154,153,153,153,153,153,185,191,154,153,153,153,153,153,169,63,0,0,0,0,0,0,224,63,0,0,0,0,0,0,240,63,0,0,0,0,0,0,0,0,154,153,153,153,153,153,233,63,0,0,0,0,0,0,240,63,0,0,0,0,0,0,240,63,51,51,51,51,51,51,211,63,0,0,0,0,0,0,224,63,0,0,0,0,0,0,224,63,0,0,0,0,0,0,224,63,0,0,0,0,0,0,35,64,0,0,0,0,0,0,38,64,0,0,0,0,0,0,62,64,0,0,0,0,0,192,88,64,0,0,0,0,0,0,240,63,0,0,0,0,0,0,0,64,0,0,0,0,0,0,8,64,0,0,0,0,0,0,16,64,0,0,0,0,0,64,207,64,0,0,0,0,0,88,219,64,0,0,0,0,0,106,232,64,0,0,0,0,0,249,245,64,0,0,0,0,0,112,199,64,0,0,0,0,0,136,211,64,0,0,0,0,0,124,229,64,0,0,0,0,0,255,244,64,0,0,0,0,0,0,26,64,0,0,0,0,0,0,32,64,0,0,0,0,0,0,62,64,0,0,0,0,0,192,88,64,0,0,0,0,0,64,207,64,0,0,0,0,0,88,219,64,0,0,0,0,0,64,239,64,0,0,0,0,0,106,248,64,0,0,0,0,0,64,191,64,0,0,0,0,0,100,201,64,0,0,0,0,0,124,229,64,154,153,153,153,153,153,185,191,0,0,0,0,0,0,0,0,0,0,0,0,0,0,240,63,0,0,0,0,0,0,0,0,0,0,0,0,0,0,240,63,0,0,0,0,0,0,240,63,51,51,51,51,51,51,211,63,0,0,0,0,0,0,224,63,0,0,0,0,0,0,224,63,0,0,0,0,0,0,18,64,0,0,0,0,0,0,22,64,0,0,0,0,0,0,62,64,0,0,0,0,0,0,240,63,0,0,0,0,0,0,0,64,0,0,0,0,0,0,8,64,0,0,0,0,0,112,199,64,0,0,0,0,0,136,211,64,0,0,0,0,0,106,232,64,0,0,0,0,0,112,183,64,0,0,0,0,0,148,193,64,0,0,0,0,0,64,223,64,154,153,153,153,153,153,185,191,0,0,0,0,0,0,0,0,0,0,0,0,0,0,240,63,0,0,0,0,0,0,224,63,0,0,0,0,0,0,224,63,0,0,0,0,0,0,8,64,0,0,0,0,0,0,16,64,0,0,0,0,0,0,16,64,0,0,0,0,0,64,191,64,0,0,0,0,0,88,203,64,0,0,0,0,0,130,228,64,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,0,0,240,191,0,0,0,0,183,29,193,4,110,59,130,9,217,38,67,13,220,118,4,19,107,107,197,23,178,77,134,26,5,80,71,30,184,237,8,38,15,240,201,34,214,214,138,47,97,203,75,43,100,155,12,53,211,134,205,49,10,160,142,60,189,189,79,56,112,219,17,76,199,198,208,72,30,224,147,69,169,253,82,65,172,173,21,95,27,176,212,91,194,150,151,86,117,139,86,82,200,54,25,106,127,43,216,110,166,13,155,99,17,16,90,103,20,64,29,121,163,93,220,125,122,123,159,112,205,102,94,116,224,182,35,152,87,171,226,156,142,141,161,145,57,144,96,149,60,192,39,139,139,221,230,143,82,251,165,130,229,230,100,134,88,91,43,190,239,70,234,186,54,96,169,183,129,125,104,179,132,45,47,173,51,48,238,169,234,22,173,164,93,11,108,160,144,109,50,212,39,112,243,208,254,86,176,221,73,75,113,217,76,27,54,199,251,6,247,195,34,32,180,206,149,61,117,202,40,128,58,242,159,157,251,246,70,187,184,251,241,166,121,255,244,246,62,225,67,235,255,229,154,205,188,232,45,208,125,236,119,112,134,52,192,109,71,48,25,75,4,61,174,86,197,57,171,6,130,39,28,27,67,35,197,61,0,46,114,32,193,42,207,157,142,18,120,128,79,22,161,166,12,27,22,187,205,31,19,235,138,1,164,246,75,5,125,208,8,8,202,205,201,12,7,171,151,120,176,182,86,124,105,144,21,113,222,141,212,117,219,221,147,107,108,192,82,111,181,230,17,98,2,251,208,102,191,70,159,94,8,91,94,90,209,125,29,87,102,96,220,83,99,48,155,77,212,45,90,73,13,11,25,68,186,22,216,64,151,198,165,172,32,219,100,168,249,253,39,165,78,224,230,161,75,176,161,191,252,173,96,187,37,139,35,182,146,150,226,178,47,43,173,138,152,54,108,142,65,16,47,131,246,13,238,135,243,93,169,153,68,64,104,157,157,102,43,144,42,123,234,148,231,29,180,224,80,0,117,228,137,38,54,233,62,59,247,237,59,107,176,243,140,118,113,247,85,80,50,250,226,77,243,254,95,240,188,198,232,237,125,194,49,203,62,207,134,214,255,203,131,134,184,213,52,155,121,209,237,189,58,220,90,160,251,216,238,224,12,105,89,253,205,109,128,219,142,96,55,198,79,100,50,150,8,122,133,139,201,126,92,173,138,115,235,176,75,119,86,13,4,79,225,16,197,75,56,54,134,70,143,43,71,66,138,123,0,92,61,102,193,88,228,64,130,85,83,93,67,81,158,59,29,37,41,38,220,33,240,0,159,44,71,29,94,40,66,77,25,54,245,80,216,50,44,118,155,63,155,107,90,59,38,214,21,3,145,203,212,7,72,237,151,10,255,240,86,14,250,160,17,16,77,189,208,20,148,155,147,25,35,134,82,29,14,86,47,241,185,75,238,245,96,109,173,248,215,112,108,252,210,32,43,226,101,61,234,230,188,27,169,235,11,6,104,239,182,187,39,215,1,166,230,211,216,128,165,222,111,157,100,218,106,205,35,196,221,208,226,192,4,246,161,205,179,235,96,201,126,141,62,189,201,144,255,185,16,182,188,180,167,171,125,176,162,251,58,174,21,230,251,170,204,192,184,167,123,221,121,163,198,96,54,155,113,125,247,159,168,91,180,146,31,70,117,150,26,22,50,136,173,11,243,140,116,45,176,129,195,48,113,133,153,144,138,93,46,141,75,89,247,171,8,84,64,182,201,80,69,230,142,78,242,251,79,74,43,221,12,71,156,192,205,67,33,125,130,123,150,96,67,127,79,70,0,114,248,91,193,118,253,11,134,104,74,22,71,108,147,48,4,97,36,45,197,101,233,75,155,17,94,86,90,21,135,112,25,24,48,109,216,28,53,61,159,2,130,32,94,6,91,6,29,11,236,27,220,15,81,166,147,55,230,187,82,51,63,157,17,62,136,128,208,58,141,208,151,36,58,205,86,32,227,235,21,45,84,246,212,41,121,38,169,197,206,59,104,193,23,29,43,204,160,0,234,200,165,80,173,214,18,77,108,210,203,107,47,223,124,118,238,219,193,203,161,227,118,214,96,231,175,240,35,234,24,237,226,238,29,189,165,240,170,160,100,244,115,134,39,249,196,155,230,253,9,253,184,137,190,224,121,141,103,198,58,128,208,219,251,132,213,139,188,154,98,150,125,158,187,176,62,147,12,173,255,151,177,16,176,175,6,13,113,171,223,43,50,166,104,54,243,162,109,102,180,188,218,123,117,184,3,93,54,181,180,64,247,177,0,0,0,0,1,0,0,0,3,0,0,0,7,0,0,0,15,0,0,0,31,0,0,0,63,0,0,0,127,0,0,0,255,0,0,0,255,1,0,0,255,3,0,0,255,7,0,0,255,15,0,0,255,31,0,0,255,63,0,0,255,127,0,0,255,255,0,0,255,255,1,0,255,255,3,0,255,255,7,0,255,255,15,0,255,255,31,0,255,255,63,0,255,255,127,0,255,255,255,0,255,255,255,1,255,255,255,3,255,255,255,7,255,255,255,15,255,255,255,31,255,255,255,63,255,255,255,127,255,255,255,255,116,14,0,0,244,14,0,0,244,15,0,0,244,17,0,0,244,21,0,0,244,29,0,0,244,45,0,0,244,77,0,0,24,0,120,58,76,70,11,60,242,204,192,60,116,252,59,61,86,73,154,61,241,93,228,61,248,163,29,62,180,231,78,62,54,157,130,62,78,220,159,62,193,174,190,62,65,132,222,62,173,194,254,62,186,101,15,63,248,0,31,63,29,233,45,63,249,219,59,63,45,162,72,63,160,17,84,63,38,15,94,63,46,143,102,63,112,149,109,63,174,51,115,63,159,135,119,63,66,184,122,63,196,242,124,63,75,103,126,63,196,69,127,63,241,186,127,63,217,237,127,63,162,253,127,63,248,255,127,63,168,9,120,57,17,119,11,59,135,139,193,59,74,113,61,60,148,82,156,60,94,8,233,60,42,83,34,61,74,118,87,61,138,227,137,61,7,140,171,61,34,154,208,61,108,239,248,61,164,52,18,62,100,112,41,62,65,21,66,62,67,11,92,62,47,56,119,62,197,191,137,62,92,97,152,62,135,112,167,62,4,220,182,62,188,145,198,62,231,126,214,62,48,144,230,62,227,177,246,62,13,104,3,63,121,107,11,63,98,89,19,63,42,40,27,63,137,206,34,63,166,67,42,63,49,127,49,63,126,121,56,63,153,43,63,63,92,143,69,63,127,159,75,63,165,87,81,63,104,180,86,63,89,179,91,63,8,83,96,63,252,146,100,63,177,115,104,63,138,246,107,63,198,29,111,63,109,236,113,63,62,102,116,63,154,143,118,63,104,109,120,63,3,5,122,63,26,92,123,63,153,120,124,63,143,96,125,63,17,26,126,63,39,171,126,63,176,25,127,63,74,107,127,63,68,165,127,63,132,204,127,63,123,229,127,63,17,244,127,63,158,251,127,63,219,254,127,63,218,255,127,63,0,0,128,63,5,12,120,56,50,131,11,58,118,186,193,58,226,203,61,59,38,207,156,59,139,32,234,59,245,102,35,60,63,100,89,60,184,127,139,60,59,23,174,60,239,114,212,60,96,140,254,60,45,46,22,61,114,237,46,61,155,127,73,61,220,223,101,61,123,4,130,61,159,250,145,61,71,207,162,61,38,127,180,61,173,6,199,61,16,98,218,61,63,141,238,61,244,193,1,62,185,160,12,62,128,224,23,62,182,126,35,62,166,120,47,62,116,203,59,62,34,116,72,62,141,111,85,62,107,186,98,62,83,81,112,62,180,48,126,62,110,42,134,62,252,92,141,62,9,174,148,62,138,27,156,62,100,163,163,62,112,67,171,62,119,249,178,62,54,195,186,62,93,158,194,62,147,136,202,62,118,127,210,62,154,128,218,62,142,137,226,62,217,151,234,62,2,169,242,62,139,186,250,62,251,100,1,63,99,106,5,63,65,108,9,63,89,105,13,63,116,96,17,63,94,80,21,63,231,55,25,63,231,21,29,63,58,233,32,63,197,176,36,63,116,107,40,63,62,24,44,63,35,182,47,63,43,68,51,63,109,193,54,63,10,45,58,63,48,134,61,63,26,204,64,63,17,254,67,63,107,27,71,63,142,35,74,63,238,21,77,63,15,242,79,63,132,183,82,63,239,101,85,63,3,253,87,63,129,124,90,63,60,228,92,63,21,52,95,63,254,107,97,63,246,139,99,63,14,148,101,63,98,132,103,63,33,93,105,63,133,30,107,63,213,200,108,63,103,92,110,63,155,217,111,63,224,64,113,63,172,146,114,63,131,207,115,63,241,247,116,63,139,12,118,63,239,13,119,63,193,252,119,63,172,217,120,63,99,165,121,63,155,96,122,63,15,12,123,63,124,168,123,63,163,54,124,63,71,183,124,63,41,43,125,63,13,147,125,63,183,239,125,63,229,65,126,63,89,138,126,63,205,201,126,63,251,0,127,63,150,48,127,63,78,89,127,63,205,123,127,63,182,152,127,63,167,176,127,63,53,196,127,63,239,211,127,63,91,224,127,63,245,233,127,63,51,241,127,63,127,246,127,63,59,250,127,63,190,252,127,63,84,254,127,63,64,255,127,63,186,255,127,63,238,255,127,63,254,255,127,63,0,0,128,63,169,12,120,55,54,134,11,57,38,198,193,57,94,226,61,58,234,237,156,58,85,101,234,58,56,170,35,59,207,219,89,59,169,226,139,59,42,178,174,59,13,91,213,59,204,219,255,59,91,25,23,60,250,46,48,60,194,45,75,60,156,20,104,60,46,113,131,60,225,202,147,60,185,22,165,60,1,84,183,60,245,129,202,60,198,159,222,60,155,172,243,60,199,211,4,61,213,71,16,61,250,49,28,61,174,145,40,61,101,102,53,61,141,175,66,61,140,108,80,61,193,156,94,61,133,63,109,61,41,84,124,61,252,236,133,61,26,232,141,61,13,27,150,61,110,133,158,61,212,38,167,61,210,254,175,61,245,12,185,61,200,80,194,61,209,201,203,61,146,119,213,61,139,89,223,61,51,111,233,61,2,184,243,61,105,51,254,61,106,112,4,62,214,223,9,62,171,103,15,62,153,7,21,62,77,191,26,62,116,142,32,62,181,116,38,62,184,113,44,62,34,133,50,62,149,174,56,62,178,237,62,62,21,66,69,62,92,171,75,62,30,41,82,62,243,186,88,62,112,96,95,62,40,25,102,62,170,228,108,62,132,194,115,62,68,178,122,62,185,217,128,62,203,98,132,62,26,244,135,62,105,141,139,62,120,46,143,62,6,215,146,62,211,134,150,62,156,61,154,62,29,251,157,62,19,191,161,62,57,137,165,62,71,89,169,62,249,46,173,62,5,10,177,62,36,234,180,62,13,207,184,62,117,184,188,62,18,166,192,62,153,151,196,62,190,140,200,62,52,133,204,62,175,128,208,62,225,126,212,62,125,127,216,62,52,130,220,62,184,134,224,62,185,140,228,62,233,147,232,62,248,155,236,62,150,164,240,62,117,173,244,62,67,182,248,62,178,190,252,62,57,99,0,63,153,102,2,63,82,105,4,63,60,107,6,63,48,108,8,63,6,108,10,63,151,106,12,63,188,103,14,63,78,99,16,63,39,93,18,63,33,85,20,63,21,75,22,63,222,62,24,63,87,48,26,63,92,31,28,63,199,11,30,63,117,245,31,63,66,220,33,63,12,192,35,63,176,160,37,63,12,126,39,63,254,87,41,63,104,46,43,63,39,1,45,63,29,208,46,63,43,155,48,63,51,98,50,63,23,37,52,63,188,227,53,63,4,158,55,63,214,83,57,63,23,5,59,63,173,177,60,63,128,89,62,63,120,252,63,63,126,154,65,63,124,51,67,63,93,199,68,63,12,86,70,63,119,223,71,63,138,99,73,63,54,226,74,63,104,91,76,63,17,207,77,63,35,61,79,63,145,165,80,63,76,8,82,63,75,101,83,63,130,188,84,63,231,13,86,63,114,89,87,63,26,159,88,63,218,222,89,63,172,24,91,63,138,76,92,63,113,122,93,63,93,162,94,63,78,196,95,63,67,224,96,63,58,246,97,63,54,6,99,63,56,16,100,63,67,20,101,63,92,18,102,63,133,10,103,63,198,252,103,63,37,233,104,63,168,207,105,63,89,176,106,63,64,139,107,63,102,96,108,63,216,47,109,63,159,249,109,63,201,189,110,63,97,124,111,63,118,53,112,63,23,233,112,63,81,151,113,63,53,64,114,63,212,227,114,63,61,130,115,63,131,27,116,63,184,175,116,63,238,62,117,63,56,201,117,63,171,78,118,63,90,207,118,63,90,75,119,63,192,194,119,63,162,53,120,63,21,164,120,63,48,14,121,63,8,116,121,63,182,213,121,63,79,51,122,63,235,140,122,63,162,226,122,63,139,52,123,63,191,130,123,63,85,205,123,63,102,20,124,63,9,88,124,63,88,152,124,63,106,213,124,63,88,15,125,63,58,70,125,63,41,122,125,63,62,171,125,63,143,217,125,63,54,5,126,63,75,46,126,63,228,84,126,63,27,121,126,63,7,155,126,63,190,186,126,63,88,216,126,63,236,243,126,63,144,13,127,63,91,37,127,63,99,59,127,63,188,79,127,63,125,98,127,63,185,115,127,63,135,131,127,63,249,145,127,63,36,159,127,63,26,171,127,63,238,181,127,63,179,191,127,63,122,200,127,63,85,208,127,63,84,215,127,63,136,221,127,63,0,227,127,63,204,231,127,63,249,235,127,63,150,239,127,63,177,242,127,63,85,245,127,63,144,247,127,63,109,249,127,63,246,250,127,63,54,252,127,63,55,253,127,63,1,254,127,63,156,254,127,63,18,255,127,63,103,255,127,63,163,255,127,63,204,255,127,63,229,255,127,63,244,255,127,63,252,255,127,63,255,255,127,63,0,0,128,63,0,0,128,63,60,12,120,54,253,134,11,56,19,201,193,56,248,231,61,57,148,245,156,57,115,118,234,57,238,186,35,58,113,249,89,58,32,251,139,58,96,216,174,58,34,148,213,58,3,23,0,59,209,82,23,59,65,125,48,59,21,150,75,59,8,157,104,59,233,200,131,59,20,58,148,59,218,161,165,59,16,0,184,59,136,84,203,59,16,159,223,59,118,223,244,59,194,138,5,60,128,32,17,60,217,48,29,60,172,187,41,60,219,192,54,60,67,64,68,60,194,57,82,60,52,173,96,60,115,154,111,60,88,1,127,60,222,112,135,60,186,157,143,60,42,7,152,60,25,173,160,60,112,143,169,60,23,174,178,60,246,8,188,60,243,159,197,60,245,114,207,60,225,129,217,60,156,204,227,60,10,83,238,60,14,21,249,60,70,9,2,61,177,165,7,61,187,95,13,61,81,55,19,61,102,44,25,61,230,62,31,61,195,110,37,61,233,187,43,61,71,38,50,61,202,173,56,61,97,82,63,61,247,19,70,61,121,242,76,61,210,237,83,61,240,5,91,61,187,58,98,61,32,140,105,61,8,250,112,61,93,132,120,61,132,21,128,61,249,246,131,61,130,230,135,61,19,228,139,61,159,239,143,61,26,9,148,61,119,48,152,61,169,101,156,61,163,168,160,61,88,249,164,61,186,87,169,61,186,195,173,61,76,61,178,61,95,196,182,61,230,88,187,61,209,250,191,61,18,170,196,61,152,102,201,61,85,48,206,61,56,7,211,61,48,235,215,61,47,220,220,61,34,218,225,61,248,228,230,61,161,252,235,61,11,33,241,61,35,82,246,61,217,143,251,61,13,109,0,62,105,24,3,62,247,201,5,62,174,129,8,62,133,63,11,62,113,3,14,62,104,205,16,62,96,157,19,62,79,115,22,62,42,79,25,62,232,48,28,62,124,24,31,62,221,5,34,62,255,248,36,62,215,241,39,62,90,240,42,62,125,244,45,62,51,254,48,62,114,13,52,62,45,34,55,62,88,60,58,62,232,91,61,62,208,128,64,62,3,171,67,62,118,218,70,62,26,15,74,62,229,72,77,62,199,135,80,62,181,203,83,62,162,20,87,62,127,98,90,62,63,181,93,62,213,12,97,62,50,105,100,62,73,202,103,62,12,48,107,62,108,154,110,62,92,9,114,62,203,124,117,62,173,244,120,62,241,112,124,62,138,241,127,62,52,187,129,62,190,127,131,62,91,70,133,62,4,15,135,62,176,217,136,62,89,166,138,62,245,116,140,62,126,69,142,62,234,23,144,62,50,236,145,62,78,194,147,62,54,154,149,62,224,115,151,62,70,79,153,62,93,44,155,62,31,11,157,62,130,235,158,62,127,205,160,62,11,177,162,62,31,150,164,62,177,124,166,62,186,100,168,62,47,78,170,62,9,57,172,62,62,37,174,62,198,18,176,62,150,1,178,62,167,241,179,62,238,226,181,62,100,213,183,62,254,200,185,62,179,189,187,62,122,179,189,62,74,170,191,62,25,162,193,62,221,154,195,62,142,148,197,62,34,143,199,62,142,138,201,62,203,134,203,62,205,131,205,62,140,129,207,62,253,127,209,62,24,127,211,62,210,126,213,62,33,127,215,62,252,127,217,62,88,129,219,62,45,131,221,62,112,133,223,62,23,136,225,62,25,139,227,62,108,142,229,62,5,146,231,62,219,149,233,62,228,153,235,62,21,158,237,62,102,162,239,62,203,166,241,62,59,171,243,62,173,175,245,62,21,180,247,62,107,184,249,62,164,188,251,62,181,192,253,62,150,196,255,62,30,228,0,63,207,229,1,63,88,231,2,63,182,232,3,63,226,233,4,63,215,234,5,63,146,235,6,63,12,236,7,63,66,236,8,63,45,236,9,63,202,235,10,63,19,235,11,63,4,234,12,63,151,232,13,63,200,230,14,63,145,228,15,63,239,225,16,63,220,222,17,63,84,219,18,63,81,215,19,63,208,210,20,63,202,205,21,63,61,200,22,63,34,194,23,63,117,187,24,63,50,180,25,63,85,172,26,63,215,163,27,63,182,154,28,63,236,144,29,63,117,134,30,63,77,123,31,63,110,111,32,63,214,98,33,63,126,85,34,63,100,71,35,63,130,56,36,63,212,40,37,63,87,24,38,63,5,7,39,63,219,244,39,63,213,225,40,63,239,205,41,63,36,185,42,63,113,163,43,63,209,140,44,63,64,117,45,63,188,92,46,63,63,67,47,63,199,40,48,63,78,13,49,63,211,240,49,63,80,211,50,63,195,180,51,63,39,149,52,63,122,116,53,63,184,82,54,63,220,47,55,63,229,11,56,63,206,230,56,63,149,192,57,63,54,153,58,63,174,112,59,63,249,70,60,63,21,28,61,63,255,239,61,63,179,194,62,63,48,148,63,63,113,100,64,63,116,51,65,63,55,1,66,63,182,205,66,63,239,152,67,63,224,98,68,63,134,43,69,63,222,242,69,63,230,184,70,63,156,125,71,63,253,64,72,63,7,3,73,63,184,195,73,63,14,131,74,63,6,65,75,63,159,253,75,63,215,184,76,63,172,114,77,63,28,43,78,63,38,226,78,63,199,151,79,63,253,75,80,63,201,254,80,63,39,176,81,63,22,96,82,63,150,14,83,63,164,187,83,63,63,103,84,63,103,17,85,63,26,186,85,63,86,97,86,63,28,7,87,63,105,171,87,63,62,78,88,63,152,239,88,63,120,143,89,63,221,45,90,63,198,202,90,63,50,102,91,63,33,0,92,63,147,152,92,63,134,47,93,63,251,196,93,63,242,88,94,63,105,235,94,63,98,124,95,63,219,11,96,63,213,153,96,63,80,38,97,63,76,177,97,63,201,58,98,63,199,194,98,63,70,73,99,63,71,206,99,63,202,81,100,63,208,211,100,63,88,84,101,63,100,211,101,63,244,80,102,63,9,205,102,63,163,71,103,63,195,192,103,63,107,56,104,63,154,174,104,63,82,35,105,63,147,150,105,63,96,8,106,63,184,120,106,63,157,231,106,63,16,85,107,63,19,193,107,63,166,43,108,63,203,148,108,63,132,252,108,63,209,98,109,63,180,199,109,63,48,43,110,63,68,141,110,63,244,237,110,63,64,77,111,63,42,171,111,63,181,7,112,63,225,98,112,63,177,188,112,63,38,21,113,63,67,108,113,63,10,194,113,63,123,22,114,63,155,105,114,63,106,187,114,63,234,11,115,63,31,91,115,63,9,169,115,63,172,245,115,63,9,65,116,63,35,139,116,63,252,211,116,63,151,27,117,63,245,97,117,63,26,167,117,63,8,235,117,63,193,45,118,63,72,111,118,63,159,175,118,63,202,238,118,63,201,44,119,63,161,105,119,63,84,165,119,63,228,223,119,63,85,25,120,63,168,81,120,63,226,136,120,63,3,191,120,63,16,244,120,63,11,40,121,63,247,90,121,63,215,140,121,63,173,189,121,63,125,237,121,63,73,28,122,63,20,74,122,63,226,118,122,63,181,162,122,63,144,205,122,63,118,247,122,63,107,32,123,63,112,72,123,63,138,111,123,63,186,149,123,63,5,187,123,63,109,223,123,63,245,2,124,63,160,37,124,63,113,71,124,63,108,104,124,63,147,136,124,63,233,167,124,63,114,198,124,63,48,228,124,63,38,1,125,63,89,29,125,63,201,56,125,63,124,83,125,63,115,109,125,63,178,134,125,63,60,159,125,63,19,183,125,63,60,206,125,63,184,228,125,63,139,250,125,63,184,15,126,63,66,36,126,63,44,56,126,63,120,75,126,63,43,94,126,63,70,112,126,63,204,129,126,63,194,146,126,63,41,163,126,63,4,179,126,63,86,194,126,63,35,209,126,63,109,223,126,63,55,237,126,63,131,250,126,63,85,7,127,63,175,19,127,63,148,31,127,63,7,43,127,63,10,54,127,63,160,64,127,63,205,74,127,63,146,84,127,63,242,93,127,63,239,102,127,63,141,111,127,63,206,119,127,63,181,127,127,63,67,135,127,63,124,142,127,63,98,149,127,63,247,155,127,63,61,162,127,63,56,168,127,63,233,173,127,63,83,179,127,63,120,184,127,63,90,189,127,63,252,193,127,63,95,198,127,63,134,202,127,63,116,206,127,63,41,210,127,63,168,213,127,63,244,216,127,63,13,220,127,63,247,222,127,63,179,225,127,63,67,228,127,63,168,230,127,63,229,232,127,63,252,234,127,63,237,236,127,63,188,238,127,63,105,240,127,63,246,241,127,63,101,243,127,63,183,244,127,63,238,245,127,63,11,247,127,63,16,248,127,63,254,248,127,63,214,249,127,63,155,250,127,63,76,251,127,63,236,251,127,63,124,252,127,63,252,252,127,63,110,253,127,63,211,253,127,63,44,254,127,63,121,254,127,63,189,254,127,63,247,254,127,63,42,255,127,63,84,255,127,63,120,255,127,63,150,255,127,63,175,255,127,63,195,255,127,63,211,255,127,63,224,255,127,63,234,255,127,63,241,255,127,63,246,255,127,63,250,255,127,63,253,255,127,63,254,255,127,63,255,255,127,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,171,15,120,53,24,135,11,55,225,201,193,55,107,233,61,56,128,247,156,56,187,122,234,56,24,191,35,57,213,0,90,57,56,1,140,57,229,225,174,57,88,162,213,57,60,33,0,58,24,97,23,58,175,144,48,58,243,175,75,58,212,190,104,58,159,222,131,58,143,85,148,58,48,196,165,58,119,42,184,58,90,136,203,58,204,221,223,58,191,42,245,58,148,183,5,59,124,85,17,59,16,111,29,59,73,4,42,59,31,21,55,59,138,161,68,59,129,169,82,59,252,44,97,59,241,43,112,59,88,166,127,59,19,206,135,59,169,6,144,59,233,124,152,59,204,48,161,59,79,34,170,59,106,81,179,59,26,190,188,59,86,104,198,59,26,80,208,59,95,117,218,59,31,216,228,59,83,120,239,59,244,85,250,59,126,184,2,60,177,100,8,60,145,47,14,60,25,25,20,60,70,33,26,60,19,72,32,60,126,141,38,60,129,241,44,60,25,116,51,60,65,21,58,60,246,212,64,60,50,179,71,60,243,175,78,60,50,203,85,60,235,4,93,60,26,93,100,60,186,211,107,60,198,104,115,60,58,28,123,60,7,119,129,60,33,111,133,60,102,118,137,60,212,140,141,60,105,178,145,60,33,231,149,60,251,42,154,60,243,125,158,60,6,224,162,60,50,81,167,60,115,209,171,60,199,96,176,60,43,255,180,60,154,172,185,60,19,105,190,60,146,52,195,60,20,15,200,60,149,248,204,60,19,241,209,60,137,248,214,60,245,14,220,60,83,52,225,60,160,104,230,60,215,171,235,60,246,253,240,60,249,94,246,60,220,206,251,60,205,166,0,61,153,109,3,61,207,59,6,61,109,17,9,61,114,238,11,61,220,210,14,61,167,190,17,61,211,177,20,61,94,172,23,61,68,174,26,61,133,183,29,61,30,200,32,61,12,224,35,61,78,255,38,61,225,37,42,61,196,83,45,61,243,136,48,61,109,197,51,61,47,9,55,61,55,84,58,61,130,166,61,61,15,0,65,61,218,96,68,61,226,200,71,61,35,56,75,61,156,174,78,61,73,44,82,61,40,177,85,61,55,61,89,61,115,208,92,61,217,106,96,61,103,12,100,61,25,181,103,61,238,100,107,61,227,27,111,61,244,217,114,61,30,159,118,61,96,107,122,61,182,62,126,61,143,12,129,61,73,253,130,61,138,241,132,61,79,233,134,61,150,228,136,61,94,227,138,61,167,229,140,61,109,235,142,61,175,244,144,61,109,1,147,61,164,17,149,61,83,37,151,61,120,60,153,61,17,87,155,61,30,117,157,61,155,150,159,61,136,187,161,61,226,227,163,61,169,15,166,61,218,62,168,61,116,113,170,61,116,167,172,61,218,224,174,61,162,29,177,61,205,93,179,61,87,161,181,61,62,232,183,61,130,50,186,61,32,128,188,61,22,209,190,61,98,37,193,61,2,125,195,61,245,215,197,61,57,54,200,61,203,151,202,61,169,252,204,61,211,100,207,61,68,208,209,61,252,62,212,61,249,176,214,61,56,38,217,61,184,158,219,61,117,26,222,61,111,153,224,61,163,27,227,61,14,161,229,61,175,41,232,61,132,181,234,61,138,68,237,61,191,214,239,61,33,108,242,61,174,4,245,61,99,160,247,61,62,63,250,61,61,225,252,61,93,134,255,61,78,23,1,62,252,108,2,62,56,196,3,62,255,28,5,62,81,119,6,62,45,211,7,62,145,48,9,62,125,143,10,62,238,239,11,62,228,81,13,62,94,181,14,62,89,26,16,62,214,128,17,62,210,232,18,62,77,82,20,62,69,189,21,62,184,41,23,62,166,151,24,62,13,7,26,62,236,119,27,62,65,234,28,62,11,94,30,62,73,211,31,62,250,73,33,62,28,194,34,62,173,59,36,62,172,182,37,62,24,51,39,62,240,176,40,62,50,48,42,62,220,176,43,62,238,50,45,62,101,182,46,62,64,59,48,62,126,193,49,62,30,73,51,62,29,210,52,62,123,92,54,62,54,232,55,62,76,117,57,62,187,3,59,62,131,147,60,62,162,36,62,62,22,183,63,62,222,74,65,62,248,223,66,62,98,118,68,62,28,14,70,62,35,167,71,62,117,65,73,62,18,221,74,62,247,121,76,62,35,24,78,62,149,183,79,62,74,88,81,62,66,250,82,62,121,157,84,62,240,65,86,62,163,231,87,62,146,142,89,62,186,54,91,62,26,224,92,62,177,138,94,62,124,54,96,62,122,227,97,62,169,145,99,62,7,65,101,62,147,241,102,62,75,163,104,62,44,86,106,62,54,10,108,62,102,191,109,62,187,117,111,62,51,45,113,62,204,229,114,62,132,159,116,62,90,90,118,62,75,22,120,62,85,211,121,62,120,145,123,62,176,80,125,62,253,16,127,62,46,105,128,62,101,74,129,62,36,44,130,62,105,14,131,62,52,241,131,62,130,212,132,62,84,184,133,62,169,156,134,62,127,129,135,62,213,102,136,62,171,76,137,62,255,50,138,62,209,25,139,62,32,1,140,62,233,232,140,62,46,209,141,62,236,185,142,62,34,163,143,62,208,140,144,62,244,118,145,62,142,97,146,62,156,76,147,62,29,56,148,62,17,36,149,62,118,16,150,62,76,253,150,62,144,234,151,62,67,216,152,62,99,198,153,62,239,180,154,62,230,163,155,62,71,147,156,62,17,131,157,62,67,115,158,62,219,99,159,62,218,84,160,62,60,70,161,62,3,56,162,62,43,42,163,62,181,28,164,62,160,15,165,62,233,2,166,62,145,246,166,62,149,234,167,62,245,222,168,62,176,211,169,62,197,200,170,62,50,190,171,62,246,179,172,62,17,170,173,62,129,160,174,62,69,151,175,62,91,142,176,62,196,133,177,62,125,125,178,62,133,117,179,62,220,109,180,62,128,102,181,62,112,95,182,62,171,88,183,62,47,82,184,62,252,75,185,62,17,70,186,62,108,64,187,62,11,59,188,62,239,53,189,62,22,49,190,62,126,44,191,62,38,40,192,62,13,36,193,62,51,32,194,62,150,28,195,62,52,25,196,62,12,22,197,62,30,19,198,62,104,16,199,62,233,13,200,62,159,11,201,62,138,9,202,62,169,7,203,62,249,5,204,62,123,4,205,62,44,3,206,62,11,2,207,62,24,1,208,62,81,0,209,62,181,255,209,62,66,255,210,62,248,254,211,62,213,254,212,62,216,254,213,62,255,254,214,62,75,255,215,62,184,255,216,62,71,0,218,62,245,0,219,62,195,1,220,62,173,2,221,62,180,3,222,62,214,4,223,62,17,6,224,62,101,7,225,62,208,8,226,62,81,10,227,62,231,11,228,62,144,13,229,62,76,15,230,62,25,17,231,62,245,18,232,62,224,20,233,62,217,22,234,62,221,24,235,62,236,26,236,62,5,29,237,62,39,31,238,62,79,33,239,62,125,35,240,62,176,37,241,62,230,39,242,62,31,42,243,62,88,44,244,62,145,46,245,62,200,48,246,62,253,50,247,62,45,53,248,62,88,55,249,62,124,57,250,62,153,59,251,62,172,61,252,62,181,63,253,62,179,65,254,62,163,67,255,62,195,34,0,63,173,163,0,63,142,36,1,63,102,165,1,63,53,38,2,63,250,166,2,63,180,39,3,63,99,168,3,63,5,41,4,63,155,169,4,63,36,42,5,63,159,170,5,63,12,43,6,63,105,171,6,63,183,43,7,63,244,171,7,63,32,44,8,63,59,172,8,63,68,44,9,63,58,172,9,63,28,44,10,63,235,171,10,63,164,43,11,63,73,171,11,63,216,42,12,63,80,170,12,63,177,41,13,63,251,168,13,63,44,40,14,63,69,167,14,63,68,38,15,63,41,165,15,63,243,35,16,63,162,162,16,63,53,33,17,63,172,159,17,63,5,30,18,63,65,156,18,63,95,26,19,63,94,152,19,63,61,22,20,63,252,147,20,63,155,17,21,63,24,143,21,63,116,12,22,63,173,137,22,63,195,6,23,63,182,131,23,63,133,0,24,63,46,125,24,63,179,249,24,63,18,118,25,63,74,242,25,63,91,110,26,63,69,234,26,63,6,102,27,63,159,225,27,63,14,93,28,63,84,216,28,63,111,83,29,63,95,206,29,63,36,73,30,63,188,195,30,63,40,62,31,63,102,184,31,63,119,50,32,63,90,172,32,63,14,38,33,63,146,159,33,63,230,24,34,63,10,146,34,63,253,10,35,63,190,131,35,63,77,252,35,63,169,116,36,63,211,236,36,63,200,100,37,63,138,220,37,63,22,84,38,63,110,203,38,63,143,66,39,63,122,185,39,63,47,48,40,63,172,166,40,63,241,28,41,63,254,146,41,63,210,8,42,63,108,126,42,63,205,243,42,63,243,104,43,63,223,221,43,63,143,82,44,63,3,199,44,63,59,59,45,63,54,175,45,63,244,34,46,63,116,150,46,63,182,9,47,63,185,124,47,63,125,239,47,63,1,98,48,63,69,212,48,63,72,70,49,63,10,184,49,63,139,41,50,63,202,154,50,63,198,11,51,63,127,124,51,63,246,236,51,63,40,93,52,63,22,205,52,63,191,60,53,63,36,172,53,63,66,27,54,63,27,138,54,63,174,248,54,63,249,102,55,63,254,212,55,63,187,66,56,63,47,176,56,63,91,29,57,63,63,138,57,63,217,246,57,63,41,99,58,63,48,207,58,63,236,58,59,63,93,166,59,63,130,17,60,63,93,124,60,63,235,230,60,63,44,81,61,63,33,187,61,63,201,36,62,63,35,142,62,63,48,247,62,63,238,95,63,63,94,200,63,63,126,48,64,63,80,152,64,63,209,255,64,63,3,103,65,63,228,205,65,63,117,52,66,63,181,154,66,63,163,0,67,63,64,102,67,63,139,203,67,63,131,48,68,63,41,149,68,63,124,249,68,63,123,93,69,63,39,193,69,63,127,36,70,63,132,135,70,63,51,234,70,63,142,76,71,63,148,174,71,63,68,16,72,63,159,113,72,63,164,210,72,63,83,51,73,63,172,147,73,63,174,243,73,63,89,83,74,63,173,178,74,63,169,17,75,63,77,112,75,63,154,206,75,63,143,44,76,63,43,138,76,63,110,231,76,63,89,68,77,63,234,160,77,63,34,253,77,63,0,89,78,63,133,180,78,63,176,15,79,63,128,106,79,63,246,196,79,63,18,31,80,63,210,120,80,63,56,210,80,63,66,43,81,63,242,131,81,63,69,220,81,63,61,52,82,63,217,139,82,63,24,227,82,63,252,57,83,63,131,144,83,63,174,230,83,63,123,60,84,63,236,145,84,63,0,231,84,63,183,59,85,63,16,144,85,63,12,228,85,63,170,55,86,63,235,138,86,63,206,221,86,63,83,48,87,63,121,130,87,63,66,212,87,63,172,37,88,63,184,118,88,63,101,199,88,63,180,23,89,63,164,103,89,63,53,183,89,63,104,6,90,63,59,85,90,63,175,163,90,63,197,241,90,63,123,63,91,63,210,140,91,63,201,217,91,63,97,38,92,63,154,114,92,63,115,190,92,63,237,9,93,63,7,85,93,63,194,159,93,63,29,234,93,63,24,52,94,63,179,125,94,63,239,198,94,63,203,15,95,63,72,88,95,63,100,160,95,63,33,232,95,63,126,47,96,63,123,118,96,63,24,189,96,63,85,3,97,63,51,73,97,63,177,142,97,63,207,211,97,63,141,24,98,63,236,92,98,63,235,160,98,63,138,228,98,63,202,39,99,63,170,106,99,63,42,173,99,63], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE);
/* memory initializer */ allocate([75,239,99,63,13,49,100,63,111,114,100,63,114,179,100,63,21,244,100,63,90,52,101,63,63,116,101,63,197,179,101,63,236,242,101,63,180,49,102,63,29,112,102,63,39,174,102,63,211,235,102,63,32,41,103,63,15,102,103,63,159,162,103,63,209,222,103,63,164,26,104,63,26,86,104,63,49,145,104,63,235,203,104,63,71,6,105,63,69,64,105,63,230,121,105,63,42,179,105,63,16,236,105,63,153,36,106,63,197,92,106,63,148,148,106,63,7,204,106,63,29,3,107,63,214,57,107,63,52,112,107,63,53,166,107,63,218,219,107,63,36,17,108,63,18,70,108,63,164,122,108,63,220,174,108,63,184,226,108,63,57,22,109,63,96,73,109,63,44,124,109,63,157,174,109,63,181,224,109,63,115,18,110,63,214,67,110,63,225,116,110,63,146,165,110,63,233,213,110,63,232,5,111,63,142,53,111,63,219,100,111,63,209,147,111,63,110,194,111,63,179,240,111,63,160,30,112,63,54,76,112,63,117,121,112,63,93,166,112,63,239,210,112,63,41,255,112,63,14,43,113,63,156,86,113,63,213,129,113,63,184,172,113,63,70,215,113,63,127,1,114,63,99,43,114,63,243,84,114,63,46,126,114,63,21,167,114,63,169,207,114,63,233,247,114,63,214,31,115,63,113,71,115,63,184,110,115,63,173,149,115,63,80,188,115,63,162,226,115,63,161,8,116,63,80,46,116,63,174,83,116,63,187,120,116,63,119,157,116,63,228,193,116,63,1,230,116,63,206,9,117,63,76,45,117,63,123,80,117,63,92,115,117,63,238,149,117,63,51,184,117,63,42,218,117,63,211,251,117,63,48,29,118,63,64,62,118,63,3,95,118,63,122,127,118,63,166,159,118,63,134,191,118,63,27,223,118,63,101,254,118,63,101,29,119,63,27,60,119,63,135,90,119,63,169,120,119,63,131,150,119,63,19,180,119,63,91,209,119,63,91,238,119,63,20,11,120,63,132,39,120,63,174,67,120,63,145,95,120,63,46,123,120,63,132,150,120,63,149,177,120,63,96,204,120,63,231,230,120,63,41,1,121,63,38,27,121,63,223,52,121,63,85,78,121,63,136,103,121,63,120,128,121,63,37,153,121,63,144,177,121,63,185,201,121,63,161,225,121,63,72,249,121,63,174,16,122,63,212,39,122,63,185,62,122,63,96,85,122,63,198,107,122,63,238,129,122,63,216,151,122,63,131,173,122,63,241,194,122,63,33,216,122,63,20,237,122,63,202,1,123,63,68,22,123,63,130,42,123,63,133,62,123,63,77,82,123,63,217,101,123,63,43,121,123,63,68,140,123,63,34,159,123,63,200,177,123,63,52,196,123,63,104,214,123,63,99,232,123,63,39,250,123,63,180,11,124,63,9,29,124,63,40,46,124,63,17,63,124,63,196,79,124,63,65,96,124,63,137,112,124,63,156,128,124,63,124,144,124,63,39,160,124,63,158,175,124,63,226,190,124,63,244,205,124,63,211,220,124,63,128,235,124,63,251,249,124,63,69,8,125,63,94,22,125,63,71,36,125,63,255,49,125,63,136,63,125,63,225,76,125,63,11,90,125,63,7,103,125,63,212,115,125,63,115,128,125,63,229,140,125,63,42,153,125,63,66,165,125,63,46,177,125,63,238,188,125,63,130,200,125,63,235,211,125,63,41,223,125,63,61,234,125,63,38,245,125,63,230,255,125,63,124,10,126,63,234,20,126,63,47,31,126,63,75,41,126,63,64,51,126,63,13,61,126,63,180,70,126,63,51,80,126,63,140,89,126,63,191,98,126,63,205,107,126,63,181,116,126,63,120,125,126,63,23,134,126,63,146,142,126,63,233,150,126,63,28,159,126,63,44,167,126,63,26,175,126,63,229,182,126,63,142,190,126,63,22,198,126,63,124,205,126,63,194,212,126,63,231,219,126,63,235,226,126,63,208,233,126,63,149,240,126,63,59,247,126,63,195,253,126,63,44,4,127,63,118,10,127,63,163,16,127,63,179,22,127,63,165,28,127,63,123,34,127,63,52,40,127,63,210,45,127,63,83,51,127,63,186,56,127,63,5,62,127,63,53,67,127,63,75,72,127,63,72,77,127,63,42,82,127,63,243,86,127,63,163,91,127,63,58,96,127,63,185,100,127,63,32,105,127,63,111,109,127,63,166,113,127,63,199,117,127,63,208,121,127,63,196,125,127,63,161,129,127,63,104,133,127,63,25,137,127,63,182,140,127,63,61,144,127,63,176,147,127,63,14,151,127,63,89,154,127,63,143,157,127,63,179,160,127,63,195,163,127,63,192,166,127,63,171,169,127,63,132,172,127,63,74,175,127,63,255,177,127,63,163,180,127,63,53,183,127,63,183,185,127,63,40,188,127,63,137,190,127,63,217,192,127,63,26,195,127,63,76,197,127,63,111,199,127,63,130,201,127,63,135,203,127,63,126,205,127,63,102,207,127,63,65,209,127,63,14,211,127,63,205,212,127,63,128,214,127,63,38,216,127,63,191,217,127,63,76,219,127,63,204,220,127,63,65,222,127,63,170,223,127,63,8,225,127,63,91,226,127,63,163,227,127,63,224,228,127,63,19,230,127,63,59,231,127,63,90,232,127,63,110,233,127,63,122,234,127,63,124,235,127,63,116,236,127,63,100,237,127,63,75,238,127,63,42,239,127,63,1,240,127,63,207,240,127,63,149,241,127,63,84,242,127,63,12,243,127,63,188,243,127,63,101,244,127,63,7,245,127,63,162,245,127,63,55,246,127,63,198,246,127,63,78,247,127,63,209,247,127,63,77,248,127,63,196,248,127,63,54,249,127,63,162,249,127,63,9,250,127,63,108,250,127,63,201,250,127,63,34,251,127,63,118,251,127,63,198,251,127,63,18,252,127,63,89,252,127,63,157,252,127,63,221,252,127,63,26,253,127,63,83,253,127,63,136,253,127,63,187,253,127,63,234,253,127,63,22,254,127,63,64,254,127,63,103,254,127,63,139,254,127,63,173,254,127,63,204,254,127,63,234,254,127,63,5,255,127,63,30,255,127,63,53,255,127,63,74,255,127,63,94,255,127,63,112,255,127,63,128,255,127,63,143,255,127,63,157,255,127,63,169,255,127,63,180,255,127,63,191,255,127,63,200,255,127,63,208,255,127,63,215,255,127,63,221,255,127,63,227,255,127,63,232,255,127,63,236,255,127,63,239,255,127,63,243,255,127,63,245,255,127,63,248,255,127,63,249,255,127,63,251,255,127,63,252,255,127,63,253,255,127,63,254,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,204,8,120,52,171,134,11,54,79,202,193,54,190,233,61,55,238,247,156,55,192,123,234,55,43,192,35,56,161,2,90,56,189,2,140,56,76,228,174,56,227,165,213,56,199,35,0,57,168,100,23,57,134,149,48,57,104,182,75,57,64,199,104,57,7,228,131,57,105,92,148,57,191,204,165,57,6,53,184,57,65,149,203,57,105,237,223,57,120,61,245,57,184,194,5,58,166,98,17,58,134,126,29,58,81,22,42,58,9,42,55,58,172,185,68,58,54,197,82,58,165,76,97,58,250,79,112,58,47,207,127,58,34,229,135,58,154,32,144,58,255,153,152,58,80,81,161,58,139,70,170,58,174,121,179,58,186,234,188,58,171,153,198,58,129,134,208,58,58,177,218,58,212,25,229,58,79,192,239,58,167,164,250,58,109,227,2,59,117,147,8,59,105,98,14,59,73,80,20,59,19,93,26,59,199,136,32,59,100,211,38,59,232,60,45,59,83,197,51,59,164,108,58,59,218,50,65,59,243,23,72,59,239,27,79,59,204,62,86,59,138,128,93,59,38,225,100,59,161,96,108,59,249,254,115,59,45,188,123,59,29,204,129,59,145,201,133,59,113,214,137,59,188,242,141,59,113,30,146,59,145,89,150,59,26,164,154,59,12,254,158,59,102,103,163,59,40,224,167,59,80,104,172,59,222,255,176,59,209,166,181,59,40,93,186,59,228,34,191,59,2,248,195,59,131,220,200,59,101,208,205,59,168,211,210,59,74,230,215,59,76,8,221,59,172,57,226,59,105,122,231,59,131,202,236,59,249,41,242,59,202,152,247,59,245,22,253,59,60,82,1,60,170,32,4,60,196,246,6,60,137,212,9,60,249,185,12,60,19,167,15,60,216,155,18,60,69,152,21,60,92,156,24,60,26,168,27,60,129,187,30,60,143,214,33,60,69,249,36,60,160,35,40,60,162,85,43,60,73,143,46,60,149,208,49,60,133,25,53,60,26,106,56,60,81,194,59,60,44,34,63,60,168,137,66,60,199,248,69,60,134,111,73,60,230,237,76,60,231,115,80,60,134,1,84,60,197,150,87,60,162,51,91,60,28,216,94,60,52,132,98,60,232,55,102,60,56,243,105,60,35,182,109,60,170,128,113,60,202,82,117,60,131,44,121,60,214,13,125,60,96,123,128,60,161,115,130,60,174,111,132,60,134,111,134,60,40,115,136,60,149,122,138,60,205,133,140,60,206,148,142,60,152,167,144,60,44,190,146,60,136,216,148,60,173,246,150,60,154,24,153,60,78,62,155,60,202,103,157,60,13,149,159,60,23,198,161,60,231,250,163,60,125,51,166,60,217,111,168,60,249,175,170,60,223,243,172,60,137,59,175,60,247,134,177,60,40,214,179,60,29,41,182,60,213,127,184,60,80,218,186,60,140,56,189,60,138,154,191,60,74,0,194,60,202,105,196,60,11,215,198,60,12,72,201,60,205,188,203,60,77,53,206,60,140,177,208,60,137,49,211,60,69,181,213,60,189,60,216,60,243,199,218,60,230,86,221,60,149,233,223,60,0,128,226,60,39,26,229,60,8,184,231,60,164,89,234,60,250,254,236,60,9,168,239,60,210,84,242,60,83,5,245,60,141,185,247,60,126,113,250,60,39,45,253,60,134,236,255,60,206,87,1,61,52,187,2,61,117,32,4,61,144,135,5,61,133,240,6,61,84,91,8,61,253,199,9,61,128,54,11,61,219,166,12,61,16,25,14,61,29,141,15,61,3,3,17,61,193,122,18,61,87,244,19,61,197,111,21,61,10,237,22,61,39,108,24,61,26,237,25,61,228,111,27,61,132,244,28,61,251,122,30,61,71,3,32,61,105,141,33,61,96,25,35,61,45,167,36,61,206,54,38,61,67,200,39,61,141,91,41,61,171,240,42,61,156,135,44,61,96,32,46,61,248,186,47,61,99,87,49,61,160,245,50,61,175,149,52,61,144,55,54,61,67,219,55,61,199,128,57,61,28,40,59,61,65,209,60,61,56,124,62,61,254,40,64,61,148,215,65,61,250,135,67,61,47,58,69,61,51,238,70,61,5,164,72,61,166,91,74,61,20,21,76,61,80,208,77,61,90,141,79,61,49,76,81,61,212,12,83,61,68,207,84,61,128,147,86,61,135,89,88,61,90,33,90,61,248,234,91,61,97,182,93,61,148,131,95,61,145,82,97,61,88,35,99,61,232,245,100,61,65,202,102,61,100,160,104,61,78,120,106,61,1,82,108,61,123,45,110,61,188,10,112,61,197,233,113,61,148,202,115,61,41,173,117,61,133,145,119,61,166,119,121,61,140,95,123,61,55,73,125,61,166,52,127,61,237,144,128,61,105,136,129,61,198,128,130,61,5,122,131,61,37,116,132,61,39,111,133,61,9,107,134,61,204,103,135,61,112,101,136,61,244,99,137,61,88,99,138,61,157,99,139,61,193,100,140,61,196,102,141,61,167,105,142,61,106,109,143,61,11,114,144,61,139,119,145,61,234,125,146,61,40,133,147,61,67,141,148,61,61,150,149,61,20,160,150,61,201,170,151,61,92,182,152,61,203,194,153,61,24,208,154,61,66,222,155,61,72,237,156,61,42,253,157,61,233,13,159,61,132,31,160,61,250,49,161,61,76,69,162,61,122,89,163,61,130,110,164,61,101,132,165,61,35,155,166,61,188,178,167,61,47,203,168,61,124,228,169,61,162,254,170,61,163,25,172,61,124,53,173,61,47,82,174,61,187,111,175,61,31,142,176,61,92,173,177,61,113,205,178,61,94,238,179,61,35,16,181,61,192,50,182,61,52,86,183,61,127,122,184,61,160,159,185,61,153,197,186,61,104,236,187,61,13,20,189,61,136,60,190,61,217,101,191,61,255,143,192,61,250,186,193,61,202,230,194,61,111,19,196,61,233,64,197,61,55,111,198,61,89,158,199,61,78,206,200,61,23,255,201,61,179,48,203,61,35,99,204,61,101,150,205,61,121,202,206,61,96,255,207,61,25,53,209,61,164,107,210,61,0,163,211,61,45,219,212,61,44,20,214,61,251,77,215,61,154,136,216,61,10,196,217,61,74,0,219,61,89,61,220,61,56,123,221,61,230,185,222,61,99,249,223,61,174,57,225,61,200,122,226,61,176,188,227,61,102,255,228,61,233,66,230,61,58,135,231,61,88,204,232,61,66,18,234,61,249,88,235,61,124,160,236,61,203,232,237,61,230,49,239,61,204,123,240,61,125,198,241,61,249,17,243,61,63,94,244,61,79,171,245,61,42,249,246,61,206,71,248,61,60,151,249,61,114,231,250,61,114,56,252,61,58,138,253,61,202,220,254,61,17,24,0,62,33,194,0,62,149,108,1,62,108,23,2,62,166,194,2,62,68,110,3,62,69,26,4,62,168,198,4,62,111,115,5,62,152,32,6,62,35,206,6,62,17,124,7,62,98,42,8,62,20,217,8,62,40,136,9,62,157,55,10,62,117,231,10,62,173,151,11,62,71,72,12,62,66,249,12,62,158,170,13,62,91,92,14,62,120,14,15,62,246,192,15,62,213,115,16,62,19,39,17,62,177,218,17,62,175,142,18,62,13,67,19,62,202,247,19,62,231,172,20,62,99,98,21,62,62,24,22,62,120,206,22,62,16,133,23,62,7,60,24,62,92,243,24,62,16,171,25,62,33,99,26,62,145,27,27,62,94,212,27,62,137,141,28,62,17,71,29,62,246,0,30,62,56,187,30,62,215,117,31,62,211,48,32,62,43,236,32,62,224,167,33,62,241,99,34,62,93,32,35,62,38,221,35,62,74,154,36,62,202,87,37,62,165,21,38,62,219,211,38,62,108,146,39,62,88,81,40,62,159,16,41,62,64,208,41,62,59,144,42,62,144,80,43,62,63,17,44,62,72,210,44,62,170,147,45,62,102,85,46,62,122,23,47,62,232,217,47,62,175,156,48,62,206,95,49,62,69,35,50,62,21,231,50,62,61,171,51,62,189,111,52,62,148,52,53,62,195,249,53,62,73,191,54,62,38,133,55,62,91,75,56,62,230,17,57,62,199,216,57,62,255,159,58,62,141,103,59,62,113,47,60,62,171,247,60,62,59,192,61,62,31,137,62,62,89,82,63,62,232,27,64,62,204,229,64,62,5,176,65,62,146,122,66,62,115,69,67,62,168,16,68,62,49,220,68,62,14,168,69,62,62,116,70,62,194,64,71,62,152,13,72,62,193,218,72,62,61,168,73,62,12,118,74,62,44,68,75,62,159,18,76,62,100,225,76,62,122,176,77,62,225,127,78,62,154,79,79,62,164,31,80,62,255,239,80,62,170,192,81,62,166,145,82,62,242,98,83,62,141,52,84,62,121,6,85,62,180,216,85,62,63,171,86,62,25,126,87,62,65,81,88,62,185,36,89,62,126,248,89,62,147,204,90,62,245,160,91,62,165,117,92,62,163,74,93,62,238,31,94,62,135,245,94,62,109,203,95,62,159,161,96,62,30,120,97,62,233,78,98,62,1,38,99,62,100,253,99,62,19,213,100,62,14,173,101,62,84,133,102,62,229,93,103,62,193,54,104,62,231,15,105,62,88,233,105,62,19,195,106,62,24,157,107,62,103,119,108,62,255,81,109,62,224,44,110,62,11,8,111,62,126,227,111,62,58,191,112,62,62,155,113,62,139,119,114,62,31,84,115,62,251,48,116,62,31,14,117,62,138,235,117,62,59,201,118,62,52,167,119,62,115,133,120,62,248,99,121,62,196,66,122,62,213,33,123,62,44,1,124,62,200,224,124,62,170,192,125,62,208,160,126,62,59,129,127,62,245,48,128,62,111,161,128,62,11,18,129,62,201,130,129,62,168,243,129,62,169,100,130,62,204,213,130,62,15,71,131,62,117,184,131,62,251,41,132,62,162,155,132,62,107,13,133,62,84,127,133,62,93,241,133,62,136,99,134,62,210,213,134,62,61,72,135,62,200,186,135,62,116,45,136,62,63,160,136,62,42,19,137,62,52,134,137,62,94,249,137,62,168,108,138,62,17,224,138,62,153,83,139,62,64,199,139,62,6,59,140,62,235,174,140,62,239,34,141,62,17,151,141,62,82,11,142,62,177,127,142,62,46,244,142,62,201,104,143,62,130,221,143,62,89,82,144,62,78,199,144,62,96,60,145,62,143,177,145,62,220,38,146,62,70,156,146,62,205,17,147,62,113,135,147,62,50,253,147,62,16,115,148,62,9,233,148,62,32,95,149,62,82,213,149,62,161,75,150,62,12,194,150,62,146,56,151,62,53,175,151,62,243,37,152,62,204,156,152,62,193,19,153,62,209,138,153,62,252,1,154,62,66,121,154,62,163,240,154,62,31,104,155,62,181,223,155,62,101,87,156,62,48,207,156,62,21,71,157,62,20,191,157,62,45,55,158,62,96,175,158,62,172,39,159,62,18,160,159,62,145,24,160,62,41,145,160,62,218,9,161,62,165,130,161,62,136,251,161,62,132,116,162,62,152,237,162,62,197,102,163,62,10,224,163,62,103,89,164,62,220,210,164,62,105,76,165,62,14,198,165,62,202,63,166,62,158,185,166,62,137,51,167,62,139,173,167,62,164,39,168,62,213,161,168,62,27,28,169,62,121,150,169,62,237,16,170,62,119,139,170,62,24,6,171,62,206,128,171,62,155,251,171,62,125,118,172,62,117,241,172,62,130,108,173,62,165,231,173,62,221,98,174,62,42,222,174,62,140,89,175,62,2,213,175,62,142,80,176,62,46,204,176,62,226,71,177,62,170,195,177,62,135,63,178,62,119,187,178,62,124,55,179,62,148,179,179,62,191,47,180,62,254,171,180,62,80,40,181,62,181,164,181,62,45,33,182,62,184,157,182,62,85,26,183,62,5,151,183,62,199,19,184,62,156,144,184,62,130,13,185,62,123,138,185,62,133,7,186,62,161,132,186,62,206,1,187,62,13,127,187,62,93,252,187,62,190,121,188,62,48,247,188,62,178,116,189,62,70,242,189,62,233,111,190,62,157,237,190,62,98,107,191,62,54,233,191,62,26,103,192,62,14,229,192,62,17,99,193,62,36,225,193,62,70,95,194,62,119,221,194,62,184,91,195,62,7,218,195,62,100,88,196,62,209,214,196,62,75,85,197,62,212,211,197,62,107,82,198,62,16,209,198,62,195,79,199,62,132,206,199,62,82,77,200,62,45,204,200,62,21,75,201,62,11,202,201,62,13,73,202,62,29,200,202,62,56,71,203,62,97,198,203,62,149,69,204,62,214,196,204,62,34,68,205,62,123,195,205,62,223,66,206,62,79,194,206,62,202,65,207,62,81,193,207,62,226,64,208,62,127,192,208,62,38,64,209,62,216,191,209,62,148,63,210,62,91,191,210,62,44,63,211,62,7,191,211,62,235,62,212,62,218,190,212,62,210,62,213,62,211,190,213,62,222,62,214,62,242,190,214,62,15,63,215,62,53,191,215,62,99,63,216,62,154,191,216,62,217,63,217,62,32,192,217,62,112,64,218,62,199,192,218,62,38,65,219,62,140,193,219,62,250,65,220,62,112,194,220,62,236,66,221,62,112,195,221,62,250,67,222,62,139,196,222,62,34,69,223,62,192,197,223,62,100,70,224,62,14,199,224,62,189,71,225,62,115,200,225,62,46,73,226,62,239,201,226,62,181,74,227,62,127,203,227,62,79,76,228,62,36,205,228,62,253,77,229,62,219,206,229,62,190,79,230,62,164,208,230,62,142,81,231,62,125,210,231,62,111,83,232,62,100,212,232,62,93,85,233,62,89,214,233,62,89,87,234,62,91,216,234,62,96,89,235,62,104,218,235,62,114,91,236,62,126,220,236,62,141,93,237,62,158,222,237,62,176,95,238,62,196,224,238,62,218,97,239,62,241,226,239,62,10,100,240,62,35,229,240,62,62,102,241,62,89,231,241,62,116,104,242,62,145,233,242,62,173,106,243,62,202,235,243,62,230,108,244,62,3,238,244,62,31,111,245,62,59,240,245,62,86,113,246,62,112,242,246,62,137,115,247,62,161,244,247,62,184,117,248,62,206,246,248,62,226,119,249,62,244,248,249,62,4,122,250,62,18,251,250,62,30,124,251,62,40,253,251,62,47,126,252,62,52,255,252,62,54,128,253,62,52,1,254,62,48,130,254,62,40,3,255,62,29,132,255,62,135,2,0,63,254,66,0,63,115,131,0,63,230,195,0,63,86,4,1,63,197,68,1,63,49,133,1,63,155,197,1,63,3,6,2,63,103,70,2,63,202,134,2,63,42,199,2,63,135,7,3,63,225,71,3,63,56,136,3,63,141,200,3,63,222,8,4,63,44,73,4,63,119,137,4,63,191,201,4,63,3,10,5,63,68,74,5,63,130,138,5,63,188,202,5,63,242,10,6,63,36,75,6,63,83,139,6,63,126,203,6,63,165,11,7,63,199,75,7,63,230,139,7,63,1,204,7,63,23,12,8,63,41,76,8,63,54,140,8,63,63,204,8,63,67,12,9,63,67,76,9,63,62,140,9,63,52,204,9,63,37,12,10,63,18,76,10,63,249,139,10,63,219,203,10,63,184,11,11,63,144,75,11,63,98,139,11,63,47,203,11,63,246,10,12,63,184,74,12,63,116,138,12,63,43,202,12,63,219,9,13,63,134,73,13,63,43,137,13,63,202,200,13,63,98,8,14,63,245,71,14,63,129,135,14,63,7,199,14,63,135,6,15,63,0,70,15,63,114,133,15,63,222,196,15,63,67,4,16,63,161,67,16,63,249,130,16,63,73,194,16,63,147,1,17,63,213,64,17,63,17,128,17,63,69,191,17,63,114,254,17,63,151,61,18,63,181,124,18,63,203,187,18,63,218,250,18,63,225,57,19,63,225,120,19,63,216,183,19,63,200,246,19,63,176,53,20,63,143,116,20,63,103,179,20,63,54,242,20,63,253,48,21,63,188,111,21,63,114,174,21,63,32,237,21,63,197,43,22,63,98,106,22,63,246,168,22,63,129,231,22,63,3,38,23,63,125,100,23,63,237,162,23,63,84,225,23,63,178,31,24,63,7,94,24,63,83,156,24,63,149,218,24,63,206,24,25,63,253,86,25,63,35,149,25,63,63,211,25,63,82,17,26,63,90,79,26,63,89,141,26,63,78,203,26,63,57,9,27,63,25,71,27,63,240,132,27,63,188,194,27,63,126,0,28,63,54,62,28,63,227,123,28,63,134,185,28,63,30,247,28,63,172,52,29,63,47,114,29,63,167,175,29,63,20,237,29,63,118,42,30,63,206,103,30,63,26,165,30,63,91,226,30,63,145,31,31,63,188,92,31,63,219,153,31,63,239,214,31,63,247,19,32,63,244,80,32,63,230,141,32,63,203,202,32,63,165,7,33,63,115,68,33,63,53,129,33,63,235,189,33,63,150,250,33,63,52,55,34,63,198,115,34,63,75,176,34,63,197,236,34,63,50,41,35,63,146,101,35,63,230,161,35,63,46,222,35,63,105,26,36,63,151,86,36,63,185,146,36,63,205,206,36,63,213,10,37,63,208,70,37,63,190,130,37,63,158,190,37,63,114,250,37,63,56,54,38,63,241,113,38,63,157,173,38,63,59,233,38,63,204,36,39,63,79,96,39,63,197,155,39,63,45,215,39,63,135,18,40,63,211,77,40,63,18,137,40,63,66,196,40,63,101,255,40,63,121,58,41,63,128,117,41,63,120,176,41,63,98,235,41,63,62,38,42,63,11,97,42,63,202,155,42,63,122,214,42,63,28,17,43,63,175,75,43,63,52,134,43,63,170,192,43,63,16,251,43,63,105,53,44,63,178,111,44,63,236,169,44,63,23,228,44,63,51,30,45,63,64,88,45,63,61,146,45,63,43,204,45,63,10,6,46,63,218,63,46,63,154,121,46,63,74,179,46,63,235,236,46,63,124,38,47,63,254,95,47,63,112,153,47,63,210,210,47,63,36,12,48,63,102,69,48,63,152,126,48,63,186,183,48,63,204,240,48,63,205,41,49,63,191,98,49,63,160,155,49,63,113,212,49,63,49,13,50,63,225,69,50,63,128,126,50,63,15,183,50,63,141,239,50,63,251,39,51,63,87,96,51,63,163,152,51,63,222,208,51,63,8,9,52,63,34,65,52,63,42,121,52,63,33,177,52,63,7,233,52,63,219,32,53,63,159,88,53,63,81,144,53,63,242,199,53,63,129,255,53,63,255,54,54,63,108,110,54,63,198,165,54,63,16,221,54,63,71,20,55,63,109,75,55,63,129,130,55,63,131,185,55,63,116,240,55,63,82,39,56,63,30,94,56,63,217,148,56,63,129,203,56,63,23,2,57,63,155,56,57,63,13,111,57,63,108,165,57,63,185,219,57,63,244,17,58,63,28,72,58,63,50,126,58,63,53,180,58,63,38,234,58,63,4,32,59,63,207,85,59,63,135,139,59,63,45,193,59,63,192,246,59,63,64,44,60,63,173,97,60,63,7,151,60,63,78,204,60,63,130,1,61,63,163,54,61,63,177,107,61,63,171,160,61,63,146,213,61,63,102,10,62,63,39,63,62,63,212,115,62,63,110,168,62,63,244,220,62,63,103,17,63,63,198,69,63,63,17,122,63,63,73,174,63,63,109,226,63,63,126,22,64,63,122,74,64,63,99,126,64,63,56,178,64,63,248,229,64,63,165,25,65,63,62,77,65,63,195,128,65,63,52,180,65,63,144,231,65,63,216,26,66,63,13,78,66,63,44,129,66,63,56,180,66,63,47,231,66,63,18,26,67,63,224,76,67,63,154,127,67,63,64,178,67,63,208,228,67,63,77,23,68,63,180,73,68,63,7,124,68,63,69,174,68,63,111,224,68,63,131,18,69,63,131,68,69,63,110,118,69,63,68,168,69,63,5,218,69,63,177,11,70,63,72,61,70,63,202,110,70,63,55,160,70,63,143,209,70,63,210,2,71,63,255,51,71,63,23,101,71,63,26,150,71,63,8,199,71,63,224,247,71,63,163,40,72,63,81,89,72,63,233,137,72,63,107,186,72,63,216,234,72,63,48,27,73,63,114,75,73,63,158,123,73,63,181,171,73,63,181,219,73,63,161,11,74,63,118,59,74,63,54,107,74,63,224,154,74,63,116,202,74,63,242,249,74,63,90,41,75,63,173,88,75,63,233,135,75,63,15,183,75,63,32,230,75,63,26,21,76,63,254,67,76,63,204,114,76,63,132,161,76,63,38,208,76,63,177,254,76,63,38,45,77,63,133,91,77,63,206,137,77,63,0,184,77,63,28,230,77,63,34,20,78,63,17,66,78,63,234,111,78,63,172,157,78,63,88,203,78,63,238,248,78,63,108,38,79,63,213,83,79,63,38,129,79,63,97,174,79,63,134,219,79,63,147,8,80,63,138,53,80,63,107,98,80,63,52,143,80,63,231,187,80,63,131,232,80,63,8,21,81,63,119,65,81,63,206,109,81,63,15,154,81,63,57,198,81,63,76,242,81,63,71,30,82,63,44,74,82,63,250,117,82,63,177,161,82,63,81,205,82,63,218,248,82,63,76,36,83,63,166,79,83,63,234,122,83,63,22,166,83,63,44,209,83,63,42,252,83,63,17,39,84,63,224,81,84,63,153,124,84,63,58,167,84,63,196,209,84,63,54,252,84,63,146,38,85,63,214,80,85,63,2,123,85,63,24,165,85,63,22,207,85,63,252,248,85,63,204,34,86,63,131,76,86,63,36,118,86,63,172,159,86,63,30,201,86,63,120,242,86,63,186,27,87,63,229,68,87,63,248,109,87,63,244,150,87,63,216,191,87,63,165,232,87,63,90,17,88,63,248,57,88,63,126,98,88,63,236,138,88,63,67,179,88,63,130,219,88,63,169,3,89,63,185,43,89,63,177,83,89,63,145,123,89,63,90,163,89,63,11,203,89,63,164,242,89,63,37,26,90,63,143,65,90,63,225,104,90,63,27,144,90,63,62,183,90,63,72,222,90,63,59,5,91,63,22,44,91,63,217,82,91,63,133,121,91,63,24,160,91,63,148,198,91,63,248,236,91,63,68,19,92,63,120,57,92,63,149,95,92,63,153,133,92,63,134,171,92,63,91,209,92,63,24,247,92,63,189,28,93,63,74,66,93,63,191,103,93,63,28,141,93,63,98,178,93,63,143,215,93,63,165,252,93,63,162,33,94,63,136,70,94,63,86,107,94,63,11,144,94,63,169,180,94,63,47,217,94,63,157,253,94,63,243,33,95,63,49,70,95,63,88,106,95,63,102,142,95,63,92,178,95,63,59,214,95,63,1,250,95,63,175,29,96,63,70,65,96,63,196,100,96,63,43,136,96,63,122,171,96,63,176,206,96,63,207,241,96,63,214,20,97,63,197,55,97,63,155,90,97,63,90,125,97,63,1,160,97,63,144,194,97,63,8,229,97,63,103,7,98,63,174,41,98,63,221,75,98,63,245,109,98,63,244,143,98,63,220,177,98,63,171,211,98,63,99,245,98,63,3,23,99,63,139,56,99,63,251,89,99,63,83,123,99,63,147,156,99,63,188,189,99,63,204,222,99,63,197,255,99,63,166,32,100,63,110,65,100,63,32,98,100,63,185,130,100,63,58,163,100,63,164,195,100,63,245,227,100,63,47,4,101,63,82,36,101,63,92,68,101,63,78,100,101,63,41,132,101,63,236,163,101,63,151,195,101,63,43,227,101,63,167,2,102,63,11,34,102,63,87,65,102,63,139,96,102,63,168,127,102,63,174,158,102,63,155,189,102,63,113,220,102,63,47,251,102,63,214,25,103,63,101,56,103,63,220,86,103,63,59,117,103,63,132,147,103,63,180,177,103,63,205,207,103,63,206,237,103,63,184,11,104,63,138,41,104,63,69,71,104,63,233,100,104,63,116,130,104,63,233,159,104,63,69,189,104,63,139,218,104,63,185,247,104,63,207,20,105,63,207,49,105,63,182,78,105,63,135,107,105,63,64,136,105,63,225,164,105,63,108,193,105,63,223,221,105,63,59,250,105,63,127,22,106,63,172,50,106,63,195,78,106,63,193,106,106,63,169,134,106,63,121,162,106,63,51,190,106,63,213,217,106,63,96,245,106,63,212,16,107,63,48,44,107,63,118,71,107,63,165,98,107,63,188,125,107,63,189,152,107,63,167,179,107,63,121,206,107,63,53,233,107,63,218,3,108,63,104,30,108,63,223,56,108,63,63,83,108,63,136,109,108,63,187,135,108,63,214,161,108,63,219,187,108,63,201,213,108,63,161,239,108,63,97,9,109,63,11,35,109,63,159,60,109,63,27,86,109,63,129,111,109,63,209,136,109,63,9,162,109,63,44,187,109,63,56,212,109,63,45,237,109,63,12,6,110,63,212,30,110,63,134,55,110,63,33,80,110,63,166,104,110,63,21,129,110,63,110,153,110,63,176,177,110,63,220,201,110,63,241,225,110,63,241,249,110,63,218,17,111,63,173,41,111,63,106,65,111,63,16,89,111,63,161,112,111,63,28,136,111,63,128,159,111,63,207,182,111,63,7,206,111,63,42,229,111,63,54,252,111,63,45,19,112,63,14,42,112,63,217,64,112,63,142,87,112,63,46,110,112,63,184,132,112,63,43,155,112,63,138,177,112,63,210,199,112,63,5,222,112,63,35,244,112,63,42,10,113,63,29,32,113,63,249,53,113,63,193,75,113,63,114,97,113,63,15,119,113,63,150,140,113,63,7,162,113,63,99,183,113,63,170,204,113,63,220,225,113,63,249,246,113,63,0,12,114,63,242,32,114,63,207,53,114,63,151,74,114,63,73,95,114,63,231,115,114,63,112,136,114,63,227,156,114,63,66,177,114,63,140,197,114,63,193,217,114,63,225,237,114,63,236,1,115,63,227,21,115,63,197,41,115,63,146,61,115,63,74,81,115,63,238,100,115,63,125,120,115,63,248,139,115,63,94,159,115,63,175,178,115,63,236,197,115,63,21,217,115,63,41,236,115,63,41,255,115,63,21,18,116,63,236,36,116,63,175,55,116,63,94,74,116,63,248,92,116,63,127,111,116,63,241,129,116,63,80,148,116,63,154,166,116,63,208,184,116,63,242,202,116,63,1,221,116,63,251,238,116,63,226,0,117,63,181,18,117,63,116,36,117,63,31,54,117,63,183,71,117,63,59,89,117,63,171,106,117,63,8,124,117,63,81,141,117,63,135,158,117,63,169,175,117,63,184,192,117,63,179,209,117,63,155,226,117,63,112,243,117,63,50,4,118,63,224,20,118,63,123,37,118,63,3,54,118,63,120,70,118,63,217,86,118,63,40,103,118,63,100,119,118,63,140,135,118,63,162,151,118,63,165,167,118,63,149,183,118,63,114,199,118,63,61,215,118,63,245,230,118,63,154,246,118,63,44,6,119,63,172,21,119,63,26,37,119,63,117,52,119,63,189,67,119,63,243,82,119,63,22,98,119,63,40,113,119,63,39,128,119,63,19,143,119,63,238,157,119,63,182,172,119,63,108,187,119,63,16,202,119,63,162,216,119,63,34,231,119,63,144,245,119,63,236,3,120,63,55,18,120,63,111,32,120,63,150,46,120,63,170,60,120,63,174,74,120,63,159,88,120,63,127,102,120,63,77,116,120,63,10,130,120,63,181,143,120,63,79,157,120,63,215,170,120,63,78,184,120,63,180,197,120,63,8,211,120,63,76,224,120,63,126,237,120,63,158,250,120,63,174,7,121,63,173,20,121,63,155,33,121,63,119,46,121,63,67,59,121,63,254,71,121,63,168,84,121,63,66,97,121,63,202,109,121,63,66,122,121,63,169,134,121,63,0,147,121,63,70,159,121,63,124,171,121,63,161,183,121,63,181,195,121,63,186,207,121,63,173,219,121,63,145,231,121,63,100,243,121,63,40,255,121,63,219,10,122,63,126,22,122,63,16,34,122,63,147,45,122,63,6,57,122,63,105,68,122,63,188,79,122,63,255,90,122,63,51,102,122,63,86,113,122,63,106,124,122,63,111,135,122,63,99,146,122,63,72,157,122,63,30,168,122,63,228,178,122,63,155,189,122,63,66,200,122,63,218,210,122,63,99,221,122,63,221,231,122,63,71,242,122,63,162,252,122,63,238,6,123,63,43,17,123,63,89,27,123,63,120,37,123,63,137,47,123,63,138,57,123,63,124,67,123,63,96,77,123,63,53,87,123,63,252,96,123,63,179,106,123,63,92,116,123,63,247,125,123,63,131,135,123,63,1,145,123,63,112,154,123,63,209,163,123,63,36,173,123,63,104,182,123,63,158,191,123,63,198,200,123,63,224,209,123,63,236,218,123,63,234,227,123,63,218,236,123,63,188,245,123,63,144,254,123,63,86,7,124,63,14,16,124,63,185,24,124,63,86,33,124,63,230,41,124,63,104,50,124,63,220,58,124,63,67,67,124,63,156,75,124,63,232,83,124,63,39,92,124,63,88,100,124,63,124,108,124,63,147,116,124,63,157,124,124,63,153,132,124,63,137,140,124,63,107,148,124,63,65,156,124,63,9,164,124,63,197,171,124,63,116,179,124,63,22,187,124,63,172,194,124,63,52,202,124,63,176,209,124,63,32,217,124,63,131,224,124,63,217,231,124,63,35,239,124,63,97,246,124,63,146,253,124,63,183,4,125,63,208,11,125,63,221,18,125,63,221,25,125,63,209,32,125,63,185,39,125,63,150,46,125,63,102,53,125,63,42,60,125,63,227,66,125,63,143,73,125,63,48,80,125,63,197,86,125,63,78,93,125,63,204,99,125,63,62,106,125,63,165,112,125,63,0,119,125,63,80,125,125,63,148,131,125,63,205,137,125,63,251,143,125,63,29,150,125,63,52,156,125,63,64,162,125,63,65,168,125,63,55,174,125,63,34,180,125,63,2,186,125,63,215,191,125,63,161,197,125,63,96,203,125,63,21,209,125,63,190,214,125,63,93,220,125,63,242,225,125,63,124,231,125,63,251,236,125,63,112,242,125,63,218,247,125,63,58,253,125,63,143,2,126,63,219,7,126,63,28,13,126,63,82,18,126,63,127,23,126,63,161,28,126,63,186,33,126,63,200,38,126,63,204,43,126,63,199,48,126,63,183,53,126,63,158,58,126,63,123,63,126,63,78,68,126,63,23,73,126,63,215,77,126,63,141,82,126,63,58,87,126,63,221,91,126,63,118,96,126,63,6,101,126,63,141,105,126,63,10,110,126,63,126,114,126,63,233,118,126,63,75,123,126,63,164,127,126,63,243,131,126,63,57,136,126,63,119,140,126,63,171,144,126,63,214,148,126,63,249,152,126,63,18,157,126,63,35,161,126,63,44,165,126,63,43,169,126,63,34,173,126,63,16,177,126,63,246,180,126,63,211,184,126,63,167,188,126,63,115,192,126,63,55,196,126,63,243,199,126,63,166,203,126,63,81,207,126,63,243,210,126,63,142,214,126,63,32,218,126,63,171,221,126,63,45,225,126,63,167,228,126,63,26,232,126,63,132,235,126,63,231,238,126,63,66,242,126,63,149,245,126,63,224,248,126,63,36,252,126,63,96,255,126,63,148,2,127,63,193,5,127,63,230,8,127,63,4,12,127,63,27,15,127,63,42,18,127,63,50,21,127,63,50,24,127,63,43,27,127,63,29,30,127,63,8,33,127,63,236,35,127,63,201,38,127,63,158,41,127,63,109,44,127,63,53,47,127,63,246,49,127,63,175,52,127,63,99,55,127,63,15,58,127,63,181,60,127,63,83,63,127,63,236,65,127,63,125,68,127,63,8,71,127,63,141,73,127,63,11,76,127,63,131,78,127,63,244,80,127,63,95,83,127,63,195,85,127,63,33,88,127,63,121,90,127,63,203,92,127,63,23,95,127,63,92,97,127,63,155,99,127,63,213,101,127,63,8,104,127,63,54,106,127,63,93,108,127,63,127,110,127,63,155,112,127,63,177,114,127,63,193,116,127,63,203,118,127,63,208,120,127,63,207,122,127,63,201,124,127,63,189,126,127,63,171,128,127,63,148,130,127,63,120,132,127,63,86,134,127,63,47,136,127,63,2,138,127,63,209,139,127,63,153,141,127,63,93,143,127,63,28,145,127,63,213,146,127,63,137,148,127,63,57,150,127,63,227,151,127,63,136,153,127,63,40,155,127,63,196,156,127,63,90,158,127,63,236,159,127,63,121,161,127,63,1,163,127,63,132,164,127,63,3,166,127,63,125,167,127,63,242,168,127,63,99,170,127,63,207,171,127,63,55,173,127,63,154,174,127,63,249,175,127,63,84,177,127,63,170,178,127,63,251,179,127,63,73,181,127,63,146,182,127,63,215,183,127,63,24,185,127,63,85,186,127,63,141,187,127,63,193,188,127,63,242,189,127,63,30,191,127,63,71,192,127,63,107,193,127,63,140,194,127,63,168,195,127,63,193,196,127,63,214,197,127,63,231,198,127,63,245,199,127,63,255,200,127,63,5,202,127,63,7,203,127,63,6,204,127,63,1,205,127,63,249,205,127,63,237,206,127,63,222,207,127,63,203,208,127,63,181,209,127,63,156,210,127,63,127,211,127,63,95,212,127,63,59,213,127,63,20,214,127,63,234,214,127,63,189,215,127,63,141,216,127,63,90,217,127,63,35,218,127,63,233,218,127,63,173,219,127,63,109,220,127,63,43,221,127,63,229,221,127,63,156,222,127,63,81,223,127,63,3,224,127,63,178,224,127,63,94,225,127,63,7,226,127,63,174,226,127,63,82,227,127,63,243,227,127,63,146,228,127,63,46,229,127,63,199,229,127,63,94,230,127,63,242,230,127,63,132,231,127,63,19,232,127,63,160,232,127,63,42,233,127,63,178,233,127,63,56,234,127,63,187,234,127,63,60,235,127,63,187,235,127,63,55,236,127,63,177,236,127,63,41,237,127,63,159,237,127,63,18,238,127,63,132,238,127,63,243,238,127,63,96,239,127,63,204,239,127,63,53,240,127,63,156,240,127,63,1,241,127,63,101,241,127,63,198,241,127,63,37,242,127,63,131,242,127,63,222,242,127,63,56,243,127,63,144,243,127,63,231,243,127,63,59,244,127,63,142,244,127,63,223,244,127,63,46,245,127,63,124,245,127,63,200,245,127,63,19,246,127,63,91,246,127,63,163,246,127,63,233,246,127,63,45,247,127,63,111,247,127,63,177,247,127,63,240,247,127,63,47,248,127,63,108,248,127,63,167,248,127,63,225,248,127,63,26,249,127,63,82,249,127,63,136,249,127,63,188,249,127,63,240,249,127,63,34,250,127,63,83,250,127,63,131,250,127,63,178,250,127,63,224,250,127,63,12,251,127,63,55,251,127,63,97,251,127,63,138,251,127,63,178,251,127,63,217,251,127,63,255,251,127,63,36,252,127,63,72,252,127,63,107,252,127,63,141,252,127,63,173,252,127,63,205,252,127,63,237,252,127,63,11,253,127,63,40,253,127,63,69,253,127,63,96,253,127,63,123,253,127,63,149,253,127,63,174,253,127,63,199,253,127,63,222,253,127,63,245,253,127,63,12,254,127,63,33,254,127,63,54,254,127,63,74,254,127,63,93,254,127,63,112,254,127,63,130,254,127,63,148,254,127,63,165,254,127,63,181,254,127,63,197,254,127,63,212,254,127,63,227,254,127,63,241,254,127,63,254,254,127,63,11,255,127,63,24,255,127,63,36,255,127,63,47,255,127,63,59,255,127,63,69,255,127,63,79,255,127,63,89,255,127,63,99,255,127,63,108,255,127,63,116,255,127,63,124,255,127,63,132,255,127,63,140,255,127,63,147,255,127,63,154,255,127,63,160,255,127,63,166,255,127,63,172,255,127,63,178,255,127,63,183,255,127,63,188,255,127,63,193,255,127,63,197,255,127,63,202,255,127,63,206,255,127,63,209,255,127,63,213,255,127,63,216,255,127,63,220,255,127,63,223,255,127,63,225,255,127,63,228,255,127,63,230,255,127,63,233,255,127,63,235,255,127,63,237,255,127,63,239,255,127,63,240,255,127,63,242,255,127,63,243,255,127,63,245,255,127,63,246,255,127,63,247,255,127,63,248,255,127,63,249,255,127,63,250,255,127,63,251,255,127,63,251,255,127,63,252,255,127,63,252,255,127,63,253,255,127,63,253,255,127,63,254,255,127,63,254,255,127,63,254,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,198,63,120,51,98,136,11,53,151,200,193,53,80,233,61,54,183,247,156,54,46,124,234,54,153,192,35,55,244,2,90,55,56,3,140,55,227,228,174,55,177,166,213,55,108,36,0,56,146,101,23,56,201,150,48,56,18,184,75,56,81,201,104,56,94,229,131,56,29,94,148,56,229,206,165,56,167,55,184,56,128,152,203,56,85,241,223,56,36,66,245,56,126,197,5,57,238,101,17,57,99,130,29,57,207,26,42,57,63,47,55,57,179,191,68,57,30,204,82,57,141,84,97,57,243,88,112,57,94,217,127,57,227,234,135,57,18,39,144,57,64,161,152,57,105,89,161,57,146,79,170,57,181,131,179,57,215,245,188,57,245,165,198,57,14,148,208,57,34,192,218,57,46,42,229,57,57,210,239,57,60,184,250,57,27,238,2,58,22,159,8,58,13,111,14,58,0,94,20,58,239,107,26,58,218,152,32,58,192,228,38,58,161,79,45,58,124,217,51,58,83,130,58,58,37,74,65,58,240,48,72,58,182,54,79,58,116,91,86,58,45,159,93,58,222,1,101,58,136,131,108,58,42,36,116,58,196,227,123,58,44,225,129,58,241,223,133,58,49,238,137,58,238,11,142,58,37,57,146,58,215,117,150,58,5,194,154,58,174,29,159,58,209,136,163,58,110,3,168,58,134,141,172,58,24,39,177,58,36,208,181,58,169,136,186,58,169,80,191,58,33,40,196,58,19,15,201,58,126,5,206,58,98,11,211,58,191,32,216,58,148,69,221,58,225,121,226,58,166,189,231,58,227,16,237,58,152,115,242,58,196,229,247,58,103,103,253,58,65,124,1,59,137,76,4,59,141,36,7,59,76,4,10,59,198,235,12,59,251,218,15,59,235,209,18,59,149,208,21,59,251,214,24,59,26,229,27,59,244,250,30,59,136,24,34,59,215,61,37,59,223,106,40,59,161,159,43,59,29,220,46,59,83,32,50,59,66,108,53,59,234,191,56,59,76,27,60,59,103,126,63,59,59,233,66,59,199,91,70,59,12,214,73,59,10,88,77,59,193,225,80,59,48,115,84,59,86,12,88,59,53,173,91,59,204,85,95,59,26,6,99,59,32,190,102,59,222,125,106,59,82,69,110,59,127,20,114,59,97,235,117,59,251,201,121,59,76,176,125,59,41,207,128,59,8,202,130,59,194,200,132,59], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+10240);
/* memory initializer */ allocate([87,203,134,59,198,209,136,59,17,220,138,59,55,234,140,59,55,252,142,59,18,18,145,59,199,43,147,59,87,73,149,59,194,106,151,59,6,144,153,59,37,185,155,59,30,230,157,59,241,22,160,59,158,75,162,59,37,132,164,59,134,192,166,59,192,0,169,59,212,68,171,59,193,140,173,59,137,216,175,59,41,40,178,59,163,123,180,59,245,210,182,59,33,46,185,59,38,141,187,59,4,240,189,59,186,86,192,59,73,193,194,59,177,47,197,59,242,161,199,59,10,24,202,59,251,145,204,59,196,15,207,59,102,145,209,59,223,22,212,59,49,160,214,59,90,45,217,59,91,190,219,59,51,83,222,59,227,235,224,59,107,136,227,59,201,40,230,59,255,204,232,59,12,117,235,59,240,32,238,59,171,208,240,59,61,132,243,59,165,59,246,59,228,246,248,59,250,181,251,59,229,120,254,59,212,159,0,60,32,5,2,60,87,108,3,60,121,213,4,60,134,64,6,60,126,173,7,60,96,28,9,60,45,141,10,60,229,255,11,60,136,116,13,60,21,235,14,60,141,99,16,60,239,221,17,60,59,90,19,60,114,216,20,60,147,88,22,60,158,218,23,60,147,94,25,60,115,228,26,60,60,108,28,60,240,245,29,60,141,129,31,60,20,15,33,60,133,158,34,60,224,47,36,60,36,195,37,60,82,88,39,60,105,239,40,60,106,136,42,60,84,35,44,60,40,192,45,60,229,94,47,60,139,255,48,60,26,162,50,60,146,70,52,60,243,236,53,60,61,149,55,60,112,63,57,60,140,235,58,60,145,153,60,60,126,73,62,60,84,251,63,60,18,175,65,60,185,100,67,60,72,28,69,60,192,213,70,60,31,145,72,60,103,78,74,60,151,13,76,60,175,206,77,60,176,145,79,60,152,86,81,60,103,29,83,60,31,230,84,60,190,176,86,60,69,125,88,60,179,75,90,60,9,28,92,60,71,238,93,60,107,194,95,60,119,152,97,60,106,112,99,60,68,74,101,60,5,38,103,60,173,3,105,60,60,227,106,60,178,196,108,60,14,168,110,60,81,141,112,60,123,116,114,60,139,93,116,60,130,72,118,60,95,53,120,60,34,36,122,60,203,20,124,60,90,7,126,60,208,251,127,60,22,249,128,60,54,245,129,60,74,242,130,60,80,240,131,60,73,239,132,60,53,239,133,60,19,240,134,60,229,241,135,60,169,244,136,60,95,248,137,60,8,253,138,60,164,2,140,60,50,9,141,60,178,16,142,60,37,25,143,60,139,34,144,60,226,44,145,60,44,56,146,60,104,68,147,60,150,81,148,60,182,95,149,60,201,110,150,60,205,126,151,60,196,143,152,60,172,161,153,60,135,180,154,60,83,200,155,60,17,221,156,60,193,242,157,60,98,9,159,60,245,32,160,60,122,57,161,60,241,82,162,60,89,109,163,60,178,136,164,60,253,164,165,60,57,194,166,60,103,224,167,60,134,255,168,60,151,31,170,60,152,64,171,60,139,98,172,60,111,133,173,60,68,169,174,60,10,206,175,60,193,243,176,60,105,26,178,60,2,66,179,60,139,106,180,60,6,148,181,60,113,190,182,60,205,233,183,60,26,22,185,60,87,67,186,60,133,113,187,60,163,160,188,60,177,208,189,60,177,1,191,60,160,51,192,60,128,102,193,60,80,154,194,60,16,207,195,60,193,4,197,60,97,59,198,60,242,114,199,60,114,171,200,60,227,228,201,60,67,31,203,60,147,90,204,60,211,150,205,60,3,212,206,60,34,18,208,60,49,81,209,60,48,145,210,60,30,210,211,60,252,19,213,60,201,86,214,60,133,154,215,60,49,223,216,60,204,36,218,60,86,107,219,60,208,178,220,60,56,251,221,60,144,68,223,60,214,142,224,60,12,218,225,60,48,38,227,60,67,115,228,60,69,193,229,60,54,16,231,60,21,96,232,60,227,176,233,60,160,2,235,60,75,85,236,60,228,168,237,60,108,253,238,60,226,82,240,60,70,169,241,60,153,0,243,60,218,88,244,60,8,178,245,60,37,12,247,60,48,103,248,60,41,195,249,60,15,32,251,60,228,125,252,60,166,220,253,60,85,60,255,60,121,78,0,61,63,255,0,61,123,176,1,61,46,98,2,61,88,20,3,61,248,198,3,61,15,122,4,61,156,45,5,61,161,225,5,61,27,150,6,61,12,75,7,61,116,0,8,61,82,182,8,61,167,108,9,61,113,35,10,61,179,218,10,61,106,146,11,61,152,74,12,61,60,3,13,61,87,188,13,61,231,117,14,61,238,47,15,61,107,234,15,61,94,165,16,61,199,96,17,61,166,28,18,61,251,216,18,61,198,149,19,61,7,83,20,61,190,16,21,61,234,206,21,61,141,141,22,61,165,76,23,61,52,12,24,61,56,204,24,61,177,140,25,61,161,77,26,61,6,15,27,61,224,208,27,61,48,147,28,61,246,85,29,61,49,25,30,61,226,220,30,61,8,161,31,61,164,101,32,61,181,42,33,61,59,240,33,61,55,182,34,61,168,124,35,61,142,67,36,61,233,10,37,61,186,210,37,61,255,154,38,61,186,99,39,61,234,44,40,61,143,246,40,61,168,192,41,61,55,139,42,61,59,86,43,61,180,33,44,61,161,237,44,61,4,186,45,61,219,134,46,61,38,84,47,61,231,33,48,61,28,240,48,61,198,190,49,61,229,141,50,61,120,93,51,61,127,45,52,61,251,253,52,61,236,206,53,61,81,160,54,61,42,114,55,61,120,68,56,61,58,23,57,61,112,234,57,61,27,190,58,61,58,146,59,61,204,102,60,61,211,59,61,61,79,17,62,61,62,231,62,61,161,189,63,61,120,148,64,61,195,107,65,61,130,67,66,61,181,27,67,61,92,244,67,61,118,205,68,61,4,167,69,61,6,129,70,61,124,91,71,61,101,54,72,61,194,17,73,61,146,237,73,61,214,201,74,61,141,166,75,61,184,131,76,61,86,97,77,61,104,63,78,61,236,29,79,61,229,252,79,61,80,220,80,61,46,188,81,61,128,156,82,61,69,125,83,61,125,94,84,61,40,64,85,61,69,34,86,61,214,4,87,61,218,231,87,61,81,203,88,61,58,175,89,61,150,147,90,61,101,120,91,61,167,93,92,61,91,67,93,61,130,41,94,61,28,16,95,61,40,247,95,61,167,222,96,61,152,198,97,61,251,174,98,61,209,151,99,61,25,129,100,61,212,106,101,61,0,85,102,61,159,63,103,61,176,42,104,61,51,22,105,61,41,2,106,61,144,238,106,61,105,219,107,61,180,200,108,61,113,182,109,61,160,164,110,61,65,147,111,61,84,130,112,61,216,113,113,61,206,97,114,61,54,82,115,61,15,67,116,61,89,52,117,61,22,38,118,61,67,24,119,61,226,10,120,61,243,253,120,61,117,241,121,61,104,229,122,61,204,217,123,61,162,206,124,61,232,195,125,61,160,185,126,61,201,175,127,61,49,83,128,61,183,206,128,61,117,74,129,61,107,198,129,61,154,66,130,61,1,191,130,61,160,59,131,61,120,184,131,61,136,53,132,61,209,178,132,61,81,48,133,61,10,174,133,61,251,43,134,61,37,170,134,61,134,40,135,61,32,167,135,61,242,37,136,61,252,164,136,61,62,36,137,61,184,163,137,61,106,35,138,61,84,163,138,61,118,35,139,61,209,163,139,61,99,36,140,61,45,165,140,61,46,38,141,61,104,167,141,61,218,40,142,61,131,170,142,61,100,44,143,61,125,174,143,61,206,48,144,61,86,179,144,61,23,54,145,61,14,185,145,61,62,60,146,61,165,191,146,61,67,67,147,61,26,199,147,61,39,75,148,61,109,207,148,61,234,83,149,61,158,216,149,61,138,93,150,61,173,226,150,61,7,104,151,61,153,237,151,61,98,115,152,61,99,249,152,61,155,127,153,61,10,6,154,61,176,140,154,61,142,19,155,61,163,154,155,61,239,33,156,61,114,169,156,61,44,49,157,61,29,185,157,61,69,65,158,61,165,201,158,61,59,82,159,61,8,219,159,61,13,100,160,61,72,237,160,61,186,118,161,61,99,0,162,61,67,138,162,61,90,20,163,61,167,158,163,61,43,41,164,61,230,179,164,61,216,62,165,61,0,202,165,61,95,85,166,61,245,224,166,61,193,108,167,61,196,248,167,61,254,132,168,61,110,17,169,61,20,158,169,61,241,42,170,61,4,184,170,61,78,69,171,61,206,210,171,61,133,96,172,61,113,238,172,61,149,124,173,61,238,10,174,61,126,153,174,61,67,40,175,61,63,183,175,61,114,70,176,61,218,213,176,61,120,101,177,61,77,245,177,61,88,133,178,61,152,21,179,61,15,166,179,61,187,54,180,61,158,199,180,61,182,88,181,61,4,234,181,61,137,123,182,61,67,13,183,61,50,159,183,61,88,49,184,61,179,195,184,61,68,86,185,61,11,233,185,61,7,124,186,61,57,15,187,61,160,162,187,61,61,54,188,61,16,202,188,61,24,94,189,61,85,242,189,61,200,134,190,61,112,27,191,61,78,176,191,61,97,69,192,61,170,218,192,61,39,112,193,61,218,5,194,61,194,155,194,61,224,49,195,61,50,200,195,61,186,94,196,61,119,245,196,61,104,140,197,61,143,35,198,61,235,186,198,61,124,82,199,61,66,234,199,61,61,130,200,61,108,26,201,61,209,178,201,61,106,75,202,61,57,228,202,61,59,125,203,61,115,22,204,61,224,175,204,61,129,73,205,61,86,227,205,61,97,125,206,61,159,23,207,61,19,178,207,61,187,76,208,61,151,231,208,61,168,130,209,61,237,29,210,61,103,185,210,61,21,85,211,61,248,240,211,61,14,141,212,61,89,41,213,61,216,197,213,61,140,98,214,61,115,255,214,61,143,156,215,61,223,57,216,61,99,215,216,61,27,117,217,61,7,19,218,61,38,177,218,61,122,79,219,61,2,238,219,61,189,140,220,61,173,43,221,61,208,202,221,61,39,106,222,61,178,9,223,61,112,169,223,61,98,73,224,61,136,233,224,61,226,137,225,61,111,42,226,61,47,203,226,61,35,108,227,61,74,13,228,61,165,174,228,61,52,80,229,61,245,241,229,61,234,147,230,61,19,54,231,61,110,216,231,61,253,122,232,61,191,29,233,61,180,192,233,61,221,99,234,61,56,7,235,61,199,170,235,61,136,78,236,61,125,242,236,61,164,150,237,61,255,58,238,61,140,223,238,61,76,132,239,61,63,41,240,61,101,206,240,61,189,115,241,61,73,25,242,61,7,191,242,61,247,100,243,61,26,11,244,61,112,177,244,61,248,87,245,61,179,254,245,61,160,165,246,61,192,76,247,61,18,244,247,61,151,155,248,61,77,67,249,61,55,235,249,61,82,147,250,61,159,59,251,61,31,228,251,61,209,140,252,61,181,53,253,61,203,222,253,61,19,136,254,61,141,49,255,61,57,219,255,61,140,66,0,62,148,151,0,62,181,236,0,62,238,65,1,62,65,151,1,62,173,236,1,62,49,66,2,62,206,151,2,62,132,237,2,62,83,67,3,62,59,153,3,62,59,239,3,62,84,69,4,62,134,155,4,62,209,241,4,62,52,72,5,62,176,158,5,62,68,245,5,62,242,75,6,62,183,162,6,62,150,249,6,62,141,80,7,62,156,167,7,62,196,254,7,62,5,86,8,62,94,173,8,62,207,4,9,62,89,92,9,62,252,179,9,62,183,11,10,62,138,99,10,62,118,187,10,62,122,19,11,62,150,107,11,62,203,195,11,62,24,28,12,62,125,116,12,62,250,204,12,62,144,37,13,62,62,126,13,62,4,215,13,62,227,47,14,62,217,136,14,62,232,225,14,62,15,59,15,62,78,148,15,62,165,237,15,62,20,71,16,62,155,160,16,62,58,250,16,62,241,83,17,62,193,173,17,62,168,7,18,62,167,97,18,62,190,187,18,62,237,21,19,62,51,112,19,62,146,202,19,62,9,37,20,62,151,127,20,62,61,218,20,62,251,52,21,62,209,143,21,62,190,234,21,62,195,69,22,62,224,160,22,62,21,252,22,62,97,87,23,62,197,178,23,62,64,14,24,62,211,105,24,62,126,197,24,62,64,33,25,62,26,125,25,62,11,217,25,62,20,53,26,62,52,145,26,62,108,237,26,62,187,73,27,62,34,166,27,62,160,2,28,62,53,95,28,62,226,187,28,62,166,24,29,62,129,117,29,62,116,210,29,62,126,47,30,62,159,140,30,62,215,233,30,62,39,71,31,62,141,164,31,62,11,2,32,62,160,95,32,62,76,189,32,62,16,27,33,62,234,120,33,62,219,214,33,62,228,52,34,62,3,147,34,62,58,241,34,62,135,79,35,62,235,173,35,62,103,12,36,62,249,106,36,62,162,201,36,62,98,40,37,62,56,135,37,62,38,230,37,62,42,69,38,62,69,164,38,62,119,3,39,62,192,98,39,62,31,194,39,62,149,33,40,62,33,129,40,62,197,224,40,62,126,64,41,62,79,160,41,62,54,0,42,62,51,96,42,62,72,192,42,62,114,32,43,62,179,128,43,62,11,225,43,62,121,65,44,62,253,161,44,62,152,2,45,62,73,99,45,62,16,196,45,62,238,36,46,62,226,133,46,62,237,230,46,62,13,72,47,62,68,169,47,62,145,10,48,62,245,107,48,62,110,205,48,62,254,46,49,62,163,144,49,62,95,242,49,62,49,84,50,62,25,182,50,62,23,24,51,62,43,122,51,62,85,220,51,62,148,62,52,62,234,160,52,62,86,3,53,62,216,101,53,62,111,200,53,62,28,43,54,62,223,141,54,62,184,240,54,62,167,83,55,62,171,182,55,62,197,25,56,62,245,124,56,62,59,224,56,62,150,67,57,62,7,167,57,62,141,10,58,62,41,110,58,62,219,209,58,62,162,53,59,62,126,153,59,62,112,253,59,62,120,97,60,62,149,197,60,62,199,41,61,62,15,142,61,62,108,242,61,62,222,86,62,62,102,187,62,62,3,32,63,62,181,132,63,62,125,233,63,62,90,78,64,62,75,179,64,62,83,24,65,62,111,125,65,62,160,226,65,62,231,71,66,62,66,173,66,62,179,18,67,62,57,120,67,62,211,221,67,62,131,67,68,62,71,169,68,62,33,15,69,62,15,117,69,62,18,219,69,62,42,65,70,62,87,167,70,62,153,13,71,62,240,115,71,62,91,218,71,62,219,64,72,62,111,167,72,62,25,14,73,62,215,116,73,62,169,219,73,62,144,66,74,62,140,169,74,62,157,16,75,62,193,119,75,62,251,222,75,62,73,70,76,62,171,173,76,62,34,21,77,62,173,124,77,62,76,228,77,62,0,76,78,62,200,179,78,62,164,27,79,62,149,131,79,62,154,235,79,62,179,83,80,62,225,187,80,62,34,36,81,62,120,140,81,62,225,244,81,62,95,93,82,62,241,197,82,62,151,46,83,62,81,151,83,62,31,0,84,62,1,105,84,62,247,209,84,62,0,59,85,62,30,164,85,62,79,13,86,62,149,118,86,62,238,223,86,62,91,73,87,62,219,178,87,62,112,28,88,62,24,134,88,62,211,239,88,62,163,89,89,62,134,195,89,62,124,45,90,62,134,151,90,62,164,1,91,62,213,107,91,62,26,214,91,62,114,64,92,62,221,170,92,62,92,21,93,62,239,127,93,62,148,234,93,62,77,85,94,62,26,192,94,62,249,42,95,62,236,149,95,62,242,0,96,62,11,108,96,62,55,215,96,62,119,66,97,62,202,173,97,62,47,25,98,62,168,132,98,62,52,240,98,62,210,91,99,62,132,199,99,62,73,51,100,62,32,159,100,62,11,11,101,62,8,119,101,62,24,227,101,62,59,79,102,62,113,187,102,62,186,39,103,62,21,148,103,62,131,0,104,62,3,109,104,62,151,217,104,62,60,70,105,62,245,178,105,62,192,31,106,62,157,140,106,62,141,249,106,62,144,102,107,62,165,211,107,62,204,64,108,62,6,174,108,62,82,27,109,62,176,136,109,62,33,246,109,62,164,99,110,62,57,209,110,62,225,62,111,62,154,172,111,62,102,26,112,62,68,136,112,62,52,246,112,62,55,100,113,62,75,210,113,62,113,64,114,62,169,174,114,62,243,28,115,62,80,139,115,62,190,249,115,62,61,104,116,62,207,214,116,62,115,69,117,62,40,180,117,62,239,34,118,62,200,145,118,62,179,0,119,62,175,111,119,62,189,222,119,62,221,77,120,62,14,189,120,62,80,44,121,62,165,155,121,62,10,11,122,62,130,122,122,62,10,234,122,62,164,89,123,62,80,201,123,62,13,57,124,62,219,168,124,62,186,24,125,62,171,136,125,62,173,248,125,62,192,104,126,62,228,216,126,62,26,73,127,62,96,185,127,62,220,20,128,62,16,77,128,62,77,133,128,62,147,189,128,62,225,245,128,62,55,46,129,62,150,102,129,62,253,158,129,62,109,215,129,62,229,15,130,62,102,72,130,62,238,128,130,62,128,185,130,62,25,242,130,62,187,42,131,62,102,99,131,62,24,156,131,62,211,212,131,62,150,13,132,62,98,70,132,62,53,127,132,62,17,184,132,62,245,240,132,62,226,41,133,62,214,98,133,62,211,155,133,62,216,212,133,62,229,13,134,62,250,70,134,62,23,128,134,62,61,185,134,62,106,242,134,62,160,43,135,62,221,100,135,62,35,158,135,62,112,215,135,62,198,16,136,62,35,74,136,62,137,131,136,62,247,188,136,62,108,246,136,62,233,47,137,62,111,105,137,62,252,162,137,62,145,220,137,62,46,22,138,62,211,79,138,62,127,137,138,62,52,195,138,62,240,252,138,62,180,54,139,62,128,112,139,62,84,170,139,62,47,228,139,62,18,30,140,62,253,87,140,62,239,145,140,62,233,203,140,62,235,5,141,62,245,63,141,62,6,122,141,62,31,180,141,62,63,238,141,62,103,40,142,62,150,98,142,62,205,156,142,62,12,215,142,62,82,17,143,62,159,75,143,62,245,133,143,62,81,192,143,62,181,250,143,62,33,53,144,62,147,111,144,62,14,170,144,62,143,228,144,62,25,31,145,62,169,89,145,62,65,148,145,62,224,206,145,62,134,9,146,62,52,68,146,62,233,126,146,62,165,185,146,62,105,244,146,62,52,47,147,62,6,106,147,62,223,164,147,62,191,223,147,62,167,26,148,62,150,85,148,62,139,144,148,62,136,203,148,62,140,6,149,62,152,65,149,62,170,124,149,62,195,183,149,62,227,242,149,62,11,46,150,62,57,105,150,62,111,164,150,62,171,223,150,62,238,26,151,62,56,86,151,62,138,145,151,62,226,204,151,62,65,8,152,62,167,67,152,62,19,127,152,62,135,186,152,62,1,246,152,62,130,49,153,62,10,109,153,62,153,168,153,62,47,228,153,62,203,31,154,62,110,91,154,62,24,151,154,62,200,210,154,62,127,14,155,62,61,74,155,62,2,134,155,62,205,193,155,62,158,253,155,62,119,57,156,62,85,117,156,62,59,177,156,62,39,237,156,62,25,41,157,62,18,101,157,62,18,161,157,62,24,221,157,62,36,25,158,62,55,85,158,62,80,145,158,62,112,205,158,62,150,9,159,62,195,69,159,62,246,129,159,62,47,190,159,62,111,250,159,62,180,54,160,62,1,115,160,62,83,175,160,62,172,235,160,62,11,40,161,62,112,100,161,62,219,160,161,62,77,221,161,62,196,25,162,62,66,86,162,62,198,146,162,62,81,207,162,62,225,11,163,62,119,72,163,62,20,133,163,62,182,193,163,62,95,254,163,62,13,59,164,62,194,119,164,62,125,180,164,62,61,241,164,62,4,46,165,62,208,106,165,62,162,167,165,62,123,228,165,62,89,33,166,62,61,94,166,62,39,155,166,62,23,216,166,62,12,21,167,62,7,82,167,62,8,143,167,62,15,204,167,62,28,9,168,62,46,70,168,62,70,131,168,62,100,192,168,62,136,253,168,62,177,58,169,62,223,119,169,62,20,181,169,62,78,242,169,62,141,47,170,62,211,108,170,62,29,170,170,62,109,231,170,62,195,36,171,62,31,98,171,62,127,159,171,62,230,220,171,62,81,26,172,62,194,87,172,62,57,149,172,62,181,210,172,62,54,16,173,62,189,77,173,62,73,139,173,62,218,200,173,62,113,6,174,62,13,68,174,62,174,129,174,62,85,191,174,62,0,253,174,62,177,58,175,62,103,120,175,62,35,182,175,62,227,243,175,62,169,49,176,62,116,111,176,62,68,173,176,62,25,235,176,62,243,40,177,62,210,102,177,62,182,164,177,62,160,226,177,62,142,32,178,62,129,94,178,62,121,156,178,62,119,218,178,62,121,24,179,62,128,86,179,62,140,148,179,62,157,210,179,62,178,16,180,62,205,78,180,62,236,140,180,62,16,203,180,62,57,9,181,62,103,71,181,62,154,133,181,62,209,195,181,62,13,2,182,62,78,64,182,62,147,126,182,62,221,188,182,62,44,251,182,62,127,57,183,62,215,119,183,62,52,182,183,62,149,244,183,62,251,50,184,62,101,113,184,62,212,175,184,62,71,238,184,62,191,44,185,62,59,107,185,62,188,169,185,62,65,232,185,62,202,38,186,62,88,101,186,62,235,163,186,62,129,226,186,62,28,33,187,62,188,95,187,62,95,158,187,62,7,221,187,62,180,27,188,62,100,90,188,62,25,153,188,62,210,215,188,62,143,22,189,62,80,85,189,62,22,148,189,62,223,210,189,62,173,17,190,62,127,80,190,62,85,143,190,62,47,206,190,62,13,13,191,62,239,75,191,62,213,138,191,62,191,201,191,62,173,8,192,62,159,71,192,62,149,134,192,62,143,197,192,62,141,4,193,62,143,67,193,62,148,130,193,62,158,193,193,62,171,0,194,62,188,63,194,62,209,126,194,62,234,189,194,62,6,253,194,62,38,60,195,62,74,123,195,62,113,186,195,62,157,249,195,62,204,56,196,62,254,119,196,62,52,183,196,62,110,246,196,62,171,53,197,62,236,116,197,62,49,180,197,62,121,243,197,62,196,50,198,62,19,114,198,62,102,177,198,62,188,240,198,62,21,48,199,62,114,111,199,62,210,174,199,62,54,238,199,62,157,45,200,62,7,109,200,62,117,172,200,62,230,235,200,62,90,43,201,62,209,106,201,62,76,170,201,62,202,233,201,62,75,41,202,62,208,104,202,62,88,168,202,62,226,231,202,62,112,39,203,62,1,103,203,62,149,166,203,62,45,230,203,62,199,37,204,62,100,101,204,62,4,165,204,62,168,228,204,62,78,36,205,62,248,99,205,62,164,163,205,62,83,227,205,62,5,35,206,62,186,98,206,62,114,162,206,62,45,226,206,62,234,33,207,62,171,97,207,62,110,161,207,62,52,225,207,62,253,32,208,62,200,96,208,62,150,160,208,62,103,224,208,62,59,32,209,62,17,96,209,62,234,159,209,62,198,223,209,62,164,31,210,62,133,95,210,62,104,159,210,62,78,223,210,62,55,31,211,62,33,95,211,62,15,159,211,62,255,222,211,62,241,30,212,62,230,94,212,62,221,158,212,62,215,222,212,62,211,30,213,62,209,94,213,62,210,158,213,62,213,222,213,62,219,30,214,62,226,94,214,62,236,158,214,62,248,222,214,62,7,31,215,62,24,95,215,62,42,159,215,62,63,223,215,62,87,31,216,62,112,95,216,62,139,159,216,62,169,223,216,62,200,31,217,62,234,95,217,62,14,160,217,62,51,224,217,62,91,32,218,62,133,96,218,62,176,160,218,62,222,224,218,62,13,33,219,62,63,97,219,62,114,161,219,62,167,225,219,62,222,33,220,62,23,98,220,62,82,162,220,62,142,226,220,62,204,34,221,62,12,99,221,62,78,163,221,62,146,227,221,62,215,35,222,62,29,100,222,62,102,164,222,62,176,228,222,62,252,36,223,62,73,101,223,62,152,165,223,62,232,229,223,62,58,38,224,62,142,102,224,62,227,166,224,62,57,231,224,62,145,39,225,62,234,103,225,62,69,168,225,62,161,232,225,62,255,40,226,62,94,105,226,62,190,169,226,62,32,234,226,62,131,42,227,62,231,106,227,62,76,171,227,62,179,235,227,62,27,44,228,62,132,108,228,62,238,172,228,62,90,237,228,62,199,45,229,62,52,110,229,62,163,174,229,62,19,239,229,62,133,47,230,62,247,111,230,62,106,176,230,62,222,240,230,62,83,49,231,62,202,113,231,62,65,178,231,62,185,242,231,62,50,51,232,62,172,115,232,62,38,180,232,62,162,244,232,62,31,53,233,62,156,117,233,62,26,182,233,62,153,246,233,62,25,55,234,62,153,119,234,62,26,184,234,62,156,248,234,62,31,57,235,62,162,121,235,62,38,186,235,62,170,250,235,62,47,59,236,62,181,123,236,62,59,188,236,62,194,252,236,62,73,61,237,62,209,125,237,62,89,190,237,62,226,254,237,62,107,63,238,62,245,127,238,62,127,192,238,62,10,1,239,62,149,65,239,62,32,130,239,62,171,194,239,62,55,3,240,62,196,67,240,62,80,132,240,62,221,196,240,62,106,5,241,62,247,69,241,62,132,134,241,62,18,199,241,62,160,7,242,62,45,72,242,62,187,136,242,62,74,201,242,62,216,9,243,62,102,74,243,62,244,138,243,62,131,203,243,62,17,12,244,62,159,76,244,62,46,141,244,62,188,205,244,62,74,14,245,62,216,78,245,62,102,143,245,62,244,207,245,62,129,16,246,62,15,81,246,62,156,145,246,62,41,210,246,62,182,18,247,62,67,83,247,62,207,147,247,62,91,212,247,62,231,20,248,62,115,85,248,62,254,149,248,62,136,214,248,62,19,23,249,62,157,87,249,62,38,152,249,62,175,216,249,62,56,25,250,62,192,89,250,62,72,154,250,62,207,218,250,62,86,27,251,62,220,91,251,62,97,156,251,62,230,220,251,62,106,29,252,62,238,93,252,62,113,158,252,62,243,222,252,62,117,31,253,62,245,95,253,62,118,160,253,62,245,224,253,62,116,33,254,62,241,97,254,62,110,162,254,62,235,226,254,62,102,35,255,62,224,99,255,62,90,164,255,62,211,228,255,62,165,18,0,63,225,50,0,63,27,83,0,63,86,115,0,63,144,147,0,63,201,179,0,63,2,212,0,63,58,244,0,63,114,20,1,63,169,52,1,63,224,84,1,63,22,117,1,63,76,149,1,63,129,181,1,63,181,213,1,63,233,245,1,63,28,22,2,63,78,54,2,63,128,86,2,63,178,118,2,63,226,150,2,63,18,183,2,63,65,215,2,63,112,247,2,63,157,23,3,63,203,55,3,63,247,87,3,63,35,120,3,63,78,152,3,63,120,184,3,63,161,216,3,63,202,248,3,63,242,24,4,63,25,57,4,63,63,89,4,63,101,121,4,63,137,153,4,63,173,185,4,63,208,217,4,63,243,249,4,63,20,26,5,63,52,58,5,63,84,90,5,63,115,122,5,63,145,154,5,63,173,186,5,63,202,218,5,63,229,250,5,63,255,26,6,63,24,59,6,63,48,91,6,63,72,123,6,63,94,155,6,63,116,187,6,63,136,219,6,63,155,251,6,63,174,27,7,63,191,59,7,63,208,91,7,63,223,123,7,63,237,155,7,63,250,187,7,63,7,220,7,63,18,252,7,63,28,28,8,63,37,60,8,63,44,92,8,63,51,124,8,63,57,156,8,63,61,188,8,63,64,220,8,63,67,252,8,63,68,28,9,63,68,60,9,63,66,92,9,63,64,124,9,63,60,156,9,63,55,188,9,63,49,220,9,63,41,252,9,63,33,28,10,63,23,60,10,63,12,92,10,63,255,123,10,63,242,155,10,63,227,187,10,63,211,219,10,63,193,251,10,63,174,27,11,63,154,59,11,63,133,91,11,63,110,123,11,63,86,155,11,63,60,187,11,63,33,219,11,63,5,251,11,63,231,26,12,63,200,58,12,63,168,90,12,63,134,122,12,63,98,154,12,63,62,186,12,63,23,218,12,63,240,249,12,63,199,25,13,63,156,57,13,63,112,89,13,63,66,121,13,63,19,153,13,63,227,184,13,63,176,216,13,63,125,248,13,63,72,24,14,63,17,56,14,63,216,87,14,63,159,119,14,63,99,151,14,63,38,183,14,63,232,214,14,63,167,246,14,63,101,22,15,63,34,54,15,63,221,85,15,63,150,117,15,63,78,149,15,63,4,181,15,63,184,212,15,63,106,244,15,63,27,20,16,63,202,51,16,63,120,83,16,63,36,115,16,63,206,146,16,63,118,178,16,63,28,210,16,63,193,241,16,63,100,17,17,63,6,49,17,63,165,80,17,63,67,112,17,63,223,143,17,63,121,175,17,63,17,207,17,63,167,238,17,63,60,14,18,63,206,45,18,63,95,77,18,63,238,108,18,63,123,140,18,63,7,172,18,63,144,203,18,63,23,235,18,63,157,10,19,63,32,42,19,63,162,73,19,63,34,105,19,63,159,136,19,63,27,168,19,63,149,199,19,63,13,231,19,63,131,6,20,63,247,37,20,63,104,69,20,63,216,100,20,63,70,132,20,63,178,163,20,63,27,195,20,63,131,226,20,63,233,1,21,63,76,33,21,63,174,64,21,63,13,96,21,63,106,127,21,63,197,158,21,63,31,190,21,63,117,221,21,63,202,252,21,63,29,28,22,63,109,59,22,63,188,90,22,63,8,122,22,63,82,153,22,63,153,184,22,63,223,215,22,63,34,247,22,63,100,22,23,63,162,53,23,63,223,84,23,63,26,116,23,63,82,147,23,63,136,178,23,63,187,209,23,63,237,240,23,63,28,16,24,63,73,47,24,63,115,78,24,63,155,109,24,63,193,140,24,63,228,171,24,63,6,203,24,63,36,234,24,63,65,9,25,63,91,40,25,63,115,71,25,63,136,102,25,63,155,133,25,63,171,164,25,63,185,195,25,63,197,226,25,63,206,1,26,63,213,32,26,63,217,63,26,63,219,94,26,63,218,125,26,63,215,156,26,63,210,187,26,63,202,218,26,63,191,249,26,63,178,24,27,63,162,55,27,63,144,86,27,63,123,117,27,63,100,148,27,63,74,179,27,63,46,210,27,63,15,241,27,63,237,15,28,63,201,46,28,63,162,77,28,63,121,108,28,63,77,139,28,63,31,170,28,63,237,200,28,63,185,231,28,63,131,6,29,63,74,37,29,63,14,68,29,63,207,98,29,63,142,129,29,63,74,160,29,63,3,191,29,63,186,221,29,63,110,252,29,63,31,27,30,63,205,57,30,63,121,88,30,63,34,119,30,63,200,149,30,63,107,180,30,63,12,211,30,63,170,241,30,63,69,16,31,63,221,46,31,63,114,77,31,63,5,108,31,63,148,138,31,63,33,169,31,63,171,199,31,63,50,230,31,63,182,4,32,63,56,35,32,63,182,65,32,63,50,96,32,63,170,126,32,63,32,157,32,63,147,187,32,63,3,218,32,63,112,248,32,63,218,22,33,63,65,53,33,63,165,83,33,63,6,114,33,63,100,144,33,63,191,174,33,63,23,205,33,63,108,235,33,63,190,9,34,63,13,40,34,63,89,70,34,63,162,100,34,63,232,130,34,63,43,161,34,63,107,191,34,63,167,221,34,63,225,251,34,63,24,26,35,63,75,56,35,63,123,86,35,63,168,116,35,63,211,146,35,63,249,176,35,63,29,207,35,63,62,237,35,63,91,11,36,63,118,41,36,63,141,71,36,63,161,101,36,63,177,131,36,63,191,161,36,63,201,191,36,63,208,221,36,63,212,251,36,63,213,25,37,63,210,55,37,63,204,85,37,63,195,115,37,63,183,145,37,63,167,175,37,63,148,205,37,63,126,235,37,63,101,9,38,63,72,39,38,63,40,69,38,63,4,99,38,63,221,128,38,63,179,158,38,63,134,188,38,63,85,218,38,63,33,248,38,63,233,21,39,63,174,51,39,63,112,81,39,63,46,111,39,63,233,140,39,63,160,170,39,63,84,200,39,63,4,230,39,63,178,3,40,63,91,33,40,63,1,63,40,63,164,92,40,63,67,122,40,63,223,151,40,63,120,181,40,63,12,211,40,63,158,240,40,63,43,14,41,63,182,43,41,63,60,73,41,63,192,102,41,63,63,132,41,63,187,161,41,63,52,191,41,63,169,220,41,63,26,250,41,63,136,23,42,63,242,52,42,63,89,82,42,63,188,111,42,63,28,141,42,63,119,170,42,63,208,199,42,63,36,229,42,63,117,2,43,63,194,31,43,63,12,61,43,63,82,90,43,63,148,119,43,63,211,148,43,63,14,178,43,63,69,207,43,63,120,236,43,63,168,9,44,63,212,38,44,63,252,67,44,63,33,97,44,63,66,126,44,63,95,155,44,63,120,184,44,63,142,213,44,63,159,242,44,63,173,15,45,63,184,44,45,63,190,73,45,63,193,102,45,63,191,131,45,63,186,160,45,63,177,189,45,63,165,218,45,63,148,247,45,63,128,20,46,63,103,49,46,63,75,78,46,63,43,107,46,63,7,136,46,63,224,164,46,63,180,193,46,63,132,222,46,63,81,251,46,63,26,24,47,63,222,52,47,63,159,81,47,63,92,110,47,63,21,139,47,63,202,167,47,63,123,196,47,63,40,225,47,63,209,253,47,63,118,26,48,63,23,55,48,63,180,83,48,63,77,112,48,63,226,140,48,63,115,169,48,63,0,198,48,63,137,226,48,63,14,255,48,63,142,27,49,63,11,56,49,63,132,84,49,63,248,112,49,63,105,141,49,63,214,169,49,63,62,198,49,63,162,226,49,63,2,255,49,63,95,27,50,63,182,55,50,63,10,84,50,63,90,112,50,63,166,140,50,63,237,168,50,63,48,197,50,63,111,225,50,63,170,253,50,63,225,25,51,63,19,54,51,63,66,82,51,63,108,110,51,63,146,138,51,63,180,166,51,63,209,194,51,63,234,222,51,63,0,251,51,63,16,23,52,63,29,51,52,63,37,79,52,63,41,107,52,63,41,135,52,63,37,163,52,63,28,191,52,63,15,219,52,63,253,246,52,63,232,18,53,63,206,46,53,63,176,74,53,63,141,102,53,63,102,130,53,63,59,158,53,63,11,186,53,63,215,213,53,63,159,241,53,63,98,13,54,63,33,41,54,63,220,68,54,63,146,96,54,63,68,124,54,63,241,151,54,63,154,179,54,63,63,207,54,63,223,234,54,63,123,6,55,63,18,34,55,63,165,61,55,63,52,89,55,63,190,116,55,63,67,144,55,63,196,171,55,63,65,199,55,63,185,226,55,63,45,254,55,63,156,25,56,63,7,53,56,63,109,80,56,63,207,107,56,63,44,135,56,63,133,162,56,63,217,189,56,63,40,217,56,63,115,244,56,63,186,15,57,63,252,42,57,63,57,70,57,63,114,97,57,63,166,124,57,63,214,151,57,63,1,179,57,63,40,206,57,63,74,233,57,63,103,4,58,63,128,31,58,63,148,58,58,63,163,85,58,63,174,112,58,63,180,139,58,63,182,166,58,63,179,193,58,63,171,220,58,63,159,247,58,63,142,18,59,63,120,45,59,63,94,72,59,63,63,99,59,63,27,126,59,63,243,152,59,63,197,179,59,63,148,206,59,63,93,233,59,63,34,4,60,63,226,30,60,63,157,57,60,63,84,84,60,63,5,111,60,63,178,137,60,63,91,164,60,63,254,190,60,63,157,217,60,63,55,244,60,63,204,14,61,63,93,41,61,63,232,67,61,63,111,94,61,63,241,120,61,63,110,147,61,63,231,173,61,63,91,200,61,63,201,226,61,63,51,253,61,63,152,23,62,63,249,49,62,63,84,76,62,63,171,102,62,63,252,128,62,63,73,155,62,63,145,181,62,63,212,207,62,63,19,234,62,63,76,4,63,63,128,30,63,63,176,56,63,63,219,82,63,63,0,109,63,63,33,135,63,63,61,161,63,63,84,187,63,63,102,213,63,63,115,239,63,63,123,9,64,63,127,35,64,63,125,61,64,63,118,87,64,63,106,113,64,63,90,139,64,63,68,165,64,63,42,191,64,63,10,217,64,63,229,242,64,63,188,12,65,63,141,38,65,63,90,64,65,63,33,90,65,63,228,115,65,63,161,141,65,63,89,167,65,63,13,193,65,63,187,218,65,63,100,244,65,63,8,14,66,63,167,39,66,63,65,65,66,63,214,90,66,63,102,116,66,63,241,141,66,63,119,167,66,63,248,192,66,63,115,218,66,63,234,243,66,63,91,13,67,63,199,38,67,63,47,64,67,63,145,89,67,63,238,114,67,63,69,140,67,63,152,165,67,63,230,190,67,63,46,216,67,63,113,241,67,63,175,10,68,63,232,35,68,63,28,61,68,63,75,86,68,63,116,111,68,63,153,136,68,63,184,161,68,63,210,186,68,63,230,211,68,63,246,236,68,63,0,6,69,63,5,31,69,63,5,56,69,63,0,81,69,63,245,105,69,63,230,130,69,63,209,155,69,63,182,180,69,63,151,205,69,63,114,230,69,63,72,255,69,63,25,24,70,63,229,48,70,63,171,73,70,63,108,98,70,63,40,123,70,63,222,147,70,63,143,172,70,63,59,197,70,63,226,221,70,63,131,246,70,63,31,15,71,63,182,39,71,63,71,64,71,63,211,88,71,63,90,113,71,63,220,137,71,63,88,162,71,63,207,186,71,63,64,211,71,63,172,235,71,63,19,4,72,63,116,28,72,63,209,52,72,63,39,77,72,63,121,101,72,63,197,125,72,63,11,150,72,63,77,174,72,63,137,198,72,63,191,222,72,63,240,246,72,63,28,15,73,63,66,39,73,63,99,63,73,63,127,87,73,63,149,111,73,63,166,135,73,63,177,159,73,63,183,183,73,63,183,207,73,63,178,231,73,63,168,255,73,63,152,23,74,63,131,47,74,63,104,71,74,63,72,95,74,63,34,119,74,63,247,142,74,63,199,166,74,63,145,190,74,63,85,214,74,63,20,238,74,63,206,5,75,63,130,29,75,63,49,53,75,63,218,76,75,63,126,100,75,63,28,124,75,63,181,147,75,63,72,171,75,63,213,194,75,63,93,218,75,63,224,241,75,63,93,9,76,63,213,32,76,63,71,56,76,63,179,79,76,63,26,103,76,63,124,126,76,63,216,149,76,63,46,173,76,63,127,196,76,63,202,219,76,63,16,243,76,63,80,10,77,63,139,33,77,63,192,56,77,63,240,79,77,63,26,103,77,63,62,126,77,63,93,149,77,63,118,172,77,63,137,195,77,63,151,218,77,63,160,241,77,63,163,8,78,63,160,31,78,63,151,54,78,63,137,77,78,63,118,100,78,63,93,123,78,63,62,146,78,63,25,169,78,63,239,191,78,63,192,214,78,63,138,237,78,63,79,4,79,63,15,27,79,63,201,49,79,63,125,72,79,63,43,95,79,63,212,117,79,63,119,140,79,63,21,163,79,63,172,185,79,63,63,208,79,63,203,230,79,63,82,253,79,63,211,19,80,63,79,42,80,63,197,64,80,63,53,87,80,63,159,109,80,63,4,132,80,63,99,154,80,63,189,176,80,63,16,199,80,63,94,221,80,63,167,243,80,63,233,9,81,63,38,32,81,63,93,54,81,63,143,76,81,63,187,98,81,63,225,120,81,63,1,143,81,63,28,165,81,63,48,187,81,63,64,209,81,63,73,231,81,63,77,253,81,63,75,19,82,63,67,41,82,63,53,63,82,63,34,85,82,63,9,107,82,63,234,128,82,63,198,150,82,63,155,172,82,63,107,194,82,63,53,216,82,63,250,237,82,63,185,3,83,63,113,25,83,63,37,47,83,63,210,68,83,63,121,90,83,63,27,112,83,63,183,133,83,63,77,155,83,63,222,176,83,63,104,198,83,63,237,219,83,63,108,241,83,63,230,6,84,63,89,28,84,63,199,49,84,63,46,71,84,63,145,92,84,63,237,113,84,63,67,135,84,63,148,156,84,63,223,177,84,63,35,199,84,63,99,220,84,63,156,241,84,63,207,6,85,63,253,27,85,63,37,49,85,63,71,70,85,63,99,91,85,63,121,112,85,63,138,133,85,63,149,154,85,63,153,175,85,63,152,196,85,63,146,217,85,63,133,238,85,63,114,3,86,63,90,24,86,63,60,45,86,63,24,66,86,63,238,86,86,63,190,107,86,63,136,128,86,63,76,149,86,63,11,170,86,63,196,190,86,63,118,211,86,63,35,232,86,63,203,252,86,63,108,17,87,63,7,38,87,63,156,58,87,63,44,79,87,63,182,99,87,63,58,120,87,63,183,140,87,63,47,161,87,63,162,181,87,63,14,202,87,63,116,222,87,63,213,242,87,63,47,7,88,63,132,27,88,63,211,47,88,63,28,68,88,63,95,88,88,63,156,108,88,63,211,128,88,63,4,149,88,63,47,169,88,63,85,189,88,63,116,209,88,63,142,229,88,63,162,249,88,63,175,13,89,63,183,33,89,63,185,53,89,63,181,73,89,63,171,93,89,63,155,113,89,63,134,133,89,63,106,153,89,63,72,173,89,63,33,193,89,63,243,212,89,63,192,232,89,63,135,252,89,63,71,16,90,63,2,36,90,63,183,55,90,63,102,75,90,63,15,95,90,63,178,114,90,63,79,134,90,63,230,153,90,63,119,173,90,63,3,193,90,63,136,212,90,63,7,232,90,63,129,251,90,63,244,14,91,63,98,34,91,63,201,53,91,63,43,73,91,63,135,92,91,63,220,111,91,63,44,131,91,63,118,150,91,63,186,169,91,63,248,188,91,63,47,208,91,63,97,227,91,63,141,246,91,63,179,9,92,63,212,28,92,63,238,47,92,63,2,67,92,63,16,86,92,63,24,105,92,63,26,124,92,63,23,143,92,63,13,162,92,63,253,180,92,63,232,199,92,63,204,218,92,63,171,237,92,63,131,0,93,63,86,19,93,63,34,38,93,63,233,56,93,63,169,75,93,63,100,94,93,63,24,113,93,63,199,131,93,63,112,150,93,63,18,169,93,63,175,187,93,63,70,206,93,63,215,224,93,63,97,243,93,63,230,5,94,63,101,24,94,63,222,42,94,63,81,61,94,63,190,79,94,63,36,98,94,63,133,116,94,63,224,134,94,63,53,153,94,63,132,171,94,63,205,189,94,63,16,208,94,63,77,226,94,63,132,244,94,63,181,6,95,63,224,24,95,63,5,43,95,63,36,61,95,63,61,79,95,63,80,97,95,63,93,115,95,63,101,133,95,63,102,151,95,63,97,169,95,63,86,187,95,63,69,205,95,63,46,223,95,63,18,241,95,63,239,2,96,63,198,20,96,63,151,38,96,63,98,56,96,63,40,74,96,63,231,91,96,63,160,109,96,63,84,127,96,63,1,145,96,63,168,162,96,63,73,180,96,63,229,197,96,63,122,215,96,63,10,233,96,63,147,250,96,63,22,12,97,63,148,29,97,63,11,47,97,63,125,64,97,63,232,81,97,63,77,99,97,63,173,116,97,63,6,134,97,63,90,151,97,63,167,168,97,63,239,185,97,63,48,203,97,63,108,220,97,63,162,237,97,63,209,254,97,63,251,15,98,63,30,33,98,63,60,50,98,63,84,67,98,63,101,84,98,63,113,101,98,63,119,118,98,63,119,135,98,63,112,152,98,63,100,169,98,63,82,186,98,63,58,203,98,63,28,220,98,63,247,236,98,63,205,253,98,63,157,14,99,63,103,31,99,63,43,48,99,63,233,64,99,63,161,81,99,63,83,98,99,63,255,114,99,63,165,131,99,63,69,148,99,63,224,164,99,63,116,181,99,63,2,198,99,63,138,214,99,63,13,231,99,63,137,247,99,63,255,7,100,63,112,24,100,63,218,40,100,63,62,57,100,63,157,73,100,63,246,89,100,63,72,106,100,63,149,122,100,63,219,138,100,63,28,155,100,63,87,171,100,63,140,187,100,63,186,203,100,63,227,219,100,63,6,236,100,63,35,252,100,63,58,12,101,63,75,28,101,63,86,44,101,63,91,60,101,63,91,76,101,63,84,92,101,63,71,108,101,63,53,124,101,63,28,140,101,63,254,155,101,63,217,171,101,63,175,187,101,63,126,203,101,63,72,219,101,63,12,235,101,63,202,250,101,63,130,10,102,63,52,26,102,63,224,41,102,63,134,57,102,63,38,73,102,63,193,88,102,63,85,104,102,63,227,119,102,63,108,135,102,63,238,150,102,63,107,166,102,63,226,181,102,63,83,197,102,63,190,212,102,63,35,228,102,63,130,243,102,63,219,2,103,63,46,18,103,63,124,33,103,63,195,48,103,63,5,64,103,63,64,79,103,63,118,94,103,63,166,109,103,63,208,124,103,63,244,139,103,63,18,155,103,63,42,170,103,63,61,185,103,63,73,200,103,63,80,215,103,63,80,230,103,63,75,245,103,63,64,4,104,63,47,19,104,63,24,34,104,63,251,48,104,63,217,63,104,63,176,78,104,63,130,93,104,63,78,108,104,63,20,123,104,63,212,137,104,63,142,152,104,63,66,167,104,63,240,181,104,63,153,196,104,63,60,211,104,63,217,225,104,63,112,240,104,63,1,255,104,63,140,13,105,63,17,28,105,63,145,42,105,63,11,57,105,63,127,71,105,63,237,85,105,63,85,100,105,63,183,114,105,63,20,129,105,63,106,143,105,63,187,157,105,63,6,172,105,63,75,186,105,63,139,200,105,63,196,214,105,63,248,228,105,63,38,243,105,63,78,1,106,63,112,15,106,63,141,29,106,63,163,43,106,63,180,57,106,63,191,71,106,63,196,85,106,63,196,99,106,63,189,113,106,63,177,127,106,63], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+20480);
/* memory initializer */ allocate([159,141,106,63,135,155,106,63,106,169,106,63,70,183,106,63,29,197,106,63,238,210,106,63,186,224,106,63,127,238,106,63,63,252,106,63,249,9,107,63,173,23,107,63,91,37,107,63,4,51,107,63,167,64,107,63,68,78,107,63,219,91,107,63,109,105,107,63,249,118,107,63,127,132,107,63,255,145,107,63,122,159,107,63,238,172,107,63,94,186,107,63,199,199,107,63,42,213,107,63,136,226,107,63,224,239,107,63,51,253,107,63,128,10,108,63,198,23,108,63,8,37,108,63,67,50,108,63,121,63,108,63,169,76,108,63,211,89,108,63,248,102,108,63,23,116,108,63,48,129,108,63,68,142,108,63,82,155,108,63,90,168,108,63,92,181,108,63,89,194,108,63,80,207,108,63,65,220,108,63,45,233,108,63,19,246,108,63,243,2,109,63,206,15,109,63,163,28,109,63,114,41,109,63,60,54,109,63,0,67,109,63,190,79,109,63,119,92,109,63,42,105,109,63,215,117,109,63,127,130,109,63,33,143,109,63,189,155,109,63,84,168,109,63,229,180,109,63,113,193,109,63,247,205,109,63,119,218,109,63,242,230,109,63,103,243,109,63,214,255,109,63,64,12,110,63,164,24,110,63,3,37,110,63,91,49,110,63,175,61,110,63,253,73,110,63,69,86,110,63,135,98,110,63,196,110,110,63,252,122,110,63,45,135,110,63,90,147,110,63,128,159,110,63,161,171,110,63,189,183,110,63,211,195,110,63,227,207,110,63,238,219,110,63,243,231,110,63,243,243,110,63,237,255,110,63,226,11,111,63,209,23,111,63,186,35,111,63,158,47,111,63,125,59,111,63,85,71,111,63,41,83,111,63,247,94,111,63,191,106,111,63,130,118,111,63,63,130,111,63,247,141,111,63,169,153,111,63,86,165,111,63,253,176,111,63,159,188,111,63,59,200,111,63,210,211,111,63,99,223,111,63,239,234,111,63,117,246,111,63,246,1,112,63,114,13,112,63,231,24,112,63,88,36,112,63,195,47,112,63,40,59,112,63,137,70,112,63,227,81,112,63,56,93,112,63,136,104,112,63,210,115,112,63,23,127,112,63,87,138,112,63,145,149,112,63,197,160,112,63,244,171,112,63,30,183,112,63,66,194,112,63,97,205,112,63,123,216,112,63,143,227,112,63,157,238,112,63,167,249,112,63,171,4,113,63,169,15,113,63,162,26,113,63,150,37,113,63,132,48,113,63,109,59,113,63,81,70,113,63,47,81,113,63,8,92,113,63,219,102,113,63,170,113,113,63,114,124,113,63,54,135,113,63,244,145,113,63,173,156,113,63,96,167,113,63,14,178,113,63,183,188,113,63,91,199,113,63,249,209,113,63,146,220,113,63,37,231,113,63,179,241,113,63,60,252,113,63,192,6,114,63,62,17,114,63,183,27,114,63,43,38,114,63,154,48,114,63,3,59,114,63,103,69,114,63,197,79,114,63,31,90,114,63,115,100,114,63,194,110,114,63,11,121,114,63,79,131,114,63,143,141,114,63,200,151,114,63,253,161,114,63,44,172,114,63,87,182,114,63,123,192,114,63,155,202,114,63,182,212,114,63,203,222,114,63,219,232,114,63,230,242,114,63,235,252,114,63,236,6,115,63,231,16,115,63,221,26,115,63,206,36,115,63,186,46,115,63,160,56,115,63,130,66,115,63,94,76,115,63,53,86,115,63,7,96,115,63,212,105,115,63,155,115,115,63,94,125,115,63,27,135,115,63,211,144,115,63,134,154,115,63,52,164,115,63,221,173,115,63,128,183,115,63,31,193,115,63,184,202,115,63,77,212,115,63,220,221,115,63,102,231,115,63,235,240,115,63,107,250,115,63,230,3,116,63,92,13,116,63,204,22,116,63,56,32,116,63,159,41,116,63,0,51,116,63,93,60,116,63,180,69,116,63,6,79,116,63,84,88,116,63,156,97,116,63,223,106,116,63,29,116,116,63,87,125,116,63,139,134,116,63,186,143,116,63,228,152,116,63,9,162,116,63,41,171,116,63,68,180,116,63,91,189,116,63,108,198,116,63,120,207,116,63,127,216,116,63,129,225,116,63,127,234,116,63,119,243,116,63,106,252,116,63,89,5,117,63,66,14,117,63,38,23,117,63,6,32,117,63,225,40,117,63,182,49,117,63,135,58,117,63,83,67,117,63,26,76,117,63,220,84,117,63,153,93,117,63,81,102,117,63,4,111,117,63,179,119,117,63,92,128,117,63,1,137,117,63,160,145,117,63,59,154,117,63,209,162,117,63,98,171,117,63,239,179,117,63,118,188,117,63,249,196,117,63,118,205,117,63,239,213,117,63,99,222,117,63,210,230,117,63,61,239,117,63,162,247,117,63,3,0,118,63,95,8,118,63,182,16,118,63,8,25,118,63,86,33,118,63,159,41,118,63,227,49,118,63,34,58,118,63,92,66,118,63,146,74,118,63,195,82,118,63,239,90,118,63,22,99,118,63,57,107,118,63,86,115,118,63,112,123,118,63,132,131,118,63,148,139,118,63,158,147,118,63,165,155,118,63,166,163,118,63,163,171,118,63,155,179,118,63,142,187,118,63,125,195,118,63,103,203,118,63,76,211,118,63,45,219,118,63,9,227,118,63,224,234,118,63,178,242,118,63,128,250,118,63,74,2,119,63,14,10,119,63,206,17,119,63,137,25,119,63,64,33,119,63,242,40,119,63,160,48,119,63,72,56,119,63,237,63,119,63,140,71,119,63,39,79,119,63,190,86,119,63,79,94,119,63,220,101,119,63,101,109,119,63,233,116,119,63,105,124,119,63,228,131,119,63,90,139,119,63,204,146,119,63,57,154,119,63,162,161,119,63,6,169,119,63,101,176,119,63,192,183,119,63,23,191,119,63,105,198,119,63,182,205,119,63,255,212,119,63,68,220,119,63,132,227,119,63,191,234,119,63,246,241,119,63,41,249,119,63,87,0,120,63,129,7,120,63,166,14,120,63,198,21,120,63,227,28,120,63,250,35,120,63,14,43,120,63,28,50,120,63,39,57,120,63,45,64,120,63,46,71,120,63,44,78,120,63,36,85,120,63,25,92,120,63,9,99,120,63,244,105,120,63,219,112,120,63,190,119,120,63,156,126,120,63,118,133,120,63,76,140,120,63,29,147,120,63,234,153,120,63,179,160,120,63,119,167,120,63,55,174,120,63,242,180,120,63,169,187,120,63,92,194,120,63,11,201,120,63,181,207,120,63,91,214,120,63,252,220,120,63,154,227,120,63,51,234,120,63,199,240,120,63,88,247,120,63,228,253,120,63,108,4,121,63,240,10,121,63,111,17,121,63,234,23,121,63,97,30,121,63,211,36,121,63,66,43,121,63,172,49,121,63,18,56,121,63,116,62,121,63,209,68,121,63,42,75,121,63,127,81,121,63,208,87,121,63,29,94,121,63,101,100,121,63,170,106,121,63,234,112,121,63,38,119,121,63,93,125,121,63,145,131,121,63,193,137,121,63,236,143,121,63,19,150,121,63,54,156,121,63,85,162,121,63,112,168,121,63,134,174,121,63,153,180,121,63,167,186,121,63,178,192,121,63,184,198,121,63,186,204,121,63,184,210,121,63,178,216,121,63,168,222,121,63,154,228,121,63,135,234,121,63,113,240,121,63,87,246,121,63,56,252,121,63,22,2,122,63,239,7,122,63,197,13,122,63,150,19,122,63,100,25,122,63,45,31,122,63,243,36,122,63,180,42,122,63,113,48,122,63,43,54,122,63,224,59,122,63,146,65,122,63,63,71,122,63,233,76,122,63,142,82,122,63,48,88,122,63,206,93,122,63,103,99,122,63,253,104,122,63,143,110,122,63,29,116,122,63,167,121,122,63,45,127,122,63,175,132,122,63,45,138,122,63,168,143,122,63,30,149,122,63,145,154,122,63,255,159,122,63,106,165,122,63,209,170,122,63,52,176,122,63,147,181,122,63,239,186,122,63,70,192,122,63,154,197,122,63,234,202,122,63,54,208,122,63,126,213,122,63,194,218,122,63,3,224,122,63,64,229,122,63,121,234,122,63,174,239,122,63,223,244,122,63,13,250,122,63,55,255,122,63,93,4,123,63,127,9,123,63,157,14,123,63,184,19,123,63,207,24,123,63,227,29,123,63,242,34,123,63,254,39,123,63,6,45,123,63,10,50,123,63,11,55,123,63,8,60,123,63,1,65,123,63,247,69,123,63,233,74,123,63,215,79,123,63,193,84,123,63,168,89,123,63,139,94,123,63,107,99,123,63,71,104,123,63,31,109,123,63,243,113,123,63,196,118,123,63,146,123,123,63,91,128,123,63,33,133,123,63,228,137,123,63,163,142,123,63,94,147,123,63,22,152,123,63,202,156,123,63,122,161,123,63,39,166,123,63,208,170,123,63,118,175,123,63,24,180,123,63,183,184,123,63,82,189,123,63,233,193,123,63,125,198,123,63,14,203,123,63,155,207,123,63,36,212,123,63,170,216,123,63,45,221,123,63,172,225,123,63,39,230,123,63,159,234,123,63,19,239,123,63,132,243,123,63,242,247,123,63,92,252,123,63,195,0,124,63,38,5,124,63,133,9,124,63,226,13,124,63,58,18,124,63,144,22,124,63,226,26,124,63,48,31,124,63,123,35,124,63,195,39,124,63,7,44,124,63,72,48,124,63,134,52,124,63,192,56,124,63,247,60,124,63,42,65,124,63,90,69,124,63,135,73,124,63,176,77,124,63,214,81,124,63,249,85,124,63,24,90,124,63,52,94,124,63,77,98,124,63,98,102,124,63,116,106,124,63,131,110,124,63,142,114,124,63,150,118,124,63,155,122,124,63,157,126,124,63,155,130,124,63,150,134,124,63,142,138,124,63,130,142,124,63,116,146,124,63,98,150,124,63,77,154,124,63,52,158,124,63,24,162,124,63,249,165,124,63,215,169,124,63,178,173,124,63,137,177,124,63,94,181,124,63,47,185,124,63,253,188,124,63,199,192,124,63,143,196,124,63,83,200,124,63,20,204,124,63,211,207,124,63,141,211,124,63,69,215,124,63,250,218,124,63,171,222,124,63,90,226,124,63,5,230,124,63,173,233,124,63,82,237,124,63,244,240,124,63,147,244,124,63,46,248,124,63,199,251,124,63,93,255,124,63,239,2,125,63,127,6,125,63,11,10,125,63,148,13,125,63,27,17,125,63,158,20,125,63,30,24,125,63,155,27,125,63,21,31,125,63,140,34,125,63,0,38,125,63,114,41,125,63,224,44,125,63,75,48,125,63,179,51,125,63,24,55,125,63,122,58,125,63,217,61,125,63,54,65,125,63,143,68,125,63,229,71,125,63,56,75,125,63,137,78,125,63,214,81,125,63,33,85,125,63,104,88,125,63,173,91,125,63,239,94,125,63,46,98,125,63,106,101,125,63,163,104,125,63,217,107,125,63,12,111,125,63,61,114,125,63,106,117,125,63,149,120,125,63,189,123,125,63,226,126,125,63,4,130,125,63,36,133,125,63,64,136,125,63,90,139,125,63,112,142,125,63,133,145,125,63,150,148,125,63,164,151,125,63,176,154,125,63,185,157,125,63,191,160,125,63,194,163,125,63,194,166,125,63,192,169,125,63,187,172,125,63,179,175,125,63,168,178,125,63,155,181,125,63,139,184,125,63,120,187,125,63,99,190,125,63,74,193,125,63,48,196,125,63,18,199,125,63,241,201,125,63,206,204,125,63,169,207,125,63,128,210,125,63,85,213,125,63,39,216,125,63,247,218,125,63,196,221,125,63,142,224,125,63,85,227,125,63,26,230,125,63,220,232,125,63,156,235,125,63,89,238,125,63,19,241,125,63,203,243,125,63,128,246,125,63,51,249,125,63,227,251,125,63,144,254,125,63,59,1,126,63,227,3,126,63,137,6,126,63,44,9,126,63,204,11,126,63,106,14,126,63,6,17,126,63,158,19,126,63,53,22,126,63,200,24,126,63,90,27,126,63,232,29,126,63,116,32,126,63,254,34,126,63,133,37,126,63,10,40,126,63,140,42,126,63,12,45,126,63,137,47,126,63,4,50,126,63,124,52,126,63,242,54,126,63,101,57,126,63,214,59,126,63,68,62,126,63,176,64,126,63,26,67,126,63,129,69,126,63,230,71,126,63,72,74,126,63,168,76,126,63,5,79,126,63,96,81,126,63,185,83,126,63,15,86,126,63,99,88,126,63,181,90,126,63,4,93,126,63,81,95,126,63,155,97,126,63,227,99,126,63,41,102,126,63,108,104,126,63,173,106,126,63,236,108,126,63,40,111,126,63,98,113,126,63,154,115,126,63,208,117,126,63,3,120,126,63,51,122,126,63,98,124,126,63,142,126,126,63,184,128,126,63,224,130,126,63,5,133,126,63,40,135,126,63,73,137,126,63,104,139,126,63,132,141,126,63,159,143,126,63,183,145,126,63,204,147,126,63,224,149,126,63,241,151,126,63,0,154,126,63,13,156,126,63,24,158,126,63,32,160,126,63,38,162,126,63,42,164,126,63,44,166,126,63,44,168,126,63,41,170,126,63,37,172,126,63,30,174,126,63,21,176,126,63,10,178,126,63,253,179,126,63,238,181,126,63,220,183,126,63,201,185,126,63,179,187,126,63,155,189,126,63,129,191,126,63,101,193,126,63,71,195,126,63,39,197,126,63,5,199,126,63,224,200,126,63,186,202,126,63,145,204,126,63,103,206,126,63,58,208,126,63,12,210,126,63,219,211,126,63,168,213,126,63,115,215,126,63,61,217,126,63,4,219,126,63,201,220,126,63,140,222,126,63,77,224,126,63,12,226,126,63,202,227,126,63,133,229,126,63,62,231,126,63,245,232,126,63,170,234,126,63,94,236,126,63,15,238,126,63,190,239,126,63,108,241,126,63,23,243,126,63,193,244,126,63,104,246,126,63,14,248,126,63,178,249,126,63,84,251,126,63,243,252,126,63,145,254,126,63,46,0,127,63,200,1,127,63,96,3,127,63,247,4,127,63,139,6,127,63,30,8,127,63,175,9,127,63,62,11,127,63,203,12,127,63,86,14,127,63,223,15,127,63,103,17,127,63,237,18,127,63,112,20,127,63,242,21,127,63,115,23,127,63,241,24,127,63,110,26,127,63,233,27,127,63,98,29,127,63,217,30,127,63,78,32,127,63,194,33,127,63,52,35,127,63,164,36,127,63,18,38,127,63,127,39,127,63,234,40,127,63,83,42,127,63,186,43,127,63,32,45,127,63,131,46,127,63,230,47,127,63,70,49,127,63,165,50,127,63,2,52,127,63,93,53,127,63,182,54,127,63,14,56,127,63,100,57,127,63,185,58,127,63,12,60,127,63,93,61,127,63,172,62,127,63,250,63,127,63,70,65,127,63,145,66,127,63,217,67,127,63,33,69,127,63,102,70,127,63,170,71,127,63,236,72,127,63,45,74,127,63,108,75,127,63,169,76,127,63,229,77,127,63,31,79,127,63,88,80,127,63,143,81,127,63,196,82,127,63,248,83,127,63,42,85,127,63,91,86,127,63,138,87,127,63,184,88,127,63,228,89,127,63,14,91,127,63,55,92,127,63,94,93,127,63,132,94,127,63,169,95,127,63,203,96,127,63,237,97,127,63,12,99,127,63,42,100,127,63,71,101,127,63,98,102,127,63,124,103,127,63,148,104,127,63,171,105,127,63,192,106,127,63,212,107,127,63,230,108,127,63,247,109,127,63,6,111,127,63,20,112,127,63,33,113,127,63,44,114,127,63,53,115,127,63,61,116,127,63,68,117,127,63,73,118,127,63,77,119,127,63,79,120,127,63,80,121,127,63,80,122,127,63,78,123,127,63,75,124,127,63,70,125,127,63,64,126,127,63,57,127,127,63,48,128,127,63,38,129,127,63,27,130,127,63,14,131,127,63,0,132,127,63,240,132,127,63,223,133,127,63,205,134,127,63,185,135,127,63,164,136,127,63,142,137,127,63,118,138,127,63,93,139,127,63,67,140,127,63,40,141,127,63,11,142,127,63,237,142,127,63,205,143,127,63,173,144,127,63,139,145,127,63,103,146,127,63,67,147,127,63,29,148,127,63,246,148,127,63,205,149,127,63,164,150,127,63,121,151,127,63,77,152,127,63,31,153,127,63,241,153,127,63,193,154,127,63,144,155,127,63,93,156,127,63,42,157,127,63,245,157,127,63,191,158,127,63,136,159,127,63,79,160,127,63,22,161,127,63,219,161,127,63,159,162,127,63,98,163,127,63,36,164,127,63,228,164,127,63,163,165,127,63,98,166,127,63,31,167,127,63,219,167,127,63,149,168,127,63,79,169,127,63,7,170,127,63,190,170,127,63,117,171,127,63,42,172,127,63,221,172,127,63,144,173,127,63,66,174,127,63,242,174,127,63,162,175,127,63,80,176,127,63,253,176,127,63,169,177,127,63,85,178,127,63,254,178,127,63,167,179,127,63,79,180,127,63,246,180,127,63,156,181,127,63,64,182,127,63,228,182,127,63,134,183,127,63,40,184,127,63,200,184,127,63,103,185,127,63,6,186,127,63,163,186,127,63,63,187,127,63,219,187,127,63,117,188,127,63,14,189,127,63,166,189,127,63,61,190,127,63,212,190,127,63,105,191,127,63,253,191,127,63,144,192,127,63,34,193,127,63,180,193,127,63,68,194,127,63,211,194,127,63,98,195,127,63,239,195,127,63,123,196,127,63,7,197,127,63,145,197,127,63,27,198,127,63,163,198,127,63,43,199,127,63,178,199,127,63,56,200,127,63,189,200,127,63,65,201,127,63,196,201,127,63,70,202,127,63,199,202,127,63,71,203,127,63,199,203,127,63,69,204,127,63,195,204,127,63,64,205,127,63,187,205,127,63,54,206,127,63,177,206,127,63,42,207,127,63,162,207,127,63,26,208,127,63,144,208,127,63,6,209,127,63,123,209,127,63,239,209,127,63,98,210,127,63,213,210,127,63,70,211,127,63,183,211,127,63,39,212,127,63,150,212,127,63,4,213,127,63,114,213,127,63,222,213,127,63,74,214,127,63,181,214,127,63,32,215,127,63,137,215,127,63,242,215,127,63,89,216,127,63,192,216,127,63,39,217,127,63,140,217,127,63,241,217,127,63,85,218,127,63,184,218,127,63,27,219,127,63,124,219,127,63,221,219,127,63,61,220,127,63,157,220,127,63,251,220,127,63,89,221,127,63,183,221,127,63,19,222,127,63,111,222,127,63,202,222,127,63,36,223,127,63,126,223,127,63,215,223,127,63,47,224,127,63,134,224,127,63,221,224,127,63,51,225,127,63,137,225,127,63,221,225,127,63,49,226,127,63,133,226,127,63,215,226,127,63,41,227,127,63,122,227,127,63,203,227,127,63,27,228,127,63,106,228,127,63,185,228,127,63,7,229,127,63,84,229,127,63,161,229,127,63,237,229,127,63,56,230,127,63,131,230,127,63,205,230,127,63,23,231,127,63,96,231,127,63,168,231,127,63,239,231,127,63,54,232,127,63,125,232,127,63,195,232,127,63,8,233,127,63,76,233,127,63,144,233,127,63,212,233,127,63,23,234,127,63,89,234,127,63,154,234,127,63,219,234,127,63,28,235,127,63,92,235,127,63,155,235,127,63,218,235,127,63,24,236,127,63,86,236,127,63,147,236,127,63,207,236,127,63,11,237,127,63,71,237,127,63,130,237,127,63,188,237,127,63,246,237,127,63,47,238,127,63,104,238,127,63,160,238,127,63,216,238,127,63,15,239,127,63,69,239,127,63,123,239,127,63,177,239,127,63,230,239,127,63,27,240,127,63,79,240,127,63,130,240,127,63,182,240,127,63,232,240,127,63,26,241,127,63,76,241,127,63,125,241,127,63,174,241,127,63,222,241,127,63,14,242,127,63,61,242,127,63,108,242,127,63,154,242,127,63,200,242,127,63,245,242,127,63,34,243,127,63,79,243,127,63,123,243,127,63,166,243,127,63,209,243,127,63,252,243,127,63,38,244,127,63,80,244,127,63,121,244,127,63,162,244,127,63,203,244,127,63,243,244,127,63,27,245,127,63,66,245,127,63,105,245,127,63,143,245,127,63,181,245,127,63,219,245,127,63,0,246,127,63,37,246,127,63,73,246,127,63,109,246,127,63,145,246,127,63,180,246,127,63,215,246,127,63,250,246,127,63,28,247,127,63,62,247,127,63,95,247,127,63,128,247,127,63,160,247,127,63,193,247,127,63,225,247,127,63,0,248,127,63,31,248,127,63,62,248,127,63,93,248,127,63,123,248,127,63,152,248,127,63,182,248,127,63,211,248,127,63,240,248,127,63,12,249,127,63,40,249,127,63,68,249,127,63,95,249,127,63,122,249,127,63,149,249,127,63,175,249,127,63,202,249,127,63,227,249,127,63,253,249,127,63,22,250,127,63,47,250,127,63,71,250,127,63,96,250,127,63,120,250,127,63,143,250,127,63,166,250,127,63,190,250,127,63,212,250,127,63,235,250,127,63,1,251,127,63,23,251,127,63,44,251,127,63,66,251,127,63,87,251,127,63,108,251,127,63,128,251,127,63,148,251,127,63,168,251,127,63,188,251,127,63,208,251,127,63,227,251,127,63,246,251,127,63,8,252,127,63,27,252,127,63,45,252,127,63,63,252,127,63,81,252,127,63,98,252,127,63,115,252,127,63,132,252,127,63,149,252,127,63,165,252,127,63,182,252,127,63,198,252,127,63,213,252,127,63,229,252,127,63,244,252,127,63,3,253,127,63,18,253,127,63,33,253,127,63,47,253,127,63,62,253,127,63,76,253,127,63,89,253,127,63,103,253,127,63,116,253,127,63,130,253,127,63,143,253,127,63,155,253,127,63,168,253,127,63,181,253,127,63,193,253,127,63,205,253,127,63,217,253,127,63,228,253,127,63,240,253,127,63,251,253,127,63,6,254,127,63,17,254,127,63,28,254,127,63,38,254,127,63,49,254,127,63,59,254,127,63,69,254,127,63,79,254,127,63,89,254,127,63,98,254,127,63,108,254,127,63,117,254,127,63,126,254,127,63,135,254,127,63,144,254,127,63,152,254,127,63,161,254,127,63,169,254,127,63,177,254,127,63,185,254,127,63,193,254,127,63,201,254,127,63,208,254,127,63,216,254,127,63,223,254,127,63,230,254,127,63,237,254,127,63,244,254,127,63,251,254,127,63,2,255,127,63,8,255,127,63,14,255,127,63,21,255,127,63,27,255,127,63,33,255,127,63,39,255,127,63,45,255,127,63,50,255,127,63,56,255,127,63,61,255,127,63,67,255,127,63,72,255,127,63,77,255,127,63,82,255,127,63,87,255,127,63,92,255,127,63,96,255,127,63,101,255,127,63,105,255,127,63,110,255,127,63,114,255,127,63,118,255,127,63,122,255,127,63,126,255,127,63,130,255,127,63,134,255,127,63,138,255,127,63,142,255,127,63,145,255,127,63,149,255,127,63,152,255,127,63,155,255,127,63,159,255,127,63,162,255,127,63,165,255,127,63,168,255,127,63,171,255,127,63,174,255,127,63,176,255,127,63,179,255,127,63,182,255,127,63,184,255,127,63,187,255,127,63,189,255,127,63,192,255,127,63,194,255,127,63,196,255,127,63,198,255,127,63,201,255,127,63,203,255,127,63,205,255,127,63,207,255,127,63,209,255,127,63,210,255,127,63,212,255,127,63,214,255,127,63,216,255,127,63,217,255,127,63,219,255,127,63,220,255,127,63,222,255,127,63,223,255,127,63,225,255,127,63,226,255,127,63,227,255,127,63,229,255,127,63,230,255,127,63,231,255,127,63,232,255,127,63,233,255,127,63,234,255,127,63,235,255,127,63,236,255,127,63,237,255,127,63,238,255,127,63,239,255,127,63,240,255,127,63,241,255,127,63,241,255,127,63,242,255,127,63,243,255,127,63,244,255,127,63,244,255,127,63,245,255,127,63,246,255,127,63,246,255,127,63,247,255,127,63,247,255,127,63,248,255,127,63,248,255,127,63,249,255,127,63,249,255,127,63,250,255,127,63,250,255,127,63,250,255,127,63,251,255,127,63,251,255,127,63,251,255,127,63,252,255,127,63,252,255,127,63,252,255,127,63,253,255,127,63,253,255,127,63,253,255,127,63,253,255,127,63,254,255,127,63,254,255,127,63,254,255,127,63,254,255,127,63,254,255,127,63,254,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,255,255,127,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,76,194,0,0,80,194,0,0,84,194,0,0,88,194,0,0,92,194,0,0,96,194,0,0,100,194,0,0,104,194,0,0,108,194,0,0,112,194,0,0,116,194,0,0,120,194,0,0,124,194,0,0,128,194,0,0,130,194,0,0,132,194,0,0,134,194,0,0,136,194,0,0,138,194,0,0,140,194,0,0,142,194,0,0,144,194,0,0,146,194,0,0,148,194,0,0,150,194,0,0,152,194,0,0,154,194,0,0,156,194,0,0,160,194,0,0,162,194,0,0,164,194,0,0,166,194,0,0,168,194,0,0,170,194,0,0,172,194,0,0,174,194,0,0,176,194,0,0,176,194,0,0,178,194,0,0,178,194,0,0,180,194,0,0,182,194,0,0,182,194,0,0,184,194,0,0,186,194,0,0,188,194,0,0,190,194,0,0,192,194,0,0,192,194,0,0,194,194,0,0,196,194,0,0,196,194,0,0,198,194,0,0,198,194,0,0,200,194,0,0,200,194,0,0,202,194,0,0,204,194,0,0,206,194,0,0,208,194,0,0,212,194,0,0,214,194,0,0,214,194,0,0,214,194,0,0,214,194,0,0,210,194,0,0,206,194,0,0,204,194,0,0,202,194,0,0,198,194,0,0,196,194,0,0,192,194,0,0,190,194,0,0,190,194,0,0,192,194,0,0,194,194,0,0,192,194,0,0,190,194,0,0,186,194,0,0,180,194,0,0,160,194,0,0,140,194,0,0,72,194,0,0,32,194,0,0,240,193,0,0,240,193,0,0,240,193,0,0,240,193,0,0,112,194,0,0,112,194,0,0,112,194,0,0,112,194,0,0,112,194,0,0,112,194,0,0,112,194,0,0,112,194,0,0,112,194,0,0,112,194,0,0,112,194,0,0,112,194,0,0,120,194,0,0,120,194,0,0,130,194,0,0,146,194,0,0,138,194,0,0,136,194,0,0,136,194,0,0,134,194,0,0,140,194,0,0,140,194,0,0,144,194,0,0,148,194,0,0,150,194,0,0,158,194,0,0,158,194,0,0,160,194,0,0,166,194,0,0,176,194,0,0,186,194,0,0,200,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,64,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,84,194,0,0,116,194,0,0,132,194,0,0,132,194,0,0,136,194,0,0,134,194,0,0,140,194,0,0,152,194,0,0,152,194,0,0,144,194,0,0,146,194,0,0,150,194,0,0,152,194,0,0,156,194,0,0,158,194,0,0,166,194,0,0,176,194,0,0,186,194,0,0,200,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,20,194,0,0,20,194,0,0,20,194,0,0,20,194,0,0,20,194,0,0,20,194,0,0,20,194,0,0,20,194,0,0,24,194,0,0,32,194,0,0,40,194,0,0,56,194,0,0,64,194,0,0,84,194,0,0,92,194,0,0,120,194,0,0,130,194,0,0,104,194,0,0,96,194,0,0,96,194,0,0,116,194,0,0,112,194,0,0,130,194,0,0,134,194,0,0,138,194,0,0,142,194,0,0,154,194,0,0,154,194,0,0,156,194,0,0,160,194,0,0,164,194,0,0,168,194,0,0,176,194,0,0,186,194,0,0,196,194,0,0,212,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,200,193,0,0,200,193,0,0,200,193,0,0,200,193,0,0,200,193,0,0,200,193,0,0,200,193,0,0,200,193,0,0,200,193,0,0,208,193,0,0,216,193,0,0,232,193,0,0,0,194,0,0,24,194,0,0,64,194,0,0,80,194,0,0,80,194,0,0,72,194,0,0,64,194,0,0,64,194,0,0,76,194,0,0,80,194,0,0,88,194,0,0,112,194,0,0,134,194,0,0,134,194,0,0,132,194,0,0,136,194,0,0,138,194,0,0,146,194,0,0,146,194,0,0,152,194,0,0,160,194,0,0,162,194,0,0,162,194,0,0,170,194,0,0,170,194,0,0,172,194,0,0,176,194,0,0,186,194,0,0,200,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,128,193,0,0,128,193,0,0,128,193,0,0,128,193,0,0,128,193,0,0,128,193,0,0,128,193,0,0,128,193,0,0,136,193,0,0,152,193,0,0,160,193,0,0,176,193,0,0,208,193,0,0,224,193,0,0,248,193,0,0,32,194,0,0,60,194,0,0,28,194,0,0,28,194,0,0,32,194,0,0,40,194,0,0,44,194,0,0,60,194,0,0,76,194,0,0,100,194,0,0,80,194,0,0,92,194,0,0,92,194,0,0,112,194,0,0,104,194,0,0,120,194,0,0,124,194,0,0,140,194,0,0,134,194,0,0,138,194,0,0,144,194,0,0,146,194,0,0,154,194,0,0,160,194,0,0,164,194,0,0,166,194,0,0,174,194,0,0,180,194,0,0,188,194,0,0,196,194,0,0,208,194,0,0,230,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,0,193,0,0,0,193,0,0,0,193,0,0,0,193,0,0,0,193,0,0,0,193,0,0,0,193,0,0,0,193,0,0,0,193,0,0,0,193,0,0,32,193,0,0,48,193,0,0,112,193,0,0,152,193,0,0,200,193,0,0,240,193,0,0,8,194,0,0,248,193,0,0,240,193,0,0,248,193,0,0,232,193,0,0,0,194,0,0,12,194,0,0,40,194,0,0,64,194,0,0,40,194,0,0,48,194,0,0,56,194,0,0,72,194,0,0,72,194,0,0,76,194,0,0,80,194,0,0,108,194,0,0,88,194,0,0,92,194,0,0,92,194,0,0,104,194,0,0,120,194,0,0,124,194,0,0,132,194,0,0,144,194,0,0,146,194,0,0,152,194,0,0,150,194,0,0,156,194,0,0,160,194,0,0,160,194,0,0,162,194,0,0,168,194,0,0,176,194,0,0,180,194,0,0,188,194,0,0,196,194,0,0,202,194,0,0,212,194,0,0,220,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,134,194,0,0,134,194,0,0,134,194,0,0,152,194,0,0,144,194,0,0,142,194,0,0,148,194,0,0,152,194,0,0,152,194,0,0,150,194,0,0,156,194,0,0,158,194,0,0,158,194,0,0,162,194,0,0,166,194,0,0,172,194,0,0,178,194,0,0,186,194,0,0,194,194,0,0,200,194,0,0,210,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,60,194,0,0,60,194,0,0,60,194,0,0,60,194,0,0,60,194,0,0,60,194,0,0,60,194,0,0,60,194,0,0,60,194,0,0,60,194,0,0,60,194,0,0,64,194,0,0,76,194,0,0,92,194,0,0,108,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,134,194,0,0,132,194,0,0,136,194,0,0,138,194,0,0,140,194,0,0,148,194,0,0,158,194,0,0,154,194,0,0,154,194,0,0,156,194,0,0,160,194,0,0,162,194,0,0,164,194,0,0,168,194,0,0,172,194,0,0,176,194,0,0,182,194,0,0,190,194,0,0,200,194,0,0,216,194,0,0,232,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,20,194,0,0,20,194,0,0,36,194,0,0,48,194,0,0,64,194,0,0,76,194,0,0,104,194,0,0,120,194,0,0,112,194,0,0,100,194,0,0,108,194,0,0,108,194,0,0,112,194,0,0,124,194,0,0,130,194,0,0,144,194,0,0,142,194,0,0,140,194,0,0,144,194,0,0,148,194,0,0,154,194,0,0,152,194,0,0,156,194,0,0,162,194,0,0,162,194,0,0,160,194,0,0,166,194,0,0,172,194,0,0,182,194,0,0,192,194,0,0,200,194,0,0,210,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,224,193,0,0,224,193,0,0,224,193,0,0,224,193,0,0,224,193,0,0,224,193,0,0,224,193,0,0,224,193,0,0,224,193,0,0,240,193,0,0,0,194,0,0,0,194,0,0,4,194,0,0,12,194,0,0,36,194,0,0,68,194,0,0,72,194,0,0,68,194,0,0,60,194,0,0,64,194,0,0,64,194,0,0,80,194,0,0,76,194,0,0,100,194,0,0,130,194,0,0,116,194,0,0,108,194,0,0,116,194,0,0,128,194,0,0,138,194,0,0,140,194,0,0,148,194,0,0,154,194,0,0,154,194,0,0,156,194,0,0,162,194,0,0,168,194,0,0,170,194,0,0,174,194,0,0,180,194,0,0,184,194,0,0,192,194,0,0,200,194,0,0,214,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,152,193,0,0,152,193,0,0,152,193,0,0,152,193,0,0,152,193,0,0,152,193,0,0,152,193,0,0,152,193,0,0,160,193,0,0,168,193,0,0,184,193,0,0,216,193,0,0,240,193,0,0,12,194,0,0,16,194,0,0,36,194,0,0,56,194,0,0,48,194,0,0,40,194,0,0,32,194,0,0,36,194,0,0,36,194,0,0,44,194,0,0,64,194,0,0,92,194,0,0,84,194,0,0,80,194,0,0,84,194,0,0,96,194,0,0,108,194,0,0,104,194,0,0,112,194,0,0,134,194,0,0,132,194,0,0,138,194,0,0,142,194,0,0,144,194,0,0,150,194,0,0,158,194,0,0,162,194,0,0,168,194,0,0,174,194,0,0,180,194,0,0,186,194,0,0,194,194,0,0,202,194,0,0,214,194,0,0,228,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,16,193,0,0,16,193,0,0,16,193,0,0,16,193,0,0,16,193,0,0,16,193,0,0,16,193,0,0,16,193,0,0,48,193,0,0,64,193,0,0,64,193,0,0,112,193,0,0,128,193,0,0,160,193,0,0,184,193,0,0,240,193,0,0,20,194,0,0,8,194,0,0,4,194,0,0,8,194,0,0,248,193,0,0,0,194,0,0,0,194,0,0,24,194,0,0,60,194,0,0,48,194,0,0,36,194,0,0,32,194,0,0,60,194,0,0,68,194,0,0,56,194,0,0,56,194,0,0,104,194,0,0,72,194,0,0,72,194,0,0,88,194,0,0,104,194,0,0,120,194,0,0,128,194,0,0,134,194,0,0,134,194,0,0,140,194,0,0,144,194,0,0,152,194,0,0,158,194,0,0,166,194,0,0,174,194,0,0,182,194,0,0,192,194,0,0,200,194,0,0,208,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,120,194,0,0,120,194,0,0,120,194,0,0,120,194,0,0,120,194,0,0,120,194,0,0,120,194,0,0,120,194,0,0,120,194,0,0,120,194,0,0,124,194,0,0,128,194,0,0,132,194,0,0,134,194,0,0,132,194,0,0,136,194,0,0,150,194,0,0,144,194,0,0,152,194,0,0,150,194,0,0,152,194,0,0,156,194,0,0,158,194,0,0,164,194,0,0,168,194,0,0,170,194,0,0,180,194,0,0,188,194,0,0,202,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,108,194,0,0,108,194,0,0,108,194,0,0,108,194,0,0,108,194,0,0,108,194,0,0,108,194,0,0,108,194,0,0,108,194,0,0,108,194,0,0,108,194,0,0,112,194,0,0,112,194,0,0,116,194,0,0,124,194,0,0,132,194,0,0,142,194,0,0,136,194,0,0,140,194,0,0,140,194,0,0,142,194,0,0,144,194,0,0,144,194,0,0,150,194,0,0,162,194,0,0,156,194,0,0,158,194,0,0,164,194,0,0,166,194,0,0,172,194,0,0,180,194,0,0,194,194,0,0,206,194,0,0,226,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,84,194,0,0,84,194,0,0,84,194,0,0,84,194,0,0,84,194,0,0,84,194,0,0,84,194,0,0,84,194,0,0,84,194,0,0,88,194,0,0,92,194,0,0,100,194,0,0,96,194,0,0,100,194,0,0,92,194,0,0,116,194,0,0,130,194,0,0,112,194,0,0,112,194,0,0,120,194,0,0,124,194,0,0,124,194,0,0,132,194,0,0,136,194,0,0,148,194,0,0,146,194,0,0,150,194,0,0,150,194,0,0,156,194,0,0,160,194,0,0,160,194,0,0,164,194,0,0,170,194,0,0,180,194,0,0,192,194,0,0,202,194,0,0,216,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,56,194,0,0,56,194,0,0,56,194,0,0,56,194,0,0,56,194,0,0,56,194,0,0,56,194,0,0,56,194,0,0,56,194,0,0,56,194,0,0,60,194,0,0,60,194,0,0,60,194,0,0,60,194,0,0,64,194,0,0,76,194,0,0,100,194,0,0,76,194,0,0,68,194,0,0,72,194,0,0,76,194,0,0,84,194,0,0,88,194,0,0,108,194,0,0,132,194,0,0,112,194,0,0,120,194,0,0,134,194,0,0,134,194,0,0,140,194,0,0,144,194,0,0,150,194,0,0,152,194,0,0,156,194,0,0,162,194,0,0,170,194,0,0,176,194,0,0,188,194,0,0,194,194,0,0,208,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,16,194,0,0,28,194,0,0,36,194,0,0,40,194,0,0,40,194,0,0,28,194,0,0,24,194,0,0,36,194,0,0,44,194,0,0,80,194,0,0,48,194,0,0,32,194,0,0,28,194,0,0,20,194,0,0,20,194,0,0,32,194,0,0,60,194,0,0,88,194,0,0,72,194,0,0,64,194,0,0,72,194,0,0,92,194,0,0,116,194,0,0,108,194,0,0,120,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,138,194,0,0,138,194,0,0,146,194,0,0,148,194,0,0,148,194,0,0,150,194,0,0,154,194,0,0,158,194,0,0,164,194,0,0,174,194,0,0,182,194,0,0,190,194,0,0,200,194,0,0,216,194,0,0,230,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,224,193,0,0,208,193,0,0,192,193,0,0,176,193,0,0,160,193,0,0,160,193,0,0,184,193,0,0,232,193,0,0,240,193,0,0,248,193,0,0,224,193,0,0,216,193,0,0,224,193,0,0,224,193,0,0,224,193,0,0,12,194,0,0,32,194,0,0,4,194,0,0,0,194,0,0,232,193,0,0,240,193,0,0,240,193,0,0,240,193,0,0,20,194,0,0,52,194,0,0,36,194,0,0,20,194,0,0,24,194,0,0,52,194,0,0,60,194,0,0,60,194,0,0,64,194,0,0,84,194,0,0,68,194,0,0,64,194,0,0,72,194,0,0,68,194,0,0,68,194,0,0,76,194,0,0,80,194,0,0,104,194,0,0,96,194,0,0,100,194,0,0,96,194,0,0,112,194,0,0,116,194,0,0,120,194,0,0,140,194,0,0,144,194,0,0,148,194,0,0,156,194,0,0,166,194,0,0,176,194,0,0,186,194,0,0,200,194,0,0,212,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,210,194,0,0,200,194,0,0,190,194,0,0,182,194,0,0,174,194,0,0,166,194,0,0,160,194,0,0,156,194,0,0,152,194,0,0,156,194,0,0,156,194,0,0,162,194,0,0,166,194,0,0,170,194,0,0,172,194,0,0,170,194,0,0,172,194,0,0,174,194,0,0,180,194,0,0,194,194,0,0,214,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,210,194], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+30720);
/* memory initializer */ allocate([0,0,200,194,0,0,190,194,0,0,180,194,0,0,170,194,0,0,162,194,0,0,154,194,0,0,146,194,0,0,140,194,0,0,134,194,0,0,134,194,0,0,136,194,0,0,150,194,0,0,146,194,0,0,140,194,0,0,138,194,0,0,140,194,0,0,144,194,0,0,150,194,0,0,158,194,0,0,168,194,0,0,166,194,0,0,168,194,0,0,172,194,0,0,176,194,0,0,178,194,0,0,178,194,0,0,186,194,0,0,196,194,0,0,210,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,200,194,0,0,190,194,0,0,180,194,0,0,170,194,0,0,160,194,0,0,152,194,0,0,142,194,0,0,136,194,0,0,136,194,0,0,130,194,0,0,124,194,0,0,124,194,0,0,120,194,0,0,120,194,0,0,128,194,0,0,130,194,0,0,128,194,0,0,116,194,0,0,120,194,0,0,124,194,0,0,128,194,0,0,132,194,0,0,136,194,0,0,146,194,0,0,146,194,0,0,148,194,0,0,150,194,0,0,152,194,0,0,162,194,0,0,166,194,0,0,170,194,0,0,176,194,0,0,178,194,0,0,184,194,0,0,190,194,0,0,200,194,0,0,216,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,160,194,0,0,150,194,0,0,142,194,0,0,136,194,0,0,130,194,0,0,124,194,0,0,120,194,0,0,116,194,0,0,116,194,0,0,116,194,0,0,116,194,0,0,108,194,0,0,96,194,0,0,100,194,0,0,84,194,0,0,72,194,0,0,104,194,0,0,80,194,0,0,72,194,0,0,72,194,0,0,80,194,0,0,84,194,0,0,88,194,0,0,104,194,0,0,134,194,0,0,124,194,0,0,134,194,0,0,136,194,0,0,144,194,0,0,150,194,0,0,156,194,0,0,160,194,0,0,162,194,0,0,162,194,0,0,164,194,0,0,170,194,0,0,178,194,0,0,180,194,0,0,186,194,0,0,194,194,0,0,202,194,0,0,214,194,0,0,228,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,130,194,0,0,116,194,0,0,108,194,0,0,100,194,0,0,96,194,0,0,92,194,0,0,92,194,0,0,96,194,0,0,96,194,0,0,100,194,0,0,92,194,0,0,84,194,0,0,80,194,0,0,60,194,0,0,48,194,0,0,48,194,0,0,72,194,0,0,48,194,0,0,36,194,0,0,28,194,0,0,28,194,0,0,40,194,0,0,32,194,0,0,56,194,0,0,76,194,0,0,68,194,0,0,72,194,0,0,84,194,0,0,88,194,0,0,124,194,0,0,112,194,0,0,116,194,0,0,120,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,140,194,0,0,146,194,0,0,148,194,0,0,150,194,0,0,152,194,0,0,150,194,0,0,158,194,0,0,170,194,0,0,178,194,0,0,182,194,0,0,192,194,0,0,204,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,80,194,0,0,72,194,0,0,68,194,0,0,68,194,0,0,64,194,0,0,64,194,0,0,64,194,0,0,68,194,0,0,72,194,0,0,72,194,0,0,68,194,0,0,56,194,0,0,44,194,0,0,28,194,0,0,12,194,0,0,4,194,0,0,24,194,0,0,16,194,0,0,0,194,0,0,232,193,0,0,0,194,0,0,0,194,0,0,0,194,0,0,12,194,0,0,48,194,0,0,28,194,0,0,24,194,0,0,24,194,0,0,56,194,0,0,72,194,0,0,52,194,0,0,56,194,0,0,84,194,0,0,72,194,0,0,72,194,0,0,72,194,0,0,88,194,0,0,88,194,0,0,84,194,0,0,84,194,0,0,96,194,0,0,100,194,0,0,108,194,0,0,132,194,0,0,140,194,0,0,144,194,0,0,148,194,0,0,158,194,0,0,166,194,0,0,170,194,0,0,180,194,0,0,194,194,0,0,228,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,210,194,0,0,200,194,0,0,190,194,0,0,180,194,0,0,172,194,0,0,160,194,0,0,150,194,0,0,150,194,0,0,158,194,0,0,160,194,0,0,158,194,0,0,160,194,0,0,162,194,0,0,164,194,0,0,176,194,0,0,190,194,0,0,206,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,216,194,0,0,206,194,0,0,196,194,0,0,186,194,0,0,176,194,0,0,166,194,0,0,158,194,0,0,156,194,0,0,150,194,0,0,142,194,0,0,134,194,0,0,136,194,0,0,146,194,0,0,146,194,0,0,144,194,0,0,146,194,0,0,150,194,0,0,154,194,0,0,160,194,0,0,164,194,0,0,176,194,0,0,186,194,0,0,200,194,0,0,214,194,0,0,228,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,210,194,0,0,202,194,0,0,192,194,0,0,180,194,0,0,172,194,0,0,162,194,0,0,154,194,0,0,146,194,0,0,138,194,0,0,132,194,0,0,116,194,0,0,120,194,0,0,132,194,0,0,128,194,0,0,120,194,0,0,130,194,0,0,132,194,0,0,140,194,0,0,144,194,0,0,152,194,0,0,162,194,0,0,160,194,0,0,168,194,0,0,180,194,0,0,190,194,0,0,204,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,214,194,0,0,206,194,0,0,194,194,0,0,184,194,0,0,176,194,0,0,166,194,0,0,158,194,0,0,148,194,0,0,140,194,0,0,132,194,0,0,108,194,0,0,84,194,0,0,104,194,0,0,120,194,0,0,92,194,0,0,88,194,0,0,88,194,0,0,88,194,0,0,104,194,0,0,116,194,0,0,120,194,0,0,144,194,0,0,140,194,0,0,144,194,0,0,150,194,0,0,156,194,0,0,160,194,0,0,162,194,0,0,160,194,0,0,166,194,0,0,166,194,0,0,176,194,0,0,186,194,0,0,200,194,0,0,214,194,0,0,230,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,200,194,0,0,190,194,0,0,180,194,0,0,170,194,0,0,160,194,0,0,150,194,0,0,140,194,0,0,132,194,0,0,120,194,0,0,96,194,0,0,64,194,0,0,48,194,0,0,64,194,0,0,56,194,0,0,56,194,0,0,44,194,0,0,56,194,0,0,64,194,0,0,64,194,0,0,76,194,0,0,104,194,0,0,104,194,0,0,108,194,0,0,112,194,0,0,120,194,0,0,120,194,0,0,116,194,0,0,116,194,0,0,130,194,0,0,128,194,0,0,130,194,0,0,136,194,0,0,140,194,0,0,148,194,0,0,150,194,0,0,156,194,0,0,162,194,0,0,172,194,0,0,190,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,200,194,0,0,190,194,0,0,180,194,0,0,170,194,0,0,160,194,0,0,150,194,0,0,140,194,0,0,130,194,0,0,116,194,0,0,92,194,0,0,68,194,0,0,28,194,0,0,4,194,0,0,32,194,0,0,12,194,0,0,0,194,0,0,24,194,0,0,32,194,0,0,4,194,0,0,12,194,0,0,20,194,0,0,56,194,0,0,36,194,0,0,52,194,0,0,48,194,0,0,56,194,0,0,40,194,0,0,52,194,0,0,56,194,0,0,80,194,0,0,72,194,0,0,72,194,0,0,72,194,0,0,88,194,0,0,88,194,0,0,92,194,0,0,100,194,0,0,120,194,0,0,128,194,0,0,132,194,0,0,136,194,0,0,140,194,0,0,152,194,0,0,162,194,0,0,180,194,0,0,200,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,196,194,0,0,180,194,0,0,170,194,0,0,164,194,0,0,166,194,0,0,160,194,0,0,156,194,0,0,168,194,0,0,158,194,0,0,160,194,0,0,166,194,0,0,174,194,0,0,178,194,0,0,182,194,0,0,186,194,0,0,198,194,0,0,212,194,0,0,234,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,196,194,0,0,180,194,0,0,170,194,0,0,160,194,0,0,150,194,0,0,140,194,0,0,136,194,0,0,148,194,0,0,144,194,0,0,148,194,0,0,154,194,0,0,160,194,0,0,164,194,0,0,170,194,0,0,174,194,0,0,184,194,0,0,178,194,0,0,182,194,0,0,190,194,0,0,200,194,0,0,212,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,196,194,0,0,180,194,0,0,166,194,0,0,150,194,0,0,142,194,0,0,124,194,0,0,128,194,0,0,134,194,0,0,120,194,0,0,128,194,0,0,134,194,0,0,140,194,0,0,146,194,0,0,154,194,0,0,162,194,0,0,168,194,0,0,166,194,0,0,170,194,0,0,178,194,0,0,180,194,0,0,186,194,0,0,196,194,0,0,208,194,0,0,218,194,0,0,228,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,206,194,0,0,192,194,0,0,176,194,0,0,162,194,0,0,150,194,0,0,136,194,0,0,104,194,0,0,88,194,0,0,96,194,0,0,88,194,0,0,96,194,0,0,96,194,0,0,104,194,0,0,112,194,0,0,124,194,0,0,132,194,0,0,148,194,0,0,138,194,0,0,144,194,0,0,144,194,0,0,150,194,0,0,148,194,0,0,154,194,0,0,162,194,0,0,162,194,0,0,164,194,0,0,168,194,0,0,174,194,0,0,186,194,0,0,192,194,0,0,198,194,0,0,208,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,216,194,0,0,204,194,0,0,192,194,0,0,182,194,0,0,170,194,0,0,160,194,0,0,148,194,0,0,136,194,0,0,112,194,0,0,76,194,0,0,56,194,0,0,64,194,0,0,56,194,0,0,44,194,0,0,52,194,0,0,60,194,0,0,60,194,0,0,68,194,0,0,64,194,0,0,96,194,0,0,84,194,0,0,92,194,0,0,104,194,0,0,100,194,0,0,124,194,0,0,104,194,0,0,112,194,0,0,132,194,0,0,128,194,0,0,134,194,0,0,140,194,0,0,140,194,0,0,148,194,0,0,154,194,0,0,168,194,0,0,172,194,0,0,178,194,0,0,182,194,0,0,186,194,0,0,188,194,0,0,202,194,0,0,218,194,0,0,236,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,216,194,0,0,206,194,0,0,196,194,0,0,186,194,0,0,176,194,0,0,166,194,0,0,156,194,0,0,146,194,0,0,136,194,0,0,112,194,0,0,84,194,0,0,48,194,0,0,12,194,0,0,24,194,0,0,24,194,0,0,8,194,0,0,8,194,0,0,16,194,0,0,32,194,0,0,36,194,0,0,48,194,0,0,76,194,0,0,52,194,0,0,56,194,0,0,60,194,0,0,56,194,0,0,88,194,0,0,72,194,0,0,68,194,0,0,72,194,0,0,72,194,0,0,72,194,0,0,76,194,0,0,88,194,0,0,100,194,0,0,104,194,0,0,112,194,0,0,132,194,0,0,132,194,0,0,132,194,0,0,128,194,0,0,130,194,0,0,136,194,0,0,154,194,0,0,164,194,0,0,174,194,0,0,190,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,214,194,0,0,204,194,0,0,194,194,0,0,184,194,0,0,174,194,0,0,166,194,0,0,156,194,0,0,150,194,0,0,164,194,0,0,158,194,0,0,166,194,0,0,170,194,0,0,178,194,0,0,184,194,0,0,190,194,0,0,196,194,0,0,202,194,0,0,210,194,0,0,218,194,0,0,226,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,212,194,0,0,200,194,0,0,190,194,0,0,180,194,0,0,172,194,0,0,162,194,0,0,156,194,0,0,148,194,0,0,138,194,0,0,148,194,0,0,148,194,0,0,152,194,0,0,158,194,0,0,166,194,0,0,168,194,0,0,172,194,0,0,178,194,0,0,184,194,0,0,194,194,0,0,186,194,0,0,200,194,0,0,206,194,0,0,214,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,212,194,0,0,200,194,0,0,190,194,0,0,180,194,0,0,174,194,0,0,166,194,0,0,160,194,0,0,150,194,0,0,138,194,0,0,112,194,0,0,132,194,0,0,132,194,0,0,136,194,0,0,140,194,0,0,148,194,0,0,156,194,0,0,158,194,0,0,162,194,0,0,162,194,0,0,166,194,0,0,168,194,0,0,174,194,0,0,186,194,0,0,192,194,0,0,198,194,0,0,206,194,0,0,214,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,216,194,0,0,206,194,0,0,196,194,0,0,186,194,0,0,178,194,0,0,170,194,0,0,164,194,0,0,156,194,0,0,142,194,0,0,120,194,0,0,92,194,0,0,104,194,0,0,104,194,0,0,88,194,0,0,88,194,0,0,92,194,0,0,108,194,0,0,116,194,0,0,120,194,0,0,140,194,0,0,132,194,0,0,132,194,0,0,134,194,0,0,140,194,0,0,144,194,0,0,150,194,0,0,156,194,0,0,168,194,0,0,168,194,0,0,168,194,0,0,176,194,0,0,182,194,0,0,180,194,0,0,190,194,0,0,196,194,0,0,204,194,0,0,206,194,0,0,212,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,216,194,0,0,206,194,0,0,196,194,0,0,188,194,0,0,180,194,0,0,174,194,0,0,164,194,0,0,158,194,0,0,146,194,0,0,134,194,0,0,104,194,0,0,60,194,0,0,72,194,0,0,52,194,0,0,36,194,0,0,52,194,0,0,64,194,0,0,48,194,0,0,48,194,0,0,68,194,0,0,88,194,0,0,76,194,0,0,64,194,0,0,60,194,0,0,68,194,0,0,72,194,0,0,76,194,0,0,100,194,0,0,104,194,0,0,112,194,0,0,124,194,0,0,138,194,0,0,140,194,0,0,138,194,0,0,142,194,0,0,148,194,0,0,156,194,0,0,164,194,0,0,180,194,0,0,190,194,0,0,202,194,0,0,210,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,202,194,0,0,194,194,0,0,186,194,0,0,180,194,0,0,170,194,0,0,160,194,0,0,154,194,0,0,144,194,0,0,130,194,0,0,96,194,0,0,64,194,0,0,20,194,0,0,32,194,0,0,16,194,0,0,8,194,0,0,32,194,0,0,72,194,0,0,60,194,0,0,24,194,0,0,36,194,0,0,60,194,0,0,24,194,0,0,12,194,0,0,28,194,0,0,24,194,0,0,44,194,0,0,32,194,0,0,52,194,0,0,72,194,0,0,52,194,0,0,48,194,0,0,60,194,0,0,72,194,0,0,92,194,0,0,64,194,0,0,64,194,0,0,80,194,0,0,132,194,0,0,140,194,0,0,152,194,0,0,164,194,0,0,180,194,0,0,194,194,0,0,210,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,216,194,0,0,206,194,0,0,196,194,0,0,186,194,0,0,172,194,0,0,158,194,0,0,152,194,0,0,166,194,0,0,162,194,0,0,170,194,0,0,174,194,0,0,178,194,0,0,186,194,0,0,196,194,0,0,204,194,0,0,214,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,216,194,0,0,206,194,0,0,196,194,0,0,186,194,0,0,172,194,0,0,158,194,0,0,142,194,0,0,154,194,0,0,148,194,0,0,154,194,0,0,158,194,0,0,162,194,0,0,168,194,0,0,170,194,0,0,180,194,0,0,184,194,0,0,186,194,0,0,184,194,0,0,196,194,0,0,202,194,0,0,216,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,216,194,0,0,206,194,0,0,196,194,0,0,186,194,0,0,174,194,0,0,156,194,0,0,136,194,0,0,130,194,0,0,132,194,0,0,120,194,0,0,130,194,0,0,134,194,0,0,140,194,0,0,146,194,0,0,150,194,0,0,156,194,0,0,164,194,0,0,164,194,0,0,166,194,0,0,168,194,0,0,182,194,0,0,186,194,0,0,196,194,0,0,204,194,0,0,212,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,200,194,0,0,190,194,0,0,180,194,0,0,164,194,0,0,148,194,0,0,120,194,0,0,100,194,0,0,104,194,0,0,96,194,0,0,76,194,0,0,80,194,0,0,80,194,0,0,88,194,0,0,88,194,0,0,104,194,0,0,132,194,0,0,108,194,0,0,112,194,0,0,124,194,0,0,132,194,0,0,138,194,0,0,146,194,0,0,158,194,0,0,166,194,0,0,168,194,0,0,160,194,0,0,162,194,0,0,162,194,0,0,164,194,0,0,176,194,0,0,184,194,0,0,196,194,0,0,210,194,0,0,226,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,214,194,0,0,204,194,0,0,194,194,0,0,184,194,0,0,168,194,0,0,158,194,0,0,138,194,0,0,100,194,0,0,60,194,0,0,80,194,0,0,60,194,0,0,48,194,0,0,52,194,0,0,72,194,0,0,80,194,0,0,40,194,0,0,40,194,0,0,84,194,0,0,44,194,0,0,44,194,0,0,64,194,0,0,76,194,0,0,96,194,0,0,92,194,0,0,80,194,0,0,100,194,0,0,108,194,0,0,116,194,0,0,120,194,0,0,134,194,0,0,142,194,0,0,156,194,0,0,166,194,0,0,172,194,0,0,188,194,0,0,196,194,0,0,206,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,200,194,0,0,190,194,0,0,180,194,0,0,168,194,0,0,156,194,0,0,140,194,0,0,116,194,0,0,76,194,0,0,36,194,0,0,32,194,0,0,24,194,0,0,32,194,0,0,56,194,0,0,80,194,0,0,76,194,0,0,36,194,0,0,32,194,0,0,56,194,0,0,32,194,0,0,24,194,0,0,24,194,0,0,36,194,0,0,56,194,0,0,36,194,0,0,56,194,0,0,60,194,0,0,44,194,0,0,44,194,0,0,52,194,0,0,36,194,0,0,52,194,0,0,96,194,0,0,134,194,0,0,136,194,0,0,166,194,0,0,174,194,0,0,180,194,0,0,190,194,0,0,204,194,0,0,214,194,0,0,226,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,218,194,0,0,210,194,0,0,202,194,0,0,192,194,0,0,182,194,0,0,168,194,0,0,154,194,0,0,164,194,0,0,164,194,0,0,170,194,0,0,178,194,0,0,188,194,0,0,200,194,0,0,212,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,212,194,0,0,206,194,0,0,196,194,0,0,184,194,0,0,170,194,0,0,160,194,0,0,142,194,0,0,150,194,0,0,144,194,0,0,152,194,0,0,160,194,0,0,168,194,0,0,172,194,0,0,178,194,0,0,186,194,0,0,200,194,0,0,214,194,0,0,226,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,214,194,0,0,208,194,0,0,202,194,0,0,194,194,0,0,184,194,0,0,176,194,0,0,168,194,0,0,160,194,0,0,128,194,0,0,132,194,0,0,124,194,0,0,128,194,0,0,132,194,0,0,138,194,0,0,146,194,0,0,154,194,0,0,166,194,0,0,166,194,0,0,172,194,0,0,182,194,0,0,196,194,0,0,208,194,0,0,222,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,214,194,0,0,208,194,0,0,202,194,0,0,194,194,0,0,184,194,0,0,180,194,0,0,168,194,0,0,148,194,0,0,100,194,0,0,104,194,0,0,80,194,0,0,92,194,0,0,88,194,0,0,72,194,0,0,80,194,0,0,72,194,0,0,80,194,0,0,124,194,0,0,120,194,0,0,138,194,0,0,152,194,0,0,154,194,0,0,156,194,0,0,156,194,0,0,158,194,0,0,164,194,0,0,176,194,0,0,188,194,0,0,200,194,0,0,212,194,0,0,222,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,212,194,0,0,204,194,0,0,196,194,0,0,190,194,0,0,180,194,0,0,170,194,0,0,166,194,0,0,156,194,0,0,140,194,0,0,72,194,0,0,72,194,0,0,36,194,0,0,48,194,0,0,68,194,0,0,60,194,0,0,72,194,0,0,72,194,0,0,48,194,0,0,92,194,0,0,56,194,0,0,60,194,0,0,64,194,0,0,64,194,0,0,88,194,0,0,68,194,0,0,68,194,0,0,104,194,0,0,120,194,0,0,142,194,0,0,162,194,0,0,174,194,0,0,184,194,0,0,194,194,0,0,204,194,0,0,216,194,0,0,228,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,212,194,0,0,204,194,0,0,196,194,0,0,190,194,0,0,180,194,0,0,170,194,0,0,166,194,0,0,156,194,0,0,140,194,0,0,52,194,0,0,44,194,0,0,36,194,0,0,60,194,0,0,72,194,0,0,76,194,0,0,72,194,0,0,68,194,0,0,52,194,0,0,60,194,0,0,36,194,0,0,48,194,0,0,36,194,0,0,28,194,0,0,44,194,0,0,24,194,0,0,20,194,0,0,32,194,0,0,36,194,0,0,48,194,0,0,72,194,0,0,104,194,0,0,130,194,0,0,146,194,0,0,158,194,0,0,170,194,0,0,184,194,0,0,194,194,0,0,202,194,0,0,210,194,0,0,218,194,0,0,226,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,214,194,0,0,200,194,0,0,190,194,0,0,174,194,0,0,162,194,0,0,170,194,0,0,166,194,0,0,176,194,0,0,186,194,0,0,200,194,0,0,214,194,0,0,228,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,214,194,0,0,202,194,0,0,190,194,0,0,176,194,0,0,166,194,0,0,152,194,0,0,146,194,0,0,144,194,0,0,158,194,0,0,168,194,0,0,180,194,0,0,190,194,0,0,200,194,0,0,210,194,0,0,220,194,0,0,230,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,208,194,0,0,196,194,0,0,184,194,0,0,174,194,0,0,162,194,0,0,140,194,0,0,130,194,0,0,120,194,0,0,134,194,0,0,142,194,0,0,148,194,0,0,160,194,0,0,170,194,0,0,182,194,0,0,190,194,0,0,198,194,0,0,206,194,0,0,216,194,0,0,222,194,0,0,228,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,206,194,0,0,194,194,0,0,180,194,0,0,170,194,0,0,152,194,0,0,112,194,0,0,96,194,0,0,88,194,0,0,112,194,0,0,120,194,0,0,116,194,0,0,96,194,0,0,124,194,0,0,130,194,0,0,146,194,0,0,148,194,0,0,154,194,0,0,150,194,0,0,156,194,0,0,162,194,0,0,172,194,0,0,174,194,0,0,176,194,0,0,182,194,0,0,188,194,0,0,196,194,0,0,206,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,200,194,0,0,194,194,0,0,184,194,0,0,172,194,0,0,162,194,0,0,158,194,0,0,140,194,0,0,100,194,0,0,76,194,0,0,60,194,0,0,76,194,0,0,104,194,0,0,112,194,0,0,96,194,0,0,84,194,0,0,72,194,0,0,104,194,0,0,80,194,0,0,72,194,0,0,72,194,0,0,84,194,0,0,92,194,0,0,128,194,0,0,138,194,0,0,142,194,0,0,170,194,0,0,164,194,0,0,156,194,0,0,162,194,0,0,170,194,0,0,190,194,0,0,204,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,200,194,0,0,194,194,0,0,184,194,0,0,170,194,0,0,166,194,0,0,158,194,0,0,144,194,0,0,68,194,0,0,32,194,0,0,44,194,0,0,44,194,0,0,88,194,0,0,96,194,0,0,76,194,0,0,72,194,0,0,32,194,0,0,44,194,0,0,24,194,0,0,16,194,0,0,12,194,0,0,20,194,0,0,24,194,0,0,20,194,0,0,48,194,0,0,88,194,0,0,112,194,0,0,100,194,0,0,112,194,0,0,140,194,0,0,150,194,0,0,168,194,0,0,184,194,0,0,206,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,204,194,0,0,190,194,0,0,178,194,0,0,164,194,0,0,166,194,0,0,168,194,0,0,180,194,0,0,184,194,0,0,198,194,0,0,214,194,0,0,226,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,214,194,0,0,202,194,0,0,190,194,0,0,178,194,0,0,166,194,0,0,144,194,0,0,148,194,0,0,156,194,0,0,170,194,0,0,176,194,0,0,176,194,0,0,180,194,0,0,184,194,0,0,196,194,0,0,210,194,0,0,222,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,218,194,0,0,206,194,0,0,194,194,0,0,186,194,0,0,174,194,0,0,162,194,0,0,140,194,0,0,140,194,0,0,134,194,0,0,150,194,0,0,146,194,0,0,152,194,0,0,158,194,0,0,162,194,0,0,166,194,0,0,176,194,0,0,178,194,0,0,194,194,0,0,206,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,214,194,0,0,200,194,0,0,188,194,0,0,176,194,0,0,166,194,0,0,150,194,0,0,124,194,0,0,108,194,0,0,108,194,0,0,124,194,0,0,132,194,0,0,112,194,0,0,120,194,0,0,134,194,0,0,134,194,0,0,154,194,0,0,152,194,0,0,162,194,0,0,176,194,0,0,172,194,0,0,184,194,0,0,192,194,0,0,204,194,0,0,218,194,0,0,232,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,196,194,0,0,184,194,0,0,172,194,0,0,162,194,0,0,146,194,0,0,96,194,0,0,80,194,0,0,60,194,0,0,92,194,0,0,112,194,0,0,104,194,0,0,80,194,0,0,76,194,0,0,52,194,0,0,68,194,0,0,72,194,0,0,84,194,0,0,88,194,0,0,116,194,0,0,142,194,0,0,140,194,0,0,138,194,0,0,156,194,0,0,158,194,0,0,174,194,0,0,180,194,0,0,192,194,0,0,208,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+40960);
/* memory initializer */ allocate([0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,206,194,0,0,192,194,0,0,180,194,0,0,172,194,0,0,156,194,0,0,140,194,0,0,76,194,0,0,40,194,0,0,60,194,0,0,64,194,0,0,92,194,0,0,88,194,0,0,88,194,0,0,84,194,0,0,40,194,0,0,12,194,0,0,224,193,0,0,4,194,0,0,24,194,0,0,20,194,0,0,48,194,0,0,60,194,0,0,68,194,0,0,88,194,0,0,124,194,0,0,136,194,0,0,156,194,0,0,164,194,0,0,178,194,0,0,188,194,0,0,198,194,0,0,208,194,0,0,218,194,0,0,228,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,200,194,0,0,180,194,0,0,158,194,0,0,170,194,0,0,162,194,0,0,164,194,0,0,164,194,0,0,178,194,0,0,188,194,0,0,198,194,0,0,206,194,0,0,218,194,0,0,230,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,194,194,0,0,170,194,0,0,144,194,0,0,148,194,0,0,140,194,0,0,140,194,0,0,140,194,0,0,152,194,0,0,170,194,0,0,182,194,0,0,186,194,0,0,194,194,0,0,206,194,0,0,218,194,0,0,230,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,224,194,0,0,186,194,0,0,162,194,0,0,136,194,0,0,120,194,0,0,112,194,0,0,112,194,0,0,100,194,0,0,124,194,0,0,140,194,0,0,154,194,0,0,164,194,0,0,180,194,0,0,186,194,0,0,196,194,0,0,208,194,0,0,218,194,0,0,226,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,226,194,0,0,200,194,0,0,186,194,0,0,168,194,0,0,124,194,0,0,104,194,0,0,64,194,0,0,84,194,0,0,88,194,0,0,80,194,0,0,80,194,0,0,100,194,0,0,128,194,0,0,132,194,0,0,152,194,0,0,166,194,0,0,162,194,0,0,170,194,0,0,170,194,0,0,180,194,0,0,190,194,0,0,196,194,0,0,202,194,0,0,206,194,0,0,212,194,0,0,216,194,0,0,222,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,210,194,0,0,190,194,0,0,172,194,0,0,148,194,0,0,84,194,0,0,72,194,0,0,24,194,0,0,44,194,0,0,68,194,0,0,44,194,0,0,40,194,0,0,28,194,0,0,28,194,0,0,56,194,0,0,80,194,0,0,100,194,0,0,96,194,0,0,144,194,0,0,138,194,0,0,148,194,0,0,162,194,0,0,174,194,0,0,184,194,0,0,188,194,0,0,194,194,0,0,198,194,0,0,204,194,0,0,210,194,0,0,216,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,216,194,0,0,198,194,0,0,180,194,0,0,152,194,0,0,132,194,0,0,52,194,0,0,44,194,0,0,36,194,0,0,48,194,0,0,60,194,0,0,44,194,0,0,60,194,0,0,32,194,0,0,240,193,0,0,248,193,0,0,248,193,0,0,28,194,0,0,4,194,0,0,32,194,0,0,36,194,0,0,44,194,0,0,84,194,0,0,108,194,0,0,140,194,0,0,146,194,0,0,154,194,0,0,158,194,0,0,164,194,0,0,168,194,0,0,174,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,182,194,0,0,152,194,0,0,150,194,0,0,170,194,0,0,186,194,0,0,196,194,0,0,208,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,182,194,0,0,140,194,0,0,140,194,0,0,150,194,0,0,172,194,0,0,178,194,0,0,188,194,0,0,196,194,0,0,202,194,0,0,212,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,190,194,0,0,160,194,0,0,112,194,0,0,130,194,0,0,128,194,0,0,148,194,0,0,166,194,0,0,176,194,0,0,182,194,0,0,190,194,0,0,198,194,0,0,206,194,0,0,214,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,190,194,0,0,160,194,0,0,104,194,0,0,92,194,0,0,68,194,0,0,132,194,0,0,136,194,0,0,142,194,0,0,156,194,0,0,156,194,0,0,160,194,0,0,176,194,0,0,170,194,0,0,178,194,0,0,194,194,0,0,200,194,0,0,210,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,190,194,0,0,160,194,0,0,84,194,0,0,80,194,0,0,36,194,0,0,108,194,0,0,108,194,0,0,68,194,0,0,104,194,0,0,96,194,0,0,124,194,0,0,172,194,0,0,158,194,0,0,180,194,0,0,186,194,0,0,196,194,0,0,206,194,0,0,214,194,0,0,224,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,194,194,0,0,182,194,0,0,146,194,0,0,52,194,0,0,32,194,0,0,4,194,0,0,84,194,0,0,116,194,0,0,68,194,0,0,88,194,0,0,72,194,0,0,72,194,0,0,112,194,0,0,80,194,0,0,134,194,0,0,148,194,0,0,162,194,0,0,184,194,0,0,192,194,0,0,200,194,0,0,210,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,226,194,0,0,212,194,0,0,198,194,0,0,184,194,0,0,154,194,0,0,160,194,0,0,176,194,0,0,194,194,0,0,212,194,0,0,230,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,232,194,0,0,218,194,0,0,204,194,0,0,190,194,0,0,178,194,0,0,148,194,0,0,144,194,0,0,176,194,0,0,174,194,0,0,190,194,0,0,204,194,0,0,218,194,0,0,232,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,232,194,0,0,218,194,0,0,204,194,0,0,190,194,0,0,178,194,0,0,150,194,0,0,132,194,0,0,148,194,0,0,154,194,0,0,156,194,0,0,172,194,0,0,174,194,0,0,180,194,0,0,192,194,0,0,210,194,0,0,230,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,230,194,0,0,216,194,0,0,202,194,0,0,188,194,0,0,176,194,0,0,132,194,0,0,96,194,0,0,116,194,0,0,140,194,0,0,130,194,0,0,156,194,0,0,144,194,0,0,166,194,0,0,168,194,0,0,186,194,0,0,196,194,0,0,210,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,210,194,0,0,190,194,0,0,178,194,0,0,164,194,0,0,100,194,0,0,80,194,0,0,80,194,0,0,108,194,0,0,96,194,0,0,108,194,0,0,104,194,0,0,138,194,0,0,134,194,0,0,176,194,0,0,164,194,0,0,164,194,0,0,178,194,0,0,188,194,0,0,200,194,0,0,216,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,202,194,0,0,192,194,0,0,180,194,0,0,166,194,0,0,154,194,0,0,88,194,0,0,44,194,0,0,24,194,0,0,72,194,0,0,64,194,0,0,80,194,0,0,64,194,0,0,40,194,0,0,40,194,0,0,76,194,0,0,80,194,0,0,84,194,0,0,108,194,0,0,130,194,0,0,142,194,0,0,156,194,0,0,170,194,0,0,190,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,240,194,0,0,210,194,0,0,172,194,0,0,136,194,0,0,156,194,0,0,158,194,0,0,180,194,0,0,200,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,240,194,0,0,210,194,0,0,172,194,0,0,132,194,0,0,146,194,0,0,154,194,0,0,176,194,0,0,192,194,0,0,210,194,0,0,230,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,240,194,0,0,210,194,0,0,184,194,0,0,160,194,0,0,116,194,0,0,128,194,0,0,136,194,0,0,160,194,0,0,174,194,0,0,184,194,0,0,200,194,0,0,220,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,240,194,0,0,208,194,0,0,182,194,0,0,158,194,0,0,80,194,0,0,112,194,0,0,88,194,0,0,128,194,0,0,138,194,0,0,154,194,0,0,160,194,0,0,164,194,0,0,168,194,0,0,170,194,0,0,174,194,0,0,176,194,0,0,180,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,236,194,0,0,200,194,0,0,174,194,0,0,154,194,0,0,68,194,0,0,72,194,0,0,48,194,0,0,104,194,0,0,116,194,0,0,116,194,0,0,134,194,0,0,130,194,0,0,120,194,0,0,120,194,0,0,120,194,0,0,130,194,0,0,136,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,230,194,0,0,196,194,0,0,168,194,0,0,120,194,0,0,68,194,0,0,48,194,0,0,24,194,0,0,56,194,0,0,68,194,0,0,68,194,0,0,56,194,0,0,28,194,0,0,20,194,0,0,28,194,0,0,32,194,0,0,40,194,0,0,44,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,176,194,0,0,148,194,0,0,154,194,0,0,164,194,0,0,164,194,0,0,170,194,0,0,180,194,0,0,188,194,0,0,198,194,0,0,208,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,176,194,0,0,132,194,0,0,140,194,0,0,162,194,0,0,160,194,0,0,162,194,0,0,168,194,0,0,176,194,0,0,182,194,0,0,186,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,176,194,0,0,116,194,0,0,124,194,0,0,140,194,0,0,142,194,0,0,148,194,0,0,154,194,0,0,160,194,0,0,166,194,0,0,170,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,172,194,0,0,120,194,0,0,124,194,0,0,120,194,0,0,120,194,0,0,104,194,0,0,80,194,0,0,72,194,0,0,72,194,0,0,80,194,0,0,88,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,236,194,0,0,216,194,0,0,168,194,0,0,84,194,0,0,72,194,0,0,72,194,0,0,72,194,0,0,92,194,0,0,60,194,0,0,52,194,0,0,32,194,0,0,32,194,0,0,32,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,236,194,0,0,200,194,0,0,146,194,0,0,44,194,0,0,20,194,0,0,40,194,0,0,44,194,0,0,84,194,0,0,24,194,0,0,20,194,0,0,12,194,0,0,12,194,0,0,24,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,200,194,0,0,182,194,0,0,168,194,0,0,148,194,0,0,160,194,0,0,160,194,0,0,160,194,0,0,160,194,0,0,160,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,200,194,0,0,182,194,0,0,168,194,0,0,148,194,0,0,136,194,0,0,136,194,0,0,136,194,0,0,136,194,0,0,136,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,200,194,0,0,172,194,0,0,156,194,0,0,140,194,0,0,112,194,0,0,52,194,0,0,240,193,0,0,168,193,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,200,194,0,0,174,194,0,0,156,194,0,0,134,194,0,0,64,194,0,0,24,194,0,0,232,193,0,0,168,193,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,200,194,0,0,172,194,0,0,138,194,0,0,96,194,0,0,52,194,0,0,12,194,0,0,4,194,0,0,232,193,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,0,220,194,0,0,200,194,0,0,166,194,0,0,142,194,0,0,64,194,0,0,216,193,0,0,24,194,0,0,20,194,0,0,8,194,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,0,192,121,196,212,236,0,0,184,232,0,0,240,236,0,0,16,237,0,0,48,237,0,0,4,0,0,0,2,0,0,0,3,0,0,0,5,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,62,180,228,51,9,145,243,51,139,178,1,52,60,32,10,52,35,26,19,52,96,169,28,52,167,215,38,52,75,175,49,52,80,59,61,52,112,135,73,52,35,160,86,52,184,146,100,52,85,109,115,52,136,159,129,52,252,11,138,52,147,4,147,52,105,146,156,52,50,191,166,52,63,149,177,52,147,31,189,52,228,105,201,52,173,128,214,52,54,113,228,52,166,73,243,52,136,140,1,53,192,247,9,53,6,239,18,53,118,123,28,53,192,166,38,53,55,123,49,53,218,3,61,53,94,76,73,53,59,97,86,53,185,79,100,53,252,37,115,53,138,121,129,53,134,227,137,53,124,217,146,53,133,100,156,53,82,142,166,53,51,97,177,53,37,232,188,53,220,46,201,53,206,65,214,53,65,46,228,53,87,2,243,53,143,102,1,54,79,207,9,54,245,195,18,54,152,77,28,54,232,117,38,54,50,71,49,54,116,204,60,54,94,17,73,54,101,34,86,54,206,12,100,54,184,222,114,54,151,83,129,54,28,187,137,54,114,174,146,54,175,54,156,54,129,93,166,54,53,45,177,54,199,176,188,54,228,243,200,54,1,3,214,54,96,235,227,54,30,187,242,54,162,64,1,55,235,166,9,55,241,152,18,55,201,31,28,55,30,69,38,55,61,19,49,55,30,149,60,55,111,214,72,55,162,227,85,55,247,201,99,55,137,151,114,55,175,45,129,55,190,146,137,55,116,131,146,55,230,8,156,55,190,44,166,55,71,249,176,55,121,121,188,55,254,184,200,55,71,196,213,55,146,168,227,55,248,115,242,55,192,26,1,56,147,126,9,56,249,109,18,56,6,242,27,56,98,20,38,56,86,223,48,56,216,93,60,56,146,155,72,56,242,164,85,56,51,135,99,56,110,80,114,56,211,7,129,56,107,106,137,56,130,88,146,56,42,219,155,56,9,252,165,56,104,197,176,56,59,66,188,56,41,126,200,56,160,133,213,56,217,101,227,56,232,44,242,56,233,244,0,57,70,86,9,57,14,67,18,57,81,196,27,57,181,227,37,57,127,171,48,57,162,38,60,57,197,96,72,57,83,102,85,57,131,68,99,57,104,9,114,57,1,226,128,57,36,66,137,57,157,45,146,57,123,173,155,57,99,203,165,57,153,145,176,57,13,11,188,57,102,67,200,57,11,71,213,57,50,35,227,57,237,229,241,57,29,207,0,58,5,46,9,58,48,24,18,58,169,150,27,58,21,179,37,58,183,119,48,58,124,239,59,58,10,38,72,58,199,39,85,58,230,1,99,58,120,194,113,58,59,188,128,58,233,25,137,58,198,2,146,58,219,127,155,58,203,154,165,58,216,93,176,58,239,211,187,58,179,8,200,58,136,8,213,58,159,224,226,58,7,159,241,58,92,169,0,59,208,5,9,59,94,237,17,59,15,105,27,59,132,130,37,59,253,67,48,59,103,184,59,59,97,235,71,59,77,233,84,59,93,191,98,59,156,123,113,59,127,150,128,59,186,241,136,59,249,215,145,59,71,82,155,59,65,106,165,59,39,42,176,59,226,156,187,59,18,206,199,59,23,202,212,59,32,158,226,59,53,88,241,59,166,131,0,60,167,221,8,60,152,194,17,60,130,59,27,60,1,82,37,60,84,16,48,60,97,129,59,60,200,176,71,60,229,170,84,60,232,124,98,60,212,52,113,60,207,112,128,60,150,201,136,60,58,173,145,60,192,36,155,60,197,57,165,60,133,246,175,60,229,101,187,60,130,147,199,60,185,139,212,60,180,91,226,60,121,17,241,60,251,93,0,61,137,181,8,61,223,151,17,61,2,14,27,61,141,33,37,61,185,220,47,61,109,74,59,61,64,118,71,61,145,108,84,61,133,58,98,61,34,238,112,61,42,75,128,61,127,161,136,61,136,130,145,61,72,247,154,61,88,9,165,61,242,194,175,61,248,46,187,61,3,89,199,61,109,77,212,61,92,25,226,61,209,202,240,61,91,56,0,62,119,141,8,62,51,109,17,62,144,224,26,62,39,241,36,62,46,169,47,62,135,19,59,62,202,59,71,62,77,46,84,62,55,248,97,62,132,167,112,62,143,37,128,62,115,121,136,62,226,87,145,62,220,201,154,62,249,216,164,62,109,143,175,62,27,248,186,62,149,30,199,62,51,15,212,62,23,215,225,62,61,132,240,62,198,18,0,63,114,101,8,63,147,66,17,63,43,179,26,63,206,192,36,63,177,117,47,63,178,220,58,63,101,1,71,63,29,240,83,63,251,181,97,63,251,96,112,63,0,0,128,63,0,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,0,0,0,0,0,0,0,0,18,0,0,0,19,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,19,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,23,0,0,0,24,0,0,0,25,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,184,237,0,0,208,0,2,0,12,165,2,0,164,22,3,0,64,23,3,0,220,23,3,0,232,57,3,0,148,74,3,0,48,75,3,0,204,75,3,0,144,92,3,0,0,102,3,0,28,106,3,0,184,106,3,0,84,107,3,0,240,107,3,0,140,108,3,0,0,0,0,0,11,0,0,0,160,2,0,0,0,3,0,0,2,0,0,0,64,156,0,0,80,195,0,0,84,238,0,0,128,238,0,0,172,238,0,0,156,239,0,0,204,239,0,0,252,239,0,0,44,243,0,0,252,239,0,0,92,246,0,0,140,246,0,0,28,0,1,0,172,9,1,0,60,19,1,0,204,28,1,0,252,28,1,0,96,3,0,0,192,3,0,0,188,32,1,0,232,32,1,0,20,33,1,0,64,33,1,0,32,4,0,0,108,33,1,0,156,33,1,0,120,4,0,0,204,33,1,0,216,4,0,0,104,43,1,0,168,54,1,0,128,71,1,0,2,0,0,0,160,119,1,0,48,120,1,0,0,2,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,16,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,35,0,0,0,21,0,0,0,9,0,0,0,0,0,0,0,0,0,0,0,30,0,0,0,20,0,0,0,8,0,0,0,0,0,0,192,0,0,160,63,25,0,0,0,12,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,20,0,0,0,9,0,0,0,253,255,255,255,0,0,0,0,0,0,0,0,20,0,0,0,9,0,0,0,252,255,255,255,0,0,0,0,0,0,0,0,20,0,0,0,9,0,0,0,252,255,255,255,0,0,0,0,0,0,0,0,20,0,0,0,6,0,0,0,250,255,255,255,0,0,0,0,0,0,0,0,20,0,0,0,3,0,0,0,246,255,255,255,0,0,0,0,0,0,0,0,18,0,0,0,1,0,0,0,242,255,255,255,0,0,0,0,0,0,0,0,18,0,0,0,0,0,0,0,240,255,255,255,0,0,0,0,0,0,0,0,18,0,0,0,254,255,255,255,240,255,255,255,0,0,0,0,0,0,0,0,12,0,0,0,254,255,255,255,236,255,255,255,0,0,0,0,0,0,0,0,90,0,0,0,90,0,0,0,95,0,0,0,95,0,0,0,95,0,0,0,95,0,0,0,105,0,0,0,105,0,0,0,105,0,0,0,105,0,0,0,105,0,0,0,105,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,232,255,255,255,226,255,255,255,216,255,255,255,216,255,255,255,211,255,255,255,211,255,255,255,211,255,255,255,253,255,255,255,248,255,255,255,243,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+51200);
/* memory initializer */ allocate([241,255,255,255,246,255,255,255,246,255,255,255,247,255,255,255,247,255,255,255,247,255,255,255,247,255,255,255,247,255,255,255,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,252,255,255,255,246,255,255,255,242,255,255,255,240,255,255,255,242,255,255,255,243,255,255,255,244,255,255,255,244,255,255,255,245,255,255,255,245,255,255,255,246,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,250,255,255,255,244,255,255,255,242,255,255,255,240,255,255,255,241,255,255,255,241,255,255,255,242,255,255,255,243,255,255,255,243,255,255,255,244,255,255,255,244,255,255,255,254,255,255,255,254,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,244,255,255,255,243,255,255,255,242,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,243,255,255,255,244,255,255,255,244,255,255,255,251,255,255,255,254,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,242,255,255,255,243,255,255,255,243,255,255,255,243,255,255,255,246,255,255,255,252,255,255,255,254,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,243,255,255,255,245,255,255,255,246,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,243,255,255,255,245,255,255,255,246,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,248,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,247,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,247,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,247,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,247,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,253,255,255,255,248,255,255,255,243,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,252,255,255,255,246,255,255,255,242,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,243,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,245,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,250,255,255,255,244,255,255,255,242,255,255,255,240,255,255,255,241,255,255,255,241,255,255,255,242,255,255,255,243,255,255,255,243,255,255,255,244,255,255,255,244,255,255,255,254,255,255,255,254,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,244,255,255,255,243,255,255,255,242,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,243,255,255,255,244,255,255,255,244,255,255,255,250,255,255,255,253,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,242,255,255,255,243,255,255,255,243,255,255,255,243,255,255,255,246,255,255,255,252,255,255,255,254,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,243,255,255,255,245,255,255,255,246,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,243,255,255,255,245,255,255,255,246,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,248,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,247,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,247,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,247,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,241,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,247,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,3,0,0,0,3,0,0,0,15,0,0,0,3,0,0,0,3,0,0,0,15,0,0,0,10,0,0,0,10,0,0,0,100,0,0,0,10,0,0,0,10,0,0,0,100,0,0,0,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,4,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,10,0,0,0,12,0,0,0,14,0,0,0,20,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,236,255,255,255,240,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,254,255,255,255,2,0,0,0,2,0,0,0,3,0,0,0,6,0,0,0,6,0,0,0,15,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,254,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,10,0,0,0,12,0,0,0,14,0,0,0,20,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,250,255,255,255,254,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,3,0,0,0,6,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,254,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,248,255,255,255,250,255,255,255,252,255,255,255,0,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,10,0,0,0,12,0,0,0,14,0,0,0,20,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,254,255,255,255,254,255,255,255,2,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,252,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,250,255,255,255,254,255,255,255,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,8,0,0,0,10,0,0,0,10,0,0,0,16,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,0,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,252,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,250,255,255,255,254,255,255,255,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,6,0,0,0,8,0,0,0,8,0,0,0,14,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,0,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,252,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,244,255,255,255,246,255,255,255,250,255,255,255,254,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,6,0,0,0,6,0,0,0,12,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,0,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,252,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,238,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,6,0,0,0,11,0,0,0,224,255,255,255,224,255,255,255,224,255,255,255,224,255,255,255,228,255,255,255,232,255,255,255,234,255,255,255,240,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,254,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,230,255,255,255,232,255,255,255,238,255,255,255,242,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,246,255,255,255,247,255,255,255,251,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,238,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,6,0,0,0,11,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,222,255,255,255,228,255,255,255,232,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,238,255,255,255,240,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,6,0,0,0,11,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,234,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,224,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,234,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,3,0,0,0,3,0,0,0,5,0,0,0,10,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,224,255,255,255,224,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,224,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,236,255,255,255,242,255,255,255,248,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,2,0,0,0,220,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,222,255,255,255,224,255,255,255,224,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,224,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,236,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,242,255,255,255,244,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,221,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,4,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,10,0,0,0,12,0,0,0,14,0,0,0,20,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,236,255,255,255,240,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,254,255,255,255,2,0,0,0,2,0,0,0,3,0,0,0,6,0,0,0,6,0,0,0,15,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,254,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,4,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,10,0,0,0,12,0,0,0,14,0,0,0,20,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,254,255,255,255,2,0,0,0,3,0,0,0,6,0,0,0,6,0,0,0,8,0,0,0,10,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,0,0,0,0,2,0,0,0,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,248,255,255,255,250,255,255,255,252,255,255,255,0,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,10,0,0,0,12,0,0,0,14,0,0,0,20,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,8,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,254,255,255,255,0,0,0,0,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,250,255,255,255,254,255,255,255,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,8,0,0,0,10,0,0,0,10,0,0,0,16,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,2,0,0,0,6,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,252,255,255,255,254,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,250,255,255,255,254,255,255,255,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,6,0,0,0,8,0,0,0,8,0,0,0,14,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,2,0,0,0,6,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,252,255,255,255,254,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,244,255,255,255,246,255,255,255,250,255,255,255,254,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,6,0,0,0,6,0,0,0,12,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,2,0,0,0,6,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,252,255,255,255,254,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,238,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,6,0,0,0,6,0,0,0,12,0,0,0,224,255,255,255,224,255,255,255,224,255,255,255,224,255,255,255,228,255,255,255,232,255,255,255,234,255,255,255,240,255,255,255,244,255,255,255,250,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,254,255,255,255,0,0,0,0,4,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,230,255,255,255,232,255,255,255,238,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,251,255,255,255,253,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,238,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,6,0,0,0,6,0,0,0,12,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,248,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,253,255,255,255,255,255,255,255,4,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,230,255,255,255,236,255,255,255,240,255,255,255,243,255,255,255,243,255,255,255,243,255,255,255,243,255,255,255,243,255,255,255,245,255,255,255,248,255,255,255,250,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,238,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,6,0,0,0,6,0,0,0,12,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,240,255,255,255,246,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,251,255,255,255,253,255,255,255,1,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,224,255,255,255,224,255,255,255,228,255,255,255,234,255,255,255,238,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,242,255,255,255,244,255,255,255,246,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,5,0,0,0,5,0,0,0,11,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,240,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,249,255,255,255,251,255,255,255,254,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,222,255,255,255,228,255,255,255,234,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,242,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,236,255,255,255,242,255,255,255,248,255,255,255,254,255,255,255,254,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,2,0,0,0,6,0,0,0,220,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,222,255,255,255,224,255,255,255,224,255,255,255,232,255,255,255,240,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,251,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,224,255,255,255,230,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,236,255,255,255,238,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,236,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,221,255,255,255,226,255,255,255,231,255,255,255,231,255,255,255,231,255,255,255,231,255,255,255,231,255,255,255,231,255,255,255,241,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,4,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,10,0,0,0,12,0,0,0,14,0,0,0,20,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,236,255,255,255,240,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,254,255,255,255,2,0,0,0,2,0,0,0,3,0,0,0,6,0,0,0,6,0,0,0,15,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,254,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,250,255,255,255,252,255,255,255,0,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,8,0,0,0,10,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,248,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,3,0,0,0,6,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,8,0,0,0,10,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,1,0,0,0,4,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,10,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,254,255,255,255,255,255,255,255,0,0,0,0,3,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,249,255,255,255,252,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,8,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,254,255,255,255,0,0,0,0,2,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,252,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,238,255,255,255,242,255,255,255,248,255,255,255,255,255,255,255,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,7,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,254,255,255,255,255,255,255,255,1,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,252,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,236,255,255,255,238,255,255,255,242,255,255,255,248,255,255,255,255,255,255,255,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,7,0,0,0,224,255,255,255,224,255,255,255,224,255,255,255,224,255,255,255,228,255,255,255,232,255,255,255,234,255,255,255,240,255,255,255,244,255,255,255,250,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,253,255,255,255,255,255,255,255,0,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,232,255,255,255,232,255,255,255,238,255,255,255,242,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,246,255,255,255,246,255,255,255,247,255,255,255,251,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,236,255,255,255,238,255,255,255,242,255,255,255,248,255,255,255,255,255,255,255,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,7,0,0,0,224,255,255,255,224,255,255,255,224,255,255,255,224,255,255,255,228,255,255,255,232,255,255,255,232,255,255,255,238,255,255,255,242,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,251,255,255,255,254,255,255,255,0,0,0,0,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,234,255,255,255,237,255,255,255,237,255,255,255,237,255,255,255,237,255,255,255,238,255,255,255,239,255,255,255,240,255,255,255,244,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,236,255,255,255,238,255,255,255,242,255,255,255,248,255,255,255,255,255,255,255,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,7,0,0,0,224,255,255,255,224,255,255,255,224,255,255,255,224,255,255,255,228,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,238,255,255,255,242,255,255,255,244,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,250,255,255,255,254,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,236,255,255,255,240,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,234,255,255,255,236,255,255,255,241,255,255,255,246,255,255,255,248,255,255,255,254,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,7,0,0,0,220,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,242,255,255,255,244,255,255,255,246,255,255,255,249,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,220,255,255,255,222,255,255,255,226,255,255,255,228,255,255,255,230,255,255,255,232,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,236,255,255,255,242,255,255,255,248,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,2,0,0,0,218,255,255,255,218,255,255,255,218,255,255,255,218,255,255,255,220,255,255,255,222,255,255,255,222,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,238,255,255,255,240,255,255,255,244,255,255,255,246,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,218,255,255,255,221,255,255,255,221,255,255,255,221,255,255,255,221,255,255,255,221,255,255,255,221,255,255,255,221,255,255,255,221,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,228,255,255,255,236,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,246,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,221,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,236,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,10,0,0,0,10,0,0,0,12,0,0,0,20,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,246,255,255,255,254,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,6,0,0,0,15,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,246,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,2,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,8,0,0,0,10,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,242,255,255,255,250,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,3,0,0,0,6,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,242,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,252,255,255,255,0,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,8,0,0,0,10,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,254,255,255,255,254,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,1,0,0,0,4,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,252,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,10,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,254,255,255,255,255,255,255,255,0,0,0,0,3,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,252,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,8,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,254,255,255,255,0,0,0,0,2,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,251,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,252,255,255,255,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,7,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,252,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,253,255,255,255,254,255,255,255,255,255,255,255,1,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,249,255,255,255,241,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+61440);
/* memory initializer */ allocate([241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,252,255,255,255,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,7,0,0,0,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,240,255,255,255,244,255,255,255,250,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,253,255,255,255,255,255,255,255,0,0,0,0,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,238,255,255,255,242,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,244,255,255,255,246,255,255,255,246,255,255,255,247,255,255,255,248,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,252,255,255,255,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,7,0,0,0,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,238,255,255,255,242,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,251,255,255,255,254,255,255,255,0,0,0,0,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,238,255,255,255,240,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,243,255,255,255,243,255,255,255,244,255,255,255,246,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,252,255,255,255,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,7,0,0,0,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,238,255,255,255,242,255,255,255,246,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,252,255,255,255,0,0,0,0,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,237,255,255,255,237,255,255,255,237,255,255,255,237,255,255,255,238,255,255,255,239,255,255,255,240,255,255,255,244,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,252,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,7,0,0,0,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,236,255,255,255,240,255,255,255,244,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,250,255,255,255,254,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,228,255,255,255,230,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,236,255,255,255,240,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,234,255,255,255,238,255,255,255,242,255,255,255,248,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,252,255,255,255,254,255,255,255,2,0,0,0,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,234,255,255,255,238,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,240,255,255,255,242,255,255,255,244,255,255,255,246,255,255,255,249,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,238,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,242,255,255,255,244,255,255,255,246,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,236,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,216,255,255,255,236,255,255,255,236,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,232,255,255,255,226,255,255,255,216,255,255,255,216,255,255,255,211,255,255,255,211,255,255,255,211,255,255,255,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,23,0,0,0,24,0,0,0,25,0,0,0,26,0,0,0,27,0,0,0,28,0,0,0,29,0,0,0,30,0,0,0,31,0,0,0,32,0,0,0,33,0,0,0,34,0,0,0,35,0,0,0,36,0,0,0,37,0,0,0,38,0,0,0,39,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,6,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,7,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,17,0,0,0,17,0,0,0,18,0,0,0,18,0,0,0,19,0,0,0,19,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,23,0,0,0,24,0,0,0,25,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,5,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,23,0,0,0,24,0,0,0,25,0,0,0,26,0,0,0,27,0,0,0,28,0,0,0,29,0,0,0,30,0,0,0,31,0,0,0,32,0,0,0,33,0,0,0,34,0,0,0,35,0,0,0,36,0,0,0,37,0,0,0,38,0,0,0,39,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,12,0,0,0,13,0,0,0,13,0,0,0,13,0,0,0,14,0,0,0,14,0,0,0,14,0,0,0,15,0,0,0,15,0,0,0,15,0,0,0,15,0,0,0,16,0,0,0,16,0,0,0,17,0,0,0,17,0,0,0,17,0,0,0,18,0,0,0,18,0,0,0,19,0,0,0,19,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,23,0,0,0,24,0,0,0,25,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,8,0,0,0,7,0,0,0,6,0,0,0,5,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,32,0,0,0,16,0,0,0,16,0,0,0,16,0,0,0,32,0,0,0,15,39,0,0,15,39,0,0,15,39,0,0,15,39,0,0,15,39,0,0,15,39,0,0,0,1,0,0,128,0,0,0,128,0,0,0,0,1,0,0,0,2,0,0,15,39,0,0,15,39,0,0,15,39,0,0,15,39,0,0,15,39,0,0,15,39,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,32,0,0,0,156,255,255,255,156,255,255,255,156,255,255,255,156,255,255,255,156,255,255,255,156,255,255,255,151,255,255,255,151,255,255,255,151,255,255,255,151,255,255,255,146,255,255,255,136,255,255,255,126,255,255,255,126,255,255,255,126,255,255,255,126,255,255,255,116,255,255,255,116,255,255,255,116,255,255,255,116,255,255,255,116,255,255,255,116,255,255,255,116,255,255,255,106,255,255,255,8,0,0,0,0,0,160,65,0,0,96,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,112,194,0,0,240,193,0,0,32,194,0,0,32,194,0,0,32,194,0,0,32,194,0,0,32,194,0,0,0,64,0,0,150,194,0,0,192,192,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,0,0,96,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,194,0,0,240,193,0,0,200,193,0,0,200,193,0,0,200,193,0,0,200,193,0,0,200,193,0,0,0,64,0,0,160,194,0,0,192,192,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,0,0,64,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,160,193,0,0,160,193,0,0,112,193,0,0,112,193,0,0,112,193,0,0,112,193,0,0,112,193,0,0,0,0,0,0,160,194,0,0,192,192,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,0,0,32,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,160,193,0,0,112,193,0,0,64,193,0,0,64,193,0,0,64,193,0,0,64,193,0,0,64,193,0,0,0,0,0,0,160,194,0,0,192,192,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,0,0,32,65,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,112,193,0,0,112,193,0,0,64,193,0,0,64,193,0,0,64,193,0,0,64,193,0,0,64,193,0,0,0,0,0,0,170,194,0,0,192,192,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,2,0,0,0,2,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,5,0,0,0,4,0,0,0,3,0,0,0,0,0,128,63,0,0,0,64,0,0,64,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,160,64,0,0,192,64,0,0,224,64,0,0,0,65,0,0,0,65,0,0,0,65,0,0,64,65,0,0,72,65,0,0,80,65,0,0,88,65,0,0,96,65,0,0,104,65,0,0,112,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,2,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,6,0,0,0,6,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,4,0,0,0,3,0,0,0,0,0,128,63,0,0,0,64,0,0,64,64,0,0,128,64,0,0,128,64,0,0,160,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,64,65,0,0,72,65,0,0,80,65,0,0,88,65,0,0,96,65,0,0,104,65,0,0,112,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,2,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,6,0,0,0,6,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,4,0,0,0,3,0,0,0,0,0,128,63,0,0,0,64,0,0,64,64,0,0,128,64,0,0,128,64,0,0,160,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,2,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,8,0,0,0,6,0,0,0,6,0,0,0,5,0,0,0,5,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,2,0,0,0,1,0,0,0,0,0,64,64,0,0,128,64,0,0,128,64,0,0,160,64,0,0,160,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,2,0,0,0,1,0,0,0,0,0,128,64,0,0,128,64,0,0,160,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,192,64,0,0,0,65,0,0,0,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,2,0,0,0,2,0,0,0,2,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,3,0,0,0,2,0,0,0,1,0,0,0,0,0,0,0,0,0,192,64,0,0,192,64,0,0,192,64,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,32,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,2,0,0,0,2,0,0,0,2,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,192,64,0,0,224,64,0,0,0,65,0,0,0,65,0,0,0,65,0,0,32,65,0,0,32,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65,0,0,0,65,0,0,0,65,0,0,32,65,0,0,32,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65,0,0,0,65,0,0,32,65,0,0,32,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65,0,0,32,65,0,0,32,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,64,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,212,54,1,0,176,55,1,0,140,56,1,0,236,57,1,0,76,59,1,0,92,61,1,0,108,63,1,0,72,64,1,0,220,66,1,0,112,69,1,0,144,84,8,0,232,54,1,0,16,55,1,0,56,55,1,0,96,55,1,0,136,55,1,0,1,0,0,0,0,1,0,0,85,109,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,4,0,0,0,85,110,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,10,0,0,0,89,110,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,25,0,0,0,99,110,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,124,110,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,196,55,1,0,236,55,1,0,20,56,1,0,60,56,1,0,100,56,1,0,1,0,0,0,0,1,0,0,188,110,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,4,0,0,0,188,111,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,10,0,0,0,192,111,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,25,0,0,0,202,111,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,227,111,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,172,56,1,0,212,56,1,0,252,56,1,0,36,57,1,0,76,57,1,0,116,57,1,0,156,57,1,0,196,57,1,0,1,0,0,0,64,0,0,0,35,112,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,1,0,0,99,112,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,9,0,0,0,99,113,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,25,0,0,0,108,113,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,133,113,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,9,0,0,0,197,113,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,25,0,0,0,206,113,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,231,113,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,58,1,0,52,58,1,0,92,58,1,0,132,58,1,0,172,58,1,0,212,58,1,0,252,58,1,0,36,59,1,0,1,0,0,0,64,0,0,0,39,114,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,1,0,0,103,114,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,9,0,0,0,103,115,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,25,0,0,0,112,115,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,137,115,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,9,0,0,0,201,115,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,25,0,0,0,210,115,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,235,115,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,124,59,1,0,164,59,1,0,204,59,1,0,244,59,1,0,28,60,1,0,68,60,1,0,108,60,1,0,148,60,1,0,188,60,1,0,228,60,1,0,12,61,1,0,52,61,1,0,1,0,0,0,8,0,0,0,43,116,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,51,116,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,115,116,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,179,116,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,32,0,0,0,51,117,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,83,117,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,18,0,0,0,211,117,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,50,0,0,0,229,117,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,23,118,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,18,0,0,0,151,118,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,50,0,0,0,169,118,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,219,118,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,140,61,1,0,180,61,1,0,220,61,1,0,4,62,1,0,44,62,1,0,84,62,1,0,124,62,1,0,164,62,1,0,204,62,1,0,244,62,1,0,28,63,1,0,68,63,1,0,1,0,0,0,8,0,0,0,91,119,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,99,119,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,163,119,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,227,119,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,32,0,0,0,99,120,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,131,120,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,18,0,0,0,3,121,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,50,0,0,0,21,121,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,71,121,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,18,0,0,0,199,121,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,50,0,0,0,217,121,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,11,122,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,63,1,0,168,63,1,0,208,63,1,0,248,63,1,0,32,64,1,0,1,0,0,0,0,1,0,0,139,122,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,4,0,0,0,139,123,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,10,0,0,0,143,123,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,25,0,0,0,153,123,3], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+71680);
/* memory initializer */ allocate([1,0,0,0,64,0,0,0,178,123,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,132,64,1,0,172,64,1,0,212,64,1,0,252,64,1,0,36,65,1,0,76,65,1,0,116,65,1,0,156,65,1,0,196,65,1,0,236,65,1,0,20,66,1,0,60,66,1,0,100,66,1,0,140,66,1,0,180,66,1,0,1,0,0,0,16,0,0,0,242,123,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,8,0,0,0,2,124,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,1,0,0,10,124,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,10,125,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,74,125,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,32,0,0,0,202,125,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,234,125,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,32,0,0,0,106,126,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,138,126,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,18,0,0,0,10,127,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,50,0,0,0,28,127,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,78,127,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,18,0,0,0,206,127,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,50,0,0,0,224,127,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,18,128,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,67,1,0,64,67,1,0,104,67,1,0,144,67,1,0,184,67,1,0,224,67,1,0,8,68,1,0,48,68,1,0,88,68,1,0,128,68,1,0,168,68,1,0,208,68,1,0,248,68,1,0,32,69,1,0,72,69,1,0,1,0,0,0,16,0,0,0,146,128,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,8,0,0,0,162,128,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,1,0,0,170,128,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,170,129,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,234,129,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,32,0,0,0,106,130,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,138,130,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,32,0,0,0,10,131,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,42,131,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,18,0,0,0,170,131,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,50,0,0,0,188,131,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,238,131,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,18,0,0,0,110,132,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,50,0,0,0,128,132,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,178,132,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,160,69,1,0,200,69,1,0,240,69,1,0,24,70,1,0,64,70,1,0,104,70,1,0,144,70,1,0,184,70,1,0,224,70,1,0,8,71,1,0,48,71,1,0,88,71,1,0,1,0,0,0,8,0,0,0,50,133,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,58,133,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,64,0,0,0,122,133,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,186,133,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,32,0,0,0,58,134,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,90,134,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,18,0,0,0,218,134,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,50,0,0,0,236,134,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,30,135,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,18,0,0,0,158,135,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,50,0,0,0,176,135,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,128,0,0,0,226,135,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,128,0,0,0,33,0,0,0,8,0,0,0,16,0,0,0,70,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,128,63,0,0,144,65,128,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,1,0,0,66,0,0,0,16,0,0,0,32,0,0,0,140,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,128,63,0,0,144,65,0,1,0,0,2,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,2,0,0,0,3,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,5,0,0,0,6,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,128,0,0,0,14,0,0,0,4,0,0,0,58,0,0,0,2,0,0,0,8,0,0,0,28,0,0,0,90,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,128,63,0,0,144,65,128,0,0,0,2,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,2,0,0,0,3,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,5,0,0,0,6,0,0,0,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,1,0,0,28,0,0,0,8,0,0,0,116,0,0,0,4,0,0,0,16,0,0,0,56,0,0,0,180,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,128,63,0,0,144,65,0,1,0,0,4,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,1,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,6,0,0,0,7,0,0,0,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,9,0,0,0,10,0,0,0,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,128,0,0,0,8,0,0,0,33,0,0,0,4,0,0,0,16,0,0,0,70,0,0,0,2,0,0,0,6,0,0,0,12,0,0,0,23,0,0,0,46,0,0,0,90,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,128,63,0,0,144,65,128,0,0,0,6,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,1,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,6,0,0,0,7,0,0,0,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,9,0,0,0,10,0,0,0,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,128,0,0,0,12,0,0,0,46,0,0,0,4,0,0,0,8,0,0,0,16,0,0,0,23,0,0,0,33,0,0,0,70,0,0,0,2,0,0,0,6,0,0,0,10,0,0,0,14,0,0,0,19,0,0,0,28,0,0,0,39,0,0,0,58,0,0,0,90,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,128,63,0,0,144,65,128,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,1,0,0,66,0,0,0,16,0,0,0,32,0,0,0,140,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,128,63,0,0,144,65,0,1,0,0,8,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,4,0,0,0,3,0,0,0,4,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,9,0,0,0,10,0,0,0,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,12,0,0,0,13,0,0,0,14], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+81944);
/* memory initializer */ allocate([2,0,0,0,0,0,0,0,0,4,0,0,93,0,0,0,23,0,0,0,116,1,0,0,6,0,0,0,46,0,0,0,186,0,0,0,238,2,0,0,14,0,0,0,33,0,0,0,65,0,0,0,130,0,0,0,4,1,0,0,44,2,0,0,3,0,0,0,10,0,0,0,18,0,0,0,28,0,0,0,39,0,0,0,55,0,0,0,79,0,0,0,111,0,0,0,158,0,0,0,220,0,0,0,56,1,0,0,208,1,0,0,138,2,0,0,82,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,64,64,0,0,144,65,0,4,0,0,8,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,4,0,0,0,3,0,0,0,4,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,0,0,0,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,9,0,0,0,10,0,0,0,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,12,0,0,0,13,0,0,0,14,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,0,8,0,0,186,0,0,0,46,0,0,0,232,2,0,0,12,0,0,0,92,0,0,0,116,1,0,0,220,5,0,0,28,0,0,0,66,0,0,0,130,0,0,0,4,1,0,0,8,2,0,0,88,4,0,0,6,0,0,0,20,0,0,0,36,0,0,0,56,0,0,0,78,0,0,0,110,0,0,0,158,0,0,0,222,0,0,0,60,1,0,0,184,1,0,0,112,2,0,0,160,3,0,0,20,5,0,0,164,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,64,64,0,0,144,65,0,8,0,0,6,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,1,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,6,0,0,0,7,0,0,0,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,9,0,0,0,10,0,0,0,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,0,2,0,0,46,0,0,0,186,0,0,0,16,0,0,0,33,0,0,0,65,0,0,0,93,0,0,0,130,0,0,0,22,1,0,0,7,0,0,0,23,0,0,0,39,0,0,0,55,0,0,0,79,0,0,0,110,0,0,0,156,0,0,0,232,0,0,0,104,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,128,63,0,0,144,65,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,66,0,0,240,65,0,0,250,67,0,0,128,63,0,0,144,65,10,0,0,0,172,119,1,0,216,119,1,0,4,120,1,0,1,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,8,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,136,120,1,0,152,145,1,0,136,120,1,0,80,166,1,0,136,120,1,0,240,175,1,0,136,120,1,0,192,185,1,0,136,120,1,0,28,202,1,0,136,120,1,0,104,207,1,0,136,120,1,0,196,212,1,0,136,120,1,0,48,218,1,0,136,120,1,0,20,236,1,0,136,120,1,0,224,242,1,0,136,120,1,0,212,249,1,0,1], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+92504);
/* memory initializer */ allocate([1], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+97540);
/* memory initializer */ allocate([1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+98568);
/* memory initializer */ allocate([1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+100620);
/* memory initializer */ allocate([1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,32,0,0,0,216,145,1,0,240,156,1,0,24,157,1,0,64,157,1,0,152,161,1,0,2,0,0,0,0,0,0,0,32,0,0,0,216,145,1,0,0,166,1,0,40,166,1,0,64,157,1,0,152,161,1,0,0,0,0,0,255,255,255,255,255,255,255,255,9,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+101776);
/* memory initializer */ allocate([1,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,8,0,0,0,16,0,0,0,32,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,231,3,0,0,4,0,0,0,8,0,0,0,16,0,0,0,32,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,81,0,0,0,98,136,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,81,0,0,0,179,136,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,158,1,0,0,0,0,0,0,0,0,0,0,0,0,0,52,158,1,0,0,0,0,0,0,0,0,0,0,0,0,0,112,158,1,0,0,0,0,0,0,0,0,0,0,0,0,0,188,158,1,0,0,0,0,0,0,0,0,0,0,0,0,0,8,159,1,0,0,0,0,0,116,159,1,0,168,159,1,0,0,0,0,0,0,0,0,0,252,159,1,0,88,160,1,0,0,0,0,0,0,0,0,0,148,160,1,0,208,160,1,0,44,161,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,4,137,3,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,40,158,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,165,162,3,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,92,158,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,22,165,3,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,152,158,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,103,165,3,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,228,158,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,184,165,3,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,48,159,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,217,166,3,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,156,159,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,42,167,3,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,208,159,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,163,167,3,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,36,160,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,76,168,3,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,128,160,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,101,168,3,0,1,0,0,0,0,160,27,225,0,160,251,96,3,0,0,0,0,0,0,0,188,160,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,214,170,3,0,1,0,0,0,0,128,217,224,0,0,145,96,4,0,0,0,0,0,0,0,248,160,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,33,1,0,0,127,171,3,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,84,161,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,88,162,1,0,0,0,0,0,0,0,0,0,0,0,0,0,140,162,1,0,0,0,0,0,0,0,0,0,0,0,0,0,200,162,1,0,0,0,0,0,0,0,0,0,0,0,0,0,20,163,1,0,0,0,0,0,0,0,0,0,0,0,0,0,96,163,1,0,0,0,0,0,204,163,1,0,0,164,1,0,0,0,0,0,0,0,0,0,84,164,1,0,176,164,1,0,0,0,0,0,0,0,0,0,236,164,1,0,56,165,1,0,148,165,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,160,172,3,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,128,162,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,65,198,3,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,180,162,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,178,200,3,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,240,162,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,3,201,3,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,60,163,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,84,201,3,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,136,163,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,117,202,3,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,244,163,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,198,202,3,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,40,164,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,63,203,3,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,124,164,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,232,203,3,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,216,164,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,1,204,3,0,1,0,0,0,0,160,59,225,0,160,251,96,4,0,0,0,0,0,0,0,20,165,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,82,204,3,0,1,0,0,0,0,128,217,224,0,0,145,96,4,0,0,0,0,0,0,0,96,165,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,33,1,0,0,251,204,3,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,188,165,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,81,0,0,0,28,206,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,81,0,0,0,109,206,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,16,0,0,0,216,145,1,0,144,166,1,0,184,166,1,0,224,166,1,0,56,171,1,0,2,0,0,0,0,0,0,0,32,0,0,0,216,145,1,0,160,175,1,0,200,175,1,0,224,166,1,0,56,171,1,0,2,0,0,0,81,0,0,0,190,206,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,81,0,0,0,15,207,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,160,167,1,0,0,0,0,0,0,0,0,0,0,0,0,0,212,167,1,0,0,0,0,0,0,0,0,0,0,0,0,0,16,168,1,0,0,0,0,0,0,0,0,0,0,0,0,0,92,168,1,0,0,0,0,0,0,0,0,0,0,0,0,0,168,168,1,0,0,0,0,0,20,169,1,0,72,169,1,0,0,0,0,0,0,0,0,0,156,169,1,0,248,169,1,0,0,0,0,0,0,0,0,0,52,170,1,0,112,170,1,0,204,170,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,96,207,3,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,200,167,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,1,233,3,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,252,167,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,114,235,3,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,56,168,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,195,235,3,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,132,168,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,20,236,3,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,208,168,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,53,237,3,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,60,169,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,134,237,3,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,112,169,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,255,237,3,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,196,169,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,168,238,3,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,32,170,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,193,238,3,0,1,0,0,0,0,160,27,225,0,160,251,96,3,0,0,0,0,0,0,0,92,170,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,50,241,3,0,1,0,0,0,0,128,217,224,0,0,145,96,4,0,0,0,0,0,0,0,152,170,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,33,1,0,0,219,241,3,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,244,170,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,248,171,1,0,0,0,0,0,0,0,0,0,0,0,0,0,44,172,1,0,0,0,0,0,0,0,0,0,0,0,0,0,104,172,1,0,0,0,0,0,0,0,0,0,0,0,0,0,180,172,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,173,1,0,0,0,0,0,108,173,1,0,160,173,1,0,0,0,0,0,0,0,0,0,244,173,1,0,80,174,1,0,0,0,0,0,0,0,0,0,140,174,1,0,216,174,1,0,52,175,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,252,242,3,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,32,172,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,157,12,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,84,172,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,14,15,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,144,172,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,95,15,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,220,172,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,176,15,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,40,173,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,209,16,4,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,148,173,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,34,17,4,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,200,173,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,155,17,4,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,28,174,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,68,18,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,120,174,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,93,18,4,0,1,0,0,0,0,160,59,225,0,160,251,96,4,0,0,0,0,0,0,0,180,174,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,174,18,4,0,1,0,0,0,0,128,217,224,0,0,145,96,4,0,0,0,0,0,0,0,0,175,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,33,1,0,0,87,19,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,92,175,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,81,0,0,0,120,20,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,81,0,0,0,201,20,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,16,0,0,0,216,145,1,0,48,176,1,0,88,176,1,0,128,176,1,0,248,180,1,0,2,0,0,0,0,0,0,0,32,0,0,0,216,145,1,0,112,185,1,0,152,185,1,0,128,176,1,0,248,180,1,0,2,0,0,0,81,0,0,0,26,21,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,81,0,0,0,107,21,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,177,1,0,0,0,0,0,0,0,0,0,0,0,0,0,116,177,1,0,0,0,0,0,0,0,0,0,0,0,0,0,176,177,1,0,0,0,0,0,0,0,0,0,0,0,0,0,252,177,1,0,0,0,0,0,0,0,0,0,0,0,0,0,72,178,1,0,0,0,0,0,180,178,1,0,232,178,1,0,0,0,0,0,0,0,0,0,60,179,1,0,152,179,1,0,0,0,0,0,0,0,0,0,212,179,1,0,48,180,1,0,140,180,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,188,21,4,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,104,177,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,93,47,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,156,177,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,206,49,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,216,177,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,31,50,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,36,178,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,112,50,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,112,178,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,145,51,4,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,220,178,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,226,51,4,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,16,179,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,91,52,4,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,100,179,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,4,53,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,192,179,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,29,53,4,0,1,0,0,0,0,184,84,225,0,160,251,96,4,0,0,0,0,0,0,0,252,179,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,169,0,0,0,198,53,4,0,1,0,0,0,0,128,217,224,0,0,145,96,4,0,0,0,0,0,0,0,88,180,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,33,1,0,0,111,54,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,180,180,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,184,181,1,0,0,0,0,0,0,0,0,0,0,0,0,0,236,181,1,0,0,0,0,0,0,0,0,0,0,0,0,0,40,182,1,0,0,0,0,0,0,0,0,0,0,0,0,0,116,182,1,0,0,0,0,0,0,0,0,0,0,0,0,0,192,182,1,0,0,0,0,0,44,183,1,0,96,183,1,0,0,0,0,0,0,0,0,0,180,183,1,0,16,184,1,0,0,0,0,0,0,0,0,0,76,184,1,0,168,184,1,0,4,185,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,144,55,4,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,224,181,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,49,81,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,20,182,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,162,83,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,80,182,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,243,83,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,156,182,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,68,84,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,232,182,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,101,85,4,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,84,183,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,182,85,4,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,136,183,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,47,86,4,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,220,183,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,216,86,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,56,184,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,241,86,4,0,1,0,0,0,0,184,84,225,0,160,251,96,4,0,0,0,0,0,0,0,116,184,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,169,0,0,0,154,87,4,0,1,0,0,0,0,128,217,224,0,0,145,96,4,0,0,0,0,0,0,0,208,184,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,33,1,0,0,67,88,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,44,185,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,81,0,0,0,100,89,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,81,0,0,0,181,89,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,16,0,0,0,0,186,1,0,24,197,1,0,24,197,1,0,64,197,1,0,64,197,1,0,2,0,0,0,0,0,0,0,32,0,0,0,0,186,1,0,244,201,1,0,244,201,1,0,64,197,1,0,64,197,1,0,0,0,0,0,255,255,255,255,255,255,255,255,10,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+105196);
/* memory initializer */ allocate([1,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,8,0,0,0,16,0,0,0,32,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,231,3,0,0,0,0,0,0,231,3,0,0,4,0,0,0,8,0,0,0,16,0,0,0,32,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,100,0,0,0,6,90,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,198,1,0,0,0,0,0,0,0,0,0,0,0,0,0,52,198,1,0,0,0,0,0,0,0,0,0,0,0,0,0,112,198,1,0,0,0,0,0,0,0,0,0,0,0,0,0,172,198,1,0,0,0,0,0,0,0,0,0,0,0,0,0,248,198,1,0,0,0,0,0,0,0,0,0,0,0,0,0,68,199,1,0,0,0,0,0,176,199,1,0,228,199,1,0,0,0,0,0,0,0,0,0,56,200,1,0,148,200,1,0,0,0,0,0,0,0,0,0,208,200,1,0,44,201,1,0,136,201,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,106,90,4,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,40,198,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,11,116,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,92,198,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,124,118,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,152,198,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,237,120,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,212,198,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,62,121,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,32,199,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,143,121,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,108,199,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,176,122,4,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,216,199,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,1,123,4,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,12,200,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,122,123,4,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,96,200,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,35,124,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,188,200,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,60,124,4,0,1,0,0,0,0,184,84,225,0,160,251,96,4,0,0,0,0,0,0,0,248,200,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,169,0,0,0,229,124,4,0,1,0,0,0,0,128,217,224,0,0,145,96,4,0,0,0,0,0,0,0,84,201,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,33,1,0,0,142,125,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,176,201,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,100,0,0,0,175,126,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,16,0,0,0,0,186,1,0,92,202,1,0,92,202,1,0,132,202,1,0,132,202,1,0,2,0,0,0,0,0,0,0,32,0,0,0,0,186,1,0,64,207,1,0,64,207,1,0,132,202,1,0,132,202,1,0,2,0,0,0,100,0,0,0,19,127,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,68,203,1,0,0,0,0,0,0,0,0,0,0,0,0,0,120,203,1,0,0,0,0,0,0,0,0,0,0,0,0,0,180,203,1,0,0,0,0,0,0,0,0,0,0,0,0,0,240,203,1,0,0,0,0,0,0,0,0,0,0,0,0,0,60,204,1,0,0,0,0,0,0,0,0,0,0,0,0,0,136,204,1,0,0,0,0,0,244,204,1,0,40,205,1,0,0,0,0,0,0,0,0,0,124,205,1,0,216,205,1,0,0,0,0,0,0,0,0,0,20,206,1,0,112,206,1,0,212,206,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,119,127,4,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,108,203,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,24,153,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,160,203,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,137,155,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,220,203,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,250,157,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,24,204,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,75,158,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,100,204,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,156,158,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,176,204,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,189,159,4,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,28,205,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,14,160,4,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,80,205,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,135,160,4,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,164,205,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,48,161,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,0,206,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,73,161,4,0,1,0,0,0,0,232,87,225,0,224,255,96,4,0,0,0,0,0,0,0,60,206,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,225,0,0,0,242,161,4,0,1,0,0,0,0,192,221,224,0,0,145,96,4,0,0,0,0,0,0,0,152,206,1,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,33,1,0,0,211,162,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,252,206,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,100,0,0,0,244,163,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,16,0,0,0,0,186,1,0,168,207,1,0,168,207,1,0,208,207,1,0,208,207,1,0,2,0,0,0,0,0,0,0,32,0,0,0,0,186,1,0,156,212,1,0,156,212,1,0,208,207,1,0,208,207,1,0,2,0,0,0,100,0,0,0,88,164,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,144,208,1,0,0,0,0,0,0,0,0,0,0,0,0,0,196,208,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,209,1,0,0,0,0,0,0,0,0,0,0,0,0,0,60,209,1,0,0,0,0,0,0,0,0,0,0,0,0,0,136,209,1,0,0,0,0,0,0,0,0,0,0,0,0,0,212,209,1,0,0,0,0,0,64,210,1,0,116,210,1,0,0,0,0,0,0,0,0,0,200,210,1,0,36,211,1,0,0,0,0,0,0,0,0,0,96,211,1,0,188,211,1,0,32,212,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,188,164,4,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,184,208,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,93,190,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,236,208,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,206,192,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,40,209,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,63,195,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,100,209,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,144,195,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,176,209,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,225,195,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,252,209,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,2,197,4,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,104,210,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,83,197,4,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,156,210,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,204,197,4,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,240,210,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,117,198,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,76,211,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,142,198,4,0,1,0,0,0,0,136,93,225,0,176,19,97,4,0,0,0,0,0,0,0,136,211,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,225,0,0,0,55,199,4,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,228,211,1,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,24,200,4,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,72,212,1,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,100,0,0,0,209,201,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,16,0,0,0,0,186,1,0,4,213,1,0,4,213,1,0,44,213,1,0,44,213,1,0,2,0,0,0,0,0,0,0,32,0,0,0,0,186,1,0,8,218,1,0,8,218,1,0,44,213,1,0,44,213,1,0,2,0,0,0,100,0,0,0,53,202,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,236,213,1,0,0,0,0,0,0,0,0,0,0,0,0,0,32,214,1,0,0,0,0,0,0,0,0,0,0,0,0,0,92,214,1,0,0,0,0,0,0,0,0,0,0,0,0,0,152,214,1,0,0,0,0,0,0,0,0,0,0,0,0,0,228,214,1,0,0,0,0,0,0,0,0,0,0,0,0,0,48,215,1,0,0,0,0,0,156,215,1,0,208,215,1,0,0,0,0,0,0,0,0,0,36,216,1,0,128,216,1,0,0,0,0,0,0,0,0,0,188,216,1,0,32,217,1,0,140,217,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,153,202,4,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,20,214,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,58,228,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,72,214,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,171,230,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,132,214,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,28,233,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,192,214,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,109,233,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,12,215,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,190,233,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,88,215,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,223,234,4,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,196,215,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,48,235,4,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,248,215,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,169,235,4,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,76,216,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,82,236,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,168,216,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,225,0,0,0,107,236,4,0,1,0,0,0,0,134,115,225,0,80,22,97,4,0,0,0,0,0,0,0,228,216,1,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,33,1,0,0,76,237,4,0,1,0,0,0,0,0,245,224,0,0,149,96,5,0,0,0,0,0,0,0,72,217,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,185,1,0,0,109,238,4,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,180,217,1,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,100,0,0,0,38,240,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,16,0,0,0,112,218,1,0,136,229,1,0,136,229,1,0,176,229,1,0,176,229,1,0,2,0,0,0,0,0,0,0,32,0,0,0,112,218,1,0,236,235,1,0,236,235,1,0,176,229,1,0,176,229,1,0,0,0,0,0,255,255,255,255,255,255,255,255,10,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+115476);
/* memory initializer */ allocate([1,0,0,0,2,0,0,0,4,0,0,0,8,0,0,0,16,0,0,0,32,0,0,0,71,0,0,0,157,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,8,0,0,0,16,0,0,0,71,0,0,0,157,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,100,0,0,0,138,240,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,230,1,0,0,0,0,0,0,0,0,0,0,0,0,0,164,230,1,0,0,0,0,0,0,0,0,0,0,0,0,0,224,230,1,0,0,0,0,0,0,0,0,0,0,0,0,0,44,231,1,0,0,0,0,0,152,231,1,0,204,231,1,0,0,0,0,0,0,0,0,0,32,232,1,0,124,232,1,0,0,0,0,0,0,0,0,0,184,232,1,0,20,233,1,0,0,0,0,0,0,0,0,0,104,233,1,0,204,233,1,0,0,0,0,0,0,0,0,0,72,234,1,0,164,234,1,0,0,235,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,238,240,4,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,152,230,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,63,241,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,204,230,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,176,243,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,8,231,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,1,244,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,84,231,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,34,245,4,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,192,231,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,115,245,4,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,244,231,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,236,245,4,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,72,232,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,149,246,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,164,232,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,174,246,4,0,1,0,0,0,0,128,208,224,0,0,118,96,4,0,0,0,0,0,0,0,224,232,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,121,0,0,0,87,247,4,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,60,233,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,208,247,4,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,144,233,1,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,177,248,4,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,244,233,1,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,169,0,0,0,106,250,4,0,1,0,0,0,0,220,125,225,0,232,51,97,4,0,0,0,0,0,0,0,112,234,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,169,0,0,0,19,251,4,0,1,0,0,0,0,96,18,225,0,128,184,96,4,0,0,0,0,0,0,0,204,234,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,1,0,0,0,49,0,0,0,188,251,4,0,1,0,0,0,0,0,152,224,0,0,16,96,6,0,0,0,0,0,0,0,40,235,1,0,0,0,0,0,24,0,0,0,23,0,0,0,25,0,0,0,22,0,0,0,26,0,0,0,21,0,0,0,27,0,0,0,20,0,0,0,28,0,0,0,19,0,0,0,29,0,0,0,18,0,0,0,30,0,0,0,17,0,0,0,31,0,0,0,16,0,0,0,32,0,0,0,15,0,0,0,33,0,0,0,14,0,0,0,34,0,0,0,13,0,0,0,35,0,0,0,12,0,0,0,36,0,0,0,11,0,0,0,37,0,0,0,10,0,0,0,38,0,0,0,9,0,0,0,39,0,0,0,8,0,0,0,40,0,0,0,7,0,0,0,41,0,0,0,6,0,0,0,42,0,0,0,5,0,0,0,43,0,0,0,4,0,0,0,44,0,0,0,3,0,0,0,45,0,0,0,2,0,0,0,46,0,0,0,1,0,0,0,47,0,0,0,0,0,0,0,48,0,0,0,2,0,0,0,100,0,0,0,237,251,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,16,0,0,0,112,218,1,0,84,236,1,0,84,236,1,0,124,236,1,0,124,236,1,0,2,0,0,0,0,0,0,0,32,0,0,0,112,218,1,0,184,242,1,0,184,242,1,0,124,236,1,0,124,236,1,0,2,0,0,0,100,0,0,0,81,252,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,60,237,1,0,0,0,0,0,0,0,0,0,0,0,0,0,112,237,1,0,0,0,0,0,0,0,0,0,0,0,0,0,172,237,1,0,0,0,0,0,0,0,0,0,0,0,0,0,248,237,1,0,0,0,0,0,100,238,1,0,152,238,1,0,0,0,0,0,0,0,0,0,236,238,1,0,72,239,1,0,0,0,0,0,0,0,0,0,132,239,1,0,224,239,1,0,0,0,0,0,0,0,0,0,52,240,1,0,152,240,1,0,0,0,0,0,0,0,0,0,20,241,1,0,112,241,1,0,204,241,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,181,252,4,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,100,237,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,6,253,4,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,152,237,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,119,255,4,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,212,237,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,200,255,4,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,32,238,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,233,0,5,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,140,238,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,58,1,5,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,192,238,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,179,1,5,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,20,239,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,92,2,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,112,239,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,117,2,5,0,1,0,0,0,0,128,208,224,0,0,118,96,4,0,0,0,0,0,0,0,172,239,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,121,0,0,0,30,3,5,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,8,240,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,151,3,5,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,92,240,1,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,120,4,5,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,192,240,1,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,169,0,0,0,49,6,5,0,1,0,0,0,0,220,125,225,0,232,51,97,4,0,0,0,0,0,0,0,60,241,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,169,0,0,0,218,6,5,0,1,0,0,0,0,96,18,225,0,128,184,96,4,0,0,0,0,0,0,0,152,241,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,1,0,0,0,49,0,0,0,131,7,5,0,1,0,0,0,0,0,152,224,0,0,16,96,6,0,0,0,0,0,0,0,244,241,1,0,0,0,0,0,24,0,0,0,23,0,0,0,25,0,0,0,22,0,0,0,26,0,0,0,21,0,0,0,27,0,0,0,20,0,0,0,28,0,0,0,19,0,0,0,29,0,0,0,18,0,0,0,30,0,0,0,17,0,0,0,31,0,0,0,16,0,0,0,32,0,0,0,15,0,0,0,33,0,0,0,14,0,0,0,34,0,0,0,13,0,0,0,35,0,0,0,12,0,0,0,36,0,0,0,11,0,0,0,37,0,0,0,10,0,0,0,38,0,0,0,9,0,0,0,39,0,0,0,8,0,0,0,40,0,0,0,7,0,0,0,41,0,0,0,6,0,0,0,42,0,0,0,5,0,0,0,43,0,0,0,4,0,0,0,44,0,0,0,3,0,0,0,45,0,0,0,2,0,0,0,46,0,0,0,1,0,0,0,47,0,0,0,0,0,0,0,48,0,0,0,2,0,0,0,100,0,0,0,180,7,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,16,0,0,0,112,218,1,0,32,243,1,0,32,243,1,0,72,243,1,0,72,243,1,0,2,0,0,0,0,0,0,0,32,0,0,0,112,218,1,0,172,249,1,0,172,249,1,0,72,243,1,0,72,243,1,0,2,0,0,0,100,0,0,0,24,8,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,244,1,0,0,0,0,0,0,0,0,0,0,0,0,0,60,244,1,0,0,0,0,0,0,0,0,0,0,0,0,0,120,244,1,0,0,0,0,0,0,0,0,0,0,0,0,0,196,244,1,0,0,0,0,0,48,245,1,0,100,245,1,0,0,0,0,0,0,0,0,0,184,245,1,0,20,246,1,0,0,0,0,0,0,0,0,0,80,246,1,0,172,246,1,0,0,0,0,0,0,0,0,0,0,247,1,0,100,247,1,0,0,0,0,0,0,0,0,0,224,247,1,0,76,248,1,0,192,248,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,124,8,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,48,244,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,205,8,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,100,244,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,62,11,5,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,160,244,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,143,11,5,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,236,244,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,176,12,5,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,88,245,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,1,13,5,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,140,245,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,122,13,5,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,224,245,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,35,14,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,60,246,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,60,14,5,0,1,0,0,0,0,128,208,224,0,0,118,96,4,0,0,0,0,0,0,0,120,246,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,121,0,0,0,229,14,5,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,212,246,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,94,15,5,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,40,247,1,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,63,16,5,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,140,247,1,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,33,1,0,0,248,17,5,0,1,0,0,0,0,24,157,225,0,24,61,97,5,0,0,0,0,0,0,0,8,248,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,105,1,0,0,25,19,5,0,1,0,0,0,0,144,27,225,0,128,184,96,5,0,0,0,0,0,0,0,116,248,1,0,0,0,0,0,9,0,0,0,8,0,0,0,10,0,0,0,7,0,0,0,11,0,0,0,6,0,0,0,12,0,0,0,5,0,0,0,13,0,0,0,4,0,0,0,14,0,0,0,3,0,0,0,15,0,0,0,2,0,0,0,16,0,0,0,1,0,0,0,17,0,0,0,0,0,0,0,18,0,0,0,1,0,0,0,49,0,0,0,130,20,5,0,1,0,0,0,0,0,152,224,0,0,16,96,6,0,0,0,0,0,0,0,232,248,1,0,0,0,0,0,24,0,0,0,23,0,0,0,25,0,0,0,22,0,0,0,26,0,0,0,21,0,0,0,27,0,0,0,20,0,0,0,28,0,0,0,19,0,0,0,29,0,0,0,18,0,0,0,30,0,0,0,17,0,0,0,31,0,0,0,16,0,0,0,32,0,0,0,15,0,0,0,33,0,0,0,14,0,0,0,34,0,0,0,13,0,0,0,35,0,0,0,12,0,0,0,36,0,0,0,11,0,0,0,37,0,0,0,10,0,0,0,38,0,0,0,9,0,0,0,39,0,0,0,8,0,0,0,40,0,0,0,7,0,0,0,41,0,0,0,6,0,0,0,42,0,0,0,5,0,0,0,43,0,0,0,4,0,0,0,44,0,0,0,3,0,0,0,45,0,0,0,2,0,0,0,46,0,0,0,1,0,0,0,47,0,0,0,0,0,0,0,48,0,0,0,2,0,0,0,100,0,0,0,179,20,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,16,0,0,0,112,218,1,0,20,250,1,0,20,250,1,0,60,250,1,0,60,250,1,0,2,0,0,0,0,0,0,0,32,0,0,0,112,218,1,0,168,0,2,0,168,0,2,0,60,250,1,0,60,250,1,0,2,0,0,0,100,0,0,0,23,21,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,252,250,1,0,0,0,0,0,0,0,0,0,0,0,0,0,48,251,1,0,0,0,0,0,0,0,0,0,0,0,0,0,108,251,1,0,0,0,0,0,0,0,0,0,0,0,0,0,184,251,1,0,0,0,0,0,36,252,1,0,88,252,1,0,0,0,0,0,0,0,0,0,172,252,1,0,8,253,1,0,0,0,0,0,0,0,0,0,68,253,1,0,160,253,1,0,0,0,0,0,0,0,0,0,244,253,1,0,88,254,1,0,0,0,0,0,0,0,0,0,212,254,1,0,72,255,1,0,188,255,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,123,21,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,36,251,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,204,21,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,88,251,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,61,24,5,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,148,251,1,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,142,24,5,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,224,251,1,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,175,25,5,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,76,252,1,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,0,26,5,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,128,252,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,121,26,5,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,212,252,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,34,27,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,48,253,1,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,59,27,5,0,1,0,0,0,0,128,208,224,0,0,118,96,4,0,0,0,0,0,0,0,108,253,1,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,121,0,0,0,228,27,5,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,200,253,1,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,93,28,5,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,28,254,1,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,62,29,5,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,128,254,1,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,105,1,0,0,247,30,5,0,1,0,0,0,128,93,176,225,0,24,61,97,5,0,0,0,0,0,0,0,252,254,1,0,0,0,0,0,9,0,0,0,8,0,0,0,10,0,0,0,7,0,0,0,11,0,0,0,6,0,0,0,12,0,0,0,5,0,0,0,13,0,0,0,4,0,0,0,14,0,0,0,3,0,0,0,15,0,0,0,2,0,0,0,16,0,0,0,1,0,0,0,17,0,0,0,0,0,0,0,18,0,0,0,2,0,0,0,105,1,0,0,96,32,5,0,1,0,0,0,0,144,27,225,0,128,184,96,5,0,0,0,0,0,0,0,112,255,1,0,0,0,0,0,9,0,0,0,8,0,0,0,10,0,0,0,7,0,0,0,11,0,0,0,6,0,0,0,12,0,0,0,5,0,0,0,13,0,0,0,4,0,0,0,14,0,0,0,3,0,0,0,15,0,0,0,2,0,0,0,16,0,0,0,1,0,0,0,17,0,0,0,0,0,0,0,18,0,0,0,1,0,0,0,49,0,0,0,201,33,5,0,1,0,0,0,0,0,152,224,0,0,16,96,6,0,0,0,0,0,0,0,228,255,1,0,0,0,0,0,24,0,0,0,23,0,0,0,25,0,0,0,22,0,0,0,26,0,0,0,21,0,0,0,27,0,0,0,20,0,0,0,28,0,0,0,19,0,0,0,29,0,0,0,18,0,0,0,30,0,0,0,17,0,0,0,31,0,0,0,16,0,0,0,32,0,0,0,15,0,0,0,33,0,0,0,14,0,0,0,34,0,0,0,13,0,0,0,35,0,0,0,12,0,0,0,36,0,0,0,11,0,0,0,37,0,0,0,10,0,0,0,38,0,0,0,9,0,0,0,39,0,0,0,8,0,0,0,40,0,0,0,7,0,0,0,41,0,0,0,6,0,0,0,42,0,0,0,5,0,0,0,43,0,0,0,4,0,0,0,44,0,0,0,3,0,0,0,45,0,0,0,2,0,0,0,46,0,0,0,1,0,0,0,47,0,0,0,0,0,0,0,48,0,0,0,2,0,0,0,100,0,0,0,250,33,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,11,0,0,0,56,5,0,0,0,3,0,0,6,0,0,0,64,156,0,0,112,17,1,0,84,238,0,0,128,238,0,0,172,238,0,0,156,239,0,0,204,239,0,0,252,239,0,0,44,243,0,0,252,239,0,0,92,246,0,0,140,246,0,0,28,0,1,0,172,9,1,0,60,19,1,0,204,28,1,0,252,28,1,0,96,3,0,0,192,3,0,0,188,32,1,0,232,32,1,0,20,33,1,0,64,33,1,0,32,4,0,0,108,33,1,0,156,33,1,0,120,4,0,0,204,33,1,0,216,4,0,0,104,43,1,0,168,54,1,0,128,71,1,0,3,0,0,0,160,119,1,0,108,1,2,0,196,1,2,0,212,26,2,0,196,1,2,0,104,55,2,0,196,1,2,0,204,61,2,0,196,1,2,0,48,68,2,0,196,1,2,0,28,86,2,0,196,1,2,0,240,92,2,0,196,99,2,0,212,124,2,0,196,99,2,0,192,142,2,0,196,99,2,0,148,149,2,0,196,99,2,0,180,154,2,0,196,99,2,0,220,159,2,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,3], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+123780);
/* memory initializer */ allocate([2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,0,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,30,0,0,0,52,27,2,0,76,38,2,0,76,38,2,0,116,38,2,0,116,38,2,0,2,0,0,0,0,0,0,0,30,0,0,0,52,27,2,0,116,42,2,0,116,42,2,0,116,38,2,0,116,38,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,180,53,2,0,180,53,2,0,220,53,2,0,220,53,2,0,0,0,0,0,255,255,255,255,255,255,255,255,7,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+134724);
/* memory initializer */ allocate([1,0,0,0,2,0,0,0,7,0,0,0,17,0,0,0,31,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,7,0,0,0,17,0,0,0,31,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,49,0,0,0,94,34,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,52,39,2,0,0,0,0,0,0,0,0,0,104,39,2,0,156,39,2,0,0,0,0,0,0,0,0,0,208,39,2,0,4,40,2,0,0,0,0,0,0,0,0,0,64,40,2,0,124,40,2,0,0,0,0,0,0,0,0,0,192,40,2,0,244,40,2,0,124,40,2,0,0,0,0,0,40,41,2,0,92,41,2,0,232,41,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,143,34,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,92,39,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,130,35,5,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,144,39,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,117,36,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,196,39,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,104,37,5,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,248,39,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,91,38,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,44,40,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,144,50,5,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,104,40,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,197,62,5,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,164,40,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,204,62,5,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,232,40,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,191,63,5,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,28,41,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,178,64,5,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,80,41,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,25,0,0,0,165,65,5,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,132,41,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,190,65,5,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,16,42,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,2,0,0,0,49,0,0,0,215,65,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,255,255,255,2,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+140360);
/* memory initializer */ allocate([32,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,8,66,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,156,54,2,0,248,54,2,0,0,0,0,0,0,0,0,0,52,55,2,0,92,41,2,0,232,41,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,169,0,0,0,12,66,5,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,196,54,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,181,66,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,32,55,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,9,0,0,0,206,66,5,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,92,55,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,15,0,0,0,52,27,2,0,200,55,2,0,200,55,2,0,240,55,2,0,240,55,2,0,2,0,0,0,0,0,0,0,30,0,0,0,52,27,2,0,240,59,2,0,240,59,2,0,240,55,2,0,240,55,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,24,60,2,0,24,60,2,0,64,60,2,0,64,60,2,0,2,0,0,0,49,0,0,0,215,66,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,176,56,2,0,0,0,0,0,0,0,0,0,228,56,2,0,24,57,2,0,0,0,0,0,0,0,0,0,76,57,2,0,128,57,2,0,0,0,0,0,0,0,0,0,188,57,2,0,248,57,2,0,0,0,0,0,0,0,0,0,60,58,2,0,112,58,2,0,248,57,2,0,0,0,0,0,164,58,2,0,216,58,2,0,100,59,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,8,67,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,216,56,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,251,67,5,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,12,57,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,238,68,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,64,57,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,225,69,5,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,116,57,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,212,70,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,168,57,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,9,83,5,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,228,57,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,62,95,5,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,32,58,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,69,95,5,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,100,58,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,56,96,5,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,152,58,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,43,97,5,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,204,58,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,25,0,0,0,30,98,5,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,0,59,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,55,98,5,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,140,59,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,2,0,0,0,49,0,0,0,80,98,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,129,98,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,61,2,0,92,61,2,0,0,0,0,0,0,0,0,0,152,61,2,0,216,58,2,0,100,59,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,169,0,0,0,133,98,5,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,40,61,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,46,99,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,132,61,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,9,0,0,0,71,99,5,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,192,61,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,15,0,0,0,52,27,2,0,44,62,2,0,44,62,2,0,84,62,2,0,84,62,2,0,2,0,0,0,0,0,0,0,30,0,0,0,52,27,2,0,84,66,2,0,84,66,2,0,84,62,2,0,84,62,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,124,66,2,0,124,66,2,0,164,66,2,0,164,66,2,0,2,0,0,0,49,0,0,0,80,99,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,63,2,0,0,0,0,0,0,0,0,0,72,63,2,0,124,63,2,0,0,0,0,0,0,0,0,0,176,63,2,0,228,63,2,0,0,0,0,0,0,0,0,0,32,64,2,0,92,64,2,0,0,0,0,0,0,0,0,0,160,64,2,0,212,64,2,0,92,64,2,0,0,0,0,0,8,65,2,0,60,65,2,0,200,65,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,129,99,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,60,63,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,116,100,5,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,112,63,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,103,101,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,164,63,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,90,102,5,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,216,63,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,77,103,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,12,64,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,130,115,5,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,72,64,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,183,127,5,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,132,64,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,190,127,5,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,200,64,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,177,128,5,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,252,64,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,164,129,5,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,48,65,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,25,0,0,0,151,130,5,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,100,65,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,176,130,5,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,240,65,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,2,0,0,0,49,0,0,0,201,130,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,250,130,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,100,67,2,0,192,67,2,0,0,0,0,0,0,0,0,0,252,67,2,0,60,65,2,0,200,65,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,169,0,0,0,254,130,5,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,140,67,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,167,131,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,232,67,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,9,0,0,0,192,131,5,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,36,68,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,15,0,0,0,144,68,2,0,168,79,2,0,168,79,2,0,208,79,2,0,208,79,2,0,2,0,0,0,0,0,0,0,30,0,0,0,144,68,2,0,64,84,2,0,64,84,2,0,208,79,2,0,208,79,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,104,84,2,0,104,84,2,0,144,84,2,0,144,84,2,0,0,0,0,0,255,255,255,255,255,255,255,255,8,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+144300);
/* memory initializer */ allocate([1,0,0,0,1,0,0,0,2,0,0,0,7,0,0,0,17,0,0,0,31,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,99,0,0,0,7,0,0,0,17,0,0,0,31,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,64,0,0,0,201,131,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,144,80,2,0,0,0,0,0,0,0,0,0,196,80,2,0,0,0,0,0,0,0,0,0,0,81,2,0,52,81,2,0,0,0,0,0,0,0,0,0,104,81,2,0,156,81,2,0,0,0,0,0,0,0,0,0,216,81,2,0,20,82,2,0,0,0,0,0,0,0,0,0,88,82,2,0,140,82,2,0,20,82,2,0,0,0,0,0,192,82,2,0,244,82,2,0,40,83,2,0,180,83,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,9,132,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,184,80,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,252,132,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,236,80,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,243,0,0,0,49,145,5,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,40,81,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,36,146,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,92,81,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,23,147,5,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,144,81,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,10,148,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,196,81,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,63,160,5,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,0,82,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,116,172,5,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,60,82,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,123,172,5,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,128,82,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,110,173,5,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,180,82,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,97,174,5,0,1,0,0,0,0,76,93,225,0,76,93,97,2,0,0,0,0,0,0,0,232,82,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,84,175,5,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,28,83,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,25,0,0,0,71,176,5,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,80,83,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,96,176,5,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,220,83,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,2,0,0,0,64,0,0,0,121,176,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,185,176,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,80,85,2,0,172,85,2,0,0,0,0,0,0,0,0,0,232,85,2,0,40,83,2,0,180,83,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,169,0,0,0,189,176,5,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,120,85,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,102,177,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,212,85,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,9,0,0,0,127,177,5,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,16,86,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,15,0,0,0,144,68,2,0,124,86,2,0,124,86,2,0,164,86,2,0,164,86,2,0,2,0,0,0,0,0,0,0,30,0,0,0,144,68,2,0,20,91,2,0,20,91,2,0,164,86,2,0,164,86,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,60,91,2,0,60,91,2,0,100,91,2,0,100,91,2,0,2,0,0,0,64,0,0,0,136,177,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,100,87,2,0,0,0,0,0,0,0,0,0,152,87,2,0,0,0,0,0,0,0,0,0,212,87,2,0,8,88,2,0,0,0,0,0,0,0,0,0,60,88,2,0,112,88,2,0,0,0,0,0,0,0,0,0,172,88,2,0,232,88,2,0,0,0,0,0,0,0,0,0,44,89,2,0,96,89,2,0,232,88,2,0,0,0,0,0,148,89,2,0,200,89,2,0,252,89,2,0,136,90,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,200,177,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,140,87,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,187,178,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,192,87,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,243,0,0,0,240,190,5,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,252,87,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,227,191,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,48,88,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,214,192,5,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,100,88,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,201,193,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,152,88,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,254,205,5,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,212,88,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,51,218,5,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,16,89,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,58,218,5,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,84,89,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,45,219,5,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,136,89,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,32,220,5,0,1,0,0,0,0,76,93,225,0,76,93,97,2,0,0,0,0,0,0,0,188,89,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,19,221,5,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,240,89,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,25,0,0,0,6,222,5,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,36,90,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,31,222,5,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,176,90,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,2,0,0,0,64,0,0,0,56,222,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,120,222,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,92,2,0,128,92,2,0,0,0,0,0,0,0,0,0,188,92,2,0,252,89,2,0,136,90,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,169,0,0,0,124,222,5,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,76,92,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,37,223,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,168,92,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,9,0,0,0,62,223,5,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,228,92,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,15,0,0,0,144,68,2,0,80,93,2,0,80,93,2,0,120,93,2,0,120,93,2,0,2,0,0,0,0,0,0,0,30,0,0,0,144,68,2,0,232,97,2,0,232,97,2,0,120,93,2,0,120,93,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,16,98,2,0,16,98,2,0,56,98,2,0,56,98,2,0,2,0,0,0,64,0,0,0,71,223,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,56,94,2,0,0,0,0,0,0,0,0,0,108,94,2,0,0,0,0,0,0,0,0,0,168,94,2,0,220,94,2,0,0,0,0,0,0,0,0,0,16,95,2,0,68,95,2,0,0,0,0,0,0,0,0,0,128,95,2,0,188,95,2,0,0,0,0,0,0,0,0,0,0,96,2,0,52,96,2,0,188,95,2,0,0,0,0,0,104,96,2,0,156,96,2,0,208,96,2,0,92,97,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,135,223,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,96,94,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,122,224,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,148,94,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,243,0,0,0,175,236,5,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,208,94,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,162,237,5,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,4,95,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,149,238,5,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,56,95,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,136,239,5,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,108,95,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,189,251,5,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,168,95,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,242,7,6,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,228,95,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,249,7,6,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,40,96,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,236,8,6,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,92,96,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,223,9,6,0,1,0,0,0,0,76,93,225,0,76,93,97,2,0,0,0,0,0,0,0,144,96,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,210,10,6,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,196,96,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,25,0,0,0,197,11,6,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,248,96,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,222,11,6,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,132,97,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,2,0,0,0,64,0,0,0,247,11,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,55,12,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,248,98,2,0,84,99,2,0,0,0,0,0,0,0,0,0,144,99,2,0,208,96,2,0,92,97,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,169,0,0,0,59,12,6,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,32,99,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,228,12,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,124,99,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,9,0,0,0,253,12,6,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,184,99,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+150948);
/* memory initializer */ allocate([2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+159812);
/* memory initializer */ allocate([2,0,0,0,0,0,0,0,15,0,0,0,52,125,2,0,76,136,2,0,76,136,2,0,116,136,2,0,116,136,2,0,2,0,0,0,0,0,0,0,30,0,0,0,52,125,2,0,228,140,2,0,228,140,2,0,116,136,2,0,116,136,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,12,141,2,0,12,141,2,0,52,141,2,0,52,141,2,0,0,0,0,0,255,255,255,255,255,255,255,255,8,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+163020);
/* memory initializer */ allocate([1,0,0,0,2,0,0,0,4,0,0,0,7,0,0,0,17,0,0,0,31,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,4,0,0,0,7,0,0,0,17,0,0,0,31,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,64,0,0,0,6,13,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,52,137,2,0,0,0,0,0,0,0,0,0,104,137,2,0,0,0,0,0,0,0,0,0,164,137,2,0,216,137,2,0,0,0,0,0,0,0,0,0,12,138,2,0,64,138,2,0,0,0,0,0,0,0,0,0,124,138,2,0,184,138,2,0,0,0,0,0,0,0,0,0,252,138,2,0,48,139,2,0,184,138,2,0,0,0,0,0,100,139,2,0,152,139,2,0,204,139,2,0,88,140,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,70,13,6,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,92,137,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,57,14,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,144,137,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,243,0,0,0,110,26,6,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,204,137,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,97,27,6,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,0,138,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,84,28,6,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,52,138,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,71,29,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,104,138,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,124,41,6,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,164,138,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,177,53,6,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,224,138,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,184,53,6,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,36,139,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,171,54,6,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,88,139,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,158,55,6,0,1,0,0,0,0,76,93,225,0,76,93,97,2,0,0,0,0,0,0,0,140,139,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,145,56,6,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,192,139,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,25,0,0,0,132,57,6,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,244,139,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,157,57,6,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,128,140,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,2,0,0,0,64,0,0,0,182,57,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,246,57,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,244,141,2,0,80,142,2,0,0,0,0,0,0,0,0,0,140,142,2,0,204,139,2,0,88,140,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,169,0,0,0,250,57,6,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,28,142,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,163,58,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,120,142,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,9,0,0,0,188,58,6,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,180,142,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,15,0,0,0,52,125,2,0,32,143,2,0,32,143,2,0,72,143,2,0,72,143,2,0,2,0,0,0,0,0,0,0,30,0,0,0,52,125,2,0,184,147,2,0,184,147,2,0,72,143,2,0,72,143,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,224,147,2,0,224,147,2,0,8,148,2,0,8,148,2,0,2,0,0,0,64,0,0,0,197,58,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,144,2,0,0,0,0,0,0,0,0,0,60,144,2,0,0,0,0,0,0,0,0,0,120,144,2,0,172,144,2,0,0,0,0,0,0,0,0,0,224,144,2,0,20,145,2,0,0,0,0,0,0,0,0,0,80,145,2,0,140,145,2,0,0,0,0,0,0,0,0,0,208,145,2,0,4,146,2,0,140,145,2,0,0,0,0,0,56,146,2,0,108,146,2,0,160,146,2,0,44,147,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,5,59,6,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,48,144,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,248,59,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,100,144,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,243,0,0,0,45,72,6,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,160,144,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,32,73,6,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,212,144,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,19,74,6,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,8,145,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,6,75,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,60,145,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,59,87,6,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,120,145,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,112,99,6,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,180,145,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,119,99,6,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,248,145,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,106,100,6,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,44,146,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,93,101,6,0,1,0,0,0,0,76,93,225,0,76,93,97,2,0,0,0,0,0,0,0,96,146,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,80,102,6,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,148,146,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,25,0,0,0,67,103,6,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,200,146,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,92,103,6,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,84,147,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,2,0,0,0,64,0,0,0,117,103,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,181,103,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,200,148,2,0,36,149,2,0,0,0,0,0,0,0,0,0,96,149,2,0,160,146,2,0,44,147,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,169,0,0,0,185,103,6,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,240,148,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,98,104,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,76,149,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,9,0,0,0,123,104,6,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,136,149,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,0,0,0,0,15,0,0,0,52,125,2,0,244,149,2,0,244,149,2,0,28,150,2,0,28,150,2,0,2,0,0,0,0,0,0,0,30,0,0,0,52,125,2,0,140,154,2,0,140,154,2,0,28,150,2,0,28,150,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,224,147,2,0,224,147,2,0,8,148,2,0,8,148,2,0,2,0,0,0,64,0,0,0,132,104,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,220,150,2,0,0,0,0,0,0,0,0,0,16,151,2,0,0,0,0,0,0,0,0,0,76,151,2,0,128,151,2,0,0,0,0,0,0,0,0,0,180,151,2,0,232,151,2,0,0,0,0,0,0,0,0,0,36,152,2,0,96,152,2,0,0,0,0,0,0,0,0,0,164,152,2,0,216,152,2,0,96,152,2,0,0,0,0,0,12,153,2,0,64,153,2,0,116,153,2,0,0,154,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,196,104,6,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,4,151,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,183,105,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,56,151,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,243,0,0,0,236,117,6,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,116,151,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,223,118,6,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,168,151,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,210,119,6,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,220,151,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,197,120,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,16,152,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,250,132,6,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,76,152,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,47,145,6,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,136,152,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,54,145,6,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,204,152,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,41,146,6,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,0,153,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,28,147,6,0,1,0,0,0,0,76,93,225,0,76,93,97,2,0,0,0,0,0,0,0,52,153,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,15,148,6,0,1,0,0,0,0,136,51,225,0,136,51,97,2,0,0,0,0,0,0,0,104,153,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,25,0,0,0,2,149,6,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,156,153,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,27,149,6,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,40,154,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,2,0,0,0,64,0,0,0,52,149,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,15,0,0,0,52,125,2,0,20,155,2,0,20,155,2,0,60,155,2,0,60,155,2,0,2,0,0,0,0,0,0,0,30,0,0,0,52,125,2,0,180,159,2,0,180,159,2,0,60,155,2,0,60,155,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,224,147,2,0,224,147,2,0,8,148,2,0,8,148,2,0,2,0,0,0,64,0,0,0,116,149,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,252,155,2,0,0,0,0,0,0,0,0,0,48,156,2,0,0,0,0,0,0,0,0,0,108,156,2,0,160,156,2,0,0,0,0,0,0,0,0,0,212,156,2,0,8,157,2,0,0,0,0,0,0,0,0,0,68,157,2,0,128,157,2,0,0,0,0,0,0,0,0,0,196,157,2,0,248,157,2,0,128,157,2,0,0,0,0,0,44,158,2,0,96,158,2,0,156,158,2,0,40,159,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,180,149,6,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,36,156,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,167,150,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,88,156,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,243,0,0,0,220,162,6,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,148,156,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,207,163,6,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,200,156,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,194,164,6,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,252,156,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,181,165,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,48,157,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,234,177,6,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,108,157,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,31,190,6,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,168,157,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,38,190,6,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,236,157,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,25,191,6,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,32,158,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,12,192,6,0,1,0,0,0,0,106,120,225,0,106,120,97,2,0,0,0,0,0,0,0,84,158,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,255,192,6,0,1,0,0,0,0,136,83,225,0,136,51,97,3,0,0,0,0,0,0,0,136,158,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,25,0,0,0,52,205,6,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,196,158,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,77,205,6,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,80,159,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,2,0,0,0,64,0,0,0,102,205,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,15,0,0,0,52,125,2,0,60,160,2,0,60,160,2,0,100,160,2,0,100,160,2,0,2,0,0,0,0,0,0,0,30,0,0,0,52,125,2,0,228,164,2,0,228,164,2,0,100,160,2,0,100,160,2,0,1,0,0,0,2,0,0,0,6,0,0,0,156,42,2,0,224,147,2,0,224,147,2,0,8,148,2,0,8,148,2,0,1,0,0,0,8,0,0,0,166,205,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,161,2,0,0,0,0,0,0,0,0,0,88,161,2,0,0,0,0,0,0,0,0,0,148,161,2,0,200,161,2,0,0,0,0,0,0,0,0,0,252,161,2,0,48,162,2,0,0,0,0,0,0,0,0,0,108,162,2,0,168,162,2,0,0,0,0,0,0,0,0,0,236,162,2,0,32,163,2,0,168,162,2,0,0,0,0,0,84,163,2,0,144,163,2,0,204,163,2,0,88,164,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,243,0,0,0,174,205,6,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,76,161,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,161,206,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,128,161,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,243,0,0,0,214,218,6,0,1,0,0,0,0,0,56,224,0,0,56,96,2,0,0,0,0,0,0,0,188,161,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,201,219,6,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,240,161,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,188,220,6,0,1,0,0,0,0,0,84,224,0,0,84,96,2,0,0,0,0,0,0,0,36,162,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,175,221,6,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,88,162,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,228,233,6,0,1,0,0,0,0,0,124,224,0,0,92,96,3,0,0,0,0,0,0,0,148,162,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,7,0,0,0,25,246,6,0,1,0,0,0,0,0,56,224,0,0,16,96,3,0,0,0,0,0,0,0,208,162,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,5,0,0,0,243,0,0,0,32,246,6,0,1,0,0,0,0,0,149,224,0,0,149,96,2,0,0,0,0,0,0,0,20,163,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,243,0,0,0,19,247,6,0,1,0,0,0,0,0,92,224,0,0,92,96,2,0,0,0,0,0,0,0,72,163,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,53,12,0,0,6,248,6,0,1,0,0,0,0,106,152,225,0,106,120,97,3,0,0,0,0,0,0,0,124,163,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,53,12,0,0,59,4,7,0,1,0,0,0,0,136,83,225,0,136,51,97,3,0,0,0,0,0,0,0,184,163,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,1,0,0,0,25,0,0,0,112,16,7,0,1,0,0,0,0,192,18,225,0,0,153,96,5,0,0,0,0,0,0,0,244,163,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,25,0,0,0,137,16,7,0,1,0,0,0,0,0,120,224,0,0,16,96,5,0,0,0,0,0,0,0,128,164,2,0,0,0,0,0,12,0,0,0,11,0,0,0,13,0,0,0,10,0,0,0,14,0,0,0,9,0,0,0,15,0,0,0,8,0,0,0,16,0,0,0,7,0,0,0,17,0,0,0,6,0,0,0,18,0,0,0,5,0,0,0,19,0,0,0,4,0,0,0,20,0,0,0,3,0,0,0,21,0,0,0,2,0,0,0,22,0,0,0,1,0,0,0,23,0,0,0,0,0,0,0,24,0,0,0,1,0,0,0,8,0,0,0,162,16,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,11,0,0,0,152,5,0,0,0,3,0,0,255,255,255,255,64,156,0,0,80,195,0,0,84,238,0,0,128,238,0,0,172,238,0,0,156,239,0,0,204,239,0,0,252,239,0,0,44,243,0,0,252,239,0,0,92,246,0,0,140,246,0,0,28,0,1,0,172,9,1,0,60,19,1,0,204,28,1,0,252,28,1,0,96,3,0,0,192,3,0,0,188,32,1,0,232,32,1,0,20,33,1,0,64,33,1,0,32,4,0,0,108,33,1,0,156,33,1,0,120,4,0,0,204,33,1,0,216,4,0,0,104,43,1,0,168,54,1,0,128,71,1,0,2,0,0,0,160,119,1,0,168,165,2,0,0,166,2,0,16,191,2,0,0,166,2,0,48,206,2,0,0,166,2,0,56,210,2,0,0,166,2,0,72,214,2,0,0,166,2,0,96,218,2,0,0,166,2,0,144,222,2,0,0,166,2,0,208,226,2,0,0,166,2,0,12,243,2,0,0,166,2,0,56,248,2,0,0,166,2,0,204,253,2,0,0,166,2,0,184,15,3,0,1], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+165448);
/* memory initializer */ allocate([1], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+176768);
/* memory initializer */ allocate([1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+177796);
/* memory initializer */ allocate([1,0,0,0,0,0,0,0,32,0,0,0,80,191,2,0,104,202,2,0,104,202,2,0,144,202,2,0,144,202,2,0,1,0,0,0,0,0,0,0,32,0,0,0,80,191,2,0,8,206,2,0,8,206,2,0,144,202,2,0,144,202,2,0,0,0,0,0,255,255,255,255,255,255,255,255,8,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+179976);
/* memory initializer */ allocate([1,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,28,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,25,0,0,0,255,255,255,255,45,0,0,0,255,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,64,0,0,0,170,16,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,80,203,2,0,0,0,0,0,0,0,0,0,0,0,0,0,132,203,2,0,0,0,0,0,0,0,0,0,0,0,0,0,184,203,2,0,0,0,0,0,0,0,0,0,0,0,0,0,244,203,2,0,0,0,0,0,0,0,0,0,0,0,0,0,48,204,2,0,0,0,0,0,124,204,2,0,216,204,2,0,0,0,0,0,0,0,0,0,20,205,2,0,80,205,2,0,172,205,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,234,16,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,120,203,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,59,17,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,172,203,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,140,17,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,224,203,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,253,19,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,28,204,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,110,22,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,88,204,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,191,22,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,164,204,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,104,23,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,0,205,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,129,23,7,0,1,0,0,0,0,32,21,225,0,32,245,96,3,0,0,0,0,0,0,0,60,205,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,242,25,7,0,1,0,0,0,0,128,211,224,0,0,122,96,4,0,0,0,0,0,0,0,120,205,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,169,0,0,0,155,26,7,0,1,0,0,0,0,0,88,224,0,0,16,96,4,0,0,0,0,0,0,0,212,205,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,64,0,0,0,68,27,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,16,0,0,0,80,191,2,0,112,206,2,0,112,206,2,0,152,206,2,0,152,206,2,0,1,0,0,0,0,0,0,0,32,0,0,0,80,191,2,0,16,210,2,0,16,210,2,0,152,206,2,0,152,206,2,0,2,0,0,0,64,0,0,0,132,27,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,88,207,2,0,0,0,0,0,0,0,0,0,0,0,0,0,140,207,2,0,0,0,0,0,0,0,0,0,0,0,0,0,192,207,2,0,0,0,0,0,0,0,0,0,0,0,0,0,252,207,2,0,0,0,0,0,0,0,0,0,0,0,0,0,56,208,2,0,0,0,0,0,132,208,2,0,224,208,2,0,0,0,0,0,0,0,0,0,28,209,2,0,88,209,2,0,180,209,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,196,27,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,128,207,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,21,28,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,180,207,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,102,28,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,232,207,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,215,30,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,36,208,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,72,33,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,96,208,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,153,33,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,172,208,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,66,34,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,8,209,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,91,34,7,0,1,0,0,0,0,32,21,225,0,32,245,96,3,0,0,0,0,0,0,0,68,209,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,204,36,7,0,1,0,0,0,0,128,211,224,0,0,122,96,4,0,0,0,0,0,0,0,128,209,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,169,0,0,0,117,37,7,0,1,0,0,0,0,0,88,224,0,0,16,96,4,0,0,0,0,0,0,0,220,209,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,64,0,0,0,30,38,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,16,0,0,0,80,191,2,0,120,210,2,0,120,210,2,0,160,210,2,0,160,210,2,0,1,0,0,0,0,0,0,0,32,0,0,0,80,191,2,0,32,214,2,0,32,214,2,0,160,210,2,0,160,210,2,0,2,0,0,0,64,0,0,0,94,38,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,96,211,2,0,0,0,0,0,0,0,0,0,0,0,0,0,148,211,2,0,0,0,0,0,0,0,0,0,0,0,0,0,200,211,2,0,0,0,0,0,0,0,0,0,0,0,0,0,4,212,2,0,0,0,0,0,0,0,0,0,0,0,0,0,64,212,2,0,0,0,0,0,140,212,2,0,232,212,2,0,0,0,0,0,0,0,0,0,36,213,2,0,104,213,2,0,196,213,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,158,38,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,136,211,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,239,38,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,188,211,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,64,39,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,240,211,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,177,41,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,44,212,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,34,44,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,104,212,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,115,44,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,180,212,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,28,45,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,16,213,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,49,0,0,0,53,45,7,0,1,0,0,0,0,176,31,225,0,32,245,96,3,0,0,0,0,0,0,0,76,213,2,0,0,0,0,0,3,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,5,0,0,0,0,0,0,0,6,0,0,0,2,0,0,0,169,0,0,0,102,45,7,0,1,0,0,0,0,128,211,224,0,0,122,96,4,0,0,0,0,0,0,0,144,213,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,169,0,0,0,15,46,7,0,1,0,0,0,0,0,88,224,0,0,16,96,4,0,0,0,0,0,0,0,236,213,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,64,0,0,0,184,46,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,16,0,0,0,80,191,2,0,136,214,2,0,136,214,2,0,176,214,2,0,176,214,2,0,1,0,0,0,0,0,0,0,32,0,0,0,80,191,2,0,56,218,2,0,56,218,2,0,176,214,2,0,176,214,2,0,2,0,0,0,64,0,0,0,248,46,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,215,2,0,0,0,0,0,0,0,0,0,0,0,0,0,164,215,2,0,0,0,0,0,0,0,0,0,0,0,0,0,216,215,2,0,0,0,0,0,0,0,0,0,0,0,0,0,20,216,2,0,0,0,0,0,0,0,0,0,0,0,0,0,80,216,2,0,0,0,0,0,156,216,2,0,248,216,2,0,0,0,0,0,0,0,0,0,52,217,2,0,128,217,2,0,220,217,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,56,47,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,152,215,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,137,47,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,204,215,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,218,47,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,0,216,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,75,50,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,60,216,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,188,52,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,120,216,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,13,53,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,196,216,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,182,53,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,32,217,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,207,53,7,0,1,0,0,0,0,32,53,225,0,32,245,96,4,0,0,0,0,0,0,0,92,217,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,32,54,7,0,1,0,0,0,0,128,211,224,0,0,122,96,4,0,0,0,0,0,0,0,168,217,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,169,0,0,0,201,54,7,0,1,0,0,0,0,0,88,224,0,0,16,96,4,0,0,0,0,0,0,0,4,218,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,64,0,0,0,114,55,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,16,0,0,0,80,191,2,0,160,218,2,0,160,218,2,0,200,218,2,0,200,218,2,0,1,0,0,0,0,0,0,0,32,0,0,0,80,191,2,0,104,222,2,0,104,222,2,0,200,218,2,0,200,218,2,0,2,0,0,0,64,0,0,0,178,55,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,136,219,2,0,0,0,0,0,0,0,0,0,0,0,0,0,188,219,2,0,0,0,0,0,0,0,0,0,0,0,0,0,240,219,2,0,0,0,0,0,0,0,0,0,0,0,0,0,44,220,2,0,0,0,0,0,0,0,0,0,0,0,0,0,104,220,2,0,0,0,0,0,180,220,2,0,16,221,2,0,0,0,0,0,0,0,0,0,76,221,2,0,152,221,2,0,252,221,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,242,55,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,176,219,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,67,56,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,228,219,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,148,56,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,24,220,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,5,59,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,84,220,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,118,61,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,144,220,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,199,61,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,220,220,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,112,62,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,56,221,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,137,62,7,0,1,0,0,0,0,224,63,225,0,224,255,96,4,0,0,0,0,0,0,0,116,221,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,225,0,0,0,218,62,7,0,1,0,0,0,0,192,221,224,0,0,145,96,4,0,0,0,0,0,0,0,192,221,2,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,33,1,0,0,187,63,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,36,222,2,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,64,0,0,0,220,64,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,16,0,0,0,80,191,2,0,208,222,2,0,208,222,2,0,248,222,2,0,248,222,2,0,1,0,0,0,0,0,0,0,32,0,0,0,80,191,2,0,168,226,2,0,168,226,2,0,248,222,2,0,248,222,2,0,2,0,0,0,64,0,0,0,28,65,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,184,223,2,0,0,0,0,0,0,0,0,0,0,0,0,0,236,223,2,0,0,0,0,0,0,0,0,0,0,0,0,0,32,224,2,0,0,0,0,0,0,0,0,0,0,0,0,0,92,224,2,0,0,0,0,0,0,0,0,0,0,0,0,0,152,224,2,0,0,0,0,0,228,224,2,0,64,225,2,0,0,0,0,0,0,0,0,0,124,225,2,0,216,225,2,0,60,226,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,92,65,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,224,223,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,173,65,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,20,224,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,254,65,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,72,224,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,111,68,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,132,224,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,224,70,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,192,224,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,49,71,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,12,225,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,218,71,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,104,225,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,243,71,7,0,1,0,0,0,0,232,87,225,0,224,255,96,4,0,0,0,0,0,0,0,164,225,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,225,0,0,0,156,72,7,0,1,0,0,0,0,192,221,224,0,0,145,96,4,0,0,0,0,0,0,0,0,226,2,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,33,1,0,0,125,73,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,100,226,2,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,64,0,0,0,158,74,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,16,0,0,0,16,227,2,0,40,238,2,0,40,238,2,0,80,238,2,0,80,238,2,0,1,0,0,0,0,0,0,0,32,0,0,0,16,227,2,0,228,242,2,0,228,242,2,0,80,238,2,0,80,238,2,0,0,0,0,0,255,255,255,255,255,255,255,255,10,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+182372);
/* memory initializer */ allocate([1,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,16,0,0,0,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,30,0,0,0,255,255,255,255,50,0,0,0,255,255,255,255,80,0,0,0,255,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,100,0,0,0,222,74,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,239,2,0,0,0,0,0,0,0,0,0,0,0,0,0,68,239,2,0,0,0,0,0,0,0,0,0,0,0,0,0,120,239,2,0,0,0,0,0,0,0,0,0,0,0,0,0,180,239,2,0,0,0,0,0,0,0,0,0,0,0,0,0,240,239,2,0,0,0,0,0,0,0,0,0,0,0,0,0,60,240,2,0,0,0,0,0,136,240,2,0,188,240,2,0,0,0,0,0,0,0,0,0,16,241,2,0,100,241,2,0,0,0,0,0,0,0,0,0,184,241,2,0,20,242,2,0,120,242,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,66,75,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,56,239,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,147,75,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,108,239,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,228,75,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,160,239,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,85,78,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,220,239,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,198,80,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,24,240,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,23,81,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,100,240,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,4,0,0,0,81,0,0,0,104,81,7,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,176,240,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,185,81,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,228,240,2,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,50,82,7,0,1,0,0,0,0,128,187,224,0,0,118,96,4,0,0,0,0,0,0,0,56,241,2,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,171,82,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,140,241,2,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,36,83,7,0,1,0,0,0,0,232,87,225,0,224,255,96,4,0,0,0,0,0,0,0,224,241,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,225,0,0,0,205,83,7,0,1,0,0,0,0,192,221,224,0,0,145,96,4,0,0,0,0,0,0,0,60,242,2,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,33,1,0,0,174,84,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,160,242,2,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,100,0,0,0,207,85,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,16,0,0,0,16,227,2,0,76,243,2,0,76,243,2,0,116,243,2,0,116,243,2,0,1,0,0,0,0,0,0,0,32,0,0,0,16,227,2,0,16,248,2,0,16,248,2,0,116,243,2,0,116,243,2,0,2,0,0,0,100,0,0,0,51,86,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,52,244,2,0,0,0,0,0,0,0,0,0,0,0,0,0,104,244,2,0,0,0,0,0,0,0,0,0,0,0,0,0,156,244,2,0,0,0,0,0,0,0,0,0,0,0,0,0,216,244,2,0,0,0,0,0,0,0,0,0,0,0,0,0,20,245,2,0,0,0,0,0,0,0,0,0,0,0,0,0,96,245,2,0,0,0,0,0,172,245,2,0,224,245,2,0,0,0,0,0,0,0,0,0,52,246,2,0,136,246,2,0,0,0,0,0,0,0,0,0,220,246,2,0,64,247,2,0,164,247,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,151,86,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,92,244,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,232,86,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,144,244,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,57,87,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,196,244,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,170,89,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,0,245,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,27,92,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,60,245,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,108,92,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,136,245,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,4,0,0,0,81,0,0,0,189,92,7,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,212,245,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,14,93,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,8,246,2,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,135,93,7,0,1,0,0,0,0,128,187,224,0,0,118,96,4,0,0,0,0,0,0,0,92,246,2,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,0,94,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,176,246,2,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,121,94,7,0,1,0,0,0,0,228,91,225,0,224,255,96,4,0,0,0,0,0,0,0,4,247,2,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,225,0,0,0,90,95,7,0,1,0,0,0,0,192,221,224,0,0,145,96,4,0,0,0,0,0,0,0,104,247,2,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,33,1,0,0,59,96,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,204,247,2,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,100,0,0,0,92,97,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,16,0,0,0,16,227,2,0,120,248,2,0,120,248,2,0,160,248,2,0,160,248,2,0,1,0,0,0,0,0,0,0,32,0,0,0,16,227,2,0,164,253,2,0,164,253,2,0,160,248,2,0,160,248,2,0,2,0,0,0,100,0,0,0,192,97,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,96,249,2,0,0,0,0,0,0,0,0,0,0,0,0,0,148,249,2,0,0,0,0,0,0,0,0,0,0,0,0,0,200,249,2,0,0,0,0,0,0,0,0,0,0,0,0,0,4,250,2,0,0,0,0,0,0,0,0,0,0,0,0,0,64,250,2,0,0,0,0,0,0,0,0,0,0,0,0,0,140,250,2,0,0,0,0,0,216,250,2,0,12,251,2,0,0,0,0,0,0,0,0,0,96,251,2,0,180,251,2,0,0,0,0,0,0,0,0,0,8,252,2,0,92,252,2,0,184,252,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,36,98,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,136,249,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,117,98,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,188,249,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,198,98,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,240,249,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,55,101,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,44,250,2,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,168,103,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,104,250,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,249,103,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,180,250,2,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,4,0,0,0,81,0,0,0,74,104,7,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,0,251,2,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,155,104,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,52,251,2,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,20,105,7,0,1,0,0,0,0,128,187,224,0,0,118,96,4,0,0,0,0,0,0,0,136,251,2,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,141,105,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,220,251,2,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,6,106,7,0,1,0,0,0,0,226,120,225,0,232,51,97,4,0,0,0,0,0,0,0,48,252,2,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,127,106,7,0,1,0,0,0,0,96,18,225,0,128,184,96,4,0,0,0,0,0,0,0,132,252,2,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,1,0,0,0,49,0,0,0,40,107,7,0,1,0,0,0,0,0,152,224,0,0,16,96,6,0,0,0,0,0,0,0,224,252,2,0,0,0,0,0,24,0,0,0,23,0,0,0,25,0,0,0,22,0,0,0,26,0,0,0,21,0,0,0,27,0,0,0,20,0,0,0,28,0,0,0,19,0,0,0,29,0,0,0,18,0,0,0,30,0,0,0,17,0,0,0,31,0,0,0,16,0,0,0,32,0,0,0,15,0,0,0,33,0,0,0,14,0,0,0,34,0,0,0,13,0,0,0,35,0,0,0,12,0,0,0,36,0,0,0,11,0,0,0,37,0,0,0,10,0,0,0,38,0,0,0,9,0,0,0,39,0,0,0,8,0,0,0,40,0,0,0,7,0,0,0,41,0,0,0,6,0,0,0,42,0,0,0,5,0,0,0,43,0,0,0,4,0,0,0,44,0,0,0,3,0,0,0,45,0,0,0,2,0,0,0,46,0,0,0,1,0,0,0,47,0,0,0,0,0,0,0,48,0,0,0,2,0,0,0,100,0,0,0,89,107,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,16,0,0,0,12,254,2,0,36,9,3,0,36,9,3,0,76,9,3,0,76,9,3,0,1,0,0,0,0,0,0,0,32,0,0,0,12,254,2,0,144,15,3,0,144,15,3,0,76,9,3,0,76,9,3,0,0,0,0,0,255,255,255,255,255,255,255,255,10,0,0,0,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+191524);
/* memory initializer */ allocate([1,0,0,0,2,0,0,0,4,0,0,0,8,0,0,0,16,0,0,0,32,0,0,0,71,0,0,0,157,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,100,0,0,0,189,107,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,10,3,0,0,0,0,0,0,0,0,0,0,0,0,0,64,10,3,0,0,0,0,0,0,0,0,0,0,0,0,0,124,10,3,0,0,0,0,0,0,0,0,0,0,0,0,0,200,10,3,0,0,0,0,0,52,11,3,0,104,11,3,0,0,0,0,0,0,0,0,0,188,11,3,0,24,12,3,0,0,0,0,0,0,0,0,0,84,12,3,0,176,12,3,0,0,0,0,0,0,0,0,0,4,13,3,0,104,13,3,0,0,0,0,0,0,0,0,0,228,13,3,0,48,14,3,0,164,14,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,33,108,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,52,10,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,114,108,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,104,10,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,227,110,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,164,10,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,52,111,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,240,10,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,85,112,7,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,92,11,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,166,112,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,144,11,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,31,113,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,228,11,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,200,113,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,64,12,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,225,113,7,0,1,0,0,0,0,128,208,224,0,0,118,96,4,0,0,0,0,0,0,0,124,12,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,121,0,0,0,138,114,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,216,12,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,3,115,7,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,44,13,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,228,115,7,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,144,13,3,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,81,0,0,0,157,117,7,0,1,0,0,0,0,24,125,225,0,24,61,97,4,0,0,0,0,0,0,0,12,14,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,105,1,0,0,238,117,7,0,1,0,0,0,0,144,27,225,0,128,184,96,5,0,0,0,0,0,0,0,88,14,3,0,0,0,0,0,9,0,0,0,8,0,0,0,10,0,0,0,7,0,0,0,11,0,0,0,6,0,0,0,12,0,0,0,5,0,0,0,13,0,0,0,4,0,0,0,14,0,0,0,3,0,0,0,15,0,0,0,2,0,0,0,16,0,0,0,1,0,0,0,17,0,0,0,0,0,0,0,18,0,0,0,1,0,0,0,49,0,0,0,87,119,7,0,1,0,0,0,0,0,152,224,0,0,16,96,6,0,0,0,0,0,0,0,204,14,3,0,0,0,0,0,24,0,0,0,23,0,0,0,25,0,0,0,22,0,0,0,26,0,0,0,21,0,0,0,27,0,0,0,20,0,0,0,28,0,0,0,19,0,0,0,29,0,0,0,18,0,0,0,30,0,0,0,17,0,0,0,31,0,0,0,16,0,0,0,32,0,0,0,15,0,0,0,33,0,0,0,14,0,0,0,34,0,0,0,13,0,0,0,35,0,0,0,12,0,0,0,36,0,0,0,11,0,0,0,37,0,0,0,10,0,0,0,38,0,0,0,9,0,0,0,39,0,0,0,8,0,0,0,40,0,0,0,7,0,0,0,41,0,0,0,6,0,0,0,42,0,0,0,5,0,0,0,43,0,0,0,4,0,0,0,44,0,0,0,3,0,0,0,45,0,0,0,2,0,0,0,46,0,0,0,1,0,0,0,47,0,0,0,0,0,0,0,48,0,0,0,2,0,0,0,100,0,0,0,136,119,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,16,0,0,0,12,254,2,0,248,15,3,0,248,15,3,0,32,16,3,0,32,16,3,0,1,0,0,0,0,0,0,0,32,0,0,0,12,254,2,0,124,22,3,0,124,22,3,0,32,16,3,0,32,16,3,0,2,0,0,0,100,0,0,0,236,119,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,224,16,3,0,0,0,0,0,0,0,0,0,0,0,0,0,20,17,3,0,0,0,0,0,0,0,0,0,0,0,0,0,80,17,3,0,0,0,0,0,0,0,0,0,0,0,0,0,156,17,3,0,0,0,0,0,8,18,3,0,60,18,3,0,0,0,0,0,0,0,0,0,144,18,3,0,236,18,3,0,0,0,0,0,0,0,0,0,40,19,3,0,132,19,3,0,0,0,0,0,0,0,0,0,216,19,3,0,60,20,3,0,0,0,0,0,0,0,0,0,184,20,3,0,28,21,3,0,144,21,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,80,120,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,8,17,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,161,120,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,60,17,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,18,123,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,120,17,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,99,123,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,196,17,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,132,124,7,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,48,18,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,213,124,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,100,18,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,78,125,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,184,18,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,247,125,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,20,19,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,16,126,7,0,1,0,0,0,0,128,208,224,0,0,118,96,4,0,0,0,0,0,0,0,80,19,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,121,0,0,0,185,126,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,172,19,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,50,127,7,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,0,20,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,19,128,7,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,100,20,3,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,225,0,0,0,204,129,7,0,1,0,0,0,0,117,153,225,0,24,61,97,4,0,0,0,0,0,0,0,224,20,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,105,1,0,0,173,130,7,0,1,0,0,0,0,144,27,225,0,128,184,96,5,0,0,0,0,0,0,0,68,21,3,0,0,0,0,0,9,0,0,0,8,0,0,0,10,0,0,0,7,0,0,0,11,0,0,0,6,0,0,0,12,0,0,0,5,0,0,0,13,0,0,0,4,0,0,0,14,0,0,0,3,0,0,0,15,0,0,0,2,0,0,0,16,0,0,0,1,0,0,0,17,0,0,0,0,0,0,0,18,0,0,0,1,0,0,0,49,0,0,0,22,132,7,0,1,0,0,0,0,0,152,224,0,0,16,96,6,0,0,0,0,0,0,0,184,21,3,0,0,0,0,0,24,0,0,0,23,0,0,0,25,0,0,0,22,0,0,0,26,0,0,0,21,0,0,0,27,0,0,0,20,0,0,0,28,0,0,0,19,0,0,0,29,0,0,0,18,0,0,0,30,0,0,0,17,0,0,0,31,0,0,0,16,0,0,0,32,0,0,0,15,0,0,0,33,0,0,0,14,0,0,0,34,0,0,0,13,0,0,0,35,0,0,0,12,0,0,0,36,0,0,0,11,0,0,0,37,0,0,0,10,0,0,0,38,0,0,0,9,0,0,0,39,0,0,0,8,0,0,0,40,0,0,0,7,0,0,0,41,0,0,0,6,0,0,0,42,0,0,0,5,0,0,0,43,0,0,0,4,0,0,0,44,0,0,0,3,0,0,0,45,0,0,0,2,0,0,0,46,0,0,0,1,0,0,0,47,0,0,0,0,0,0,0,48,0,0,0,2,0,0,0,100,0,0,0,71,132,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,11,0,0,0,248,5,0,0,0,3,0,0,2,0,0,0,144,101,0,0,64,156,0,0,84,238,0,0,128,238,0,0,172,238,0,0,156,239,0,0,204,239,0,0,252,239,0,0,44,243,0,0,252,239,0,0,92,246,0,0,140,246,0,0,28,0,1,0,172,9,1,0,60,19,1,0,204,28,1,0,252,28,1,0,96,3,0,0,192,3,0,0,188,32,1,0,232,32,1,0,20,33,1,0,64,33,1,0,32,4,0,0,108,33,1,0,156,33,1,0,88,6,0,0,204,33,1,0,216,4,0,0,104,43,1,0,168,54,1,0,128,71,1,0,2,0,0,0,160,119,1,0,48,120,1,0,11,0,0,0,184,6,0,0,0,3,0,0,255,255,255,255,144,101,0,0,64,156,0,0,84,238,0,0,128,238,0,0,172,238,0,0,156,239,0,0,204,239,0,0,252,239,0,0,44,243,0,0,252,239,0,0,92,246,0,0,140,246,0,0,28,0,1,0,172,9,1,0,60,19,1,0,204,28,1,0,252,28,1,0,96,3,0,0,192,3,0,0,188,32,1,0,232,32,1,0,20,33,1,0,64,33,1,0,32,4,0,0,108,33,1,0,156,33,1,0,88,6,0,0,204,33,1,0,216,4,0,0,0,0,0,0,168,54,1,0,128,71,1,0,2,0,0,0,160,119,1,0,168,165,2,0,3,0,0,0,24,7,0,0,56,7,0,0,2,0,0,0,56,74,0,0,144,101,0,0,120,24,3,0,132,24,3,0,144,24,3,0,156,239,0,0,204,239,0,0,224,24,3,0,224,24,3,0,224,24,3,0,240,25,3,0,32,26,3,0,80,29,3,0,80,29,3,0,128,32,3,0,204,28,1,0,176,35,3,0,88,7,0,0,88,7,0,0,240,36,3,0,240,36,3,0,252,36,3,0,252,36,3,0,120,7,0,0,12,37,3,0,28,37,3,0,152,7,0,0,204,33,1,0,184,7,0,0,44,37,3,0,168,54,1,0,128,71,1,0,2,0,0,0,236,40,3,0,12,41,3,0,0,4,0,0,0,2,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,4,0,0,30,0,0,0,25,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,25,0,0,0,22,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,20,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,15,0,0,0,0,0,0,0,242,255,255,255,0,0,0,0,0,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,246,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,246,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,246,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,236,255,255,255,246,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,0,0,0,10,0,0,0,255,255,255,255,10,0,0,0,10,0,0,0,255,255,255,255,20,0,0,0,20,0,0,0,255,255,255,255,20,0,0,0,20,0,0,0,255,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,4,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,12,0,0,0,12,0,0,0,14,0,0,0,20,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,6,0,0,0,8,0,0,0,8,0,0,0,15,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,6,0,0,0,8,0,0,0,15,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,10,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,10,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,244,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,249,255,255,255,251,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,6,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,238,255,255,255,238,255,255,255,238,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,4,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,12,0,0,0,12,0,0,0,14,0,0,0,20,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,6,0,0,0,8,0,0,0,8,0,0,0,15,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,4,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,8,0,0,0,10,0,0,0,12,0,0,0,20,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,4,0,0,0,5,0,0,0,6,0,0,0,8,0,0,0,8,0,0,0,15,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,8,0,0,0,12,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,244,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,6,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,244,255,255,255,244,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,250,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,4,0,0,0,6,0,0,0,8,0,0,0,8,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,14,0,0,0,20,0,0,0,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,254,255,255,255,254,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,8,0,0,0,8,0,0,0,15,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,4,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,8,0,0,0,10,0,0,0,12,0,0,0,20,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,8,0,0,0,8,0,0,0,15,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,250,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,8,0,0,0,12,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,244,255,255,255,236,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,5,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,6,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,244,255,255,255,244,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,250,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,8,0,0,0,9,0,0,0,9,0,0,0,10,0,0,0,10,0,0,0,11,0,0,0,11,0,0,0,12,0,0,0,12,0,0,0,13,0,0,0,13,0,0,0,14,0,0,0,14,0,0,0,15,0,0,0,15,0,0,0,16,0,0,0,16,0,0,0,17,0,0,0,17,0,0,0,17,0,0,0,18,0,0,0,18,0,0,0,19,0,0,0,19,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,23,0,0,0,24,0,0,0,25,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,6,0,0,0,7,0,0,0,7,0,0,0,6,0,0,0,6,0,0,0,5,0,0,0,5,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,14,0,0,0,15,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,23,0,0,0,24,0,0,0,0,1,0,0,0,1,0,0,15,39,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,156,255,255,255,156,255,255,255,156,255,255,255,151,255,255,255,126,255,255,255,126,255,255,255,126,255,255,255,116,255,255,255,4,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,6,0,0,0,5,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,64,64,0,0,64,64,0,0,128,64,0,0,128,64,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,4,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,6,0,0,0,5,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,0,64,0,0,64,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,5,0,0,0,4,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,128,64,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,0,65,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,244,40,3,0,0,41,3,0,9,0,0,0,3,0,0,0,3,0,0,0,9,0,0,0,9,0,0,0,9,0,0,0,136,120,1,0,36,41,3,0,136,120,1,0,212,45,3,0,136,120,1,0,244,50,3,0,2,0,0,0,0,0,0,0,32,0,0,0,0,186,1,0,68,41,3,0,68,41,3,0,108,41,3,0,108,41,3,0,2,0,0,0,100,0,0,0,171,132,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,44,42,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,96,42,3,0,0,0,0,0,0,0,0,0,0,0,0,0,156,42,3,0,0,0,0,0,0,0,0,0,0,0,0,0,232,42,3,0,0,0,0,0,0,0,0,0,0,0,0,0,52,43,3,0,0,0,0,0,160,43,3,0,212,43,3,0,0,0,0,0,0,0,0,0,40,44,3,0,132,44,3,0,0,0,0,0,0,0,0,0,192,44,3,0,244,44,3,0,88,45,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,15,133,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,84,42,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,176,158,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,136,42,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,33,161,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,196,42,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,114,161,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,16,43,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,195,161,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,92,43,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,228,162,7,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,200,43,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,53,163,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,252,43,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,174,163,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,80,44,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,87,164,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,172,44,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,81,0,0,0,112,164,7,0,1,0,0,0,0,176,19,225,0,176,19,97,2,0,0,0,0,0,0,0,232,44,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,225,0,0,0,193,164,7,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,28,45,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,162,165,7,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,128,45,3,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,0,0,0,0,32,0,0,0,0,186,1,0,20,46,3,0,20,46,3,0,60,46,3,0,60,46,3,0,2,0,0,0,0,0,0,0,32,0,0,0,0,186,1,0,204,50,3,0,204,50,3,0,60,46,3,0,60,46,3,0,2,0,0,0,100,0,0,0,91,167,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,252,46,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,47,3,0,0,0,0,0,0,0,0,0,0,0,0,0,108,47,3,0,0,0,0,0,0,0,0,0,0,0,0,0,184,47,3,0,0,0,0,0,0,0,0,0,0,0,0,0,4,48,3,0,0,0,0,0,112,48,3,0,164,48,3,0,0,0,0,0,0,0,0,0,248,48,3,0,84,49,3,0,0,0,0,0,0,0,0,0,144,49,3,0,236,49,3,0,80,50,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,191,167,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,36,47,3,0,0,0,0,0,1], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+198432);
/* memory initializer */ allocate([2,0,0,0,4,0,0,0,113,2,0,0,96,193,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,88,47,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,209,195,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,148,47,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,34,196,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,224,47,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,115,196,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,44,48,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,148,197,7,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,152,48,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,229,197,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,204,48,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,94,198,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,32,49,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,7,199,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,124,49,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,32,199,7,0,1,0,0,0,0,136,93,225,0,176,19,97,4,0,0,0,0,0,0,0,184,49,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,225,0,0,0,201,199,7,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,20,50,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,170,200,7,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,120,50,3,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,100,0,0,0,99,202,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,0,0,0,0,32,0,0,0,112,218,1,0,52,51,3,0,52,51,3,0,92,51,3,0,92,51,3,0,2,0,0,0,0,0,0,0,32,0,0,0,112,218,1,0,192,57,3,0,192,57,3,0,92,51,3,0,92,51,3,0,2,0,0,0,100,0,0,0,199,202,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,28,52,3,0,0,0,0,0,0,0,0,0,0,0,0,0,80,52,3,0,0,0,0,0,0,0,0,0,0,0,0,0,140,52,3,0,0,0,0,0,0,0,0,0,0,0,0,0,216,52,3,0,0,0,0,0,68,53,3,0,120,53,3,0,0,0,0,0,0,0,0,0,204,53,3,0,40,54,3,0,0,0,0,0,0,0,0,0,100,54,3,0,192,54,3,0,0,0,0,0,0,0,0,0,20,55,3,0,120,55,3,0,0,0,0,0,0,0,0,0,244,55,3,0,96,56,3,0,212,56,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,43,203,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,68,52,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,124,203,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,120,52,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,237,205,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,180,52,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,62,206,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,0,53,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,95,207,7,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,108,53,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,176,207,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,160,53,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,41,208,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,244,53,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,210,208,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,80,54,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,235,208,7,0,1,0,0,0,0,128,208,224,0,0,118,96,4,0,0,0,0,0,0,0,140,54,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,121,0,0,0,148,209,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,232,54,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,13,210,7,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,60,55,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,238,210,7,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,160,55,3,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,33,1,0,0,167,212,7,0,1,0,0,0,0,24,157,225,0,24,61,97,5,0,0,0,0,0,0,0,28,56,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,105,1,0,0,200,213,7,0,1,0,0,0,0,144,27,225,0,128,184,96,5,0,0,0,0,0,0,0,136,56,3,0,0,0,0,0,9,0,0,0,8,0,0,0,10,0,0,0,7,0,0,0,11,0,0,0,6,0,0,0,12,0,0,0,5,0,0,0,13,0,0,0,4,0,0,0,14,0,0,0,3,0,0,0,15,0,0,0,2,0,0,0,16,0,0,0,1,0,0,0,17,0,0,0,0,0,0,0,18,0,0,0,1,0,0,0,49,0,0,0,49,215,7,0,1,0,0,0,0,0,152,224,0,0,16,96,6,0,0,0,0,0,0,0,252,56,3,0,0,0,0,0,24,0,0,0,23,0,0,0,25,0,0,0,22,0,0,0,26,0,0,0,21,0,0,0,27,0,0,0,20,0,0,0,28,0,0,0,19,0,0,0,29,0,0,0,18,0,0,0,30,0,0,0,17,0,0,0,31,0,0,0,16,0,0,0,32,0,0,0,15,0,0,0,33,0,0,0,14,0,0,0,34,0,0,0,13,0,0,0,35,0,0,0,12,0,0,0,36,0,0,0,11,0,0,0,37,0,0,0,10,0,0,0,38,0,0,0,9,0,0,0,39,0,0,0,8,0,0,0,40,0,0,0,7,0,0,0,41,0,0,0,6,0,0,0,42,0,0,0,5,0,0,0,43,0,0,0,4,0,0,0,44,0,0,0,3,0,0,0,45,0,0,0,2,0,0,0,46,0,0,0,1,0,0,0,47,0,0,0,0,0,0,0,48,0,0,0,2,0,0,0,100,0,0,0,98,215,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,216,7,0,0,56,7,0,0,255,255,255,255,56,74,0,0,144,101,0,0,120,24,3,0,132,24,3,0,144,24,3,0,156,239,0,0,204,239,0,0,224,24,3,0,224,24,3,0,224,24,3,0,240,25,3,0,32,26,3,0,80,29,3,0,80,29,3,0,128,32,3,0,204,28,1,0,176,35,3,0,88,7,0,0,88,7,0,0,240,36,3,0,240,36,3,0,252,36,3,0,252,36,3,0,120,7,0,0,12,37,3,0,28,37,3,0,152,7,0,0,204,33,1,0,184,7,0,0,44,37,3,0,168,54,1,0,128,71,1,0,2,0,0,0,236,40,3,0,132,58,3,0,0,166,2,0,156,58,3,0,0,166,2,0,124,62,3,0,0,166,2,0,168,67,3,0,1,0,0,0,0,0,0,0,32,0,0,0,80,191,2,0,188,58,3,0,188,58,3,0,228,58,3,0,228,58,3,0,2,0,0,0,64,0,0,0,198,215,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,164,59,3,0,0,0,0,0,0,0,0,0,0,0,0,0,216,59,3,0,0,0,0,0,0,0,0,0,0,0,0,0,12,60,3,0,0,0,0,0,0,0,0,0,0,0,0,0,72,60,3,0,0,0,0,0,0,0,0,0,0,0,0,0,132,60,3,0,0,0,0,0,208,60,3,0,44,61,3,0,0,0,0,0,0,0,0,0,104,61,3,0,156,61,3,0,0,62,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,6,216,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,204,59,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,87,216,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,0,60,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,168,216,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,52,60,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,25,219,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,112,60,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,138,221,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,172,60,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,219,221,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,248,60,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,132,222,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,84,61,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,81,0,0,0,157,222,7,0,1,0,0,0,0,176,19,225,0,176,19,97,2,0,0,0,0,0,0,0,144,61,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,225,0,0,0,238,222,7,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,196,61,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,207,223,7,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,40,62,3,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,1,0,0,0,0,0,0,0,32,0,0,0,16,227,2,0,188,62,3,0,188,62,3,0,228,62,3,0,228,62,3,0,1,0,0,0,0,0,0,0,32,0,0,0,16,227,2,0,128,67,3,0,128,67,3,0,228,62,3,0,228,62,3,0,2,0,0,0,100,0,0,0,136,225,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,164,63,3,0,0,0,0,0,0,0,0,0,0,0,0,0,216,63,3,0,0,0,0,0,0,0,0,0,0,0,0,0,12,64,3,0,0,0,0,0,0,0,0,0,0,0,0,0,72,64,3,0,0,0,0,0,0,0,0,0,0,0,0,0,132,64,3,0,0,0,0,0,0,0,0,0,0,0,0,0,208,64,3,0,0,0,0,0,28,65,3,0,80,65,3,0,0,0,0,0,0,0,0,0,164,65,3,0,248,65,3,0,0,0,0,0,0,0,0,0,76,66,3,0,176,66,3,0,20,67,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,236,225,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,204,63,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,61,226,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,0,64,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,142,226,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,52,64,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,255,228,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,112,64,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,112,231,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,172,64,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,193,231,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,248,64,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,4,0,0,0,81,0,0,0,18,232,7,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,68,65,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,99,232,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,120,65,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,220,232,7,0,1,0,0,0,0,128,187,224,0,0,118,96,4,0,0,0,0,0,0,0,204,65,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,85,233,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,32,66,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,206,233,7,0,1,0,0,0,0,228,91,225,0,224,255,96,4,0,0,0,0,0,0,0,116,66,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,225,0,0,0,175,234,7,0,1,0,0,0,0,192,221,224,0,0,145,96,4,0,0,0,0,0,0,0,216,66,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,33,1,0,0,144,235,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,60,67,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,100,0,0,0,177,236,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,32,0,0,0,12,254,2,0,232,67,3,0,232,67,3,0,16,68,3,0,16,68,3,0,1,0,0,0,0,0,0,0,32,0,0,0,12,254,2,0,108,74,3,0,108,74,3,0,16,68,3,0,16,68,3,0,2,0,0,0,100,0,0,0,21,237,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,208,68,3,0,0,0,0,0,0,0,0,0,0,0,0,0,4,69,3,0,0,0,0,0,0,0,0,0,0,0,0,0,64,69,3,0,0,0,0,0,0,0,0,0,0,0,0,0,140,69,3,0,0,0,0,0,248,69,3,0,44,70,3,0,0,0,0,0,0,0,0,0,128,70,3,0,220,70,3,0,0,0,0,0,0,0,0,0,24,71,3,0,116,71,3,0,0,0,0,0,0,0,0,0,200,71,3,0,44,72,3,0,0,0,0,0,0,0,0,0,168,72,3,0,12,73,3,0,128,73,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,121,237,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,248,68,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,202,237,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,44,69,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,59,240,7,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,104,69,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,140,240,7,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,180,69,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,173,241,7,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,32,70,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,254,241,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,84,70,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,119,242,7,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,168,70,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,32,243,7,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,4,71,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,57,243,7,0,1,0,0,0,0,128,208,224,0,0,118,96,4,0,0,0,0,0,0,0,64,71,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,121,0,0,0,226,243,7,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,156,71,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,91,244,7,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,240,71,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,60,245,7,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,84,72,3,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,225,0,0,0,245,246,7,0,1,0,0,0,0,117,153,225,0,24,61,97,4,0,0,0,0,0,0,0,208,72,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,105,1,0,0,214,247,7,0,1,0,0,0,0,144,27,225,0,128,184,96,5,0,0,0,0,0,0,0,52,73,3,0,0,0,0,0,9,0,0,0,8,0,0,0,10,0,0,0,7,0,0,0,11,0,0,0,6,0,0,0,12,0,0,0,5,0,0,0,13,0,0,0,4,0,0,0,14,0,0,0,3,0,0,0,15,0,0,0,2,0,0,0,16,0,0,0,1,0,0,0,17,0,0,0,0,0,0,0,18,0,0,0,1,0,0,0,49,0,0,0,63,249,7,0,1,0,0,0,0,0,152,224,0,0,16,96,6,0,0,0,0,0,0,0,168,73,3,0,0,0,0,0,24,0,0,0,23,0,0,0,25,0,0,0,22,0,0,0,26,0,0,0,21,0,0,0,27,0,0,0,20,0,0,0,28,0,0,0,19,0,0,0,29,0,0,0,18,0,0,0,30,0,0,0,17,0,0,0,31,0,0,0,16,0,0,0,32,0,0,0,15,0,0,0,33,0,0,0,14,0,0,0,34,0,0,0,13,0,0,0,35,0,0,0,12,0,0,0,36,0,0,0,11,0,0,0,37,0,0,0,10,0,0,0,38,0,0,0,9,0,0,0,39,0,0,0,8,0,0,0,40,0,0,0,7,0,0,0,41,0,0,0,6,0,0,0,42,0,0,0,5,0,0,0,43,0,0,0,4,0,0,0,44,0,0,0,3,0,0,0,45,0,0,0,2,0,0,0,46,0,0,0,1,0,0,0,47,0,0,0,0,0,0,0,48,0,0,0,2,0,0,0,100,0,0,0,112,249,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,248,7,0,0,56,7,0,0,2,0,0,0,152,58,0,0,56,74,0,0,120,24,3,0,132,24,3,0,144,24,3,0,156,239,0,0,204,239,0,0,224,24,3,0,224,24,3,0,224,24,3,0,240,25,3,0,32,26,3,0,80,29,3,0,80,29,3,0,128,32,3,0,204,28,1,0,176,35,3,0,88,7,0,0,88,7,0,0,240,36,3,0,240,36,3,0,252,36,3,0,252,36,3,0,120,7,0,0,12,37,3,0,28,37,3,0,24,8,0,0,204,33,1,0,184,7,0,0,44,37,3,0,168,54,1,0,128,71,1,0,2,0,0,0,236,40,3,0,12,41,3,0,3,0,0,0,56,8,0,0,56,7,0,0,255,255,255,255,152,58,0,0,56,74,0,0,120,24,3,0,132,24,3,0,144,24,3,0,156,239,0,0,204,239,0,0,224,24,3,0,224,24,3,0,224,24,3,0,240,25,3,0,32,26,3,0,80,29,3,0,80,29,3,0,128,32,3,0,204,28,1,0,176,35,3,0,88,7,0,0,88,7,0,0,240,36,3,0,240,36,3,0,252,36,3,0,252,36,3,0,120,7,0,0,12,37,3,0,28,37,3,0,24,8,0,0,204,33,1,0,184,7,0,0,44,37,3,0,168,54,1,0,128,71,1,0,2,0,0,0,236,40,3,0,132,58,3,0,2,0,0,0,88,8,0,0,112,8,0,0,2,0,0,0,40,35,0,0,152,58,0,0,104,76,3,0,104,76,3,0,112,76,3,0,156,239,0,0,204,239,0,0,172,76,3,0,0,0,0,0,172,76,3,0,120,77,3,0,144,77,3,0,144,77,3,0,0,0,0,0,0,0,0,0,204,28,1,0,176,35,3,0,136,8,0,0,0,0,0,0,244,79,3,0,244,79,3,0,252,79,3,0,252,79,3,0,160,8,0,0,4,80,3,0,16,80,3,0,184,8,0,0,204,33,1,0,208,8,0,0,28,80,3,0,168,54,1,0,128,71,1,0,1,0,0,0,236,82,3,0,248,82,3,0,0,2,0,0,0,2,0,0,30,0,0,0,25,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,30,0,0,0,25,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,20,0,0,0,0,0,0,0,242,255,255,255,0,0,0,0,0,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,246,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,0,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,246,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,236,255,255,255,240,255,255,255,246,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,10,0,0,0,10,0,0,0,255,255,255,255,10,0,0,0,10,0,0,0,255,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,4,0,0,0,10,0,0,0,10,0,0,0,12,0,0,0,12,0,0,0,12,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,10,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,99,0,0,0,99,0,0,0,99,0,0,0,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,4,0,0,0,10,0,0,0,10,0,0,0,12,0,0,0,12,0,0,0,12,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,99,0,0,0,99,0,0,0,99,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,244,255,255,255,244,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,99,0,0,0,99,0,0,0,99,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,99,0,0,0,99,0,0,0,99,0,0,0,64,0,0,0,64,0,0,0,8,0,0,0,8,0,0,0,156,255,255,255,156,255,255,255,151,255,255,255,126,255,255,255,126,255,255,255,116,255,255,255,4,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,6,0,0,0,5,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,4,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,6,0,0,0,5,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,128,63,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,0,0,198,66,240,82,3,0,6,0,0,0,6,0,0,0,136,120,1,0,8,83,3,0,136,120,1,0,184,87,3,0,2,0,0,0,0,0,0,0,32,0,0,0,0,186,1,0,40,83,3,0,40,83,3,0,80,83,3,0,80,83,3,0,2,0,0,0,100,0,0,0,212,249,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,84,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,68,84,3,0,0,0,0,0,0,0,0,0,0,0,0,0,128,84,3,0,0,0,0,0,0,0,0,0,0,0,0,0,204,84,3,0,0,0,0,0,0,0,0,0,0,0,0,0,24,85,3,0,0,0,0,0,132,85,3,0,184,85,3,0,0,0,0,0,0,0,0,0,12,86,3,0,104,86,3,0,0,0,0,0,0,0,0,0,164,86,3,0,216,86,3,0,60,87,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,56,250,7,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,56,84,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,217,19,8,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,108,84,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,74,22,8,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,168,84,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,155,22,8,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,244,84,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,236,22,8,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,64,85,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,13,24,8,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,172,85,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,94,24,8,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,224,85,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,215,24,8,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,52,86,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,128,25,8,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,144,86,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,81,0,0,0,153,25,8,0,1,0,0,0,0,176,19,225,0,176,19,97,2,0,0,0,0,0,0,0,204,86,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,225,0,0,0,234,25,8,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,0,87,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+208676);
/* memory initializer */ allocate([1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,203,26,8,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,100,87,3,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,0,0,0,0,32,0,0,0,0,186,1,0,216,87,3,0,216,87,3,0,0,88,3,0,0,88,3,0,2,0,0,0,100,0,0,0,132,28,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,192,88,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,244,88,3,0,0,0,0,0,0,0,0,0,0,0,0,0,48,89,3,0,0,0,0,0,0,0,0,0,0,0,0,0,124,89,3,0,0,0,0,0,0,0,0,0,0,0,0,0,200,89,3,0,0,0,0,0,52,90,3,0,104,90,3,0,0,0,0,0,0,0,0,0,188,90,3,0,24,91,3,0,0,0,0,0,0,0,0,0,84,91,3,0,176,91,3,0,20,92,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,0,0,0,161,25,0,0,232,28,8,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,232,88,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,137,54,8,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,28,89,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,250,56,8,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,88,89,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,75,57,8,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,164,89,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,33,1,0,0,156,57,8,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,240,89,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,4,0,0,0,81,0,0,0,189,58,8,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,92,90,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,14,59,8,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,144,90,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,169,0,0,0,135,59,8,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,228,90,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,48,60,8,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,64,91,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,169,0,0,0,73,60,8,0,1,0,0,0,0,136,93,225,0,176,19,97,4,0,0,0,0,0,0,0,124,91,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,225,0,0,0,242,60,8,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,216,91,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,211,61,8,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,60,92,3,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,2,0,0,0,232,8,0,0,112,8,0,0,255,255,255,255,40,35,0,0,152,58,0,0,104,76,3,0,104,76,3,0,112,76,3,0,156,239,0,0,204,239,0,0,172,76,3,0,0,0,0,0,172,76,3,0,120,77,3,0,144,77,3,0,144,77,3,0,0,0,0,0,0,0,0,0,204,28,1,0,176,35,3,0,136,8,0,0,0,0,0,0,244,79,3,0,244,79,3,0,252,79,3,0,252,79,3,0,160,8,0,0,4,80,3,0,16,80,3,0,184,8,0,0,204,33,1,0,208,8,0,0,28,80,3,0,168,54,1,0,128,71,1,0,1,0,0,0,236,82,3,0,44,93,3,0,0,166,2,0,60,93,3,0,0,166,2,0,28,97,3,0,1,0,0,0,0,0,0,0,32,0,0,0,80,191,2,0,92,93,3,0,92,93,3,0,132,93,3,0,132,93,3,0,2,0,0,0,64,0,0,0,140,63,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,68,94,3,0,0,0,0,0,0,0,0,0,0,0,0,0,120,94,3,0,0,0,0,0,0,0,0,0,0,0,0,0,172,94,3,0,0,0,0,0,0,0,0,0,0,0,0,0,232,94,3,0,0,0,0,0,0,0,0,0,0,0,0,0,36,95,3,0,0,0,0,0,112,95,3,0,204,95,3,0,0,0,0,0,0,0,0,0,8,96,3,0,60,96,3,0,160,96,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,204,63,8,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,108,94,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,29,64,8,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,160,94,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,110,64,8,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,212,94,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,223,66,8,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,16,95,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,80,69,8,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,76,95,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,169,0,0,0,161,69,8,0,1,0,0,0,0,0,158,224,0,0,84,96,4,0,0,0,0,0,0,0,152,95,3,0,0,0,0,0,6,0,0,0,5,0,0,0,7,0,0,0,4,0,0,0,8,0,0,0,3,0,0,0,9,0,0,0,2,0,0,0,10,0,0,0,1,0,0,0,11,0,0,0,0,0,0,0,12,0,0,0,2,0,0,0,25,0,0,0,74,70,8,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,244,95,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,81,0,0,0,99,70,8,0,1,0,0,0,0,176,19,225,0,176,19,97,2,0,0,0,0,0,0,0,48,96,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,225,0,0,0,180,70,8,0,1,0,0,0,0,96,242,224,0,0,149,96,4,0,0,0,0,0,0,0,100,96,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,185,1,0,0,149,71,8,0,1,0,0,0,0,0,116,224,0,0,16,96,5,0,0,0,0,0,0,0,200,96,3,0,0,0,0,0,10,0,0,0,9,0,0,0,11,0,0,0,8,0,0,0,12,0,0,0,7,0,0,0,13,0,0,0,6,0,0,0,14,0,0,0,5,0,0,0,15,0,0,0,4,0,0,0,16,0,0,0,3,0,0,0,17,0,0,0,2,0,0,0,18,0,0,0,1,0,0,0,19,0,0,0,0,0,0,0,20,0,0,0,1,0,0,0,0,0,0,0,32,0,0,0,16,227,2,0,60,97,3,0,60,97,3,0,100,97,3,0,100,97,3,0,2,0,0,0,100,0,0,0,78,73,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,36,98,3,0,0,0,0,0,0,0,0,0,0,0,0,0,88,98,3,0,0,0,0,0,0,0,0,0,0,0,0,0,140,98,3,0,0,0,0,0,0,0,0,0,0,0,0,0,200,98,3,0,0,0,0,0,0,0,0,0,0,0,0,0,4,99,3,0,0,0,0,0,0,0,0,0,0,0,0,0,80,99,3,0,0,0,0,0,156,99,3,0,208,99,3,0,0,0,0,0,0,0,0,0,36,100,3,0,120,100,3,0,0,0,0,0,0,0,0,0,204,100,3,0,48,101,3,0,148,101,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,81,0,0,0,178,73,8,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,76,98,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,81,0,0,0,3,74,8,0,1,0,0,0,0,0,16,224,0,0,16,96,2,0,0,0,0,0,0,0,128,98,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,113,2,0,0,84,74,8,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,180,98,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,113,2,0,0,197,76,8,0,1,0,0,0,0,0,48,224,0,0,16,96,3,0,0,0,0,0,0,0,240,98,3,0,0,0,0,0,2,0,0,0,1,0,0,0,3,0,0,0,0,0,0,0,4,0,0,0,2,0,0,0,81,0,0,0,54,79,8,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,44,99,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,2,0,0,0,81,0,0,0,135,79,8,0,1,0,0,0,0,0,80,224,0,0,16,96,4,0,0,0,0,0,0,0,120,99,3,0,0,0,0,0,4,0,0,0,3,0,0,0,5,0,0,0,2,0,0,0,6,0,0,0,1,0,0,0,7,0,0,0,0,0,0,0,8,0,0,0,4,0,0,0,81,0,0,0,216,79,8,0,1,0,0,0,0,0,118,224,0,0,118,96,2,0,0,0,0,0,0,0,196,99,3,0,0,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,2,0,0,0,121,0,0,0,41,80,8,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,248,99,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,162,80,8,0,1,0,0,0,0,128,187,224,0,0,118,96,4,0,0,0,0,0,0,0,76,100,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,121,0,0,0,27,81,8,0,1,0,0,0,0,0,84,224,0,0,16,96,4,0,0,0,0,0,0,0,160,100,3,0,0,0,0,0,5,0,0,0,4,0,0,0,6,0,0,0,3,0,0,0,7,0,0,0,2,0,0,0,8,0,0,0,1,0,0,0,9,0,0,0,0,0,0,0,10,0,0,0,2,0,0,0,225,0,0,0,148,81,8,0,1,0,0,0,0,228,91,225,0,224,255,96,4,0,0,0,0,0,0,0,244,100,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,225,0,0,0,117,82,8,0,1,0,0,0,0,192,221,224,0,0,145,96,4,0,0,0,0,0,0,0,88,101,3,0,0,0,0,0,7,0,0,0,6,0,0,0,8,0,0,0,5,0,0,0,9,0,0,0,4,0,0,0,10,0,0,0,3,0,0,0,11,0,0,0,2,0,0,0,12,0,0,0,1,0,0,0,13,0,0,0,0,0,0,0,14,0,0,0,2,0,0,0,33,1,0,0,86,83,8,0,1,0,0,0,0,0,112,224,0,0,16,96,5,0,0,0,0,0,0,0,188,101,3,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,16,0,0,0,2,0,0,0,0,9,0,0,24,9,0,0,2,0,0,0,64,31,0,0,40,35,0,0,156,102,3,0,156,102,3,0,164,102,3,0,156,239,0,0,204,239,0,0,224,102,3,0,0,0,0,0,224,102,3,0,120,77,3,0,172,103,3,0,172,103,3,0,0,0,0,0,0,0,0,0,204,28,1,0,176,35,3,0,136,8,0,0,0,0,0,0,244,79,3,0,244,79,3,0,252,79,3,0,252,79,3,0,48,9,0,0,4,80,3,0,16,80,3,0,64,9,0,0,204,33,1,0,208,8,0,0,28,80,3,0,168,54,1,0,128,71,1,0,1,0,0,0,16,106,3,0,248,82,3,0,0,2,0,0,0,2,0,0,32,0,0,0,25,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,30,0,0,0,25,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,20,0,0,0,0,0,0,0,242,255,255,255,0,0,0,0,0,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,250,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,250,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,246,255,255,255,246,255,255,255,250,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,4,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,10,0,0,0,10,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,99,0,0,0,99,0,0,0,99,0,0,0,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,4,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,10,0,0,0,10,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,251,255,255,255,251,255,255,255,251,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,232,255,255,255,236,255,255,255,242,255,255,255,246,255,255,255,250,255,255,255,248,255,255,255,248,255,255,255,250,255,255,255,250,255,255,255,99,0,0,0,99,0,0,0,99,0,0,0,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,241,255,255,255,244,255,255,255,246,255,255,255,248,255,255,255,0,0,0,0,2,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,99,0,0,0,99,0,0,0,99,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,234,255,255,255,236,255,255,255,242,255,255,255,244,255,255,255,244,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,246,255,255,255,99,0,0,0,99,0,0,0,99,0,0,0,226,255,255,255,226,255,255,255,226,255,255,255,226,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,230,255,255,255,232,255,255,255,99,0,0,0,99,0,0,0,99,0,0,0,20,106,3,0,6,0,0,0,6,0,0,0,2,0,0,0,88,9,0,0,24,9,0,0,255,255,255,255,64,31,0,0,40,35,0,0,156,102,3,0,156,102,3,0,164,102,3,0,156,239,0,0,204,239,0,0,224,102,3,0,0,0,0,0,224,102,3,0,120,77,3,0,172,103,3,0,172,103,3,0,0,0,0,0,0,0,0,0,204,28,1,0,176,35,3,0,136,8,0,0,0,0,0,0,244,79,3,0,244,79,3,0,252,79,3,0,252,79,3,0,48,9,0,0,4,80,3,0,16,80,3,0,64,9,0,0,204,33,1,0,208,8,0,0,28,80,3,0,168,54,1,0,128,71,1,0,1,0,0,0,16,106,3,0,44,93,3,0,11,0,0,0,112,9,0,0,0,3,0,0,2,0,0,0,80,195,0,0,64,13,3,0,84,238,0,0,128,238,0,0,172,238,0,0,156,239,0,0,204,239,0,0,252,239,0,0,44,243,0,0,252,239,0,0,92,246,0,0,140,246,0,0,28,0,1,0,172,9,1,0,60,19,1,0,204,28,1,0,252,28,1,0,96,3,0,0,192,3,0,0,188,32,1,0,232,32,1,0,20,33,1,0,64,33,1,0,32,4,0,0,108,33,1,0,156,33,1,0,120,4,0,0,204,33,1,0,216,4,0,0,104,43,1,0,168,54,1,0,128,71,1,0,2,0,0,0,160,119,1,0,48,120,1,0,11,0,0,0,112,9,0,0,0,3,0,0,255,255,255,255,80,195,0,0,64,13,3,0,84,238,0,0,128,238,0,0,172,238,0,0,156,239,0,0,204,239,0,0,252,239,0,0,44,243,0,0,252,239,0,0,92,246,0,0,140,246,0,0,28,0,1,0,172,9,1,0,60,19,1,0,204,28,1,0,252,28,1,0,96,3,0,0,192,3,0,0,188,32,1,0,232,32,1,0,20,33,1,0,64,33,1,0,32,4,0,0,108,33,1,0,156,33,1,0,120,4,0,0,204,33,1,0,216,4,0,0,0,0,0,0,168,54,1,0,128,71,1,0,2,0,0,0,160,119,1,0,168,165,2,0,2,0,0,0,112,9,0,0,24,9,0,0,2,0,0,0,0,0,0,0,64,31,0,0,156,102,3,0,156,102,3,0,164,102,3,0,156,239,0,0,204,239,0,0,224,102,3,0,0,0,0,0,224,102,3,0,120,77,3,0,172,103,3,0,172,103,3,0,0,0,0,0,0,0,0,0,204,28,1,0,176,35,3,0,136,8,0,0,0,0,0,0,244,79,3,0,244,79,3,0,252,79,3,0,252,79,3,0,48,9,0,0,4,80,3,0,16,80,3,0,64,9,0,0,204,33,1,0,208,8,0,0,28,80,3,0,168,54,1,0,128,71,1,0,1,0,0,0,16,106,3,0,248,82,3,0,2,0,0,0,112,9,0,0,24,9,0,0,255,255,255,255,0,0,0,0,64,31,0,0,156,102,3,0,156,102,3,0,164,102,3,0,156,239,0,0,204,239,0,0,224,102,3,0,0,0,0,0,224,102,3,0,120,77,3,0,172,103,3,0,172,103,3,0,0,0,0,0,0,0,0,0,204,28,1,0,176,35,3,0,136,8,0,0,0,0,0,0,244,79,3,0,244,79,3,0,252,79,3,0,252,79,3,0,48,9,0,0,4,80,3,0,16,80,3,0,64,9,0,0,204,33,1,0,208,8,0,0,28,80,3,0,168,54,1,0,128,71,1,0,1,0,0,0,16,106,3,0,44,93,3,0,88,105,112,104,46,79,114,103,32,108,105,98,86,111,114,98,105,115,32,73,32,50,48,49,53,48,49,48,53,32,40,226,155,132,226,155,132,226,155,132,226,155,132,41,0,7,7,7,11,6,6,7,11,7,6,6,10,12,10,10,13,7,7,8,11,7,7,7,11,7,6,7,10,11,10,10,13,10,10,9,12,9,9,9,11,8,8,8,11,13,11,10,14,15,15,14,15,15,14,13,14,15,12,12,17,17,17,17,17,7,7,6,9,6,6,6,9,7,6,6,8,11,11,10,12,7,7,7,9,7,6,6,9,7,6,6,9,13,10,10,11,10,9,8,10,9,8,8,10,8,8,7,9,13,12,10,11,17,14,14,13,15,14,12,13,17,13,12,15,17,17,14,17,7,6,6,7,6,6,5,7,6,6,6,6,11,9,9,9,7,7,6,7,7,6,6,7,6,6,6,6,10,9,8,9,10,9,8,8,9,8,7,8,8,7,6,8,11,10,9,10,17,17,12,15,15,15,12,14,14,14,10,12,15,13,12,13,11,10,8,10,11,10,8,8,10,9,7,7,10,9,9,11,11,11,9,10,11,10,8,9,10,8,6,8,10,9,9,11,14,13,10,12,12,11,10,10,8,7,8,10,10,11,11,12,17,17,15,17,17,17,17,17,17,13,12,17,17,17,14,17,2,2,2,2,0,0,0,0,3,2,3,2,3,3,0,0,0,0,0,0,0,0,0,0,3,3,4,3,4,3,4,4,5,4,5,4,6,5,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,4,3,5,3,5,3,5,4,6,5,6,5,7,6,6,7,7,9,9,11,11,16,11,14,10,11,11,13,16,15,15,15,15,15,15,15,15,15,6,7,7,12,6,6,7,12,7,6,6,10,15,12,11,13,7,7,8,13,7,7,8,12,7,7,7,11,12,12,11,13,10,9,9,11,9,9,9,10,10,8,8,12,14,12,12,14,11,11,12,14,11,12,11,15,15,12,13,15,15,15,15,15,6,6,7,10,6,6,6,11,7,6,6,9,14,12,11,13,7,7,7,10,6,6,7,9,7,7,6,10,13,12,10,12,9,9,9,11,9,9,8,9,9,8,8,10,13,12,10,12,12,12,11,13,12,12,11,12,15,13,12,15,15,15,14,14,6,6,6,8,6,6,5,6,7,7,6,5,11,10,9,8,7,6,6,7,6,6,5,6,7,7,6,6,11,10,9,8,8,8,8,9,8,8,7,8,8,8,6,7,11,10,9,9,14,11,10,14,14,11,10,15,13,11,9,11,15,12,12,11,11,9,8,8,10,9,8,9,11,10,9,8,12,11,12,11,13,10,8,9,11,10,8,9,10,9,8,9,10,8,12,12,15,11,10,10,13,11,10,10,8,8,7,12,10,9,11,12,15,12,11,15,13,11,11,15,12,14,11,13,15,15,13,13,2,2,2,2,0,0,0,0,2,2,3,3,3,3,0,0,0,0,0,0,0,0,0,0,4,3,4,3,4,3,5,3,5,4,5,4,6,4,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,4,3,5,3,5,3,6,4,7,4,7,5,7,6,7,6,7,8,10,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,12,12,10,7,8,13,9,6,7,11,10,8,8,12,17,17,17,17,7,5,5,9,6,4,4,8,8,5,5,8,16,14,13,16,7,5,5,7,6,3,3,5,8,5,4,7,14,12,12,15,10,7,8,9,7,5,5,6,9,6,5,5,15,12,9,10,8,13,17,17,8,11,17,17,11,13,17,17,17,17,17,17,6,10,16,17,6,10,15,17,8,10,16,17,17,17,17,17,9,13,15,17,8,11,17,17,10,12,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,6,11,15,17,7,10,15,17,8,10,17,17,17,15,17,17,4,8,13,17,4,7,13,17,6,8,15,17,16,15,17,17,6,11,15,17,6,9,13,17,8,10,17,17,15,17,17,17,16,17,17,17,12,14,15,17,13,14,15,17,17,17,17,17,5,10,14,17,5,9,14,17,7,9,15,17,15,15,17,17,3,7,12,17,3,6,11,17,5,7,13,17,12,12,17,17,5,9,14,17,3,7,11,17,5,8,13,17,13,11,16,17,12,17,17,17,9,14,15,17,10,11,14,17,16,14,17,17,8,12,17,17,8,12,17,17,10,12,17,17,17,17,17,17,5,10,17,17,5,9,15,17,7,9,17,17,13,13,17,17,7,11,17,17,6,10,15,17,7,9,15,17,12,11,17,17,12,15,17,17,11,14,17,17,11,10,15,17,17,16,17,17,0,3,3,3,3,3,3,3,3,0,0,0,0,0,0,0,0,0,3,3,3,4,4,4,4,5,4,5,4,5,4,6,4,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,5,3,5,3,5,4,5,4,5,5,5,5,6,5,6,5,6,5,6,5,6,5,7,8,9,11,13,13,13,13,13,13,13,13,13,13,13,13,0,3,3,2,3,3,4,3,4,0,0,0,0,0,0,0,0,0,3,4,3,6,3,6,3,6,3,7,3,8,4,9,4,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,7,2,7,3,8,4,9,5,9,8,10,11,11,12,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,13,13,13,13,7,5,5,9,9,6,6,9,12,8,7,8,11,8,9,15,6,3,3,7,7,4,3,6,9,6,5,6,8,6,8,15,8,5,5,9,8,5,4,6,10,7,5,5,11,8,7,15,14,15,13,13,13,13,8,11,15,10,7,6,11,9,10,15,5,6,8,15,6,9,10,15,10,11,12,15,15,15,15,15,4,6,7,15,6,7,8,15,9,8,9,15,15,15,15,15,6,8,9,15,7,7,8,15,10,9,10,15,15,15,15,15,15,13,15,15,15,10,11,15,15,13,13,15,15,15,15,15,4,6,7,15,6,8,9,15,10,10,12,15,15,15,15,15,2,5,6,15,5,6,7,15,8,6,7,15,15,15,15,15,5,6,8,15,5,6,7,15,9,6,7,15,15,15,15,15,14,12,13,15,12,10,11,15,15,15,15,15,15,15,15,15,7,8,9,15,9,10,10,15,15,14,14,15,15,15,15,15,5,6,7,15,7,8,9,15,12,9,10,15,15,15,15,15,7,7,9,15,7,7,8,15,12,8,9,15,15,15,15,15,13,13,14,15,12,11,12,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,13,13,13,15,15,15,15,15,15,15,15,15,15,15,15,15,15,12,13,15,15,12,13,15,15,14,15,15,15,15,15,15,15,15,15,15,15,15,13,15,15,15,15,15,15,15,15,15,0,2,3,3,3,3,4,3,4,0,0,0,0,0,0,0,0,0,3,4,3,4,3,5,3,6,3,6,4,6,4,7,5,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,5,2,5,3,5,3,6,3,6,4,7,6,7,8,7,9,8,9,9,9,10,9,11,13,11,13,10,10,13,13,13,13,13,13,12,12,12,12,0,3,3,3,3,2,4,3,4,0,0,0,0,0,0,0,0,0,2,3,3,4,3,4,4,5,4,6,5,6,7,6,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,2,4,3,6,3,7,3,8,5,8,6,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,7,1,6,3,7,2,4,5,7,1,6,12,16,4,12,15,16,9,15,16,16,16,16,16,16,2,5,11,16,5,11,13,16,9,13,16,16,16,16,16,16,4,8,12,16,5,9,12,16,9,13,15,16,16,16,16,16,15,16,16,16,11,14,13,16,12,15,16,16,16,16,16,15,7,6,9,17,7,6,8,17,12,9,11,16,16,16,16,16,5,4,7,16,5,3,6,14,9,6,8,15,16,16,16,16,5,4,6,13,3,2,4,11,7,4,6,13,16,11,10,14,12,12,12,16,9,7,10,15,12,9,11,16,16,15,15,16,5,5,5,5,5,5,6,5,6,5,6,5,6,5,6,5,6,5,6,5,6,5,6,5,6,5,6,6,6,6,7,6,7,6,7,6,7,6,7,6,7,6,8,6,8,6,8,7,8,7,8,7,8,7,9,7,9,8,9,8,9,8,10,8,10,9,10,9,10,9,11,9,11,9,10,10,11,10,11,10,11,11,11,11,11,11,12,13,14,14,14,15,15,16,16,16,17,15,16,15,16,16,17,17,16,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,2,5,5,5,5,5,5,4,5,5,5,5,5,5,5,5,6,5,6,5,6,5,7,6,7,6,7,6,8,6,8,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,3,5,3,6,4,6,4,7,4,7,4,7,4,8,4,8,4,9,5,9,5,9,5,9,6,10,6,10,6,11,7,10,7,10,8,11,9,11,9,11,10,11,11,12,11,11,12,15,15,12,14,11,14,12,14,11,14,13,14,12,14,11,14,11,14,12,14,11,14,11,14,13,13,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,0,4,5,4,5,4,5,3,5,3,5,3,5,4,4,4,5,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,3,3,4,4,4,4,5,4,5,4,6,5,7,5,7,6,8,6,8,6,9,7,9,7,10,7,9,8,11,8,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,8,3,8,4,8,4,8,6,8,5,8,4,8,4,8,6,8,7,8,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,0,4,4,4,4,4,4,4,4,4,4,4,4,4,5,4,5,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,3,5,4,6,4,6,4,7,4,7,4,8,4,8,4,9,4,9,4,10,4,10,5,10,5,11,5,12,6,12,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,1,6,3,7,3,8,4,8,5,8,8,8,9,7,8,8,7,7,7,8,9,10,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,9,9,1,3,4,7,2,5,6,7,1,4,10,19,3,8,13,19,7,12,19,19,19,19,19,19,2,6,11,19,8,13,19,19,9,11,19,19,19,19,19,19,6,7,13,19,9,13,19,19,10,13,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,3,6,10,17,4,8,11,20,8,10,11,20,20,20,20,20,2,4,8,18,4,6,8,17,7,8,10,20,20,17,20,20,3,5,8,17,3,4,6,17,8,8,10,17,17,12,16,20,13,13,15,20,10,10,12,20,15,14,15,20,20,20,19,19,5,5,6,5,6,5,6,5,6,5,6,5,6,5,6,5,7,5,7,5,7,5,7,5,7,5,7,5,8,5,8,5,8,5,8,5,8,6,8,6,8,6,9,6,9,6,9,6,9,6,9,7,9,7,9,7,9,7,10,7,10,8,10,8,10,8,10,8,10,8,11,8,11,8,11,8,11,8,11,9,12,9,12,9,12,9,12,9,12,10,12,10,13,11,13,11,14,12,14,13,15,14,16,14,17,15,18,16,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,2,5,5,4,5,4,5,4,5,5,5,5,5,5,6,5,6,5,6,5,7,6,7,6,7,6,8,6,9,7,9,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,3,5,3,5,3,6,3,6,4,6,4,7,4,7,5,8,5,8,6,9,7,9,7,9,8,10,9,10,9,11,10,11,11,11,11,11,11,12,12,12,13,12,13,12,14,12,15,12,14,12,16,13,17,13,17,14,17,14,16,13,17,14,17,14,17,15,17,15,15,16,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,16,16,16,16,16,16,16,16,16,16,0,4,5,4,6,4,8,3,9,3,9,2,9,3,8,4,9,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,1,5,3,5,3,5,4,7,5,10,7,10,7,12,10,14,10,14,9,14,11,14,14,14,13,13,13,13,13,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,0,4,4,4,4,4,4,4,5,3,5,3,5,4,6,4,6,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,3,6,3,6,4,7,4,7,4,7,4,8,4,8,4,8,4,8,4,9,4,9,5,10,5,10,7,10,8,10,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,2,4,3,4,4,4,5,4,7,5,8,5,11,6,10,6,12,7,12,7,12,8,12,8,12,10,12,12,12,12,12,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,4,5,6,11,5,5,6,10,7,7,6,6,14,13,9,9,6,6,6,10,6,6,6,9,8,7,7,9,14,12,8,11,8,7,7,11,8,8,7,11,9,9,7,9,13,11,9,13,19,19,18,19,15,16,16,19,11,11,10,13,10,10,9,15,5,5,6,13,6,6,6,11,8,7,6,7,14,11,10,11,6,6,6,12,7,6,6,11,8,7,7,11,13,11,9,11,9,7,6,12,8,7,6,12,9,8,8,11,13,10,7,13,19,19,17,19,17,14,14,19,12,10,8,12,13,10,9,16,7,8,7,12,7,7,7,11,8,7,7,8,12,12,11,11,8,8,7,12,8,7,6,11,8,7,7,10,10,11,10,11,9,8,8,13,9,8,7,12,10,9,7,11,9,8,7,11,18,18,15,18,18,16,17,18,15,11,10,18,11,9,9,18,16,16,13,16,12,11,10,16,12,11,9,6,15,12,11,13,16,16,14,14,13,11,12,16,12,9,9,13,13,10,10,12,17,18,17,17,14,15,14,16,14,12,14,15,12,10,11,12,18,18,18,18,18,18,18,18,18,12,13,18,16,11,9,18,1,3,2,3,0,0,0,0,2,3,2,3,3,3,0,0,0,0,0,0,0,0,0,0,3,3,3,4,3,4,4,4,4,4,5,5,5,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,4,2,4,3,5,4,5,5,5,5,6,6,6,6,6,6,6,7,7,8,6,9,7,12,11,16,13,16,12,15,13,15,12,14,12,15,15,15,2,10,8,14,7,12,11,14,1,5,3,7,4,9,7,13,1,4,2,6,3,7,5,7,1,5,7,21,5,8,9,21,10,9,12,20,20,16,20,20,4,8,9,20,6,8,9,20,11,11,13,20,20,15,17,20,9,11,14,20,8,10,15,20,11,13,15,20,20,20,20,20,20,20,20,20,13,20,20,20,18,18,20,20,20,20,20,20,3,6,8,20,6,7,9,20,10,9,12,20,20,20,20,20,5,7,9,20,6,6,9,20,10,9,12,20,20,20,20,20,8,10,13,20,8,9,12,20,11,10,12,20,20,20,20,20,18,20,20,20,15,17,18,20,18,17,18,20,20,20,20,20,7,10,12,20,8,9,11,20,14,13,14,20,20,20,20,20,6,9,12,20,7,8,11,20,12,11,13,20,20,20,20,20,9,11,15,20,8,10,14,20,12,11,14,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,11,16,18,20,15,15,17,20,20,17,20,20,20,20,20,20,9,14,16,20,12,12,15,20,17,15,18,20,20,20,20,20,16,19,18,20,15,16,20,20,17,17,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,20,2,3,7,13,4,4,7,15,8,6,9,17,21,16,15,21,2,5,7,11,5,5,7,14,9,7,10,16,17,15,16,21,4,7,10,17,7,7,9,15,11,9,11,16,21,18,15,21,18,21,21,21,15,17,17,19,21,19,18,20,21,21,21,20,5,5,5,5,6,5,6,5,6,5,6,5,6,5,6,5,6,5,6,5,6,5,6,5,7,5,7,5,7,5,7,5,8,6,8,6,8,6,9,6,9,6,10,6,10,6,11,6,11,7,11,7,12,7,12,7,12,7,12,7,12,7,12,7,12,7,12,8,13,8,12,8,12,8,13,8,13,9,13,9,13,9,13,9,12,10,12,10,13,10,14,11,14,12,14,13,14,13,14,14,15,16,15,15,15,14,15,17,21,22,22,21,22,22,22,22,22,22,21,21,21,21,21,21,21,21,21,21,2,5,5,4,5,4,5,4,5,4,6,5,6,5,6,5,6,5,7,5,7,6,8,6,8,6,8,6,9,6,9,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,5,8,4,9,4,9,4,9,4,9,4,9,4,9,4,9,4,9,4,9,4,8,4,8,4,9,5,9,5,9,5,9,5,9,6,10,6,10,7,10,8,11,9,11,11,12,13,12,14,13,15,13,15,14,16,14,17,15,17,15,15,16,16,15,16,16,16,15,18,16,15,17,17,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,19,1,5,5,5,5,5,5,5,6,5,6,5,6,5,6,5,6,6,7,7,7,7,8,7,8,8,9,8,10,9,10,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,3,4,3,4,4,5,4,5,4,5,5,6,5,6,5,7,5,7,6,7,6,8,7,8,7,8,7,9,8,9,9,9,9,10,10,10,11,9,12,9,12,9,15,10,14,9,13,10,13,10,12,10,12,10,13,10,12,11,13,11,14,12,13,13,14,14,13,14,15,14,16,13,13,14,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,15,15,0,4,5,4,5,3,5,3,5,3,5,4,4,4,4,5,5,5], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+218916);
/* memory initializer */ allocate([3,3,4,3,4,4,4,4,5,5,5,5,5,6,5,7,5,8,6,8,6,9,7,10,7,10,8,10,8,11,9,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,7,3,8,3,10,3,8,3,9,3,8,4,9,4,9,5,9,6,10,6,9,7,11,7,12,9,13,10,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,0,4,5,4,5,4,5,4,5,3,5,3,5,3,5,4,5,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,2,4,2,5,3,5,4,6,6,6,7,7,8,7,8,7,8,7,9,8,9,8,9,8,10,8,11,9,12,9,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,5,2,6,3,6,4,7,4,7,5,9,5,11,6,11,6,11,7,11,6,11,6,11,9,11,8,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,2,6,8,9,7,11,13,13,1,3,5,5,6,6,12,10,1,2,3,6,4,7,5,7,3,3,6,16,5,5,7,16,9,8,11,16,16,16,16,16,5,5,8,16,5,5,7,16,8,7,9,16,16,16,16,16,9,9,12,16,6,8,11,16,9,10,11,16,16,16,16,16,16,16,16,16,13,16,16,16,15,16,16,16,16,16,16,16,5,4,7,16,6,5,8,16,9,8,10,16,16,16,16,16,5,5,7,15,5,4,6,15,7,6,8,16,16,16,16,16,9,9,11,15,7,7,9,16,8,8,9,16,16,16,16,16,16,16,16,16,15,15,15,16,15,15,14,16,16,16,16,16,8,8,11,16,8,9,10,16,11,10,14,16,16,16,16,16,6,8,10,16,6,7,10,16,8,8,11,16,14,16,16,16,10,11,14,16,9,9,11,16,10,10,11,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,15,16,16,16,16,16,16,16,16,16,16,16,12,16,15,16,12,14,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,16,2,4,7,13,4,5,7,15,8,7,10,16,16,14,16,16,2,4,7,16,3,4,7,14,8,8,10,16,16,16,15,16,6,8,11,16,7,7,9,16,11,9,13,16,16,16,15,16,16,16,16,16,14,16,16,16,16,16,16,16,16,16,16,16,5,5,5,5,5,5,6,5,6,5,6,5,6,5,6,5,6,5,7,5,7,5,7,5,8,5,8,5,8,5,9,5,9,6,10,6,10,6,11,6,11,6,11,6,11,6,11,6,11,6,11,6,12,7,11,7,11,7,11,7,11,7,10,7,11,7,11,7,12,7,11,8,11,8,11,8,11,8,13,8,12,9,11,9,11,9,11,10,12,10,12,9,12,10,12,11,14,12,16,12,12,11,14,16,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17,16,16,16,16,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,6,6,6,6,6,6,7,6,7,6,7,6,7,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,5,7,5,7,4,7,4,8,4,8,4,8,4,8,3,8,4,9,4,9,4,9,4,9,4,9,5,9,5,9,6,9,7,9,8,9,9,9,10,9,11,9,14,9,15,10,15,10,15,10,15,10,15,11,15,10,14,12,14,11,14,13,14,13,15,15,15,12,15,15,15,13,15,13,15,13,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,14,2,4,5,4,5,4,5,4,5,5,5,5,5,5,6,5,6,5,6,6,6,6,7,7,7,7,7,7,8,8,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,4,3,4,3,4,4,5,4,5,5,5,6,6,6,7,6,8,6,8,6,9,7,10,7,10,7,10,7,12,7,12,7,12,9,12,11,12,10,12,10,12,11,12,12,12,10,12,10,12,10,12,9,12,11,12,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,10,10,12,12,12,12,12,10,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,0,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,5,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,3,3,4,4,4,4,4,4,5,5,5,5,6,6,7,6,7,6,8,6,9,7,9,7,9,9,11,9,12,10,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,6,3,7,3,7,5,7,7,7,7,7,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,0,3,4,4,4,4,4,4,4,4,5,4,5,4,5,4,4,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,2,4,3,4,4,4,5,5,6,5,6,5,7,6,6,6,7,7,7,8,9,9,9,12,10,11,10,10,12,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,6,5,7,5,7,7,7,7,7,5,7,5,7,5,7,5,7,7,7,7,7,4,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,6,6,6,6,6,6,6,6,6,1,2,3,6,5,4,7,7,3,3,3,14,5,4,4,11,8,6,6,10,17,12,11,17,6,5,5,15,5,3,4,11,8,5,5,8,16,9,10,14,10,8,9,17,8,6,6,13,10,7,7,10,16,11,13,14,17,17,17,17,17,16,16,16,16,15,16,16,16,16,16,16,2,4,6,17,4,5,7,17,8,7,10,17,17,17,17,17,3,4,6,15,3,3,6,15,7,6,9,17,17,17,17,17,6,8,10,17,6,6,8,16,9,8,10,17,17,15,16,17,17,17,17,17,12,15,15,16,12,15,15,16,16,16,16,16,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,6,5,6,6,6,6,5,6,6,7,6,7,6,7,6,7,6,8,7,8,7,8,7,8,7,8,7,9,7,9,7,9,7,9,8,9,8,10,8,10,8,10,7,10,6,10,8,10,8,11,7,10,7,11,8,11,11,12,12,11,11,12,11,13,11,13,11,13,12,15,12,13,13,14,14,14,14,14,15,15,15,16,14,17,19,19,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,18,2,4,5,4,5,4,5,4,5,5,5,5,5,5,6,5,6,5,6,6,7,6,7,6,8,7,8,7,8,7,8,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,3,5,3,5,4,5,4,5,4,5,5,5,5,6,5,6,5,7,5,8,6,8,6,8,6,8,6,8,7,9,7,9,7,11,9,11,11,12,11,14,12,14,16,14,16,13,16,14,16,12,15,13,16,14,16,13,14,12,15,13,15,13,13,13,15,12,14,14,15,13,15,12,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,0,4,5,4,4,4,5,4,4,4,5,4,5,4,5,3,5,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,3,4,3,4,4,5,4,5,4,6,4,6,5,6,5,7,5,7,6,8,6,8,6,8,7,8,7,9,7,9,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,3,3,3,4,3,4,4,5,5,6,6,7,7,7,8,8,11,8,9,9,9,10,11,11,11,9,10,10,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,0,4,4,4,4,4,4,3,4,4,4,4,4,5,4,5,5,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,3,4,3,5,4,6,4,6,5,7,6,7,6,8,6,8,7,9,8,10,8,12,9,13,10,15,10,15,11,14,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,8,4,8,4,8,4,8,5,8,5,8,6,8,4,8,4,8,5,8,5,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,10,9,12,15,12,13,16,14,16,7,1,5,14,7,10,13,16,16,9,4,6,16,8,11,16,16,16,14,4,7,16,9,12,14,16,16,10,5,7,14,9,12,14,15,15,13,8,9,14,10,12,13,14,15,13,9,9,7,6,8,11,12,12,14,8,8,5,4,5,8,11,12,16,10,10,6,5,6,8,9,10,5,6,12,14,12,14,16,17,18,4,2,5,11,7,10,12,14,15,9,4,5,11,7,10,13,15,18,15,6,7,5,6,8,11,13,16,11,5,6,5,5,6,9,13,15,12,5,7,6,5,6,9,12,14,12,6,7,8,6,7,9,12,13,14,8,8,7,5,5,8,10,12,16,9,9,8,6,6,7,9,9,1,4,4,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,7,9,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,0,0,0,0,0,0,7,10,9,0,0,0,0,0,0,8,10,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,0,0,0,0,0,0,8,10,10,0,0,0,0,0,0,8,9,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,10,10,0,0,0,0,0,0,9,9,11,0,0,0,0,0,0,10,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,10,10,0,0,0,0,0,0,9,11,9,0,0,0,0,0,0,10,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,0,0,0,0,0,0,8,10,10,0,0,0,0,0,0,8,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,10,10,0,0,0,0,0,0,10,11,11,0,0,0,0,0,0,9,9,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,10,10,0,0,0,0,0,0,10,11,11,0,0,0,0,0,0,9,11,9], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+229158);
/* memory initializer */ allocate([1,4,4,7,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,5,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,5,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,7,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,7,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,2,3,7,7,0,0,0,0,0,0,0,6,6,0,0,0,0,0,0,0,6,6,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,9,8,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,6,6,6,6,8,8,0,0,0,6,6,7,7,9,9,0,0,0,6,6,7,7,9,9,0,0,0,7,7,8,8,10,10,0,0,0,7,7,8,8,10,10,0,0,0,9,9,9,9,10,10,0,0,0,9,9,9,9,10,10,0,0,0,10,10,10,10,11,11,0,0,0,0,0,10,10,11,11,1,4,3,6,6,7,7,8,8,8,8,9,9,10,10,10,10,0,0,0,7,7,8,8,9,9,9,9,10,10,10,10,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,10,10,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,11,12,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,11,11,0,0,0,8,8,9,9,9,9,10,10,10,10,11,11,12,12,0,0,0,8,8,9,9,9,9,10,10,10,11,11,11,12,12,0,0,0,9,9,10,9,10,10,10,10,11,11,11,11,12,12,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,0,0,9,9,10,10,10,11,11,11,12,12,13,13,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,13,13,0,0,0,0,0,10,10,11,10,11,11,11,12,13,12,13,13,0,0,0,0,0,0,0,11,10,11,11,12,12,12,12,13,13,0,0,0,0,0,0,0,11,11,12,12,12,12,13,13,13,14,0,0,0,0,0,0,0,11,11,12,12,12,12,13,13,13,14,0,0,0,0,0,0,0,12,12,12,13,13,13,13,13,14,14,0,0,0,0,0,0,0,0,0,12,12,13,12,13,13,14,14,1,4,4,7,6,6,7,6,6,4,6,6,10,9,9,11,9,9,4,6,6,10,9,9,10,9,9,7,10,10,11,11,11,12,11,11,7,9,9,11,11,10,11,10,10,7,9,9,11,10,11,11,10,10,7,10,10,11,11,11,12,11,11,7,9,9,11,10,10,11,10,10,7,9,9,11,10,10,11,10,10,1,4,4,6,6,7,7,8,8,8,8,10,10,10,7,6,8,8,8,8,8,8,10,10,10,7,6,7,7,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,8,8,8,8,9,9,9,9,10,10,10,8,8,8,8,9,9,9,9,10,10,10,9,9,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,1,4,4,6,6,7,7,8,8,9,9,10,10,6,5,5,7,7,8,8,8,8,9,9,11,11,7,5,5,7,7,8,8,8,8,9,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,12,12,9,9,9,10,10,10,11,11,11,12,0,13,13,9,9,9,9,10,10,11,11,11,12,0,0,0,10,10,10,10,11,11,12,12,12,13,0,0,0,10,10,10,10,11,11,12,12,13,12,0,0,0,14,14,11,10,11,12,12,13,13,14,0,0,0,15,15,11,11,12,11,12,12,14,13,0,0,0,0,0,12,12,12,12,13,13,14,14,0,0,0,0,0,13,13,12,12,13,13,13,14,2,3,3,5,5,6,6,6,5,5,6,6,6,5,5,6,6,6,5,5,6,6,6,5,5,1,7,7,11,11,8,11,11,11,11,4,11,3,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,7,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,8,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,1,4,4,6,6,8,8,9,10,10,11,11,11,6,5,5,7,7,8,8,9,10,9,11,11,12,5,5,5,7,7,8,9,10,10,12,12,14,13,15,7,7,8,8,9,10,11,11,10,12,10,11,15,7,8,8,8,9,9,11,11,13,12,12,13,15,10,10,8,8,10,10,12,12,11,14,10,10,15,11,11,8,8,10,10,12,13,13,14,15,13,15,15,15,10,10,10,10,12,12,13,12,13,10,15,15,15,10,10,11,10,13,11,13,13,15,13,15,15,15,13,13,10,11,11,11,12,10,14,11,15,15,14,14,13,10,10,12,11,13,13,14,14,15,15,15,15,15,11,11,11,11,12,11,15,12,15,15,15,15,15,12,12,11,11,14,12,13,14,3,4,3,6,6,7,7,8,8,9,9,9,9,9,9,9,9,10,11,11,6,6,7,7,8,8,9,9,9,9,9,9,9,9,10,10,10,6,6,7,7,8,8,9,9,9,9,9,9,9,9,10,10,10,7,7,7,8,8,8,9,9,9,9,9,9,10,9,10,11,10,7,6,7,7,8,8,9,9,9,9,9,9,9,10,10,10,11,7,7,8,8,8,8,9,9,9,9,9,9,9,9,10,10,10,7,7,8,8,8,8,9,9,9,9,9,9,9,10,11,11,11,8,8,8,8,8,8,9,9,9,9,9,9,9,9,11,10,10,11,11,8,8,8,9,9,9,9,9,9,10,9,10,10,10,10,11,11,9,9,9,9,9,9,9,9,9,9,9,9,11,11,10,11,11,9,9,9,9,9,9,9,9,9,10,10,10,10,11,10,11,11,9,9,9,9,9,9,9,9,9,9,9,9,11,10,10,11,11,11,11,9,9,9,9,9,9,9,9,10,10,10,11,11,10,11,11,11,9,10,10,9,9,9,9,9,9,9,10,11,11,11,11,11,11,9,9,9,9,9,9,9,9,9,9,11,11,11,11,11,11,11,10,10,9,9,9,9,9,9,9,9,11,11,11,10,11,11,11,11,11,9,9,9,10,9,9,9,9,1,4,5,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,7,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,0,0,0,0,0,0,7,9,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,0,0,0,0,0,0,8,10,9,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,10,9,0,0,0,0,0,0,9,9,10,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,8,10,9,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,9,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,8,9,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,10,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,9,10,9], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+238237);
/* memory initializer */ allocate([1,4,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,5,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,5,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,7,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,4,7,7,0,0,0,0,0,4,4,7,7,0,0,0,0,0,4,5,7,7,0,0,0,0,0,6,7,8,8,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,10,9,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,1,4,3,6,6,7,7,9,9,0,5,5,7,7,8,7,9,9,0,5,5,7,7,8,8,9,9,0,7,7,8,8,8,8,10,10,0,0,0,8,8,8,8,10,10,0,0,0,9,9,9,9,10,10,0,0,0,9,9,9,9,10,10,0,0,0,10,10,10,10,11,11,0,0,0,0,0,10,10,11,11,1,4,4,6,6,8,8,9,9,8,8,9,9,10,10,11,11,0,6,6,7,7,8,8,9,9,9,9,10,10,11,11,12,12,0,6,5,7,7,8,8,9,9,9,9,10,10,11,11,12,12,0,7,7,7,7,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,7,7,8,8,9,9,10,10,11,11,11,11,12,12,0,0,0,8,8,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,8,8,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,9,9,9,9,10,10,10,10,11,11,12,12,13,13,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,13,13,0,0,0,0,0,9,9,10,10,11,11,12,12,13,13,13,13,0,0,0,0,0,9,9,10,10,11,11,12,12,12,13,13,13,0,0,0,0,0,10,10,11,11,11,11,12,12,13,13,14,14,0,0,0,0,0,0,0,11,11,11,11,12,12,13,13,14,14,0,0,0,0,0,0,0,11,11,12,12,13,13,13,13,14,14,0,0,0,0,0,0,0,11,11,12,12,13,13,13,13,14,14,0,0,0,0,0,0,0,12,12,12,13,13,13,14,14,14,14,0,0,0,0,0,0,0,0,0,12,12,13,13,14,14,14,14,1,4,4,7,6,6,7,6,6,4,7,6,10,9,9,11,9,9,4,6,7,10,9,9,11,9,9,7,10,10,10,11,11,11,11,10,6,9,9,11,10,10,11,10,10,6,9,9,11,10,11,11,10,10,7,11,11,11,11,11,12,11,11,7,9,9,11,10,10,12,10,10,7,9,9,11,10,10,11,10,10,2,4,4,5,5,7,7,7,7,8,8,10,5,5,6,6,7,7,8,8,8,8,10,5,5,6,6,7,7,8,8,8,8,10,7,7,7,7,8,8,8,8,8,8,10,10,10,7,7,7,7,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,9,9,10,10,10,10,10,8,8,8,8,9,9,10,10,10,10,10,9,9,9,9,8,9,10,10,10,10,10,8,9,8,8,9,8,1,4,4,6,6,7,7,7,7,9,9,10,10,7,5,5,7,7,8,8,8,8,10,9,11,10,7,5,5,7,7,8,8,8,8,9,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,12,12,9,9,9,10,10,10,11,11,12,12,0,13,13,9,9,9,9,10,10,11,11,12,12,0,0,0,10,10,10,10,11,11,12,12,12,13,0,0,0,10,10,10,10,11,11,12,12,12,12,0,0,0,14,14,11,11,11,11,12,13,13,13,0,0,0,14,14,11,10,11,11,12,12,13,13,0,0,0,0,0,12,12,12,12,13,13,13,14,0,0,0,0,0,13,12,12,12,13,13,13,14,2,4,4,4,5,6,5,5,5,5,6,5,5,5,5,6,5,5,5,5,6,6,6,5,5,1,4,4,12,11,13,13,14,14,4,7,7,11,13,14,14,14,14,3,8,3,14,14,14,14,14,14,14,10,12,14,14,14,14,14,14,14,14,5,14,8,14,14,14,14,14,12,14,13,14,14,14,14,14,14,14,13,14,10,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,14,1,4,4,6,6,8,8,9,9,10,11,11,11,6,5,5,7,7,8,8,10,10,10,11,11,11,6,5,5,7,7,8,8,10,10,11,12,12,12,14,7,7,7,8,9,9,11,11,11,12,11,12,17,7,7,8,7,9,9,11,11,12,12,12,12,14,11,11,8,8,10,10,11,12,12,13,11,12,14,11,11,8,8,10,10,11,12,12,13,13,12,14,15,14,10,10,10,10,11,12,12,12,12,11,14,13,16,10,10,10,9,12,11,12,12,13,14,14,15,14,14,13,10,10,11,11,12,11,13,11,14,12,15,13,14,11,10,12,10,12,12,13,13,13,13,14,15,15,12,12,11,11,12,11,13,12,14,14,14,14,17,12,12,11,10,13,11,13,13,3,4,4,6,6,7,7,8,8,8,8,9,9,9,9,9,9,10,6,6,6,6,7,7,8,8,8,9,9,9,9,9,9,9,10,6,6,6,6,7,7,8,8,8,8,9,9,9,9,9,9,10,7,7,7,7,8,8,8,8,9,9,9,9,9,9,9,9,10,10,10,7,7,7,8,8,8,9,9,9,9,9,9,9,9,10,10,10,8,8,8,8,8,8,9,9,9,9,9,9,9,9,10,10,10,8,8,8,8,8,8,9,9,9,9,9,9,9,9,11,10,11,8,8,8,8,8,8,9,9,9,9,9,9,9,9,10,10,10,11,11,8,8,8,8,9,9,9,9,9,9,9,9,11,10,11,11,11,9,9,9,9,9,9,9,9,9,9,9,9,10,11,10,11,11,9,9,9,9,9,9,9,9,9,9,9,10,11,11,10,11,11,9,9,9,9,9,9,9,9,9,9,9,9,11,10,11,11,11,11,11,9,9,9,9,9,9,9,9,9,9,10,11,11,11,11,11,11,9,10,10,10,9,9,9,9,9,9,11,10,11,11,11,11,11,9,9,9,9,9,9,9,9,9,9,11,11,11,11,11,11,11,10,10,9,9,9,9,9,9,9,9,10,11,11,11,11,11,11,11,11,9,9,9,9,9,9,9,9,4,4,7,8,7,8,10,12,17,3,1,6,6,7,8,10,12,15,7,6,9,9,9,11,12,14,17,8,6,9,6,7,9,11,13,17,7,6,9,7,7,8,9,12,15,8,8,10,8,7,7,7,10,14,9,10,12,10,8,8,8,10,14,11,13,15,13,12,11,11,12,16,17,18,18,19,20,18,16,16,20,3,3,8,8,8,8,10,12,14,3,2,6,7,7,8,10,12,16,7,6,7,9,8,10,12,14,16,8,6,8,4,5,7,9,11,13,7,6,8,5,6,7,9,11,14,8,8,10,7,7,6,8,10,13,9,11,12,9,9,7,8,10,12,10,13,15,11,11,10,9,10,13,13,16,17,14,15,14,13,14,17,9,8,12,11,12,13,14,14,16,6,1,5,6,6,9,12,14,17,9,4,5,9,7,9,13,15,16,8,5,8,6,8,10,13,17,17,9,6,7,7,8,9,13,15,17,11,8,9,9,9,10,12,16,16,13,7,8,7,7,9,12,14,15,13,6,7,5,5,7,10,13,13,14,7,8,5,6,7,9,10,12,6,6,12,13,13,14,16,17,17,4,2,5,8,7,9,12,15,15,9,4,5,9,7,9,12,16,18,11,6,7,4,6,8,11,14,18,10,5,6,5,5,7,10,14,17,10,5,7,7,6,7,10,13,16,11,5,7,7,7,8,10,12,15,13,6,7,5,5,7,9,12,13,16,8,9,6,6,7,9,10,12,1,5,5,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,8,10,9,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,10,11,0,0,0,0,0,0,9,11,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,11,9,0,0,0,0,0,0,9,10,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,11,10,0,0,0,0,0,0,9,9,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,10,0,0,0,0,0,0,9,10,11,0,0,0,0,0,0,9,11,10], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+247353);
/* memory initializer */ allocate([1,4,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,5,5,7,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,5,6,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,7,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,7,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,2,8,7,0,0,0,0,0,0,0,6,6,0,0,0,0,0,0,0,6,6,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,6,6,6,6,8,8,0,0,0,7,7,7,7,9,9,0,0,0,7,7,7,7,9,9,0,0,0,7,7,7,8,9,9,0,0,0,7,7,7,7,9,9,0,0,0,9,9,8,8,10,10,0,0,0,8,9,8,8,10,10,0,0,0,10,10,9,9,10,10,0,0,0,0,0,9,9,10,10,1,4,3,6,6,8,7,8,8,8,8,9,9,10,10,11,11,0,0,0,7,7,8,8,9,9,9,9,9,10,10,10,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,10,10,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,11,11,0,0,0,8,8,9,9,9,9,10,10,10,10,11,11,12,12,0,0,0,8,8,9,9,9,9,10,10,10,10,11,11,12,12,0,0,0,9,9,9,9,10,10,10,10,11,11,11,12,12,12,0,0,0,0,0,9,9,10,10,10,10,11,11,11,11,12,12,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,13,13,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,13,13,0,0,0,0,0,10,10,11,11,11,11,11,12,12,12,13,13,0,0,0,0,0,0,0,11,10,11,11,11,11,12,12,13,13,0,0,0,0,0,0,0,11,11,12,11,12,12,12,12,13,13,0,0,0,0,0,0,0,11,11,11,12,12,12,12,13,13,13,0,0,0,0,0,0,0,12,12,12,12,12,13,13,13,14,14,0,0,0,0,0,0,0,0,0,12,12,12,12,13,13,14,14,1,4,4,7,6,6,7,6,6,4,7,7,10,9,9,10,9,9,4,6,7,10,9,9,11,9,9,7,10,10,11,11,11,12,10,11,6,9,9,11,10,11,11,10,10,6,9,9,11,10,11,11,10,10,7,11,10,12,11,11,11,11,11,7,9,9,10,10,10,11,11,10,6,9,9,11,10,10,11,10,10,2,3,3,6,6,7,7,7,7,7,8,10,10,10,6,6,7,7,8,8,8,8,10,10,10,6,6,7,7,8,8,8,8,10,10,10,7,7,7,7,8,8,8,8,10,10,10,7,7,7,7,8,8,8,8,10,10,10,8,7,8,8,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,10,10,10,10,10,9,9,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,1,4,4,6,6,7,7,7,7,8,8,9,9,7,5,5,7,7,8,8,8,8,9,9,10,10,7,5,6,7,7,8,8,8,8,9,9,10,10,0,8,8,8,8,9,9,9,9,10,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,12,12,9,9,10,10,10,10,11,11,11,11,0,13,13,9,9,9,9,10,10,11,11,11,12,0,0,0,10,10,10,10,11,11,11,11,12,12,0,0,0,10,10,9,9,11,11,11,12,12,12,0,0,0,13,13,10,10,11,11,12,12,13,13,0,0,0,14,14,10,10,11,11,12,12,13,13,0,0,0,0,0,11,11,11,11,13,12,13,13,0,0,0,0,0,12,12,11,11,12,12,13,13,2,3,3,5,5,6,6,6,5,5,6,6,6,5,5,6,6,6,5,5,6,6,6,5,5,1,5,5,10,10,6,9,8,10,10,6,10,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,8,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,1,4,4,6,6,7,7,9,9,11,12,13,12,6,5,5,7,7,8,8,10,9,12,12,12,12,6,5,5,7,7,8,8,10,9,12,11,11,13,16,7,7,8,8,9,9,10,10,12,12,13,12,16,7,7,8,7,9,9,10,10,11,12,12,13,16,10,10,8,8,10,10,11,12,12,12,13,13,16,11,10,8,7,11,10,11,11,12,11,13,13,16,16,16,10,10,10,10,11,11,13,12,13,13,16,16,16,11,9,11,9,15,13,12,13,13,13,16,16,16,15,13,11,11,12,13,12,12,14,13,16,16,16,14,13,11,11,13,12,14,13,13,13,16,16,16,16,16,13,13,13,12,14,13,14,14,16,16,16,16,16,13,13,12,12,14,14,15,13,2,4,4,6,6,7,7,7,7,8,8,8,8,8,8,8,8,10,10,10,7,7,7,8,8,8,9,9,9,9,9,9,9,9,10,10,10,7,7,7,7,8,8,9,9,9,9,9,9,9,9,10,10,10,7,7,8,8,8,8,9,9,9,9,9,10,9,9,10,10,10,7,7,8,8,9,8,9,9,9,9,10,9,9,10,10,10,10,8,8,8,8,9,8,9,9,9,9,9,10,9,10,10,10,10,7,7,8,8,9,9,9,9,9,9,10,9,10,10,10,10,10,8,8,8,9,9,9,9,9,9,9,10,10,10,9,11,10,10,10,10,8,8,9,9,9,9,9,10,9,9,9,10,10,10,10,11,11,9,9,9,9,9,9,9,9,10,9,9,10,11,10,10,11,11,9,9,9,9,9,9,9,9,9,9,10,9,11,11,10,11,11,9,9,9,9,9,9,9,9,9,9,10,9,11,10,10,11,11,11,11,9,9,9,9,9,9,9,9,9,9,10,10,10,11,11,11,11,9,10,9,10,9,9,9,9,10,9,10,11,10,11,10,10,10,10,10,9,9,9,10,9,9,9,10,11,11,10,11,11,10,11,10,10,10,9,9,9,9,10,9,9,10,11,10,11,11,11,11,10,11,10,10,9,10,9,9,9,10,1,5,5,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,7,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,9,8,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,8,10,9,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,8,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,9,9,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,9,10,10], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+256249);
/* memory initializer */ allocate([1,4,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,5,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,5,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,7,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,7,7,0,0,0,0,0,5,4,7,7,0,0,0,0,0,5,5,7,7,0,0,0,0,0,6,7,8,8,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,9,10,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,1,4,3,6,6,7,7,9,9,0,5,5,7,7,8,7,9,9,0,5,5,7,7,8,8,9,9,0,7,7,8,8,8,8,10,10,0,0,0,8,8,8,8,10,10,0,0,0,9,9,9,9,11,11,0,0,0,9,9,9,9,11,11,0,0,0,10,10,10,10,11,11,0,0,0,0,0,9,9,11,11,1,4,4,6,6,8,8,8,8,8,8,9,9,10,10,11,11,0,6,6,7,7,8,8,9,9,9,9,10,10,10,11,11,11,0,5,6,7,7,8,8,9,9,9,9,10,10,10,11,11,11,0,7,7,8,8,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,8,8,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,8,8,9,9,10,10,10,10,11,11,11,11,12,12,0,0,0,8,8,9,9,10,10,10,10,11,11,11,11,12,12,0,0,0,9,9,9,9,10,10,10,10,11,11,12,12,12,13,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,13,13,0,0,0,0,0,9,9,10,10,11,11,11,11,12,12,13,13,0,0,0,0,0,9,9,10,10,11,10,11,11,12,12,13,13,0,0,0,0,0,10,10,10,10,11,11,12,12,12,13,13,13,0,0,0,0,0,0,0,10,10,11,11,12,12,12,13,13,13,0,0,0,0,0,0,0,11,11,12,12,12,12,13,13,14,14,0,0,0,0,0,0,0,11,11,12,11,12,12,13,13,13,13,0,0,0,0,0,0,0,12,12,12,12,13,13,13,13,14,14,0,0,0,0,0,0,0,0,0,12,12,12,12,13,13,14,14,1,4,4,7,6,6,7,6,6,4,7,7,10,9,9,11,9,9,4,7,7,10,9,9,11,9,9,7,10,10,10,11,11,11,10,10,6,9,9,11,11,10,11,10,10,6,9,9,11,10,11,11,10,10,7,11,10,11,11,11,11,11,11,6,9,9,11,10,10,11,11,10,6,9,9,11,10,10,11,10,11,2,4,4,6,6,7,7,7,7,7,8,9,5,5,6,6,7,7,8,8,8,8,9,5,5,6,6,7,7,8,8,8,8,10,7,7,7,7,7,7,8,8,8,8,10,10,10,7,7,7,7,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,10,10,10,10,10,9,9,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,1,4,4,6,6,7,7,7,7,8,8,9,9,7,5,5,7,7,8,8,8,8,9,9,10,10,7,6,5,7,7,8,8,8,8,9,9,10,10,0,8,8,8,8,9,9,9,9,10,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,12,12,9,9,10,10,10,10,11,11,11,11,0,13,13,9,9,9,9,10,10,11,11,11,12,0,0,0,9,10,10,10,11,11,12,11,12,12,0,0,0,10,10,9,9,11,11,12,12,12,12,0,0,0,13,13,10,10,11,11,12,12,13,13,0,0,0,14,14,10,10,11,11,12,12,13,13,0,0,0,0,0,11,12,11,11,13,12,13,13,0,0,0,0,0,12,12,11,11,13,12,14,14,2,4,4,4,4,6,5,5,5,5,6,5,5,5,5,6,6,6,5,5,6,6,6,5,5,1,3,3,11,11,11,11,11,11,3,7,6,11,11,11,11,11,11,4,8,7,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,1,4,4,6,6,7,7,9,9,10,11,12,12,6,5,5,7,7,8,8,10,10,12,11,12,12,6,5,5,7,7,8,8,10,10,12,11,12,12,17,7,7,8,8,9,9,10,10,12,12,13,13,18,7,7,8,7,9,9,10,10,12,12,12,13,19,10,10,8,8,10,10,11,11,12,12,13,14,19,11,10,8,7,10,10,11,11,12,12,13,12,19,19,19,10,10,10,10,11,11,12,12,13,13,19,19,19,11,9,11,9,14,12,13,12,13,13,19,20,18,13,14,11,11,12,12,13,13,14,13,20,20,20,15,13,11,10,13,11,13,13,14,13,20,20,20,20,20,13,14,12,12,13,13,13,13,20,20,20,20,20,13,13,12,12,16,13,15,13,2,5,5,6,6,7,7,7,7,8,8,8,8,8,8,8,8,10,6,6,7,7,7,7,8,8,9,9,9,9,9,9,9,9,10,6,6,7,7,8,7,8,8,9,9,9,9,9,9,9,9,10,7,7,7,7,8,8,8,9,9,9,9,9,9,9,9,9,10,10,10,7,7,8,8,9,8,9,9,9,9,10,9,9,10,10,10,11,8,8,8,8,9,9,9,9,9,9,9,10,9,10,10,10,10,8,8,8,8,9,9,9,9,9,9,9,9,10,10,11,10,10,8,8,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,11,11,8,8,9,9,9,9,9,9,9,9,9,10,11,11,11,11,11,9,9,9,9,9,9,9,9,10,9,10,9,11,11,10,11,11,9,9,9,9,9,9,9,9,9,9,10,9,11,11,10,11,11,9,9,9,9,9,9,9,9,9,9,10,9,11,10,11,11,11,11,11,9,9,10,9,9,9,9,9,9,9,10,11,10,11,11,11,11,10,10,10,10,9,9,9,9,9,9,10,11,11,11,11,11,11,9,10,9,9,9,9,9,9,9,9,11,11,10,11,11,11,10,10,10,9,9,9,9,9,9,9,9,10,11,10,11,11,11,11,11,11,9,9,9,9,9,9,9,9,5,4,8,9,8,9,10,12,15,4,1,5,5,6,8,11,12,12,8,5,8,9,9,11,13,12,12,9,5,8,5,7,9,12,13,13,8,6,8,7,7,9,11,11,11,9,7,9,7,7,7,7,10,12,10,10,11,9,8,7,7,9,11,11,12,13,12,11,9,8,9,11,13,16,16,15,15,12,10,11,12,5,4,9,10,9,10,11,12,13,4,1,5,7,7,9,11,12,14,8,5,7,9,8,10,13,13,13,10,7,9,4,6,7,10,12,14,9,6,7,6,6,7,10,12,12,9,8,9,7,6,7,8,11,12,11,11,11,9,8,7,8,10,12,12,13,14,12,11,9,9,9,12,12,17,17,15,16,12,10,11,13,6,8,13,12,13,14,15,16,16,4,2,4,7,6,8,11,13,15,10,4,4,8,6,8,11,14,17,11,5,6,5,6,8,12,14,17,11,5,5,6,5,7,10,13,16,12,6,7,8,7,8,10,13,15,13,8,8,7,7,8,10,12,15,15,7,7,5,5,7,9,12,14,15,8,8,6,6,7,8,10,11,4,7,13,14,14,15,16,18,18,4,2,5,8,7,9,12,15,15,10,4,5,10,6,8,11,15,17,12,5,7,5,6,8,11,14,17,11,5,6,6,5,6,9,13,17,12,6,7,6,5,6,8,12,14,14,7,8,6,6,7,9,11,14,14,8,9,6,5,6,9,11,13,16,10,10,7,6,7,8,10,11,2,4,4,0,0,0,0,0,0,5,7,6,0,0,0,0,0,0,5,6,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,8,9,10,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,8,9,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,7,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,8,10,9,0,0,0,0,0,0,8,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,9,9], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+265365);
/* memory initializer */ allocate([2,3,4,6,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,5,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,2,7,7,0,0,0,0,0,13,13,6,6,0,0,0,0,0,12,0,6,6,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,8,9,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,11,10,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,6,5,6,6,8,8,0,0,0,7,7,7,7,9,9,0,0,0,7,7,7,7,9,9,0,0,0,7,7,8,8,10,10,0,0,0,7,7,8,8,10,10,0,0,0,9,9,8,8,10,10,0,0,0,8,8,8,8,10,10,0,0,0,10,10,9,9,11,11,0,0,0,0,0,9,9,11,11,1,4,3,6,6,7,7,8,8,8,8,9,9,10,10,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,10,10,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,10,10,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,12,11,0,0,0,8,8,9,9,9,10,10,10,10,10,11,11,12,12,0,0,0,8,8,9,9,10,9,10,10,10,10,11,11,12,12,0,0,0,9,9,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,0,0,9,9,10,10,10,11,11,11,12,12,13,13,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,13,13,0,0,0,0,0,10,10,10,10,11,11,12,12,12,12,13,13,0,0,0,0,0,0,0,10,10,11,11,12,12,12,12,13,13,0,0,0,0,0,0,0,11,11,12,12,12,12,13,13,13,13,0,0,0,0,0,0,0,11,11,11,11,12,12,13,13,13,13,0,0,0,0,0,0,0,12,12,12,12,12,12,13,13,14,14,0,0,0,0,0,0,0,0,0,12,12,12,12,13,13,14,14,1,4,4,7,6,6,7,6,6,4,7,7,10,9,9,11,9,9,4,7,7,10,9,9,11,9,9,6,10,10,11,11,11,11,10,10,6,9,9,11,10,10,11,10,10,6,9,9,11,10,11,11,10,10,7,11,10,11,11,11,12,11,11,7,9,9,11,10,10,11,11,10,6,9,9,10,10,10,12,10,11,2,3,3,6,6,7,7,7,7,8,8,10,10,10,6,6,7,7,8,8,8,8,10,10,10,6,6,7,7,8,8,8,8,10,10,10,7,7,7,7,8,8,8,8,10,10,10,7,7,7,7,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,10,10,10,10,10,9,9,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,1,4,4,6,6,7,7,7,7,8,8,10,9,7,5,6,7,7,8,8,8,8,9,9,10,10,7,5,5,7,7,8,8,8,8,9,9,10,10,0,8,8,8,8,9,9,9,9,10,10,11,10,0,8,8,8,8,9,9,9,9,10,10,11,11,0,12,12,9,9,9,10,10,10,11,11,11,11,0,13,13,9,9,9,9,10,10,11,11,11,11,0,0,0,10,10,10,10,11,11,12,11,12,12,0,0,0,10,10,10,9,11,11,12,11,13,12,0,0,0,13,13,10,10,11,11,12,12,13,13,0,0,0,14,14,10,10,11,11,12,12,13,13,0,0,0,0,0,11,12,11,11,12,12,14,13,0,0,0,0,0,12,11,11,11,13,10,14,13,2,3,3,5,5,6,6,6,5,5,6,6,6,5,5,6,6,6,5,5,6,6,6,5,5,1,4,3,10,10,10,10,10,10,10,10,10,10,4,8,6,10,10,10,10,10,10,10,10,10,10,4,8,7,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,1,4,4,6,5,7,7,9,9,10,10,12,12,6,5,5,7,7,8,8,10,10,12,11,12,12,6,5,5,7,7,8,8,10,10,11,11,12,12,15,7,7,8,8,9,9,11,11,12,12,13,12,15,8,8,8,7,9,9,10,10,12,12,13,13,16,11,10,8,8,10,10,11,11,12,12,13,13,16,11,11,9,8,11,10,11,11,12,12,13,12,16,16,16,10,11,10,11,12,12,12,12,13,13,16,16,16,11,9,11,9,14,12,12,12,13,13,16,16,16,12,14,11,12,12,12,13,13,14,13,16,16,16,15,13,12,10,13,10,13,14,13,13,16,16,16,16,16,13,14,12,13,13,12,13,13,16,16,16,16,16,13,12,12,11,14,12,15,13,2,4,4,6,6,6,6,7,7,8,8,8,8,8,8,8,8,10,10,10,7,7,7,7,8,8,9,9,9,9,9,9,9,9,10,10,10,7,7,8,7,8,8,9,9,9,9,9,9,9,9,10,10,10,7,7,8,8,8,9,9,9,9,9,9,10,9,9,10,10,10,7,7,8,8,9,8,9,9,9,9,10,9,9,10,10,11,11,8,8,8,8,9,9,9,9,9,9,10,9,9,10,10,10,10,8,8,8,8,9,9,9,9,9,9,9,9,10,10,11,11,11,8,8,9,9,9,9,9,9,9,9,9,9,10,10,10,10,11,11,11,8,8,9,9,9,9,10,9,9,9,9,9,11,11,11,11,11,9,9,9,9,9,9,9,9,9,9,9,9,11,10,10,11,11,9,9,9,9,9,9,9,9,9,10,10,10,10,11,10,11,11,9,9,9,9,9,9,9,9,9,10,10,9,10,10,11,11,11,11,11,9,9,9,10,9,9,9,9,9,9,10,11,11,11,11,11,11,10,10,10,10,9,9,9,9,9,9,10,11,11,11,11,11,11,9,10,9,9,9,9,10,9,9,9,11,11,11,11,11,11,11,10,10,9,9,9,9,9,9,10,9,11,11,10,11,11,11,11,10,11,9,9,9,9,9,9,9,9,1,5,5,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,7,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,9,8,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,9,10,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,8,10,9,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,8,9,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,9,10,9], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+274261);
/* memory initializer */ allocate([2,3,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,7,7,0,0,0,0,0,5,5,6,6,0,0,0,0,0,5,5,7,7,0,0,0,0,0,7,7,7,7,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,8,9,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,6,6,7,7,9,9,0,6,6,7,7,8,8,9,9,0,6,6,7,7,8,8,9,9,0,7,7,8,8,8,8,10,10,0,0,0,8,8,8,8,10,10,0,0,0,8,8,9,9,11,11,0,0,0,9,9,9,9,11,11,0,0,0,10,10,10,10,11,11,0,0,0,0,0,9,9,11,11,2,3,3,6,6,7,7,8,8,8,8,9,9,10,10,11,11,0,5,5,6,6,8,8,9,9,9,9,10,10,10,10,11,11,0,5,5,6,6,8,8,9,9,9,9,10,10,10,10,11,11,0,7,7,7,7,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,8,8,8,8,9,9,10,10,10,11,11,11,12,12,0,0,0,8,8,8,8,9,9,10,10,10,10,11,11,12,12,0,0,0,9,9,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,13,13,0,0,0,0,0,9,9,9,9,10,10,11,11,12,12,13,13,0,0,0,0,0,9,9,9,9,10,10,11,11,12,12,13,13,0,0,0,0,0,9,9,10,10,11,11,12,12,12,12,13,13,0,0,0,0,0,0,0,10,10,11,11,12,12,12,12,13,13,0,0,0,0,0,0,0,11,11,11,11,12,12,13,13,13,13,0,0,0,0,0,0,0,11,11,11,11,12,12,13,13,13,13,0,0,0,0,0,0,0,11,11,12,12,12,12,13,13,14,14,0,0,0,0,0,0,0,0,0,12,12,12,12,13,13,14,14,1,4,4,7,6,6,7,6,6,4,7,7,10,9,9,11,9,9,4,7,7,10,9,9,11,9,9,7,10,10,10,11,11,11,10,10,6,9,9,11,11,10,11,10,10,6,9,9,11,10,11,11,10,10,7,11,11,11,11,11,11,11,11,6,9,9,11,10,10,11,11,10,6,9,9,10,10,10,11,10,11,2,4,4,6,6,7,7,7,7,8,8,10,5,5,6,6,7,7,8,8,8,8,10,5,5,6,6,7,7,8,8,8,8,10,7,7,7,7,8,8,8,8,8,8,10,10,10,7,7,7,7,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,9,8,10,10,10,10,10,8,8,8,8,8,8,10,10,10,10,10,9,9,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,1,4,4,6,6,7,7,7,7,8,8,9,9,7,5,5,7,7,8,8,8,8,9,9,10,10,7,5,6,7,7,8,8,8,8,9,9,11,10,0,8,8,8,8,9,9,9,9,10,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,12,12,9,9,10,10,10,10,11,11,11,11,0,13,13,9,9,9,9,10,10,11,11,12,12,0,0,0,9,10,9,10,11,11,12,11,13,12,0,0,0,10,10,9,9,11,11,12,12,13,12,0,0,0,13,13,10,10,11,11,12,12,13,13,0,0,0,14,14,10,10,11,11,12,12,13,13,0,0,0,0,0,11,12,11,11,12,13,14,13,0,0,0,0,0,12,12,11,11,13,12,14,13,2,4,4,4,5,6,5,5,5,5,6,5,5,5,5,6,5,5,5,5,6,6,6,5,5,1,3,3,13,13,13,13,13,13,13,13,13,13,3,6,6,13,13,13,13,13,13,13,13,13,13,4,8,7,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,1,4,4,6,6,7,7,9,9,10,11,12,12,6,5,5,7,7,8,7,10,10,11,11,12,12,6,5,5,7,7,8,8,10,10,11,11,12,12,16,7,7,8,8,9,9,11,11,12,12,13,13,17,7,7,8,7,9,9,11,10,12,12,13,13,19,11,10,8,8,10,10,11,11,12,12,13,13,19,11,11,9,7,11,10,11,11,12,12,13,12,19,19,19,10,10,10,10,11,12,12,12,13,14,18,19,19,11,9,11,9,13,12,12,12,13,13,19,20,19,13,15,11,11,12,12,13,13,14,13,18,19,20,15,13,12,10,13,10,13,13,13,14,20,20,20,20,20,13,14,12,12,13,12,13,13,20,20,20,20,20,13,12,12,12,14,12,14,13,2,5,5,6,6,7,6,7,7,8,8,8,8,8,8,8,8,10,6,6,7,7,7,7,8,8,9,9,9,9,9,9,9,9,10,6,6,7,7,8,8,8,8,9,9,9,9,9,9,9,9,10,7,7,7,7,8,8,8,9,9,9,9,9,9,9,9,9,10,10,10,7,7,8,8,9,9,9,9,9,9,9,9,9,9,10,11,11,8,8,8,8,9,9,9,9,9,9,10,10,9,10,10,10,10,8,8,8,8,9,9,9,9,9,9,9,9,10,10,11,10,10,8,8,9,9,9,9,9,9,9,9,9,9,10,9,10,10,10,11,11,8,8,9,9,9,9,9,9,9,9,9,9,11,11,11,11,11,9,9,9,9,9,9,9,9,10,9,10,9,11,11,11,11,11,9,8,9,9,9,9,9,9,9,10,10,9,11,11,10,11,11,9,9,9,9,9,9,9,9,9,10,10,9,11,11,11,11,11,11,11,9,9,10,9,9,9,9,10,9,10,10,11,10,11,11,11,11,9,10,10,10,9,9,9,9,9,9,10,11,11,11,11,11,11,9,9,9,9,9,9,9,9,10,9,11,11,10,11,11,11,11,10,10,9,9,9,9,9,9,10,9,10,11,10,11,11,11,11,11,11,9,9,10,9,9,9,9,9,5,5,9,10,9,9,10,11,12,5,1,5,6,6,7,10,12,14,9,5,6,8,8,10,12,14,14,10,5,8,5,6,8,11,13,14,9,5,7,6,6,8,10,12,11,9,7,9,7,6,6,7,10,10,10,9,12,9,8,7,7,10,12,11,11,13,12,10,9,8,9,11,11,14,15,15,13,11,9,9,11,5,4,8,10,9,9,10,11,12,4,2,5,6,6,8,10,11,13,8,4,6,8,7,9,12,12,14,10,6,8,4,5,6,9,11,12,9,5,6,5,5,6,9,11,11,9,7,9,6,5,5,7,10,10,10,9,11,8,7,6,7,9,11,11,12,13,10,10,9,8,9,11,11,15,15,12,13,11,9,10,11,11,9,13,12,12,11,12,12,13,15,8,2,11,4,8,5,7,10,12,15,13,7,10,9,8,8,10,13,17,17,11,4,12,5,9,5,8,11,14,16,12,6,8,7,6,6,8,11,13,16,11,4,9,5,6,4,6,10,13,16,11,6,11,7,7,6,7,10,13,15,13,9,12,9,8,6,8,10,12,14,14,10,10,8,6,5,6,9,11,13,15,11,11,9,6,5,6,8,9,12,2,4,4,0,0,0,0,0,0,5,6,6,0,0,0,0,0,0,5,6,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,6,8,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,8,9,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,9,9], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+282921);
/* memory initializer */ allocate([1,4,4,0,0,0,7,7,0,0,0,7,7,0,0,0,8,8,0,0,0,0,0,0,0,4,6,6,0,0,0,8,8,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,4,6,6,0,0,0,8,8,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,8,8,0,0,0,11,11,0,0,0,11,11,0,0,0,12,11,0,0,0,0,0,0,0,7,8,8,0,0,0,10,11,0,0,0,11,11,0,0,0,11,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,8,0,0,0,11,11,0,0,0,11,11,0,0,0,12,12,0,0,0,0,0,0,0,6,8,8,0,0,0,10,11,0,0,0,10,11,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,9,9,0,0,0,11,12,0,0,0,11,12,0,0,0,12,11,0,0,0,0,0,0,0,8,10,9,0,0,0,12,11,0,0,0,12,11,0,0,0,11,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,4,3,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,6,6,0,0,0,0,0,6,6,6,6,0,0,0,0,0,6,6,6,6,0,0,0,0,0,7,7,6,6,0,0,0,0,0,0,0,6,7,0,0,0,0,0,0,0,7,8,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,6,6,7,7,9,9,0,7,7,7,7,7,7,9,9,0,7,7,7,7,7,7,9,9,0,8,8,7,7,8,8,10,10,0,0,0,7,7,8,8,10,10,0,0,0,9,9,8,8,10,10,0,0,0,9,9,8,8,10,10,0,0,0,10,10,9,9,11,11,0,0,0,0,0,9,9,11,11,1,4,3,6,6,8,8,9,9,9,9,9,9,10,10,11,11,0,7,7,7,7,8,8,9,9,9,9,10,10,11,11,12,11,0,7,7,7,7,8,8,9,9,9,9,10,10,11,11,11,12,0,8,8,7,7,9,9,10,10,9,9,10,10,11,11,12,12,0,0,0,7,7,9,9,10,10,10,9,10,10,11,11,12,12,0,0,0,8,8,9,9,10,10,10,10,11,11,11,11,12,12,0,0,0,8,8,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,9,9,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,13,13,0,0,0,0,0,9,9,10,10,11,11,11,11,12,12,13,13,0,0,0,0,0,9,9,10,10,11,11,11,11,12,12,13,13,0,0,0,0,0,10,10,10,10,11,11,12,12,13,12,13,13,0,0,0,0,0,0,0,10,10,11,11,12,12,13,13,13,13,0,0,0,0,0,0,0,11,11,12,12,12,12,13,13,13,14,0,0,0,0,0,0,0,11,11,12,12,12,12,13,13,13,14,0,0,0,0,0,0,0,12,12,12,12,13,13,13,13,14,14,0,0,0,0,0,0,0,0,0,12,12,13,13,13,13,14,14,1,4,4,7,6,6,7,6,6,4,7,7,10,9,9,11,9,9,4,7,7,10,9,9,10,9,9,7,10,10,11,10,11,11,10,11,6,9,9,11,10,10,11,10,10,6,9,9,11,10,11,11,10,10,7,11,10,11,11,11,12,11,11,6,9,9,11,10,10,11,11,10,6,9,9,11,10,10,12,10,11,2,3,4,6,6,7,7,7,7,7,7,9,7,7,6,6,7,7,8,8,8,8,9,6,6,6,6,7,7,8,8,8,8,10,7,7,7,7,7,7,8,8,8,8,10,10,10,7,7,7,7,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,7,8,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,10,10,10,10,10,9,9,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,1,4,4,6,6,7,7,7,7,8,8,9,9,6,5,5,7,7,8,8,8,8,9,9,10,10,7,6,5,7,7,8,8,8,8,9,9,10,10,0,8,8,8,8,9,9,9,9,10,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,12,12,9,9,10,10,10,10,11,11,11,11,0,13,13,9,9,10,10,10,10,11,11,12,12,0,0,0,10,10,10,10,11,11,12,12,12,13,0,0,0,10,10,10,10,11,11,12,12,12,12,0,0,0,14,14,10,11,11,11,12,12,13,13,0,0,0,14,14,11,10,11,11,13,12,13,13,0,0,0,0,0,12,12,11,12,13,12,14,14,0,0,0,0,0,12,12,12,12,13,12,14,14,2,4,4,5,4,6,5,5,5,5,6,5,5,5,5,6,5,5,5,5,6,6,6,5,5,1,5,4,12,12,12,12,12,12,12,12,12,12,4,9,8,11,11,11,11,11,11,11,11,11,11,2,8,7,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,1,4,4,6,6,7,6,8,8,10,9,10,10,6,5,5,7,7,8,7,10,9,11,11,12,13,6,5,5,7,7,8,8,10,10,11,11,13,13,18,8,8,8,8,9,9,10,10,12,12,12,13,18,8,8,8,8,9,9,10,10,12,12,13,13,18,11,11,8,8,10,10,11,11,12,11,13,12,18,11,11,9,7,10,10,11,11,11,12,12,13,17,17,17,10,10,11,11,12,12,12,10,12,12,17,17,17,11,10,11,10,13,12,11,12,12,12,17,17,17,15,14,11,11,12,11,13,10,13,12,17,17,17,14,14,12,10,11,11,13,13,13,13,17,17,16,17,16,13,13,12,10,13,10,14,13,17,16,17,16,17,13,12,12,10,13,11,14,14,2,4,4,6,6,7,7,7,7,7,7,8,8,8,8,8,8,10,7,7,7,7,7,7,8,8,8,8,9,9,9,9,9,9,10,7,7,7,7,8,8,8,8,8,8,9,9,9,9,9,9,10,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,10,10,10,8,7,8,8,8,8,9,9,9,9,9,9,9,9,10,11,11,8,8,8,8,9,9,9,9,9,9,10,9,9,9,10,11,10,8,8,8,8,9,9,9,9,9,9,9,10,10,10,10,11,10,8,8,9,9,9,9,9,9,10,9,9,10,9,10,11,10,11,11,11,8,8,9,9,9,9,9,9,9,9,10,10,11,11,11,11,11,9,9,9,9,9,9,10,9,9,9,10,10,11,11,11,11,11,9,9,9,9,9,9,9,9,9,10,9,10,11,11,11,11,11,9,9,9,9,10,10,9,9,9,10,10,10,11,11,11,11,11,11,11,9,9,9,10,9,9,10,10,10,10,11,11,10,11,11,11,11,10,9,10,10,9,9,9,9,10,10,11,10,11,11,11,11,11,9,9,9,9,10,9,10,10,10,10,11,10,11,11,11,11,11,10,10,9,9,10,9,10,10,10,10,10,10,10,11,11,11,11,11,11,9,9,10,9,10,9,10,10,6,6,12,10,10,10,9,10,12,12,6,1,10,5,6,6,7,9,11,14,12,9,8,11,7,8,9,11,13,15,10,5,12,7,8,7,9,12,14,15,10,6,7,8,5,6,7,9,12,14,9,6,8,7,6,6,7,9,12,12,9,7,9,9,7,6,6,7,10,10,10,9,10,11,8,7,6,6,8,10,12,11,13,13,11,10,8,8,8,10,11,13,15,15,14,13,10,8,8,9,10,9,13,11,14,10,12,13,13,14,7,2,12,5,10,5,7,10,12,14,12,6,9,8,7,7,9,11,13,16,10,4,12,5,10,6,8,12,14,16,12,6,8,7,6,5,7,11,12,16,10,4,8,5,6,4,6,9,13,16,10,6,10,7,7,6,7,9,13,15,12,9,11,9,8,6,7,10,12,14,14,11,10,9,6,5,6,9,11,13,15,13,11,10,6,5,6,8,9,11,2,4,4,0,0,0,0,0,0,5,6,6,0,0,0,0,0,0,5,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,6,7,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,6,8,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,8,8,9,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,7,9,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,7,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,9,8], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+291843);
/* memory initializer */ allocate([2,5,5,0,0,0,5,5,0,0,0,5,5,0,0,0,7,8,0,0,0,0,0,0,0,5,6,6,0,0,0,7,7,0,0,0,7,7,0,0,0,10,10,0,0,0,0,0,0,0,5,6,6,0,0,0,7,7,0,0,0,7,7,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,7,7,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,5,7,7,0,0,0,7,7,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,7,7,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,5,7,7,0,0,0,7,7,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,10,10,0,0,0,9,9,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,8,10,10,0,0,0,9,9,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,4,3,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,3,6,6,0,0,0,0,0,4,4,6,6,0,0,0,0,0,4,4,6,6,0,0,0,0,0,5,5,6,6,0,0,0,0,0,0,0,6,6,0,0,0,0,0,0,0,7,8,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,4,6,6,7,7,9,9,0,5,5,7,7,7,8,9,9,0,5,5,7,7,8,8,9,9,0,7,7,8,8,8,8,10,10,0,0,0,8,8,8,8,10,10,0,0,0,9,9,9,9,10,10,0,0,0,9,9,9,9,10,10,0,0,0,10,10,10,10,11,11,0,0,0,0,0,10,10,11,11,2,3,3,6,6,7,7,8,8,8,8,9,9,10,10,11,10,0,5,5,7,7,8,8,9,9,9,9,10,10,10,10,11,11,0,5,5,7,7,8,8,9,9,9,9,10,10,10,10,11,11,0,6,6,7,7,8,8,9,9,9,9,10,10,11,11,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,11,12,0,0,0,8,8,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,8,8,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,9,9,9,9,10,10,10,10,11,10,11,11,12,12,0,0,0,0,0,9,9,10,10,10,10,11,11,11,11,12,12,0,0,0,0,0,9,8,9,9,10,10,11,11,12,12,12,12,0,0,0,0,0,8,8,9,9,10,10,11,11,12,11,12,12,0,0,0,0,0,9,10,10,10,11,11,11,11,12,12,13,13,0,0,0,0,0,0,0,10,10,10,10,11,11,12,12,13,13,0,0,0,0,0,0,0,11,11,11,11,12,12,12,12,13,13,0,0,0,0,0,0,0,11,11,11,11,12,12,12,12,13,13,0,0,0,0,0,0,0,11,11,12,12,12,12,13,13,13,13,0,0,0,0,0,0,0,0,0,12,12,12,12,13,13,13,13,1,4,4,7,6,6,7,6,6,4,7,7,10,9,9,11,9,9,4,7,7,10,9,9,11,9,9,7,10,10,11,11,10,12,11,11,6,9,9,11,10,10,11,10,10,6,9,9,11,10,10,11,10,10,7,11,11,11,11,11,12,11,11,6,9,9,11,10,10,11,10,10,6,9,9,11,10,10,11,10,10,2,4,4,6,6,7,7,7,7,8,8,10,5,5,6,6,7,7,8,8,8,8,10,5,5,6,6,7,7,8,8,8,8,10,6,6,7,7,8,8,8,8,8,8,10,10,10,7,7,8,7,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,7,8,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,10,10,10,10,10,9,9,8,8,9,8,10,10,10,10,10,8,8,8,8,8,8,1,4,4,6,6,7,7,8,8,9,9,10,10,6,5,5,7,7,8,8,8,8,9,9,10,10,7,5,5,7,7,8,8,8,8,9,9,11,10,0,8,8,8,8,9,9,9,9,10,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,12,12,9,9,10,10,10,10,11,11,11,12,0,13,13,9,9,10,10,10,10,11,11,12,12,0,0,0,10,10,10,10,11,11,12,12,12,12,0,0,0,10,10,10,10,11,11,12,12,12,12,0,0,0,14,14,11,11,11,11,12,12,13,13,0,0,0,14,14,11,11,11,11,12,12,13,13,0,0,0,0,0,12,12,12,12,13,13,14,13,0,0,0,0,0,13,13,12,12,13,12,14,13,2,4,4,5,5,6,5,5,5,5,6,4,5,5,5,6,5,5,5,5,6,6,6,5,5,1,4,4,12,12,12,12,12,12,12,12,12,12,4,9,8,12,12,12,12,12,12,12,12,12,12,2,9,7,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,1,4,4,6,6,7,7,8,7,9,9,10,10,10,10,6,5,5,7,7,8,8,10,8,11,10,12,12,13,13,6,5,5,7,7,8,8,10,9,11,11,12,12,13,12,18,8,8,8,8,9,9,10,9,11,10,12,12,13,13,18,8,8,8,8,9,9,10,10,11,11,13,12,14,13,18,11,11,9,9,10,10,11,11,11,12,13,12,13,14,18,11,11,9,8,11,10,11,11,11,11,12,12,14,13,18,18,18,10,11,10,11,12,12,12,12,13,12,14,13,18,18,18,10,11,11,9,12,11,12,12,12,13,13,13,18,18,17,14,14,11,11,12,12,13,12,14,12,14,13,18,18,18,14,14,11,10,12,9,12,13,13,13,13,13,18,18,17,16,18,13,13,12,12,13,11,14,12,14,14,17,18,18,17,18,13,12,13,10,12,11,14,14,14,14,17,18,18,18,18,15,16,12,12,13,10,14,12,14,15,18,18,18,16,17,16,14,12,11,13,10,13,13,14,15,2,5,5,6,6,7,7,7,7,7,7,8,8,8,8,8,8,10,6,6,7,7,8,7,8,8,8,8,8,9,9,9,9,9,10,6,6,7,7,7,7,8,8,8,8,9,9,9,9,9,9,10,7,7,7,7,8,8,8,8,9,9,9,9,9,9,9,9,10,10,10,7,7,8,8,8,9,9,9,9,9,9,9,9,9,11,11,11,8,8,8,8,9,9,9,9,9,9,9,9,9,9,10,10,10,8,8,8,8,9,9,9,9,9,9,9,9,9,9,10,10,10,8,9,9,9,9,9,9,9,9,9,9,9,10,9,10,10,10,11,11,9,9,9,9,9,9,9,9,9,9,9,9,11,10,11,11,11,9,9,9,9,9,9,10,10,9,9,10,9,11,10,11,11,11,9,9,9,9,9,9,9,9,10,10,10,9,11,11,11,11,11,9,9,9,9,10,10,9,9,9,9,10,9,11,11,11,11,11,11,11,9,9,9,9,9,9,10,10,10,10,11,11,11,11,11,11,11,10,9,10,10,9,10,9,9,10,9,11,10,10,11,11,11,11,9,10,9,9,9,9,10,10,10,10,11,11,11,11,11,11,10,10,10,9,9,10,9,10,9,10,10,10,10,11,11,11,11,11,11,11,9,9,9,9,9,10,10,10,5,6,11,11,11,11,10,10,12,11,5,2,11,5,6,6,7,9,11,13,13,10,7,11,6,7,8,9,10,12,11,5,11,6,8,7,9,11,14,15,11,6,6,8,4,5,7,8,10,13,10,5,7,7,5,5,6,8,10,11,10,7,7,8,6,5,5,7,9,9,11,8,8,11,8,7,6,6,7,9,12,11,10,13,9,9,7,7,7,9,11,13,12,15,12,11,9,8,8,8,4,7,14,10,15,10,12,15,16,15,4,2,11,5,10,6,8,11,14,14,14,10,7,11,6,8,10,11,13,15,9,4,11,5,9,6,9,12,14,15,14,9,6,9,4,5,7,10,12,13,9,5,7,6,5,5,7,10,13,13,10,8,9,8,7,6,8,10,14,14,13,11,10,10,7,7,8,11,14,15,13,12,9,9,6,5,7,10,14,17,15,13,11,10,6,6,7,9,12,17,2,4,4,0,0,0,0,0,0,5,6,6,0,0,0,0,0,0,5,6,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,6,8,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,8,9,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,7,7,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,8,8,0,0,0,0,0,0,8,9,9,0,0,0,0,0,0,8,9,9], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+301328);
/* memory initializer */ allocate([2,5,5,0,0,0,5,5,0,0,0,5,5,0,0,0,7,7,0,0,0,0,0,0,0,5,6,6,0,0,0,7,7,0,0,0,7,7,0,0,0,10,10,0,0,0,0,0,0,0,5,6,6,0,0,0,7,7,0,0,0,7,7,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,0,0,0,7,7,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,5,7,8,0,0,0,7,7,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,7,7,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,5,7,7,0,0,0,7,7,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,10,10,0,0,0,9,9,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,8,10,10,0,0,0,9,9,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,3,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,5,4,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,5,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,3,6,6,0,0,0,0,0,4,4,6,6,0,0,0,0,0,4,4,6,6,0,0,0,0,0,5,5,6,6,0,0,0,0,0,0,0,6,6,0,0,0,0,0,0,0,7,8,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,3,6,6,7,7,9,9,0,4,4,6,6,7,7,9,9,0,4,5,6,6,7,7,9,9,0,6,6,7,7,8,8,10,10,0,0,0,7,7,8,8,10,9,0,0,0,9,8,8,8,10,10,0,0,0,8,8,8,8,10,10,0,0,0,10,10,9,9,11,11,0,0,0,0,0,9,9,10,10,2,4,4,6,6,8,8,9,9,8,8,9,9,10,10,11,11,0,4,4,6,6,8,8,9,9,9,9,10,10,11,11,11,11,0,4,4,7,6,8,8,9,9,9,9,10,10,11,11,11,11,0,6,6,7,7,8,8,9,9,9,9,10,10,11,11,11,12,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,8,8,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,8,8,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,9,9,9,9,10,10,10,10,11,11,11,11,12,12,0,0,0,0,0,9,9,10,10,10,10,11,11,11,11,12,12,0,0,0,0,0,9,9,9,10,10,10,11,11,11,11,12,12,0,0,0,0,0,9,9,9,9,10,10,11,11,11,12,12,12,0,0,0,0,0,10,10,10,10,11,11,11,11,12,12,13,12,0,0,0,0,0,0,0,10,10,11,11,11,11,12,12,12,12,0,0,0,0,0,0,0,11,11,11,11,12,12,12,12,13,13,0,0,0,0,0,0,0,11,11,11,11,12,12,12,12,13,13,0,0,0,0,0,0,0,12,12,12,12,12,12,13,13,13,13,0,0,0,0,0,0,0,0,0,12,12,12,12,12,13,13,13,1,4,4,7,6,6,7,6,6,4,7,7,10,9,9,11,9,9,4,7,7,10,9,9,11,9,9,7,10,10,11,11,10,11,11,11,6,9,9,11,10,10,11,10,10,6,9,9,11,10,10,11,10,10,7,11,11,12,11,11,12,11,11,6,9,9,11,10,10,11,10,10,6,9,9,11,10,10,11,10,10,2,4,4,6,6,7,7,7,7,8,8,10,5,5,6,6,7,7,8,8,8,8,10,5,5,6,6,7,7,8,8,8,8,10,6,6,7,7,8,8,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,8,7,8,8,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,10,10,10,10,10,9,9,8,8,9,8,10,10,10,10,10,8,8,8,8,9,9,1,4,4,6,6,7,7,8,8,9,9,10,10,6,5,5,7,7,8,8,8,8,9,10,11,11,7,5,5,7,7,8,8,9,9,10,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,12,12,9,9,9,9,10,10,10,10,11,11,0,13,13,9,9,10,9,10,10,11,11,11,12,0,0,0,10,10,10,10,10,10,11,11,12,12,0,0,0,10,10,10,10,10,10,11,11,12,12,0,0,0,14,14,11,11,11,11,12,12,12,12,0,0,0,14,14,11,11,11,11,12,12,12,13,0,0,0,0,0,12,12,12,12,12,12,13,13,0,0,0,0,0,13,12,12,12,12,12,13,13,2,4,4,5,5,6,5,5,5,5,6,5,4,5,5,6,5,5,5,5,6,6,6,5,5,1,3,3,12,12,12,12,12,12,12,12,12,12,4,7,7,12,12,12,12,12,12,12,12,12,12,3,8,8,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,1,4,4,5,5,7,7,9,8,10,9,10,10,10,10,6,5,5,7,7,9,8,10,9,11,10,12,12,13,13,6,5,5,7,7,9,9,10,10,11,11,12,12,12,13,19,8,8,8,8,9,9,10,10,12,11,12,12,13,13,19,8,8,8,8,9,9,11,11,12,12,13,13,13,13,19,12,12,9,9,11,11,11,11,12,11,13,12,13,13,18,12,12,9,9,11,10,11,11,12,12,12,13,13,14,19,18,18,11,11,11,11,12,12,13,12,13,13,14,14,16,18,18,11,11,11,10,12,11,13,13,13,13,13,14,17,18,18,14,15,11,12,12,13,13,13,13,14,14,14,18,18,18,15,15,12,10,13,10,13,13,13,13,13,14,18,17,18,17,18,12,13,12,13,13,13,14,14,16,14,18,17,18,18,17,13,12,13,10,12,12,14,14,14,14,17,18,18,18,18,14,15,12,12,13,12,14,14,15,15,18,18,18,17,18,15,14,12,11,12,12,14,14,14,15,2,5,5,6,6,7,7,7,7,8,8,8,8,8,8,8,8,9,9,9,9,11,6,6,7,7,8,8,8,8,9,9,9,9,9,9,9,9,10,10,10,10,11,6,6,7,7,8,8,8,8,9,9,9,9,9,9,10,9,10,10,10,10,11,7,7,7,7,8,8,9,9,9,9,9,9,9,10,10,10,10,10,10,10,12,11,11,7,7,8,8,9,9,9,9,9,9,10,10,10,10,10,10,10,10,12,11,12,8,8,8,8,9,9,9,9,9,10,10,10,10,10,10,10,10,10,11,11,11,8,8,8,8,9,9,9,9,10,10,10,10,10,10,10,10,10,10,11,11,12,9,9,9,9,9,9,10,9,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,11,12,11,11,11,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,12,12,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,12,11,12,11,11,11,9,10,10,10,10,10,10,10,10,10,10,10,10,10,11,12,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,12,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,12,11,11,12,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,12,12,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,12,12,12,11,11,11,12,11,11,11,10,10,10,10,10,10,10,10,10,10,10,12,11,12,12,12,12,12,11,12,11,11,10,10,10,10,10,10,10,10,10,10,12,12,12,12,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,4,7,11,11,11,11,10,11,12,11,5,2,11,5,6,6,7,9,11,12,11,9,6,10,6,7,8,9,10,11,11,5,11,7,8,8,9,11,13,14,11,6,5,8,4,5,7,8,10,11,10,6,7,7,5,5,6,8,9,11,10,7,8,9,6,6,6,7,8,9,11,9,9,11,7,7,6,6,7,9,12,12,10,13,9,8,7,7,7,8,11,13,11,14,11,10,9,8,7,7,5,8,10,14,11,11,12,16,15,17,5,5,7,9,7,8,10,13,17,17,7,5,5,10,5,7,8,11,13,15,10,8,10,8,8,8,11,15,18,18,8,5,5,8,3,4,6,10,14,16,9,7,6,7,4,3,5,9,14,18,10,9,8,10,6,5,6,9,14,18,12,12,11,12,8,7,8,11,14,18,14,13,12,10,7,5,6,9,14,18,14,14,13,10,6,5,6,8,11,16,2,4,4,0,0,0,0,0,0,4,7,7,0,0,0,0,0,0,4,6,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,7,7,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,7,8,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,7,7,0,0,0,0,0,0,7,9,8,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,7,7,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,10,11,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,8,10,9,0,0,0,0,0,0,9,10,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,7,7,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,11,10,0,0,0,0,0,0,8,9,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,9,9,0,0,0,0,0,0,9,10,10,0,0,0,0,0,0,9,11,10], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+310869);
/* memory initializer */ allocate([2,4,4,0,0,0,5,5,0,0,0,5,5,0,0,0,8,7,0,0,0,0,0,0,0,4,6,6,0,0,0,8,8,0,0,0,8,7,0,0,0,10,10,0,0,0,0,0,0,0,4,6,6,0,0,0,8,8,0,0,0,7,8,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,0,0,0,8,8,0,0,0,8,8,0,0,0,10,10,0,0,0,0,0,0,0,5,7,8,0,0,0,8,8,0,0,0,8,8,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,0,0,0,8,8,0,0,0,8,8,0,0,0,10,10,0,0,0,0,0,0,0,5,8,8,0,0,0,8,8,0,0,0,8,8,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,10,10,0,0,0,10,10,0,0,0,9,10,0,0,0,11,10,0,0,0,0,0,0,0,8,10,10,0,0,0,10,10,0,0,0,10,10,0,0,0,10,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,4,3,5,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,5,5,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,5,5,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,6,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,6,6,8,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,3,6,6,0,0,0,0,0,4,4,6,6,0,0,0,0,0,4,4,6,6,0,0,0,0,0,5,5,6,6,0,0,0,0,0,0,0,6,6,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,8,7,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,2,4,3,6,6,7,7,9,9,0,4,4,6,6,7,7,9,9,0,4,4,6,6,7,7,9,9,0,6,6,7,7,7,7,9,9,0,0,0,7,6,7,7,9,9,0,0,0,8,8,8,8,10,10,0,0,0,8,8,8,8,10,10,0,0,0,9,9,9,9,10,10,0,0,0,0,0,9,9,10,10,2,4,4,6,6,8,8,9,9,9,9,10,10,10,10,11,11,0,4,4,6,6,8,8,9,9,9,9,10,10,11,11,12,12,0,4,4,6,6,8,8,9,9,9,9,10,10,11,11,12,12,0,6,6,7,7,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,12,12,0,0,0,7,7,9,9,10,10,10,10,11,11,11,11,12,12,0,0,0,7,7,8,9,10,10,10,10,11,11,11,11,12,12,0,0,0,8,8,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,0,0,9,9,10,10,10,10,11,11,12,12,12,12,0,0,0,0,0,9,9,9,10,10,10,11,11,12,12,12,12,0,0,0,0,0,10,10,10,10,11,11,11,12,12,12,13,13,0,0,0,0,0,0,0,10,10,11,11,11,11,12,12,13,13,0,0,0,0,0,0,0,11,11,11,11,12,12,12,13,13,13,0,0,0,0,0,0,0,11,11,11,11,12,12,12,12,13,13,0,0,0,0,0,0,0,12,12,12,12,13,12,13,13,13,13,0,0,0,0,0,0,0,0,0,12,12,12,12,13,13,13,13,1,4,4,7,6,6,7,6,6,4,7,7,10,9,9,11,9,9,4,7,7,10,9,9,11,9,9,7,10,10,11,11,10,11,11,11,6,9,9,11,10,10,11,10,10,6,9,9,11,10,10,11,10,10,7,11,11,12,11,11,12,11,11,6,9,9,11,10,10,11,10,10,6,9,9,11,10,10,11,10,10,2,4,4,6,6,7,7,8,8,8,8,10,5,5,6,6,7,7,8,8,8,8,10,5,5,6,6,7,7,8,8,8,8,10,6,6,7,7,8,8,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,7,7,8,8,8,8,8,8,10,10,10,8,8,8,8,8,8,8,9,10,10,10,10,10,8,8,8,8,8,8,10,10,10,10,10,9,9,8,8,8,8,10,10,10,10,10,8,8,8,8,8,8,1,4,4,6,6,7,7,8,8,9,9,10,10,6,5,5,7,7,8,8,8,9,10,10,10,10,7,5,5,7,7,8,8,9,9,10,10,10,10,0,8,8,8,8,9,9,9,9,10,10,11,11,0,8,8,8,8,9,9,9,9,10,10,11,11,0,12,12,9,9,9,10,10,10,10,10,11,11,0,13,13,9,9,9,9,10,10,11,11,11,11,0,0,0,10,10,10,10,10,10,11,11,11,11,0,0,0,10,10,10,10,10,10,11,11,12,12,0,0,0,14,14,11,11,11,11,12,12,12,12,0,0,0,14,14,11,11,11,11,12,12,12,12,0,0,0,0,0,12,12,12,12,12,12,13,13,0,0,0,0,0,12,12,12,12,12,12,13,13,2,4,4,5,5,6,5,5,5,5,6,4,5,5,5,6,5,5,5,5,6,6,6,5,5,1,3,3,13,13,13,13,13,13,13,13,13,13,13,13,4,7,7,13,13,13,13,13,13,13,13,13,13,13,13,3,8,6,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,12,12,12,12,12,1,4,4,5,5,7,7,9,8,10,9,10,10,11,10,11,11,6,5,5,7,7,8,9,10,10,11,10,12,11,12,11,13,12,6,5,5,7,7,9,9,10,10,11,11,12,12,13,12,13,13,18,8,8,8,8,9,9,10,11,11,11,12,11,13,11,13,12,18,8,8,8,8,10,10,11,11,12,12,13,13,13,13,13,14,18,12,12,9,9,11,11,11,11,12,12,13,12,13,12,13,13,20,13,12,9,9,11,11,11,11,12,12,13,13,13,14,14,13,20,18,19,11,12,11,11,12,12,13,13,13,13,13,13,14,13,18,19,19,12,11,11,11,12,12,13,12,13,13,13,14,14,13,18,17,19,14,15,12,12,12,13,13,13,14,14,14,14,14,14,19,19,19,16,15,12,11,13,12,14,14,14,13,13,14,14,14,19,18,19,18,19,13,13,13,13,14,14,14,13,14,14,14,14,18,17,19,19,19,13,13,13,11,13,11,13,14,14,14,14,14,19,17,17,18,18,16,16,13,13,13,13,14,13,15,15,14,14,19,19,17,17,18,16,16,13,11,14,10,13,12,14,14,14,14,19,19,19,19,19,18,17,13,14,13,11,14,13,14,14,15,15,19,19,19,17,19,18,18,14,13,12,11,14,11,15,15,15,15,3,5,5,6,6,7,7,7,7,8,8,8,8,8,8,8,8,8,8,8,9,11,5,6,7,7,8,7,8,8,8,8,9,9,9,9,9,9,9,9,9,9,11,5,5,7,7,7,7,8,8,8,8,9,9,9,9,9,9,9,9,9,9,11,7,7,7,7,8,8,8,8,9,9,9,9,9,9,9,9,9,10,9,10,11,11,11,7,7,8,8,8,8,9,9,9,9,9,9,10,10,10,10,10,10,11,11,11,8,8,8,8,9,9,9,9,9,9,9,10,10,10,10,10,10,10,11,11,11,8,8,8,8,9,9,9,9,9,9,10,10,10,10,10,10,10,10,11,11,11,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,11,11,11,11,11,9,9,9,9,9,9,10,9,10,10,10,10,10,10,10,10,11,11,11,11,11,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,9,9,10,9,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,9,9,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,3,8,9,13,10,12,12,12,12,12,6,4,6,8,6,8,10,10,11,12,8,5,4,10,4,7,8,9,10,11,13,8,10,8,9,9,11,12,13,14,10,6,4,9,3,5,6,8,10,11,11,8,6,9,5,5,6,7,9,11,12,9,7,11,6,6,6,7,8,10,12,11,9,12,7,7,6,6,7,9,13,12,10,13,9,8,7,7,7,8,11,15,11,15,11,10,9,8,7,7,3,9,11,11,13,14,19,17,17,19,5,4,5,8,10,10,13,16,18,19,7,4,4,5,8,9,12,14,17,19,8,6,5,5,7,7,10,13,16,18,10,8,7,6,5,5,8,11,17,19,11,9,7,7,5,4,5,8,17,19,13,11,8,7,7,5,5,7,16,18,14,13,8,6,6,5,5,7,16,18,18,16,10,8,8,7,7,9,16,18,18,18,12,10,10,9,9,10,17,18,1,5,5,0,5,5,0,5,5,5,8,7,0,9,9,0,9,8,5,7,8,0,9,9,0,8,9,0,0,0,0,0,0,0,0,0,5,9,8,0,8,8,0,8,8,5,8,9,0,8,8,0,8,8,0,0,0,0,0,0,0,0,0,5,9,9,0,8,8,0,8,8,5,9,9,0,8,8,0,8,8,3,5,5,8,8,0,5,5,8,8,0,5,5,8,8,0,7,7,9,9,0,0,0,9,9,5,7,7,9,9,0,8,8,10,10,0,8,7,10,9,0,10,10,11,11,0,0,0,11,11,5,7,7,9,9,0,8,8,10,10,0,7,8,9,10,0,10,10,11,11,0,0,0,11,11,8,9,9,11,11,0,11,11,12,12,0,11,10,12,12,0,13,14,14,14,0,0,0,14,13,8,9,9,11,11,0,11,11,12,12,0,10,11,12,12,0,14,13,14,14,0,0,0,13,14,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,11,10,0,7,7,10,10,0,7,7,10,10,0,9,9,11,10,0,0,0,11,11,5,7,8,10,11,0,7,7,10,10,0,7,7,10,10,0,9,9,10,11,0,0,0,11,11,8,10,9,12,12,0,10,10,12,12,0,10,10,12,12,0,12,12,13,13,0,0,0,13,13,8,9,10,12,12,0,10,10,11,12,0,10,10,12,12,0,12,12,13,13,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,11,11,0,7,7,10,10,0,7,7,10,10,0,9,9,10,11,0,0,0,11,10,5,8,8,11,11,0,7,7,10,10,0,7,7,10,10,0,9,9,11,11,0,0,0,10,11,8,10,10,12,12,0,10,10,12,12,0,10,10,12,12,0,12,13,13,13,0,0,0,14,13,8,10,10,12,12,0,10,10,12,12,0,10,10,12,12,0,13,12,13,13,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,10,10,14,13,0,9,9,13,12,0,9,9,12,12,0,10,10,12,12,0,0,0,12,12,7,10,10,13,14,0,9,9,12,13,0,9,9,12,12,0,10,10,12,12,0,0,0,12,12,9,11,11,14,13,0,11,10,14,13,0,11,11,13,13,0,12,12,13,13,0,0,0,13,13,9,11,11,13,14,0,10,11,13,14,0,11,11,13,13,0,12,12,13,13,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,11,11,14,14,0,11,11,13,13,0,11,10,13,13,0,12,12,13,13,0,0,0,13,13,9,11,11,14,14,0,11,11,13,13,0,10,11,13,13,0,12,12,14,13,0,0,0,13,13,2,3,4,6,6,7,7,9,9,0,4,4,6,6,7,7,9,10,0,4,4,6,6,7,7,10,9,0,5,5,7,7,8,8,10,10,0,0,0,7,6,8,8,10,10,0,0,0,7,7,9,9,11,11,0,0,0,7,7,9,9,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,4,4,6,6,7,7,8,8,8,8,9,9,9,10,10,10,0,4,4,6,6,8,8,9,9,9,9,10,10,10,10,11,11,0,4,4,6,6,8,8,9,9,9,9,10,10,10,10,11,11,0,6,6,7,7,8,8,9,9,9,9,10,10,11,11,11,11,0,0,0,7,7,8,8,9,9,9,9,10,10,11,11,11,11,0,0,0,7,7,9,9,10,10,10,10,11,11,11,11,12,12,0,0,0,7,7,9,9,10,10,10,10,11,11,11,11,12,12,0,0,0,7,7,8,8,9,9,10,10,11,11,12,12,12,12,0,0,0,0,0,8,8,9,9,10,10,11,11,12,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,4,4,5,7,7,6,7,7,4,6,6,9,9,10,10,10,9,4,6,6,9,10,9,10,9,10,6,9,9,10,12,11,10,11,11,7,10,9,11,12,12,12,12,12,7,10,10,11,12,12,12,12,12,6,10,10,10,12,12,11,12,12,7,9,10,11,12,12,12,12,12,7,10,9,12,12,12,12,12,12,3,5,4,6,6,7,7,8,8,8,8,11,4,4,6,6,7,7,8,8,8,8,11,4,4,6,6,7,7,8,8,8,8,11,6,6,6,6,8,8,8,8,9,9,11,11,11,6,6,7,8,8,8,8,9,11,11,11,7,7,8,8,8,8,8,8,11,11,11,7,7,8,8,8,8,8,8,11,11,11,8,8,8,8,8,8,8,8,11,11,11,10,10,8,8,8,8,8,8,11,11,11,10,10,8,8,8,8,8,8,11,11,11,10,10,7,7,8,8,8,8,1,4,4,6,6,8,8,8,8,10,9,10,10,6,5,5,7,7,9,9,9,9,10,10,11,11,6,5,5,7,7,9,9,10,9,11,10,11,11,0,6,6,7,7,9,9,10,10,11,11,12,12,0,7,7,7,7,9,9,10,10,11,11,12,12,0,11,11,8,8,10,10,11,11,12,12,12,12,0,11,12,9,8,10,10,11,11,12,12,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,4,4,5,5,5,4,4,5,5,5,4,4,5,5,6,5,5,5,5,6,6,6,5,5,1,4,4,6,6,8,8,8,8,10,10,11,10,6,5,5,7,7,8,8,9,9,10,10,12,11,6,5,5,7,7,8,8,9,9,10,10,12,11,21,7,7,7,7,9,9,10,10,11,11,12,12,21,7,7,7,7,9,9,10,10,11,11,12,12,21,12,12,9,9,10,10,11,11,11,11,12,12,21,12,12,9,9,10,10,11,11,12,12,12,12,21,21,21,11,11,10,10,11,12,12,12,13,13,21,21,21,11,11,10,10,12,12,12,12,13,13,21,21,21,15,15,11,11,12,12,13,13,13,13,21,21,21,15,16,11,11,12,12,13,13,14,14,21,21,21,21,20,13,13,13,13,13,13,14,14,20,20,20,20,20,13,13,13,13,13,13,14,14,3,5,5,6,6,7,7,7,7,7,7,9,5,5,6,6,7,7,7,7,8,7,8,5,5,6,6,7,7,7,7,7,7,9,6,6,7,7,7,7,8,7,7,8,9,9,9,7,7,7,7,7,7,7,8,9,9,9,7,7,7,7,8,8,8,8,9,9,9,7,7,7,7,7,7,8,8,9,9,9,8,8,8,8,7,7,8,8,9,9,9,9,8,8,8,7,7,8,8,9,9,9,8,8,8,8,7,7,8,8,9,9,9,8,8,7,7,7,7,8,8,1,4,4,7,7,8,8,7,7,8,7,9,8,10,9,6,5,5,8,8,9,9,8,8,9,9,11,10,11,10,6,5,5,8,8,9,9,8,8,9,9,10,10,11,11,18,8,8,9,8,10,10,9,9,10,10,10,10,11,10,18,8,8,9,9,10,10,9,9,10,10,11,11,12,12,18,12,13,9,10,10,10,9,10,10,10,11,11,12,11,18,13,13,9,9,10,10,10,10,10,10,11,11,12,12,18,18,18,10,10,9,9,11,11,11,11,11,12,12,12,18,18,18,10,9,10,9,11,10,11,11,11,11,13,12,18,18,18,14,13,10,10,11,11,12,12,12,12,12,12,18,18,18,14,13,10,10,11,10,12,12,12,12,12,12,18,18,18,18,18,12,12,11,11,12,12,13,13,13,14,18,18,18,18,18,12,12,11,11,12,11,13,13,14,13,18,18,18,18,18,16,16,11,12,12,13,13,13,14,13,18,18,18,18,18,16,15,12,11,12,11,13,11,15,14,3,5,5,6,6,7,7,7,7,8,7,8,8,8,8,8,8,8,8,8,8,10,6,6,7,7,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,10,6,6,7,7,8,8,8,8,8,8,9,8,9,9,9,9,9,9,9,9,10,7,7,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,10,11,11,8,7,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,11,11,11,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,11,11,11,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,11,11,11,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,11,11,11,11,11,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,9,11,11,11,11,11,9,9,9,9,9,9,10,9,9,10,9,10,9,9,10,9,11,11,11,11,11,9,9,9,9,9,9,9,10,10,10,10,9,10,10,9,10,11,11,11,11,11,9,9,9,9,10,10,10,9,10,10,10,10,9,10,10,9,11,11,11,11,11,11,11,9,9,9,9,10,10,10,10,9,10,10,10,10,10,11,11,11,11,11,11,11,10,9,10,10,10,10,10,10,10,9,10,9,10,10,11,11,11,11,11,11,11,10,9,10,9,10,10,9,10,10,10,10,10,10,10,11,11,11,11,11,11,11,10,10,10,10,10,10,10,9,10,10,10,10,10,9,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,9,10,10,11,11,11,11,11,11,11,11,11,10,10,10,9,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,10,11,9,10,10,10,10,10,10,10,10,10,1,3,3,11,11,11,11,11,11,11,11,11,11,4,7,7,11,11,11,11,11,11,11,11,11,11,5,8,9,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,1,4,4,7,7,7,7,7,6,8,8,8,8,6,6,6,8,8,8,8,8,7,9,8,10,10,5,6,6,8,8,9,9,8,8,10,10,10,10,16,9,9,9,9,9,9,9,8,10,9,11,11,16,8,9,9,9,9,9,9,9,10,10,11,11,16,13,13,9,9,10,9,9,10,11,11,11,12,16,13,14,9,8,10,8,9,9,10,10,12,11,16,14,16,9,9,9,9,11,11,12,11,12,11,16,16,16,9,7,9,6,11,11,11,10,11,11,16,16,16,11,12,9,10,11,11,12,11,13,13,16,16,16,12,11,10,7,12,10,12,12,12,12,16,16,15,16,16,10,11,10,11,13,13,14,12,16,16,16,15,15,12,10,11,11,13,11,12,13,2,4,3,4,5,5,5,6,6,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,3,8,11,13,14,14,13,13,16,14,6,3,4,7,9,9,10,11,14,13,10,4,3,5,7,7,9,10,13,15,12,7,4,4,6,6,8,10,13,15,12,8,6,6,6,6,8,10,13,14,11,9,7,6,6,6,7,8,12,11,13,10,9,8,7,6,6,7,11,11,13,11,10,9,9,7,7,6,10,11,13,13,13,13,13,11,9,8,10,12,12,15,15,16,15,12,11,10,10,12,4,11,12,14,15,15,17,17,18,18,5,6,6,8,9,10,13,17,18,19,7,5,4,6,8,9,11,15,19,19,8,6,5,5,6,7,11,14,16,17,9,7,7,6,7,7,10,13,15,19,10,8,7,6,7,6,7,9,14,16,12,10,9,7,7,6,4,5,10,15,14,13,11,7,6,6,4,2,7,13,16,16,15,9,8,8,8,6,9,13,19,19,17,12,11,10,10,9,11,14,1,5,5,0,5,5,0,5,5,5,8,7,0,9,9,0,9,8,5,7,8,0,9,9,0,8,9,0,0,0,0,0,0,0,0,0,5,9,9,0,8,8,0,8,8,5,8,9,0,8,8,0,8,8,0,0,0,0,0,0,0,0,0,5,9,9,0,8,8,0,8,8,5,8,9,0,8,8,0,8,8,3,5,5,8,8,0,5,5,8,8,0,5,5,8,8,0,7,7,9,9,0,0,0,9,9,5,7,7,9,9,0,8,8,10,10,0,8,7,10,9,0,10,10,11,11,0,0,0,11,11,5,7,7,9,9,0,8,8,10,10,0,7,8,9,10,0,10,10,11,11,0,0,0,11,11,8,9,9,11,10,0,11,11,12,12,0,11,10,12,12,0,13,14,14,14,0,0,0,14,13,8,9,9,10,11,0,11,11,12,12,0,10,11,12,12,0,13,13,14,14,0,0,0,13,14,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,11,10,0,7,7,10,10,0,7,7,10,10,0,9,9,11,10,0,0,0,11,11,5,7,8,10,11,0,7,7,10,10,0,7,7,10,10,0,9,9,10,11,0,0,0,11,11,8,10,9,12,12,0,10,10,12,12,0,10,10,12,12,0,12,12,13,13,0,0,0,13,13,8,9,10,12,12,0,10,10,12,12,0,10,10,11,12,0,12,12,13,13,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,8,11,11,0,7,7,10,10,0,7,7,10,10,0,9,9,10,11,0,0,0,11,10,5,8,8,10,11,0,7,7,10,10,0,7,7,10,10,0,9,9,11,10,0,0,0,10,11,9,10,10,12,12,0,10,10,12,12,0,10,10,12,12,0,12,13,13,13,0,0,0,13,12,9,10,10,12,12,0,10,10,12,12,0,10,10,12,12,0,13,12,13,13,0,0,0,12,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,10,10,14,13,0,9,9,12,12,0,9,9,12,12,0,10,10,12,12,0,0,0,12,12,7,10,10,13,14,0,9,9,12,13,0,9,9,12,12,0,10,10,12,12,0,0,0,12,12,9,11,11,14,13,0,11,10,13,12,0,11,11,13,13,0,12,12,13,13,0,0,0,13,13,9,11,11,13,14,0,10,11,12,13,0,11,11,13,13,0,12,12,13,13,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,11,11,14,14,0,10,11,13,13,0,11,10,13,13,0,12,12,13,13,0,0,0,13,12,9,11,11,14,14,0,11,10,13,13,0,10,11,13,13,0,12,12,14,13,0,0,0,13,13,2,4,4,5,5,7,7,9,9,0,4,4,6,6,7,7,9,9,0,4,4,6,6,7,7,9,9,0,5,5,6,6,8,8,10,10,0,0,0,6,6,8,8,10,10,0,0,0,7,7,9,9,10,10,0,0,0,7,7,8,8,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,4,4,5,5,7,7,8,8,8,8,9,9,10,10,11,11,0,4,4,6,6,7,7,8,8,9,9,10,10,11,11,12,12,0,4,4,6,6,7,7,8,8,9,9,10,10,11,11,12,12,0,5,5,6,6,8,8,9,9,9,9,10,10,11,12,12,12,0,0,0,6,6,8,7,9,9,9,9,10,10,11,11,12,12,0,0,0,7,7,8,8,9,9,10,10,11,11,12,12,13,12,0,0,0,7,7,8,8,9,9,10,10,11,11,12,12,12,13,0,0,0,7,7,8,8,9,9,10,10,11,11,12,12,13,13,0,0,0,0,0,8,8,9,9,10,10,11,11,12,12,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,4,4,5,7,7,6,7,7,4,6,7,10,10,10,10,10,9,4,6,6,10,10,10,10,9,10,5,10,10,9,11,12,10,11,12,7,10,10,11,12,12,12,12,12,7,10,10,11,12,12,12,12,12,6,10,10,10,12,12,11,12,12,7,10,10,12,12,12,12,11,12,7,10,10,11,12,12,12,12,12,3,5,5,6,6,7,7,8,8,8,8,11,4,4,6,6,7,7,8,8,9,9,11,4,4,6,6,7,7,8,8,9,9,12,5,5,6,6,7,7,9,9,9,9,12,12,12,6,6,7,7,9,9,9,9,11,11,11,7,7,7,7,8,8,9,9,11,11,11,7,7,7,7,8,8,9,9,11,11,11,7,7,8,8,8,8,9,9,11,11,11,11,11,8,8,8,8,8,9,11,11,11,11,11,8,8,8,8,8,8,11,11,11,11,11,7,7,8,8,8,8,1,4,4,6,6,7,7,8,7,9,8,10,10,6,5,5,7,7,8,8,9,9,9,10,11,11,7,5,5,7,7,8,8,9,9,10,10,11,11,0,7,7,7,7,9,8,9,9,10,10,11,11,0,8,8,7,7,8,9,9,9,10,10,11,11,0,11,11,9,9,10,10,11,10,11,11,12,12,0,12,12,9,9,10,10,11,11,11,11,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,4,4,5,5,5,4,4,5,5,5,4,4,5,5,6,5,5,5,5,6,6,6,5,5,1,4,4,6,6,7,8,9,9,10,10,12,11,6,5,5,7,7,8,8,9,10,11,11,12,12,7,5,5,7,7,8,8,10,10,11,11,12,12,20,7,7,7,7,8,9,10,10,11,11,12,13,20,7,7,7,7,9,9,10,10,11,12,13,13,20,11,11,8,8,9,9,11,11,12,12,13,13,20,11,11,8,8,9,9,11,11,12,12,13,13,20,20,20,10,10,10,10,12,12,13,13,13,13,20,20,20,10,10,10,10,12,12,13,13,13,14,20,20,20,14,14,11,11,12,12,13,13,14,14,20,20,20,14,14,11,11,12,12,13,13,14,14,20,20,20,20,19,13,13,13,13,14,14,15,14,19,19,19,19,19,13,13,13,13,14,14,15,15,4,5,5,6,6,7,7,7,7,7,7,8,6,6,7,7,7,7,7,7,7,7,8,6,6,6,7,7,7,7,7,7,7,8,6,6,7,7,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,8,8,7,7,7,7,7,7,8,8,8,8,8,7,7,7,7,7,7,8,8,8,8,8,7,7,7,7,7,7,1,4,4,7,7,8,8,8,7,9,8,9,9,10,10,6,5,5,7,7,9,9,8,8,10,9,11,10,12,11,6,5,5,8,7,9,9,8,8,10,10,11,11,12,11,19,8,8,8,8,10,10,9,9,10,10,11,11,12,11,19,8,8,8,8,10,10,9,9,10,10,11,11,12,12,19,12,12,9,9,10,10,9,10,10,10,11,11,12,12,19,12,12,9,9,10,10,10,10,10,10,12,12,12,12,19,19,19,9,9,9,9,11,10,11,11,12,11,13,13,19,19,19,9,9,9,9,11,10,11,11,11,12,13,13,19,19,19,13,13,10,10,11,11,12,12,12,12,13,12,19,19,19,14,13,10,10,11,11,12,12,12,13,13,13,19,19,19,19,19,12,12,12,11,12,13,14,13,13,13,19,19,19,19,19,12,12,12,11,12,12,13,14,13,14,19,19,19,19,19,16,16,12,13,12,13,13,14,15,14,19,18,18,18,18,16,15,12,11,12,11,14,12,14,14,3,5,5,7,6,7,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,10,6,6,7,7,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,10,6,6,7,7,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,10,7,7,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,11,10,10,10,9,9,9,9,9,9,9,9,9,9,10,9,9,10,9,9,10,11,10,11,10,9,9,9,9,9,9,9,10,10,10,9,10,9,9,9,9,11,10,11,10,10,9,9,9,9,9,9,10,9,9,10,9,9,10,9,9,10,11,10,10,11,10,9,9,9,9,9,10,10,9,10,10,10,10,9,10,10,10,10,10,10,11,11,11,10,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,10,10,10,10,10,10,10,10,10,10,10,10,9,10,10,9,10,11,11,10,11,10,11,10,9,10,10,9,10,10,10,10,10,10,10,10,10,10,11,11,11,11,10,11,11,10,10,10,10,10,10,9,10,9,10,10,9,10,9,10,10,10,11,10,11,10,11,11,10,10,10,10,10,10,9,10,10,10,10,10,10,10,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,10,11,11,10,10,10,10,9,9,10,10,9,9,10,9,10,10,10,10,11,11,10,10,10,10,10,10,10,9,9,10,10,10,9,9,10,10,10,10,10,11,10,11,10,10,10,10,10,10,9,10,10,10,10,10,10,10,10,10,1,3,3,11,11,11,11,11,11,11,11,11,11,4,6,6,11,11,11,11,11,11,11,11,11,11,4,7,7,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,1,4,4,7,7,7,7,7,6,8,8,8,8,6,6,6,8,8,9,8,8,7,9,8,11,10,5,6,6,8,8,9,8,8,8,10,9,11,11,16,8,8,9,8,9,9,9,8,10,9,11,10,16,8,8,9,9,10,10,9,9,10,10,11,11,16,13,13,9,9,10,10,9,10,11,11,12,11,16,13,13,9,8,10,9,10,10,10,10,11,11,16,14,16,8,9,9,9,11,10,11,11,12,11,16,16,16,9,7,10,7,11,10,11,11,12,11,16,16,16,12,12,9,10,11,11,12,11,12,12,16,16,16,12,10,10,7,11,8,12,11,12,12,16,16,15,16,16,11,12,10,10,12,11,12,12,16,16,16,15,15,11,11,10,10,12,12,12,12,2,4,3,4,5,5,5,6,6,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,3,8,11,13,15,14,14,13,15,14,6,4,5,7,9,10,11,11,14,13,10,4,3,5,7,8,9,10,13,13,12,7,4,4,5,6,8,9,12,14,13,9,6,5,5,6,8,9,12,14,12,9,7,6,5,5,6,8,11,11,12,11,9,8,7,6,6,7,10,11,13,11,10,9,8,7,6,6,9,11,13,13,12,12,12,10,9,8,9,11,12,14,15,15,14,12,11,10,10,12,4,11,13,14,15,15,18,17,19,17,5,6,8,9,10,10,12,15,19,19,6,6,6,6,8,8,11,14,18,19,8,6,5,4,6,7,10,13,16,17,9,7,6,5,6,7,9,12,15,19,10,8,7,6,6,6,7,9,13,15,12,10,9,8,7,6,4,5,10,15,13,13,11,8,6,6,4,2,7,12,17,15,16,10,8,8,7,6,9,12,19,18,17,13,11,10,10,9,11,14,1,5,5,0,5,5,0,5,5,5,7,7,0,9,8,0,9,8,6,7,7,0,8,9,0,8,9,0,0,0,0,0,0,0,0,0,5,9,8,0,8,8,0,8,8,5,8,9,0,8,8,0,8,8,0,0,0,0,0,0,0,0,0,5,9,8,0,8,8,0,8,8,5,8,9,0,8,8,0,8,8,3,5,5,8,8,0,5,5,8,8,0,5,5,8,8,0,7,7,9,9,0,0,0,9,9,5,7,7,9,9,0,8,7,10,9,0,8,7,10,9,0,10,10,11,11,0,0,0,11,11,5,7,7,9,9,0,7,8,9,10,0,7,8,9,10,0,10,10,11,11,0,0,0,11,11,8,9,9,11,10,0,11,10,12,11,0,11,10,12,12,0,13,13,14,14,0,0,0,14,13,8,9,9,10,11,0,10,11,12,12,0,10,11,12,12,0,13,13,14,14,0,0,0,13,14,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,11,10,0,7,7,10,10,0,7,7,10,10,0,9,9,10,10,0,0,0,11,10,5,7,8,10,11,0,7,7,10,10,0,7,7,10,10,0,9,9,10,10,0,0,0,10,10,8,10,9,12,12,0,10,10,12,11,0,10,10,12,12,0,12,12,13,12,0,0,0,13,12,8,9,10,12,12,0,10,10,11,12,0,10,10,11,12,0,12,12,13,13,0,0,0,12,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,7,11,10,0,7,7,10,10,0,7,7,10,10,0,9,9,10,11,0,0,0,10,10,6,7,8,10,11,0,7,7,10,10,0,7,7,10,10,0,9,9,10,10,0,0,0,10,10,9,10,9,12,12,0,10,10,12,12,0,10,10,12,11,0,12,12,13,13,0,0,0,13,12,8,9,10,12,12,0,10,10,12,12,0,10,10,11,12,0,12,12,13,13,0,0,0,12,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,10,10,13,13,0,9,9,12,12,0,9,9,12,12,0,10,10,12,12,0,0,0,12,12,7,10,10,13,13,0,9,9,12,12,0,9,9,12,12,0,10,10,12,12,0,0,0,12,12,9,11,11,14,13,0,10,10,13,12,0,11,10,13,12,0,12,12,13,12,0,0,0,13,13,9,11,11,13,14,0,10,11,12,13,0,10,11,13,13,0,12,12,12,13,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,11,11,14,14,0,10,11,13,13,0,11,10,13,13,0,11,12,13,13,0,0,0,13,12,9,11,11,14,14,0,11,10,13,13,0,10,11,13,13,0,12,12,13,13,0,0,0,12,13,2,4,4,5,5,7,7,9,9,0,4,4,6,6,7,7,9,9,0,4,4,6,6,7,7,9,9,0,5,5,6,6,8,8,10,10,0,0,0,6,6,8,8,10,10,0,0,0,7,7,9,9,10,10,0,0,0,7,7,8,8,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,4,4,5,5,7,7,8,8,8,8,9,9,10,10,11,11,0,4,4,6,6,7,7,8,8,9,8,10,10,11,11,11,11,0,4,4,6,6,7,7,8,8,9,9,10,10,11,11,11,11,0,6,5,6,6,7,7,9,9,9,9,10,10,11,11,12,12,0,0,0,6,6,7,7,9,9,9,9,10,10,11,11,12,12,0,0,0,7,7,8,8,9,9,10,10,11,11,11,12,12,12,0,0,0,7,7,8,8,9,9,10,10,11,11,11,12,12,12,0,0,0,7,7,8,8,9,9,10,10,11,11,12,12,13,13,0,0,0,0,0,8,8,9,9,10,10,11,11,12,12,13,13], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+320562);
/* memory initializer */ allocate([1,4,4,5,7,7,6,7,7,4,7,6,10,10,10,10,10,10,4,6,6,10,10,10,10,9,10,5,10,10,9,11,11,10,11,11,7,10,10,11,12,12,12,12,12,7,10,10,11,12,12,12,12,12,6,10,10,10,12,12,10,12,12,7,10,10,11,12,12,12,12,12,7,10,10,11,12,12,12,12,12,3,5,5,6,6,7,7,8,8,8,8,11,4,5,6,6,7,7,8,8,8,8,11,5,5,6,6,7,7,8,8,8,9,12,5,5,6,6,7,7,8,8,9,9,12,12,12,6,6,7,7,8,8,9,9,11,11,11,6,6,7,7,8,8,8,8,11,11,11,6,6,7,7,8,8,8,8,11,11,11,7,7,7,8,8,8,8,8,11,11,11,11,11,7,7,8,8,8,8,11,11,11,11,11,7,7,7,7,8,8,11,11,11,11,11,7,7,7,7,8,8,1,4,4,6,6,7,7,8,8,9,9,10,10,6,5,5,7,7,8,8,9,9,10,10,11,11,6,5,5,7,7,8,8,9,9,10,10,11,11,0,7,7,7,7,9,9,10,10,10,10,11,11,0,7,7,7,7,9,9,10,10,10,10,11,11,0,11,11,9,9,10,10,11,11,11,11,12,12,0,12,12,9,9,10,10,11,11,12,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,4,4,5,5,5,4,4,5,5,5,4,4,5,5,6,5,5,5,5,6,6,6,5,5,1,4,4,6,6,8,7,9,9,10,10,12,12,6,5,5,7,7,8,8,10,10,11,11,12,12,7,5,5,7,7,8,8,10,10,11,11,12,12,21,7,7,7,7,8,9,10,10,11,11,12,12,21,7,7,7,7,9,9,10,10,12,12,13,13,21,11,11,8,8,9,9,11,11,12,12,13,13,21,11,11,8,8,9,9,11,11,12,12,13,13,21,21,21,10,10,10,10,11,11,12,13,13,13,21,21,21,10,10,10,10,11,11,13,13,14,13,21,21,21,13,13,11,11,12,12,13,13,14,14,21,21,21,14,14,11,11,12,12,13,13,14,14,21,21,21,21,20,13,13,13,12,14,14,16,15,20,20,20,20,20,13,13,13,13,14,13,15,15,4,5,6,6,6,7,7,7,7,7,7,8,6,6,6,7,7,7,7,7,7,7,8,6,6,6,6,7,7,7,7,7,7,8,6,6,7,7,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,8,8,7,7,7,7,7,7,8,8,8,8,8,7,7,7,7,7,7,8,8,8,8,8,7,7,7,7,7,7,1,4,4,7,6,8,8,8,7,9,8,10,10,11,10,6,5,5,7,7,9,9,8,8,10,10,11,11,12,11,6,5,5,7,7,9,9,9,9,10,10,11,11,12,12,20,8,8,8,8,9,9,9,9,10,10,11,11,12,12,20,8,8,8,8,10,9,9,9,10,10,11,11,12,12,20,12,12,9,9,10,10,10,10,10,11,12,12,12,12,20,12,12,9,9,10,10,10,10,11,11,12,12,13,13,20,20,20,9,9,9,9,11,10,11,11,12,12,12,13,20,19,19,9,9,9,9,11,11,11,12,12,12,13,13,19,19,19,13,13,10,10,11,11,12,12,13,13,13,13,19,19,19,14,13,11,10,11,11,12,12,12,13,13,13,19,19,19,19,19,12,12,12,12,13,13,13,13,14,13,19,19,19,19,19,12,12,12,11,12,12,13,14,14,14,19,19,19,19,19,16,15,13,12,13,13,13,14,14,14,19,19,19,19,19,17,17,13,12,13,11,14,13,15,15,4,5,5,7,7,7,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,10,6,6,7,7,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,10,6,6,7,7,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,10,7,7,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,9,9,10,9,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,9,9,10,10,10,9,9,9,9,9,10,10,10,10,10,10,10,9,9,9,9,9,10,10,10,9,9,9,9,9,9,10,10,10,10,10,10,10,9,10,10,9,10,10,10,10,9,10,9,10,10,9,10,10,10,10,10,10,10,9,10,10,10,10,10,10,9,9,10,10,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,9,9,9,10,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,9,9,10,9,10,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,9,9,10,9,9,9,10,10,10,10,10,10,10,10,10,10,10,9,9,9,9,9,9,10,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,9,10,9,9,10,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,9,9,10,10,9,10,9,9,1,4,3,11,11,11,11,11,11,11,11,11,11,11,11,11,11,4,7,7,11,11,11,11,11,11,11,11,11,11,11,11,11,11,4,8,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,1,4,4,7,6,7,7,7,7,8,8,9,9,10,10,10,10,11,11,6,6,6,8,8,9,8,8,7,10,8,11,10,12,11,12,12,13,13,5,5,6,8,8,9,9,8,8,10,9,11,11,12,12,13,13,13,13,17,8,8,9,9,9,9,9,9,10,9,12,10,12,12,13,12,13,13,17,9,8,9,9,9,9,9,9,10,10,12,12,12,12,13,13,13,13,17,13,13,9,9,10,10,10,10,11,11,12,11,13,12,13,13,14,15,17,13,13,9,8,10,9,10,10,11,11,12,12,14,13,15,13,14,15,17,17,17,9,10,9,10,11,11,12,12,12,12,13,13,14,14,15,15,17,17,17,9,8,9,8,11,11,12,12,12,12,14,13,14,14,14,15,17,17,17,12,14,9,10,11,11,12,12,14,13,13,14,15,13,15,15,17,17,17,13,11,10,8,11,9,13,12,13,13,13,13,13,14,14,14,17,17,17,17,17,11,12,11,11,13,13,14,13,15,14,13,15,16,15,17,17,17,17,17,11,11,12,8,13,12,14,13,17,14,15,14,15,14,17,17,17,17,17,15,15,12,12,12,12,13,14,14,14,15,14,17,14,17,17,17,17,17,16,17,12,12,13,12,13,13,14,14,14,14,14,14,17,17,17,17,17,17,17,14,14,13,12,13,13,15,15,14,13,15,17,17,17,17,17,17,17,17,13,14,13,13,13,13,14,15,15,15,14,15,17,17,17,17,17,17,17,16,15,13,14,13,13,14,14,15,14,14,16,17,17,17,17,17,17,17,16,16,13,14,13,13,14,14,15,14,15,14,2,4,4,4,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,3,8,12,13,14,14,14,13,14,14,6,4,5,8,10,10,11,11,14,13,9,5,4,5,7,8,9,10,13,13,12,7,5,4,5,6,8,9,12,13,13,9,6,5,5,5,7,9,11,14,12,10,7,6,5,4,6,7,10,11,12,11,9,8,7,5,5,6,10,10,13,12,10,9,8,6,6,5,8,10,14,13,12,12,11,10,9,7,8,10,12,13,14,14,13,12,11,9,9,10,5,13,18,16,17,17,19,18,19,19,5,7,10,11,12,12,13,16,17,18,6,6,7,7,9,9,10,14,17,19,8,7,6,5,6,7,9,12,19,17,8,7,7,6,5,6,8,11,15,19,9,8,7,6,5,5,6,8,13,15,11,10,8,8,7,5,4,4,10,14,12,13,11,9,7,6,4,2,6,12,18,16,16,13,8,7,7,5,8,13,16,17,18,15,11,9,9,8,10,13,1,5,5,0,5,5,0,5,5,6,8,8,0,9,8,0,9,8,6,8,8,0,8,9,0,8,9,0,0,0,0,0,0,0,0,0,5,8,8,0,7,7,0,8,8,5,8,8,0,7,8,0,8,8,0,0,0,0,0,0,0,0,0,5,9,8,0,8,8,0,7,7,5,8,9,0,8,8,0,7,7,3,5,5,8,8,0,5,5,8,8,0,5,5,8,8,0,7,7,9,9,0,0,0,9,9,6,7,7,9,8,0,8,8,9,9,0,8,7,9,9,0,9,10,10,10,0,0,0,11,10,6,7,7,8,9,0,8,8,9,9,0,7,8,9,9,0,10,9,11,10,0,0,0,10,10,8,9,8,10,10,0,10,10,12,11,0,10,10,11,11,0,12,13,13,13,0,0,0,13,12,8,8,9,10,10,0,10,10,11,12,0,10,10,11,11,0,13,12,13,13,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,8,7,10,10,0,7,7,10,9,0,7,7,10,10,0,9,9,10,10,0,0,0,10,10,6,7,8,10,10,0,7,7,9,10,0,7,7,10,10,0,9,9,10,10,0,0,0,10,10,8,9,9,11,11,0,10,10,11,11,0,10,10,11,11,0,12,12,12,12,0,0,0,12,12,8,9,10,11,11,0,9,10,11,11,0,10,10,11,11,0,12,12,12,12,0,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,8,7,10,10,0,7,7,10,10,0,7,7,10,9,0,9,9,10,10,0,0,0,10,10,6,7,8,10,10,0,7,7,10,10,0,7,7,9,10,0,9,9,10,10,0,0,0,10,10,8,10,9,12,11,0,10,10,12,11,0,10,9,11,11,0,11,12,12,12,0,0,0,12,12,8,9,10,11,12,0,10,10,11,11,0,9,10,11,11,0,12,11,12,12,0,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,10,9,12,12,0,9,9,12,11,0,9,9,11,11,0,10,10,12,11,0,0,0,11,12,7,9,10,12,12,0,9,9,11,12,0,9,9,11,11,0,10,10,11,12,0,0,0,11,11,9,11,10,13,12,0,10,10,12,12,0,10,10,12,12,0,11,11,12,12,0,0,0,13,12,9,10,11,12,13,0,10,10,12,12,0,10,10,12,12,0,11,12,12,12,0,0,0,12,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,11,10,13,13,0,10,10,12,12,0,10,10,12,12,0,11,12,12,12,0,0,0,12,12,9,10,11,13,13,0,10,10,12,12,0,10,10,12,12,0,12,11,13,12,0,0,0,12,12,3,4,4,5,5,6,6,8,8,0,4,4,5,5,6,7,8,8,0,4,4,5,5,7,7,8,8,0,5,5,6,6,7,7,9,9,0,0,0,6,6,7,7,9,9,0,0,0,7,7,8,8,9,9,0,0,0,7,7,8,8,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,10,10,0,5,4,5,5,7,7,8,8,8,8,9,9,10,10,11,11,0,5,5,6,6,7,7,8,8,8,8,9,9,10,10,11,11,0,6,5,6,6,7,7,8,8,9,9,10,10,11,11,11,12,0,0,0,6,6,7,7,8,8,9,9,10,10,11,11,12,12,0,0,0,7,7,7,7,9,9,9,9,10,10,11,11,12,12,0,0,0,7,7,7,8,9,9,9,9,10,10,11,11,12,12,0,0,0,7,7,8,8,9,9,10,10,11,11,12,12,13,13,0,0,0,0,0,8,8,9,9,10,10,11,11,12,12,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,4,4,5,7,7,6,7,7,4,7,6,9,10,10,10,10,9,4,6,7,9,10,10,10,9,10,5,9,9,9,11,11,10,11,11,7,10,9,11,12,11,12,12,12,7,9,10,11,11,12,12,12,12,6,10,10,10,12,12,10,12,11,7,10,10,11,12,12,11,12,12,7,10,10,11,12,12,12,12,12,4,5,5,6,6,7,7,7,7,7,7,11,5,5,6,6,7,7,7,7,8,8,11,5,5,6,6,7,7,7,7,8,8,11,5,5,6,6,7,7,8,8,8,8,11,11,11,6,6,7,7,7,8,8,8,11,11,11,6,6,7,7,7,8,8,8,11,11,11,6,6,7,7,7,7,8,8,11,11,11,7,7,7,7,7,7,8,8,11,11,11,10,10,7,7,7,7,8,8,11,11,11,11,11,7,7,7,7,7,7,11,11,11,11,11,7,7,7,7,7,7,2,4,4,6,6,7,7,7,7,8,8,9,9,5,4,4,6,6,8,8,9,9,9,9,10,10,6,4,4,6,6,8,8,9,9,9,9,10,10,0,6,6,7,7,8,8,9,9,10,10,11,11,0,6,6,7,7,8,8,9,9,10,10,11,11,0,10,10,8,8,9,9,10,10,11,11,12,12,0,11,11,8,8,9,9,10,10,11,11,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,4,4,4,5,5,5,4,4,5,5,5,4,4,5,5,5,5,5,5,5,5,5,5,5,5,2,4,4,6,6,7,7,8,8,10,10,11,11,6,4,4,6,6,8,8,9,9,10,10,12,12,6,4,5,6,6,8,8,9,9,10,10,12,12,20,6,6,6,6,8,8,9,10,11,11,12,12,20,6,6,6,6,8,8,10,10,11,11,12,12,20,10,10,7,7,9,9,10,10,11,11,12,12,20,11,11,7,7,9,9,10,10,11,11,12,12,20,20,20,9,9,9,9,11,11,12,12,13,13,20,20,20,9,9,9,9,11,11,12,12,13,13,20,20,20,13,13,10,10,11,11,12,13,13,13,20,20,20,13,13,10,10,11,11,12,13,13,13,20,20,20,20,19,12,12,12,12,13,13,14,15,19,19,19,19,19,12,12,12,12,13,13,14,14,5,6,6,6,6,7,7,7,7,7,7,7,6,6,6,6,7,7,7,7,7,7,7,6,6,6,6,7,7,7,7,7,7,8,6,6,6,6,7,7,7,7,7,7,8,8,8,6,6,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,7,7,7,7,7,7,7,7,8,8,8,8,8,7,7,7,7,7,7,8,8,8,8,8,7,7,7,7,7,7,8,8,8,8,8,7,7,7,7,7,7,1,4,4,7,6,8,8,8,8,9,9,10,10,11,10,6,5,5,7,7,9,9,8,9,10,10,11,11,12,12,6,5,5,7,7,9,9,9,9,10,10,11,11,12,12,21,7,8,8,8,9,9,9,9,10,10,11,11,12,12,21,8,8,8,8,9,9,9,9,10,10,11,11,12,12,21,11,12,9,9,10,10,10,10,10,11,11,12,12,12,21,12,12,9,8,10,10,10,10,11,11,12,12,13,13,21,21,21,9,9,9,9,11,11,11,11,12,12,12,13,21,20,20,9,9,9,9,10,11,11,11,12,12,13,13,20,20,20,13,13,10,10,11,11,12,12,13,13,13,13,20,20,20,13,13,10,10,11,11,12,12,13,13,13,13,20,20,20,20,20,12,12,12,12,12,12,13,13,14,14,20,20,20,20,20,12,12,12,11,13,12,13,13,14,14,20,20,20,20,20,15,16,13,12,13,13,14,13,14,14,20,20,20,20,20,16,15,12,12,13,12,14,13,14,14,4,6,6,7,7,7,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,10,6,6,7,7,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,10,6,6,7,7,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,10,7,7,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,9,9,9,9,9,9,9,9,9,9,9,9,10,9,9,9,10,10,10,10,10,10,10,9,9,9,9,9,9,10,9,9,9,9,9,9,9,10,10,10,10,10,10,10,9,9,9,10,10,10,10,10,9,9,9,9,9,9,10,10,10,10,10,10,10,9,9,10,9,10,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,9,9,10,10,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,9,9,10,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,9,9,10,10,9,9,10,9,9,9,10,10,10,10,10,10,10,10,10,10,10,9,9,10,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,9,9,9,9,10,9,9,9,9,9,1,4,3,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,4,5,6,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,4,6,6,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,1,4,4,7,7,7,7,8,7,9,8,9,9,10,10,11,11,11,11,6,5,5,8,8,9,9,9,8,10,9,11,10,12,12,13,12,13,13,5,5,5,8,8,9,9,9,9,10,10,11,11,12,12,13,12,13,13,17,8,8,9,9,9,9,9,9,10,10,12,11,13,12,13,13,13,13,18,8,8,9,9,9,9,9,9,11,11,12,12,13,13,13,13,13,13,17,13,12,9,9,10,10,10,10,11,11,12,12,12,13,13,13,14,14,18,13,12,9,9,10,10,10,10,11,11,12,12,13,13,13,14,14,14,17,18,18,10,10,10,10,11,11,11,12,12,12,14,13,14,13,13,14,18,18,18,10,9,10,9,11,11,12,12,12,12,13,13,15,14,14,14,18,18,16,13,14,10,11,11,11,12,13,13,13,13,14,13,13,14,14,18,18,18,14,12,11,9,11,10,13,12,13,13,13,14,14,14,13,14,18,18,17,18,18,11,12,12,12,13,13,14,13,14,14,13,14,14,14,18,18,18,18,17,12,10,12,9,13,11,13,14,14,14,14,14,15,14,18,18,17,17,18,14,15,12,13,13,13,14,13,14,14,15,14,15,14,18,17,18,18,18,15,15,12,10,14,10,14,14,13,13,14,14,14,14,18,16,18,18,18,18,17,14,14,13,14,14,13,13,14,14,14,15,15,18,18,18,18,17,17,17,14,14,14,12,14,13,14,14,15,14,15,14,18,18,18,18,18,18,18,17,16,13,13,13,14,14,14,14,15,16,15,18,18,18,18,18,18,18,17,17,13,13,13,13,14,13,14,15,15,15,2,4,4,5,5,5,5,5,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,3,8,12,14,15,15,15,13,15,15,6,5,8,10,12,12,13,12,14,13,10,6,5,6,8,9,11,11,13,13,13,8,5,4,5,6,8,10,11,13,14,10,7,5,4,5,7,9,11,12,13,11,8,6,5,4,5,7,9,11,12,11,10,8,7,5,4,5,9,10,13,13,11,10,8,6,5,4,7,9,15,14,13,12,10,9,8,7,8,9,12,12,14,13,12,11,10,9,8,9,4,3,7,9,12,16,16,3,2,5,7,11,14,15,7,4,5,6,9,12,15,8,5,5,5,8,10,14,9,7,6,6,8,10,12,12,10,10,7,6,8,10,15,12,10,6,4,7,9,1,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,5,5,0,7,7,0,8,8,0,9,9,0,12,12,0,8,8,0,9,9,0,13,13,0,8,8,0,6,6,0,11,11,0,12,12,0,12,12,0,14,14,0,11,12,0,12,12,0,15,15,0,12,12,0,5,5,0,5,5,0,6,6,0,7,7,0,10,10,0,6,6,0,7,7,0,11,11,0,6,6,0,7,7,0,11,11,0,12,11,0,11,11,0,14,14,0,10,10,0,12,12,0,15,15,0,12,12,0,6,6,0,12,12,0,12,12,0,12,12,0,14,14,0,11,11,0,12,12,0,16,16,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,8,0,12,12,0,12,12,0,12,12,0,15,15,0,12,12,0,11,11,0,16,16,0,11,11,0,6,6,0,12,12,0,12,12,0,13,13,0,15,15,0,12,12,0,13,13,0,15,15,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,0,9,9,0,9,9,0,10,10,0,9,9,0,10,10,0,10,10,0,10,10,0,10,10,0,7,7,0,7,7,0,6,6,0,8,8,0,7,7,0,8,8,0,8,8,0,7,7,0,8,8,0,7,7,0,9,9,0,8,9,0,10,10,0,9,9,0,10,10,0,10,11,0,9,9,0,10,10,0,9,9,0,11,11,0,12,12,0,12,12,0,11,11,0,12,12,0,13,13,0,12,12,0,13,13,0,8,8,0,12,12,0,12,12,0,13,13,0,13,13,0,13,13,0,13,13,0,13,13,0,13,13,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,11,11,0,12,12,0,13,13,0,12,12,0,13,13,0,13,13,0,12,12,0,12,12,0,9,9,0,12,12,0,13,13,0,14,14,0,13,13,0,14,14,0,14,14,0,13,13,0,14,14,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,6,6,6,8,8,6,8,8,7,9,9,10,11,11,8,8,8,7,9,9,11,12,12,9,9,9,6,7,7,10,11,11,10,11,11,10,11,11,13,13,13,12,12,12,10,12,11,14,14,14,12,12,12,6,5,5,9,6,6,9,6,6,9,7,7,12,10,10,11,7,6,9,7,7,13,11,11,12,7,7,7,8,8,12,10,10,12,10,10,11,10,10,15,13,13,13,9,9,12,11,11,15,14,14,15,11,11,8,7,7,12,11,11,12,11,11,11,11,11,14,13,14,14,12,12,12,11,11,16,15,15,14,12,12,0,10,10,0,12,12,0,12,12,0,11,11,0,14,14,0,11,11,0,11,11,0,15,15,0,11,11,7,8,8,13,11,11,12,10,10,12,11,11,15,13,13,14,11,11,12,10,10,16,14,14,15,10,10,9,7,7,13,11,12,13,12,11,12,11,11,15,14,14,14,12,12,13,12,12,16,15,15,15,12,12,0,11,11,0,12,12,0,12,13,0,12,12,0,15,15,0,12,12,0,12,12,0,16,15,0,12,12,2,3,4,9,9,10,12,12,12,11,10,12,12,13,12,11,13,12,11,11,11,12,12,12,11,11,13,13,13,13,11,12,12,14,14,12,13,13,13,13,11,13,13,13,13,11,13,13,13,13,11,13,13,13,13,11,12,12,14,14,12,13,13,12,12,11,13,13,13,13,11,13,13,12,12,11,13,13,13,13,12,12,13,14,14,12,13,13,12,12,11,13,13,13,13,11,13,13,12,12,11,13,13,13,13,12,13,13,14,14,12,13,13,12,12,11,13,13,13,13,11,13,13,12,12,11,10,10,10,10,12,10,10,11,11,12,9,9,11,11,13,11,11,10,10,13,10,10,10,10,13,11,11,12,12,13,10,10,12,12,14,12,11,12,12,13,11,11,11,12,13,12,12,12,12,13,11,11,12,12,13,10,10,12,12,14,11,11,12,12,13,11,11,12,12,13,11,11,12,12,14,12,12,12,12,14,10,10,11,11,14,12,11,11,11,13,11,11,11,11,13,12,12,11,11,14,12,12,12,11,14,10,10,11,11,14,12,11,11,11,13,11,11,11,11,13,12,12,11,11,11,11,11,10,10,12,10,11,9,9,12,12,12,11,11,13,12,12,9,9,13,13,13,10,10,13,13,13,12,12,13,13,13,14,14,13,12,12,11,11,14,13,13,12,12,14,13,13,11,11,13,13,13,12,11,13,13,13,14,14,13,12,12,10,10,14,13,13,11,11,13,13,13,10,10,13,13,13,11,11,14,13,13,14,14,14,12,12,10,10,13,13,13,11,11,13,13,13,10,10,13,13,13,11,11,14,13,13,14,14,14,13,13,10,10,13,13,13,11,11,13,13,13,10,10,14,12,12,8,8,14,12,12,9,9,14,11,11,9,9,14,12,12,8,8,14,12,12,7,7,15,13,13,10,10,15,12,12,10,10,15,13,13,10,10,15,12,13,9,9,15,13,13,10,10,15,13,13,10,10,15,12,12,10,10,15,13,13,10,10,15,13,13,9,9,15,13,13,10,10,15,13,13,10,10,15,12,12,10,10,15,13,13,9,9,14,13,12,9,9,14,13,13,9,9,15,13,13,10,10,15,12,12,10,10,15,13,13,9,9,15,13,13,9,9,14,13,13,9,9,14,12,12,8,8,13,13,13,8,8,14,14,13,9,9,14,14,13,7,7,14,14,14,8,8,14,14,14,10,10,15,14,14,12,12,14,14,14,9,9,15,14,14,10,10,14,14,14,9,9,14,14,14,10,9,15,14,14,12,12,14,14,14,9,9,15,14,14,10,10,14,14,14,9,9,15,14,15,9,9,15,14,14,11,11,14,14,14,8,8,14,14,14,9,9,14,14,14,8,8,14,15,14,10,10,15,14,14,11,11,14,14,14,8,8,15,14,14,9,9,14,14,14,8,8,12,12,12,13,13,16,16,15,12,12,17,16,16,13,13,17,16,16,11,11,17,16,16,12,12,17,16,17,13,13,17,16,16,14,14,17,17,16,12,12,18,16,16,13,13,17,16,17,12,12,17,17,17,13,13,18,16,16,14,14,18,17,17,12,12,17,17,17,13,13,18,17,17,13,13,17,17,17,13,13,17,16,16,14,14,17,17,17,12,12,16,16,17,13,13,17,17,16,12,12,18,17,17,13,13,18,16,16,14,14,18,17,17,12,12,19,16,17,13,13,17,16,17,12,12,13,14,14,10,10,16,14,14,13,13,17,15,15,14,14,17,14,14,13,13,16,14,14,13,13,17,16,15,14,14,16,16,16,15,15,17,15,15,14,14,17,15,15,14,14,17,15,15,14,14,17,16,15,14,14,16,16,16,15,15,18,15,15,13,13,16,16,15,14,14,17,15,15,14,13,17,15,15,14,14,16,16,16,15,15,18,15,14,13,13,17,15,15,14,14,18,14,15,13,13,18,15,15,14,14,16,16,16,15,15,17,15,15,13,13,17,15,15,14,14,17,15,15,13,13,13,11,11,10,10,16,14,14,13,13,17,14,15,14,14,17,15,15,12,12,17,14,14,12,12,16,15,15,14,14,16,14,14,14,14,16,15,15,14,14,16,15,15,14,14,16,15,15,14,14,16,15,15,14,14,16,15,14,15,15,17,15,15,14,14,17,15,15,14,14,17,15,15,14,14,17,15,16,14,14,16,14,14,14,14,17,15,15,13,13,17,15,15,13,13,16,15,15,13,13,17,16,16,14,14,17,15,14,15,14,17,15,15,13,13,17,15,15,13,13,17,15,15,13,13,14,14,14,9,9,14,14,14,18,19,14,15,15,19,18,14,14,14,19,19,15,14,14,19,19,15,16,16,19,19,15,16,16,19,19,15,15,15,19,19,15,16,16,19,20,15,15,15,19,19,15,15,15,19,19,15,16,16,20,20,15,15,15,18,19,15,15,16,19,20,15,15,15,19,18,15,15,15,18,18,15,16,16,21,20,15,15,15,19,19,15,15,15,19,19,15,15,14,19,20,15,15,15,20,19,15,16,16,19,20,15,15,15,19,19,15,15,15,20,21,15,14,15,19,19,14,12,12,9,9,14,14,15,21,19,14,14,14,18,19,14,15,15,19,20,14,14,14,19,19,15,15,15,19,20,15,15,14,21,19,15,15,15,20,19,15,14,15,20,21,15,15,15,18,18,15,15,15,20,21,16,14,14,18,19,15,15,15,20,19,15,15,15,18,21,15,15,15,19,19,15,15,15,19,20,16,15,14,20,19,15,16,15,19,19,15,15,15,19,0,14,15,15,19,19,15,15,15,19,19,15,15,14,20,19,15,15,15,20,19,15,15,15,19,19,15,15,15,20,19,12,12,12,13,13,16,15,16,11,11,16,16,16,12,12,17,16,16,11,11,17,16,16,12,11,17,17,17,13,13,18,16,16,14,14,18,18,17,13,13,17,16,16,13,13,17,17,17,13,13,17,16,17,12,12,17,15,16,13,13,17,16,17,12,12,17,16,16,13,12,17,16,16,12,12,18,17,17,13,13,18,16,16,13,14,18,17,17,12,12,17,16,16,12,12,17,17,17,12,12,18,17,17,13,13,17,16,16,14,14,17,17,17,12,12,17,16,16,12,12,18,17,17,12,12,13,14,14,9,9,16,14,14,13,13,16,15,15,14,14,16,14,14,13,13,16,14,14,13,13,17,16,15,15,15,16,15,16,16,15,17,15,15,14,14,17,15,15,15,15,17,15,15,14,14,17,15,15,14,14,16,15,16,16,16,17,15,15,14,14,16,15,15,14,15,16,15,15,14,14,17,15,15,15,15,16,16,16,15,16,18,15,14,13,14,17,15,15,14,14,17,14,14,13,13,17,15,15,14,14,16,15,15,15,15,17,15,14,14,14,17,15,15,14,14,17,14,14,13,13,13,11,11,11,11,16,14,14,12,12,16,14,14,13,13,16,14,14,12,12,16,14,14,12,12,16,15,15,13,13,17,14,14,14,14,17,15,15,13,13,16,15,15,14,13,16,15,15,13,13,16,15,15,13,13,16,14,14,14,14,16,15,15,13,13,16,14,15,13,13,17,15,15,13,13,17,15,15,13,13,16,14,14,14,14,17,15,15,12,12,17,14,15,13,13,17,15,15,12,12,16,15,15,13,13,17,14,14,14,14,17,15,15,12,12,17,15,15,13,13,16,15,15,12,12,14,15,15,8,8,14,14,14,19,18,14,15,15,19,20,14,14,14,19,19,14,14,15,19,20,15,16,15,19,21,15,16,16,21,19,15,15,15,20,19,15,16,16,19,20,15,15,15,19,18,15,16,15,20,19,15,16,16,19,20,15,15,15,19,19,15,16,15,20,20,14,15,15,19,19,15,15,15,21,19,15,17,16,19,20,15,14,15,0,21,15,15,15,19,20,14,14,14,19,19,15,15,15,20,19,15,16,16,19,19,15,15,15,19,18,15,15,15,20,19,14,14,15,18,18,14,12,12,9,9,14,14,14,18,18,14,14,14,18,18,14,15,14,19,18,14,14,14,19,18,15,15,15,19,20,15,14,14,18,18,15,15,15,20,19,15,15,15,18,20,15,15,15,19,18,15,15,15,19,19,15,14,14,19,21,15,15,15,20,20,15,15,15,18,19,14,15,15,19,20,15,15,15,20,19,15,14,14,19,21,15,15,15,18,19,15,14,15,20,19,14,15,15,21,21,14,15,15,19,20,15,14,14,19,20,15,15,15,19,20,15,15,14,20,20,14,15,15,20,19,13,12,12,13,13,17,16,16,11,11,17,16,16,12,12,18,17,16,11,11,18,16,16,11,11,17,17,17,13,13,18,16,16,13,13,18,17,17,12,12,18,16,16,13,13,18,17,17,12,12,18,17,17,13,13,18,16,16,14,14,18,16,17,12,12,18,17,17,13,13,17,17,17,12,12,17,17,17,12,12,17,16,15,13,13,18,16,16,11,11,17,16,16,12,12,17,16,17,11,11,18,17,17,13,12,17,16,16,13,13,17,17,17,12,12,17,16,17,12,12,18,17,17,11,11,14,14,14,9,9,16,14,14,13,13,17,15,15,14,14,17,14,14,13,13,16,14,14,13,13,17,15,15,14,14,16,16,16,16,15,18,15,15,14,14,17,16,15,15,15,17,15,15,14,14,17,15,15,14,15,16,16,16,15,16,18,15,15,14,14,17,15,15,14,15,17,15,15,14,14,17,15,15,14,14,16,16,16,15,16,17,14,14,13,13,17,15,15,14,14,18,15,15,13,13,17,15,15,14,14,16,16,16,15,15,17,14,14,13,13,17,15,15,14,14,17,14,14,13,13,13,11,11,11,11,16,14,14,12,12,16,14,14,12,13,17,15,14,11,11,17,14,14,11,11,17,15,15,13,14,17,14,14,14,14,17,15,15,13,13,17,14,14,13,13,17,15,15,13,13,17,15,15,13,13,17,14,14,14,14,17,15,15,13,13,18,14,15,13,13,17,15,15,13,13,16,15,15,13,13,17,14,14,13,13,17,15,15,12,12,16,14,14,12,12,16,15,15,12,12,17,16,15,13,13,17,14,14,13,13,17,15,15,12,12,16,15,15,12,12,16,15,15,12,12,13,15,15,8,8,14,14,14,18,19,14,15,15,19,20,14,14,14,18,18,14,15,15,18,18,15,16,16,19,19,15,16,17,20,20,15,15,15,19,19,15,16,16,18,20,15,15,15,19,19,15,15,16,18,18,15,17,16,19,19,15,15,15,18,21,15,16,16,21,20,15,15,15,19,21,15,16,15,20,19,15,16,17,20,20,15,15,15,19,19,15,16,16,21,20,15,15,15,19,20,15,15,15,19,19,15,16,16,20,19,15,15,15,19,19,15,16,15,20,21,15,15,15,21,19,14,12,12,8,8,14,14,14,20,18,14,13,13,19,19,14,14,14,19,18,15,14,14,19,20,14,15,15,20,20,15,14,14,21,20,15,15,15,20,20,15,15,14,21,19,15,15,15,19,19,15,15,15,19,20,15,14,14,20,20,15,15,15,19,20,15,14,14,19,20,15,15,15,20,20,15,15,15,20,19,15,14,14,20,21,15,15,15,20,21,15,14,14,20,0,15,16,15,20,21,15,15,15,19,20,15,14,14,19,19,15,15,15,19,20,15,15,15,19,19,15,15,15,18,20,13,12,12,13,13,18,16,17,12,12,17,16,16,12,12,17,17,16,11,11,18,16,16,11,11,17,17,18,13,13,18,16,16,14,14,18,17,17,13,13,18,16,16,13,13,18,17,17,12,12,17,17,16,13,13,17,16,16,13,14,18,17,17,12,12,18,16,16,12,13,17,16,17,12,12,17,18,17,13,13,18,16,16,13,13,18,17,17,12,12,17,16,16,12,12,17,17,17,11,11,17,16,17,12,12,17,16,16,13,13,17,16,16,11,11,17,16,16,12,12,18,16,17,11,11,14,14,14,9,9,16,14,15,13,13,17,15,15,14,14,17,14,14,12,12,16,14,14,13,13,18,15,15,15,15,17,15,16,15,16,18,15,15,14,14,17,15,16,15,15,17,15,15,14,14,18,15,15,14,14,16,16,16,16,15,17,15,15,14,14,16,15,15,14,14,17,15,15,14,14,17,15,15,14,14,17,16,16,15,15,17,15,14,13,13,17,15,15,14,14,17,15,15,13,13,17,15,15,14,14,16,16,16,15,15,18,15,14,14,14,17,15,15,14,14,18,15,15,13,13,13,12,12,11,11,16,14,14,12,12,16,14,14,13,13,17,15,15,12,12,17,14,14,12,12,17,15,15,14,14,17,14,14,14,14,17,15,15,13,13,17,15,14,13,13,17,15,15,13,13,17,15,15,13,13,16,14,14,14,14,17,15,15,13,13,16,14,14,13,13,16,15,15,13,13,17,15,16,13,13,17,14,14,14,13,17,15,15,12,12,16,15,14,12,12,17,15,15,12,12,16,15,16,13,13,16,14,14,14,13,17,15,15,12,12,16,14,14,12,12,17,15,15,12,12,14,15,15,8,8,14,14,14,18,18,14,15,15,19,18,14,14,14,18,18,14,15,15,19,20,15,16,15,21,18,15,16,16,18,0,15,15,15,19,20,15,16,16,20,0,15,16,15,19,18,15,15,15,19,19,15,16,16,21,19,15,15,15,19,19,15,16,16,20,20,15,15,15,19,19,15,15,15,19,18,15,16,16,20,20,15,14,15,20,19,15,15,15,19,20,15,15,15,19,19,15,16,15,19,20,15,16,16,19,20,15,15,15,19,19,15,16,15,20,20,15,15,15,20,18,13,12,12,8,8,14,14,14,19,20,14,14,14,19,19,14,15,15,20,20,14,14,14,18,19,15,15,15,20,0,15,14,14,18,20,15,15,15,19,19,15,15,15,21,19,15,15,15,19,20,15,15,15,20,21,15,14,14,20,19,15,15,15,20,19,15,15,14,21,19,15,15,15,19,18,15,15,15,20,19,15,14,14,19,19,15,15,16,20,19,15,15,15,20,0,15,15,15,19,21,15,15,15,22,20,15,14,14,22,19,15,15,15,19,20,15,14,14,20,19,14,15,15,19,21,1,7,7,14,14,6,8,8,15,16,7,8,8,16,15,0,14,14,17,17,0,14,14,16,16,7,9,9,16,16,10,11,11,17,18,9,8,8,16,16,0,14,14,19,19,0,14,14,17,16,8,9,9,16,16,12,12,12,17,17,10,9,9,16,16,0,15,14,18,20,0,14,14,17,17,0,15,15,18,17,0,21,0,0,21,0,13,13,17,17,0,17,17,0,0,0,15,15,17,17,0,15,15,17,18,0,0,0,0,21,0,13,13,17,17,0,18,18,0,21,0,16,15,17,18,6,7,7,14,14,9,10,10,16,16,11,10,10,15,15,0,21,0,20,21,0,0,0,18,20,10,10,10,15,16,12,13,13,18,18,12,11,11,15,15,0,0,0,20,20,0,0,21,19,19,12,11,11,15,15,15,14,14,18,18,13,11,11,15,16,0,0,0,20,19,0,0,0,20,21,0,0,20,19,19,0,0,0,0,0,0,20,0,17,18,0,0,21,0,0,0,0,0,21,0,0,21,0,20,19,0,0,0,0,0,0,21,0,18,18,0,0,0,21,0,0,0,0,0,20,7,6,6,13,13,9,6,6,12,12,9,7,7,14,14,0,10,10,12,12,0,11,11,15,15,9,7,7,14,14,12,9,9,14,14,10,7,7,14,13,0,11,11,16,15,0,11,11,14,14,9,7,7,14,14,13,10,10,14,14,11,7,7,14,13,0,11,11,16,16,0,11,11,14,14,0,12,12,16,16,0,19,0,17,18,0,10,10,14,14,0,15,14,0,0,0,12,12,14,14,0,12,12,15,15,0,20,0,18,19,0,10,10,14,14,0,16,15,0,20,0,13,13,14,14,0,11,11,13,13,0,12,13,16,16,0,12,12,16,16,0,16,16,0,21,0,17,18,0,0,0,12,12,16,16,0,15,15,18,0,0,12,12,16,16,0,17,16,21,21,0,16,17,0,0,0,13,13,17,16,0,16,16,20,21,0,12,12,17,16,0,17,17,0,21,0,17,17,21,21,0,17,18,0,0,0,0,0,0,0,0,15,15,0,0,0,18,21,0,0,0,18,19,0,0,0,18,17,21,21,0,0,0,0,0,0,16,16,0,0,0,0,0,0,0,0,19,19,0,0,0,11,11,12,12,0,11,11,10,10,0,12,12,13,13,0,12,12,9,9,0,14,14,13,13,0,12,12,13,13,0,14,14,12,13,0,11,11,12,12,0,13,13,13], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+330920);
/* memory initializer */ allocate([13,0,13,13,13,13,0,12,12,13,13,0,14,14,12,12,0,11,11,12,12,0,14,13,14,14,0,13,13,13,13,0,15,15,14,15,0,0,0,16,16,0,12,12,13,13,0,16,17,20,21,0,14,13,12,12,0,14,14,14,14,0,21,0,16,16,0,12,12,13,13,0,18,17,21,0,0,14,14,13,13,7,8,8,17,17,11,10,10,18,18,12,10,10,17,17,0,15,15,20,18,0,15,15,17,17,11,9,9,17,17,14,12,12,19,19,13,9,9,16,16,0,15,14,0,19,0,14,14,16,16,12,10,10,20,18,16,13,13,21,20,14,10,10,17,17,0,15,15,21,20,0,15,14,17,17,0,15,15,21,21,0,0,21,0,0,0,13,13,18,18,0,19,16,0,0,0,15,15,17,16,0,16,16,0,21,0,0,0,0,21,0,13,14,18,17,0,20,19,0,0,0,15,15,18,18,8,7,7,15,15,12,11,11,17,16,13,11,11,16,16,0,0,0,21,20,0,0,0,0,20,11,10,10,17,17,14,13,13,19,18,14,11,11,16,16,0,20,0,21,19,0,0,21,0,20,12,11,11,17,17,16,15,15,0,19,14,11,11,17,16,0,21,0,0,19,0,0,0,21,20,0,0,21,20,0,0,0,0,0,0,0,0,0,19,21,0,0,0,0,0,0,0,0,19,20,0,0,0,20,21,0,0,0,0,0,0,20,0,19,21,0,0,0,0,0,0,0,0,21,20,11,10,9,15,15,14,11,11,15,15,14,11,11,16,16,0,14,14,14,14,0,16,15,17,16,13,11,11,16,16,16,13,13,16,16,15,10,10,15,15,0,14,15,17,17,0,14,14,16,15,13,11,11,16,16,17,15,14,16,16,15,10,10,15,15,0,15,15,17,18,0,15,15,16,16,0,16,16,17,17,0,21,0,21,20,0,13,13,15,15,0,18,18,0,21,0,15,15,15,15,0,16,16,17,17,0,0,0,0,18,0,13,13,15,15,0,19,18,0,0,0,15,15,16,16,0,12,12,15,15,0,13,13,17,17,0,13,13,17,18,0,16,17,21,0,0,20,18,0,0,0,13,13,17,17,0,15,15,0,18,0,12,12,17,18,0,16,16,0,0,0,17,17,21,0,0,13,13,18,18,0,16,16,21,21,0,12,12,17,18,0,16,17,21,0,0,17,17,0,21,0,17,18,0,0,0,0,0,0,0,0,16,15,0,21,0,21,19,0,0,0,18,18,0,0,0,18,19,0,0,0,0,0,0,0,0,16,16,21,21,0,20,19,0,0,0,19,21,0,21,0,12,12,15,15,0,12,12,15,16,0,13,13,16,16,0,14,14,15,15,0,16,15,17,17,0,13,13,17,17,0,15,15,16,18,0,12,12,16,16,0,14,14,17,17,0,15,14,16,16,0,13,13,16,16,0,16,15,17,17,0,12,12,16,16,0,15,15,18,18,0,14,14,17,16,0,16,16,17,18,0,0,0,20,21,0,13,13,16,17,0,17,17,0,0,0,15,15,16,16,0,15,16,17,17,0,0,0,19,0,0,13,13,15,16,0,19,18,0,0,0,16,15,16,17,8,8,8,17,17,13,11,10,17,18,13,10,10,17,17,0,15,15,20,19,0,15,15,17,17,12,10,10,19,18,15,12,12,20,18,14,10,10,17,16,0,15,15,20,20,0,14,15,16,16,13,10,10,17,17,17,14,14,0,18,15,10,10,17,17,0,16,15,20,20,0,14,14,17,17,0,15,16,20,20,0,0,21,0,0,0,13,13,17,17,0,18,17,0,0,0,15,16,17,18,0,15,15,18,21,0,0,0,21,0,0,13,13,18,18,0,19,19,0,0,0,16,16,18,17,9,8,8,15,15,12,11,11,16,16,13,11,11,16,15,0,0,0,0,21,0,21,0,19,19,12,11,11,17,18,15,13,13,18,19,14,11,11,16,16,0,0,21,21,19,0,0,0,21,20,13,11,11,18,17,17,14,15,20,21,15,11,12,16,16,0,0,0,20,0,0,0,21,0,19,0,0,0,0,19,0,0,0,0,0,0,21,21,19,19,0,0,0,21,0,0,0,0,19,21,0,0,0,19,20,0,0,0,21,0,0,0,21,19,19,0,0,0,0,0,0,0,0,21,20,0,11,11,15,15,0,12,12,15,16,0,12,12,16,16,0,15,15,16,15,0,16,16,17,17,0,12,12,17,17,0,14,14,17,17,0,11,11,16,16,0,15,15,19,18,0,15,15,16,16,0,12,12,17,16,0,14,15,16,16,0,11,11,15,15,0,16,16,18,19,0,15,15,15,16,0,17,17,18,20,0,21,0,21,19,0,14,14,16,16,0,18,18,0,0,0,16,16,15,15,0,16,16,18,17,0,0,0,19,20,0,14,14,16,16,0,19,19,0,0,0,16,17,15,15,0,12,12,14,15,0,13,13,16,17,0,12,12,17,17,0,17,16,0,0,0,18,17,21,0,0,13,13,19,17,0,15,15,20,21,0,12,12,17,17,0,17,17,0,0,0,17,17,0,0,0,13,13,17,18,0,16,16,21,0,0,12,12,17,17,0,17,17,0,0,0,17,17,0,0,0,18,21,0,0,0,0,0,0,0,0,15,15,21,0,0,20,21,0,0,0,18,19,0,0,0,18,17,0,0,0,0,0,0,0,0,16,16,21,0,0,21,21,0,0,0,18,19,0,0,0,12,12,16,16,0,13,13,16,17,0,13,13,17,16,0,14,14,16,16,0,16,15,19,18,0,13,13,17,17,0,15,15,18,18,0,12,12,16,16,0,15,15,18,19,0,15,15,17,16,0,13,13,17,17,0,16,16,18,17,0,12,12,17,16,0,15,15,18,18,0,15,15,17,17,0,16,16,0,19,0,0,0,0,0,0,14,14,16,17,0,18,18,0,0,0,15,15,17,17,0,16,16,21,19,0,21,0,21,21,0,13,14,16,16,0,19,19,0,0,0,15,16,16,16,0,11,11,17,16,0,15,14,19,18,0,14,14,19,19,0,18,17,18,20,0,17,17,18,19,0,13,13,17,17,0,16,17,21,18,0,13,13,17,16,0,18,17,19,0,0,16,17,18,18,0,12,12,19,18,0,18,18,20,20,0,13,13,17,17,0,17,17,21,0,0,16,17,17,18,0,18,17,19,18,0,0,0,0,0,0,14,14,17,17,0,19,19,21,0,0,16,16,16,17,0,17,17,19,20,0,0,0,0,21,0,15,15,17,18,0,21,21,0,0,0,17,17,17,18,0,10,10,15,15,0,15,14,17,18,0,14,14,16,16,0,0,0,18,0,0,21,0,19,0,0,13,13,17,16,0,17,17,18,0,0,14,14,16,15,0,0,0,21,0,0,21,0,19,18,0,13,13,17,17,0,18,18,20,20,0,15,15,16,16,0,0,0,21,21,0,0,0,20,20,0,0,0,19,0,0,0,0,0,0,0,21,20,18,18,0,0,0,0,0,0,0,0,0,20,0,0,0,0,20,0,0,0,0,0,0,0,0,19,18,0,0,0,0,21,0,0,0,18,20,0,18,19,16,17,0,21,19,17,17,0,0,21,18,18,0,0,21,20,19,0,0,0,20,20,0,0,21,17,17,0,0,0,19,19,0,20,20,17,17,0,0,0,0,20,0,0,20,18,18,0,21,20,17,17,0,0,0,20,21,0,19,0,17,17,0,0,21,0,0,0,20,0,18,19,0,0,0,21,21,0,0,0,0,21,0,20,20,17,17,0,0,0,0,0,0,21,0,18,17,0,0,0,20,19,0,0,0,0,21,0,20,20,17,17,0,0,0,0,0,0,21,21,18,18,0,12,12,15,14,0,14,14,17,17,0,14,14,17,16,0,18,18,21,0,0,19,20,0,0,0,13,13,18,17,0,16,16,19,18,0,13,13,17,17,0,17,17,0,0,0,17,17,21,0,0,13,13,17,17,0,17,17,21,20,0,13,13,18,17,0,18,19,21,21,0,19,18,0,0,0,18,17,0,0,0,0,0,0,0,0,15,16,0,0,0,21,21,0,0,0,20,18,21,0,0,17,18,0,0,0,0,0,0,0,0,15,16,0,0,0,0,20,0,0,0,0,19,0,0,0,15,15,18,19,0,18,17,21,0,0,16,18,0,20,0,17,18,21,0,0,18,20,0,0,0,16,16,21,21,0,19,20,21,0,0,16,15,0,21,0,18,20,0,0,0,18,19,0,0,0,16,15,21,21,0,21,0,0,0,0,16,15,21,0,0,20,19,0,0,0,18,21,21,0,0,20,18,0,0,0,0,0,0,0,0,16,16,0,20,0,21,0,0,0,0,17,18,20,21,0,18,18,21,21,0,0,0,0,0,0,16,16,20,0,0,0,21,0,0,0,21,18,0,0,0,12,12,20,17,0,15,15,19,18,0,14,14,19,18,0,18,17,21,19,0,17,17,21,17,0,13,13,21,19,0,16,17,20,19,0,13,13,16,16,0,17,17,20,21,0,16,16,19,17,0,13,13,18,18,0,17,19,19,19,0,13,13,17,17,0,18,18,0,19,0,16,17,18,18,0,16,17,19,21,0,0,0,0,0,0,15,15,16,17,0,20,19,21,0,0,17,17,17,17,0,17,17,21,19,0,0,0,0,0,0,15,15,17,17,0,21,0,0,0,0,18,18,17,17,0,10,10,15,15,0,15,15,17,17,0,15,14,16,16,0,0,0,21,19,0,21,21,19,21,0,13,13,17,16,0,17,17,18,19,0,14,15,16,15,0,0,0,21,19,0,21,21,18,19,0,14,14,16,17,0,18,18,18,19,0,15,15,15,16,0,0,21,0,21,0,0,0,19,20,0,0,0,21,19,0,0,0,0,0,0,21,21,19,17,0,0,0,0,0,0,0,0,21,21,0,21,0,0,21,0,0,0,0,0,0,21,21,19,18,0,0,0,0,0,0,0,0,0,19,0,21,18,18,17,0,21,0,20,20,0,0,0,18,20,0,0,21,18,21,0,0,0,21,18,0,0,0,0,19,0,0,0,21,21,0,20,21,17,19,0,21,0,21,0,0,21,0,18,18,0,20,21,17,18,0,0,0,21,19,0,20,21,17,18,0,0,0,21,21,0,0,0,20,19,0,0,0,21,21,0,0,0,0,0,0,21,21,19,18,0,0,0,0,0,0,0,21,19,18,0,21,21,19,0,0,0,0,21,0,0,21,21,18,17,0,0,0,0,0,0,21,0,21,18,0,12,12,14,14,0,15,14,17,17,0,14,14,17,16,0,19,17,0,0,0,19,19,0,0,0,13,13,17,17,0,17,17,20,20,0,13,13,18,18,0,18,17,0,0,0,18,21,0,0,0,13,13,17,17,0,18,18,21,20,0,14,14,18,19,0,19,18,21,0,0,19,19,0,0,0,20,18,20,0,0,0,0,0,0,0,15,16,0,0,0,21,21,0,0,0,19,19,0,0,0,18,18,0,0,0,0,0,0,0,0,16,16,0,21,0,0,0,0,0,0,19,20,0,0,0,15,15,20,21,0,17,17,21,21,0,17,17,0,0,0,19,18,0,0,0,18,19,0,0,0,17,16,0,21,0,0,20,0,0,0,16,16,0,20,0,19,19,0,21,0,19,18,0,21,0,16,16,0,0,0,21,21,0,0,0,16,16,0,0,0,21,21,0,0,0,19,19,0,0,0,20,0,0,0,0,0,0,0,0,0,17,17,0,21,0,0,20,0,0,0,20,18,21,21,0,19,18,0,20,0,0,0,0,0,0,16,17,21,0,0,0,21,0,0,0,19,20,21,20,2,3,3,3,3,3,3,1,7,7,6,8,8,7,8,8,7,9,9,11,11,11,9,8,8,7,9,9,11,12,11,9,9,9,6,7,7,10,11,11,10,10,10,10,11,11,15,14,14,12,12,12,11,11,11,14,14,14,12,12,12,5,6,6,8,5,5,8,7,7,8,8,8,12,10,10,10,7,7,8,7,7,12,10,10,10,7,7,6,7,7,12,11,11,12,10,10,11,10,10,14,14,13,13,10,10,11,10,10,16,14,14,14,11,10,7,7,7,13,12,12,12,12,11,11,11,11,15,14,17,13,12,12,12,11,11,15,15,15,14,13,13,10,9,9,14,12,11,13,11,11,12,11,11,16,15,14,14,11,11,12,11,11,17,14,14,15,11,11,7,8,8,12,11,11,13,10,10,11,10,10,17,14,13,14,10,10,12,10,10,18,15,15,14,10,10,8,7,7,13,12,12,13,11,11,12,11,11,16,14,15,14,12,12,12,11,11,18,16,16,14,12,12,11,10,10,13,12,11,13,11,11,13,12,12,0,15,14,14,11,11,13,11,11,16,15,15,15,11,11,2,6,7,6,8,8,7,7,8,7,8,8,9,9,9,8,7,7,8,8,8,9,9,9,9,8,8,6,6,6,9,7,7,9,7,7,9,8,8,10,8,8,10,8,8,10,8,8,10,9,8,10,8,8,7,6,6,9,6,6,9,6,6,9,7,7,10,8,8,10,6,6,9,7,7,10,8,8,10,6,6,7,7,7,11,9,9,11,9,9,10,9,9,12,10,10,12,8,8,11,9,9,13,9,10,12,8,8,8,7,7,11,9,10,11,10,10,10,9,9,11,11,11,11,9,9,11,10,9,12,11,11,11,9,10,10,8,8,11,9,10,11,9,9,11,9,9,12,10,10,11,9,9,11,9,9,12,10,11,11,9,9,8,8,8,12,9,9,12,9,9,11,9,9,13,9,9,13,8,8,12,9,9,13,10,10,12,8,8,9,7,7,11,10,10,11,10,10,11,10,10,12,11,11,11,10,9,11,10,10,11,11,11,11,9,9,11,9,9,12,10,10,11,10,10,12,10,10,11,11,11,11,9,9,11,10,10,12,11,11,11,9,9,1,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,1,3,2,5,4,7,7,8,8,9,9,10,10,11,11,12,12,13,13,14,14,15,15,15,15,3,5,4,5,4,5,4,5,5,5,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,2,3,6,7,9,13,17,3,2,5,7,9,13,17,6,5,5,6,9,12,16,7,7,6,6,7,10,13,10,10,9,7,6,10,13,13,13,12,10,10,11,15,17,17,17,14,14,15,17,1,3,2,3,1,3,3,8,8,10,10,10,10,10,10,10,10,5,7,5,9,8,10,10,10,10,11,10,11,10,5,5,7,8,9,10,10,11,10,10,11,10,11,10,10,10,11,11,11,11,11,11,11,10,11,11,10,10,10,10,11,11,11,11,11,10,11,11,11,11,11,11,11,11,12,11,10,11,11,11,11,11,11,11,11,11,11,11,11,10,10,11,11,12,11,11,11,11,11,11,12,11,11,11,10,11,11,11,11,11,11,11,11,10,11,11,10,11,10,11,11,11,11,11,11,11,11,11,11,12,11,11,12,12,11,11,11,11,11,11,11,11,11,11,11,11,12,11,10,11,11,11,11,11,11,11,12,11,13,11,11,11,11,11,11,11,11,11,11,11,12,11,13,1,4,4,7,7,4,5,6,7,7,4,6,5,7,7,7,6,7,6,7,7,7,6,7,6,1,4,4,4,4,4,4,4,4,3,3,7,8,10,13,16,3,2,5,7,9,13,16,6,4,4,6,10,14,15,7,5,5,7,10,13,14,9,8,9,9,9,11,13,12,11,12,9,7,8,11,14,12,10,6,5,7,10,1,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,5,5,0,7,7,0,8,8,0,9,9,0,12,12,0,8,8,0,9,9,0,12,12,0,8,8,0,6,6,0,11,11,0,12,12,0,12,12,0,15,15,0,11,11,0,12,12,0,15,15,0,12,12,0,5,5,0,5,5,0,6,6,0,7,7,0,11,11,0,6,6,0,7,7,0,10,11,0,6,6,0,7,7,0,11,11,0,12,12,0,11,11,0,15,15,0,10,10,0,12,12,0,15,15,0,12,12,0,6,6,0,12,12,0,12,12,0,12,12,0,15,15,0,11,11,0,12,12,0,15,15,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,8,0,12,12,0,12,12,0,12,12,0,15,15,0,12,12,0,11,12,0,15,16,0,11,11,0,6,6,0,11,12,0,12,12,0,12,12,0,16,15,0,12,12,0,13,12,0,15,14,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,0,9,9,0,9,9,0,10,10,0,9,9,0,10,10,0,10,10,0,9,9,0,10,10,0,7,7,0,7,7,0,6,6,0,8,8,0,7,7,0,8,8,0,8,9,0,8,8,0,8,8,0,7,7,0,9,9,0,8,8,0,10,10,0,9,9,0,10,10,0,10,10,0,9,9,0,10,10,0,9,9,0,11,11,0,11,11,0,12,12,0,11,11,0,12,12,0,13,13,0,12,12,0,13,12,0,8,8,0,12,12,0,12,12,0,13,13,0,12,12,0,13,13,0,13,13,0,13,13,0,13,13,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,11,11,0,12,12,0,13,13,0,12,12,0,13,13,0,13,13,0,12,12,0,12,12,0,8,8,0,12,12,0,12,12,0,13,13,0,13,13,0,13,14,0,14,13,0,13,13,0,13,13,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,6,6,7,8,8,7,8,8,7,9,9,10,12,11,9,8,8,7,9,9,11,12,12,9,9,9,6,7,7,10,11,11,10,11,11,10,11,11,13,13,14,12,12,12,11,11,11,14,14,14,12,12,12,6,5,5,9,6,5,9,6,6,9,7,7,12,10,10,11,6,6,10,7,7,13,10,10,12,7,7,7,8,8,12,10,10,12,10,10,11,10,10,15,13,13,13,9,9,12,11,11,16,13,13,15,11,11,8,7,7,12,12,12,12,11,11,12,11,11,14,14,14,14,12,12,12,12,12,16,15,15,14,12,12,0,10,10,0,12,12,0,12,12,0,11,11,0,14,14,0,11,11,0,12,12,0,15,15,0,11,11,8,8,8,13,11,11,13,10,10,13,11,11,15,13,13,14,11,11,12,10,10,16,14,14,14,10,10,9,7,7,13,11,11,13,11,11,12,11,11,16,14,14,14,12,12,13,12,12,15,14,14,15,13,12,0,11,11,0,12,12,0,12,12,0,12,12,0,15,15,0,12,12,0,13,12,0,14,15,0,12,12,2,4,4,8,8,10,12,12,11,11,9,11,11,12,13,11,12,12,11,11,11,12,12,12,12,10,13,12,13,13,11,12,12,13,13,11,12,12,13,13,11,12,13,13,13,11,13,13,13,13,10,13,13,12,13,11,12,12,14,14,11,13,12,12,12,11,12,12,13,13,11,13,13,12,12,11,13,13,13,13,11,12,12,13,13,11,13,13,12,12,11,12,12,13,13,11,13,13,12,12,11,13,13,13,13,11,12,12,14,14,11,13,13,12,12,11,12,12,13,13,11,13,13,12,12,11,10,10,10,10,12,10,10,11,11,11,8,8,11,11,13,10,10,10,10,12,10,10,10,10,13,11,11,11,11,13,10,10,11,11,13,11,11,12,12,13,11,11,11,11,13,11,11,12,12,13,11,11,12,12,13,10,10,11,11,13,11,11,11,11,13,11,10,11,11,13,11,11,11,11,13,11,11,11,11,13,10,10,11,11,13,11,11,11,11,12,10,11,11,11,13,11,11,11,11,13,11,11,11,11,13,10,10,11,11,13,11,11,11,11,13,11,11,11,11,13,11,11,11,11,11,10,10,10,10,12,10,10,9,9,12,12,12,11,11,13,12,12,9,9,13,12,12,10,10,12,12,12,12,12,13,13,13,14,14,13,12,12,11,11,13,13,13,12,12,13,12,12,11,11,13,12,13,11,11,13,13,13,14,14,13,12,12,10,10,13,13,13,11,11,13,12,12,10,10,13,13,13,11,11,13,13,13,14,14,13,12,12,10,10,13,13,13,11,11,13,12,13,10,10,13,13,13,11,11,13,13,13,14,14,13,12,12,10,10,13,13,13,11,11,13,13,12,10,10,14,12,12,8,8,14,12,12,9,9,14,11,11,9,9,14,12,12,8,8,14,11,11,7,7,14,13,13,10,10,15,12,12,10,10,15,13,13,10,10,15,12,12,9,9,15,13,13,10,10,15,13,13,10,10,15,12,12,10,10,15,13,13,10,10,14,12,12,9,9,14,13,13,9,9,14,13,13,9,9,15,12,12,9,9,15,13,13,9,9,14,12,12,9,9,14,13,13,9,9,14,13,13,9,9,15,12,12,9,9,14,13,13,9,9,14,12,12,9,9,14,13,13,9,9,13,12,12,8,8,13,13,13,8,8,14,13,13,9,9,13,13,13,7,7,14,13,13,8,8,14,14,14,10,10,14,14,14,11,11,14,14,14,9,9,14,14,14,10,10,14,14,14,9,9,14,14,14,10,9,15,14,14,11,11,14,14,14,9,9,14,14,14,10,10,14,14,14,9,9,14,14,14,9,9,15,14,14,11,11,14,14,14,8,8,14,14,14,9,9,14,14,14,8,8,14,14,14,9,9,15,14,14,11,11,14,14,14,8,8,14,14,14,9,9,14,14,14,8,8,12,12,12,13,13,16,15,15,11,11,16,15,16,12,12,17,16,16,11,11,17,15,15,12,11,16,16,16,12,13,16,15,15,13,13,16,16,16,12,12,16,16,15,13,13,16,16,16,12,12,16,16,16,13,13,17,16,16,14,14,17,17,16,12,12,17,16,16,13,13,17,17,16,12,13,16,16,17,13,12,17,16,16,14,13,17,16,16,12,12,17,16,16,12,12,17,16,17,12,12,17,17,17,13,13,16,16,16,13,14,17,17,16,12,12,16,16,16,13,13,17,17,17,12,12,13,14,14,10,10,16,14,14,12,12,16,15,15,14,14,16,14,14,12,12,15,14,14,13,13,17,15,15,14,13,16,16,15,15,15,16,15,15,14,14,16,15,15,14,14,17,15,15,14,14,16,15,15,14,14,16,16,15,15,15,17,15,15,13,13,16,15,15,14,14,17,15,15,13,13,17,15,15,14,14,16,15,15,15,15,16,14,14,13,13,16,15,15,14,14,16,14,14,13,13,17,15,15,14,14,16,16,15,15,15,17,14,14,13,13,16,15,15,14,14,17,14,14,13,13,13,11,11,10,10,16,14,14,13,13,15,14,14,13,13,16,14,14,12,12,16,14,14,12,12,15,15,15,14,14,16,14,14,14,14,16,15,14,14,14,16,14,14,14,14,16,15,15,14,13,16,15,15,14,14,16,14,14,14,14,17,15,15,14,14,16,14,14,14,14,16,15,15,13,14,16,15,15,14,14,16,14,14,14,14,16,15,15,13,13,16,14,14,13,13,16,15,15,13,13,16,15,15,14,14,16,14,14,14,14,17,15,15,13,13,16,15,14,13,13,17,15,15,13,13,14,14,14,9,9,14,14,14,17,17,14,15,15,18,18,14,14,14,18,19,14,14,14,18,18,15,15,15,19,18,15,16,15,18,20,15,15,15,18,19,15,15,15,19,19,15,15,15,18,20,15,15,15,18,19,15,15,16,20,18,15,15,15,18,18,15,15,15,19,19,15,15,15,18,19,15,15,15,18,19,15,15,15,19,19,14,15,14,19,19,15,15,15,20,19,15,14,14,19,18,14,15,15,18,19,15,15,16,20,20,14,14,14,18,19,15,15,15,19,18,14,14,14,18,18,14,12,12,9,9,13,14,14,18,18,14,13,13,18,19,14,14,14,18,18,14,14,14,18,18,15,15,15,19,19,15,14,14,19,18,14,15,15,19,18,15,14,14,18,18,15,15,15,19,18,14,15,15,19,19,15,14,14,19,18,14,15,15,19,18,15,14,14,19,18,14,15,15,19,18,15,15,15,21,18,15,14,14,19,18,14,15,15,18,19,14,15,14,20,19,14,15,15,18,19,14,15,15,19,19,15,14,14,19,20,14,15,15,18,18,14,14,14,19,19,14,15,15,19,18,12,12,12,13,13,16,15,15,11,11,16,15,15,12,12,16,16,16,11,11,16,15,15,11,11,16,16,16,13,13,17,16,16,13,13,17,17,17,12,12,16,16,16,13,13,17,16,17,13,12,15,16,16,12,12,16,15,15,13,13,17,16,16,12,12,16,16,15,12,12,16,16,16,12,12,17,17,16,13,12,16,16,16,13,13,17,16,16,12,12,17,16,16,12,12,17,17,16,12,12,16,17,16,12,12,17,15,15,13,13,17,16,16,12,12,16,16,16,12,12,16,16,16,12,12,13,13,13,9,9,15,14,14,13,13,16,15,14,14,14,16,14,14,13,13,15,14,14,13,13,17,15,15,14,14,16,15,15,15,15,16,15,15,14,14,16,15,15,15,15,17,15,15,14,14,16,15,15,14,14,16,15,15,15,15,17,14,15,14,14,16,15,15,14,14,17,15,15,13,14,17,15,15,14,14,16,15,15,15,15,17,14,14,13,13,16,15,15,14,14,17,14,14,13,13,17,15,15,14,14,16,15,16,15,15,17,14,14,13,13,16,15,15,14,14,18,14,14,13,13,13,11,11,11,11,15,14,14,12,12,15,14,14,13,13,16,14,14,12,12,16,13,14,12,12,16,15,15,13,13,16,14,14,14,14,16,15,15,13,13,16,14,14,13,13,16,14,15,13,13,15,15,15,13,13,16,14,14,14,13,16,14,14,13,13,16,14,14,13,13,16,15,15,13,13,16,15,15,13,13,16,14,14,14,14,16,15,15,12,12,16,14,14,13,13,16,15,15,12,12,16,15,15,13,13,16,14,14,14,14,17,15,14,12,12,16,14,14,13,13,16,15,15,12,12,14,14,14,8,8,14,14,14,17,18,14,15,15,17,18,14,14,14,17,18,14,14,14,18,18,14,15,15,18,18,14,16,15,19,19,15,15,15,18,19,15,16,15,20,19,15,15,15,18,18,14,15,15,18,19,15,16,16,20,19,15,15,15,19,17,14,15,15,20,18,14,15,15,18,18,14,15,15,18,19,14,15,15,19,20,14,14,14,18,18,14,15,15,18,19,14,14,14,18,19,14,15,15,19,18,15,16,16,20,21,14,14,15,19,19,14,15,15,19,19,14,14,14,19,18,13,12,12,9,9,13,14,14,18,19,14,14,14,18,19,14,14,14,18,18,14,14,14,18,18,14,15,15,19,19,15,14,14,19,18,15,15,15,19,19,15,14,14,19,20,14,15,15,18,19,14,15,15,20,18,15,14,14,18,18,14,15,15,18,18,14,14,14,19,19,14,15,15,18,18,14,15,15,19,18,15,14,14,19,19,14,15,15,19,18,15,14,14,19,18,14,14,15,18,19,14,15,15,19,18,15,14,14,18,19,14,15,14,19,20,14,14,14,19,19,14,15,15,19,19,12,12,12,13,13,16,16,16,11,11,16,16,16,12,12,17,16,16,11,11,17,15,15,11,11,16,16,16,13,13,17,15,16,13,13,16,16,16,12,12,17,16,16,13,13,17,17,16,12,12,17,17,16,13,13,17,16,16,13,13,17,17,17,12,12,17,16,16,13,13,17,17,17,12,12,16,16,16,12,12,17,15,15,13,13,17,16,16,11,11,17,16,16,12,12,16,16,16,11,11,16,17,16,12,12,17,16,16,13,13,17,17,16,12,12,17,17,16,12,12,17,16,16,11,11,13,14,14,9,9,16,14,14,13,13,16,14,15,14,14,16,14,14,12,12,16,14,14,13,13,17,15,15,14,14,16,15,15,15,15,17,15,15,14,14,16,15,15,14,14,17,15,15,14,14,16,15,15,14,14,16,15,15,15,16,17,14,15,14,14,16,15,15,14,14,17,15,15,14,14,16,15,15,14,14,16,15,15,15,15,17,14,14,13,13,16,15,15,14,14,16,14,14,13,13,17,15,15,14,14,16,16,15,15,15,17,14,14,13,13,16,15,15,14,14,17,14,14,13,13,13,11,11,10,10,16,14,14,12,12,15,13,13,13,12,16,14,14,11,11,16,14,14,11,11,16,14,15,13,14,16,14,14,13,13,16,15,15,13,13,16,14,14,13,13,16,15,15,13,13,16,15,15,13,13,17,14,14,14,14,17,15,15,13,13,16,14,15,13,13,16,15,15,13,13,16,15,15,13,13,16,14,14,13,13,17,15,15,12,12,16,14,14,12,12,16,15,15,12,12,16,15,15,13,13,16,14,14,13,13,17,15,15,12,12,17,14,14,12,12,16,15,15,12,12,13,14,14,8,8,13,14,14,18,18,13,15,15,17,18,14,14,14,18,19,14,14,14,19,18,14,15,15,19,18,15,15,16,21,18,15,15,15,19,19,14,16,16,19,19,14,15,15,18,19,14,15,15,19,20,14,16,16,19,18,15,15,15,18,19,14,15,15,19,18,15,15,15,18,18,15,15,15,20,18,15,16,16,20,19,14,15,14,18,19,14,15,16,19,20,14,15,15,19,18,15,15,15,19,18,15,16,16,20,19,15,14,14,18,18,14,15,15,19,19,14,15,15,18,18,13,12,12,8,8,13,14,14,19,18,14,13,13,20,18,14,14,14,19,18,14,13,13,18,19,14,15,15,20,19,15,14,14,19,19,14,15,15,19,18,15,14,14,20,20,15,15,15,19,18,14,15,15,19,18,15,14,14,19,18,14,15,15,20,19,14,14,14,20,19,14,15,15,19,18,15,15,15,18,18,15,14,14,18,18,14,15,15,19,19,14,14,14,19,19,14,15,15,19,19,15,15,15,19,18,15,14,14,20,19,15,15,15,19,19,14,14,14,20,19,14,15,15,20,20,12,12,12,13,13,17,16,16,11,11,16,16,15,12,12,17,16,16,11,11,17,15,15,11,11,17,17,17,13,13,17,16,16,13,13,17,17,17,12,12,17,16,16,13,13,17,17,16,12,13,16,17,16,13,13,17,16,15,13,13,17,16,16,12,12,17,16,16,12,13,17,16,17,12,12,17,17,17,12,12,17,16,15,13,13,17,16,16,12,12,17,16,16,12,12,17,16,16,11,11,16,16,16,12,12,17,15,15,13,13,17,16,15,11,11,16,16,16,12,12,17,16,16,11,11,13,14,14,9,9,16,14,14,13,13,16,14,15,14,14,16,14,14,12,12,16,14,14,13,13,17,15,15,14,15,16,15,15,15,15,17,15,15,14,14,16,15,15,15,14,16,15,15,14,14,16,15,15,14,14,16,15,16,15,15,17,15,14,14,14,16,15,15,14,14,17,15,15,13,13,16,15,15,14,14,16,16,16,15,15,17,14,14,13,13,16,15,15,14,14,18,14,15,13,13,16,15,15,14,14,16,16,15,15,15,16,14,14,13,13,16,15,15,14,14,17,14,15,13,13,13,11,11,10,10,15,14,14,12,12,15,14,14,13,13,16,14,14,12,12,16,13,14,12,12,16,14,15,14,13,16,14,14,14,14,16,15,15,13,13,16,14,14,13,13,16,15,15,13,13,15,15,15,13,13,16,14,14,14,14,17,15,15,13,13,16,14,14,13,13,16,15,15,13,13,16,15,15,13,13,16,14,14,13,13,17,15,15,12,12,16,14,14,12,12,16,14,15,12,12,16,15,15,13,13,16,14,14,13,13,17,15,15,12,12,16,14,14,12,12,16,15,15,12,12,14,14,14,8,8,14,14,14,17,17,14,15,15,18,18,14,14,14,18,17,14,14,14,18,18,14,15,15,18,20,15,16,15,19,18,15,15,15,19,18,15,15,16,19,18,15,15,15,18,18,14,15,15,18,18,15,16,16,18,19,15,15,15,18,18,15,15,15,19,20,15,15,15,18,18,15,15,15,18,18,15,16,16,19,19,15,14,15,19,19,15,15,15,19,20,14,14,15,18,18,15,15,15,19,19,15,16,16,19,19,15,15,14,18,19,15,15,15,20,20,15,15,14,18,18,13,12,12,8,8,13,14,14,18,18,14,14,14,18,18,14,14,14,18,20,14,14,14,18,18,14,15,15,19,18,15,14,14,18,19,15,15,15,18,19,15,14,14,18,19,15,15,15,18,18,14,15,14,18,19,15,14,14,21,19,15,15,15,19,18,14,14,14,19,18,14,15,15,19,18,15,15,15,20,19,15,14,14,20,18,14,15,15,18,19,14,14,14,19,18,14,15,15,18,19,15,15,15,18,19,15,14,14,19,19,15,15,15,19,19,14,14,14,19,20,14,15,15,18,19,2,6,6,14,14,6,8,8,14,14,7,7,7,14,14,0,13,13,15,16,0,13,13,15,15,7,8,8,15,15,9,10,10,16,16,9,8,8,14,15,0,13,13,17,17,0,13,13,16,16,8,8,8,15,15,12,11,11,16,16,9,8,8,14,14,0,13,13,17,17,0,13,13,15,15,0,14,14,16,16,0,0,0,18,19,0,12,12,16,15,0,16,16,0,20,0,14,14,16,16,0,14,14,17,17,0,0,0,19,19,0,12,12,15,15,0,18,17,21,21,0,14,14,16,16,5,7,7,12,13,9,10,9,14,14,11,10,10,14,14,0,0,0,18,17,0,20,21,18,18,9,10,10,14,14,12,12,12,17,16,12,10,10,14,14,0,20,20,18,17,0,21,21,17,17,11,10,10,14,14,15,13,13,18,18,13,11,11,14,14,0,20,0,18,18,0,20,21,18,17,0,21,0,18,19,0,0,0,0,21,0,21,20,16,17,0,0,0,21,21,0,0,0,20,18,0,20,0,17,18,0,0,0,0,0,0,0,20,16,17,0,0,0,20,0,0,0,0,18,18,6,6,6,13,13,8,5,5,11,11,9,6,6,13,13,0,9,9,12,12,0,10,10,14,14,9,7,7,13,13,12,9,9,13,13,10,6,6,13,13,0,10,10,14,14,0,10,10,13,13,9,7,7,13,13,13,10,10,13,13,11,6,6,13,13,0,10,10,15,15,0,10,10,13,13,0,12,11,15,15,0,20,19,17,16,0,9,9,13,13,0,13,13,20,19,0,11,11,13,13,0,11,11,15,15,0,20,19,17,17,0,10,10,13,13,0,14,15,0,21,0,12,12,13,13,0,10,10,12,12,0,11,11,15,15,0,11,11,15,15,0,15,15,20,20,0,16,16,0,0,0,11,11,15,15,0,14,14,17,17,0,11,11,15,15,0,15,15,20,21,0,16,16,21,21,0,12,12,15,15,0,15,15,18,20,0,11,11,16,15,0,15,15,21,21,0,16,16,0,21,0,16,16,0,0,0,0,0,0,0,0,14,14,21,21,0,17,18,0,0,0,16,17,20,0,0,16,16,0,0,0,0,0,0,0,0,15,15,20,20,0,19,18,0,21,0,18,17,0,0,0,10,10,11,11,0,10,10,10,10,0,11,11,12,12,0,11,11,9,9,0,13,13,12,12,0,11,11,12,12,0,13,13,12,12,0,10,10,12,12,0,12,12,13,13,0,12,12,12,12,0,11,11,12,12,0,13,13,12,12,0,10,10,12,12,0,13,13,13,13,0,12,12,12,12,0,14,13,13,13,0,19,21,15,15,0,12,11,12,12,0,16,15,19,19,0,13,13,11,11,0,13,13,13,13,0,0,21,15,16,0,12,12,12,12,0,16,16,19,21,0,13,13,12,12,7,7,7,16,16,11,9,9,16,16,12,9,9,16,16,0,13,13,16,16,0,14,14,17,16,11,9,9,16,16,14,12,11,17,17,13,8,9,15,15,0,13,13,19,19,0,13,13,16,15,12,10,10,17,17,15,12,12,19,18,14,9,9,17,16,0,14,14,18,0,0,14,13,16,16,0,14,15,18,17,0,21,0,19,21,0,12,12,16,16,0,16,16,0,0,0,14,14,16,16,0,14,14,18,18,0,0,21,20,0,0,13,13,16,17,0,18,18,0,0,0,15,14,17,16,8,7,7,14,14,11,10,10,15,15,13,10,10,15,15,0,21,20,19,19,0,21,0,17,18,11,10,10,15,16,14,12,12,18,18,14,11,11,15,14,0,21,20,18,19,0,0,21,18,18,12,11,11,16,16,16,14,14,18,20,14,11,11,16,15,0,20,20,19,19,0,0,20,18,18,0,21,0,18,19,0,0,0,0,0,0,20,20,17,18,0,0,0,20,20,0,0,0,19,19,0,0,0,20,18,0,0,0,0,0,0,0,21,18,18,0,21,21,0,21,0,0,0,19,20,11,9,9,14,14,13,10,10,14,14,13,11,11,15,15,0,13,13,13,13,0,14,14,16,16,13,11,11,15,15,16,12,12,15,15,14,10,10,14,14,0,14,14,16,16,0,14,14,15,15,13,10,10,15,15,17,13,14,15,16,15,10,10,15,15,0,14,14,17,16,0,14,14,15,15,0,15,15,17,17,0,0,21,18,18,0,13,13,15,15,0,16,16,21,20,0,14,14,15,14,0,15,14,16,17,0,0,20,20,19,0,13,13,15,15,0,19,18,0,0,0,15,15,15,15,0,11,11,14,14,0,12,12,16,16,0,12,12,16,16,0,15,16,21,21,0,16,17,21,0,0,12,12,17,16,0,14,14,18,19,0,11,11,16,16,0,15,15,20,21,0,16,16,21,0,0,12,12,17,16,0,15,15,19,19,0,12,12,16,17,0,16,15,0,0,0,16,16,0,0,0,17,17,0,21,0,0,0,0,0,0,14,15,20,0,0,17,17,0,0,0,17,17,0,0,0,17,16,0,0,0,0,0,0,0,0,15,15,0,0,0,18,18,0,0,0,18,17,0,0,0,11,11,14,14,0,12,12,15,15,0,12,12,15,15,0,13,13,14,14,0,14,14,17,17,0,12,12,16,16,0,14,14,16,16,0,11,11,15,15,0,13,13,16,17,0,13,13,16,16,0,12,12,15,15,0,14,14,17,16,0,11,11,15,15,0,14,14,17,17,0,13,13,16,16,0,15,15,17,18,0,21,20,20,21,0,12,12,15,15,0,16,16,20,21,0,14,14,15,15,0,14,14,17,17,0,0,0,18,19,0,12,13,15,15,0,18,17,21,0,0,14,15,15,15,8,8,8,16,16,12,10,10,16,16,13,9,9,16,16,0,14,14,18,17,0,14,14,16,17,12,10,10,18,17,14,12,11,18,18,14,9,9,16,16,0,13,13,18,18,0,13,13,17,16,12,9,9,16,17,17,13,13,17,17,14,9,9,15,15,0,14,14,20,19,0,13,13,16,16,0,15,15,19,18,0,0,0,20,19,0,12,13,17,17,0,16,16,20,0,0,14,14,16,17,0,14,14,19,18,0,0,0,20,20,0,13,13,16,16,0,18,17,0,0,0,15,15,16,16,9,7,7,14,14,12,10,10,15,15,13,10,10,15,15,0,21,0,18,19,0,20,21,19,18,12,10,10,16,15,15,13,13,18,18,14,11,11,15,15,0,0,0,19,18,0,0,21,18,18,13,11,11,15,15,16,14,14,17,19,15,11,11,15,15,0,21,21,20,18,0,0,21,18,18,0,0,21,21,19,0,0,0,0,0,0,19,20,18,17,0,0,0,21,21,0,21,0,20,18,0,0,21,19,19,0,0,0,0,0,0,20,21,17,17,0,0,0,0,0,0,21,0,18,20,0,10,10,14,14,0,11,11,15,15,0,11,11,15,15,0,14,14,15,15,0,15,15,16,16,0,11,12,16,16,0,13,13,16,16,0,11,11,15,15,0,14,14,17,17,0,14,14,15,15,0,11,11,16,15,0,14,14,15,15,0,11,11,15,15,0,15,15,17,17,0,14,14,15,15,0,16,16,18,18,0,0,0,20,19,0,14,13,16,15,0,17,17,21,0,0,15,15,15,15,0,16,15,17,16,0,20,0,20,18,0,13,14,15,15,0,19,18,0,21,0,15,15,15,15,0,11,11,14,14,0,12,12,16,16,0,12,12,16,16,0,16,15,20,21,0,17,16,0,0,0,12,12,16,16,0,14,14,18,18,0,11,11,16,16,0,15,15,21,20,0,16,16,0,0,0,12,12,16,17,0,15,14,19,19,0,11,12,16,16,0,15,15,21,0,0,16,16,0,0,0,16,17,0,0,0,0,0,0,0,0,15,15,21,0,0,17,17,0,0,0,17,17,0,0,0,17,16,0,0,0,0,0,0,0,0,15,15,0,20,0,19,20,0,0,0,17,17,0,0,0,12,12,15,15,0,12,12,15,15,0,12,12,16,16,0,13,13,15,15,0,15,15,17,17,0,13,13,17,16,0,14,14,17,17,0,11,11,16,16,0,14,14,17,17,0,13,13,16,16,0,12,12,16,16,0,15,15,16,17,0,11,11,15,16,0,14,14,17,17,0,13,14,16,16,0,15,15,18,18,0,21,20,20,19,0,13,13,16,17,0,16,16,0,0,0,14,14,16,16,0,15,15,18,18,0,0,0,20,19,0,13,13,16,16,0,17,17,0,0,0,14,14,16,16,0,11,11,16,16,0,13,13,18,17,0,13,13,17,17,0,16,16,17,17,0,16,16,17,18,0,12,12,17,17,0,15,15,18,18,0,12,12,16,16,0,16,16,19,19,0,15,15,16,17,0,12,12,17,17,0,17,17,18,18,0,12,12,17,17,0,16,16,19,19,0,15,16,17,17,0,16,16,18,17,0,0,0,21,21,0,13,13,16,16,0,17,17,0,20,0,15,15,16,17,0,16,16,19,18,0,0,21,20,21,0,14,14,17,16,0,20,0,0,0,0,15,16,16,17,0,9,9,14,14,0,13,13,16,16,0,14,14,15,15,0,0,20,19,19,0,0,0,19,19,0,12,12,15,15,0,15,16,19,18,0,14,14,15,15,0,21,0,18,18,0,20,0,17,18,0,13,13,16,16,0,17,17,17,19,0,14,14,16,15,0,21,20,20,19,0,0,0,19,19,0,0,0,19,18,0,0,0,0,0,0,20,20,17,18,0,0,0,21,21,0,0,0,18,18,0,21,0,18,19,0,0,0,0,0,0,20,21,18,18,0,0,0,20,21,0,0,0,19,19,0,18,18,15,15,0,20,21,17,17,0,19,21,17,17,0,0,0,17,18,0,0,0,20,19,0,19,19,17,17,0,0,0,18,18,0,19,20,16,17,0,0,21,20,20,0,19,20,19,18,0,19,20,16,16,0,0,0,18,19,0,19,20,17,17,0,0,21,0,20,0,21,21,17,19,0,20,0,19,20,0,0,0,20,0,0,19,18,17,16,0,0,0,0,0,0,0,20,17,17,0,20,21,18,20,0,0,0,0,21,0,19,20,17,17,0,0,0,0,0,0,20,21,17,17,0,11,11,14,14,0,13,13,16,17,0,13,13,16,16,0,17,17,0,21,0,18,17,21,0,0,13,13,16,16,0,15,15,18,18,0,12,12,16,16,0,17,16,21,0,0,17,17,0,0,0,12,12,17,17,0,17,17,19,21,0,13,12,16,16,0,17,17,0,0,0,17,17,0,0,0,18,17,0,21,0,0,0,0,0,0,15,15,20,0,0,20,18,0,0,0,17,18,0,0,0,16,17,0,0,0,0,0,0,0,0,15,15,0,0,0,19,19,0,0,0,18,18,0,0,0,14,14,18,18,0,16,16,0,21,0,16,16,21,21,0,17,17,0,20,0,17,17,20,0,0,16,15,0,0,0,20,20,0,0,0,15,15,20,20,0,17,17,21,0,0,17,18,20,20,0,15,15,20,20,0,18,18,0,0,0,15,15,19,20,0,17,18,0,0,0,17,17,20,20,0,18,17,21,0,0,0,0,0,21,0,15,15,20,20,0,19,19], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+341160);
/* memory initializer */ allocate([17,17,21,0,0,17,17,0,0,0,0,0,21,0,0,15,15,19,19,0,20,21,0,0,0,18,17,21,21,0,12,12,16,16,0,14,14,17,17,0,13,13,17,18,0,16,16,18,17,0,16,16,18,18,0,13,13,18,18,0,15,16,19,18,0,13,13,16,16,0,16,16,20,18,0,16,16,17,17,0,12,13,17,17,0,17,16,18,18,0,12,12,16,16,0,17,16,20,19,0,16,16,16,16,0,16,17,18,20,0,0,0,21,20,0,14,14,17,16,0,19,18,0,20,0,16,16,17,16,0,16,16,17,18,0,0,21,21,21,0,14,14,16,16,0,20,20,21,0,0,16,16,16,16,0,10,10,14,14,0,14,14,15,16,0,14,14,15,15,0,0,21,18,18,0,0,21,18,19,0,13,13,16,16,0,16,16,18,18,0,14,14,15,15,0,21,0,18,18,0,21,0,18,18,0,13,13,16,16,0,17,17,19,20,0,14,14,15,15,0,0,0,18,20,0,0,21,18,18,0,0,21,19,18,0,0,0,0,0,0,20,21,18,17,0,0,0,21,21,0,0,0,19,19,0,21,0,18,19,0,0,0,0,0,0,21,20,17,17,0,0,21,20,0,0,0,0,19,19,0,19,20,15,16,0,0,20,18,17,0,20,21,17,18,0,21,0,18,18,0,0,0,19,19,0,20,20,17,18,0,0,0,18,19,0,20,20,18,17,0,0,0,0,20,0,0,21,17,18,0,20,21,17,17,0,0,0,18,18,0,19,19,17,17,0,0,0,21,21,0,20,20,17,17,0,0,0,21,19,0,0,0,20,19,0,21,20,17,18,0,0,0,0,0,0,0,20,18,17,0,21,20,18,18,0,0,0,20,21,0,20,20,17,17,0,0,0,0,0,0,20,0,17,17,0,11,11,13,14,0,13,13,16,16,0,13,13,16,16,0,17,17,0,0,0,17,18,0,0,0,13,13,16,16,0,15,16,18,18,0,13,13,16,17,0,16,17,20,0,0,17,18,20,0,0,13,13,17,17,0,16,16,20,21,0,13,13,16,16,0,17,17,21,0,0,17,18,0,0,0,17,18,0,21,0,0,0,0,0,0,15,15,20,0,0,19,19,0,0,0,17,17,0,0,0,18,17,21,20,0,0,0,0,0,0,16,16,20,21,0,21,20,0,21,0,19,21,0,0,0,15,15,0,0,0,16,17,0,19,0,16,16,0,0,0,17,17,0,0,0,19,18,0,0,0,16,16,20,20,0,20,18,21,0,0,15,15,21,21,0,18,18,0,0,0,18,19,0,0,0,16,15,0,21,0,20,19,0,0,0,16,16,0,0,0,20,18,0,21,0,17,18,21,0,0,18,19,0,0,0,0,0,0,0,0,16,16,20,20,0,19,20,0,0,0,17,17,0,0,0,18,17,20,21,0,0,0,0,0,0,16,16,0,20,0,20,22,0,0,0,18,18,0,22,2,3,3,3,3,3,3,1,6,6,6,8,8,7,8,8,7,9,8,10,11,11,9,8,8,7,8,8,11,11,11,9,8,8,6,7,7,10,10,10,10,10,10,10,10,10,14,13,13,12,11,11,10,10,10,14,14,13,13,11,11,6,6,6,8,5,5,8,7,7,8,7,7,11,9,9,9,7,7,8,7,7,12,10,10,10,7,7,7,8,8,12,11,11,12,10,10,11,10,10,14,13,13,13,10,10,11,10,11,16,14,14,13,10,10,7,8,7,12,12,12,12,11,11,12,11,11,16,14,15,13,12,12,11,11,11,17,15,14,14,13,13,10,9,9,13,11,11,13,11,11,12,11,11,16,14,13,14,11,11,12,11,11,16,15,14,14,11,11,7,8,8,12,11,11,12,10,10,12,10,10,16,14,13,13,11,11,12,10,10,16,14,14,13,10,10,8,8,8,12,12,12,12,11,11,12,11,11,16,14,15,14,12,12,12,11,11,16,15,15,14,12,12,10,10,10,13,11,11,13,11,11,12,12,12,16,14,14,14,11,11,12,11,11,17,14,15,14,11,11,2,7,7,7,8,8,7,7,7,7,8,8,8,8,9,8,7,7,8,8,8,9,9,9,9,7,7,6,6,6,9,7,7,9,7,7,9,8,8,10,8,8,10,8,8,10,8,8,10,8,8,10,8,8,7,6,6,9,6,6,9,6,6,9,7,7,10,8,8,9,6,6,9,7,7,10,8,8,9,7,7,7,8,8,11,9,9,11,9,9,11,9,9,12,9,9,12,8,8,12,9,9,12,10,9,12,8,8,8,7,7,10,9,9,11,9,9,11,9,9,11,11,10,11,9,9,11,10,9,11,10,11,11,9,9,10,8,8,11,9,9,11,9,9,11,9,9,11,10,10,11,9,9,11,9,9,11,10,10,11,9,9,9,8,8,12,9,9,12,9,9,11,9,9,12,9,9,12,8,8,12,9,9,12,9,9,12,8,8,9,7,7,11,9,10,11,10,9,11,9,9,11,11,11,11,9,9,11,10,10,11,11,11,11,9,9,10,9,9,11,9,9,11,10,10,11,10,9,11,10,10,11,9,9,11,10,10,11,10,11,11,9,9,1,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,7,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,1,3,2,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,12,14,14,14,15,15,3,4,4,5,4,5,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,2,3,6,7,10,14,16,3,2,5,7,11,14,17,6,5,5,7,10,12,14,7,7,6,6,7,9,13,10,11,9,6,6,9,11,15,15,13,10,9,10,12,18,18,16,14,12,13,16,1,3,2,3,1,3,4,7,7,8,8,9,9,9,10,10,10,5,6,5,8,7,9,8,9,9,10,9,11,10,5,5,7,7,8,8,9,9,9,9,10,10,11,8,9,8,10,9,10,9,10,9,11,10,11,10,8,8,9,9,10,9,10,9,11,10,11,10,11,10,11,11,11,11,11,11,11,11,11,11,11,11,10,11,11,11,12,11,11,11,11,11,11,10,12,12,12,12,12,12,12,11,12,12,12,11,11,11,12,12,12,12,12,12,12,11,12,11,12,11,11,13,12,12,12,13,12,12,12,12,11,12,11,11,13,13,13,12,12,12,12,12,12,11,11,11,10,13,13,13,12,13,12,13,11,13,10,12,11,11,13,13,12,13,12,12,12,12,11,12,11,11,11,1,4,4,6,6,5,5,5,7,5,5,5,5,6,7,7,6,7,7,7,6,7,7,7,7,1,4,4,4,4,4,4,4,4,4,5,7,8,10,13,14,4,2,4,6,8,11,12,7,4,3,5,8,12,14,8,5,4,4,8,12,12,9,7,7,7,9,10,11,13,11,11,9,7,8,10,13,11,10,6,5,7,9,1,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,4,4,0,7,7,0,8,8,0,9,9,0,12,12,0,8,8,0,9,9,0,12,12,0,8,8,0,6,6,0,11,11,0,11,11,0,12,12,0,14,14,0,11,11,0,12,12,0,14,14,0,11,11,0,6,6,0,6,5,0,7,6,0,7,7,0,10,10,0,6,6,0,7,7,0,10,10,0,7,7,0,7,7,0,10,10,0,11,11,0,11,11,0,14,14,0,10,10,0,12,12,0,14,14,0,12,12,0,6,6,0,11,11,0,11,11,0,12,12,0,14,14,0,11,11,0,12,12,0,15,15,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,8,0,11,11,0,11,11,0,12,12,0,15,15,0,12,12,0,11,11,0,15,15,0,11,11,0,6,6,0,11,11,0,12,12,0,12,12,0,15,15,0,11,11,0,12,12,0,14,14,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,3,3,0,8,8,0,8,8,0,10,10,0,9,9,0,10,10,0,10,10,0,9,9,0,10,10,0,7,7,0,7,7,0,7,7,0,8,8,0,8,8,0,8,8,0,9,9,0,8,8,0,8,8,0,7,7,0,8,8,0,8,8,0,10,10,0,9,9,0,9,9,0,10,10,0,9,9,0,10,10,0,8,8,0,11,11,0,11,11,0,12,12,0,11,11,0,12,12,0,12,12,0,12,12,0,12,12,0,8,8,0,11,11,0,11,11,0,13,12,0,12,12,0,13,12,0,13,13,0,12,12,0,13,13,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,8,0,11,11,0,11,11,0,13,12,0,12,12,0,12,12,0,12,12,0,11,11,0,12,12,0,8,8,0,12,12,0,12,12,0,13,13,0,12,12,0,13,13,0,13,13,0,12,13,0,13,13,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,6,6,6,7,7,7,8,8,7,8,8,10,11,11,9,8,8,7,9,9,11,12,12,9,8,8,6,7,7,9,11,11,10,11,11,10,11,11,13,13,13,11,12,12,10,11,11,13,14,14,12,12,12,6,6,6,8,6,6,8,6,6,9,7,7,12,10,10,10,6,6,9,7,7,12,10,10,11,7,6,7,8,8,12,10,10,12,10,10,11,10,10,15,13,13,13,10,10,12,11,11,15,13,13,14,11,11,8,7,7,12,11,11,12,11,11,11,11,11,14,14,14,13,12,12,12,11,11,16,15,15,14,12,12,0,10,10,0,11,11,0,12,12,0,11,11,0,14,14,0,11,11,0,11,11,0,15,15,0,11,11,7,8,8,13,10,10,12,10,10,12,11,11,15,13,13,14,11,11,12,10,10,16,14,14,14,10,10,8,7,7,12,11,11,13,11,11,12,11,11,15,14,14,14,12,12,13,12,12,15,14,14,15,12,12,0,11,11,0,12,12,0,12,12,0,12,12,0,15,15,0,12,12,0,12,12,0,15,14,0,12,12,2,3,4,7,7,10,12,12,12,12,10,11,11,13,13,11,12,12,11,11,12,12,12,12,12,11,13,13,13,13,12,12,12,13,14,12,13,13,13,13,12,13,13,13,13,12,13,13,13,13,11,13,13,13,13,12,12,12,14,14,12,13,13,12,12,12,12,13,13,13,12,13,13,13,13,12,13,13,13,13,12,12,12,14,14,12,13,13,12,12,12,13,13,13,13,12,13,13,12,12,12,13,13,13,13,12,12,12,14,14,12,13,13,12,12,12,13,13,13,13,12,13,13,12,12,10,10,11,10,10,11,11,11,11,11,11,9,9,10,10,12,11,11,10,10,12,10,10,10,10,13,12,12,12,12,13,11,11,11,11,13,12,12,12,12,13,11,11,11,11,13,12,12,12,12,13,12,12,12,12,13,11,11,11,11,13,12,12,12,12,13,11,11,11,11,13,12,12,11,11,13,12,12,11,11,13,11,11,11,11,13,12,12,11,11,13,11,11,11,11,13,12,12,11,11,13,12,12,11,11,13,11,11,11,11,13,12,12,11,11,13,11,11,11,11,13,12,12,11,11,11,11,11,10,10,11,11,11,9,9,11,12,12,11,11,12,12,12,9,9,13,13,13,10,10,13,13,13,11,11,13,13,13,14,14,13,13,13,11,10,13,13,14,12,12,13,13,13,11,11,13,13,13,11,11,13,13,13,14,14,13,13,13,10,10,13,13,13,11,11,13,13,13,10,10,13,14,13,11,11,13,14,14,14,14,13,13,13,10,10,13,14,14,11,11,13,13,13,10,10,13,14,14,11,11,13,13,13,14,14,14,13,13,10,10,13,14,14,11,11,13,13,13,10,10,14,12,12,9,9,14,12,12,9,9,14,11,11,9,9,14,12,12,8,8,14,11,11,7,7,15,13,13,10,10,15,12,12,10,10,15,13,13,10,10,15,12,12,10,10,15,13,13,10,10,15,13,13,10,10,15,12,12,10,10,15,13,13,10,10,15,12,12,10,10,15,13,13,10,10,15,13,13,10,10,15,12,12,10,10,15,13,13,9,9,15,12,12,9,9,14,13,13,9,9,15,13,13,10,10,15,12,12,10,10,15,13,13,9,9,15,12,12,9,9,15,13,13,9,9,13,12,12,9,9,13,13,13,8,8,13,13,13,9,9,13,13,13,7,7,14,13,13,8,8,14,14,14,10,10,15,14,14,11,11,14,14,14,9,9,15,14,14,10,10,15,14,14,9,9,14,14,14,10,10,15,14,14,11,11,15,14,14,9,9,14,14,14,10,10,14,14,14,9,9,15,14,15,10,10,15,14,14,11,11,14,14,14,9,9,14,14,14,9,9,14,14,14,8,8,15,14,14,10,10,15,14,14,11,11,14,14,14,9,9,15,14,14,9,9,14,14,14,8,8,12,12,12,13,13,16,16,16,11,11,17,16,16,12,12,17,16,16,11,11,17,16,16,11,11,17,17,16,13,13,17,16,16,13,13,18,17,16,12,12,17,16,16,13,13,17,16,17,12,12,18,17,17,13,13,17,16,16,14,14,18,17,17,12,12,18,16,16,13,13,17,17,17,13,12,17,17,17,13,13,17,16,16,13,13,18,17,17,12,12,17,16,16,13,12,17,17,17,12,12,18,17,17,13,13,18,16,16,14,14,18,17,17,12,12,17,17,17,13,13,18,17,18,12,12,13,14,14,10,10,16,14,14,13,13,17,15,15,14,14,17,14,14,12,13,16,14,14,13,13,17,15,15,14,14,16,16,16,15,15,17,15,15,14,14,17,16,16,14,15,17,15,15,14,14,17,15,16,14,14,17,16,16,15,15,17,15,15,13,13,17,15,15,14,14,18,15,15,13,14,17,15,15,14,14,16,16,16,15,15,17,15,15,13,13,17,15,15,14,14,17,15,15,13,13,17,15,15,14,14,16,16,16,15,15,17,15,15,13,13,17,15,15,14,14,18,15,15,13,13,13,11,11,10,10,16,14,14,13,12,16,14,14,13,13,16,15,14,12,12,16,14,14,12,12,16,15,15,14,14,16,14,14,14,14,17,15,15,13,13,16,15,15,14,14,17,15,15,13,14,17,15,15,14,14,17,15,14,14,14,17,15,15,13,13,17,15,15,14,14,17,15,15,13,13,17,15,15,14,14,17,14,14,14,14,17,15,15,13,13,17,15,15,13,13,17,15,15,13,13,17,15,15,14,14,17,15,15,14,14,17,15,15,13,13,17,15,15,13,13,17,15,15,13,13,14,14,15,8,8,14,14,14,19,19,14,15,15,18,19,14,14,14,19,18,14,14,14,19,19,15,15,15,19,18,15,16,16,19,19,15,15,15,19,19,15,16,16,20,19,15,15,15,19,19,15,15,15,19,19,16,16,16,20,19,15,15,15,19,18,15,16,16,20,19,15,15,15,18,18,15,15,15,19,20,15,16,16,19,19,15,15,15,20,19,15,15,15,20,19,15,15,15,19,18,15,15,15,19,19,15,16,16,19,20,15,15,15,19,19,15,15,15,19,20,15,15,15,19,19,14,12,12,9,9,14,14,14,19,19,14,14,14,19,19,14,14,15,20,19,15,14,14,18,19,15,15,15,19,19,15,15,14,20,19,15,15,15,20,19,15,15,14,20,19,15,15,15,20,19,15,15,15,19,20,15,14,14,19,20,15,15,15,20,20,15,14,14,20,19,15,15,15,19,19,15,15,15,19,19,15,14,14,19,19,15,15,15,19,20,15,15,15,20,20,15,15,15,19,19,15,15,15,20,19,16,14,14,19,19,15,15,15,20,19,15,14,15,20,19,14,15,15,20,19,12,12,12,13,13,16,16,16,11,11,16,16,16,12,12,17,16,16,11,11,17,15,16,11,11,17,17,17,13,13,18,16,17,13,13,18,17,17,13,12,17,16,17,13,13,17,17,17,13,13,16,16,16,12,12,17,16,16,13,13,17,16,16,12,12,17,16,16,12,13,17,17,17,12,12,17,17,17,13,13,18,16,16,13,13,18,17,17,12,12,18,17,17,12,12,17,17,17,12,12,17,17,17,12,12,17,16,16,13,13,17,17,17,12,12,17,16,16,12,12,17,17,17,12,12,13,14,14,9,9,16,14,14,13,13,16,15,15,14,14,17,14,14,13,13,16,14,14,13,13,17,15,15,15,15,16,16,16,15,15,17,15,15,14,14,17,15,15,15,15,17,15,15,14,14,17,15,15,14,14,16,16,16,15,15,17,15,15,14,14,17,15,15,14,14,17,15,15,14,14,17,15,15,14,14,16,16,16,15,15,18,15,15,14,13,17,15,15,14,14,17,15,15,13,13,17,15,15,14,14,16,16,16,15,15,17,15,15,14,13,17,15,15,14,14,17,15,15,13,13,13,11,11,11,11,16,14,14,12,12,16,14,14,13,13,16,15,14,12,12,17,14,14,12,12,17,15,15,13,13,17,14,14,14,14,17,15,15,13,13,17,14,15,14,13,17,15,15,13,13,16,15,15,13,13,16,14,14,14,14,17,15,15,13,13,16,14,14,13,13,16,15,15,13,13,17,15,15,13,13,17,14,14,14,14,17,15,15,12,12,17,15,15,13,13,17,15,15,12,12,16,15,15,13,13,17,14,14,13,14,17,15,15,12,12,17,14,14,13,13,17,15,15,12,12,14,14,14,8,8,14,14,14,18,18,14,15,15,19,19,14,14,14,19,19,14,15,14,18,19,15,15,15,18,19,15,16,16,20,20,15,15,15,19,20,15,16,16,19,20,15,15,15,19,20,15,15,16,19,19,15,16,16,20,20,15,15,15,20,19,15,16,16,20,19,15,15,15,19,20,15,15,15,19,19,15,16,16,20,19,15,15,15,19,19,15,16,15,20,19,15,15,15,19,19,15,15,15,19,20,15,16,16,20,20,15,15,15,19,19,15,15,15,20,20,15,15,15,19,19,14,12,12,9,9,14,14,14,18,18,14,14,14,19,20,14,14,14,18,18,14,14,14,18,19,15,15,15,19,20,15,14,14,19,19,15,15,15,19,19,15,14,15,19,19,15,15,15,18,20,15,15,15,19,19,15,14,14,19,19,15,15,15,20,19,15,15,14,20,20,15,15,15,19,19,15,15,15,19,19,15,14,14,19,19,15,15,15,19,19,15,14,14,19,20,14,15,15,19,19,15,15,15,19,19,15,14,14,20,19,15,15,15,19,19,15,14,14,20,19,15,15,15,19,19,13,12,12,13,13,17,17,16,11,11,16,16,16,12,12,17,17,16,11,11,17,16,16,11,11,17,17,17,13,13,17,16,16,13,13,18,17,17,12,12,17,16,16,13,13,18,17,17,12,12,18,17,17,13,13,18,16,17,13,13,17,17,17,12,12,18,17,17,13,13,18,17,17,12,12,17,16,17,12,12,17,16,16,13,13,17,16,16,11,11,17,16,16,12,12,17,17,17,11,11,17,17,17,12,12,18,16,16,13,13,18,17,17,12,11,17,16,16,12,12,18,17,17,11,11,13,14,14,9,9,16,14,14,13,13,16,15,15,14,14,17,14,14,12,12,16,14,14,13,13,17,15,15,14,14,17,16,16,15,16,18,15,15,14,14,17,15,15,14,14,17,15,15,14,14,18,15,15,14,14,16,16,16,15,16,18,15,15,14,14,17,16,15,14,14,18,15,15,14,14,17,15,15,14,14,17,16,16,15,15,18,14,15,13,13,17,15,15,14,14,18,15,15,13,13,17,15,15,14,14,17,16,15,15,15,17,15,15,13,13,17,15,15,14,14,18,15,15,13,13,13,11,11,10,10,16,14,14,12,12,16,14,14,12,12,17,14,15,11,11,17,14,14,11,11,17,15,15,13,13,17,14,14,14,13,17,15,15,13,13,16,15,15,13,13,17,15,15,13,13,17,15,15,13,13,17,14,14,14,14,17,15,15,13,13,17,14,15,13,13,16,15,15,13,13,17,15,15,13,13,17,14,14,13,13,17,15,15,12,12,16,14,14,12,12,17,15,15,12,12,17,15,15,13,13,17,14,14,13,13,17,15,15,12,12,17,14,14,12,12,17,15,15,12,12,13,15,14,8,8,14,14,14,19,19,14,15,15,18,19,14,14,14,18,19,14,15,14,19,19,15,16,15,19,19,15,16,16,19,20,15,15,15,19,19,15,16,16,19,19,15,16,16,19,19,15,15,15,19,19,15,16,16,20,20,15,15,15,19,19,15,15,15,19,19,15,15,15,19,19,15,15,15,19,19,15,16,16,20,19,15,15,15,19,19,15,15,15,19,19,15,15,15,19,19,15,16,15,19,19,15,16,16,21,19,15,15,15,20,20,15,15,15,20,21,15,15,15,19,20,14,12,12,8,8,14,14,14,19,19,14,13,13,19,19,14,14,14,19,19,14,13,14,19,19,15,15,15,20,20,15,14,14,20,19,15,15,15,19,20,15,14,14,19,20,15,15,15,20,19,15,15,15,19,20,15,14,14,20,20,15,15,15,20,19,15,14,14,19,19,15,15,15,19,19,15,15,15,20,19,15,14,14,21,19,15,15,15,20,21,15,14,14,21,19,15,15,15,19,19,15,15,15,20,20,15,14,14,19,21,15,15,15,19,19,15,14,14,19,20,15,15,15,19,19,13,12,12,13,13,17,16,16,11,11,17,16,15,12,12,18,16,16,11,11,17,16,16,11,11,18,17,17,13,13,18,16,16,13,13,17,17,17,12,13,18,17,16,13,13,18,17,17,13,13,17,17,17,13,13,17,16,16,13,13,18,16,17,12,12,17,16,16,13,12,17,17,17,12,12,18,17,17,13,12,18,16,16,13,13,18,17,17,12,12,17,16,16,12,12,17,17,17,11,11,17,16,16,12,12,17,16,16,13,13,17,16,16,11,11,17,16,16,12,12,17,17,17,11,11,13,14,14,9,9,16,14,14,13,13,16,15,15,14,14,17,14,14,12,12,16,14,14,13,13,17,15,15,14,14,17,15,16,15,15,17,15,15,14,14,17,15,16,14,15,18,15,15,14,14,17,15,15,14,14,16,16,16,15,15,18,15,15,13,14,17,15,15,14,14,18,15,15,14,14,17,15,15,14,14,17,16,16,15,15,17,15,15,13,13,17,15,15,14,14,18,15,15,13,13,17,15,15,14,14,17,16,16,15,15,17,15,15,13,13,17,15,15,14,14,18,15,15,13,13,13,11,11,10,10,16,14,14,12,12,16,14,14,13,13,17,14,14,11,11,17,14,14,12,12,17,15,15,14,14,17,14,14,14,14,17,15,15,13,13,17,15,14,13,13,16,15,15,13,13,16,15,15,13,13,17,14,14,14,14,17,15,15,13,13,17,14,14,13,13,16,15,15,13,13,16,15,15,13,13,17,14,14,13,13,17,15,15,12,12,17,14,14,12,12,16,15,15,12,12,17,15,15,13,13,17,14,14,13,13,17,15,15,12,12,17,14,14,12,12,16,15,15,12,12,14,14,14,8,8,14,14,14,18,18,14,15,15,19,18,14,14,14,18,18,14,14,14,18,19,15,16,15,19,19,15,17,16,20,20,15,15,15,19,19,15,16,16,19,19,15,15,15,19,19,15,16,15,18,19,15,16,16,20,20,15,15,15,19,19,15,16,16,19,20,15,15,15,19,19,15,15,16,19,19,15,16,16,20,20,15,15,15,19,19,15,15,15,19,20,15,15,15,19,19,15,15,15,19,19,15,16,16,20,20,15,15,15,19,20,15,16,16,20,20,15,15,15,19,19,13,12,12,8,8,14,14,14,19,20,14,14,14,19,19,14,14,14,18,19,14,14,14,19,20,15,15,15,19,20,15,14,14,21,20,15,15,15,20,20,15,15,14,19,19,15,15,15,19,19,15,15,15,19,19,15,14,14,19,20,15,15,15,19,20,15,14,14,19,19,15,15,15,19,19,15,15,15,19,19,16,14,14,19,19,15,15,15,20,20,15,14,14,21,19,15,15,15,19,19,15,15,15,19,20,16,14,14,19,20,15,15,15,19,19,15,14,14,19,19,15,15,15,20,19,2,6,6,14,14,6,7,7,14,14,7,7,7,14,14,0,13,13,16,16,0,13,13,15,14,7,8,8,15,15,9,10,10,16,16,9,8,8,15,15,0,13,13,17,16,0,13,13,15,16,8,8,8,15,15,12,11,11,16,16,9,8,8,14,14,0,13,13,17,18,0,13,13,15,15,0,14,14,16,16,0,0,0,19,18,0,12,12,16,15,0,15,16,0,20,0,14,14,16,16,0,14,14,17,17,0,0,0,19,18,0,12,12,15,15,0,17,17,0,20,0,14,14,16,16,5,6,7,12,12,9,9,9,14,14,10,10,10,14,14,0,21,21,18,17,0,20,20,18,17,9,10,10,14,14,12,12,12,16,16,12,10,10,14,14,0,20,19,18,17,0,0,20,17,18,11,10,10,14,14,14,13,13,18,18,13,11,11,14,14,0,20,20,17,18,0,21,21,17,17,0,21,0,18,18,0,0,0,0,0,0,20,19,16,17,0,0,0,19,19,0,0,0,18,18,0,21,21,18,18,0,0,0,0,0,0,20,20,16,17,0,0,0,21,21,0,0,0,18,19,6,6,6,13,12,8,6,6,11,11,8,6,6,13,13,0,9,9,11,11,0,11,10,14,14,9,7,7,13,13,11,9,9,13,13,10,6,6,13,13,0,10,10,14,15,0,10,10,13,13,9,7,7,13,13,13,10,9,13,13,10,6,6,13,13,0,10,10,15,14,0,10,10,13,13,0,11,11,15,15,0,19,20,17,17,0,9,9,13,13,0,13,13,20,20,0,11,11,13,13,0,11,11,15,15,0,19,19,17,17,0,10,10,13,13,0,15,15,20,20,0,12,12,13,13,0,10,10,12,12,0,11,11,15,15,0,11,11,15,15,0,15,15,20,0,0,16,16,0,21,0,11,11,15,15,0,14,14,18,17,0,11,11,15,15,0,15,16,19,20,0,16,16,21,21,0,12,12,15,15,0,15,14,18,18,0,11,11,16,16,0,15,15,21,21,0,16,15,0,0,0,16,16,21,0,0,0,0,0,0,0,14,14,20,20,0,18,18,0,0,0,16,17,21,0,0,16,16,21,21,0,0,0,0,0,0,15,15,21,21,0,20,19,0,21,0,17,17,0,0,0,10,10,12,11,0,10,10,10,11,0,11,11,12,12,0,11,11,9,9,0,13,13,11,12,0,11,11,12,12,0,13,13,12,12,0,10,10,12,12,0,12,12,13,13,0,12,12,12,12,0,11,11,12,12,0,13,13,12,12,0,10,10,12,12,0,13,13,14,14,0,12,12,12,12,0,14,14,14,13,0,19,20,15,15,0,12,11,12,12,0,15,15,21,20,0,13,13,11,11,0,13,13,13,13,0,19,0,15,15,0,12,12,12,12,0,17,16,19,0,0,13,13,12,12,7,7,7,16,16,11,9,9,15,15,12,9,9,16,16,0,13,13,15,14,0,14,14,17,16,10,9,9,16,16,14,11,11,17,16,12,9,8,15,15,0,13,13,18,18,0,13,13,15,15,12,10,10,18,17,15,12,12,17,17,14,9,9,16,16,0,13,13,18,19,0,14,13,17,16,0,14,14,18,18,0,0,0,20,21,0,12,12,16,16,0,16,16,20,21,0,14,14,17,16,0,14,14,18,19,0,0,0,19,21,0,13,13,17,17,0,17,17,0,21,0,15,15,16,16,8,7,7,14,14,11,10,10,15,15,12,10,10,15,15,0,20,20,18,18,0,0,0,17,17,11,10,10,16,16,14,12,12,18,17,14,11,11,15,15,0,20,21,18,18,0,0,19,18,17,12,10,10,16,16,17,14,14,19,19,14,11,11,15,15,0,21,21,19,19,0,21,20,19,18,0,21,0,18,19,0,0,0,0,0,0,20,20,18,17,0,21,0,0,0,0,0,0,19,18,0,0,0,18,19,0,0,0,0,0,0,0,21,17,18,0,0,0,0,21,0,0,21,18,19,11,9,9,14,14,13,10,10,13,13,13,11,11,15,15,0,13,13,12,12,0,15,15,16,16,13,10,10,15,15,16,12,12,15,15,15,10,10,15,15,0,14,13,16,15,0,14,13,15,15,13,10,10,15,15,18,14,14,15,15,15,10,10,14,15,0,14,14,16,16,0,14,14,16,15,0,15,15,17,16,0,21,0,18,18,0,12,13,15,15,0,16,16,0,0,0,14,14,15,15,0,15,15,16,16,0,21,20,18,18,0,13,13,15,15,0,19,18,0,0,0,15,15,15,15,0,11,11,13,13,0,12,12,16,16,0,12,12,16,16,0,15,16,20,0,0,16,17,0,0,0,12,12,16,16,0,14,14,18,18,0,11,11,16,17,0,15,15,20,0,0,16,16,0,0,0,12,12,16,16,0,15,15,19,19,0,11,11,17,17,0,16,16,21,0,0,16,16,0,0,0,17,17,20,20,0,0,0,0,0,0,15,15,20,0,0,17,18,0,0,0,17,17,0,0,0,16,16,0,21,0,0,0,0,0,0,15,15,21,0,0,19,18,0,0,0,18,17,0,0,0,11,11,14,14,0,11,11,15,15,0,12,12,16,16,0,13,13,14,14,0,14,14,17,17,0,12,12,16,16,0,14,14,16,16,0,11,11,16,15,0,13,13,16,17,0,13,13,16,16,0,12,12,15,16,0,15,14,16,16,0,11,11,15,15,0,14,14,17,17,0,13,13,16,16,0,15,14,18,18,0,21,0,19,19,0,13,13,15,15,0,16,16,20,20,0,14,14,16,15,0,14,14,17,17,0,21,0,20,18,0,13,13,15,15,0,17,17,0,0,0,14,14,16,15,8,8,8,16,16,12,9,9,16,16,13,9,9,16,16,0,14,14,18,17,0,14,14,16,17,12,10,10,18,17,14,11,11,18,18,14,9,9,16,16,0,13,13,18,18,0,13,13,17,16,12,9,9,16,17,17,13,13,16,16,14,9,9,15,15,0,14,14,20,20,0,13,13,15,15,0,15,14,18,18,0,0,0,20,21,0,12,13,16,17,0,16,16,20,21,0,14,14,16,17,0,14,14,18,17,0,0,0,20,21,0,13,13,16,16,0,19,17,0,21,0,14,15,16,16,8,7,7,14,13,12,10,10,15,15,13,10,10,15,15,0,21,21,18,19,0,20,21,18,18,12,10,10,16,15,15,12,12,17,17,14,11,11,15,15,0,21,21,19,18,0,0,21,17,18,13,11,11,15,15,16,13,13,18,19,15,11,11,15,14,0,21,0,19,19,0,0,21,18,18,0,0,21,19,19,0,0,0,0,0,0,20,19,17,17,0,0,0,21,0,0,21,0,18,19,0,0,20,20,19,0,0,0,0,0,0,21,20,18,17,0,0,0,0,20,0,0,0,18,19,0,10,10,15,14,0,11,11,14,14,0,11,11,15,16,0,14,14,15,15,0,15,15,16,16,0,11,11,16,16,0,14,13,16,16,0,11,11,15,15,0,14,14,16,16,0,14,14,15,15,0,11,11,15,15,0,13,13,15,15,0,11,11,15,15,0,15,15,18,17,0,14,14,15,15,0,15,16,18,18,0,0,0,20,20,0,14,13,16,15,0,17,17,21,0,0,15,15,15,15,0,16,15,17,17,0,0,0,19,19,0,13,13,15,15,0,20,19,0,0,0,15,15,15,15,0,11,11,13,13,0,12,12,16,16,0,12,12,16,16,0,15,15,21,21,0,17,16,0,0,0,12,12,16,16,0,14,14,17,17,0,11,11,16,16,0,15,15,0,0,0,16,16,21,0,0,12,12,17,16,0,14,15,20,20,0,11,11,16,16,0,15,15,0,20,0,16,16,0,21,0,16,17,21,0,0,0,0,0,0,0,15,15,0,21,0,18,18,0,0,0,17,16,0,0,0,17,17,21,0,0,0,0,0,0,0,15,15,0,20,0,19,20,21,0,0,17,18,0,0,0,12,12,15,15,0,12,12,15,15,0,12,12,16,16,0,13,13,15,15,0,15,15,17,17,0,13,12,17,16,0,14,14,17,16,0,11,11,16,16,0,14,14,17,17,0,14,14,17,17,0,12,12,16,16,0,15,15,17,17,0,11,11,16,16,0,14,14,17,17,0,14,14,16,16,0,15,15,18,17,0,0,0,19,0,0,13,13,16,16,0,16,16,0,21,0,14,14,16,16,0,15,15,18,17,0,0,0,19,19,0,13,13,16,16,0,18,17,0,21,0,14,15,16,16,0,11,11,16,16,0,13,13,17,17,0,13,13,17,17,0,16,16,16,17,0,16,16,18,18,0,12,12,17,17,0,16,15,18,17,0,12,12,16,16,0,16,15,19,19,0,16,15,17,17,0,12,12,17,18,0,16,16,18,18,0,12,12,16,16,0,16,16,19,19,0,15,16,17,17,0,15,16,18,18,0,0,0,20,20,0,13,13,16,16,0,18,18,21,20,0,15,15,16,16,0,16,16,19,18,0,0,0,19,20,0,14,14,17,17,0,19,19,0,21,0,15,16,16,16,0,9,9,14,14,0,13,13,15,15,0,14,14,15,15,0,0,21,19,19,0,0,21,18,18,0,12,12,15,15,0,15,15,18,18,0,14,13,15,15,0,21,21,18,19,0,21,20,18,18,0,13,13,16,16,0,17,17,18,19,0,14,14,15,15,0,0,21,19,19,0,21,20,18,19,0,20,20,19,19,0,0,0,0,0,0,19,20,17,17,0,0,0,21,21,0,21,0,18,20,0,21,0,18,21,0,0,0,0,0,0,21,21,19,18,0,0,0,0,0,0,0,0,19,19,0,18,18,15,15,0,18,20,17,16,0,20,0,17,17,0,21,0,17,17,0,21,20,19,20,0,19,19,16,16,0,21,21,17,18,0,19,19,17,17,0,20,21,21,21,0,20,20,18,18,0,19,19,16,16,0,0,21,18,19,0,18,19,16,17,0,21,21,19,20,0,21,19,18,18,0,21,20,19,21,0,0,0,20,21,0,19,19,17,16,0,0,0,0,0,0,21,20,17,17,0,20,21,19,18,0,0,0,0,21,0,19,18,16,17,0,0,0,0,0,0,20,20,17,17,0,11,11,14,14,0,13,13,16,16,0,13,13,16,16,0,17,17,21,0,0,17,18,0,0,0,12,12,16,16,0,15,15,17,18,0,12,12,16,16,0,16,16,0,20,0,17,17,0,21,0,12,12,17,17,0,16,16,19,20,0,12,12,17,17,0,17,17,0,20,0,17,17,0,0,0,17,17,21,0,0,0,0,0,0,0,15,15,0,20,0,19,19,0,0,0,18,18,0,0,0,17,17,0,0,0,0,0,0,0,0,15,15,0,0,0,20,19,0,0,0,19,18,0,0,0,14,14,21,19,0,16,16,20,21,0,16,16,20,20,0,17,17,20,0,0,17,17,20,20,0,15,15,20,20,0,19,18,20,0,0,15,15,20,20,0,17,18,21,20,0,17,17,20,21,0,15,15,19,19,0,19,18,21,21,0,15,15,19,20,0,17,18,0,0,0,17,17,20,20,0,17,18,20,21,0,0,0,0,0,0,15,15,20,20,0,19,19,0,0,0,17,17,19,21,0,17,17,0,21,0,0,0,0,21,0,15,15,20,19,0,0,20,0,0,0,17,17,21,20,0,12,12,16,16,0,14,14,17,17,0,13,13,17,17,0,16,16,17,18,0,17,16,18,18,0,13,13,18,17,0,15,16,19,18,0,13,13,16,16,0,16,16,19,19,0,16,16,17,17,0,13,12,17,17,0,16,16,18,17,0,12,12,16,16,0,17,17,19,18,0,16,15,16,16,0,16,17,18,19,0,0,0,20,20,0,14,14,17,16,0,18,18,21,0,0,16,16,16,16,0,16,16,18,17,0,0,21,21,21,0,14,14,16,16,0,21,20,21,0,0,16,16,16,16,0,10,10,14,14,0,14,14,15,16,0,14,14,15,15,0,0,21,18,18,0,0,21,18,19,0,13,13,16,16,0,16,16,18,17,0,14,14,15,15,0,20,0,18,18,0,21,0,18,17,0,13,13,16,15,0,17,17,19,19,0,14,14,15,15,0,20,20,18,19,0,0,0,18,17,0,0,21,18,18,0,0,0,0,0,0,20,21,18,17,0,0,0,0,0,0,0,0,19,19,0,0,21,18,18,0,0,0,0,0,0,21,0,18,17,0,0,0,0,21,0,0,0,19,20,0,19,19,16,16,0,0,21,18,17,0,21,0,18,18,0,20,0,19,18,0,21,20,19,19,0,21,19,17,18,0,0,21,19,19,0,21,19,18,18,0,21,0,20,18,0,0,21,18,18,0,20,21,17,17,0,21,0,18,18,0,21,19,17,17,0,21,0,0,20,0,0,20,17,18,0,0,0,19,20,0,0,0,20,19,0,19,21,17,18,0,21,0,0,0,0,21,21,18,17,0,0,21,18,18,0,0,0,0,21,0,20,19,16,17,0,0,0,0,0,0,21,20,17,17,0,11,11,13,13,0,13,13,16,16,0,13,13,16,16,0,17,17,0,21,0,18,19,21,0,0,12,12,16,16,0,15,15,19,18,0,13,13,16,16,0,16,17,21,19,0,17,17,21,21,0,13,13,16,16,0,16,16,20,18,0,13,13,16,16,0,17,17,0,0,0,18,18,0,0,0,18,17,0,20,0,0,0,0,0,0,15,15,21,21,0,19,18,0,0,0,17,17,21,21,0,17,17,0,0,0,0,0,0,0,0,15,15,20,21,0,20,20,0,0,0,19,19,0,0,0,14,15,21,19,0,16,16,0,21,0,17,16,21,21,0,17,18,21,20,0,18,18,0,21,0,16,16,0,20,0,19,19,0,0,0,16,15,0,20,0,18,18,0,0,0,17,17,0,21,0,16,16,20,20,0,20,19,0,0,0,15,16,21,22,0,18,18,0,0,0,18,17,0,0,0,18,18,0,0,0,0,0,0,0,0,16,16,21,20,0,19,20,0,0,0,18,17,21,0,0,17,18,0,0,0,0,0,0,0,0,16,16,0,20,0,0,20,0,0,0,18,18,22,0,2,3,3,3,3,3,3,1,6,6,7,8,8,7,8,8,7,9,8,10,11,11,9,8,8,7,8,8,11,11,11,9,8,8,6,7,7,10,10,10,10,10,10,10,10,10,14,13,13,12,11,11,10,10,10,14,14,13,12,11,11,6,6,6,8,5,5,8,7,7,9,7,7,11,10,10,9,7,7,9,7,7,12,10,10,10,7,7,7,8,8,12,11,10,12,10,10,11,10,10,15,13,13,13,10,10,11,10,10,17,14,13,13,10,10,7,7,7,12,11,12,12,11,11,12,11,11,16,14,14,13,12,12,12,11,11,17,15,14,14,12,12,10,9,9,13,11,11,13,11,11,13,11,11,17,14,13,14,11,11,12,11,11,16,15,14,14,11,11,7,8,8,12,11,11,12,10,10,12,10,10,15,13,13,14,11,10,12,10,10,16,14,14,14,10,10,8,7,7,12,11,11,12,11,11,12,11,11,17,14,14,14,12,12,12,11,11,16,15,15,14,12,12,10,10,10,13,11,11,13,11,11,13,11,12,16,14,14,14,11,11,13,12,11,16,15,15,14,11,11,2,7,7,7,7,7,7,7,7,7,8,8,9,8,8,8,7,7,8,8,8,9,8,8,9,7,7,6,6,6,9,8,7,9,7,7,9,8,8,10,8,8,10,8,8,10,8,8,10,8,8,10,8,8,7,6,6,9,6,6,9,7,7,9,7,7,10,8,8,9,6,6,9,7,7,10,8,8,9,7,7,7,8,8,11,9,9,11,9,9,11,8,9,12,9,9,12,8,8,11,9,9,12,9,9,12,8,8,8,7,7,10,9,9,10,10,9,10,9,9,11,10,10,11,9,9,11,9,9,11,10,11,11,9,9,10,8,8,11,9,9,10,9,9,11,9,9,11,10,10,11,9,9,11,9,9,11,10,10,11,9,9,9,8,8,11,9,9,12,9,9,11,9,9,12,9,9,12,8,8,12,9,9,12,9,9,12,8,8,9,7,7,11,9,9,11,10,10,11,9,9,11,11,11,11,9,9,11,10,10,11,11,11,11,9,9,10,9,9,11,9,9,11,10,10,11,9,9,11,10,10,11,9,9,11,9,10,11,10,10,11,9,9,1,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,7,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,7,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,1,3,2,5,4,7,7,8,8,9,9,10,10,11,11,12,12,13,13,13,14,16,16,16,16,3,4,4,5,4,5,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,3,7,7,9,13,16,3,2,4,6,10,13,17,7,4,4,6,9,12,14,7,6,6,5,7,9,12,10,10,9,6,6,9,12,14,14,13,9,8,10,11,18,18,15,13,11,10,11,1,3,2,3,1,4,4,7,7,8,8,9,9,10,10,11,11,4,6,5,8,6,9,8,10,9,10,10,11,10,5,5,6,6,8,8,9,9,10,10,10,10,11,7,8,8,9,8,10,9,10,9,11,10,11,10,7,8,8,8,10,9,10,10,10,10,11,10,11,9,10,10,11,11,11,11,12,11,12,11,12,11,9,10,10,11,11,11,11,11,11,11,12,11,12,11,11,11,12,12,12,12,12,12,12,12,12,11,11,12,11,12,12,12,12,12,12,12,12,11,12,12,12,12,12,13,12,13,12,12,12,12,12,12,12,12,12,13,13,13,13,12,13,12,12,12,12,12,13,13,12,13,12,13,12,13,12,12,12,12,13,13,13,13,13,13,12,12,12,12,12,11,12,1,4,4,6,6,5,5,5,6,6,5,6,5,6,6,6,6,7,7,7,6,7,6,7,7,1,4,4,4,4,4,4,4,4,4,4,12,9,8,12,15,17,4,2,11,6,5,9,13,15,11,7,8,7,7,10,14,13,8,5,7,5,5,8,12,12,8,4,7,4,3,6,11,12,11,8,9,7,6,8,11,12,15,13,14,12,9,7,10,13,16,12,17,12,7,5,8,11,1,2,2], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+351400);
/* memory initializer */ allocate([1,4,4,0,0,0,8,8,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,10,10,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,0,0,0,6,6,0,0,0,7,7,0,0,0,8,8,0,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,8,0,0,0,8,8,0,0,0,10,10,0,0,0,9,9,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,10,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,10,10,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,8,0,0,0,10,10,0,0,0,11,11,0,0,0,12,12,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,10,10,0,0,0,13,13,0,0,0,0,0,0,0,0,13,13,0,0,0,0,0,0,0,0,12,12,0,0,0,13,13,0,0,0,0,0,0,0,0,13,13,0,0,0,0,0,0,0,0,12,12,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,12,12,0,0,0,12,12,0,0,0,0,0,0,0,0,13,13,0,0,0,0,0,0,0,0,12,11,0,0,0,12,12,0,0,0,0,0,0,0,0,13,13,0,0,0,0,0,0,0,0,12,12,0,0,0,13,13,0,0,0,0,0,0,0,0,13,13,0,0,0,0,0,0,0,0,12,12,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,8,0,0,0,10,10,0,0,0,11,11,0,0,0,12,12,0,0,0,0,0,0,0,0,13,13,0,0,0,0,0,0,0,0,12,12,0,0,0,13,13,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,10,10,0,0,0,13,13,0,0,0,0,0,0,0,0,14,13,0,0,0,0,0,0,0,0,13,12,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,12,12,0,0,0,12,12,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,0,12,12,0,0,0,13,13,0,0,0,0,0,0,0,0,13,13,0,0,0,0,0,0,0,0,12,12,0,0,0,12,12,0,0,0,0,0,0,0,0,13,13,0,0,0,0,0,0,0,0,12,12,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,12,12,0,0,0,12,12,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,0,11,11,0,0,0,14,14,0,0,0,0,0,0,0,0,13,13,0,0,0,0,0,0,0,0,12,12,0,0,0,12,13,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,0,11,11,0,0,0,14,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,12,12,0,0,0,13,13,0,0,0,0,0,0,0,0,13,13,0,0,0,0,0,0,0,0,12,12,0,0,0,13,13,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,0,12,12,0,0,0,14,14,0,0,0,0,0,0,0,0,14,14,0,0,0,0,0,0,0,0,12,12,0,0,0,13,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,5,5,6,7,7,0,8,8,6,9,9,8,11,11,0,8,8,0,9,9,0,12,12,0,8,8,5,7,7,7,10,10,0,12,12,8,11,11,9,12,12,0,11,12,0,12,12,0,15,15,0,12,12,0,6,6,0,6,6,0,7,7,0,7,7,0,10,10,0,7,7,0,8,8,0,11,11,0,7,7,6,7,7,10,9,9,0,11,10,10,9,9,12,12,12,0,10,10,0,11,11,0,13,13,0,11,11,7,6,6,10,10,10,0,11,11,11,11,11,12,12,12,0,11,11,0,12,12,0,15,15,0,11,11,0,11,11,0,11,11,0,12,12,0,12,12,0,14,14,0,12,12,0,12,12,0,15,15,0,11,11,0,8,8,0,10,10,0,11,11,0,11,11,0,12,12,0,12,12,0,11,11,0,15,15,0,11,11,0,6,6,0,10,10,0,12,12,0,10,10,0,13,13,0,12,12,0,13,13,0,14,14,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,3,0,8,8,0,8,8,0,9,9,0,9,9,0,9,9,0,9,9,0,9,9,0,8,8,0,6,6,0,7,7,0,7,7,0,8,8,0,8,8,0,8,8,0,8,8,0,8,8,0,8,8,0,6,6,0,6,6,0,6,6,0,8,8,0,9,9,0,7,7,0,8,8,0,9,9,0,6,6,0,8,8,0,9,9,0,9,9,0,10,10,0,10,10,0,10,10,0,10,10,0,11,11,0,9,9,0,7,7,0,10,10,0,10,10,0,12,11,0,12,12,0,11,11,0,11,11,0,12,12,0,10,10,0,7,7,0,10,10,0,10,10,0,12,12,0,11,12,0,11,11,0,11,11,0,11,11,0,10,10,0,8,8,0,9,9,0,9,9,0,10,10,0,10,10,0,10,9,0,10,10,0,10,10,0,9,9,0,6,6,0,10,10,0,10,10,0,11,11,0,12,12,0,11,11,0,11,11,0,12,12,0,11,11,0,7,7,0,9,9,0,9,9,0,11,11,0,11,11,0,10,10,0,10,10,0,11,11,0,9,9,1,6,6,6,7,7,7,8,8,7,8,8,10,11,11,9,8,8,7,8,8,11,11,11,9,8,8,6,7,7,9,11,11,9,11,11,10,11,11,12,13,13,11,12,12,10,11,11,13,14,14,12,12,12,6,6,6,8,6,6,8,7,7,9,7,7,11,10,10,10,6,6,9,7,7,12,10,10,11,6,7,7,7,7,11,10,10,12,10,10,11,10,10,14,13,13,13,10,10,12,11,11,15,13,13,14,10,10,8,7,7,12,11,11,12,11,11,11,11,11,14,14,14,13,12,12,12,11,11,15,15,15,13,12,12,0,10,10,0,11,11,0,11,11,0,11,11,0,14,14,0,11,11,0,11,11,0,15,15,0,11,11,7,8,8,12,10,10,12,10,10,12,11,11,15,13,13,14,11,11,12,10,10,16,14,14,14,10,10,8,7,7,12,11,11,12,11,11,12,11,11,16,14,14,14,12,12,13,12,12,15,14,14,15,12,12,0,11,11,0,12,12,0,12,12,0,12,12,0,15,15,0,12,12,0,12,12,0,14,14,0,12,12,3,4,4,8,8,11,9,9,12,12,11,10,10,12,12,12,10,10,11,11,12,12,12,12,12,12,11,11,13,13,12,12,12,13,13,12,10,10,12,12,12,11,11,13,13,12,13,13,13,13,12,11,11,13,13,12,12,12,13,13,12,10,10,12,12,12,11,11,13,13,12,13,13,12,12,12,11,11,13,13,12,13,13,13,13,12,11,11,12,12,12,11,11,12,12,12,13,13,12,12,12,13,13,13,13,12,13,13,13,13,13,13,13,12,12,12,13,13,13,13,12,13,13,12,12,11,8,8,10,10,12,11,11,11,11,12,10,10,10,10,13,11,11,10,10,13,11,11,10,10,13,12,12,12,12,13,11,11,11,11,13,12,12,11,11,13,12,12,11,11,13,12,12,12,11,13,12,12,12,12,13,11,11,11,11,13,12,12,11,11,13,11,12,11,11,13,12,12,11,11,14,12,12,11,11,13,11,11,11,11,14,12,12,11,11,13,11,12,10,10,14,12,12,11,11,14,12,12,11,11,14,11,11,11,11,14,12,12,11,11,13,12,12,11,11,14,12,12,11,11,11,8,8,10,10,12,7,7,10,10,12,9,9,11,11,13,9,9,9,9,13,13,13,10,10,13,9,9,12,12,13,13,13,12,12,13,9,8,11,11,13,10,10,12,12,14,13,13,11,11,13,9,9,11,11,13,13,13,12,12,13,9,9,10,10,13,10,10,11,11,13,13,13,10,10,14,10,10,11,11,14,14,14,12,12,13,9,9,10,10,13,10,10,11,11,14,13,14,10,10,14,14,14,11,12,14,14,14,14,14,14,13,13,10,10,13,14,14,11,11,14,14,14,10,10,14,9,9,9,9,14,9,9,9,9,14,10,10,9,9,14,10,10,8,8,14,11,11,8,8,15,11,11,10,10,15,12,12,10,10,15,10,10,10,10,15,11,11,10,10,15,13,13,10,10,15,11,11,10,10,15,12,12,10,10,15,10,10,10,10,15,11,11,10,10,15,13,13,10,10,15,11,11,10,10,15,12,12,10,10,15,11,11,9,9,15,11,11,9,9,15,13,13,9,9,15,13,13,10,10,15,12,12,10,10,15,13,13,10,10,15,13,12,9,9,15,13,13,9,9,14,12,12,9,9,14,13,13,9,9,14,13,13,9,9,14,13,13,7,7,14,13,13,8,8,15,14,14,10,10,15,14,14,10,10,15,14,14,10,10,15,14,14,10,10,15,14,14,9,9,15,14,14,10,10,15,14,14,10,10,14,14,14,9,9,15,14,14,10,10,14,14,14,9,9,15,14,14,10,10,15,14,14,10,10,14,14,14,9,9,14,14,14,9,9,14,14,14,8,8,15,14,14,10,10,15,14,14,11,11,15,14,14,9,9,15,14,14,9,9,14,14,14,8,8,13,9,9,12,12,17,11,11,12,12,17,12,12,12,12,17,12,12,11,11,18,15,15,12,12,17,12,12,12,12,17,14,15,13,13,17,12,12,12,12,17,13,13,12,13,17,15,15,12,12,18,13,13,13,13,18,15,15,13,13,18,12,12,12,12,18,13,13,13,13,18,15,15,12,12,18,13,13,12,12,18,15,15,13,13,18,13,13,12,12,17,13,13,12,12,17,15,15,12,12,18,15,15,13,13,18,15,15,13,14,18,15,16,12,12,18,15,15,12,12,18,16,16,12,12,13,8,8,10,10,14,15,14,11,11,14,15,15,12,12,15,14,14,12,11,15,15,15,12,12,15,15,15,12,12,15,15,15,13,13,15,15,15,12,12,15,15,15,13,13,15,15,15,13,13,15,15,15,13,13,15,15,16,13,13,15,15,15,12,12,15,15,15,13,13,15,15,15,13,13,15,15,15,13,13,15,15,15,13,13,15,15,14,12,12,15,15,15,12,12,16,15,14,12,12,16,15,15,13,13,16,16,16,13,13,16,15,15,12,12,15,15,15,13,13,15,15,15,12,12,13,12,12,10,10,14,14,14,11,11,15,14,14,12,12,15,14,14,11,11,15,14,14,11,11,15,15,15,13,13,15,14,14,13,13,15,15,15,12,12,15,14,15,13,13,16,15,15,12,12,15,15,15,13,13,16,14,14,13,13,15,15,15,12,12,15,15,15,13,13,16,15,15,12,12,16,15,15,12,12,16,14,14,13,13,15,15,15,11,11,15,15,15,12,12,16,15,15,11,11,16,15,15,13,13,16,14,15,14,14,16,15,15,12,12,16,15,14,12,12,16,15,15,12,12,14,10,10,9,9,14,11,11,12,12,14,12,12,13,13,14,12,12,12,12,15,14,14,13,13,15,13,13,14,14,15,14,14,15,15,15,12,12,13,13,15,13,13,14,14,15,14,14,13,13,15,13,13,13,14,15,14,14,15,15,15,12,12,13,13,15,13,13,14,14,15,14,14,13,13,15,13,13,14,14,15,14,14,15,15,15,13,13,12,12,15,13,13,13,13,15,14,14,13,12,15,15,15,14,15,15,15,14,20,20,15,14,14,13,13,15,14,14,13,13,15,14,14,13,13,14,12,12,9,9,14,14,14,12,12,14,13,13,12,13,14,14,14,12,12,15,14,14,12,12,15,14,14,14,13,15,14,14,14,14,15,14,14,13,13,15,14,14,13,13,15,15,15,14,14,15,14,14,13,13,15,14,14,14,14,15,14,14,13,13,15,14,14,13,13,15,15,15,15,14,15,15,15,13,13,15,14,14,14,14,15,14,14,13,13,15,14,14,13,13,14,15,15,14,14,15,15,15,14,14,15,14,14,14,14,15,15,15,14,14,15,14,14,13,14,15,15,15,14,14,13,10,10,12,12,17,11,11,12,12,17,12,12,12,12,17,12,12,11,11,17,15,15,12,11,18,13,13,13,13,18,15,15,13,13,17,12,12,12,12,18,13,13,13,13,17,15,15,12,12,17,12,12,12,12,17,15,15,13,13,17,12,12,12,12,17,13,13,12,12,17,15,15,12,12,18,14,13,12,12,18,15,15,13,13,18,13,13,12,12,18,13,13,12,12,18,16,16,12,12,18,16,16,12,12,18,15,15,13,13,18,16,16,12,12,17,15,15,12,12,17,16,16,12,12,13,8,8,10,10,14,14,15,12,12,14,15,15,12,12,15,14,14,12,12,15,15,14,12,12,15,15,15,13,13,15,15,15,13,13,15,15,15,12,12,16,15,15,13,13,16,15,15,13,13,15,15,15,12,12,15,15,15,14,14,15,15,15,12,12,15,15,15,13,13,16,15,15,13,13,15,15,15,13,13,16,15,15,13,13,15,15,14,12,12,15,15,15,12,12,16,14,15,13,13,16,15,15,13,13,15,16,15,13,13,16,15,14,13,13,16,15,15,13,13,16,15,15,13,13,13,12,12,11,11,14,14,14,11,11,14,14,14,12,12,15,14,14,11,11,16,14,14,11,11,15,15,15,12,13,16,14,14,13,13,15,15,15,12,12,15,14,14,13,13,16,15,15,12,12,15,15,15,12,12,15,14,14,13,13,15,15,15,12,12,15,14,14,12,12,16,15,15,12,12,16,15,15,12,12,16,14,14,13,13,15,15,15,11,11,15,15,14,12,12,16,15,15,11,11,16,15,15,12,12,16,14,14,13,13,16,15,15,11,11,16,14,14,12,12,16,15,15,11,11,14,10,10,9,9,14,11,11,12,12,14,12,12,13,14,14,12,12,12,12,14,14,14,13,13,15,13,13,14,14,15,14,14,15,15,15,12,12,13,13,15,13,13,14,14,15,15,15,14,14,15,13,13,14,14,15,15,15,15,15,15,12,12,13,13,15,13,13,14,14,15,14,14,13,13,15,13,13,14,14,15,14,14,15,15,15,12,12,13,13,15,13,13,13,13,14,14,14,13,13,15,15,15,14,15,15,15,15,21,19,15,14,14,13,13,15,14,14,14,14,14,14,14,13,13,14,12,12,9,9,14,14,14,12,12,14,14,13,13,13,14,14,14,12,12,14,14,14,12,12,15,14,14,13,13,15,14,14,14,14,15,14,14,13,13,15,14,14,13,13,15,15,15,15,15,15,14,14,13,13,15,14,14,14,14,15,14,14,13,13,15,14,14,13,13,14,15,15,15,15,15,14,15,13,13,15,14,14,14,14,15,14,14,13,13,15,14,14,13,13,14,15,15,14,14,15,15,15,14,14,15,14,14,14,14,15,15,15,15,15,15,14,14,14,13,14,15,15,14,14,13,10,10,12,12,18,12,12,12,12,17,12,12,12,12,18,13,13,11,11,18,15,14,11,11,17,13,13,13,13,18,15,15,12,12,18,12,12,12,12,17,13,13,12,12,18,15,15,12,12,18,13,13,13,12,18,15,15,13,13,18,13,13,12,12,18,13,13,12,12,18,15,15,12,12,17,13,13,12,12,17,15,15,12,12,17,12,12,11,11,17,13,13,11,11,17,15,15,11,11,18,16,16,12,12,18,15,15,13,13,18,15,15,11,11,17,15,15,12,12,18,15,15,11,11,13,8,8,10,10,14,14,14,11,11,15,15,15,12,12,15,14,14,11,11,16,14,14,12,12,15,15,15,12,12,15,15,15,13,13,15,15,15,12,12,15,15,15,12,12,16,15,15,13,13,15,15,15,12,12,15,15,15,13,13,16,15,15,12,12,15,15,15,12,12,16,15,15,13,13,16,15,15,12,12,15,15,15,13,13,15,14,14,12,12,15,15,15,12,12,16,15,14,12,12,16,15,15,13,13,16,16,16,13,13,16,14,15,13,13,15,15,15,13,13,16,15,15,12,12,13,12,12,10,10,14,14,14,11,11,15,14,14,12,12,15,14,14,11,11,16,14,14,11,11,15,14,15,12,12,15,14,14,13,13,15,15,15,12,12,15,14,14,12,12,15,14,15,12,12,15,15,15,12,12,16,14,14,13,13,15,15,15,11,12,16,14,14,12,12,16,15,15,12,12,15,15,15,12,12,16,14,14,12,12,15,15,15,11,11,15,14,14,11,12,15,15,14,11,11,16,15,15,12,12,16,14,14,13,13,16,15,15,11,11,16,14,14,12,12,16,15,15,11,11,13,10,10,8,8,14,12,12,12,12,14,12,12,13,13,14,12,12,12,12,14,14,14,13,13,15,13,13,14,14,15,15,14,15,15,15,13,13,13,13,15,13,13,14,14,15,14,15,14,14,15,13,13,13,13,15,15,15,15,15,15,12,12,13,12,15,13,13,14,14,15,14,14,13,13,15,13,13,14,13,15,15,15,16,16,15,13,13,12,12,15,13,13,13,13,14,14,14,12,12,15,15,15,14,14,15,15,15,20,20,15,14,14,13,13,15,15,14,14,14,15,14,14,13,13,13,12,12,9,9,14,13,13,12,12,14,13,13,12,12,14,14,14,12,12,14,14,14,13,13,15,14,14,13,13,15,14,14,14,14,15,15,14,12,12,15,14,14,13,13,15,14,15,14,15,15,14,14,13,13,15,14,14,14,14,15,14,14,12,12,15,14,14,13,13,14,15,14,15,14,15,14,14,13,13,15,14,14,14,14,15,14,14,12,12,15,14,14,13,13,15,15,15,14,14,15,15,15,14,14,16,14,14,14,14,15,15,15,14,14,15,14,14,14,14,14,15,15,14,14,13,13,13,12,13,17,15,15,12,12,17,15,15,12,12,18,15,15,11,11,17,16,16,11,11,18,16,16,13,13,18,17,16,13,13,18,16,16,12,12,18,16,16,12,12,18,17,17,12,12,17,16,16,12,13,17,16,16,12,13,17,16,16,12,12,17,16,16,12,12,18,17,16,12,12,18,16,16,12,12,17,16,17,12,12,18,15,15,11,11,18,15,15,12,12,17,17,17,11,11,17,17,17,12,12,17,16,16,13,13,18,16,16,11,11,18,16,16,12,12,18,17,16,11,11,14,14,14,10,10,16,15,14,11,11,16,15,15,12,12,16,14,14,12,12,17,14,14,13,13,17,15,15,13,13,17,15,15,14,14,16,15,15,12,12,16,15,15,13,13,18,15,15,14,14,16,15,15,12,12,16,15,15,14,14,16,15,15,12,12,16,15,15,13,13,17,15,15,13,13,17,15,15,13,13,17,15,15,14,14,16,14,14,12,12,17,15,15,12,12,18,15,15,13,13,17,15,15,14,14,17,16,16,15,15,17,15,14,13,13,17,15,15,14,14,17,15,15,13,13,14,12,12,11,11,15,14,14,12,12,16,14,14,12,12,16,14,14,11,11,17,14,14,12,12,16,15,14,13,13,16,14,14,13,13,16,15,15,12,12,16,14,14,13,13,17,15,15,13,13,16,15,15,13,13,17,14,14,13,13,16,15,15,12,12,16,14,14,12,12,16,15,15,12,12,17,15,15,12,12,17,14,14,13,13,16,15,15,12,12,16,14,14,12,12,16,15,15,12,12,17,15,15,13,13,17,14,14,13,13,17,15,15,12,12,17,14,14,12,12,17,15,15,12,12,14,14,14,8,8,14,14,14,13,13,14,15,15,14,14,14,14,14,14,14,15,15,15,19,19,15,15,15,14,14,15,15,16,20,19,15,15,15,14,14,15,16,16,15,15,15,15,15,19,19,15,15,15,14,14,15,16,16,19,20,15,15,15,14,14,15,15,15,15,15,15,15,15,19,19,15,15,15,15,15,15,15,16,19,20,15,14,15,14,14,15,15,15,15,15,15,15,15,20,19,15,15,15,21,19,15,16,16,20,20,15,15,14,19,19,15,15,16,20,21,15,15,15,20,19,13,12,12,9,9,14,14,14,12,12,14,13,13,13,13,14,14,14,13,13,15,14,14,20,19,15,14,14,14,13,15,14,14,19,19,15,15,14,13,13,15,14,14,14,14,15,15,15,19,20,15,14,14,13,13,15,14,14,20,19,14,15,14,13,13,15,14,14,14,13,15,15,15,19,20,15,15,14,14,14,15,14,14,21,19,15,15,15,13,13,15,14,14,14,14,14,15,15,20,20,15,15,15,21,20,15,14,14,19,20,15,15,15,20,20,15,14,14,19,20,15,15,15,21,19,2,6,6,14,14,6,7,7,14,14,7,7,7,15,15,0,13,13,16,16,0,13,13,15,15,7,8,8,15,15,9,10,10,17,16,9,8,8,15,15,0,13,13,18,17,0,13,13,16,16,8,8,8,15,15,12,11,11,16,17,9,8,8,14,14,0,13,13,18,17,0,13,13,16,15,0,14,14,18,17,0,20,22,18,20,0,12,12,16,16,0,16,16,22,20,0,14,14,16,16,0,14,14,17,17,0,22,22,22,19,0,12,13,16,16,0,17,17,0,0,0,15,15,16,16,5,7,7,13,13,9,9,9,15,14,10,10,10,14,14,0,21,21,18,17,0,21,22,18,17,9,10,10,14,14,12,12,12,17,17,12,10,10,14,14,0,19,21,18,17,0,20,22,18,18,11,10,10,14,14,14,13,13,18,17,12,11,11,14,14,0,22,19,17,18,0,20,0,18,17,0,22,21,17,17,0,0,0,0,0,0,20,22,17,17,0,22,0,21,19,0,22,0,18,18,0,0,22,18,19,0,0,0,0,0,0,19,21,17,17,0,0,0,20,20,0,0,0,18,18,6,6,6,13,12,8,6,6,11,11,8,6,6,13,13,0,9,9,11,11,0,11,11,14,14,9,7,7,13,13,11,9,9,13,13,10,6,6,13,13,0,10,10,14,14,0,10,10,13,13,9,7,7,13,14,13,9,9,13,13,10,6,6,13,12,0,11,11,15,15,0,10,10,13,13,0,12,12,15,15,0,19,0,17,17,0,9,9,13,13,0,13,14,19,20,0,11,11,13,13,0,11,11,14,14,0,19,20,17,18,0,10,10,13,13,0,15,15,21,19,0,12,12,13,13,0,10,10,12,13,0,11,11,15,15,0,11,11,15,15,0,15,15,22,0,0,16,17,22,0,0,11,11,15,15,0,14,14,18,17,0,11,11,15,16,0,15,15,22,21,0,16,16,0,20,0,12,12,16,15,0,15,14,19,19,0,11,11,16,16,0,15,15,21,0,0,16,15,0,0,0,16,16,22,21,0,0,0,0,0,0,15,15,20,20,0,18,18,0,0,0,16,17,0,0,0,17,17,0,22,0,0,0,0,0,0,15,15,21,22,0,20,18,0,0,0,18,17,22,0,0,10,10,12,11,0,10,10,10,10,0,11,11,12,12,0,11,11,9,9,0,13,13,12,12,0,11,11,12,12,0,13,13,12,12,0,10,10,12,12,0,13,12,13,13,0,12,12,12,12,0,11,11,12,12,0,13,13,12,12,0,10,10,12,12,0,13,13,13,14,0,12,12,12,12,0,13,14,14,14,0,20,21,15,15,0,12,11,12,12,0,15,16,20,22,0,13,12,11,11,0,13,13,14,13,0,20,0,16,15,0,12,12,12,12,0,16,16,22,21,0,13,13,12,12,6,7,7,16,16,11,9,9,15,15,12,9,9,16,16,0,13,13,14,14,0,14,14,16,17,10,9,9,16,16,14,12,12,16,16,12,9,9,15,15,0,13,13,18,18,0,13,13,15,16,12,10,10,17,18,15,12,12,17,17,13,9,9,16,16,0,13,13,17,18,0,14,14,16,16,0,15,15,18,18,0,22,0,20,20,0,12,12,16,16,0,16,16,20,22,0,14,14,16,16,0,15,14,18,18,0,0,22,19,21,0,13,13,16,17,0,17,17,22,22,0,15,15,16,16,7,7,7,14,14,11,10,10,15,15,12,10,10,15,14,0,22,0,18,18,0,0,21,17,18,11,10,10,15,15,14,12,12,17,17,14,11,11,15,15,0,22,20,18,18,0,0,20,18,17,12,10,10,16,16,17,14,14,19,18,14,11,11,15,15,0,21,22,19,19,0,21,22,18,18,0,22,0,19,21,0,0,0,0,0,0,22,22,18,17,0,0,0,21,20,0,22,22,20,19,0,0,22,20,20,0,0,0,0,0,0,20,21,17,17,0,0,22,21,21,0,0,0,18,18,10,9,9,14,14,13,10,10,13,13,13,10,11,14,14,0,13,13,12,12,0,15,15,16,16,13,10,10,15,15,15,12,12,14,14,15,10,10,14,15,0,14,14,16,15,0,14,14,15,15,13,10,10,15,15,18,13,13,15,15,15,10,10,14,15,0,14,14,16,16,0,14,14,15,15,0,15,15,16,16,0,22,0,18,18,0,12,13,14,14,0,17,17,22,0,0,14,14,14,14,0,15,15,16,16,0,22,0,18,17,0,13,13,14,14,0,19,18,21,22,0,15,15,14,14,0,11,11,13,13,0,12,12,16,16,0,12,12,16,16,0,15,16,21,0,0,16,17,0,22,0,12,12,16,16,0,14,14,17,18,0,11,11,16,16,0,15,15,21,22,0,16,16,0,0,0,12,12,16,16,0,15,15,0,19,0,12,12,16,17,0,16,16,22,0,0,16,16,0,22,0,17,17,0,22,0,0,0,0,0,0,15,15,20,19,0,18,18,0,0,0,17,18,0,0,0,17,17,0,0,0,0,0,0,0,0,15,15,0,22,0,20,18,0,0,0,18,18,22,22,0,11,11,14,14,0,12,12,14,14,0,12,12,15,15,0,13,13,14,14,0,14,14,17,16,0,12,12,16,16,0,14,14,16,16,0,11,11,15,15,0,13,13,16,16,0,13,13,15,15,0,12,12,15,15,0,15,14,16,16,0,11,11,15,15,0,14,14,17,17,0,13,13,15,15,0,15,15,17,17,0,0,0,19,18,0,13,12,15,15,0,16,16,0,0,0,14,14,15,15,0,14,14,16,17,0,22,0,18,18,0,13,13,15,15,0,17,17,0,0,0,14,14,15,15,8,8,8,16,16,12,10,10,16,16,13,9,9,16,16,0,14,14,17,17,0,14,14,17,16,12,10,10,18,17,14,11,11,18,18,14,9,10,16,16,0,13,13,18,19,0,14,13,16,16,12,9,9,16,16,17,13,13,17,17,14,9,9,15,15,0,14,14,19,20,0,13,13,15,15,0,15,15,18,19,0,0,22,22,22,0,13,13,17,17,0,16,16,19,21,0,14,14,16,16,0,14,14,18,18,0,0,0,0,0,0,13,13,16,16,0,18,18,0,0,0,15,15,16,16,8,7,7,14,14,12,10,10,15,15,13,10,10,15,14,0,22,0,18,18,0,22,0,18,18,12,10,10,16,15,15,12,12,17,17,14,11,11,15,15,0,20,21,19,18,0,0,0,17,18,13,11,11,15,15,16,13,13,18,18,15,11,11,14,14,0,22,21,19,19,0,21,22,18,18,0,22,22,20,18,0,0,0,0,0,0,22,19,17,17,0,0,0,22,21,0,0,22,19,17,0,0,22,19,19,0,0,0,0,0,0,22,21,18,17,0,0,0,22,0,0,0,0,19,19,0,10,10,14,14,0,11,11,15,14,0,11,11,15,15,0,14,14,15,14,0,15,15,16,16,0,11,11,16,16,0,13,13,16,16,0,11,11,15,15,0,14,14,17,16,0,14,14,15,15,0,11,11,16,16,0,14,13,15,15,0,11,11,15,15,0,15,15,17,17,0,14,14,15,14,0,16,16,17,17,0,0,22,18,18,0,13,13,15,15,0,17,17,22,0,0,15,15,15,14,0,15,16,16,17,0,0,22,18,19,0,13,13,15,15,0,20,18,21,0,0,15,15,14,14,0,11,11,13,13,0,12,12,16,16,0,12,12,16,15,0,15,16,22,22,0,17,17,0,0,0,12,12,16,16,0,14,14,18,18,0,11,11,16,16,0,15,16,22,20,0,16,16,0,22,0,12,12,16,16,0,15,15,18,20,0,11,11,16,16,0,15,15,0,0,0,16,16,0,0,0,17,17,22,0,0,0,0,0,0,0,15,15,0,21,0,18,18,0,0,0,17,16,0,0,0,17,17,22,22,0,0,0,0,0,0,15,15,21,0,0,20,22,0,0,0,18,18,0,0,0,12,12,15,15,0,12,12,15,15,0,12,12,16,16,0,13,13,15,15,0,15,15,17,17,0,13,12,16,16,0,14,14,16,16,0,12,11,16,16,0,14,14,17,17,0,14,14,16,16,0,12,12,16,16,0,15,15,17,16,0,11,11,15,16,0,14,14,17,17,0,14,14,16,16,0,15,15,18,18,0,0,0,22,19,0,13,13,15,16,0,16,17,0,0,0,14,14,16,16,0,15,15,18,17,0,0,0,20,20,0,13,13,16,15,0,17,17,22,22,0,14,14,15,15,0,11,11,16,16,0,13,13,16,17,0,13,13,17,18,0,16,16,17,17,0,17,17,18,18,0,12,12,17,17,0,16,15,18,18,0,12,12,16,16,0,16,16,18,18,0,15,15,17,17,0,12,12,17,17,0,16,16,19,18,0,12,12,16,17,0,16,16,19,19,0,15,16,16,17,0,16,16,19,17,0,0,0,20,22,0,13,13,16,16,0,19,18,21,0,0,15,15,16,16,0,16,16,18,18,0,0,0,22,21,0,14,14,16,16,0,21,19,21,22,0,16,16,16,16,0,9,9,14,14,0,13,13,15,15,0,14,14,15,15,0,0,20,18,19,0,0,22,18,18,0,12,12,15,15,0,15,15,17,18,0,14,13,14,14,0,20,0,18,18,0,21,0,18,17,0,13,13,15,16,0,17,17,18,18,0,14,14,15,15,0,22,22,20,19,0,20,21,18,18,0,20,22,19,19,0,0,0,0,0,0,20,20,17,17,0,0,22,22,21,0,22,0,18,18,0,20,22,19,19,0,0,0,0,0,0,21,21,17,18,0,0,0,21,20,0,0,22,19,18,0,18,18,15,15,0,22,21,17,16,0,0,22,17,17,0,20,22,18,18,0,0,22,20,20,0,21,19,16,16,0,21,21,18,18,0,19,19,17,17,0,0,22,19,19,0,22,20,17,17,0,21,19,16,16,0,22,22,19,18,0,19,20,16,16,0,22,21,19,21,0,21,22,17,18,0,21,20,18,18,0,0,0,19,20,0,20,19,16,16,0,22,22,0,0,0,21,21,17,16,0,22,20,19,18,0,0,0,20,20,0,20,19,16,16,0,0,0,0,0,0,21,22,17,17,0,11,11,13,13,0,13,13,15,16,0,13,13,16,16,0,17,18,21,0,0,17,18,0,0,0,12,12,15,16,0,15,15,19,18,0,12,12,16,16,0,17,17,22,0,0,17,17,0,22,0,12,12,17,16,0,16,16,19,20,0,12,12,16,16,0,17,17,0,0,0,17,17,0,21,0,17,16,22,0,0,0,0,0,0,0,15,15,20,22,0,20,18,0,0,0,18,18,0,0,0,17,17,21,0,0,0,0,0,0,0,15,15,21,22,0,19,20,22,0,0,19,18,0,0,0,14,14,18,18,0,16,16,22,20,0,16,16,22,19,0,17,17,20,22,0,19,19,0,0,0,15,15,20,0,0,18,21,0,20,0,15,15,21,20,0,18,17,0,0,0,17,17,0,22,0,15,15,19,19,0,19,18,0,0,0,15,15,20,0,0,18,18,22,22,0,17,17,0,20,0,18,18,0,0,0,0,22,0,0,0,15,15,19,20,0,20,19,0,0,0,17,17,20,21,0,17,18,20,22,0,0,0,0,22,0,15,15,20,20,0,22,20,0,0,0,17,18,20,0,0,12,12,17,16,0,14,14,17,17,0,13,13,17,17,0,16,16,18,18,0,17,16,17,17,0,13,13,17,17,0,15,16,18,18,0,13,13,16,16,0,16,16,18,18,0,16,16,17,16,0,13,13,16,16,0,17,17,18,17,0,12,12,15,16,0,17,17,19,19,0,16,16,16,16,0,16,17,19,18,0,0,0,21,22,0,14,14,16,16,0,18,18,0,22,0,16,16,16,16,0,16,16,18,17,0,0,0,21,20,0,14,14,16,16,0,21,22,22,0,0,16,16,16,16,0,9,9,14,13,0,13,14,15,16,0,14,13,15,14,0,22,0,18,18,0,21,0,17,18,0,13,13,15,15,0,15,16,18,17,0,14,14,15,14,0,20,22,18,18,0,22,21,17,17,0,13,13,15,15,0,17,17,19,19,0,14,14,14,14,0,0,22,18,18,0,0,22,17,17,0,0,22,19,20,0,0,0,0,0,0,21,20,17,16,0,0,0,21,22,0,0,0,18,19,0,0,0,18,18,0,0,0,0,0,0,22,0,17,17,0,0,0,20,22,0,0,0,18,19,0,18,19,16,16,0,22,20,17,17,0,22,22,17,18,0,22,22,18,17,0,0,22,18,19,0,20,20,17,18,0,0,22,19,18,0,22,22,17,17,0,22,0,19,19,0,0,22,18,18,0,20,22,17,17,0,0,22,18,18,0,19,20,17,17,0,22,0,20,19,0,22,21,17,17,0,0,0,18,18,0,0,0,22,19,0,20,0,17,17,0,22,0,0,22,0,0,20,17,18,0,22,0,19,19,0,0,0,0,19,0,19,21,17,17,0,0,0,0,0,0,20,21,17,16,0,11,11,13,13,0,13,13,16,16,0,13,13,15,16,0,17,17,21,22,0,17,18,0,0,0,12,12,16,16,0,15,15,18,18,0,13,13,16,16,0,17,16,21,21,0,17,17,0,0,0,13,13,16,16,0,16,16,19,18,0,13,13,16,16,0,17,17,0,22,0,17,18,20,22,0,17,18,0,0,0,0,0,0,0,0,15,15,20,0,0,18,19,0,0,0,17,17,0,0,0,18,17,22,0,0,0,0,0,0,0,15,16,21,20,0,20,20,0,0,0,18,19,0,0,0,15,15,22,22,0,17,16,20,22,0,17,17,20,22,0,18,18,0,21,0,19,18,0,0,0,16,16,20,20,0,19,19,22,0,0,15,16,21,22,0,18,19,22,0,0,17,18,0,0,0,16,16,22,0,0,19,19,0,21,0,15,16,20,0,0,18,18,0,22,0,18,17,0,0,0,18,18,0,0,0,0,0,0,0,0,16,16,22,21,0,20,21,0,0,0,17,18,22,0,0,18,18,0,0,0,0,0,0,0,0,16,16,20,19,0,22,21,0,0,0,18,18,22,22,2,3,3,3,3,3,3,1,7,7,7,8,8,7,8,8,7,9,9,10,11,11,9,8,8,7,8,9,11,11,11,9,8,8,6,7,7,10,10,10,10,10,10,10,10,10,14,14,14,12,11,11,10,11,11,15,14,14,13,11,11,6,6,6,8,5,5,8,7,7,8,7,7,11,10,10,9,7,7,9,7,7,12,10,10,10,7,7,6,8,7,12,10,10,12,10,10,11,10,10,15,14,13,13,10,10,11,10,10,16,14,14,14,10,10,7,7,7,12,11,11,12,11,11,11,11,11,16,14,14,13,12,12,11,11,11], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+361716);
/* memory initializer */ allocate([17,15,15,14,12,12,10,9,9,13,11,11,13,11,11,12,11,11,16,14,13,14,11,11,12,11,11,17,15,14,14,11,11,7,8,8,12,11,11,12,10,10,12,10,10,16,13,14,13,10,10,11,10,10,17,14,14,14,10,10,7,7,7,12,11,11,12,11,11,12,11,11,15,14,15,14,12,12,12,11,11,17,15,15,14,12,12,10,10,9,13,11,11,13,11,11,13,11,11,16,14,14,14,11,11,13,11,11,16,15,15,15,11,11,2,6,6,7,7,7,7,7,7,7,8,8,9,9,9,8,7,7,8,8,8,9,9,9,9,8,8,6,7,7,9,8,8,9,7,7,9,8,8,10,8,8,10,8,8,10,8,8,10,8,9,10,8,8,7,6,6,8,6,6,9,6,6,9,7,7,10,8,8,9,6,6,9,7,7,10,9,8,9,7,7,7,7,7,11,8,8,11,9,9,10,9,9,12,9,9,12,8,8,11,9,9,12,9,9,12,8,8,8,7,7,10,9,9,10,9,9,10,9,9,11,10,11,11,9,9,11,9,9,11,11,11,11,9,9,10,8,8,11,9,9,10,9,9,11,9,9,11,10,10,11,9,9,11,9,9,12,10,10,11,9,9,8,8,8,11,9,9,12,9,9,11,9,9,12,9,9,12,8,8,12,9,9,12,9,10,12,8,8,9,7,7,11,9,9,11,10,10,11,9,9,11,11,11,11,9,9,11,10,10,12,11,11,11,9,10,10,9,9,11,9,9,11,10,10,11,10,10,11,11,11,11,9,9,11,9,10,11,11,11,11,9,9,1,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,1,9,9,6,9,9,5,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,1,3,2,5,4,7,7,8,8,9,9,10,10,11,11,12,12,13,13,14,14,15,15,15,15,3,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,4,9,8,8,10,13,16,4,2,9,5,7,10,14,18,9,7,6,5,7,9,12,16,7,5,5,3,5,8,11,13,8,7,7,5,5,7,9,11,10,10,9,8,6,6,8,10,13,14,13,11,9,8,9,10,17,18,16,14,11,10,10,10,1,3,2,3,1,4,4,7,7,8,8,9,9,10,10,11,11,4,6,5,8,7,9,8,10,9,11,10,11,11,4,5,6,7,8,8,9,9,10,10,10,10,11,8,9,8,10,8,10,9,11,10,11,11,11,11,8,8,9,8,10,9,10,10,11,11,11,11,11,9,10,10,11,11,11,11,11,11,12,11,12,11,9,10,10,10,11,11,11,11,11,11,12,11,12,10,11,11,12,11,12,12,12,12,12,12,12,12,10,11,11,11,11,12,12,12,13,12,12,12,12,11,12,12,12,12,13,13,12,12,12,12,12,12,11,12,12,12,12,13,13,12,13,12,12,12,12,12,13,13,13,13,13,13,12,13,12,13,12,12,12,13,13,13,13,13,13,13,12,13,12,12,12,2,4,4,5,5,4,5,5,6,5,4,5,5,5,6,5,5,6,6,6,5,6,5,6,6,1,4,4,4,4,4,4,4,4,4,5,16,9,9,12,17,18,4,2,18,6,5,9,13,15,10,7,7,6,7,9,13,13,8,5,6,5,5,7,11,12,8,4,7,4,3,6,10,12,11,8,9,7,6,8,11,12,15,13,13,11,9,7,10,12,16,12,16,12,6,5,8,11,1,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,7,7,0,0,0,8,8,0,0,0,8,8,0,0,0,11,11,0,0,0,0,0,0,0,0,10,9,0,0,0,0,0,0,0,0,9,9,0,0,0,10,11,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,0,11,11,0,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,0,0,0,5,5,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,0,5,5,0,0,0,8,8,0,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,0,5,6,0,0,0,7,7,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,11,11,0,0,0,9,9,0,0,0,10,10,0,0,0,10,10,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,11,12,0,0,0,0,0,0,0,0,11,11,0,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,7,7,0,0,0,8,8,0,0,0,10,10,0,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,7,7,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,7,7,0,0,0,8,8,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,7,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,7,7,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,10,0,0,0,9,9,0,0,0,10,10,0,0,0,11,12,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,8,8,0,0,0,11,11,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,9,9,0,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,10,0,0,0,9,9,0,0,0,10,10,0,0,0,12,12,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,0,10,10,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,5,5,5,8,8,0,8,8,6,9,9,8,10,10,0,8,8,0,9,9,0,12,12,0,8,8,4,7,7,6,10,10,0,12,12,7,11,11,9,12,12,0,12,12,0,13,13,0,15,15,0,12,12,0,7,7,0,7,7,0,8,8,0,8,8,0,10,10,0,7,7,0,8,8,0,11,11,0,7,7,5,7,7,9,9,9,0,11,10,9,9,9,11,12,12,0,10,10,0,11,11,0,13,13,0,11,11,6,7,7,9,10,10,0,12,12,10,11,11,11,12,12,0,12,12,0,13,13,0,15,15,0,12,12,0,10,10,0,11,11,0,11,11,0,12,12,0,13,13,0,11,11,0,12,12,0,15,15,0,11,11,0,8,8,0,10,10,0,12,12,0,11,11,0,12,12,0,12,12,0,12,12,0,15,15,0,11,11,0,7,7,0,10,10,0,12,12,0,10,10,0,12,13,0,12,12,0,13,13,0,14,14,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,4,4,0,8,8,0,8,8,0,9,9,0,10,10,0,8,8,0,9,9,0,10,10,0,8,8,0,7,7,0,8,8,0,8,8,0,8,8,0,8,8,0,8,8,0,8,8,0,8,8,0,8,8,0,7,7,0,6,6,0,7,7,0,7,7,0,10,10,0,6,6,0,7,7,0,10,10,0,6,5,0,8,8,0,7,7,0,8,8,0,8,8,0,9,9,0,7,7,0,8,8,0,9,9,0,7,7,0,6,6,0,9,10,0,10,10,0,10,10,0,11,11,0,9,9,0,10,10,0,11,11,0,9,9,0,8,8,0,8,8,0,8,8,0,9,9,0,9,9,0,8,8,0,8,8,0,9,9,0,7,7,0,8,8,0,7,7,0,7,7,0,8,8,0,9,9,0,7,7,0,7,7,0,9,9,0,6,6,0,6,6,0,10,10,0,10,10,0,10,10,0,12,12,0,9,9,0,10,10,0,12,12,0,9,9,0,8,8,0,7,7,0,8,8,0,8,8,0,9,9,0,7,7,0,8,8,0,9,9,0,7,7,1,6,6,7,7,7,7,7,7,7,8,8,10,11,11,9,8,8,8,8,8,11,11,11,10,8,8,5,7,7,9,11,11,10,11,11,10,11,11,12,13,14,11,12,12,10,11,11,13,14,14,12,12,12,5,6,6,8,6,6,8,7,7,8,7,7,11,10,10,10,7,7,9,7,7,12,11,11,11,7,7,7,7,7,11,10,10,12,10,10,11,10,10,15,13,13,13,10,10,12,11,11,15,13,13,14,11,11,7,7,7,11,11,11,12,11,11,12,11,11,14,14,14,14,12,12,12,12,12,16,15,15,14,12,12,0,10,10,0,11,11,0,11,12,0,11,11,0,14,14,0,11,11,0,12,12,0,15,15,0,11,11,8,8,8,12,10,10,12,10,10,13,11,11,15,13,13,14,11,11,12,10,10,16,14,14,14,10,10,8,7,7,12,11,11,13,11,11,12,11,11,15,14,14,14,12,12,13,12,12,15,14,14,15,12,12,0,11,11,0,12,12,0,12,12,0,12,12,0,15,15,0,12,12,0,13,13,0,14,15,0,12,12,3,4,5,8,8,12,10,10,12,12,12,10,10,12,12,13,11,11,12,12,13,12,12,12,12,13,10,10,13,13,13,13,13,13,13,13,10,10,13,13,13,11,11,13,13,14,13,13,12,12,13,10,10,13,13,13,13,13,13,13,13,10,10,12,12,13,11,11,13,13,13,13,13,12,12,13,12,12,13,13,13,13,13,13,13,14,11,11,12,12,14,12,12,13,12,14,14,14,12,12,13,14,14,13,13,14,13,13,13,13,14,14,14,12,12,14,13,13,13,13,14,14,14,12,12,12,8,8,11,11,12,12,12,11,11,12,11,11,10,10,13,12,12,10,10,13,12,12,10,10,13,12,12,12,12,14,12,12,12,12,13,13,13,11,11,14,12,12,11,11,14,12,12,12,12,14,12,12,12,12,13,12,12,12,12,13,13,13,11,11,14,12,12,11,11,14,12,12,12,12,14,13,13,12,12,14,12,12,12,11,14,13,13,11,11,14,13,12,11,11,14,13,13,11,11,14,13,13,12,12,14,12,12,12,12,15,13,13,12,12,14,12,12,11,11,14,13,13,11,11,12,9,9,10,10,12,7,7,11,11,12,9,9,12,12,13,10,10,10,10,14,14,14,11,11,13,9,9,12,12,14,14,14,12,12,13,8,8,11,11,14,9,9,12,12,14,14,14,11,11,13,9,9,12,12,14,14,14,12,12,14,8,8,11,11,14,9,9,12,12,14,14,14,11,11,14,10,10,12,12,14,14,14,13,13,14,9,9,11,11,14,10,10,12,12,14,14,14,11,11,14,14,15,12,12,15,14,14,14,14,15,14,14,11,11,14,14,14,12,12,14,14,14,11,11,14,11,11,10,10,14,10,10,10,10,14,10,10,10,10,15,11,11,9,9,14,12,12,9,9,15,11,11,11,11,15,13,13,11,11,15,10,10,10,10,15,11,11,10,10,15,13,13,11,11,15,11,11,11,11,15,13,13,11,11,15,10,10,10,10,15,11,11,10,10,15,13,13,11,11,15,12,12,11,11,15,13,13,11,11,15,11,11,10,10,15,12,12,10,10,15,13,13,10,10,15,14,14,11,11,15,13,13,11,11,15,14,14,10,11,15,13,13,10,10,15,13,14,10,10,14,13,13,10,10,14,13,13,10,10,14,13,13,10,10,14,13,13,9,9,14,14,14,9,9,15,14,14,11,11,15,14,14,10,10,15,14,14,10,10,15,14,14,11,11,15,14,14,10,10,15,14,14,11,11,15,14,14,10,10,14,14,14,10,10,15,14,14,10,10,14,14,14,10,10,15,14,14,11,11,15,14,14,11,11,14,14,14,10,10,15,14,14,10,10,14,14,14,9,9,15,15,15,11,11,15,14,14,12,12,15,15,14,10,10,15,14,14,10,10,14,15,15,9,9,14,10,10,12,12,17,9,9,12,12,17,10,10,13,13,17,11,11,12,12,18,14,14,12,12,17,10,10,13,13,17,14,14,12,12,17,9,9,12,12,17,11,11,12,12,17,14,14,12,12,18,10,10,13,13,18,14,14,13,13,18,9,9,12,12,18,10,10,13,13,18,14,14,12,12,18,11,11,13,13,18,14,14,13,13,18,10,10,12,12,17,11,11,12,12,17,14,14,12,12,18,15,15,13,13,18,14,14,14,14,18,15,15,12,12,18,14,14,12,12,18,15,15,12,12,13,7,7,11,11,14,15,15,11,11,14,15,15,12,12,14,15,15,11,11,15,15,15,11,11,14,15,15,12,12,14,15,15,12,12,14,15,15,11,11,14,15,15,11,11,15,15,15,12,12,14,15,15,12,12,14,15,15,12,12,14,15,15,11,11,14,15,15,11,11,15,15,15,12,12,15,15,15,12,12,14,15,15,12,12,14,15,14,12,12,14,15,15,11,11,15,14,14,12,12,15,15,15,12,12,15,16,16,12,12,15,15,15,12,12,15,15,15,12,12,15,15,15,12,12,13,13,13,11,10,14,14,15,11,11,14,14,14,12,12,15,14,14,10,10,15,15,15,11,11,14,15,15,12,12,14,14,14,11,11,14,15,15,11,11,14,15,15,12,12,15,15,15,11,11,14,15,15,12,12,14,14,14,12,12,14,15,15,11,11,14,15,15,12,12,15,15,15,11,11,15,15,15,12,12,15,14,14,12,12,14,15,15,11,11,14,15,15,11,11,15,15,15,10,10,15,15,16,12,12,15,15,15,14,14,15,15,15,11,11,15,15,15,12,12,15,15,15,11,11,14,11,11,10,10,15,9,9,12,12,15,10,10,12,12,15,11,11,11,11,15,14,14,12,12,15,10,10,13,13,15,14,14,12,12,15,9,9,12,12,15,10,10,13,13,15,13,13,12,11,15,10,10,12,12,15,14,14,12,12,15,9,9,11,11,15,11,11,12,12,15,13,13,11,11,15,11,11,13,13,15,13,14,13,14,15,11,11,11,11,15,11,11,12,12,15,14,14,11,11,15,14,14,13,13,15,14,14,20,20,15,14,14,12,12,15,14,14,12,12,15,14,14,11,11,14,13,13,10,10,14,13,13,12,12,14,14,13,12,12,15,14,14,12,12,15,14,14,11,11,15,14,14,12,12,15,14,14,13,13,15,14,14,12,11,15,14,14,11,11,15,14,14,13,13,15,14,14,12,12,15,14,14,13,13,15,14,14,12,11,15,14,14,12,12,15,14,14,13,13,15,14,14,13,13,15,14,14,12,12,15,14,14,12,12,15,14,14,12,12,15,15,15,13,13,15,15,15,13,13,15,14,14,13,13,15,15,15,13,13,15,14,15,12,12,15,15,15,13,13,14,10,10,12,13,17,9,9,12,12,17,10,10,13,13,17,11,11,12,12,18,14,14,12,12,18,10,10,13,13,18,14,14,12,12,17,9,9,12,12,18,10,11,13,13,18,14,14,12,12,17,10,10,12,12,17,14,14,12,12,17,9,9,12,12,17,11,11,12,12,17,14,14,12,12,18,11,11,12,12,18,14,14,13,13,18,11,11,12,12,18,11,11,12,12,18,14,14,12,12,18,15,15,12,12,18,14,14,13,13,18,15,15,12,12,17,14,14,12,12,17,15,15,12,12,13,7,7,11,11,14,15,15,11,11,14,15,15,11,11,14,15,14,12,12,15,15,15,12,11,14,15,15,12,12,14,15,15,12,12,14,15,15,11,11,14,15,15,11,11,15,15,15,13,13,14,15,15,11,11,14,15,15,13,12,14,15,15,11,11,14,15,15,11,11,15,15,15,13,13,14,15,15,12,12,15,15,15,12,12,15,15,15,11,11,15,15,15,11,11,15,15,15,12,12,15,15,15,13,13,15,16,16,12,12,15,15,15,12,13,15,15,15,12,12,15,15,15,12,12,13,13,13,11,11,14,14,14,11,11,14,14,14,12,12,14,14,14,10,10,15,14,14,11,11,14,15,15,12,12,14,14,14,12,12,14,15,15,11,11,14,15,14,12,12,15,14,14,11,11,14,15,15,12,12,14,14,14,11,11,14,15,15,11,11,14,14,14,12,12,15,15,14,11,11,15,15,15,12,12,15,14,14,12,12,14,15,15,11,11,14,15,14,11,11,15,15,15,10,10,15,15,15,12,12,15,14,14,14,13,15,15,15,11,11,15,15,15,11,11,15,15,15,10,10,14,11,11,10,10,15,9,9,12,12,15,10,10,12,12,15,11,11,11,11,15,14,14,12,12,15,10,10,13,13,15,13,13,12,12,15,9,9,12,12,15,11,11,13,13,15,14,14,12,12,15,10,10,13,13,15,13,14,12,12,15,9,9,12,12,15,10,10,13,13,15,13,13,11,11,15,11,11,13,13,15,14,14,13,13,15,10,10,11,11,15,11,11,12,12,15,14,14,11,11,15,14,14,13,13,15,14,14,21,20,15,14,14,11,11,15,14,14,12,12,15,14,14,11,11,14,13,13,10,10,14,13,13,11,11,15,14,14,12,12,15,14,14,12,12,14,14,14,12,12,15,14,14,12,12,15,14,14,13,13,14,14,14,11,11,15,14,14,11,11,15,14,14,13,13,15,14,14,12,12,15,14,14,13,13,14,14,14,11,11,15,14,14,11,11,14,14,14,13,13,15,14,14,12,12,15,14,14,12,12,15,14,14,12,12,15,14,14,12,12,14,14,14,13,13,15,15,15,13,13,16,14,14,12,13,15,15,15,13,13,15,14,14,12,12,15,15,15,13,13,15,11,11,13,12,18,10,10,12,12,17,11,11,12,12,18,12,12,11,11,18,14,14,12,12,18,11,11,13,13,17,14,14,12,12,18,10,10,12,12,18,12,12,12,12,18,14,15,12,12,18,11,11,13,13,18,14,14,12,12,17,10,10,12,12,18,11,11,12,12,18,15,14,12,12,17,12,12,12,12,17,14,14,12,12,17,11,11,11,11,17,12,12,12,11,17,15,15,11,11,18,15,15,12,12,18,14,15,13,13,18,15,15,11,11,17,15,15,12,12,18,15,15,11,11,14,9,9,11,11,14,15,15,11,11,15,15,15,11,11,15,15,15,12,11,15,15,15,12,12,15,15,15,11,11,15,15,15,13,13,14,15,15,11,11,15,15,15,11,11,15,15,15,13,13,15,15,15,11,11,15,15,15,13,13,15,15,15,11,11,15,15,15,11,11,15,15,15,13,13,15,15,15,12,12,15,15,15,13,13,15,15,14,11,11,15,15,15,12,12,15,15,15,12,12,16,15,15,13,13,15,16,16,13,13,16,15,15,12,12,15,15,15,13,12,15,15,15,12,12,13,12,12,11,11,14,14,14,11,11,14,14,14,12,12,15,14,14,11,11,15,14,14,12,12,15,14,14,12,12,15,14,14,12,12,14,15,15,11,11,15,14,14,12,12,15,14,14,11,11,15,14,14,12,12,15,14,14,12,12,14,15,15,11,11,15,14,14,12,12,15,14,14,11,11,15,15,15,12,12,15,14,14,12,12,15,15,15,11,11,15,14,14,11,11,15,14,15,11,11,15,15,15,12,12,15,14,14,13,13,16,15,15,11,11,15,14,14,12,12,15,15,15,11,11,14,11,11,9,9,15,10,10,12,12,14,11,11,12,12,15,12,12,12,12,15,14,14,13,13,15,11,11,13,13,15,14,14,13,13,15,10,10,12,12,15,12,12,13,13,15,14,14,13,13,15,11,11,12,12,15,14,14,13,13,14,10,10,12,12,15,12,12,13,13,15,14,14,12,12,15,12,12,13,13,15,14,14,15,15,15,11,11,12,12,15,12,12,12,13,15,14,14,12,12,15,15,15,14,14,15,14,14,20,20,15,14,14,12,12,15,14,14,13,13,15,14,14,12,12,14,13,13,10,10,14,13,13,11,11,14,13,13,12,12,14,14,14,12,12,15,14,14,13,13,15,14,14,12,12,14,14,14,14,14,14,14,14,11,11,15,14,14,12,12,15,14,14,14,14,15,14,14,12,12,14,14,14,14,14,14,14,14,11,11,15,14,14,12,12,14,14,14,14,14,15,14,14,12,12,15,14,14,13,13,15,14,14,12,12,15,14,14,12,12,14,14,14,14,13,15,15,15,14,14,15,14,14,13,13,15,15,15,14,14,15,14,14,13,13,15,15,15,13,13,14,13,13,13,13,18,15,15,12,12,18,15,15,13,12,18,15,16,11,11,18,16,17,12,12,18,15,15,13,13,18,17,17,12,12,18,15,15,12,12,17,15,15,12,12,18,17,17,12,12,18,15,15,13,13,18,16,17,12,12,17,15,15,12,12,18,15,15,12,12,18,16,17,11,12,18,16,16,12,12,17,16,17,12,12,18,15,15,11,11,18,15,15,12,12,18,17,17,11,11,17,17,17,12,12,18,16,16,13,13,18,17,17,11,11,18,16,16,12,12,18,17,17,11,11,15,14,14,11,11,16,15,15,11,11,16,15,15,12,12,16,15,15,12,12,17,15,15,14,13,16,15,15,12,12,17,15,15,14,14,16,15,15,11,11,16,15,15,12,12,18,15,15,13,13,16,15,15,11,11,17,15,15,14,14,16,15,15,11,11,16,15,15,12,12,17,15,15,13,13,16,15,15,12,12,17,16,15,14,14,16,14,15,12,12,16,15,15,12,12,18,15,15,13,13,17,15,15,14,14,17,16,16,15,15,17,15,15,13,13,17,15,15,14,14,18,15,15,13,13,15,12,13,11,11,15,14,14,12,12,16,14,14,12,12,16,14,14,12,12,16,14,14,12,12,16,14,14,13,12,17,14,14,13,13,16,15,15,12,12,16,14,14,12,12,17,14,14,12,12,16,14,14,12,12,17,14,14,13,13,15,15,15,12,12,16,14,14,12,12,17,14,14,12,12,17,15,15,12,12,17,14,14,13,13,16,15,15,12,12,16,14,14,12,12,17,15,15,12,12,18,15,15,13,13,17,14,14,13,13,17,15,15,12,12,17,14,14,12,12,17,15,15,12,12,14,15,15,9,9,15,15,15,12,12,15,15,15,13,13,15,15,15,14,14,15,15,15,19,19,15,15,16,13,13,15,15,16,19,20,15,15,15,13,12,15,16,16,14,14,15,15,15,19,19,15,15,15,13,13,15,16,15,20,19,14,15,15,13,13,15,15,15,14,14,15,15,15,19,19,15,15,15,14,14,15,16,16,19,20,15,15,15,14,14,15,15,15,14,14,15,15,15,19,19,15,15,15,20,19,15,16,16,20,19,15,15,15,19,19,15,16,16,20,20,15,15,15,19,20,14,13,13,10,10,14,14,14,11,11,14,14,14,12,12,15,14,14,13,13,15,14,14,19,20,15,14,14,12,12,14,14,14,20,19,14,14,14,11,11,15,14,14,12,12,15,14,14,20,20,15,14,14,12,12,14,14,14,20,19,14,14,14,11,11,15,14,14,12,12,15,14,14,19,20,15,14,14,13,13,15,14,14,22,19,15,15,14,12,12,15,14,14,13,13,14,15,15,22,20,15,15,15,20,20,15,14,14,21,20,15,15,15,20,21,15,14,14,20,20,14,15,15,20,20,2,6,6,14,14,6,7,7,14,14,7,7,7,15,15,0,12,12,15,15,0,13,13,15,15,7,8,8,15,15,10,10,10,16,16,9,8,8,15,15,0,13,13,18,17,0,13,13,16,16,8,8,8,15,15,12,11,11,16,16,9,8,8,15,15,0,13,13,18,18,0,13,13,16,16,0,14,14,17,17,0,20,0,19,20,0,12,12,16,16,0,16,16,20,22,0,14,14,16,16,0,14,14,17,17,0,20,22,20,19,0,13,13,15,16,0,17,18,0,21,0,15,15,16,16,5,7,7,13,13,8,9,9,14,14,10,10,10,14,14,0,20,22,18,18,0,22,21,18,17,9,10,10,14,14,12,12,12,17,17,12,10,10,14,14,0,0,20,17,17,0,22,21,17,18,11,10,10,14,14,14,13,13,18,18,12,11,11,14,14,0,22,21,18,19,0,20,0,17,17,0,22,0,18,18,0,0,0,0,0,0,20,20,17,17,0,22,0,22,21,0,21,0,19,18,0,22,22,18,18,0,0,0,0,0,0,21,0,17,17,0,22,0,20,20,0,0,0,19,18,6,6,6,12,12,8,6,6,10,10,8,6,6,13,12,0,10,10,11,11,0,11,11,13,13,8,7,7,13,13,11,9,9,13,13,10,6,6,12,12,0,10,10,14,14,0,10,10,13,13,9,7,7,13,13,12,10,10,13,13,10,6,6,12,12,0,11,11,15,15,0,10,10,13,13,0,12,12,15,14,0,19,20,16,17,0,9,9,13,13,0,14,14,20,21,0,12,11,13,12,0,12,12,15,14,0,20,19,17,17,0,10,10,12,13,0,15,15,22,21,0,12,12,12,13,0,10,10,12,12,0,11,11,15,15,0,11,11,15,15,0,15,15,22,22,0,16,17,0,0,0,11,11,15,15,0,14,14,18,18,0,11,11,16,16,0,16,15,0,21,0,16,16,0,0,0,12,12,15,15,0,14,14,19,19,0,11,11,15,15,0,15,15,22,0,0,16,16,22,0,0,16,16,0,21,0,0,0,0,0,0,15,15,19,20,0,18,18,0,0,0,17,17,0,0,0,17,17,0,0,0,0,0,0,0,0,16,15,22,21,0,20,20,0,0,0,18,18,0,0,0,10,10,12,12,0,10,10,11,11,0,11,11,12,12,0,11,11,9,9,0,13,12,12,12,0,11,11,13,13,0,13,13,12,12,0,10,10,12,12,0,13,12,13,13,0,12,12,12,12,0,11,11,13,13,0,13,13,12,12,0,10,10,12,12,0,13,13,14,13,0,12,12,12,12,0,14,13,13,14,0,20,21,15,15,0,11,11,12,12,0,15,16,20,20,0,12,13,10,10,0,13,13,14,13,0,20,20,15,15,0,11,11,12,12,0,16,17,21,21,0,13,13,11,11,6,7,7,16,15,11,9,9,14,15,12,9,9,16,16,0,13,13,15,15,0,14,14,17,17,10,9,9,16,16,14,12,12,16,16,12,9,9,15,15,0,13,13,17,18,0,13,13,15,15,12,10,10,17,17,15,12,12,17,17,13,9,9,16,16,0,13,13,18,19,0,14,14,16,16,0,15,15,18,18,0,0,0,20,19,0,12,12,17,16,0,16,17,0,21,0,14,15,16,16,0,15,15,18,18,0,0,22,19,21,0,13,13,16,16,0,18,17,22,22,0,15,15,16,16,7,7,7,13,13,11,10,10,15,15,12,10,10,14,14,0,21,0,18,17,0,21,22,18,18,11,10,10,15,15,14,12,12,17,17,14,11,11,14,14,0,21,20,18,18,0,22,21,18,17,12,11,10,16,16,16,14,14,17,19,14,11,11,15,15,0,0,22,19,19,0,21,22,18,18,0,21,0,18,19,0,0,0,22,0,0,22,21,17,17,0,0,0,20,22,0,0,21,18,18,0,0,0,19,20,0,0,0,0,0,0,0,21,17,17,0,0,0,22,21,0,0,0,19,19,10,9,9,14,13,13,10,10,12,12,13,10,10,14,14,0,13,13,12,12,0,15,14,16,15,13,10,10,14,14,15,12,12,14,14,15,10,10,14,14,0,14,14,15,15,0,14,13,14,14,13,10,10,15,15,17,13,13,15,15,14,10,10,14,14,0,14,14,15,16,0,14,14,15,15,0,15,15,16,16,0,21,22,17,18,0,12,12,14,14,0,17,17,20,21,0,14,14,14,14,0,15,15,16,16,0,21,22,18,18,0,13,13,14,14,0,18,18,22,0,0,15,15,14,14,0,11,11,13,13,0,12,12,16,15,0,12,12,16,16,0,16,16,0,0,0,16,17,0,22,0,12,12,16,16,0,14,14,17,18,0,11,11,16,16,0,15,15,0,21,0,16,16,21,22,0,12,12,16,16,0,15,15,19,19,0,12,12,17,16,0,16,16,21,22,0,16,16,0,0,0,17,17,0,22,0,0,0,0,0,0,15,15,19,20,0,17,19,0,0,0,17,17,22,0,0,17,17,0,22,0,0,0,0,0,0,15,15,21,0,0,19,20,0,0,0,19,18,22,0,0,11,12,14,14,0,11,11,14,14,0,12,12,15,15,0,13,13,13,13,0,14,14,16,16,0,12,12,15,15,0,14,14,16,15,0,11,11,15,15,0,13,13,16,16,0,13,13,15,15,0,12,12,15,15,0,15,14,16,16,0,11,11,15,15,0,14,14,17,17,0,13,13,15,15,0,15,15,16,16,0,0,0,18,18,0,12,12,14,14,0,16,16,22,0,0,14,14,15,15,0,15,15,16,17,0,21,22,18,18,0,13,13,15,14,0,18,17,22,0,0,14,14,15,15,8,8,8,16,15,12,10,10,16,15,12,10,10,16,16,0,14,14,16,17,0,14,14,17,16,12,10,10,17,18,14,12,12,18,18,14,10,10,16,16,0,14,14,18,18,0,14,14,16,16,12,9,9,16,16,17,13,13,16,17,14,9,9,15,15,0,14,14,18,19,0,13,13,15,15,0,15,15,18,19,0,0,0,22,21,0,13,13,16,16,0,16,16,22,0,0,15,15,16,16,0,14,14,18,17,0,0,0,20,0,0,13,13,16,16,0,18,18,0,0,0,15,15,16,16,8,7,7,13,13,12,10,10,15,15,12,10,10,14,14,0,22,22,19,18,0,0,0,18,18,12,10,10,15,15,14,13,13,17,17,14,11,11,15,15,0,19,20,18,18,0,22,21,17,18,13,11,11,15,15,16,13,13,18,18,14,11,11,14,15,0,22,21,20,19,0,22,21,17,17,0,0,22,19,18,0,0,0,0,0,0,22,20,17,17,0,0,0,21,20,0,0,0,19,17,0,0,22,19,19,0,0,0,0,0,0,22,20,18,17,0,0,0,0,0,0,0,0,18,18,0,10,10,14,14,0,11,11,14,14,0,11,11,15,15,0,14,14,14,14,0,15,15,16,16,0,11,11,16,16,0,13,13,16,16,0,11,11,15,15,0,14,14,16,16,0,14,14,15,15,0,11,11,15,15,0,13,13,15,15,0,10,10,15,15,0,15,15,17,17,0,14,14,14,14,0,16,16,16,16,0,0,22,19,19,0,13,13,14,14,0,17,17,0,0,0,15,15,14,14,0,16,16,17,17,0,0,22,18,18,0,13,13,14,14,0,21,18,0,0,0,15,15,14,14,0,11,11,13,13,0,12,12,15,15,0,12,12,16,15,0,16,16,0,0,0,17,17,22,22,0,12,12,16,16,0,14,14,18,18,0,11,12,16,16,0,15,16,0,21,0,16,16,22,21,0,12,12,16,16,0,15,15,19,20,0,11,12,16,16,0,15,15,20,22,0,16,16,0,22,0,17,17,22,0,0,0,0,0,0,0,15,15,21,22,0,19,18,0,0,0,17,17,0,0,0,17,17,0,22,0,0,0,0,0,0,16,15,22,0,0,19,19,0,0,0,17,18,0,0,0,12,12,15,15,0,12,12,15,15,0,12,12,15,15,0,13,13,14,14,0,15,15,16,17,0,12,12,16,16,0,14,14,16,16,0,12,11,15,16], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+371956);
/* memory initializer */ allocate([0,14,14,16,17,0,14,14,16,16,0,13,12,16,16,0,15,15,16,16,0,11,11,15,15,0,14,14,16,16,0,14,14,15,15,0,15,15,18,17,0,0,22,0,20,0,13,13,15,15,0,16,17,22,22,0,14,14,15,15,0,15,15,17,18,0,20,0,19,19,0,13,13,15,15,0,18,18,22,0,0,14,14,15,15,0,11,11,16,16,0,14,14,17,16,0,13,13,17,17,0,16,16,17,17,0,17,17,18,19,0,12,12,16,17,0,15,15,18,18,0,12,12,16,16,0,16,16,19,18,0,16,16,17,16,0,12,13,17,17,0,17,16,18,17,0,13,12,16,16,0,16,16,18,19,0,16,16,16,17,0,16,16,18,18,0,22,0,22,22,0,13,13,16,16,0,19,18,22,20,0,16,15,16,16,0,16,17,18,18,0,0,0,22,20,0,14,14,16,16,0,19,19,0,0,0,16,16,16,16,0,9,9,13,13,0,13,13,15,15,0,14,14,15,15,0,0,22,17,18,0,22,0,18,19,0,12,12,15,15,0,15,16,17,17,0,14,14,14,14,0,22,0,18,18,0,21,22,17,17,0,13,13,15,15,0,17,17,17,18,0,14,14,15,15,0,22,21,21,19,0,20,21,17,17,0,21,21,19,18,0,0,0,0,0,0,21,21,17,17,0,0,0,22,22,0,0,22,19,18,0,0,21,19,18,0,0,0,0,22,0,19,20,17,17,0,0,0,0,22,0,0,0,19,18,0,19,19,15,16,0,21,19,16,17,0,0,21,17,17,0,0,22,17,17,0,22,22,18,19,0,20,20,16,16,0,0,22,18,18,0,20,19,16,17,0,22,21,20,19,0,0,21,17,17,0,21,20,17,17,0,0,0,18,18,0,19,19,17,16,0,22,0,19,19,0,21,22,17,18,0,0,22,19,18,0,0,0,19,20,0,19,19,16,16,0,22,22,22,0,0,20,22,16,16,0,22,20,18,19,0,0,0,20,19,0,20,20,16,16,0,0,0,0,0,0,22,20,17,16,0,11,11,13,13,0,14,13,15,15,0,13,13,16,15,0,18,17,21,0,0,18,18,21,0,0,12,12,15,15,0,15,16,17,18,0,12,12,15,15,0,17,17,22,20,0,17,18,22,0,0,12,12,17,16,0,16,17,19,19,0,13,13,16,16,0,17,17,0,22,0,17,17,0,21,0,18,18,20,22,0,0,0,0,0,0,15,15,21,20,0,20,19,0,0,0,18,18,22,0,0,17,17,22,0,0,0,0,0,0,0,15,16,20,22,0,20,21,0,0,0,19,18,0,0,0,15,15,19,19,0,17,16,20,20,0,16,17,20,21,0,18,17,0,0,0,19,19,0,0,0,15,15,21,19,0,19,19,0,0,0,15,15,22,22,0,18,18,0,22,0,17,18,22,21,0,15,15,20,19,0,19,19,0,0,0,15,15,20,22,0,18,19,20,0,0,18,17,21,21,0,18,18,19,22,0,0,0,0,0,0,15,15,20,19,0,19,19,0,0,0,18,18,21,22,0,18,18,22,0,0,0,0,0,0,0,15,15,19,20,0,21,21,0,0,0,17,17,20,20,0,12,12,17,17,0,14,14,16,17,0,13,14,17,17,0,16,16,17,17,0,17,17,17,19,0,13,13,17,17,0,16,16,18,18,0,13,13,16,16,0,16,16,18,18,0,16,16,17,17,0,13,13,17,17,0,17,17,18,17,0,12,12,15,16,0,17,18,19,20,0,16,16,16,16,0,17,16,18,19,0,0,22,21,22,0,14,14,16,16,0,19,19,0,0,0,16,16,16,16,0,16,16,18,17,0,0,22,21,21,0,14,14,16,16,0,22,20,22,0,0,16,16,15,15,0,9,9,13,13,0,14,14,15,15,0,14,14,14,14,0,22,22,18,18,0,0,22,18,18,0,12,12,15,15,0,16,16,18,17,0,14,14,14,14,0,20,21,18,18,0,22,21,17,17,0,13,13,15,15,0,17,17,18,18,0,14,14,14,14,0,0,21,18,19,0,0,22,17,17,0,22,22,19,18,0,0,0,0,0,0,19,21,17,17,0,0,0,22,20,0,0,21,18,19,0,0,22,18,18,0,0,0,0,22,0,20,22,17,17,0,0,0,20,22,0,0,0,18,18,0,19,21,16,16,0,20,22,16,17,0,20,0,17,17,0,22,0,18,17,0,21,0,18,19,0,20,20,17,17,0,22,0,18,18,0,21,20,17,17,0,0,20,20,19,0,0,21,18,17,0,21,21,17,17,0,22,0,18,17,0,19,19,17,17,0,0,22,20,21,0,0,21,17,17,0,22,0,18,18,0,0,0,20,22,0,20,19,16,16,0,0,0,0,0,0,22,22,17,17,0,22,0,18,19,0,0,0,21,20,0,19,21,16,17,0,0,0,0,0,0,22,22,17,16,0,11,11,13,13,0,13,13,15,15,0,13,13,15,15,0,17,17,22,21,0,18,18,22,0,0,12,13,16,15,0,15,16,18,18,0,13,13,16,16,0,17,17,0,22,0,17,17,22,22,0,13,13,16,16,0,16,16,19,18,0,13,13,16,16,0,18,17,0,20,0,18,17,20,0,0,17,17,21,0,0,0,0,0,0,0,15,15,21,22,0,19,20,0,0,0,18,18,0,0,0,18,17,0,0,0,0,0,0,0,0,16,16,22,22,0,20,20,0,0,0,21,19,0,0,0,15,15,20,19,0,16,16,22,20,0,17,17,0,22,0,18,18,0,22,0,19,17,0,0,0,15,16,22,20,0,18,19,0,0,0,16,16,22,20,0,18,18,0,22,0,18,18,22,0,0,16,16,21,20,0,19,20,0,22,0,16,16,0,22,0,18,18,0,22,0,18,18,0,21,0,19,18,0,22,0,0,0,0,0,0,16,16,21,20,0,20,0,0,0,0,18,18,21,0,0,18,18,0,0,0,0,0,0,0,0,16,16,21,19,0,0,0,0,0,0,18,18,0,21,2,3,3,3,3,3,3,1,6,6,7,7,7,7,8,8,7,9,9,11,11,11,9,8,8,8,9,9,12,11,11,9,8,8,6,7,7,10,11,10,10,10,10,11,11,10,14,13,14,12,11,11,11,11,11,15,14,14,13,12,12,5,6,6,8,5,5,8,7,7,8,8,8,12,10,10,9,7,7,9,7,8,12,10,10,10,7,7,7,8,8,12,10,10,12,10,10,11,10,10,15,13,13,13,10,10,11,10,10,16,13,14,14,10,10,7,7,7,12,11,11,12,11,11,11,11,11,16,15,15,14,12,12,12,11,11,16,15,16,14,12,12,10,9,9,14,11,11,13,11,11,12,11,11,16,14,14,14,11,11,12,11,11,17,15,15,14,11,11,7,8,8,12,11,11,12,10,10,12,10,10,16,14,13,14,10,10,12,10,10,17,14,14,14,10,10,8,7,7,13,11,11,12,11,11,12,11,11,16,15,14,14,12,12,12,11,11,16,15,14,15,12,12,11,10,10,13,11,11,13,12,11,13,11,11,17,14,14,14,11,11,13,11,11,17,14,15,14,11,11,2,6,6,7,7,7,7,7,7,7,8,8,9,9,9,9,7,7,8,8,8,9,9,9,9,7,8,6,7,7,8,8,8,8,8,8,9,8,8,10,9,9,10,8,8,10,8,8,10,9,9,10,8,8,6,6,6,8,6,6,8,7,7,8,7,7,10,8,8,9,7,7,9,7,7,10,8,9,9,7,7,7,7,7,10,8,8,11,8,8,10,8,8,12,9,9,12,8,8,11,9,9,12,9,9,11,8,8,7,7,7,10,9,9,10,9,9,10,9,9,11,10,10,10,9,9,11,9,9,11,10,10,11,9,9,9,8,8,10,9,9,10,9,9,11,9,9,11,10,10,11,9,9,11,9,9,11,10,10,11,9,9,8,8,8,11,9,9,11,9,9,11,9,9,12,9,9,12,8,8,12,9,9,12,9,9,12,8,8,8,7,7,10,9,9,10,9,9,11,9,9,11,11,11,11,9,9,11,10,10,11,11,11,11,9,9,10,9,9,11,9,9,11,9,10,11,10,9,11,10,10,11,9,9,11,9,10,11,10,10,11,9,9,1,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,1,9,9,6,9,9,5,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,7,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,1,3,2,5,4,7,7,8,8,9,9,10,10,11,11,12,12,13,13,14,14,15,15,15,15,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,4,13,9,9,12,15,17,4,2,18,5,7,10,14,18,11,8,6,5,6,8,11,14,8,5,5,3,5,8,11,13,9,6,7,5,5,7,9,10,11,10,9,8,6,6,8,10,14,14,11,11,9,8,9,10,17,17,14,13,10,9,10,10,1,3,2,3,1,4,4,8,8,8,8,9,9,10,10,10,10,4,6,5,8,7,9,9,9,9,10,9,11,9,4,5,6,7,8,9,9,9,9,9,10,9,10,8,9,8,9,8,10,9,11,9,12,10,12,10,8,8,9,8,9,9,10,9,11,10,12,10,12,9,10,10,11,10,12,11,12,11,12,12,12,12,9,10,10,11,11,11,11,11,12,12,12,12,12,10,11,11,12,12,12,12,12,12,12,12,12,12,10,11,11,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,13,12,13,12,13,12,12,11,12,12,12,12,12,12,13,12,12,12,12,12,12,12,12,13,13,12,13,12,13,12,13,12,12,12,13,12,13,12,13,12,13,12,13,12,12,12,3,4,4,5,5,4,4,5,5,5,4,5,4,5,5,5,5,6,5,6,5,6,5,6,5,1,4,4,4,4,4,4,4,4,3,5,16,9,9,13,18,21,4,2,21,6,6,10,15,21,16,19,6,5,7,10,13,16,8,6,5,4,4,8,13,16,8,5,6,4,4,7,12,15,13,10,9,7,7,9,13,16,18,15,13,12,9,7,10,14,21,18,13,13,7,5,8,12,1,2,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,9,9,0,0,0,8,8,0,0,0,9,9,0,0,0,12,12,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,0,11,11,0,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,0,0,0,5,5,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,0,5,5,0,0,0,7,7,0,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,0,5,5,0,0,0,7,7,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,11,11,0,0,0,9,9,0,0,0,10,10,0,0,0,10,10,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,12,12,0,0,0,0,0,0,0,0,11,11,0,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,7,7,0,0,0,8,8,0,0,0,10,10,0,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,10,0,0,0,7,7,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,0,0,0,7,7,0,0,0,8,8,0,0,0,10,11,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,0,0,0,0,0,7,7,0,0,0,9,9,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,5,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,10,0,0,0,7,7,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,7,8,0,0,0,9,9,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,9,9,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,10,0,0,0,9,9,0,0,0,10,10,0,0,0,11,11,0,0,0,0,0,0,0,0,9,9,0,0,0,0,0,0,0,0,8,8,0,0,0,10,10,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,9,9,0,0,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,7,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,10,10,0,0,0,9,9,0,0,0,10,10,0,0,0,12,12,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,9,9,0,0,0,10,10,0,0,0,0,0,0,0,0,10,10,0,0,0,0,0,0,0,0,8,8,0,0,0,10,10,0,0,0,0,0,0,0,0,11,11,0,0,0,0,0,0,0,0,10,10,0,0,0,11,11,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,6,6,5,7,8,0,8,8,6,9,9,7,10,10,0,8,8,0,9,9,0,12,12,0,8,8,4,7,7,6,10,10,0,12,12,7,11,11,8,12,12,0,12,12,0,13,12,0,15,15,0,12,12,0,7,7,0,7,7,0,7,7,0,8,8,0,10,10,0,7,7,0,8,8,0,11,11,0,7,7,5,7,7,8,9,9,0,10,10,8,9,9,11,11,11,0,10,9,0,11,11,0,13,13,0,10,10,6,7,7,8,10,10,0,12,12,9,10,10,10,12,12,0,12,12,0,12,12,0,15,15,0,12,12,0,10,10,0,11,11,0,11,11,0,11,11,0,13,13,0,11,11,0,11,11,0,15,15,0,10,10,0,8,8,0,10,10,0,12,12,0,11,11,0,12,12,0,12,12,0,12,12,0,15,15,0,11,11,0,7,7,0,10,10,0,12,12,0,10,10,0,12,12,0,12,12,0,13,13,0,14,14,0,12,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,5,5,0,8,8,0,8,8,0,9,9,0,10,10,0,8,8,0,8,8,0,10,10,0,8,8,0,7,7,0,8,8,0,7,7,0,8,8,0,8,8,0,8,8,0,8,8,0,8,8,0,8,8,0,7,7,0,6,6,0,7,7,0,7,7,0,10,10,0,6,6,0,7,7,0,10,10,0,5,5,0,8,8,0,7,7,0,8,8,0,8,8,0,9,9,0,7,7,0,8,8,0,9,9,0,7,7,0,6,6,0,9,10,0,10,10,0,10,10,0,11,11,0,9,9,0,10,10,0,11,11,0,9,9,0,8,8,0,8,8,0,8,8,0,9,9,0,9,9,0,7,7,0,8,8,0,9,9,0,7,7,0,8,8,0,7,7,0,7,7,0,8,8,0,9,9,0,7,7,0,7,7,0,8,8,0,6,6,0,6,6,0,10,10,0,10,10,0,10,10,0,12,12,0,9,9,0,10,10,0,12,12,0,9,9,0,8,8,0,7,7,0,7,7,0,8,8,0,9,9,0,7,7,0,8,8,0,9,9,0,6,6,1,6,6,6,7,7,7,8,8,7,8,8,10,11,11,9,8,8,8,8,8,11,11,12,9,8,8,5,7,7,9,11,11,10,11,11,10,11,11,12,14,14,11,12,12,10,12,12,13,14,14,12,12,12,5,6,6,7,6,6,8,7,7,8,7,7,11,10,10,10,7,7,9,8,8,12,11,11,10,7,7,7,7,7,11,10,10,12,10,10,11,10,10,15,13,13,13,10,10,12,11,11,15,13,13,14,11,11,7,7,7,11,11,11,12,11,11,12,11,11,14,14,14,13,12,12,12,12,12,16,15,15,14,12,12,0,10,10,0,11,11,0,12,12,0,11,11,0,14,14,0,11,11,0,12,12,0,15,15,0,11,11,7,8,8,12,11,10,12,10,10,12,11,11,15,13,13,14,11,11,12,10,10,16,14,14,14,10,10,8,7,7,12,11,11,12,11,11,12,11,11,15,14,14,14,12,12,13,12,12,15,14,14,15,13,13,0,11,11,0,12,12,0,12,12,0,12,12,0,15,15,0,12,12,0,13,13,0,15,14,0,12,12,4,5,5,9,9,12,9,9,12,12,12,10,10,13,13,13,11,11,12,12,13,13,13,12,12,13,10,10,13,13,13,13,13,13,13,13,10,10,13,12,13,11,11,13,13,13,14,14,13,12,13,10,10,13,13,12,13,13,13,13,13,10,10,12,12,13,11,11,13,13,13,14,14,12,12,13,12,12,13,13,13,13,13,13,13,13,11,11,12,12,13,11,11,13,13,13,14,14,12,12,13,14,14,13,13,14,13,13,14,14,14,14,14,12,12,13,14,14,13,13,14,14,14,12,12,12,8,8,12,12,13,12,12,11,11,13,11,11,11,11,14,12,12,11,11,14,12,12,10,11,14,12,12,12,12,14,12,12,12,12,13,13,13,11,11,14,12,12,11,11,14,12,12,12,12,14,12,12,12,12,14,12,12,12,12,14,13,13,11,11,14,12,12,11,11,14,12,12,12,12,14,13,13,12,12,14,12,12,12,12,14,13,13,11,11,14,12,12,11,11,14,13,13,11,11,15,13,13,12,12,14,12,12,12,12,15,13,13,12,12,14,12,12,11,11,15,13,13,11,11,12,9,9,11,11,13,7,7,11,11,13,8,8,12,12,14,10,10,10,10,14,14,14,11,11,14,8,8,12,12,14,14,14,12,12,14,7,7,11,11,14,9,9,12,12,14,14,14,11,11,14,8,8,12,12,14,14,14,12,12,14,7,7,11,11,14,9,9,12,12,14,14,14,11,11,14,10,10,12,12,14,14,14,13,13,14,9,9,11,11,14,10,10,12,11,15,14,14,11,11,14,15,15,12,12,15,14,14,14,14,15,14,14,11,11,15,14,14,12,12,15,14,14,11,11,14,11,11,10,10,15,10,10,10,10,15,10,10,10,10,15,11,11,9,9,15,12,13,9,9,15,11,11,11,11,15,13,13,11,11,15,10,10,10,10,15,11,11,10,10,15,13,13,11,11,15,11,11,11,11,15,13,13,11,11,15,10,10,10,10,15,11,11,10,10,15,13,13,10,11,15,12,12,11,11,15,13,13,11,10,15,11,11,10,10,15,11,12,10,9,15,13,13,10,10,15,14,14,11,11,15,13,13,11,11,15,14,14,10,10,15,13,13,10,10,15,14,14,10,10,14,13,13,10,10,15,13,13,10,10,15,13,13,10,10,14,14,14,8,9,15,14,14,9,9,15,14,14,11,11,15,14,14,10,10,15,14,14,10,10,15,14,14,11,11,15,14,14,10,10,15,14,14,11,11,15,14,14,10,10,15,14,14,10,10,15,14,14,10,10,15,14,14,9,9,15,14,14,11,11,15,14,14,11,11,15,14,14,10,10,15,14,14,10,10,14,14,14,9,9,15,15,15,11,11,15,14,14,12,12,15,15,15,10,10,15,14,15,10,10,15,15,15,9,9,15,10,10,13,13,17,8,8,12,12,17,10,9,13,13,18,11,11,12,12,18,14,14,12,12,17,9,9,13,13,17,13,13,12,12,18,8,8,12,12,18,10,10,12,12,18,14,14,12,12,18,10,10,13,13,18,13,13,13,13,18,9,9,12,12,18,10,10,13,13,18,14,14,12,12,18,11,11,13,13,18,14,14,13,13,18,10,10,12,12,17,11,11,12,12,18,14,14,12,12,18,14,14,13,13,18,14,14,13,13,19,14,15,12,12,18,14,14,12,12,18,15,15,12,12,13,7,7,11,11,14,15,15,11,11,14,16,15,11,11,14,15,15,11,11,14,15,15,11,11,14,15,15,11,12,14,15,15,12,12,13,15,15,11,11,14,15,15,11,11,15,15,15,12,12,14,15,15,12,12,14,16,16,12,12,14,15,15,11,11,14,15,15,11,11,15,15,15,12,12,15,15,15,12,12,14,15,15,12,12,14,15,15,11,11,14,15,15,11,11,15,14,15,12,12,15,15,15,12,12,15,16,16,12,12,15,15,15,12,12,14,15,15,12,12,15,15,15,12,12,13,13,13,11,11,14,14,15,11,11,14,14,14,12,12,14,15,15,10,10,15,15,15,11,11,14,15,15,12,12,14,14,14,11,11,14,15,15,11,11,14,15,15,12,12,15,15,15,11,11,14,15,15,12,12,14,14,15,11,11,14,15,15,11,11,14,15,15,12,12,15,15,15,11,11,15,15,15,12,12,14,15,15,12,12,14,15,15,10,10,14,15,15,11,11,15,15,15,10,10,15,15,15,12,12,15,15,15,14,14,15,15,15,11,11,15,15,15,11,11,15,15,15,11,11,14,10,10,10,10,15,9,9,12,11,15,10,10,12,12,15,11,11,11,11,15,13,13,12,12,16,10,10,12,12,15,13,13,12,12,15,9,9,11,11,15,10,10,13,12,15,13,13,11,11,15,10,10,12,12,15,13,13,12,12,15,9,9,11,11,15,10,10,12,12,15,13,13,11,11,15,11,11,12,12,15,13,13,13,13,15,10,10,11,11,15,11,11,12,12,15,13,14,11,11,15,14,14,13,13,16,14,14,20,19,15,14,14,11,11,15,13,14,12,12,15,14,14,11,11,14,13,13,10,10,14,14,13,11,11,15,13,14,12,12,15,14,14,12,12,15,14,14,11,11,15,14,14,12,12,15,15,14,13,13,15,14,14,11,11,15,14,14,11,11,15,14,14,13,13,15,14,14,12,12,15,14,14,13,13,15,14,14,11,11,15,14,14,11,11,15,14,14,13,13,15,14,14,12,12,15,14,14,12,12,15,14,14,12,12,15,14,14,11,11,15,15,15,12,12,15,15,15,13,13,16,14,14,12,12,15,15,15,13,13,15,15,15,12,12,15,15,15,12,12,14,10,10,13,13,17,9,9,12,12,17,9,9,13,13,17,11,11,12,12,18,14,14,12,12,18,10,10,13,13,18,14,13,12,12,18,9,9,12,12,18,10,10,12,13,18,14,14,12,12,17,9,9,12,12,17,13,14,12,12,17,9,9,12,12,17,10,10,12,12,17,14,14,11,11,18,11,11,12,12,18,14,14,12,13,18,10,10,12,12,18,11,11,12,12,18,14,14,11,11,18,15,15,12,12,18,14,14,13,13,18,14,15,12,12,17,14,14,12,12,17,15,15,12,12,13,7,7,11,11,14,15,15,11,11,14,15,15,11,11,14,15,15,11,11,14,15,15,11,11,14,15,15,11,11,14,15,15,12,12,14,15,15,11,11,14,15,15,11,11,15,15,15,12,12,14,15,15,11,11,14,15,15,12,12,14,15,15,11,11,15,15,15,11,11,15,15,15,12,12,14,15,15,12,12,14,15,16,12,12,14,15,15,11,11,14,15,15,11,11,15,15,15,12,12,15,15,15,12,12,15,16,16,12,12,15,15,15,12,12,15,15,15,12,12,15,15,15,12,12,13,13,13,12,12,14,14,14,11,11,14,14,14,12,12,14,14,14,10,10,15,15,15,11,11,14,15,15,12,12,14,14,14,11,11,14,15,15,11,11,14,14,14,12,12,15,15,14,11,11,14,15,15,12,12,14,14,14,11,11,14,15,15,11,11,14,14,14,11,11,15,14,14,10,10,14,15,15,12,12,14,14,14,12,12,14,15,15,10,10,14,15,15,11,11,15,15,15,10,10,15,15,15,12,12,15,14,14,13,13,15,15,15,10,10,15,14,14,11,11,15,15,15,10,10,14,10,10,10,10,14,9,9,12,12,15,10,10,12,12,14,11,11,11,11,15,13,14,12,12,15,10,10,13,13,15,13,13,12,12,15,9,9,12,12,15,10,10,13,13,15,13,14,11,11,15,10,10,12,12,15,13,13,12,12,15,9,9,11,11,15,10,10,12,12,15,13,13,11,11,15,11,11,12,12,15,13,13,13,13,15,10,10,11,11,15,11,11,12,12,15,14,14,11,11,15,14,14,13,13,15,14,14,20,19,15,14,14,11,11,15,14,14,12,12,15,14,14,11,11,14,13,13,11,11,15,13,13,11,11,15,14,13,12,12,15,14,14,11,12,15,14,14,11,11,15,14,14,12,12,14,14,14,13,13,15,14,14,11,11,15,14,14,11,11,15,14,14,13,13,15,14,14,12,12,15,14,14,13,13,14,14,14,11,11,15,14,14,11,11,15,14,14,13,13,15,14,14,12,12,15,14,14,12,12,15,14,14,12,12,15,14,14,11,11,14,14,14,12,12,15,15,15,13,13,16,14,14,12,12,15,15,15,13,13,15,14,14,12,12,15,15,15,12,12,15,11,11,13,13,18,10,10,12,12,17,11,11,12,12,18,12,12,11,11,18,14,14,12,12,18,10,10,13,13,18,14,14,12,12,18,10,10,12,12,18,11,11,12,12,18,14,14,12,12,18,11,11,12,13,18,14,14,12,12,18,10,10,12,12,18,11,11,12,12,18,14,14,11,11,18,11,11,12,12,18,14,14,12,12,17,10,10,11,11,17,12,12,11,11,17,14,14,11,11,18,15,15,12,12,18,14,14,13,13,18,15,15,11,11,18,15,14,12,12,18,15,15,11,11,14,8,8,11,11,14,15,15,10,10,14,15,15,11,11,14,15,15,11,11,15,15,15,12,12,15,15,15,11,11,15,15,15,12,12,14,15,15,10,10,15,15,15,11,11,15,15,15,12,12,15,15,15,11,11,15,15,15,13,13,14,15,15,10,10,15,15,15,11,11,15,15,15,12,12,15,15,15,12,12,15,16,16,12,12,15,14,14,11,11,15,15,15,11,11,15,15,15,12,12,16,15,15,13,13,15,16,16,13,13,16,15,15,12,12,15,15,15,12,12,15,15,15,12,12,14,13,13,11,11,14,14,14,11,11,14,14,14,12,12,15,14,14,11,11,15,15,14,11,11,15,14,14,12,12,15,14,14,12,12,14,15,15,11,11,15,14,14,12,12,15,14,14,11,11,15,14,15,12,12,15,14,14,12,12,14,15,15,11,11,15,14,14,11,11,15,14,14,11,11,15,15,14,12,12,15,14,14,12,12,15,15,15,10,11,15,14,14,11,11,15,15,15,10,10,15,15,15,12,12,16,14,14,13,13,15,15,15,11,11,15,14,14,11,11,15,15,15,11,11,14,11,11,9,9,14,10,10,12,12,15,11,11,12,12,15,12,12,12,12,15,14,14,13,13,15,11,11,12,12,15,14,14,13,13,14,10,10,12,12,15,11,11,13,13,15,14,14,12,12,15,10,10,12,12,14,14,14,13,13,14,10,10,11,11,15,11,11,12,12,15,14,14,12,12,15,12,12,13,13,15,14,14,14,14,15,11,11,11,11,15,12,11,12,12,15,14,14,11,11,15,15,15,13,14,15,14,14,20,19,15,14,14,12,12,15,14,14,13,13,15,14,14,12,12,14,13,13,10,10,14,13,13,11,11,14,13,13,11,11,15,14,14,12,12,15,14,14,12,12,15,14,14,12,11,14,14,14,13,13,15,14,14,11,11,15,14,14,11,11,15,14,14,14,14,15,14,14,11,12,15,14,14,13,13,14,14,14,11,11,15,14,14,11,11,15,14,14,14,14,15,14,14,12,12,15,14,14,13,13,15,14,14,11,11,14,14,14,12,12,15,14,14,13,13,15,15,15,13,13,15,14,14,13,13,15,15,15,13,13,15,14,14,13,13,15,15,15,13,13,15,14,14,13,13,18,15,15,12,12,18,15,15,12,12,18,16,16,11,11,18,17,17,12,12,18,15,15,13,13,18,17,17,12,12,18,15,15,12,12,18,15,16,12,12,18,17,17,12,12,18,15,15,13,12,17,16,17,12,12,17,15,15,11,12,18,15,15,12,12,18,17,17,11,11,18,16,16,12,12,18,17,16,12,12,18,15,15,11,11,18,15,15,12,12,18,17,17,11,11,18,17,17,12,12,18,16,16,13,13,18,17,17,11,11,17,16,16,11,11,18,17,17,11,11,15,15,15,11,11,16,15,15,11,11,16,15,15,11,11,16,15,15,12,12,17,15,15,14,14,16,15,15,11,11,17,15,15,14,14,16,15,15,11,11,16,15,15,12,12,18,15,15,13,13,16,15,15,11,11,17,15,15,14,14,16,15,15,11,11,16,15,15,12,12,17,15,15,13,13,16,15,15,12,12,17,16,15,14,14,16,15,15,11,11,16,15,15,12,12,18,15,15,13,13,17,15,15,14,14,17,16,16,15,15,18,14,15,13,13,18,15,15,14,14,18,15,15,13,13,15,13,13,12,12,15,14,14,12,12,16,14,14,12,12,16,14,14,12,12,17,14,15,12,12,16,14,14,12,12,17,14,14,13,13,16,15,15,12,12,16,14,14,12,12,17,14,14,12,12,16,14,14,12,12,17,14,14,13,13,15,15,15,11,11,16,14,14,12,12,17,14,14,12,12,16,15,15,12,12,17,14,14,13,12,16,15,15,11,11,16,14,14,12,12,17,15,15,11,11,17,15,15,13,13,17,14,14,13,13,18,15,15,12,12,17,14,14,12,12,17,15,15,12,12,14,15,15,9,9,14,15,15,12,12,15,16,15,13,13,15,15,15,14,14,15,15,15,21,19,15,15,15,13,13,15,15,15,19,19,15,15,15,12,12,15,16,16,14,14,15,15,15,19,19,15,16,15,13,13,15,16,16,19,20,15,15,15,12,13,15,16,16,14,14,15,15,15,20,19,15,15,15,14,14,15,16,16,19,19,15,15,15,14,13,15,15,15,14,14,15,15,15,19,19,15,16,16,20,19,15,17,16,21,20,15,15,15,20,19,15,16,16,20,20,15,15,15,19,20,14,13,13,10,10,14,14,14,11,11,14,14,14,12,12,15,14,14,13,13,15,15,14,20,20,15,14,14,12,12,14,14,14,19,19,15,14,14,11,11,15,14,14,12,12,15,14,14,20,19,15,14,14,12,12,14,14,14,20,20,14,14,14,11,11,15,14,14,12,12,15,14,14,20,21,15,14,14,13,13,15,14,14,20,20,15,14,14,12,12,15,14,14,13,13,14,15,15,20,20,15,15,15,20,19,15,14,14,20,19,15,15,15,20,20,15,14,14,21,20,15,15,15,20,20,1,7,6,15,15,7,8,8,15,15,8,8,8,15,15,0,13,13,16,16,0,14,14,16,16,7,9,9,16,16,10,11,11,17,17,10,8,8,15,16,0,14,14,18,18,0,14,14,16,16,9,9,9,16,16,12,11,11,17,17,10,9,9,15,15,0,14,14,19,19,0,14,14,16,16,0,15,15,18,17,0,0,0,20,20,0,13,13,16,16,0,17,17,22,20,0,15,15,17,17,0,15,15,18,18,0,22,21,20,21,0,13,13,16,16,0,18,18,0,22,0,15,15,17,17,6,7,7,13,13,9,10,10,15,15,11,10,10,15,15,0,21,22,18,18,0,0,0,18,18,10,10,10,15,15,12,13,13,17,17,12,11,11,15,15,0,22,22,18,18,0,0,21,18,18,12,11,11,15,15,15,14,14,18,18,13,11,11,15,15,0,0,21,18,19,0,21,22,18,19,0,22,0,18,19,0,0,0,0,0,0,21,21,18,18,0,22,0,0,21,0,0,0,19,18,0,0,0,18,19,0,0,0,0,0,0,20,20,18,17,0,0,22,0,21,0,0,0,19,19,6,6,6,13,13,8,6,6,11,11,9,7,7,13,13,0,10,10,11,11,0,12,12,14,14,9,8,8,14,14,12,10,10,13,13,10,7,7,13,13,0,11,11,15,15,0,11,11,13,13,9,8,8,14,14,13,10,10,13,14,11,7,7,13,13,0,11,11,15], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+382196);
/* memory initializer */ allocate([15,0,11,11,13,13,0,12,12,15,15,0,21,21,17,17,0,10,10,13,13,0,14,14,20,20,0,12,12,13,13,0,12,12,15,15,0,21,22,17,18,0,10,10,13,13,0,16,16,20,21,0,12,12,13,13,0,11,11,13,13,0,12,12,16,16,0,12,12,16,16,0,16,16,0,21,0,17,18,0,0,0,12,12,15,15,0,15,15,18,18,0,12,12,16,16,0,16,16,21,22,0,17,17,22,21,0,12,12,16,16,0,15,15,19,19,0,12,12,16,16,0,16,16,22,22,0,17,16,22,0,0,17,18,0,0,0,0,0,0,0,0,15,15,21,20,0,19,20,0,22,0,18,18,0,0,0,18,17,0,0,0,0,0,0,0,0,16,16,22,21,0,20,20,0,22,0,20,19,0,0,0,11,11,12,12,0,10,10,11,11,0,11,11,12,12,0,12,12,10,10,0,13,13,12,12,0,11,11,13,13,0,13,13,12,12,0,10,10,12,12,0,13,13,14,13,0,12,12,12,12,0,12,12,13,13,0,14,14,13,13,0,10,10,12,12,0,13,13,14,14,0,13,12,12,12,0,14,14,14,14,0,21,21,16,16,0,12,12,12,12,0,16,16,20,21,0,13,13,11,11,0,14,14,14,14,0,20,20,16,15,0,12,12,12,12,0,17,17,20,20,0,13,13,11,11,7,8,8,16,16,11,10,10,15,15,12,10,10,17,17,0,14,14,16,15,0,15,15,17,17,11,9,9,16,16,14,12,12,17,17,13,9,9,16,15,0,14,14,19,18,0,14,14,16,16,12,10,10,17,18,16,13,13,17,18,14,10,10,16,16,0,14,14,19,19,0,14,15,17,17,0,15,15,18,19,0,0,0,20,20,0,13,13,17,17,0,17,18,0,22,0,15,15,16,17,0,15,15,18,18,0,0,0,20,21,0,14,14,17,17,0,19,18,0,0,0,16,16,17,17,8,7,7,14,14,12,11,11,15,15,13,11,11,15,15,0,0,0,18,19,0,21,20,18,18,12,10,11,15,16,14,13,13,18,18,14,11,11,15,15,0,20,20,19,18,0,20,0,18,18,13,11,11,16,16,17,15,15,19,19,14,12,12,15,15,0,21,0,18,20,0,22,22,18,19,0,22,22,19,19,0,0,0,0,0,0,21,22,19,18,0,0,0,0,21,0,0,0,19,19,0,0,22,20,20,0,0,0,0,0,0,22,0,18,18,0,0,0,0,22,0,0,0,19,20,11,10,10,14,14,14,11,11,13,13,14,11,11,15,15,0,14,13,12,12,0,15,15,16,16,13,11,11,15,15,16,13,13,15,15,15,10,10,14,15,0,14,14,16,16,0,14,14,15,15,13,11,11,15,15,18,14,14,15,15,15,10,10,15,14,0,14,14,16,16,0,14,14,15,15,0,15,15,17,16,0,21,22,18,18,0,13,13,14,14,0,18,17,20,21,0,15,15,14,14,0,15,16,16,17,0,0,0,19,18,0,13,13,15,14,0,19,19,0,0,0,15,15,14,14,0,12,12,14,13,0,13,13,16,16,0,12,12,16,16,0,16,16,22,0,0,17,18,0,22,0,13,13,16,16,0,15,15,18,18,0,12,12,16,16,0,16,16,22,22,0,17,17,0,0,0,13,13,17,17,0,16,16,19,20,0,12,12,17,17,0,17,17,22,0,0,17,17,22,21,0,18,18,0,0,0,0,0,0,0,0,16,16,21,21,0,19,19,0,0,0,18,18,0,22,0,18,18,0,22,0,0,0,0,0,0,16,16,22,0,0,20,20,0,0,0,19,18,0,0,0,12,12,15,15,0,12,12,15,14,0,13,13,15,15,0,14,14,14,14,0,15,15,16,16,0,13,13,15,16,0,15,15,16,16,0,12,12,15,15,0,14,14,16,16,0,14,14,15,15,0,13,13,15,16,0,15,15,16,16,0,12,12,15,15,0,15,15,17,17,0,14,14,15,15,0,15,15,17,17,0,21,21,19,19,0,13,13,14,14,0,17,17,22,0,0,14,14,15,15,0,15,15,17,17,0,22,0,18,20,0,13,13,15,15,0,18,18,0,22,0,15,15,14,15,8,8,8,17,16,12,10,10,16,16,13,10,10,17,16,0,15,15,17,17,0,15,15,17,17,12,11,11,18,18,15,12,12,18,18,15,10,10,16,17,0,14,14,18,18,0,14,14,17,17,13,10,10,16,16,17,14,14,17,17,15,10,10,16,15,0,15,15,19,20,0,14,14,15,16,0,16,16,19,19,0,0,0,21,22,0,13,13,17,17,0,18,17,0,21,0,15,15,17,17,0,15,15,18,19,0,0,22,0,21,0,13,13,16,17,0,19,19,0,22,0,16,15,16,16,9,8,8,14,14,12,11,11,15,15,13,11,11,15,15,0,21,20,19,18,0,0,0,19,18,12,11,11,16,15,15,13,13,17,18,14,11,11,15,15,0,22,22,19,18,0,22,21,18,18,14,11,11,15,15,17,14,14,18,18,15,12,12,15,15,0,22,22,20,19,0,0,21,18,18,0,0,22,20,20,0,0,0,0,0,0,20,21,18,18,0,0,0,21,21,0,0,0,20,19,0,22,21,19,19,0,0,0,0,0,0,0,22,17,18,0,0,22,0,22,0,22,0,19,19,0,11,11,15,15,0,11,11,14,14,0,12,12,15,15,0,15,15,14,14,0,16,16,16,16,0,12,12,16,16,0,14,14,16,16,0,11,11,15,15,0,15,15,17,17,0,15,15,15,15,0,12,12,16,16,0,14,14,15,15,0,11,11,15,15,0,15,15,17,17,0,15,15,14,15,0,16,16,17,17,0,0,0,19,19,0,14,14,15,15,0,18,18,21,0,0,15,15,14,15,0,16,16,17,17,0,21,0,19,19,0,14,14,15,15,0,20,20,22,0,0,16,15,14,14,0,12,12,13,13,0,12,12,16,16,0,12,12,16,16,0,16,16,22,21,0,18,17,21,0,0,13,13,16,16,0,15,15,18,19,0,12,12,16,16,0,16,17,22,0,0,17,17,0,22,0,13,13,17,16,0,15,15,19,19,0,12,12,16,16,0,16,16,21,20,0,17,16,22,0,0,18,18,22,21,0,0,0,0,0,0,15,16,21,21,0,19,19,0,0,0,18,17,0,0,0,18,18,21,0,0,0,0,0,0,0,16,16,22,22,0,20,21,0,0,0,18,19,0,22,0,13,13,16,16,0,12,12,15,15,0,13,13,16,16,0,14,14,15,15,0,15,15,17,17,0,13,13,17,16,0,15,15,17,17,0,12,12,16,16,0,15,15,17,17,0,14,14,16,16,0,13,13,16,17,0,15,15,17,17,0,12,12,16,16,0,14,14,17,17,0,14,14,16,16,0,16,16,17,17,0,21,0,21,19,0,13,13,16,16,0,17,17,0,0,0,15,15,16,16,0,16,15,18,18,0,22,0,20,20,0,13,13,15,15,0,18,18,0,0,0,15,15,15,15,0,12,12,17,17,0,14,14,17,17,0,14,14,17,17,0,17,17,18,17,0,17,17,19,18,0,13,13,17,17,0,16,16,18,18,0,13,13,16,16,0,17,17,19,19,0,16,16,17,17,0,13,13,18,18,0,17,17,18,18,0,13,13,17,17,0,17,17,19,19,0,16,17,17,17,0,17,17,19,19,0,21,0,21,19,0,14,14,16,16,0,20,19,0,21,0,16,16,16,16,0,17,18,19,19,0,0,0,0,21,0,15,15,16,17,0,21,20,0,0,0,17,18,16,17,0,9,9,14,14,0,14,14,15,16,0,14,14,15,15,0,0,0,18,18,0,21,0,18,19,0,12,12,15,15,0,16,16,17,17,0,14,14,14,14,0,22,0,19,18,0,22,0,17,18,0,14,14,16,15,0,18,18,19,18,0,14,15,15,15,0,0,21,20,20,0,0,0,18,18,0,21,21,19,19,0,0,0,0,0,0,21,21,18,18,0,22,0,20,20,0,22,0,19,19,0,22,0,19,20,0,0,0,0,0,0,0,21,17,18,0,0,0,22,22,0,0,0,19,18,0,18,20,16,16,0,21,20,17,17,0,0,21,18,18,0,22,21,18,18,0,0,22,19,19,0,20,20,17,17,0,0,0,18,18,0,19,20,17,17,0,22,0,19,21,0,22,21,18,18,0,20,19,17,18,0,0,0,19,19,0,20,20,17,17,0,22,22,21,21,0,20,0,18,18,0,22,22,18,18,0,0,0,20,22,0,20,20,16,16,0,0,0,21,0,0,21,20,16,17,0,22,0,19,20,0,0,0,21,20,0,19,21,17,17,0,0,0,0,0,0,21,21,17,17,0,12,12,13,13,0,14,14,16,16,0,14,14,16,16,0,18,18,0,0,0,19,18,22,0,0,13,13,16,16,0,16,16,18,18,0,13,13,16,16,0,17,18,21,0,0,18,18,21,0,0,13,13,16,16,0,17,17,19,20,0,13,13,16,17,0,18,18,21,0,0,18,18,21,0,0,18,19,0,21,0,0,0,0,0,0,16,16,21,20,0,20,20,0,0,0,18,19,0,0,0,18,18,0,0,0,0,0,0,0,0,16,16,0,21,0,22,22,0,0,0,19,19,0,0,0,16,16,19,20,0,17,16,22,21,0,17,17,21,20,0,19,18,0,22,0,19,19,22,22,0,16,15,22,22,0,19,19,0,21,0,15,15,20,20,0,18,19,0,21,0,18,18,22,22,0,16,16,21,20,0,20,19,21,22,0,16,15,20,20,0,19,19,0,22,0,18,18,21,0,0,19,18,21,22,0,0,0,0,0,0,16,16,19,21,0,20,22,0,22,0,18,18,20,21,0,19,18,0,22,0,0,0,22,0,0,16,16,20,20,0,21,21,0,0,0,18,18,21,0,0,12,12,17,17,0,15,14,17,17,0,14,14,18,18,0,17,17,17,18,0,18,18,18,18,0,13,13,18,18,0,16,17,19,18,0,13,13,16,17,0,17,17,18,19,0,17,17,17,17,0,13,13,17,17,0,17,18,18,18,0,13,13,16,16,0,18,18,19,20,0,16,17,17,16,0,17,18,19,18,0,0,0,22,21,0,15,15,16,16,0,20,20,21,22,0,17,17,16,16,0,16,17,18,18,0,0,0,21,21,0,15,15,16,16,0,21,20,0,0,0,17,17,16,16,0,10,10,14,14,0,14,14,15,15,0,14,14,15,15,0,22,0,18,18,0,0,0,19,19,0,13,13,15,16,0,17,16,18,18,0,14,14,15,15,0,21,21,19,18,0,22,21,18,17,0,14,14,15,15,0,18,18,19,18,0,15,15,14,14,0,22,21,19,19,0,22,21,17,18,0,0,0,19,19,0,0,0,0,0,0,20,22,17,17,0,0,22,22,20,0,0,0,19,18,0,21,22,19,18,0,0,0,0,0,0,22,22,17,18,0,0,0,21,22,0,0,0,19,18,0,20,20,17,17,0,21,21,17,18,0,21,22,18,18,0,21,0,18,18,0,22,0,19,19,0,19,21,18,18,0,0,22,18,18,0,22,21,17,17,0,22,0,20,20,0,0,0,18,18,0,22,21,18,18,0,21,0,19,19,0,20,21,17,17,0,0,22,22,20,0,21,22,17,17,0,0,21,19,18,0,0,0,21,21,0,21,20,16,17,0,0,0,0,0,0,21,0,17,17,0,21,0,19,20,0,0,0,20,22,0,20,20,17,17,0,0,0,0,0,0,21,21,17,17,0,12,12,13,13,0,14,14,16,16,0,14,14,16,16,0,18,18,21,0,0,19,19,22,0,0,13,13,16,16,0,16,16,18,18,0,13,13,16,16,0,18,18,21,22,0,18,18,0,22,0,13,13,16,16,0,17,17,20,18,0,13,13,16,16,0,19,18,0,22,0,18,18,22,21,0,18,19,0,0,0,0,0,0,0,0,16,16,21,21,0,21,21,0,0,0,18,19,0,0,0,19,19,21,0,0,0,0,0,0,0,16,16,0,21,0,20,20,0,0,0,20,20,0,0,0,16,16,21,20,0,18,17,21,22,0,17,18,0,21,0,18,19,22,22,0,19,19,0,22,0,16,17,21,22,0,20,19,0,0,0,16,16,20,21,0,19,19,0,0,0,19,19,0,22,0,17,17,21,21,0,19,20,0,0,0,16,16,0,20,0,19,20,0,21,0,18,18,0,22,0,19,20,22,22,0,0,0,0,22,0,17,17,0,21,0,21,21,0,0,0,18,19,23,21,0,20,19,0,0,0,0,0,0,0,0,17,17,0,20,0,0,0,0,0,0,19,19,23,22,2,3,3,3,3,3,3,1,7,7,7,8,8,7,8,8,7,9,9,11,11,11,9,8,8,8,9,9,12,11,12,9,8,8,6,7,7,10,11,11,10,10,10,11,11,11,14,14,14,12,11,12,11,11,11,15,15,14,13,12,12,5,6,6,8,5,5,8,7,7,8,7,7,12,10,10,10,7,6,9,8,8,12,10,10,10,6,6,7,8,8,12,10,10,12,10,10,11,10,10,16,14,14,13,10,10,12,10,10,15,14,14,14,10,10,7,7,7,13,11,11,13,11,11,12,11,11,16,14,14,14,12,12,12,11,11,18,15,15,14,12,12,10,9,10,14,11,11,13,11,11,12,11,11,17,14,14,14,11,11,13,11,11,16,15,15,14,11,11,7,8,8,13,11,11,12,10,10,12,10,10,16,14,13,13,10,10,12,10,10,17,14,14,14,10,10,8,7,7,12,11,11,13,11,11,12,11,11,16,15,14,14,12,12,12,11,11,16,15,15,14,12,12,11,10,10,14,11,11,13,11,11,13,11,11,17,14,14,14,11,11,13,11,11,18,14,15,15,11,10,2,6,6,6,7,7,7,7,7,7,8,8,9,9,9,9,7,7,8,8,8,9,9,9,9,8,8,6,7,7,8,8,8,8,8,8,9,8,8,9,8,9,9,8,8,10,8,8,10,9,9,10,8,8,6,6,6,8,6,6,8,7,7,8,7,7,10,8,8,9,7,7,9,7,7,10,8,8,9,7,7,7,7,7,10,8,8,11,9,9,10,9,9,11,9,9,11,8,8,11,9,9,12,9,9,12,8,8,7,7,7,10,9,9,10,9,9,10,9,9,11,10,10,10,9,9,11,9,10,11,10,11,10,9,9,9,8,8,10,9,9,10,9,9,11,9,9,11,10,10,11,9,9,11,9,9,11,10,10,11,9,9,8,8,8,11,9,9,11,9,9,11,9,9,12,9,9,12,8,8,11,9,9,12,9,9,12,8,8,8,7,7,10,9,9,10,9,9,10,9,9,11,11,11,11,9,9,11,10,10,11,11,11,11,9,9,10,9,9,11,9,9,11,9,10,11,10,10,11,10,10,11,9,9,11,10,10,11,10,10,11,9,9,1,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,1,9,9,7,9,9,8,8,9,9,9,9,9,9,9,8,9,9,7,9,9,9,9,9,9,9,9,7,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,6,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,5,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,8,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,5,10,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,8,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,1,3,2,5,4,7,7,8,8,9,9,10,10,11,11,12,12,13,13,14,14,15,15,15,15,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,5,13,9,9,12,16,18,4,2,20,6,7,10,15,20,10,7,5,5,6,8,10,13,8,5,5,3,5,7,10,11,9,7,6,5,5,7,9,9,11,10,8,7,6,6,8,8,15,15,10,10,9,7,8,9,17,19,13,12,10,8,9,9,1,3,2,3,1,4,4,8,8,9,8,9,9,10,10,10,10,4,6,5,8,7,9,9,9,9,10,9,10,10,4,5,6,7,8,9,9,9,9,9,10,9,10,8,9,8,9,8,10,9,11,9,12,10,11,10,8,8,9,8,9,9,10,9,11,10,11,10,12,9,10,10,11,10,11,11,12,11,12,12,12,12,9,10,10,11,11,11,11,11,12,12,12,12,12,10,11,11,12,12,12,12,12,12,12,12,12,12,10,11,11,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,13,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,12,12,11,13,12,12,12,13,12,12,12,12,12,12,12,3,4,4,5,5,4,4,5,5,5,4,5,4,5,5,5,5,6,5,6,5,6,5,6,5,1,4,4,4,4,4,4,4,4,4,7,12,14,15,18,20,20,5,3,4,6,9,11,15,19,9,4,3,4,7,9,13,18,11,6,3,3,5,8,13,19,14,9,6,5,7,10,16,20,16,11,9,8,10,10,14,16,21,14,13,11,8,7,11,14,21,14,13,9,6,5,10,12,2,5,5,5,7,7,5,7,7,5,7,7,7,8,9,7,9,9,5,7,7,7,9,9,7,9,8,5,7,8,8,9,10,8,9,10,8,9,10,9,10,12,10,11,11,8,10,10,10,11,11,9,11,11,5,8,7,8,9,9,8,10,9,8,10,10,9,11,11,10,11,11,8,10,9,10,11,11,9,12,10,5,8,8,7,9,10,8,10,9,7,9,9,9,10,11,9,11,11,8,10,9,10,11,11,10,11,11,7,9,9,9,10,11,9,11,11,9,9,11,10,10,13,11,11,12,9,11,11,11,12,13,11,13,12,7,9,9,9,11,11,9,11,10,9,11,10,10,11,12,11,13,12,9,11,11,11,12,13,11,13,11,5,8,8,8,9,10,7,10,9,8,9,10,10,11,11,10,11,11,7,9,9,9,11,11,9,11,10,7,9,9,9,10,11,9,11,11,9,11,11,11,11,13,11,13,12,9,10,11,11,12,13,10,12,11,7,9,9,9,11,11,9,11,10,9,11,11,11,12,13,11,13,12,9,11,9,11,12,11,10,13,10,4,6,6,9,9,6,7,8,10,10,6,8,7,10,10,8,10,10,12,13,8,10,10,13,12,6,7,8,10,10,7,8,9,10,11,8,9,9,11,11,10,10,11,12,14,10,11,11,14,13,6,8,7,10,10,8,9,9,11,11,7,9,8,11,10,10,11,11,13,14,10,11,10,14,12,9,10,10,12,12,10,10,11,12,13,10,11,11,13,13,12,12,13,12,15,13,14,13,15,14,9,10,10,12,12,10,11,11,13,13,10,11,10,13,12,13,13,14,14,15,12,13,12,15,12,6,7,8,10,11,8,9,10,11,12,8,9,9,11,12,10,11,12,13,14,10,11,11,14,13,8,9,10,11,12,9,10,11,12,13,9,10,11,12,13,11,12,13,13,15,12,12,13,15,14,8,9,9,12,12,9,10,11,12,13,9,10,10,13,12,12,12,13,14,15,11,12,12,14,14,11,11,12,13,14,11,12,13,13,15,12,13,13,14,15,14,13,15,14,16,14,15,15,16,16,11,12,11,14,13,12,13,13,15,14,11,13,12,14,13,14,15,15,15,16,13,14,14,16,14,6,8,7,11,10,8,9,9,11,12,8,10,9,12,11,10,11,11,13,14,10,12,11,14,13,8,9,9,12,12,9,10,10,12,13,9,11,10,13,12,11,12,12,13,14,12,13,12,15,14,8,10,9,12,11,9,11,10,13,12,9,11,10,13,12,12,13,12,14,15,11,13,12,15,13,11,11,12,13,14,11,12,13,13,15,12,13,13,14,15,13,14,14,14,16,14,15,15,16,16,11,12,11,14,13,12,13,13,15,14,11,13,12,15,13,14,15,15,16,16,13,15,13,16,14,9,10,11,12,14,11,11,12,13,15,11,12,12,13,14,13,14,15,15,17,13,14,14,15,16,11,11,12,13,15,12,12,13,14,16,12,13,13,14,15,14,14,16,15,17,15,15,15,16,17,11,12,12,14,14,12,13,13,15,16,12,13,13,15,15,15,15,15,16,17,14,15,15,16,16,14,14,15,15,17,14,15,15,15,17,15,15,16,16,17,16,16,17,16,18,17,17,17,18,18,14,15,14,16,16,15,15,16,17,17,14,15,15,17,16,17,17,17,18,18,16,16,16,17,17,9,11,10,14,12,11,12,12,14,13,11,12,11,15,13,13,14,14,16,15,13,15,14,17,15,11,12,12,15,14,12,13,13,15,15,12,13,13,15,15,14,15,15,16,16,15,15,15,17,16,11,12,11,15,13,12,13,13,15,14,12,13,12,16,14,15,15,15,17,16,14,15,14,17,15,14,14,15,16,16,14,15,15,16,16,15,16,15,17,17,16,16,16,17,17,17,17,17,18,17,14,15,14,16,15,15,15,15,17,16,15,15,15,17,15,17,17,17,18,18,16,17,16,18,16,6,8,8,11,11,8,9,9,11,12,8,9,9,12,11,10,11,11,13,14,10,12,11,14,13,7,9,9,11,12,9,10,10,12,13,9,10,10,13,13,11,11,12,13,15,11,12,12,15,14,8,9,9,12,11,9,11,10,13,13,9,11,10,13,12,12,13,12,14,15,11,13,12,15,13,10,11,12,13,14,11,12,12,13,15,12,12,13,14,15,13,13,14,14,16,14,15,15,16,16,11,12,11,14,13,12,13,13,15,14,11,13,12,15,13,14,15,15,15,16,13,14,14,16,14,7,9,9,11,12,9,10,11,12,13,9,10,10,13,12,11,12,12,14,15,11,12,12,15,14,9,9,11,11,13,10,10,12,12,14,10,11,12,13,14,12,12,13,14,16,12,13,13,15,15,9,11,10,13,13,10,12,12,13,14,10,12,11,14,13,12,13,13,15,16,12,13,13,15,14,11,11,13,13,15,12,12,14,13,16,13,13,13,14,15,14,14,15,14,17,15,15,15,16,16,12,13,12,15,14,13,14,14,15,15,12,14,13,16,14,15,15,16,16,17,14,15,14,17,15,7,9,9,12,11,9,10,10,12,13,9,11,10,13,12,11,12,12,14,14,11,13,12,15,14,9,10,10,13,12,10,10,11,12,13,10,12,11,14,13,12,12,13,13,15,12,14,13,16,15,9,10,10,13,12,11,11,12,13,13,10,12,10,14,12,13,13,13,15,15,12,13,12,15,13,11,12,12,14,14,12,12,13,14,15,13,14,13,15,15,14,13,15,13,16,15,16,15,17,16,12,13,12,14,14,13,14,14,15,15,12,13,12,15,14,15,15,16,16,17,14,15,13,16,13,10,11,12,13,14,11,12,13,14,15,12,13,13,15,15,14,14,15,15,17,14,15,15,16,16,12,12,13,12,15,12,12,14,13,16,13,13,14,14,16,14,14,16,15,17,15,15,16,16,17,12,13,13,15,15,13,14,14,16,16,13,14,13,16,15,15,16,16,17,17,14,15,15,17,16,14,14,15,14,17,15,15,16,15,17,15,15,16,15,17,16,16,17,16,18,17,17,17,17,18,14,15,15,17,16,15,16,16,17,17,15,16,15,17,16,17,17,17,18,18,16,17,16,18,17,10,12,11,14,14,12,13,13,15,15,12,13,12,15,14,14,15,15,16,16,14,15,15,17,16,11,13,12,15,14,12,13,13,15,15,13,14,13,16,14,15,15,15,16,16,15,16,15,17,16,12,13,13,15,15,13,14,14,16,16,12,14,13,16,15,15,16,16,17,17,15,16,15,17,16,14,15,15,16,16,14,15,15,16,16,15,16,16,17,16,16,16,16,16,17,17,18,17,18,17,14,15,15,17,16,15,16,16,17,17,15,16,15,17,16,17,17,18,18,18,16,17,16,18,16,6,8,8,11,11,8,9,9,11,12,8,9,9,12,11,10,11,12,13,14,10,11,11,14,13,8,9,9,11,12,9,10,11,12,13,9,10,11,13,13,11,12,13,13,15,12,12,12,15,14,7,9,9,12,11,9,10,10,13,13,9,10,10,13,12,11,12,12,14,15,11,12,11,15,13,11,11,12,13,14,11,12,13,13,15,12,13,13,14,15,13,14,14,14,16,14,15,15,16,16,10,12,11,14,13,12,13,12,14,14,11,12,12,15,13,14,15,15,16,16,13,14,13,16,14,7,9,9,11,12,9,10,11,12,13,9,10,10,13,12,11,12,13,14,15,11,12,12,14,14,9,10,10,12,13,10,10,12,12,14,11,12,11,13,13,12,12,14,13,15,13,13,13,15,15,9,10,10,12,13,10,11,12,13,14,10,11,10,13,12,13,13,14,15,16,12,13,12,15,13,12,13,13,14,14,12,12,13,14,15,13,14,14,15,15,14,13,15,13,16,15,16,15,17,16,11,12,12,14,14,13,13,14,15,15,12,13,12,15,14,15,15,16,16,17,14,14,13,16,13,7,9,9,12,11,9,10,10,12,13,9,11,10,13,12,11,12,12,14,15,11,12,12,15,14,9,10,11,13,13,10,11,12,13,14,10,12,12,14,13,12,13,13,14,16,12,13,13,16,15,9,11,9,13,11,10,12,11,13,13,10,12,10,14,12,12,13,13,15,15,12,13,12,16,14,12,12,13,14,15,12,13,14,14,15,13,14,14,15,15,14,14,15,15,17,15,16,15,17,16,11,13,11,15,13,13,14,13,15,14,12,14,12,16,13,15,15,15,16,16,14,15,14,17,14,10,11,12,14,14,12,12,13,14,15,12,13,13,15,15,14,15,15,16,17,14,15,15,16,16,12,12,13,15,15,13,13,14,15,16,13,14,14,16,16,15,15,16,16,17,15,16,16,17,17,11,12,13,14,15,13,13,14,15,16,12,13,13,15,15,15,15,16,16,17,15,15,15,16,16,14,15,15,16,17,15,15,16,16,17,15,16,16,17,17,16,16,17,16,18,17,17,17,18,18,14,15,15,16,16,15,16,16,16,17,15,15,15,16,16,17,17,17,18,18,16,16,16,17,16,10,12,11,14,13,12,13,13,15,15,11,13,12,15,14,14,15,15,16,16,14,15,14,17,15,12,13,13,15,15,13,13,14,16,16,13,14,14,16,16,15,15,15,16,17,15,16,16,17,17,12,13,12,15,12,13,14,13,16,14,12,14,12,16,13,15,16,15,17,16,14,16,14,17,15,14,15,15,16,17,15,15,16,17,17,15,16,16,17,17,16,16,17,17,18,17,18,17,18,18,14,15,14,17,14,15,16,15,17,15,15,16,15,17,15,17,17,17,18,17,16,17,16,18,16,9,11,11,14,14,11,12,12,14,14,11,12,12,15,14,13,14,14,16,16,13,15,14,16,16,10,11,12,14,14,11,12,13,15,15,12,13,13,15,15,13,14,15,16,17,14,15,15,17,16,11,12,12,15,14,12,13,13,15,15,12,13,13,15,15,14,15,15,16,16,14,15,15,17,16,12,13,14,15,16,13,14,14,15,16,13,14,15,16,16,15,15,16,16,18,16,16,16,18,17,14,14,14,16,15,15,15,15,17,16,14,15,15,17,16,16,17,17,18,17,16,16,16,18,16,10,12,12,14,14,11,12,13,15,15,12,13,13,15,15,13,14,15,16,17,14,15,15,17,16,11,12,13,14,15,12,12,14,15,16,13,13,14,15,16,14,14,15,16,17,15,15,16,17,17,12,13,13,15,15,13,14,14,16,16,13,14,13,16,15,15,16,15,17,17,15,16,15,17,16,13,13,15,14,17,14,13,16,15,17,15,14,16,15,17,15,15,17,16,18,16,16,17,17,18,14,15,15,17,16,15,16,16,17,17,15,16,15,17,16,17,17,17,18,18,16,17,16,18,17,10,12,11,14,14,11,12,13,15,15,12,13,12,15,15,14,15,15,16,16,14,15,15,17,16,11,12,12,15,15,12,13,13,15,15,13,14,13,16,15,14,15,15,16,16,15,16,15,17,16,11,13,13,15,15,13,14,14,15,15,12,14,13,16,15,15,16,15,17,17,15,16,15,17,16,13,15,14,16,16,14,15,14,16,16,15,16,15,17,16,15,16,16,16,17,16,17,16,18,17,14,15,15,16,16,15,16,16,17,17,15,15,15,17,16,17,17,17,18,18,16,16,16,18,16,12,13,13,15,16,13,14,14,15,16,13,14,14,16,16,15,15,16,16,18,15,16,16,17,17,13,13,14,15,16,14,14,15,15,17,14,15,15,16,17,15,15,17,16,18,16,16,17,17,17,13,14,14,16,16,14,15,15,17,17,14,15,14,17,16,16,17,16,17,18,16,17,16,18,17,15,15,16,14,17,16,15,17,14,18,16,16,16,15,18,16,16,18,15,19,18,18,18,17,19,15,16,16,18,17,16,17,17,18,17,16,17,16,18,17,18,18,18,19,19,17,18,16,18,17,11,12,12,15,15,13,13,14,15,16,13,14,13,16,15,15,16,16,16,17,15,16,16,17,16,12,14,13,16,15,13,13,14,15,16,14,15,14,17,15,15,15,16,16,17,16,17,16,18,17,12,13,14,15,16,14,15,15,16,16,13,14,13,16,15,16,16,16,17,17,15,16,15,17,15,15,16,15,17,16,15,15,15,16,16,16,17,16,18,16,16,15,16,15,17,17,18,17,18,17,15,15,16,17,17,16,16,17,17,17,15,16,15,17,16,18,18,18,18,18,16,17,16,18,15,9,11,11,14,14,11,12,12,14,15,10,12,12,15,14,13,14,15,16,16,13,14,14,16,16,11,12,12,14,15,12,12,13,15,15,12,13,13,15,15,14,15,15,16,17,14,15,15,16,16,10,12,12,14,14,12,13,13,15,15,11,13,12,15,15,14,15,15,16,17,13,15,14,16,16,14,14,14,15,16,14,15,15,16,17,14,15,15,16,17,16,16,17,16,18,16,17,17,17,17,12,14,13,16,15,13,15,14,16,16,13,14,14,16,15,16,16,16,17,17,15,16,15,17,16,10,11,11,14,14,12,12,13,14,15,11,13,12,15,14,14,15,15,16,17,14,15,15,16,16,12,13,13,15,15,12,13,14,15,16,13,14,14,15,15,15,15,16,16,17,15,15,16,17,17,11,12,12,15,15,13,13,14,15,16,12,13,13,15,15,15,15,16,16,17,14,15,15,16,16,14,15,15,16,16,15,15,15,16,17,15,16,16,17,17,16,16,17,16,18,17,17,17,17,18,13,14,15,16,16,15,15,16,16,17,14,14,14,16,16,16,16,17,17,18,16,16,16,17,16,10,12,12,14,14,12,13,13,15,15,11,13,12,15,15,14,15,15,16,17,13,15,14,17,16,12,13,13,15,15,13,13,14,15,16,13,14,14,16,16,15,15,16,16,17,15,15,16,17,17,11,13,12,15,14,13,14,13,16,15,12,14,12,16,15,15,16,15,17,17,14,15,14,17,16,14,15,15,16,17,15,15,16,16,17,15,16,16,17,17,16,16,17,17,18,17,17,17,18,18,13,15,13,17,14,14,16,14,17,16,14,15,13,17,15,16,17,16,18,17,15,17,15,18,16,11,12,12,15,15,13,13,14,15,16,13,14,13,16,15,15,16,16,16,17,15,16,16,17,16,12,14,13,16,15,13,13,14,15,16,14,15,15,16,16,16,15,16,16,17,16,16,16,17,17,12,13,14,15,16,14,14,15,15,17,13,14,13,16,15,16,16,17,17,18,15,16,15,17,15,15,16,15,17,17,15,15,16,16,17,16,17,16,17,17,16,15,17,15,18,17,18,17,18,18,15,15,16,16,17,16,16,17,16,18,15,15,15,16,16,17,17,18,17,18,16,16,15,17,15,12,13,13,15,15,13,14,14,16,16,13,14,14,16,16,15,16,16,17,18,15,16,15,18,16,13,14,14,16,16,14,14,15,16,17,14,15,15,17,17,16,16,17,17,18,16,16,17,18,17,13,14,13,16,14,14,15,15,17,16,14,15,14,17,15,16,17,17,18,17,15,17,15,18,16,15,16,16,17,17,16,16,17,17,18,16,17,17,18,18,17,16,18,17,19,18,18,18,18,18,15,16,15,17,14,16,16,16,18,15,16,17,15,18,14,18,18,18,18,17,17,18,16,19,15,1,5,6,5,7,8,5,8,7,5,7,8,7,8,10,8,10,10,5,8,7,8,10,10,7,10,8,6,8,9,8,10,11,9,10,10,9,10,11,10,11,12,11,12,12,9,11,10,11,12,12,10,12,11,6,9,8,9,10,10,8,11,10,9,10,11,10,11,12,11,12,12,9,11,10,11,12,12,10,12,11,6,9,9,8,10,11,9,11,10,8,10,10,10,10,12,11,12,12,9,11,10,11,12,12,10,12,11,8,10,10,10,11,12,10,12,11,10,10,12,11,11,13,12,13,13,10,12,11,12,13,13,11,13,11,7,10,10,10,11,12,10,12,11,10,12,11,11,11,12,12,14,13,10,12,12,12,14,14,11,13,11,6,9,9,9,10,11,8,11,10,9,10,11,10,11,12,11,12,12,8,11,10,11,12,12,10,12,10,7,10,10,10,11,12,10,12,11,10,12,12,11,11,13,12,13,13,10,11,12,12,13,14,11,12,11,8,10,10,10,11,12,10,12,11,10,11,12,11,11,13,12,13,13,10,12,10,12,13,13,11,13,11,5,6,6,6,7,7,6,7,7,6,7,7,7,7,8,7,8,8,6,7,7,7,8,8,7,8,7,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,9,7,8,7,8,8,8,8,8,8,8,8,8,8,9,9,8,9,9,8,8,8,8,9,9,8,9,8,6,8,8,7,8,8,7,8,8,7,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,7,8,8,8,9,9,8,9,9,8,8,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,7,8,8,8,8,9,8,9,8,8,8,8,8,9,9,9,9,9,8,9,8,9,9,9,8,9,9,6,8,8,7,8,8,7,8,8,8,8,8,8,8,9,8,9,9,7,8,8,8,9,9,8,9,8,7,8,8,8,8,9,8,9,8,8,8,9,8,9,9,9,9,9,8,8,8,9,9,9,8,9,9,7,8,8,8,9,9,8,9,9,8,9,9,9,9,9,9,9,9,8,9,8,9,9,9,9,9,9,1,5,5,5,7,9,5,9,7,5,7,8,7,7,10,9,10,10,5,8,7,9,10,10,7,10,7,6,8,9,9,10,12,9,11,11,9,10,11,11,11,13,12,13,13,9,11,11,11,12,13,11,13,11,6,9,8,9,11,11,9,12,10,9,11,11,11,11,13,11,13,12,9,11,10,12,13,13,11,13,11,6,9,9,8,10,11,9,12,11,9,10,11,10,10,12,11,13,13,9,11,11,11,13,12,11,13,11,8,10,10,9,10,12,10,12,11,10,10,12,10,10,13,12,13,13,10,12,11,12,13,13,10,13,10,7,10,10,11,11,13,11,14,11,10,12,11,11,11,13,13,14,13,10,12,12,14,14,14,11,14,11,6,9,9,9,11,12,8,11,10,9,11,11,11,11,13,11,12,13,8,11,10,11,13,13,10,12,10,7,10,10,11,11,14,11,13,11,10,12,12,11,11,14,14,14,14,10,11,12,13,13,14,11,13,11,8,10,10,10,11,12,9,12,10,10,11,12,11,10,13,12,13,13,10,12,10,12,13,13,11,13,10,5,7,7,10,10,7,8,9,10,11,7,9,8,11,10,9,10,10,11,11,9,10,10,11,11,7,9,9,10,10,8,9,10,10,11,9,10,10,11,11,10,10,11,11,11,10,11,11,12,12,7,9,9,10,10,9,10,10,11,11,8,10,9,11,10,10,11,11,11,11,10,11,10,11,11,10,10,10,11,11,10,10,11,11,11,11,11,11,11,11,11,11,12,11,12,11,12,11,12,12,10,10,10,11,11,10,11,11,11,11,10,11,10,11,11,11,12,11,12,12,11,12,11,12,11,8,9,9,11,11,9,10,10,11,12,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,9,10,10,11,11,10,10,11,11,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,9,10,10,11,11,10,11,11,12,12,10,11,10,12,12,11,12,12,12,12,11,12,12,12,12,11,11,11,12,12,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,8,9,9,11,11,9,10,10,11,11,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,9,10,10,11,11,10,10,11,12,12,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,9,10,10,11,11,10,11,11,12,12,10,11,10,12,11,11,12,12,12,12,11,12,11,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,13,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,13,13,13,12,12,12,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,12,13,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,10,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,13,12,12,12,12,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,12,12,12,12,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,12,13,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,8,9,9,11,11,9,10,10,11,11,9,10,10,12,11,10,11,11,12,12,10,11,11,12,12,9,10,10,11,11,10,10,11,11,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,9,10,10,11,11,10,11,11,12,12,10,11,10,12,12,11,12,12,12,12,11,12,12,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,9,10,10,11,11,10,10,11,12,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,10,10,11,11,12,11,11,12,12,12,11,11,12,12,12,11,11,12,12,13,12,12,12,12,12,10,11,11,12,12,11,12,11,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,11,11,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,13,12,13,12,12,13,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,8,10,10,11,11,10,11,11,12,12,10,11,10,12,12,11,12,12,12,12,11,12,12,12,12,10,11,10,12,12,10,10,11,12,12,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,10,11,11,12,12,11,12,12,12,12,10,12,11,12,12,12,12,12,13,13,12,13,12,13,12,11,12,12,12,12,11,12,12,12,13,12,12,12,13,13,12,12,13,12,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,12,11,11,11,12,12,11,12,12,12,13,11,12,12,12,12,12,12,12,13,13,12,12,13,13,13,11,12,12,12,12,12,12,12,12,13,12,12,13,13,13,12,12,13,13,13,13,13,13,13,13,11,12,12,12,12,12,13,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,13,13,13,12,12,12,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,10,11,11,12,12,11,12,12,12,13,11,12,12,13,12,12,13,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,11,12,12,13,12,12,13,12,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,8,9,9,11,11,9,10,10,11,12,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,9,10,10,11,11,10,10,11,12,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,9,10,10,11,11,10,11,11,12,12,10,11,10,12,12,11,12,12,12,12,11,12,11,12,12,11,11,11,12,12,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,8,10,10,11,11,10,10,11,12,12,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,10,11,11,12,12,10,11,12,12,12,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,10,10,11,12,12,11,12,12,12,12,10,11,10,12,12,12,12,12,13,13,12,12,12,13,12,11,12,12,12,12,11,12,12,12,13,12,12,12,13,13,12,12,13,12,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,11,12,12,13,12,12,13,13,13,13,12,13,12,13,12,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,11,12,12,12,12,11,12,11,12,12,10,11,11,12,12,11,11,12,12,12,11,11,12,12,12,12,12,12,12,13,12,12,12,13,12,10,11,10,12,11,11,12,11,12,12,11,12,11,12,12,12,12,12,12,12,12,12,11,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,13,13,13,13,11,12,11,12,12,12,12,12,13,12,12,12,12,12,12,12,13,12,13,13,12,12,12,13,12,10,11,11,12,12,11,12,12,12,13,11,12,12,13,12,12,12,13,13,13,12,13,13,13,13,11,12,12,12,13,12,12,13,13,13,12,12,13,13,13,13,13,13,13,13,13,13,13,13,13,11,12,12,12,12,12,12,13,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,13,13,13,13,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,13,13,13,12,13,12,13,13,11,12,12,12,12,12,12,13,13,13,12,12,13,13,13,12,13,13,13,13,12,13,13,13,13,11,12,12,12,12,12,13,12,13,13,12,12,12,13,12,13,13,13,13,13,12,13,12,13,13,12,12,12,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,12,13,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,11,12,12,13,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,12,12,12,12,12,12,12,13,13,13,12,13,13,13,13,12,13,13,13,13,13,13,13], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+392436);
/* memory initializer */ allocate([13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,10,11,11,12,12,11,12,12,12,13,11,12,12,13,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,13,13,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,13,13,13,12,12,12,12,13,12,12,13,13,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,12,12,12,13,13,13,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,10,11,11,12,12,11,12,12,12,13,11,12,12,13,12,12,13,13,13,13,12,13,12,13,13,11,12,12,13,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,13,13,13,13,13,11,12,12,13,12,12,13,12,13,13,12,13,12,13,13,13,13,13,13,13,12,13,13,13,13,12,12,12,13,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,12,12,12,12,12,12,12,13,12,13,12,13,13,13,13,12,13,13,12,13,13,13,13,13,13,12,12,12,12,12,12,13,13,13,13,12,13,12,13,13,13,13,13,13,13,12,13,13,13,12,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,13,12,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,12,13,13,13,13,13,13,13,13,11,12,12,12,12,12,13,12,13,13,12,13,12,13,13,12,13,13,13,13,12,13,12,13,13,12,12,12,12,12,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,12,13,13,13,12,12,13,13,13,11,12,11,12,12,12,12,12,13,13,11,12,12,13,13,12,13,13,13,13,12,13,12,13,13,12,12,12,12,12,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,13,13,13,13,13,12,13,13,13,13,10,11,11,12,12,11,12,12,12,13,11,12,12,13,12,12,12,13,13,13,12,13,13,13,13,11,12,12,13,13,12,12,13,13,13,12,12,13,13,13,12,13,13,13,13,13,13,13,13,13,11,12,12,13,12,12,13,12,13,13,12,12,12,13,13,12,13,13,13,13,13,13,13,13,13,12,12,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,13,13,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,10,12,11,12,12,11,12,12,12,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,13,12,12,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,13,13,13,11,12,12,13,12,12,12,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,12,13,13,12,13,12,13,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,12,13,13,13,13,13,12,13,12,13,13,13,13,13,13,13,12,13,13,13,13,10,11,11,12,12,11,12,12,12,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,13,13,13,12,13,13,13,13,12,12,13,13,13,13,13,13,13,13,11,12,12,12,12,12,13,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,12,13,13,12,12,12,12,13,12,13,13,13,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,12,12,12,12,12,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,12,13,13,13,13,11,12,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,13,12,12,12,13,12,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,12,12,12,12,12,12,12,13,13,13,12,13,13,13,13,13,13,13,12,13,13,13,13,13,13,12,12,12,12,12,12,13,13,13,13,12,13,12,13,12,13,13,13,13,13,13,13,13,13,12,1,6,6,10,10,6,7,9,11,13,5,9,7,13,11,8,11,12,13,15,8,12,11,15,13,6,7,8,11,11,7,8,10,11,13,9,10,10,13,13,11,11,13,12,16,12,13,13,16,15,6,8,7,11,11,9,10,10,13,13,7,10,7,13,11,12,13,13,15,16,11,13,11,16,12,10,11,11,11,13,11,11,13,12,15,13,13,13,14,15,13,12,15,12,17,15,16,16,16,16,10,11,11,14,11,13,13,13,15,14,11,13,11,15,12,15,15,16,16,16,13,15,12,17,12,6,8,9,12,12,9,10,12,13,15,9,11,11,15,14,12,13,15,16,18,13,14,14,17,16,9,10,11,13,14,11,10,13,14,16,11,12,12,15,15,14,13,16,15,18,14,15,15,17,17,9,11,11,14,14,11,12,13,15,16,11,13,11,15,14,15,15,15,17,18,14,15,14,17,15,13,14,14,15,16,14,14,15,15,17,15,16,15,17,17,16,16,17,15,19,17,18,18,19,18,13,14,14,16,15,15,15,16,17,17,14,15,14,18,15,17,17,17,19,19,16,17,15,19,16,6,9,8,13,12,9,11,11,14,15,9,12,10,15,13,13,14,14,16,17,12,15,13,18,16,9,11,11,14,14,11,11,13,14,15,11,13,12,16,15,14,14,15,15,18,14,15,15,18,17,9,11,10,14,13,11,12,12,15,15,11,13,10,16,14,14,15,15,16,18,14,16,13,18,15,13,14,14,16,16,14,14,15,15,17,15,16,15,17,17,16,16,17,16,19,17,18,17,18,19,13,14,14,16,15,15,15,15,17,17,14,15,14,17,15,17,17,17,18,19,16,17,15,19,15,11,13,13,15,16,13,14,15,16,18,14,15,15,17,17,16,16,18,18,20,17,18,17,19,20,13,14,14,16,17,15,15,16,17,18,15,16,16,17,17,18,17,19,18,19,18,18,18,19,21,14,14,15,16,17,15,15,16,18,18,15,16,16,17,18,18,18,19,19,21,18,19,19,22,20,16,16,17,17,19,17,17,17,18,20,17,18,18,20,19,19,19,20,19,0,19,19,20,20,21,17,17,17,19,18,18,18,20,19,19,18,18,18,20,20,19,19,20,20,20,20,21,20,21,19,11,13,13,16,15,14,15,15,17,17,14,15,14,18,16,16,18,18,20,19,16,19,17,21,18,13,14,15,16,17,15,15,16,18,18,15,16,15,19,18,18,18,18,19,19,18,18,18,22,20,13,14,14,16,16,15,16,16,18,17,15,16,15,18,17,18,18,18,19,19,17,18,17,21,18,16,17,17,18,18,17,18,19,19,19,18,20,18,19,19,19,20,21,19,21,20,20,20,0,21,16,17,17,19,19,18,18,18,19,21,17,18,18,19,18,20,19,21,20,21,19,20,20,22,19,7,9,9,13,13,8,10,11,14,15,9,12,11,15,14,11,13,14,16,17,13,15,14,17,16,8,10,11,14,14,10,10,12,14,16,11,12,12,16,15,13,12,15,15,18,14,15,15,19,17,9,11,11,14,14,11,12,12,15,15,11,13,11,16,14,14,15,14,17,17,14,16,14,18,15,12,13,14,15,16,13,13,15,14,17,15,15,15,17,17,15,14,17,14,19,17,18,18,19,18,13,14,14,16,16,15,15,15,17,17,14,15,14,18,15,17,18,17,18,17,16,18,16,19,15,7,10,10,13,13,9,10,12,14,15,10,12,11,15,14,12,13,14,16,17,13,15,14,18,16,10,10,12,13,14,10,10,13,13,16,12,12,13,15,15,13,12,15,15,18,15,15,16,18,17,10,11,11,14,14,12,13,13,15,16,10,13,10,16,14,14,15,15,17,17,14,15,13,17,15,13,13,14,15,16,14,13,15,14,18,15,15,16,16,17,16,15,18,15,18,17,18,18,18,18,13,15,14,17,16,15,16,16,17,17,14,15,13,17,15,17,17,18,18,18,16,17,14,20,14,8,10,10,14,14,11,11,13,14,16,11,13,11,16,14,14,15,16,16,18,14,16,15,18,16,10,12,11,15,14,11,11,13,14,16,13,14,13,16,15,15,14,16,15,19,16,17,16,20,18,10,11,12,14,15,13,13,14,16,16,11,14,11,16,14,16,16,17,18,19,15,17,14,20,15,14,15,14,17,16,13,14,15,15,18,16,17,16,19,18,16,15,18,15,19,18,19,18,21,21,14,14,15,16,17,16,16,17,18,18,13,15,14,17,15,18,18,19,18,22,16,18,15,21,15,12,13,14,16,16,14,14,16,16,18,14,15,15,17,18,16,16,18,18,20,18,18,17,20,20,13,14,15,15,17,15,14,16,16,18,16,16,16,17,19,17,15,18,17,21,18,18,18,19,19,14,15,15,18,17,15,16,16,18,19,15,16,15,18,18,17,18,18,20,21,17,19,17,20,19,16,16,17,16,19,17,17,18,17,20,18,18,18,18,19,19,18,20,17,22,20,20,19,20,20,17,17,18,18,19,18,18,20,21,20,17,18,17,20,20,21,21,21,21,21,19,21,18,22,20,11,13,13,17,16,14,14,16,16,18,14,16,14,18,16,17,18,19,19,20,18,19,18,21,19,14,15,14,17,16,14,14,16,18,18,16,17,16,18,17,18,17,19,18,20,19,19,18,20,20,13,14,15,16,17,16,16,17,18,19,14,16,14,19,17,18,19,18,20,20,18,20,17,21,18,17,17,17,19,18,16,17,18,18,19,18,19,18,21,21,18,18,20,17,21,19,20,20,22,21,16,17,18,18,19,18,18,19,21,20,16,17,17,20,18,21,21,22,21,22,18,21,18,0,18,7,9,9,13,13,9,11,12,14,15,8,11,10,15,14,13,14,15,16,18,11,14,13,17,15,9,11,11,14,14,11,11,13,14,16,11,12,12,15,15,14,14,16,15,18,14,14,15,17,17,8,11,10,14,14,11,12,12,15,15,10,12,10,16,14,14,15,15,17,18,13,15,12,18,15,13,14,14,16,16,14,14,15,15,17,15,15,15,16,17,16,15,17,15,19,17,17,17,18,18,12,14,13,16,15,15,15,15,17,17,13,15,13,17,14,17,18,18,18,19,15,17,14,19,14,8,10,10,14,14,11,11,13,14,16,11,13,11,16,14,14,15,16,17,19,14,16,15,18,17,10,12,11,15,14,11,11,14,14,17,13,14,13,17,15,15,14,17,15,19,16,17,16,19,17,10,11,12,14,15,13,13,14,15,17,11,13,11,17,14,16,16,17,18,19,15,16,14,18,15,14,15,14,16,16,13,14,15,15,18,16,16,16,18,18,16,15,18,15,20,18,19,18,21,18,14,14,15,16,17,16,16,17,17,18,13,15,14,17,16,19,19,19,19,19,15,18,15,20,15,7,10,10,13,13,10,11,12,14,15,9,12,10,15,14,13,14,15,16,17,12,15,13,17,16,10,11,11,14,14,10,10,13,14,16,12,13,13,16,15,14,13,16,15,18,15,15,16,17,17,10,12,10,14,13,12,13,12,15,15,10,13,10,16,13,15,16,15,17,18,13,16,12,18,15,13,14,14,16,17,14,13,15,15,18,15,16,15,17,17,16,14,17,15,19,17,18,18,19,19,13,15,13,17,14,15,15,15,18,17,14,15,13,17,14,18,17,18,18,19,15,17,15,19,15,11,13,13,16,17,14,14,16,16,18,14,16,15,18,17,17,18,19,18,21,18,18,17,20,18,13,15,14,17,16,14,14,16,17,18,16,17,16,19,17,18,17,19,18,22,18,19,19,21,21,13,14,15,16,18,16,16,17,17,20,14,16,14,18,17,18,18,19,19,21,17,18,17,21,18,17,18,17,19,18,16,17,17,18,19,18,18,18,22,22,18,17,19,17,0,20,21,19,21,20,17,17,18,18,21,18,18,18,19,21,17,17,17,19,19,20,20,22,21,21,19,20,18,20,17,12,14,13,17,16,14,15,15,17,18,14,16,14,18,16,17,18,18,21,20,16,18,16,21,18,14,15,15,17,17,15,15,16,18,18,15,17,16,18,18,17,17,19,19,20,18,19,18,20,19,14,15,14,17,15,15,16,16,18,17,15,16,14,19,15,18,18,18,19,20,17,20,15,21,17,16,17,18,18,19,17,17,18,18,20,18,19,18,19,21,19,18,19,19,21,20,0,19,21,20,16,17,16,19,16,18,18,18,19,19,17,18,17,20,17,19,20,20,22,0,19,20,17,21,17,11,13,14,16,17,14,15,15,17,18,14,15,15,18,18,16,17,17,19,20,16,18,17,19,21,13,14,15,17,17,14,15,16,17,19,15,16,16,18,19,16,17,18,19,21,17,18,20,21,21,13,15,15,17,17,15,16,16,18,19,15,16,16,18,19,17,17,18,19,22,17,19,18,22,19,15,16,17,19,19,16,17,18,18,20,17,18,18,19,20,19,18,20,18,22,20,19,19,22,21,16,17,17,18,19,18,18,18,19,20,17,18,18,20,19,20,19,20,22,20,19,20,21,21,20,12,14,14,16,16,13,14,16,17,18,14,16,15,18,18,15,17,17,19,19,17,18,18,19,19,13,14,15,16,17,14,14,16,16,20,15,16,16,17,19,16,15,18,17,20,18,17,19,19,19,14,15,15,17,17,16,16,16,18,18,15,16,15,19,18,17,18,18,20,21,17,18,17,21,18,16,15,17,17,19,17,15,18,17,20,19,17,18,19,20,18,16,19,17,22,20,19,20,19,20,17,17,18,19,19,18,18,19,20,20,17,18,17,18,18,21,21,20,20,21,18,20,17,21,19,11,14,14,16,17,15,14,16,17,19,14,16,14,18,17,18,18,19,19,21,17,19,18,20,20,13,15,14,17,17,14,14,16,17,18,16,17,16,19,18,18,17,19,18,20,18,21,18,20,20,13,15,15,16,17,16,16,17,18,19,14,16,15,19,18,19,19,19,21,20,18,19,17,20,18,16,17,16,19,18,16,17,17,19,20,17,19,18,20,19,18,17,21,18,0,21,20,20,0,20,17,17,18,18,19,18,19,19,20,22,16,17,17,20,18,21,22,20,20,22,18,22,18,22,18,12,14,14,17,17,14,15,16,17,19,14,16,15,17,17,17,17,18,18,21,17,19,17,20,19,14,15,15,16,18,15,14,16,16,19,16,17,16,19,18,17,16,20,17,20,18,20,19,19,20,14,15,15,18,17,16,16,17,18,19,14,16,15,19,17,18,21,18,19,21,17,18,17,19,18,17,17,18,17,20,17,16,18,17,21,18,19,19,19,19,18,17,19,17,20,20,21,20,21,20,17,17,17,19,19,19,18,18,20,21,16,18,16,19,18,20,20,21,21,20,18,19,16,0,17,12,14,14,17,17,15,15,18,17,19,15,18,15,20,16,20,19,21,18,22,20,20,20,22,19,14,16,14,20,17,14,15,17,17,20,18,18,17,20,18,18,17,19,17,21,20,21,20,0,21,14,15,16,17,19,18,17,19,18,21,14,18,15,21,17,21,20,21,20,0,18,21,17,21,17,18,19,17,20,18,16,17,17,19,19,19,21,20,0,20,18,17,21,17,0,22,0,21,0,22,17,17,19,18,20,20,20,21,19,22,16,17,18,20,18,22,22,0,22,0,17,21,17,22,17,11,14,13,16,16,14,15,15,17,18,14,15,14,18,17,17,18,18,19,20,16,17,17,21,19,13,14,15,17,17,15,16,16,18,18,15,16,16,19,18,18,18,18,19,20,17,18,18,20,19,13,15,14,17,17,15,16,16,17,18,14,16,15,19,17,17,18,19,21,21,17,18,17,20,18,16,17,17,19,19,17,18,19,19,20,18,19,18,21,21,21,20,19,21,22,20,20,19,21,20,15,17,16,19,19,17,18,18,20,21,16,18,17,20,18,19,19,21,21,21,19,19,19,20,18,11,14,13,17,16,14,14,16,16,19,14,16,15,19,16,18,18,18,19,22,17,18,17,20,19,13,15,14,17,17,15,15,16,17,19,16,17,16,20,18,18,17,19,18,21,19,19,18,22,0,13,14,15,17,18,16,16,17,17,19,14,16,15,19,18,18,19,19,20,21,18,18,17,20,18,17,18,17,20,18,16,17,17,18,20,18,19,18,20,20,18,18,21,17,21,20,21,21,0,19,16,16,18,18,19,19,18,20,19,20,16,17,17,20,18,21,20,21,22,22,18,20,17,21,17,12,14,14,17,16,14,15,16,18,18,13,15,14,18,17,17,18,18,19,19,15,17,16,19,19,14,15,15,17,17,15,15,16,18,19,15,16,16,19,18,17,17,18,18,20,18,18,18,21,20,13,15,14,17,16,15,16,15,18,18,14,16,14,18,17,18,18,18,19,21,16,18,16,20,17,17,18,17,18,19,17,17,18,18,19,18,19,19,21,19,19,18,20,18,21,21,20,20,21,20,16,17,15,20,17,17,19,17,19,19,17,18,15,20,17,19,20,19,21,22,17,20,16,0,17,12,14,14,17,18,16,15,18,16,20,16,18,15,21,17,20,18,21,19,22,19,21,19,0,19,14,16,15,19,17,14,15,17,16,21,18,19,18,21,17,19,17,21,17,22,20,21,21,0,21,14,15,16,17,19,18,17,19,18,21,14,17,15,20,17,21,22,21,20,22,18,21,17,21,17,17,19,17,21,18,16,17,17,19,20,19,21,20,21,20,17,18,20,17,21,0,22,20,21,22,17,17,20,18,21,21,20,22,20,21,16,17,17,21,19,0,22,0,21,21,18,22,17,21,17,12,14,14,17,16,14,15,16,17,18,14,16,15,18,17,17,17,20,19,20,16,18,17,21,18,14,15,15,17,17,14,15,16,17,19,16,17,16,18,18,17,16,19,18,19,18,19,18,21,20,14,15,15,18,17,16,16,16,19,18,15,16,14,20,16,18,18,19,19,20,16,19,16,21,17,17,17,18,19,19,16,16,18,18,19,19,19,18,20,20,18,16,19,18,20,22,21,20,19,20,16,18,17,20,16,18,19,18,19,18,16,18,16,20,17,21,20,21,20,20,18,19,17,21,16,2,3,3,3,3,3,3,1,5,5,5,7,9,5,9,7,5,7,8,7,7,10,9,9,10,5,8,7,9,10,9,7,10,7,6,9,9,9,10,12,10,12,11,9,10,11,11,10,13,12,12,13,10,11,11,12,13,13,11,13,11,6,9,9,10,11,12,9,12,11,10,11,11,11,11,13,12,13,13,9,11,10,12,13,13,11,13,10,6,9,10,9,11,12,10,12,11,9,10,11,10,10,13,11,13,13,10,11,11,12,13,12,11,13,11,7,9,10,9,10,12,10,11,11,10,10,11,10,10,12,12,11,12,10,11,10,12,12,12,10,12,10,7,10,10,11,11,13,11,13,11,10,12,11,11,10,13,13,14,13,10,11,12,13,13,14,11,13,10,6,10,9,10,11,12,9,12,11,9,11,11,11,11,13,12,12,13,9,11,10,12,13,13,10,13,10,7,10,10,11,11,14,11,13,11,10,12,11,11,10,14,13,14,13,10,11,12,13,13,14,11,13,10,7,10,9,10,10,12,9,12,10,10,11,11,10,10,12,12,12,12,9,11,10,11,12,12,10,12,9,2,6,6,5,7,8,5,8,7,6,7,7,7,7,8,8,8,8,6,7,7,7,8,8,7,8,7,6,8,8,8,9,10,8,9,9,8,9,9,9,9,10,10,10,10,8,9,9,10,10,10,9,10,10,6,8,8,8,9,9,8,10,9,9,9,9,9,9,10,10,10,10,8,9,9,10,10,10,9,10,9,6,8,9,8,9,9,8,9,9,8,9,9,9,9,10,9,10,10,8,9,9,9,10,10,9,10,9,7,8,9,8,9,9,9,9,9,8,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,7,9,9,9,10,10,9,10,10,9,10,9,9,9,10,10,10,10,9,10,9,10,10,10,9,10,9,6,8,8,8,9,9,8,9,9,8,9,9,9,9,10,9,10,10,8,9,9,9,10,10,9,10,9,7,9,9,9,10,10,9,10,9,9,9,10,10,9,10,10,10,10,9,9,9,10,10,10,9,10,9,7,9,8,8,9,9,8,9,9,9,9,9,9,9,9,9,9,9,8,9,8,9,9,9,9,9,9,1,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,1,7,7,6,9,9,7,9,9,6,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,7,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,7,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,1,2,3,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13,14,14,14,14,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,3,7,12,14,14,16,18,19,6,2,4,6,8,9,12,14,12,3,3,5,7,8,11,13,13,6,4,5,7,8,10,11,14,8,7,7,7,7,9,10,15,9,8,7,7,6,8,9,17,11,11,10,9,8,9,9,19,14,13,11,10,9,9,9,1,3,2,3,1,4,4,8,8,10,10,10,10,9,8,11,11,4,6,5,8,6,10,10,10,10,10,9,10,9,4,5,6,6,9,10,10,10,10,9,10,9,10,8,9,8,9,8,9,9,10,9,11,10,12,10,8,8,9,8,9,9,9,9,10,10,11,10,12,9,10,10,11,10,11,10,12,11,12,11,13,11,9,10,10,10,11,10,11,11,12,11,12,11,12,11,12,12,12,12,13,12,13,12,13,12,13,13,11,12,12,12,12,12,12,12,13,13,13,13,13,12,12,12,13,13,13,13,13,13,13,13,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,12,13,13,13,14,14,13,13,13,13,13,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,4,4,4,5,5,4,5,5,5,5,4,5,4,4,5,5,5,5,5,5,5,5,5,5,5,1,4,4,4,4,4,4,4,4,2,8,13,15,16,18,21,22,5,4,6,8,10,12,17,21,9,5,5,6,8,11,15,19,11,6,5,5,6,7,12,14,14,8,7,5,4,4,9,11,16,11,9,7,4,3,7,10,22,15,14,12,8,7,9,11,21,16,15,12,9,5,6,8,2,5,5,5,7,7,5,7,7,5,7,7,7,8,9,7,9,9,5,7,7,7,9,9,7,9,8,5,7,8,8,9,10,8,9,9,8,9,10,9,10,12,10,11,11,8,9,10,10,11,11,9,11,11,5,8,7,8,9,9,8,10,9,8,10,9,9,11,11,10,11,11,8,10,9,10,11,11,9,12,10,5,8,8,7,9,10,8,10,9,7,9,9,9,10,11,9,11,11,8,10,10,10,11,11,10,12,11,7,9,9,9,10,11,9,11,11,9,9,11,10,10,13,11,11,12,9,11,11,11,12,13,11,13,12,7,9,9,9,11,11,9,12,10,9,11,10,10,11,12,11,13,12,9,11,11,11,13,13,11,13,11,5,8,8,8,9,10,7,10,9,8,10,10,10,11,11,10,11,11,7,9,9,9,11,11,9,11,10,7,9,9,9,10,12,9,11,11,9,11,11,11,11,13,11,13,13,9,10,11,11,12,13,10,12,11,7,9,9,9,11,11,9,11,10,9,11,11,11,12,13,11,13,12,9,11,9,11,12,11,10,13,10,4,6,6,9,9,6,7,8,10,10,6,8,7,10,10,8,10,10,12,13,8,10,10,13,12,6,8,8,10,10,7,8,9,10,11,8,9,9,11,11,10,10,11,12,13,10,11,11,14,13,6,8,8,10,10,8,9,9,11,11,7,9,8,11,10,10,11,11,13,14,10,11,10,13,12,9,10,10,12,12,10,10,11,12,13,10,11,11,13,13,12,12,13,12,15,13,14,13,15,14,9,10,10,13,12,10,11,11,13,13,10,11,10,13,12,13,13,14,14,15,12,13,12,15,12,6,8,8,10,11,8,9,10,11,12,8,9,9,11,11,10,11,12,13,14,10,11,11,14,13,8,9,9,11,12,9,10,11,12,13,9,10,11,12,13,11,11,13,13,15,11,12,12,14,14,8,9,9,12,12,9,10,11,12,13,9,10,10,13,12,11,12,13,14,15,11,12,12,14,14,11,11,12,13,14,11,12,13,13,15,12,13,13,14,15,13,13,14,14,16,14,15,15,16,16,11,12,11,14,13,12,13,13,14,14,11,13,12,14,13,14,15,15,16,16,13,14,14,16,14,6,8,8,11,10,8,9,9,12,11,8,10,9,12,11,10,11,11,13,13,10,12,11,14,13,8,9,9,12,12,9,10,10,12,13,9,11,10,13,12,11,12,12,14,14,11,13,12,15,14,8,9,9,12,11,9,10,10,13,12,9,11,10,13,12,12,12,12,14,14,11,13,12,15,13,11,11,12,13,14,11,12,13,13,14,12,13,13,14,15,13,13,14,14,16,14,15,15,16,16,11,12,11,14,13,12,13,13,15,14,11,13,12,15,13,14,15,15,16,16,13,15,13,16,14,9,10,11,12,13,11,11,12,13,14,11,12,12,13,14,13,13,14,14,16,13,14,14,15,16,11,11,12,13,14,12,12,13,14,15,12,13,13,14,15,14,14,15,15,17,14,15,15,16,17,11,12,12,14,14,12,13,13,14,15,12,13,12,15,15,14,15,15,16,17,14,15,15,16,16,13,14,14,15,16,14,14,15,15,17,15,15,15,16,17,16,16,17,16,18,16,17,17,18,18,13,14,14,16,15,14,15,15,17,16,14,15,15,16,16,16,17,17,18,18,16,16,16,17,16,9,11,10,13,12,11,12,12,14,13,11,12,11,15,13,13,14,14,16,15,13,14,13,17,14,11,12,12,14,14,12,12,13,15,15,12,13,13,15,14,14,14,15,16,16,14,15,15,17,16,11,12,11,14,13,12,13,13,15,14,12,13,12,15,13,14,15,15,16,16,14,15,14,17,15,13,14,14,15,16,14,15,15,16,17,14,15,15,16,17,16,16,16,17,17,16,17,17,18,18,13,15,14,16,15,15,15,15,17,16,14,15,14,17,15,16,17,17,18,18,16,17,16,18,16,6,8,8,11,11,8,9,9,11,12,8,9,9,12,11,10,11,11,13,14,10,12,11,14,13,7,9,9,11,12,9,10,10,12,13,9,10,10,13,12,11,11,12,13,15,11,12,12,15,14,8,9,9,12,11,9,10,10,13,13,9,11,10,13,12,12,12,12,14,15,11,13,12,15,13,10,11,11,13,14,11,12,12,13,15,11,12,12,14,14,13,13,14,14,16,14,15,14,16,16,11,12,11,14,13,12,13,13,15,14,11,13,12,15,13,14,15,15,16,16,13,14,14,16,14,8,9,9,11,12,9,10,11,12,13,9,10,10,13,12,11,12,13,14,15,11,12,12,15,14,9,9,11,11,13,10,10,12,12,14,10,10,11,13,14,12,12,13,14,16,12,13,13,15,15,9,11,10,13,12,10,11,11,13,14,10,12,11,14,13,12,13,13,15,16,12,13,13,15,15,11,11,13,13,15,12,12,14,13,15,13,13,14,14,15,14,14,15,14,17,15,15,15,16,16,12,13,12,15,14,13,14,14,15,15,12,14,13,15,14,15,15,15,17,17,14,15,14,17,15,7,9,9,12,11,9,10,10,12,12,9,11,10,13,12,11,12,12,14,14,11,13,12,15,14,9,10,10,12,12,10,10,11,12,13,10,11,11,14,13,12,12,13,14,15,12,13,13,16,15,9,10,10,13,12,10,11,11,13,13,10,11,10,14,12,13,13,13,15,15,12,13,12,15,14,11,12,12,14,14,12,12,13,14,15,13,14,13,15,15,14,13,15,14,16,15,16,15,17,16,12,12,12,14,14,13,13,14,15,15,12,13,12,15,14,15,15,16,16,17,14,15,14,17,14,10,11,12,13,14,11,12,13,14,15,11,12,13,14,15,13,14,15,15,17,14,15,15,16,16,11,12,13,12,15,12,12,14,13,16,13,13,14,13,16,14,14,16,14,18,15,15,16,16,17,12,13,12,15,15,13,14,14,15,16,13,14,13,16,15,15,15,16,17,18,15,15,15,17,16,14,14,15,14,17,15,14,16,14,17,15,15,16,15,18,16,16,17,16,19,17,17,17,17,18,14,15,15,17,16,15,16,16,17,17,15,16,15,18,16,17,17,18,18,18,16,17,16,18,17,10,11,11,14,13,11,12,12,15,14,11,13,12,15,14,14,15,15,16,16,14,15,15,17,16,11,12,12,15,14,12,13,13,15,14,13,14,13,16,14,14,15,15,16,16,15,16,15,18,16,11,13,12,15,15,13,14,14,15,15,12,14,13,16,15,15,16,16,17,17,15,16,15,17,16,14,15,14,16,16,14,15,15,16,16,15,16,15,17,16,16,16,17,16,17,17,18,17,19,18,14,15,15,17,16,15,16,16,17,17,15,15,15,18,16,17,18,18,18,18,16,17,16,19,16,6,8,8,11,11,8,9,9,11,12,8,9,9,12,11,10,11,12,13,14,10,11,11,14,13,8,9,9,11,12,9,10,11,12,13,9,10,10,13,13,11,12,13,13,15,11,12,12,15,14,7,9,9,12,11,9,10,10,12,13,9,10,10,13,12,11,12,12,14,15,11,12,11,14,13,11,11,12,13,14,11,12,13,13,15,12,13,13,14,15,13,14,14,14,16,14,15,15,16,16,10,11,11,14,13,11,12,12,14,14,11,12,12,15,13,14,14,14,16,16,13,14,13,16,14,7,9,9,11,12,9,10,10,12,13,9,10,10,12,12,11,12,13,14,15,11,12,12,14,14,9,10,10,12,13,10,10,11,12,14,10,11,11,13,13,12,12,13,14,15,13,13,13,15,15,9,10,10,12,12,10,11,11,13,14,10,11,10,13,12,12,13,13,15,16,12,13,12,15,14,11,12,13,14,14,12,12,13,14,15,13,14,13,15,15,14,14,15,14,17,15,16,15,17,16,11,12,12,14,14,13,13,13,15,15,12,13,12,15,14,15,15,15,16,17,14,15,14,16,14,8,9,9,12,11,9,10,10,12,13,9,11,10,13,12,11,12,12,14,15,11,12,12,15,14,9,10,11,13,13,10,11,12,13,14,10,11,11,14,13,12,13,13,15,15,12,13,13,16,15,9,11,9,13,11,10,11,10,14,13,10,12,10,14,12,12,13,13,15,15,12,13,12,16,14,12,12,13,14,15,12,13,14,14,16,13,14,14,15,15,14,14,15,15,17,15,16,15,17,16,11,13,11,15,13,13,14,13,15,14,12,14,12,16,13,15,15,15,16,16,14,15,14,17,14,10,11,11,13,14,11,12,13,14,15,11,12,12,14,15,14,14,15,16,17,14,15,15,16,16,11,12,13,14,15,12,13,14,15,16,13,14,14,15,16,15,15,16,16,18,15,16,16,17,17,11,12,12,14,15,13,13,14,14,16,12,13,13,15,15,15,15,16,16,18,14,15,15,16,16,14,15,15,16,17,15,15,16,16,17,15,16,16,17,17,16,16,17,16,19,17,18,17,18,18,14,14,15,16,16,15,15,16,16,17,14,15,15,16,16,17,17,18,18,19,16,17,16,17,16,10,12,11,14,13,11,13,12,15,14,11,13,12,15,14,14,15,15,16,16,13,15,14,17,15,12,13,13,15,15,13,13,14,15,16,13,14,14,16,16,14,15,15,17,17,15,16,16,17,17,11,13,12,15,12,13,14,13,16,13,12,14,12,16,13,15,16,15,17,16,14,16,14,18,14,14,15,15,16,17,15,15,16,16,17,15,16,16,17,17,16,16,17,17,18,17,18,17,18,18,14,15,14,17,14,15,16,15,18,15,15,16,15,18,14,17,17,17,18,17,16,17,16,19,16,9,11,11,13,13,10,12,12,14,14,11,12,12,15,14,13,14,14,16,16,13,14,14,16,16,10,11,12,14,14,11,12,13,14,15,12,13,13,15,15,13,14,15,16,16,14,15,15,17,16,11,12,12,15,14,12,13,13,15,15,12,13,12,15,15,14,15,15,16,17,14,15,14,17,16,12,13,14,15,16,13,13,14,15,16,13,14,15,16,16,14,15,16,16,18,15,16,16,18,18,13,14,14,16,15,14,15,15,17,16,14,15,15,17,16,16,17,17,18,18,16,17,16,18,17,10,12,12,14,14,11,12,13,15,15,12,13,13,15,15,13,14,15,16,17,14,15,15,17,16,11,11,13,14,15,12,12,14,15,16,13,13,14,15,16,14,14,15,16,17,15,15,16,17,17,12,13,12,15,15,13,14,14,16,16,13,14,13,16,15,15,16,15,17,17,15,16,15,18,16,13,12,15,14,17,14,13,16,14,17,14,14,16,15,18,15,14,17,16,18,16,16,17,17,18,14,15,15,17,16,15,16,16,17,17,15,16,15,18,16,17,17,17,18,18,16,17,16,19,17,10,11,11,14,14,11,12,12,15,15,11,13,12,15,15,14,15,14,16,16,14,15,15,17,16,11,12,12,15,14,12,12,13,15,15,13,14,13,16,15,14,15,15,16,16,15,16,15,18,17,11,13,12,15,15,13,14,13,15,15,12,14,13,16,15,15,16,15,17,17,15,16,15,18,16,13,14,13,16,16,14,15,14,16,16,14,15,15,17,16,16,16,16,16,18,16,18,17,19,18,14,15,15,17,16,15,16,16,17,17,15,15,15,17,16,17,17,18,18,19,16,17,16,18,16,12,13,13,15,16,13,14,14,16,17,13,14,14,16,16,15,15,16,17,18,15,16,16,18,17,13,13,14,14,17,14,14,15,15,17,14,14,15,16,17,15,15,17,16,18,16,17,17,18,18,13,14,14,17,16,14,15,15,17,17,14,15,14,17,16,16,17,17,18,18,16,17,16,18,17,15,14,16,13,18,16,15,17,14,19,16,16,17,15,18,17,16,18,15,19,18,18,18,17,19,15,16,16,18,17,16,17,17,18,18,16,17,16,19,17,18,19,18,19,19,17,18,17,20,18,11,12,12,15,15,13,13,14,15,16,13,14,13,16,15,15,16,16,17,17,15,16,16,18,17,12,14,13,16,15,13,13,14,15,16,14,15,14,17,16,16,16,16,16,17,16,17,17,19,17,12,13,14,16,16,14,15,15,16,17,13,15,13,17,15,16,17,17,18,18,16,17,16,18,16,15,16,15,17,16,15,15,15,17,17,16,17,16,18,17,17,16,17,16,18,18,19,18,20,18,15,16,16,17,17,16,17,17,18,18,15,16,15,18,17,18,18,19,19,19,17,18,16,19,16,9,11,11,13,13,11,12,12,14,15,10,12,12,14,14,13,14,14,16,16,13,14,14,16,16,11,12,12,14,14,12,12,13,15,15,12,13,13,15,15,14,15,15,16,17,14,15,15,16,16,10,12,11,14,14,12,13,13,15,15,11,13,12,15,14,14,15,15,16,17,13,15,14,17,16,13,14,14,15,16,14,15,15,16,17,14,15,15,16,17,16,16,17,17,18,16,17,17,18,18,12,14,13,16,15,13,15,14,17,16,13,14,13,17,15,15,16,16,18,18,15,16,15,18,16,10,11,11,14,14,11,12,13,14,15,11,12,12,15,15,14,15,15,16,17,14,15,15,16,16,11,12,13,15,15,12,13,14,15,16,13,14,14,15,16,15,15,16,16,18,15,15,16,17,17,11,12,12,14,15,13,13,14,15,16,12,13,13,15,15,15,15,16,17,18,14,15,15,17,16,14,15,15,16,17,15,15,16,16,17,15,16,16,17,17,16,16,17,16,19,17,17,18,19,18,13,13,14,16,16,14,15,16,17,17,14,14,15,16,16,16,16,17,18,18,16,16,16,18,16,10,12,12,14,14,12,13,13,15,15,11,13,12,15,15,14,15,15,16,17,13,15,14,17,16,12,13,13,15,15,13,13,14,15,16,13,14,14,16,16,15,15,16,17,18,15,15,16,17,17,11,13,12,15,14,13,14,13,16,15,12,14,12,16,14,15,16,15,17,17,14,16,14,17,16,14,15,15,16,17,15,15,16,16,18,15,16,16,17,17,16,17,17,17,19,17,17,17,18,18,13,15,12,17,14,14,16,14,17,15,14,15,13,17,14,16,17,16,18,17,15,17,14,19,15,11,12,12,15,15,13,13,14,15,16,13,14,13,16,15,15,16,16,17,18,15,16,16,17,17,12,14,13,16,16,13,13,15,15,17,14,15,15,17,16,16,16,17,16,19,16,17,17,18,18,12,13,14,15,16,14,14,15,16,17,13,14,13,16,15,16,17,17,18,19,15,16,16,17,16,15,16,16,18,17,15,15,16,17,18,16,17,17,18,18,16,16,18,16,19,18,19,19,20,19,15,15,16,16,17,16,16,17,17,18,15,15,15,17,16,18,18,19,18,20,17,17,16,18,16,12,13,13,16,15,13,14,14,16,16,13,14,14,16,16,15,16,16,17,18,15,16,15,18,17,13,14,14,16,16,14,15,15,16,17,14,15,15,17,17,16,17,17,18,18,16,17,17,18,18,13,14,13,17,14,14,15,14,17,16,14,15,14,17,15,16,17,17,18,18,15,17,15,19,15,16,16,16,17,18,16,16,17,17,19,16,17,17,18,19,17,17,18,18,20,18,18,18,19,19,15,16,14,18,13,16,17,16,19,15,16,17,15,19,14,18,18,18,19,17,17,18,16,20,15,1,5,5,5,7,8,5,8,7,5,7,8,8,8,10,8,10,10,5,8,7,8,10,10,8,10,8,6,8,9,8,10,12,9,11,11,9,10,11,11,11,13,12,13,13,9,11,11,11,13,13,11,13,12,6,9,8,9,11,11,8,12,10,9,11,11,11,12,13,11,13,13,9,11,10,11,13,13,11,13,11,5,9,9,8,11,11,9,12,11,8,10,11,10,11,13,11,13,13,9,11,11,11,13,13,11,13,12,8,10,11,10,12,13,10,13,12,10,10,13,11,11,14,12,13,14,11,13,12,13,14,14,12,14,12,8,11,10,11,12,13,11,14,12,10,13,12,12,12,13,13,15,14,11,12,13,13,14,15,12,14,12,5,9,9,9,11,12,8,11,11,9,11,11,11,12,13,11,13,13,8,11,10,11,13,13,10,13,11,8,10,11,11,12,14,11,13,12,11,13,12,12,12,14,13,15,14,10,12,13,13,14,15,12,13,12,8,11,10,10,12,13,10,13,12,11,12,13,12,12,14,13,14,14,10,13,10,12,14,13,11,14,11,5,7,7,6,7,7,6,7,7,6,7,7,7,8,8,7,8,8,6,7,7,7,8,8,7,8,8,7,7,8,7,8,8,7,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,7,8,7,7,8,8,7,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,6,8,8,7,8,8,7,8,8,7,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,7,8,8,8,8,9,8,9,9,8,8,9,8,9,9,9,9,9,8,9,9,9,9,9,9,9,9,7,8,8,8,9,9,8,9,8,8,8,8,8,9,9,9,9,9,8,9,8,9,9,9,9,9,9,6,8,8,7,8,8,7,8,8,8,8,8,8,8,9,8,9,9,7,8,8,8,9,9,8,9,8,7,8,8,8,8,9,8,9,8,8,8,9,9,9,9,9,9,9,8,8,8,9,9,9,8,9,9,7,8,8,8,9,9,8,9,8,8,9,9,9,9,9,9,9,9,8,9,8,9,9,9,9,9,9,2,5,5,5,7,8,5,8,7,5,7,7,7,7,9,7,9,9,5,7,7,8,9,9,7,9,7,6,8,8,8,9,10,8,9,9,8,9,10,9,9,11,10,11,11,8,9,9,10,11,11,9,11,10,6,8,8,8,9,9,8,10,9,8,9,9,9,10,11,10,11,10,8,10,9,10,11,11,9,11,9,6,8,8,7,9,9,8,10,9,7,9,9,9,9,10,9,10,10,8,9,9,9,10,10,9,11,10,7,9,9,8,10,10,9,10,10,9,9,10,10,10,11,10,11,11,9,10,10,10,11,11,10,11,10,7,9,9,9,9,10,9,10,9,8,10,9,9,9,11,10,11,11,9,10,10,10,11,11,9,11,9,6,8,8,8,9,10,7,9,9,8,9,9,9,10,10,9,10,10,7,9,9,9,10,10,9,10,9,7,9,9,9,9,10,9,10,9,9,10,10,9,9,11,10,11,11,8,9,10,10,11,11,9,11,9,7,9,9,9,10,10,8,10,10,9,10,10,10,10,11,10,11,11,9,10,9,10,11,11,10,11,10,6,8,8,10,10,8,9,9,10,11,8,10,9,11,10,9,10,10,11,11,9,10,10,11,11,8,9,9,10,10,9,9,10,11,11,10,10,10,11,11,10,11,11,11,11,10,11,11,11,11,8,9,9,11,10,10,10,10,11,11,9,10,9,11,11,10,11,11,11,11,10,11,10,11,11,10,10,11,11,11,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,10,11,10,11,11,11,11,11,11,11,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,8,9,10,11,11,10,10,11,11,11,10,10,10,11,11,10,11,11,12,12,10,11,11,12,12,10,10,11,11,11,10,10,11,11,12,11,11,11,12,12,11,11,12,12,12,11,11,12,12,12,10,10,10,11,11,11,11,11,12,12,10,11,11,12,12,11,12,12,12,12,11,12,11,12,12,11,11,11,11,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,11,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,8,10,9,11,11,10,10,10,11,11,10,11,10,11,11,10,11,11,12,12,10,11,11,12,12,10,10,10,11,11,10,11,11,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,10,11,10,11,11,11,11,11,12,12,10,11,10,12,11,11,12,11,12,12,11,12,11,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,11,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,10,11,11,11,12,11,11,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,11,12,11,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,12,12,12,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,13,13,12,12,12,13,12,10,11], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+402676);
/* memory initializer */ allocate([11,12,11,11,11,12,12,12,11,12,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,11,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,13,12,13,12,8,10,10,11,11,10,10,11,11,11,10,11,10,11,11,10,11,11,12,12,10,11,11,12,12,9,10,11,11,11,10,10,11,12,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,10,11,10,11,11,11,11,11,12,12,10,11,11,12,12,11,12,12,12,12,11,12,11,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,10,11,11,12,12,11,11,12,12,12,11,11,12,12,12,11,11,12,12,12,12,12,12,12,12,10,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,9,10,10,11,11,10,11,11,12,12,10,11,11,12,11,11,12,12,12,12,11,12,12,12,12,10,11,11,12,12,11,11,11,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,10,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,12,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,12,12,13,13,12,13,12,13,13,12,12,12,12,12,12,12,12,12,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,12,12,12,12,12,12,12,13,13,13,12,13,12,13,13,12,13,13,13,13,12,13,13,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,12,12,13,12,13,13,12,13,12,13,13,12,12,12,12,12,12,12,13,13,13,12,12,13,13,13,12,13,13,12,13,13,13,13,13,13,12,12,12,12,12,12,13,12,13,13,12,13,12,13,12,12,13,13,13,13,12,13,13,13,13,8,10,10,11,11,10,10,11,11,11,9,11,10,11,11,10,11,11,12,12,10,11,11,12,12,10,10,11,11,11,10,11,11,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,9,11,10,11,11,10,11,11,12,12,10,11,10,12,12,11,12,12,12,12,11,12,11,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,10,11,11,12,12,11,11,12,12,12,11,11,11,12,12,12,12,12,12,12,11,12,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,11,12,12,12,12,11,12,11,12,12,10,11,11,12,12,11,11,12,12,12,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,10,11,11,12,12,11,12,11,12,12,11,12,11,12,12,12,12,12,12,12,11,12,11,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,12,13,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,13,13,13,12,12,13,13,13,12,13,13,12,13,12,13,13,13,13,12,12,12,12,12,12,12,13,13,13,12,12,12,13,12,12,13,13,13,13,12,13,13,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,13,12,13,12,13,12,12,12,12,12,12,12,12,13,13,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,12,12,12,12,12,12,13,12,13,12,12,13,12,13,12,13,13,13,13,13,12,13,13,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,12,13,13,13,13,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,13,12,12,12,13,12,11,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,13,13,12,12,13,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,12,12,13,12,13,13,12,13,12,13,13,12,12,12,12,12,12,12,13,12,13,12,12,13,13,13,12,12,13,13,13,13,13,13,13,13,12,12,12,12,12,12,13,13,13,13,12,13,12,13,12,12,13,13,13,13,12,13,13,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,12,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,12,13,13,12,13,12,13,13,12,12,12,12,12,12,12,12,12,13,12,13,12,13,13,12,13,13,13,13,12,13,13,13,13,12,12,12,12,12,12,13,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,12,13,13,11,12,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,13,12,13,13,12,12,12,12,12,12,12,13,12,13,12,12,13,12,13,12,12,13,12,13,12,13,13,13,13,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,13,13,13,13,12,13,12,13,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,12,12,13,12,13,12,10,11,11,12,12,11,12,12,12,12,11,12,11,12,12,11,12,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,13,11,12,11,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,13,13,12,13,12,13,13,13,13,12,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,12,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,12,13,13,12,12,12,13,13,12,12,12,12,12,12,12,13,13,13,12,12,13,13,13,12,12,13,13,13,12,13,13,13,13,12,12,12,12,12,12,12,13,13,13,12,12,12,13,12,12,13,13,13,13,12,13,13,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,12,13,13,13,11,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,13,12,13,12,13,13,12,12,12,12,12,12,12,12,13,13,12,13,12,13,13,12,13,13,13,13,13,13,13,13,13,12,12,12,12,12,12,13,12,13,13,12,13,12,13,12,12,13,13,13,13,12,13,12,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,12,12,12,11,12,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,13,12,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,12,13,12,13,13,13,13,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,13,12,13,12,13,12,2,6,6,10,10,5,7,8,11,12,5,8,7,12,11,9,11,11,13,15,9,11,11,15,13,6,7,8,11,11,7,7,9,11,13,8,9,9,13,12,11,11,12,12,15,11,12,12,15,14,6,8,7,11,11,8,9,9,12,13,7,9,7,13,11,11,12,12,14,15,11,12,11,15,12,10,11,11,12,14,10,11,12,12,15,12,13,13,14,15,13,12,14,12,16,15,15,15,16,16,10,11,11,14,12,12,13,13,15,14,10,12,11,15,12,15,15,15,16,17,13,14,12,17,12,6,8,8,12,12,8,9,10,13,13,8,9,9,13,13,12,12,13,15,16,12,13,13,16,15,8,9,10,12,13,9,9,11,13,14,10,11,11,14,14,13,13,14,15,16,13,14,14,16,16,8,10,9,13,13,10,11,11,14,14,9,10,10,14,13,13,14,14,16,17,13,13,13,16,15,12,13,13,14,16,13,13,14,14,16,14,14,14,16,16,15,15,16,15,18,16,17,17,18,18,12,13,13,15,15,14,14,14,16,16,13,14,13,16,15,16,16,17,18,18,15,16,15,18,15,6,8,8,12,12,8,9,9,13,13,8,10,9,13,13,12,13,13,15,16,12,13,12,16,15,8,9,10,13,13,9,10,10,13,14,10,11,11,14,14,13,13,13,15,16,13,14,14,17,16,8,10,9,13,13,10,11,11,14,14,9,11,9,14,13,13,14,14,16,16,13,14,13,16,14,12,13,13,15,16,13,13,14,15,16,14,14,14,16,16,15,15,16,15,18,17,17,17,18,18,12,13,13,16,14,14,14,14,16,16,13,14,13,16,14,16,17,17,18,18,15,16,15,18,15,11,12,13,14,16,13,13,14,15,17,13,14,14,16,17,16,16,17,17,19,16,17,17,18,19,13,13,14,16,16,14,14,15,16,17,14,15,15,17,17,17,16,17,17,19,17,17,18,19,19,13,14,14,16,16,14,14,15,17,18,14,15,14,17,17,17,17,18,18,19,17,17,17,18,19,16,16,16,17,18,17,17,17,18,19,17,17,17,18,19,18,18,19,18,20,19,20,19,21,20,16,17,17,18,18,17,17,18,19,19,17,17,17,19,18,19,19,19,19,20,19,19,19,20,19,11,13,12,16,14,13,14,14,17,16,13,14,13,17,15,16,17,17,18,18,16,17,16,19,17,13,14,14,16,16,14,14,14,17,17,14,15,15,17,16,17,17,17,19,19,17,18,17,19,18,13,14,13,17,16,14,15,15,17,17,14,15,14,18,16,17,17,17,19,19,17,17,16,19,17,16,17,17,18,19,17,17,17,18,18,17,18,17,19,18,18,19,18,19,19,19,20,19,20,20,16,17,16,18,17,17,17,17,18,18,17,18,17,19,17,19,19,19,19,20,18,19,19,20,18,6,8,8,12,12,8,9,9,13,13,8,10,9,13,13,11,13,13,15,16,12,13,13,16,15,8,9,9,13,13,9,9,10,13,14,10,11,11,14,14,12,12,13,14,16,13,14,14,17,16,8,10,9,13,13,10,11,11,14,14,9,11,10,14,13,13,14,14,16,16,13,14,13,16,15,12,13,13,14,16,12,13,14,14,16,13,14,14,16,16,15,14,16,15,18,16,17,17,18,17,12,13,13,16,15,14,14,14,16,16,13,14,13,16,15,16,16,17,17,17,15,16,15,18,15,7,9,9,13,13,9,9,11,13,14,9,10,10,14,13,12,13,14,15,16,12,14,13,17,15,9,9,10,13,14,10,9,11,13,15,11,11,11,14,14,13,12,14,14,17,14,14,14,17,16,9,10,10,14,13,11,11,11,14,14,10,11,10,15,13,14,14,14,16,17,13,14,13,17,14,13,13,14,14,16,13,13,14,14,17,14,14,14,16,16,15,14,16,15,18,17,17,17,18,18,13,14,13,16,15,14,14,15,17,16,13,14,13,17,15,17,16,17,17,17,15,16,14,18,14,7,9,9,13,13,9,10,10,13,14,9,11,10,14,13,13,14,14,16,16,13,14,14,17,15,9,10,10,14,13,9,10,11,13,14,11,12,11,15,14,13,13,14,14,16,14,15,15,17,17,9,10,10,14,14,11,12,12,14,15,10,11,10,15,13,14,15,15,17,17,14,15,13,17,14,13,14,13,16,16,13,13,14,15,16,14,15,15,17,17,15,14,16,15,18,17,18,17,20,18,13,14,14,16,16,15,15,15,17,17,13,14,13,17,15,17,17,18,18,18,15,16,14,19,14,12,13,13,15,16,13,13,15,16,17,13,14,14,16,16,15,15,17,17,19,16,17,17,19,18,13,13,14,15,17,14,13,15,15,17,14,15,15,16,17,16,15,18,16,19,17,17,17,18,19,13,14,14,17,16,14,15,15,17,17,14,15,14,17,16,17,17,17,18,19,16,17,16,19,17,16,16,17,16,18,16,16,17,16,19,17,17,18,18,19,18,17,18,17,21,19,19,19,20,19,16,17,17,18,18,17,17,18,18,19,16,17,16,18,18,19,19,19,19,20,18,18,17,20,18,11,13,13,16,15,13,14,14,16,17,13,15,14,17,16,16,17,17,18,18,17,17,17,19,18,13,14,13,17,16,14,13,14,16,17,15,16,15,18,16,17,16,17,17,19,18,18,18,20,18,13,14,14,16,17,15,15,15,17,18,14,15,14,18,16,18,18,18,19,20,17,18,16,20,17,16,17,16,18,18,16,16,17,18,18,17,18,18,19,18,18,17,19,17,20,19,20,19,22,20,16,16,17,18,18,18,17,17,19,19,16,17,16,18,17,19,20,19,22,21,18,19,18,21,17,6,8,8,12,12,8,9,10,13,13,8,9,9,13,13,12,13,13,15,16,11,13,13,16,15,8,9,10,13,13,9,10,11,13,14,10,11,11,14,14,13,13,14,15,16,13,14,14,16,16,8,9,9,13,13,10,11,11,14,14,9,10,9,14,13,13,14,14,16,17,12,14,12,16,14,12,13,13,15,16,13,13,14,15,16,13,14,14,15,17,15,15,16,15,18,16,16,17,17,17,12,13,13,16,14,13,14,14,16,16,12,14,13,16,14,16,17,17,18,18,15,15,14,18,14,7,9,9,13,13,9,10,11,13,14,9,10,10,14,13,13,14,14,15,17,13,14,14,16,15,9,10,10,14,14,10,10,11,13,15,11,12,12,15,14,14,13,15,14,17,14,15,15,17,17,9,10,10,13,14,11,11,12,14,15,9,11,10,14,13,14,15,15,16,18,13,14,13,16,14,13,14,14,16,16,13,13,14,15,17,15,15,15,16,17,15,14,16,15,18,17,17,18,19,18,13,14,14,16,16,14,15,15,17,17,13,14,13,16,15,17,17,18,18,18,15,16,14,18,15,7,9,9,13,13,9,10,10,13,14,9,11,10,14,13,12,13,14,15,16,12,14,13,16,15,9,10,10,13,14,10,10,11,13,14,11,11,11,15,14,13,13,14,14,16,14,14,14,17,16,9,10,9,14,13,11,11,11,14,14,10,11,9,15,13,14,14,14,16,16,13,14,12,17,14,13,13,14,15,16,13,13,14,15,16,14,15,14,16,17,15,14,16,14,18,16,17,17,18,18,13,14,13,16,14,14,14,14,16,16,13,14,13,17,14,17,17,17,18,18,15,16,14,18,15,11,13,13,16,16,13,14,15,16,17,13,14,14,17,16,16,17,17,18,19,17,17,17,19,18,13,14,14,17,17,13,13,15,16,18,15,15,15,17,17,17,16,18,17,20,18,17,18,19,19,13,14,14,16,17,15,15,16,16,18,14,15,14,16,16,17,17,18,18,20,17,18,16,18,17,16,17,16,19,18,16,16,17,18,19,18,18,18,19,19,18,17,18,17,21,20,19,19,21,21,16,16,17,18,18,17,17,18,19,19,16,17,16,19,18,20,20,20,19,21,18,18,17,20,18,12,13,13,16,15,13,14,14,16,16,13,14,13,17,16,16,17,17,18,18,15,17,15,19,17,13,14,14,16,17,14,14,15,16,17,14,15,15,17,17,16,16,17,17,18,17,17,17,19,19,13,14,13,17,15,14,15,15,17,16,14,15,13,17,15,17,18,17,19,18,16,17,15,20,16,16,17,17,18,18,16,16,17,18,18,17,18,17,19,18,17,17,18,18,20,19,20,19,20,19,16,16,16,19,16,17,17,17,19,18,16,17,16,19,16,19,19,19,19,19,18,19,17,19,17,11,13,13,16,16,13,14,14,17,17,13,14,14,17,17,15,17,17,19,19,16,18,17,20,19,12,14,14,17,17,13,14,15,17,18,14,15,15,17,18,16,16,17,18,20,17,18,18,20,18,13,14,14,17,17,14,15,15,17,18,14,15,15,17,17,17,18,17,19,19,17,18,17,19,19,15,16,16,18,18,15,16,17,18,19,16,17,17,19,19,17,17,18,18,21,18,19,19,21,19,16,17,17,18,18,17,17,18,19,19,17,18,17,19,19,19,19,19,20,20,18,19,18,21,19,12,13,13,16,16,13,14,14,16,17,13,15,14,17,16,15,16,17,17,19,16,17,17,19,18,13,13,14,16,17,14,13,15,16,17,14,15,15,17,17,15,15,17,17,20,17,17,18,19,18,13,14,14,17,16,15,15,15,17,18,14,15,14,17,16,17,17,17,18,18,16,17,16,19,17,16,15,17,17,19,16,15,17,16,19,17,16,17,18,19,17,16,19,16,20,19,18,19,19,19,16,17,17,18,18,17,17,17,18,19,16,17,16,19,18,20,19,19,20,19,18,18,17,20,17,11,13,13,16,16,13,14,15,16,17,14,15,14,18,16,17,17,17,18,21,17,18,17,20,19,13,14,14,17,16,13,14,15,16,18,15,16,15,18,17,17,16,17,17,19,17,18,18,20,19,13,14,14,16,17,15,15,16,17,18,14,15,14,18,17,17,18,18,19,20,17,18,16,19,17,16,17,15,19,18,16,16,16,18,18,17,18,17,20,19,18,17,18,17,20,20,20,19,22,20,16,17,17,18,19,18,18,18,19,20,16,17,16,19,18,20,19,19,20,20,18,19,17,20,17,13,14,14,16,17,14,14,16,16,18,14,16,15,17,16,16,16,17,17,18,17,17,16,19,18,14,14,15,16,17,14,14,16,16,18,16,16,16,17,17,16,15,17,16,19,18,18,18,19,19,14,15,15,17,17,15,16,16,17,18,14,16,14,18,16,17,17,18,18,19,16,17,16,19,17,16,16,17,16,18,16,16,17,16,19,18,18,18,17,18,17,16,18,16,20,19,19,19,19,19,16,17,17,18,18,17,17,18,19,19,16,17,16,19,17,18,19,19,19,20,17,18,16,20,16,11,14,13,17,17,14,14,16,16,18,14,16,14,19,16,18,18,19,18,19,18,19,18,21,18,13,15,14,18,16,14,14,16,16,18,16,17,16,19,17,18,16,19,17,20,19,19,19,21,19,13,14,15,17,18,17,16,17,17,19,14,16,14,18,16,20,19,19,20,21,18,19,16,21,17,17,18,16,19,17,16,16,17,18,18,19,19,18,21,18,17,17,18,17,20,20,20,20,22,20,17,17,18,18,20,19,19,19,18,20,16,17,17,19,19,21,21,21,20,21,17,19,17,23,17,11,13,13,16,16,13,14,14,17,17,13,14,14,17,17,16,17,17,19,20,15,16,16,19,19,13,14,14,16,17,14,15,15,17,18,14,15,15,17,17,17,17,18,19,19,17,17,18,19,19,13,14,14,17,16,14,15,15,17,17,13,15,14,18,17,17,18,18,19,20,16,17,16,19,18,16,16,17,18,18,17,17,17,18,19,17,18,17,19,19,19,19,19,19,20,19,20,19,20,20,15,16,16,18,17,16,17,17,20,18,15,16,16,19,17,19,19,19,20,20,17,18,17,21,17,11,13,13,16,16,13,14,15,16,17,13,15,14,17,16,17,17,18,18,20,17,17,17,19,19,13,14,14,17,17,14,14,15,17,18,15,15,15,18,17,17,17,18,17,20,18,18,17,20,18,13,14,14,16,17,15,15,16,17,18,14,15,13,17,17,17,18,18,19,20,17,17,16,19,17,16,17,17,18,18,16,16,17,18,18,18,18,18,19,19,18,17,19,18,21,19,20,20,20,20,16,15,17,18,18,17,17,18,18,20,16,16,16,18,17,20,19,20,21,22,17,18,17,20,17,12,13,13,16,16,13,14,15,16,17,13,14,14,17,16,16,17,18,18,19,15,16,16,19,18,13,14,14,16,17,14,14,15,16,17,14,15,15,17,17,16,16,17,17,19,17,17,17,19,18,13,14,13,17,16,14,15,15,17,17,13,15,13,17,16,17,17,17,19,19,15,17,15,19,17,16,17,17,18,18,16,16,17,17,19,17,18,17,19,19,18,17,19,17,19,19,19,19,20,19,15,17,15,19,16,17,17,16,19,18,16,17,15,18,16,19,19,19,20,19,17,19,16,19,16,11,14,14,17,17,15,14,16,16,18,15,16,14,18,16,18,18,19,18,21,18,19,18,20,18,13,15,14,18,17,14,14,16,16,18,16,17,16,19,17,17,17,19,17,22,19,19,19,21,19,13,14,15,17,18,17,16,17,17,19,14,16,14,18,16,19,19,19,20,21,18,18,16,20,17,17,18,16,19,18,15,17,17,19,19,19,19,18,21,19,18,17,20,17,21,22,21,20,21,21,17,16,19,18,20,19,18,19,18,20,16,17,16,19,18,21,20,21,19,23,18,19,16,20,17,13,14,14,17,16,14,14,15,16,18,14,16,14,17,16,16,16,17,17,19,16,17,16,19,17,14,15,15,17,17,14,14,16,16,17,15,16,16,18,17,16,16,17,17,19,17,18,17,19,18,14,15,14,17,16,16,16,16,17,17,14,16,14,17,16,18,18,18,18,19,16,17,15,19,16,17,17,17,18,18,16,15,17,17,18,18,18,18,19,19,17,16,18,16,19,19,19,19,19,19,16,17,16,19,16,18,18,17,19,18,16,17,16,19,16,19,19,20,19,19,17,18,16,20,16,2,3,3,3,3,3,3,1,5,5,5,7,9,5,9,7,5,7,8,7,7,10,9,10,10,5,8,7,9,10,10,7,10,7,6,9,9,9,10,12,9,11,11,9,10,11,11,11,13,12,13,13,9,11,11,12,13,13,11,13,11,6,9,9,9,11,11,9,12,10,9,11,11,11,11,13,12,13,13,9,11,10,12,13,13,11,13,11,6,9,9,9,11,12,9,12,11,9,10,11,10,10,13,12,13,13,9,11,11,12,13,12,11,13,11,7,9,10,9,10,12,10,12,11,10,10,12,10,10,12,12,12,13,10,11,11,12,12,13,10,12,10,7,10,10,11,11,14,11,14,11,10,12,11,11,11,14,14,14,14,10,11,12,14,14,14,11,14,11,6,9,9,9,11,12,9,12,11,9,11,11,11,11,13,12,12,13,9,11,10,12,13,13,10,13,10,7,10,10,11,11,14,11,14,11,10,12,11,11,11,14,14,15,14,10,11,12,13,14,15,11,14,11,7,10,9,10,11,12,9,12,10,10,11,11,10,10,12,12,13,12,9,12,10,12,13,12,10,12,10,2,6,6,6,7,8,6,8,7,6,7,7,7,7,8,7,8,8,6,7,7,7,8,8,7,8,7,6,8,8,8,9,9,8,9,9,8,9,9,9,9,10,9,10,10,8,9,9,9,10,10,9,10,9,6,8,8,8,9,9,8,9,9,8,9,9,9,9,10,9,10,10,8,9,9,9,10,9,9,10,9,6,8,8,8,9,9,8,9,9,8,9,9,9,9,10,9,9,10,8,9,9,9,10,9,9,10,9,7,8,8,8,9,9,8,9,9,8,8,9,9,9,9,9,9,9,8,9,9,9,10,9,9,9,9,7,9,9,9,9,10,9,10,9,9,9,9,9,9,10,10,10,10,9,9,9,10,10,10,9,10,9,6,8,8,8,9,9,8,9,9,8,9,9,9,9,10,9,10,10,8,9,9,9,10,9,9,10,9,7,9,9,9,9,10,9,10,9,9,9,9,9,9,10,10,10,10,9,9,9,10,10,10,9,10,9,7,8,8,8,9,9,8,9,9,8,9,9,9,9,10,9,9,10,8,9,8,9,9,9,9,10,9,1,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,1,4,5,5,10,10,5,10,10,5,10,10,10,10,10,10,10,10,5,10,10,10,10,10,10,10,10,7,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,6,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,6,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,6,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,9,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,11,11,11,11,11,11,1,2,3,4,5,7,7,8,8,9,9,10,10,11,11,12,12,13,13,14,14,15,15,15,15,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,2,7,13,15,16,17,19,20,6,3,4,7,9,10,12,15,13,4,3,4,7,8,11,13,14,7,4,4,6,7,10,11,16,9,7,6,7,8,9,10,16,9,8,7,7,6,8,8,18,12,10,10,9,8,8,9,20,14,13,12,11,8,9,9,2,3,1,3,1,4,4,7,7,10,10,12,12,12,12,13,12,5,5,5,8,6,11,9,12,12,13,12,12,12,4,5,5,6,8,9,11,12,12,13,12,12,12,7,7,8,9,9,11,8,12,9,12,12,12,12,7,8,8,9,9,8,11,9,12,12,12,11,12,10,10,10,11,11,11,11,11,10,11,11,12,11,10,10,10,11,11,11,11,10,11,11,11,11,12,11,11,11,12,11,12,11,12,11,13,11,13,11,11,11,11,11,12,11,12,10,13,11,12,11,13,12,12,12,13,12,13,13,13,12,14,12,14,13,12,12,12,12,13,13,13,12,14,12,14,13,14,13,14,14,14,14,14,14,14,14,15,14,15,14,13,14,13,14,14,14,14,14,15,14,14,14,15,4,4,4,5,5,4,5,5,5,5,4,5,5,5,5,5,5,5,4,5,5,5,5,5,4,1,3,2,5,5,6,6,6,6,3,9,14,16,17,19,22,22,5,4,6,9,11,13,17,20,9,5,5,6,9,11,15,19,11,7,5,5,7,9,13,17,14,9,7,6,6,7,11,14,16,11,9,7,6,4,4,8,19,15,13,11,9,4,3,4,21,16,16,15,12,6,4,4,2,5,5,4,7,7,4,7,7,5,7,7,7,8,9,7,9,9,5,7,7,7,9,9,7,9,8,6,7,8,8,9,10,8,9,10,8,9,10,10,10,12,10,11,11,8,10,10,10,11,12,10,11,11,6,8,7,8,10,9,8,10,9,8,10,10,10,11,11,10,12,11,8,10,9,10,11,11,10,12,10,5,8,8,8,10,10,8,10,10,7,9,10,9,10,11,9,11,11,8,10,10,10,11,12,10,12,11,7,9,9,9,10,11,9,11,11,9,9,11,10,11,12,11,11,12,9,11,11,11,12,12,11,12,12,7,9,9,10,11,11,10,12,11,9,11,10,11,11,12,11,13,12,10,11,11,12,13,13,11,13,11,5,8,8,8,10,10,8,10,10,8,10,10,10,11,12,10,12,11,7,10,9,9,11,11,9,11,10,7,9,9,10,11,12,10,11,11,10,11,11,11,11,13,12,13,13,9,10,11,11,12,13,11,12,11,7,9,9,9,11,11,9,11,10,9,11,11,11,12,12,11,12,12,9,11,9,11,12,11,10,12,11,4,6,6,9,9,6,8,8,10,10,6,8,8,10,10,8,10,10,12,13,8,10,10,13,12,6,8,8,10,10,8,8,9,10,11,8,9,9,11,11,10,10,11,12,13,10,11,11,13,13,6,8,8,10,10,8,9,9,11,11,8,9,8,11,10,10,11,11,13,13,10,11,10,13,12,9,10,10,12,12,10,10,11,12,13,10,11,11,13,13,12,12,13,12,15,13,13,13,15,14,9,10,10,12,12,10,11,11,13,13,10,11,10,13,12,12,13,13,14,15,12,13,12,15,12,6,8,8,10,11,8,9,10,11,12,8,9,9,11,11,10,11,12,13,14,10,11,11,13,13,8,9,9,11,12,9,10,11,12,13,9,10,10,12,13,11,12,13,13,15,11,12,12,14,14,8,9,9,11,12,9,10,11,12,13,9,10,10,13,12,11,12,13,14,15,11,12,12,14,13,10,11,12,13,14,11,12,13,13,15,12,13,13,14,14,13,13,14,14,16,14,15,14,16,15,10,12,11,14,13,12,12,13,14,14,11,12,12,14,14,14,14,15,15,16,13,14,14,16,14,6,8,8,11,10,8,9,9,11,11,8,10,9,12,11,10,11,11,13,13,10,12,11,14,13,8,9,9,12,11,9,10,10,12,13,9,11,10,13,12,11,12,12,14,14,11,13,12,15,14,8,9,9,12,11,9,10,10,13,12,9,11,10,13,12,11,12,12,14,14,11,13,12,15,13,10,11,12,13,14,11,12,13,13,14,12,13,12,14,14,13,13,14,14,16,14,15,14,16,16,10,12,11,14,13,12,13,13,14,14,11,13,12,15,13,14,14,15,16,16,13,14,13,16,14,9,10,11,12,13,11,11,12,13,14,11,11,12,13,14,13,13,14,14,16,13,14,14,15,15,11,11,12,13,14,12,12,13,13,15,12,13,13,14,15,14,14,15,15,17,14,14,15,16,16,11,12,12,13,14,12,12,13,14,15,12,13,12,14,15,14,14,15,15,17,14,15,14,16,16,13,14,14,15,16,14,14,15,15,17,14,15,15,16,16,15,16,17,16,18,16,17,16,17,17,13,14,14,16,15,14,15,15,16,16,14,15,14,16,15,16,16,17,17,18,16,16,16,17,16,9,11,10,13,12,11,12,11,14,13,11,12,11,14,13,13,14,14,16,15,13,14,13,16,14,11,12,12,14,13,12,12,13,14,14,12,13,13,15,14,14,14,15,16,16,14,15,14,17,15,11,12,11,14,13,12,13,13,15,14,12,13,12,15,13,14,15,14,16,16,14,15,14,17,15,13,14,14,15,16,14,14,15,16,16,14,15,15,16,16,15,16,16,16,17,16,16,16,17,17,13,14,14,16,15,14,15,15,17,16,14,15,14,17,15,16,17,17,17,17,16,16,16,18,16,6,8,8,11,11,8,9,9,11,12,8,9,9,12,11,10,11,11,13,14,10,11,11,14,13,8,9,9,11,12,9,10,10,12,13,9,10,10,13,12,11,11,12,13,15,11,12,12,15,14,8,9,9,12,11,9,10,11,12,13,9,11,10,13,12,11,12,12,14,15,11,13,12,15,14,10,11,11,13,14,11,12,12,13,14,11,12,12,14,14,13,13,14,14,16,13,14,14,16,15,11,12,11,14,13,12,13,13,14,14,11,13,12,14,13,14,14,15,16,16,13,14,14,16,14,8,9,9,11,12,9,10,10,12,13,9,10,10,13,12,11,12,12,14,15,11,12,12,14,14,9,9,10,11,13,10,10,12,12,14,10,10,11,13,13,12,12,13,14,16,12,12,13,15,15,9,10,10,13,12,10,11,11,13,14,10,12,11,14,13,12,13,13,15,15,12,13,13,15,15,11,11,12,13,15,12,12,13,13,15,12,13,13,14,15,14,14,15,15,17,14,15,15,16,16,11,13,12,15,14,13,13,13,15,15,12,14,13,15,14,15,15,15,16,16,14,15,15,17,15,7,9,9,12,11,9,10,10,12,12,9,11,10,13,12,11,12,12,14,14,11,13,12,15,14,9,10,10,12,12,10,10,11,12,13,10,11,11,14,13,12,12,13,14,15,12,13,13,15,14,9,10,10,12,12,10,11,11,13,13,10,11,10,14,12,12,13,13,15,15,12,13,12,15,13,11,12,12,14,14,12,12,13,14,15,12,13,13,15,15,14,13,14,13,16,14,15,15,16,16,11,12,12,14,14,13,13,14,15,15,12,13,12,15,14,15,15,15,16,16,14,15,14,17,14,10,11,12,13,14,11,12,13,14,15,11,12,12,14,15,13,14,15,15,17,14,14,14,16,16,11,12,13,12,15,12,12,14,13,16,13,13,14,13,16,14,14,15,14,17,15,15,15,15,17,11,13,12,15,15,13,13,14,15,16,12,14,13,16,15,15,15,15,17,17,15,15,15,17,16,14,14,15,14,16,14,14,16,14,17,15,15,15,14,17,16,16,17,15,18,17,17,17,16,18,14,15,15,17,16,15,16,16,17,17,15,16,15,17,16,17,17,17,18,18,16,17,16,18,17,10,11,11,14,13,11,12,12,14,14,11,13,12,15,14,14,14,14,16,16,14,15,14,16,15,11,12,12,15,13,12,13,13,15,14,13,14,13,16,14,14,15,15,16,16,15,16,15,17,16,11,13,12,15,14,13,13,14,15,15,12,14,13,16,14,15,15,15,17,17,14,16,15,17,16,14,14,14,16,15,14,15,15,16,16,15,16,15,17,16,16,16,16,16,17,16,17,17,18,17,14,15,15,16,16,15,15,16,17,16,14,15,15,17,16,17,17,17,18,18,16,17,16,18,16,6,8,8,11,11,8,9,9,11,12,8,9,9,12,11,10,11,12,13,14,10,11,11,14,13,8,9,9,11,12,9,10,11,12,13,9,11,10,13,12,11,12,13,14,15,11,12,12,15,14,8,9,9,12,11,9,10,10,12,13,9,10,10,13,12,11,12,12,14,15,11,12,12,14,13,11,11,12,13,14,11,12,13,13,15,12,13,13,14,14,13,14,14,14,16,14,15,14,16,16,10,11,11,14,13,11,12,12,14,14,11,12,12,14,13,13,14,14,15,16,13,14,13,16,14,7,9,9,11,11,9,10,11,12,13,9,10,10,12,12,11,12,13,14,15,11,12,12,14,14,9,10,10,12,12,10,10,11,12,13,10,11,11,13,13,12,12,13,13,15,12,13,13,15,15,9,10,10,12,12,10,11,11,13,13,10,11,10,13,12,12,13,13,14,15,12,13,12,15,13,11,12,12,14,14,12,12,13,14,15,13,14,13,15,15,14,13,15,13,16,15,15,15,16,16,11,12,12,14,14,12,13,13,14,15,12,13,12,15,14,14,15,15,16,17,13,14,13,16,13,8,9,9,12,11,9,10,10,12,13,9,10,10,13,12,11,12,12,14,15,11,12,12,15,14,9,10,10,12,13,10,11,12,13,14,10,11,11,14,13,12,13,13,15,15,12,13,13,15,15,9,10,9,13,11,10,11,10,13,13,10,12,10,14,12,12,13,12,15,15,12,13,12,15,14,11,12,13,14,15,12,13,14,14,15,13,13,13,15,15,14,15,15,15,17,15,15,15,16,16,11,12,11,15,13,12,13,13,15,14,12,13,12,16,13,14,15,15,16,16,14,15,14,17,14,10,11,11,13,14,11,12,13,14,15,11,12,12,14,14,14,14,15,15,17,14,14,14,15,16,11,12,13,14,15,12,13,14,14,16,13,14,13,15,15,14,15,16,15,17,15,15,15,17,17,11,12,12,13,15,13,13,14,14,16,12,13,13,14,15,15,15,15,16,17,14,15,15,16,16,14,15,15,16,16,14,15,15,16,17,15,15,16,16,17,16,16,17,16,18,17,17,17,18,18,14,14,15,15,16,15,15,15,16,17,14,15,15,16,16,16,17,17,17,18,16,16,16,17,16,10,11,11,14,13,11,13,12,15,14,11,13,12,15,14,14,15,14,16,16,13,15,14,17,15,11,12,13,15,15,12,13,14,15,16,13,14,13,16,15,15,15,15,16,17,15,15,15,17,16,11,13,11,15,12,13,14,13,16,13,12,14,12,16,13,15,15,15,17,15,14,16,14,17,14,14,15,15,16,17,15,15,16,16,17,15,16,15,17,17,16,16,17,17,18,16,17,17,18,18,14,15,14,17,13,15,16,15,17,15,15,16,15,17,14,16,17,16,18,16,16,17,16,18,15,9,11,11,13,13,10,12,12,14,14,11,12,12,14,14,13,14,14,15,16,13,14,14,16,16,10,11,12,14,14,11,12,13,14,15,11,13,13,15,15,13,14,14,15,16,14,15,15,16,16,11,12,12,14,14,12,13,13,15,15,12,13,12,15,14,14,15,15,16,16,14,15,14,17,16,12,13,13,15,16,13,13,14,15,16,13,14,14,16,16,14,15,16,16,17,15,16,16,17,17,13,14,14,16,15,14,15,15,17,16,14,15,14,17,15,16,16,17,17,17,16,16,16,18,16,10,11,12,14,14,11,12,13,14,15,11,13,12,15,15,13,14,15,16,16,14,15,15,17,16,11,11,13,14,15,12,12,14,14,16,12,13,14,15,15,14,14,15,16,17,15,15,15,17,17,12,13,12,15,15,13,14,14,16,15,13,14,13,16,15,15,16,15,17,17,15,16,15,17,16,13,12,15,14,16,14,13,15,14,17,14,13,15,15,17,15,14,17,15,18,16,15,17,17,18,14,15,15,17,16,15,16,16,17,17,15,16,15,17,16,16,17,17,18,18,16,17,16,18,17,10,11,11,14,14,11,12,12,14,15,11,13,12,15,14,13,14,14,16,16,14,15,14,16,16,11,12,12,14,14,12,12,13,15,15,12,13,13,15,15,14,14,15,16,16,14,15,15,17,16,11,12,12,15,15,13,13,13,15,15,12,13,13,15,15,15,15,15,17,17,14,15,15,17,16,13,14,13,16,15,14,14,14,16,16,14,15,14,17,16,15,15,16,16,17,16,17,16,18,17,14,15,15,16,16,15,15,15,17,17,14,15,15,17,16,16,17,17,18,18,16,17,16,18,16,12,13,13,15,15,13,14,14,16,16,13,14,14,16,16,14,15,16,16,18,15,16,16,17,17,13,13,14,14,16,14,14,15,15,17,14,14,15,15,17,15,15,17,15,18,16,16,17,17,18,13,14,14,16,16,14,15,15,16,17,14,15,15,17,16,16,17,16,17,18,16,17,16,18,17,15,14,16,13,18,16,15,17,14,18,16,15,17,14,18,17,16,18,15,19,17,17,18,16,19,15,16,16,17,17,16,17,17,18,18,16,17,16,18,17,18,18,18,19,18,17,18,17,19,17,11,12,12,15,15,13,13,14,15,16,13,14,13,16,15,15,15,15,16,17,15,16,15,17,16,12,13,13,15,15,13,13,14,15,16,14,15,14,16,15,15,15,16,16,17,16,16,16,18,17,12,13,13,15,15,14,14,15,16,16,13,14,13,16,15,16,16,16,17,17,15,16,15,18,16,15,15,15,17,15,14,15,15,16,16,16,17,16,17,16,16,16,17,16,17,17,18,17,19,18,15,15,16,17,17,16,16,16,17,17,15,16,15,17,16,17,18,18,18,18,16,17,16,18,16,9,11,11,13,13,11,12,12,14,14,10,12,12,14,14,13,14,14,15,16,13,14,14,16,15,11,12,12,14,14,12,12,13,14,15,12,13,13,15,15,14,14,15,16,17,14,15,15,16,16,10,12,11,14,14,11,13,13,15,15,11,13,12,15,14,14,14,15,16,16,13,14,14,16,15,13,14,14,15,16,14,14,15,15,17,14,15,15,16,17,16,16,16,16,18,16,16,17,17,17,12,13,13,16,15,13,14,14,16,16,12,14,13,16,15,15,16,16,17,17,14,16,15,17,16,10,11,11,14,14,11,12,13,14,15,11,12,12,15,14,14,14,15,16,16,13,14,14,16,16,11,12,12,14,15,12,13,14,15,15,13,13,13,15,15,14,15,15,16,17,15,15,15,16,17,11,12,12,14,14,12,13,13,15,15,12,13,12,15,15,14,15,15,16,17,14,15,14,16,16,14,14,15,16,16,14,15,15,16,17,15,16,15,17,17,16,16,17,16,18,16,17,17,18,18,13,13,14,15,16,14,14,15,16,17,14,14,14,16,15,16,16,17,17,18,15,16,15,17,16,10,12,11,14,14,11,13,13,15,15,11,13,12,15,15,14,15,15,16,16,13,15,14,16,16,12,12,13,15,15,13,13,14,15,16,13,14,14,16,15,15,15,16,16,17,15,15,15,17,17,11,13,11,15,14,12,14,13,16,15,12,14,12,16,14,15,15,15,17,17,14,15,14,17,15,14,15,15,16,17,15,15,16,16,17,15,16,16,17,17,16,16,17,17,18,16,17,17,18,18,13,14,12,16,14,14,15,13,17,15,14,15,13,17,14,16,17,15,18,17,15,17,14,18,15,11,12,12,14,15,13,13,14,15,16], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+412916);
/* memory initializer */ allocate([13,14,13,16,15,15,15,16,16,17,15,15,15,16,16,12,13,13,15,15,13,13,14,15,16,14,15,14,16,16,15,15,16,16,18,16,16,16,18,17,12,13,13,15,15,14,14,15,15,16,13,14,13,15,15,16,16,16,17,18,15,16,15,17,16,15,16,15,17,16,15,15,16,16,17,16,17,16,17,17,16,16,17,16,18,17,18,18,18,18,14,15,15,15,17,16,15,17,16,17,14,15,15,16,16,17,17,18,18,19,16,16,16,17,16,12,13,13,15,15,13,14,14,16,16,13,14,14,16,16,15,16,16,17,17,15,16,15,18,16,13,14,14,16,16,14,15,15,16,17,14,15,15,17,16,16,16,17,17,18,16,17,16,18,18,13,14,13,16,14,14,15,14,17,15,14,15,14,17,14,16,17,16,18,17,15,17,15,18,15,15,16,16,17,18,16,16,17,17,18,16,17,17,17,18,17,17,18,18,19,17,18,18,19,18,15,16,14,17,13,16,17,15,18,14,16,17,15,18,14,18,18,17,19,16,17,18,16,19,15,2,5,5,4,7,7,4,7,7,5,7,8,7,8,10,8,9,9,5,7,7,8,9,9,7,10,8,5,7,8,8,9,10,8,10,10,8,9,10,10,10,12,10,12,12,8,10,10,10,12,12,10,12,11,5,8,7,8,10,10,8,10,9,8,10,10,10,11,12,10,12,12,8,10,9,10,12,12,10,12,10,5,8,8,7,10,10,8,10,10,7,9,10,9,10,12,10,12,12,8,10,10,10,12,12,10,12,11,7,9,10,9,11,12,10,12,11,9,9,12,11,10,14,12,12,13,10,12,11,12,13,13,11,14,12,7,10,9,10,11,11,10,12,11,9,11,11,11,11,13,12,14,13,10,12,12,12,14,14,11,14,12,5,8,8,8,10,10,7,10,10,8,10,10,10,11,12,10,12,12,7,10,9,10,12,12,9,12,10,7,9,10,10,11,12,10,11,11,10,12,12,11,12,14,12,14,14,9,11,11,12,13,14,11,13,11,7,10,9,10,11,12,9,12,11,10,11,12,11,12,14,12,13,13,9,12,9,12,13,12,11,14,10,6,7,7,7,7,7,7,7,7,7,7,8,7,8,8,7,8,8,7,8,7,7,8,8,7,8,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,8,9,8,8,8,8,8,8,8,9,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,8,8,8,8,8,8,9,8,8,9,8,7,8,8,7,8,8,7,8,8,7,8,8,8,8,8,8,8,8,8,8,8,8,9,8,8,9,8,7,8,8,8,8,9,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,9,9,9,8,9,9,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,7,8,8,7,8,8,7,8,8,8,8,8,8,8,9,8,8,9,7,8,8,8,8,8,8,8,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,7,8,8,8,8,8,8,9,8,8,8,8,8,9,9,9,9,9,8,8,8,8,9,9,8,9,8,1,5,5,5,7,8,5,8,7,5,7,8,7,8,10,8,10,10,5,8,7,8,10,10,7,10,8,6,8,9,9,10,12,9,11,11,9,10,11,11,11,13,11,13,13,9,11,11,11,12,13,11,13,11,6,9,8,9,11,11,9,12,10,9,11,11,11,11,13,11,13,13,9,11,10,11,13,13,11,13,11,6,9,9,8,10,11,9,12,11,8,10,11,10,11,13,11,13,13,9,11,11,11,13,12,11,13,11,8,10,10,9,11,12,10,12,12,10,10,12,11,11,14,12,13,14,10,12,12,12,13,13,11,14,11,8,11,10,11,12,13,11,14,12,10,12,11,11,12,14,13,15,14,10,12,12,13,14,15,12,14,12,5,9,9,9,11,12,8,11,10,9,11,11,11,11,13,11,12,13,8,11,10,11,13,13,10,13,11,8,10,11,11,12,14,11,13,12,10,12,12,12,12,14,14,15,14,10,11,12,13,14,15,11,14,12,8,10,10,10,12,12,9,12,11,10,12,12,11,11,14,12,13,13,10,12,10,12,14,13,11,13,11,7,8,8,10,10,8,9,9,10,11,8,9,9,10,10,9,10,10,11,11,9,10,10,11,11,8,9,9,10,10,9,9,10,11,11,9,10,10,11,11,10,10,11,11,11,10,11,11,11,11,8,9,9,10,10,9,10,10,11,11,9,10,9,11,11,10,11,11,11,11,10,11,10,11,11,10,10,10,11,11,10,11,11,11,11,10,11,11,11,11,11,11,11,11,12,11,11,11,11,12,10,10,10,11,11,10,11,11,11,11,10,11,11,11,11,11,11,11,12,11,11,11,11,12,11,8,9,10,11,11,9,10,11,11,11,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,10,10,10,11,11,10,10,11,11,12,10,11,11,12,12,11,11,12,12,12,11,11,12,12,12,10,10,10,11,11,10,11,11,12,12,10,11,11,12,11,11,12,12,12,12,11,12,11,12,12,11,11,11,11,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,8,10,9,11,11,9,10,10,11,11,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,10,10,10,11,11,10,11,11,12,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,10,10,10,11,11,10,11,11,12,12,10,11,10,12,11,11,12,11,12,12,11,12,11,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,11,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,10,11,11,11,12,11,11,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,12,12,13,13,11,12,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,12,12,12,12,12,12,12,12,13,12,12,13,13,13,12,13,13,13,13,12,13,13,13,13,12,12,12,12,12,12,12,13,13,13,12,12,12,13,12,12,13,13,13,13,12,13,13,13,13,10,11,11,12,11,11,11,11,12,12,11,12,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,11,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,11,12,12,12,12,12,12,12,11,12,11,12,12,12,12,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,13,13,13,12,12,12,12,12,12,12,12,13,13,12,13,12,13,12,12,13,13,13,13,13,13,13,13,13,8,10,10,11,11,9,10,10,11,11,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,9,10,10,11,11,10,10,11,11,12,10,11,11,12,12,11,11,12,12,12,11,11,12,12,12,10,10,10,11,11,10,11,11,12,12,10,11,10,12,11,11,12,11,12,12,11,12,11,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,11,11,11,12,11,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,9,10,10,11,11,10,11,11,11,12,10,11,11,12,12,11,11,11,12,12,11,11,11,12,12,10,10,11,11,12,11,11,12,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,10,11,11,12,12,11,11,11,12,12,11,12,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,9,10,10,11,11,10,11,11,12,12,10,11,11,12,11,11,12,11,12,12,11,12,11,12,12,10,11,11,12,12,11,11,11,12,12,11,12,11,12,12,11,12,12,12,12,12,12,12,12,12,10,11,11,12,12,11,12,11,12,12,11,12,11,12,12,12,12,12,13,12,12,12,12,12,12,11,12,11,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,11,12,12,12,12,12,12,12,13,12,11,12,12,12,12,12,12,12,13,12,12,12,12,13,12,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,11,11,12,12,12,12,12,12,12,13,12,12,12,12,12,12,12,13,12,13,12,12,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,12,13,13,12,13,12,13,13,12,12,12,12,12,12,12,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,12,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,12,12,12,12,12,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,12,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,8,10,10,11,11,9,10,10,11,11,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,10,10,10,11,11,10,11,11,11,12,10,11,11,12,12,11,11,12,12,12,11,11,12,12,12,9,10,10,11,11,10,11,11,12,12,10,11,10,12,11,11,12,11,12,12,11,12,11,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,11,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,10,11,11,12,12,11,11,12,12,12,11,11,11,12,12,12,12,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,12,12,13,12,11,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,13,12,12,12,12,13,12,9,10,10,11,11,10,11,11,12,12,10,11,11,12,12,11,11,11,12,12,11,12,11,12,12,10,11,11,12,12,11,11,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,10,11,10,12,11,11,11,11,12,12,11,12,11,12,12,11,12,12,12,12,11,12,11,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,11,12,11,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,13,12,12,12,12,13,12,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,12,12,12,13,13,13,13,12,13,12,13,13,12,12,12,12,13,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,12,12,12,13,13,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,11,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,13,12,13,12,13,12,12,12,12,12,13,12,12,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,12,12,12,13,13,13,13,12,13,12,13,12,13,13,13,13,13,13,13,13,13,13,10,11,11,12,12,10,11,11,12,12,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,11,11,12,12,11,12,12,12,12,11,12,11,13,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,13,12,13,13,11,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,13,13,13,12,13,13,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,11,12,12,12,11,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,12,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,12,13,13,12,12,12,12,13,12,12,13,12,13,12,12,13,13,13,12,12,13,13,13,12,13,13,13,13,12,12,12,12,13,12,12,13,13,13,12,12,12,13,13,13,13,13,13,13,12,13,13,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,11,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,12,13,13,12,13,12,13,13,12,12,12,12,12,12,12,12,13,13,12,13,12,13,13,12,13,13,13,13,13,13,13,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,12,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,13,13,13,13,12,13,12,13,13,12,12,12,12,12,12,12,13,12,13,12,13,13,13,13,12,13,13,12,13,13,13,13,13,13,12,12,12,12,12,12,13,13,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,10,11,11,12,12,11,12,12,12,13,11,12,12,13,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,13,13,12,13,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,13,12,12,12,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,12,12,12,12,12,12,12,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,10,11,11,12,12,10,11,11,12,12,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,11,12,12,11,11,12,12,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,10,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,13,13,12,12,12,13,12,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,12,13,13,13,11,12,12,12,12,12,12,12,13,13,11,12,12,13,12,12,13,13,13,13,12,13,12,13,13,12,12,12,12,13,12,12,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,13,12,12,12,13,13,13,12,12,12,13,13,13,13,13,13,13,12,13,13,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,12,13,13,13,11,12,11,12,12,12,12,12,13,13,11,12,12,13,12,12,13,12,13,13,12,13,12,13,13,12,12,12,12,12,12,12,13,13,13,12,13,12,13,13,12,13,13,13,13,13,13,13,13,13,12,12,12,13,12,12,13,12,13,13,12,13,12,13,13,13,13,13,13,13,12,13,12,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,13,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,13,13,13,12,13,12,13,13,12,12,12,12,13,12,12,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,12,12,12,13,13,13,13,12,13,12,13,13,13,13,13,13,13,13,13,13,13,13,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,12,13,13,13,12,13,13,13,13,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,13,12,13,13,12,13,12,13,13,12,12,12,12,12,12,13,13,13,13,12,13,13,13,13,13,13,13,13,13,13,13,13,13,13,12,12,12,12,12,12,13,13,13,13,12,13,12,13,12,13,13,13,13,13,13,13,13,13,12,2,6,6,9,9,5,7,8,10,11,5,8,7,11,10,8,10,11,12,13,8,11,10,13,12,6,7,8,10,11,7,8,10,10,12,8,9,9,12,11,10,10,12,11,14,10,11,12,14,13,6,8,7,11,10,8,9,9,11,12,7,10,8,12,10,10,12,12,13,14,10,12,10,14,11,9,10,11,11,12,10,10,11,11,13,11,12,12,13,13,12,11,13,11,15,13,14,13,14,14,9,11,10,12,11,11,12,12,13,13,10,11,10,13,11,13,13,14,14,14,12,13,11,14,11,7,8,9,11,12,9,9,11,12,13,9,10,10,13,12,11,12,13,13,15,11,12,12,14,14,9,10,10,12,13,10,10,12,12,14,11,11,11,13,13,12,12,13,13,15,12,13,13,15,14,9,10,10,12,13,10,11,11,13,14,10,12,11,14,13,12,13,13,14,15,12,13,13,15,14,12,12,13,13,14,12,13,13,13,15,13,14,14,14,15,14,14,15,14,16,14,15,15,16,16,12,13,13,14,14,13,13,14,15,14,12,13,13,15,14,14,15,15,15,16,14,15,14,16,14,7,9,8,12,11,9,10,10,12,13,9,11,9,13,12,11,12,12,14,14,11,13,12,15,13,9,10,10,13,12,10,11,12,13,14,10,12,11,14,13,12,13,13,14,15,13,13,13,15,14,9,10,10,13,12,11,11,11,13,13,10,12,10,14,12,13,13,13,14,15,12,13,12,15,13,12,13,13,14,14,12,13,13,14,15,13,14,13,15,15,14,14,15,14,16,14,15,15,16,15,12,13,12,14,13,13,13,13,15,14,12,13,13,15,13,14,15,15,16,15,14,15,14,16,14,11,12,12,13,14,12,13,14,14,15,12,13,13,14,15,14,14,15,15,16,14,15,15,16,16,12,13,13,14,15,13,13,14,14,16,13,14,14,15,15,15,15,16,15,17,15,15,15,16,16,12,13,13,14,15,13,14,14,15,16,13,14,14,15,15,15,15,16,16,17,15,15,15,17,16,14,15,15,16,16,15,15,16,15,16,15,16,16,16,17,16,16,17,16,18,16,16,17,18,17,14,15,15,16,16,15,16,16,16,17,15,16,15,17,16,16,17,17,17,18,16,16,16,17,16,11,12,12,14,13,12,13,13,15,14,12,14,13,15,14,14,15,15,16,16,14,15,14,16,15,12,13,13,15,14,13,14,14,15,15,13,14,14,16,15,15,15,15,16,16,15,16,15,17,16,12,13,13,15,14,13,14,14,15,15,13,14,13,16,14,15,15,15,16,16,15,15,15,17,15,14,15,15,16,16,15,15,15,16,16,15,16,16,17,17,16,16,17,17,17,16,17,17,18,17,14,15,15,16,15,15,15,16,16,16,15,15,15,17,15,17,17,17,18,17,16,17,16,18,16,6,9,9,12,12,8,10,10,12,13,9,11,10,13,12,10,12,12,14,14,11,13,12,14,14,8,10,10,12,12,9,10,11,12,14,10,11,11,13,13,12,12,13,13,15,12,13,13,15,14,9,10,10,13,13,10,11,11,13,13,10,12,10,14,13,12,13,13,14,15,12,13,13,15,14,11,12,12,13,14,12,12,13,13,15,12,13,13,14,14,13,13,14,13,16,14,15,15,16,15,11,12,12,14,14,13,13,13,15,14,12,13,13,15,14,14,15,15,16,15,14,14,14,16,14,7,9,10,12,12,9,10,11,13,13,9,11,10,13,13,11,12,13,14,15,12,13,13,15,14,9,10,11,12,13,10,10,12,13,14,11,11,12,14,14,12,12,14,14,15,13,13,13,15,15,9,11,11,13,13,11,12,12,14,14,10,12,10,14,13,13,14,13,15,15,12,14,13,15,14,12,12,13,13,15,12,12,14,13,15,13,14,14,15,15,14,14,15,14,17,14,15,15,16,16,12,13,13,15,14,13,14,14,15,15,12,14,13,15,14,14,15,15,16,16,14,15,14,16,14,7,10,10,12,12,10,11,11,12,13,10,12,10,14,12,12,13,13,14,15,12,13,13,15,14,9,11,10,13,12,10,10,12,12,14,11,13,12,14,13,13,13,14,13,15,13,14,14,15,14,10,11,11,13,13,12,12,12,13,14,10,12,10,14,12,13,14,14,15,15,13,14,13,15,13,12,13,13,14,14,12,12,13,14,15,13,14,14,15,15,13,13,14,13,15,14,15,15,16,16,12,13,13,14,14,13,14,14,15,15,12,13,13,15,13,15,15,15,16,16,13,14,13,16,13,11,12,13,14,14,12,13,14,14,15,12,13,13,15,15,14,14,15,15,17,14,15,15,16,16,12,13,14,14,15,13,13,14,14,16,13,14,14,15,16,14,14,16,15,17,15,15,16,16,16,12,13,13,15,15,13,14,14,15,16,13,14,14,15,16,15,15,16,17,17,15,16,15,17,16,14,15,15,15,16,15,15,16,15,17,15,15,16,16,17,16,16,16,16,18,16,16,17,17,17,14,15,15,16,16,15,16,16,16,17,15,16,15,17,16,16,17,17,17,17,16,17,16,18,17,11,12,12,14,14,13,13,14,14,15,13,14,13,15,14,14,15,15,15,16,14,15,15,17,15,12,13,13,15,14,13,13,14,15,15,14,15,14,16,15,15,15,15,15,16,15,16,15,17,16,12,13,13,15,15,14,14,14,15,16,13,14,13,16,15,15,15,16,16,17,15,16,15,17,15,14,15,15,16,16,14,15,15,16,16,15,16,16,17,16,15,15,16,15,17,16,17,17,18,17,14,15,15,16,16,15,16,16,16,17,14,15,15,17,16,17,17,17,17,18,15,16,16,18,15,6,9,9,12,12,9,10,11,12,13,8,10,10,13,12,11,12,13,14,14,10,12,12,14,13,9,10,10,12,13,10,10,12,13,14,10,11,11,13,13,12,13,13,14,15,12,13,13,15,14,8,10,10,12,12,10,11,11,13,13,9,11,10,13,13,12,13,13,14,15,12,13,12,15,13,11,12,12,14,14,12,13,13,13,15,13,13,13,14,15,14,14,15,14,16,14,15,15,15,15,11,12,12,14,13,12,13,13,15,14,12,13,12,15,13,14,14,15,16,16,13,14,13,16,13,7,10,10,12,12,10,10,12,12,14,10,11,11,13,12,12,13,13,13,15,12,13,13,15,14,10,11,11,13,13,10,10,12,12,14,12,12,12,14,13,13,13,14,13,15,13,14,14,15,14,9,10,11,13,13,11,12,12,13,14,10,12,10,14,12,13,13,14,14,15,13,13,12,15,13,12,13,13,14,14,12,13,13,14,15,13,14,14,15,15,13,13,15,13,16,15,15,15,16,16,12,13,13,14,14,13,14,14,15,15,12,13,12,15,14,15,15,15,16,16,13,14,13,15,13,7,10,9,12,12,9,10,11,13,13,9,11,10,13,13,11,13,13,14,15,11,13,12,15,14,9,11,11,13,13,10,10,12,13,14,11,12,12,14,14,12,13,14,14,15,13,13,13,15,15,9,11,10,13,12,11,12,11,14,14,10,12,10,14,13,13,14,13,15,15,12,14,12,15,14,12,13,13,14,15,13,13,14,14,15,13,14,14,15,15,14,14,15,14,17,14,15,15,16,16,12,13,12,15,13,13,14,14,15,15,12,14,13,15,13,14,15,15,16,16,14,15,14,16,14,11,12,12,14,14,13,13,14,14,15,13,14,13,15,15,14,15,15,16,17,14,15,15,16,15,12,13,13,15,15,13,13,14,15,16,14,14,14,16,15,15,15,16,15,17,15,16,15,17,16,12,13,13,14,15,14,14,15,15,16,13,14,13,15,15,15,15,16,16,17,15,15,15,16,15,14,15,15,16,16,14,15,15,16,17,15,16,16,17,17,16,15,16,15,17,16,17,17,17,17,14,15,15,15,16,15,15,16,16,17,14,15,15,16,16,16,16,17,17,18,15,16,15,17,15,11,13,12,14,14,12,13,13,15,15,12,14,13,15,14,14,15,15,16,16,14,15,14,16,15,12,13,13,15,15,13,14,14,15,16,13,14,14,16,16,15,15,16,16,17,15,16,15,17,16,12,13,13,15,14,13,14,14,16,15,13,14,13,16,14,15,16,15,17,16,15,15,14,18,15,14,15,15,16,16,15,15,16,16,17,15,16,15,17,16,16,16,17,17,18,16,17,17,18,17,14,15,15,16,15,15,16,15,17,16,15,15,15,17,15,16,17,17,18,17,16,17,16,18,15,10,12,12,14,14,12,13,13,14,14,12,13,13,14,14,13,14,14,15,15,13,14,14,16,15,11,12,13,14,14,12,13,13,15,15,12,13,13,15,15,13,14,15,15,16,14,15,15,16,16,12,13,13,14,14,13,13,14,15,15,13,14,13,15,15,14,15,15,16,16,14,15,14,16,15,13,14,14,15,15,13,14,14,15,16,14,14,15,16,16,14,15,15,15,17,15,16,16,17,17,13,14,14,15,15,14,15,15,16,16,14,15,15,16,16,15,16,16,16,17,15,16,15,17,16,11,12,12,14,14,12,13,13,14,15,12,13,13,15,14,13,14,14,15,16,13,14,14,16,15,12,13,13,14,15,13,13,14,15,15,13,14,14,15,15,14,14,15,15,17,14,15,15,16,16,12,13,13,15,15,13,14,14,15,15,13,14,13,15,15,14,15,15,16,17,14,15,15,16,16,13,13,14,15,16,14,14,15,15,16,14,15,15,16,16,15,15,16,15,18,15,16,16,17,17,14,15,15,16,16,15,15,15,16,16,14,15,15,17,16,16,16,16,17,17,15,16,16,17,16,10,12,12,14,14,12,13,13,14,15,12,13,13,15,14,14,14,15,15,16,14,15,14,16,15,12,13,13,15,14,13,13,14,15,15,13,14,14,15,15,14,14,15,15,16,14,15,15,16,16,12,13,13,15,15,13,14,14,15,16,13,14,13,15,14,15,15,15,16,16,14,15,15,16,15,13,14,14,16,15,14,14,14,15,16,14,15,15,16,16,15,15,16,15,17,16,17,16,17,17,14,14,15,15,16,15,15,16,16,16,14,15,14,16,15,16,16,16,17,17,15,16,15,17,15,11,13,13,14,15,13,13,14,15,15,13,14,13,15,15,14,15,15,15,16,14,15,15,17,15,13,13,14,15,15,13,14,15,15,16,14,14,14,16,16,15,14,16,15,17,15,16,16,17,16,13,14,14,15,15,14,14,14,16,16,13,15,14,16,15,15,15,16,17,17,15,16,15,17,16,14,15,15,15,16,15,15,16,15,17,15,16,16,16,17,16,16,17,15,18,16,17,17,17,17,14,15,15,16,16,15,16,16,17,17,15,16,15,17,16,16,17,17,18,18,16,17,15,18,16,10,12,12,14,14,13,13,14,14,15,13,14,13,15,14,14,15,15,15,16,15,15,15,16,15,12,13,13,15,14,12,12,14,14,15,14,15,14,16,15,15,14,15,14,17,15,16,16,17,16,12,13,13,14,15,14,14,15,15,16,13,14,12,16,14,15,16,16,16,17,15,16,14,17,15,14,15,14,16,15,14,14,15,15,15,15,16,15,17,16,15,14,16,14,16,16,17,17,18,17,14,14,15,15,16,15,16,16,16,17,14,15,14,16,15,16,16,17,17,17,15,16,14,17,14,10,12,12,14,13,12,13,13,14,14,11,13,12,14,14,13,14,14,15,16,13,14,14,16,15,12,13,13,14,14,13,13,14,15,15,13,14,13,15,15,14,14,15,15,16,14,15,15,16,16,11,13,12,14,14,12,13,13,15,15,12,13,13,15,15,14,15,15,16,16,13,14,14,16,15,13,14,14,15,15,14,15,15,15,16,14,15,15,16,16,15,16,16,16,17,16,16,16,17,17,13,14,14,15,15,14,15,15,16,16,13,14,14,16,15,15,16,16,17,17,15,15,15,17,15,11,12,12,14,14,12,13,13,14,15,12,13,13,15,14,14,14,15,15,16,14,14,14,16,15,12,13,13,15,14,13,13,14,15,15,13,14,14,16,15,14,15,15,15,16,15,15,15,16,16,12,13,13,14,15,13,13,14,15,15,13,14,13,15,15,15,15,15,16,16,14,15,14,16,15,14,14,15,16,16,14,15,15,15,16,15,16,15,16,16,15,15,16,15,17,16,16,16,17,17,13,14,14,15,16,14,15,15,16,16,14,14,14,16,16,16,16,16,17,17,15,15,15,17,15,11,12,12,14,14,12,13,13,14,15,12,13,13,15,14,14,14,14,15,16,13,14,14,16,15,12,13,13,15,15,13,13,14,15,16,13,14,14,15,15,14,15,15,16,17,14,15,15,17,16,12,13,13,15,14,13,14,14,15,15,13,14,13,15,15,14,15,15,16,16,14,15,14,17,15,14,15,15,16,16,14,15,15,16,17,15,15,15,17,17,15,16,16,16,17,16,17,16,17,17,13,15,14,16,15,14,15,15,16,16,14,15,14,16,15,16,16,16,17,17,15,16,15,17,15,10,12,12,14,14,13,13,14,14,15,13,14,13,15,14,14,15,15,15,17,14,15,15,16,15,12,13,13,15,14,12,12,14,14,15,14,15,14,16,15,15,14,16,15,17,15,16,16,17,16,12,13,13,14,15,14,14,15,15,16,12,14,12,15,14,15,16,16,16,17,15,16,14,17,14,14,15,14,16,16,14,14,15,15,16,15,16,16,17,16,15,14,16,14,17,16,17,17,18,17,14,14,15,15,16,15,15,16,16,17,14,15,14,16,15,16,17,17,17,18,15,16,14,17,14,11,13,13,15,14,13,13,14,15,15,12,14,13,15,15,14,15,15,15,17,14,15,14,16,15,13,14,14,15,15,13,14,15,15,16,14,15,14,16,16,15,15,16,16,17,15,16,16,17,17,13,14,13,15,15,14,14,14,16,16,13,15,14,16,15,15,16,16,17,17,15,16,14,17,15,15,15,15,16,17,15,15,16,16,17,15,16,16,17,17,16,15,17,16,17,17,17,17,18,18,14,15,15,17,15,15,16,16,17,16,15,16,15,17,15,16,17,17,17,17,16,17,15,18,15,2,3,3,3,3,3,3,2,5,6,5,7,8,5,8,7,5,7,7,7,7,9,8,9,9,5,7,7,8,9,9,7,9,7,6,8,8,8,9,10,8,9,9,8,9,10,9,9,11,10,10,11,8,10,9,10,10,11,9,10,10,6,8,8,8,9,9,8,10,9,8,9,10,9,10,10,10,11,10,8,10,9,10,11,10,9,11,9,6,8,8,7,9,9,8,9,9,7,9,9,9,9,10,9,10,10,8,9,9,9,10,10,9,10,9,7,9,9,9,10,10,9,10,10,9,9,10,10,9,11,10,11,11,9,10,10,10,11,11,10,11,10,6,9,8,9,10,10,9,10,9,8,10,10,9,9,10,10,11,11,9,10,10,10,11,11,9,11,9,6,8,8,8,9,9,7,9,9,8,9,9,9,9,10,9,10,10,7,9,9,9,10,10,9,10,9,6,8,9,9,9,10,9,10,10,9,10,10,9,9,11,10,11,11,8,10,10,10,11,11,9,10,9,7,9,9,9,10,10,9,10,10,9,10,10,10,10,11,10,11,11,9,10,9,10,11,11,10,11,9,4,7,7,6,7,8,6,8,7,7,7,8,7,7,8,8,8,8,7,7,7,8,8,8,7,8,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,7,8,8,8,8,9,8,9,8,8,8,8,8,8,9,8,9,9,8,8,8,9,9,9,8,9,9,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,9,9,8,9,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,8,9,9,8,8,8,9,9,9,8,9,8,1,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,1,5,5,4,10,10,5,10,10,5,10,10,10,10,10,10,10,10,5,10,10,10,10,10,9,10,10,6,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,7,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,6,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,6,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,10,11,11,1,3,2,4,5,7,7,8,8,9,9,10,10,11,11,12,12,13,13,14,14,15,15,15,15,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,2,7,14,16,17,17,18,20,6,3,5,8,10,11,13,15,13,5,3,5,8,9,11,12,15,7,4,3,5,7,9,11,16,10,7,5,6,7,9,10,17,11,8,7,7,6,8,8,19,13,11,9,9,8,8,9,20,14,13,11,10,8,9,9,3,9,15,17,20,21,22,23,5,5,7,9,11,13,17,20,9,5,5,6,8,10,15,18,11,7,5,4,6,9,13,17,14,9,7,5,6,7,10,14,17,10,8,6,6,4,5,8,20,14,13,10,8,4,3,4,23,17,16,14,12,6,4,4,2,5,5,4,7,7,4,7,7,5,7,7,7,8,9,7,9,9,5,7,7,7,9,9,7,9,8,6,7,8,8,9,10,8,9,10,8,9,10,10,10,12,10,11,12,8,10,10,10,11,12,10,11,11,6,8,7,8,10,9,8,10,9,8,10,10,10,11,11,10,12,11,8,10,9,10,12,11,10,12,10,5,8,8,8,10,10,8,10,10,7,9,10,9,10,11,9,11,11,8,10,10,10,12,12,10,12,11,7,9,9,9,10,11,9,11,11,9,9,11,10,11,12,10,11,12,9,11,11,11,12,12,11,12,12,7,9,9,10,11,11,10,12,11,9,11,10,11,11,12,11,13,12,10,11,11,12,13,13,11,13,11,5,8,8,8,10,10,8,10,10,8,10,10,10,11,12,10,12,11,7,10,9,9,11,11,9,11,10,7,9,9,10,11,12,10,11,11,10,11,11,11,11,13,12,13,13,9,10,11,12,12,13,11,12,11,7,9,9,9,11,11,9,11,10,9,11,11,11,12,12,11,12,12,9,11,9,10,12,11,10,12,11,4,6,6,9,9,6,8,8,10,10,6,8,8,10,10,8,9,10,12,12,8,10,9,12,12,6,8,8,10,10,8,8,9,10,11,8,9,9,11,11,9,10,11,12,13,10,11,11,13,13,6,8,8,10,10,8,9,9,11,11,8,9,8,11,10,10,11,11,13,13,9,11,10,13,12,9,10,10,12,12,10,10,11,12,13,10,11,11,13,13,12,12,13,12,15,12,13,13,15,14,9,10,10,12,12,10,11,11,13,13,10,11,10,13,12,12,13,13,14,15,12,13,12,15,12,7,8,8,10,11,8,9,10,11,12,8,9,9,11,11,10,11,11,13,14,10,11,11,13,13,8,9,9,11,12,9,10,11,11,13,9,10,10,12,12,11,11,12,13,15,11,12,12,14,14,8,9,9,11,11,9,10,11,12,13,9,10,10,12,12,11,12,12,14,15,11,12,12,14,14,10,11,12,13,13,11,12,12,13,14,12,12,12,14,14,13,13,14,14,16,14,14,14,16,15,10,11,11,13,13,12,12,12,14,14,11,12,12,14,13,14,14,14,15,16,13,14,13,16,14,7,8,8,11,10,8,9,9,11,11,8,10,9,12,11,10,11,11,13,13,10,11,11,14,13,8,9,9,12,11,9,10,10,12,12,9,11,10,13,12,11,12,12,13,14,11,12,12,15,14,8,9,9,12,11,9,10,10,12,12,9,11,10,13,11,11,12,12,14,14,11,12,12,14,13,10,11,11,13,13,11,12,12,13,14,12,13,12,14,14,13,13,14,14,16,13,14,14,16,15,10,11,11,13,13,12,12,12,14,14,11,12,12,14,13,13,14,14,15,15,13,14,13,16,14,9,10,11,12,13,11,11,12,12,14,11,11,12,13,14,13,13,14,14,16,13,13,14,15,15,11,11,12,12,14,12,12,13,13,15,12,12,13,13,15,14,14,15,15,16,14,14,14,15,16,11,12,12,13,14,12,12,13,14,15,12,13,12,14,14,14,14,15,15,16,14,14,14,16,16,13,13,14,15,16,14,14,15,15,16,14,15,15,16,16,15,15,16,16,18,16,16,16,17,17,13,14,14,15,15,14,14,15,16,16,14,15,14,16,16,16,16,16,17,18,15,16,16,17,16,9,11,10,13,12,11,12,11,14,13,11,12,11,14,12,13,14,13,15,14,13,14,13,16,14,11,12,12,14,13,12,12,13,14,14,12,13,12,15,14,14,14,14,16,16,14,15,14,17,15,11,12,11,14,12,12,13,12,15,13,12,13,12,15,13,14,14,14,16,15,14,15,14,16,15,13,14,14,15,15,14,14,15,16,16,14,15,14,16,16,15,15,16,16,17,16,16,16,17,17,13,14,14,16,15,14,15,15,16,16,14,15,14,17,15,16,16,16,17,17,15,16,15,18,16,7,8,8,10,11,8,9,9,11,12,8,9,9,12,11,10,11,11,13,14,10,11,11,14,13,8,9,9,11,11,9,10,10,12,12,9,10,10,12,12,11,12,12,13,14,11,12,12,14,14,8,9,9,12,11,9,10,11,12,13,9,11,10,13,12,11,12,12,14,14,11,12,12,14,13,10,11,11,13,13,11,12,12,13,14,11,12,12,14,14,13,13,14,14,16,13,14,14,16,15,10,12,11,13,13,12,12,12,14,14,11,12,12,14,13,14,14,14,15,16,13,14,14,16,14,8,9,9,11,11,9,10,10,12,12,9,10,10,12,12,11,11,12,13,14,11,12,12,14,14,9,9,10,11,12,10,10,11,12,13,10,10,11,12,13,12,12,13,14,15,12,12,13,14,15,9,10,10,12,12,10,11,11,13,13,10,11,11,13,13,12,13,13,15,15,12,13,13,15,14,11,11,12,13,14,12,12,13,13,15,12,12,13,14,15,14,14,15,14,16,14,14,15,15,16,11,12,12,14,14,12,13,13,15,15,12,13,13,15,14,14,15,15,16,16,14,15,14,17,15,8,9,9,11,11,9,10,10,12,12,9,11,10,13,12,11,12,12,14,14,11,13,12,15,13,9,10,10,12,12,10,10,11,12,13,10,12,11,13,13,12,12,13,13,15,12,13,13,15,14,9,10,10,12,12,11,11,12,13,13,10,12,10,13,12,12,13,13,15,15,12,13,13,15,13,11,12,12,14,14,12,12,13,14,14,12,13,13,15,14,13,13,14,13,16,14,15,14,16,16,11,12,12,14,14,13,13,13,15,15,12,13,12,15,14,14,15,15,16,17,14,15,13,16,13,10,11,11,13,14,11,12,12,13,15,11,12,12,14,14,13,14,14,15,16,13,14,14,16,16,11,11,12,12,14,12,12,13,13,15,12,13,13,13,15,14,14,15,14,17,14,14,15,15,16,11,12,12,14,14,12,13,13,15,15,12,13,13,15,15,14,15,15,16,17,14,15,15,16,16,13,14,14,14,16,14,14,15,14,17,14,15,15,14,17,16,16,17,15,18,16,16,17,16,18,13,14,14,16,16,14,15,15,17,16,14,15,15,17,16,16,17,17,18,18,16,17,16,18,17,10,11,11,14,13,11,12,12,14,14,11,13,12,15,14,14,14,14,16,15,14,15,14,16,15,11,12,12,14,13,12,13,13,15,14,13,14,13,15,14,14,15,15,16,16,14,15,15,17,15,11,12,12,14,14,13,13,13,15,15,12,13,13,15,14,15,15,15,17,17,14,15,15,17,15,13,14,14,16,15,14,15,15,16,16,15,15,15,17,16,16,16,16,16,17,16,17,16,18,17,14,14,14,16,16,15,15,15,16,16,14,15,14,17,16,16,17,17,17,18,16,17,16,18,16,7,8,8,11,11,8,9,9,11,12,8,9,9,12,11,10,11,11,13,14,10,11,11,14,13,8,9,9,11,12,9,10,11,12,13,9,11,10,13,12,11,12,12,13,14,11,12,12,14,14,8,9,9,11,11,9,10,10,12,12,9,10,10,13,12,11,12,12,14,14,11,12,11,14,13,10,11,12,13,13,11,12,12,13,14,12,13,12,14,14,13,13,14,14,16,13,14,14,16,15,10,11,11,13,13,11,12,12,14,14,11,12,12,14,13,13,14,14,15,16,13,14,13,16,14,8,9,9,11,11,9,10,11,12,13,9,10,10,12,12,11,12,13,13,14,11,12,12,14,14,9,10,10,12,12,10,10,11,12,13,11,12,11,13,13,12,12,13,13,15,12,13,13,15,15,9,10,10,12,12,10,11,12,13,14,10,11,10,13,12,12,13,13,14,15,12,13,12,15,13,12,12,12,14,14,12,12,13,14,15,13,13,13,15,15,14,14,15,13,16,14,15,15,16,16,11,12,12,14,14,12,13,13,14,15,12,13,12,14,14,14,14,15,16,16,13,14,13,16,14,8,9,9,11,11,9,10,10,12,12,9,10,10,12,12,11,12,12,14,14,11,12,11,14,14,9,10,10,12,12,10,11,11,13,13,10,11,11,13,13,12,13,13,14,15,12,13,13,15,14,9,10,9,12,11,10,11,10,13,12,10,11,10,13,12,12,13,12,15,14,12,13,12,15,14,11,12,12,14,14,12,13,13,14,15,12,13,13,15,15,14,14,15,15,17,14,15,15,16,16,11,12,11,14,13,12,13,12,15,14,12,13,12,15,13,14,15,14,16,15,13], "i8", ALLOC_NONE, Runtime.GLOBAL_BASE+423156);
/* memory initializer */ allocate([15,14,17,14,10,11,11,13,14,11,12,13,13,15,11,12,12,14,14,14,14,15,15,17,13,14,14,15,16,11,12,12,14,14,12,12,13,14,15,13,13,13,15,15,14,15,15,15,17,15,15,15,16,16,11,12,12,13,14,13,13,14,14,15,12,13,13,14,15,14,15,15,16,17,14,15,15,16,16,14,14,14,16,16,14,14,15,15,17,15,15,15,17,16,16,16,17,16,18,16,17,17,18,17,13,14,14,15,16,14,15,15,16,17,14,15,15,16,16,16,17,17,17,18,16,16,16,17,16,10,11,11,14,13,11,12,12,14,14,11,12,12,15,13,13,14,14,16,15,13,14,14,16,15,11,12,12,14,14,12,13,13,15,15,12,13,13,15,15,14,15,15,16,17,14,15,15,17,16,11,12,11,14,12,12,13,13,15,13,12,13,12,15,13,14,15,15,16,15,14,15,14,17,14,13,14,14,16,16,14,15,15,16,17,14,15,15,16,17,16,16,17,17,18,16,17,17,18,18,13,14,14,16,13,14,15,15,17,14,14,15,14,17,14,16,17,16,17,16,16,17,16,18,15,8,11,11,13,13,10,12,12,14,14,11,12,12,14,14,13,13,14,15,16,13,14,14,16,15,10,11,11,14,14,11,12,12,14,15,11,12,12,15,14,13,14,14,15,16,13,14,14,16,16,11,12,12,14,14,12,13,13,15,15,12,13,12,15,14,14,14,15,16,16,14,15,14,16,16,12,13,13,15,15,12,13,14,15,16,13,14,14,16,16,14,15,15,16,17,15,15,16,17,17,13,14,14,16,15,14,15,15,16,16,14,15,14,16,16,16,16,16,17,17,15,16,16,18,16,10,11,11,13,14,11,12,12,14,15,11,12,12,15,14,13,14,14,16,16,13,14,14,16,16,11,11,12,14,14,12,12,13,14,15,12,13,13,15,15,14,14,15,15,17,14,14,15,16,16,11,12,12,15,14,12,13,13,15,15,12,13,13,15,15,14,15,15,17,17,14,15,15,17,16,13,12,14,14,16,13,13,15,14,17,14,13,15,15,17,15,14,16,15,18,16,15,16,16,18,13,14,14,16,16,14,15,15,17,17,14,15,15,17,16,16,17,17,18,18,16,17,16,18,17,10,11,11,14,13,11,12,12,14,14,11,13,12,15,14,13,14,14,15,16,13,14,14,16,16,11,12,12,14,14,12,13,13,14,15,12,13,13,15,15,14,14,15,15,16,14,15,15,17,16,11,12,12,14,14,13,13,13,15,15,12,13,13,15,14,14,15,15,16,17,14,15,14,17,15,13,14,13,16,15,14,14,14,15,16,14,15,14,16,16,15,15,16,16,17,16,16,16,18,17,14,14,14,16,16,15,15,15,17,16,14,15,14,17,16,16,16,17,17,18,16,17,16,18,16,11,13,13,15,15,12,13,14,15,16,12,14,14,15,15,14,15,15,16,17,14,15,15,17,17,12,13,14,14,16,13,14,14,14,16,14,14,14,15,16,15,15,16,15,18,15,16,16,17,17,13,14,14,16,16,14,14,15,16,16,14,15,14,16,16,15,16,16,17,18,15,16,16,18,17,14,14,16,13,17,15,15,16,14,18,15,15,16,14,18,16,16,18,15,19,17,17,18,16,18,15,16,15,17,17,15,16,17,18,18,16,16,16,18,17,17,18,18,19,19,17,18,17,19,18,11,12,12,15,14,13,13,14,15,16,13,14,13,16,14,15,15,15,16,17,15,16,15,17,16,12,13,13,15,14,13,13,14,15,15,14,15,14,16,15,15,15,16,16,17,16,16,16,18,17,12,13,13,15,15,14,14,15,16,16,13,14,13,16,15,16,16,16,17,18,15,16,15,17,16,14,15,14,17,15,14,15,15,16,16,15,16,15,17,16,16,15,16,15,17,17,18,17,18,17,15,15,15,16,17,16,16,16,17,17,15,16,15,17,16,17,18,18,18,18,16,17,16,18,15,8,11,11,13,13,11,12,12,14,14,10,12,12,14,14,13,14,14,15,16,13,14,13,16,15,11,12,12,14,14,12,12,13,14,15,12,13,13,15,15,14,14,15,15,16,14,14,14,16,16,10,11,11,14,14,11,12,12,14,15,11,12,12,15,14,13,14,14,16,16,13,14,14,16,15,13,14,14,15,16,14,14,15,16,16,14,15,15,16,16,15,16,16,16,18,16,16,16,17,17,12,13,13,15,15,13,14,14,16,16,12,14,13,16,15,15,16,15,17,17,14,16,15,17,16,10,11,11,13,14,11,12,13,14,15,11,13,12,14,14,14,14,15,16,16,13,14,14,16,16,11,12,12,14,14,12,13,13,14,15,13,14,13,15,15,14,15,15,16,17,14,15,15,17,16,11,12,12,14,14,12,13,13,15,15,12,13,12,15,14,14,15,15,16,17,14,15,15,16,16,14,14,14,16,16,14,14,15,16,16,15,15,15,16,16,16,16,17,16,18,16,17,17,18,18,13,13,14,15,16,14,14,15,16,17,13,14,14,16,16,16,16,17,17,18,15,16,15,17,16,10,11,11,14,13,11,12,12,14,14,11,12,12,15,14,13,14,14,16,16,13,14,14,16,16,11,12,12,14,14,12,13,13,15,15,12,13,13,15,15,14,15,15,16,17,14,15,15,17,16,11,12,11,14,14,12,13,13,15,15,12,13,12,15,14,14,15,14,16,16,14,15,14,17,16,14,14,14,16,16,14,15,15,16,17,14,15,15,17,17,16,16,17,17,18,16,17,17,18,18,13,14,12,16,14,14,15,13,17,15,13,15,13,17,14,16,16,15,18,16,15,17,14,18,15,11,12,12,14,15,13,13,14,14,16,13,14,13,15,14,15,15,16,16,17,15,16,15,17,16,12,13,13,15,15,13,13,14,15,16,14,15,14,16,16,15,15,16,15,18,16,16,16,18,17,12,13,13,15,15,14,14,15,15,16,13,14,13,15,15,16,16,16,16,18,15,16,15,17,16,15,15,15,17,16,15,15,16,16,17,16,16,16,18,17,16,16,17,15,18,17,18,17,19,18,14,14,15,15,17,15,15,16,16,17,14,15,15,16,16,17,17,18,17,19,16,17,15,17,15,11,13,12,15,15,12,14,14,15,15,12,14,13,16,15,15,15,15,17,17,14,15,15,17,16,12,14,14,16,16,14,14,15,16,16,14,14,14,16,16,15,16,17,17,18,15,16,16,18,17,12,14,13,16,14,13,14,14,16,15,13,15,14,16,14,15,16,16,17,17,15,16,15,18,15,15,15,16,17,17,15,16,16,17,18,16,16,16,18,18,17,17,18,18,19,17,17,18,19,19,14,15,14,17,13,15,16,15,18,14,15,16,15,18,14,17,18,17,18,16,16,18,16,19,15,2,5,5,5,7,7,5,7,7,5,7,7,7,8,9,7,9,9,5,7,7,7,9,9,7,9,8,5,7,8,7,9,10,8,9,9,8,9,10,9,10,12,10,11,11,8,10,9,10,11,12,9,11,10,5,8,7,8,10,9,7,10,9,8,9,10,9,10,11,10,12,11,8,10,9,10,11,11,9,12,10,5,8,8,7,9,10,8,10,9,7,9,10,9,10,11,9,11,11,8,10,9,10,11,11,10,12,10,7,9,10,9,10,12,9,11,11,9,9,12,11,10,13,11,11,13,10,12,11,11,13,13,11,13,12,7,9,9,9,11,11,9,12,11,9,11,10,10,11,12,11,13,12,9,11,11,12,13,13,11,13,11,5,8,8,8,9,10,7,10,9,8,9,10,10,10,12,10,11,11,7,10,9,9,11,11,9,11,10,7,9,9,9,11,12,9,11,11,9,11,11,11,11,13,12,13,13,9,10,11,11,12,13,10,12,11,7,10,9,9,11,11,9,12,10,10,11,12,11,12,13,12,13,13,9,12,9,11,13,11,10,13,10,6,7,7,7,7,8,7,8,7,7,7,8,7,8,8,8,8,8,7,8,7,7,8,8,7,8,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,8,9,8,8,8,8,9,9,8,9,9,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,8,8,8,8,9,9,8,9,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,7,8,8,8,8,8,8,8,8,8,8,8,8,8,9,8,9,9,8,8,8,8,8,9,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,9,9,8,9,9,8,8,8,8,9,8,8,9,8,2,5,5,4,7,8,4,8,7,5,7,8,7,7,10,8,9,9,5,7,7,8,9,9,7,10,7,5,7,8,8,9,11,8,10,10,8,9,10,10,10,12,11,12,12,8,10,10,10,12,12,10,12,11,5,8,7,8,10,10,8,11,9,8,10,10,10,11,12,10,12,12,8,10,9,11,12,12,10,12,10,5,8,8,7,10,10,8,11,10,7,9,10,9,10,12,10,12,12,8,10,10,10,12,12,10,12,11,7,9,10,9,11,12,10,12,11,9,9,12,10,10,13,12,12,13,10,12,11,12,13,13,11,13,11,7,10,9,10,11,12,10,13,11,9,11,11,11,11,13,12,14,13,10,11,11,12,14,14,11,14,11,5,8,8,8,10,11,7,10,10,8,10,10,10,11,12,10,12,12,7,10,9,10,12,12,9,12,10,7,9,10,10,11,13,10,12,11,10,11,11,11,11,14,12,14,14,9,11,11,12,13,14,11,13,11,7,10,9,10,11,12,9,12,10,10,11,12,11,11,13,12,13,13,9,12,9,12,13,12,10,13,10,7,9,9,10,10,9,10,10,10,11,9,10,10,11,10,9,10,10,11,11,9,10,10,11,11,9,10,10,11,11,10,10,10,11,11,10,10,10,11,11,10,11,11,11,11,10,11,11,11,11,9,10,10,11,11,10,10,10,11,11,9,10,10,11,11,10,11,11,11,11,10,11,11,11,11,10,11,11,11,11,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,11,12,10,11,11,11,11,11,11,11,11,11,10,11,11,11,11,11,11,11,11,11,11,11,11,11,11,9,10,10,11,11,10,10,11,11,11,10,10,11,11,11,10,11,11,11,12,10,11,11,12,12,10,10,11,11,11,10,11,11,11,12,11,11,11,12,12,11,11,12,12,12,11,11,12,12,12,10,11,11,11,11,11,11,11,12,12,10,11,11,12,12,11,12,11,12,12,11,12,11,12,12,11,11,11,11,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,9,10,10,11,11,10,11,10,11,11,10,11,10,11,11,10,11,11,12,12,10,11,11,12,11,10,11,11,11,11,10,11,11,11,12,11,11,11,12,12,11,11,12,12,12,11,11,11,12,12,10,11,10,11,11,11,11,11,12,12,10,11,11,12,11,11,12,11,12,12,11,12,11,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,11,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,10,11,11,11,11,11,11,11,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,12,12,12,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,10,11,11,11,11,11,11,11,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,13,12,13,12,9,10,10,11,11,10,10,11,11,11,10,11,10,11,11,10,11,11,12,12,10,11,11,12,12,10,10,11,11,11,10,11,11,11,12,10,11,11,12,12,11,11,12,12,12,11,11,11,12,12,10,11,10,11,11,11,11,11,12,12,10,11,11,12,11,11,12,11,12,12,11,12,11,12,12,11,11,11,11,12,11,11,12,12,12,11,12,12,12,12,11,12,12,12,12,11,12,12,12,12,11,11,11,12,11,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,10,10,11,11,11,10,11,11,12,12,10,11,11,12,12,11,11,11,12,12,11,11,12,12,12,10,11,11,11,12,11,11,12,12,12,11,11,12,12,12,11,11,12,12,12,11,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,9,10,10,11,11,10,11,11,11,12,10,11,11,12,11,11,12,11,12,12,11,12,11,12,12,10,11,11,12,11,11,11,11,12,12,11,12,11,12,12,11,12,12,12,12,11,12,12,12,12,10,11,11,12,12,11,12,11,12,12,11,12,11,12,12,12,12,12,12,12,11,12,12,12,12,11,12,11,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,12,13,13,13,13,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,13,13,13,12,13,13,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,13,13,12,13,12,13,13,13,13,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,13,13,13,12,13,13,13,13,9,10,10,11,11,10,10,11,11,11,10,11,10,11,11,10,11,11,12,12,10,11,11,12,12,10,11,11,11,11,10,11,11,12,12,11,11,11,12,12,11,11,12,12,12,11,11,12,12,12,10,11,10,11,11,10,11,11,12,12,10,11,11,12,11,11,12,11,12,12,11,11,11,12,12,11,11,11,11,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,11,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,11,12,12,12,12,9,10,10,11,11,10,11,11,11,12,10,11,11,12,11,11,11,12,12,12,11,11,12,12,12,10,11,11,12,12,11,11,12,12,12,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,10,11,11,12,12,11,11,11,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,10,11,10,11,11,10,11,11,12,12,10,11,11,12,12,11,11,11,12,12,11,12,11,12,12,11,11,11,12,12,11,11,12,12,12,11,11,12,12,12,11,12,12,12,12,11,12,12,12,12,10,11,11,12,11,11,12,11,12,12,11,12,11,12,12,11,12,12,12,12,11,12,11,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,13,12,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,12,12,13,13,12,13,13,12,13,12,13,13,13,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,13,13,12,13,12,13,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,12,12,12,12,12,12,12,13,13,12,12,12,12,13,12,13,13,13,13,12,13,13,13,13,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,13,13,13,12,13,13,13,12,10,11,11,12,12,11,11,11,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,11,12,12,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,12,12,13,13,13,12,12,12,12,12,12,12,12,12,13,12,12,12,13,12,12,13,13,13,13,12,13,13,13,13,10,11,11,12,12,11,12,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,13,11,12,11,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,12,13,12,13,12,13,13,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,13,12,13,12,13,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,12,13,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,13,12,13,12,12,12,12,13,12,12,13,12,13,12,13,13,12,13,12,12,12,12,12,12,13,13,13,12,12,12,12,13,12,12,13,13,13,13,12,13,13,13,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,13,13,13,11,12,12,12,12,12,12,12,12,13,12,12,12,13,12,12,13,12,13,13,12,13,12,13,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,12,13,13,13,12,12,12,12,12,12,12,13,12,13,12,12,12,12,13,12,12,13,13,13,12,12,13,12,13,12,10,11,11,12,12,11,11,11,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,11,11,11,12,12,11,11,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,11,12,12,12,12,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,10,11,11,12,12,11,11,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,11,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,13,11,11,11,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,13,12,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,12,12,13,13,13,12,12,12,12,12,12,12,12,12,13,12,12,12,12,12,12,12,13,13,13,12,12,12,13,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,13,11,12,11,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,13,13,13,13,12,13,13,13,13,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,12,13,13,12,13,12,13,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,12,13,12,12,13,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,13,12,12,12,13,12,12,12,12,12,12,12,12,12,12,13,12,12,12,13,13,12,12,13,12,13,12,13,13,13,13,12,12,12,12,12,12,12,13,12,13,12,12,12,12,12,12,13,13,12,12,12,13,12,12,12,11,11,11,12,12,11,12,12,12,12,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,11,12,12,12,12,12,12,12,12,13,12,12,12,12,13,12,12,13,13,13,12,12,12,13,13,11,12,12,12,12,12,12,12,12,12,12,12,12,12,12,12,13,12,13,13,12,13,12,13,12,12,12,12,12,12,12,12,12,12,13,12,13,12,13,13,12,13,13,12,13,12,13,13,13,13,12,12,12,12,12,12,12,12,13,12,12,13,12,13,12,12,13,12,13,12,12,13,12,13,12,2,6,6,9,9,5,7,8,10,11,5,8,7,11,10,8,10,11,12,13,8,11,10,13,12,6,7,8,10,11,7,8,10,10,12,8,9,9,12,12,10,10,12,12,14,10,12,12,14,13,6,8,7,11,10,8,9,9,12,12,7,10,8,12,11,10,12,12,13,14,10,12,10,14,12,9,10,11,11,13,10,10,11,11,13,11,12,12,13,14,12,12,13,11,15,13,14,14,15,14,9,11,10,13,11,11,12,12,13,13,10,11,10,13,11,13,14,14,15,15,12,13,12,15,11,6,8,9,11,12,8,9,11,12,13,8,10,10,13,13,11,12,13,14,15,11,12,13,14,14,9,9,10,12,13,10,10,12,12,14,10,11,11,13,14,12,12,14,14,15,13,13,14,15,15,9,10,10,13,13,10,11,11,13,14,10,11,10,14,13,13,13,14,15,15,12,14,13,15,14,12,12,13,13,14,12,13,14,13,15,13,14,14,15,15,14,14,15,14,16,15,15,15,16,16,12,13,13,14,14,13,14,14,15,15,12,14,13,15,14,14,15,15,16,16,14,15,14,16,14,6,9,8,12,11,8,10,10,13,13,8,11,9,13,12,11,12,12,14,14,11,13,12,15,14,9,10,10,13,13,10,10,11,13,14,10,12,11,14,13,12,13,14,14,15,13,13,13,15,14,9,10,9,13,12,10,11,11,14,13,10,12,10,14,12,13,14,13,15,15,12,14,12,15,14,12,13,13,14,14,13,13,13,14,15,13,14,14,15,15,14,14,15,14,16,14,15,15,16,16,12,13,12,14,13,13,14,14,15,15,12,14,13,15,13,15,15,15,16,16,14,15,14,16,14,11,12,12,13,14,12,13,14,14,16,12,13,13,15,15,14,14,16,15,17,14,15,15,16,16,12,13,14,14,15,13,13,15,15,16,14,14,14,15,16,15,15,16,16,17,15,15,16,16,17,13,13,13,15,15,14,14,15,15,16,13,14,14,15,16,15,15,16,16,17,15,16,15,17,16,14,15,15,16,16,15,15,16,16,17,15,16,16,17,17,16,16,17,16,18,16,17,17,17,17,15,15,15,16,16,15,16,16,17,17,15,16,16,17,16,16,17,17,18,18,16,17,16,17,16,11,12,12,15,13,13,13,13,15,15,12,14,13,16,14,14,15,15,16,16,14,15,14,17,15,13,13,13,15,14,13,14,14,16,15,14,14,14,16,15,15,15,16,16,17,15,16,15,17,16,12,14,13,15,14,14,14,14,16,15,13,14,13,16,15,15,16,16,17,16,15,16,15,17,16,15,15,15,16,16,15,15,16,16,17,15,16,16,17,17,16,16,17,17,17,17,17,17,18,17,14,15,15,16,16,15,16,16,17,16,15,16,15,17,16,17,17,17,18,17,16,17,16,18,16,6,9,9,12,12,8,10,10,12,13,8,10,10,13,12,10,12,12,14,15,11,13,12,15,14,8,9,10,12,13,9,10,11,13,14,10,11,11,14,13,12,12,13,14,15,12,13,13,15,15,8,10,10,13,13,10,11,11,13,14,10,12,10,14,13,12,13,13,15,15,12,14,13,15,14,11,12,12,13,14,12,12,13,13,15,12,13,13,15,15,14,13,15,14,16,14,15,15,16,16,12,13,13,14,14,13,13,14,15,14,12,14,13,15,14,14,15,15,16,15,14,15,14,16,14,7,9,10,12,12,9,10,11,13,14,9,11,10,13,13,11,12,13,14,15,12,13,13,15,14,9,10,11,12,13,10,10,12,13,14,11,11,12,14,14,12,12,14,14,15,13,13,14,15,15,9,11,11,13,13,11,12,12,14,14,10,12,10,14,13,13,14,14,15,15,13,14,13,16,14,12,12,13,14,15,13,13,14,14,16,13,14,14,15,15,14,14,15,14,17,14,15,15,16,16,12,13,13,15,14,13,14,14,15,15,13,14,13,16,14,15,15,15,16,16,14,15,14,16,14,7,10,9,13,12,10,11,12,12,14,10,12,11,14,12,12,13,13,14,15,12,14,13,15,14,9,11,10,13,13,10,11,12,13,14,12,13,12,15,13,13,13,14,13,15,13,14,14,16,15,10,11,11,13,13,12,12,13,14,14,11,12,11,14,13,14,14,14,15,16,13,14,13,16,13,12,13,13,14,14,12,13,13,14,15,14,14,14,15,15,14,13,15,13,16,15,15,15,17,16,13,13,13,14,14,14,14,14,15,15,12,13,13,15,14,15,16,16,16,16,14,15,14,16,13,11,12,13,14,15,12,13,14,15,16,13,14,14,15,15,14,14,15,15,17,14,15,15,16,16,13,13,14,14,15,13,13,15,14,16,14,14,15,15,16,15,14,16,15,17,15,16,16,16,17,13,14,14,15,15,14,14,15,16,16,13,15,14,16,16,15,16,16,17,17,15,16,15,17,16,14,15,15,15,17,15,15,16,15,17,15,16,16,16,17,16,16,17,16,18,17,17,17,17,18,15,15,15,17,16,15,16,16,17,17,15,16,16,17,16,16,17,17,18,18,16,17,16,18,17,11,13,12,15,14,13,13,14,15,15,13,14,13,16,14,15,15,15,16,16,15,16,15,17,16,13,14,13,15,14,13,13,14,15,15,14,15,14,16,15,15,15,16,16,16,15,16,15,18,16,13,14,14,15,15,14,15,15,15,16,13,15,13,16,15,15,16,16,17,17,15,16,15,17,16,15,15,15,16,16,15,15,15,16,17,16,16,16,17,16,16,16,17,16,17,17,17,17,18,17,15,15,15,16,16,16,16,16,17,17,15,16,15,17,16,17,17,17,18,18,16,17,16,17,15,6,9,9,12,12,8,10,10,12,13,8,10,10,13,12,11,12,13,14,15,10,12,12,14,14,9,10,10,13,13,10,10,12,13,14,10,11,11,14,13,12,13,14,14,15,12,13,13,15,15,8,10,9,13,12,10,11,11,13,14,9,11,10,14,13,12,13,13,15,15,12,13,12,15,14,12,13,13,14,14,12,13,13,14,15,13,14,14,14,15,14,14,15,14,16,14,15,15,16,16,11,12,12,14,13,13,13,13,15,15,12,13,12,15,13,14,15,15,16,16,14,15,14,16,14,7,9,10,12,13,10,10,12,12,14,10,12,11,14,13,12,13,14,14,15,12,13,13,15,14,10,11,11,13,13,11,11,12,13,14,12,13,12,14,14,13,13,14,13,16,14,14,14,15,15,9,10,11,13,14,12,12,13,13,15,10,12,10,14,13,13,14,14,15,16,13,14,13,15,13,13,14,13,14,15,12,13,13,14,15,14,14,14,15,15,14,13,15,13,16,15,16,16,16,16,12,13,13,14,14,14,14,14,15,15,12,13,13,15,14,15,15,16,16,16,14,15,13,16,13,7,10,9,12,12,9,10,11,13,13,9,11,10,14,13,12,13,13,14,15,11,13,12,15,14,9,11,11,13,13,10,10,12,13,14,11,12,12,14,14,13,13,14,14,16,13,14,14,16,15,9,11,10,13,12,11,12,11,14,14,10,12,10,14,13,13,14,13,15,15,12,14,12,16,14,12,13,13,14,15,13,13,14,14,16,13,14,14,15,15,14,14,15,14,16,15,15,15,16,16,12,13,12,15,14,13,14,14,15,15,12,14,13,16,14,14,15,15,16,16,14,15,14,17,14,11,12,13,14,15,13,13,14,14,16,13,14,13,15,15,15,15,16,16,17,15,15,15,16,16,13,14,13,15,15,13,13,15,15,16,14,15,15,16,16,15,15,16,15,17,16,16,16,17,17,13,13,14,14,15,14,14,15,15,16,13,14,13,15,15,15,16,16,16,17,15,16,15,16,16,15,15,15,16,16,15,15,16,16,17,16,16,16,17,17,16,16,17,16,18,17,17,17,18,18,15,15,15,16,16,16,16,16,17,17,15,15,15,16,16,17,17,17,17,18,16,16,16,17,15,11,13,12,15,14,13,13,14,15,15,12,14,13,16,14,14,15,15,16,16,14,15,14,16,15,13,14,14,15,15,13,14,14,16,16,14,15,14,16,16,15,15,16,17,17,15,16,16,17,17,13,14,13,15,14,14,14,14,16,15,13,15,13,16,14,15,16,15,17,16,15,16,14,17,15,14,16,15,16,17,15,16,16,16,17,15,16,16,17,17,16,16,17,17,18,16,17,17,18,17,14,15,15,17,15,15,16,16,17,16,15,16,15,17,15,16,17,17,18,17,16,17,16,18,15,10,12,12,14,14,12,13,13,15,15,12,13,13,15,15,13,14,14,15,16,14,15,14,16,16,12,13,13,15,15,12,13,14,15,15,13,14,14,15,15,14,14,15,16,17,14,15,15,17,16,12,13,13,15,15,13,14,14,15,16,13,14,14,16,15,14,15,15,16,17,14,15,15,17,16,13,14,14,15,16,14,14,15,15,16,14,15,15,16,16,15,15,16,16,17,15,16,16,17,17,14,15,15,16,16,15,15,15,16,16,15,15,15,16,16,16,17,16,17,17,16,16,16,18,16,11,12,12,14,14,12,13,14,15,15,12,13,13,15,15,13,14,15,16,16,14,15,15,16,16,12,13,13,15,15,13,13,14,15,16,13,14,14,15,16,14,14,15,16,17,15,15,15,16,17,12,13,13,15,15,13,14,14,15,16,13,14,14,16,15,15,15,15,16,17,15,16,15,17,16,14,14,15,15,16,14,14,15,15,17,15,15,16,16,17,15,15,16,15,18,16,16,16,17,17,14,15,15,16,16,15,16,16,17,17,15,15,15,17,16,16,17,16,17,17,16,16,16,18,16,11,12,12,14,14,13,13,14,15,15,13,14,13,15,15,14,15,15,16,16,14,15,15,16,16,12,13,13,15,15,13,13,14,15,15,14,14,14,16,15,15,15,15,15,16,15,16,15,17,16,12,13,13,15,15,14,14,15,15,16,13,14,13,16,15,15,15,16,16,17,15,16,15,17,15,14,15,14,16,16,14,15,15,16,16,15,16,15,17,16,15,15,16,15,17,16,17,16,17,17,14,15,15,16,16,15,16,16,16,17,14,15,15,16,16,16,17,17,17,18,16,16,16,17,16,12,13,13,15,15,13,13,14,15,16,13,14,14,16,15,14,15,15,16,17,14,15,15,17,16,13,14,14,15,16,14,14,15,15,17,14,15,15,16,16,15,14,16,15,17,15,16,16,17,17,13,14,14,16,16,14,15,15,16,16,14,15,14,16,16,15,16,16,17,17,15,16,15,17,16,15,15,16,15,17,15,15,16,15,17,15,16,16,16,17,16,15,17,15,18,17,17,17,17,17,15,15,15,17,17,16,16,16,17,17,15,16,15,17,17,16,17,17,18,18,16,17,15,18,15,11,12,12,15,15,13,13,15,14,16,13,14,13,16,14,15,15,16,16,17,15,16,15,17,15,12,14,13,16,14,13,13,14,14,16,14,15,14,16,15,15,15,16,15,17,16,16,16,17,16,12,13,14,15,16,15,15,15,15,16,13,15,13,16,14,16,16,16,17,17,15,16,15,17,15,15,16,15,16,15,14,14,15,16,16,16,16,16,17,16,15,15,16,15,17,17,17,17,18,17,15,15,15,16,16,16,16,16,16,17,14,15,15,17,16,17,17,17,17,18,15,16,15,18,14,10,12,12,14,14,12,13,13,15,15,12,13,13,15,15,14,14,15,15,16,13,15,14,16,16,12,13,13,15,15,13,14,14,15,16,13,14,14,15,15,14,15,15,16,17,14,15,15,17,16,12,13,13,15,15,13,14,14,15,15,12,14,13,15,15,14,15,15,16,17,14,15,14,17,15,14,15,15,16,16,14,15,15,16,17,15