import ItemBase from './item-base.js';
require('classlist-polyfill');
let media = require('./modal.js');
import TmUtils from '../tmUtils.js';
import ItemCommon from '../itemCommon.js';
import Tooltip from 'tooltip.js';

export default class Item {
    constructor(canvas, canvasWidth, canvasHeight, xPos, yPos, itemData, range) {
        // super(canvas);
        this.canvas = canvas;
        this.canvasWidth = canvasWidth;
        this.canvasHeight = canvasHeight;
        this.xPos = xPos;
        this.yPos = yPos;
        this.itemData = itemData;
        this.range = range;
        this.itemDiv = {};
        this.moreInfo = {}        
        this.init();
    }

    init() {
        let iconsDiv = document.createElement('div');
        iconsDiv.setAttribute('class', 'item-icons-container');
        
        // Info icon.
        if (this.itemData.hasOwnProperty('moreInfo')) {
            let infoIcon = document.createElement('span');
            infoIcon.setAttribute('class', `icon-info icon more-info-icon-${this.itemData.id}`);
            ItemCommon.addTabable(this.itemData, infoIcon);

            this.moreInfo = new Tooltip(infoIcon, {
                title: `${this.itemData.moreInfo.description}`,
                trigger: 'manual',
                placement: 'right',
                boundariesElement: document.getElementById('stretchy-outer'),
                html: true,
                container: this.canvas
            });

            infoIcon.addEventListener('click', (e) => {
                this.moreInfo.toggle();
            })

            infoIcon.addEventListener('keydown', (e) => {
                if (e.which === 13 || e.which === 32) {
                   this.moreInfo.toggle();
                }
            });

            iconsDiv.appendChild(infoIcon);
        }

        // Play, or media, icon.
        if (this.itemData.hasOwnProperty('media')) {
            let playIcon = document.createElement('span');
            playIcon.setAttribute('class', 'icon-play2 icon');
            playIcon.setAttribute('title', 'Play media');
            ItemCommon.addTabable(this.itemData, playIcon);

            playIcon.addEventListener('click', () => {
                media.show(this.itemData);
            });

            playIcon.addEventListener('keydown', (e) => {
                if (e.which === 13 || e.which === 32) {
                   media.show(this.itemData);
                }
            });

            iconsDiv.appendChild(playIcon);
        }

        // Olink.
        let olink = ItemCommon.createOlink(this.itemData)
        if (olink) {
            ItemCommon.addTabable(this.itemData, olink);
            iconsDiv.appendChild(olink);
        }
        
        // Progress check box.
        let checkBox = ItemCommon.createCheckbox(this.itemData)
        ItemCommon.addTabable(this.itemData, checkBox);
        iconsDiv.appendChild(checkBox);

        // Title.
        let label = document.createTextNode(this.itemData.title);
        let labelSpan = document.createElement('span');
        labelSpan.setAttribute('class', 'item-title-default');
        labelSpan.appendChild(label);
        
        this.itemDiv = document.createElement('div');
        this.itemDiv.setAttribute('class', 'item item-default');
        this.itemDiv.setAttribute('id', `item-${this.itemData.id}`);
        if (this.itemData.style.curColour !== undefined && this.itemData.style.curColour !== '') {
            this.itemDiv.classList.add(`item-default-${this.itemData.style.curColour}`);
        }

        this.itemDiv.addEventListener('click', (e) => {
            ItemCommon.increaseItemZindex(e.target);
        })

        // Create the y-marker.
        let marker = document.createElement('img');
        marker.setAttribute('src', 'src/images/y-marker.png');
        marker.setAttribute('class', 'y-marker');
        this.itemDiv.appendChild(marker);

        this.itemDiv.appendChild(labelSpan);
        this.itemDiv.appendChild(iconsDiv);
        this.canvas.appendChild(this.itemDiv);

        marker.style.top = (this.itemDiv.offsetHeight / 2) - 3 + 'px';
    }

    update(xPos, yPos) {
        this.xPos = xPos;
        this.yPos = yPos;

        // Calculate the x axis start and end position.
        let logStart = Math.log10(this.itemData.xStart || this.range.x.min),
            logEnd = Math.log10(this.itemData.xEnd),
            logMin = Math.log10(this.range.x.min),
            logRange = Math.log10(this.range.x.max) - logMin;
        let xStart = (logStart-logMin) / logRange * this.xPos;
        let xEnd = (logEnd - logMin) / logRange * this.xPos;

        this.itemDiv.style.left = xStart + 'px';
        // -1 for the right border width;
        this.itemDiv.style.width = (xEnd - xStart) - 1 + 'px';

        // Calculate the y axis start and end position.
        logStart = Math.log10(this.itemData.yStart || this.range.y.min);
        // logEnd = Math.log10(this.itemData.yEnd);
        logMin = Math.log10(this.range.y.min);
        logRange = Math.log10(this.range.y.max) - logMin;
        let yStart = this.canvasHeight - (logStart - logMin) / logRange * this.yPos;
        // Adjust for the marker.
        yStart -= (this.itemDiv.offsetHeight / 2) + 1;
        this.itemDiv.style.top = yStart + 'px';

        ItemCommon.setTabable(this.itemDiv, this.itemData.tabable);
    }

    show() {
        this.itemDiv.style.display = 'block';
    }

    hide() {
        this.itemDiv.style.display = 'none';
    }

    /**
     * Deletes this instance's div from the DOM.
     * TODO: Currently there is no way to add it back from within this instance.
     * @memberof Item
     */
    delete() {
        this.itemDiv.parentNode.removeChild(this.itemDiv);
    }

    /**
     * Returns the div width of this item.
     * 
     * @returns {Number} Width the div.
     * 
     * @memberof Item
     */
    getWidth() {
        return this.itemDiv.offsetWidth;
    }

    
    /** Hides the more info pop-up box.
     * 
     * @memberof ItemNarrow
     */
    hideInfo() {
        this.moreInfo.hide();
    }
}