import Item from './item.js';
import ItemNarrow from './item-narrow.js';
import ItemDragDrop from './item-activity.js';
import ItemLabel from './item-label.js';

/**
 * Handles the display and updating of items on the supplied container.
 * 
 * @export
 * @class ItemControl
 */
export default class ItemControl {
    constructor(config, itemData, saverInstance) {
        // Required - container must be set.
        // TODO: Add validation.
        this.container = config.container;

        // Required - itemData
        // TODO: validate!?
        this.itemData = itemData;

        // Optional - set the categories to display.
        if (config.displayCats !== undefined) {
        	this.setDisplayCategories(config.displayCats);
        } else {
        	this.displayCats = 'all';
        }

        this.canvasWidth = config.canvasWidth || '';
        this.canvasHeight = config.canvasHeight || '';
        this.xPos = config.xPos || '';
        this.yPos = config.yPos || '';
        this.range = config.range || '';

        this.saver = saverInstance;
    }

    /**
     * Displays all the items.
     * 
     * @memberof ItemControl
     */
    display(xPos, yPos) {
        this.xPos = xPos;
        this.yPos = yPos;
        for (let i = 0; i < this.itemData.length; i++) {
            // Check for a reset external to this class.
            if (this.itemData[i].hasOwnProperty('reset')) {
                this.resetItem(i);
            } else {
                if (this.itemData[i].hasOwnProperty('instance')) {
                    this.checkWidths(i);
                }
            }
            // Create the item instance if it doesn't exist.
            if (this.itemData[i].instance === undefined) {
                switch (this.itemData[i].style.curTemplate) {
                    case 'default':
                        this.itemData[i].instance = new Item(this.container, this.canvasWidth, this.canvasHeight, this.xPos, this.yPos, this.itemData[i], this.range);
                        break;
                    case 'activity':
                        this.itemData[i].instance = new ItemDragDrop(this.container, this.canvasWidth, this.canvasHeight, this.xPos, this.yPos, this.itemData[i], this.range);
                        break;
                    case 'item-narrow':
                        this.itemData[i].instance = new ItemNarrow(this.container, this.canvasWidth, this.canvasHeight, this.xPos, this.yPos, this.itemData[i], this.range);
                        break;
                    case 'label':
                        this.itemData[i].instance = new ItemLabel(this.container, this.canvasWidth, this.canvasHeight, this.xPos, this.yPos, this.itemData[i], this.range);
                        break;
                    default:
                        this.itemData[i].instance = new Item(this.container, this.canvasWidth, this.canvasHeight, this.xPos, this.yPos, this.itemData[i], this.range);
                        break;
                }
            }

            if (this.displayCats.indexOf(this.itemData[i].category) !== -1) {
                this.itemData[i].instance.show();
            } else {
                this.itemData[i].instance.hide();
            }

            this.itemData[i].instance.update(this.xPos, this.yPos);
        }
    }

    resetItem(i) {
        // console.log('resetItem()');
        // TODO: Should all this be done in the item class?

        // Close the info pop-up before deleting the instance.
        if (typeof this.itemData[i].instance.hideInfo == 'function') {
            this.itemData[i].instance.hideInfo();
        }

        this.itemData[i].instance.delete();
        delete this.itemData[i].reset;
        delete this.itemData[i].instance;
    }

    /**
     * Sets the categories that will be displayed. 
     * 
     * @param {Array} categories 
     * @returns 
     * 
     * @memberof ItemControl
     */
    setDisplayCategories(categories) {
        if (categories.constructor !== Array) {
            // console.warn('config.displayCats is set but is not an array. All categories will be displayed.')
            return;
        } else {
        	this.displayCats = categories;
        }
    }

    checkWidths(idx) {
        // Check if there are any width break points set.
        for (let [i, width] of this.itemData[idx].style.width.breakPoints.entries()) {

            // Found a default style, no need to go any further.
            if (this.itemData[idx].style.width.breakPoints[i] === 'default') {
                this.setCurrentStyle(i, idx);
                return;
            }

            // Is the visual width less than a break point? If so, do something.
            if (this.itemData[idx].instance.getWidth() <= this.itemData[idx].style.width.breakPoints[i]) {
                this.setCurrentStyle(i, idx);
                return;
            }
        }

        // None of the width break points apply. Check if anything has changed so as to not update unnecessarily.
        if (this.itemData[idx].style.curColour !== this.itemData[idx].style.defColour || 
                    this.itemData[idx].style.curTemplate !== this.itemData[idx].style.defTemplate) {
            this.itemData[idx].style.curTemplate = this.itemData[idx].style.defTemplate;
            this.itemData[idx].style.curColour = this.itemData[idx].style.defColour;
            this.resetItem(idx);
        }
    }

    /**
     *Sets the current style from an item's colour and style data.
     * TODO: This came before setStyle(), could setStyle() be used instead?
     * 
     * @param {any} i 
     * @param {any} idx 
     * @memberof ItemControl
     */
    setCurrentStyle(i, idx) {
        let reset = false;

        // TODO: Not sure why this code change doesn't work at the moment. The intention was to not change style unnecessarily,
        // got something to do with undefined. Will review this later.
        // if (this.itemData[idx].style.curTemplate != this.itemData[idx].style.width.templates[i]) {
        //     // console.log('-------------', this.itemData[idx].title);
        //     // console.log('different template', this.itemData[idx].style.curTemplate, ' - ', this.itemData[idx].style.width.templates[i]);
        //     // this.itemData[idx].style.curTemplate = this.itemData[idx].style.width.templates[i];
        //     reset = true;
        // }
        // if (this.itemData[idx].style.curColour != this.itemData[idx].style.width.colours[i]) {
        //     // console.log('different colour', this.itemData[idx].style.curColour, ' - ', this.itemData[idx].style.width.colours[i]);
        //     // this.itemData[idx].style.curColour = this.itemData[idx].style.width.colours[i];
        //     reset = true;
        // }

        // // Only need to reset if something has changed.
        // if (reset) {
        //     this.itemData[idx].style.curTemplate = this.itemData[idx].style.width.templates[i];
        //     this.itemData[idx].style.curColour = this.itemData[idx].style.width.colours[i];
        //     this.resetItem(idx);
        // }

        this.itemData[idx].style.curTemplate = this.itemData[idx].style.width.templates[i];
        this.itemData[idx].style.curColour = this.itemData[idx].style.width.colours[i];
        this.resetItem(idx);
    }

    /**
     * Set the style (template and colour) for a particular item id.
     * 
     * @param {Number} idx - Id of the item to be updated.
     * @param {String} style - An itemStyles object literal name.
     * 
     * @memberof ItemControl
     */
    setStyle(id, style) {
        if (itemStyles.hasOwnProperty(style)) {
            this.itemData[id].style = itemStyles[style];
            this.resetItem(id);
            // Refresh the display.
            this.display(this.xPos, this.yPos);
        }
    }
}