export default class Scale {
    constructor(config) {
        this.container = config.container;
        this.vertical = config.vertical || false;
        this.rtl = config.rightToLeft || false;
        this.min = config.minVal;
        this.max = config.maxVal;
        this.skipLabels = config.skipLabels || false;

        // Set the right classes.
        if (!this.vertical) {
            this.tickClass = config.tickClass || 'tick';
            this.tick5Class = config.tick5Class || 'tick-5';
            this.tickLabel = config.tickLabel || 'tick-label';
        } else {
            this.tickClass = config.tickClass || 'tick-v';
            this.tick5Class = config.tick5Class || 'tick-5-v';
            this.tickLabel = config.tickLabel || 'tick-label-v';
        }

        this.showTickLabel = config.showTickLabel || true;
    }

    setMin(value) {
        this.min = value;
    }

    setMax(value) {
        this.max = value;
    }

    /**
     * Top level method to draw a scale.
     * 
     * @memberof Scale
     */
    drawScale() {
        let min = this.min;
        let max = this.max;
        let firstTic = min;
        let logMin = Math.log10(min);
        let range = max - min;
        let pos = 0;
        let allTics = [];
        let tickValue = 0;
        let tickSpacing = 0;
        
        // Get the length, or height, of the div for the scale.
        if (!this.vertical) {
            this.scaleLength = this.container.offsetWidth;
        } else {
            this.scaleLength = this.container.offsetHeight;
        }

        // Empty the scale container - clear all the ticks.
        while (this.container.childNodes[0]) {
            this.container.removeChild(this.container.childNodes[0]);
        }

        // Calculate the value of the first tick if it doesn't fall on 1Ex, i.e. if it's 2Ex.
        if (logMin < Math.ceil(logMin)) {
            firstTic = Math.pow(10, Math.ceil(logMin));
        };

        // If skipping labels get pixel spacing of first two ticks to determine if they are close enough to skip a label.
        if (this.skipLabels) {
            let firstTicPos = this.scalePosition(0, this.scaleLength, min, max, firstTic);
            let secondTicPos = this.scalePosition(0, this.scaleLength, min, max, firstTic * 10);
            tickSpacing = secondTicPos - firstTicPos;
        }
            
        // Draw the ticks.    
        for (let i = firstTic; i <= max; i *= 10) {
            pos = this.scalePosition(0, this.scaleLength, min, max, i);
            allTics.push(pos);
            this.drawTick(pos);

            tickValue = Math.floor(Math.log10(i));
            if (this.showTickLabel) {
                if (this.skipLabels) {
                    if (tickSpacing > this.skipLabels.lessThan) {
                        this.drawTickLabel(pos, tickValue);
                    } else if (tickSpacing < this.skipLabels.lessThan && tickValue % this.skipLabels.num === 0) {
                        this.drawTickLabel(pos, tickValue);
                    }
                } else {
                    this.drawTickLabel(pos, tickValue);
                }
            }
        }

        let extraLabelsX = [
            {   
                value: 0.000000001,
                row: 2,
                text: '1 nanosecond'
            },
            {   
                value: 0.000001,
                row: 3,
                text: '1 microsecond'
            },
            {   
                value: 0.001,
                row: 2,
                text: '1 millisecond'
            },
            {
                value: 1,
                row: 3,
                text: '1 second'
            },
            {
                value: 60,
                row: 2,
                text: '1 minute'
            },
            {
                value: 3600,
                row: 3,
                text: '1 hour'
            },
            {
                value: 31536000,
                row: 2,
                text: '1 year'
            },
            {
                value: 3153600000,
                row: 3,
                text: '100 years'
            },
            {
                value: 3153600000000,
                row: 2,
                text: '1 million years'
            },
            {
                value: 31536000000000000,
                row: 3,
                text: '1 billion years'
            }
        ];

        if (!this.vertical) {
            for (let label of extraLabelsX) {
                pos = this.scalePosition(0, this.scaleLength, min, max, label.value);
                this.drawTickPlus(pos, label.row);
                this.drawTickLabelPlus(pos, label.row, label.text);
            }


            // draw a tickPlus.
            // pos = this.scalePosition(0, this.scaleLength, min, max, 0.001)
            // this.drawTickPlus(pos);
            // this.drawTickLabelPlus(pos, '1 millisecond');

            // pos = this.scalePosition(0, this.scaleLength, min, max, 1)
            // this.drawTickPlus(pos);
            // this.drawTickLabelPlus(pos, '1 second');

            // pos = this.scalePosition(0, this.scaleLength, min, max, 3600)
            // this.drawTickPlus(pos);
            // this.drawTickLabelPlus(pos, '1 hour');

            // pos = this.scalePosition(0, this.scaleLength, min, max, 31536000)
            // this.drawTickPlus(pos);
            // this.drawTickLabelPlus(pos, '1 year');
            
            // pos = this.scalePosition(0, this.scaleLength, min, max, 3153600000)
            // this.drawTickPlus(pos);
            // this.drawTickLabelPlus(pos, '100 years');

            // pos = this.scalePosition(0, this.scaleLength, min, max, 31536000000000)
            // this.drawTickPlus(pos);
            // this.drawTickLabelPlus(pos, '1 million years');
        }

        return allTics;
    }

    /**
     * Draws a tick mark on the scale.
     * 
     * @param {any} pos 
     * @param {any} tickValue 
     * 
     * @memberof Scale
     */
    drawTick(pos) {
        let tick = document.createElement('div');
        tick.setAttribute('class', this.tickClass);

        if (!this.vertical) {
            tick.style.left = pos + 'px';
        } else {
            tick.style.top = Number(this.scaleLength - pos) + 'px';
        }

        this.container.appendChild(tick);
    }

    drawTickLabel(pos, tickValue) {
        let tickLabelDiv = document.createElement('div');
        tickLabelDiv.setAttribute('class', this.tickLabel);
        tickLabelDiv.innerHTML = `10<sup>${tickValue}</sup>`;
        this.container.appendChild(tickLabelDiv);
        if (!this.vertical) {
            tickLabelDiv.style.left = pos - tickLabelDiv.offsetWidth / 2 + 'px';
        } else {
            tickLabelDiv.style.top = Number(this.scaleLength - pos - (tickLabelDiv.offsetHeight / 2)) + 'px';
        }
    }

    drawTickPlus(pos, row) {
        let tick = document.createElement('div');
        tick.setAttribute('class', `${this.tickClass}-row-${row}`);

        if (!this.vertical) {
            tick.style.left = pos + 'px';
        } else {
            tick.style.top = Number(this.scaleLength - pos) + 'px';
        }

        this.container.appendChild(tick);
    }

    drawTickLabelPlus(pos, row, text) {
        let tickLabelDiv = document.createElement('div');
        tickLabelDiv.setAttribute('class', `${this.tickLabel}-row-${row}`);
        tickLabelDiv.innerHTML = text;
        this.container.appendChild(tickLabelDiv);
        if (!this.vertical) {
            tickLabelDiv.style.left = pos - tickLabelDiv.offsetWidth / 2 + 'px';
        } else {
            tickLabelDiv.style.top = Number(this.scaleLength - pos - (tickLabelDiv.offsetHeight / 2)) + 'px';
        }
    }

    scalePosition(minp, maxp, minv, maxv, value) {
        // position will be between 0 and 100
        // var minp = 0;
        // var maxp = 100;

        // The result should be between min and max values.
        var minv = Math.log(minv);
        var maxv = Math.log(maxv);

        // calculate adjustment factor
        var scale = (maxv-minv) / (maxp-minp);

        return (Math.log(value)-minv) / scale + minp;
    }

    /**
     * Calculates natural log tick points.
     * 
     * @param {any} mn 
     * @param {any} mx 
     * @returns 
     * 
     * @memberof Scale
     */
    calculateTicks(mn, mx,) {
      
        return {
            min: mn,
            max: mx,
            range: mx - mn,
            ticFirst: ticFirst,
            tic: d,
            tic5: d5
        }
    }
};



// Math.log10() polyfil for older browsers.
Math.log10 = Math.log10 || function(x) {
  return Math.floor(Math.log(x) * Math.LOG10E);
};