/**
 * @fileOverview OU Core defines the OU namespace and some core/misc functionality
 *
 * OU should be the only global variable used
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @namespace OU namespace, holds the entire framework together in a single entity
 */
var OU = new function() {
    this.obj = {}; // required for some activities that need to call themselves from the page, ie in drop down selectors
    this.head = document.getElementsByTagName('head')[0];
    this.requiredFiles = 0;
    this.__mini = false;
    this.fileCount = 0;
    this._buttonCount = 0;
    this.IS_IPAD = navigator.userAgent.match(/iPad/i);
    this.IS_IPHONE = ((navigator.userAgent.match(/iPhone/i)) || (navigator.userAgent.match(/iPod/i)));
    /** @namespace Utilities namespace*/
    this.util = function() {
    };
    /** @namespace Activities namespace*/
    this.activity = function() {
    };
    this.menus = [];
    this.controllers = [];
    this._tabbables = [];
    this.FileLoaded = {};
    this.preOnLoad = [];
    this.preLoadImg = [];
    this._debug = /debug/.test(window.location);

    //determine library location
    var loadScript = document.getElementsByTagName('script')[0];
    this.libraryPath = loadScript.src.substr(0, loadScript.src.indexOf('js/ou.js'));
    this.jsPath = this.libraryPath + 'js/';
    this.cssPath = this.libraryPath + 'css/';

    this.loadingTicks = 0;

    // Set Device Pixel Ratio value to use...
    // On an iPad 3, the device pixel ratio is 2, because it renders content with resolution 1024x768, onto a 2048x1436 screen.
    // We use a viewport scale of 0.505 to force the browser to give us the full screen resolution
    // Note: we do not use 0.5 as you would expect, because this makes the iPad 3 do a lot of additional calculations, which make it slow and it crashes
    this.dpr = 1; // Set DPR to the normal value of 1
    if (false && window.devicePixelRatio === 2 && window === window.top) { // Note, we only use DPR of 2 if we are not in an iFrame, because the viewport scale is ignored when in iframe
        var viewportmeta = document.querySelector('meta[name="viewport"]');
        if (viewportmeta) {
            viewportmeta.content = 'width=device-width, minimum-scale=0.505, maximum-scale=0.505, initial-scale=0.505';
            this.dpr = 2;
        }
    }
    /*
     this.dpr = window.devicePixelRatio || 1; // Set DPR to the normal value of 1
     if (this.dpr !== 1) {
     var vpzoom = (1 / this.dpr) + 0.05,
     viewportmeta = document.querySelector('meta[name="viewport"]');
     if (viewportmeta) {
     viewportmeta.content = 'width=device-width, minimum-scale=' + vpzoom + ', maximum-scale=' + vpzoom + ', initial-scale=' + vpzoom;
     }
     }
     //*/
    this.controlHeight = 40 * this.dpr; // set the controlHeight according to the DPR
    //*/
    return this;
};
/* Define some constants
 */
OU._enableResize = true;
/* Activity states */
OU.LOADING = 0;
OU.PAUSED = 1;
OU.RUNNING = 2;
/* Set up Layer zIndex constants
 * Each activity has several layers - these constants help define the zIndex for each layer.
 * In addition each Activity has an activity zIndex Offset - so that activities as a whole can be layered
 */
OU.DATA_LEVEL = 20;
OU.CONTROL_LEVEL = 50;
OU.CONTROLLER_LEVEL = 90;
OU.OVERLAY_CONTROLLER_LEVEL = 8000;
OU.POP_UP_LEVEL = 9000;
OU.CLOSE_LEVEL = 9500;
OU.DEBUG_LEVEL = 10000;
OU.ABSOLUTE_TOP = 50000;
/**
 * Inherit the prototype methods from one constructor into another.
 *
 * Usage:
 * <pre>
 * function ParentClass(a, b) { }
 * ParentClass.prototype.foo = function(a) { }
 *
 * function ChildClass(a, b, c) {
 *   goog.base(this, a, b);
 * }
 * goog.inherits(ChildClass, ParentClass);
 *
 * var child = new ChildClass('a', 'b', 'see');
 * child.foo(); // works
 * </pre>
 *
 * In addition, a superclass' implementation of a method can be invoked
 * as follows:
 *
 * <pre>
 * ChildClass.prototype.foo = function(a) {
 *   ChildClass.superClass_.foo.call(this, a);
 *   // other code
 * };
 * </pre>
 *
 * @param {Function} childCtor Child class.
 * @param {Function} parentCtor Parent class.
 */
OU.inherits = function(childCtor, parentCtor) {
    var tempCtor = function() {
    };
    tempCtor.prototype = parentCtor.prototype;
    childCtor.superClass_ = parentCtor.prototype;
    childCtor.prototype = new tempCtor();
    childCtor.prototype.constructor = childCtor;
};
/**
 * Call up to the superclass.
 *
 * If this is called from a constructor, then this calls the superclass
 * contructor with arguments 1-N.
 *
 * If this is called from a prototype method, then you must pass
 * the name of the method as the second argument to this function. If
 * you do not, you will get a runtime error. This calls the superclass'
 * method with arguments 2-N.
 *
 * This function only works if you use goog.inherits to express
 * inheritance relationships between your classes.
 *
 * This function is a compiler primitive. At compile-time, the
 * compiler will do macro expansion to remove a lot of
 * the extra overhead that this function introduces. The compiler
 * will also enforce a lot of the assumptions that this function
 * makes, and treat it as a compiler error if you break them.
 *
 * @param {!Object} me Should always be "this".
 * @param {*=} opt_methodName The method name if calling a super method.
 * @param {...*} var_args The rest of the arguments.
 * @return {*} The return value of the superclass method.
 */
OU.base = function(me, opt_methodName, var_args) {
    var caller = arguments.callee.caller;
    if (caller.superClass_) {
        // This is a constructor. Call the superclass constructor.
        return caller.superClass_.constructor.apply(
                me, Array.prototype.slice.call(arguments, 1));
    }
    var args = Array.prototype.slice.call(arguments, 2);
    var foundCaller = false;
    for (var ctor = me.constructor;
            ctor; ctor = ctor.superClass_ && ctor.superClass_.constructor) {
        if (ctor.prototype[opt_methodName] === caller) {
            foundCaller = true;
        }
        else if (foundCaller) {
            return ctor.prototype[opt_methodName].apply(me, args);
        }
    }
    // If we did not find the caller in the prototype chain,
    // then one of two things happened:
    // 1) The caller is an instance method.
    // 2) This method was not called by the right caller.
    if (me[opt_methodName] === caller) {
        return me.constructor.prototype[opt_methodName].apply(me, args);
    }
    else {
        throw Error('OU.base called from a method of one name (' + me[opt_methodName] + ') to a method of a different name:' + caller);
    }
};
// Fake console support if the browser is rubbish (IE)
if (typeof console === 'undefined') {
    console = {
        log: function() {
        },
        error: function() {
        }
    };
}
/**
 * Log - outputs a message to a debug window or the console log
 *
 * To switch on the debug window(Div) you can add debug into the URL (ie. index.html?debug=true )
 * or for testing in eBooks, manually override the state of OU._debug at line 22 of this file
 *
 * @param {String} s The message to output
 */
OU.log = function(s) {
    if (this._debug) {
        if (this.debugDiv) {
            if (this._cachedDebug) {
                s = this._cachedDebug + s + '<br/>';
                this._cachedDebug = null;
            }
            this.debugDiv.innerHTML += s + '<br/>';
        }
        else {
            if (this._cachedDebug)
                this._cachedDebug += s + '<br/>';
            else
                this._cachedDebug = s + '<br/>';
        }
    }
    else {
        console.log(s);
    }
};
/**
 * OU.require - provides a dependancy mechanism
 *
 * @param {String} obj name of the javascript function(class) required
 */
OU.require = function(obj) {
    var fn = obj.toLowerCase().replace(/\./g, '/'),
            src = fn.replace(/^ou\//, this.jsPath) + ".js";
    if (this.__mini && obj.match(/^OU\.util/))
        return;
    if (!this.FileLoaded[fn]) {
        this.FileLoaded[fn] = true;
        this.loadScript(src);
    }
};
OU.loadScript = function(src) {
    this.requiredFiles++;
    var inc = document.createElement('script');
    inc.src = src;
    inc.type = 'text/javascript';
    if (document.all) { // Fix for IE's lack of 'onload' support
        inc.onreadystatechange = function() {
            if (inc.readyState === 'complete' || inc.readyState === 'loaded') {
                OU.fileCount++;
            }
        };
    }
    else {
        inc.onload = function() {
            OU.fileCount++;
        };
    }
    this.head.appendChild(inc);
};
/** Inserts a stylesheet link into the web page
 *
 * @param - {String} relativeURL path of the CSS file relative to this web page
 */
OU.loadCSS = function(relativeURL) {
    var l = document.createElement('link');
    l.href = relativeURL;
    l.rel = "stylesheet";
    l.type = "text/css";
    this.head.appendChild(l);
};
/**
 * Adds a title to the page
 */
OU.loadTitle = function(title) {
    var t = document.createElement('title');
    t.innerHTML = title;
    this.head.appendChild(t);
};

/**
 *  Loads a theme, as defined in the data.js file (or 'default' if undefined)
 *
 *  Adds 2 files to the head of the page:
 *  <ul>
 *  <li>A CSS file from /css/themes/
 *  <li>A Javascript file from /js/themes/
 *  </ul>
 */
OU.loadTheme = function(dataTheme) {
    var theme = dataTheme || 'default', cssTheme,
            inc = document.createElement('script');
    cssTheme = this.cssPath + "themes/" + theme + ".css";
    this.loadCSS(cssTheme);
    OU.requiredFiles++;
    inc.src = this.jsPath + "themes/" + theme + ".js";
    inc.type = 'text/javascript';
    if (document.all) { // Fix for IE's lack of 'onload' support
        inc.onreadystatechange = function() {
            if (inc.readyState === 'complete' || inc.readyState === 'loaded') {
                OU.fileCount++;
            }
        };
    }
    else {
        inc.onload = function() {
            OU.fileCount++;
        };
    }
    this.head.appendChild(inc);

    /* Now load any extra style specified in the data - not currently used
     if(data.extraStyle) {
     extraStyle=document.createElement('script');
     OU.requiredFiles++;
     extraStyle.src =  "data/" + data.extraStyle;
     extraStyle.type = 'text/javascript';
     extraStyle.onload = function () {
     OU.fileCount++;
     };
     if (document.all) { // Fix for IE's lack of 'onload' support
     extraStyle.onreadystatechange = function () {
     if (extraStyle.readyState=='complete' || extraStyle.readyState=='loaded') {
     OU.fileCount++;
     }
     }
     }
     this.head.appendChild(extraStyle);
     }
     /*/
};
/**
 * loadFonts - if data.font is defined, then insert the relevant code in the page head
 */
OU.loadFonts = function(fonts) {
    var n, cssBuilder = "", cssBuilderB = "";
    if (!fonts)
        return;
    OU.fontsDone = 0;
    for (var i = 0; i < fonts.length; i++) {
        n = fonts[i].name;
        //{name:'economica',url:'data/Economica-Regular.ttf'}
        cssBuilder = cssBuilder + "@font-face {font-family:" + n + "; src: url( " + fonts[i].url + " ) format('truetype');}\n";
        cssBuilderB = cssBuilderB + "." + n + "{font-family:'" + n + "';}\n";
    }
    var st = document.createElement('style');
    st.innerHTML = cssBuilder + cssBuilderB;
    st.type = 'text/css';
    st.async = 'true';
    var s = document.getElementsByTagName('style')[0] ? document.getElementsByTagName('style')[0] : document.getElementsByTagName('script')[0];
    s.parentNode.insertBefore(st, s);
};

OU.waitFonts = function() {
    OU.fontsDone = 1;
};

/**
 * Used in conjunction with OU.require - implements a loop which waits until all the required files are loaded and then executes a given function
 *
 * @requires OU.require
 * @param {function} fn - the function to call when all files are loaded
 */
OU.onLoad = function(fn) {
    if (fn) {
        this.onLoadFn = fn;
        OU.loadingTicks = 0;
    }
    else {
        fn = this.onLoadFn;
        OU.loadingTicks++;
    }
    if (this.fileCount >= this.requiredFiles || this.loadingTicks > 100) {
        if (this.loadingTicks > 100) {
            OU.log('Problem loading all required files - loaded ' + this.fileCount + ' of ' + this.requiredFiles);
            for (var fnInc in this.FileLoaded) {
                if (!this.FileLoaded[fnInc]) {
                    console.log('Not loaded: ' + fnInc);
                }
                else {
                    console.log('Loaded: ' + fnInc);
                }
            }
        }
        for (var i = this.preOnLoad.length; i--; )
            this.preOnLoad[i]();
        fn();
    }
    else {
        setTimeout('OU.onLoad();', 40);
    }
};
/** Sets up the HTML page ready for use - n other words removes it's current contents and scrolls to the top of the page
 *
 *  In addition, this function checks for DEBUG state and if valid creates the debug 'console'(DIV)
 */
OU.initPage = function() {
    if (!OU.__pageInitDone) {
        OU.__pageInitDone = true;
        document.body.innerHTML = '';
        window.scrollTo(0, 0);
        OU._closeBDone = false;
        if (this._debug) {
            var d = this.debugDiv = document.createElement('div');
            d.style.zIndex = '500000';
            d.style.background = '#fff';
            d.style.position = 'absolute';
            d.style.border = '1px solid #c00';
            d.style.padding = '10px';
            d.height = 500;
            d.style.overflow = 'auto';
            document.body.appendChild(d);
        }
        OU.ensureScreenWrapped();
    }
};
/** Removes an activity that has been loaded via a controller.
 *  Including any sub elements that have the same zOffset
 *
 *  @param {int} zIndex - zIndex to remove
 *  @param {String} controllerInstance - reference to the relevant controller
 */
OU.removeActivity = function(zIndex, controllerInstance) { // removes an activity and all "removables" with the same zOffset
    var i, c = OU.controllers[controllerInstance], s = c.section;
    OU.clean(zIndex);
    for (i = s.activityObjects.length; i--; ) {
        if (s.activityObjects[i] && s.activityObjects[i]._zOffset === zIndex) {
            if (s.activityObjects[i].remove)
                s.activityObjects[i].remove();
            delete s.activityObjects[i];
            s.activityObjects[i] = null;
        }
    }
    for (i = s.activities.length; i--; ) {
        if (s.activities[i]._zOffset === zIndex)
            s.activities.splice(i, 1);
    }
};
/** Removes all objects that been pushed into the removables array that have the given zOffset
 *
 * @param {int} zOffset
 */
OU.clean = function(zOffset) {
    var i, r;
    for (i = OU.removables.length; i--; ) {
        r = OU.removables[i];
        if (r && (zOffset === undefined || r._clicktype === zOffset)) {
            r.remove();
            delete r;
            OU.removables[i] = null;
        }
    }
};
/**
 * Adds an object to the removables array and adds some extra house keeping variables for use when removing
 * @param {object} removableObject - the object to be added
 * @param {string} type - a 'type' identifier
 */
OU.addRemovable = function(removableObject, type) {
    var controller, instanceName;
    if (removableObject) {
        if (removableObject.controller) {
            controller = removableObject.controller;
        }
        else if (removableObject.container && removableObject.container.controller) {
            controller = removableObject.container.controller;
        }
        instanceName = controller ? controller.instance : 'nocontroller';
        removableObject._controllerInstance = instanceName;
        OU.removables.push(removableObject, type);
    }
};

/** Removes objects from the removables array that are "under" the given controller
 *
 * @param {object} controller
 */
OU.cleanSection = function(controller) {
    var i, r;
    for (i = OU.removables.length; i--; ) {
        r = OU.removables[i];
        if (r) { // may have already been removed
            if (r._controllerInstance === controller.instance) {
                if (OU.obj[r.instance])
                    OU.obj[r.instance] = null; // remove activities from object array
                r.remove();
                delete r;
                OU.removables[i] = null;
            }
        }
    }
};
/**
 * Provides a globally accessible way to step to a specific menu item's action
 */
OU.stepMenu = function(menuId, levelId, buttonId) {
    OU.menus[menuId].step(levelId, buttonId);
    return false;
};
/**
 * Stores details of an override section (overriding a section removes all activities and inserts new ones into a controller section)
 *
 * @param {Controller section definition JSON} s defines the new controller Section definition to recall later
 * @returns array index number of the new entry
 */
OU.storeOverride = function(s) {
    if (OU._overrideArray === undefined)
        OU._overrideArray = [];
    OU._overrideArray.push(s);
    return OU._overrideArray.length - 1;
};
/**
 * Replaces an existing activity with one that has been previously stored in the overrideArray using OU.storeOverride
 *
 * @param {int} n - index of the new activity in the override array
 * @param {int} oldZ - zIndex of the activity to remove
 * @param {controller Object} controller - reference to the controller this is happening under
 */
OU.replaceActivity = function(n, oldZ, controller) {
    OU.removeActivity(oldZ, controller.controllerId);
    var a = OU._overrideArray[n];
    if (a !== undefined)
        controller.addActivity(a, true);
    return false;
};
/**
 * A function to stop default event handlers from triggering, used in OU.nobbleDoneBar
 *
 * @param {Event} ev - the event details
 */
OU.preventDefault = function(ev) {
    ev.preventDefault();
};
/**
 * Captures touch events for a given object and prevents the event from being bubbled up to the 'browser / rendering engine'
 *
 * This is specifically used to stop the "Done" bar from appearing in an Apple iBooks - ePub format document media overlay.
 *
 * @param {Object} o - the DOM object to nobble, ie a DIV,CANVAS, etc.
 */
OU.nobbleDoneBar = function(o) {
    o.addEventListener("touchstart", OU.preventDefault, false);
    o.addEventListener("touchend", OU.preventDefault, false);
    o.addEventListener("touchmove", OU.preventDefault, false);
};
/**
 * @class Typed Array -  array class - inherits Array prototype
 *
 * Holds an array of objects - with the addition of type (id)
 * - array can then be "cleaned" of a specific type
 *
 * @augments Array
 */
OU.util.TypedArray = function() {
    Array.call(this);
    this.prototype = new Array();
    this.base = Array.prototype;
};
OU.util.TypedArray.prototype.push = function(o, id) {
    if (id !== undefined)
        o._clicktype = id;
    this.base.push.call(this, o);
};
OU.util.TypedArray.prototype.removeType = function(id) { // removes objects from the array that have a specific 'id'
    var i;
    for (i = this.length; i--; ) {
        if (this[i] && this[i]._clicktype === id) {
            this[i] = null;
            // this.base.splice.call(this, i, 1);
        }
    }
};
OU.util.TypedArray.prototype.removeIdx = function(i) {
    this[i] = null;
    this.base.splice.call(this, i, 1);
};
OU.removables = new OU.util.TypedArray();
OU.disableResize = function() {
    OU._enableResize = false;
};
OU.enableResize = function() {
    OU._enableResize = true;
};
/**
 * A function to monitor the outer frame of our main web page
 * On a change of size, the callback function 'fn' is called
 *
 * Note: we use this loop, because the default window.onorientationchange event is suppressed on some mobile platforms
 *
 * This function only executes once, as we only want one loop
 *
 * @param {function} fn - callback function to execute when a resize happens
 */
OU.onOrientChange = function(fn, force) {
    if (OU._OCHandlerSet || force !== true)
        return;
    OU._OCHandlerSet = true;
    OU._oldInnerWidth = window.innerWidth;
    OU._oldOuterWidth = window.outerWidth;
    OU._oldInnerHeight = window.innerHeight;
    OU._oldOuterHeight = window.outerHeight;
    if (OU._onOChangeInterval) {
        window.clearInterval(OU._onOChangeInterval);
    }
    OU._onOChangeInterval = window.setInterval(function() {
        try {
            if (OU._enableResize &&
                    (OU._oldInnerWidth !== window.innerWidth
                            || OU._oldOuterWidth !== window.outerWidth
                            || OU._oldInnerHeight !== window.innerHeight
                            || OU._oldOuterHeight !== window.outerHeight)) {
                OU._closeButtonDone = false;
                fn();
            }
            OU._oldInnerWidth = window.innerWidth;
            OU._oldOuterWidth = window.outerWidth;
            OU._oldInnerHeight = window.innerHeight;
            OU._oldOuterHeight = window.outerHeight;
        }
        catch (e) {
            if (OU._enableResize &&
                    (OU._oldInnerWidth !== window.innerWidth
                            || OU._oldInnerHeight !== window.innerHeight)) {
                OU._closeButtonDone = false;
                fn();
            }
            OU._oldInnerWidth = window.innerWidth;
            OU._oldInnerHeight = window.innerHeight;
        }
    }, 40);
    window.onorientationchange = fn;
};
/**
 * Determines whether html5 local storage is available
 */
OU.supportsLocalStorage = function() {
    try {
        if (!localStorage) {
            return false;
        }
        // Actually attempt to use storage <= Private browsing in Safari has broken localStorage
        localStorage['___safariTest'] = 'AppleFail';
        return true;
    }
    catch (e) {
        return false;
    }
};
OU._localStorage = OU.supportsLocalStorage();
if (OU._localStorage) {
    OU.LocalStorage = {
        save: function(i, v) {
            localStorage[i] = v;
        },
        load: function(i) {
            return localStorage[i];
        }
    };
}
else {
    OU.LocalStorage = {
        save: function(i, v) {
            document.cookie = (i) + "=" + encodeURIComponent(v);
        },
        load: function(i) {
            var s = "; " + document.cookie + ";",
                    p = s.indexOf("; " + i + "=");
            if (p < 0)
                return "";
            p = p + i.length + 3;
            var p2 = s.indexOf(";", p + 1);
            return decodeURIComponent(s.substring(p, p2));
        }
    };
}
/**
 * VLE data storage functionality
 */
/**
 * Get saved data from the VLE or revert to localStorage if not available.
 * For further details of VLE functionality see VLE.get_server_data() in vleapi.1.js
 *
 * @param user If true, stores data for current user
 * @param names Array of names
 * @param ok Function that is called if the data is retrieved OK.
 * @param error Function that is called if there is an error. (optional)
 * @param activityid Activity id (Optional: omit to use current activity)
 * @param itemid Document item id (Optional; omit to use current document)
 * @param courseid Course numeric id (Optional; omit to use current course)
 */
OU.VLEget = function(user, names, ok, error, activityid, itemid, courseid) {
    if (VLE && VLE.serverversion) {
        VLE.get_server_data(user, names, ok || function() {
        }, error || function() {
        }, activityid, itemid, courseid);
    }
    else { // No VLE, so revert to localStorage
        var values = [];
        for (var i = names.length; i--; ) {
            values[names[i]] = OU.LocalStorage.load(names[i]);
        }
        if (ok) {
            ok(values);
        }
    }
};
/**
 * Stores data on the VLE or reverts to localStorage if VLE not available.
 * For further details of VLE functionality see VLE.set_server_data() in vleapi.1.js
 *
 * @param user If true, stores data for current user
 * @param values JavaScript object containing the key/value pairs to set
 * @param ok Function that is called if the data is set OK. (optional)
 * @param error Function that is called if there is an error. (optional)
 * @param previousvalues Previous values (optional)
 * @param retry Function that is called if previous values changed (optional)
 * @param activityid Activity id (optional: omit to use current activity)
 * @param itemid Document item id (optional; omit to use current document)
 * @param courseid Course numeric id (optional; omit to use current course)
 */
OU.VLEset = function(user, values, ok, error, previousvalues, retry,
        activityid, itemid, courseid) {
    if (VLE && VLE.serverversion) {
        VLE.set_server_data(user, values, ok || function() {
        }, error || function() {
        }, previousvalues, retry, activityid, itemid, courseid);
    }
    // Also(or) store the values in localStorage
    for (var name in values) {
        OU.LocalStorage.save(name, String(values[name]));
    }
};

/**
 * Resizes the close button - when activities are in an epub, a close button is
 * added to the top right of the activity to allow the user to return to the main book content
 *
 * @param {OU.util.Activity} activity -
 */
OU.resizeCloseButton = function(activity) {
    var bd = 45 * OU.dpr;
    if (activity.closeLayer) {
        activity.closeLayer.resize({
            x: activity.w - bd,
            y: 0,
            w: bd,
            h: bd
        });
        activity.closeLayer.context.closeIcon({
            x: (bd - (15 * OU.dpr)) / 2,
            y: (bd - (15 * OU.dpr)) / 2,
            r: (15 * OU.dpr)
        });
        activity.closeLayer.events.clickable.length = 0;
        activity.closeLayer.events.clickable.push(new OU.util.InvisibleButton({
            x: 0,
            y: 0,
            w: bd,
            h: bd,
            onClick: function() {
                window.location = OU.epubBack;
            }
        }));
    }
};
/**
 * loads the back reference for use in ePubs, so the epub knows which page to return to on activiy close
 */
OU.loadBackRef = function() {
    var inc = document.createElement('script');
    inc.src = 'data/back.js';
    inc.type = 'text/javascript';
    this.head.appendChild(inc);
};
/**
 * position and render a close button if we are in an ePub
 */
OU.closeButton = function(activity) {
    var re = /back=(.*)/i, matches,
            url = window.location.toString();
    if (OU._closeBDone)
        return;
    OU._closeBDone = true;
    if (this.widget && !window.widget && window === window.top) {
        if (!this.epubBack) {
            this.epubBack = '../chunk001.html'; // default fall-back option
            matches = re.exec(url);
            if (matches) {
                this.epubBack = matches[1];
            }
            else {
                OU.loadBackRef();
            }
        }
        activity.closeLayer = new OU.util.Layer({
            container: activity,
            x: activity.w - 45,
            y: 0,
            w: 45,
            h: 45,
            'id': '_closeLayer',
            hasEvents: true,
            zIndex: OU.CLOSE_LEVEL,
            dontRegister: true
        });
        activity.closeLayer.context.closeIcon({
            x: 15,
            y: 15,
            r: 15
        });
        activity.closeLayer.events.clickable.push(new OU.util.InvisibleButton({
            x: 0,
            y: 0,
            w: 45,
            h: 45,
            onClick: function() {
                window.location = OU.epubBack;
            }
        }));
    }
};
/**
 * Combines the mouse and touch events to present a unified interface to activities
 */
OU.combineMouseTouch = function(e) { // combine Mouse and Touch events into a common structure { pageX, pageY }
    if (e.touches !== undefined && e.touches[0] !== undefined) {
        return {
            pageX: e.touches[0].pageX,
            pageY: e.touches[0].pageY,
            touches: e.touches
        };
    }
    //when on mobile safari, the coordinates information is inside the targetTouches object
    if (e.targetTouches !== undefined)
        if (e.targetTouches[0] !== undefined)
            e = e.targetTouches[0];
    if (e.pageX !== undefined && e.pageY !== undefined)
        return {
            pageX: e.pageX,
            pageY: e.pageY
        };
    var element = (!document.compatMode || document.compatMode === 'CSS1Compat') ? document.documentElement : document.body;
    return {
        pageX: e.clientX + element.scrollLeft,
        pageY: e.clientY + element.scrollTop
    };
};
/**
 * function to return the distance between 2 points in 2d
 *
 * @param {object} params - json object containing x1,y1, x2,y2
 * @returns {double} distance between points
 */
OU.distance2D = function(params) {
    var dx = params.x2 - params.x1, dy = params.y2 - params.y1;
    return Math.sqrt((dx * dx) + (dy * dy));
};
/**
 * @class Holds font information and provides a correctly formatted font description for setting canvas->context fonts
 *
 * @param {object} p - contains elements: size, weight, style, family
 */
OU.font = function(p) {
    this.size = p.size || OU.theme.fontSize;
    this.weight = p.weight || '';
    this.style = p.style || '';
    if (p.family !== undefined)
        this.family = p.family + ',' + OU.theme.font;
    else
        this.family = OU.theme.font;
    /**
     * generates a string correctly formatted for setting canvas->context fonts
     *
     * @returns {string}
     */
    this.str = function() {
        return this.style + ' ' + this.weight + ' ' + this.size + 'px ' + this.family;
    };
};
/**
 * extracts any GET variables from the page's URL string
 * ie. of the form:   index.html?a=1&b=2
 * @returns {array} array of key-value pairs
 */
OU.getUrlVars = function() { // extract GET vars from current URL
    if (OU._urlVars)
        return OU._urlVars;
    OU._urlVars = [];
    window.location.href.replace(/[?&]+([^=&]+)=([^&]*)/gi, function(m, key, value) {
        OU._urlVars[key] = value;
    });
    return OU._urlVars;
};

OU.addClass = function(obj, className) {
    var name = obj.className.replace(className, '').trim();
    obj.className = name + ' ' + className;
};
OU.removeClass = function(obj, className) {
    obj.className = obj.className.replace(className, '').trim();
};
OU.ensureScreenWrapped = function() {
    if (!OU._screenWrapper) {
        OU._screenWrapper = new OU.util.Div({
            screenWrapper: true,
            style: "overflow:hidden;"
        });
    }
};
// Extensions to the canvas-context object
if (typeof CanvasRenderingContext2D === 'object' || typeof CanvasRenderingContext2D === 'function') {
    /**
     * draws a round cornered rectangle
     */
    CanvasRenderingContext2D.prototype.roundRect = function(x, y, w, h, r) {
        var r2d = Math.PI / 180;
        r = r || 0;
        if (w < 2 * r)
            r = w / 2;
        if (h < 2 * r)
            r = h / 2;
        this.beginPath();
        this.moveTo(x + r, y);
        this.lineTo(x + w - r, y);
        this.arc(x + w - r, y + r, r, r2d * 270, r2d * 360, false);
        this.lineTo(x + w, y + h - r);
        this.arc(x + w - r, y + h - r, r, r2d * 0, r2d * 90, false);
        this.lineTo(x + r, y + h);
        this.arc(x + r, y + h - r, r, r2d * 90, r2d * 180, false);
        this.lineTo(x, y + r);
        this.arc(x + r, y + r, r, r2d * 180, r2d * 270, false);
        this.closePath();
    };
    /**
     * Rounded Rectangle with a square Top Left Corner
     *
     * @param {int} x
     * @param {int} y
     * @param {int} w
     * @param {int} h
     * @param {int} r radius of corner
     */
    CanvasRenderingContext2D.prototype.roundRectTLC = function(x, y, w, h, r) {
        var r2d = Math.PI / 180;
        if (w < 2 * r)
            r = w / 2;
        if (h < 2 * r)
            r = h / 2;
        this.beginPath();
        this.moveTo(x, y);
        this.lineTo(x + w - r, y);
        this.arc(x + w - r, y + r, r, r2d * 270, r2d * 360, false);
        this.lineTo(x + w, y + h - r);
        this.arc(x + w - r, y + h - r, r, r2d * 0, r2d * 90, false);
        this.lineTo(x + r, y + h);
        this.arc(x + r, y + h - r, r, r2d * 90, r2d * 180, false);
        this.lineTo(x, y);
        this.closePath();
    };
    /**
     * Draws a gradient on a rounded rectangle
     *
     * All params are optional, and default to theme defaults & full canvas if not specified
     *
     * @param {object} params containing x,y,w,h,radius,col1,col2, alpha
     */
    CanvasRenderingContext2D.prototype.gradRect = function(params) {
        if (params === undefined)
            params = {};
        var x = params.x || 0, y = params.y || 0, w = params.w || this.canvas.width, h = params.h || this.canvas.height,
                r = params.radius || 0, col1 = params.col1 || OU.theme.bgGradTopCol, col2 = params.col2 || OU.theme.bgGradBotCol, gradient = this.createLinearGradient(x, y, x, y + h);
        gradient.addColorStop(0, col1);
        gradient.addColorStop(1, col2);
        this.save();
        if (params.alpha !== undefined)
            this.globalAlpha = params.alpha;
        this.fillStyle = gradient;
        this.roundRect(x, y, w, h, r);
        this.fill();
        this.restore();
    };
    /**
     * Renders a background(rectangle) based on various parameters
     *
     * @param {object} params - various rendering parameters, all are optional:
     * <ul>
     * <li><strong>{boolean} clear</strong>: if true, clears the background using clearRect (can be used in conjuction with other options.</li>
     * <li><strong>{DOM Image} img</strong>: DOM Image object, draws the image in the background. Note this should not be a filename.</li>
     * <li><strong>{string} col</strong>: fill style, typically in format '#c00' or 'rgba(1,1,1,1)'</li>
     * <li><strong>{string} RGB</strong>: rgb values excluding the alpha part, in a string ie '255,255,255' </li>
     * <li><strong>{double} alpha</strong>: alpha value to use in conjunction with RGB values (0 to 1)</li>
     * <li><strong>{CanvasGradient} gradient</strong>: a canvas gradient object to define a gradient</li>
     * <li><strong>{boolean} shadow</strong>: adds CSS shadowing to the background (this is relatively slow, not advised in time critical use)</li>
     * <li><strong>{double} padding</strong>: amount of padding to reduce the background by</li>
     * <li><strong>{double} radius</strong>: amount of radius to apply, n/a to images or clear</li>
     * <li><strong>{string} borderCol</strong>: if defined then adds a border with this colour</li>
     * <li><strong>{boolean} gloss</strong>: if true, adds a gloss effect over the top of the background</li>
     * </ul>
     * @param {object} dims - a json object that has elements: x,y,w,h to define the area to render it to
     */
    CanvasRenderingContext2D.prototype.background = function(params, dims) {
        if (params === undefined)
            return;
        var x = dims.x, y = dims.y, w = dims.w, h = dims.h, fStyle = null, alpha = params.alpha || 1, g;
        if (params.clear)
            this.clearRect(x, y, w, h);
        if (params.img) {
            this.clearRect(x, y, w, h);
            this.drawImage(params.img, x, y, w, h);
            return;
        }
        if (params.col)
            fStyle = params.col;
        if (params.RGB)
            fStyle = 'rgba(' + params.RGB + ',' + alpha + ')';
        if (params.gradient)
            fStyle = params.gradient;
        if (!fStyle)
            return;
        this.save();
        this.beginPath();
        if (params.shadow) {
            this.shadowOffsetX = 2;
            this.shadowOffsetY = 2;
            this.shadowBlur = 20;
            this.shadowColor = '#666';
        }
        if (params.padding) {
            x = x + params.padding;
            y = y + params.padding;
            w = w - 2 * params.padding;
            h = h - 2 * params.padding;
        }
        this.fillStyle = fStyle;
        if (params.radius !== undefined) {
            this.roundRect(x, y, w, h, params.radius);
            this.fill();
        }
        else {
            this.fillRect(x, y, w, h);
            if (params.borderCol !== undefined) {
                this.rect(x, y, w, h);
            }
        }
        if (params.gloss) {
            g = this.createLinearGradient(x, y, x, y + h);
            g.addColorStop(0, 'rgba(255,255,255,0.52)');
            g.addColorStop(0.49, 'rgba(255,255,255,0.26)');
            g.addColorStop(0.5, 'rgba(255,255,255,0.13)');
            g.addColorStop(1, 'rgba(255,255,255,0.01)');
            this.fillStyle = g;
            this.fill();
        }
        if (params.borderCol !== undefined) {
            this.strokeStyle = params.borderCol;
            this.stroke();
        }
        this.closePath();
        this.restore();
    };
}
// Extend built in javascript Date object to make life easier
/**
 * Increases a date by a number of years
 * @param {int} n - number of years to add
 */
Date.prototype.incYear = function(n) {
    this.setFullYear(parseInt(this.getFullYear()) + (n || 1));
};
/**
 * Increases a date by a number of months
 * @param {int} n - number of months to add
 */
Date.prototype.incMonth = function(n) {
    var nM = parseInt(this.getMonth()) + (n || 1),
            dY = (nM / 12) | 0;
    if (dY >= 1) {
        this.incYear(dY);
        this.setMonth(nM % 12);
    }
    else {
        this.setMonth(nM);
    }
};
/**
 * Increases a date by a number of days
 * @param {int} n - number of days to add
 */
Date.prototype.incDay = function(n) {
    this.setTime(this.valueOf() + (n || 1) * 86400000);
};
/**
 * Decreases a date by a number of years
 * @param {int} n - number of years to take off
 */
Date.prototype.decYear = function(n) {
    this.setFullYear(parseInt(this.getFullYear()) - (n || 1));
};
/**
 * Decreases a date by a number of months
 * @param {int} n - number of months to take off
 */
Date.prototype.decMonth = function(n) {
    var nM = parseInt(this.getMonth()) - (n || 1);
    if (nM < 0) {
        this.decYear((Math.abs(nM) / 12) | 0);
        this.setMonth(12 - (Math.abs(nM) % 12));
    }
    else {
        this.setMonth(nM);
    }
};
/**
 * Decreases a date by a number of days
 * @param {int} n - number of days to take off
 */
Date.prototype.decDay = function(n) {
    this.setTime(this.valueOf() - (n || 1) * 86400000);
};
/**
 * Extends Number to round to N decimal places
 * @param {int} n - number of decimals to round the number off to
 */
Number.prototype.nDecimals = function(n) {
    var m = Math.pow(10, n),
            a = this > 0 ? 0.5 : -0.5,
            t = ((this * m) + a) | 0;
    return t / m;
};
/**
 * Split & trim a string
 * @param {string} on - Specifies the character, or the regular expression, to use for splitting the string.
 */
String.prototype.breaker = function(on) {
    var i, ws = this.split(on);
    for (i = 0; i < ws.length; i++)
        ws[i] = ws[i].trim();
    return ws;
};
/**
 * A slightly better version of typeof, this one can detect arrays.
 * @param {object} obj
 * @returns typeof object, including 'array' if the object is an array
 */
function typeOf(obj) {
    if (typeof (obj) === 'object') {
        if (obj.length)
            return 'array';
        else
            return 'object';
    }
    return typeof (obj);
}
;
/**
 * Register a Messenger Object - ie an object that can recieve messages via the browser message system
 *
 * @param {Object} obj - the object in question
 */
OU.registerMessenger = function(obj) {
    if (!OU._messengerRegister)
        OU._messengerRegister = [];
    var arrayIndex = OU._messengerRegister.length;
    OU._messengerRegister[arrayIndex] = obj;
    obj.__messengerArrayIndex = arrayIndex; // add the index to the obj, so it can remove itself later
    obj.__messengerParams = [];
    obj.defineActivityAPI = function(paramsArray) {
        OU.addMessengerParams(paramsArray, this);
    };
};
/**
 * Register Parameters that can be viewed or set by a message post
 *
 * @param {Object} paramsArray - the param(s), either in an array or a single parameter object
 * @param {Object} obj - the object in question
 *
 * Usage:
 * params should be specified with the following format
 *
 *  {
 *      name: "paramName",
 *      getFunction: <function reference> // reference to the function that gets this param's value
 *      setFunction: <function reference> // reference to the function that sets this param's value
 *  }
 */
OU.addMessengerParams = function(paramsArray, obj) {
    var i;
    if (typeOf(paramsArray) === 'object') {
        obj.__messengerParams.push(paramsArray);
    }
    else {
        for (i = paramsArray.length; i--; ) {
            obj.__messengerParams.push(paramsArray[i]);
        }
    }
};
/**
 * Remove a Messenger Object
 *
 * @param {Object} obj - the object in question
 */
OU.removeMessenger = function(obj) {
    OU._messengerRegister[obj.__messengerArrayIndex] = null;
};
/**
 * List the Messenger Objects and return as a message response
 *
 */
OU.listMessengers = function() {
    var messenger, i, j, count = 0, params, paramsArray, reply = [], actType;
    for (i = OU._messengerRegister.length; i--; ) {
        messenger = OU._messengerRegister[i];
        if (messenger) {
            params = messenger.__messengerParams;
            paramsArray = [];
            for (j = params.length; j--; ) {
                paramsArray.push({
                    name: params[j].name,
                    gettable: (params[j].getFunction !== null && params[j].getFunction !== undefined),
                    settable: (params[j].setFunction !== null && params[j].setFunction !== undefined)
                });
            }
            if (messenger.controllerActivity)
                actType = messenger.controllerActivity.type;
            if (params.length > 0) {
                count++;
                reply.push({
                    id: i,
                    instance: messenger.instance,
                    type: actType,
                    params: paramsArray
                });
            }
        }

    }
    if (count === 0) {
        return "No Activities";
    }
    return reply;
};
/**
 * get or set a parameter of a Messenger object (typically an Activity), IO via Messaging
 * @param {object} data - Data passed via the window.postMessage function
 * @param {object} event - the message event itself
 * @private
 */
OU._getSetMessengerParamMessaging = function(data, event) {
    var param, params, j, error = 'Invalid Activity ID', messenger = OU._messengerRegister[data.activityId];
    if (messenger) {
        params = messenger.__messengerParams;
        for (j = params.length; j--; ) {
            param = params[j];
            if (param.name === data.param) {
                if (data.action === 'set') {
                    if (param.setFunction) {
                        param.setFunction(data.value);
                        return "Success";
                    }
                    else {
                        error = 'Parameter not settable';
                    }
                }
                if (data.action === 'get') {
                    if (param.getFunction) {
                        return param.getFunction(data.value);
                    }
                    else {
                        error = 'Parameter not gettable';
                    }
                }
                if (data.action === 'monitor') {
                    if (param.monitor) {
                        if (!data.value)
                            data.value = {};
                        data.value.callback = function(content) {
                            var reply = {};
                            reply.action = data.action;
                            reply.param = data.param;
                            reply.content = content;
                            event.source.postMessage(reply, event.origin);
                        };
                        return param.monitor(data.value);
                    }
                    else {
                        error = 'Parameter not monitorable';
                    }
                }
            }
        }
        error = 'Invalid Parameter name';
    }
    return "Failed: " + error;
};
/**
 * get or set a parameter of a Messenger object (typically an Activity), IO via "internal" activity
 * @param {object} data - Data passed
 * @private
 */
OU._getSetMessengerParamInternal = function(data) {
    var param, params, j, error = 'Invalid Activity ID', messenger = OU._messengerRegister[data.activityId];
    if (messenger) {
        params = messenger.__messengerParams;
        for (j = params.length; j--; ) {
            param = params[j];
            if (param.name === data.param) {
                if (data.action === 'set') {
                    if (param.setFunction) {
                        param.setFunction(data.value);
                        return "Success";
                    }
                    else {
                        error = 'Parameter not settable';
                    }
                }
                if (data.action === 'get') {
                    if (param.getFunction) {
                        return param.getFunction(data.value);
                    }
                    else {
                        error = 'Parameter not gettable';
                    }
                }
                if (data.action === 'monitor') {
                    if (param.monitor) {
                        if (data.value)
                            return param.monitor(data.value);
                    }
                    else {
                        error = 'Parameter not monitorable';
                    }
                }
            }
        }
        error = 'Invalid Parameter name';
    }
    return "Failed: " + error;
};
/**
 * Recieves messages from other windows via the window.postMessage() function
 *
 * For example, recieves a message from a parent window, when we are in an iFrame
 *
 * @param {Object} event - the message received
 */
OU._msgRecieve = function(event) {
    // validate event.source
    var data = event.data, reply;

    if (data) {
        reply = {
            action: data.action,
            param: data.param
        };
        switch (data.action) {
            default:
            case 'list':
                reply.content = OU.listMessengers();
                break;
            case 'get':
            case 'set':
            case 'monitor':
                reply.content = OU._getSetMessengerParamMessaging(data, event);
                break;
            case 'hello':
                reply.content = "hiya";
                break;
        }
    }

    event.source.postMessage(reply, event.origin);
};
try {
    window.addEventListener("message", OU._msgRecieve); // add event listener to enable message receipt from other windows
}
catch (e) {
    console.log('WINDOW MESSAGING NOT SUPPORTED');
}

/**
 * @class Activity Class - defines the generic structure and behaviour of all activities.
 * All activities should extend this class
 *
 * @param {Object} data - Holds the data content for a specific instance of the activity
 * @param {String} instance - A unique identifier name for this instance, defaults to 'a1'
 * @param {OU.util.Controller} controller - A reference to the controller that initialised this instance, if undefined, the superclass will generate a new controller and create the reference to it as 'this.controller'
 */
OU.util.Activity = function(data, instance, controller) {
    this.instance = instance || 'a1';
    this.data = data || {};
    this._prefO = data.preferredOrientation;
    this.controller = controller;
    this.state = OU.LOADING;
    if (this.controller && this.controller.activityDefs[this.instance]) {
        this.controllerActivity = this.controller.activityDefs[this.instance];
        this.dataDir = this.controllerActivity.data;
        this.zOffset = this.controllerActivity._zOffset;
    }
    else {
        this.dataDir = 'data/';
        this.zOffset = 0;
    }
    this.init();
    if (this.data.css) {
        this.loadCSSFile(this.data.css);
    }
    if (this.data.activityOptions && this.data.activityOptions.css) {
        this.loadCSSFile(this.data.activityOptions.css);
    }
    this._resizeable = true;
    if (this._resizeNeeded)
        this.resize();
};
/**
 * Performs some house keeping and set up for generic activities
 *
 * @private
 **/
OU.util.Activity.prototype.init = function() {
    var self = this, re = /accessible/;
    OU.initPage();
    OU.addRemovable(this, this.zOffset); // All activities are removables
    OU.registerMessenger(this); // Register all activities as message receivers/transmitters
    if (!re.test(window.location) && this.browserTest()) {
        this.getsize_();
        this.forceController_();
        if (typeof this.data.activityInstructions === 'object'
                && typeof (this.instructionsView) === 'function'
                && (typeof this.data.activityInstructions.autoLoad === 'undefined' || this.data.activityInstructions.autoLoad === true))
            this.instructionsView();
        if (typeof (this.canvasView) === 'function') // Do any extended functionality
            this.canvasView();
        if (typeof (this.view) === 'function') // Do any extended functionality
            this.view();
        if (this.data.header)
            this.header(this.data.header);

        OU.onOrientChange(function() {
            OU.resetButtonGroupSizes(); // reset the button font sizes
            if (OU._screenWrapper) {
                OU._screenWrapper.resize({
                });
            }
            self.resize();
        }, true);
    }
    else {
        this.accessibleView();
    }
};
OU.util.Activity.prototype.browserTest = function() {
    var meetsRequirements = true;

    // Test for canvas support
    try {
        var canvas = document.createElement('canvas');
        if (canvas) {
            if (typeof canvas.getContext !== 'function') {
                meetsRequirements = false;
                console.log('FAILURE: Browser does not support canvas context');
            }
        }
        else {
            meetsRequirements = false;
            console.log('FAILURE: Browser does not support canvas');
        }
    }
    catch (e) {
        meetsRequirements = false;
        console.log('FAILURE: Browser does not support canvas (exception)');
    }
    return meetsRequirements;
};
/**
 * Ensure that every activity has a controller
 * If not already in a controller, then initalise one and put this activity in it
 * @private
 */
OU.util.Activity.prototype.forceController_ = function() {
    var className = '';

    if (this.baseController === undefined && this.controller === undefined) {
        if (OU.baseController) {
            this.controller = OU.baseController;
        }
        else {
            this.controller = new OU.util.Controller({
                sections: [
                    {
                        activities: []
                    }
                ]
            });
        }

        this.controller.section.activityObjects.push(this);
        if (this.constructor && this.constructor.toString) { // determine the activities className
            className = /(OU\..*)\.prototype/.exec(this.constructor.toString())[1];
        }

        this.controllerActivity = {//} this.controller.activities[this.instance]  = {
            data: 'data/',
            type: className
        }; // push this into the controllers activity list
        this.controller.section.activities.push(this.controllerActivity);
    }
};
/** Changes the page headers
 *  Any Activity can call this and it will change the page header of the base Controller
 *  If no current header is in place, one will be added and all sub activities resized accordingly
 *
 *  @param {object} h - contains optional h1 and h2 elements ie. {h1:"header 1",h2:"header 2"}
 */
OU.util.Activity.prototype.header = function(h) {
    OU.baseController.header(h);
};
/**
 * canvasView
 *
 * Intended to be overridden by the sub class
 */
OU.util.Activity.prototype.canvasView = function() {
};
/**
 * view
 * As we are moving away from Canvas, I thought this would be more appropriate.
 * Intended to be overridden by the sub class
 */
OU.util.Activity.prototype.view = function() {
};
/**
 * instructionsView
 *
 * @param {object} instructions
 */
OU.util.Activity.prototype.instructionsView = function() {
    var instructionText = this.data.activityInstructions.instructions || null;
    var buttonText = this.data.activityInstructions.button || 'Start';
    var background = this.data.activityInstructions.background || null;

    if (instructionText && instructionText.length) {
        var instructions = document.createElement('div');
        instructions.setAttribute('class', 'activity_instructions');
        instructions.setAttribute('id', 'activity_instructions');
        instructions.style.zIndex = OU.POP_UP_LEVEL;

        if (background) {
            if (background === 'transparent') {
                instructions.style.backgroundColor = 'rgba(255, 255, 255, 0.9)';
            } else {
                instructions.style.background = 'url(data/' + background + ') #FFF';
                instructions.style.backgroundSize = 'cover';
            }
        }

        var infoContainer = document.createElement('div');
        infoContainer.setAttribute('class', 'instructions_container');

        var info = document.createElement('div');
        info.setAttribute('class', 'instructions');
        info.innerHTML = instructionText.join('<br />');

        infoContainer.appendChild(info);

        var buttonContainer = document.createElement('div');
        buttonContainer.setAttribute('class', 'button-container');

        var button = document.createElement('button');
        button.setAttribute('class', 'button');
        button.setAttribute('style', 'margin-top:15px;');
        button.textContent = buttonText;
        OU.events.addListener(instructions, function() {
            if (instructions) {
                document.body.removeChild(instructions);
                instructions = null;
            }
        });
        OU.events.addListener(button, function() {
            if (instructions) {
                document.body.removeChild(instructions);
                instructions = null;
            }
        });

        buttonContainer.appendChild(button);

        instructions.appendChild(infoContainer);
        instructions.appendChild(buttonContainer);

        document.body.appendChild(instructions);

        this.viewResize = function(self) {
            return function() {
                var padding = 15;
                infoContainer.style.top = padding + 'px';
                infoContainer.style.left = padding + 'px';
                infoContainer.style.width = (self.w - (padding * 2)) + 'px';
                infoContainer.style.height = 'auto'; // (self.h - button.getBoundingClientRect().height - ((padding + 10) * 2)) + 'px';

                info.style.marginTop = '';
                info.style.marginLeft = '';
                info.style.width = '50%';
                if (self.w <= 480) {
                    info.style.fontSize = '13px';
                } else if (self.w <= 600) {
                    info.style.fontSize = '15px';
                } else if (self.w <= 800) {
                    info.style.fontSize = '17px';
                } else {
                    info.style.fontSize = '19px';
                }

                while (parseInt(info.style.width) < 100) {
                    if (infoContainer.scrollHeight > infoContainer.clientHeight || infoContainer.scrollHeight > infoContainer.offsetHeight) {
                        info.style.width = parseInt(info.style.width) + 5 + '%';
                    } else {
                        break;
                    }
                }

                if (parseInt(info.style.width) >= 100) {
                    info.style.width = '';
                }

                var infoContainerBCR = infoContainer.getBoundingClientRect();
                var infoBCRHeight = info.getBoundingClientRect().height;
//                if (infoContainerBCR.height > infoBCRHeight) {
                info.style.marginTop = ((((self.h - button.getBoundingClientRect().height - ((padding + 10) * 2)) - infoBCRHeight) / 2) | 0) + 'px';
                //              }
                info.style.marginLeft = (((window.innerWidth - info.getBoundingClientRect().width) / 2) | 0) + 'px';
            };
        }(this);

        this.viewResize();
    }
};
OU.util.Activity.prototype.removeInstructionsView = function() {
    var insDiv = document.getElementById('activity_instructions');
    if (insDiv) {
        insDiv.parentNode.removeChild(insDiv);
        insDiv = null;
    }
};
/**
 * Remove - house keeping tidy up for all activities.
 * Optionally extended by the sub class. Do not override
 */
OU.util.Activity.prototype.remove = function() {
    OU.removeMessenger(this);
    var objs = document.getElementsByClassName('activity_instructions');
    for (var i = objs.length; i--; ) {
        document.body.removeChild(objs[i]);
    }
};
/** accessibleView
 *
 *  Intended to be extended or overidden by the sub class if required
 */
OU.util.Activity.prototype.accessibleView = function() {
    var accessible = document.createElement('div');
    accessible.innerHTML = '<div style="border-radius:20px; margin: 20px; color:#222;background:#eee; font-size:1.4em;padding:20px;" id="accessibleView">Unfortunately your current browser is unable to support this HTML5 interactive.<br/><br/>For the full experience of this material, please use an alternative browser. We recommend that you use Google Chrome on a desktop or tablet.</div>';
    document.body.appendChild(accessible);
};
/**
 * Called whenever the activity's outer 'frame' changes size
 * Gets the activities new outer dimensions
 * Moves the closeButton if applicable
 * Checks to see if the activity has a preferred orientation, if so displays relevant message if required
 *
 * Intended to be extended (not overridden) by the sub class
 *
 * @param {object} params - optional new dimensions containing elements x,y,w,h if defined
 */
OU.util.Activity.prototype.resize = function(params) { // intended to be extended by the sub class
    this.getsize_(params);
    OU.resizeCloseButton(this);
    if (this._prefO) { //TODO need to make this only work for mobile devices or display a 'resize browser' message on desktops
        if ((this._prefO === 'landscape' && this.h > this.w) || (this._prefO === 'portrait' && this.h < this.w)) {
            if (OU._oWarn) {
                OU._oWarn.resize();
            }
            else {
                OU._oWarn = new OU.util.Layer({
                    container: this,
                    zIndex: OU.ABSOLUTE_TOP
                });
            }
            var ctx = OU._oWarn.context;
            ctx.fillStyle = '#fff';
            ctx.fillRect(0, 0, this.w, this.h);
            new OU.util.DynText({
                x: this.w * .1,
                w: this.w * .8,
                h: this.h,
                txt: ["This activity is best", "viewed in " + this._prefO + " mode."],
                colour: '#666',
                align: 'center',
                context: ctx
            });
        }
        else {
            if (OU._oWarn) {
                OU._oWarn.remove();
                OU._oWarn = null;
            }
        }
    }

    if (typeof this.viewResize === 'function') {
        this.viewResize();
    }
};
/**
 * Loads additional CSS into the page from an additional file
 *
 * @param {string} f - filename relative to the data folder
 */
OU.util.Activity.prototype.loadCSSFile = function(f) {
    console.log('loading css:' + f);
    OU.loadCSS(this.dataDir + f);
};
/**
 * Sets the activities dimensions and position, either from params passed, the controller, or the outer container of the activity
 * @private
 */
OU.util.Activity.prototype.getsize_ = function(params) {
    params = params || {};
    if (this.controllerActivity !== undefined) {
        var a = this.controllerActivity;
        if (a._dims) {
            this.x = params.x || a._dims.x;
            this.y = params.y || a._dims.y;
            this.w = params.w || a._dims.w;
            this.h = params.h || a._dims.h;
            return;
        }
    }
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || window.innerWidth || 480;
    this.h = params.h || window.innerHeight || 360;
};
/**
 * @class Tabbable - the base class for objects that are tabbable for accessibility
 *
 * @Usage - The Tabbable class is designed to be extended by an object sub class (ie. OU.util.BaseButton)
 *
 * The sub class should override the following functions:
 *
 *      focus()         // called when the object has Tab-focus
 *      blur()          // called when the object loses Tab-focus
 *      hit()           // called when the object has focus and the Enter or Space key are hit (must return False)
 *      arrowKey(k)     // called when any other button is pressed
 *
 * NOTE: The sub class object should only be initialised once and then removed.
 * Automatic removal is performed by the controller when an activity is removed.
 * However, if you are creating new instances of your sub-class object (ie. with each render), then you must remove the old object first, using the remove() function.
 *
 */
OU.util.Tabbable = function(params) {
    this.txt = params.tabbableText || params.txt || 'button';
    this.tabIndex = params.tabIndex || 100;
    OU.util.Tabbable.prototype.initTabbable = function() {
        var newI;
        this._tabbable = true;
        this._tabCount = OU._tabCount = (OU._tabCount || 0) + 1;
        this.position(this, this._tabCount);
        if (!OU.__tabsDiv) {
            OU.__tabsDiv = document.createElement('div');
            OU.__tabsDiv.setAttribute('id', "__tabsDiv");
            document.body.appendChild(OU.__tabsDiv);
        }
        if (!document.getElementById("__tabsDiv")) {
            document.body.appendChild(OU.__tabsDiv);
        }
        newI = document.createElement('input');
        newI.setAttribute('id', '_tabbable_' + this._tabCount);
        newI.setAttribute('value', params.txt);
        newI.setAttribute('tabIndex', this.tabIndex);
        newI.setAttribute('onKeyDown', 'return OU._tabbables[' + this._tabCount + ']._key(event);');
        newI.setAttribute('onFocus', 'return OU._tabbables[' + this._tabCount + '].focus();');
        newI.setAttribute('onBlur', 'return OU._tabbables[' + this._tabCount + '].blur();');
        OU.__tabsDiv.appendChild(newI);
        this.hiddenObj = newI;
        OU.addRemovable(this);
        delete newI;
    };
    OU.util.Tabbable.prototype.position = function(item, index) {
        OU._tabbables[index] = item;
    };
    OU.util.Tabbable.prototype.remove = function() {
        if (OU.__tabsDiv && this.hiddenObj) {
            OU.__tabsDiv.removeChild(this.hiddenObj);
            this.hiddenObj = null;
            if (!OU.__tabsDiv.hasChildNodes() && OU.__tabsDiv.parentNode) {
                // last entry has been deleted lets kill the layer...
                OU.__tabsDiv.parentNode.removeChild(OU.__tabsDiv);
            }
            //  OU.__tabsDiv.removeChild(this.hiddenObj);
            this.hiddenObj = null;
        }
    };
    OU.util.Tabbable.prototype.setFocus = function(a) {
        if (a === false) // assume true if not present (undefined)
            this.hiddenObj.blur();
        else
            this.hiddenObj.focus();
    };
    OU.util.Tabbable.prototype.focus = function() { // called when the button gets focus- override with subclass
    };
    OU.util.Tabbable.prototype.blur = function() { // called when the button loses focus - override with subclass
    };
    OU.util.Tabbable.prototype.hit = function() { //  called when enter key is hit on the focussed button - override with subclass
        return false;
    };
    OU.util.Tabbable.prototype.arrowKey = function(k) { // called when an arrow key is pressed- override with subclass
        // return true to prevent Events also being triggered by keypress
    };
    /**
     * @private
     */
    OU.util.Tabbable.prototype._key = function(e) { //  called on keypress
        var k = null;
        if (window.event)
            k = window.event.keyCode;
        else if (e)
            k = e.which || e.keyCode;
        if (k === 13 || k === 32) {
            this.hit();
            return;
        }
        if (e.keyCode > 36 && e.keyCode < 41) {
            e._dealtWith = this.arrowKey(e.keyCode);
            return;
        }
    };
    if (!params.disableTab)
        this.initTabbable();
};
/**
 * bootStrap - Once we have all the default code loaded and the data.js file for the top activity, load the top activity and run it.
 */
OU.bootStrap = function() {
    //Load in the starting data.json file, which defines the top activity type and includes its data
    var xhr = new XMLHttpRequest();
    if (typeof xhr.overrideMimeType !== "function") {
        xhr.overrideMimeType = function() {
            console.log("Warning, XMLHttpRequest object doesn't support method overrideMimeType()");
        };
    }
    xhr.overrideMimeType("application/json");
    xhr.open("GET", "data/data.json", true);
    xhr.onreadystatechange = function() {
        if (xhr.readyState === 4) {
            var data = null;
            try {
                data = JSON.parse(xhr.responseText);
            }
            catch (e_json) {
                console.log('Error: JSON.parse failed, check your activity\'s data.json... ' + e_json);
                return;
            }

            var standardOptions = data.standardOptions;
            if (typeof standardOptions !== "object" || null === standardOptions) {
                console.log('Error: "standardOptions" required in JSON - processing aborted');
                return;
            }

            var activityOptions = data.activityOptions;
            if (typeof activityOptions !== "object" || null === activityOptions) {
                console.log('Error: "activityOptions" required in JSON - processing aborted');
                return;
            }

            var activityContent = data.activityContent;
            if (typeof activityContent !== "object" || null === activityContent) {
                console.log('Error: "activityContent" required in JSON - processing aborted');
                return;
            }

            (function() {
                var whiteList = [
                    "standardOptions",
                    "activityOptions",
                    "activityContent",
                    "activityInstructions"
                ];
                var excessProperties = [];

                try {
                    Object.keys(data).forEach(function(item) {
                        if (-1 === whiteList.indexOf(item)) {
                            excessProperties.push(item);
                        }
                    });
                }
                catch (e) {
                    console.log("Warning: Couldn't determine excessProperties in JSON list (Possible old browser)");
                }

                if (excessProperties.length > 0) {
                    console.log("Warning: excess properties in JSON... " + excessProperties.join(" "));
                }
            }());

            OU.initPage();

            var actType = "OU.activity." + standardOptions.activityType;
            console.log('[lib.ver. 170314] activity type: ' + actType);
            OU.require(actType); // load in the activity code

            var pageTitle = standardOptions.title;
            if (!pageTitle) {
                pageTitle = actType.substr(actType.lastIndexOf('.') + 1);
            }
            OU.loadTitle(pageTitle || 'Activity'); // Set page title
            OU.loadTheme(data.theme); // Load the Theme
            OU.loadFonts(data.font); // Load extra fonts if required
            // Run the activity when it loads
            OU.onLoad(function() {
                // Run the application instance
                var fn = eval(actType);

                OU.topActivity = new fn(data);
            });
        }
    };
    xhr.setRequestHeader("Cache-Control", "no-cache");
    xhr.send(null);

// initialise the DomEvents handler
    OU.events = new OU.util.DomEvents(this);
    OU.events.attachToDom(document.body);

};
/**
 * Loads in a data.json file using an XMLHttpRequest and runs an activity using the data
 * @param {OU.activity.Object} fn - the Activity function to run
 * @param {String} dataFile - filename and path ie. data/data.json
 */
OU.runActivityWithData = function(fn, dataFile, instance, controller) {
    var xhr = new XMLHttpRequest();
    if (typeof xhr.overrideMimeType !== "function") {
        xhr.overrideMimeType = function() {
            console.log("Warning, XMLHttpRequest object doesn't support method overrideMimeType()");
        };
    }
    xhr.overrideMimeType("application/json");
    xhr.open("GET", dataFile, true);
    xhr.onreadystatechange = function() {
        if (xhr.readyState === 4) {
            var data = JSON.parse(xhr.responseText),
                    newActivity = controller.activityDefs[instance],
                    ao = new fn(data, instance, controller);
            controller.section.activityObjects.push(ao);
            ao.controllerActivity = newActivity;
            newActivity.objRef = ao;
        }
    };
    xhr.setRequestHeader("Cache-Control", "no-cache");
    xhr.send(null);
};

OU.loadCSS(OU.cssPath + 'style.css'); // load the default stylesheet
//OU.loadScript('data/data.js'); // load the data for the top activity
OU.require('OU.util.Controller'); // required by Activity class
OU.require('OU.util.Layer');   // required by Close Button
OU.require('OU.util.DynText'); // required by Close Button
OU.require('OU.util.Button'); // required by Close Button
OU.require('OU.util.DomEvents'); // required
/*
 *  Load the Theme and the base Activity class, as these are always required
 */
OU.onLoad(OU.bootStrap);


