/**
 * @fileoverview Controller Class
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
OU.require('OU.util.Div');
OU.require('OU.util.PopUpInfo');
/**
 *
 * @class Base Controller Class - provides the guts of all controller type activities.
 * Facilitates the loading/unloading of multiple activities, including housekeeping
 * functionality such as removing canvases, divs, etc.
 *
 * @extends OU.util.Activity
 */
OU.util.Controller = function(baseControllerData, instance, controller) {
    /**
     * Main starting point when in a browser that supports canvas
     */
    OU.util.Controller.prototype.canvasView = function() {
        this.controllerId = OU.controllers.length; // get the menuId for this menu activity
        OU.controllers[this.controllerId] = this;    // and store this menu in the controllers array of menus
        OU.initPage = function() {
        };   //Override the initPage function to stop each sub activity overwriting the controller
        this.menuHeight = 0;
        this.headerHeight = 0;
        this.footerHeight = 0;
        this.tabHeight = 0;
        this.padding = 0;
        this.menus = [];
        this.activities = [];
        this.activityDefs = {};
        this.sections = this.data.sections || [{
                activities: []
            }];
        this.dontsave = this.data.dontsave || false;
        this.chgSectionMaxCountdown = 1500;
        this.audioDiv = null;
        this.initSections();
        this.audioStep = 1;
        this.watchAudio();
        OU.addRemovable(this, this.zOffset);
    };
    /**
     * Removal function to be used if controllers are nested, and a sub controller is removed by a parent
     */
    OU.util.Controller.prototype.remove = function() {
        if (this.controller !== undefined) {
            this.clearSection();
            if (this.audioDiv) {
                this.audioDiv.remove();
            }
        }
        OU.controllers[this.controllerId] = null;
    };
    /**
     * Initialises the sections when the controller starts up
     * <ul>
     * <li>Calls InitSection for each section</li>
     * <li>Sets current section (remembering where we where last time)</li>
     * <li>Waits for relevant code to load and then calls loadSection</li>
     * </ul>
     *
     * @private
     */
    OU.util.Controller.prototype.initSections = function() {
        var i, s,
                ns = this.numSections = this.sections.length,
                self = this;
        this.sectionNum = 0;
        for (i = ns; i--; ) {
            s = this.sections[i];
            s.sectionId = i;
            this.initSection(s, i);
        }
        if (!this.dontsave) {
            this.sectionNum = parseInt(OU.LocalStorage.load("OU.control.sectionID." + this.instance + "." + this.data.SaveID) || 0);
        }
        this.subtitleson_ = OU.LocalStorage.load("OU.control.subtitlesOn") === 'true';
        if (this.sections[this.sectionNum] === undefined) {
            this.sectionNum = 0;
        }
        this.section = this.sections[this.sectionNum];
        OU.onLoad(function() {
            self.loadSection();
        }); // wait for all required js files to load
    };
    /**
     * Initialises a section - looks at the activities required for this section and loads the activity code using OU.require
     *
     * @private
     * @param {object} s - Section to init
     * @param {int} i - Section index, used to auto generate instance names for activities
     */
    OU.util.Controller.prototype.initSection = function(s, i) {
        var j, a;
        if (i === undefined)
            i = 'x';
        if (s.options === undefined)
            s.options = {};
        s.activityObjects = [];
        for (j = s.activities.length; j--; ) {
            a = s.activities[j];
            a._instName = '_act' + i + '_' + j;
            a._dataLoaded = false;
            if (a.type.indexOf('OU.activity') !== 0) {
                a.type = "OU.activity." + a.type;
            }
            OU.require(a.type); // load the relevant Javscript for this activity
            this.activities.push(a);
        }
        if (s.options.chgSectionMaxCountdown)
            this.chgSectionMaxCountdown = s.options.chgSectionMaxCountdown;
    };
    /**
     * Defines the dimensions for all activities in a section
     * @param {object} s - the section to resize
     * @private
     */
    OU.util.Controller.prototype.resizeSection = function(s) {
        var j, a,
                h = this.h - this.menuHeight - this.headerHeight - this.footerHeight - this.tabHeight - 2 * this.padding,
                y = this.y + this.menuHeight + this.headerHeight + this.tabHeight + this.padding;
        if (s.activities.length > 0) {
            for (j = s.activities.length; j--; ) {
                a = s.activities[j];
                if (a !== undefined && a.dims !== undefined) {
                    if (this.w > this.h || s.options.maintainLayout === true) { // landscape
                        if (!a.dims.landscape) {
                            a._dims = {
                                x: this.x + this.w * (a.dims.x || 0) + this.padding,
                                y: y + h * (a.dims.y || 0),
                                w: this.w * (a.dims.w || 1) - 2 * this.padding,
                                h: h * (a.dims.h || 1)
                            };
                        }
                        else {
                            a._dims = {
                                x: this.x + this.w * (a.dims.landscape.x || 0) + this.padding,
                                y: y + h * (a.dims.landscape.y || 0),
                                w: this.w * (a.dims.landscape.w || 1) - 2 * this.padding,
                                h: h * (a.dims.landscape.h || 1)
                            };
                        }
                    }
                    else { // portrait
                        if (!a.dims.portrait) {
                            a._dims = {
                                x: this.x + this.w * (a.dims.y || 0) + this.padding,
                                y: y + h * (a.dims.x || 0),
                                w: this.w * (a.dims.h || 1) - 2 * this.padding,
                                h: h * (a.dims.w || 1)
                            };
                        }
                        else {
                            a._dims = {
                                x: this.x + this.w * (a.dims.portrait.x || 0) + this.padding,
                                y: y + h * (a.dims.portrait.y || 0),
                                w: this.w * (a.dims.portrait.w || 1) - 2 * this.padding,
                                h: h * (a.dims.portrait.h || 1)
                            };
                        }
                    }
                }
                else { // no dims set, so use fullscreen
                    a._dims = {
                        x: this.x + this.padding,
                        y: y,
                        w: this.w - 2 * this.padding,
                        h: h
                    };
                }
            }
        }
    };
    /**
     * Resizes all sections by calling resizeSection for each one.
     * @private
     */
    OU.util.Controller.prototype.resizeSections = function() {
        var i, s;
        if (this.headerDiv) {
            this.headerDiv.resize({
                w: this.w,
                h: this.headerHeight
            });
        }
        this.resizeSection(this.section); // resize current section, in case it is an overriden section and therefore not in the array.
        if (this.sections !== undefined && this.sections.length > 0) {
            for (i = this.sections.length; i--; ) {
                s = this.sections[i];
                if (s !== this.section) // no need to do this.section again
                    if (s.activityObjects.length > 0) {
                        this.resizeSection(s);
                    }
            }
        }
    };
    /**
     * Resizes the controller, resizes each section, then calls the resize function for all live activities
     *
     * @private
     */
    OU.util.Controller.prototype.resize = function() {
        OU.util.Controller.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, i, a,
                j, s = this.section;
        if (this.headerDiv) {
            this.headerDiv.resize({
                w: this.w,
                h: "auto"
            });
        }
        if (this.audioDiv) {
            this.audioDiv.resize({
                x: this.x + 40,
                y: this.y + this.h - bH,
                w: this.w / 2,
                h: bH
            });
        }
        if (this.subtitleDiv) {
            this.subtitleDiv.resize({
                x: OU.baseController.x + 20,
                y: OU.baseController.y + OU.baseController.h * .8,
                w: OU.baseController.w - 40,
                h: 'auto'
            });
        }
        this.resizeSections();
        for (i = s.activityObjects.length; i--; ) {
            if (s.activityObjects[i] !== null) {
                a = s.activityObjects[i];
                if (a._resizeable)
                    a.resize();
                else
                    a._resizeNeeded = true;
            }
        }
        for (j = s.activities.length; j--; ) {
            a = s.activities[j];
            if (a.frame !== undefined) {
                a.frame.resize({
                    x: a._dims.x - 2 * OU.dpr,
                    y: a._dims.y - 42 * OU.dpr,
                    w: a._dims.w,
                    h: a._dims.h + 40 * OU.dpr
                });
            }
        }
    };
    /** To be overwritten by the sub class  */
    OU.util.Controller.prototype.render = function() {
    };
    /**
     * Overrides the current section using activity definitions held in an array. Useful, when you are using some HTML code to load a new section
     *
     * @param {int} n - array index of the section to recall
     */
    OU.util.Controller.prototype.recallOverride = function(n) {
        var s = OU._overrideArray[n];
        if (s !== undefined)
            this.overrideSection(s);
        return false;
    };
    /**
     * Adds an activity using an activity definition held in an array. Useful, when you are using some HTML code to load a new activity
     *
     * @param {int} n - array index of the activity to recall
     */
    OU.util.Controller.prototype.recallPopup = function(n) {
        var a = OU._overrideArray[n];
        if (a !== undefined)
            this.addActivity(a);
        return false;
    };
    /**
     * Removes all current activities and loads some different ones.
     * @param {object} s - A controller section definition, containing 1 or more activity definitions
     */
    OU.util.Controller.prototype.overrideSection = function(s) {
        if (s === undefined)
            return;
        this.clearSection();
        this.section = s;
        this.initSection(this.section);
        this.loadSection();
    };
    /**
     * Loads the current Section
     * <ul>
     * <li> Looks at each activity and loads the relevant data file</li>
     * <li> Ensures everything is loaded before continuing onto render and startSection </li>
     * </ul>
     * @private
     */
    OU.util.Controller.prototype.loadSection = function() {
        var a, j, s = this.section, self = this;
        OU.LocalStorage.save("OU.control.sectionID." + this.instance + "." + this.data.SaveID, this.sectionNum);
        for (j = s.activities.length; j--; ) {
            a = s.activities[j];
            a._zOffset = this.zOffset + 100 * (j + 1);
            if (!a._dataLoaded) {
                this.loadDataJSON(a);
                return;
            }
        }
        if (OU.requiredFiles > OU.fileCount) {
            setTimeout(function() {
                self.loadSection();
            }, 40);
            return;
        }
        this.render();
        this.startSection();
    };
    /**
     * Starts the section by Running all the Activities and inserting any audio for this section
     *
     * @private
     */
    OU.util.Controller.prototype.startSection = function() {
        var j, a, f, ao, s = this.section;
        this.header(s.header);
        for (j = s.activities.length; j--; ) {
            a = s.activities[j];
            this.activityDefs[a._instName] = a;
            f = eval(a.type);
            s.activityObjects.push(ao = new f(a._data, a._instName, this));
            a.objRef = ao;
            ao.controllerActivity = a;
            ao._zOffset = a._zOffset;
            if (a.frame !== undefined)
                this.activityFrame(a);
        }
        this.resize();
        this.audioStep = 1;
        if (s.audio !== undefined) {
            this.insertAudio(s.audio[0]);
        }
        else {
            this.removeAudio();
        }
    };
    /**
     * Loads the data for a specific activity.
     * Inserts a script tag to the page head to load the data.js file
     * @param {OU.util.Activity} activity
     * @private
     */
    OU.util.Controller.prototype.loadDataJSON = function(activity) {
        var self = this;
        activity._dataLoaded = false;

        if (activity.inline) {
            // loads directly into the object bypassing page injection
            activity._data = activity.inline;
            activity._dataLoaded = true;
            self.loadSection();
        }
        else {
            var xhr = new XMLHttpRequest();

            xhr.overrideMimeType("application/json");
            xhr.open("GET", activity.data + 'data.json', true);
            xhr.onreadystatechange = function() {
                if (xhr.readyState === 4) {
                    var data = JSON.parse(xhr.responseText);
                    activity._data = data;
                    activity._dataLoaded = true;
                    self.loadSection();
                }
            };
            xhr.setRequestHeader("Cache-Control", "no-cache");
            xhr.send(null);
        }
    };

    /**
     * Adds an Activity to the controller, without clearing current section
     * @param {object} act - minimum required elements are:
     * <ul>
     * <li>type - the activity type, ie. "Slideshow"</li>
     * <li>data - the relative URL of the data directory for this activity</li>
     * </ul>
     * Optional elements:
     * <ul>
     * <li>dims - a Json object defining the activities location - if not included, then the activity will be full screen(inside controller headers/footers)</li>
     * </ul>
     *
     * @param {boolean} noFrame - set to true if you do not want a frame around the activity, which includes a close button
     */
    OU.util.Controller.prototype.addActivity = function(act, noFrame) {
        var newActivity, s = this.section, j = s.activities.length, i = s.sectionId,
                controller = this,
                h = this.h - this.menuHeight - this.headerHeight - this.footerHeight - this.tabHeight - 2 * this.padding,
                y = this.y + this.menuHeight + this.headerHeight + this.tabHeight + this.padding;
        newActivity = s.activities[j] = act;
        newActivity._instName = '_act' + i + '_' + j;
        newActivity._dataLoaded = false;
        newActivity._zOffset = this.zOffset + 100 * (j + 1);
        if (newActivity.type.indexOf('OU.activity') !== 0) {
            newActivity.type = "OU.activity." + newActivity.type;
        }
        this.activityDefs[newActivity._instName] = newActivity;
        if (newActivity.dims) {
            if (this.w > this.h || s.options.maintainLayout === true) {
                newActivity._dims = {
                    x: this.x + this.w * (newActivity.dims.x || 0),
                    y: y + h * (newActivity.dims.y || 0),
                    w: this.w * (newActivity.dims.w || 1),
                    h: h * (newActivity.dims.h || 1)
                };
            }
            else {
                newActivity._dims = {
                    x: this.x + this.w * (newActivity.dims.y || 0),
                    y: y + h * (newActivity.dims.x || 0),
                    w: this.w * (newActivity.dims.h || 1),
                    h: h * (newActivity.dims.w || 1)
                };
            }
        }
        else if (s.options.fullScreen !== undefined && !s.options.fullScreen) {
            newActivity._dims = {
                x: this.x,
                y: y,
                w: this.w,
                h: h
            };
        }
        if (noFrame === undefined || noFrame === false)
            this.activityFrame(newActivity);

        OU.require(newActivity.type); // load the relevant Javscript for this activity
        OU.onLoad(function() { // make sure we wait for activity code to load, including any dependencies
            var fn = eval(newActivity.type);
            OU.runActivityWithData(fn, newActivity.data + 'data.json', newActivity._instName, controller);
        });

        return newActivity;
    };
    /**
     * Removes an activity and any 'removables' with the same zOffset
     * @param {OU.util.Activity} act - a reference to the activity to remove
     */
    OU.util.Controller.prototype.removeActivity = function(act) {
        OU.removeActivity(act._zOffset, this.controllerId);
    };
    /**
     * Renders a 'frame' around a popup activity
     * @param {OU.util.Activity} a - activity associated with the frame
     * @private
     */
    OU.util.Controller.prototype.activityFrame = function(a) {
        // Add window frame
        var self = this;
        a.frame = new OU.util.Div({
            x: a._dims.x - 2 * OU.dpr,
            y: a._dims.y - 42 * OU.dpr,
            w: a._dims.w,
            h: a._dims.h + 40 * OU.dpr,
            style: 'border: 2px solid #444; background: #fff',
            container: {
                zOffset: a._zOffset,
                controller: self // needs reference to self, so that Clean can remove frame on close
            }
        });
        OU._closerCount = (OU._closerCount || 0) + 1;
        // Add Close Icon
        a.frame.div.innerHTML = '<a id="closeCross' + OU._closerCount + '" class="popupClose"></a>';
        var cross = document.getElementById('closeCross' + OU._closerCount);
        if (cross) {
            controllerId = this.controllerId;
            OU.events.addListener(cross, function() {
                OU.removeActivity(a._zOffset, controllerId);
            });
        }
        else {
            console.log('Error finding close cross in add activity frame');
        }
    };

    /**
     * Process an array of API requests if defined
     *
     * @param {object[]} requests - array of API requests (see API function below for spec of requests)
     */
    OU.util.Controller.prototype.processAPI = function(requests) {
        var i;
        if (requests) {
            for (i = requests.length; i--; )
                this.API(requests[i]);
        }
    };
    /**
     * Controller API, facilitates a single interface to handle some controller functions, such as:
     * <ul>
     * <li>Adding/removing activities / sections</li>
     * <li>Inter-Activity messages</li>
     * <li>Changing headers</li>
     * </ul>
     *
     * @param {Object} request - request to process, has the following format:
     * <ul>
     * <li><strong>{string} type</strong>: type of request [activityAPI | header | addActivity | overrideSection]</li>
     * <li><strong>{string} targetInstance</strong>: (optional) a single activity instance to target if request type is activityAPI</li>
     * <li><strong>{string} targetType</strong>: (optional) an activity type to target if request type is activityAPI</li>
     * <li><strong>{object} details</strong>: defines the details of the request</li>
     * </ul>
     *
     */
    OU.util.Controller.prototype.API = function(request) {
        var i, messenger;
        switch (request.type) {
            case 'activityAPI':
                for (i = OU._messengerRegister.length; i--; ) {
                    messenger = OU._messengerRegister[i];
                    if (messenger) {
                        if (request.targetInstance && messenger.instance === request.targetInstance) {
                            request.details.activityId = i;
                            OU._getSetMessengerParamInternal(request.details);
                        }
                        if (request.targetType && messenger.controllerActivity && messenger.controllerActivity.type === request.targetType) {
                            request.details.activityId = i;
                            OU._getSetMessengerParamInternal(request.details);
                        }
                    }
                }
                break;
            case 'header':
                this.header(request.details);
                break;
            case 'addActivity':
                this.addActivity(request.details, !(request.details.frame || true));
                break;
            case 'overrideSection':
                this.overrideSection(request.details);
                break;
        }
    };
    /**
     * Adds or updates the current header - the header is a div which is placed at the top of the page.
     * the div then contains H1 and/or H2 tags
     * @param {object} header - header info with optional h1&h2 elements ie { h1:'Main header', h2:'Sub header'}.
     * If header is undefined, then the current headers are removed.
     */
    OU.util.Controller.prototype.header = function(header) {
        var h = '';
        if (!this.baseController) {
            // The base controller (OU.baseController) is the only controller that can set the header,
            // so if we are a sub controller, then bubble the request up to the base (or outer) controller
            OU.baseController.header(header);
        }
        else {
            if (header) {
                if (!this.headerDiv) {
                    this.headerDiv = new OU.util.Div({
                        container: this,
                        y: this.y,
                        h: "auto", //OU.controlHeight,
                        htmlClass: "OUheader",
                        showScroll: false
                    });
                }
                if (header.h1 && header.h1 !== '')
                    h = h + "<h1>" + header.h1 + "</h1>";
                if (header.h2 && header.h2 !== '')
                    h = h + "<h2>" + header.h2 + "</h2>";
                if (header.h3 && header.h3 !== '')
                    h = h + "<h3>" + header.h3 + "</h3>";
                this.headerDiv.div.innerHTML = h;
                this.headerDiv.resize({
                    h: "auto"
                });
                this.headerHeight = this.headerDiv.div.clientHeight;
            }
            else {
                if (this.headerDiv)
                    this.headerDiv.remove();
                this.headerDiv = null;
                this.headerHeight = 0;
            }
            // call resize to resize all activities underneath this outer controller
            this.resize();
        }
    };
    /**
     * Adds or updates the current audio
     * @param {object} audio -audio definition direct from section data
     */
    OU.util.Controller.prototype.insertAudio = function(audio) {
        var i, h, nTop, bH = OU.controlHeight, self = this, audioElement;
        this.removeAudio();
        if (audio === undefined || audio.sources === undefined)
            return;
        if (this.audioDiv === null) {
            this.audioDiv = new OU.util.Div({
                id: 'audioDiv',
                x: this.x + 40,
                y: this.y + this.h - bH,
                w: this.w / 2,
                h: bH,
                zIndex: OU.CONTROLLER_LEVEL + 1,
                container: this,
                style: 'overflow:hidden; vertical-align:bottom'
            });
        }
        audioElement = document.createElement('audio');
        audioElement.setAttribute('id', '_caud_' + this.instance);
        audioElement.setAttribute('style', 'float:left;');
        audioElement.controls = "controls";
        for (i = audio.sources.length; i--; ) {
            var source = document.createElement('source');
            source.type = audio.sources[i].type;
            source.src = audio.sources[i].src;
            audioElement.appendChild(source);
        }
        audioElement.load();
        this.audioDiv.div.appendChild(audioElement);
        this.audio = audioElement;
        if (!this.audio) {
            return;
        }
        nTop = (bH - this.audio.offsetHeight) / 2;
        nTop = nTop < 0 ? 0 : nTop;
        if (audio.hasSubTitles) {
            if (!this.audioAnchors)
                this.audioAnchors = [];
            this.audioAnchors.push(new OU.util.Anchor({
                container: this,
                parent: this.audioDiv.div,
                classes: this.subtitleson_ ? "subtitleButton subtitlesOn" : "subtitleButton",
                id: "subtitleButton_",
                tabIndex: (20000),
                txt: "Subs",
                onClick: function() {
                    self.toggleSubtitles();
                }
            }));
        }
        if (audio.hasTranscript) {

            if (!this.audioAnchors)
                this.audioAnchors = [];
            this.audioAnchors.push(new OU.util.Anchor({
                container: this,
                parent: this.audioDiv.div,
                classes: "subtitleButton",
                id: "transcriptButton_",
                tabIndex: (20001),
                txt: "Transcript",
                onClick: function() {
                    self.toggleTranscript();
                }
            }));
        }
        this.audioDiv.resize({
            y: this.y + this.h - bH + nTop,
            w: this.audio.offsetWidth + (audio.hasSubTitles ? 60 : 0) + (audio.hasTranscript ? 120 : 0)
        });
        if (audio.cues) {
            audio.cues.sort(function(a, b) {
                return b.start - a.start;
            });
            this.audioListener = this.audio.addEventListener("timeupdate", this.audioListener = function() {
                self.audioTimeUpdate();
            }, false);
        }
        setTimeout(function() {
            self.audio.play();
        }, 1500);
    };
    /**
     * Updates the current audio position. Called by the timeupdate event listener
     * @private
     */
    OU.util.Controller.prototype.audioTimeUpdate = function() {
        this.audioTime = this.audio.currentTime;
    };
    /**
     * Switches subtitles on or off, so that they are either displayed or not if present in the audio data
     * @param {boolean} onoff - true or false (if undefined, then toggled the current state
     */
    OU.util.Controller.prototype.toggleSubtitles = function(onoff) {
        var subButton = document.getElementById("subtitleButton_");
        this.subtitleson_ = onoff === undefined ? !this.subtitleson_ : onoff;
        OU.LocalStorage.save("OU.control.subtitlesOn", this.subtitleson_);
        if (subButton) {
            if (this.subtitleson_)
                subButton.setAttribute("class", "subtitleButton subtitlesOn");
            else
                subButton.setAttribute("class", "subtitleButton");
        }
    };
    /**
     * Loads audio transcript into a div - if the div is already open, then it closes it instead
     */
    OU.util.Controller.prototype.toggleTranscript = function() {
        var s = this.audioStep || 1, c, i, audio = this.section.audio[s - 1], h = "", self = this;
        if (this.transcriptDiv && this.transcriptDiv.html.div) {
            this.transcriptDiv.remove();
            this.transcriptDiv = null;
        }
        else {
            if (audio && audio.cues) {
                for (i = audio.cues.length; i--; ) {
                    c = audio.cues[i];
                    if (c.subtitle)
                        h = h + "<p>" + c.subtitle + "</p>";
                }
                if (h !== "") {
                    this.transcriptDiv = new OU.util.PopUpInfo({
                        container: this,
                        txt: "<h1>Transcript</h1>" + h,
                        onClose: function() {
                            self.transcriptDiv = null;
                        }
                    });
                }
            }
        }
    };
    /**
     * Monitors the current audio position and triggers audio cue points if they are defined in the data.
     * Also displays sub titles if present and the subtitles are switched on.
     */
    OU.util.Controller.prototype.watchAudio = function() {
        var i, cues, s = this.section, c, rc = null, t = this.audioTime, self = this, a, act;
        if (s !== undefined && s.audio !== undefined && s.audio !== null && this.audioStep >= 1 && s.audio[this.audioStep - 1] !== undefined) {
            cues = s.audio[this.audioStep - 1].cues;
            if (this.audioTime !== -1) {
                if (cues !== undefined) {
                    for (i = cues.length; i--; ) {
                        c = cues[i];
                        if (c.start < t) {
                            rc = null;
                            if (c.end === undefined || c.end > t)
                                rc = c;
                        }
                        if (c.start > t)
                            i = 0;
                    }
                    if (rc) {
                        if (rc.activity) {
                            a = s.activities[rc.activity - 1];
                            if (a) {
                                act = a.objRef;
                                if (act && act.step) {
                                    act.step(rc.step); // Call the activity with step N
                                }
                            }
                        }
                        if (s.audio.hasSubTitles && rc) {
                            if (this.subtitleson_ && rc.subtitle) {
                                if (!this.subtitleDiv) {
                                    this.subtitleDiv = new OU.util.Div({
                                        container: this,
                                        x: OU.baseController.x + 20,
                                        y: OU.baseController.y + OU.baseController.h * .8,
                                        w: OU.baseController.w - 40,
                                        h: 'auto',
                                        zIndex: OU.ABSOLUTE_TOP,
                                        htmlClass: "OUSubtitles",
                                        showScroll: false
                                    });
                                }
                                this.subtitleDiv.div.innerHTML = rc.subtitle;
                            }
                            else if (this.subtitleDiv) {
                                this.subtitleDiv.remove();
                                this.subtitleDiv = undefined;
                            }
                        }
                    }
                }
            }
        }
        setTimeout(function() {
            self.watchAudio();
        }, 40);
    };
    /**
     * Removes the current audio if present
     */
    OU.util.Controller.prototype.removeAudio = function() {
        if (this.audio)
            this.audio.pause();
        if (this.audioListener !== undefined)
            this.audio.removeEventListener("timeupdate", this.audioListener, false);
        this.audioTime = -1;
        if (this.audioDiv)
            this.audioDiv.remove();
        this.audioListener = undefined;
        if (this.audioDiv)
            this.audioDiv.div.innerHTML = '';
        this.audioDiv = null;

        if (this.subtitleDiv) {
            this.subtitleDiv.remove();
            this.subtitleDiv = undefined;
        }
        if (this.audioAnchors) {
            for (var i = this.audioAnchors.length; i--; ) {
                this.audioAnchors[i].remove();
            }
            this.audioAnchors.length = 0;
        }
    };
    /**
     * If a section has multiple audios defined, this function moves to a specific audio
     * @param {int} n - the audio number to jump to (if undefined, then jumps to the next one)
     */
    OU.util.Controller.prototype.stepAudio = function(n) { // Jump to Audio
        if (n === undefined)
            n = this.audioStep + 1; // if n not defined, then progress to next
        if (this.audio && this.audio !== undefined && n >= 1)
            this.insertAudio(this.section.audio[n - 1]);
        this.audioStep = n;
    };
    /**
     * Changes the controller section, clears current section and loads a new one.
     * Note this function acts immediately - see chgSection() for a delayed step.
     * @param {int} n - index of the section to change to
     */
    OU.util.Controller.prototype.step = function(n) { // Jump to section n
        var i;
        if (n > this.sections.length)
            n = 1;
        if (n > 0 && n <= this.sections.length) {
            this.clearSection();
            if (this.section !== undefined) {
                for (i = this.section.activityObjects.length; i--; ) {
                    this.section.activityObjects[i] = null;
                }
                this.section.activityObjects.length = 0;
                this.sectionNum = n - 1;
                this.section = this.sections[this.sectionNum];
                this.loadSection();
            }
        }
    };
    /**
     * Clears the current section, removes all open activities that are under this controller.
     * Includes some fairly aggressive deletion code to aid garbage collection
     * @private
     */
    OU.util.Controller.prototype.clearSection = function() {
        var i, s = this.section;
        OU.cleanSection(this);
        if (OU._overrideArray)
            OU._overrideArray.length = 0;
        if (s) {
            for (i = s.activityObjects.length; i--; ) {
                if (s.activityObjects[i] !== null) {
                    OU.obj[s.activityObjects[i].instance] = null;
                    this.activityDefs[s.activityObjects[i].instance] = null;
                    // kill the reference back to the controller...
                    s.activityObjects[i].controller = null;
                    s.activityObjects[i].controllerActivity = null;
                    s.activityObjects[i].data = null;
                    // kill the active object
                    delete s.activityObjects[i];
                    s.activityObjects[i] = null;
                }
            }
            this.section.activityObjects.length = 0;
            for (i = s.activities.length; i--; ) {
                delete s.activities[i].objRef;
                s.activities[i].objRef = null;
            }
        }
    };
    /**
     * Changes the controller section, after a delay.
     * The delay allows a user to select multiple new sections in quick seccession without actually loading each section.
     * Once the user stops changing section and a small delay elapses, then the section is actually changed.
     *
     * @param {int} newTarget - index of the section to change to
     */
    OU.util.Controller.prototype.chgSection = function(newTarget) {
        var self = this;
        if (newTarget < 0 || newTarget > this.sections.length - 1)
            return;
        if (newTarget !== undefined) { //maybe =0 which is valid
            this.sectionNum = newTarget;
            this.render();
            if (this.chgSectionCountdown_ > 80) {
                this.chgSectionCountdown_ = this.chgSectionMaxCountdown;
                return; // don't loop again if already looping
            }
            this.chgSectionCountdown_ = this.chgSectionMaxCountdown;
        }
        else {
            this.chgSectionCountdown_ = this.chgSectionCountdown_ - 40;
        }
        if (this.chgSectionCountdown_ < 40) {
            this.clearSection();
            this.section = this.sections[this.sectionNum];
            this.loadSection();
        }
        else {
            setTimeout(function() {
                self.chgSection();
            }, 40);
        }
    };
    /**
     * A short cut to chgSection - increments section by 1
     */
    OU.util.Controller.prototype.nextSection = function() {
        this.controlLayer.events.flush();
        this.chgSection(this.sectionNum + 1);
    };
    /**
     * A short cut to chgSection - decrements section by 1
     */
    OU.util.Controller.prototype.prevSection = function() {
        this.controlLayer.events.flush();
        this.chgSection(this.sectionNum - 1);
    };
    if (!OU.baseController) {
        OU.baseController = this;
        this.baseController = true;

        // This is the base Controller, so we need to set it's type, as it isn't done elsewhere
        if (this.constructor && this.constructor.toString) { // determine the activities className
            var className = /(OU\..*)\.prototype/.exec(this.constructor.toString())[1];
        }
        this.controllerActivity = {//} this.controller.activities[this.instance]  = {
            data: 'data/',
            type: className || 'Unknown Type'
        }; // push this into the controllers activity list
    }
    OU.base(this, baseControllerData, instance, controller);
};
OU.inherits(OU.util.Controller, OU.util.Activity);
