/**
 * @fileOverview Draggable Object
 *
 * Note: The calling container should provide callback functions to handle the change of state
 *       This object should be pushed into a clickable array to capture events
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @class draggable class -intended to be extended by canvas based object classes that need a draggable behaviour
 * @param {object} params  - parameters to define the class:
 * <ul>
 * <li><strong>{int} x</strong>: x position</li>
 * <li><strong>{int} y</strong>: y position</li>
 * <li><strong>{int} w</strong>: width </li>
 * <li><strong>{int} h</strong>: height</li>
 * <li><strong>{function} onStart</strong>: function to call when a drag starts</li>
 * <li><strong>{function} onMove</strong>: function to call when drag started, and has moved</li>
 * <li><strong>{function} onEnd</strong>: function to call when drag ends</li>
 * <li><strong>{boolean} disabled</strong>: set true to disable the drag functionality</li>
 * <li><strong>{object} me</strong>: reference of parent to pass back in the callback functions</li>
 * </ul>
 */
OU.util.Draggable = function(params) {
    this.events = params.events;
    this.x = params.x;
    this.y = params.y;
    this.h = params.h;
    this.w = params.w;
    this.dragId = 0;
    this.onStart = params.onStart || function() {
    };
    this.onMove = params.onMove || function() {
    };
    this.onEnd = params.onEnd || function() {
    };
    this.me = params.me;
    this.disabled = params.disabled || false;
};
/**
 * Evant handler called by Events, provides the guts of the class
 * @private
 */
OU.util.Draggable.prototype.isHit = function(x, y, pressed) {
    if (this.disabled) {
        return;
    }
    if (pressed) { // mouse down
        if (this.dragId === 0 && !this.events.dragTaken) { // not in drag and drag not already taken by another object  - so start drag if mouse over me
            if (x > this.x && x < (this.x + this.w) && y > this.y && y < (this.y + this.h)) {
                this.dragStart = {
                    "x": x,
                    "y": y
                };
                this.dragId = this.events.dragId;
                this.events.dragTaken = true; // let other draggable object know that this drag is taken
                this.onStart({
                    "me": this.me,
                    "x": this.x,
                    "y": this.y
                });
                return;
            }
        }
        else if (this.dragId === this.events.dragId) { // still in this objects drag so move to new location
            this.x += x - this.dragStart.x;
            this.y += y - this.dragStart.y;
            this.dragStart = {
                "x": x,
                "y": y
            };
            this.onMove({
                "me": this.me,
                "x": this.x,
                "y": this.y
            });
            return;
        }
    }
    // If you get this far, then should not be in a drag or drag is over
    if (this.dragId !== 0) {
        this.dragId = 0;
        this.onEnd({
            "me": this.me,
            "x": this.x,
            "y": this.y
        });
    }
};
/**
 * Resizes the object
 * @param {object} p  - parameters to define the new position:
 * <ul>
 * <li><strong>{int} x</strong>: x position</li>
 * <li><strong>{int} y</strong>: y position</li>
 * <li><strong>{int} w</strong>: width </li>
 * <li><strong>{int} h</strong>: height</li>
 * </ul>
 */
OU.util.Draggable.prototype.resize = function(p) {
    this.x = p.x || 0;
    this.y = p.y || 0;
    this.w = p.w || 100;
    this.h = p.h || 40;
};

