/**
 * @fileoverview  Tessellator Activity
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
OU.require('OU.util.Controller');
OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.Div');
OU.require('OU.util.Menu');
OU.require('OU.util.Instruction');
/**
 * @class Tessellator Activity - A tiled view controller
 *
 * Note, the guts of the controller are in OU.util.Controller.
 * @extends OU.util.Tessellator
 * @param {object} controllerData - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.Tessellator = function(controllerData, instance, controller) { //NOTE: controllerData is intentionally not called data, so as not to be confused with the global variable data, that we use to load activity data.
    this.standardOptions = controllerData.standardOptions;
    this.activityOptions = controllerData.activityOptions;
    this.activityContent = controllerData.activityContent;

    var tessellator = this;
    var newTileset = null;
    var tilesetBreadcrumb = [];
    var clearTiles = false;
    var currentTileset;
    var backIcon;
    var backIconDiv;
    var allowFlips;
    var screenWrap;

    tessellator.SQUARE = 0;
    tessellator.HALF_HOR = 1;
    tessellator.HALF_VERT = 2;
    tessellator.QUARTER = 3;
    tessellator.DOUBLE_HOR = 4;
    tessellator.DOUBLE_VERT = 5;

    tessellator.TILE_ASPECT_RATIO = 1.5;

    /**
     * Starting point
     */
    OU.activity.Tessellator.prototype.canvasView = function() {
        OU.activity.Tessellator.superClass_.canvasView.call(this); // call the parent class canvasView

        OU.addClass(document.body, 'tessellatorBody');

        screenWrap = new OU.util.Div({
            x: 0,
            y: 0,
            w: this.w,
            h: this.h,
            style: "overflow:hidden"
        });
        allowFlips = this.activityOptions.allowFlips === false ? false : true; // default to true if undefined

        currentTileset = this.tiles = this.loadTiles(this.activityContent);

        backIconDiv = new OU.util.Div({
            x: tessellator.w - 100,
            w: 100,
            h: 60,
            zIndex: OU.ABSOLUTE_TOP + 1000,
            htmlClass: 'tessellatorBack',
            showScroll: false
        });
        backIcon = backIconDiv.div;
        OU.events.addListener(backIcon, function() {
            tessellator.back();
        }, 'tap');

        this.render();

        OU.events.addDragListener({
            onDrag: function(dragEvent) {
                tessellator.physics({
                    action: 'drag',
                    dx: dragEvent.dx
                });
            },
            endDrag: function(dragEvent) {
                tessellator.physics({
                    action: 'fling',
                    dx: dragEvent.dx
                });
            }
        });
    };
    /**
     * Handle the physics of the motion (ie. drag, fling, etc.)
     * Note, actual animation is done with CSS transforms
     */
    OU.activity.Tessellator.prototype.physics = function(params) {
        params = params || {};
        console.log('physics: ' + params.action + ' ' + params.dx);
        switch (params.action) {
            default: // no action, do nowt
                break;
            case 'fling':
                // at end of drag, let's simulate a bit of momentum by adding the same jump as the last drag distance
                this.xOffset = this.xOffset + this.lastDragDelta;
                this.lastDragDelta = 0;
                break;
            case 'drag':
                this.xOffset = this.xOffset + params.dx;
                this.lastDragDelta = params.dx;
                break;
            case 'reset':
                this.lastDragDelta = 0;
                this.xOffset = 0;
                break;
        }
        this.render();
    };
    /**
     * Resize the control bar
     */
    OU.activity.Tessellator.prototype.resize = function() {
        OU.activity.Tessellator.superClass_.resize.call(this); // call the parent class resize

        //determine constraints
        this.maxW = this.w * .95; // 95% allows for a glimpse on right side that there is more content
        this.maxH = this.h - this.headerHeight;

        // determine number of columns
        if (this.maxW <= 440) {
            this.visibleCols = 1;
        } else if (this.maxW <= 650) {
            this.visibleCols = 2;
        } else {
            this.visibleCols = 3;
        }
        this.unitWidth = this.maxW / this.visibleCols;
        this.unitHeight = this.unitWidth / tessellator.TILE_ASPECT_RATIO;
        this.visibleRows = (this.maxH / this.unitHeight) | 0;

        this.tileOffsetX = this.x;
        this.tileOffsetY = this.headerHeight + (this.maxH - this.visibleRows * this.unitHeight) / 2;

        if (screenWrap) {
            screenWrap.resize({
                x: 0,
                y: 0,
                w: this.w,
                h: this.h
            });
        }
        if (backIconDiv) {
            backIconDiv.resize({
                x: tessellator.w - 100,
                y: 0,
                w: 100,
                h: 60
            });
        }
        this.physics({
            action: 'reset'
        });
    };
    /**
     * Render the tiles
     */
    OU.activity.Tessellator.prototype.render = function() {
        this.tessellate();
        if (backIcon) {
            if (tilesetBreadcrumb.length > 0) { // we have a breadcrumb trail, so show the back icon
                if (backIcon.style.display !== 'block') {
                    setTimeout(function() {
                        backIcon.style.display = 'block';
                    }, 3000);
                }
            }
            else { // no breadcrumb so hide the back icon
                backIcon.style.display = 'none';
            }
        }
    };
    /**
     * Load the tiles
     */
    OU.activity.Tessellator.prototype.loadTiles = function(tileData) {
        var i, tiles = [];
        for (i = tileData.length; i--; ) {
            tiles.push(new this.Tile(tileData[i]));
        }
        return tiles;
    };

    OU.activity.Tessellator.prototype.back = function() {
        this.clearSection();
        if (tilesetBreadcrumb.length > 0) {
            newTileset = tilesetBreadcrumb[tilesetBreadcrumb.length - 1];
            tilesetBreadcrumb.pop();
        }
        this.render();
    };
    /**
     * Tessellate the tiles
     */
    OU.activity.Tessellator.prototype.tessellate = function() {
        var i, tiles = currentTileset;
        if (tiles) {
            // remove all tiles from placement
            for (i = tiles.length; i--; ) {
                tiles[i].displace();
            }
        }
        if (newTileset) {
            tiles = currentTileset = newTileset;
            newTileset = null;
        }
        if (clearTiles) {
            tiles = currentTileset = null;
            clearTiles = false;
        }
        if (tiles) {
            this.rightBorder = 0;
            this.maxRight = 0;

            // determine positions of all tiles
            for (i = tiles.length; i--; ) {
                tiles[i].position(tiles);
            }

            if (this.rightBorder < this.w) {
                this.tileOffsetX = (this.w - this.rightBorder) / 2;
                this.xOffset = 0;
            }
            else {
                if (this.xOffset !== 0) {
                    if (this.xOffset < -(this.maxRight - this.w + 40)) {
                        this.xOffset = -(this.maxRight - this.w + 40);
                    }
                    if (this.xOffset > 0) {
                        this.xOffset = 0;
                    }
                }
            }
//            console.log('xOffset:' + this.xOffset);
            // actually move the tiles into place.
            for (i = tiles.length; i--; ) {
                tiles[i].place();
            }
        }
    };
    /**
     * @class Tile - holds a single tile
     */
    OU.activity.Tessellator.prototype.Tile = function(data) {
        var self = this;
        this.titleHTML = data.titleHTML;
        this.activityPopup = data.activityPopup;
        this.controllerAPI = data.controllerAPI;
        this.img = data.img || "";
        this.div = new OU.util.Div({
            x: -tessellator.w, // place new div off screen - it will be moved when it's tessellated
            y: 0,
            w: tessellator.w / 10,
            h: tessellator.h / 10,
            htmlClass: 'tessellatorTile',
            showScroll: false,
            parentDiv: screenWrap
        });
        switch (data.shape) {
            default:
            case tessellator.SQUARE:
            case "SQUARE":
                this.unitWidth = 1;
                this.unitHeight = 1;
                this.shape = tessellator.SQUARE;
                this.shapeClass = "TessSquare";
                break;
            case tessellator.HALF_HOR:
            case "HALF_HOR":
                this.unitWidth = 1;
                this.unitHeight = 0.5;
                this.shape = tessellator.HALF_HOR;
                this.shapeClass = "TessHalfHor";
                break;
            case tessellator.HALF_VERT:
            case "HALF_VERT":
                this.unitWidth = 0.5;
                this.unitHeight = 1;
                this.shape = tessellator.HALF_VERT;
                this.shapeClass = "TessHalfVert";
                break;
            case tessellator.QUARTER:
            case "QUARTER":
                this.unitWidth = 0.5;
                this.unitHeight = 0.5;
                this.shape = tessellator.QUARTER;
                this.shapeClass = "TessQuarter";
                break;
        }
        this.div.html('<img src="data/' + this.img + '"/><div>' + this.titleHTML + '</div>');

        OU.events.addListener(this.div.div, function() {
            self.hit();
        }, 'tap');

        if (data.tiles) { // if this tile has sub tiles, then recursively load them
            this.tiles = tessellator.loadTiles(data.tiles);
        }
        OU.activity.Tessellator.prototype.Tile.prototype.hit = function() {
            var tile = this;
            if (this.tiles) { // has sub tiles, so load 'em
                tilesetBreadcrumb.push(currentTileset);
                newTileset = this.tiles;
                tessellator.physics({
                    action: 'reset'
                });
            }
            else {
                if (this.controllerAPI) {
                    tilesetBreadcrumb.push(currentTileset);
                    clearTiles = true;
                    tessellator.render();
                    setTimeout(function() {
                        tessellator.processAPI(tile.controllerAPI);
                    }, 1000);
                }
                if (this.activityPopup) {
                    tilesetBreadcrumb.push(currentTileset);
                    clearTiles = true;
                    tessellator.render();
                    setTimeout(function() {
                        tessellator.addActivity(tile.activityPopup);
                    }, 1000);
                }
            }
        };
        /**
         * Slot in the tile to the first gap we find.
         * Attempt to place as far left as possible with a preference for the top.
         */
        OU.activity.Tessellator.prototype.Tile.prototype.position = function(tiles) {
            var xu = 0, yu, rightBorder, firstPosition, found = false;
            this.w = this.unitWidth * tessellator.unitWidth;
            this.h = this.unitHeight * tessellator.unitHeight;
            do {
                for (yu = 0; yu < tessellator.visibleRows; yu += 0.5) {
                    this.x = xu * tessellator.unitWidth;
                    this.y = yu * tessellator.unitHeight;
                    if (this.y + this.h <= tessellator.unitHeight * tessellator.visibleRows) {

                        if (!this.collides(tiles)) {
                            rightBorder = this.x + this.w;
                            found = true;
                            firstPosition = {
                                x: this.x,
                                y: this.y,
                                w: this.w,
                                h: this.h,
                                xu: xu
                            };

                            yu = tessellator.visibleRows + 1;
                        }
                    }
                }
                xu += 0.5;
            }
            while (!found && xu < 1000); // should never get this far
            this.div.removeClass("flipped");
//
            // If it's a half type, then try it flipped to see if it fits further left
            if (allowFlips && firstPosition && this.x > 0 && (this.shape === tessellator.HALF_HOR || this.shape === tessellator.HALF_VERT)) {
                found = false;
                xu = 0;
                this.w = this.unitHeight * tessellator.unitWidth;
                this.h = this.unitWidth * tessellator.unitHeight;
                do {
                    for (yu = 0; yu < tessellator.visibleRows; yu += 0.5) {
                        this.x = xu * tessellator.unitWidth;
                        this.y = yu * tessellator.unitHeight;
                        if (this.y + this.h <= tessellator.unitHeight * tessellator.visibleRows) {
                            if (!this.collides(tiles)) {
                                if (this.x + this.w < rightBorder) {
                                    //Flipping tile so flip class
                                    this.div.addClass("flipped");

                                    firstPosition = {
                                        x: this.x,
                                        y: this.y,
                                        w: this.w,
                                        h: this.h,
                                        xu: xu
                                    };
                                    rightBorder = this.x + this.w;
                                    yu = tessellator.visibleRows + 1;
                                    found = true;
                                }
                            }
                        }
                    }
                    xu += 0.5;
                }
                while (!found && xu <= firstPosition.xu + 1); // should never get this far
            }
            this.div.addClass(this.shapeClass);

            if (firstPosition) {
                this.x = firstPosition.x;
                this.y = firstPosition.y;
                this.w = firstPosition.w;
                this.h = firstPosition.h;
            }
            else {
                this.x = 0;
                this.y = 0;
            }

            this.isPositioned = true;

            if (rightBorder > tessellator.rightBorder) {
                tessellator.rightBorder = rightBorder;
            }

            var newX = tessellator.tileOffsetX + this.x + 6,
                    newW = this.w - 12,
                    farRight = newX + newW;
            if (farRight > tessellator.maxRight) {
                tessellator.maxRight = farRight;
            }
        };
        OU.activity.Tessellator.prototype.Tile.prototype.place = function() {
            var newX = tessellator.tileOffsetX + tessellator.xOffset + this.x + 6,
                    newY = tessellator.tileOffsetY + this.y + 6,
                    newW = this.w - 12,
                    newH = this.h - 12;
            this.div.resize({
                x: newX,
                y: newY,
                w: newW,
                h: newH
            });
            this.div.addClass('visible');
        };
        OU.activity.Tessellator.prototype.Tile.prototype.displace = function() {
            this.div.resize({
                x: -tessellator.w,
                y: 0,
                w: tessellator.w / 10,
                h: tessellator.h / 10
                        /*
                         y: tessellator.tileOffsetY + this.y + 6,
                         w: this.w - 12,
                         h: this.h - 12
                         //*/
            });
            this.div.removeClass('visible');
            this.isPositioned = false;
        };
        /**
         * Does this tile collide with any others that are already placed?
         * @returns {boolean}
         */
        OU.activity.Tessellator.prototype.Tile.prototype.collides = function(tiles) {
            var i, comparedTile;
            for (i = tiles.length; i--; ) {
                comparedTile = tiles[i];
                if (comparedTile.isPositioned) {
                    if (!(this.x + this.w <= comparedTile.x || comparedTile.x + comparedTile.w <= this.x
                            || this.y + this.h <= comparedTile.y || comparedTile.y + comparedTile.h <= this.y)) {
                        return true;
                    }
                }
            }
            return false;
        };
    };

    OU.base(this, controllerData, instance, controller);
};
if (OU.util.Controller) { // If superclass is already defined, then we can call inherits
    OU.inherits(OU.activity.Tessellator, OU.util.Controller);
}
else { // if controller class is no defined yet, then add the inherits call to the preOnLoad queue
    OU.preOnLoad.push(function() {
        OU.inherits(OU.activity.Tessellator, OU.util.Controller);
    });
}
