/**
 * @fileOverview varControlBank - controller for a row of varControls
 *
 * A bit clunky as it covers various usage options
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.varControl');
/**
 * @class A Bank of VarControls, allows the user to view/modify a set of variables using optional sliders, and buttons.
 * The order of the Var Controls can also be changed using a draggable section for each varcontroller if required.
 * 
 * @param {object} params - options:
 * <ul>
 * <li><strong>{double} x:</strong> X Co-ordinate</li>
 * <li><strong>{double} y:</strong> Y Co-ordinate</li>
 * <li><strong>{double} w:</strong> Width</li>
 * <li><strong>{double} h:</strong> Height (Note this is the button height and does not include the height of the pop-up elements)</li>
 * <li><strong>{int} tabIndex :</strong> Tab index, as the controls are inherently tabbable</li>
 * <li><strong>{boolean} fillWidth:</strong> set true to stop width being restricted, default:false</li>
 * <li><strong>{object[]} vars:</strong> Array of Variables, each must be an object with a field "val" that holds its value</li>
 * <li><strong>{boolean} valInButton:</strong> Set true to display the value of each control in its button, default:false</li>
 * <li><strong>{function} callback:</strong> Callback function that is called whenever the value changes.</li>
 * <li><strong>{OU.util.Layer} layer:</strong> Layer to render to</li>
 * <li><strong>{object} parent:</strong> Reference to the calling object, typically an OU.util.Activity</li>
 * <li><strong>{boolean} hasOnOff:</strong> If true include an On/Off button, default:false</li>
 * <li><strong>{boolean} hasSlider:</strong> If true, include a slider, default:false </li>
 * <li><strong>{boolean} hasVal:</strong> If true display the value as an element in the pop-up, default:false</li>
 * <li><strong>{boolean} hasGrip:</strong> If true, include a draggable grip area, default:false</li>
 * </ul>
 */
OU.util.varControlBank = function ( params ) {
    this.params = params;
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || 300;
    this.h = params.h || 100;
    this.tabIndex = params.tabIndex || 100;
    this.fillWidth = params.fillWidth || false;
    this.vars = params.vars || [];
    this.valInButton = params.valInButton || false;
    this.stackH = 0;
    this.numVars = this.vars.length;
    this.layer = params.layer;
    this.doRender = false;
    if (params.hasOnOff!==undefined)
        this.hasOnOff = params.hasOnOff;
    else
        this.hasOnOff = false;
    if (params.hasSlider!==undefined)
        this.hasSlider = params.hasSlider;
    else
        this.hasSlider = true;
    if (params.hasVal!==undefined)
        this.hasVal = params.hasVal;
    else
        this.hasVal = true;
    if (params.hasGrip!==undefined)
        this.hasGrip = params.hasGrip;
    else
        this.hasGrip = false;
    this.init();
};
/**
 * Init the bank and all the varControls
 * @private
 */
OU.util.varControlBank.prototype.init = function () {
    var i;
    this.controlW = this.w / this.numVars;
    if (!this.fillWidth && this.controlW > 80)
        this.controlW = 80;
    if (this.hasOnOff)
        this.stackH = this.stackH + this.h * .8;
    if (this.hasSlider)
        this.stackH = this.stackH + this.h * 3;
    if (this.hasVal)
        this.stackH = this.stackH + this.h * .8;
    if (this.hasGrip)
        this.stackH = this.stackH + this.h * .8;
    // render variable controls
    this.controls = [];
    for (i = this.numVars; i--;) {
        this.controls.push(new OU.util.varControl({
            x:i * this.controlW + this.x,
            y:this.y,
            w:this.controlW,
            h:this.h,
            stackH:this.stackH,
            layerOffset:this.params.layerOffset,
            variable:this.vars[i],
            valInButton:this.valInButton,
            layer:this.layer,
            instance:'cont' + i,
            bank:this,
            clickable:this.params.clickable,
            callback:this.params.callback,
            hasGrip:this.params.hasGrip,
            hasVal:this.params.hasVal,
            hasSlider:this.params.hasSlider,
            hasOnOff:this.params.hasOnOff,
            parent:this.params.parent,
            sortOrder:i,
            tabIndex: this.tabIndex+i*10
        }));
    }
    this.w = this.numVars * this.controlW;
    if (this.params.hasGrip) {
        var cBank = this;
        setInterval(function () {
            if (cBank.doRender)
                cBank.performRender();
            cBank.params.callback(cBank.params.parent);
        }, 20);
    }
};
/**
 * Resizes the bank and controls
 * @param {object} dims - new dimensions
 */
OU.util.varControlBank.prototype.resize = function ( dims ) {
    var i;
    this.x = dims.x || this.x || 0;
    this.y = dims.y || this.y || 0;
    this.w = dims.w || this.w || 300;
    this.h = dims.h || this.h || OU.controlHeight;
    this.controlW = this.w / this.numVars;
    if (!this.fillWidth && this.controlW > 80)
        this.controlW = 80;
    this.w = this.numVars * this.controlW;
    if (this.hasOnOff)
        this.stackH = this.stackH + this.h * .8;
    if (this.hasSlider)
        this.stackH = this.stackH + this.h * 3;
    if (this.hasVal)
        this.stackH = this.stackH + this.h * .8;
    if (this.hasGrip)
        this.stackH = this.stackH + this.h * .8;
    for (i = this.numVars; i--;) {
        this.controls[i].resize({
            x:(this.numVars - i - 1) * this.controlW + this.x,
            y:this.y,
            w:this.controlW,
            h:this.h,
            layerOffset:dims.layerOffset
        });
    }
};
/**
 * Renders the bank using the appropriate method depending on set up.
 */
OU.util.varControlBank.prototype.render = function () {
    if (this.hasGrip)
        this.doRender = true;
    else
        this.performRender();
};
/**
 * Actually render the controls
 * @private
 */
OU.util.varControlBank.prototype.performRender = function () {
    var control, i, numControls = this.controls.length, shuffling = false;
    this.layer.clear(this.x, this.y - this.stackH, this.w, this.h + this.stackH);
    if (this.params.hasGrip) {
        // render layer controls
        for (i = numControls; i--;) {
            // animate shuffle if out of place
            control = this.controls[i];
            if (control.draggable.dragId==0) { // item is not being dragged, so shuffle to appropriate position
                var targetX = control.sortOrder * this.controlW;
                if (control.x!=targetX) {
                    shuffling = true;
                    var newX = control.x + (targetX - control.x) / 2;
                    if (Math.abs(targetX - newX) < 0.2)
                        newX = targetX;
                    control.move(newX);
                }
            }
            else {
                shuffling = true;
            }
            // render item
            control.render();
        }
        if (!shuffling) {
            this.doRender = false;
        }
    }
    else {
        for (i = numControls; i--;)
            this.controls[i].render();
    }
};
/**
 * Closes all the controls
 */
OU.util.varControlBank.prototype.close = function () {
    if (this.controls.length > 0)
        this.controls[0].closeControls();
};

