/*
 * @fileoverview 2D Vector object, provides an easy way to get useful info about 2 vectors.
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

/**
 * @class 2D Vector object
 * @param {object} points - defines two points in the vector ie (x1,y1) and (x2,y2)
 */
OU.util.Vector2D = function(points) {
    var r2d = 180 / Math.PI;
    this.x1=points.x1 || 0;
    this.x2=points.x2 || 0;
    this.y1=points.y1 || 0;
    this.y2=points.y2 || 0;

    this.dX = this.x2-this.x1; // difference in X points (delta X)
    this.dY = this.y2-this.y1; // difference in Y points (delta Y)

    this.opposite = Math.abs(this.dY); // length of 'opposite' in trigonometry terms
    this.adjacent = Math.abs(this.dX); // length of 'adjacent' in trigonometry terms
    this.hypotenuse = Math.sqrt(this.opposite * this.opposite + this.adjacent * this.adjacent); // length of 'hypotenuse' in trigonometry terms

    // definition of the vector in terms of the equation: Y=aX+b
    this.a = this.dY/this.dX;
    this.b = this.y1 - (this.a*this.x1);

    // Determine the angle of this vector compared to the horizontal
    // ie. in terms of the trigonometry angle of the triangle formed by this vector and the horizontal
    // Note this is always positive
    if (this.hypotenuse > 0) {
        this.trigRadians = Math.asin(this.opposite / this.hypotenuse);
        this.trigDegrees = this.trigRadians * r2d;
    }

    // Determine the angle of the vector in relation to "12 O'Clock"
    // Value between 0 and 2xPI radians (or 0 and 360 degrees)
    if(this.x1>this.x2) {
        if(this.y1>this.y2) {
            this.radians = this.trigRadians+Math.PI*1.5;
        }
        else {
            this.radians = Math.PI*1.5 - this.trigRadians;
        }
    }
    else {
        if(this.y1>this.y2) {
            this.radians = Math.PI*0.5-this.trigRadians;
        }
        else {
            this.radians = this.trigRadians+Math.PI*0.5;
        }
    }
    this.degrees = this.radians * r2d;

    /**
     * returns the Y value at a given x
     * @param {double} x - in X vae
     * @returns {double} Y value
     */
    OU.util.Vector2D.prototype.Yvalue = function(x) {
        return this.a*x + this.b;
    };
    /**
     * Calculates the angle between this vector and another vector
     * @param {OU.util.Vector2D} vector - the vector to compare to
     * @returns {object} angle in degrees and radians ie. {degrees:180, radians: Math.PI}
     */
    OU.util.Vector2D.prototype.angleToVector = function(vector) {
        var degrees,radians;
        if(this.degrees>vector.degrees) {
            radians=this.radians-vector.radians;
        }
        else  {
            radians=vector.radians-this.radians;
        }
        if(radians>Math.PI)
            radians = Math.PI*2 - radians;
        degrees = radians*r2d;

        if(degrees>90) {
            degrees = 180-degrees;
            radians = Math.PI-radians;
        } //*/
        return {
            radians: radians,
            degrees: degrees
        };
    };
    /**
     * Calcultes the intersection point and angle between this vector and another
     * @param {OU.util.Vector2D} vector - the vector to compare to
     * @returns {object} intersection ie {x:10,y:10, angle: {degrees:180, radians: Math.PI} }
     */
    OU.util.Vector2D.prototype.intersectPoint = function(vector) {
        var x,smallAngle,bigAngle;
        if(this.a-vector.a === 0)
            return null;
        x = (vector.b-this.b)/(this.a-vector.a);
        smallAngle = this.angleToVector(vector);
        bigAngle = 180-smallAngle;
        return {
            x: x,
            y: this.Yvalue(x),
            angle: smallAngle,
            oppositeAngle:bigAngle
        };
    };
};

