/**
 * @fileOverview Drag2Group - user drags a serious of items onto various group boxes
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Instruction');
OU.require('OU.util.Draggable');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
OU.require('OU.util.ModalMessage');
OU.require('OU.util.ModalInfo');

KEY_LEFT = 37;
KEY_UP = 38;
KEY_RIGHT = 39;
KEY_DOWN = 40;

/**
 * @class Drag2Group - user drags a serious of items onto various group boxes
 * @extends OU.util.Activity
 * @param {object} data - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.Drag2Group = function(data, instance, controller) {
    var allowLimitMessage = false;
    /**
     * Starting point when running with HTML5 Canvas Support
     */
    OU.activity.Drag2Group.prototype.canvasView = function() {
        var self = this, bH = OU.controlHeight;
        this.showHelper = false;
        this.bgLayer = new OU.util.Layer({
            container: this
        });
        this.bgLayer.canvas.style.backgroundColor = this.data.activityOptions.backgroundColour;
        this.messageLayer = new OU.util.Layer({
            container: this
        });
        this.itemsLayer = new OU.util.Layer({
            container: this,
            hasEvents: true
        });
        this.itemsLayer.context.font = "bold 20px " + OU.theme.font;
        if (this.data.preload === true) {
            this.controlLayer = new OU.util.Layer({
                x: this.x + this.w / 2 - bH,
                y: this.y + this.h - bH,
                w: bH * 2,
                h: bH,
                container: this,
                hasEvents: true
            });
        }
        allowLimitMessage = this.data.activityOptions.allowLimitMessage || false; // defaults to false if not present
        this.images = {};
        this.images.data = this.data;
        this.images.image = this.data.activityOptions.background;
        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.images,
            onLoad: function() {
                self.start();
                self.started = true;
            }
        });
        this.helper = {
            correct: new Image(),
            incorrect: new Image()
        };
        this.helper.correct.src = OU.cssPath + 'correct.png';
        this.helper.correct.onload = function() {
        };
        this.helper.incorrect.src = OU.cssPath + 'incorrect.png';
        this.helper.incorrect.onload = function() {
        };
    };
    /**
     * resize all layers,
     * re-init the scale,
     * re-render
     */
    OU.activity.Drag2Group.prototype.resize = function() {
        OU.activity.Drag2Group.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight;
        if (!this.started)
            return;
        this.bgLayer.resize();
        this.bgLayer.canvas.style.backgroundColor = this.data.activityOptions.backgroundColour;
        this.messageLayer.resize();
        this.itemsLayer.resize();
        if (this.data.preload === true) {
            this.controlLayer.resize({
                x: this.x + this.w / 2 - bH,
                y: this.y + this.h - bH,
                w: bH * 2,
                h: bH
            });
            this.doneButton.render();
        }
        this.initScale();
        for (var i = this.visibles.length; i--; ) {
            this.visibles[i].resize();
        }
        this.doRender = true;
    };
    /**
     * Ensure render loop stops on removal
     */
    OU.activity.Drag2Group.prototype.remove = function() {
        OU.activity.Drag2Group.superClass_.remove.call(this); // call the superclass method
        this.doRender = false;
    };
    /**
     * initScale - calculates the scale factor of the activity based on the current dimensions.
     * Then resize the groups.
     */
    OU.activity.Drag2Group.prototype.initScale = function() {
        var i, j, g, item, bg = this.images.imgElement,
                fitW = this.w / bg.width,
                fitH = this.h / bg.height,
                s = this.scale = (fitW < fitH ? fitW : fitH),
                x = (this.w - bg.width * s) / 2,
                y = (this.h - bg.height * s) / 2;
        this.bgLayer.context.drawImage(bg, x, y, bg.width * s, bg.height * s);
        if (this.data._doneResize === undefined) {
            this.data._doneResize = true;
            for (i = this.groups.length; i--; ) { // scale group dimensions to background
                g = this.groups[i];
                g._x = g.x;
                g._y = g.y;
                g._w = g.w;
                g._h = g.h;
                g.c = '#0f0';
            }
        }
        for (i = this.groups.length; i--; ) { // scale group dimensions to background
            g = this.groups[i];
            g.x = x + g._x * s;
            g.y = y + g._y * s;
            g.w = g._w * s;
            g.h = g._h * s;
            // scale items if they are image based
            for (j = g.items.length; j--; ) {
                item = g.items[j];
                if (item.imgElement) {
                    item.w = item.imgElement.width * this.scale;
                    item.h = item.imgElement.height * this.scale;
                }
            }
        }
    };
    /**
     * Starts the activity after the image loader has finished
     */
    OU.activity.Drag2Group.prototype.start = function() {
        var self = this, bH = OU.controlHeight, i, j, g, item,
                ctx = this.itemsLayer.context;
        var font = new OU.font({
            size: 20,
        });
        ctx.font = font.str();

        //Init data
        this.visibles = []; // onscreen items
        this.invisibles = []; // offscreen items
        this.items = [];
        this.groups = [];
        for (j = this.data.activityContent.length; j--; ) {
            g = this.data.activityContent[j];
            g.handle = {};
            this.groups.push(g);
            for (i = g.items.length; i--; ) {
                item = g.items[i];
                item.group = g;
                item.visible = false;
                if (item.imgElement) {
                    item.w = item.imgElement.width * this.scale;
                    item.h = item.imgElement.height * this.scale;
                }
                else {
                    item.w = ctx.measureText(item.label, 0, 0).width + 20;
                    item.h = 30;
                }
                this.items.push(item);
            }
        }
        this.initScale();
        // randomise order of the items
        this.items.sort(function() {
            return (Math.round(Math.random()) - 0.5);
        });


        if (this.data.preload === true) {
            this.doneButton = new OU.util.ControlButton({
                txt: "I'm Done",
                x: 0,
                y: 0,
                w: bH * 2,
                h: bH,
                layer: this.controlLayer,
                onClick: function() {
                    self.feedback();
                }
            });
        }
        this.restart();
    };
    /**
     * Resets activity to the starting point.
     * Shows the instructions, and starts the render loop.
     */
    OU.activity.Drag2Group.prototype.restart = function() {
        var targetX, targetY, shrinkArea, infoSize = 2 * OU.controlHeight;
        this._feedbackRendered = false;

        // populate the invisibles array
        this.populateInvisibles();

        if (this.data.infoButtonLocation === 'topRight')
            shrinkArea = {
                x: this.w - infoSize,
                y: OU.controlHeight,
                w: infoSize,
                h: infoSize
            };
        if (this.data.infoButtonLocation === 'bottomRight')
            shrinkArea = {
                x: this.w - infoSize,
                y: this.h - infoSize,
                w: infoSize,
                h: infoSize
            };
        if (this.data.infoButtonLocation === 'bottomLeft')
            shrinkArea = {
                x: OU.controlHeight,
                y: this.h - infoSize,
                w: infoSize,
                h: infoSize
            };
        if (this.data.preload === true) {
            this.popAllItems();
            if (this.data.instructions !== undefined) {
                new OU.util.Instruction({
                    container: this,
                    message: this.data.instructions,
                    shrinkArea: shrinkArea // defaults to topLeft if undefined
                });
            }
        }
        else {
            this.visibles.length = 0;
            this.popItem(); // add first item to the screen
            if (this.data.defaultStartingPosition !== undefined && this.data.defaultStartingPosition.x !== 0) {
                targetX = (this.data.defaultStartingPosition.x * this.scale);
                targetY = (this.data.defaultStartingPosition.y * this.scale);
            }
            else {
                targetX = this.w / 2;
                targetY = 75;
            }
            if (this.data.instructions !== undefined) {
                new OU.util.Instruction({
                    container: this,
                    message: this.data.instructions,
                    duration: this.data.instructionDuration,
                    point: {
                        x: targetX,
                        y: targetY
                    },
                    shrinkArea: shrinkArea // defaults to topLeft if undefined
                });
            }
        }
        // start render loop
        this.doRender = true;
        this.renderLoop();
    };
    /**
     * Generates an Item object for every item in the list and pushes to the invisibles array
     */
    OU.activity.Drag2Group.prototype.populateInvisibles = function() {
        var i, item, x, y;
        this.visibles.length = 0;
        this.invisibles.length = 0;
        for (i = this.items.length; i--; ) {
            item = this.items[i];
            if (item.init !== undefined) {
                x = item.init.x * this.scale;
                y = item.init.y * this.scale;
            }
            else {
                x = (this.w - item.w) / 2;
                y = 10;
            }
            this.invisibles.push(new this.Item({
                "events": this.itemsLayer.events,
                "item": item,
                "sortOrder": this.popped,
                "order": this.popped,
                "name": item.label,
                "color": '#fcfcfc',
                "x": x,
                "y": y,
                "w": item.w,
                "h": item.h,
                parent: this
            }));
        }
    };
    /**
     * Pushes all the items from the items queue (array) into the visibles array,
     * effectively putting them on screen
     */
    OU.activity.Drag2Group.prototype.popAllItems = function() {
        this.visibles = this.invisibles;
        this.invisibles = [];
    };
    /**
     * Pushes the next item in the items queue(array) into the visibles array,
     * making it visible and draggable onscreen
     */
    OU.activity.Drag2Group.prototype.popItem = function() {
        var item = this.invisibles.pop();
        if (item === undefined) {
            return;
        }
        this.visibles.push(item);
    };
    /**
     * There can be only one!
     * We only want 1 item ungrouped, so put all others back in the invisibles array
     */
    OU.activity.Drag2Group.prototype.highlander = function() {
        for (var i = this.visibles.length; i--; ) {
            if (!this.visibles[i].g) {
                var item = this.visibles.splice(i, 1);
                this.invisibles.push(item[0]);
            }
        }
        this.popItem();
    };
    /**
     * Renders the current view, and loops every 40 milliseconds, to enable animation of items
     */
    OU.activity.Drag2Group.prototype.renderLoop = function() {
        var self = this;
        if (this.itemsLayer.context) { // only do render loop if context is still valid, otherwise the activity has been killed
            if (this.doRender)
                this.render();
            setTimeout(function() {
                self.renderLoop();
            }, 40);
        }
    };
    /**
     * calculates the distance from the centre of an item to the centre of a group
     * @param {object} item
     * @param {object} group
     */
    OU.activity.Drag2Group.prototype.distance = function(item, group) { // returns the distance between the centres of 2 objects
        var iX = item.x + (item.w / 2),
                iY = item.y + (item.h / 2),
                gX = group.x + (group.w / 2),
                gY = group.y + (group.h / 2),
                dx = gX - iX,
                dy = gY - iY;
        return Math.sqrt(dx * dx + dy * dy);
    };
    /**
     * Calculates the closest group, by finding shortest distance to the given item
     * @param {object} item
     */
    OU.activity.Drag2Group.prototype.nearestGroup = function(item) {
        var closest = null, i, g, d, shortest = 99999999;
        for (i = this.groups.length; i--; ) { // reset group counts
            g = this.groups[i];
            if (this.collide(item, g)) {
                d = this.distance(item, g);
                if (d < shortest) {
                    shortest = d;
                    closest = g;
                }
            }
        }
        return closest;
    };
    /**
     * Determines if 2 objects collide.
     * Both objects should have x,y,w,h elements
     * @param {object} a
     * @param {object} b
     */
    OU.activity.Drag2Group.prototype.collide = function(a, b) {
        var ArX = a.x + a.w,
                AbY = a.y + a.h,
                BrX = b.x + b.w,
                BbY = b.y + b.h,
                collide = true;
        if (a.x > BrX)
            collide = false;
        if (ArX < b.x)
            collide = false;
        if (a.y > BbY)
            collide = false;
        if (AbY < b.y)
            collide = false;
        return collide;
    };
    /**
     * Outputs a message to the messageLayer.
     * Parameters are used to initalise the message.
     * The function is then called without parameters for each render loop,
     * to re-display the message for a given length of time.
     * (Not sure why I did it like this - its a bit clunky!)
     * @param {string} message
     * @param {double} duration - amount of time to show the message
     */
    OU.activity.Drag2Group.prototype.message = function(message, duration) {
        var ctx = this.messageLayer.context,
                now = new Date().getTime();
        if (message) {
            this._message = message;
            this._messageStart = new Date().getTime();
            this._duration = duration || 2000;
        }
        if (this._message === undefined)
            return;
        ctx.clearRect(0, 0, this.w, this.h);
        if (now - this._messageStart > this._duration)
            return;
        ctx.globalAlpha = 1 - (now - this._messageStart) / this._duration;
        new OU.util.DynText({
            txt: this._message,
            context: ctx,
            x: this.w / 4,
            y: this.h * .75,
            w: this.w / 2,
            h: this.h * .2,
            background: {
                col: '#ccc',
                radius: 20
            }
        });
        this.doRender = true;
    };
    /**
     * Renders the current view.
     * Also performs the animation of items as it iterates through them to render them.
     */
    OU.activity.Drag2Group.prototype.render = function() {
        // render dragged items
        var events = this.itemsLayer.events,
                visible, g, i, j, targetX, targetY, newX, newY, colW,
                inGroups = 0, shuffling = false;
        this.itemsLayer.clear();
        events.clickable.length = 0;
        for (i = this.groups.length; i--; ) { // reset group counts
            g = this.groups[i];
            g.nItems = 0;
            g.rItems = 0;
            g.totalW = 0;
            g.totalH = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
            g.runningH = 0;
            g.columns = 0;
            g.column = 0;
            g.lastColumn = 0;
            g.colItems = 0;
            /* Debug - Show where groups are
             var ctx = this.itemsLayer.context;
             ctx.rect(g.x,g.y,g.w,g.h);
             ctx.fillStyle='#c00';
             ctx.fill();
             // */
        }
        for (i = this.visibles.length; i--; ) {
            visible = this.visibles[i];
            // animate shuffle if out of place
            visible.g = null; //this.groups[this.groups.length-1];
            visible.itmN = 0;
            visible.colN = 0;
            if (visible.draggable.dragId === 0 && !visible.hasFocus) { // item is not being dragged (or has Tab Focus), so shuffle to appropriate position
                for (j = this.groups.length; j--; ) {
                    g = this.groups[j];
                    if ((this.collide(g, visible) && !visible.lock) || g.handle === visible.lock) {
                        if (!visible.lock) {
                            g = this.nearestGroup(visible);
                        }
                        if (g.maxItems === undefined || g.maxItems < 1 || g.nItems < g.maxItems) {
                            visible.g = g;
                            visible.lock = g.handle;
                            visible.itmN = g.nItems;
                            g.nItems++;
                            //g.c='#f00';
                            inGroups++;
                        }
                        else {
                            //g.c='#000';
                            if (allowLimitMessage) {
                                this.message('Only ' + g.maxItems + ' item' + (g.maxItems > 1 ? 's' : '') + ' allowed in this area');
                            }
                        }
                        j = 0;
                    }
                }
            }
            else {
                shuffling = true;
            }
            //            this.bgLayer.context.strokeStyle=g.c;
            //            this.bgLayer.context.strokeRect(g.x,g.y,g.w,g.h);
        }
        var maxH = -1;
        for (i = this.visibles.length; i--; ) {
            visible = this.visibles[i];
            if (visible.h > maxH)
                maxH = visible.h;
            if (visible.draggable.dragId === 0) {
                g = visible.g;
                if (g !== null) {
                    g.totalW = g.totalW + visible.w;
                    if (g.totalH[g.columns] + visible.h > g.h && visible.h < g.h) {
                        g.columns++;
                    }
                    g.totalH[g.columns] = g.totalH[g.columns] + visible.h;
                }
            }
        }
        //   var s=this.scale;
        for (i = this.visibles.length; i--; ) {
            visible = this.visibles[i];
            if (visible.draggable.dragId === 0 && !visible.hasFocus) {
                g = visible.g;
                if (g !== null) {
                    colW = g.w / (g.columns + 1);
                    if (g.runningH + visible.h > g.h && visible.h < g.h) {
                        g.column++;
                        g.runningH = 0;
                    }
                    targetX = g.x + (g.column * colW) + (colW - visible.w) / 2;
                    targetY = g.y + ((g.h - g.totalH[g.column]) / 2) + g.runningH;
                    g.runningH = g.runningH + visible.h + 10;
                    g.rItems++;
                }
                else {
                    if (this.data.defaultStartingPosition !== undefined && this.data.defaultStartingPosition.x !== 0) {
                        targetX = (this.data.defaultStartingPosition.x * this.scale) - visible.w / 2;
                        targetY = (this.data.defaultStartingPosition.y * this.scale) - visible.h / 2;
                    }
                    else {
                        targetX = (this.w - visible.w) / 2;
                        targetY = 10 + visible.h / 2;
                    }
                }
                if (visible.x !== targetX || visible.y !== targetY) {
                    shuffling = true;
                    newX = visible.x + (targetX - visible.x) / 2;
                    newY = visible.y + (targetY - visible.y) / 2;
                    if (Math.abs(targetX - newX) < 0.2)
                        newX = targetX;
                    if (Math.abs(targetY - newY) < 0.2)
                        newY = targetY;
                    visible.move(newX, newY);
                }
            }
            visible.render();
        }
        if (this.data.preload !== true && inGroups === this.visibles.length) {
            this.popItem();
        }
        if (inGroups < this.visibles.length - 1) { // too many items out of groups
            this.highlander(); // There can be only 1 item ungrouped
        }
        if (!shuffling) {
            this.doRender = false;
        }
        this.message();
        if (this.data.preload !== true && inGroups >= this.items.length)
            this.feedback();
    };
    /**
     * Show feedback once all items are placed in groups.
     */
    OU.activity.Drag2Group.prototype.feedback = function() {
        var i, item, incorrect = 0;
        this.showHelper = true;

        for (i = this.visibles.length; i--; ) {
            item = this.visibles[i];
//            item.render();
            if (item.g !== item.item.group) {
                incorrect++;
            }
        }

        if (incorrect > 0) {
            if (!this._feedbackRendered) {
                new OU.util.ModalMessage({
                    html: '<div style="padding: 10px; font-size: 20px; text-align: center;">' + this.data.activityOptions.feedback.incorrect.join('<br />') + '</div>',
                    timeout: ((typeof this.data.activityOptions.feedback.timeout !== 'undefined') ? this.data.activityOptions.feedback.timeout : 5000)
                });
                this._feedbackRendered = true;
            }
        }
        else {
            if (!OU.modalInfo) {
                OU.modalInfo = new OU.util.ModalInfo();
            }
            OU.modalInfo.view({
                html: this.data.activityOptions.feedback.correct,
                width: this.w * .66
            });
        }

    };
    /**
     * @class Item object class - each draggable item is handled by this class
     * @extends Tabbable
     * @param {object} params - various parameters to initialise the item
     */
    OU.activity.Drag2Group.prototype.Item = function(params) {
        this.events = params.events;
        this.item = params.item;
        this.lock = false;
        this.order = params.order;
        this.sortOrder = params.sortOrder;
        this.parent = params.parent;
        this.container = this.parent.container;
        this.name = params.name;
        this.color = params.color;
        this.x = params.x;
        this.y = params.y;
        this.origW = params.w;
        this.origH = params.h;
        this.imgElement = params.item.imgElement;
        this.startX = this.x;
        this.startY = this.y;
        OU.activity.Drag2Group.prototype.Item.prototype.resize = function() {
            var scale = this.parent.scale;
            if (this.imgElement) {
                this.w = this.imgElement.width * scale;
                this.h = this.imgElement.height * scale;
            }
            else {
                this.w = this.origW * scale;
                this.h = this.origH * scale;
            }
            if (this.draggable) {
                this.draggable.w = this.w;
                this.draggable.h = this.h;
            }
        };
        /**
         * renders the item, as either an image or text depending on data given
         */
        OU.activity.Drag2Group.prototype.Item.prototype.render = function() {
            var showFeedback = this.parent.showHelper && this.g,
                    ctx = this.parent.itemsLayer.context, bg,
                    sF = 1, //(this.g === null ? 1.4 : (this.draggable.dragId !== 0 ? 1.4 : 1)),
                    x = this.x - (this.w * (sF - 1) / 2),
                    y = this.y - (this.h * (sF - 1) / 2),
                    w = this.w * sF,
                    h = this.h * sF;
            if (this.imgElement !== undefined) {
                ctx.drawImage(this.imgElement, x, y, w, h);
            }
            else {
                if (this.hasFocus) {
                    bg = {
                        "borderCol": "#444",
                        "col": '#c00',
                        "radius": h / 2
                    };
                }
                else {
                    bg = {
                        "borderCol": "#444",
                        "col": this.color,
                        "radius": h / 2
                    };
                }
                new OU.util.DynText({
                    "txt": this.name,
                    "x": x,
                    "y": y,
                    "w": w,
                    "h": h,
                    "context": ctx,
                    background: bg,
                    fontWeight: "bold",
                    fontSize: 14 * OU.dpr
                });
            }

            if (showFeedback) {
                ctx.drawImage(this.parent.helper[((this.g === this.item.group ? '' : 'in') + 'correct')], x + w, y + ((h / 2) - (((h < 15) ? h : 15) / 2)), ((h < 15) ? h : 15), ((h < 15) ? h : 15));
            }

            this.parent.itemsLayer.events.clickable.push(this.draggable);
        };
        /**
         * Focus function is called when the item receives Tab focus.
         * Sets focus state to true, and re-renders
         */
        OU.activity.Drag2Group.prototype.Item.prototype.focus = function() {
            this.lock = false;
            this.hasFocus = true;
            this.render();
            return false;
        };
        /**
         * Blur function is called when the item loses Tab focus.
         * Sets focus state to false, and re-renders
         */
        OU.activity.Drag2Group.prototype.Item.prototype.blur = function() {
            this.hasFocus = false;
            this.render();
            return false;
        };
        /**
         * arrowKey function is called if the item has Tab focus and an arrow key is pressed.
         * Moves the item accordingly and makes the parent activity render
         * @param {int} k - the key code of the arrow key pressed
         */
        OU.activity.Drag2Group.prototype.Item.prototype.arrowKey = function(k) {
            switch (k) {
                case KEY_UP:
                    this.move(this.x, this.y - 10);
                    break;
                case KEY_DOWN:
                    this.move(this.x, this.y + 10);
                    break;
                case KEY_LEFT:
                    this.move(this.x - 10, this.y);
                    break;
                case KEY_RIGHT:
                    this.move(this.x + 10, this.y);
                    break;
            }
            this.parent.doRender = true;
        };
        /**
         * startDrag is called when the item is "picked up", ie starts to be dragged
         */
        OU.activity.Drag2Group.prototype.Item.prototype.startDrag = function(p) {
            //sort the items so that the latest dragged is at the end of the queue, therefore render last next time (on top)
            p.me.parent.visibles.sort(function(a, b) {
                return b.draggable.dragId - a.draggable.dragId;
            });
            p.me.parent.view = 'norm';
            p.me.parent.doRender = true;
        };
        /**
         * endDrag is called when the item is "dropped", ie when it is released from drag
         */
        OU.activity.Drag2Group.prototype.Item.prototype.endDrag = function(p) {
            p.me.lock = false;
            p.me.parent.doRender = true;
        };
        /**
         * newPos, called when in mid drag to update to the latest position
         * @param {object} p - new position
         */
        OU.activity.Drag2Group.prototype.Item.prototype.newPos = function(p) {
            p.me.x = p.x;
            p.me.y = p.y;
            p.me.doRender = true;
        };
        /**
         * Moves the item to X,Y co-ordinates - also moves the corresponding draggable item to keep them in-sync
         * @param {double} x
         * @param {double} y
         */
        OU.activity.Drag2Group.prototype.Item.prototype.move = function(x, y) {
            this.x = x;
            this.y = y;
            this.draggable.x = x;
            this.draggable.y = y;
        };
        this.resize();
        this.draggable = new OU.util.Draggable({
            "me": this,
            "events": this.parent.itemsLayer.events,
            "x": this.x,
            "y": this.y,
            "h": this.h,
            "w": this.w,
            "onStart": this.startDrag,
            "onEnd": this.endDrag,
            "onMove": this.newPos
        });
        OU.base(this, params);
    };
    OU.inherits(OU.activity.Drag2Group.prototype.Item, OU.util.Tabbable);

    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Drag2Group, OU.util.Activity);
