/**
 * @fileOverview TileRotation - provides a zoomable view of a series of images that have been tiled into smaller images
 *               with the option of zooming in
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @class TileRotation - provides a zoomable view of a series of images that have been tiled 
 *              
 * @param {object} params - options:
 * <ul>
 * <li><strong>{object} container:</strong> (required) The calling object, typically an OU.util.Acitivity </li>
 * <li><strong>{canvas.context} context:</strong> (required) Context of the canvas/layer to render to</li>
 * <li><strong>{object} window:</strong> defines the dimensions with elements</li>
 * <ul>
 * <li><strong>{int} x:</strong> X co-ordinate</li>
 * <li><strong>{int} y:</strong> Y co-ordinate</li>
 * <li><strong>{int} w:</strong> Width</li>
 * <li><strong>{int} h:</strong> Height</li>
 * </ul>
 * <li><strong>{object} image:</strong> dimensions of the original image before being cut into tiles, with:</li>
 * <ul>
 * <li><strong>{int} w:</strong> Width</li>
 * <li><strong>{int} h:</strong> Height</li>
 * </ul>
 * <li><strong>{object} imageBase:</strong> a json structure that defines the images (see data.js format)</li>
 * <li><strong>{int} zoomLevels:</strong> (required) number of zoom levels in the zoomify files</li>
 * <li><strong>{boolean} useTileGroups:</strong> true if the Zoomify files are in TileGroup folders, defaults to true</li>
 * <li><strong>{int} tileSize:</strong> size of the tiles in pixels, defaults to 256</li>
 * <li><strong>{function} zoomCallback:</strong> Callback function, called with new zoom value, ie to set a slider in the calling activity</li>
 * </ul>
 */
OU.util.TileRotation = function ( params ) {
    var NOTLOADED = 0, LOADING = 1, LOADED = 2;
    this.context = params.context;
    this.container = params.container;
    this.imageBase = params.imageBase || [];
    this.numImages = this.imageBase.length;
    this.imageIdx = 0;
    this.window = params.window || { // viewing window on the canvas
        w:1024,
        h:768
    };
    this.v = {}; // viewport of the image (section of the image being displayed)
    this.dims = params.image || {}; // image dims
    this.useTileGroups=params.useTileGroups===undefined?true:params.useTileGroups;
    this.tileSize = params.tileSize || 256;
    this.nZ = params.zoomLevels || 0; // total zoom levels
    this.z = 0; // current zoom level
    this.s = 0; // scale (0 <= s <=1) - % of total zoom range
    this.x = this.y = 0; // offsets
    this.doRender = false;
    this.zoomCallback = params.zoomCallback || function () {
    };
    this.hasDragged = false;
    this.inDrag = false;
    this.mouseDown = false;
    this.history = [];
    this.inertia = {
        x:0,
        y:0
    };
    this.dataDir = this.container.dataDir;
    /**
     * @private
     */
    OU.util.TileRotation.prototype.init = function () {
        var fn,fileType,zoomLevels=this.nZ,squareTileSize = this.tileSize * Math.pow(2, zoomLevels - 1),
        i, z, x, y, nX = 1,nY=1, fitWidth, fitHeight,
        p2m = this.imageBase[this.imageIdx].pixelsPerMM,
        URLVars = OU.getUrlVars(),tileGroup=0,tGCount=0,
        uncroppedW,uncroppedH,pX,pY;

        if (!this.dims.w || this.dims.w < 1 || !this.dims.h || this.dims.h < 1) {// if image size not specified, calculate from the max zoom & tile size
            this.dims.w = this.dims.h = squareTileSize;
        }
        pX = this.propX = this.dims.w/squareTileSize;
        pY = this.propY = this.dims.h/squareTileSize;

        uncroppedW = this.uncroppedW = (this.dims.w);
        uncroppedH = this.uncroppedH = (this.dims.h);

        fitWidth = this.window.w / (uncroppedW);
        fitHeight = this.window.h / (uncroppedH);
        
            this.viewAllScale = fitWidth < fitHeight ? fitWidth : fitHeight;

        this.scaleFactor = 1 - this.viewAllScale; // factor to normalise zoom scale to a range of 0 to 1
        if (URLVars['zoom']) {
            this.s = parseFloat(URLVars['zoom']);
            this.s = this.s < 0 ? 0 : (this.s > 1 ? 1 : this.s);
        }
        if (URLVars['x']) {
            this.x = this.container.w / 2 - parseFloat(URLVars['x']) * (p2m * (this.s * this.scaleFactor + this.viewAllScale));
        }
        else {
            this.x = (this.window.w - uncroppedW * this.viewAllScale) / 2;
        }
        if (URLVars['y']) {
            this.y = (this.container.h - OU.controlHeight) / 2 - parseFloat(URLVars['y']) * (p2m * (this.s * this.scaleFactor + this.viewAllScale));
        }
        else {
            this.y = (this.window.h - uncroppedH * this.viewAllScale) / 2;
        }

        //set up array of image place holders
        for (i = this.numImages; i--;) {
            nX = nY = 1;
            tileGroup=tGCount=0;
            this.imageBase[i]._tiles = [];
            fileType = this.imageBase[i].fileType || '.jpg';
            for (z = 0; z < zoomLevels; z++) {
                this.imageBase[i]._tiles[z] = [];
                for (x = 0; x<(nX*pX)+1; x++) {
                    this.imageBase[i]._tiles[z][x] = [];
                }
                nX = nX * 2;
            }
            nX = nY = 1;

            for (z = 0; z < zoomLevels; z++) {
                for (y = 0;y<(nY*pY); y++) {
                    for (x = 0; x<(nX*pX); x++) {
                        if(this.useTileGroups){
                            fn=this.dataDir + this.imageBase[i].fileBase + '/TileGroup'+tileGroup+'/'+z + '-' + x + '-' + y + fileType;
                            tGCount++;
                            if(tGCount>255) {
                                tGCount=0;
                                tileGroup++;
                            }
                        }
                        else {
                            fn=this.dataDir + this.imageBase[i].fileBase + z + '_' + x + '_' + y + fileType;
                        }
                        this.imageBase[i]._tiles[z][x][y] = {
                            file:fn,
                            loadState:NOTLOADED
                        };
                    }
                }
                nX = nX * 2;
                nY = nY * 2;
            }
            this.loadTile(this.imageBase[i]._tiles[0][0][0]);
        }
        //load initial image
        this._tiles = this.imageBase[this.imageIdx]._tiles;
        this.rough = this._tiles[0][0][0].image;
        if(this.dims.w>this.dims.h)
            this.roughSF = this.rough.width/(this.tileSize);
        else
            this.roughSF = this.rough.height/(this.tileSize);
        this.context.font = '18px ' + OU.theme.font;
        this.context.lineWidth = 2;
        this.context.strokeStyle = '#c00';
        this.context.fillStyle = '#c00';
        this.context.textAlign = 'center';
        this.renderCycle();
        this.purge();
    };
    /**
     * Changes the image (this is what makes the rotation effect)
     * @param {int} i - index of the image
     */
    OU.util.TileRotation.prototype.changeImage = function ( i ) {
        this.imageIdx = i < 0 ? this.numImages - 1 : i > this.numImages - 1 ? 0 : i;
        this._tiles = this.imageBase[this.imageIdx]._tiles;
        this.rough = this._tiles[0][0][0].image;
        this.doRender = true;
    };
    /**
     * Enable Loading - allows tiles to be loaded
     */
    OU.util.TileRotation.prototype.enableLoading = function () {
        this.disableLoad = false;
        this.doRender=true;
    };
    /**
     * Disable Loading - stops tiles being loaded - used when rotation ZoomRotation objects
     */
    OU.util.TileRotation.prototype.disableLoading = function () {
        this.disableLoad = true;
    };
    /**
     * Loads a specific tile
     * @param {object} tile - a "tile" object
     * @private
     */
    OU.util.TileRotation.prototype.loadTile = function ( tile ) {
        if (tile.loadState==NOTLOADED && this.disableLoad!==true) {
            var self = this;
            tile.loadState = LOADING;
            tile.image = new Image();
            tile.image.src = tile.file;
            tile.image.onload = function () {
                tile.loadState = LOADED;
                tile.width=tile.image.width;
                tile.height=tile.image.height;
                self.doRender = true;
            };
            tile.image.onerror = function(e) {
            };
        }
    };
    /**
     *  Purges images from memory if they haven't been used in the last second - iPad soon crashes if you don't do this
     *  @private
     */
    OU.util.TileRotation.prototype.purge = function () {
        var self = this, i, t, nI, z, x, y, tile, now = new Date().getTime();
        for (i = this.numImages; i--;) {
            nI = 1;
            t = this.imageBase[i]._tiles;
            for (z = 0; z < this.nZ; z++) {
                for (x = nI; x--;) {
                    for (y = nI; y--;) {
                        tile = t[z]?t[z][x]?t[z][x][y]:undefined:undefined;
                        if (tile!==undefined && tile.image!==undefined && tile.loadState==LOADED) {
                            if (tile.lastUsed < now - 1000 && (z!=0 || x!=0 || y!=0)) {
                                tile.lastUsed = now;
                                tile.loadState = NOTLOADED;
                                tile.image = undefined;
                            }
                        }
                    }
                }
                nI = nI * 2;
            }
        }
        setTimeout(function () {
            self.purge();
        }, 40);
    };
    /**
     * Event handler function
     * @private
     */
    OU.util.TileRotation.prototype.isHit = function ( x, y, evState, ev ) {
        var dX, dY,
        uncroppedW = this.uncroppedW,
        uncroppedH = this.uncroppedH,
        s = this.s * this.scaleFactor + this.viewAllScale,
        xO = this.x,
        yO = this.y;
        if (evState!==undefined)
            this.mouseDown = evState;
        if (this.mouseDown) {
            if (this.inDrag) {
                if (this.measureOn) {
                    this.renderMeasure = true;
                    this.measureEndX = (x - xO) / s;
                    this.measureEndY = (y - yO) / s;
                }
                else {
                    dX = x - this.dragStartX;
                    dY = y - this.dragStartY;
                    this.x = (this.x + dX) % (uncroppedW * s);
                    this.y = (this.y + dY) % (uncroppedH * s);
                    this.dragStartX = x;
                    this.dragStartY = y;
                }
                this.doRender = true;
            }
            else {
                this.inertia = {
                    x:0,
                    y:0
                };
                this.dragStartX = x;
                this.dragStartY = y;
                this.inDrag = true;
                if (this.measureOn) {
                    this.measureStartX = (x - xO) / s;
                    this.measureStartY = (y - yO) / s;
                }
            }
        }
        else {
            if (this.inDrag && this.history.length > 0) {
                this.inertia = {  // if mouse/touch just ended, then apply inertia
                    x:(this.x - this.history[0].x) / 4,
                    y:(this.y - this.history[0].y) / 4
                };
            }
            this.inDrag = false;
            if (this.circleOn)
                this.doRender = true;
        }
    };
    /**
     * Stops any movement and rendering.
     * Don't think this is actually used any more
     */
    OU.util.TileRotation.prototype.stop = function () {
        this.inertia = {
            x:0,
            y:0
        };
        this.doRender = false;
    };
    /**
     * Sets new scale
     * @param {double} scale - new scale value between 0 and 1
     * @param {object} centre - point to keep as centre of the zoom change in form eg. {x:100,y:200}
     */
    OU.util.TileRotation.prototype.scale = function ( scale, centre ) {
        var ns = scale > 1 ? 1 : (scale < 0 ? 0 : scale), // ensure new scale is within limits
        nnS = ns * this.scaleFactor + this.viewAllScale,
        noS = this.s * this.scaleFactor + this.viewAllScale,
        dS = nnS / noS,
        cX = this.window.w / 2, // center zoom on middle of window
        cY = this.window.h / 2, offX, offY;
        if (centre) { // if zooming via pinch, then center on touch point
            cX = centre.x;
            cY = centre.y;
            this.history.length = 0;// zero history so that pinch release doesn't trigger inertia movement
        }
        offX = cX - this.x,
        offY = cY - this.y;
        this.x += offX - (offX * dS); // set new position to maintain zoom center
        this.y += offY - (offY * dS);
        this.s = ns; // set new Scale
        this.doRender = true;
    };
    /**
     * Render cycle, also performs the physics of the movement after touch
     * @private
     */
    OU.util.TileRotation.prototype.renderCycle = function () {
        var self = this,s,
        uncroppedW = (this.dims.w),
        uncroppedH = (this.dims.h);
        if (this.history.length > 4)
            this.history.shift();
        this.history.push({
            x:this.x,
            y:this.y
        });
        if (!this.mouseDown && (this.inertia.x!=0 || this.inertia.y!=0)) {
            s = this.s * this.scaleFactor + this.viewAllScale;
            this.x = (this.x + this.inertia.x) % (uncroppedW * s);
            this.y = (this.y + this.inertia.y) % (uncroppedH * s);
            this.inertia.x = this.inertia.x * .9;
            this.inertia.y = this.inertia.y * .9;
            this.inertia.x = this.inertia.x < 2 && this.inertia.x > -2 ? 0 : this.inertia.x;
            this.inertia.y = this.inertia.y < 2 && this.inertia.y > -2 ? 0 : this.inertia.y;
            this.doRender = true;
        }
        if (this.doRender)
            this.render();
        setTimeout(function () {
            self.renderCycle();
        }, 40);
    };
    /**
     * Renders the current view
     * @private
     */
    OU.util.TileRotation.prototype.render = function () {
        var nX,nY, x1, x2, y1, y2, x, y, ax, tile, rts, tX, tY,tW,tH,zoomLevels=this.nZ,now = new Date().getTime(),
        ctx = this.context,
        uncroppedW = this.uncroppedW,
        uncroppedH = this.uncroppedH,
        s = this.s * this.scaleFactor + this.viewAllScale, // factored scale between 100% and max zoom
        z = (zoomLevels * ((1 - this.viewAllScale) * s + this.viewAllScale)) | 0; // calcs relevant zoom level for best quality images
        z = z >= zoomLevels - 1 ? zoomLevels - 1 : z;

        if(!this.rough)
            return;

        if(this.dims.w<this.dims.h) {
            do {
                z++;
                nX = Math.pow(2, z); // number of tiles across or down @ zoom level
                rts = (uncroppedW / nX)/this.propX; // relative tile size @ zoom level
            }
            while (rts * s > 1.05 * this.tileSize && z < zoomLevels-1);
            if(z==zoomLevels)
                nX=nX/2;
            nY = (nX*uncroppedH/uncroppedW) | 0;
            rts = (uncroppedW / nX)/this.propX; // relative tile size @ zoom level
        }
        else {
            do {
                z++;
                nY = Math.pow(2, z); // number of tiles across or down @ zoom level
                rts = (uncroppedH / nY)/this.propY; // relative tile size @ zoom level
            }
            while (rts * s > 1.05 * this.tileSize && z < zoomLevels-1);
            if(z==zoomLevels)
                nY=nY/2;
            nX = (nY*uncroppedW/uncroppedH) | 0;
            rts = (uncroppedH / nY)/this.propY; // relative tile size @ zoom level
        }
        this.z = z = z >= zoomLevels-1 ? zoomLevels-1 : z;

        if (this.y > 0)
            this.y = 0;
        if (this.y < this.window.h - this.dims.h * s)
            this.y = this.window.h - this.dims.h * s;

        // calc viewport dims & location
        this.v.x = -this.x / s;
        this.v.y = -this.y / s;
        this.v.w = this.window.w / s;
        this.v.h = this.window.h / s;

        x1 = (this.v.x / rts) | 0;
        x2 = (((this.v.x + this.v.w) / rts) | 0) + 1;
        y1 = (this.v.y / rts) | 0;
        y2 = (((this.v.y + this.v.h) / rts) | 0) + 1;

        x1 = x1 < 0 ? 0 : x1 > nX - 1 ? nX - 1 : x1;
        x2 = x2 < 0 ? 0 : x2 > nX - 1 ? nX - 1 : x2;

        y1 = y1 < 0 ? 0 : y1 > nY - 1 ? nY - 1 : y1;
        y2 = y2 < 0 ? 0 : y2 > nY? nY: y2;
        ctx.clearRect(0, 0, this.window.w, this.window.h);

        ctx.drawImage(this.rough, this.x, this.y, uncroppedW* s, uncroppedH* s);

        for (x = x1; x <= x2; x++) { // render images if loaded, otherwise load relevant tiles
            for (y = y1; y <= y2; y++) {
                ax = x;
                ax = ax < 0 ? ax + nX : ax;
                if (this._tiles[z][ax]) {
                    tile = this._tiles[z][ax][y];
                    if (tile) {
                        if (tile.loadState==LOADED) {
                            tile.lastUsed = now;
                            tX = this.x + x * rts * s;
                            tY = this.y + y * rts * s;
                            tW= (tile.width/this.tileSize)*rts * s;
                            tH= (tile.height/this.tileSize)*rts * s;
                            ctx.drawImage(tile.image, tX, tY, tW, tH);
                        }
                        else if (tile.loadState==NOTLOADED) {
                            this.loadTile(tile);
                        }
                    }
                }
            }
        }
        this.doRender = false;
    };
    this.init();
};
