/**
 * @fileOverview Microscope Rotation x4
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.Slider');
OU.require('OU.util.Layer');
OU.require('OU.util.TileViewer');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Rotation4X = function ( data, instance, controller ) {
    OU.activity.Rotation4X.prototype.canvasView = function () {
        this.bgLayer = new OU.util.Layer({
            container:this,
            id:'bg'
        });
        this.bgLayer.context.fillStyle = '#fff';
        this.bgLayer.context.fillRect(0, 0, this.w, this.h);
        OU.rotations = [];
        this.leftRotation = new this.Rotation({
            rotation:this.data.rotations[0],
            x:0,
            w:this.w / 2,
            h:this.h,
            mscope:this
        });
        this.rightRotation = new this.Rotation({
            rotation:this.data.rotations[1],
            x:this.w / 2,
            w:this.w / 2,
            h:this.h,
            mscope:this
        });
    };
    OU.activity.Rotation4X.prototype.resize = function () {
        OU.activity.Rotation4X.superClass_.resize.call(this); // call the parent class resize 
        this.init();
        this.bgLayer.resize();
    };
    OU.activity.Rotation4X.prototype.Rotation = function ( params ) {
        var rotation = this;
        this.rotationIdx = OU.rotations.length;
        OU.rotations.push(this);
        this.mscope = params.mscope;
        this.x = params.x;
        this.title = params.rotation.title || '';
        this.pplData = params.rotation.pplData || '';
        this.xplData = params.rotation.xplData || '';
        this.slideWidth = params.w * .9;
        this.slideHeight = (params.h / 2) * .9;
        this.slideWidth = this.slideHeight = this.slideWidth < this.slideHeight ? this.slideWidth : this.slideHeight;
        this.rightX = this.leftX = (this.mscope.w / 2 - this.slideWidth) / 2;
        this.leftY = (this.mscope.h * .5 - this.slideHeight) / 2;
        this.rightY = this.mscope.h * .5 + (this.mscope.h * .5 - this.slideHeight) / 2;
        this.leftCenterX = this.leftX + this.slideWidth / 2;
        this.rightCenterX = this.rightX + this.slideWidth / 2;
        this.leftCenterY = this.leftY + this.slideHeight / 2;
        this.rightCenterY = this.mscope.h * .5 + this.leftY + this.slideHeight / 2;
        this.numberOfImagesInFullRotation = params.rotation.numberOfImagesInFullRotation || 72;
        this.reuseCount = params.rotation.reuseCount || 2;
        this.degrees = 0;
        OU.activity.Rotation4X.prototype.Rotation.prototype.init = function () {
            var i, imageLayer, ctx, scaleX, scaleY, scaleInPixels;
            // Create 2 DIVs to contain the left and right image sets
            this.leftBox = document.createElement("div"),
                this.rightBox = document.createElement("div");
            this.leftBox.style.width = this.rightBox.style.width = this.slideWidth + "px";
            this.leftBox.style.height = this.rightBox.style.height = this.slideHeight + "px";
            this.leftBox.style.left = this.x + this.leftX + "px";
            this.rightBox.style.left = this.x + this.rightX + "px";
            this.leftBox.style.top = this.leftY + "px";
            this.rightBox.style.top = this.rightY + "px";
            this.leftBox.style.zIndex = this.rightBox.style.zIndex = OU.POP_UP_LEVEL + 1;
            this.leftBox.className = this.rightBox.className = "microscopeRotation";
            document.body.appendChild(this.leftBox);
            document.body.appendChild(this.rightBox);
            // Load the images into the divs, incrementing the rotation per slide
            this.segmentAngle = 2.0 * Math.PI / this.numberOfImagesInFullRotation;
            for (i = 0; i < this.numberOfImagesInFullRotation / this.reuseCount; i++) {
                // Add PPL image
                imageLayer = document.createElement("img");
                imageLayer.setAttribute("width", rotation.slideWidth);
                imageLayer.setAttribute("height", rotation.slideHeight);
                imageLayer.style.visibility = "hidden";
                imageLayer.src = this.pplData + (((i + 1) < 10) ? "0" : "") + (i + 1) + ".jpg";
                imageLayer.style.zIndex = OU.POP_UP_LEVEL + 1 + i;
                this.leftBox.appendChild(imageLayer);
                // Add XPL image
                imageLayer = document.createElement("img");
                imageLayer.setAttribute("width", rotation.slideWidth);
                imageLayer.setAttribute("height", rotation.slideHeight);
                imageLayer.style.visibility = "hidden";
                imageLayer.src = this.xplData + (((i + 1) < 10) ? "0" : "") + (i + 1) + ".jpg";
                imageLayer.style.zIndex = OU.POP_UP_LEVEL + 1 + i;
                this.rightBox.appendChild(imageLayer);
            }
            // Add a control layer to capture touch/mouse events & contain other visual elements
            this.controlLayer = new OU.util.Layer({
                container:this.mscope,
                zIndex:OU.POP_UP_LEVEL + this.numberOfImagesInFullRotation + 10,
                hasEvents:true,
                x:this.x,
                w:this.mscope.w / 2,
                h:this.mscope.h
            });
            ctx = this.controlLayer.context;
            ctx.font = 'bold ' + this.slideWidth * .05 + 'px ' + OU.theme.font;
            this.controlLayer.events.clickable.push(this);
            // Title
            ctx.textAlign = 'left';
            ctx.fillText(this.title, 5, this.mscope.h * .05);
            // scale
            scaleInPixels = 234 * (this.slideWidth / 460);
            scaleY = this.mscope.h * .5;
            scaleX = this.mscope.w / 4 - scaleInPixels / 2;
            ctx.save();
            ctx.beginPath();
            ctx.textAlign = 'center';
            ctx.fillText('1mm', this.mscope.w / 4, scaleY - 10);
            ctx.moveTo(scaleX, scaleY - 10);
            ctx.lineTo(scaleX, scaleY + 10);
            ctx.moveTo(scaleX, scaleY);
            ctx.lineTo(scaleX + scaleInPixels, scaleY);
            ctx.moveTo(scaleX + scaleInPixels, scaleY - 10);
            ctx.lineTo(scaleX + scaleInPixels, scaleY + 10);
            ctx.lineWidth = 2;
            ctx.strokeStyle = '#000';
            ctx.stroke();
            ctx.restore();
            // Crosshairs
            ctx.beginPath();
            ctx.moveTo(this.leftX, this.leftY + this.slideHeight / 2);
            ctx.lineTo(this.leftX + this.slideWidth, this.leftY + this.slideHeight / 2);
            ctx.moveTo(this.leftX + this.slideWidth / 2, this.leftY);
            ctx.lineTo(this.leftX + this.slideWidth / 2, this.leftY + this.slideHeight);
            ctx.moveTo(this.rightX, this.rightY + this.slideHeight / 2);
            ctx.lineTo(this.rightX + this.slideWidth, this.rightY + this.slideHeight / 2);
            ctx.moveTo(this.rightX + this.slideWidth / 2, this.rightY);
            ctx.lineTo(this.rightX + this.slideWidth / 2, this.rightY + this.slideHeight);
            ctx.strokeStyle = '#fff';
            ctx.stroke();
            this.doRender = true;
        };
        OU.activity.Rotation4X.prototype.Rotation.prototype.renderCycle = function () {
            if (this.doRender)
                this.performRender();
            var rotIdx = this.rotationIdx;
            setTimeout(function () {
                OU.rotations[rotIdx].renderCycle();
            }, 20);
        };
        OU.activity.Rotation4X.prototype.Rotation.prototype.render = function () {
            this.doRender = true;
        };
        OU.activity.Rotation4X.prototype.Rotation.prototype.performRender = function () {
            var i, leftLayer, rightLayer, rotationDegrees = this.degrees, temp,
                r2d = Math.PI / 180, ctx = this.controlLayer.context,
                nImgs = this.numberOfImagesInFullRotation / this.reuseCount,
                indexOfLowerLayer = parseInt(this.degrees / this.segmentAngle),
                indexOfUpperLayer = (indexOfLowerLayer + 1),
                opacityOfLowerLayer = 1,
                opacityOfUpperLayer = (this.degrees - indexOfLowerLayer * this.segmentAngle) / this.segmentAngle;
            this.doRender = false;
            indexOfLowerLayer = indexOfLowerLayer % nImgs;
            indexOfUpperLayer = indexOfUpperLayer % nImgs;
            if (indexOfLowerLayer > indexOfUpperLayer) { // Swap order if required
                temp = indexOfUpperLayer;
                indexOfUpperLayer = indexOfLowerLayer;
                indexOfLowerLayer = temp;
                opacityOfUpperLayer = 1 - opacityOfUpperLayer;
            }
            // step through slides and set visibility,opacity & rotation accordingly
            for (i = nImgs; i--;) {
                leftLayer = this.leftBox.childNodes[i];
                rightLayer = this.rightBox.childNodes[i];
                if (i==indexOfLowerLayer) {
                    if (leftLayer.style.visibility!="visible")
                        leftLayer.style.visibility = rightLayer.style.visibility = "visible";
                    leftLayer.style.webkitTransform = rightLayer.style.webkitTransform = "rotate3d(0,0,1," + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.MozTransform = rightLayer.style.MozTransform = "rotate(" + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.opacity = rightLayer.style.opacity = opacityOfLowerLayer;
                } else if (i==indexOfUpperLayer) {
                    if (leftLayer.style.visibility!="visible")
                        rightLayer.style.visibility = leftLayer.style.visibility = "visible";
                    leftLayer.style.webkitTransform = rightLayer.style.webkitTransform = "rotate3d(0,0,1," + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.MozTransform = rightLayer.style.MozTransform = "rotate(" + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.opacity = rightLayer.style.opacity = opacityOfUpperLayer;
                }
                else {
                    if (leftLayer.style.visibility!="hidden")
                        rightLayer.style.visibility = leftLayer.style.visibility = "hidden";
                }
            }
            ctx.clearRect(this.mscope.w / 16, this.leftY + this.slideWidth * .9, 40, this.slideWidth * .1);
            ctx.fillText((this.degrees / r2d | 0) + '°', this.mscope.w / 16, this.leftY + this.slideWidth * .95);
        };
        OU.activity.Rotation4X.prototype.Rotation.prototype.isHit = function ( x, y, evState ) {
            var dX, dY, oldTheta, newTheta;
            if (evState) {
                if (y < this.rightY)
                    dY = y - this.leftCenterY;
                else
                    dY = y - this.rightCenterY;
                if (x < this.rightX)
                    dX = x - this.leftCenterX;
                else
                    dX = x - this.rightCenterX;
                this.distance = Math.sqrt(dX * dX + dY * dY);
                if (this.distance > this.slideWidth / 2)
                    return;
                if (this.inDrag) {
                    oldTheta = Math.atan2(this.startdY, this.startdX),
                        newTheta = Math.atan2(dY, dX);
                    this.degrees += newTheta - oldTheta;
                    this.degrees = (this.degrees + (Math.PI * 2)) % (Math.PI * 2);
                    this.startdY = dY;
                    this.startdX = dX;
                    this.render();
                }
                else {
                    this.startdY = dY;
                    this.startdX = dX;
                    this.inDrag = true;
                }
            }
            else {
                this.inDrag = false;
            }
        };
        OU.activity.Rotation4X.prototype.Rotation.prototype.close = function () {
            this.controlLayer.remove();
            this.leftBox.parentNode.removeChild(this.leftBox);
            this.rightBox.parentNode.removeChild(this.rightBox);
            this.bgLayer.remove();
            this.mscope.inRotationView = false;
        };
        this.init();
        this.renderCycle();
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Rotation4X, OU.util.Activity);

