/**
 * @fileoveriew Menu - places a div element on the page and renders HTML input buttons to build a menu structure
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Anchor');
/**
 * @class An HTML based Menu, implements a menu without canvas
 * 
 * @param {object} params - options:
 * <ul>
 * <li><strong>{object} container:</strong> calling object, typically an OU.util.Activity</li>
 * <li><strong>{object} menus:</strong> defines the menus - this is a nested structure which has the following elements. (Typically a reference to the data.js structure)</li>
     * <ul>
     * <li><strong>{int} levels:</strong> Number of levels to the menu structure</li>
     * <li><strong>{object} sections:</strong> an array of nested menus (See data.js structure for examples)</li>
     * <li><strong>{boolean} autoLoad:</strong> Load/perform the "hit action" for the selected menu on init</li>
     * <li><strong>{string} style:</strong> horizontal (default) or vertical style</li>
     * <li><strong>{boolean} shadow:</strong> </li>
     * </ul>
 * <li><strong>{string} title:</strong> Menu title</li>
 * <li><strong>{int} x:</strong> X coordinate of the menu div</li>
 * <li><strong>{int} y:</strong> Y coordinate of the menu div</li>
 * <li><strong>{int} w:</strong> Width of the menu div</li>
 * <li><strong>{int} h:</strong> Height of the menu div</li>
 * <li><strong>{int} zIndex:</strong> zIndex of the menu div</li>
 * <li><strong>{boolean} showHighlight:</strong> If true, then current selected menu is highlighted</li>
 * <li><strong>{boolean} dontRegister:</strong> Set true to make the menu not get removed by a controller, default is false</li>
 * </ul>
 */
OU.util.Menu = function ( params ) {
    this.menus = params.menus;
    this.container = params.container;
    this.controller = params.container.controller;
    this.register = params.dontRegister || false;
    this.x = params.x || this.container.x || 0;
    this.y = params.y || this.container.y || 0;
    this.w = params.w || this.container.w;
    this.h = params.h || this.container.h;
    this.showHighlight = params.showHighlight===false ? false : true;
    this.style = this.menus.style || 'horizontal';
    this.shadow = this.menus.shadow || false;
    this.title = params.title || '';
    this.zIndex = params.zIndex || OU.CONTROLLER_LEVEL - 1;
    this.Anchors = [];
    /**
     * @private
     */
    OU.util.Menu.prototype.init = function () {
        var menus = this.menus, i,
        autoLoad = this.menus.autoLoad;
        this.menuId = OU.menus.length; // get the menuId for this menu activity
        OU.menus[this.menuId] = this;    // and store this menu in the controllers array of menus
        this.menuDiv = new OU.util.Div({
            dontRegister:this.register,
            x:this.x,
            y:this.y,
            w:this.w,
            h:"auto",
            zIndex:this.zIndex,
            container:this.container,
            showScroll:false
        });
        OU.nobbleDoneBar(this.menuDiv.div);
        menus.selected = [];
        for (i = menus.levels; i--;)
            menus.selected[i] = OU.LocalStorage.load("ou.menus." + this.container.instance + "." + this.menuId + "._" + i) || 0;
        this.render(this.menuId==0 && autoLoad);
    };
    /**
     * @private
     */
    OU.util.Menu.prototype.removeAnchors = function () {
        var i;
        for (i = this.Anchors.length; i--;) { // remove existing anchors
            this.Anchors[i].remove();
        }
        this.Anchors = []; // clear array
    };
    /**
     * Removes the menus, and corresponding Anchors
     */
    OU.util.Menu.prototype.remove = function () {
        this.removed = true;
        this.removeAnchors();
    };
    /**
     * Resizes the menu div
     * @param {object} params - new dims:
     * <ul>
     * <li><strong>{int} x:</strong> X coordinate of the menu div</li>
     * <li><strong>{int} y:</strong> Y coordinate of the menu div</li>
     * <li><strong>{int} w:</strong> Width of the menu div</li>
     * <li><strong>{int} h:</strong> Height of the menu div</li>
     * </ul>
     */
    OU.util.Menu.prototype.resize = function ( params ) {
        this.x = params.x || this.container.x || 0;
        this.y = params.y || this.container.y || 0;
        this.w = params.w || this.container.w;
        this.h = params.h || this.container.h;
        this.menuDiv.resize({
            x:this.x,
            y:this.y,
            w:this.w,
            h:"auto"
        });
    };
    /**
     * Renders the menus, displaying sub level menus according to selections, etc.
     * @param {boolean} doLoad - performs the action of the selected button if true
     * @private
     */
    OU.util.Menu.prototype.render = function ( doLoad ) {
        var menus = this.menus, bH = OU.controlHeight, i, j, h, menu = menus, button, section,
        bFn, self = this, classes, br, mDiv;
        if (this.removed) // menu may have been removed during a Step->Section Override if so, then don't render again
            return;
        /**
         * We have Nobbled the Done Bar for the Menu's DIV (in other words we are capturing and preventing the default events)
         * So, we need to handle the events ourselves to make the buttons work.
         * bFn handles the event and calls the relevant step
         * @private
         */
        bFn = function () {
            var id = this.getAttribute("id"),
            bits = id.split("."),
            nBits = bits.length;
            self.step(parseInt(bits[nBits - 2]), parseInt(bits[nBits - 1]));
        };
        this.removeAnchors();
        this.menuDiv.div.innerHTML = '';
        for (i = 0; i < menus.levels; i++) {
            mDiv = document.createElement('div');
            mDiv.setAttribute("style", "height: " + bH + "px");
            this.menuDiv.div.appendChild(mDiv);            
            if (this.title!='')
                mDiv.innerHTML = '<strong>' + this.title + '</strong><br/>';
            for (j = 0; j < menu.sections.length; j++) {
                section = menu.sections[j];
                if (section.type=='button') {
                    button = section;
                    if ((section.step==parseInt(menu.selected[0]) + 1) && this.showHighlight)
                        classes = '_abutton highlighted';
                    else
                        classes = '_abutton';
                    this.Anchors.push(new OU.util.Anchor({
                        container:this.container,
                        parent:mDiv,
                        classes:classes,
                        id:"ou.menu." + this.menuId + "." + i + "." + j,
                        tabIndex:(i * 20 + j + 10),
                        txt:button.title,
                        img:button.img,
                        onClick:bFn
                    }));
                    if (this.style=='vertical') {
                        br = document.createElement('br');
                        br.setAttribute("clear", "left");
                        mDiv.appendChild(br);
                    }
                    if (doLoad && j==menus.selected[i] && button.menu===undefined)
                        this.loadContent(button);
                }
                else {
                    h = document.createElement(section.type);
                    h.innerHTML = section.content;
                    mDiv.appendChild(h);
                }
            }
            menu = menu.sections[menus.selected[i]].menu;
            if (menu===undefined)
                i = menus.levels;
        }
    };
    /**
     * Performs the button action, ie. loads content, switches sub menus, etc.
     * @private
     */
    OU.util.Menu.prototype.loadContent = function ( button ) {
        if (button.step!==undefined) {
            this.container.controller.step(button.step);
        }
        if (button.controller!==undefined) {
            this.container.controller.overrideSection(button.controller);
        }
        if (button.activityPopup!==undefined) {
            this.container.controller.addActivity(button.activityPopup);
        }
    };
    /**
     * Provides programmatic access to the button actions
     * @param {int} levelId - The level of the button to hit
     * @param {int} buttonId - The button index of the button to hit
     */
    OU.util.Menu.prototype.step = function ( levelId, buttonId ) {
        var menus = this.menus, menu, i, button;
        menus.selected[levelId] = buttonId;
        menu = menus;
        for (i = 0; i < menus.levels; i++) {
            if (menus.selected[i] > menu.sections.length - 1)
                menus.selected[i] = menu.sections.length - 1;
            button = menu.sections[menus.selected[i]];
            if (button.menu===undefined) {
                this.loadContent(button);
            }
            else {
                menu = button.menu;
            }
        }
        this.render(false);
        for (i = menus.levels; i--;) {
            OU.LocalStorage.save("ou.menus." + this.container.instance + "." + this.menuId + "._" + i, menus.selected[i]);
        }
    };
    this.init();
};

