/**
 * OU.util.Spinner Object
 * @fileOverview The Spinner Control
 *
 * @param {object} params
 *
 * @author Martin Donnelly
 * @modified Nigel Clarke <nigel.clarke@pentahedra.com>
 */
OU.require('OU.util.HtmlBox');
/**
 * @class Pop-up spinner, allows the user to select a number from a spinning wheel UI.
 * And returns the value to the calling object on completion.
 * 
 * @param {object} params - options:
 * <ul>
 * <li><strong>{object} container:</strong> (required) The calling object, typically an OU.util.Acitivity </li>
 * <li><strong>{int} min:</strong> Minimum number in the spinner range</li>
 * <li><strong>{int} max:</strong> Maximum number in the spinner range</li>
 * <li><strong>{int} currentVal:</strong> Starting selected number in the range (defaults to 1)</li>
 * <li><strong>{function} onReturn:</strong> Function to call with final value when spinner is closed</li>
 * <li><strong>{double} x:</strong> X co-ordinate</li>
 * <li><strong>{double} y:</strong> Y co-ordinate</li>
 * <li><strong>{double} w:</strong> Width</li>
 * <li><strong>{double} h:</strong> Height</li>
 * <li><strong>{int} maxH:</strong> Maximum Height of font? (default: 100)</li>
 * <li><strong>{int} font:</strong> Font Height (default: 20)</li>
 * <li><strong>{double} feather:</strong> The "gap" in pixels around the spinner that is touchable (defaults to 50)</li>
 * </ul>
 */
OU.util.Spinner = function ( params ) {
    if (params.context===undefined) { // deprecated - doesn't appear to be used any more
        alert('Incorrect use of Spinner Object - Context Required');
        return;
    }
    this.instance = 'spinner';
    this.params = params;
    this.x = params.x || 1;
    this.x = this.x < 0 ? 1 : this.x;
    this.y = params.y || 0;
    this.w = params.w || 200;
    this.h = params.h || 100;
    this.spinnerLayer = new OU.util.Layer({
        container:params.container,
        hasEvents:true,
        instance:this.instance
    });
    this.context = this.spinnerLayer.context;
    this.layer = this.spinnerLayer;
    this.maxH = params.maxH || 100;
    this.font = params.font || 20;
    this.execute = params.execute || null;
    this.min = params.min || 1;
    this.max = params.max || 20;
    this.currentVal = params.currentVal || 1;
    this.feather = params.feather || 50;
    this.yO = 0;
    this.displayed = false;
    this.touched = false;
    this.touchReady = false;
    this.flicked = false;
    this.doanimation = false;
    this.animationstop = false;
    this.bounceBack = false;
    this.cur = this.currentVal;
    this.context.font = (this.font.size) + "px " + OU.theme.font;
    this.tripWidth = this.context.measureText(this.max).width;
    this.nudge = this.font.size * 0.1;
    this.xOffset = 50;
    this.sizeRegulator = 4.4631;
    if ((this.font.size * this.sizeRegulator ) > this.maxH) this.font.size = this.maxH / this.sizeRegulator;
    this.bigBox = {
        w:this.tripWidth * 1.629,
        h:this.font.size * 4.4631,
        x:this.xOffset + (this.x - (this.tripWidth * 1.629) < 0 ? 1 : this.x - (this.tripWidth * 1.629)),
        y:this.y
        };
    this.inBox = {
        w:this.bigBox.w - (this.nudge * 2),
        h:this.bigBox.h - (this.nudge * 2),
        x:this.nudge,
        y:this.nudge
        };
    this.lensBox = {
        w:this.inBox.w,
        h:this.font.size + this.nudge,
        x:this.inBox.x,
        y:(this.bigBox.h / 2) - (this.font.size / 2)
    };
    this.touchBubble = {
        x:this.bigBox.x - this.feather,
        y:this.bigBox.y - this.feather,
        w:this.bigBox.w + (this.feather * 2),
        h:this.bigBox.h + (this.feather * 2)
        };
    this.innertouchBubble = {
        x:this.bigBox.x + this.inBox.x,
        y:this.bigBox.y + this.inBox.y,
        w:this.inBox.w,
        h:this.inBox.h
        };
    this.FrictionEpsilon = 1e-2;
    this.boxH = Math.floor(this.inBox.h / 3);
    this.yO = (((this.cur - 1) * Math.floor(this.boxH))   ) * -1;
    this.closeTimer = 0;
    this.hold = 0;
    this.noTouching=0;
    this.virgin=true;
    this.spinnerLayer.events.moveRight = function () {
        return null;
    };
    this.spinnerLayer.events.moveLeft = function () {
        return null;
    };
    this.onReturn = this.params.onReturn || function () {
    };
    this.cl1 = ["rgba(119,123,137,1)", "rgba(49,55,77,1)", "rgba(29,36,60,1)", "rgba(33,34,41,1)"];
    this.cl2 = ['#38393E', '#F2F3F3', '#ffffff', '#D3D3D3'];
    this.cl3 = ['rgba(110,130,229,0.22)', 'rgba(110,130,229,0.36)', 'rgba(255,255,255,0.59)', 'rgba(110,130,229,0.43)', 'rgba(110,130,229,0.56)'];
    this.cl4 = ["rgba(28,28,28,0.75)", "rgba(210,210,210,0.75)", "#1F2335"];
    this.friction = 0.97;
    this.doRender = true;
    this.tutor = this.params.tutor || 0;
    this.parent = this.params.parent || undefined;
    this.d = 0;
    this.prevCur = 0;
    this.buildGradients();
    this.renderLoop();
    this.called = 0;
};
/**
 * Defines some gradient to use later
 * @private
 */
OU.util.Spinner.prototype.buildGradients = function () {
    var ctx = this.context, gBig, gWheel, gLens;
    if (this.gBig===undefined) {
        gBig = ctx.createLinearGradient(this.bigBox.x, this.bigBox.y, this.bigBox.x, this.bigBox.y + this.bigBox.h);
        gBig.addColorStop(0, this.cl1[0]);
        gBig.addColorStop(0.50, this.cl1[1]);
        gBig.addColorStop(0.51, this.cl1[2]);
        gBig.addColorStop(1, this.cl1[3]);
        this.gBig = gBig;
    }
    if (this.gWheel===undefined) {
        gWheel = ctx.createLinearGradient(this.inBox.x, this.inBox.y, this.inBox.x, this.inBox.y + this.inBox.h);
        gWheel.addColorStop(0, this.cl2[0]);
        gWheel.addColorStop(0.21, this.cl2[1]);
        gWheel.addColorStop(0.32, this.cl2[2]);
        gWheel.addColorStop(0.47, this.cl2[3]);
        gWheel.addColorStop(0.63, this.cl2[3]);
        gWheel.addColorStop(0.68, this.cl2[2]);
        gWheel.addColorStop(0.79, this.cl2[1]);
        gWheel.addColorStop(1, this.cl2[0]);
        this.gWheel = gWheel;
    }
    if (this.gLens===undefined) {
        gLens = ctx.createLinearGradient(this.lensBox.x, this.lensBox.y, this.lensBox.x, this.lensBox.y + this.lensBox.h);
        gLens.addColorStop(0, this.cl3[0]);
        gLens.addColorStop(0.49, this.cl3[1]);
        gLens.addColorStop(0.50, this.cl3[2]);
        gLens.addColorStop(0.51, this.cl3[3]);
        gLens.addColorStop(1, this.cl3[4]);
        this.gLens = gLens;
    }
};
/**
 * Renders the spinner
 */
OU.util.Spinner.prototype.render = function () {
    var ctx = this.context;
    var aW, gBig, gWheel, gLens, cl = this.boxH / 2, startPos = this.boxH, cY, cx, chr, t, ch = OU.dpr;
    ctx.font = this.font.size + "px " + this.font.family;
    gBig = this.gBig;
    gWheel = this.gWheel;
    gLens = this.gLens;
    ctx.save();
    ctx.fillStyle = gBig;
    ctx.roundRect(this.bigBox.x, this.bigBox.y, this.bigBox.w, this.bigBox.h, 5);
    ctx.fill();
    ctx.translate(this.bigBox.x, this.bigBox.y);
    ctx.fillStyle = gWheel;
    ctx.strokeStyle = "#778396";
    //   ctx.save();
    ctx.roundRect(this.inBox.x, this.inBox.y, this.inBox.w, this.inBox.h, 5);
    ctx.save();
    ctx.translate(this.inBox.x, this.inBox.y);
    ctx.clip();
    ctx.stroke();
    ctx.fill();
    ctx.strokeStyle = 'rgba(255,255,255,0.20)';
    ctx.lineWidth = "1";
    ctx.strokeStyle = '#ff0000';
    ctx.fillStyle = "#1F2335";
    for (t = this.min; t <= this.max; t++) {
        chr = t;
        cY = startPos + (this.boxH * (t - 1)) + this.yO;
        if ((cY > (-1) * this.boxH) && (cY < this.boxH * 4)) {
            aW = ctx.measureText(chr).width;
            cx = (this.inBox.w - aW ) / 2;
            if ((cY + cl) < this.boxH) {
                ctx.fillStyle = this.cl4[0];
                ctx.fillText(chr, cx, cY + cl + ch);
                ctx.fillStyle = this.cl4[1];
                ctx.fillText(chr, cx, cY + cl - ch);
                ctx.fillStyle = this.cl4[2];
                ctx.fillText(chr, cx, cY + cl);
            }
            else if (((cY + cl) >= this.boxH) && ((cY + cl) < this.boxH)) {
                ctx.fillStyle = this.cl4[1];
                ctx.fillText(chr, cx + ch, cY + cl);
                ctx.fillStyle = this.cl4[0];
                ctx.fillText(chr, cx - ch, cY + cl);
                ctx.fillStyle = this.cl4[2];
                ctx.fillText(chr, cx, cY + cl);
            }
            else {
                ctx.fillStyle = this.cl4[1];
                ctx.fillText(chr, cx, cY + cl + ch);
                ctx.fillStyle = this.cl4[0];
                ctx.fillText(chr, cx, cY + cl - ch);
                ctx.fillStyle = this.cl4[2];
                ctx.fillText(chr, cx, cY + cl);
            }
        }
    }
    ctx.restore();
    ctx.strokeStyle = "#778396";
    ctx.fillStyle = gLens;
    ctx.fillRect(this.lensBox.x, this.lensBox.y, this.lensBox.w, this.lensBox.h);
    ctx.strokeRect(this.lensBox.x, this.lensBox.y, this.lensBox.w, this.lensBox.h);
    ctx.restore();
    this.called += 1;
    this.touchReady = true;
};
/**
 * Stops the spinner and removes it.
 * Note, does not return a value to the calling object.
 */
OU.util.Spinner.prototype.removeSpinner = function () {
    this.doRender = false;
    this.layer.remove();
};
/**
 * Returns the selected value via a callback to the original calling object.
 * And removes the spinner.
 */
OU.util.Spinner.prototype.returnValue = function () {
    var rVal = this.cur, self = this;
    self.removeSpinner();
    //if (this.execute) this.execute(rVal);
    this.onReturn(rVal);
};
/**
 * Calculates what the current value really is
 * @private
 */
OU.util.Spinner.prototype.findCur = function () {
    this.prevCur = this.cur;
    for (var t = this.min, p, yOabs; t <= this.max; t++) {
        p = (t - 1) * this.boxH;
        yOabs = Math.abs(this.yO);
        if ((yOabs > p - 3) && (yOabs < p + 3)) {
            this.cur = t;
            if (this.cur==1 && this.prevCur==this.max) {

                // fixes the max -> 1 issue
                this.cur = this.max;
                p = ((this.cur - 1) * this.boxH);
                this.yO = p * (-1);
            }
            return;
        }
    }
};
/**
 * The main display loop
 * @param me  - reference back to the spinner (Odd usage - should be using self to avoid scoping issues, not passing "me")
 * @private
 */
OU.util.Spinner.prototype.renderLoop = function ( me ) {
    var self = me || this, d, usd, a, sign,events=this.layer.events,
    tx = events.x,
    ty = events.y,
    touching=events.pressed || events.touched,
    tb = this.touchBubble,
    itb = this.innertouchBubble, cP, force, yOabs, dabs;
    if (!this.doRender) return;
    if (!touching) {
        if (this.touched) {
            yOabs = Math.abs(this.yO);
            if ((yOabs % this.boxH > 0) || (yOabs > this.boxH * (this.max - 1))) {
                this.doanimation = true;
                this.touched = false;
            }
        }
        this.prevY = null;

        this.noTouching++;
        if(!this.virgin && this.noTouching>20) { // spinner has been touched and then not touched (in or outside the bubble) for 20 cycles (1 seconds), so close
            events.flush();
            this.doRender = false;
            setTimeout(function () {
                self.returnValue();
            }, 5);
            return;
        }
    }
    else { // touching
        this.noTouching=0;
        this.virgin=false;

        if (this.touchReady) {
            if (((tx > tb.x && tx < (tb.x + tb.w)) && (ty > tb.y && ty < (tb.y + tb.h)))) { // is touch inside the bubble?
                // spinner has been dragged, so move it
                this.touched = true;
                this.flicked = false;
                this.bounceBack = false;
                if (this.prevY===null)
                    this.prevY = events.y;
                if ((this.yO < ((this.max) * this.boxH) * -1) || (this.yO > 0)) this.d = ( this.layer.events.y - this.prevY) * 0.15;
                else this.d = ( events.y - this.prevY) * this.friction;
                this.prevY = events.y;
            }
            else { // touch is outside the bubble, so return value and remove spinner
                events.flush();
                this.doRender = false;
                setTimeout(function () {
                    self.returnValue();
                }, 5);
                return;
            }
        }
    }
    dabs = Math.abs(this.d);
    if (!touching && dabs > 5 && !this.flicked) { // delta of drag is >5 so initiate flick
        this.flicked = true;
        this.bounceBack = false;
        this.doanimation = false;
        this.hold = 0;
    }
    if (touching && dabs===0 && !this.flicked) {
        this.hold++;
    }
    if (!touching && dabs===0 && !this.flicked) {
        if (this.hold>0 && this.hold < 5) {
            // wheel has been tapped, so spin to the number hit
            cP = ( (events.lastTouch.y - itb.y ) / this.boxH) | 0;
            if (cP===0) {
                this.d = (this.boxH / 3);
                this.flicked = true;
                this.bounceBack = false;
                this.doanimation = false;
            }
            if (cP===1) {
                events.flush();
                this.doRender = false;
                setTimeout(function () {
                    self.returnValue();
                }, 5);
                return;
            }
            if (cP >= 2) {
                this.d = (-1) * (this.boxH / 3);
                this.flicked = true;
                this.bounceBack = false;
                this.doanimation = false;
            }
        }
        this.hold = 0;
    }
    if (!this.bounceBack)
        this.calculate();
    if (this.doanimation && !touching) {
        force = false;
        yOabs = Math.abs(this.yO);
        if (yOabs > this.boxH * (this.max - 1)) {
            d = yOabs - this.boxH * (this.max - 1);
            force = true;
        }
        else
            d = this.yO % this.boxH;
        usd = Math.abs(d);
        sign = d && d / usd;
        if (((this.boxH - usd < 1.0) && !force) || this.animationstop) {
            this.doanimation = false;
            this.animationstop = false;
            this.yO = sign * (((yOabs | 0) / this.boxH) * this.boxH);
            this.findCur();
        }
        else {
            if ((usd > this.boxH / 2) && !force) {
                a = sign * ((this.boxH - usd) / 2);
                if (Math.abs(a) < this.FrictionEpsilon) this.animationstop = true;
            }
            else {
                a = (usd / 2);
                if (Math.abs(a) < this.FrictionEpsilon) this.animationstop = true;
            }
            this.yO += a;
        }
    }
    else {
        // we've been retouched so kill the animation
        this.doanimation = false;
    }
    if ((this.yO) > 0)
        this.yO = 0;
    if ((this.yO) < ((this.max ) * this.boxH) * (-1) && this.yO!==0) {
        this.yO = ((this.max - 1) * this.boxH) * (-1);
    }
    this.prevCur = this.cur;
    this.findCur();
    this.render();
    if (this.cur==2 && this.tutor==2) {
        // console.log("bleep!");
        this.parent.killTutorPopup();
        var params = {
            x:this.parent.tutorSteps[2].x,
            y:this.parent.tutorSteps[2].y,
            w:this.parent.tutorSteps[2].w,
            h:this.parent.tutorSteps[2].h,
            txt:this.parent.tutorSteps[2].txt,
            k:3
        };
        this.parent.tutorPopup(params);
        this.parent.tutor = this.tutor = 3;
    }
    if (!this.displayed) {
        this.displayed = true;
    }
    if (this.doRender) {
        setTimeout(function () {
            self.renderLoop();
        }, 40);
    }
};
/**
 * Calculates inertia / movement
 * @private
 */
OU.util.Spinner.prototype.calculate = function () {
    var p, tq, fs = this.boxH;
    if (Math.abs(this.d) > 0) {
        if (!this.flicked) this.yO += this.d;
        if (this.flicked) {
            this.yO += this.d;
            if (Math.abs(this.d) < 1.9) {
                // 3/4's rule..
                tq = (fs / 4) * 3;
                p = (this.yO % fs);
                if (Math.abs(p) < tq) {
                    if (!this.doanimation) {
                        this.doanimation = true;
                        this.flicked = false;
                        this.d = 0;
                    }
                    this.d /= 2;
                    if (Math.abs(this.d) < 0.01) {
                        this.d = 0;
                        this.bounceBack = false;
                        this.yO = Math.floor(this.yO);
                    }
                    this.d *= -1;
                }
                if (Math.abs(Math.floor(this.yO % fs))===Math.abs(fs)) {
                    if (this.d < 0) this.yO -= fs;
                    this.d = 0;
                    this.flicked = false;
                }
            }
            else {
                this.d *= 0.75;
            }
        }
    }
    if ((this.yO) > 0) {
        this.yO = 0;
    }
    if ((this.yO) < ((this.max ) * this.boxH) * (-1) && this.yO!==0) {
        this.yO = ((this.max - 1) * this.boxH) * (-1);
    }
    this.findCur();
};







