/**
 * @fileOverview ImageComparison - Image comparison activity, allows the user to overlay multuple images over a target to determine a match.
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

// Load in any util elements that are required
OU.require('OU.util.Layer');

/**
 * @class ImageComparison - Image comparison activity, allows the user to overlay multuple images over a target to determine a match.
 * @extends OU.util.Activity
 * @param {Object} data - Holds the data content for a specific instance of the activity
 * @param {String} instance - (Optional) A unique identifier name for this instance, defaults to 'a1'
 * @param {OU.util.Controller} controller - (Optional) A reference to the controller that initialised this instance, if undefined, the superclass will generate a new controller and create the reference to it as 'this.controller'
 */
OU.activity.ImageComparison = function(data, instance, controller) {

    /**
     * canvasView - this is the function that is first called when the activity is launched,
     *              assuming you are in a browser that supports canvas.
     *
     * Typical tasks for this function are:
     * <ul>
     * <li> Define and initialise and activity wide variables</li>
     * <li> Initialise Layers & Divs</li>
     * <li> Call a loading function</li>
     * <li> Initiating the activity by calling any methods that</li>
     * </ul>
     */
    OU.activity.ImageComparison.prototype.canvasView = function() {

        // Most activities should have a background layer
        this.bgLayer = new OU.util.Layer({
            container: this
        });
        this.bgLayer.context.gradRect(); // draw default background

        this.titleDiv = new OU.util.Div({
            container: this,
            h: 'auto'
        });

        this.imageLayer = new OU.util.Layer({
            container: this,
            hasEvents: true
        });
        this.imageLayer.events.clickable.push(this);
        this.thumbDiv = new OU.util.Div({
            container: this,
            overflow: 'auto',
            style: "text-align:center;"
        });

        this.xOffset = 0;
        this.yOffset = 0;
        this.selectedIdx = undefined;

        this.loadImages(); // start by loading images
    };
    OU.activity.ImageComparison.prototype.loadImages = function() {
        var self = this;
        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.data,
            onLoad: function() {
                self.findTarget();
                self.resize();
                self.loadThumbs();
            }
        });

    };
    OU.activity.ImageComparison.prototype.renderTitle = function() {
        var ins = "Select a thumbnail on the right to compare.";
        if (this.selectedIdx !== undefined)
            ins = "Comparing with: <strong>" + this.images[this.selectedIdx].name + "</strong>";
        this.titleDiv.div.innerHTML = "<div style='padding: 5px 10px; background:#444;color:#eee;'>" + this.data.activityTitle + "</div><p style='padding: 5px 10px'><u>" + this.sampleTitle + "</u></p><p style='float:right;padding: 5px 20px'><u>" + this.data.libraryTitle + "</u></p><p style='padding: 5px 10px'>" + ins + "</p>";
    };
    OU.activity.ImageComparison.prototype.findTarget = function() {
        var i;
        this.images = this.data.images;
        // Firstly determine which image is the target
        var targetName = OU.LocalStorage.load('comparisonImageTargetName') || this.data.targetImageName;
        this.sampleTitle = OU.LocalStorage.load('comparisonImageTitle') || this.data.mainImageTitle;
        this.renderTitle();
        if (targetName) {
            for (i = this.images.length; i--; ) {
                if (this.images[i].name === targetName)
                    this.targetImage = this.images[i];
            }
        }
        if (!this.targetImage) {
            console.log('using default target of first image instead of ' + targetName);
            this.targetImage = this.images[0];
        }

        // set width of thumbnail div, based on thumbnail widths
        this.thumbnailWidth = this.targetImage.thumb.image.width;
        this.fullImageWidth = this.targetImage.full.image.width;


    };
    OU.activity.ImageComparison.prototype.isHit = function(x, y, state) {
        var dx, dy;
        if (state) {
            if (this.inDrag) {
                dx = x - this.startX;
                dy = y - this.startY;
                this.xOffset += dx;
                this.yOffset += dy;
                this.startX = x;
                this.startY = y;
                this.render();
            }
            else {
                this.inDrag = true;
                this.startX = x;
                this.startY = y;
            }
        }
        else {
            this.inDrag = false;
        }
    };
    OU.activity.ImageComparison.prototype.render = function() {
        var ctx = this.imageLayer.context,
                image = this.targetImage.full.image,
                comparisonImage = this.selectedIdx !== undefined ? this.images[this.selectedIdx].blue.image : null,
                w = image.width * this.scaleFactor,
                h = image.height * this.scaleFactor,
                x = ((this.imageLayer.w - w) / 2) | 0,
                y = ((this.imageLayer.h - h) / 2) | 0;
        this.imageLayer.clear();
        ctx.drawImage(image, x, y, w, h);
        if (this.selectedIdx !== undefined)
            ctx.drawImage(comparisonImage, x + this.xOffset | 0, y + this.yOffset | 0, w, h);
    };
    OU.activity.ImageComparison.prototype.changeImage = function(idx) {
        var imgObj = document.getElementById('imageSelector_' + this.selectedIdx);
        if (imgObj)
            imgObj.className = 'imageSelector';
        if (idx === -1) {
            this.selectedIdx = undefined;
        }
        else {
            this.selectedIdx = idx;
            imgObj = document.getElementById('imageSelector_' + this.selectedIdx);
            imgObj.className = 'imageSelected';
            this.xOffset = Math.random() * 20 - 10;
            this.yOffset = Math.random() * 20 - 10;
            this.xOffset += this.xOffset > 0 ? 10 : -10;
            this.yOffset += this.yOffset > 0 ? 10 : -10;
        }
        this.render();
        this.renderTitle();
    };
    OU.activity.ImageComparison.prototype.loadThumbs = function() {
        var i, self = this,
                html = '<img class="imageSelector" src="' + this.dataDir + '/clear.png" onclick="OU.__selectThumb(-1)" style="margin 5px auto;"/><br clear="both" />';

        OU.__selectThumb = function(i) {
            self.changeImage(i);
        };

        for (i = 0; i < this.images.length; i++) {
            html = html + '<img id="imageSelector_' + i + '" class="imageSelector" src="' + this.dataDir + '/blue/' + this.images[i].thumb.img + '" onclick="OU.__selectThumb(' + i + ')" style="margin 5px auto;"/><br clear="both" />';
        }

        this.thumbDiv.div.innerHTML = html;
    };
    /**
     * resize - called whenever the outer bounds of the activity change, ie when the browser window is resized, or a mobile device is rotated.
     */
    OU.activity.ImageComparison.prototype.resize = function() {
        OU.activity.ImageComparison.superClass_.resize.call(this); // call the superclass resize

        if (this.w - this.thumbnailWidth - 20 < this.fullImageWidth * 1.2) { // we need to reduce the main image to fit it in
            this.scaleFactor = (this.w - this.thumbnailWidth - 20) * 0.8 / this.fullImageWidth;
        }
        else {
            this.scaleFactor = 1;
        }
        var imageLayerWidth = this.w - this.thumbnailWidth - 40;
        this.bgLayer.resize();
        this.bgLayer.context.gradRect();

        this.titleDiv.resize({
            x: this.x,
            y: this.y,
            w: this.w,
            h: 'auto'
        });
        var titleHeight = this.titleDiv.div.clientHeight;
        this.imageLayer.resize({
            x: this.x,
            y: this.y + titleHeight,
            w: imageLayerWidth,
            h: this.h - titleHeight
        });
        this.thumbDiv.resize({
            x: this.x + imageLayerWidth,
            y: this.y + titleHeight,
            w: this.w - imageLayerWidth,
            h: this.h - titleHeight
        });
        this.render();
    };

    // call the superclass's constructor
    OU.base(this, data, instance, controller);
};
// Call our inherits function to implement the class inheritance
OU.inherits(OU.activity.ImageComparison, OU.util.Activity);
