/**
 * @fileoverview Anchor - class to handle Anchor Links (A tags) which extends the Tabbable class
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 *
 * @class Inserts an A tag into a parent Div, extends tabbable to make Hyperlinks tabbable on all browsers/platforms
 * @param {object} params - params to define it:
 * <ul>
 * <li><strong>{object} container</strong>: (required) reference to the anchors container (typically the activity calling it)</li>
 * <li><strong>{DOM object} parent</strong>: reference to the parent DOM object that the anchor is going to be appended to (typically a DIV)</li>
 * <li><strong>{string} txt</strong>: (optional) text contents of the anchor is no image present</li>
 * <li><strong>{string} img</strong>: (optional) Image URL if using an image in the anchor</li>
 * <li><strong>{function} onClick</strong>: Function to call when the anchor is "hit"</li>
 * <li><strong>{string} id</strong>: (optional) ID to add to the anchor</li>
 * <li><strong>{string} classes</strong>: (optional) CSS style to add to the anchor</li>
 * </ul>
 * @extends OU.util.Tabbable
 */
OU.util.Anchor = function ( params ) {
    var self=this;
    this.container = params.container;
    this.parent = params.parent;
    this.txt = params.txt||'';
    this.onClick=params.onClick || function() {};

    this.touch = function(ev) {
        var e=OU.combineMouseTouch(ev);
        if(e.pageX>0)
            self.onClick.call(self.a);
    };
    this.a = document.createElement('a');
    if(params.id)
        this.a.setAttribute('id', params.id);
    if(params.classes)
        this.a.setAttribute('class', params.classes);
    this.parent.appendChild(this.a);

    if (!params.img) {
        this.a.innerHTML=this.txt;
    }
    else {
        this.a.innerHTML= "<img src='" + params.img + "' alt='" + this.txt + "' title='" + this.txt + "'/>";
    }
    if(params.onClick) {
        this._handlersSet=true;
        this.a.addEventListener("mousedown", this.onClick, false);
        this.a.addEventListener("touchstart", this.touch, false);
    }
    /**
     * Change visual appearance of anchor to show it has focus
     * @private
     */
    OU.util.Anchor.prototype.focus = function () {
        this.a.setAttribute("style","background: #c00;color:#fff");
        return false;
    };
    /**
     * Change visual appearance of anchor to show it does not have focus
     * @private
     */
    OU.util.Anchor.prototype.blur = function () {
        this.a.setAttribute("style","");
        return false;
    };
    /**
     * Perform onClick function when hit
     * @private
     */
    OU.util.Anchor.prototype.hit = function () {
        this.onClick.call(this.a); // call the onClick function as if it was being called by the A tag itself.
        return false;
    };
    /**
     * Extends the super class remove, removes the event handlers associated with this anchor
     */
    OU.util.Anchor.prototype.remove = function () {
        OU.util.Anchor.superClass_.remove.call(this); // call the parent class remove
        if(this._handlersSet) {
            this.a.removeEventListener("mousedown", this.onClick, false);
            this.a.removeEventListener("touchstart", this.touch, false);
        }
    };
    OU.base(this, params);
};
OU.inherits(OU.util.Anchor, OU.util.Tabbable);