/**
 * @fileOverview Allows user to type in a function
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @class Adds a DIV element to the page and allows the user to enter a 'function'
 *
 * @param {object} params, various params to define the behaviour:
 * <ul>
 * <li><strong>{OU.util.Activity} container</strong>: (required) reference to the parent activity</li>
 * <li><strong>x</strong>: x position (optional, defaults to 25% of container width)</li>
 * <li><strong>y</strong>: y position (optional, defaults to 25% of container height)</li>
 * <li><strong>w</strong>: width (optional, defaults to 50% width of container)</li>
 * <li><strong>{int} zIndex</strong>: (optional) zIndex to apply to this div(Gets added to the activity zOffset</li>
 * <li><strong>{string} style</strong>: (optional) CSS style to add to the div</li>
 * </ul>
 */
OU.util.FunctionEditor = function(params) {
    if (params === undefined)
        params = {};
    this.div = null;
    this.container = params.container || {};
    this.zIndexStart = params.zIndex || OU.ABSOLUTE_TOP;
    this.zOffset = this.container.zOffset || 0;
    this.zIndexStart = this.zIndexStart + this.zOffset;
    this.params = params;
    this.style = params.style || '';
    this.addStyle = '';
    this.value = params.value || '';
    this.onSubmit = params.onSubmit || function() {
    };
    /**
     * sets everything up and adds the div element into the parent DOM object
     * @private
     */
    OU.util.FunctionEditor.prototype.init = function(params) {
        var self = this, container = this.container,
                x = params.x || container.x + container.w * .25,
                y = params.y || container.y + container.h * .25,
                w = params.w || container.w * .5 || window.innerWidth || 480;

        this.x = x;
        this.y = y;
        this.w = w;

        if (OU._popupHighlander)
            OU._popupHighlander.remove();
        OU._popupHighlander = this;
        this.div = document.createElement("div");
        this.div.setAttribute("style", this.addStyle + 'left:' + this.x + 'px; top:' + this.y + 'px;' + 'width:' + this.w + 'px; height:auto; ' + this.style);
        this.div.setAttribute("class", "_overlaydiv htmleditor hidden3d" + (this.params.htmlClass || ''));
        this.div.style.zIndex = this.zIndexStart;
        document.body.appendChild(this.div);

        this.setFunctions();
        this.div.innerHTML = "<div><p>Provide a function for a new column in terms of existing column values.<br/> e.g. ln(A^2)+B&nbsp;&nbsp;&nbsp;&nbsp;or&nbsp;&nbsp;&nbsp;&nbsp;cos(X)+sin(X)</p></div>"
                + "<input type=text id='_htmlInput' onkeyup='OU._FunctionEditor.change();' value='" + this.value + "'/>"
                + "<div id='_funcs' style='width:365px; float:right'><p><input type=submit value='+' onclick='OU._FunctionEditor.insChar(\"+\");'/>"
                + "<input type=submit value='-' onclick='OU._FunctionEditor.insChar(\" - \");'/>"
                + "<input type=submit value='×' onclick='OU._FunctionEditor.insChar(\" × \");'/>"
                + "<input type=submit value='÷' onclick='OU._FunctionEditor.insChar(\" ÷ \");'/>"
                + "<input type=submit value='^' onclick='OU._FunctionEditor.insChar(\"^\");'/><br/>"
                + "<input type=submit value='ln' onclick='OU._FunctionEditor.insFn(\"ln\");'/>"
                + "<input type=submit value='log10' onclick='OU._FunctionEditor.insFn(\"log10\");'/>"
                + "<input type=submit value='exp' onclick='OU._FunctionEditor.insFn(\"exp\");'/>"
                + "<input type=submit value='(' onclick='OU._FunctionEditor.insChar(\"(\");'/>"
                + "<input type=submit value=')' onclick='OU._FunctionEditor.insChar(\")\");'/><br/>"
                + "<input type=submit value='abs' onclick='OU._FunctionEditor.insFn(\"abs\");'/>"
                + "<input type=submit value='1/x' onclick='OU._FunctionEditor.insFn(\"1/\");'/>"
                + "<input type=submit value='sqrt' onclick='OU._FunctionEditor.insFn(\"sqrt\");'/>"
                + "<input type=submit value='π' onclick='OU._FunctionEditor.insChar(\"π\");'/>"
                + "<input type=submit value='e' onclick='OU._FunctionEditor.insChar(\"e\");'/><br/>"
                + "<input type=submit value='sin' onclick='OU._FunctionEditor.insFn(\"sin\");'/>"
                + "<input type=submit value='cos' onclick='OU._FunctionEditor.insFn(\"cos\");'/>"
                + "<input type=submit value='tan' onclick='OU._FunctionEditor.insFn(\"tan\");'/><br/>"
                + "<input type=submit value='asin' onclick='OU._FunctionEditor.insFn(\"asin\");'/>"
                + "<input type=submit value='acos' onclick='OU._FunctionEditor.insFn(\"acos\");'/>"
                + "<input type=submit value='atan' onclick='OU._FunctionEditor.insFn(\"atan\");'/>"

                + "</p></div><div id='_keypad'><p>"
                + "<input type=submit value='7' onclick='OU._FunctionEditor.insChar(\"7\");'/>"
                + "<input type=submit value='8' onclick='OU._FunctionEditor.insChar(\"8\");'/>"
                + "<input type=submit value='9' onclick='OU._FunctionEditor.insChar(\"9\");'/><br/>"
                + "<input type=submit value='4' onclick='OU._FunctionEditor.insChar(\"4\");'/>"
                + "<input type=submit value='5' onclick='OU._FunctionEditor.insChar(\"5\");'/>"
                + "<input type=submit value='6' onclick='OU._FunctionEditor.insChar(\"6\");'/><br/>"
                + "<input type=submit value='1' onclick='OU._FunctionEditor.insChar(\"1\");'/>"
                + "<input type=submit value='2' onclick='OU._FunctionEditor.insChar(\"2\");'/>"
                + "<input type=submit value='3' onclick='OU._FunctionEditor.insChar(\"3\");'/><br/>"
                + "<input type=submit value='0' onclick='OU._FunctionEditor.insChar(\"0\");'/>"
                + "<input type=submit value='.' onclick='OU._FunctionEditor.insChar(\".\");'/>"
                + "<input type=submit value='E' onclick='OU._FunctionEditor.insChar(\"E\");'/><br/>"
                + "</p></div><p><br clear='both'/><span id='_errorMsg' style='color:#c00'></span><input id='_doneButton' type=submit value='Populate' onclick='OU._FunctionEditor.done();' style='margin:10px 0 0 10px;float:right;'/><input type=submit value='Cancel' onclick='OU._FunctionEditor.cancel();' style='margin:10px 0;float:right;'/></p>";
        OU.addRemovable(this, this.zOffset);
        setTimeout(function() {
            OU.addClass(self.div, "show3d");
            document.getElementById('_htmlInput').focus();
        }, 20);
    };

    OU.util.FunctionEditor.prototype.done = function() {
        this.onSubmit(this.value);
        this.remove();
    };
    OU.util.FunctionEditor.prototype.cancel = function() {
        this.remove();
    };
    /**
     * Initialise the functions that will action the buttons
     */
    OU.util.FunctionEditor.prototype.setFunctions = function() {
        var self = this;
        OU._FunctionEditor = {
            insFn: function(fn) {
                var inBox = document.getElementById('_htmlInput'),
                        val = inBox.value,
                        start = inBox.selectionStart, end = inBox.selectionEnd,
                        chunk1 = val.substr(0, start),
                        chunk2 = val.substr(start, end - start),
                        chunk3 = val.substr(end);

                self.value = chunk1 + fn + "(" + chunk2 + ")" + chunk3;
                inBox.value = self.value;
                inBox.selectionEnd = inBox.selectionStart = start + fn.length + 1;
                inBox.focus();
                this.change();
                return false;
            },
            insChar: function(c) {
                var inBox = document.getElementById('_htmlInput'),
                        val = inBox.value,
                        start = inBox.selectionStart,
                        chunk1 = val.substr(0, start),
                        chunk2 = val.substr(start);

                self.value = chunk1 + c + chunk2;
                document.getElementById('_htmlInput').value = self.value;
                inBox.selectionEnd = inBox.selectionStart = start + c.length;
                inBox.focus();
                this.change();
                return false;
            },
            change: function() {
                var doneButton = document.getElementById('_doneButton'),
                        inBox = document.getElementById('_htmlInput'),
                        start = inBox.selectionStart,
                        end = inBox.selectionEnd,
                        val = inBox.value, invalidMsg,
                        valid = true, i, invalids = ";[]{}$@'\"?><~`%£&|!=_",
                        replaceStrs = ['SIN', 'COS', 'TAN', 'ASIN', 'ACOS', 'ATAN', 'ABS', 'LN', 'LOG10', 'EXP', 'SQRT', '/', '*'],
                        withStrs = ['sin', 'cos', 'tan', 'asin', 'acos', 'atan', 'abs', 'ln', 'log10', 'exp', 'sqrt', '÷', '×'];
                for (i = invalids.length; i--; ) { // remove invalid chars
                    val = val.replace(invalids.charAt(i), '');
                }
                val = val.toUpperCase();
                for (i = replaceStrs.length; i--; ) { // replace strs
                    val = val.replace(replaceStrs[i], withStrs[i]);
                }
                // now determine if the function is valid
                if (val.split("(").length !== val.split(")").length) {
                    valid = false;
                    invalidMsg = 'Unmatched brackets';
                }
                if (valid) {
                    try {
                        var result = OU._popupHighlander.container.evalFn(val, [11, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]); // try it with demo data

//                        console.log('eval result:'+result);
                    }
                    catch (e) {
                        if (e.message === "missing ; before statement")
                            invalidMsg = "Missing operator between elements";
                        else
                            invalidMsg = e.message;
                        valid = false;
                    }
                }
                if (valid) {
                    document.getElementById('_errorMsg').innerHTML = '';
                    doneButton.title = 'Your function appears valid, click to populate.';
                    doneButton.class = 'validFn';
                    doneButton.disabled = false;
                }
                else {
                    document.getElementById('_errorMsg').innerHTML = invalidMsg;
                    doneButton.title = 'Please correct the error first';
                    doneButton.class = 'invalidFn';
                    doneButton.disabled = true;
                }
                OU._popupHighlander.value = inBox.value = val;
                inBox.selectionStart=start;
                inBox.selectionEnd=end;
                return false;
            },
            done: function() {
                self.done();
                return false;
            },
            cancel: function() {
                self.cancel();
                return false;
            }
        };
    };
    /**
     * Resizes the div
     * @param {object} p - optional resize parameters, all default to current value if not specified:
     * <ul>
     * <li><strong>x</strong>: x position</li>
     * <li><strong>y</strong>: y position</li>
     * <li><strong>w</strong>: width </li>
     * <li><strong>h</strong>: height</li>
     * </ul>
     */
    OU.util.FunctionEditor.prototype.resize = function(p) {
        p=p || {};
        var x = p.x || this.container.x+this.container.w*.25,
                y = p.y || this.container.y+this.container.h*.25,
                w = p.w || this.container.w*.5;
        if (x === this.x && y === this.y && w === this.w)
            return; // nothing to do
        this.div.setAttribute("style", this.addStyle + 'left:' + x + 'px; top:' + y + 'px;' + 'width:' + w + 'px; height:auto; ' + this.style);
        if (this.zIndexStart)
            this.div.style.zIndex = this.zIndexStart;
        this.x = x;
        this.y = y;
        this.w = w;
        if (this._scroll) {
            this._scroll.resize({
                x: x + w,
                y: y,
                w: 20,
                h: 'auto'
            });
            this.handleScroll();
        }
    };

    /**
     * Removes the div from the page and tidies up event handlers, scroll bars , etc.
     */
    OU.util.FunctionEditor.prototype.remove = function() {
        var self = this;
        // remove scrollbar
        if (this._scroll) {
            this.div.removeEventListener("scroll", this._scrollListener, false);
            this.div.removeEventListener("DOMSubtreeModified", this._scrollListener, false);
            this._scroll.remove();
        }
        this._scrollctx = null;
        this._scroll = null;
        if (this.div) {
            OU.removeClass(this.div, 'show3d');
            if (this.div.parentNode) {
                setTimeout(function() { // remove after css transform has finished
                    self.div.parentNode.removeChild(self.div);
                }, 600);
            }
        }
        OU._popupHighlander = null;
    };
    /**
     * Handles the manual scroll bar
     * @private
     */
    OU.util.FunctionEditor.prototype.handleScroll = function() {
        if (!this._scroll || this.h === 'auto')
            return;
        var contentHeight = this.div.scrollHeight,
                scrollTop = this.div.scrollTop,
                h = this.h,
                visiblePerc = h / contentHeight,
                barTopPerc = scrollTop / contentHeight,
                barTop = h * barTopPerc,
                barHeight = h * visiblePerc;

        this._scroll.clear();
        if (contentHeight <= h)
            return;
        this.hasScrollBar = true;
        this._scrollctx.fillStyle = "rgba(128,128,128,0.2)";
        this._scrollctx.fillRect(5, 0, 10, h);
        this._scrollctx.fillStyle = "rgba(0,0,0,0.2)";
        this._scrollctx.fillRect(5, barTop, 10, barHeight);
    };
    this.init(params);
};
