/**
 * @fileOverview TabSlide - Left side tab that slides a bigger panel in to show further info
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Div');
/**
 * @class TabSlide - Left side tab that slides a bigger HTML panel in to show further info.
 * Multiple tabs can be used at once. The zIndex is used to "activate" each tab
 * 
 * @param {object} params - options:
 * <ul>
 * <li><strong>{object} container:</strong> (required) The calling object, typically an OU.util.Acitivity </li>
 * <li><strong>{int} y:</strong> Y co-ordinate</li>
 * <li><strong>{int} w:</strong> Width</li>
 * <li><strong>{int} h:</strong> Height</li>
 * <li><strong>{string} title:</strong> Title to display on the tab</li>
 * <li><strong>{string} tabCol:</strong> Background colour of the tab (Default: '#444')</li>
 * <li><strong>{string} textCol:</strong> Font colour of the tab (Default: '#fff')</li>
 * <li><strong>{int} zIndex:</strong> zIndex of the tab (useful when multiple tabs used at once.</li>
 * </ul> 
 */
OU.util.TabSlide = function( params ) {
    var bH = OU.controlHeight;

    this.params = params;
    this.title = params.title || "Info";
    this.html = params.html || "<h1>Oops!</h1><p>Should be some content here...</p>";
    this.tabCol = params.tabCol || '#444';
    this.textCol = params.textCol || '#fff';
    this.container = params.container;
    this.zIndex = params.zIndex || OU.DATA_LEVEL+1,
    this.w = params.w || this.container.w-2*bH;
    this.h = params.h || this.container.h-2*bH;
    this.x = bH-this.w-1;
    this.y = params.y || bH;
    this.isOpen=false;

    /**
     * @private
     */
    OU.util.TabSlide.prototype.init = function() {
        this.div = new OU.util.Div({
            x:this.x,
            y:this.y,
            w:this.w,
            h:this.h,
            htmlClass: "tabSlide",
            zIndex: this.zIndex,
            container: this.container
        });

        if(OU._tSlides===undefined)
            OU._tSlides=[];
        OU._tSlides.push(this);
        this.slideIdx = OU._tSlides.length-1;

        this.div.div.innerHTML = "<div onclick='OU._tSlides["+this.slideIdx+"].toggle();' class='tabSlideTab tsClosed' style='background: "
            + this.tabCol+"; color: "+this.textCol+"; width:"+(this.h-30)+"px; margin:"+(this.h/2-20)+"px -"+(this.h/2-20)+"px;'>"
            + this.title+"<span style='padding-left:30px;'>\u2195</span></div><div class='content'>"
            + this.html +'</div>';
    };
    /**
     * Activate this tab.
     * Changes the zIndex of this tab to move it to the top of all tabs currently being used
     */
    OU.util.TabSlide.prototype.toTop = function() {
        var i,slide;
        for(i=OU._tSlides.length; i--;) {
            slide=OU._tSlides[i];
            slide.div.opacity(0);
            slide.div.zIndex(slide.zIndex);
            slide.targX=slide.x=bH-slide.w-1;
            slide.div.div.style.left = (bH-slide.w-1)+'px';
        }
        this.div.opacity(1);
        this.div.zIndex(OU.CONTROL_LEVEL-1);
    };
    /**
     * Animates the slide of the tab in and out
     * @private
     */
    OU.util.TabSlide.prototype.slide = function() {
        var self = this;
        OU.disableResize();
        if(this.x-this.targX<0.01 && this.x-this.targX>-0.01) {
            this.x = this.targX;
            OU.enableResize();
            return;
        }
        this.x = this.x + (this.targX-this.x)/2;
        this.div.div.style.left=this.x+'px';
        setTimeout(function() {
            self.slide();
        },40);
    };
    /**
     * Open or close the tab
     */
    OU.util.TabSlide.prototype.toggle = function() {
        if(this.isOpen)
            this.targX=bH-this.w-1;
        else
            this.targX=(this.container.w-this.w)/2;
        this.slide();
        this.isOpen = !this.isOpen;
    };
    this.init();
};
