/**
 * @fileOverview Audio with Cue points
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 * @author Will Rawes
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Div');
OU.require('OU.util.Layer');
/**
 * @class Audio activity with Cue points
 * @extends OU.util.Activity
 * @param {object} data - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.AudioCues = function(data,instance,controller) {

    /**
     * Starting point when running with HTML5 Canvas Support
     */
    OU.activity.AudioCues.prototype.canvasView = function() {
        this.bgLayer = new OU.util.Layer({
            container:this,
            hasEvents: true
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer

        this.subtitleson_ = OU.LocalStorage.load("OU.control.subtitlesOn")==='true' || false;

        this.initAudio();
    };
    /**
     * Tidy up when the activity is removed
     */
    OU.activity.AudioCues.prototype.remove = function() {
        OU.activity.AudioCues.superClass_.remove.call(this); // call the superclass method
        if(this.subToggleHandler) {
            this._subButton.removeEventListener("mousedown",this.subToggleHandler , false);
            this._subButton.removeEventListener("touchstart",this.subToggleHandler , false);
        }
        if(this.transToggleHandler) {
            this._transButton.removeEventListener("mousedown",this.transToggleHandler , false);
            this._transButton.removeEventListener("touchstart",this.transToggleHandler , false);
        }
    };
    /**
     * Resize the acitivy and related divs
     */
    OU.activity.AudioCues.prototype.resize = function() {
        OU.activity.AudioCues.superClass_.resize.call(this); // call the parent class resize

        this.bgLayer.resize();
        this.bgLayer.context.gradRect();
        if(this.htmlDiv)
            this.htmlDiv.resize({
                x: this.x,
                y:this.y,
                w:this.w,
                h:'auto'
            });
        if(this.audioDiv)
            this.audioDiv.resize({
                x:0,
                y:this.y+this.h-this.audioDiv.div.clientHeight,
                w:this.w,
                h:'auto'
            });
        if(this.subtitleDiv) {
            this.subtitleDiv.resize({
                x: OU.baseController.x+20,
                y: OU.baseController.y+OU.baseController.h*.8,
                w: OU.baseController.w-40,
                h: 'auto'
            });
        }
        if(this.transcriptDiv && this.transcriptDiv.html.div) {
            this.transcriptDiv.remove();
        }
    };
    /**
     * Initial the divs, audio element and event handlers
     */
    OU.activity.AudioCues.prototype.initAudio = function() {
        var i,audio = this.data.audio,self=this,audioElement,vY;

        this.htmlDiv = new OU.util.Div({
            x: this.x,
            y:this.y,
            w:this.w,
            h:'auto',
            container:this
        });
        if (this.data.html && this.data.html!=='') {
            this.htmlDiv.div.innerHTML = this.data.html;
        }
        if (this.htmlDiv)
            vY = this.y + this.htmlDiv.div.clientHeight + 10;
        else
            vY = this.y + (this.h - 40) / 2;

        this.audioDiv = new OU.util.Div({
            x:0,
            y:this.y+this.h-OU.controlHeight,
            w:this.w,
            h:'auto',
            container:this
        });

        this.time=0;
        this.cues = this.data.audio.cues;
        if(this.cues) {
            this.cues.sort(function(a,b) {
                return b.start-a.start;
            });
        }

        audioElement = document.createElement('audio');
        audioElement.setAttribute('id', '_caud_' + this.instance);
        audioElement.setAttribute('style', 'float:left; margin: auto 10px;');
        audioElement.controls = "controls";
        for (i = audio.sources.length; i--;) {
            var source = document.createElement('source');
            source.type = audio.sources[i].type;
            source.src = audio.sources[i].src;
            audioElement.appendChild(source);
        }
        audioElement.load();
        this.audioDiv.div.appendChild(audioElement);
        this.audio = audioElement;
        this.audio.play();

        this.audio.addEventListener("timeupdate", function() {
            self.timeUpdate();
        }, false);

        if(audio.hasSubTitles) {
            this._subButton = document.createElement('span');
            this.subToggleHandler = function() {
                self.toggleSubtitles();
            };
            this._subButton.setAttribute("id", "subtitleButton_");
            if(this.subtitleson_)
                this._subButton.setAttribute("class", "subtitleButton subtitlesOn");
            else
                this._subButton.setAttribute("class", "subtitleButton");
            this._subButton.setAttribute("title", "Toggle Subtitles");
            this._subButton.innerHTML="Subs";
            this._subButton.addEventListener("mousedown",this.subToggleHandler , false);
            this._subButton.addEventListener("touchstart",this.subToggleHandler , false);
            this.audioDiv.div.appendChild(this._subButton);
        }
        if(audio.hasTranscript) {
            this._transButton = document.createElement('span');
            this.transToggleHandler = function() {
                self.toggleTranscript();
            };
            this._transButton.setAttribute("id", "subtitleButton_");
            this._transButton.setAttribute("class", "subtitleButton");
            this._transButton.setAttribute("title", "Open Transcript");
            this._transButton.innerHTML="Transcript";
            this._transButton.addEventListener("mousedown",this.transToggleHandler , false);
            this._transButton.addEventListener("touchstart",this.transToggleHandler , false);
            this.audioDiv.div.appendChild(this._transButton);
        }

        this.audioDiv.resize({
            x:0,
            y:this.y+this.h-this.audioDiv.div.clientHeight,
            w:this.w,
            h:'auto'
        });

        if(this.cues)
            this.watch();
    };
    /**
     * Turns subtitles on and off
     * @param {boolean} onoff - unsurprisingly true for on, false for off!
     */
    OU.activity.AudioCues.prototype.toggleSubtitles = function (onoff) {
        var subButton = document.getElementById("subtitleButton_");
        this.subtitleson_=onoff===undefined?!this.subtitleson_:onoff;
        OU.LocalStorage.save("OU.control.subtitlesOn",this.subtitleson_);
        if(subButton) {
            if(this.subtitleson_)
                subButton.setAttribute("class", "subtitleButton subtitlesOn");
            else
                subButton.setAttribute("class", "subtitleButton");
        }
    };
    /**
     * Show the audio transcript in a pop-up div.
     * If pop-up already open, then close it.
     */
    OU.activity.AudioCues.prototype.toggleTranscript = function () {
        var c,i,h="",self=this;
        if(this.transcriptDiv) {
            this.transcriptDiv.remove();
            this.transcriptDiv=null;
        }
        else {
            if(this.cues) {
                for(i=this.cues.length; i--;) {
                    c=this.cues[i];
                    if(c.subtitle)
                        h = h + "<p>"+c.subtitle+"</p>";
                }
                if(h!=="") {
                    this.transcriptDiv = new OU.util.PopUpInfo({
                        container: this,
                        txt:"<h1>Transcript</h1>"+h,
                        onClose: function() {
                            self.transcriptDiv=null;
                        }
                    });
                }
            }
        }
    };
    /**
     * Jump the audio to the point in time
     * @param {double} timePoint - time in seconds
     */
    OU.activity.AudioCues.prototype.jumpTo = function(timePoint) {
        this.audio.currentTime = timePoint;
    };
    /**
     * Monitoring loop that compares the current time against the array of cue points.
     */
    OU.activity.AudioCues.prototype.watch = function() {
        var i,cues=this.cues,c,t=this.time,self=this;
        if(!cues)
            return;
        for(i=cues.length; i--;) {
            c=cues[i];
            c.start=c.startTime || c.start; //backward compatible
            c.end=c.endTime || c.end; //backward compatible
            if(c.start && c.start<t) {
                if (c.end && c.end>t) {
                    if(c.html && c.html!==this._lastHTML) {// retained for backward compatability only - we should really be using activity Steps rather than inserting raw html
                        this.htmlDiv.div.innerHTML=this._lastHTML=c.html;
                    }
                    if(this.subtitleson_ && c.subtitle) {
                        if(!this.subtitleDiv) {
                            this.subtitleDiv = new OU.util.Div({
                                container:this,
                                x: OU.baseController.x+20,
                                y: OU.baseController.y+OU.baseController.h*.8,
                                w: OU.baseController.w-40,
                                h: 'auto',
                                zIndex:OU.ABSOLUTE_TOP,
                                htmlClass:"OUSubtitles",
                                showScroll:false
                            });
                        }
                        this.subtitleDiv.div.innerHTML=c.subtitle;
                    }
                    else if(this.subtitleDiv) {
                        this.subtitleDiv.remove();
                        this.subtitleDiv=undefined;
                    }
                }
            }
        }

        setTimeout(function() {
            self.watch();
        },40);

    };
    /**
     * Updates the activities reference of the current time, called by the audio event handler
     */
    OU.activity.AudioCues.prototype.timeUpdate = function() {
        this.time = this.audio.currentTime;
    };
    OU.base(this,data,instance,controller);
};
OU.inherits(OU.activity.AudioCues,OU.util.Activity);