/**
 * @fileOverview Quiz Activity - multiple choice
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

// Load in any util elements that are required
OU.require('OU.util.Div');
OU.require('OU.util.ImageLoader');

OU.activity.Quiz = function (data, instance, controller) {
    this.standardOptions = data.standardOptions;
    this.activityOptions = data.activityOptions;
    this.activityContent = data.activityContent;
    
    OU.activity.Quiz.prototype.canvasView = function() {
        OU.activity.Quiz.superClass_.canvasView.call(this); // call the parent class canvasView
        
        this.config = this.verifyOptions(this.activityOptions);
        this.rounds = this.verifyContent(this.activityContent);
        
        //console.log("config:\n" + JSON.stringify(this.config, null, 2));
        //console.log("rounds:\n" + JSON.stringify(this.rounds, null, 2));
        
        this.currentRound         = 0;
        this.roundLimit           = 0; // Clamped to the first round; to disable the clamp use: this.rounds.length - 1;
        this.suppressRoundEndText = true;
        
        if ((0 === this.roundLimit) && (this.rounds.length > 1)) {
            console.log("Warning: The quiz activity is currently limited to one round of questions, subsequent rounds will be ignored.");
        }
        
        this.currentQuestion      = 0;
        this.displayAnswer        = null;
        
        this.quizDiv = new OU.util.Div({
            container: this
        });
        
        this.render();
    };
    
    OU.activity.Quiz.prototype.accessibleView = function() {
        OU.activity.Quiz.superClass_.accessibleView.call(this); // call the superclass method is you want to extend it, remove this line to override the method instead
    };
    
    OU.activity.Quiz.prototype.resize = function() {
        OU.activity.Quiz.superClass_.resize.call(this); // call the superclass resize
        
        if (this.quizDiv !== null) {
            this.quizDiv.resize({
                x: this.x,
                y: this.y,
                w: this.w,
                h: this.h
            });
            this.render();
        }
    };
    
    OU.activity.Quiz.prototype.remove = function() {
        OU.activity.Quiz.superClass_.remove.call(this); // call the superclass remove
        
        if (this.quizDiv !== null) {
            this.quizDiv.remove();
            this.quizDiv = null;
        }
    };
    
    
    
    
    
    OU.activity.Quiz.prototype.verifyObject = function (object, specification) {
        var inSpecification = function (property) {
            var index = null;
            var i;
            for (i = 0; i < specification.length; i++) {
                if (property === specification[i].name) {
                    index = i;
                    break;
                }
            }
            return index;
        };

        var hasCorrectType = function (specificationIndex, obj, property) {
            var specificationEntry = specification[specificationIndex];
            var hasType = false;
            
            if ("array" === specificationEntry.type) {
                hasType = Array.isArray(obj[property]);
            }
            else {
                hasType = (typeof obj[property] === specificationEntry.type);
            }
            
            return hasType;
        };
        
        var verifiedObject = {};
        var approved       = true;
        var errorMessage   = null;
        
        Object.keys(object).forEach(function (property, index) {
            if (!approved) {
                // Abort further processing if we've already fallen foul of something in the specification.
                return;
            }

            var specificationIndex = inSpecification(property);
            if (null === specificationIndex) {
                errorMessage = "unknown property '" + property + "'";
                approved     = false;
                return;
            }

            if (!hasCorrectType(specificationIndex, object, property)) {
                errorMessage = "unexpected type for '" + property + "' - should be '" + specification[specificationIndex].type + "'";
                approved     = false;
                return;
            }
            
            verifiedObject[property] = object[property];
        });
        
        return {
            verifiedObject: verifiedObject,
            errorMessage:   errorMessage       
        };
    };

    OU.activity.Quiz.prototype.verifyOptions = function (rawOptions) {
        var optionsSpecification = [
            {
                name: "endText",
                type: "string"
            }
        ];
        
        var verification = this.verifyObject(rawOptions, optionsSpecification);
        
        var result = {};
        
        if (verification.errorMessage !== null) {
            console.log("Quiz.verifyOptions() Warning: options failed verification, " + verification.errorMessage);
        }
        else {
            result = verification.verifiedObject;
        }
        
        return result;
    };
    
    OU.activity.Quiz.prototype.verifyContent = function (rawContent) {
        var roundSpecification = [
            {
                name: "title",
                type: "string"
            },
            {
                name: "questions",
                type: "array"
            }
        ];
        
        var results = [];
        
        var self = this;
        rawContent.forEach(function (item, index) {
            var verification = self.verifyObject(item, roundSpecification);
            
            if (verification.errorMessage !== null) {
                console.log("Quiz.verifyContent() Warning: dropping round " + index + " for failing verification, " + verification.errorMessage);
            }
            else {
                results.push(item);
            }
        });
        
        return results;        
    };
    
    OU.activity.Quiz.prototype.nextQuestion = function () {
        var round = this.rounds[this.currentRound];
        
        if (this.currentQuestion + 1 <= round.questions.length - 1) {
            this.currentQuestion++;
            this.displayAnswer = null;
        }
        else {
            // No more questions in this round, move to the next round;
            this.currentRound++;
            this.currentQuestion = 0;
            this.displayAnswer   = null;
        }
    };
    
    OU.activity.Quiz.prototype.restartQuiz = function () {
        this.currentRound    = 0;
        this.currentQuestion = 0;
        this.displayAnswer   = null;
    };
    
    OU.activity.Quiz.prototype.render = function () {
        var self = this;
        
        var disableSelection = function (element) {
            var styles = [
                "-webkit-touch-callout: none;",
                "-webkit-user-select: none;",
                "-khtml-user-select: none;",
                "-moz-user-select: none;",
                "-ms-user-select: none;",
                "user-select: none;"
            ];

            element.setAttribute("style", styles.join(" "));
        };
        
        var renderChosenAnswer = function (rootElement, answer) {
            while (rootElement.lastChild) {
                rootElement.removeChild(rootElement.lastChild);
            }

            var choiceElement = document.createElement("div");
            choiceElement.style.marginBottom = "6px";
            choiceElement.style.position     = "relative";

            var one = document.createElement("span");
            var two = document.createElement("span");
            one.textContent     = "You chose: ";
            one.style.fontStyle = "italic";
            two.textContent     = answer.answer;
            
            var close = document.createElement("span");
            close.textContent = "Retry";
            disableSelection(close);
            close.style.cssFloat     = "right";
            close.style.border       = "1px solid black";
            close.style.borderRadius = "10px";
            close.style.paddingLeft  = "10px";
            close.style.paddingRight = "10px";
            close.style.cursor       = "pointer";
            
            OU.events.addListener(close, (function () {
                return function (event) {
                    event.preventDefault();
                    self.displayAnswer = null;
                    self.render();
                };
            }()), "tap");
            
            choiceElement.appendChild(one);
            choiceElement.appendChild(two);
            choiceElement.appendChild(close);

            var verdictElement = document.createElement("div");
            verdictElement.textContent = (0 === answer.score) ? "Incorrect" : "Correct";
            verdictElement.style.marginBottom = "6px";
            
            var feedbackElement = document.createElement("div");
            feedbackElement.textContent = answer.feedback;
            
            rootElement.appendChild(choiceElement);
            rootElement.appendChild(verdictElement);
            rootElement.appendChild(feedbackElement);
        };
        
        this.quizDiv.clear();
        
        // Enable the vertical scrollbar where it becomes necessary.
        this.quizDiv.div.style.overflowY = "auto";
        
        var topContainer = document.createElement("div");
        topContainer.style.padding = "20px";
        
        if (this.currentRound > this.roundLimit) {
            var endTextElement    = document.createElement("div");
            endTextElement.textContent = this.config.endText;
            endTextElement.style.marginBottom = "60px";
            topContainer.appendChild(endTextElement);
            this.quizDiv.div.appendChild(topContainer);
            var restartButtonContainer = document.createElement("div");
            var restartButtonElement   = document.createElement("span");
            restartButtonElement.textContent = "Restart";
            disableSelection(restartButtonElement);
            restartButtonElement.style.cursor       = "pointer";
            restartButtonElement.style.border       = "1px solid black";
            restartButtonElement.style.borderRadius = "10px";
            restartButtonElement.style.textAlign    = "center";
            restartButtonElement.style.paddingLeft  = "10px";
            restartButtonElement.style.paddingRight = "10px";
            restartButtonContainer.style.textAlign  = "center";
            OU.events.addListener(restartButtonElement, (function () {
                return function (event) {
                    event.preventDefault();
                    self.restartQuiz();
                    self.render();
                };
            }()), "tap");
            restartButtonContainer.appendChild(restartButtonElement);
            topContainer.appendChild(restartButtonContainer);
            return;
        }
        
        var round = this.rounds[this.currentRound];
        
        // If the provided round title is empty, display the round number.
        var roundTitleText = (0 === round.title.length) ? ("Round " + (this.currentRound + 1)) : round.title;
        
        var roundProgress = {
            current: this.currentQuestion + 1,
            ofTotal: round.questions.length
        };
        
        var questionText = "Q" + roundProgress.current + " of " + roundProgress.ofTotal + ": " + round.questions[this.currentQuestion].question;
        
        var activityTitleElement    = document.createElement("div");
        var roundTitleElement       = document.createElement("div");
        var questionElement         = document.createElement("div");
        var answersElement          = document.createElement("div");
        var continueButtonContainer = document.createElement("div");
        var continueButtonElement   = document.createElement("span");
        
        activityTitleElement.textContent         = this.standardOptions.title;
        activityTitleElement.style.marginBottom  = "10px";
        activityTitleElement.style.fontSize      = "20px";
        activityTitleElement.style.fontWeight    = "bold";
        roundTitleElement.textContent            = roundTitleText;
        roundTitleElement.style.marginBottom     = "10px";
        roundTitleElement.style.fontSize         = "16px";
        roundTitleElement.style.fontWeight       = "bold";
        questionElement.textContent              = questionText;
        questionElement.style.marginBottom       = "30px";
        answersElement.style.marginBottom        = "40px";
        continueButtonElement.textContent        = "Continue";
        disableSelection(continueButtonElement);
        continueButtonElement.style.cursor       = "pointer";
        continueButtonElement.style.border       = "1px solid black";
        continueButtonElement.style.borderRadius = "10px";
        continueButtonElement.style.textAlign    = "center";
        continueButtonElement.style.paddingLeft  = "10px";
        continueButtonElement.style.paddingRight = "10px";
        continueButtonContainer.style.textAlign  = "center";
        
        OU.events.addListener(continueButtonElement, (function () {
            return function (event) {
                event.preventDefault();
                self.nextQuestion();
                self.render();
            };
        }()), "tap");
        
        var answerInstructionElement = document.createElement("div");
        answerInstructionElement.textContent         = "Select one answer";
        answerInstructionElement.style.fontStyle     = "italic";
        answerInstructionElement.style.marginBottom  = "6px";
        answersElement.appendChild(answerInstructionElement);
        
        if (this.displayAnswer !== null) {
            renderChosenAnswer(answersElement, this.displayAnswer);
        }
        else {
            var answers = round.questions[this.currentQuestion].answers;
            answers.forEach(function (answer, index) {
                var answerElement = document.createElement("div");
                answerElement.textContent = answer.answer;
                disableSelection(answerElement);
                answerElement.style.cursor = "pointer";
                if (index !== answer.length - 1) {
                    answerElement.style.marginBottom = "4px";
                }

                OU.events.addListener(answerElement, (function (answer) {
                    return function (event) {
                        event.preventDefault();
                        self.displayAnswer = answer;
                        renderChosenAnswer(answersElement, answer);
                    };
                }(answer)), "tap");

                answersElement.appendChild(answerElement);
            });
        }
        
        topContainer.appendChild(activityTitleElement);
        
        if (false === this.suppressRoundEndText) {
            topContainer.appendChild(roundTitleElement);
        }
        
        topContainer.appendChild(questionElement);
        topContainer.appendChild(answersElement);
        continueButtonContainer.appendChild(continueButtonElement);
        topContainer.appendChild(continueButtonContainer);
        
        this.quizDiv.div.appendChild(topContainer);
    };
    
    
    // Call the superclass's constructor
    OU.base(this, data, instance, controller);
};

// Call our inherits function to implement the class inheritance
OU.inherits(OU.activity.Quiz, OU.util.Activity);
