/**
 * @fileOverview TileZoom - "Tiled" image viewer - views images generated by the Google Maps Image Cutter
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.Slider');
OU.require('OU.util.Layer');
OU.require('OU.util.TileViewer');

KEY_LEFT = 37;
KEY_UP = 38;
KEY_RIGHT = 39;
KEY_DOWN = 40;
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.TileZoom = function(data, instance, controller) {

    OU.activity.TileZoom.prototype.canvasView = function() {
        var bH = OU.controlHeight, self = this;
        this.bgLayer = new OU.util.Layer({
            container: this,
            id: 'bg'
        });
        this.bgLayer.context.gradRect();
        this.imageLayer = new OU.util.Layer({
            container: this,
            id: 'canvas',
            h: this.h - bH,
            hasEvents: true,
            pinch: this.pinch,
            pinchMe: this
        });
        this.controlsLayer = new OU.util.Layer({
            container: this,
            id: 'oucontrols',
            y: this.h - bH,
            h: bH,
            hasEvents: true,
            keyPress: function(k) {
                self.keyPress(k);
            }
        });
        this.tileView = new OU.util.TileViewer({
            imageBase: this.data.activityContent.images,
            zoomLevels: this.data.activityContent.zoomLevels,
            window: {
                w: this.w,
                h: this.h - bH
            },
            container: this,
            context: this.imageLayer.context,
            useGoogleMapsFileConvention: this.data.GoogleFileConvention,
            zoomCallback: this.setSlider
        });
        this.imageLayer.events.clickable.push(this.tileView);
        this.initControls();
    };
    OU.activity.TileZoom.prototype.keyPress = function(key) {
        var dX = 0, dY = 0;
        switch (key) {
            case KEY_UP:
                dY = -10;
                break;
            case KEY_DOWN:
                dY = 10;
                break;
            case KEY_LEFT:
                dX = -10;
                break;
            case KEY_RIGHT:
                dX = 10;
                break;
        }
        this.tileView.x += dX;
        this.tileView.y += dY;
        this.tileView.doRender = true;
    };
    OU.activity.TileZoom.prototype.resize = function() {
        OU.activity.TileZoom.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight;
        this.bgLayer.resize();
        this.bgLayer.context.gradRect();
        this.imageLayer.resize({
            h: this.h - bH
        });
        this.controlsLayer.resize({
            y: this.h - bH,
            h: bH
        });
        this.zoomSlider.resize({
            x: 0,
            y: 0,
            w: this.w,
            h: bH
        });
        this.tileView.w = {
            w: this.w,
            h: this.h - bH
        };
        this.tileView.render();
    };
    OU.activity.TileZoom.prototype.initControls = function() {
        var ctx = this.controlsLayer.context, bH = OU.controlHeight,
                clickable = this.controlsLayer.events.clickable;
        clickable.length = 0;
        this.tileView.measureOn = false;
        this.zoomSlider = new OU.util.Slider({
            container: this,
            sliderPos: 0,
            instance: 'zoom' + this.instance,
            x: 0,
            y: 0,
            w: this.w,
            h: bH,
            sliderHeight: bH / 2,
            drawContainer: false,
            callback: this.setZoom,
            callbackParam: this,
            background: {
                clear: true
            },
            context: ctx
        });
        clickable.push(this.zoomSlider);
        this.renderSlider();
    };
    OU.activity.TileZoom.prototype.renderControls = function() {
        this.controlsLayer.context.clearRect(0, 0, this.w, OU.controlHeight);
        this.renderSlider();
    };
    OU.activity.TileZoom.prototype.renderSlider = function() {
        var bH = OU.controlHeight, self = this;
        this.controlsLayer.context.clearRect(bH * 2 * this.data.activityContent.images.length, 0, this.w - bH * 2 * (1 + this.data.activityContent.images.length), bH);

        this.zoomSlider.render();
        setTimeout(function() {
            self.renderSlider();
        }, 40);
    };
    OU.activity.TileZoom.prototype.setZoom = function(val, me) { // Called when scrubBar is moved
        me.zoomSlider.render();
        me.tileView.scale(val);
    };
    OU.activity.TileZoom.prototype.setSlider = function(val, me) { // Called when TileViewer changed via mousewheel, etc.
        me.zoomSlider.setTarget(val);
    };
    OU.activity.TileZoom.prototype.pinch = function(e, s, x, y, dx, dy, me) { // called when pinch event detected
        var ns = ((e - s) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x: x,
            y: y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.TileZoom, OU.util.Activity);
