/**
 * @fileOverview Queue - Allows user to order elements in a queue and gives feedback against a possible order
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Div');
OU.require('OU.util.Events');
OU.require('OU.util.Draggable');
OU.require('OU.util.ModalMessage');
OU.require('OU.util.ModalInfo');

KEY_LEFT = 37;
KEY_UP = 38;
KEY_RIGHT = 39;
KEY_DOWN = 40;

/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Queue = function(data, instance, controller) {
    // TODO: Maybe seperate the directional options rather than inline if statements
    OU.activity.Queue.prototype.view = function() {
        this.direction = (typeof this.data.activityOptions.direction === 'string') ? this.data.activityOptions.direction : 'vertical';

        this.maxAttempts = (typeof this.data.activityOptions.attempts !== 'undefined') ? this.data.activityOptions.attempts : this.data.activityContent.length;
        this.attempts = this.maxAttempts;
        this.attempting = false;
        this.itemMoved = false;
        this.showHelper = false;

        this.setSizes();

        this.bgLayer = new OU.util.Div({
            container: this,
            showScroll: false
        });

        this.layer = new OU.util.Div({
            container: this.bgLayer,
            style: "background-color: " + (typeof this.data.activityOptions.bgColor === 'string' ? this.data.activityOptions.bgColor : '#ccc') + "; border-radius: " + this.listArea.padding + "px;",
            showScroll: false
        });
        this.layer.events = new OU.util.Events({
            target: this.layer.div
        });

        this.renderBackdrop();

        if (this.data.activityContent.length) {
            this.items = [];
            for (var i = 0; i < this.data.activityContent.length; i++) {
                this.items.push(new this.Item({
                    queue: this,
                    item: this.data.activityContent[i],
                    order: i
                }));
            }

            this.items = this.shuffle(this.items);

            this.renderItems();
        }
    };

    OU.activity.Queue.prototype.checkAnswers = function() {
        var correct = 0;
        var incorrect = 0;

        for (var i = 0; i < this.items.length; i++) {
            var item = this.items[i];
            if (i === item.order) {
                correct++;
            } else {
                incorrect++;
            }
            if (this.attempts <= 0 && this.showHelper) {
                var feedbackPosition = typeof this.data.activityOptions.feedback.position !== 'undefined' ? this.data.activityOptions.feedback.position : 'top right';

                var status = document.createElement('div');
                status.style.position = 'absolute';
                status.status = (i === item.order ? '' : 'in') + 'correct';

                for (var fP = 0; fP < feedbackPosition.split(' ').length; fP++) {
                    status.style[feedbackPosition.split(' ')[fP]] = '2px';
                }
                var statusImage = document.createElement('img');
                statusImage.setAttribute("src", OU.cssPath + (i === item.order ? '' : 'in') + "correct.png");
                statusImage.style.maxWidth = '100%';
                statusImage.style.maxHeight = '100%';
                status.appendChild(statusImage);

                if (typeof item.layer.div.lastChild.status !== 'undefined') {
                    item.layer.div.replaceChild(status, item.layer.div.lastChild);
                } else {
                    item.layer.div.appendChild(status);
                }
            }
        }

        if (incorrect > 0) {
            if (this.attempts === 0) {
                this.attempts--;

                var answer = document.createElement('div');
                answer.style.textAlign = 'center';

                var accept = document.createElement('button');
                accept.textContent = this.data.activityOptions.feedback.helper.options.accept;
                accept.style.padding = '5px 15px';
                accept.style.marginRight = '40px';
                accept.style.marginBottom = '15px';
                accept.style.borderRadius = '15px';
                accept.style.background = '#fff';

                var decline = document.createElement('button');
                decline.textContent = this.data.activityOptions.feedback.helper.options.decline;
                decline.style.padding = '5px 15px';
                decline.style.marginBottom = '15px';
                decline.style.borderRadius = '15px';
                decline.style.background = '#fff';

                answer.appendChild(accept);
                answer.appendChild(decline);

                OU.events.addListener(accept, function(self) {
                    return function() {
                        self.showHelper = true;
                        self.checkAnswers();

                        self.message.close();
                    };
                }(this));

                OU.events.addListener(decline, function(self) {
                    return function() {
                        self.attempts = self.maxAttempts;

                        self.message.close();
                    };
                }(this));

                this.message = new OU.util.ModalMessage({
                    html: '<div style="padding: 10px; font-size: 20px; text-align: center;">' + this.data.activityOptions.feedback.helper.message.join('<br />') + '</div>',
                    noModalClose: true
                });

                this.message.messageLayer.div.appendChild(answer);
            }
        } else {
            if (!this.showHelper) {
                this.showHelper = true;
                this.attempts = -1;
                this.checkAnswers();
            } else {
                if (!OU.modalInfo) {
                    OU.modalInfo = new OU.util.ModalInfo();
                }
                OU.modalInfo.view({
                    html: this.data.activityOptions.feedback.correct,
                    width: this.w * .66
                });
            }
        }
    };

    OU.activity.Queue.prototype.Item = function(params) {
        this.queue = params.queue || null;
        this.layer = new OU.util.Div({
            container: this.queue.layer,
            style: "overflow:hidden;background-color: " + (typeof this.queue.data.activityOptions.itemBgColor === 'string' ? this.queue.data.activityOptions.itemBgColor : 'rgba(255,255,255,0.5)') + "; border-radius: 5px;",
            showScroll: false,
            htmlClass: 'mousePoint'
        });
        this.hasFocus = false;
        this.startX = this.layer.x;
        this.startY = this.layer.y;
        this.draggable = null;
        this.params = params.item || null;
        this.name = this.params.name || '';
        this.order = params.order || 0;
        this.label = null;
        this.image = null;

        OU.activity.Queue.prototype.Item.prototype.renderLabel = function() {
            this.label = {
                text: this.name,
                layer: new OU.util.Div({
                    container: this.layer,
                    innerHTML: this.name,
                    style: "overflow:hidden;color: " + (typeof this.queue.data.activityOptions.itemColor === 'string' ? this.queue.data.activityOptions.itemColor : '#666') + "; text-align: center;",
                    showScroll: false
                })
            };
        };

        OU.activity.Queue.prototype.Item.prototype.renderImage = function() {
            if (typeof this.params.image !== 'undefined') {
                var position = (typeof this.params.image.position === 'undefined') ? 'first' : this.params.image.position;
                var src = (typeof this.params.image.src === 'undefined') ? '' : this.params.image.src;
                this.image = {
                    src: 'data/' + src,
                    position: position,
                    layer: new OU.util.Div({
                        container: this.layer,
                        style: "overflow:hidden;text-align: center;",
                        showScroll: false
                    })
                };

                var img = document.createElement('img');
                img.setAttribute('src', this.queue.dataDir + src);
                img.style.maxWidth = '100%';
                img.style.maxHeight = '100%';

                this.image.layer.div.appendChild(img);
            }
        };

        this.renderImage();

        if (!this.image || this.image.position !== 'center') {
            this.renderLabel();
        }

        OU.activity.Queue.prototype.Item.prototype.resize = function(p) {
            this.fontSize = null;

            this.layer.resize({
                x: p.x,
                y: p.y,
                h: p.h,
                w: p.w
            });

            this.draggable.x = p.x;
            this.draggable.y = p.y;
            this.draggable.h = p.h;
            this.draggable.w = p.w;

            this.startX = p.x;
            this.startY = p.y;

            if (this.label !== null) {
                this.label.layer.resize({
                    x: p.w * (this.image !== null && this.image.position === 'first' ? (this.queue.direction === 'vertical' ? .34 : 0) : 0),
                    y: p.h * (this.image !== null && this.image.position === 'first' ? (this.queue.direction === 'horizontal' ? .34 : 0) : 0),
                    w: p.w * (this.image !== null ? (this.queue.direction === 'vertical' ? .66 : 1) : 1),
                    h: p.h * (this.image !== null ? (this.queue.direction === 'horizontal' ? .66 : 1) : 1)
                });
            }

            if (this.image !== null) {
                this.image.layer.resize({
                    x: p.w * (this.image.position === 'last' ? (this.queue.direction === 'vertical' ? .67 : 0) : 0),
                    y: p.h * (this.image.position === 'last' ? (this.queue.direction === 'horizontal' ? .67 : 0) : 0),
                    w: p.w * (this.image.position === 'center' ? 1 : (this.queue.direction === 'vertical' ? .33 : 1)),
                    h: p.h * (this.image.position === 'center' ? 1 : (this.queue.direction === 'horizontal' ? .33 : 1))
                });
            }

            if (this.label !== null) {
                this.fontSize = this.label.layer.fitContent();
            }

            return this.fontSize;
        };

        OU.activity.Queue.prototype.Item.prototype.onStart = function(p) {
            p.me.queue.positionItems();

            p.me.queue.itemMoved = false;
            p.me.queue.attempting = true;

            p.me.layer.zOffset = p.me.layer.zIndexStart;
            p.me.layer.zIndexStart += 10;
        };

        OU.activity.Queue.prototype.Item.prototype.onMove = function(p) {
            p.me.layer.resize({
                x: (p.me.queue.direction === 'vertical') ? p.me.layer.x : p.x,
                y: (p.me.queue.direction === 'horizontal') ? p.me.layer.y : p.y,
                w: p.me.layer.w,
                h: p.me.layer.h
            });

            if (this.hasFocus) {
                p.me.layer.div.style.backgroundColor = "#c00";
            } else {
                p.me.layer.div.style.backgroundColor = "rgba(255,255,255,0.9)";
            }

            var moveItem = null;
            var itemA = null;
            for (var i = 0; i < p.me.queue.items.length; i++) {
                var itemB = p.me.queue.items[i];
                if (itemA !== null) {
                    if (itemA.order === p.me.order) {
                        if (itemB.layer[((p.me.queue.direction === 'vertical') ? 'y' : 'x')] <= itemA.layer[((p.me.queue.direction === 'vertical') ? 'y' : 'x')]) {
                            moveItem = itemB;
                            break;
                        }
                    }
                    if (itemB.order === p.me.order) {
                        if (itemA.layer[((p.me.queue.direction === 'vertical') ? 'y' : 'x')] >= itemB.layer[((p.me.queue.direction === 'vertical') ? 'y' : 'x')]) {
                            moveItem = itemA;
                            break;
                        }
                    }
                }
                itemA = p.me.queue.items[i];
            }

            p.me.queue.items.sort(function(a, b) {
                return a.layer[((p.me.queue.direction === 'vertical') ? 'y' : 'x')] - b.layer[((p.me.queue.direction === 'vertical') ? 'y' : 'x')];
            });

            if (moveItem !== null) {
                p.me.queue.itemMoved = true;
                if (p.me.queue.attempting) {
                    p.me.queue.attempts--;
                    p.me.queue.attempting = false;
                }
                var index = 0;
                for (var i = 0; i < p.me.queue.items.length; i++) {
                    index = i;
                    if (p.me.queue.items[i].order === moveItem.order) {
                        break;
                    }
                }

                moveItem.layer.resize({
                    x: (p.me.queue.direction === 'horizontal') ? (p.me.queue.listArea.padding + (p.me.queue.listArea.itemSize * index)) : moveItem.layer.x,
                    y: (p.me.queue.direction === 'vertical') ? (p.me.queue.listArea.padding + (p.me.queue.listArea.itemSize * index)) : moveItem.layer.y,
                    w: moveItem.layer.w,
                    h: moveItem.layer.h
                });
            }
        };

        OU.activity.Queue.prototype.Item.prototype.onEnd = function(p) {
            p.me.layer.zIndexStart = p.me.layer.zOffset;
            p.me.queue.checkAnswers();
            p.me.queue.positionItems();
        };

        this.draggable = new OU.util.Draggable({
            "me": this,
            "events": this.queue.layer.events,
            "x": this.layer.x,
            "y": this.layer.y,
            "h": this.layer.h,
            "w": this.layer.w,
            "onStart": this.onStart,
            "onEnd": this.onEnd,
            "onMove": this.onMove
        });

        this.queue.layer.events.clickable.push(this.draggable);

        OU.activity.Queue.prototype.Item.prototype.arrowKey = function(k) {
            var x = this.layer.x;
            var y = this.layer.y;
            switch (k) {
                case KEY_UP:
                    if (this.queue.direction === 'vertical') {
                        y = this.layer.y - 10;
                        y = y < 0 ? 0 : y;
                    }
                    break;
                case KEY_DOWN:
                    if (this.queue.direction === 'vertical') {
                        y = this.layer.y + 10;
                        y = y > this.queue.layer.h - this.layer.h ? this.queue.layer.h - this.layer.h : y;
                    }
                    break;
                case KEY_LEFT:
                    if (this.queue.direction === 'horizontal') {
                        x = this.layer.x - 10;
                        x = x < 0 ? 0 : x;
                    }
                    break;
                case KEY_RIGHT:
                    if (this.queue.direction === 'horizontal') {
                        x = this.layer.x + 10;
                        x = x > this.queue.layer.w - this.layer.w ? this.queue.layer.w - this.layer.w : x;
                    }
                    break;
            }

            this.onMove({
                x: x,
                y: y,
                me: this
            });
        };

        OU.activity.Queue.prototype.Item.prototype.focus = function() {
            this.hasFocus = true;
            this.layer.div.style.backgroundColor = "#c00";
            if (this.label) {
                this.label.layer.div.style.color = "#fff";
            }
        };

        OU.activity.Queue.prototype.Item.prototype.blur = function() {
            this.hasFocus = false;
            this.layer.div.style.backgroundColor = "rgba(255,255,255,0.5)";
            if (this.label) {
                this.label.layer.div.style.color = "#666";
            }
            this.queue.positionItems();
        };

        OU.base(this, params);
    };
    OU.inherits(OU.activity.Queue.prototype.Item, OU.util.Tabbable);

    OU.activity.Queue.prototype.shuffle = function(items) {
        var tmp, current, top = items.length;
        if (top)
            while (--top) {
                current = Math.floor(Math.random() * (top + 1));
                tmp = items[current];
                items[current] = items[top];
                items[top] = tmp;
            }

        return items;
    };

    OU.activity.Queue.prototype.renderBackdrop = function() {
        var background = this.data.activityOptions.background;
        if (background) {
            if (background === 'transparent') {
                this.bgLayer.div.style.backgroundColor = 'rgba(255, 255, 255, 0.9)';
            } else {
                this.bgLayer.div.style.background = 'url(' + this.dataDir + background + ') #FFF';
                this.bgLayer.div.style.backgroundSize = 'cover';
            }
        }

        this.layer.resize({
            x: this.listArea.x,
            y: this.listArea.y,
            w: this.listArea.w,
            h: this.listArea.h
        });
    };

    OU.activity.Queue.prototype.renderItems = function() {
        this.minFontSize = null;
        for (var i = 0; i < this.items.length; i++) {
            var item = this.items[i];

            item.position(item, (i + 1));

            var fontSize = item.resize({
                x: this.listArea.padding + ((this.direction === 'horizontal') ? (this.listArea.itemSize * i) : 0),
                y: this.listArea.padding + ((this.direction === 'vertical') ? (this.listArea.itemSize * i) : 0),
                w: (this.direction === 'horizontal') ? this.listArea.itemSize * .9 : this.listArea.w - this.listArea.padding * 2,
                h: (this.direction === 'vertical') ? this.listArea.itemSize * .9 : this.listArea.h - this.listArea.padding * 2
            });

            this.minFontSize = (this.minFontSize) ? (fontSize && fontSize < this.minFontSize) ? fontSize : this.minFontSize : fontSize;
        }

        if (this.minFontSize === null) {
            this.minFontSize = 13;
        }

        for (var i = 0; i < this.items.length; i++) {
            var item = this.items[i];

            if (item.label !== null) {
                item.label.layer.div.style.fontSize = this.minFontSize + 'px';
                item.label.layer.div.style.lineHeight = this.minFontSize + 'px';
            }
        }
    };

    OU.activity.Queue.prototype.positionItems = function() {
        for (var i = 0; i < this.items.length; i++) {
            var item = this.items[i];

            item.position(item, (i + 1));

            item.layer.resize({
                x: this.listArea.padding + ((this.direction === 'horizontal') ? (this.listArea.itemSize * i) : 0),
                y: this.listArea.padding + ((this.direction === 'vertical') ? (this.listArea.itemSize * i) : 0),
                w: (this.direction === 'horizontal') ? this.listArea.itemSize * .9 : this.listArea.w - this.listArea.padding * 2,
                h: (this.direction === 'vertical') ? this.listArea.itemSize * .9 : this.listArea.h - this.listArea.padding * 2
            });

            item.draggable.x = item.layer.x;
            item.draggable.y = item.layer.y;
            item.draggable.h = item.layer.h;
            item.draggable.w = item.layer.w;

            item.startX = item.layer.x;
            item.startY = item.layer.y;
        }
    };

    OU.activity.Queue.prototype.setSizes = function() {
        var minWidth = 360;
        var maxWidth = 800;

        var maxSizeWidth = null;
        var maxSizeHeight = null;
        if (typeof this.data.activityOptions.width !== 'undefined') {
            maxSizeWidth = parseInt(this.data.activityOptions.width);
        }
        if (typeof this.data.activityOptions.height !== 'undefined') {
            maxSizeHeight = parseInt(this.data.activityOptions.height);
        }

        var percentage = ((this.w - minWidth) / (maxWidth - minWidth));
        var min = Math.min(this.h / 40, this.w / 20);
        var width = ((this.w < minWidth) ? this.w : (this.w < maxWidth) ? (this.w * (1 + percentage * (.5 - 1))) : this.w * .5) - this.w * .05;
        var height = this.h * .9;

        this.listArea = {
            x: (maxSizeWidth && width > maxSizeWidth) ? (this.w - maxSizeWidth) / 2 : this.w * .025 + ((this.w < minWidth) ? 0 : (this.w < maxWidth) ? (this.w * (0 + percentage * (.25 - 0))) : this.w * .25),
            y: (maxSizeHeight && height > maxSizeHeight) ? (this.h - maxSizeHeight) / 2 : this.h * .05,
            w: (maxSizeWidth && width > maxSizeWidth) ? maxSizeWidth : width,
            h: (maxSizeHeight && height > maxSizeHeight) ? maxSizeHeight : height,
            padding: min,
            itemSize: (((this.direction === 'horizontal') ? ((maxSizeWidth) ? (width > maxSizeWidth) ? maxSizeWidth : width : width) : ((maxSizeHeight) ? (height > maxSizeHeight) ? maxSizeHeight : height : height)) - min * 2) / this.data.activityContent.length
        };
    };

    OU.activity.Queue.prototype.resize = function() {
        OU.activity.Queue.superClass_.resize.call(this); // call the parent class resize
        this.setSizes();

        this.bgLayer.resize();

        this.renderBackdrop();
        this.renderItems();
    };

    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Queue, OU.util.Activity);
