/**
 * @fileOverview Div - Adds a div to the body, and positions accordingly
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @class Adds a DIV element to the page
 *
 * @param {object} params, various params to define the behaviour:
 * <ul>
 * <li><strong>{OU.util.Activity} container</strong>: (required) reference to the parent activity</li>
 * <li><strong>{string} innerHTML</strong>: (optional) html to go inside the div as it is inserted into the page.</li>
 * <li><strong>x</strong>: x position (optional, defaults to 0)</li>
 * <li><strong>y</strong>: y position (optional, defaults to 0)</li>
 * <li><strong>w</strong>: width (optional, defaults to width of container)</li>
 * <li><strong>h</strong>: height (optional, defaults to width of container)</li>
 * <li><strong>{int} zIndex</strong>: (optional) zIndex to apply to this div(Gets added to the activity zOffset</li>
 * <li><strong>{string} style</strong>: (optional) CSS style to add to the div</li>
 * <li><strong>{boolean} showScroll </strong>: (optional, defaults to true) adds a visible scroll bar, as some mobile devices don't show by default </li>
 * <li><strong>{string} overflow</strong>: (optional) CSS overflow value</li>
 * <li><strong>{DOM object} beforeDiv</strong>: (optional) insert the new Div before this one, defaults to end of current body</li>
 * </ul>
 */
OU.util.Div = function(params) {
    var _transform = "";
    if (params === undefined)
        params = {};
    this.div = null;
    this.container = params.container || {};
    this.zIndexStart = params.zIndex || OU.DATA_LEVEL;
    this.zOffset = this.container.zOffset || 0;
    this.zIndexStart = this.zIndexStart + this.zOffset;
    this.params = params;
    this.style = params.style || '';
    this.addStyle = '';
    this._scroll = null;
    this.showScroll = params.showScroll === undefined ? true : params.showScroll;
    if (params.overflow)
        this.addStyle = 'overflow:' + params.overflow + '; ';
    /**
     * sets everything up and adds the div element into the parent DOM object
     * @private
     */
    OU.util.Div.prototype.init = function(params) {
        var self = this, container = this.container, target,
                x = params.x || 0,
                y = params.y || 0,
                w = params.w || container.w || window.innerWidth || 480,
                h = params.h || container.h || window.innerHeight || 360,
                hp = h;
        if (h !== "auto") {
            hp = h + 'px';
        }

        this.x = x;
        this.y = y;
        this.w = w;
        this.h = h;

        this.div = document.createElement("div");
        this.div.setAttribute("style", this.addStyle + 'left:' + x + 'px; top:' + y + 'px;' + 'width:' + w + 'px; height:' + hp + '; ' + this.style);
        this.div.setAttribute("class", "_overlaydiv " + (this.params.htmlClass || ''));
        this.div.style.zIndex = this.zIndexStart;
        if (!params.screenWrapper) {
            //OU.ensureScreenWrapped();
        }
        var parentDiv = OU._screenWrapper ? OU._screenWrapper.div : document.body;

        target = params.parentDiv || this.container.div || parentDiv;
        if (params.beforeDiv) {
            target.insertBefore(this.div, document.getElementById(params.beforeDiv));
        }
        else {
            target.appendChild(this.div);
        }
        if (this.params.innerHTML)
            this.div.innerHTML = this.params.innerHTML;
        if (this.showScroll && OU.IS_IPAD && parentDiv !== document.body) {
            this._scroll = new OU.util.Layer({
                container: this.container,
                zIndex: 10000,
                x: x + w,
                y: y,
                w: 20,
                h: h
            });
            this._scrollctx = this._scroll.context;
            this._scrollListener = function() {
                self.handleScroll();
            };
            this.div.addEventListener("scroll", this._scrollListener, false, 0, true);
            this.div.addEventListener("DOMSubtreeModified", this._scrollListener, false, 0, true);
            this.handleScroll();
        }

        if (params.dontRegister === undefined || params.dontRegister === false) {
            OU.addRemovable(this, this.zOffset);
        }
        if (params.transform) {
            this.transform(params.transform);
        }
    };
    /**
     * Resizes the div
     * @param {object} p - optional resize parameters, all default to current value if not specified:
     * <ul>
     * <li><strong>x</strong>: x position</li>
     * <li><strong>y</strong>: y position</li>
     * <li><strong>w</strong>: width </li>
     * <li><strong>h</strong>: height</li>
     * </ul>
     */
    OU.util.Div.prototype.resize = function(p) {
        var p = p || {};
        var x = typeof p.x === 'undefined' ? this.x : p.x,
                y = typeof p.y === 'undefined' ? this.y : p.y,
                w = p.w || this.container.w || window.innerWidth || 480,
                h = p.h || this.container.h || window.innerHeight || 360,
                hp = h;

        if (x === this.x && y === this.y && w === this.w && h === this.h)
            return; // nothing to do
        this.x = x = x | 0;
        this.y = y = y | 0;
        this.w = w = w | 0;
        this.h = h = h | 0;
        if (h !== "auto") {
            hp = h + 'px';
        }
        this.div.setAttribute("style", this.addStyle + 'left:' + x + 'px; top:' + y + 'px;' + 'width:' + w + 'px; height:' + hp + '; ' + this.style);
        if (this.zIndexStart)
            this.div.style.zIndex = this.zIndexStart;
        if (this._scroll) {
            this._scroll.resize({
                x: x + w,
                y: y,
                w: 20,
                h: h
            });
            this.handleScroll();
        }
    };
    /**
     * @deprecated - should use resize
     * @private
     */
    OU.util.Div.prototype.move = function(p) {
        return this.resize(p);
    };
    /**
     * Animated move of the div to a new location
     *
     * @param {object} p - optional new size/location parameters, all default to current value if not specified:
     * <ul>
     * <li><strong>x</strong>: x position</li>
     * <li><strong>y</strong>: y position</li>
     * <li><strong>w</strong>: width </li>
     * <li><strong>h</strong>: height</li>
     * </ul>     */
    OU.util.Div.prototype.slideTo = function(p) {
        this.slideTargetX = p.x || this.x;
        this.slideTargetY = p.y || this.y;
        this.slideTargetW = p.w || this.w;
        this.slideTargetH = p.h || this.h;
        if (this.moving === undefined) {
            this.moving = true;
            this.animate();
        }
    };
    /**
     * performs to actual movement of the slideTo function
     * @private
     */
    OU.util.Div.prototype.animate = function() {
        var self = this, p = {
            x: this.x + ((this.slideTargetX - this.x) / 2) | 0,
            y: this.y + ((this.slideTargetY - this.y) / 2) | 0,
            w: this.w + ((this.slideTargetW - this.w) / 2) | 0,
            h: this.h + ((this.slideTargetH - this.h) / 2) | 0
        };
        if (Math.abs(this.x - this.slideTargetX) > 2 || Math.abs(this.y - this.slideTargetY) > 2) {
            this.move(p);
            setTimeout(function() {
                self.animate();
            }, 40);
        }
        else {
            this.move({
                x: this.slideTargetX,
                y: this.slideTargetY,
                w: this.slideTargetW,
                h: this.slideTargetH
            });
            this.moving = undefined;
        }
    };
    /**
     * Updates the div's contents with new html
     */
    OU.util.Div.prototype.html = function(html) {
        this.div.innerHTML = html;
    };
    /**
     * Updates the div's contents with new html
     */
    OU.util.Div.prototype.appendChild = function(obj) {
        this.div.appendChild(obj);
    };
    OU.util.Div.prototype.fitContent = function() {
        var self = this;
        var getFontSize = function() {
            return parseInt(document.defaultView.getComputedStyle(self.div, null).getPropertyValue('font-size').slice(0, -2));
        };
        var resizeText = function(inc) {
            var fontSize = getFontSize();
            var elNewFontSize = (fontSize - (inc ? -6 : 6)) + 'px';

            self.div.style.fontSize = elNewFontSize;
            self.div.style.lineHeight = elNewFontSize;
        };

        while (this.div.scrollHeight <= this.div.offsetHeight || this.div.scrollWidth <= this.div.offsetWidth) {
            resizeText(true);
        }
        while (this.div.scrollHeight > this.div.offsetHeight || this.div.scrollWidth > this.div.offsetWidth) {
            resizeText();
        }
        return getFontSize();
    };
    /**
     * Set new opacity value for the div
     * @param {double} o - new value between 0 and 1
     */
    OU.util.Div.prototype.opacity = function(o) {
        this.div.style.opacity = o;
        this.div.style.MozOpacity = o;
    };
    /**
     * Sets new zIndex for the div
     * @param {int} z - new zindex value
     */
    OU.util.Div.prototype.zIndex = function(z) {
        console.log(z + this.zOffset);
        this.div.style.zIndex = z + this.zOffset;
    };
    /**
     * Adds a class to the div
     */
    OU.util.Div.prototype.addClass = function(className) {
        OU.addClass(this.div, className);
    };
    /**
     * Removes a class from the div
     */
    OU.util.Div.prototype.removeClass = function(className) {
        OU.removeClass(this.div, className);
    };
    /**
     * Empties the contents of the div
     */
    OU.util.Div.prototype.clear = function() {
        this.div.innerHTML = '';
    };
    /**
     * Removes the div from the page and tidies up event handlers, scroll bars , etc.
     */
    OU.util.Div.prototype.remove = function() {
        // remove scrollbar
        if (this._scroll) {
            this.div.removeEventListener("scroll", this._scrollListener, false);
            this.div.removeEventListener("DOMSubtreeModified", this._scrollListener, false);
            this._scroll.remove();
        }
        this._scrollctx = null;
        this._scroll = null;
        if (this.div.parentNode) {
            this.div.parentNode.removeChild(this.div);
        }
    };
    /**
     * Handles the manual scroll bar
     * @private
     */
    OU.util.Div.prototype.handleScroll = function() {
        if (!this._scroll || this.h === 'auto')
            return;
        var contentHeight = this.div.scrollHeight,
                scrollTop = this.div.scrollTop,
                h = this.h,
                visiblePerc = h / contentHeight,
                barTopPerc = scrollTop / contentHeight,
                barTop = h * barTopPerc,
                barHeight = h * visiblePerc;

        this._scroll.clear();
        if (contentHeight <= h)
            return;
        this.hasScrollBar = true;
        this._scrollctx.fillStyle = "rgba(128,128,128,0.2)";
        this._scrollctx.fillRect(5, 0, 10, h);
        this._scrollctx.fillStyle = "rgba(0,0,0,0.2)";
        this._scrollctx.fillRect(5, barTop, 10, barHeight);
    };

    OU.util.Div.prototype.transform = function(params) {
        var translate = params.translate,
                rotate = params.rotate,
                origin = params.origin,
                rotateBit = "", translateBit = "";
        _transform = "";
        if (translate) {
            translate.x = (translate.x || 0) | 0;
            translate.y = (translate.y || 0) | 0;
            translate.z = (translate.z || 0) | 0;
            if (translate.x !== 0 || translate.y !== 0 || translate.z !== 0) {
                translateBit = " translate3d(" + translate.x + "px," + translate.y + "px," + translate.z + "px)";
            }
        }
        if (origin) {
            origin.x = (origin.x || 0) | 0;
            origin.y = (origin.y || 0) | 0;
            origin.z = (origin.z || 0) | 0;
            if (origin.x !== 0 || origin.y !== 0 || origin.z !== 0) {
                _transform += " transform-origin: " + origin.x + "px " + origin.y + "px " + origin.z + "px;"
                _transform += " -webkit-transform-origin: " + origin.x + "px " + origin.y + "px " + origin.z + "px;"
                _transform += " -moz-transform-origin: " + origin.x + "px " + origin.y + "px " + origin.z + "px;"
                _transform += " -ms-transform-origin: " + origin.x + "px " + origin.y + "px " + origin.z + "px;"
            }
        }
        if (rotate) {
            rotate.x = (rotate.x || 0) | 0;
            rotate.y = (rotate.y || 0) | 0;
            rotate.z = (rotate.z || 0) | 0;
            if (rotate.x !== 0) {
                rotateBit += " rotateX(" + rotate.x + "deg)";
            }
            if (rotate.y !== 0) {
                rotateBit += " rotateY(" + rotate.y + "deg)";
            }
            if (rotate.z !== 0) {
                rotateBit += " rotateZ(" + rotate.z + "deg)";
            }
        }
        if (rotateBit !== "" || translateBit !== "") {
            _transform += " transform:" + rotateBit + translateBit + ";";
            _transform += " -webkit-transform:" + rotateBit + translateBit + ";";
            _transform += " -moz-transform:" + rotateBit + translateBit + ";";
            _transform += " -ms-transform:" + rotateBit + translateBit + ";";
        }


        var hp = this.h;
        if (this.h !== "auto") {
            hp = this.h + 'px';
        }
        this.div.setAttribute("style", this.addStyle + 'left:' + this.x + 'px; top:' + this.y + 'px;' + 'width:' + this.w + 'px; height:' + hp + '; ' + this.style + _transform);
        if (this.zIndexStart) {
            this.div.style.zIndex = this.zIndexStart;
        }
    };
    this.init(params);
};
