/**
 * @fileOverview ImageLoader - loads images into a JSON structure and calls a function on completion
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 *
 */
/**
 * @class Recurse down through a given json object and load any images that are found.
 * Requires images to be identified using field names as "image" (formerly either "fileName" or "img")
 * Once all images are loaded, a given callback function is called.
 * Every element in "data" that has an image reference will have an additional field "imgElement" that is a DOM image object // formerly "image"
 *
 * @param {object} params - Options:
 * <ul>
 * <li><strong>{object} data:</strong> (required) The json structure to tranverse</li>
 * <li><strong>{function} onLoad:</strong> (required) The callback function, called when loading is complete</li>
 * <li><strong>{object} container:</strong> (required) A reference to the calling container, typically an OU.util.Activity</li>
 * <li><strong>{object} progressWindow:</strong> (optional) parameters for a DynText call, to move the progress message to a particular location, etc.</li>
 * </ul>
 */
OU.util.ImageLoader = function(params) {
    var LOAD_TIMEOUT = 60000;
    this.required = 0;
    this.loaded = 0;
    this.complete = false;
    this.jItems = [];
    this.started = new Date().getTime();
    this.onLoadFn = params.onLoad;
    this.container = params.container || {};
    this.data = params.data;
    this.progressWindow = params.progressWindow;
    this.halted = false;
    OU.addRemovable(this, this.container.zOffset);
    /**
     * @private
     */
    OU.util.ImageLoader.prototype.init = function() {
        this.progressLayer = new OU.util.Layer({
            container: this.container,
            'id': 'loadingLayer' + this.container.instance,
            zIndex: OU.CONTROLLER_LEVEL - 10
        });
        var ctx = this.progressLayer.context;
        ctx.font = 16 * OU.dpr + 'px serif';
        ctx.textAlign = 'center';
        ctx.strokeStyle = '#999';
        ctx.fillStyle = '#0c0';
        this.dataDir = this.container.dataDir;
        this.progress();
        this.load(this.data);
        while (this.jItems.length > 0) {
            this.load(this.jItems.pop());
        }
        this.wait();
    };
    /**
     * @private
     * @param {object or array} obj
     */
    OU.util.ImageLoader.prototype.load = function(obj) {
        var i;
        var child;
        var fileName;
        var self = this;
        if (!((typeof obj === "object") || Array.isArray(obj)) || (null === obj)) {
            return;
        }

        if (Array.isArray(obj)) {
            for (i = obj.length; i--; ) {
                child = obj[i];
                if (typeof child === 'object' || Array.isArray(child)) {
                    this.jItems.push(child);
                }
            }
        }
        else if (typeof obj === 'object') {
            Object.keys(obj).forEach(function(key) {
                child = obj[key];
                if (child && !child._imageLoaderBeenHere && (typeof child === 'object' || Array.isArray(child))) {
                    self.jItems.push(child);
                }
            });
        }
        fileName = (typeof obj.image === "string") ? obj.image : "";
        if (fileName !== "") {
            this.required++;
            obj.imgElement = new Image();
            obj.imgElement.onload = function() {
                self.loaded++;
                self.progress();
            };
            obj.imgElement.src = this.dataDir + fileName;
        }
        obj._imageLoaderBeenHere = true; // dataset may contain references to itself, so keep track of processed objects
    };
    /**
     * @private
     */
    OU.util.ImageLoader.prototype.wait = function() {
        if (this.halted)
            return;
        var self = this, now = new Date().getTime();
        if (this.loaded >= this.required) {
            this.progressLayer.remove();
            this.complete = true;
            this.onLoadFn();
        }
        else {
            if (now - this.started > LOAD_TIMEOUT) {
                alert("There was a problem loading files - load time exceeded. This activity may not work correctly.");
                this.progressLayer.remove();
                this.onLoadFn();
            }
            else {
                setTimeout(function() {
                    self.wait();
                }, 40);
            }
        }
    };
    /**
     * Removes the loader.
     * Halts the wait loop and ensures the onLoad function is never called
     */
    OU.util.ImageLoader.prototype.remove = function() {
        this.required++; // ensure never triggers onLoadFn
        this.halted = true;
        this.data = undefined;
    };
    /**
     * Displays the progress so far
     * @private
     */
    OU.util.ImageLoader.prototype.progress = function() {
        var pl = this.progressLayer, ctx = pl.context,
                perc = this.loaded / this.required * 100 | 0,
                x = pl.w / 4,
                y = pl.h / 2 - 40 * OU.dpr, w = pl.w / 2, h = 10 * OU.dpr;
        ctx.clearRect(0, 0, this.progressLayer.w, this.progressLayer.h);
        if (this.progressWindow) {
            this.progressWindow.txt = 'Loading... ' + perc + '%';
            this.progressWindow.context = ctx;
            new OU.util.DynText(this.progressWindow);
        }
        else {
            ctx.rect(x, y, w, h);
            ctx.stroke();
            ctx.fillRect(x, y, w * perc / 100, h);
            ctx.save();
            ctx.fillStyle = '#222';
            ctx.fillText('Loading... ' + perc + '%', this.progressLayer.w / 2, this.progressLayer.h / 2);
            ctx.restore();
        }
    };
    this.init();
};


