/**
 * @fileOverview Layer Object
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Events');
OU.require('OU.util.Div');

/**
 * @class A 'layer' object, adds a canvas element to the page and gives us extra functionality to make manipulting the canvas easier
 *
 * @param {object} params - provides the following initialisation options (optional unless otherwise specified):
 * <ul>
 * <li><strong>{object} container</strong>: (required) a reference back to the activity/container that initialised this layer</li>
 * <li><strong>{int} zIndex</strong>: specify a zIndex to change the z-order relative to other layers. Note this will be added to the activity zOffset</li>
 * <li><strong>{boolean} hasEvents</strong>: if true an event handler (OU.util.Events) will be added</li>
 * <li><strong>{boolean} dontRegister</strong>: if true then the layer will not be added to the removables array and therefore not removed automatically by the controller </li>
 * <li><strong>{string} id</strong>: an identifier which is added to the canvas tag (mainly for debug purposes)</li>
 * <li><strong>{int} x</strong>: the X co-ordinate of the canvas relative to the activity</li>
 * <li><strong>{int} y</strong>: the Y co-ordinate of the canvas relative to the activity</li>
 * <li><strong>{int} w</strong>: the width of the canvas in pixels</li>
 * <li><strong>{int} h</strong>: the height of the canvas in pixels</li>
 * <li><strong>{DOM element} insertInside</strong>: a DOM object (DIV, etc.) to insert the canvas into, defaults to the normal page</li>
 * <li><strong>{DOM element} beforeDiv</strong>: a DOM object (DIV, etc.) to insert the canvas before, defaults to ust appending to the normal page </li>
 * <li><strong>{string} themeClass</strong>: add a style class, to apply specific CSS to the canvas tag</li>
 * </ul>
 */
OU.util.Layer = function(params) {
    params = params || {};
    this.container = params.container || {};
    this.canvas = null;
    this.context = null;
    this.events = null;
    this._zIndex = params.zIndex === undefined ? OU.DATA_LEVEL : params.zIndex;
    this.zOffset = this.container.zOffset || 0;
    this._zIndex = this._zIndex + this.zOffset;
    var _transform = "";
    OU.util.Layer.prototype.init = function() {
        if (params.canvas) { // use an existing canvas
            this.externalCanvas = true;
            this.canvas = params.canvas;
            this.context = this.canvas.getContext("2d");
            this.context.lineWidth = 0.75;
            this.context.textBaseline = 'middle';
        }
        else {
            this.addCanvas(params);
        }
        if (params.hasEvents) {
            this.events = new OU.util.Events({
                target: this.canvas,
                pinch: params.pinch,
                pinchMe: params.pinchMe,
                keyPress: params.keyPress
            });
        }
        if (params.dontRegister === undefined) {
            OU.addRemovable(this, this.zOffset);
        }
        if (params.transform) {
            this.transform(params.transform);
        }
    };
    OU.util.Layer.prototype.transform = function(params) {
        var translate = params.translate,
                rotate = params.rotate,
                origin = params.origin,
                rotateBit = "", translateBit = "", originBit = "";
        _transform = "";
        if (translate) {
            translate.x = (translate.x || 0) | 0;
            translate.y = (translate.y || 0) | 0;
            translate.z = (translate.z || 0) | 0;
            if (translate.x !== 0 || translate.y !== 0 || translate.z !== 0) {
                translateBit = " translate3d(" + translate.x + "px," + translate.y + "px," + translate.z + "px)";
            }
        }
        if (origin) {
            origin.x = (origin.x || 0) | 0;
            origin.y = (origin.y || 0) | 0;
            origin.z = (origin.z || 0) | 0;
            if (origin.x !== 0 || origin.y !== 0 || origin.z !== 0) {
                _transform += " transform-origin: " + origin.x + "px " + origin.y + "px " + origin.z + "px;"
                _transform += " -webkit-transform-origin: " + origin.x + "px " + origin.y + "px " + origin.z + "px;"
                _transform += " -moz-transform-origin: " + origin.x + "px " + origin.y + "px " + origin.z + "px;"
                _transform += " -ms-transform-origin: " + origin.x + "px " + origin.y + "px " + origin.z + "px;"
            }
        }
        if (rotate) {
            rotate.x = (rotate.x || 0) | 0;
            rotate.y = (rotate.y || 0) | 0;
            rotate.z = (rotate.z || 0) | 0;
            if (rotate.x !== 0) {
                rotateBit += " rotateX(" + rotate.x + "deg)";
            }
            if (rotate.y !== 0) {
                rotateBit += " rotateY(" + rotate.y + "deg)";
            }
            if (rotate.z !== 0) {
                rotateBit += " rotateZ(" + rotate.z + "deg)";
            }
        }
        if (rotateBit !== "" || translateBit !== "") {
            _transform += " transform:" + rotateBit + translateBit + ";";
            _transform += " -webkit-transform:" + rotateBit + translateBit + ";";
            _transform += " -moz-transform:" + rotateBit + translateBit + ";";
            _transform += " -ms-transform:" + rotateBit + translateBit + ";";
        }

        this.canvas.setAttribute("style", 'left:' + this.x + 'px; top:' + this.y + "px; " + _transform);
        if (this._zIndex) {
            this.canvas.style.zIndex = this._zIndex;
        }
    };
    /**
     * Resizes the layer (ie the canvas element within the page).
     * Action is performed in a single move
     * @param {object} param - optional new dimensions
     * <ul>
     * <li><strong>x</strong>: left position in Pixels (defaults to current value or 0)</li>
     * <li><strong>y</strong>: top position in pixels (defaults to current value or 0)</li>
     * <li><strong>w</strong>: width in pixels (defaults to current value or width of containing activity)</li>
     * <li><strong>h</strong>: height in pixels (defaults to current value or height of containing activity)</li>
     * </ul>
     */
    OU.util.Layer.prototype.resize = function(param) {
        if (!this.externalCanvas) { // don't resize external canvases
            this.move(this.getDims(param));
            this.context.textBaseline = 'middle';
        }
    };
    /**
     * actually performs the resize/move
     * @private
     */
    OU.util.Layer.prototype.move = function(p) {
        this.x = p.x === undefined ? this.x : p.x;
        this.y = p.y === undefined ? this.y : p.y;
        this.w = p.w || this.w;
        this.h = p.h || this.h;
        this.x = this.x | 0;
        this.y = this.y | 0;
        this.w = this.w | 0;
        this.h = this.h | 0;
        this.canvas.setAttribute("style", 'left:' + this.x + 'px; top:' + this.y + 'px;');
        if (p.h)
            this.canvas.setAttribute("height", this.h);
        if (p.w)
            this.canvas.setAttribute("width", this.w);
        if (this._zIndex)
            this.canvas.style.zIndex = this._zIndex;
        this.context.textBaseline = 'middle';
    };
    /**
     * Performs an animated move of the layer/canvas to a new X,Y position
     * @param {object} param - X,Y position, ie {x:10,y:20}
     */
    OU.util.Layer.prototype.slideTo = function(param) {
        this.slideTargetX = param.x === undefined ? this.x : param.x;
        this.slideTargetY = param.y === undefined ? this.y : param.y;
        if (this.moving === undefined) {
            this.moving = true;
            this.animate();
        }
    };
    /**
     * performs the actual animation of a slideTo call
     * @private
     */
    OU.util.Layer.prototype.animate = function() {
        var self = this, p = {
            x: this.x + ((this.slideTargetX - this.x) / 8) | 0,
            y: this.y + ((this.slideTargetY - this.y) / 8) | 0
        };
        if (this.canvas) {
            if (Math.abs(this.x - this.slideTargetX) > 2 || Math.abs(this.y - this.slideTargetY) > 2) {
                this.move(p);
                setTimeout(function() {
                    self.animate();
                }, 40);
            }
            else {
                this.move({
                    x: this.slideTargetX,
                    y: this.slideTargetY
                });
                this.moving = undefined;
            }
        }
    };
    /**
     * Returns the new dimensions from a given params object, using defaults if not present
     * @private
     */
    OU.util.Layer.prototype.getDims = function(params) {
        params = params || {};
        return {
            x: params.x === undefined ? (this.container.x || 0) : params.x,
            y: params.y === undefined ? (this.container.y || 0) : params.y,
            w: params.w || (this.container ? this.container.w : false) || window.innerWidth || 480,
            h: params.h || (this.container ? this.container.h : false) || window.innerHeight || 360
        };
    };
    /**
     * Inserts the canvas element into the page
     * @private
     */
    OU.util.Layer.prototype.addCanvas = function(params) {
        params = params || {};
        var container = this.container,
                dims = this.getDims(params),
                x = dims.x | 0,
                y = dims.y | 0,
                w = dims.w | 0,
                h = dims.h | 0,
                instance = container.instance || '';
        this.id = params.id || '';
        this.canvas = document.createElement("canvas");
        this.canvas.setAttribute("id", instance + this.id);
        this.canvas.setAttribute("style", 'left:' + x + 'px; top:' + y + 'px;');
        this.canvas.setAttribute("height", h);
        this.canvas.setAttribute("width", w);
        params.themeClass = params.themeClass || "";
        this.canvas.setAttribute("class", "fastease " + params.themeClass);

        if (this._zIndex) {
            this.canvas.style.zIndex = this._zIndex;
        }
//        OU.ensureScreenWrapped();
        var parentDiv = OU._screenWrapper ? OU._screenWrapper.div : document.body;
        if (parentDiv === document.body) {
            console.error('WARNING: Initialising a Layer (canvas) with no screenwrapper');
        }
        this.parentNode = params.insertInside || params.parentDiv || parentDiv;
        if (params.beforeDiv !== undefined) {
            this.parentNode.insertBefore(this.canvas, document.getElementById(params.beforeDiv));
        }
        else {
            this.parentNode.appendChild(this.canvas);
        }
        this.context = this.canvas.getContext("2d");
        this.context.lineWidth = 0.75;
        this.context.textBaseline = 'middle';
        OU.closeButton(container);
        this.x = x;
        this.y = y;
        this.w = w;
        this.h = h;
    };
    /**
     * Sets the opacity of the layer/canvas using css style
     * @param {double} opacity value of opacity between 0 (transparent) and 1 (fully opaque)
     */
    OU.util.Layer.prototype.opacity = function(opacity) {
        this.canvas.style.opacity = opacity;
    };
    /**
     * Sets the zIndex of the layer/canvas
     * @param {int} z new zIndex value
     */
    OU.util.Layer.prototype.zIndex = function(z) {
        this._zIndex = z;
        this.canvas.style.zIndex = z;
    };
    /**
     * Clears the canvas - equivalent to calling context.clearRect(x,y,w,h)
     * Defaults to the entire canvas area if parameters omitted
     * @param {int} x - optional x position of the rectangle to clear
     * @param {int} y - optional y position of the rectangle to clear
     * @param {int} w - optional width of the rectangle to clear
     * @param {int} h - optional height of the rectangle to clear
     */
    OU.util.Layer.prototype.clear = function(x, y, w, h) {
        if (this.context)
            this.context.clearRect(x || 0, y || 0, w || this.w, h || this.h);
    };
    /**
     * Removes the layer and nulls some of the internal elements to aid garbage collection
     */
    OU.util.Layer.prototype.remove = function() {
        if (this.events) {
            this.events.remove();
        }
        if (this.canvas && this.parentNode) {
            this.parentNode.removeChild(this.canvas);
        }
        delete this.canvas;
        delete this.context;
        delete this.events;
        this.canvas = null;
        this.context = null;
        this.container = null;
    };
    this.init();
};